/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.internal.blobstore;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Provider;
import javax.inject.Singleton;
import org.sonatype.nexus.blobstore.BlobStoreDescriptor;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreConfiguration;
import org.sonatype.nexus.blobstore.api.BlobStoreCreatedEvent;
import org.sonatype.nexus.blobstore.api.BlobStoreDeletedEvent;
import org.sonatype.nexus.blobstore.api.BlobStoreException;
import org.sonatype.nexus.blobstore.api.BlobStoreManager;
import org.sonatype.nexus.blobstore.api.BlobStoreUpdatedEvent;
import org.sonatype.nexus.blobstore.file.FileBlobStoreConfigurationBuilder;
import org.sonatype.nexus.common.event.EventAware;
import org.sonatype.nexus.common.event.EventManager;
import org.sonatype.nexus.common.node.NodeAccess;
import org.sonatype.nexus.common.stateguard.Guarded;
import org.sonatype.nexus.common.stateguard.StateGuardLifecycleSupport;
import org.sonatype.nexus.jmx.reflect.ManagedObject;
import org.sonatype.nexus.orient.freeze.DatabaseFreezeService;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationCreatedEvent;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationDeletedEvent;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationEvent;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationStore;
import org.sonatype.nexus.repository.internal.blobstore.BlobStoreConfigurationUpdatedEvent;
import org.sonatype.nexus.repository.manager.RepositoryManager;

@Named
@Singleton
@ManagedObject
public class BlobStoreManagerImpl
extends StateGuardLifecycleSupport
implements BlobStoreManager,
EventAware {
    private final EventManager eventManager;
    private final Map<String, BlobStore> stores = Maps.newConcurrentMap();
    private final BlobStoreConfigurationStore store;
    private final Map<String, BlobStoreDescriptor> blobStoreDescriptors;
    private final Map<String, Provider<BlobStore>> blobStorePrototypes;
    private final DatabaseFreezeService databaseFreezeService;
    private final BooleanSupplier provisionDefaults;
    private final Provider<RepositoryManager> repositoryManagerProvider;

    @Inject
    public BlobStoreManagerImpl(EventManager eventManager, BlobStoreConfigurationStore store, Map<String, BlobStoreDescriptor> blobStoreDescriptors, Map<String, Provider<BlobStore>> blobStorePrototypes, DatabaseFreezeService databaseFreezeService, Provider<RepositoryManager> repositoryManagerProvider, NodeAccess nodeAccess, @Nullable @Named(value="${nexus.blobstore.provisionDefaults}") Boolean provisionDefaults) {
        this.eventManager = (EventManager)Preconditions.checkNotNull((Object)eventManager);
        this.store = (BlobStoreConfigurationStore)Preconditions.checkNotNull((Object)store);
        this.blobStoreDescriptors = (Map)Preconditions.checkNotNull(blobStoreDescriptors);
        this.blobStorePrototypes = (Map)Preconditions.checkNotNull(blobStorePrototypes);
        this.databaseFreezeService = (DatabaseFreezeService)Preconditions.checkNotNull((Object)databaseFreezeService);
        this.repositoryManagerProvider = (Provider)Preconditions.checkNotNull(repositoryManagerProvider);
        this.provisionDefaults = provisionDefaults != null ? provisionDefaults::booleanValue : () -> !nodeAccess.isClustered();
    }

    protected void doStart() throws Exception {
        List<BlobStoreConfiguration> configurations = this.store.list();
        if (configurations.isEmpty() && this.provisionDefaults.getAsBoolean()) {
            this.log.debug("No BlobStores configured; provisioning default BlobStore");
            this.store.create(new FileBlobStoreConfigurationBuilder("default").build());
            configurations = this.store.list();
        }
        this.log.debug("Restoring {} BlobStores", (Object)configurations.size());
        for (BlobStoreConfiguration blobStoreConfiguration : configurations) {
            this.log.debug("Restoring BlobStore: {}", (Object)blobStoreConfiguration);
            try {
                BlobStore blobStore = this.newBlobStore(blobStoreConfiguration);
                this.track(blobStoreConfiguration.getName(), blobStore);
            }
            catch (Exception e) {
                this.log.error("Unable to restore BlobStore {}", (Object)blobStoreConfiguration, (Object)e);
            }
        }
        this.log.debug("Starting {} BlobStores", (Object)this.stores.size());
        for (Map.Entry entry : this.stores.entrySet()) {
            String name = (String)entry.getKey();
            BlobStore blobStore = (BlobStore)entry.getValue();
            this.log.debug("Starting BlobStore: {}", (Object)name);
            try {
                blobStore.start();
            }
            catch (Exception e) {
                this.log.error("Unable to start BlobStore {}", (Object)name, (Object)e);
            }
        }
    }

    protected void doStop() throws Exception {
        if (this.stores.isEmpty()) {
            this.log.debug("No BlobStores defined");
            return;
        }
        this.log.debug("Stopping {} BlobStores", (Object)this.stores.size());
        for (Map.Entry<String, BlobStore> entry : this.stores.entrySet()) {
            String name = entry.getKey();
            BlobStore store = entry.getValue();
            this.log.debug("Stopping blob-store: {}", (Object)name);
            store.stop();
        }
        this.stores.clear();
    }

    @Guarded(by={"STARTED"})
    public Iterable<BlobStore> browse() {
        return ImmutableList.copyOf(this.stores.values());
    }

    @Guarded(by={"STARTED"})
    public BlobStore create(BlobStoreConfiguration configuration) throws Exception {
        Preconditions.checkNotNull((Object)configuration);
        this.log.debug("Creating BlobStore: {} with attributes: {}", (Object)configuration.getName(), (Object)configuration.getAttributes());
        BlobStoreDescriptor blobStoreDescriptor = this.blobStoreDescriptors.get(configuration.getType());
        blobStoreDescriptor.validateConfig(configuration);
        BlobStore blobStore = this.newBlobStore(configuration);
        try {
            this.store.create(configuration);
        }
        catch (Exception e) {
            try {
                blobStore.remove();
            }
            catch (Exception removeException) {
                this.log.error("Error removing BlobStore {} after create failed", (Object)configuration.getName(), (Object)removeException);
            }
            throw e;
        }
        this.track(configuration.getName(), blobStore);
        blobStore.start();
        this.eventManager.post((Object)new BlobStoreCreatedEvent(blobStore));
        return blobStore;
    }

    @Guarded(by={"STARTED"})
    public BlobStore update(BlobStoreConfiguration configuration) throws Exception {
        Preconditions.checkNotNull((Object)configuration);
        BlobStore blobStore = this.get(configuration.getName());
        Preconditions.checkNotNull((Object)blobStore);
        this.log.debug("Updating BlobStore: {} with attributes: {}", (Object)configuration.getName(), (Object)configuration.getAttributes());
        BlobStoreDescriptor blobStoreDescriptor = this.blobStoreDescriptors.get(configuration.getType());
        blobStoreDescriptor.validateConfig(configuration);
        BlobStoreConfiguration currentConfig = blobStore.getBlobStoreConfiguration();
        blobStore.stop();
        try {
            blobStore.init(configuration);
            blobStore.start();
            this.store.update(configuration);
            this.eventManager.post((Object)new BlobStoreUpdatedEvent(blobStore));
        }
        catch (Exception e) {
            this.log.error("Failed to update configuration", (Throwable)e);
            if (blobStore.isStarted()) {
                blobStore.stop();
            }
            blobStore.init(currentConfig);
            blobStore.start();
            throw new BlobStoreException("Failed to start blob store with new configuration.", null);
        }
        return blobStore;
    }

    @Guarded(by={"STARTED"})
    @Nullable
    public BlobStore get(String name) {
        Preconditions.checkNotNull((Object)name);
        return this.stores.get(name);
    }

    @Guarded(by={"STARTED"})
    public void delete(String name) throws Exception {
        Preconditions.checkNotNull((Object)name);
        if (((RepositoryManager)this.repositoryManagerProvider.get()).isBlobstoreUsed(name)) {
            throw new IllegalStateException("BlobStore " + name + " is in use and cannot be deleted");
        }
        this.forceDelete(name);
    }

    @Guarded(by={"STARTED"})
    public void forceDelete(String name) throws Exception {
        Preconditions.checkNotNull((Object)name);
        this.databaseFreezeService.checkUnfrozen("Unable to delete a BlobStore while database is frozen.");
        BlobStore blobStore = this.blobStore(name);
        this.log.debug("Deleting BlobStore: {}", (Object)name);
        blobStore.stop();
        blobStore.remove();
        this.untrack(name);
        this.store.delete(blobStore.getBlobStoreConfiguration());
        this.eventManager.post((Object)new BlobStoreDeletedEvent(blobStore));
    }

    public boolean exists(String name) {
        return this.stores.keySet().stream().anyMatch(key -> key.equalsIgnoreCase(name));
    }

    private BlobStore newBlobStore(BlobStoreConfiguration blobStoreConfiguration) throws Exception {
        BlobStore blobStore = (BlobStore)this.blobStorePrototypes.get(blobStoreConfiguration.getType()).get();
        blobStore.init(blobStoreConfiguration);
        return blobStore;
    }

    @VisibleForTesting
    BlobStore blobStore(String name) {
        BlobStore blobStore = this.stores.get(name);
        Preconditions.checkState((blobStore != null ? 1 : 0) != 0, (String)"Missing BlobStore: %s", (Object)name);
        return blobStore;
    }

    @VisibleForTesting
    void track(String name, BlobStore blobStore) {
        this.log.debug("Tracking: {}", (Object)name);
        this.stores.put(name, blobStore);
    }

    private void untrack(String name) {
        this.log.debug("Untracking: {}", (Object)name);
        this.stores.remove(name);
    }

    @Subscribe
    public void on(BlobStoreConfigurationCreatedEvent event) {
        this.handleRemoteOnly(event, evt -> {
            String name = evt.getName();
            if (!this.stores.containsKey(name)) {
                this.store.list().stream().filter(c -> c.getName().equals(name)).findFirst().ifPresent(c -> {
                    try {
                        BlobStore blobStore = this.newBlobStore((BlobStoreConfiguration)c);
                        this.track(name, blobStore);
                        blobStore.start();
                    }
                    catch (Exception e) {
                        this.log.warn("create blob store from remote event failed: {}", (Object)name, (Object)e);
                    }
                });
            }
        });
    }

    @Subscribe
    public void on(BlobStoreConfigurationDeletedEvent event) {
        this.handleRemoteOnly(event, evt -> {
            try {
                String name = evt.getName();
                if (this.stores.containsKey(name)) {
                    BlobStore blobStore = this.blobStore(name);
                    blobStore.stop();
                    blobStore.remove();
                    this.untrack(name);
                }
            }
            catch (Exception e) {
                this.log.warn("delete blob store from remote event failed: {}", (Object)evt.getName(), (Object)e);
            }
        });
    }

    @Subscribe
    public void on(BlobStoreConfigurationUpdatedEvent event) {
        this.handleRemoteOnly(event, evt -> {
            try {
                String name = evt.getName();
                if (this.stores.containsKey(name)) {
                    BlobStore blobStore = this.blobStore(name);
                    blobStore.stop();
                    blobStore.init((BlobStoreConfiguration)event.getEntity());
                    blobStore.start();
                }
            }
            catch (Exception e) {
                this.log.warn("update blob store from remote event failed: {}", (Object)evt.getName(), (Object)e);
            }
        });
    }

    private void handleRemoteOnly(BlobStoreConfigurationEvent event, Consumer<BlobStoreConfigurationEvent> consumer) {
        this.log.trace("handling: {}", (Object)event);
        if (!event.isLocal()) {
            consumer.accept(event);
        }
    }

    public long blobStoreUsageCount(String blobStoreName) {
        long count = 0L;
        for (BlobStore otherBlobStore : this.stores.values()) {
            BlobStoreConfiguration otherBlobStoreConfig = otherBlobStore.getBlobStoreConfiguration();
            BlobStoreDescriptor otherBlobStoreDescriptor = this.blobStoreDescriptors.get(otherBlobStoreConfig.getType());
            if (!otherBlobStoreDescriptor.configHasDependencyOn(otherBlobStoreConfig, blobStoreName)) continue;
            ++count;
        }
        return count;
    }

    public boolean isPromotable(String blobStoreName) {
        BlobStore blobStore = this.get(blobStoreName);
        return blobStore != null && blobStore.isGroupable() && blobStore.isWritable() && !this.store.findParent(blobStore.getBlobStoreConfiguration().getName()).isPresent();
    }

    public Optional<String> getParent(String blobStoreName) {
        BlobStore blobStore = this.get(blobStoreName);
        return blobStore == null ? Optional.empty() : this.store.findParent(blobStoreName).map(BlobStoreConfiguration::getName);
    }
}

