/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.blobstore.file;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.hash.HashCode;
import com.squareup.tape.QueueFile;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileStore;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.sonatype.nexus.blobstore.BlobIdLocationResolver;
import org.sonatype.nexus.blobstore.BlobStoreSupport;
import org.sonatype.nexus.blobstore.BlobSupport;
import org.sonatype.nexus.blobstore.StreamMetrics;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobAttributes;
import org.sonatype.nexus.blobstore.api.BlobId;
import org.sonatype.nexus.blobstore.api.BlobMetrics;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreConfiguration;
import org.sonatype.nexus.blobstore.api.BlobStoreException;
import org.sonatype.nexus.blobstore.api.BlobStoreMetrics;
import org.sonatype.nexus.blobstore.api.BlobStoreUsageChecker;
import org.sonatype.nexus.blobstore.file.FileAttributesLocation;
import org.sonatype.nexus.blobstore.file.FileBlobAttributes;
import org.sonatype.nexus.blobstore.file.internal.BlobCollisionException;
import org.sonatype.nexus.blobstore.file.internal.FileBlobStoreMetricsStore;
import org.sonatype.nexus.blobstore.file.internal.FileOperations;
import org.sonatype.nexus.common.app.ApplicationDirectories;
import org.sonatype.nexus.common.io.DirectoryHelper;
import org.sonatype.nexus.common.log.DryRunPrefix;
import org.sonatype.nexus.common.node.NodeAccess;
import org.sonatype.nexus.common.property.PropertiesFile;
import org.sonatype.nexus.common.property.SystemPropertiesHelper;
import org.sonatype.nexus.common.stateguard.Guarded;
import org.sonatype.nexus.logging.task.ProgressLogIntervalHelper;
import org.sonatype.nexus.scheduling.CancelableHelper;
import org.sonatype.nexus.scheduling.TaskInterruptedException;

@Named(value="File")
public class FileBlobStore
extends BlobStoreSupport<FileAttributesLocation> {
    public static final String BASEDIR = "blobs";
    public static final String TYPE = "File";
    public static final String BLOB_CONTENT_SUFFIX = ".bytes";
    @VisibleForTesting
    public static final String CONFIG_KEY = "file";
    @VisibleForTesting
    public static final String PATH_KEY = "path";
    @VisibleForTesting
    public static final String METADATA_FILENAME = "metadata.properties";
    @VisibleForTesting
    public static final String TYPE_KEY = "type";
    @VisibleForTesting
    public static final String TYPE_V1 = "file/1";
    @VisibleForTesting
    public static final String REBUILD_DELETED_BLOB_INDEX_KEY = "rebuildDeletedBlobIndex";
    @VisibleForTesting
    public static final String DELETIONS_FILENAME = "deletions.index";
    private static final boolean RETRY_ON_COLLISION = SystemPropertiesHelper.getBoolean((String)"nexus.blobstore.retryOnCollision", (boolean)true);
    @VisibleForTesting
    static final int MAX_COLLISION_RETRIES = 8;
    private Path contentDir;
    private final FileOperations fileOperations;
    private final Path basedir;
    private FileBlobStoreMetricsStore metricsStore;
    private LoadingCache<BlobId, FileBlob> liveBlobs;
    private QueueFile deletedBlobIndex;
    private final NodeAccess nodeAccess;
    private boolean supportsHardLinkCopy;
    private boolean supportsAtomicMove;

    @Inject
    public FileBlobStore(BlobIdLocationResolver blobIdLocationResolver, FileOperations fileOperations, ApplicationDirectories directories, FileBlobStoreMetricsStore metricsStore, NodeAccess nodeAccess, DryRunPrefix dryRunPrefix) {
        super(blobIdLocationResolver, dryRunPrefix);
        this.fileOperations = (FileOperations)Preconditions.checkNotNull((Object)fileOperations);
        this.basedir = directories.getWorkDirectory(BASEDIR).toPath();
        this.metricsStore = (FileBlobStoreMetricsStore)((Object)Preconditions.checkNotNull((Object)((Object)metricsStore)));
        this.nodeAccess = (NodeAccess)Preconditions.checkNotNull((Object)nodeAccess);
        this.supportsHardLinkCopy = true;
        this.supportsAtomicMove = true;
    }

    @VisibleForTesting
    public FileBlobStore(Path contentDir, BlobIdLocationResolver blobIdLocationResolver, FileOperations fileOperations, FileBlobStoreMetricsStore metricsStore, BlobStoreConfiguration configuration, ApplicationDirectories directories, NodeAccess nodeAccess, DryRunPrefix dryRunPrefix) {
        this(blobIdLocationResolver, fileOperations, directories, metricsStore, nodeAccess, dryRunPrefix);
        this.contentDir = (Path)Preconditions.checkNotNull((Object)contentDir);
        this.blobStoreConfiguration = (BlobStoreConfiguration)Preconditions.checkNotNull((Object)configuration);
    }

    protected void doStart() throws Exception {
        Path storageDir = this.getAbsoluteBlobDir();
        PropertiesFile metadata = new PropertiesFile(storageDir.resolve(METADATA_FILENAME).toFile());
        if (metadata.getFile().exists()) {
            metadata.load();
            String type = metadata.getProperty(TYPE_KEY);
            Preconditions.checkState((boolean)TYPE_V1.equals(type), (String)"Unsupported blob store type/version: %s in %s", (Object)type, (Object)metadata.getFile());
        } else {
            metadata.setProperty(TYPE_KEY, TYPE_V1);
            metadata.store();
        }
        this.liveBlobs = CacheBuilder.newBuilder().weakValues().build(CacheLoader.from(arg_0 -> FileBlob.new(this, arg_0)));
        File deletedIndexFile = storageDir.resolve(this.getDeletionsFilename()).toFile();
        try {
            this.maybeUpgradeLegacyIndexFile(deletedIndexFile.toPath());
            this.deletedBlobIndex = new QueueFile(deletedIndexFile);
        }
        catch (IOException e) {
            this.log.error("Unable to load deletions index file {}, run the compact blobstore task to rebuild", (Object)deletedIndexFile, (Object)e);
            this.createEmptyDeletionsIndex(deletedIndexFile);
            this.deletedBlobIndex = new QueueFile(deletedIndexFile);
            metadata.setProperty(REBUILD_DELETED_BLOB_INDEX_KEY, "true");
            metadata.store();
        }
        this.metricsStore.setStorageDir(storageDir);
        this.metricsStore.setBlobStore((BlobStore)this);
        this.metricsStore.start();
    }

    private void maybeUpgradeLegacyIndexFile(Path deletedIndexPath) throws IOException {
        Path legacyDeletionsIndex = deletedIndexPath.getParent().resolve(DELETIONS_FILENAME);
        if (!Files.exists(deletedIndexPath, new LinkOption[0]) && Files.exists(legacyDeletionsIndex, new LinkOption[0])) {
            this.log.info("Found 'deletions.index' file in blob store {}, renaming to {}", (Object)this.getAbsoluteBlobDir(), (Object)deletedIndexPath);
            Files.move(legacyDeletionsIndex, deletedIndexPath, new CopyOption[0]);
        }
    }

    private String getDeletionsFilename() {
        return String.valueOf(this.nodeAccess.getId()) + "-" + DELETIONS_FILENAME;
    }

    protected void doStop() throws Exception {
        this.liveBlobs = null;
        try {
            this.deletedBlobIndex.close();
        }
        finally {
            this.deletedBlobIndex = null;
            this.metricsStore.stop();
        }
    }

    @VisibleForTesting
    Path contentPath(BlobId id) {
        return this.contentDir.resolve(String.valueOf(this.blobIdLocationResolver.getLocation(id)) + BLOB_CONTENT_SUFFIX);
    }

    @VisibleForTesting
    Path attributePath(BlobId id) {
        return this.contentDir.resolve(String.valueOf(this.blobIdLocationResolver.getLocation(id)) + ".properties");
    }

    protected String attributePathString(BlobId blobId) {
        return this.attributePath(blobId).toString();
    }

    private Path temporaryContentPath(BlobId id, UUID suffix) {
        return this.contentDir.resolve(String.valueOf(this.blobIdLocationResolver.getTemporaryLocation(id)) + "." + suffix + BLOB_CONTENT_SUFFIX);
    }

    private Path temporaryAttributePath(BlobId id, UUID suffix) {
        return this.contentDir.resolve(String.valueOf(this.blobIdLocationResolver.getTemporaryLocation(id)) + "." + suffix + ".properties");
    }

    protected Blob doCreate(InputStream blobData, Map<String, String> headers, @Nullable BlobId blobId) {
        return this.create(headers, destination -> this.fileOperations.create(destination, blobData), blobId);
    }

    @Guarded(by={"STARTED"})
    public Blob create(Path sourceFile, Map<String, String> headers, long size, HashCode sha1) {
        Preconditions.checkNotNull((Object)sourceFile);
        Preconditions.checkNotNull((Object)sha1);
        Preconditions.checkArgument((boolean)Files.exists(sourceFile, new LinkOption[0]));
        return this.create(headers, destination -> {
            this.fileOperations.hardLink(sourceFile, destination);
            return new StreamMetrics(size, sha1.toString());
        }, null);
    }

    private Blob create(Map<String, String> headers, BlobIngester ingester, BlobId blobId) {
        int retries = 0;
        while (retries <= 8) {
            try {
                return this.tryCreate(headers, ingester, blobId);
            }
            catch (BlobCollisionException e) {
                this.log.warn("BlobId collision: {} already exists{}", (Object)e.getBlobId(), (Object)(retries < 8 ? ", retrying with new BlobId" : "!"));
                ++retries;
            }
        }
        throw new BlobStoreException("Cannot find free BlobId", null);
    }

    private Blob tryCreate(Map<String, String> headers, BlobIngester ingester, BlobId reusedBlobId) {
        BlobId blobId = this.getBlobId(headers, reusedBlobId);
        boolean isDirectPath = Boolean.parseBoolean(headers.getOrDefault("BlobStore.direct-path", "false"));
        Long existingSize = isDirectPath && this.exists(blobId) ? this.getContentSizeForDeletion(blobId) : null;
        Path blobPath = this.contentPath(blobId);
        Path attributePath = this.attributePath(blobId);
        UUID uuidSuffix = UUID.randomUUID();
        Path temporaryBlobPath = this.temporaryContentPath(blobId, uuidSuffix);
        Path temporaryAttributePath = this.temporaryAttributePath(blobId, uuidSuffix);
        FileBlob blob = (FileBlob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
        Lock lock = blob.lock();
        try {
            boolean wouldCollide = this.fileOperations.exists(blobPath);
            if (reusedBlobId == null && RETRY_ON_COLLISION && wouldCollide && !isDirectPath) {
                throw new BlobCollisionException(blobId);
            }
            try {
                this.log.debug("Writing blob {} to {}", (Object)blobId, (Object)blobPath);
                StreamMetrics streamMetrics = ingester.ingestTo(temporaryBlobPath);
                BlobMetrics metrics = new BlobMetrics(new DateTime(), streamMetrics.getSha1(), streamMetrics.getSize());
                blob.refresh(headers, metrics);
                FileBlobAttributes blobAttributes = new FileBlobAttributes(temporaryAttributePath, headers, metrics);
                blobAttributes.store();
                if (existingSize != null) {
                    this.overwrite(temporaryBlobPath, blobPath);
                    this.overwrite(temporaryAttributePath, attributePath);
                    this.metricsStore.recordDeletion(existingSize);
                } else {
                    this.move(temporaryBlobPath, blobPath);
                    this.move(temporaryAttributePath, attributePath);
                }
                this.metricsStore.recordAddition(blobAttributes.getMetrics().getContentSize());
                FileBlob fileBlob = blob;
                return fileBlob;
            }
            catch (Exception e) {
                this.fileOperations.deleteQuietly(temporaryAttributePath);
                this.fileOperations.deleteQuietly(temporaryBlobPath);
                this.fileOperations.deleteQuietly(attributePath);
                this.fileOperations.deleteQuietly(blobPath);
                throw new BlobStoreException((Throwable)e, blobId);
            }
        }
        finally {
            lock.unlock();
        }
    }

    @Guarded(by={"STARTED"})
    public Blob copy(BlobId blobId, Map<String, String> headers) {
        Blob sourceBlob = (Blob)Preconditions.checkNotNull((Object)this.get(blobId));
        Path sourcePath = this.contentPath(sourceBlob.getId());
        if (this.supportsHardLinkCopy) {
            try {
                return this.create(headers, destination -> {
                    this.fileOperations.hardLink(sourcePath, destination);
                    BlobMetrics metrics = sourceBlob.getMetrics();
                    return new StreamMetrics(metrics.getContentSize(), metrics.getSha1Hash());
                }, null);
            }
            catch (BlobStoreException e) {
                this.supportsHardLinkCopy = false;
                this.log.trace("Disabling copy by hard link for blob store {}, could not hard link blob {}", new Object[]{this.blobStoreConfiguration.getName(), sourceBlob.getId(), e});
            }
        }
        this.log.trace("Using fallback mechanism for blob store {}, copying blob {}", (Object)this.blobStoreConfiguration.getName(), (Object)sourceBlob.getId());
        return this.create(headers, destination -> {
            this.fileOperations.copy(sourcePath, destination);
            BlobMetrics metrics = sourceBlob.getMetrics();
            return new StreamMetrics(metrics.getContentSize(), metrics.getSha1Hash());
        }, null);
    }

    @Nullable
    @Guarded(by={"STARTED"})
    public Blob get(BlobId blobId) {
        return this.get(blobId, false);
    }

    @Nullable
    @Guarded(by={"STARTED"})
    public Blob get(BlobId blobId, boolean includeDeleted) {
        FileBlob blob;
        block10: {
            Preconditions.checkNotNull((Object)blobId);
            blob = (FileBlob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
            if (blob.isStale()) {
                Lock lock = blob.lock();
                try {
                    if (!blob.isStale()) break block10;
                    FileBlobAttributes blobAttributes = this.getFileBlobAttributes(blobId);
                    if (blobAttributes == null) {
                        return null;
                    }
                    if (blobAttributes.isDeleted() && !includeDeleted) {
                        this.log.warn("Attempt to access soft-deleted blob {} ({}), reason: {}", new Object[]{blobId, blobAttributes.getPath(), blobAttributes.getDeletedReason()});
                        return null;
                    }
                    try {
                        blob.refresh(blobAttributes.getHeaders(), blobAttributes.getMetrics());
                    }
                    catch (Exception e) {
                        throw new BlobStoreException((Throwable)e, blobId);
                    }
                }
                finally {
                    lock.unlock();
                }
            }
        }
        this.log.debug("Accessing blob {}", (Object)blobId);
        return blob;
    }

    protected boolean doDelete(BlobId blobId, String reason) {
        FileBlob blob = (FileBlob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
        Lock lock = blob.lock();
        try {
            this.log.debug("Soft deleting blob {}", (Object)blobId);
            FileBlobAttributes blobAttributes = this.getFileBlobAttributes(blobId);
            if (blobAttributes == null) {
                this.log.warn("Attempt to mark-for-delete non-existent blob {}, hard deleting instead", (Object)blobId);
                boolean bl = this.deleteHard(blobId);
                return bl;
            }
            if (blobAttributes.isDeleted()) {
                this.log.debug("Attempt to delete already-deleted blob {}", (Object)blobId);
                return false;
            }
            blobAttributes.setDeleted(true);
            blobAttributes.setDeletedReason(reason);
            blobAttributes.store();
            this.deletedBlobIndex.add(blobId.toString().getBytes(StandardCharsets.UTF_8));
            blob.markStale();
            return true;
        }
        catch (Exception e) {
            throw new BlobStoreException((Throwable)e, blobId);
        }
        finally {
            lock.unlock();
        }
    }

    protected boolean doDeleteHard(BlobId blobId) {
        FileBlob blob = (FileBlob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
        Lock lock = blob.lock();
        try {
            this.log.debug("Hard deleting blob {}", (Object)blobId);
            Path attributePath = this.attributePath(blobId);
            Long contentSize = this.getContentSizeForDeletion(blobId);
            Path blobPath = this.contentPath(blobId);
            boolean blobDeleted = this.delete(blobPath);
            this.delete(attributePath);
            if (blobDeleted && contentSize != null) {
                this.metricsStore.recordDeletion(contentSize);
            }
            boolean bl = blobDeleted;
            return bl;
        }
        catch (Exception e) {
            throw new BlobStoreException((Throwable)e, blobId);
        }
        finally {
            lock.unlock();
            this.liveBlobs.invalidate((Object)blobId);
        }
    }

    @Nullable
    private Long getContentSizeForDeletion(BlobId blobId) {
        return Optional.ofNullable(this.getFileBlobAttributes(blobId)).map(BlobAttributes::getMetrics).map(BlobMetrics::getContentSize).orElse(null);
    }

    @Guarded(by={"STARTED"})
    public BlobStoreMetrics getMetrics() {
        return this.metricsStore.getMetrics();
    }

    protected void doCompact(@Nullable BlobStoreUsageChecker inUseChecker) {
        try {
            this.maybeRebuildDeletedBlobIndex();
            this.log.info("Begin deleted blobs processing");
            ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);
            int counter = 0;
            int numBlobs = this.deletedBlobIndex.size();
            while (counter < numBlobs) {
                CancelableHelper.checkCancellation();
                byte[] bytes = this.deletedBlobIndex.peek();
                if (bytes == null) {
                    return;
                }
                this.deletedBlobIndex.remove();
                BlobId blobId = new BlobId(new String(bytes, StandardCharsets.UTF_8));
                FileBlob blob = (FileBlob)((Object)this.liveBlobs.getIfPresent((Object)blobId));
                if (blob == null || blob.isStale()) {
                    this.maybeCompactBlob(inUseChecker, blobId);
                } else {
                    this.deletedBlobIndex.add(bytes);
                }
                progressLogger.info("Elapsed time: {}, processed: {}/{}", new Object[]{progressLogger.getElapsed(), counter + 1, numBlobs});
                ++counter;
            }
            progressLogger.flush();
        }
        catch (BlobStoreException | TaskInterruptedException e) {
            throw e;
        }
        catch (Exception e) {
            throw new BlobStoreException((Throwable)e, null);
        }
    }

    private void maybeCompactBlob(@Nullable BlobStoreUsageChecker inUseChecker, BlobId blobId) throws IOException {
        Optional<FileBlobAttributes> attributesOption = Optional.ofNullable((FileBlobAttributes)this.getBlobAttributes(blobId));
        if (!attributesOption.isPresent() || !this.undelete(inUseChecker, blobId, (BlobAttributes)attributesOption.get(), false)) {
            this.log.debug("Hard deleting blob id: {}, in blob store: {}", (Object)blobId, (Object)this.blobStoreConfiguration.getName());
            this.deleteHard(blobId);
        }
    }

    public boolean isStorageAvailable() {
        try {
            boolean result;
            FileStore fileStore = Files.getFileStore(this.contentDir);
            long usableSpace = fileStore.getUsableSpace();
            boolean readOnly = fileStore.isReadOnly();
            boolean bl = result = !readOnly && usableSpace > 0L;
            if (!result) {
                this.log.warn("File blob store '{}' is not writable. Read only: {}. Usable space: {}", new Object[]{this.getBlobStoreConfiguration().getName(), readOnly, usableSpace});
            }
            return result;
        }
        catch (IOException e) {
            this.log.warn("File blob store '{}' is not writable.", (Object)this.getBlobStoreConfiguration().getName(), (Object)e);
            return false;
        }
    }

    protected void doInit(BlobStoreConfiguration configuration) {
        try {
            Path blobDir = this.getAbsoluteBlobDir();
            Path content = blobDir.resolve("content");
            DirectoryHelper.mkdir((Path)content);
            this.contentDir = content;
            this.setConfiguredBlobStorePath(this.getRelativeBlobDir());
        }
        catch (Exception e) {
            throw new BlobStoreException("Unable to initialize blob store directory structure: " + this.getConfiguredBlobStorePath(), (Throwable)e, null);
        }
    }

    private void checkExists(Path path, BlobId blobId) throws IOException {
        if (!this.fileOperations.exists(path)) {
            this.log.warn("Can't open input stream to blob {} as file {} not found", (Object)blobId, (Object)path);
            throw new BlobStoreException("Blob has been deleted", blobId);
        }
    }

    public boolean exists(BlobId blobId) {
        Preconditions.checkNotNull((Object)blobId);
        if (!this.fileOperations.exists(this.attributePath(blobId))) {
            this.log.debug("Blob {} was not found during existence check", (Object)blobId);
            return false;
        }
        return true;
    }

    private boolean delete(Path path) throws IOException {
        boolean deleted = this.fileOperations.delete(path);
        if (deleted) {
            this.log.debug("Deleted {}", (Object)path);
        } else {
            this.log.error("No file to delete found at {}", (Object)path, (Object)new FileNotFoundException(path.toString()));
        }
        return deleted;
    }

    private void move(Path source, Path target) throws IOException {
        if (this.supportsAtomicMove) {
            try {
                this.fileOperations.copyIfLocked(source, target, this.fileOperations::moveAtomic);
                return;
            }
            catch (AtomicMoveNotSupportedException e) {
                this.supportsAtomicMove = false;
                this.log.warn("Disabling atomic moves for blob store {}, could not move {} to {}, reason deleted: {}", new Object[]{this.blobStoreConfiguration.getName(), source, target, e.getReason()});
            }
        }
        this.log.trace("Using normal move for blob store {}, moving {} to {}", new Object[]{this.blobStoreConfiguration.getName(), source, target});
        this.fileOperations.copyIfLocked(source, target, this.fileOperations::move);
    }

    private void overwrite(Path source, Path target) throws IOException {
        if (this.supportsAtomicMove) {
            try {
                this.fileOperations.copyIfLocked(source, target, this.fileOperations::overwriteAtomic);
                return;
            }
            catch (AtomicMoveNotSupportedException e) {
                this.supportsAtomicMove = false;
                this.log.warn("Disabling atomic moves for blob store {}, could not overwrite {} with {}, reason deleted: {}", new Object[]{this.blobStoreConfiguration.getName(), source, target, e.getReason()});
            }
        }
        this.log.trace("Using normal overwrite for blob store {}, overwriting {} with {}", new Object[]{this.blobStoreConfiguration.getName(), source, target});
        this.fileOperations.copyIfLocked(source, target, this.fileOperations::overwrite);
    }

    private void setConfiguredBlobStorePath(Path path) {
        this.blobStoreConfiguration.attributes(CONFIG_KEY).set(PATH_KEY, (Object)path.toString());
    }

    private Path getConfiguredBlobStorePath() {
        return Paths.get(this.blobStoreConfiguration.attributes(CONFIG_KEY).require(PATH_KEY).toString(), new String[0]);
    }

    @Guarded(by={"NEW", "STOPPED", "FAILED"})
    public void remove() {
        try {
            Path blobDir = this.getAbsoluteBlobDir();
            if (this.fileOperations.deleteEmptyDirectory(this.contentDir)) {
                this.metricsStore.remove();
                this.fileOperations.deleteQuietly(blobDir.resolve(METADATA_FILENAME));
                File[] files = blobDir.toFile().listFiles((dir, name) -> name.endsWith(DELETIONS_FILENAME));
                if (files != null) {
                    Arrays.stream(files).map(File::toPath).forEach(this.fileOperations::deleteQuietly);
                } else {
                    this.log.warn("Unable to cleanup file(s) for Deletions Index");
                }
                if (!this.fileOperations.deleteEmptyDirectory(blobDir)) {
                    this.log.warn("Unable to delete non-empty blob store directory {}", (Object)blobDir);
                }
            } else {
                this.log.warn("Unable to delete non-empty blob store content directory {}", (Object)this.contentDir);
            }
        }
        catch (Exception e) {
            throw new BlobStoreException((Throwable)e, null);
        }
    }

    @VisibleForTesting
    Path getAbsoluteBlobDir() throws IOException {
        Path configurationPath = this.getConfiguredBlobStorePath();
        if (configurationPath.isAbsolute()) {
            return configurationPath;
        }
        Path normalizedBase = this.basedir.toRealPath(new LinkOption[0]).normalize();
        Path normalizedPath = configurationPath.normalize();
        return normalizedBase.resolve(normalizedPath);
    }

    @VisibleForTesting
    Path getRelativeBlobDir() throws IOException {
        Path configurationPath = this.getConfiguredBlobStorePath();
        if (configurationPath.isAbsolute()) {
            Path normalizedBase = this.basedir.toRealPath(new LinkOption[0]).normalize();
            Path normalizedPath = configurationPath.toRealPath(new LinkOption[0]).normalize();
            if (normalizedPath.startsWith(normalizedBase)) {
                return normalizedBase.relativize(normalizedPath);
            }
        }
        return configurationPath;
    }

    @VisibleForTesting
    void maybeRebuildDeletedBlobIndex() throws IOException {
        PropertiesFile metadata = new PropertiesFile(this.getAbsoluteBlobDir().resolve(METADATA_FILENAME).toFile());
        metadata.load();
        String deletedBlobIndexRebuildRequired = metadata.getProperty(REBUILD_DELETED_BLOB_INDEX_KEY, "false");
        if (Boolean.parseBoolean(deletedBlobIndexRebuildRequired)) {
            Path deletedIndex = this.getAbsoluteBlobDir().resolve(this.getDeletionsFilename());
            this.log.warn("Clearing deletions index file {} for rebuild", (Object)deletedIndex);
            this.deletedBlobIndex.clear();
            if (!this.nodeAccess.isOldestNode()) {
                this.log.info("Skipping deletion index rebuild because this is not the oldest node.");
                return;
            }
            this.log.warn("Rebuilding deletions index file {}", (Object)deletedIndex);
            ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);
            AtomicInteger processed = new AtomicInteger();
            int softDeletedBlobsFound = this.getBlobIdStream().map(this::getFileBlobAttributes).filter(Objects::nonNull).mapToInt(attributes -> {
                block5: {
                    block4: {
                        try {
                            if (!attributes.isDeleted()) break block4;
                            String blobId = this.getBlobIdFromAttributeFilePath(new FileAttributesLocation(attributes.getPath()));
                            this.deletedBlobIndex.add(blobId.getBytes(StandardCharsets.UTF_8));
                        }
                        catch (IOException e) {
                            try {
                                this.log.warn("Failed to add blobId to index from attribute file {}", (Object)attributes.getPath(), (Object)e);
                            }
                            catch (Throwable throwable) {
                                progressLogger.info("Elapsed time: {}, processed: {}, deleted: {}", new Object[]{progressLogger.getElapsed(), processed.incrementAndGet(), this.deletedBlobIndex.size()});
                                throw throwable;
                            }
                            progressLogger.info("Elapsed time: {}, processed: {}, deleted: {}", new Object[]{progressLogger.getElapsed(), processed.incrementAndGet(), this.deletedBlobIndex.size()});
                            break block5;
                        }
                        progressLogger.info("Elapsed time: {}, processed: {}, deleted: {}", new Object[]{progressLogger.getElapsed(), processed.incrementAndGet(), this.deletedBlobIndex.size()});
                        return 1;
                    }
                    progressLogger.info("Elapsed time: {}, processed: {}, deleted: {}", new Object[]{progressLogger.getElapsed(), processed.incrementAndGet(), this.deletedBlobIndex.size()});
                }
                return 0;
            }).sum();
            progressLogger.flush();
            this.log.warn("Elapsed time: {}, Added {} soft deleted blob(s) to index file {}", new Object[]{progressLogger.getElapsed(), softDeletedBlobsFound, deletedIndex});
            metadata.remove((Object)REBUILD_DELETED_BLOB_INDEX_KEY);
            metadata.store();
        } else {
            this.log.info("Deletions index file rebuild not required");
        }
    }

    private Stream<Path> getAttributeFilePaths() throws IOException {
        return this.getAttributeFilePaths("");
    }

    private Stream<Path> getAttributeFilePaths(String prefix) throws IOException {
        Path parent = this.contentDir.resolve(prefix);
        if (!parent.toFile().exists()) {
            return Stream.empty();
        }
        return Files.walk(parent, FileVisitOption.FOLLOW_LINKS).filter(this::isNonTemporaryAttributeFile);
    }

    private boolean isNonTemporaryAttributeFile(Path path) {
        File attributeFile = path.toFile();
        return attributeFile.isFile() && attributeFile.getName().endsWith(".properties") && !attributeFile.getName().startsWith("tmp$");
    }

    private void createEmptyDeletionsIndex(File deletionsIndex) throws IOException {
        Path tempFile = Files.createTempFile(DELETIONS_FILENAME, "tmp", new FileAttribute[0]);
        Files.delete(tempFile);
        try {
            new QueueFile(tempFile.toFile()).close();
            Throwable throwable = null;
            Object var4_5 = null;
            try (RandomAccessFile raf = new RandomAccessFile(deletionsIndex, "rw");){
                raf.setLength(0L);
                raf.write(Files.readAllBytes(tempFile));
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        finally {
            Files.deleteIfExists(tempFile);
        }
    }

    @VisibleForTesting
    void setLiveBlobs(LoadingCache<BlobId, FileBlob> liveBlobs) {
        this.liveBlobs = liveBlobs;
    }

    public Stream<BlobId> getBlobIdStream() {
        try {
            return this.getAttributeFilePaths().map(FileAttributesLocation::new).map(arg_0 -> ((FileBlobStore)this).getBlobIdFromAttributeFilePath(arg_0)).map(BlobId::new);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public Stream<BlobId> getDirectPathBlobIdStream(String prefix) {
        Preconditions.checkArgument((!prefix.contains("..") ? 1 : 0) != 0, (Object)"path traversal not allowed");
        try {
            return this.getAttributeFilePaths("directpath/" + prefix).map(this::toBlobName).filter(Objects::nonNull).map(this::toBlobId);
        }
        catch (IOException e) {
            this.log.error("Caught IOException during getDirectPathBlobIdStream for {}", (Object)prefix, (Object)e);
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    Path getContentDir() {
        return this.contentDir;
    }

    @Nullable
    @VisibleForTesting
    String toBlobName(Path path) {
        try {
            String pathStr = this.contentDir.resolve("directpath").relativize(path).toString().replace(File.separatorChar, '/');
            return StringUtils.removeEnd((String)pathStr, (String)".properties");
        }
        catch (Exception ex) {
            this.log.debug("Attempting to create blob name from path {}, but caught Exception", (Object)path, (Object)ex);
            return null;
        }
    }

    @Nullable
    public BlobAttributes getBlobAttributes(BlobId blobId) {
        FileBlobAttributes blobAttributes;
        block3: {
            Path blobPath = this.attributePath(blobId);
            try {
                blobAttributes = new FileBlobAttributes(blobPath);
                if (blobAttributes.load()) break block3;
                this.log.warn("Attempt to access non-existent blob {} ({})", (Object)blobId, (Object)this.attributePath(blobId));
                return null;
            }
            catch (Exception e) {
                this.log.error("Unable to load BlobAttributes for blob id: {}, path: {}, exception: {}", new Object[]{blobId, blobPath, e.getMessage(), this.log.isDebugEnabled() ? e : null});
                return null;
            }
        }
        return blobAttributes;
    }

    public BlobAttributes getBlobAttributes(FileAttributesLocation attributesFilePath) throws IOException {
        FileBlobAttributes fileBlobAttributes = new FileBlobAttributes(attributesFilePath.getPath());
        fileBlobAttributes.load();
        return fileBlobAttributes;
    }

    @Nullable
    private FileBlobAttributes getFileBlobAttributes(BlobId blobId) {
        return (FileBlobAttributes)this.getBlobAttributes(blobId);
    }

    private BlobId toBlobId(String blobName) {
        ImmutableMap headers = ImmutableMap.of((Object)"BlobStore.blob-name", (Object)blobName, (Object)"BlobStore.direct-path", (Object)"true");
        return this.blobIdLocationResolver.fromHeaders((Map)headers);
    }

    public void setBlobAttributes(BlobId blobId, BlobAttributes blobAttributes) {
        try {
            FileBlobAttributes fileBlobAttributes = this.getFileBlobAttributes(blobId);
            fileBlobAttributes.updateFrom(blobAttributes);
            fileBlobAttributes.store();
        }
        catch (Exception e) {
            this.log.error("Unable to set BlobAttributes for blob id: {}, exception: {}", new Object[]{blobId, e.getMessage(), this.log.isDebugEnabled() ? e : null});
        }
    }

    private static interface BlobIngester {
        public StreamMetrics ingestTo(Path var1) throws IOException;
    }

    class FileBlob
    extends BlobSupport {
        FileBlob(BlobId blobId) {
            super(blobId);
        }

        public InputStream getInputStream() {
            Path contentPath = FileBlobStore.this.contentPath(this.getId());
            try {
                FileBlobStore.this.checkExists(contentPath, this.getId());
                return new BufferedInputStream(FileBlobStore.this.fileOperations.openInputStream(contentPath));
            }
            catch (BlobStoreException e) {
                this.markStale();
                throw e;
            }
            catch (Exception e) {
                throw new BlobStoreException((Throwable)e, this.getId());
            }
        }
    }
}

