/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.util;

import com.google.common.base.Charsets;
import com.google.common.collect.Sets;
import com.google.common.primitives.Ints;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import org.apache.lucene.index.CheckIndex;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.store.SimpleFSDirectory;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardPath;
import org.elasticsearch.index.shard.ShardStateMetaData;

public class MultiDataPathUpgrader {
    private final NodeEnvironment nodeEnvironment;
    private final ESLogger logger = Loggers.getLogger(this.getClass());

    public MultiDataPathUpgrader(NodeEnvironment nodeEnvironment) {
        this.nodeEnvironment = nodeEnvironment;
    }

    public void upgrade(ShardId shard, ShardPath targetPath) throws IOException {
        Object[] paths = this.nodeEnvironment.availableShardPaths(shard);
        if (!this.isTargetPathConfigured((Path[])paths, targetPath)) {
            throw new IllegalArgumentException("shard path must be one of the shards data paths");
        }
        assert (this.needsUpgrading(shard)) : "Should not upgrade a path that needs no upgrading";
        this.logger.info("{} upgrading multi data dir to {}", shard, targetPath.getDataPath());
        ShardStateMetaData loaded = ShardStateMetaData.FORMAT.loadLatestState(this.logger, (Path[])paths);
        if (loaded == null) {
            throw new IllegalStateException(shard + " no shard state found in any of: " + Arrays.toString(paths) + " please check and remove them if possible");
        }
        this.logger.info("{} loaded shard state {}", shard, loaded);
        ShardStateMetaData.FORMAT.write(loaded, loaded.version, targetPath.getShardStatePath());
        Files.createDirectories(targetPath.resolveIndex(), new FileAttribute[0]);
        try (SimpleFSDirectory directory = new SimpleFSDirectory(targetPath.resolveIndex());){
            try (Lock lock = directory.obtainLock("write.lock");){
                this.upgradeFiles(shard, targetPath, targetPath.resolveIndex(), "index", (Path[])paths);
            }
            catch (LockObtainFailedException ex) {
                throw new IllegalStateException("Can't obtain lock on " + targetPath.resolveIndex(), ex);
            }
        }
        this.upgradeFiles(shard, targetPath, targetPath.resolveTranslog(), "translog", (Path[])paths);
        this.logger.info("{} wipe upgraded directories", shard);
        for (Object path : paths) {
            if (path.equals(targetPath.getShardStatePath())) continue;
            this.logger.info("{} wipe shard directories: [{}]", shard, path);
            IOUtils.rm(new Path[]{path});
        }
        if (FileSystemUtils.files(targetPath.resolveIndex()).length == 0) {
            throw new IllegalStateException("index folder [" + targetPath.resolveIndex() + "] is empty");
        }
        if (FileSystemUtils.files(targetPath.resolveTranslog()).length == 0) {
            throw new IllegalStateException("translog folder [" + targetPath.resolveTranslog() + "] is empty");
        }
    }

    public void checkIndex(ShardPath targetPath) throws IOException {
        BytesStreamOutput os = new BytesStreamOutput();
        PrintStream out = new PrintStream((OutputStream)os, false, Charsets.UTF_8.name());
        try (SimpleFSDirectory directory = new SimpleFSDirectory(targetPath.resolveIndex());
             CheckIndex checkIndex = new CheckIndex(directory);){
            checkIndex.setInfoStream(out);
            CheckIndex.Status status = checkIndex.checkIndex();
            out.flush();
            if (!status.clean) {
                this.logger.warn("check index [failure]\n{}", new String(os.bytes().toBytes(), Charsets.UTF_8));
                throw new IllegalStateException("index check failure");
            }
        }
    }

    public boolean needsUpgrading(ShardId shard) {
        Path[] paths = this.nodeEnvironment.availableShardPaths(shard);
        if (paths.length > 1) {
            int numPathsExist = 0;
            for (Path path : paths) {
                if (!Files.exists(path.resolve("_state"), new LinkOption[0]) || ++numPathsExist <= 1) continue;
                return true;
            }
        }
        return false;
    }

    public ShardPath pickShardPath(ShardId shard) throws IOException {
        NodeEnvironment.NodePath[] paths;
        if (!this.needsUpgrading(shard)) {
            throw new IllegalStateException("Shard doesn't need upgrading");
        }
        for (NodeEnvironment.NodePath path : paths = this.nodeEnvironment.nodePaths()) {
            Files.createDirectories(path.resolve(shard), new FileAttribute[0]);
        }
        ShardFileInfo[] shardFileInfo = this.getShardFileInfo(shard, paths);
        long totalBytesUsedByShard = 0L;
        long leastUsableSpace = Long.MAX_VALUE;
        long mostUsableSpace = Long.MIN_VALUE;
        assert (shardFileInfo.length == this.nodeEnvironment.availableShardPaths(shard).length);
        for (ShardFileInfo info : shardFileInfo) {
            totalBytesUsedByShard += info.spaceUsedByShard;
            leastUsableSpace = Math.min(leastUsableSpace, info.usableSpace + info.spaceUsedByShard);
            mostUsableSpace = Math.max(mostUsableSpace, info.usableSpace + info.spaceUsedByShard);
        }
        if (mostUsableSpace < totalBytesUsedByShard) {
            throw new IllegalStateException("Can't upgrade path available space: " + new ByteSizeValue(mostUsableSpace) + " required space: " + new ByteSizeValue(totalBytesUsedByShard));
        }
        ShardFileInfo target = shardFileInfo[0];
        if (leastUsableSpace >= 2L * totalBytesUsedByShard) {
            for (ShardFileInfo info : shardFileInfo) {
                if (info.spaceUsedByShard <= target.spaceUsedByShard) continue;
                target = info;
            }
        } else {
            for (ShardFileInfo info : shardFileInfo) {
                if (info.usableSpace <= target.usableSpace) continue;
                target = info;
            }
        }
        return new ShardPath(false, target.path, target.path, "_na_", shard);
    }

    private ShardFileInfo[] getShardFileInfo(ShardId shard, NodeEnvironment.NodePath[] paths) throws IOException {
        ShardFileInfo[] info = new ShardFileInfo[paths.length];
        for (int i = 0; i < info.length; ++i) {
            Path path = paths[i].resolve(shard);
            long usabelSpace = this.getUsabelSpace(paths[i]);
            info[i] = new ShardFileInfo(path, usabelSpace, this.getSpaceUsedByShard(path));
        }
        return info;
    }

    protected long getSpaceUsedByShard(Path path) throws IOException {
        final long[] spaceUsedByShard = new long[]{0L};
        if (Files.exists(path, new LinkOption[0])) {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (attrs.isRegularFile()) {
                        spaceUsedByShard[0] = spaceUsedByShard[0] + attrs.size();
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        return spaceUsedByShard[0];
    }

    protected long getUsabelSpace(NodeEnvironment.NodePath path) throws IOException {
        FileStore fileStore = path.fileStore;
        return fileStore.getUsableSpace();
    }

    private void upgradeFiles(ShardId shard, ShardPath targetPath, Path targetDir, String folderName, Path[] paths) throws IOException {
        ArrayList<Path> movedFiles = new ArrayList<Path>();
        for (Path path : paths) {
            Path sourceDir;
            if (path.equals(targetPath.getDataPath()) || !Files.exists(sourceDir = path.resolve(folderName), new LinkOption[0])) continue;
            this.logger.info("{} upgrading [{}] from [{}] to [{}]", shard, folderName, sourceDir, targetDir);
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(sourceDir);){
                Files.createDirectories(targetDir, new FileAttribute[0]);
                for (Path file : stream) {
                    if ("write.lock".equals(file.getFileName().toString()) || Files.isDirectory(file, new LinkOption[0])) continue;
                    this.logger.info("{} move file [{}] size: [{}]", shard, file.getFileName(), Files.size(file));
                    Path targetFile = targetDir.resolve(file.getFileName());
                    Path targetTempFile = Files.createTempFile(targetDir, "upgrade_", "_" + file.getFileName().toString(), new FileAttribute[0]);
                    Files.copy(file, targetTempFile, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
                    Files.move(targetTempFile, targetFile, StandardCopyOption.ATOMIC_MOVE);
                    Files.delete(file);
                    movedFiles.add(targetFile);
                }
            }
        }
        if (!movedFiles.isEmpty()) {
            this.logger.info("{} fsync files", shard);
            for (Path moved : movedFiles) {
                this.logger.info("{} syncing [{}]", shard, moved.getFileName());
                IOUtils.fsync(moved, false);
            }
            this.logger.info("{} syncing directory [{}]", shard, targetDir);
            IOUtils.fsync(targetDir, true);
        }
    }

    private boolean isTargetPathConfigured(Path[] paths, ShardPath targetPath) {
        for (Path path : paths) {
            if (!path.equals(targetPath.getDataPath())) continue;
            return true;
        }
        return false;
    }

    public static void upgradeMultiDataPath(NodeEnvironment nodeEnv, ESLogger logger) throws IOException {
        if (nodeEnv.nodeDataPaths().length > 1) {
            MultiDataPathUpgrader upgrader = new MultiDataPathUpgrader(nodeEnv);
            Set<String> allIndices = nodeEnv.findAllIndices();
            for (String index : allIndices) {
                for (ShardId shardId : MultiDataPathUpgrader.findAllShardIds(nodeEnv.indexPaths(new Index(index)))) {
                    ShardLock lock = nodeEnv.shardLock(shardId, 0L);
                    Throwable throwable = null;
                    try {
                        if (upgrader.needsUpgrading(shardId)) {
                            ShardPath shardPath = upgrader.pickShardPath(shardId);
                            upgrader.upgrade(shardId, shardPath);
                            if (!Files.exists(shardPath.resolveIndex(), new LinkOption[0])) continue;
                            upgrader.checkIndex(shardPath);
                            continue;
                        }
                        logger.debug("{} no upgrade needed - already upgraded", shardId);
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (lock == null) continue;
                        if (throwable != null) {
                            try {
                                lock.close();
                            }
                            catch (Throwable x2) {
                                throwable.addSuppressed(x2);
                            }
                            continue;
                        }
                        lock.close();
                    }
                }
            }
        }
    }

    private static Set<ShardId> findAllShardIds(Path ... locations) throws IOException {
        HashSet shardIds = Sets.newHashSet();
        for (Path location : locations) {
            if (!Files.isDirectory(location, new LinkOption[0])) continue;
            shardIds.addAll(MultiDataPathUpgrader.findAllShardsForIndex(location));
        }
        return shardIds;
    }

    private static Set<ShardId> findAllShardsForIndex(Path indexPath) throws IOException {
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        if (Files.isDirectory(indexPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                String currentIndex = indexPath.getFileName().toString();
                for (Path shardPath : stream) {
                    Integer shardId;
                    if (!Files.isDirectory(shardPath, new LinkOption[0]) || (shardId = Ints.tryParse((String)shardPath.getFileName().toString())) == null) continue;
                    ShardId id = new ShardId(currentIndex, (int)shardId);
                    shardIds.add(id);
                }
            }
        }
        return shardIds;
    }

    static class ShardFileInfo {
        final Path path;
        final long usableSpace;
        final long spaceUsedByShard;

        ShardFileInfo(Path path, long usableSpace, long spaceUsedByShard) {
            this.path = path;
            this.usableSpace = usableSpace;
            this.spaceUsedByShard = spaceUsedByShard;
        }
    }
}

