/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.cache.request;

import com.carrotsearch.hppc.ObjectHashSet;
import com.carrotsearch.hppc.ObjectSet;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.cache.Weigher;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.RamUsageEstimator;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.lucene.index.ElasticsearchDirectoryReader;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.MemorySizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.search.internal.SearchContext;
import org.elasticsearch.search.internal.ShardSearchRequest;
import org.elasticsearch.search.query.QueryPhase;
import org.elasticsearch.search.query.QuerySearchResult;
import org.elasticsearch.threadpool.ThreadPool;

public class IndicesRequestCache
extends AbstractComponent
implements RemovalListener<Key, Value> {
    public static final String INDEX_CACHE_REQUEST_ENABLED = "index.requests.cache.enable";
    @Deprecated
    public static final String DEPRECATED_INDEX_CACHE_REQUEST_ENABLED = "index.cache.query.enable";
    public static final String INDICES_CACHE_REQUEST_CLEAN_INTERVAL = "indices.requests.cache.clean_interval";
    public static final String INDICES_CACHE_QUERY_SIZE = "indices.requests.cache.size";
    @Deprecated
    public static final String DEPRECATED_INDICES_CACHE_QUERY_SIZE = "indices.cache.query.size";
    public static final String INDICES_CACHE_QUERY_EXPIRE = "indices.requests.cache.expire";
    public static final String INDICES_CACHE_QUERY_CONCURRENCY_LEVEL = "indices.requests.cache.concurrency_level";
    private static final Set<SearchType> CACHEABLE_SEARCH_TYPES = EnumSet.of(SearchType.QUERY_THEN_FETCH, SearchType.QUERY_AND_FETCH);
    private final ThreadPool threadPool;
    private final ClusterService clusterService;
    private final TimeValue cleanInterval;
    private final Reaper reaper;
    final ConcurrentMap<CleanupKey, Boolean> registeredClosedListeners = ConcurrentCollections.newConcurrentMap();
    final Set<CleanupKey> keysToClean = ConcurrentCollections.newConcurrentSet();
    private final String size;
    private final TimeValue expire;
    private final int concurrencyLevel;
    private volatile Cache<Key, Value> cache;

    @Inject
    public IndicesRequestCache(Settings settings, ClusterService clusterService, ThreadPool threadPool) {
        super(settings);
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        this.cleanInterval = settings.getAsTime(INDICES_CACHE_REQUEST_CLEAN_INTERVAL, TimeValue.timeValueSeconds(60L));
        String size = settings.get(INDICES_CACHE_QUERY_SIZE);
        if (size == null && (size = settings.get(DEPRECATED_INDICES_CACHE_QUERY_SIZE)) != null) {
            this.deprecationLogger.deprecated("The [indices.cache.query.size] settings is now deprecated, use [indices.requests.cache.size] instead", new Object[0]);
        }
        if (size == null) {
            size = "1%";
        }
        this.size = size;
        this.expire = settings.getAsTime(INDICES_CACHE_QUERY_EXPIRE, null);
        this.concurrencyLevel = settings.getAsInt(INDICES_CACHE_QUERY_CONCURRENCY_LEVEL, (Integer)16);
        if (this.concurrencyLevel <= 0) {
            throw new IllegalArgumentException("concurrency_level must be > 0 but was: " + this.concurrencyLevel);
        }
        this.buildCache();
        this.reaper = new Reaper();
        threadPool.schedule(this.cleanInterval, "same", this.reaper);
    }

    private boolean isCacheEnabled(Settings settings, boolean defaultEnable) {
        Boolean enable = settings.getAsBoolean(INDEX_CACHE_REQUEST_ENABLED, null);
        if (enable == null && (enable = settings.getAsBoolean(DEPRECATED_INDEX_CACHE_REQUEST_ENABLED, null)) != null) {
            this.deprecationLogger.deprecated("The [index.cache.query.enable] settings is now deprecated, use [index.requests.cache.enable] instead", new Object[0]);
        }
        if (enable == null) {
            enable = defaultEnable;
        }
        return enable;
    }

    private void buildCache() {
        long sizeInBytes = MemorySizeValue.parseBytesSizeValueOrHeapRatio(this.size, INDICES_CACHE_QUERY_SIZE).bytes();
        CacheBuilder cacheBuilder = CacheBuilder.newBuilder().maximumWeight(sizeInBytes).weigher((Weigher)new QueryCacheWeigher()).removalListener((RemovalListener)this);
        cacheBuilder.concurrencyLevel(this.concurrencyLevel);
        if (this.expire != null) {
            cacheBuilder.expireAfterAccess(this.expire.millis(), TimeUnit.MILLISECONDS);
        }
        this.cache = cacheBuilder.build();
    }

    public void close() {
        this.reaper.close();
        this.cache.invalidateAll();
    }

    public void clear(IndexShard shard) {
        if (shard == null) {
            return;
        }
        this.keysToClean.add(new CleanupKey(shard, -1L));
        this.logger.trace("{} explicit cache clear", shard.shardId());
        this.reaper.reap();
    }

    public void onRemoval(RemovalNotification<Key, Value> notification) {
        if (notification.getKey() == null) {
            return;
        }
        ((Key)notification.getKey()).shard.requestCache().onRemoval(notification);
    }

    public boolean canCache(ShardSearchRequest request, SearchContext context) {
        if (Strings.hasLength(request.templateSource())) {
            return false;
        }
        if (context.size() != 0) {
            return false;
        }
        if (!CACHEABLE_SEARCH_TYPES.contains((Object)context.searchType())) {
            return false;
        }
        IndexMetaData index = this.clusterService.state().getMetaData().index(request.index());
        if (index == null) {
            return false;
        }
        if (request.requestCache() == null ? !this.isCacheEnabled(index.getSettings(), Boolean.FALSE) : request.requestCache() == false) {
            return false;
        }
        if (!(context.searcher().getIndexReader() instanceof DirectoryReader)) {
            return false;
        }
        return !context.nowInMillisUsed();
    }

    public void loadIntoContext(ShardSearchRequest request, SearchContext context, QueryPhase queryPhase) throws Exception {
        assert (this.canCache(request, context));
        Key key = IndicesRequestCache.buildKey(request, context);
        Loader loader = new Loader(queryPhase, context, key);
        Value value = (Value)this.cache.get((Object)key, (Callable)loader);
        if (loader.isLoaded()) {
            Boolean previous;
            key.shard.requestCache().onMiss();
            CleanupKey cleanupKey = new CleanupKey(context.indexShard(), ((DirectoryReader)context.searcher().getIndexReader()).getVersion());
            if (!this.registeredClosedListeners.containsKey(cleanupKey) && (previous = this.registeredClosedListeners.putIfAbsent(cleanupKey, Boolean.TRUE)) == null) {
                ElasticsearchDirectoryReader.addReaderCloseListener(context.searcher().getDirectoryReader(), cleanupKey);
            }
        } else {
            key.shard.requestCache().onHit();
            QuerySearchResult result = context.queryResult();
            result.readFromWithId(context.id(), value.reference.streamInput());
            result.shardTarget(context.shardTarget());
        }
    }

    private static Key buildKey(ShardSearchRequest request, SearchContext context) throws Exception {
        return new Key(context.indexShard(), ((DirectoryReader)context.searcher().getIndexReader()).getVersion(), request.cacheKey());
    }

    private class Reaper
    implements Runnable {
        private final ObjectSet<CleanupKey> currentKeysToClean = new ObjectHashSet<CleanupKey>();
        private final ObjectSet<IndexShard> currentFullClean = new ObjectHashSet<IndexShard>();
        private volatile boolean closed;

        private Reaper() {
        }

        void close() {
            this.closed = true;
        }

        @Override
        public void run() {
            if (this.closed) {
                return;
            }
            if (IndicesRequestCache.this.keysToClean.isEmpty()) {
                this.schedule();
                return;
            }
            try {
                IndicesRequestCache.this.threadPool.executor("generic").execute(new Runnable(){

                    @Override
                    public void run() {
                        Reaper.this.reap();
                        Reaper.this.schedule();
                    }
                });
            }
            catch (EsRejectedExecutionException ex) {
                IndicesRequestCache.this.logger.debug("Can not run ReaderCleaner - execution rejected", ex, new Object[0]);
            }
        }

        private void schedule() {
            try {
                IndicesRequestCache.this.threadPool.schedule(IndicesRequestCache.this.cleanInterval, "same", this);
            }
            catch (EsRejectedExecutionException ex) {
                IndicesRequestCache.this.logger.debug("Can not schedule ReaderCleaner - execution rejected", ex, new Object[0]);
            }
        }

        synchronized void reap() {
            this.currentKeysToClean.clear();
            this.currentFullClean.clear();
            Iterator<CleanupKey> iterator = IndicesRequestCache.this.keysToClean.iterator();
            while (iterator.hasNext()) {
                CleanupKey cleanupKey = iterator.next();
                iterator.remove();
                if (cleanupKey.readerVersion == -1L || cleanupKey.indexShard.state() == IndexShardState.CLOSED) {
                    this.currentFullClean.add(cleanupKey.indexShard);
                    continue;
                }
                this.currentKeysToClean.add(cleanupKey);
            }
            if (!this.currentKeysToClean.isEmpty() || !this.currentFullClean.isEmpty()) {
                CleanupKey lookupKey = new CleanupKey(null, -1L);
                Iterator iterator2 = IndicesRequestCache.this.cache.asMap().keySet().iterator();
                while (iterator2.hasNext()) {
                    Key key = (Key)iterator2.next();
                    if (this.currentFullClean.contains(key.shard)) {
                        iterator2.remove();
                        continue;
                    }
                    lookupKey.indexShard = key.shard;
                    lookupKey.readerVersion = key.readerVersion;
                    if (!this.currentKeysToClean.contains(lookupKey)) continue;
                    iterator2.remove();
                }
            }
            IndicesRequestCache.this.cache.cleanUp();
            this.currentKeysToClean.clear();
            this.currentFullClean.clear();
        }
    }

    private class CleanupKey
    implements IndexReader.ReaderClosedListener {
        IndexShard indexShard;
        long readerVersion;

        private CleanupKey(IndexShard indexShard, long readerVersion) {
            this.indexShard = indexShard;
            this.readerVersion = readerVersion;
        }

        @Override
        public void onClose(IndexReader reader) {
            Boolean remove = (Boolean)IndicesRequestCache.this.registeredClosedListeners.remove(this);
            if (remove != null) {
                IndicesRequestCache.this.keysToClean.add(this);
            }
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            CleanupKey that = (CleanupKey)o;
            if (this.readerVersion != that.readerVersion) {
                return false;
            }
            return this.indexShard.equals(that.indexShard);
        }

        public int hashCode() {
            int result = this.indexShard.hashCode();
            result = 31 * result + (int)(this.readerVersion ^ this.readerVersion >>> 32);
            return result;
        }
    }

    public static class Key
    implements Accountable {
        public final IndexShard shard;
        public final long readerVersion;
        public final BytesReference value;

        Key(IndexShard shard, long readerVersion, BytesReference value) {
            this.shard = shard;
            this.readerVersion = readerVersion;
            this.value = value;
        }

        @Override
        public long ramBytesUsed() {
            return RamUsageEstimator.NUM_BYTES_OBJECT_REF + 8 + this.value.length();
        }

        @Override
        public Collection<Accountable> getChildResources() {
            return Collections.emptyList();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            Key key = (Key)o;
            if (this.readerVersion != key.readerVersion) {
                return false;
            }
            if (!this.shard.equals(key.shard)) {
                return false;
            }
            return this.value.equals(key.value);
        }

        public int hashCode() {
            int result = this.shard.hashCode();
            result = 31 * result + (int)(this.readerVersion ^ this.readerVersion >>> 32);
            result = 31 * result + this.value.hashCode();
            return result;
        }
    }

    public static class Value
    implements Accountable {
        final BytesReference reference;
        final long ramBytesUsed;

        public Value(BytesReference reference, long ramBytesUsed) {
            this.reference = reference;
            this.ramBytesUsed = ramBytesUsed;
        }

        @Override
        public long ramBytesUsed() {
            return this.ramBytesUsed;
        }

        @Override
        public Collection<Accountable> getChildResources() {
            return Collections.emptyList();
        }
    }

    private static class Loader
    implements Callable<Value> {
        private final QueryPhase queryPhase;
        private final SearchContext context;
        private final Key key;
        private boolean loaded;

        Loader(QueryPhase queryPhase, SearchContext context, Key key) {
            this.queryPhase = queryPhase;
            this.context = context;
            this.key = key;
        }

        public boolean isLoaded() {
            return this.loaded;
        }

        @Override
        public Value call() throws Exception {
            this.queryPhase.execute(this.context);
            int expectedSizeInBytes = 512;
            try (BytesStreamOutput out = new BytesStreamOutput(512);){
                this.context.queryResult().writeToNoId(out);
                BytesReference reference = out.bytes();
                this.loaded = true;
                Value value = new Value(reference, out.ramBytesUsed());
                this.key.shard.requestCache().onCached(this.key, value);
                Value value2 = value;
                return value2;
            }
        }
    }

    private static class QueryCacheWeigher
    implements Weigher<Key, Value> {
        private QueryCacheWeigher() {
        }

        public int weigh(Key key, Value value) {
            return (int)(key.ramBytesUsed() + value.ramBytesUsed());
        }
    }
}

