/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery;

import com.google.common.collect.Iterables;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexFormatTooNewException;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.RateLimiter;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.StopWatch;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.compress.CompressorFactory;
import org.elasticsearch.common.lease.Releasables;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.CancellableThreads;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.index.deletionpolicy.SnapshotIndexCommit;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.engine.RecoveryEngineException;
import org.elasticsearch.index.shard.IllegalIndexShardStateException;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardClosedException;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.store.StoreFileMetaData;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.recovery.RecoverFilesRecoveryException;
import org.elasticsearch.indices.recovery.RecoveryCleanFilesRequest;
import org.elasticsearch.indices.recovery.RecoveryFileChunkRequest;
import org.elasticsearch.indices.recovery.RecoveryFilesInfoRequest;
import org.elasticsearch.indices.recovery.RecoveryFinalizeRecoveryRequest;
import org.elasticsearch.indices.recovery.RecoveryPrepareForTranslogOperationsRequest;
import org.elasticsearch.indices.recovery.RecoveryResponse;
import org.elasticsearch.indices.recovery.RecoverySettings;
import org.elasticsearch.indices.recovery.RecoveryTranslogOperationsRequest;
import org.elasticsearch.indices.recovery.StartRecoveryRequest;
import org.elasticsearch.transport.EmptyTransportResponseHandler;
import org.elasticsearch.transport.RemoteTransportException;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportService;

public class RecoverySourceHandler {
    protected final ESLogger logger;
    private final IndexShard shard;
    private final String indexName;
    private final int shardId;
    private final StartRecoveryRequest request;
    private final RecoverySettings recoverySettings;
    private final TransportService transportService;
    protected final RecoveryResponse response;
    private final CancellableThreads cancellableThreads = new CancellableThreads(){

        @Override
        protected void onCancel(String reason, @Nullable Throwable suppressedException) {
            ElasticsearchException e = RecoverySourceHandler.this.shard.state() == IndexShardState.CLOSED ? new IndexShardClosedException(RecoverySourceHandler.this.shard.shardId(), "shard is closed and recovery was canceled reason [" + reason + "]") : new CancellableThreads.ExecutionCancelledException("recovery was canceled reason [" + reason + "]");
            if (suppressedException != null) {
                e.addSuppressed(suppressedException);
            }
            throw e;
        }
    };

    public RecoverySourceHandler(IndexShard shard, StartRecoveryRequest request, RecoverySettings recoverySettings, TransportService transportService, ESLogger logger) {
        this.shard = shard;
        this.request = request;
        this.recoverySettings = recoverySettings;
        this.logger = logger;
        this.transportService = transportService;
        this.indexName = this.request.shardId().index().name();
        this.shardId = this.request.shardId().id();
        this.response = new RecoveryResponse();
    }

    public RecoveryResponse recoverToTarget() {
        Engine engine = this.shard.engine();
        assert (engine.getTranslog() != null) : "translog must not be null";
        try (Translog.View translogView = engine.getTranslog().newView();){
            SnapshotIndexCommit phase1Snapshot;
            this.logger.trace("captured translog id [{}] for recovery", translogView.minTranslogGeneration());
            try {
                phase1Snapshot = this.shard.snapshotIndex(false);
            }
            catch (Throwable e) {
                IOUtils.closeWhileHandlingException(translogView);
                throw new RecoveryEngineException(this.shard.shardId(), 1, "Snapshot failed", e);
            }
            try {
                this.phase1(phase1Snapshot, translogView);
            }
            catch (Throwable e) {
                try {
                    throw new RecoveryEngineException(this.shard.shardId(), 1, "phase1 failed", e);
                }
                catch (Throwable throwable) {
                    Releasables.closeWhileHandlingException(phase1Snapshot);
                    throw throwable;
                }
            }
            Releasables.closeWhileHandlingException(phase1Snapshot);
            this.logger.trace("snapshot translog for recovery. current size is [{}]", translogView.totalOperations());
            try (Translog.Snapshot phase2Snapshot = translogView.snapshot();){
                this.phase2(phase2Snapshot);
            }
            catch (Throwable e) {
                throw new RecoveryEngineException(this.shard.shardId(), 2, "phase2 failed", e);
            }
            this.finalizeRecovery();
        }
        return this.response;
    }

    public void phase1(final SnapshotIndexCommit snapshot, final Translog.View translogView) {
        this.cancellableThreads.checkForCancel();
        long totalSize = 0L;
        long existingTotalSize = 0L;
        final Store store = this.shard.store();
        store.incRef();
        try {
            boolean recoverWithSyncId;
            Store.MetadataSnapshot recoverySourceMetadata;
            StopWatch stopWatch = new StopWatch().start();
            try {
                recoverySourceMetadata = store.getMetadata(snapshot);
            }
            catch (CorruptIndexException | IndexFormatTooNewException | IndexFormatTooOldException ex) {
                this.shard.engine().failEngine("recovery", ex);
                throw ex;
            }
            for (String name : snapshot.getFiles()) {
                StoreFileMetaData md = recoverySourceMetadata.get(name);
                if (md != null) continue;
                this.logger.info("Snapshot differs from actual index for file: {} meta: {}", name, recoverySourceMetadata.asMap());
                throw new CorruptIndexException("Snapshot differs from actual index - maybe index was removed metadata has " + recoverySourceMetadata.asMap().size() + " files", name);
            }
            String recoverySourceSyncId = recoverySourceMetadata.getSyncId();
            String recoveryTargetSyncId = this.request.metadataSnapshot().getSyncId();
            boolean bl = recoverWithSyncId = recoverySourceSyncId != null && recoverySourceSyncId.equals(recoveryTargetSyncId);
            if (recoverWithSyncId) {
                long numDocsSource;
                long numDocsTarget = this.request.metadataSnapshot().getNumDocs();
                if (numDocsTarget != (numDocsSource = recoverySourceMetadata.getNumDocs())) {
                    throw new IllegalStateException("try to recover " + this.request.shardId() + " from primary shard with sync id but number of docs differ: " + numDocsTarget + " (" + this.request.sourceNode().getName() + ", primary) vs " + numDocsSource + "(" + this.request.targetNode().getName() + ")");
                }
                this.logger.trace("[{}][{}] skipping [phase1] to {} - identical sync id [{}] found on both source and target", this.indexName, this.shardId, this.request.targetNode(), recoverySourceSyncId);
            } else {
                Store.RecoveryDiff diff = recoverySourceMetadata.recoveryDiff(this.request.metadataSnapshot());
                for (StoreFileMetaData md : diff.identical) {
                    this.response.phase1ExistingFileNames.add(md.name());
                    this.response.phase1ExistingFileSizes.add(md.length());
                    existingTotalSize += md.length();
                    if (this.logger.isTraceEnabled()) {
                        this.logger.trace("[{}][{}] recovery [phase1] to {}: not recovering [{}], exists in local store and has checksum [{}], size [{}]", this.indexName, this.shardId, this.request.targetNode(), md.name(), md.checksum(), md.length());
                    }
                    totalSize += md.length();
                }
                for (StoreFileMetaData md : Iterables.concat(diff.different, diff.missing)) {
                    if (this.request.metadataSnapshot().asMap().containsKey(md.name())) {
                        this.logger.trace("[{}][{}] recovery [phase1] to {}: recovering [{}], exists in local store, but is different: remote [{}], local [{}]", this.indexName, this.shardId, this.request.targetNode(), md.name(), this.request.metadataSnapshot().asMap().get(md.name()), md);
                    } else {
                        this.logger.trace("[{}][{}] recovery [phase1] to {}: recovering [{}], does not exists in remote", this.indexName, this.shardId, this.request.targetNode(), md.name());
                    }
                    this.response.phase1FileNames.add(md.name());
                    this.response.phase1FileSizes.add(md.length());
                    totalSize += md.length();
                }
                this.response.phase1TotalSize = totalSize;
                this.response.phase1ExistingTotalSize = existingTotalSize;
                this.logger.trace("[{}][{}] recovery [phase1] to {}: recovering_files [{}] with total_size [{}], reusing_files [{}] with total_size [{}]", this.indexName, this.shardId, this.request.targetNode(), this.response.phase1FileNames.size(), new ByteSizeValue(totalSize), this.response.phase1ExistingFileNames.size(), new ByteSizeValue(existingTotalSize));
                this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

                    @Override
                    public void run() throws InterruptedException {
                        RecoveryFilesInfoRequest recoveryInfoFilesRequest = new RecoveryFilesInfoRequest(RecoverySourceHandler.this.request.recoveryId(), RecoverySourceHandler.this.request.shardId(), RecoverySourceHandler.this.response.phase1FileNames, RecoverySourceHandler.this.response.phase1FileSizes, RecoverySourceHandler.this.response.phase1ExistingFileNames, RecoverySourceHandler.this.response.phase1ExistingFileSizes, translogView.totalOperations());
                        RecoverySourceHandler.this.transportService.submitRequest(RecoverySourceHandler.this.request.targetNode(), "internal:index/shard/recovery/filesInfo", recoveryInfoFilesRequest, TransportRequestOptions.builder().withTimeout(RecoverySourceHandler.this.recoverySettings.internalActionTimeout()).build(), EmptyTransportResponseHandler.INSTANCE_SAME).txGet();
                    }
                });
                final CountDownLatch latch = new CountDownLatch(this.response.phase1FileNames.size());
                final CopyOnWriteArrayList exceptions = new CopyOnWriteArrayList();
                final AtomicReference corruptedEngine = new AtomicReference();
                int fileIndex = 0;
                final AtomicLong bytesSinceLastPause = new AtomicLong();
                for (final String name : this.response.phase1FileNames) {
                    long fileSize = this.response.phase1FileSizes.get(fileIndex);
                    ThreadPoolExecutor pool = fileSize > RecoverySettings.SMALL_FILE_CUTOFF_BYTES ? this.recoverySettings.concurrentStreamPool() : this.recoverySettings.concurrentSmallFileStreamPool();
                    pool.execute(new AbstractRunnable(){

                        @Override
                        public void onFailure(Throwable t) {
                            RecoverySourceHandler.this.logger.debug("Failed to transfer file [" + name + "] on recovery", new Object[0]);
                        }

                        @Override
                        public void onAfter() {
                            latch.countDown();
                        }

                        /*
                         * WARNING - Removed try catching itself - possible behaviour change.
                         */
                        @Override
                        protected void doRun() {
                            RecoverySourceHandler.this.cancellableThreads.checkForCancel();
                            store.incRef();
                            StoreFileMetaData md = recoverySourceMetadata.get(name);
                            try (IndexInput indexInput = store.directory().openInput(name, IOContext.READONCE);){
                                int BUFFER_SIZE = (int)Math.max(1L, RecoverySourceHandler.this.recoverySettings.fileChunkSize().bytes());
                                byte[] buf = new byte[BUFFER_SIZE];
                                boolean shouldCompressRequest = RecoverySourceHandler.this.recoverySettings.compress();
                                if (CompressorFactory.isCompressed(indexInput)) {
                                    shouldCompressRequest = false;
                                }
                                long len = indexInput.length();
                                long readCount = 0L;
                                final TransportRequestOptions requestOptions = TransportRequestOptions.builder().withCompress(shouldCompressRequest).withType(TransportRequestOptions.Type.RECOVERY).withTimeout(RecoverySourceHandler.this.recoverySettings.internalActionTimeout()).build();
                                while (readCount < len) {
                                    long bytes;
                                    if (RecoverySourceHandler.this.shard.state() == IndexShardState.CLOSED) {
                                        throw new IndexShardClosedException(RecoverySourceHandler.this.shard.shardId());
                                    }
                                    int toRead = readCount + (long)BUFFER_SIZE > len ? (int)(len - readCount) : BUFFER_SIZE;
                                    long position = indexInput.getFilePointer();
                                    RateLimiter rl = RecoverySourceHandler.this.recoverySettings.rateLimiter();
                                    long throttleTimeInNanos = 0L;
                                    if (rl != null && (bytes = bytesSinceLastPause.addAndGet(toRead)) > rl.getMinPauseCheckBytes()) {
                                        bytesSinceLastPause.addAndGet(-bytes);
                                        throttleTimeInNanos = rl.pause(bytes);
                                        RecoverySourceHandler.this.shard.recoveryStats().addThrottleTime(throttleTimeInNanos);
                                    }
                                    indexInput.readBytes(buf, 0, toRead, false);
                                    BytesArray content = new BytesArray(buf, 0, toRead);
                                    boolean lastChunk = (readCount += (long)toRead) == len;
                                    final RecoveryFileChunkRequest fileChunkRequest = new RecoveryFileChunkRequest(RecoverySourceHandler.this.request.recoveryId(), RecoverySourceHandler.this.request.shardId(), md, position, content, lastChunk, translogView.totalOperations(), throttleTimeInNanos);
                                    RecoverySourceHandler.this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

                                        @Override
                                        public void run() throws InterruptedException {
                                            RecoverySourceHandler.this.transportService.submitRequest(RecoverySourceHandler.this.request.targetNode(), "internal:index/shard/recovery/file_chunk", fileChunkRequest, requestOptions, EmptyTransportResponseHandler.INSTANCE_SAME).txGet();
                                        }
                                    });
                                }
                            }
                            catch (Throwable e) {
                                IOException corruptIndexException = ExceptionsHelper.unwrapCorruption(e);
                                if (corruptIndexException != null) {
                                    if (!store.checkIntegrityNoException(md)) {
                                        RecoverySourceHandler.this.logger.warn("{} Corrupted file detected {} checksum mismatch", RecoverySourceHandler.this.shard.shardId(), md);
                                        if (!corruptedEngine.compareAndSet(null, corruptIndexException)) {
                                            ((Throwable)corruptedEngine.get()).addSuppressed(e);
                                        }
                                    } else {
                                        RemoteTransportException exception = new RemoteTransportException("File corruption occurred on recovery but checksums are ok", null);
                                        exception.addSuppressed(e);
                                        exceptions.add(0, exception);
                                        RecoverySourceHandler.this.logger.warn("{} Remote file corruption on node {}, recovering {}. local checksum OK", corruptIndexException, RecoverySourceHandler.this.shard.shardId(), RecoverySourceHandler.this.request.targetNode(), md);
                                    }
                                } else {
                                    exceptions.add(0, e);
                                }
                            }
                            finally {
                                store.decRef();
                            }
                        }
                    });
                    ++fileIndex;
                }
                this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

                    @Override
                    public void run() throws InterruptedException {
                        latch.await();
                    }
                });
                if (corruptedEngine.get() != null) {
                    this.shard.engine().failEngine("recovery", (Throwable)corruptedEngine.get());
                    throw (Throwable)corruptedEngine.get();
                }
                ExceptionsHelper.rethrowAndSuppress(exceptions);
                this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

                    @Override
                    public void run() throws InterruptedException {
                        try {
                            RecoverySourceHandler.this.transportService.submitRequest(RecoverySourceHandler.this.request.targetNode(), "internal:index/shard/recovery/clean_files", new RecoveryCleanFilesRequest(RecoverySourceHandler.this.request.recoveryId(), RecoverySourceHandler.this.shard.shardId(), recoverySourceMetadata, translogView.totalOperations()), TransportRequestOptions.builder().withTimeout(RecoverySourceHandler.this.recoverySettings.internalActionTimeout()).build(), EmptyTransportResponseHandler.INSTANCE_SAME).txGet();
                        }
                        catch (RemoteTransportException remoteException) {
                            IOException corruptIndexException = ExceptionsHelper.unwrapCorruption(remoteException);
                            if (corruptIndexException != null) {
                                try {
                                    Store.MetadataSnapshot recoverySourceMetadata2 = store.getMetadata(snapshot);
                                    StoreFileMetaData[] metadata = (StoreFileMetaData[])Iterables.toArray((Iterable)recoverySourceMetadata2, StoreFileMetaData.class);
                                    ArrayUtil.timSort(metadata, new Comparator<StoreFileMetaData>(){

                                        @Override
                                        public int compare(StoreFileMetaData o1, StoreFileMetaData o2) {
                                            return Long.compare(o1.length(), o2.length());
                                        }
                                    });
                                    for (StoreFileMetaData md : metadata) {
                                        RecoverySourceHandler.this.logger.debug("{} checking integrity for file {} after remove corruption exception", RecoverySourceHandler.this.shard.shardId(), md);
                                        if (store.checkIntegrityNoException(md)) continue;
                                        RecoverySourceHandler.this.shard.engine().failEngine("recovery", corruptIndexException);
                                        RecoverySourceHandler.this.logger.warn("{} Corrupted file detected {} checksum mismatch", RecoverySourceHandler.this.shard.shardId(), md);
                                        throw corruptIndexException;
                                    }
                                }
                                catch (IOException ex) {
                                    remoteException.addSuppressed(ex);
                                    throw remoteException;
                                }
                                RemoteTransportException exception = new RemoteTransportException("File corruption occurred on recovery but checksums are ok", null);
                                exception.addSuppressed(remoteException);
                                RecoverySourceHandler.this.logger.warn("{} Remote file corruption during finalization on node {}, recovering {}. local checksum OK", corruptIndexException, RecoverySourceHandler.this.shard.shardId(), RecoverySourceHandler.this.request.targetNode());
                                throw exception;
                            }
                            throw remoteException;
                        }
                    }
                });
            }
            this.prepareTargetForTranslog(translogView);
            this.logger.trace("[{}][{}] recovery [phase1] to {}: took [{}]", this.indexName, this.shardId, this.request.targetNode(), stopWatch.totalTime());
            this.response.phase1Time = stopWatch.totalTime().millis();
        }
        catch (Throwable e) {
            throw new RecoverFilesRecoveryException(this.request.shardId(), this.response.phase1FileNames.size(), new ByteSizeValue(totalSize), e);
        }
        finally {
            store.decRef();
        }
    }

    protected void prepareTargetForTranslog(final Translog.View translogView) {
        StopWatch stopWatch = new StopWatch().start();
        this.logger.trace("{} recovery [phase1] to {}: prepare remote engine for translog", this.request.shardId(), this.request.targetNode());
        long startEngineStart = stopWatch.totalTime().millis();
        this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

            @Override
            public void run() throws InterruptedException {
                RecoverySourceHandler.this.transportService.submitRequest(RecoverySourceHandler.this.request.targetNode(), "internal:index/shard/recovery/prepare_translog", new RecoveryPrepareForTranslogOperationsRequest(RecoverySourceHandler.this.request.recoveryId(), RecoverySourceHandler.this.request.shardId(), translogView.totalOperations()), TransportRequestOptions.builder().withTimeout(RecoverySourceHandler.this.recoverySettings.internalActionTimeout()).build(), EmptyTransportResponseHandler.INSTANCE_SAME).txGet();
            }
        });
        stopWatch.stop();
        this.response.startTime = stopWatch.totalTime().millis() - startEngineStart;
        this.logger.trace("{} recovery [phase1] to {}: remote engine start took [{}]", this.request.shardId(), this.request.targetNode(), stopWatch.totalTime());
    }

    public void phase2(Translog.Snapshot snapshot) {
        if (this.shard.state() == IndexShardState.CLOSED) {
            throw new IndexShardClosedException(this.request.shardId());
        }
        this.cancellableThreads.checkForCancel();
        StopWatch stopWatch = new StopWatch().start();
        this.logger.trace("{} recovery [phase2] to {}: sending transaction log operations", this.request.shardId(), this.request.targetNode());
        int totalOperations = this.sendSnapshot(snapshot);
        stopWatch.stop();
        this.logger.trace("{} recovery [phase2] to {}: took [{}]", this.request.shardId(), this.request.targetNode(), stopWatch.totalTime());
        this.response.phase2Time = stopWatch.totalTime().millis();
        this.response.phase2Operations = totalOperations;
    }

    public void finalizeRecovery() {
        if (this.shard.state() == IndexShardState.CLOSED) {
            throw new IndexShardClosedException(this.request.shardId());
        }
        this.cancellableThreads.checkForCancel();
        StopWatch stopWatch = new StopWatch().start();
        this.logger.trace("[{}][{}] finalizing recovery to {}", this.indexName, this.shardId, this.request.targetNode());
        this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

            @Override
            public void run() throws InterruptedException {
                RecoverySourceHandler.this.transportService.submitRequest(RecoverySourceHandler.this.request.targetNode(), "internal:index/shard/recovery/finalize", new RecoveryFinalizeRecoveryRequest(RecoverySourceHandler.this.request.recoveryId(), RecoverySourceHandler.this.request.shardId()), TransportRequestOptions.builder().withTimeout(RecoverySourceHandler.this.recoverySettings.internalActionLongTimeout()).build(), EmptyTransportResponseHandler.INSTANCE_SAME).txGet();
            }
        });
        if (this.request.markAsRelocated()) {
            try {
                this.shard.relocated("to " + this.request.targetNode());
            }
            catch (IllegalIndexShardStateException illegalIndexShardStateException) {
                // empty catch block
            }
        }
        stopWatch.stop();
        this.logger.trace("[{}][{}] finalizing recovery to {}: took [{}]", this.indexName, this.shardId, this.request.targetNode(), stopWatch.totalTime());
    }

    protected int sendSnapshot(final Translog.Snapshot snapshot) {
        Translog.Operation operation;
        int ops = 0;
        long size = 0L;
        int totalOperations = 0;
        final ArrayList<Translog.Operation> operations = new ArrayList<Translog.Operation>();
        try {
            operation = snapshot.next();
        }
        catch (IOException ex) {
            throw new ElasticsearchException("failed to get next operation from translog", (Throwable)ex, new Object[0]);
        }
        final TransportRequestOptions recoveryOptions = TransportRequestOptions.builder().withCompress(this.recoverySettings.compress()).withType(TransportRequestOptions.Type.RECOVERY).withTimeout(this.recoverySettings.internalActionLongTimeout()).build();
        if (operation == null) {
            this.logger.trace("[{}][{}] no translog operations to send to {}", this.indexName, this.shardId, this.request.targetNode());
        }
        while (operation != null) {
            if (this.shard.state() == IndexShardState.CLOSED) {
                throw new IndexShardClosedException(this.request.shardId());
            }
            this.cancellableThreads.checkForCancel();
            operations.add(operation);
            ++totalOperations;
            if (++ops >= this.recoverySettings.translogOps() || (size += operation.estimateSize()) >= this.recoverySettings.translogSize().bytes()) {
                this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

                    @Override
                    public void run() throws InterruptedException {
                        RecoveryTranslogOperationsRequest translogOperationsRequest = new RecoveryTranslogOperationsRequest(RecoverySourceHandler.this.request.recoveryId(), RecoverySourceHandler.this.request.shardId(), operations, snapshot.estimatedTotalOperations());
                        RecoverySourceHandler.this.transportService.submitRequest(RecoverySourceHandler.this.request.targetNode(), "internal:index/shard/recovery/translog_ops", translogOperationsRequest, recoveryOptions, EmptyTransportResponseHandler.INSTANCE_SAME).txGet();
                    }
                });
                if (this.logger.isTraceEnabled()) {
                    this.logger.trace("[{}][{}] sent batch of [{}][{}] (total: [{}]) translog operations to {}", this.indexName, this.shardId, ops, new ByteSizeValue(size), snapshot.estimatedTotalOperations(), this.request.targetNode());
                }
                ops = 0;
                size = 0L;
                operations.clear();
            }
            try {
                operation = snapshot.next();
            }
            catch (IOException ex) {
                throw new ElasticsearchException("failed to get next operation from translog", (Throwable)ex, new Object[0]);
            }
        }
        if (!operations.isEmpty()) {
            this.cancellableThreads.execute(new CancellableThreads.Interruptable(){

                @Override
                public void run() throws InterruptedException {
                    RecoveryTranslogOperationsRequest translogOperationsRequest = new RecoveryTranslogOperationsRequest(RecoverySourceHandler.this.request.recoveryId(), RecoverySourceHandler.this.request.shardId(), operations, snapshot.estimatedTotalOperations());
                    RecoverySourceHandler.this.transportService.submitRequest(RecoverySourceHandler.this.request.targetNode(), "internal:index/shard/recovery/translog_ops", translogOperationsRequest, recoveryOptions, EmptyTransportResponseHandler.INSTANCE_SAME).txGet();
                }
            });
        }
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("[{}][{}] sent final batch of [{}][{}] (total: [{}]) translog operations to {}", this.indexName, this.shardId, ops, new ByteSizeValue(size), snapshot.estimatedTotalOperations(), this.request.targetNode());
        }
        return totalOperations;
    }

    public void cancel(String reason) {
        this.cancellableThreads.cancel(reason);
    }

    public String toString() {
        return "ShardRecoveryHandler{shardId=" + this.request.shardId() + ", sourceNode=" + this.request.sourceNode() + ", targetNode=" + this.request.targetNode() + '}';
    }
}

