/*
 * Copyright (c) 2008-present Sonatype, Inc.
 *
 * All rights reserved. Includes the third-party code listed at http://links.sonatype.com/products/nexus/pro/attributions
 * Sonatype and Sonatype Nexus are trademarks of Sonatype, Inc. Apache Maven is a trademark of the Apache Foundation.
 * M2Eclipse is a trademark of the Eclipse Foundation. All other trademarks are the property of their respective owners.
 */
/*global Ext, NX*/

/**
 * User Token controller.
 *
 * @since 3.0
 */
Ext.define('NX.proui.controller.UserToken', {
  extend: 'NX.app.Controller',
  requires: [
    'NX.Conditions',
    'NX.Messages',
    'NX.Permissions',
    'NX.Security',
    'NX.State',
    'NX.I18n'
  ],

  views: [
    'usertoken.UserTokenCurrent',
    'usertoken.UserTokenDetails',
    'usertoken.UserTokenSettings',
    'usertoken.UserTokenUser'
  ],
  refs: [
    {
      ref: 'userUserTokenPanel',
      selector: 'nx-coreui-user-feature nx-proui-usertoken-user'
    }
  ],

  /**
   * @override
   */
  init: function () {
    var me = this;

    me.getApplication().getIconController().addIcons({
      'feature-usertoken': {
        file: 'key.png',
        variants: ['x16', 'x32']
      },
      'usertoken-default': {
        file: 'key.png',
        variants: ['x16', 'x32']
      }
    });

    me.getApplication().getFeaturesController().registerFeature([
      {
        mode: 'admin',
        path: '/Security/User Token',
        view: { xtype: 'nx-proui-usertoken-settings' },
        text: NX.I18n.get('UserToken_Text'),
        description: NX.I18n.get('UserToken_Description'),
        iconConfig: {
          file: 'key.png',
          variants: ['x16', 'x32']
        },
        visible: function () {
          return NX.Permissions.check('nexus:usertoken-settings:read');
        }
      },
      {
        mode: 'user',
        path: '/User Token',
        text: NX.I18n.get('UserToken_Text'),
        description: NX.I18n.get('UserToken_Description'),
        view: { xtype: 'nx-proui-usertoken-current' },
        iconConfig: {
          file: 'key.png',
          variants: ['x16', 'x32']
        },
        visible: function () {
          var userTokenState = NX.State.getValue('usertoken');
          return NX.Permissions.check('nexus:usertoken-current:read') && userTokenState && userTokenState.enabled;
        }
      }
    ], me);

    me.listen({
      component: {
        'nx-proui-usertoken-settings button[action=resetall]': {
          click: me.resetAll,
          afterrender: me.bindResetAllButton
        },
        'nx-proui-usertoken-settings nx-settingsform': {
          loaded: me.onLoaded,
          submitted: me.onSubmitted
        },
        'nx-proui-usertoken-current button[action=access]': {
          click: me.accessCurrent,
          afterrender: me.bindAccessButton
        },
        'nx-proui-usertoken-current button[action=reset]': {
          click: me.resetCurrent,
          afterrender: me.bindResetCurrentButton
        },
        'nx-coreui-user-list': {
          beforerender: me.bindUserUserTokenTab,
          selection: me.onUserSelection
        },
        'nx-proui-usertoken-user button[action=reset]': {
          click: me.reset,
          afterrender: me.bindResetButton
        }
      }
    });
  },

  /**
   * @private
   */
  onLoaded: function (form, action) {
    var resetAllForm = form.up('nx-proui-usertoken-settings').down('#resetAll');

    if (action.result.data.enabled) {
      resetAllForm.show();
    }
    else {
      resetAllForm.hide();
    }
  },

  /**
   * @private
   */
  onSubmitted: function (form, action) {
    NX.State.setValue('usertoken', Ext.apply(
        NX.State.getValue('usertoken', {}),
        { enabled: action.result.data.enabled }
    ));
  },

  /**
   * @private
   */
  onUserSelection: function (list, model) {
    var panel = this.getUserUserTokenPanel();

    if (Ext.isDefined(model) && panel) {
      panel.down('form').loadRecord(model);
    }
  },

  /**
   * @private
   * Show/Hide user token tab if user token is enabled.
   */
  bindUserUserTokenTab: function (list) {
    var me = this;
    list.mon(
        NX.Conditions.watchState('usertoken', function (userToken) {
          return userToken && userToken.enabled;
        }),
        {
          satisfied: function () {
            var panel = me.getUserUserTokenPanel();

            if (!panel) {
              me.fireEvent('userSettingsAddTab', {xtype: 'nx-proui-usertoken-user'});
            }
          },
          unsatisfied: function () {
            var panel = me.getUserUserTokenPanel();

            if (panel) {
              me.fireEvent('userSettingsRemoveTab', {xtype: 'nx-proui-usertoken-user'});
            }
          },
          scope: me
        }
    );
  },

  /**
   * @override
   * @private
   * Enable 'Reset all user tokens' when user has 'nexus:usertoken-users:delete' permission.
   */
  bindResetAllButton: function (button) {
    button.mon(
        NX.Conditions.and(
            NX.Conditions.isPermitted('nexus:usertoken-users:delete')
        ),
        {
          satisfied: function () {
            button.enable();
          },
          unsatisfied: function () {
            button.disable();
          }
        }
    );
  },

  /**
   * @override
   * @private
   * Enable 'Access user token' when user has 'nexus:usertoken-current:read' permission.
   */
  bindAccessButton: function (button) {
    button.mon(
        NX.Conditions.and(
            NX.Conditions.isPermitted('nexus:usertoken-current:read')
        ),
        {
          satisfied: function () {
            button.enable();
          },
          unsatisfied: function () {
            button.disable();
          }
        }
    );
  },

  /**
   * @override
   * @private
   * Enable 'Reset user token' when user has 'nexus:usertoken-current:delete' permission.
   */
  bindResetCurrentButton: function (button) {
    button.mon(
        NX.Conditions.and(
            NX.Conditions.isPermitted('nexus:usertoken-current:delete')
        ),
        {
          satisfied: function () {
            button.enable();
          },
          unsatisfied: function () {
            button.disable();
          }
        }
    );
  },

  /**
   * @override
   * @private
   * Enable 'Reset user token' when user has 'nexus:usertoken-user:delete' permission.
   */
  bindResetButton: function (button) {
    button.mon(
        NX.Conditions.and(
            NX.Conditions.isPermitted('nexus:usertoken-user:delete')
        ),
        {
          satisfied: function () {
            button.enable();
          },
          unsatisfied: function () {
            button.disable();
          }
        }
    );
  },

  /**
   * @private
   */
  resetAll: function () {
    NX.Security.doWithAuthenticationToken(
        NX.I18n.format('UserToken_Authentication_Validation', 'Resetting'),
        {
          success: function (authToken) {
            NX.direct.usertoken_UserToken.resetAll(authToken, function (response) {
              if (Ext.isDefined(response) && response.success) {
                NX.Messages.add({ text: NX.I18n.get('UserToken_ResetAll_Success'), type: 'success' });
              }
            });
          }
        }
    );
  },

  /**
   * @private
   */
  accessCurrent: function () {
    NX.Security.doWithAuthenticationToken(
      NX.I18n.format('UserToken_Authentication_Validation', 'Accessing'),
        {
          success: function (authToken) {
            NX.direct.usertoken_UserToken.retrieveCurrent(authToken, function (response) {
              if (Ext.isDefined(response) && response.success) {
                Ext.widget('nx-proui-usertoken-details', response.data);
              }
            });
          }
        }
    );
  },

  /**
   * @private
   */
  resetCurrent: function () {
    NX.Security.doWithAuthenticationToken(
      NX.I18n.format('UserToken_Authentication_Validation', 'Resetting'),
        {
          success: function (authToken) {
            NX.direct.usertoken_UserToken.resetCurrent(authToken, function (response) {
              if (Ext.isDefined(response) && response.success) {
                NX.Messages.add({ text: NX.I18n.get('UserToken_Reset_Success'), type: 'success' });
              }
            });
          }
        }
    );
  },

  /**
   * @private
   */
  reset: function (button) {
    var form = button.up('form'),
        userModel = form.getRecord();
    NX.Security.doWithAuthenticationToken(
      NX.I18n.format('UserToken_Authentication_Validation', 'Resetting'),
        {
          success: function (authToken) {
            NX.direct.usertoken_UserToken.reset(authToken, userModel.get('userId'), function (response) {
              var errorMessage = '';
              if (Ext.isDefined(response) && response.success) {
                NX.Messages.add({
                  text: NX.I18n.format('UserToken_ResetById_Message', userModel.get('userId')),
                  type: 'success'
                });
              }
              else if (Ext.isDefined(response.errors)) {
                Object.keys(response.errors).forEach(function(key) {
                  errorMessage += key + ' ' + response.errors[key] + ". ";
                });
                NX.Messages.warning(errorMessage);
              }
            });
          }
        }
    );
  }

});
