/*
 * character.ts
 *
 * Copyright (C) 2022 by RStudio, PBC
 *
 * Unless you have received this program directly from RStudio pursuant
 * to the terms of a commercial license agreement with RStudio, then
 * this program is licensed to you under the terms of version 3 of the
 * GNU Affero General Public License. This program is distributed WITHOUT
 * ANY EXPRESS OR IMPLIED WARRANTY, INCLUDING THOSE OF NON-INFRINGEMENT,
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Please refer to the
 * AGPL (http://www.gnu.org/licenses/agpl-3.0.txt) for more details.
 *
 */

// Maps unicode characters to their latex equivalent.
// Generated from:
// https://www.w3.org/2003/entities/2007xml/unicode.xml
// and
// http://www.w3.org/Math/characters/unicode.xml
//
// When both xml files contained entries for a unicode character,
// the definition provided by entities/2007xml was preferred
export interface CharacterMap {
  id: string;
  decimal: number;
  character: string;
  latex: string;
  ungrouped?: boolean;
}

const characters: { [key: number]: CharacterMap } = {
  34: { id: 'U00022', decimal: 34, character: '"', latex: '"' },
  35: { id: 'U00023', decimal: 35, character: '#', latex: '\\#' },
  36: { id: 'U00024', decimal: 36, character: '$', latex: '\\$' },
  94: { id: 'U0005e', decimal: 94, character: '^', latex: '\\textasciicircum' },
  95: { id: 'U0005f', decimal: 95, character: '_', latex: '\\_' },
  123: { id: 'U0007b', decimal: 123, character: '{', latex: '\\textbraceleft' },
  125: { id: 'U0007d', decimal: 125, character: '}', latex: '\\textbraceright' },
  256: {
    id: 'U00100',
    decimal: 256,
    character: 'Ā',
    latex: '\\={A}',
  },
  257: {
    id: 'U00101',
    decimal: 257,
    character: 'ā',
    latex: '\\={a}',
  },
  258: {
    id: 'U00102',
    decimal: 258,
    character: 'Ă',
    latex: '\\u{A}',
  },
  259: {
    id: 'U00103',
    decimal: 259,
    character: 'ă',
    latex: '\\u{a}',
  },
  260: {
    id: 'U00104',
    decimal: 260,
    character: 'Ą',
    latex: '\\k{A}',
  },
  261: {
    id: 'U00105',
    decimal: 261,
    character: 'ą',
    latex: '\\k{a}',
  },
  262: {
    id: 'U00106',
    decimal: 262,
    character: 'Ć',
    latex: "\\'{C}",
  },
  263: {
    id: 'U00107',
    decimal: 263,
    character: 'ć',
    latex: "\\'{c}",
  },
  264: {
    id: 'U00108',
    decimal: 264,
    character: 'Ĉ',
    latex: '\\^{C}',
  },
  265: {
    id: 'U00109',
    decimal: 265,
    character: 'ĉ',
    latex: '\\^{c}',
  },
  266: {
    id: 'U0010A',
    decimal: 266,
    character: 'Ċ',
    latex: '\\.{C}',
  },
  267: {
    id: 'U0010B',
    decimal: 267,
    character: 'ċ',
    latex: '\\.{c}',
  },
  268: {
    id: 'U0010C',
    decimal: 268,
    character: 'Č',
    latex: '\\v{C}',
  },
  269: {
    id: 'U0010D',
    decimal: 269,
    character: 'č',
    latex: '\\v{c}',
  },
  270: {
    id: 'U0010E',
    decimal: 270,
    character: 'Ď',
    latex: '\\v{D}',
  },
  271: {
    id: 'U0010F',
    decimal: 271,
    character: 'ď',
    latex: '\\v{d}',
  },
  272: {
    id: 'U00110',
    decimal: 272,
    character: 'Đ',
    latex: '\\DJ',
  },
  273: {
    id: 'U00111',
    decimal: 273,
    character: 'đ',
    latex: '\\dj',
  },
  274: {
    id: 'U00112',
    decimal: 274,
    character: 'Ē',
    latex: '\\={E}',
  },
  275: {
    id: 'U00113',
    decimal: 275,
    character: 'ē',
    latex: '\\={e}',
  },
  276: {
    id: 'U00114',
    decimal: 276,
    character: 'Ĕ',
    latex: '\\u{E}',
  },
  277: {
    id: 'U00115',
    decimal: 277,
    character: 'ĕ',
    latex: '\\u{e}',
  },
  278: {
    id: 'U00116',
    decimal: 278,
    character: 'Ė',
    latex: '\\.{E}',
  },
  279: {
    id: 'U00117',
    decimal: 279,
    character: 'ė',
    latex: '\\.{e}',
  },
  280: {
    id: 'U00118',
    decimal: 280,
    character: 'Ę',
    latex: '\\k{E}',
  },
  281: {
    id: 'U00119',
    decimal: 281,
    character: 'ę',
    latex: '\\k{e}',
  },
  282: {
    id: 'U0011A',
    decimal: 282,
    character: 'Ě',
    latex: '\\v{E}',
  },
  283: {
    id: 'U0011B',
    decimal: 283,
    character: 'ě',
    latex: '\\v{e}',
  },
  284: {
    id: 'U0011C',
    decimal: 284,
    character: 'Ĝ',
    latex: '\\^{G}',
  },
  285: {
    id: 'U0011D',
    decimal: 285,
    character: 'ĝ',
    latex: '\\^{g}',
  },
  286: {
    id: 'U0011E',
    decimal: 286,
    character: 'Ğ',
    latex: '\\u{G}',
  },
  287: {
    id: 'U0011F',
    decimal: 287,
    character: 'ğ',
    latex: '\\u{g}',
  },
  288: {
    id: 'U00120',
    decimal: 288,
    character: 'Ġ',
    latex: '\\.{G}',
  },
  289: {
    id: 'U00121',
    decimal: 289,
    character: 'ġ',
    latex: '\\.{g}',
  },
  290: {
    id: 'U00122',
    decimal: 290,
    character: 'Ģ',
    latex: '\\c{G}',
  },
  291: {
    id: 'U00123',
    decimal: 291,
    character: 'ģ',
    latex: '\\c{g}',
  },
  292: {
    id: 'U00124',
    decimal: 292,
    character: 'Ĥ',
    latex: '\\^{H}',
  },
  293: {
    id: 'U00125',
    decimal: 293,
    character: 'ĥ',
    latex: '\\^{h}',
  },
  296: {
    id: 'U00128',
    decimal: 296,
    character: 'Ĩ',
    latex: '\\~{I}',
  },
  297: {
    id: 'U00129',
    decimal: 297,
    character: 'ĩ',
    latex: '\\~{\\i}',
  },
  298: {
    id: 'U0012A',
    decimal: 298,
    character: 'Ī',
    latex: '\\={I}',
  },
  299: {
    id: 'U0012B',
    decimal: 299,
    character: 'ī',
    latex: '\\={\\i}',
  },
  300: {
    id: 'U0012C',
    decimal: 300,
    character: 'Ĭ',
    latex: '\\u{I}',
  },
  301: {
    id: 'U0012D',
    decimal: 301,
    character: 'ĭ',
    latex: '\\u{\\i}',
  },
  302: {
    id: 'U0012E',
    decimal: 302,
    character: 'Į',
    latex: '\\k{I}',
  },
  303: {
    id: 'U0012F',
    decimal: 303,
    character: 'į',
    latex: '\\k{i}',
  },
  304: {
    id: 'U00130',
    decimal: 304,
    character: 'İ',
    latex: '\\.{I}',
  },
  305: {
    id: 'U00131',
    decimal: 305,
    character: 'ı',
    latex: '\\i',
  },
  306: {
    id: 'U00132',
    decimal: 306,
    character: 'Ĳ',
    latex: 'IJ',
  },
  307: {
    id: 'U00133',
    decimal: 307,
    character: 'ĳ',
    latex: 'ij',
  },
  308: {
    id: 'U00134',
    decimal: 308,
    character: 'Ĵ',
    latex: '\\^{J}',
  },
  309: {
    id: 'U00135',
    decimal: 309,
    character: 'ĵ',
    latex: '\\^{\\j}',
  },
  310: {
    id: 'U00136',
    decimal: 310,
    character: 'Ķ',
    latex: '\\c{K}',
  },
  311: {
    id: 'U00137',
    decimal: 311,
    character: 'ķ',
    latex: '\\c{k}',
  },
  313: {
    id: 'U00139',
    decimal: 313,
    character: 'Ĺ',
    latex: "\\'{L}",
  },
  314: {
    id: 'U0013A',
    decimal: 314,
    character: 'ĺ',
    latex: "\\'{l}",
  },
  315: {
    id: 'U0013B',
    decimal: 315,
    character: 'Ļ',
    latex: '\\c{L}',
  },
  316: {
    id: 'U0013C',
    decimal: 316,
    character: 'ļ',
    latex: '\\c{l}',
  },
  317: {
    id: 'U0013D',
    decimal: 317,
    character: 'Ľ',
    latex: '\\v{L}',
  },
  318: {
    id: 'U0013E',
    decimal: 318,
    character: 'ľ',
    latex: '\\v{l}',
  },
  321: {
    id: 'U00141',
    decimal: 321,
    character: 'Ł',
    latex: '\\L',
  },
  322: {
    id: 'U00142',
    decimal: 322,
    character: 'ł',
    latex: '\\l',
  },
  323: {
    id: 'U00143',
    decimal: 323,
    character: 'Ń',
    latex: "\\'{N}",
  },
  324: {
    id: 'U00144',
    decimal: 324,
    character: 'ń',
    latex: "\\'{n}",
  },
  325: {
    id: 'U00145',
    decimal: 325,
    character: 'Ņ',
    latex: '\\c{N}',
  },
  326: {
    id: 'U00146',
    decimal: 326,
    character: 'ņ',
    latex: '\\c{n}',
  },
  327: {
    id: 'U00147',
    decimal: 327,
    character: 'Ň',
    latex: '\\v{N}',
  },
  328: {
    id: 'U00148',
    decimal: 328,
    character: 'ň',
    latex: '\\v{n}',
  },
  329: {
    id: 'U00149',
    decimal: 329,
    character: 'ŉ',
    latex: "'n",
  },
  330: {
    id: 'U0014A',
    decimal: 330,
    character: 'Ŋ',
    latex: '\\NG',
  },
  331: {
    id: 'U0014B',
    decimal: 331,
    character: 'ŋ',
    latex: '\\ng',
  },
  332: {
    id: 'U0014C',
    decimal: 332,
    character: 'Ō',
    latex: '\\={O}',
  },
  333: {
    id: 'U0014D',
    decimal: 333,
    character: 'ō',
    latex: '\\={o}',
  },
  334: {
    id: 'U0014E',
    decimal: 334,
    character: 'Ŏ',
    latex: '\\u{O}',
  },
  335: {
    id: 'U0014F',
    decimal: 335,
    character: 'ŏ',
    latex: '\\u{o}',
  },
  336: {
    id: 'U00150',
    decimal: 336,
    character: 'Ő',
    latex: '{\\H O}',
  },
  337: {
    id: 'U00151',
    decimal: 337,
    character: 'ő',
    latex: '{\\H o}',
  },
  338: {
    id: 'U00152',
    decimal: 338,
    character: 'Œ',
    latex: '\\OE',
  },
  339: {
    id: 'U00153',
    decimal: 339,
    character: 'œ',
    latex: '\\oe',
  },
  340: {
    id: 'U00154',
    decimal: 340,
    character: 'Ŕ',
    latex: "\\'{R}",
  },
  341: {
    id: 'U00155',
    decimal: 341,
    character: 'ŕ',
    latex: "\\'{r}",
  },
  342: {
    id: 'U00156',
    decimal: 342,
    character: 'Ŗ',
    latex: '\\c{R}',
  },
  343: {
    id: 'U00157',
    decimal: 343,
    character: 'ŗ',
    latex: '\\c{r}',
  },
  344: {
    id: 'U00158',
    decimal: 344,
    character: 'Ř',
    latex: '\\v{R}',
  },
  345: {
    id: 'U00159',
    decimal: 345,
    character: 'ř',
    latex: '\\v{r}',
  },
  346: {
    id: 'U0015A',
    decimal: 346,
    character: 'Ś',
    latex: "\\'{S}",
  },
  347: {
    id: 'U0015B',
    decimal: 347,
    character: 'ś',
    latex: "\\'{s}",
  },
  348: {
    id: 'U0015C',
    decimal: 348,
    character: 'Ŝ',
    latex: '\\^{S}',
  },
  349: {
    id: 'U0015D',
    decimal: 349,
    character: 'ŝ',
    latex: '\\^{s}',
  },
  350: {
    id: 'U0015E',
    decimal: 350,
    character: 'Ş',
    latex: '\\c{S}',
  },
  351: {
    id: 'U0015F',
    decimal: 351,
    character: 'ş',
    latex: '\\c{s}',
  },
  352: {
    id: 'U00160',
    decimal: 352,
    character: 'Š',
    latex: '\\v{S}',
  },
  353: {
    id: 'U00161',
    decimal: 353,
    character: 'š',
    latex: '\\v{s}',
  },
  354: {
    id: 'U00162',
    decimal: 354,
    character: 'Ţ',
    latex: '\\c{T}',
  },
  355: {
    id: 'U00163',
    decimal: 355,
    character: 'ţ',
    latex: '\\c{t}',
  },
  356: {
    id: 'U00164',
    decimal: 356,
    character: 'Ť',
    latex: '\\v{T}',
  },
  357: {
    id: 'U00165',
    decimal: 357,
    character: 'ť',
    latex: '\\v{t}',
  },
  360: {
    id: 'U00168',
    decimal: 360,
    character: 'Ũ',
    latex: '\\~{U}',
  },
  361: {
    id: 'U00169',
    decimal: 361,
    character: 'ũ',
    latex: '\\~{u}',
  },
  362: {
    id: 'U0016A',
    decimal: 362,
    character: 'Ū',
    latex: '\\={U}',
  },
  363: {
    id: 'U0016B',
    decimal: 363,
    character: 'ū',
    latex: '\\={u}',
  },
  364: {
    id: 'U0016C',
    decimal: 364,
    character: 'Ŭ',
    latex: '\\u{U}',
  },
  365: {
    id: 'U0016D',
    decimal: 365,
    character: 'ŭ',
    latex: '\\u{u}',
  },
  366: {
    id: 'U0016E',
    decimal: 366,
    character: 'Ů',
    latex: '\\r{U}',
  },
  367: {
    id: 'U0016F',
    decimal: 367,
    character: 'ů',
    latex: '\\r{u}',
  },
  368: {
    id: 'U00170',
    decimal: 368,
    character: 'Ű',
    latex: '{\\H U}',
  },
  369: {
    id: 'U00171',
    decimal: 369,
    character: 'ű',
    latex: '{\\H u}',
  },
  370: {
    id: 'U00172',
    decimal: 370,
    character: 'Ų',
    latex: '\\k{U}',
  },
  371: {
    id: 'U00173',
    decimal: 371,
    character: 'ų',
    latex: '\\k{u}',
  },
  372: {
    id: 'U00174',
    decimal: 372,
    character: 'Ŵ',
    latex: '\\^{W}',
  },
  373: {
    id: 'U00175',
    decimal: 373,
    character: 'ŵ',
    latex: '\\^{w}',
  },
  374: {
    id: 'U00176',
    decimal: 374,
    character: 'Ŷ',
    latex: '\\^{Y}',
  },
  375: {
    id: 'U00177',
    decimal: 375,
    character: 'ŷ',
    latex: '\\^{y}',
  },
  376: {
    id: 'U00178',
    decimal: 376,
    character: 'Ÿ',
    latex: '\\"{Y}',
  },
  377: {
    id: 'U00179',
    decimal: 377,
    character: 'Ź',
    latex: "\\'{Z}",
  },
  378: {
    id: 'U0017A',
    decimal: 378,
    character: 'ź',
    latex: "\\'{z}",
  },
  379: {
    id: 'U0017B',
    decimal: 379,
    character: 'Ż',
    latex: '\\.{Z}',
  },
  380: {
    id: 'U0017C',
    decimal: 380,
    character: 'ż',
    latex: '\\.{z}',
  },
  381: {
    id: 'U0017D',
    decimal: 381,
    character: 'Ž',
    latex: '\\v{Z}',
  },
  382: {
    id: 'U0017E',
    decimal: 382,
    character: 'ž',
    latex: '\\v{z}',
  },
  402: {
    id: 'U00192',
    decimal: 402,
    character: 'ƒ',
    latex: '\\textflorin',
  },
  501: {
    id: 'U001F5',
    decimal: 501,
    character: 'ǵ',
    latex: "\\'{g}",
  },
  609: {
    id: 'U00261',
    decimal: 609,
    character: 'ɡ',
    latex: 'g',
  },
  700: {
    id: 'U002BC',
    decimal: 700,
    character: 'ʼ',
    latex: "'",
  },
  711: {
    id: 'U002C7',
    decimal: 711,
    character: 'ˇ',
    latex: '\\textasciicaron',
  },
  714: {
    id: 'U002ca',
    decimal: 714,
    character: 'ˊ',
    latex: '\\textasciitilde',
  },
  728: {
    id: 'U002D8',
    decimal: 728,
    character: '˘',
    latex: '\\textasciibreve',
  },
  729: {
    id: 'U002D9',
    decimal: 729,
    character: '˙',
    latex: '\\textperiodcentered',
  },
  730: {
    id: 'U002DA',
    decimal: 730,
    character: '˚',
    latex: '\\r{}',
  },
  731: {
    id: 'U002DB',
    decimal: 731,
    character: '˛',
    latex: '\\k{}',
  },
  732: {
    id: 'U002DC',
    decimal: 732,
    character: '˜',
    latex: '\\texttildelow',
  },
  733: {
    id: 'U002dd',
    decimal: 733,
    character: '˝',
    latex: '\\textacutedbl',
  },
  768: {
    id: 'U00300',
    decimal: 768,
    character: '̀',
    latex: '\\`',
    ungrouped: true,
  },
  769: {
    id: 'U00301',
    decimal: 769,
    character: '́',
    latex: "\\'",
    ungrouped: true,
  },
  770: {
    id: 'U00302',
    decimal: 770,
    character: '̂',
    latex: '\\^',
    ungrouped: true,
  },
  771: {
    id: 'U00303',
    decimal: 771,
    character: '̃',
    latex: '\\~',
    ungrouped: true,
  },
  772: {
    id: 'U00304',
    decimal: 772,
    character: '̄',
    latex: '\\=',
    ungrouped: true,
  },
  774: {
    id: 'U00306',
    decimal: 774,
    character: '̆',
    latex: '\\u',
    ungrouped: true,
  },
  775: {
    id: 'U00307',
    decimal: 775,
    character: '̇',
    latex: '\\.',
    ungrouped: true,
  },
  776: {
    id: 'U00308',
    decimal: 776,
    character: '̈',
    latex: '\\"',
    ungrouped: true,
  },
  778: {
    id: 'U0030A',
    decimal: 778,
    character: '̊',
    latex: '\\r',
    ungrouped: true,
  },
  780: {
    id: 'U0030C',
    decimal: 780,
    character: '̌',
    latex: '\\v',
    ungrouped: true,
  },
  807: {
    id: 'U00327',
    decimal: 807,
    character: '̧',
    latex: '\\c',
    ungrouped: true,
  },
  808: {
    id: 'U00328',
    decimal: 808,
    character: '̨',
    latex: '\\k',
    ungrouped: true,
  },
  902: {
    id: 'U00386',
    decimal: 902,
    character: 'Ά',
    latex: "\\'{A}",
  },
  904: {
    id: 'U00388',
    decimal: 904,
    character: 'Έ',
    latex: "\\'{E}",
  },
  905: {
    id: 'U00389',
    decimal: 905,
    character: 'Ή',
    latex: "\\'{H}",
  },
  906: {
    id: 'U0038A',
    decimal: 906,
    character: 'Ί',
    latex: "\\'{}{I}",
  },
  908: {
    id: 'U0038C',
    decimal: 908,
    character: 'Ό',
    latex: "\\'{}O",
  },
  910: {
    id: 'U0038E',
    decimal: 910,
    character: 'Ύ',
    latex: "$\\mathrm{'Y}$",
  },
  911: {
    id: 'U0038F',
    decimal: 911,
    character: 'Ώ',
    latex: "$\\mathrm{'\\Omega}$",
  },
  924: {
    id: 'U0039C',
    decimal: 924,
    character: 'Μ',
    latex: 'M',
  },
  925: {
    id: 'U0039D',
    decimal: 925,
    character: 'Ν',
    latex: 'N',
  },
  927: {
    id: 'U0039F',
    decimal: 927,
    character: 'Ο',
    latex: 'O',
  },
  938: {
    id: 'U003AA',
    decimal: 938,
    character: 'Ϊ',
    latex: '$\\mathrm{\\ddot{I}}$',
  },
  939: {
    id: 'U003AB',
    decimal: 939,
    character: 'Ϋ',
    latex: '$\\mathrm{\\ddot{Y}}$',
  },
  940: {
    id: 'U003AC',
    decimal: 940,
    character: 'ά',
    latex: "\\'{$\\alpha$}",
  },
  959: {
    id: 'U003BF',
    decimal: 959,
    character: 'ο',
    latex: 'o',
  },
  972: {
    id: 'U003CC',
    decimal: 972,
    character: 'ό',
    latex: "\\'{o}",
  },
  8194: {
    id: 'U02002',
    decimal: 8194,
    character: ' ',
    latex: '\\hspace{0.6em}',
  },
  8195: {
    id: 'U02003',
    decimal: 8195,
    character: ' ',
    latex: '\\hspace{1em}',
  },
  8196: {
    id: 'U02004',
    decimal: 8196,
    character: ' ',
    latex: '\\hspace{0.33em}',
  },
  8197: {
    id: 'U02005',
    decimal: 8197,
    character: ' ',
    latex: '\\hspace{0.25em}',
  },
  8198: {
    id: 'U02006',
    decimal: 8198,
    character: ' ',
    latex: '\\hspace{0.166em}',
  },
  8199: {
    id: 'U02007',
    decimal: 8199,
    character: ' ',
    latex: '\\hphantom{0}',
  },
  8200: {
    id: 'U02008',
    decimal: 8200,
    character: ' ',
    latex: '\\hphantom{,}',
  },
  8201: {
    id: 'U02009',
    decimal: 8201,
    character: ' ',
    latex: '\\hspace{0.167em}',
  },
  8204: {
    id: 'U00200c',
    decimal: 8204,
    character: '‌',
    latex: '\\textcompwordmark',
  },
  8208: {
    id: 'U02010',
    decimal: 8208,
    character: '‐',
    latex: '-',
  },
  8209: {
    id: 'U2011',
    decimal: 8209,
    character: '‑',
    latex: '\\nobreakdash-'
  },
  8211: {
    id: 'U02013',
    decimal: 8211,
    character: '–',
    latex: '\\textendash',
  },
  8212: {
    id: 'U02014',
    decimal: 8212,
    character: '—',
    latex: '\\textemdash',
  },
  8213: {
    id: 'U02015',
    decimal: 8213,
    character: '―',
    latex: '\\rule{1em}{1pt}',
  },
  8216: {
    id: 'U02018',
    decimal: 8216,
    character: '‘',
    latex: '\\textquoteleft',
  },
  8217: {
    id: 'U02019',
    decimal: 8217,
    character: '’',
    latex: '\\textquoteright',
  },
  8218: {
    id: 'U0201A',
    decimal: 8218,
    character: '‚',
    latex: ',',
  },
  8220: {
    id: 'U0201C',
    decimal: 8220,
    character: '“',
    latex: '\\textquotedblleft',
  },
  8221: {
    id: 'U0201D',
    decimal: 8221,
    character: '”',
    latex: '\\textquotedblright',
  },
  8222: {
    id: 'U0201E',
    decimal: 8222,
    character: '„',
    latex: ',,',
  },
  8224: {
    id: 'U02020',
    decimal: 8224,
    character: '†',
    latex: '\\textdagger',
  },
  8225: {
    id: 'U02021',
    decimal: 8225,
    character: '‡',
    latex: '\\textdaggerdbl',
  },
  8226: {
    id: 'U02022',
    decimal: 8226,
    character: '•',
    latex: '\\textbullet',
  },
  8228: {
    id: 'U02024',
    decimal: 8228,
    character: '․',
    latex: '.',
  },
  8229: {
    id: 'U02025',
    decimal: 8229,
    character: '‥',
    latex: '..',
  },
  8230: {
    id: 'U02026',
    decimal: 8230,
    character: '…',
    latex: '\\ldots',
  },
  8240: {
    id: 'U02030',
    decimal: 8240,
    character: '‰',
    latex: '\\textperthousand',
  },
  8241: {
    id: 'U02031',
    decimal: 8241,
    character: '‱',
    latex: '\\textpertenthousand',
  },
  8242: {
    id: 'U02032',
    decimal: 8242,
    character: '′',
    latex: "{'}",
  },
  8243: {
    id: 'U02033',
    decimal: 8243,
    character: '″',
    latex: "{''}",
  },
  8244: {
    id: 'U02034',
    decimal: 8244,
    character: '‴',
    latex: "{'''}",
  },
  8249: {
    id: 'U02039',
    decimal: 8249,
    character: '‹',
    latex: '\\guilsinglleft',
  },
  8250: {
    id: 'U0203A',
    decimal: 8250,
    character: '›',
    latex: '\\guilsinglright',
  },
  8279: {
    id: 'U02057',
    decimal: 8279,
    character: '⁗',
    latex: "''''",
  },
  8288: {
    id: 'U02060',
    decimal: 8288,
    character: '⁠',
    latex: '\\nolinebreak',
  },
  8364: {
    id: 'U020AC',
    decimal: 8364,
    character: '€',
    latex: '\\mbox{\\texteuro}',
  },
  8450: {
    id: 'U02102',
    decimal: 8450,
    character: 'ℂ',
    latex: '$\\mathbb{C}$',
  },
  8460: {
    id: 'U0210C',
    decimal: 8460,
    character: 'ℌ',
    latex: '$\\mathfrak{H}$',
  },
  8461: {
    id: 'U0210D',
    decimal: 8461,
    character: 'ℍ',
    latex: '$\\mathbb{H}$',
  },
  8465: {
    id: 'U02111',
    decimal: 8465,
    character: 'ℑ',
    latex: '$\\mathfrak{I}$',
  },
  8469: {
    id: 'U02115',
    decimal: 8469,
    character: 'ℕ',
    latex: '$\\mathbb{N}$',
  },
  8473: {
    id: 'U02119',
    decimal: 8473,
    character: 'ℙ',
    latex: '$\\mathbb{P}$',
  },
  8474: {
    id: 'U0211A',
    decimal: 8474,
    character: 'ℚ',
    latex: '$\\mathbb{Q}$',
  },
  8476: {
    id: 'U0211C',
    decimal: 8476,
    character: 'ℜ',
    latex: '$\\mathfrak{R}$',
  },
  8477: {
    id: 'U0211D',
    decimal: 8477,
    character: 'ℝ',
    latex: '$\\mathbb{R}$',
  },
  8482: {
    id: 'U02122',
    decimal: 8482,
    character: '™',
    latex: '\\texttrademark',
  },
  8484: {
    id: 'U02124',
    decimal: 8484,
    character: 'ℤ',
    latex: '$\\mathbb{Z}$',
  },
  8488: {
    id: 'U02128',
    decimal: 8488,
    character: 'ℨ',
    latex: '$\\mathfrak{Z}$',
  },
  8491: {
    id: 'U0212B',
    decimal: 8491,
    character: 'Å',
    latex: '\\AA',
  },
  8493: {
    id: 'U0212D',
    decimal: 8493,
    character: 'ℭ',
    latex: '$\\mathfrak{C}$',
  },
  8722: {
    id: 'U02212',
    decimal: 8722,
    character: '−',
    latex: '-',
  },
  8762: {
    id: 'U0223A',
    decimal: 8762,
    character: '∺',
    latex: '$\\mathbin{{:}\\!\\!{-}\\!\\!{:}}$',
  },
  8788: {
    id: 'U02254',
    decimal: 8788,
    character: '≔',
    latex: ':=',
  },
  8789: {
    id: 'U02255',
    decimal: 8789,
    character: '≕',
    latex: '=:',
  },
  8942: {
    id: 'U022EE',
    decimal: 8942,
    character: '⋮',
    latex: '\\vdots',
  },
  8982: {
    id: 'U02316',
    decimal: 8982,
    character: '⌖',
    latex: '$\\mathchar"2208$',
  },
  9251: {
    id: 'U02423',
    decimal: 9251,
    character: '␣',
    latex: '\\textvisiblespace',
  },
  9645: {
    id: 'U025AD',
    decimal: 9645,
    character: '▭',
    latex: '\\fbox{~~}',
  },
  11005: {
    id: 'U02AFD-020E5',
    decimal: 11005,
    character: '⫽',
    latex: '{\\rlap{\\textbackslash}{{/}\\!\\!{/}}}',
  },
  64256: {
    id: 'U0FB00',
    decimal: 64256,
    character: 'ﬀ',
    latex: 'ff',
  },
  64257: {
    id: 'U0FB01',
    decimal: 64257,
    character: 'ﬁ',
    latex: 'fi',
  },
  64258: {
    id: 'U0FB02',
    decimal: 64258,
    character: 'ﬂ',
    latex: 'fl',
  },
  64259: {
    id: 'U0FB03',
    decimal: 64259,
    character: 'ﬃ',
    latex: 'ffi',
  },
  64260: {
    id: 'U0FB04',
    decimal: 64260,
    character: 'ﬄ',
    latex: 'ffl',
  },
  119808: {
    id: 'U1D400',
    decimal: 119808,
    character: '퐀',
    latex: '$\\mathbf{A}$',
  },
  119809: {
    id: 'U1D401',
    decimal: 119809,
    character: '퐁',
    latex: '$\\mathbf{B}$',
  },
  119810: {
    id: 'U1D402',
    decimal: 119810,
    character: '퐂',
    latex: '$\\mathbf{C}$',
  },
  119811: {
    id: 'U1D403',
    decimal: 119811,
    character: '퐃',
    latex: '$\\mathbf{D}$',
  },
  119812: {
    id: 'U1D404',
    decimal: 119812,
    character: '퐄',
    latex: '$\\mathbf{E}$',
  },
  119813: {
    id: 'U1D405',
    decimal: 119813,
    character: '퐅',
    latex: '{$\\mathbf{F}$}',
  },
  119814: {
    id: 'U1D406',
    decimal: 119814,
    character: '퐆',
    latex: '$\\mathbf{G}$',
  },
  119815: {
    id: 'U1D407',
    decimal: 119815,
    character: '퐇',
    latex: '$\\mathbf{H}$',
  },
  119816: {
    id: 'U1D408',
    decimal: 119816,
    character: '퐈',
    latex: '$\\mathbf{I}$',
  },
  119817: {
    id: 'U1D409',
    decimal: 119817,
    character: '퐉',
    latex: '$\\mathbf{J}$',
  },
  119818: {
    id: 'U1D40A',
    decimal: 119818,
    character: '퐊',
    latex: '$\\mathbf{K}$',
  },
  119819: {
    id: 'U1D40B',
    decimal: 119819,
    character: '퐋',
    latex: '$\\mathbf{L}$',
  },
  119820: {
    id: 'U1D40C',
    decimal: 119820,
    character: '퐌',
    latex: '$\\mathbf{M}$',
  },
  119821: {
    id: 'U1D40D',
    decimal: 119821,
    character: '퐍',
    latex: '$\\mathbf{N}$',
  },
  119822: {
    id: 'U1D40E',
    decimal: 119822,
    character: '퐎',
    latex: '$\\mathbf{O}$',
  },
  119823: {
    id: 'U1D40F',
    decimal: 119823,
    character: '퐏',
    latex: '$\\mathbf{P}$',
  },
  119824: {
    id: 'U1D410',
    decimal: 119824,
    character: '퐐',
    latex: '$\\mathbf{Q}$',
  },
  119825: {
    id: 'U1D411',
    decimal: 119825,
    character: '퐑',
    latex: '$\\mathbf{R}$',
  },
  119826: {
    id: 'U1D412',
    decimal: 119826,
    character: '퐒',
    latex: '$\\mathbf{S}$',
  },
  119827: {
    id: 'U1D413',
    decimal: 119827,
    character: '퐓',
    latex: '$\\mathbf{T}$',
  },
  119828: {
    id: 'U1D414',
    decimal: 119828,
    character: '퐔',
    latex: '$\\mathbf{U}$',
  },
  119829: {
    id: 'U1D415',
    decimal: 119829,
    character: '퐕',
    latex: '$\\mathbf{V}$',
  },
  119830: {
    id: 'U1D416',
    decimal: 119830,
    character: '퐖',
    latex: '$\\mathbf{W}$',
  },
  119831: {
    id: 'U1D417',
    decimal: 119831,
    character: '퐗',
    latex: '$\\mathbf{X}$',
  },
  119832: {
    id: 'U1D418',
    decimal: 119832,
    character: '퐘',
    latex: '$\\mathbf{Y}$',
  },
  119833: {
    id: 'U1D419',
    decimal: 119833,
    character: '퐙',
    latex: '$\\mathbf{Z}$',
  },
  119834: {
    id: 'U1D41A',
    decimal: 119834,
    character: '퐚',
    latex: '$\\mathbf{a}$',
  },
  119835: {
    id: 'U1D41B',
    decimal: 119835,
    character: '퐛',
    latex: '$\\mathbf{b}$',
  },
  119836: {
    id: 'U1D41C',
    decimal: 119836,
    character: '퐜',
    latex: '$\\mathbf{c}$',
  },
  119837: {
    id: 'U1D41D',
    decimal: 119837,
    character: '퐝',
    latex: '$\\mathbf{d}$',
  },
  119838: {
    id: 'U1D41E',
    decimal: 119838,
    character: '퐞',
    latex: '$\\mathbf{e}$',
  },
  119839: {
    id: 'U1D41F',
    decimal: 119839,
    character: '퐟',
    latex: '{$\\mathbf{f}$}',
  },
  119840: {
    id: 'U1D420',
    decimal: 119840,
    character: '퐠',
    latex: '$\\mathbf{g}$',
  },
  119841: {
    id: 'U1D421',
    decimal: 119841,
    character: '퐡',
    latex: '$\\mathbf{h}$',
  },
  119842: {
    id: 'U1D422',
    decimal: 119842,
    character: '퐢',
    latex: '$\\mathbf{i}$',
  },
  119843: {
    id: 'U1D423',
    decimal: 119843,
    character: '퐣',
    latex: '$\\mathbf{j}$',
  },
  119844: {
    id: 'U1D424',
    decimal: 119844,
    character: '퐤',
    latex: '$\\mathbf{k}$',
  },
  119845: {
    id: 'U1D425',
    decimal: 119845,
    character: '퐥',
    latex: '$\\mathbf{l}$',
  },
  119846: {
    id: 'U1D426',
    decimal: 119846,
    character: '퐦',
    latex: '$\\mathbf{m}$',
  },
  119847: {
    id: 'U1D427',
    decimal: 119847,
    character: '퐧',
    latex: '$\\mathbf{n}$',
  },
  119848: {
    id: 'U1D428',
    decimal: 119848,
    character: '퐨',
    latex: '$\\mathbf{o}$',
  },
  119849: {
    id: 'U1D429',
    decimal: 119849,
    character: '퐩',
    latex: '$\\mathbf{p}$',
  },
  119850: {
    id: 'U1D42A',
    decimal: 119850,
    character: '퐪',
    latex: '$\\mathbf{q}$',
  },
  119851: {
    id: 'U1D42B',
    decimal: 119851,
    character: '퐫',
    latex: '$\\mathbf{r}$',
  },
  119852: {
    id: 'U1D42C',
    decimal: 119852,
    character: '퐬',
    latex: '$\\mathbf{s}$',
  },
  119853: {
    id: 'U1D42D',
    decimal: 119853,
    character: '퐭',
    latex: '$\\mathbf{t}$',
  },
  119854: {
    id: 'U1D42E',
    decimal: 119854,
    character: '퐮',
    latex: '$\\mathbf{u}$',
  },
  119855: {
    id: 'U1D42F',
    decimal: 119855,
    character: '퐯',
    latex: '$\\mathbf{v}$',
  },
  119856: {
    id: 'U1D430',
    decimal: 119856,
    character: '퐰',
    latex: '$\\mathbf{w}$',
  },
  119857: {
    id: 'U1D431',
    decimal: 119857,
    character: '퐱',
    latex: '$\\mathbf{x}$',
  },
  119858: {
    id: 'U1D432',
    decimal: 119858,
    character: '퐲',
    latex: '$\\mathbf{y}$',
  },
  119859: {
    id: 'U1D433',
    decimal: 119859,
    character: '퐳',
    latex: '$\\mathbf{z}$',
  },
  120068: {
    id: 'U1D504',
    decimal: 120068,
    character: '프',
    latex: '$\\mathfrak{A}$',
  },
  120069: {
    id: 'U1D505',
    decimal: 120069,
    character: '픅',
    latex: '$\\mathfrak{B}$',
  },
  120071: {
    id: 'U1D507',
    decimal: 120071,
    character: '픇',
    latex: '$\\mathfrak{D}$',
  },
  120072: {
    id: 'U1D508',
    decimal: 120072,
    character: '픈',
    latex: '$\\mathfrak{E}$',
  },
  120073: {
    id: 'U1D509',
    decimal: 120073,
    character: '픉',
    latex: '$\\mathfrak{F}$',
  },
  120074: {
    id: 'U1D50A',
    decimal: 120074,
    character: '픊',
    latex: '$\\mathfrak{G}$',
  },
  120077: {
    id: 'U1D50D',
    decimal: 120077,
    character: '픍',
    latex: '$\\mathfrak{J}$',
  },
  120078: {
    id: 'U1D50E',
    decimal: 120078,
    character: '픎',
    latex: '$\\mathfrak{K}$',
  },
  120079: {
    id: 'U1D50F',
    decimal: 120079,
    character: '픏',
    latex: '$\\mathfrak{L}$',
  },
  120080: {
    id: 'U1D510',
    decimal: 120080,
    character: '픐',
    latex: '$\\mathfrak{M}$',
  },
  120081: {
    id: 'U1D511',
    decimal: 120081,
    character: '픑',
    latex: '$\\mathfrak{N}$',
  },
  120082: {
    id: 'U1D512',
    decimal: 120082,
    character: '픒',
    latex: '$\\mathfrak{O}$',
  },
  120083: {
    id: 'U1D513',
    decimal: 120083,
    character: '픓',
    latex: '$\\mathfrak{P}$',
  },
  120084: {
    id: 'U1D514',
    decimal: 120084,
    character: '픔',
    latex: '$\\mathfrak{Q}$',
  },
  120086: {
    id: 'U1D516',
    decimal: 120086,
    character: '픖',
    latex: '$\\mathfrak{S}$',
  },
  120087: {
    id: 'U1D517',
    decimal: 120087,
    character: '픗',
    latex: '$\\mathfrak{T}$',
  },
  120088: {
    id: 'U1D518',
    decimal: 120088,
    character: '픘',
    latex: '$\\mathfrak{U}$',
  },
  120089: {
    id: 'U1D519',
    decimal: 120089,
    character: '픙',
    latex: '$\\mathfrak{V}$',
  },
  120090: {
    id: 'U1D51A',
    decimal: 120090,
    character: '픚',
    latex: '$\\mathfrak{W}$',
  },
  120091: {
    id: 'U1D51B',
    decimal: 120091,
    character: '픛',
    latex: '$\\mathfrak{X}$',
  },
  120092: {
    id: 'U1D51C',
    decimal: 120092,
    character: '픜',
    latex: '$\\mathfrak{Y}$',
  },
  120094: {
    id: 'U1D51E',
    decimal: 120094,
    character: '픞',
    latex: '$\\mathfrak{a}$',
  },
  120095: {
    id: 'U1D51F',
    decimal: 120095,
    character: '픟',
    latex: '$\\mathfrak{b}$',
  },
  120096: {
    id: 'U1D520',
    decimal: 120096,
    character: '픠',
    latex: '$\\mathfrak{c}$',
  },
  120097: {
    id: 'U1D521',
    decimal: 120097,
    character: '픡',
    latex: '$\\mathfrak{d}$',
  },
  120098: {
    id: 'U1D522',
    decimal: 120098,
    character: '픢',
    latex: '$\\mathfrak{e}$',
  },
  120099: {
    id: 'U1D523',
    decimal: 120099,
    character: '픣',
    latex: '$\\mathfrak{f}$',
  },
  120100: {
    id: 'U1D524',
    decimal: 120100,
    character: '픤',
    latex: '$\\mathfrak{g}$',
  },
  120101: {
    id: 'U1D525',
    decimal: 120101,
    character: '픥',
    latex: '$\\mathfrak{h}$',
  },
  120102: {
    id: 'U1D526',
    decimal: 120102,
    character: '픦',
    latex: '$\\mathfrak{i}$',
  },
  120103: {
    id: 'U1D527',
    decimal: 120103,
    character: '픧',
    latex: '$\\mathfrak{j}$',
  },
  120104: {
    id: 'U1D528',
    decimal: 120104,
    character: '픨',
    latex: '$\\mathfrak{k}$',
  },
  120105: {
    id: 'U1D529',
    decimal: 120105,
    character: '픩',
    latex: '$\\mathfrak{l}$',
  },
  120106: {
    id: 'U1D52A',
    decimal: 120106,
    character: '픪',
    latex: '$\\mathfrak{m}$',
  },
  120107: {
    id: 'U1D52B',
    decimal: 120107,
    character: '픫',
    latex: '$\\mathfrak{n}$',
  },
  120108: {
    id: 'U1D52C',
    decimal: 120108,
    character: '픬',
    latex: '$\\mathfrak{o}$',
  },
  120109: {
    id: 'U1D52D',
    decimal: 120109,
    character: '픭',
    latex: '$\\mathfrak{p}$',
  },
  120110: {
    id: 'U1D52E',
    decimal: 120110,
    character: '픮',
    latex: '$\\mathfrak{q}$',
  },
  120111: {
    id: 'U1D52F',
    decimal: 120111,
    character: '픯',
    latex: '$\\mathfrak{r}$',
  },
  120112: {
    id: 'U1D530',
    decimal: 120112,
    character: '픰',
    latex: '$\\mathfrak{s}$',
  },
  120113: {
    id: 'U1D531',
    decimal: 120113,
    character: '픱',
    latex: '$\\mathfrak{t}$',
  },
  120114: {
    id: 'U1D532',
    decimal: 120114,
    character: '픲',
    latex: '$\\mathfrak{u}$',
  },
  120115: {
    id: 'U1D533',
    decimal: 120115,
    character: '픳',
    latex: '$\\mathfrak{v}$',
  },
  120116: {
    id: 'U1D534',
    decimal: 120116,
    character: '픴',
    latex: '$\\mathfrak{w}$',
  },
  120117: {
    id: 'U1D535',
    decimal: 120117,
    character: '픵',
    latex: '$\\mathfrak{x}$',
  },
  120118: {
    id: 'U1D536',
    decimal: 120118,
    character: '픶',
    latex: '$\\mathfrak{y}$',
  },
  120119: {
    id: 'U1D537',
    decimal: 120119,
    character: '픷',
    latex: '$\\mathfrak{z}$',
  },
  120120: {
    id: 'U1D538',
    decimal: 120120,
    character: '픸',
    latex: '$\\mathbb{A}$',
  },
  120121: {
    id: 'U1D539',
    decimal: 120121,
    character: '픹',
    latex: '$\\mathbb{B}$',
  },
  120123: {
    id: 'U1D53B',
    decimal: 120123,
    character: '픻',
    latex: '$\\mathbb{D}$',
  },
  120124: {
    id: 'U1D53C',
    decimal: 120124,
    character: '피',
    latex: '$\\mathbb{E}$',
  },
  120125: {
    id: 'U1D53D',
    decimal: 120125,
    character: '픽',
    latex: '$\\mathbb{F}$',
  },
  120126: {
    id: 'U1D53E',
    decimal: 120126,
    character: '픾',
    latex: '$\\mathbb{G}$',
  },
  120128: {
    id: 'U1D540',
    decimal: 120128,
    character: '핀',
    latex: '$\\mathbb{I}$',
  },
  120129: {
    id: 'U1D541',
    decimal: 120129,
    character: '핁',
    latex: '$\\mathbb{J}$',
  },
  120130: {
    id: 'U1D542',
    decimal: 120130,
    character: '핂',
    latex: '$\\mathbb{K}$',
  },
  120131: {
    id: 'U1D543',
    decimal: 120131,
    character: '핃',
    latex: '$\\mathbb{L}$',
  },
  120132: {
    id: 'U1D544',
    decimal: 120132,
    character: '필',
    latex: '$\\mathbb{M}$',
  },
  120134: {
    id: 'U1D546',
    decimal: 120134,
    character: '핆',
    latex: '$\\mathbb{O}$',
  },
  120138: {
    id: 'U1D54A',
    decimal: 120138,
    character: '핊',
    latex: '$\\mathbb{S}$',
  },
  120139: {
    id: 'U1D54B',
    decimal: 120139,
    character: '핋',
    latex: '$\\mathbb{T}$',
  },
  120140: {
    id: 'U1D54C',
    decimal: 120140,
    character: '핌',
    latex: '$\\mathbb{U}$',
  },
  120141: {
    id: 'U1D54D',
    decimal: 120141,
    character: '핍',
    latex: '$\\mathbb{V}$',
  },
  120142: {
    id: 'U1D54E',
    decimal: 120142,
    character: '핎',
    latex: '$\\mathbb{W}$',
  },
  120143: {
    id: 'U1D54F',
    decimal: 120143,
    character: '핏',
    latex: '$\\mathbb{X}$',
  },
  120144: {
    id: 'U1D550',
    decimal: 120144,
    character: '핐',
    latex: '$\\mathbb{Y}$',
  },
  120146: {
    id: 'U1D552',
    decimal: 120146,
    character: '핒',
    latex: '$\\mathbb{a}$',
  },
  120147: {
    id: 'U1D553',
    decimal: 120147,
    character: '핓',
    latex: '$\\mathbb{b}$',
  },
  120148: {
    id: 'U1D554',
    decimal: 120148,
    character: '핔',
    latex: '$\\mathbb{c}$',
  },
  120149: {
    id: 'U1D555',
    decimal: 120149,
    character: '핕',
    latex: '$\\mathbb{d}$',
  },
  120150: {
    id: 'U1D556',
    decimal: 120150,
    character: '핖',
    latex: '$\\mathbb{e}$',
  },
  120151: {
    id: 'U1D557',
    decimal: 120151,
    character: '핗',
    latex: '$\\mathbb{f}$',
  },
  120152: {
    id: 'U1D558',
    decimal: 120152,
    character: '하',
    latex: '$\\mathbb{g}$',
  },
  120153: {
    id: 'U1D559',
    decimal: 120153,
    character: '학',
    latex: '$\\mathbb{h}$',
  },
  120154: {
    id: 'U1D55A',
    decimal: 120154,
    character: '핚',
    latex: '$\\mathbb{i}$',
  },
  120155: {
    id: 'U1D55B',
    decimal: 120155,
    character: '핛',
    latex: '$\\mathbb{j}$',
  },
  120156: {
    id: 'U1D55C',
    decimal: 120156,
    character: '한',
    latex: '$\\mathbb{k}$',
  },
  120157: {
    id: 'U1D55D',
    decimal: 120157,
    character: '핝',
    latex: '$\\mathbb{l}$',
  },
  120158: {
    id: 'U1D55E',
    decimal: 120158,
    character: '핞',
    latex: '$\\mathbb{m}$',
  },
  120159: {
    id: 'U1D55F',
    decimal: 120159,
    character: '핟',
    latex: '$\\mathbb{n}$',
  },
  120160: {
    id: 'U1D560',
    decimal: 120160,
    character: '할',
    latex: '$\\mathbb{o}$',
  },
  120161: {
    id: 'U1D561',
    decimal: 120161,
    character: '핡',
    latex: '$\\mathbb{p}$',
  },
  120162: {
    id: 'U1D562',
    decimal: 120162,
    character: '핢',
    latex: '$\\mathbb{q}$',
  },
  120163: {
    id: 'U1D563',
    decimal: 120163,
    character: '핣',
    latex: '$\\mathbb{r}$',
  },
  120164: {
    id: 'U1D564',
    decimal: 120164,
    character: '핤',
    latex: '$\\mathbb{s}$',
  },
  120165: {
    id: 'U1D565',
    decimal: 120165,
    character: '핥',
    latex: '$\\mathbb{t}$',
  },
  120166: {
    id: 'U1D566',
    decimal: 120166,
    character: '핦',
    latex: '$\\mathbb{u}$',
  },
  120167: {
    id: 'U1D567',
    decimal: 120167,
    character: '핧',
    latex: '$\\mathbb{v}$',
  },
  120168: {
    id: 'U1D568',
    decimal: 120168,
    character: '함',
    latex: '$\\mathbb{w}$',
  },
  120169: {
    id: 'U1D569',
    decimal: 120169,
    character: '합',
    latex: '$\\mathbb{x}$',
  },
  120170: {
    id: 'U1D56A',
    decimal: 120170,
    character: '핪',
    latex: '$\\mathbb{y}$',
  },
  120171: {
    id: 'U1D56B',
    decimal: 120171,
    character: '핫',
    latex: '$\\mathbb{z}$',
  },
  120224: {
    id: 'U1D5A0',
    decimal: 120224,
    character: '햠',
    latex: '$\\mathsf{A}$',
  },
  120225: {
    id: 'U1D5A1',
    decimal: 120225,
    character: '햡',
    latex: '$\\mathsf{B}$',
  },
  120226: {
    id: 'U1D5A2',
    decimal: 120226,
    character: '햢',
    latex: '$\\mathsf{C}$',
  },
  120227: {
    id: 'U1D5A3',
    decimal: 120227,
    character: '햣',
    latex: '$\\mathsf{D}$',
  },
  120228: {
    id: 'U1D5A4',
    decimal: 120228,
    character: '햤',
    latex: '$\\mathsf{E}$',
  },
  120229: {
    id: 'U1D5A5',
    decimal: 120229,
    character: '향',
    latex: '$\\mathsf{F}$',
  },
  120230: {
    id: 'U1D5A6',
    decimal: 120230,
    character: '햦',
    latex: '$\\mathsf{G}$',
  },
  120231: {
    id: 'U1D5A7',
    decimal: 120231,
    character: '햧',
    latex: '$\\mathsf{H}$',
  },
  120232: {
    id: 'U1D5A8',
    decimal: 120232,
    character: '햨',
    latex: '$\\mathsf{I}$',
  },
  120233: {
    id: 'U1D5A9',
    decimal: 120233,
    character: '햩',
    latex: '$\\mathsf{J}$',
  },
  120234: {
    id: 'U1D5AA',
    decimal: 120234,
    character: '햪',
    latex: '$\\mathsf{K}$',
  },
  120235: {
    id: 'U1D5AB',
    decimal: 120235,
    character: '햫',
    latex: '$\\mathsf{L}$',
  },
  120236: {
    id: 'U1D5AC',
    decimal: 120236,
    character: '햬',
    latex: '$\\mathsf{M}$',
  },
  120237: {
    id: 'U1D5AD',
    decimal: 120237,
    character: '햭',
    latex: '$\\mathsf{N}$',
  },
  120238: {
    id: 'U1D5AE',
    decimal: 120238,
    character: '햮',
    latex: '$\\mathsf{O}$',
  },
  120239: {
    id: 'U1D5AF',
    decimal: 120239,
    character: '햯',
    latex: '$\\mathsf{P}$',
  },
  120240: {
    id: 'U1D5B0',
    decimal: 120240,
    character: '햰',
    latex: '$\\mathsf{Q}$',
  },
  120241: {
    id: 'U1D5B1',
    decimal: 120241,
    character: '햱',
    latex: '$\\mathsf{R}$',
  },
  120242: {
    id: 'U1D5B2',
    decimal: 120242,
    character: '햲',
    latex: '$\\mathsf{S}$',
  },
  120243: {
    id: 'U1D5B3',
    decimal: 120243,
    character: '햳',
    latex: '$\\mathsf{T}$',
  },
  120244: {
    id: 'U1D5B4',
    decimal: 120244,
    character: '햴',
    latex: '$\\mathsf{U}$',
  },
  120245: {
    id: 'U1D5B5',
    decimal: 120245,
    character: '햵',
    latex: '$\\mathsf{V}$',
  },
  120246: {
    id: 'U1D5B6',
    decimal: 120246,
    character: '햶',
    latex: '$\\mathsf{W}$',
  },
  120247: {
    id: 'U1D5B7',
    decimal: 120247,
    character: '햷',
    latex: '$\\mathsf{X}$',
  },
  120248: {
    id: 'U1D5B8',
    decimal: 120248,
    character: '햸',
    latex: '$\\mathsf{Y}$',
  },
  120249: {
    id: 'U1D5B9',
    decimal: 120249,
    character: '햹',
    latex: '$\\mathsf{Z}$',
  },
  120250: {
    id: 'U1D5BA',
    decimal: 120250,
    character: '햺',
    latex: '$\\mathsf{a}$',
  },
  120251: {
    id: 'U1D5BB',
    decimal: 120251,
    character: '햻',
    latex: '$\\mathsf{b}$',
  },
  120252: {
    id: 'U1D5BC',
    decimal: 120252,
    character: '햼',
    latex: '$\\mathsf{c}$',
  },
  120253: {
    id: 'U1D5BD',
    decimal: 120253,
    character: '햽',
    latex: '$\\mathsf{d}$',
  },
  120254: {
    id: 'U1D5BE',
    decimal: 120254,
    character: '햾',
    latex: '$\\mathsf{e}$',
  },
  120255: {
    id: 'U1D5BF',
    decimal: 120255,
    character: '햿',
    latex: '$\\mathsf{f}$',
  },
  120256: {
    id: 'U1D5C0',
    decimal: 120256,
    character: '헀',
    latex: '$\\mathsf{g}$',
  },
  120257: {
    id: 'U1D5C1',
    decimal: 120257,
    character: '헁',
    latex: '$\\mathsf{h}$',
  },
  120258: {
    id: 'U1D5C2',
    decimal: 120258,
    character: '헂',
    latex: '$\\mathsf{i}$',
  },
  120259: {
    id: 'U1D5C3',
    decimal: 120259,
    character: '헃',
    latex: '$\\mathsf{j}$',
  },
  120260: {
    id: 'U1D5C4',
    decimal: 120260,
    character: '헄',
    latex: '$\\mathsf{k}$',
  },
  120261: {
    id: 'U1D5C5',
    decimal: 120261,
    character: '헅',
    latex: '$\\mathsf{l}$',
  },
  120262: {
    id: 'U1D5C6',
    decimal: 120262,
    character: '헆',
    latex: '$\\mathsf{m}$',
  },
  120263: {
    id: 'U1D5C7',
    decimal: 120263,
    character: '헇',
    latex: '$\\mathsf{n}$',
  },
  120264: {
    id: 'U1D5C8',
    decimal: 120264,
    character: '허',
    latex: '$\\mathsf{o}$',
  },
  120265: {
    id: 'U1D5C9',
    decimal: 120265,
    character: '헉',
    latex: '$\\mathsf{p}$',
  },
  120266: {
    id: 'U1D5CA',
    decimal: 120266,
    character: '헊',
    latex: '$\\mathsf{q}$',
  },
  120267: {
    id: 'U1D5CB',
    decimal: 120267,
    character: '헋',
    latex: '$\\mathsf{r}$',
  },
  120268: {
    id: 'U1D5CC',
    decimal: 120268,
    character: '헌',
    latex: '$\\mathsf{s}$',
  },
  120269: {
    id: 'U1D5CD',
    decimal: 120269,
    character: '헍',
    latex: '$\\mathsf{t}$',
  },
  120270: {
    id: 'U1D5CE',
    decimal: 120270,
    character: '헎',
    latex: '$\\mathsf{u}$',
  },
  120271: {
    id: 'U1D5CF',
    decimal: 120271,
    character: '헏',
    latex: '$\\mathsf{v}$',
  },
  120272: {
    id: 'U1D5D0',
    decimal: 120272,
    character: '헐',
    latex: '$\\mathsf{w}$',
  },
  120273: {
    id: 'U1D5D1',
    decimal: 120273,
    character: '헑',
    latex: '$\\mathsf{x}$',
  },
  120274: {
    id: 'U1D5D2',
    decimal: 120274,
    character: '헒',
    latex: '$\\mathsf{y}$',
  },
  120275: {
    id: 'U1D5D3',
    decimal: 120275,
    character: '헓',
    latex: '$\\mathsf{z}$',
  },
  120432: {
    id: 'U1D670',
    decimal: 120432,
    character: '홰',
    latex: '$\\mathtt{A}$',
  },
  120433: {
    id: 'U1D671',
    decimal: 120433,
    character: '홱',
    latex: '$\\mathtt{B}$',
  },
  120434: {
    id: 'U1D672',
    decimal: 120434,
    character: '홲',
    latex: '$\\mathtt{C}$',
  },
  120435: {
    id: 'U1D673',
    decimal: 120435,
    character: '홳',
    latex: '$\\mathtt{D}$',
  },
  120436: {
    id: 'U1D674',
    decimal: 120436,
    character: '홴',
    latex: '$\\mathtt{E}$',
  },
  120437: {
    id: 'U1D675',
    decimal: 120437,
    character: '홵',
    latex: '$\\mathtt{F}$',
  },
  120438: {
    id: 'U1D676',
    decimal: 120438,
    character: '홶',
    latex: '$\\mathtt{G}$',
  },
  120439: {
    id: 'U1D677',
    decimal: 120439,
    character: '홷',
    latex: '$\\mathtt{H}$',
  },
  120440: {
    id: 'U1D678',
    decimal: 120440,
    character: '홸',
    latex: '$\\mathtt{I}$',
  },
  120441: {
    id: 'U1D679',
    decimal: 120441,
    character: '홹',
    latex: '$\\mathtt{J}$',
  },
  120442: {
    id: 'U1D67A',
    decimal: 120442,
    character: '홺',
    latex: '$\\mathtt{K}$',
  },
  120443: {
    id: 'U1D67B',
    decimal: 120443,
    character: '홻',
    latex: '$\\mathtt{L}$',
  },
  120444: {
    id: 'U1D67C',
    decimal: 120444,
    character: '홼',
    latex: '$\\mathtt{M}$',
  },
  120445: {
    id: 'U1D67D',
    decimal: 120445,
    character: '홽',
    latex: '$\\mathtt{N}$',
  },
  120446: {
    id: 'U1D67E',
    decimal: 120446,
    character: '홾',
    latex: '$\\mathtt{O}$',
  },
  120447: {
    id: 'U1D67F',
    decimal: 120447,
    character: '홿',
    latex: '$\\mathtt{P}$',
  },
  120448: {
    id: 'U1D680',
    decimal: 120448,
    character: '횀',
    latex: '$\\mathtt{Q}$',
  },
  120449: {
    id: 'U1D681',
    decimal: 120449,
    character: '횁',
    latex: '$\\mathtt{R}$',
  },
  120450: {
    id: 'U1D682',
    decimal: 120450,
    character: '횂',
    latex: '$\\mathtt{S}$',
  },
  120451: {
    id: 'U1D683',
    decimal: 120451,
    character: '횃',
    latex: '$\\mathtt{T}$',
  },
  120452: {
    id: 'U1D684',
    decimal: 120452,
    character: '횄',
    latex: '$\\mathtt{U}$',
  },
  120453: {
    id: 'U1D685',
    decimal: 120453,
    character: '횅',
    latex: '$\\mathtt{V}$',
  },
  120454: {
    id: 'U1D686',
    decimal: 120454,
    character: '횆',
    latex: '$\\mathtt{W}$',
  },
  120455: {
    id: 'U1D687',
    decimal: 120455,
    character: '횇',
    latex: '$\\mathtt{X}$',
  },
  120456: {
    id: 'U1D688',
    decimal: 120456,
    character: '횈',
    latex: '$\\mathtt{Y}$',
  },
  120457: {
    id: 'U1D689',
    decimal: 120457,
    character: '횉',
    latex: '$\\mathtt{Z}$',
  },
  120458: {
    id: 'U1D68A',
    decimal: 120458,
    character: '횊',
    latex: '$\\mathtt{a}$',
  },
  120459: {
    id: 'U1D68B',
    decimal: 120459,
    character: '횋',
    latex: '$\\mathtt{b}$',
  },
  120460: {
    id: 'U1D68C',
    decimal: 120460,
    character: '회',
    latex: '$\\mathtt{c}$',
  },
  120461: {
    id: 'U1D68D',
    decimal: 120461,
    character: '획',
    latex: '$\\mathtt{d}$',
  },
  120462: {
    id: 'U1D68E',
    decimal: 120462,
    character: '횎',
    latex: '$\\mathtt{e}$',
  },
  120463: {
    id: 'U1D68F',
    decimal: 120463,
    character: '횏',
    latex: '$\\mathtt{f}$',
  },
  120464: {
    id: 'U1D690',
    decimal: 120464,
    character: '횐',
    latex: '$\\mathtt{g}$',
  },
  120465: {
    id: 'U1D691',
    decimal: 120465,
    character: '횑',
    latex: '$\\mathtt{h}$',
  },
  120466: {
    id: 'U1D692',
    decimal: 120466,
    character: '횒',
    latex: '$\\mathtt{i}$',
  },
  120467: {
    id: 'U1D693',
    decimal: 120467,
    character: '횓',
    latex: '$\\mathtt{j}$',
  },
  120468: {
    id: 'U1D694',
    decimal: 120468,
    character: '횔',
    latex: '$\\mathtt{k}$',
  },
  120469: {
    id: 'U1D695',
    decimal: 120469,
    character: '횕',
    latex: '$\\mathtt{l}$',
  },
  120470: {
    id: 'U1D696',
    decimal: 120470,
    character: '횖',
    latex: '$\\mathtt{m}$',
  },
  120471: {
    id: 'U1D697',
    decimal: 120471,
    character: '횗',
    latex: '$\\mathtt{n}$',
  },
  120472: {
    id: 'U1D698',
    decimal: 120472,
    character: '횘',
    latex: '$\\mathtt{o}$',
  },
  120473: {
    id: 'U1D699',
    decimal: 120473,
    character: '횙',
    latex: '$\\mathtt{p}$',
  },
  120474: {
    id: 'U1D69A',
    decimal: 120474,
    character: '횚',
    latex: '$\\mathtt{q}$',
  },
  120475: {
    id: 'U1D69B',
    decimal: 120475,
    character: '횛',
    latex: '$\\mathtt{r}$',
  },
  120476: {
    id: 'U1D69C',
    decimal: 120476,
    character: '횜',
    latex: '$\\mathtt{s}$',
  },
  120477: {
    id: 'U1D69D',
    decimal: 120477,
    character: '횝',
    latex: '$\\mathtt{t}$',
  },
  120478: {
    id: 'U1D69E',
    decimal: 120478,
    character: '횞',
    latex: '$\\mathtt{u}$',
  },
  120479: {
    id: 'U1D69F',
    decimal: 120479,
    character: '횟',
    latex: '$\\mathtt{v}$',
  },
  120480: {
    id: 'U1D6A0',
    decimal: 120480,
    character: '횠',
    latex: '$\\mathtt{w}$',
  },
  120481: {
    id: 'U1D6A1',
    decimal: 120481,
    character: '횡',
    latex: '$\\mathtt{x}$',
  },
  120482: {
    id: 'U1D6A2',
    decimal: 120482,
    character: '횢',
    latex: '$\\mathtt{y}$',
  },
  120483: {
    id: 'U1D6A3',
    decimal: 120483,
    character: '횣',
    latex: '$\\mathtt{z}$',
  },
  120490: {
    id: 'U1D6AA',
    decimal: 120490,
    character: '횪',
    latex: '$\\mathbf{\\Gamma}$',
  },
  120491: {
    id: 'U1D6AB',
    decimal: 120491,
    character: '횫',
    latex: '$\\mathbf{\\Delta}$',
  },
  120495: {
    id: 'U1D6AF',
    decimal: 120495,
    character: '횯',
    latex: '$\\mathbf{\\Theta}$',
  },
  120498: {
    id: 'U1D6B2',
    decimal: 120498,
    character: '횲',
    latex: '$\\mathbf{\\Lambda}$',
  },
  120499: {
    id: 'U1D6B3',
    decimal: 120499,
    character: '횳',
    latex: '$\\mathbf{M}$',
  },
  120500: {
    id: 'U1D6B4',
    decimal: 120500,
    character: '횴',
    latex: 'N',
  },
  120501: {
    id: 'U1D6B5',
    decimal: 120501,
    character: '횵',
    latex: '$\\mathbf{\\Xi}$',
  },
  120502: {
    id: 'U1D6B6',
    decimal: 120502,
    character: '횶',
    latex: 'O',
  },
  120503: {
    id: 'U1D6B7',
    decimal: 120503,
    character: '횷',
    latex: '$\\mathbf{\\Pi}$',
  },
  120505: {
    id: 'U1D6B9',
    decimal: 120505,
    character: '횹',
    latex: '$\\mathbf{\\vartheta}$',
  },
  120506: {
    id: 'U1D6BA',
    decimal: 120506,
    character: '횺',
    latex: '$\\mathbf{\\Sigma}$',
  },
  120508: {
    id: 'U1D6BC',
    decimal: 120508,
    character: '횼',
    latex: '$\\mathbf{\\Upsilon}$',
  },
  120509: {
    id: 'U1D6BD',
    decimal: 120509,
    character: '횽',
    latex: '$\\mathbf{\\Phi}$',
  },
  120511: {
    id: 'U1D6BF',
    decimal: 120511,
    character: '횿',
    latex: '$\\mathbf{\\Psi}$',
  },
  120512: {
    id: 'U1D6C0',
    decimal: 120512,
    character: '훀',
    latex: '$\\mathbf{\\Omega}$',
  },
  120513: {
    id: 'U1D6C1',
    decimal: 120513,
    character: '훁',
    latex: '$\\mathbf{\\nabla}$',
  },
  120514: {
    id: 'U1D6C2',
    decimal: 120514,
    character: '훂',
    latex: '$\\mathbf{\\alpha}$',
  },
  120515: {
    id: 'U1D6C3',
    decimal: 120515,
    character: '훃',
    latex: '$\\mathbf{\\beta}$',
  },
  120516: {
    id: 'U1D6C4',
    decimal: 120516,
    character: '후',
    latex: '$\\mathbf{\\gamma}$',
  },
  120517: {
    id: 'U1D6C5',
    decimal: 120517,
    character: '훅',
    latex: '$\\mathbf{\\delta}$',
  },
  120518: {
    id: 'U1D6C6',
    decimal: 120518,
    character: '훆',
    latex: '$\\mathbf{\\epsilon}$',
  },
  120519: {
    id: 'U1D6C7',
    decimal: 120519,
    character: '훇',
    latex: '$\\mathbf{\\zeta}$',
  },
  120520: {
    id: 'U1D6C8',
    decimal: 120520,
    character: '훈',
    latex: '$\\mathbf{\\eta}$',
  },
  120521: {
    id: 'U1D6C9',
    decimal: 120521,
    character: '훉',
    latex: '$\\mathbf{\\theta}$',
  },
  120522: {
    id: 'U1D6CA',
    decimal: 120522,
    character: '훊',
    latex: '$\\mathbf{\\iota}$',
  },
  120523: {
    id: 'U1D6CB',
    decimal: 120523,
    character: '훋',
    latex: '$\\mathbf{\\kappa}$',
  },
  120524: {
    id: 'U1D6CC',
    decimal: 120524,
    character: '훌',
    latex: '$\\mathbf{\\lambda}$',
  },
  120525: {
    id: 'U1D6CD',
    decimal: 120525,
    character: '훍',
    latex: '$\\mathbf{\\mu}$',
  },
  120526: {
    id: 'U1D6CE',
    decimal: 120526,
    character: '훎',
    latex: '$\\mathbf{\\nu}$',
  },
  120527: {
    id: 'U1D6CF',
    decimal: 120527,
    character: '훏',
    latex: '$\\mathbf{\\xi}$',
  },
  120528: {
    id: 'U1D6D0',
    decimal: 120528,
    character: '훐',
    latex: '$\\mathbf{o}$',
  },
  120529: {
    id: 'U1D6D1',
    decimal: 120529,
    character: '훑',
    latex: '$\\mathbf{\\pi}$',
  },
  120530: {
    id: 'U1D6D2',
    decimal: 120530,
    character: '훒',
    latex: '$\\mathbf{\\rho}$',
  },
  120531: {
    id: 'U1D6D3',
    decimal: 120531,
    character: '훓',
    latex: '$\\mathbf{\\varsigma}$',
  },
  120532: {
    id: 'U1D6D4',
    decimal: 120532,
    character: '훔',
    latex: '$\\mathbf{\\sigma}$',
  },
  120533: {
    id: 'U1D6D5',
    decimal: 120533,
    character: '훕',
    latex: '$\\mathbf{\\tau}$',
  },
  120534: {
    id: 'U1D6D6',
    decimal: 120534,
    character: '훖',
    latex: '$\\mathbf{\\upsilon}$',
  },
  120535: {
    id: 'U1D6D7',
    decimal: 120535,
    character: '훗',
    latex: '$\\mathbf{\\phi}$',
  },
  120536: {
    id: 'U1D6D8',
    decimal: 120536,
    character: '훘',
    latex: '$\\mathbf{\\chi}$',
  },
  120537: {
    id: 'U1D6D9',
    decimal: 120537,
    character: '훙',
    latex: '$\\mathbf{\\psi}$',
  },
  120538: {
    id: 'U1D6DA',
    decimal: 120538,
    character: '훚',
    latex: '$\\mathbf{\\omega}$',
  },
  120540: {
    id: 'U1D6DC',
    decimal: 120540,
    character: '훜',
    latex: '$\\mathbf{\\varepsilon}$',
  },
  120541: {
    id: 'U1D6DD',
    decimal: 120541,
    character: '훝',
    latex: '$\\mathbf{\\vartheta}$',
  },
  120542: {
    id: 'U1D6DE',
    decimal: 120542,
    character: '훞',
    latex: '$\\mathbf{\\varkappa}$',
  },
  120543: {
    id: 'U1D6DF',
    decimal: 120543,
    character: '훟',
    latex: '$\\mathbf{\\phi}$',
  },
  120544: {
    id: 'U1D6E0',
    decimal: 120544,
    character: '훠',
    latex: '$\\mathbf{\\varrho}$',
  },
  120545: {
    id: 'U1D6E1',
    decimal: 120545,
    character: '훡',
    latex: '$\\mathbf{\\varpi}$',
  },
  120558: {
    id: 'U1D6EE',
    decimal: 120558,
    character: '훮',
    latex: 'N',
  },
  120560: {
    id: 'U1D6F0',
    decimal: 120560,
    character: '훰',
    latex: 'O',
  },
  120618: {
    id: 'U1D72A',
    decimal: 120618,
    character: '휪',
    latex: 'O',
  },
  120676: {
    id: 'U1D764',
    decimal: 120676,
    character: '흤',
    latex: 'O',
  },
  120734: {
    id: 'U1D79E',
    decimal: 120734,
    character: '힞',
    latex: 'O',
  },
  120782: {
    id: 'U1D7CE',
    decimal: 120782,
    character: 'ퟎ',
    latex: '$\\mathbf{0}$',
  },
  120783: {
    id: 'U1D7CF',
    decimal: 120783,
    character: 'ퟏ',
    latex: '$\\mathbf{1}$',
  },
  120784: {
    id: 'U1D7D0',
    decimal: 120784,
    character: 'ퟐ',
    latex: '$\\mathbf{2}$',
  },
  120785: {
    id: 'U1D7D1',
    decimal: 120785,
    character: 'ퟑ',
    latex: '$\\mathbf{3}$',
  },
  120786: {
    id: 'U1D7D2',
    decimal: 120786,
    character: 'ퟒ',
    latex: '$\\mathbf{4}$',
  },
  120787: {
    id: 'U1D7D3',
    decimal: 120787,
    character: 'ퟓ',
    latex: '$\\mathbf{5}$',
  },
  120788: {
    id: 'U1D7D4',
    decimal: 120788,
    character: 'ퟔ',
    latex: '$\\mathbf{6}$',
  },
  120789: {
    id: 'U1D7D5',
    decimal: 120789,
    character: 'ퟕ',
    latex: '$\\mathbf{7}$',
  },
  120790: {
    id: 'U1D7D6',
    decimal: 120790,
    character: 'ퟖ',
    latex: '$\\mathbf{8}$',
  },
  120791: {
    id: 'U1D7D7',
    decimal: 120791,
    character: 'ퟗ',
    latex: '$\\mathbf{9}$',
  },
  120792: {
    id: 'U1D7D8',
    decimal: 120792,
    character: 'ퟘ',
    latex: '$\\mathbb{0}$',
  },
  120793: {
    id: 'U1D7D9',
    decimal: 120793,
    character: 'ퟙ',
    latex: '$\\mathbb{1}$',
  },
  120794: {
    id: 'U1D7DA',
    decimal: 120794,
    character: 'ퟚ',
    latex: '$\\mathbb{2}$',
  },
  120795: {
    id: 'U1D7DB',
    decimal: 120795,
    character: 'ퟛ',
    latex: '$\\mathbb{3}$',
  },
  120796: {
    id: 'U1D7DC',
    decimal: 120796,
    character: 'ퟜ',
    latex: '$\\mathbb{4}$',
  },
  120797: {
    id: 'U1D7DD',
    decimal: 120797,
    character: 'ퟝ',
    latex: '$\\mathbb{5}$',
  },
  120798: {
    id: 'U1D7DE',
    decimal: 120798,
    character: 'ퟞ',
    latex: '$\\mathbb{6}$',
  },
  120799: {
    id: 'U1D7DF',
    decimal: 120799,
    character: 'ퟟ',
    latex: '$\\mathbb{7}$',
  },
  120800: {
    id: 'U1D7E0',
    decimal: 120800,
    character: 'ퟠ',
    latex: '$\\mathbb{8}$',
  },
  120801: {
    id: 'U1D7E1',
    decimal: 120801,
    character: 'ퟡ',
    latex: '$\\mathbb{9}$',
  },
  120802: {
    id: 'U1D7E2',
    decimal: 120802,
    character: 'ퟢ',
    latex: '$\\mathsf{0}$',
  },
  120803: {
    id: 'U1D7E3',
    decimal: 120803,
    character: 'ퟣ',
    latex: '$\\mathsf{1}$',
  },
  120804: {
    id: 'U1D7E4',
    decimal: 120804,
    character: 'ퟤ',
    latex: '$\\mathsf{2}$',
  },
  120805: {
    id: 'U1D7E5',
    decimal: 120805,
    character: 'ퟥ',
    latex: '$\\mathsf{3}$',
  },
  120806: {
    id: 'U1D7E6',
    decimal: 120806,
    character: 'ퟦ',
    latex: '$\\mathsf{4}$',
  },
  120807: {
    id: 'U1D7E7',
    decimal: 120807,
    character: 'ퟧ',
    latex: '$\\mathsf{5}$',
  },
  120808: {
    id: 'U1D7E8',
    decimal: 120808,
    character: 'ퟨ',
    latex: '$\\mathsf{6}$',
  },
  120809: {
    id: 'U1D7E9',
    decimal: 120809,
    character: 'ퟩ',
    latex: '$\\mathsf{7}$',
  },
  120810: {
    id: 'U1D7EA',
    decimal: 120810,
    character: 'ퟪ',
    latex: '$\\mathsf{8}$',
  },
  120811: {
    id: 'U1D7EB',
    decimal: 120811,
    character: 'ퟫ',
    latex: '$\\mathsf{9}$',
  },
  120822: {
    id: 'U1D7F6',
    decimal: 120822,
    character: 'ퟶ',
    latex: '$\\mathtt{0}$',
  },
  120823: {
    id: 'U1D7F7',
    decimal: 120823,
    character: 'ퟷ',
    latex: '$\\mathtt{1}$',
  },
  120824: {
    id: 'U1D7F8',
    decimal: 120824,
    character: 'ퟸ',
    latex: '$\\mathtt{2}$',
  },
  120825: {
    id: 'U1D7F9',
    decimal: 120825,
    character: 'ퟹ',
    latex: '$\\mathtt{3}$',
  },
  120826: {
    id: 'U1D7FA',
    decimal: 120826,
    character: 'ퟺ',
    latex: '$\\mathtt{4}$',
  },
  120827: {
    id: 'U1D7FB',
    decimal: 120827,
    character: 'ퟻ',
    latex: '$\\mathtt{5}$',
  },
  120828: {
    id: 'U1D7FC',
    decimal: 120828,
    character: '퟼',
    latex: '$\\mathtt{6}$',
  },
  120829: {
    id: 'U1D7FD',
    decimal: 120829,
    character: '퟽',
    latex: '$\\mathtt{7}$',
  },
  120830: {
    id: 'U1D7FE',
    decimal: 120830,
    character: '퟾',
    latex: '$\\mathtt{8}$',
  },
  120831: {
    id: 'U1D7FF',
    decimal: 120831,
    character: '퟿',
    latex: '$\\mathtt{9}$',
  },
};

export default characters;
