package com.syntevo.plugin.jira.commit.messagesource;

import java.util.*;
import java.util.List;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.plugin.jira.transport.*;

/**
 * @author syntevo GmbH
 */
final class JiraCommitMessageSourceDialog extends AbstractOkCancelDialog {

	// Fields =================================================================

	private final JiraIssueLoader issueLoader;
	private final JiraQueryConfiguration queryConfiguration;

	private List<JiraIssue> issues;
	private Table table;
	private String commitMessage;

	// Setup ==================================================================

	public JiraCommitMessageSourceDialog(@NotNull List<JiraIssue> issues, JiraIssueLoader issueLoader, JiraQueryConfiguration queryConfiguration) {
		this.issues = issues;
		this.issueLoader = issueLoader;

		this.queryConfiguration = queryConfiguration;

		getOkAction().setName("Select");
	}

	// Implemented ============================================================

	@NotNull
	@Override
	public String getTitle() {
		return "Select JIRA Issue";
	}

	@NotNull
	@Override
	public Control createComponent(@NotNull Composite parent) {
		final Composite panel = new Composite(parent, SWT.NONE) {
			@Override
			public Point computeSize(int wHint, int hHint, boolean changed) {
				return new Point(600, 400);
			}
		};
		panel.setLayout(new FillLayout());

		table = new Table(panel, SWT.BORDER | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI);
		table.setHeaderVisible(true);

		new TableColumn(table, SWT.RIGHT).setText("Key");
		new TableColumn(table, SWT.NONE).setText("Summary");
		new TableColumn(table, SWT.NONE).setText("Status");

		table.addListener(SWT.DefaultSelection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				performOk();
			}
		});

		updateTableFromIssues();
		pack();
		return panel;
	}

	@Override
	protected void ok() throws DialogValidationFailedException {
		final int[] selection = table.getSelectionIndices();
		if (selection.length == 0) {
			throw new DialogValidationFailedException(table, "Please select one or more issue.", "Or press Cancel to abort.");
		}

		final List<JiraIssue> selectedIssues = new ArrayList<>();
		for (int row : selection) {
			selectedIssues.add(issues.get(row));
		}

		commitMessage = createCommitMessage(selectedIssues);
		super.ok();
	}

	@NotNull
	@Override
	public Button initButtonPanel(@NotNull ButtonPanel buttonPanel) {
		buttonPanel.addButton(new ButtonAction("Refresh", BUTTON_TYPE_OTHER) {
			@Override
			public void performAction() {
				issueLoader.load(queryConfiguration, true);
			}
		});
		return super.initButtonPanel(buttonPanel);
	}

	// Accessing ==============================================================

	public String getCommitMessage() {
		return commitMessage;
	}

	public void setIssues(@NotNull List<JiraIssue> issues) {
		final boolean firstNonEmptySet = this.issues.size() > 0 && issues.size() > 0;

		final Set<String> selectedKeys = new HashSet<>();
		for (int row : table.getSelectionIndices()) {
			selectedKeys.add(this.issues.get(row).getKey());
		}

		this.issues = issues;

		updateTableFromIssues();
		if (firstNonEmptySet) {
			pack();
		}

		if (selectedKeys.size() > 0) {
			final List<Integer> indicesToSelect = new ArrayList<>();
			int i = 0;
			for (JiraIssue issue : issues) {
				if (selectedKeys.contains(issue.getKey())) {
					indicesToSelect.add(i);
				}
				i++;
			}

			final int[] indicesToSelectArray = new int[indicesToSelect.size()];
			for (i = 0; i < indicesToSelectArray.length; i++) {
				indicesToSelectArray[i] = indicesToSelect.get(i).intValue();
			}

			table.setSelection(indicesToSelectArray);
		}
	}

	// Utils ==================================================================

	private void updateTableFromIssues() {
		table.setRedraw(false);
		try {
			table.setItemCount(issues.size());

			final TableItem[] items = table.getItems();
			for (int i = 0; i < items.length; i++) {
				final TableItem item = items[i];
				final JiraIssue issue = issues.get(i);
				item.setText(0, issue.getKey());
				item.setText(1, issue.getSummary());
				item.setText(2, issue.getStatus().getName());
			}
		}
		finally {
			table.setRedraw(true);
		}
	}

	private void pack() {
		for (TableColumn column : table.getColumns()) {
			column.pack();
		}
	}

	private static String createCommitMessage(List<JiraIssue> issues) {
		final StringBuilder message = new StringBuilder();
		for (final Iterator it = issues.iterator(); it.hasNext(); ) {
			final JiraIssue issue = (JiraIssue)it.next();
			message.append(issue.getKey());
			message.append(": ");
			message.append(issue.getSummary());
			if (it.hasNext()) {
				message.append("\n");
			}
		}

		return message.toString();
	}
}