package com.syntevo.plugin.jira.commit.messagesource;

import java.io.*;
import java.util.*;
import javax.xml.soap.*;

import com.syntevo.plugin.jira.*;
import com.syntevo.plugin.jira.transport.*;

/**
 * @author syntevo GmbH
 */
final class JiraIssueLoader extends Thread {

	// Constants ==============================================================

	private static final int UNRELEASED_VERSIONS_TO_DISPLAY;

	// Static =================================================================

	static {
		int unreleasedVersions = 3;
		try {
			unreleasedVersions = Integer.parseInt(System.getProperty("smartsvn.plugin.jira.unreleasedVersionsToDisplay", System.getProperty("smartsvn.plugin.jira.unreleased-versions-to-display", String.valueOf(unreleasedVersions))));
		}
		catch (NumberFormatException ex) {
			// Ignore
		}
		UNRELEASED_VERSIONS_TO_DISPLAY = unreleasedVersions;
	}

	// Fields =================================================================

	private final JiraIssueList issueList = new JiraIssueList();
	private final JiraIssueTable table;

	private JiraQueryConfiguration configuration;
	private boolean forceRefresh;
	private boolean aborted;
	private JiraIssueLoaderTableUpdater tableUpdater;

	// Setup ==================================================================

	public JiraIssueLoader(JiraIssueTable table) {
		this.table = table;
	}

	// Implemented ============================================================

	@Override
	public void run() {
		JiraQueryConfiguration configuration = null;
		JiraSoapClient soapClient;
		for (; ; ) {
			try {
				synchronized (this) {
					while (this.configuration == null && !aborted) {
						try {
							wait();
						}
						catch (InterruptedException ex) {
							// Ignore
						}
					}

					if (aborted) {
						return;
					}

					if (areConfigurationsEqual(configuration, this.configuration) && !forceRefresh) {
						this.configuration = null;
						continue;
					}

					configuration = this.configuration;
					forceRefresh = false;

					issueList.clear();

					if (configuration == null) {
						continue;
					}

					soapClient = new JiraSoapClient(configuration.getConnection());

					final JiraQueryClient queryClient = new JiraQueryClient(configuration.getConnection());
					tableUpdater = new JiraIssueLoaderTableUpdater(queryClient, issueList, table);
				}

				try {
					try {
						if (!soapClient.login()) {
							synchronized (this) {
								this.configuration = configuration = null;
							}
							continue;
						}
					}
					catch (SOAPException ex) {
						JiraPlugin.LOGGER.error(ex.getMessage(), ex);
						configuration.getConnection().showError("Connection failed.", ex);
						synchronized (this) {
							this.configuration = configuration = null;
						}
						continue;
					}

					final String projectId;
					final String username;
					final List<JiraVersion> unreleasedVersions;
					try {
						final List<JiraVersion> unreleasedVersionsByProjectKeyInOrder = soapClient.getUnreleasedVersionsByProjectKeyInOrder(configuration.getProjectKey());
						synchronized (this) {
							unreleasedVersions = unreleasedVersionsByProjectKeyInOrder;
						}

						if (configuration.isLoadAll()) {
							projectId = soapClient.getProjectIDByKey(configuration.getProjectKey());
							tableUpdater.load(projectId, null, Collections.<JiraIssueStatus>emptyList(), Collections.<String>emptyList(), 2);
							continue;
						}

						projectId = soapClient.getProjectIDByKey(configuration.getProjectKey());
						username = soapClient.getLoggedInUsername();
						tableUpdater.load(projectId, username, Collections.singletonList(JiraIssueStatus.IN_PROGRESS), Collections.<String>emptyList(), 0);
					}
					finally {
						soapClient.logout();
					}

					if (unreleasedVersions.isEmpty() || UNRELEASED_VERSIONS_TO_DISPLAY == 0) {
						tableUpdater.load(projectId, username, Arrays.asList(JiraIssueStatus.OPEN, JiraIssueStatus.REOPENED), Collections.<String>emptyList(), 1);
					}
					else {
						final List<String> firstUnreleasedVersions = new ArrayList<>();
						for (int index = 0; index < Math.min(UNRELEASED_VERSIONS_TO_DISPLAY, unreleasedVersions.size()); index++) {
							firstUnreleasedVersions.add(unreleasedVersions.get(index).getId());
						}

						tableUpdater.load(projectId, username, Arrays.asList(JiraIssueStatus.OPEN, JiraIssueStatus.REOPENED), firstUnreleasedVersions, 1);
					}
				}
				catch (SOAPException | IOException ex) {
					JiraPlugin.LOGGER.error(ex.getMessage(), ex);
					configuration.getConnection().showError("Connection failed.", ex);
				}

				synchronized (this) {
					this.configuration = null;
				}
			}
			catch (Throwable ex) {
				JiraPlugin.LOGGER.error(ex.getMessage(), ex);
			}
		}
	}

	// Accessing ==============================================================

	public synchronized void load(JiraQueryConfiguration configuration, boolean forceRefresh) {
		this.configuration = configuration;
		this.forceRefresh = forceRefresh;
		notifyAll();
	}

	public synchronized void abortAndCleanup() {
		aborted = true;
		if (tableUpdater != null) {
			tableUpdater.abort();
		}
		notifyAll();
	}

	// Utils ==================================================================

	private static boolean areConfigurationsEqual(JiraQueryConfiguration configuration1, JiraQueryConfiguration configuration2) {
		if (configuration1 == null) {
			return configuration2 == null;
		}

		if (configuration2 == null) {
			return false;
		}

		return configuration1.getConnection().getUrl().equals(configuration2.getConnection().getUrl()) && configuration1.getProjectKey().equals(configuration2.getProjectKey());
	}
}
