package com.syntevo.plugin.jira.transport;

import java.util.*;

import org.xml.sax.*;
import org.xml.sax.helpers.*;

/**
 * @author syntevo GmbH
 */
final class JiraIssueXmlHandler extends DefaultHandler {

	// Fields =================================================================

	private final List<String> currentElements = new ArrayList<>();
	private final StringBuffer characters = new StringBuffer();
	private final List<JiraIssue> issues = new ArrayList<>();

	private JiraIssue currentIssue;
	private Attributes currentAttributes;

	// Implemented ============================================================

	@Override
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
		final boolean inIssueBefore = isInIssue();
		currentElements.add(qName);
		final boolean inIssueAfter = isInIssue();

		characters.setLength(0);
		currentAttributes = new AttributesImpl(attributes);

		if (!inIssueBefore && inIssueAfter) {
			currentIssue = new JiraIssue();
		}
	}

	@Override
	public void endElement(String uri, String localName, String qName) throws SAXException {
		if (currentElements.size() == 0) {
			throw new SAXException("Unexpected endElement.");
		}

		final String lastElement = currentElements.get(currentElements.size() - 1);
		if (!areEqual(qName, lastElement)) {
			throw new SAXException("Unexpected endElement encountered, expected " + currentElements + ", actual '" + qName + "'.");
		}

		currentElements.remove(currentElements.size() - 1);

		if (currentIssue == null) {
			return;
		}

		if (JiraIssue.FIELD_KEY.equals(qName)) {
			currentIssue.setKey(getAttributeExisting("id"), getCharactersTrimmed());
		}
		else if (JiraIssue.FIELD_SUMMARY.equals(qName)) {
			currentIssue.setSummary(getCharactersTrimmed());
		}
		else if (JiraIssue.FIELD_ASSIGNEE.equals(qName)) {
			currentIssue.setAssignee(getAttributeExisting("username"));
		}
		else if (JiraIssue.FIELD_TYPE.equals(qName)) {
			currentIssue.setType(JiraIssueType.parseFromId(getAttributeExisting("id")));
		}
		else if (JiraIssue.FIELD_STATUS.equals(qName)) {
			currentIssue.setStatus(JiraIssueStatus.parseFromId(getAttributeExisting("id")));
		}
		else if (JiraIssue.FIELD_RESOLUTION.equals(qName)) {
			currentIssue.setResolution(JiraIssueResolution.parseFromId(getAttributeExisting("id")));
		}

		if (!isInIssue()) {
			if (currentIssue.getId() != null && currentIssue.getKey() != null) {
				issues.add(currentIssue);
			}
			currentIssue = null;
		}
	}

	@Override
	public void characters(char[] ch, int start, int length) throws SAXException {
		characters.append(ch, start, length);
	}

	// Accessing ==============================================================

	public List<JiraIssue> getIssues() {
		return Collections.unmodifiableList(issues);
	}

	// Utils ==================================================================

	private String getAttributeExisting(String key) throws SAXException {
		final int index = currentAttributes.getIndex(key);
		if (index == -1) {
			throw new SAXException("Attribute '" + key + "' expected, but not found.");
		}
		return currentAttributes.getValue(index);
	}

	private boolean isInIssue() {
		if (!hasElement("rss", 0)) {
			return false;
		}

		return hasElement("rss", 0) && hasElement("channel", 1) && hasElement("item", 2);
	}

	private boolean hasElement(String name, int position) {
		if (currentElements.size() <= position) {
			return false;
		}

		return areEqual(currentElements.get(position), name);
	}

	private String getCharactersTrimmed() {
		return characters.toString().trim();
	}

	private static boolean areEqual(String str1, String str2) {
		if (str1 == null) {
			return str2 == null;
		}

		return str1.equals(str2);
	}
}