package com.syntevo.plugin.jira.transport;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.net.ssl.*;
import javax.xml.parsers.*;

import org.xml.sax.*;

import com.syntevo.plugin.common.bugtracker.transport.*;
import com.syntevo.plugin.jira.*;

/**
 * @author syntevo GmbH
 */
public final class JiraQueryClient extends JiraClient {

	// Fields =================================================================

	private final SAXParserFactory saxParserFactory;

	private int currentIndex;
	private String queryUrl;

	// Setup ==================================================================

	public JiraQueryClient(BugtrackerConnection connection) {
		super(connection);

		saxParserFactory = SAXParserFactory.newInstance();
	}

	// Accessing ==============================================================

	public List<JiraIssue> getFirstIssues(String projectId, String assignee, List<JiraIssueStatus> stati, List<String> fixVersions, int count) throws IOException {
		final BugtrackerConnection connection = getBugtrackerConnection();
		queryUrl = createQueryUrl(connection.getUrl(), connection.getUsername(), connection.getPassword(), projectId, assignee, fixVersions, stati);
		currentIndex = 0;
		return getNextIssues(count);
	}

	public List<JiraIssue> getNextIssues(int count) throws IOException {
		final URL url = createUrl(queryUrl, currentIndex, count);
		currentIndex += count;
		return readIssues(url);
	}

	// Utils ==================================================================

	private List<JiraIssue> readIssues(URL url) throws IOException {
		url = createConnectableURL(url);
		JiraPlugin.LOGGER.debug("Sending request: " + url);
		try {
			final SAXParser parser = saxParserFactory.newSAXParser();
			final JiraIssueXmlHandler handler = new JiraIssueXmlHandler();
			parser.parse(url.openStream(), handler);
			getBugtrackerConnection().acknowledgeSSLClientCertificate(true);
			return handler.getIssues();
		}
		catch (ParserConfigurationException | SAXException ex) {
			throw new IOException(ex.getMessage());
		}
		catch (SSLException ex) {
			getBugtrackerConnection().acknowledgeSSLClientCertificate(false);
			throw ex;
		}
	}

	private static String createQueryUrl(String baseUrl, String username, String password, String projectId, String assignee, List<String> fixVersions, List<JiraIssueStatus> stati) throws UnsupportedEncodingException {
		// http://localhost:8080/secure/IssueNavigator.jspa?reset=true&&pid=10040&fixfor=11731&status=3&assigneeSelect=specificuser&assignee=user&sorter/field=issuekey&sorter/order=DESC

		final StringBuilder builder = new StringBuilder();
		builder.append(baseUrl);
		builder.append("sr/jira.issueviews:searchrequest-xml/temp/SearchRequest.xml?sorter/field=issuekey&sorter/order=DESC");
		builder.append("&os_username=");
		builder.append(URLEncoder.encode(username, "UTF-8"));
		builder.append("&os_password=");
		builder.append(URLEncoder.encode(password, "UTF-8"));
		builder.append("&pid=");
		builder.append(projectId);
		if (assignee != null) {
			builder.append("&assigneeSelect=specificuser&assignee=");
			builder.append(URLEncoder.encode(assignee, "UTF-8"));
		}
		for (JiraIssueStatus status : stati) {
			if (status.getJiraId() == null) {
				continue;
			}
			
			builder.append("&status=");
			builder.append(status.getJiraId());
		}
		for (String fixVersion : fixVersions) {
			builder.append("&fixfor=");
			builder.append(URLEncoder.encode(fixVersion, "UTF-8"));
		}
		return builder.toString();
	}

	private static URL createUrl(String queryUrl, int currentIndex, int count) throws MalformedURLException {
		final StringBuilder builder = new StringBuilder();
		builder.append(queryUrl);
		builder.append("&tempMax=");
		builder.append(count);
		builder.append("&pager/start=");
		builder.append(currentIndex);
		return new URL(builder.toString());
	}
}