package com.syntevo.plugin.precommithook;

import java.io.*;
import java.util.*;
import java.util.concurrent.*;

import org.apache.subversion.javahl.*;

import com.syntevo.openapi.deprecated.log.*;
import com.syntevo.openapi.deprecated.smartsvn.command.*;
import com.syntevo.openapi.deprecated.smartsvn.command.commit.CommitItem;
import com.syntevo.openapi.deprecated.smartsvn.command.commit.*;
import com.syntevo.openapi.deprecated.util.*;

/**
 * @author syntevo GmbH
 */
class PreCommitHookCommitPhase implements ICommitPreCommandPhase {

	// Constants ==============================================================

	private static final Logger LOG = LoggerFactory.createLogger(PreCommitHookPlugin.PLUGIN_ID);

	// Fields =================================================================

	private final ICommitPreCommandPhase wrappedPhase;
	private final File preCommitHook;

	// Setup ==================================================================

	public PreCommitHookCommitPhase(ICommitPreCommandPhase wrappedPhase, File preCommitHook) {
		this.wrappedPhase = wrappedPhase;
		this.preCommitHook = preCommitHook;
	}

	// Implemented ============================================================

	@Override
	public List<CommitPacket> process(List<CommitPacket> commitSets, SvnClients svnClients) throws SubversionException {
		commitSets = wrappedPhase.process(commitSets, svnClients);
		
		for (CommitPacket set : commitSets) {
			final File workingCopyRoot = set.getCommitRoot().getFile();

			final File pathsFile;
			try {
				pathsFile = File.createTempFile("smartsvn", null);
			}
			catch (IOException ex) {
				LOG.error(ex.getMessage());
				continue;
			}

			try {
				fillPathsFile(pathsFile, set);

				final ProcessBuilder processBuilder = new ProcessBuilder(preCommitHook.getAbsolutePath(), pathsFile.getAbsolutePath());
				processBuilder.directory(workingCopyRoot);

				final Process process = processBuilder.start();
				final Semaphore threadWaiter = new Semaphore(0);
				final PreCommitHookStreamGobbler stdoutGobbler = new PreCommitHookStreamGobbler("stdout", process.getInputStream(), threadWaiter);
				final PreCommitHookStreamGobbler stderrGobbler = new PreCommitHookStreamGobbler("stderr", process.getErrorStream(), threadWaiter);

				new Thread(stdoutGobbler).start();
				new Thread(stderrGobbler).start();

				final int retCode = process.waitFor();
				threadWaiter.acquire(2);

				if (retCode != 0) {
					throw new PluginException("Pre-commit hook '" + preCommitHook.getAbsolutePath() + "' failed with return code " + retCode + ".");
				}
			}
			catch (IOException | InterruptedException ex) {
				LOG.error(ex.getMessage());
			}
			finally {
				if (!pathsFile.delete()) {
					LOG.warn("Could not delete '" + pathsFile + "'.");
				}
			}
		}

		return commitSets;
	}

	// Utils ==================================================================

	private static void fillPathsFile(File pathsFile, CommitPacket packet) throws IOException {
		try (PrintWriter writer = new PrintWriter(new FileWriter(pathsFile))) {
			for (CommitItem item : packet.getItems()) {
				writer.println(item.getPath().getFile());
			}
		}
	}
}