package com.syntevo.plugin.removeemptydirectories;

import java.io.*;
import java.util.*;

import org.apache.subversion.javahl.*;
import org.apache.subversion.javahl.callback.*;
import org.apache.subversion.javahl.types.*;

import com.syntevo.openapi.deprecated.log.*;
import com.syntevo.openapi.deprecated.smartsvn.command.*;
import com.syntevo.openapi.deprecated.smartsvn.file.*;
import com.syntevo.openapi.deprecated.smartsvn.thread.*;
import com.syntevo.openapi.deprecated.util.*;

/**
 * @author syntevo GmbH
 */
final class REDSvnCommand implements ISvnCommand {

	// Constants ==============================================================

	private static final Logger LOG = LoggerFactory.createLogger(REDSvnCommand.class.getName());

	// Fields =================================================================

	private final SvnDirectory directory;

	// Setup ==================================================================

	public REDSvnCommand(SvnDirectory directory) {
		this.directory = directory;
	}

	// Implemented ============================================================

	@Override
	public void perform(SvnClients svnClients, ProgressViewer progressViewer) throws SubversionException {
		LOG.info("Removing empty directories within " + directory.getPath());

		final Map<File, List<File>> directoryToChildren = new HashMap<>();
		final Set<File> removableDirectories = new HashSet<>();
		fillDirectoryToChildren(directory, directoryToChildren, removableDirectories, svnClients);
		final List<File> emptyDirectories = determineEmptyDirectories(directoryToChildren, removableDirectories);
		removeEmptyDirectories(emptyDirectories, svnClients, progressViewer);
	}

	// Utils ==================================================================

	private static void fillDirectoryToChildren(SvnDirectory directory, final Map<File, List<File>> directoryToChildren, final Set<File> removableDirectories, SvnClients svnClients) throws SubversionException {
		final String baseLocation = directory.getPath().getFile().getPath();
		final ISVNClient svnClient = svnClients.createClient();
		try {
			svnClient.status(baseLocation, Depth.infinity, false, true, true, false, null, new StatusCallback() {
				@Override
				public void doStatus(String path, Status status) {
					final File file = new File(status.getPath());
					if (status.getNodeKind() == NodeKind.dir) {
						List<File> children = directoryToChildren.get(file);
						if (children == null) {
							children = new ArrayList<>();
							directoryToChildren.put(file, children);
						}

						if (isRemovableDirectory(status)) {
							removableDirectories.add(file);
						}
					}
					final Status.Kind contentStatus = status.getTextStatus();
					if (contentStatus == Status.Kind.deleted) {
						return;
					}

					final File parent = file.getParentFile();
					if (parent == null) {
						return;
					}

					List<File> children = directoryToChildren.get(parent);
					if (children == null) {
						children = new ArrayList<>();
						directoryToChildren.put(parent, children);
					}

					children.add(file);
				}
			});
		} finally {
			svnClient.dispose();
		}
	}

	private static List<File> determineEmptyDirectories(Map<File, List<File>> directoryToChildren, Set<File> removableDirectories) {
		final List<File> emptyDirectories = new ArrayList<>();
		for (boolean continu = true; continu;) {
			continu = false;

			for (File directory : new HashSet<>(directoryToChildren.keySet())) {
				if (directoryToChildren.get(directory).size() > 0 || !removableDirectories.contains(directory)) {
					continue;
				}

				emptyDirectories.add(directory);
				directoryToChildren.remove(directory);
				final List<File> parentChildren = directoryToChildren.get(directory.getParentFile());
				if (parentChildren == null) {
					continue;
				}

				parentChildren.remove(directory);
				continu = true;
			}
		}
		return emptyDirectories;
	}

	private static void removeEmptyDirectories(List<File> emptyDirectories, SvnClients svnClients, ProgressViewer progressViewer) throws SubversionException {
		// Reverse the empty directories, so we will process children before parents.
		Collections.sort(emptyDirectories, new Comparator<File>() {
			@Override
			public int compare(File o1, File o2) {
				return o1.getAbsolutePath().compareTo(o2.getAbsolutePath());
			}
		});
		Collections.reverse(emptyDirectories);

		final ISVNClient svnClient = svnClients.createClient();
		try {
			for (int index = 0; index < emptyDirectories.size(); index++) {
				final File directory = emptyDirectories.get(index);
				final String directoryPath = directory.getAbsolutePath();

				progressViewer.setProgress(index, emptyDirectories.size());
				progressViewer.setMessage("Processing " + directory);

				final boolean[] removed = new boolean[1];

				try {
					svnClient.status(directoryPath, Depth.immediates, false, true, true, false, null, new StatusCallback() {
						@Override
						public void doStatus(String path, Status status) {
							final Status.Kind contentStatus = status.getTextStatus();
							final File file = new File(status.getPath());
							if (file.equals(directory)) {
								if (!isRemovableDirectory(status)) {
									throw PluginException.createWrappedRuntimeException("Directory '" + directory + "' is modified.");
								}
								removed[0] = contentStatus == Status.Kind.deleted;
								return;
							}

							if (contentStatus != Status.Kind.deleted) {
								throw PluginException.createWrappedRuntimeException("Directory '" + directory + "' is not empty now, but was before.");
							}
						}
					});
				} catch (RuntimeException rex) {
					if (rex.getCause() instanceof PluginException) {
						throw (PluginException) rex.getCause();
					}
					throw rex;
				}

				if (removed[0]) {
					continue;
				}
				LOG.info("Scheduling empty directory " + directoryPath + " for removal.");
				svnClient.remove(Collections.singleton(directoryPath), false, false, null, null, null);
			}
		} finally {
			svnClient.dispose();
		}
	}

	private static boolean isRemovableDirectory(Status status) {
		if (status.getNodeKind() != NodeKind.dir) {
			return false;
		}

		final Status.Kind contentsStatus = status.getTextStatus();
		final Status.Kind propertiesStatus =  status.getPropStatus();
		return contentsStatus == Status.Kind.deleted ||
		       (contentsStatus == Status.Kind.normal &&
		        (propertiesStatus == Status.Kind.none || propertiesStatus == Status.Kind.normal));
	}
}
