package com.syntevo.plugin.tagmultiple;

import java.util.*;

import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.openapi.deprecated.smartsvn.*;
import com.syntevo.openapi.deprecated.smartsvn.action.*;
import com.syntevo.openapi.deprecated.smartsvn.file.*;
import com.syntevo.openapi.deprecated.smartsvn.file.gui.*;
import com.syntevo.openapi.deprecated.smartsvn.settings.*;

/**
 * @author syntevo GmbH
 */
final class TagMultipleProjectRootsAction implements ISvnFileDirectoryAction {

	// Constants ==============================================================

	public static final String ID = TagMultipleProjectRootsPlugin.ID + ".TagMultipleProjectRoots";

	private static final int MAX_TAGS_TO_REMEMBER = 10;

	private static final String SETTINGS_KEY_FIX_EXTERNALS = "fixExternals";
	private static final String SETTINGS_KEY_PREVIOUS_TAGS = "previousTags";
	private static final String SETTINGS_KEY_COMMIT_MESSAGE = "commitMessage";

	// Fields =================================================================

	private final SmartSvnServices services;
	private final Settings settings;

	// Setup ==================================================================

	public TagMultipleProjectRootsAction(SmartSvnServices services) {
		this.services = services;
		this.settings = services.getSettingsServices().getGlobalSettingsProvider().getSettings(TagMultipleProjectRootsPlugin.ID);
	}

	// Implemented ============================================================

	@Override
	public boolean isApplicable(SelectedSvnFiles selectedFiles, SelectedSvnDirectories selectedDirectories) {
		final Set<? extends SvnDirectory> directories = selectedDirectories.getDirectories();
		if (directories.isEmpty()) {
			return false;
		}

		for (SvnDirectory svnDirectory : directories) {
			if (svnDirectory.getPath().getRelativePath().length() > 0) {
				return false;
			}
		}

		return true;
	}

	@Override
	public void execute(SelectedSvnFiles selectedFiles, SvnEntryQuery entryQuery, ActionServices services) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void execute(SelectedSvnDirectories selectedDirectories, SvnEntryQuery entryQuery, final ActionServices services) {
		final Set<? extends SvnDirectory> svnDirectories = selectedDirectories.getDirectories();
		final List<String> previousTags = getPreviousTags();
		final TagMultipleProjectRootsDialog dialog = new TagMultipleProjectRootsDialog(previousTags, SettingsUtils.getBooleanValue(SETTINGS_KEY_FIX_EXTERNALS, false, settings),
		                                                                               settings.getString(SETTINGS_KEY_COMMIT_MESSAGE, null), this.services.getGuiSpacings());
		services.getDialogDisplayer().showAsync(dialog, new IDialogResultHandler() {
			@Override
			public void handleDialogResult(int value) {
				if (value != AbstractDialog.RESULT_OK) {
					return;
				}

				final String tagName = dialog.getTagName();
				final boolean fixExternals = dialog.isFixExternals();
				final String commitMessage = dialog.getCommitMessage();
				setPreviousTags(tagName, previousTags);
				SettingsUtils.setBooleanValue(SETTINGS_KEY_FIX_EXTERNALS, fixExternals, settings);
				settings.putString(SETTINGS_KEY_COMMIT_MESSAGE, commitMessage);

				services.scheduleCommand(new TagMultipleProjectRootsSvnCommand(tagName, svnDirectories, fixExternals, TagMultipleProjectRootsAction.this.services.getTagBranchLayouts(), commitMessage), "Tag with '" + tagName + "'");
			}
		});
	}

	// Utils ==================================================================

	private List<String> getPreviousTags() {
		return SettingsUtils.getStrings(SETTINGS_KEY_PREVIOUS_TAGS, settings);
	}

	private void setPreviousTags(String tagName, List<String> previousTags) {
		previousTags.remove(tagName);
		previousTags.add(0, tagName);
		for (int i = previousTags.size(); i-- > MAX_TAGS_TO_REMEMBER;) {
			previousTags.remove(i);
		}
		SettingsUtils.setStrings(SETTINGS_KEY_PREVIOUS_TAGS, previousTags, settings);
	}
}
