package com.syntevo.plugin.trac.commit.messagesource;

import java.util.*;
import java.util.List;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.openapi.deprecated.smartsvn.command.commit.*;
import com.syntevo.plugin.trac.transport.*;

/**
 * @author syntevo GmbH
 */
final class TracCommitMessageSourceDialog extends AbstractOkCancelDialog {

	// Fields =================================================================

	private final TracIssueLoader issueLoader;
	private final TracQueryConfiguration queryConfiguration;
	private final DialogDisplayer dialogDisplayer;
	private final CommitBugtraqProperties bugtraqProperties;

	private List<TracIssue> issues;
	private Table table;
	private String commitMessage;
	private String commitMessagePattern;

	// Setup ==================================================================

	public TracCommitMessageSourceDialog(@Nullable List<TracIssue> issues, @NotNull TracIssueLoader issueLoader,
	                                     @Nullable TracQueryConfiguration queryConfiguration, @Nullable String commitMessagePattern,
	                                     @NotNull CommitBugtraqProperties bugtraqProperties, @NotNull DialogDisplayer dialogDisplayer) {
		this.issues = issues;
		this.issueLoader = issueLoader;
		this.queryConfiguration = queryConfiguration;
		this.commitMessagePattern = commitMessagePattern;
		this.bugtraqProperties = bugtraqProperties;
		this.dialogDisplayer = dialogDisplayer;

		getOkAction().setName("Select");
	}

	// Implemented ============================================================

	@NotNull
	@Override
	public String getTitle() {
		return "Select Trac Issue";
	}

	@NotNull
	@Override
	public Control createComponent(@NotNull Composite parent) {
		final Composite panel = new Composite(parent, SWT.NONE) {
			@Override
			public Point computeSize(int wHint, int hHint, boolean changed) {
				return new Point(600, 400);
			}
		};
		panel.setLayout(new FillLayout());

		table = new Table(panel, SWT.BORDER | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI);
		table.setHeaderVisible(true);

		new TableColumn(table, SWT.RIGHT).setText("ID");
		new TableColumn(table, SWT.NONE).setText("Summary");
		new TableColumn(table, SWT.NONE).setText("Status");
		new TableColumn(table, SWT.NONE).setText("Milestone");
		new TableColumn(table, SWT.NONE).setText("Version");

		table.addListener(SWT.DefaultSelection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				performOk();
			}
		});

		updateTableFromIssues();
		pack();
		return panel;
	}

	@Override
	protected void ok() throws DialogValidationFailedException {
		final int[] selection = table.getSelectionIndices();
		if (selection.length == 0) {
			throw new DialogValidationFailedException(table, "Please select one or more issue.", "Or press Cancel to abort.");
		}

		final List<TracIssue> selectedIssues = new ArrayList<>();
		for (int row : selection) {
			selectedIssues.add(issues.get(row));
		}

		if (commitMessagePattern != null) {
			commitMessagePattern = commitMessagePattern.trim();
		}

		commitMessage = createCommitMessage(selectedIssues, commitMessagePattern, bugtraqProperties);
		super.ok();
	}

	@NotNull
	@Override
	public Button initButtonPanel(@NotNull ButtonPanel buttonPanel) {
		buttonPanel.addButton(new ButtonAction("Refresh", BUTTON_TYPE_OTHER) {
			@Override
			public void performAction() {
				issueLoader.load(queryConfiguration, true);
			}
		});

		if (bugtraqProperties.getMessage() == null) {
			buttonPanel.addButton(new ButtonAction("Configure", BUTTON_TYPE_OTHER) {
				@Override
				public void performAction() {
					configureMessagePattern();
				}
			});
		}

		return super.initButtonPanel(buttonPanel);
	}

	// Accessing ==============================================================

	@NotNull
	public String getCommitMessage() {
		return commitMessage;
	}

	@Nullable
	public String getCommitMessagePattern() {
		return commitMessagePattern;
	}

	public void setIssues(@NotNull List<TracIssue> issues) {
		final boolean firstNonEmptySet = this.issues.size() > 0 && issues.size() > 0;

		final Set<Long> selectedKeys = new HashSet<>();
		for (int row : table.getSelectionIndices()) {
			selectedKeys.add(this.issues.get(row).getId());
		}

		this.issues = issues;

		updateTableFromIssues();
		if (firstNonEmptySet) {
			pack();
		}

		if (selectedKeys.size() > 0) {
			final List<Integer> indicesToSelect = new ArrayList<>();
			int i = 0;
			for (TracIssue issue : issues) {
				if (selectedKeys.contains(issue.getId())) {
					indicesToSelect.add(i);
				}
				i++;
			}

			final int[] indicesToSelectArray = new int[indicesToSelect.size()];
			for (i = 0; i < indicesToSelectArray.length; i++) {
				indicesToSelectArray[i] = indicesToSelect.get(i).intValue();
			}

			table.setSelection(indicesToSelectArray);
		}
	}

	// Utils ==================================================================

	private void updateTableFromIssues() {
		table.setRedraw(false);
		try {
			table.setItemCount(issues.size());

			final TableItem[] items = table.getItems();
			for (int i = 0; i < items.length; i++) {
				final TableItem item = items[i];
				final TracIssue issue = issues.get(i);
				item.setText(0, String.valueOf(issue.getId()));
				item.setText(1, issue.getSummary());
				item.setText(2, issue.getStatus().getName());
				item.setText(3, issue.getMilestoneName());
				item.setText(4, issue.getVersionName());
			}
		}
		finally {
			table.setRedraw(true);
		}
	}

	private void pack() {
		for (TableColumn column : table.getColumns()) {
			column.pack();
		}
	}

	private void configureMessagePattern() {
		final TracConfigureCommitMessagePatternDialog messagePatternDialog = new TracConfigureCommitMessagePatternDialog(commitMessagePattern);
		dialogDisplayer.showAsync(messagePatternDialog, new IDialogResultHandler() {
			@Override
			public void handleDialogResult(int value) {
				if (value != AbstractDialog.RESULT_OK) {
					return;
				}

				commitMessagePattern = messagePatternDialog.getCommitMessagePattern();
			}
		});
	}

	@NotNull
	private static String createCommitMessage(@NotNull List<TracIssue> issues, @Nullable String commitMessagePattern, @NotNull CommitBugtraqProperties bugtraqProperties) {
		final StringBuilder idBuilder = new StringBuilder();
		final StringBuilder messageBuilder = new StringBuilder();
		final String bugtraqMessage = bugtraqProperties.getMessage();

		if (bugtraqMessage != null) {
			for (final Iterator it = issues.iterator(); it.hasNext(); ) {
				final TracIssue issue = (TracIssue)it.next();

				idBuilder.append(bugtraqMessage.replace(TracConfigureCommitMessagePatternDialog.PLACEHOLDER_BUGID, String.valueOf(issue.getId())));

				if (it.hasNext()) {
					idBuilder.append(",");
				}
			}
		}

		for (final Iterator it = issues.iterator(); it.hasNext(); ) {
			final TracIssue issue = (TracIssue)it.next();

			if (bugtraqMessage == null && commitMessagePattern != null && !commitMessagePattern.isEmpty()) {
				String message = commitMessagePattern.replace(TracConfigureCommitMessagePatternDialog.PLACEHOLDER_BUGID, String.valueOf(issue.getId()));
				message = message.replace(TracConfigureCommitMessagePatternDialog.PLACEHOLDER_MESSAGE, issue.getSummary());

				messageBuilder.append(message);
			}
			else {
				messageBuilder.append(issue.getSummary());
			}

			if (it.hasNext()) {
				messageBuilder.append("\n");
			}
		}

		if (bugtraqMessage != null) {
			if (bugtraqProperties.isAppend()) {
				return messageBuilder.toString() + "\n" + idBuilder.toString();
			}
			else {
				return idBuilder.toString() + "\n" + messageBuilder.toString();
			}
		}

		return messageBuilder.toString();
	}
}