package com.syntevo.plugin.trac.commit.messagesource;

import java.util.*;

import org.jetbrains.annotations.*;

import com.syntevo.plugin.trac.*;
import com.syntevo.plugin.trac.transport.*;

/**
 * @author syntevo GmbH
 */
final class TracIssueLoader extends Thread {

	// Constants ==============================================================

	private static final int UNRELEASED_VERSIONS_TO_DISPLAY;
	private static final int INCOMPLETE_MILESTONES_TO_DISPLAY;

	// Static =================================================================

	static {
		int unreleasedVersions = 3;
		try {
			unreleasedVersions = Integer.parseInt(System.getProperty("smartsvn.plugin.trac.unreleased-versions-to-display", String.valueOf(unreleasedVersions)));
		}
		catch (NumberFormatException ex) {
			// Ignore
		}

		UNRELEASED_VERSIONS_TO_DISPLAY = unreleasedVersions;

		int incompleteMilestones = 3;

		try {
			incompleteMilestones = Integer.parseInt(System.getProperty("smartsvn.plugin.trac.incomplete-milestones-to-display", String.valueOf(incompleteMilestones)));
		}
		catch (NumberFormatException ex) {
			// Ignore
		}

		INCOMPLETE_MILESTONES_TO_DISPLAY = incompleteMilestones;
	}

	// Fields =================================================================

	private final TracIssueList issueList = new TracIssueList();
	private final TracIssueTable table;

	private TracQueryConfiguration configuration;
	private boolean forceRefresh;
	private boolean aborted;
	private TracIssueLoaderTableUpdater tableUpdater;

	// Setup ==================================================================

	public TracIssueLoader(@NotNull TracIssueTable table) {
		this.table = table;
	}

	// Implemented ============================================================

	@Override
	public void run() {
		TracQueryConfiguration configuration = null;
		TracJsonRpcClient queryClient;

		for (; ; ) {
			try {
				synchronized (this) {
					while (this.configuration == null && !aborted) {
						try {
							wait();
						}
						catch (InterruptedException ex) {
							// Ignore
						}
					}

					if (aborted) {
						return;
					}

					if (areConfigurationsEqual(configuration, this.configuration) && !forceRefresh) {
						continue;
					}

					configuration = this.configuration;
					forceRefresh = false;

					issueList.clear();

					if (configuration == null) {
						continue;
					}

					queryClient = new TracJsonRpcClient(configuration.getConnection());
					tableUpdater = new TracIssueLoaderTableUpdater(queryClient, issueList, table);
				}

				try {
					try {
						if (configuration.getConnection().getUsername() == null) {
							synchronized (this) {
								this.configuration = configuration = null;
							}

							continue;
						}

						queryClient.login();
					}
					catch (Exception ex) {
						TracPlugin.LOGGER.error(ex.getMessage(), ex);
						configuration.getConnection().showError("Connection failed.", ex);

						synchronized (this) {
							this.configuration = configuration = null;
						}

						continue;
					}

					final String username;
					final List<TracIssueVersion> unreleasedVersions;
					final List<TracIssueMilestone> incompleteMilestones;
					final List<TracIssueVersion> unreleasedVersionsInOrder = queryClient.getUnreleasedVersionsInOrder();
					final List<TracIssueMilestone> incompleteMilestonesInOrder = queryClient.getIncompleteMilestonesInOrder();

					synchronized (this) {
						unreleasedVersions = restrictUnreleasedVersions(unreleasedVersionsInOrder);
						incompleteMilestones = restrictIncompleteMilestones(incompleteMilestonesInOrder);
					}

					if (configuration.isLoadAll()) {
						tableUpdater.load(null, Collections.<TracIssueStatus>emptyList(), Collections.<TracIssueVersion>emptyList(), Collections.<TracIssueMilestone>emptyList());
						continue;
					}

					username = configuration.getConnection().getUsername();
					tableUpdater.load(username, Arrays.asList(TracIssueStatus.ACCEPTED), Collections.<TracIssueVersion>emptyList(), Collections.<TracIssueMilestone>emptyList());
					tableUpdater.load(username, Arrays.asList(TracIssueStatus.ASSIGNED, TracIssueStatus.REOPENED, TracIssueStatus.NEW), unreleasedVersions, incompleteMilestones);
				}
				catch (Exception ex) {
					TracPlugin.LOGGER.error(ex.getMessage(), ex);
					configuration.getConnection().showError("Connection failed.", ex);
				}

				synchronized (this) {
					this.configuration = null;
				}
			}
			catch (Throwable ex) {
				TracPlugin.LOGGER.error(ex.getMessage(), ex);
			}
		}
	}

	// Accessing ==============================================================

	public synchronized void load(@Nullable TracQueryConfiguration configuration, boolean forceRefresh) {
		this.configuration = configuration;
		this.forceRefresh = forceRefresh;
		notifyAll();
	}

	public synchronized void abortAndCleanup() {
		this.aborted = true;

		if (tableUpdater != null) {
			tableUpdater.abort();
		}

		notifyAll();
	}

	// Utils ==================================================================

	@NotNull
	private static List<TracIssueVersion> restrictUnreleasedVersions(@NotNull List<TracIssueVersion> unreleasedVersions) {
		if (unreleasedVersions.size() == 0) {
			return Collections.emptyList();
		}

		final List<TracIssueVersion> firstUnreleasedVersions = new ArrayList<>();

		for (int index = 0; index < Math.min(UNRELEASED_VERSIONS_TO_DISPLAY, unreleasedVersions.size()); index++) {
			firstUnreleasedVersions.add(unreleasedVersions.get(index));
		}

		return firstUnreleasedVersions;
	}

	@NotNull
	private static List<TracIssueMilestone> restrictIncompleteMilestones(@NotNull List<TracIssueMilestone> incompleteMilestones) {
		if (incompleteMilestones.size() == 0) {
			return Collections.emptyList();
		}

		final List<TracIssueMilestone> firstIncompleteMilestones = new ArrayList<>();

		for (int index = 0; index < Math.min(INCOMPLETE_MILESTONES_TO_DISPLAY, incompleteMilestones.size()); index++) {
			firstIncompleteMilestones.add(incompleteMilestones.get(index));
		}

		return firstIncompleteMilestones;
	}
	private static boolean areConfigurationsEqual(@Nullable TracQueryConfiguration configuration1, @Nullable TracQueryConfiguration configuration2) {
		if (configuration1 == null) {
			return configuration2 == null;
		}

		return configuration2 != null && configuration1.getConnection().getUrl().equals(configuration2.getConnection().getUrl());
	}
}