package com.syntevo.plugin.trac.commit.workflow;

import java.util.*;

import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.gui.*;
import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.openapi.deprecated.smartsvn.command.commit.*;
import com.syntevo.openapi.deprecated.smartsvn.settings.*;
import com.syntevo.openapi.deprecated.smartsvn.thread.*;
import com.syntevo.openapi.deprecated.util.*;
import com.syntevo.plugin.trac.*;
import com.syntevo.plugin.trac.commit.*;
import com.syntevo.plugin.trac.transport.*;

/**
 * @author syntevo GmbH
 */
final class TracFilesPhase extends FilterCommitPacketPhase {

	// Constants ==============================================================

	private static final boolean RESOLVE_ENABLED = "true".equals(System.getProperty("smartsvn.plugin.trac.show-resolveIssue-dialog", "true"));

	// Fields =================================================================

	private final SettingsServices settingsServices;
	private final JobExecutor jobExecutor;
	private final DialogDisplayer dialogDisplayer;
	private final GuiSpacings spacings;

	private List<TracResolvePacket> selectedResolvePackets;

	// Setup ==================================================================

	public TracFilesPhase(@NotNull ICommitPacketPhase internalPhase, @NotNull SettingsServices settingsServices, @NotNull JobExecutor jobExecutor,
	                      @NotNull GuiSpacings spacings, DialogDisplayer dialogDisplayer) {
		super(internalPhase);

		this.settingsServices = settingsServices;
		this.jobExecutor = jobExecutor;
		this.dialogDisplayer = dialogDisplayer;
		this.spacings = spacings;
	}

	// Implemented ============================================================

	@Override
	public void updateTo(@NotNull final CommitPacketPhaseState state, @NotNull final Runnable successRunnable) {
		super.updateTo(state, new Runnable() {
			@Override
			public void run() {
				selectedResolvePackets = null;

				if (!RESOLVE_ENABLED) {
					successRunnable.run();
					return;
				}

				final Set<TracResolvablePacket> allResolvablePackets = new HashSet<>();

				for (CommitPacket packet : new ArrayList<>(state.getPackets())) {
					allResolvablePackets.addAll(extractResolvablePackets(packet, settingsServices, spacings, dialogDisplayer));
				}

				if (allResolvablePackets.isEmpty()) {
					successRunnable.run();
					return;
				}

				processResolvablePackets(allResolvablePackets, jobExecutor, successRunnable, dialogDisplayer);
			}
		});
	}

	// Accessing ==============================================================

	@Nullable
	public List<TracResolvePacket> getResolvePackets() {
		UiUtils.assertThreadAccess(); // We don't sync on selectedResolvePackets, so we must touch that only in the GUI thread.
		return selectedResolvePackets != null ? Collections.unmodifiableList(selectedResolvePackets) : null;
	}

	// Utils ==================================================================

	private void processResolvablePackets(@NotNull Set<TracResolvablePacket> allResolvablePackets,
	                                                                   @NotNull JobExecutor jobExecutor, final Runnable successRunnable, @NotNull final DialogDisplayer dialogDisplayer) {
		TracPlugin.LOGGER.debug("Filtering resolvable packets");

		final List<TracResolvablePacket> resolvablePacketsInOrder = new ArrayList<>(allResolvablePackets);
		Collections.sort(resolvablePacketsInOrder, new Comparator<TracResolvablePacket>() {
			@Override
			public int compare(TracResolvablePacket o1, TracResolvablePacket o2) {
				return o1.getIssueId().compareTo(o2.getIssueId());
			}
		});

		jobExecutor.executeAsync("Trac", new IJob() {
			@Override
			public IJobSuccessRunnable execute(@NotNull ProgressViewer progressViewer) {
				int progress = 0;

				final List<TracResolvablePacket> resolvablePackets = new ArrayList<>();
				for (TracResolvablePacket packet : resolvablePacketsInOrder) {
					progressViewer.setMessage("Querying " + packet.getConnection().getUrl());
					progressViewer.setProgress(progress++, resolvablePacketsInOrder.size());

					try {
						final TracJsonRpcClient client = new TracJsonRpcClient(packet.getConnection());
						final TracIssue issue = client.getIssueById(packet.getIssueId());

						if (issue == null) {
							TracPlugin.LOGGER.debug(" ... " + packet + " not exists in Trac");
							continue;
						}

						if (!isResolvable(issue)) {
							TracPlugin.LOGGER.debug(" ... " + packet + " not resolvable (status=" + issue.getStatus() + ")");
							continue;
						}

						final List<TracIssueVersion> unreleasedVersions = client.getUnreleasedVersionsInOrder();
						final List<TracIssueMilestone> incompleteMilestones = client.getIncompleteMilestonesInOrder();

						packet.setIssue(issue);
						packet.setUnreleasedVersions(unreleasedVersions);
						packet.setIncompleteMilestones(incompleteMilestones);
						resolvablePackets.add(packet);
						TracPlugin.LOGGER.debug(" ... " + packet + " is resolvable.");
					}
					catch (Exception ex) {
						TracPlugin.LOGGER.error(ex.getMessage(), ex);
						dialogDisplayer.showErrorDialogSync("Trac", ex.getMessage(), null);
					}
				}

				selectedResolvePackets = chooseResolvePackets(resolvablePackets);
				return new IJobSuccessRunnable() {
					@Override
					public void runInEdt(@NotNull DialogDisplayer dialogDisplayer) {
						if (selectedResolvePackets != null) {
							successRunnable.run();
						}
					}
				};
			}
		});
	}

	@Nullable
	private List<TracResolvePacket> chooseResolvePackets(@NotNull List<TracResolvablePacket> resolvablePackets) {
		final List<TracResolvePacket> selectedResolvePackets = new ArrayList<>();

		for (final TracResolvablePacket packet : resolvablePackets) {
			final TracResolveIssueDialog resolveIssueDialog = dialogDisplayer.showSync(new IDialogFactory<TracResolveIssueDialog>() {
				@NotNull
				@Override
				public TracResolveIssueDialog createDialog() {
					return TracResolveIssueDialog.getInstance(packet, spacings);
				}
			});
			if (resolveIssueDialog == null) {
				TracPlugin.LOGGER.debug(" ... " + packet + " cancelled");
				return null;
			}

			if (!resolveIssueDialog.doResolve()) {
				TracPlugin.LOGGER.debug(" ... " + packet + " not selected");
				continue;
			}

			TracPlugin.LOGGER.debug(" ... " + packet + " selected");
			final String fixVersion = resolveIssueDialog.getFixVersion();
			final String fixMilestone = resolveIssueDialog.getFixMilestone();

			selectedResolvePackets.add(new TracResolvePacket(packet.getConnection(), packet.getIssueId(), fixVersion, fixMilestone));
		}

		return selectedResolvePackets;
	}

	private static boolean isResolvable(@NotNull TracIssue issue) {
		final TracIssueStatus status = issue.getStatus();
		return status == TracIssueStatus.ACCEPTED || status == TracIssueStatus.ASSIGNED || status == TracIssueStatus.REOPENED || status == TracIssueStatus.NEW;
	}

	@NotNull
	private static Collection<TracResolvablePacket> extractResolvablePackets(@NotNull CommitPacket packet, @NotNull SettingsServices settingsServices,
	                                                                         @NotNull GuiSpacings spacings, @NotNull DialogDisplayer dialogDisplayer) {
		TracPlugin.LOGGER.debug("Extracting resolvable packets for " + packet.getCommitRoot());
		final CommitBugtraqProperties bugtraqProperties = packet.getBugtraqProperties();

		if (bugtraqProperties == null) {
			TracPlugin.LOGGER.debug(" ... no bugtraq-properties");
			return Collections.emptyList();
		}

		final String urlStr = bugtraqProperties.getURL();

		if (urlStr == null) {
			TracPlugin.LOGGER.debug(" ... no bugtraq URL");
			return Collections.emptyList();
		}

		final TracUrl url = TracUrl.parseFromBugtraqUrls(urlStr);

		if (url == null) {
			TracPlugin.LOGGER.debug(" ... no valid Trac URL");
			return Collections.emptyList();
		}

		TracPlugin.LOGGER.debug(" ... URL is " + url);

		final String message = packet.getMessage();
		final List<TracResolvablePacket> resolvePackets = new ArrayList<>();
		final TracUIConnection connection = new TracUIConnection(url.getBaseUrl(), settingsServices, true, spacings, dialogDisplayer);

		TracPlugin.LOGGER.debug("Parsing message '" + message + "'");

		if (message != null) {
			final List<String> issueIds = bugtraqProperties.parseIssueIds(message);

			if (issueIds == null) {
				return Collections.emptyList();
			}

			for (String idStr : issueIds) {
				try {
					resolvePackets.add(new TracResolvablePacket(Integer.valueOf(idStr), connection));
					TracPlugin.LOGGER.debug(" ... found issue " + idStr);
				}
				catch (NumberFormatException e) {
					TracPlugin.LOGGER.error(" ... skip invalid issue id " + idStr);
				}
			}
		}

		return resolvePackets;
	}
}