package com.syntevo.plugin.trac.commit.workflow;

import java.util.List;

import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.gui.*;
import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.openapi.deprecated.util.*;
import com.syntevo.plugin.trac.transport.*;

/**
 * @author syntevo GmbH
 */
final class TracResolveIssueDialog extends AbstractOkCancelDialog {

	// Constants ==============================================================

	private static final boolean USE_EMPTY_VERSION = true;
	private static final boolean USE_EMPTY_MILESTONE = true;

	// Static =================================================================

	@NotNull
	public static TracResolveIssueDialog getInstance(@NotNull TracResolvablePacket packet, @NotNull GuiSpacings spacings) {
		final List<TracIssueVersion> unreleasedVersions = packet.getUnreleasedVersions();
		final List<TracIssueMilestone> incompleteMilestones = packet.getIncompleteMilestones();
		final TracIssue issue = packet.getIssue();
		final String defaultVersion = issue.getVersionName();
		final String defaultMilestone = issue.getMilestoneName();
		String summary = issue.getSummary();
		String message = "Do you want to resolve issue " + issue.getId();

		if (summary != null) {
			if (summary.length() > 50) {
				summary = summary.substring(0, 47) + "...";
			}

			message = message + ": " + summary;
		}

		message = message + "?";

		return new TracResolveIssueDialog(message, unreleasedVersions, incompleteMilestones, defaultVersion, defaultMilestone, spacings);
	}

	// Fields =================================================================

	private final String message;
	private final List<TracIssueVersion> unreleasedVersions;
	private final List<TracIssueMilestone> incompleteMilestones;
	private final GuiSpacings spacings;

	private Button resolveButton;
	private Combo fixVersionComboBox;
	private Combo fixMilestoneComboBox;
	private Button dontResolveButton;

	private final String defaultVersion;
	private final String defaultMilestone;

	private boolean doResolve;
	private String fixVersion;
	private String fixMilestone;

	// Setup ==================================================================

	private TracResolveIssueDialog(@NotNull String message, @NotNull List<TracIssueVersion> unreleasedVersions, @NotNull List<TracIssueMilestone> incompleteMilestones,
	                               @Nullable String defaultVersion, @Nullable String defaultMilestone, @NotNull GuiSpacings spacings) {
		this.message = message;
		this.unreleasedVersions = unreleasedVersions;
		this.incompleteMilestones = incompleteMilestones;
		this.defaultVersion = defaultVersion;
		this.defaultMilestone = defaultMilestone;
		this.spacings = spacings;
	}

	// Implemented ============================================================

	@NotNull
	@Override
	public String getTitle() {
		return "Resolve Trac Issue";
	}

	@NotNull
	@Override
	public Control createComponent(@NotNull Composite parent) {
		final Composite panel = new Composite(parent, SWT.NONE);

		final GridLayout layout = new GridLayout(2, false);
		layout.horizontalSpacing = spacings.getXRelated();
		layout.verticalSpacing = spacings.getYUnrelated();
		panel.setLayout(layout);

		final Control messageLabel = UiUtils.createLabel(message, panel);
		messageLabel.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 2, 1));

		final Control emptyLabel = UiUtils.createLabel("", panel);
		emptyLabel.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 2, 1));

		dontResolveButton = UiUtils.createRadioButton("Don't mark as resolved", panel);
		dontResolveButton.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, false, false, 2, 1));
		dontResolveButton.setSelection(true);

		resolveButton = UiUtils.createRadioButton("Mark as resolved", panel);
		resolveButton.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, false, false, 2, 1));

		final Label resolveVersion = UiUtils.createLabel("Version:", panel);
		resolveVersion.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, false));

		fixVersionComboBox = new Combo(panel, SWT.BORDER | SWT.READ_ONLY);
		fixVersionComboBox.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

		final String[] versionNames = new String[unreleasedVersions.size()];
		int indexToSelect = -1;

		for (int i = 0; i < versionNames.length; i++) {
			final TracIssueVersion version = unreleasedVersions.get(i);
			versionNames[i] = version.getName();

			if (indexToSelect < 0 && versionNames[i].equals(defaultVersion)) {
				indexToSelect = i;
			}
		}

		fixVersionComboBox.setItems(versionNames);

		if (indexToSelect >= 0) {
			fixVersionComboBox.select(indexToSelect);
		}

		if (USE_EMPTY_VERSION) {
			fixVersionComboBox.add("", 0);
		}

		final Label resolveMilestone = UiUtils.createLabel("Milestone:", panel);
		resolveMilestone.setLayoutData(new GridData(SWT.RIGHT, SWT.FILL, false, false));

		fixMilestoneComboBox = new Combo(panel, SWT.BORDER | SWT.READ_ONLY);
		fixMilestoneComboBox.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

		final String[] milestoneNames = new String[incompleteMilestones.size()];
		indexToSelect = -1;

		for (int i = 0; i < milestoneNames.length; i++) {
			final TracIssueMilestone milestone = incompleteMilestones.get(i);
			milestoneNames[i] = milestone.getName();

			if (indexToSelect < 0 && milestoneNames[i].equals(defaultMilestone)) {
				indexToSelect = i;
			}
		}

		fixMilestoneComboBox.setItems(milestoneNames);

		if (indexToSelect >= 0) {
			fixMilestoneComboBox.select(indexToSelect);
		}

		if (USE_EMPTY_MILESTONE) {
			fixMilestoneComboBox.add("", 0);
		}

		final SelectionAdapter selectionListener = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				updateEnabledStates();
			}
		};

		resolveButton.addSelectionListener(selectionListener);
		dontResolveButton.addSelectionListener(selectionListener);

		updateEnabledStates();

		return panel;
	}

	@Override
	public void dialogShowing() {
		super.dialogShowing();

		if (fixVersionComboBox.isEnabled()) {
			fixVersionComboBox.setFocus();
		}
		else if (fixMilestoneComboBox.isEnabled()) {
			fixMilestoneComboBox.setFocus();
		}
		else {
			dontResolveButton.setFocus();
		}
	}

	@Override
	protected void ok() throws DialogValidationFailedException {
		doResolve = resolveButton.getSelection();
		fixVersion = null;
		fixMilestone = null;

		if (doResolve) {
			int selIdx = fixVersionComboBox.getSelectionIndex();

			if (selIdx > -1) {
				fixVersion = fixVersionComboBox.getItem(fixVersionComboBox.getSelectionIndex());
			}

			selIdx = fixMilestoneComboBox.getSelectionIndex();

			if (selIdx > -1) {
				fixMilestone = fixMilestoneComboBox.getItem(fixMilestoneComboBox.getSelectionIndex());
			}
		}

		super.ok();
	}

	// Accessing ==============================================================

	@Nullable
	public String getFixVersion() {
		if (fixVersion == null || fixVersion.isEmpty()) {
			return null;
		}

		return fixVersion;
	}

	@Nullable
	public String getFixMilestone() {
		if (fixMilestone == null || fixMilestone.isEmpty()) {
			return null;
		}

		return fixMilestone;
	}

	public boolean doResolve() {
		return doResolve;
	}

	// Utils ==================================================================

	private void updateEnabledStates() {
		fixVersionComboBox.setEnabled(resolveButton.getSelection());
		fixMilestoneComboBox.setEnabled(resolveButton.getSelection());
	}
}