package com.syntevo.plugin.trac.transport;

import java.io.*;
import java.net.*;
import java.text.*;
import java.util.*;

import org.jetbrains.annotations.*;
import org.json.simple.*;
import org.json.simple.parser.*;
import org.json.simple.parser.ParseException;

import com.syntevo.openapi.deprecated.util.*;
import com.syntevo.plugin.common.bugtracker.transport.*;
import com.syntevo.plugin.trac.*;

/**
 * @author syntevo GmbH
 */
public final class TracJsonRpcClient extends TracClient {

	// Constants ==============================================================

	private static final String UTF8 = "UTF-8";

	private static final String JSON_OPTION_PREFIX = "&";
	private static final String JSON_OPTION_ASSIGNMENT = "=";

	private static final String JSON_FIELD_ID = "id";
	private static final String JSON_FIELD_MILESTONE = "milestone";
	private static final String JSON_FIELD_JSONCLASS = "__jsonclass__";
	private static final String JSON_FIELD_METHOD = "method";
	private static final String JSON_FIELD_PARAMS = "params";

	private static final String JSON_FIELD_ERROR = "error";
	private static final String JSON_FIELD_MESSAGE = "message";
	private static final String JSON_FIELD_NAME = "name";
	private static final String JSON_FIELD_DESCRIPTION = "description";
	private static final String JSON_FIELD_CODE = "code";

	private static final String JSON_FIELD_TIME = "time";
	private static final String JSON_FIELD_DUE = "due";
	private static final String JSON_FIELD_COMPLETED = "completed";

	private static final String JSON_FIELD_RESULT = "result";
	private static final String JSON_FIELD_MAX = "max";
	private static final String JSON_FIELD_ORDER = "order";
	private static final String JSON_FIELD_PAGE = "page";

	private static final String JSON_METHOD_SYSTEM_MULTICALL = "system.multicall";
	private static final String JSON_METHOD_TICKET_GET = "ticket.get";
	private static final String JSON_METHOD_TICKET_QUERY = "ticket.query";
	private static final String JSON_METHOD_TICKET_UPDATE = "ticket.update";

	private static final String JSON_METHOD_TICKET_VERSION_GET = "ticket.version.get";
	private static final String JSON_METHOD_TICKET_VERSION_GET_ALL = "ticket.version.getAll";

	private static final String JSON_METHOD_TICKET_MILESTONE_GET = "ticket.milestone.get";
	private static final String JSON_METHOD_TICKET_MILESTONE_GET_ALL = "ticket.milestone.getAll";

	private static final String JSON_METHOD_GET_API_VERSION = "system.getAPIVersion";

	private static final String JSON_TIMESTAMP_PATTERN = "yyyy-MM-ddHH:mm:ss";

	// Fields =================================================================

	private int maxCount;
	private int currentPage;
	private boolean hasNextPage;
	private String queryParams = "";

	// Setup ==================================================================

	public TracJsonRpcClient(@NotNull BugtrackerConnection connection) {
		super(connection);
	}

	// Accessing ==============================================================

	public void login() throws JsonRpcException, IOException {
		final JSONObject requestObj = createJSONRequest(JSON_METHOD_GET_API_VERSION);
		sendProcessedJSONRequest(requestObj, JSONArray.class);
	}

	@NotNull
	public List<TracIssue> getFirstIssues(@Nullable String assignee, @NotNull List<TracIssueStatus> states,
	                                      @NotNull List<TracIssueVersion> fixVersions, @NotNull List<TracIssueMilestone> fixMilestones, int maxCount)
			throws IOException, JsonRpcException {
		this.maxCount = maxCount;
		queryParams = createQueryMainParams(assignee, fixVersions, fixMilestones, states);
		currentPage = 1;

		return getNextIssues();
	}

	@Nullable
	public TracIssue getIssueById(@NotNull Integer id) throws IOException {
		final JSONObject requestObj = createJSONRequest(JSON_METHOD_TICKET_GET, id);

		try {
			final JSONArray result = Assert.assertNotNull(sendProcessedJSONRequest(requestObj, JSONArray.class));
			return createTracIssue(result);
		}
		catch (Exception e) {
			return null;
		}
	}

	@NotNull
	public List<TracIssueMilestone> getIncompleteMilestonesInOrder() throws IOException, JsonRpcException {
		final JSONObject requestObj = createJSONRequest(JSON_METHOD_TICKET_MILESTONE_GET_ALL);
		JSONArray result = sendProcessedJSONRequest(requestObj, JSONArray.class);

		if (result == null) {
			return Collections.emptyList();
		}

		final List<JSONObject> requests = new LinkedList<>();

		for (Object obj : result) {
			final String milestone = getObjectOfTypeNotNull(obj, String.class);
			requests.add(createJSONRequest(JSON_METHOD_TICKET_MILESTONE_GET, milestone));
		}

		result = Assert.assertNotNull(sendMultiCallJSONRequest(requests));
		final List<TracIssueMilestone> milestones = new LinkedList<>();

		for (Object obj : result) {
			final JSONObject jsonObject = getJSONObjectNotNull(obj);
			final JSONObject resultObject = getObjectOfTypeNotNull(jsonObject, JSON_FIELD_RESULT, JSONObject.class);
			final String name = getObjectOfTypeNotNull(resultObject, JSON_FIELD_NAME, String.class);
			final String desc = getObjectOfTypeNotNull(resultObject, JSON_FIELD_DESCRIPTION, String.class);
			final Object dueObj = resultObject.get(JSON_FIELD_DUE);
			final Object completedObj = resultObject.get(JSON_FIELD_COMPLETED);
			final Date dueTime = getNotMandatoryDateFromObject(dueObj);
			final Date completedTime = getNotMandatoryDateFromObject(completedObj);

			if (completedTime == null || completedTime.after(new Date())) {
				milestones.add(new TracIssueMilestone(name, desc, dueTime, completedTime));
			}
		}

		return milestones;
	}

	@NotNull
	public List<TracIssueVersion> getUnreleasedVersionsInOrder() throws IOException, JsonRpcException {
		final JSONObject requestObj = createJSONRequest(JSON_METHOD_TICKET_VERSION_GET_ALL);
		JSONArray result = sendProcessedJSONRequest(requestObj, JSONArray.class);

		if (result == null) {
			return Collections.emptyList();
		}

		final List<JSONObject> requests = new LinkedList<>();

		for (Object obj : result) {
			final String version = getObjectOfTypeNotNull(obj, String.class);
			requests.add(createJSONRequest(JSON_METHOD_TICKET_VERSION_GET, version));
		}

		result = Assert.assertNotNull(sendMultiCallJSONRequest(requests));

		final List<TracIssueVersion> versions = new LinkedList<>();

		for (Object obj : result) {
			final JSONObject jsonObject = getJSONObjectNotNull(obj);
			final JSONObject resultObject = getObjectOfTypeNotNull(jsonObject, JSON_FIELD_RESULT, JSONObject.class);
			final String name = getObjectOfTypeNotNull(resultObject, JSON_FIELD_NAME, String.class);
			final String desc = getObjectOfTypeNotNull(resultObject, JSON_FIELD_DESCRIPTION, String.class);
			final Object timeObj = resultObject.get(JSON_FIELD_TIME);
			final Date time = getNotMandatoryDateFromObject(timeObj);

			if (time == null || time.after(new Date())) {
				versions.add(new TracIssueVersion(name, desc, time));
			}
		}

		return versions;
	}

	@SuppressWarnings("unchecked")
	public void resolveIssue(@NotNull Integer issueKey, @NotNull String username, @NotNull TracIssueResolution resolution, @Nullable String fixVersion, @Nullable String fixMilestone)
			throws JsonRpcException, IOException {
		//array ticket.update(int id, string comment, struct attributes={}, boolean notify=False, string author="", dateTime.iso8601 when=None)

		final JSONObject attributes = new JSONObject();
		attributes.put(TracIssue.FIELD_RESOLUTION, resolution.toString());
		attributes.put(TracIssue.FIELD_STATUS, TracIssueStatus.CLOSED.toString());

		if (fixVersion != null) {
			attributes.put(TracIssue.FIELD_VERSION, fixVersion);
		}

		if (fixMilestone != null) {
			attributes.put(TracIssue.FIELD_MILESTONE, fixMilestone);
		}

		final JSONObject requestObj = createJSONRequest(JSON_METHOD_TICKET_UPDATE, issueKey, null, attributes, Boolean.FALSE, username, null);
		sendJSONRequest(requestObj);
	}

	@NotNull
	public List<TracIssue> getNextIssues() throws IOException, JsonRpcException {
		int max = maxCount;

		// first issues
		if (currentPage == 1) {
			max++;
		}
		else if (!hasNextPage) {
			return Collections.emptyList();
		}

		final StringBuilder builder = new StringBuilder(queryParams);
		builder.append(JSON_OPTION_PREFIX);
		builder.append(JSON_FIELD_PAGE);
		builder.append(JSON_OPTION_ASSIGNMENT);
		builder.append(currentPage);

		builder.append(JSON_OPTION_PREFIX);
		builder.append(JSON_FIELD_MAX);
		builder.append(JSON_OPTION_ASSIGNMENT);
		builder.append(max);

		final List<TracIssue> results;

		try {
			results = readIssues(builder.toString());
		}
		catch (JsonRpcException e) {
			// Page n is beyond the number of	pages in the query
			if (e.getErrorCode().longValue() == -32603) {
				hasNextPage = false;
				return Collections.emptyList();
			}

			throw e;
		}

		// only by getting first issues
		if (currentPage == 1) {
			if (results.size() > maxCount) {
				results.remove(maxCount);
				hasNextPage = true;
			}
			else {
				hasNextPage = false;
			}
		}

		currentPage++;

		return results;
	}

	// Utils ==================================================================

	@SuppressWarnings("unchecked")
	@NotNull
	private JSONObject sendJSONRequest(@NotNull JSONObject request) throws JsonRpcException, IOException {
		request.put(JSON_FIELD_ID, System.currentTimeMillis());

		final StringWriter out = new StringWriter();
		request.writeJSONString(out);

		final String jsonText = out.toString();

		TracPlugin.LOGGER.debug("Sending request: " + jsonText);

		final URL url = new URL(getBugtrackerConnection().getUrl());
		final URLConnection urlConnection = createConnectableURL(url).openConnection();

		final OutputStreamWriter writer = new OutputStreamWriter(urlConnection.getOutputStream(), "UTF-8");
		writer.write(jsonText);
		writer.flush();
		writer.close();

		final ByteArrayOutputStream bos = new ByteArrayOutputStream();
		copyStream(urlConnection.getInputStream(), bos);

		final String content = new String(bos.toByteArray(), "UTF-8");
		TracPlugin.LOGGER.debug("Received reply: " + content);

		final JSONParser parser = new JSONParser();
		final Object replyObj;

		try {
			replyObj = parser.parse(content);
			getBugtrackerConnection().acknowledgeSSLClientCertificate(true);
		}
		catch (ParseException ex) {
			throw new IOException(ex);
		}

		final JSONObject reply = getJSONObjectNotNull(replyObj);
		final Long requestId = getObjectOfType(request, JSON_FIELD_ID, Long.class);
		final Long replyId = getObjectOfType(reply, JSON_FIELD_ID, Long.class);
		final JSONObject errorObj = getObjectOfType(reply, JSON_FIELD_ERROR, JSONObject.class);

		if (errorObj != null) {
			final String errorMsg = getObjectOfTypeNotNull(errorObj, JSON_FIELD_MESSAGE, String.class);
			final String errorName = getObjectOfTypeNotNull(errorObj, JSON_FIELD_NAME, String.class);
			final Long errorCode = getObjectOfTypeNotNull(errorObj, JSON_FIELD_CODE, Long.class);

			//throw new IOException(errorName + ": " + errorMsg + " (" + errorCode + ")");
			throw new JsonRpcException(errorMsg, errorName, errorCode);
		}

		if (!CompareUtils.areEqual(requestId, replyId)) {
			throw new IOException("Reply could not be assigned on the basis of the id.");
		}

		return reply;
	}

	private static void copyStream(@NotNull InputStream inputStream, @NotNull OutputStream outputStream) throws IOException {
		try {
			final byte[] buffer = new byte[64 * 1024];
			for (int readBytes = inputStream.read(buffer); readBytes > 0; readBytes = inputStream.read(buffer)) {
				outputStream.write(buffer, 0, readBytes);
			}
		}
		finally {
			outputStream.flush();
		}
	}

	@Nullable
	private <CLASS> CLASS sendProcessedJSONRequest(@NotNull JSONObject request, @NotNull Class<CLASS> replyClazz) throws IOException, JsonRpcException {
		return getObjectOfType(sendJSONRequest(request), JSON_FIELD_RESULT, replyClazz);
	}

	@Nullable
	private JSONArray sendMultiCallJSONRequest(@NotNull List<JSONObject> requests) throws IOException, JsonRpcException {
		final JSONObject mainRequest = createJSONRequest(JSON_METHOD_SYSTEM_MULTICALL, requests.toArray());
		return sendProcessedJSONRequest(mainRequest, JSONArray.class);
	}

	@NotNull
	private List<TracIssue> readIssues(@NotNull String queryParams) throws IOException, JsonRpcException {
		final JSONObject requestObj = createJSONRequest(JSON_METHOD_TICKET_QUERY, queryParams);
		final JSONArray ticketIds = sendProcessedJSONRequest(requestObj, JSONArray.class);

		if (ticketIds == null || ticketIds.size() == 0) {
			return Collections.emptyList();
		}

		final List<JSONObject> ticketRequests = new LinkedList<>();

		for (Object obj : ticketIds) {
			final Long id = getObjectOfTypeNotNull(obj, Long.class);
			ticketRequests.add(createJSONRequest(JSON_METHOD_TICKET_GET, id));
		}

		final JSONArray tickets = Assert.assertNotNull(sendMultiCallJSONRequest(ticketRequests));
		final List<TracIssue> issues = new LinkedList<>();

		for (Object obj : tickets) {
			final JSONObject reply = getJSONObjectNotNull(obj);
			issues.add(createTracIssue(reply));
		}

		return issues;
	}

	@SuppressWarnings("unchecked")
	@NotNull
	private static JSONObject createJSONRequest(@NotNull String method, @NotNull Object... params) {
		final JSONObject jsonObj = new JSONObject();
		jsonObj.put(JSON_FIELD_METHOD, method);

		if (params.length > 0) {
			final JSONArray array = new JSONArray();
			array.addAll(Arrays.asList(params));

			jsonObj.put(JSON_FIELD_PARAMS, array);
		}

		return jsonObj;
	}

	@NotNull
	private static TracIssue createTracIssue(@NotNull JSONArray jsonArray) throws IOException {
		final Long id = getObjectOfTypeNotNull(jsonArray.get(0), Long.class);
		final TracIssue issue = new TracIssue(id.longValue());
		final JSONObject attributes = getJSONObjectNotNull(jsonArray.get(3));

		final String statusStr = getObjectOfTypeNotNull(attributes, TracIssue.FIELD_STATUS, String.class);
		issue.setStatus(TracIssueStatus.parseFromId(statusStr));

		final JSONObject changeTimeObj = getObjectOfTypeNotNull(attributes, TracIssue.FIELD_CHANGETIME, JSONObject.class);
		issue.setChangeTime(getParsedDateFromJSONObject(changeTimeObj));

		issue.setDescription(getObjectOfType(attributes, TracIssue.FIELD_DESCRIPTION, String.class));
		issue.setReporter(getObjectOfType(attributes, TracIssue.FIELD_REPORTER, String.class));
		issue.setCc(getObjectOfType(attributes, TracIssue.FIELD_CC, String.class));
		issue.setResolution(getObjectOfType(attributes, TracIssue.FIELD_RESOLUTION, String.class));

		final JSONObject createdTimeObj = getObjectOfTypeNotNull(attributes, TracIssue.FIELD_CREATEDTIME, JSONObject.class);
		issue.setCreatedTime(getParsedDateFromJSONObject(createdTimeObj));

		issue.setComponent(getObjectOfType(attributes, TracIssue.FIELD_COMPONENT, String.class));
		issue.setSummary(getObjectOfType(attributes, TracIssue.FIELD_SUMMARY, String.class));
		issue.setPriority(getObjectOfType(attributes, TracIssue.FIELD_PRIORITY, String.class));
		issue.setKeywords(getObjectOfType(attributes, TracIssue.FIELD_KEYWORDS, String.class));
		issue.setVersionName(getObjectOfType(attributes, TracIssue.FIELD_VERSION, String.class));
		issue.setMilestoneName(getObjectOfType(attributes, TracIssue.FIELD_MILESTONE, String.class));
		issue.setOwner(getObjectOfType(attributes, TracIssue.FIELD_OWNER, String.class));
		issue.setType(getObjectOfType(attributes, TracIssue.FIELD_TYPE, String.class));

		return issue;
	}

	@Nullable
	private static Date getNotMandatoryDateFromObject(@NotNull Object dateObject) throws IOException {
		if (dateObject instanceof Number) {
			return null;
		}

		return getParsedDateFromJSONObject(getJSONObjectNotNull(dateObject));
	}

	@Nullable
	private static Date getParsedDateFromJSONObject(@NotNull JSONObject jsonObject) throws IOException {
		final JSONArray dateArray = getObjectOfTypeNotNull(jsonObject, JSON_FIELD_JSONCLASS, JSONArray.class);
		final SimpleDateFormat dateFormat = new SimpleDateFormat(JSON_TIMESTAMP_PATTERN);
		String dateStr = getObjectOfTypeNotNull(dateArray.get(1), String.class);
		dateStr = dateStr.replace("T", "");

		// JSONRPC uses UTC as time zone
		dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));

		try {
			return dateFormat.parse(dateStr);
		}
		catch (java.text.ParseException e) {
			TracPlugin.LOGGER.error("Couldn't parse the date string '" + dateStr + "'.");
			return null;
		}
	}

	@NotNull
	private static TracIssue createTracIssue(@NotNull JSONObject jsonObject) throws IOException {
		return createTracIssue(getObjectOfTypeNotNull(jsonObject, JSON_FIELD_RESULT, JSONArray.class));
	}

	private static String createQueryMainParams(@Nullable String assignee, @NotNull List<TracIssueVersion> fixVersions,
	                                            @NotNull List<TracIssueMilestone> fixMilestones, @NotNull List<TracIssueStatus> states)
			throws UnsupportedEncodingException {
		final StringBuilder builder = new StringBuilder();

		builder.append(JSON_FIELD_ORDER);
		builder.append(JSON_OPTION_ASSIGNMENT);
		builder.append(JSON_FIELD_MILESTONE);

		if (assignee != null) {
			builder.append(JSON_OPTION_PREFIX);
			builder.append(TracIssue.FIELD_OWNER);
			builder.append(JSON_OPTION_ASSIGNMENT);
			builder.append(URLEncoder.encode(assignee, UTF8));
		}

		for (TracIssueStatus status : states) {
			builder.append(JSON_OPTION_PREFIX);
			builder.append(TracIssue.FIELD_STATUS);
			builder.append(JSON_OPTION_ASSIGNMENT);
			builder.append(status.toString());
		}

		for (TracIssueVersion fixVersion : fixVersions) {
			builder.append(JSON_OPTION_PREFIX);
			builder.append(TracIssue.FIELD_VERSION);
			builder.append(JSON_OPTION_ASSIGNMENT);

			if (fixVersion != null) {
				builder.append(URLEncoder.encode(fixVersion.getName(), UTF8));
			}
		}

		for (TracIssueMilestone fixMilestone : fixMilestones) {
			builder.append(JSON_OPTION_PREFIX);
			builder.append(TracIssue.FIELD_MILESTONE);
			builder.append(JSON_OPTION_ASSIGNMENT);

			if (fixMilestone != null) {
				builder.append(URLEncoder.encode(fixMilestone.getName(), UTF8));
			}
		}

		return builder.toString();
	}

	@NotNull
	private static JSONObject getJSONObjectNotNull(@NotNull Object object) throws IOException {
		return getObjectOfTypeNotNull(object, JSONObject.class);
	}

	@Nullable
	private static <CLASS> CLASS getObjectOfType(@NotNull JSONObject parent, @NotNull String key, @NotNull Class<CLASS> clazz) throws IOException {
		final Object obj = parent.get(key);
		if (obj == null) {
			return null;
		}

		return getObjectOfTypeNotNull(parent, key, clazz);
	}

	@NotNull
	private static <CLASS> CLASS getObjectOfTypeNotNull(@NotNull JSONObject parent, @NotNull String key, @NotNull Class<CLASS> clazz) throws IOException {
		final Object obj = parent.get(key);
		if (obj == null) {
			throw new IOException("Object '" + key + "' not found.");
		}

		if (!clazz.isInstance(obj)) {
			throw new IOException("Object '" + key + "' has unexpected type.");
		}

		//noinspection unchecked
		return ((CLASS)obj);
	}

	@NotNull
	private static <CLASS> CLASS getObjectOfTypeNotNull(@NotNull Object object, @NotNull Class<CLASS> clazz) throws IOException {
		if (!clazz.isInstance(object)) {
			throw new IOException("Object has unexpected type.");
		}

		//noinspection unchecked
		return ((CLASS)object);
	}
}