package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)BufferedDrawingCanvas.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This is just a DrawingCanvasComponent that uses double buffering for smooth
 * drawing.  This would probably be the typical approach if users are doing
 * anything update-intensive.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class BufferedDrawingCanvasComponent extends DrawingCanvasComponent implements ComponentListener {

	/**
	 * The image used for double buffering.
	 */
	transient protected Image offScreenImage;

	/**
	 * The graphics used for double buffering.
	 */
	transient protected Graphics offScreenGraphics;

	/**
	 * Constructs a new BufferedDrawingCanvasComponent.
	 */
	public BufferedDrawingCanvasComponent() {
		super();
		this.addComponentListener(this);
	}
	/**
	 * Constructs a new BufferedDrawingCanvasComponent, associating it with the given DrawingCanvas
	 * @see DrawingCanvas
	 *
	 * @param canvas the canvas this component will be associated with
	 */
	public BufferedDrawingCanvasComponent(DrawingCanvas canvas) {
		super(canvas);
		this.addComponentListener(this);
	}
	/**
	 * Sent when this component is hidden by something else. Currently does nothing.
	 *
	 * @param e the event which triggered this method
	 */
	public void componentHidden(ComponentEvent e) {}
	/**
	 * Sent when this component is moved. Currently does nothing.
	 *
	 * @param e the event which triggered this method
	 */
	public void componentMoved(ComponentEvent e) {}
	/**
	 * Sent when this component resized; resets the buffer.
	 *
	 * @param e the event which triggered this method
	 */
	public void componentResized(ComponentEvent e) {
		resetBuffer();
	}
	/**
	 * Sent when this component is shown. Currently does nothing.
	 *
	 * @param e the event which triggered this method
	 */
	public void componentShown(ComponentEvent e) {}
	/**
	 * Creates the buffer.
	 */
	public void createBuffer() {
		offScreenImage = createImage( getSize().width, getSize().height );
		offScreenGraphics = offScreenImage.getGraphics();
	}
	/** 
	 * Paints the component.
	 * @see #update
	 * 
	 * @param g the Graphics within which to paint.
	 */
	public void paint( Graphics g ) {
		if ( offScreenGraphics == null ) createBuffer();
		super.paint( offScreenGraphics );
		g.drawImage( offScreenImage, 0, 0, this );
	}
	/**
	 * Reset the buffer
	 */
	public void resetBuffer() {
		offScreenImage = null;
		offScreenGraphics = null;
	}
	/** 
	 * Updates the component. This method is called in
	 * response to a call to repaint. You can assume that
	 * the background is not cleared.
	 * @see #paint
	 * 
	 * @param g the Graphics object within which to update
	 */
	public void update( Graphics g ) {
		if ( offScreenGraphics == null ) createBuffer();
		paint( g );
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)CanvasPalette.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.awt.event.*;
import com.rolemodelsoft.drawlet.*;

/**
 * Represents a set of objects, tied to some visual representation
 * and a method for user invocation, associated with a
 * <code>DrawingCanvas</code> which they act on.
 *
 * @version 	1.1.6, 12/28/98
 */

 public abstract class CanvasPalette extends Panel implements ActionListener {
	/**
	 * The canvas on which the tools operate.
	 */
	protected DrawingCanvas canvas;

	/**
	 * Called if an action occurs in the receiver.
	 *
	 * @param evt the event which triggered this method
	 */
	public abstract void actionPerformed(ActionEvent evt);
	/**
	 * Add a button to the receiver, assigning it the given name.
	 * 
	 * @param label the name to give the <code>Button</code>
	 * @return	the <code>Button</code> created
	 */
	protected Button addButton(String label) {
		Button button = new Button(label);
		button.addActionListener(this);
		add(button);
		return button;
	}
	/**
	 * Answer the <code>DrawingCanvas</code> to which we are applying tools.
	 * 
	 * @return	the DrawingCanvas to which we are applying tools
	 */
	public DrawingCanvas getCanvas() {
		return canvas;
	}
	/**
	 * Answer the index of the given <code>Component</code>.  
	 * 
	 * @param comp the <code>Component</code> we're looking for
	 * @return an integer representing the index of the <code>Component</code>;
	 * -1 if the <code>Component</code> is not found.
	 */
	protected int indexOf(Component comp) {
		Component myComponents[] = getComponents();
		for (int i=0; i < myComponents.length; i++) 
			if (comp == myComponents[i]) return i;
		return -1; //we don't expect this to happen
	}
	/**
	 * Set the <code>DrawingCanvas</code> to which we are applying tools.
	 *
	 * @param canvas the <code>DrawingCanvas</code> to which tools will be applied
	 */
	public void setCanvas(DrawingCanvas newCanvas) {
		this.canvas = newCanvas;
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)ImageButton.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

/**
 * A button which uses an image instead of a label.
 *
 * @version 	1.1.6, 12/30/98
 */

public class ColorButton extends Component implements MouseListener {
	/**
	 * The label (not visible) for this button.
	 */
	protected String command = getDefaultCommand();

	/**
	 * The image for this button.
	 */
	protected Color color = getDefaultColor();

	/**
	 * The actionListener member variable, for dispatching events.
	 */
	protected ActionListener actionListener = null;

	 /**
	  * Flag that is set when the mouse is pressed inside.
	  */
	protected boolean isPressed = false;

	 /**
	  * Flag that is set when the mouse is inside
	  */
	protected boolean isInside = false;

	 /**
	  * Flag indicating the current value of highlighting.
	  */
	protected boolean isHighlighted = false;
	/**
	 * Create a new, default <code>ColorButton</code>
	 */
	public ColorButton() {
		super();
		addMouseListener( this );
	}
	/**
	 * Create a new <code>ColorButton</code> initialized with the given color.
	 *
	 * @param color the color to be used.
	 */
	public ColorButton( Color color ) {
		this();
		this.color = color;
	}
	/**
	 * Create a new <code>ColorButton</code> initialized with the given command.
	 *
	 * @param command the command that will be passed in action events.
	 */
	public ColorButton( String command ) {
		this();
		this.command = command;
	}
	/**
	 * Create a new <code>ColorButton</code> initialized with the given command and color.
	 *
	 * @param command the command that will be passed in action events.
	 * @param color the color to be used.
	 */
	public ColorButton( String command, Color color ) {
		this();
		this.command = command;
		this.color = color;
}
	/**
	 * Add the given <code>ActionListener</code> to my set of listeners.
	 *
	 * @param listener the <code>ActionListener</code> to add. 
	 */
	public void addActionListener( ActionListener listener ) {
		actionListener = AWTEventMulticaster.add( actionListener, listener );
	}
	/**
	 * Answers the <code>Color</code> this button represents and displays.
	 * 
	 * @return the Color this button holds.
	 */
	public Color getColor() {
		return color;
	}
	/**
	 * Answer the command associated with this receiver.
	 * 
	 * @return a String representing the command passed in action events generated
	 * by this button.
	 */
	public String getCommand() {
		return command;
	}
	/**
	 * Answers the default color for the receiver (intended for initialization).
	 * 
	 * @return the default Color for this button to use
	 */
	protected Color getDefaultColor() {
		return Color.white;
	}
	/**
	 * Answers the default command for the receiver (intended for initialization).
	 * 
	 * @return a String representing the default command to use.
	 */
	protected String getDefaultCommand() {
		return "ColorButton";
	}
	/**
	 * Answer the size this color button must be displayed at.
	 *
	 * @return an integer representing the minimum size for this color button
	 */
	public Dimension getMinimumSize() {
		return new Dimension( 16, 16 );
	}
	/**
	 * Answer the size this color button prefers to be displayed at.
	 *
	 * @return an integer representing the preferred size for this color button
	 */
	public Dimension getPreferredSize() {
		return getMinimumSize();
	}
	/**
	 * Answer whether or not the receiver is to be drawn highlighted.
	 * 
	 * @return	a boolean value of <code>true</code> if the receiver is to be drawn highlighted;
	 *			or <code>false</code> otherwise.
	 */
	public boolean isHighlighted() {
		return isHighlighted;
	}
	/**
	 * Mouse events are handled in <code>MousePressed</code> and <code>MouseReleased</code>.
	 * 
	 * @param e the event.
	 */
	public void mouseClicked( MouseEvent e ) {
	}
	/**
	 * If the mouse moves inside of the button, set the <code>isInside</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mouseEntered( MouseEvent e ) {
		isInside = true;
	}
	/**
	 * If the mouse moves outside of the button, reset the <code>isInside</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mouseExited( MouseEvent e ) {
		isInside = false;
	}
	/**
	 * If the mouse is pressed inside of the button, set the <code>isPressed</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mousePressed( MouseEvent e ) {
		if ( isInside ) {
			isPressed = true;
		}
	}
	/**
	 * If the mouse is released, check to see if it is inside of the button, and if it was
	 * originally pressed inside of the button. If it was, it was an action event, so call
	 * <code>processActionEvent()</code>. Then reset the <code>isPressed</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mouseReleased( MouseEvent e ) {
		if ( isInside && isPressed ) {
			processActionEvent();
		}
		isPressed = false;
	}
	/**
	 * Paint the button.
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paint( Graphics g ) {
		if ( isHighlighted() )
			paintHighlighted( g );
		else
			paintUnhighlighted( g );
	}
	/**
	 * Paint the button highlight.
	 * 
	 * @param g the graphics object to use in painting.
	 * @param firstColor the <code>Color</code> to paint the top and left-hand sides.
	 * @param secondColor the <code>Color</code> to paint the bottom and right-hand sides.
	 */
	public void paintHighlight( Graphics g, Color firstColor, Color secondColor ) {
		int rightX = getSize().width - 1;
		int bottomY = getSize().height - 1;
		g.setColor( firstColor );
		g.drawLine( 0, 0, rightX, 0 );
		g.drawLine( 0, 0, 0, bottomY );
		g.setColor( secondColor );
		g.drawLine( 0, bottomY, rightX, bottomY );
		g.drawLine( rightX, 0, rightX, bottomY );
	}
	/**
	 * Paint the button highlighted (down).
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paintHighlighted( Graphics g ) {
		paintHighlight( g, Color.darkGray, Color.white );
		if ( color == null ) {
			g.setColor( Color.black );
			paintLabel( g, 1 );
		} else {
			g.setColor( color );
			g.fillRect( 3, 3, getSize().width - 4, getSize().height - 4 );
		}
	}
	/**
	 * Paint the label form, offsetting it by the given amount.
	 * 
	 * @param g the graphics object to use in painting.
	 * @param moveBy an integer representing the amount to offset the label by.
	 */
	public void paintLabel(Graphics g, int moveBy) {
		Dimension dimension = getSize();
		FontMetrics metrics = g.getFontMetrics();
		int width = metrics.stringWidth(command);
		int height = metrics.getHeight();
		g.drawString(command, ( (dimension.width - width) / 2 ) + moveBy, ( (dimension.height - height) / 2 + metrics.getAscent() ) + moveBy );
	}
	/**
	 * Paint the button unhighlighted (not down).
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paintUnhighlighted( Graphics g ) {
		paintHighlight( g, Color.white, Color.darkGray );
		if ( color == null ) {
			g.setColor( Color.black );
			paintLabel( g, 0 );
		} else {
			g.setColor( color );
			g.fillRect( 2, 2, getSize().width - 4, getSize().height - 4 );
		}
	}
	/**
	 * Create an <code>ActionEvent</code> and pass it to everyone listening for it.
	 */
	public void processActionEvent() {
		if ( actionListener != null ) {
			actionListener.actionPerformed(
				new ActionEvent( this, ActionEvent.ACTION_PERFORMED, command )
			);
		}
	}
	/**
	 * Remove the given <code>ActionListener</code> from my set of listeners.
	 * 
	 * @param listener the <code>ActionListener</code> to remove. 
	 */
	public void removeActionListener( ActionListener listener ) {
		actionListener = AWTEventMulticaster.remove( actionListener, listener );
	}
	/**
	 * Set the <code>Color</code> this button is associated with.
	 * 
	 * @param color the color this button should use.
	 */
	public void setColor( Color color ) {
		this.color = color;
	}
	/**
	 * Set the command associated with this receiver.
	 * 
	 * @param command the command to be passed in action events generated.
	 * by this button.
	 */
	public void setCommand( String command ) {
		this.command = command;
	}
	/**
	 * Set whether or not the receiver is to be drawn highlighted.
	 * 
	 * @param highlighted <code>true</code> if the button should be drawn highlighted;
	 * <code>false</code> otherwise.
	 */
	public void setHighlighted( boolean highlighted ) {
		this.isHighlighted = highlighted;
		repaint();
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)DrawingCanvasComponent.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.util.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.awt.datatransfer.*;

import java.beans.*;

/**
 * This provides basic functionality necessary to provide a meaningful
 * working version of a DrawingCanvas that can be tied to a Component in an AWT
 * Application.  It is expected that this would serve as the base for these sort
 * of Components, but not required.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class DrawingCanvasComponent extends Canvas {
	
	/**
	 * The drawing canvas we are displaying/manipulating.
	 */
	protected DrawingCanvas canvas;
	
	/**
	 * Create a default <code>DrawingCanvasComponent</code>.
	 */
	public DrawingCanvasComponent() {
		this(new SimpleDrawingCanvas());
	}
	/**
	 * Constructs a new <code>DrawingCanvasComponent</code>, associating
	 * it with the given <code>DrawingCanvas</code>
	 *
	 * @param canvas the canvas this component will be associated with.
	 */
	public DrawingCanvasComponent(DrawingCanvas canvas) {
		super();
		this.setCanvas(canvas);
		if (canvas instanceof MouseListener)
			this.addMouseListener((MouseListener)canvas);
		if (canvas instanceof MouseMotionListener)
			this.addMouseMotionListener((MouseMotionListener)canvas);
		if (canvas instanceof KeyListener)
			this.addKeyListener((KeyListener)canvas);
	}
	/**
	 * @return the DrawingCanvas we are drawing/manipulating
	 */
	public DrawingCanvas getCanvas() {
		return canvas;
	}
	/** 
	 * Gets the preferred size of the receiver.
	 *
	 * @return a dimension object indicating the receiver's preferred size.
	 */
	public Dimension getPreferredSize() {
		return getCanvas().getBounds().getSize();
	}
	/**
	 * Tell the AWT I am able to receive the focus.
	 *
	 * @return boolean value of <code>true</code> if the
	 * receiver is focus traversable;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isFocusTraversable() {
		return true;
	}
	/** 
	 * Paints the receiver.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g) {
		canvas.paint(g);
	}
	/** 
	 * Repaints part of the receiver. This will result in a
	 * call to update as soon as possible.
	 * 
	 * @param rectangle is the region to be repainted
	 */
	public void repaint(Rectangle rectangle) {
		/*
		 * Note that we fudge by a pixel to avoid
		 * differences in the way various drawing primitives determine 
		 * where to start/stop drawing between lines and fills.
		 */
		repaint(rectangle.x, rectangle.y, rectangle.width + 1, rectangle.height + 1);
	}
	/**
	 * Set the <code>DrawingCanvas</code> the receiver is associated with.
	 *
	 * @param canvas the DrawingCanvas we should draw/manipulate
	 */
	protected void setCanvas(DrawingCanvas canvas) {
		if (canvas instanceof ComponentHolder)
			((ComponentHolder)canvas).setComponent(this);
		this.canvas = canvas;
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)ImageButton.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

/**
 * A button which uses an image instead of a label.
 *
 * @version 	1.1.6, 12/30/98
 */

public class ImageButton extends Component implements MouseListener {
	/**
	 * The label (not visible) for this button.
	 */
	protected String command = getDefaultCommand();

	/**
	 * The image for this button.
	 */
	protected Image image = getDefaultImage();

	/**
	 * The actionListener member variable, for dispatching events.
	 */
	protected ActionListener actionListener = null;

	 /**
	  * Flag that is set when the mouse is pressed inside.
	  */
	protected boolean isPressed = false;

	 /**
	  * Flag that is set when the mouse is inside
	  */
	protected boolean isInside = false;

	 /**
	  * Flag that is set when the button should be highlighted.
	  */
	protected boolean isHighlighted = false;
	/**
	 * Create a new, default <code>ImageButton</code>.
	 */
	public ImageButton() {
		super();
		addMouseListener( this );
	}
	/**
	 * Create a new <code>ImageButton</code> and initalize it with the given <code>Image</code>.
	 *
	 * @param image the image to be used.
	 */
	public ImageButton( Image image ) {
		this();
		this.image = image;
		waitForImage( this, image );
	}
	/**
	 * Create a new <code>ImageButton</code> and initalize it with the given command.
	 *
	 * @param command the command that will be passed in action events.
	 */
	public ImageButton( String command ) {
		this();
		this.command = command;
	}
	/**
	 * Create a new <code>ImageButton</code> and initalize it with the
	 * given command and <code>Image</code>.
	 *
	 *
	 * @param command the command that will be passed in action events.
	 * @param image the image to be used.
	 */
	public ImageButton( String command, Image image ) {
		this();
		this.command = command;
		this.image = image;
		waitForImage( this, image );
}
	/**
	 * Add the given <code>ActionListener</code> to my set of listeners.
	 *
	 * @param listener the action listener to add. 
	 */
	public void addActionListener( ActionListener listener ) {
		actionListener = AWTEventMulticaster.add( actionListener, listener );
	}
	/**
	 * Answer the command associated with this receiver.
	 * 
	 * @return a String representing the command passed in action events generated
	 * by this button.
	 */
	public String getCommand() {
		return command;
	}
	/**
	 * Answers the default command for the receiver (intended for initialization).
	 * 
	 * @return a String representing the default command to use.
	 */
	protected String getDefaultCommand() {
		return "ImageButton";
	}
	/**
	 * Answers the default image for the receiver (intended for initialization.
	 * If this is changed to something other than null, it will also need to
	 * wait for the image.
	 * 
	 * @return the default Image for this button to use
	 */
	protected Image getDefaultImage() {
		return null;
	}
	/**
	 * Answers the current image this button is associated with.
	 * 
	 * @return the Image this button is using.
	 */
	public Image getImage() {
		return image;
	}
	/**
	 * Answer the size this <code>ImageButton</code> must be displayed at.
	 *
	 * @return an integer representing the minimum size for this image button
	 */
	public Dimension getMinimumSize() {
		return new Dimension(
			image.getWidth( this ) + 4,
			image.getHeight( this ) + 4
		);
	}
	/**
	 * Answer the size this <code>ImageButton</code> prefers to be displayed at.
	 *
	 * @return an integer representing the preferred size for this <code>ImageButton</code>.
	 */
	public Dimension getPreferredSize() {
		return getMinimumSize();
	}
	/**
	 * Returns whether this button is highlighted (down) or not.
	 * 
	 * @return a boolean representing the button's current
	 * highlight state.
	 */
	public boolean isHighlighted() {
		return isHighlighted;
	}
	/**
	 * Mouse 'action' events are handled in <code>MousePressed</code> and <code>MouseReleased</code>.
	 * 
	 * @param e the event.
	 */
	public void mouseClicked( MouseEvent e ) {
	}
	/**
	 * If the mouse moves inside of the button, set the <code>isInside</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mouseEntered( MouseEvent e ) {
		isInside = true;
		if ( isPressed )
			repaint();
	}
	/**
	 * If the mouse moves outside of the button, reset the <code>isInside</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mouseExited( MouseEvent e ) {
		isInside = false;
		if ( isPressed )
			repaint();
	}
	/**
	 * If the mouse is pressed inside of the button, set the <code>isPressed</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mousePressed( MouseEvent e ) { 
		if ( isInside ) {
			isPressed = true;
			repaint();
		}
	}
	/**
	 * If the mouse is released, check to see if it is inside of the button, and if it was
	 * originally pressed inside of the button. If it was, it was an action event, so call
	 * <code>processActionEvent()</code>. Then reset the <code>isPressed</code> flag.
	 * 
	 * @param e the event.
	 */
	public void mouseReleased( MouseEvent e ) {
		if ( isInside && isPressed ) {
			processActionEvent();
		}
		isPressed = false;
		repaint();
	}
	/**
	 * Paint the button.
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paint( Graphics g ) {
		if ( isHighlighted() || ( isPressed && isInside ) )
			paintHighlighted( g );
		else
			paintUnhighlighted( g );
	}
	/**
	 * Paint the button highlight.
	 * 
	 * @param g the graphics object to use in painting.
	 * @param firstColor the <code>Color</code> to paint the top and left-hand sides.
	 * @param secondColor the <code>Color</code> to paint the bottom and right-hand sides.
	 */
	public void paintHighlight( Graphics g, Color firstColor, Color secondColor ) {
		int rightX = getSize().width - 1;
		int bottomY = getSize().height - 1;
		g.setColor( firstColor );
		g.drawLine( 0, 0, rightX, 0 );
		g.drawLine( 0, 0, 0, bottomY );
		g.setColor( secondColor );
		g.drawLine( 0, bottomY, rightX, bottomY );
		g.drawLine( rightX, 0, rightX, bottomY );
	}
	/**
	 * Paint the button highlighted (down).
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paintHighlighted( Graphics g ) {
		paintHighlight( g, Color.darkGray, Color.white );
		g.drawImage( image, 3, 3, this );
	}
	/**
	 * Paint the button unhighlighted (not down).
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paintUnhighlighted( Graphics g ) {
		paintHighlight( g, Color.white, Color.darkGray );
		g.drawImage( image, 2, 2, this );
	}
	/**
	 * Create an <code>ActionEvent</code> and pass it to everyone listening for it.
	 */
	public void processActionEvent() {
		if ( actionListener != null ) {
			actionListener.actionPerformed(
				new ActionEvent( this, ActionEvent.ACTION_PERFORMED, command )
			);
		}
	}
	/**
	 * Remove the given <code>ActionListener</code> from my set of listeners.
	 * 
	 * @param l ActionListener
	 */
	public void removeActionListener( ActionListener listener ) {
		actionListener = AWTEventMulticaster.remove( actionListener, listener );
	}
	/**
	 * Set the command associated with the receiver.
	 * 
	 * @param command the command to be passed in action events generated.
	 * by this button.
	 */
	public void setCommand( String command ) {
		this.command = command;
	}
	/**
	 * Set whether this button is highlighted (down) or not.
	 * 
	 * @param isHighlighted a boolean representing the button's
	 * new highlight state.
	 */
	public void setHighlight( boolean isHighlighted ) {
		this.isHighlighted = isHighlighted;
		repaint();
	}
	/**
	 * Set the <code>Image</code> this button is associated with.
	 * 
	 * @param image the image this button should use.
	 */
	public void setImage( Image image ) {
		this.image = image;
		waitForImage( this, image );
	}
	/**
	 * Wait for the image to be available.
	 *
	 * @param component the component the image is associated with.
	 * @param image the image to wait for.
	 */
	protected static void waitForImage( Component component, Image image ) {
		MediaTracker tracker = new MediaTracker( component );
		try {
			tracker.addImage( image, 0 );
			tracker.waitForID( 0 );
		}
		catch ( InterruptedException e ) {
			e.printStackTrace();
		}
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)PaintableViewer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import com.rolemodelsoft.drawlet.*;

/**
 * A component which can show any Paintable.
 *
 * @version 	1.1.6, 12/30/98
 */
public class PaintableViewer extends Component {
	protected Paintable paintable;
	/**
	 * Create a <code>PaintableViewer</code>.
	 */
	public PaintableViewer() {
		super();
	}
	/**
	 * Create a <code>PaintableViewer</code> initially showing the given <code>Paintable</code>.
	 *
	 * @param paintable the thing to view
	 */
	public PaintableViewer(Paintable paintable) {
		super();
		this.paintable = paintable;
	}
	/**
	 * Answer the size the receiver must be displayed at.
	 *
	 * @return an integer representing the minimum size for the receiver.
	 */
	public Dimension getMinimumSize() {
		if (paintable == null)
			return super.getMinimumSize();
		else {
			Rectangle rect = (new Rectangle(0,0)).union(paintable.getBounds());
			rect.grow(10, 10);
			return rect.getSize();
		}
	}
	/**
	 * @return the paintable assocatied with the receiver.
	 */
	public Paintable getPaintable() {
		return paintable;
	}
	/**
	 * Answer the size the receiver prefers to be displayed at.
	 *
	 * @return an integer representing the preferred size for the receiver
	 */
	public Dimension getPreferredSize() {
		if (paintable == null)
			return super.getPreferredSize();
		else {
			Rectangle rect = (new Rectangle(0,0)).union(paintable.getBounds());
			rect.grow(10, 10);
			return rect.getSize();
		}
	}
	/**
	 * Paint the thing we're viewing.
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paint( Graphics g ) {
		paintBackground(g);
		if (paintable != null)
			paintable.paint(g);
	}
	/**
	 * Paint the background of the component.
	 * 
	 * @param g the graphics object to use in painting.
	 */
	public void paintBackground( Graphics g ) {
		Dimension dimension = getSize();
		g.setColor( getBackground() );
		g.fillRect( 0, 0, dimension.width, dimension.height );
	}
	/**
	 * Set the thing we are viewing and repaint.
	 *
	 * @param paintable the <code>Paintable</code> to be associated with this viewer.
	 */
	public void setPaintable(Paintable paintable) {
		this.paintable = paintable;
		repaint();
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)StylePalette.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.lang.reflect.*;
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.SimpleDrawing;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleShape;
import com.rolemodelsoft.drawlet.shapes.lines.Line;
import com.rolemodelsoft.drawlet.text.TextLabel;

/**
 * Represents a set of objects, including ColorButtons,
 * Setters, and a StyleViewer, associated with a
 * <code>DrawingCanvas</code> which they act on.
 *
 * @version 	1.1.6, 12/28/98
 */

 public class StylePalette extends CanvasPalette {

	/**
	 * The list of tools that correspond to the color buttons.
	 */
	protected Vector colors = new Vector(5);

	/**
	 * The currently selected color.
	 */
	protected Color color = Color.black;

	/**
	 * The list of tools that correspond to the setter buttons.
	 */
	protected Vector setters = new Vector(5);

	/**
	 * The currently selected colorButton.
	 */
	protected ColorButton currentColorButton;

	/**
	 * A flag tracking whether there are any color buttons yet or not.
	 */
	protected boolean areButtons;

	/**
	 * The drawing associated with this palette.
	 */
	protected Drawing drawing;

	/**
	 * The viewer which identifies the current style.
	 */
	protected Component viewer;
	/** 
	 * Creates a new <code>StylePalette</code> and associates it
	 * with the given <code>DrawingCanvas</code>.
	 *
	 * @param canvas the canvas on which to apply tools.
	 */
	public StylePalette(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
	/**
	 * Called if an action occurs in the palette.
	 * Set the style of the canvas as appropriate.
	 *
	 * @param evt the event
	 */
	public void actionPerformed(ActionEvent evt) {
		int index = indexOf((Component) evt.getSource());
		int numColors = colors.size();
		if (index < numColors) {
			color = (Color) colors.elementAt(index);
			currentColorButton.setHighlighted( false );
			currentColorButton = (ColorButton)evt.getSource();
			currentColorButton.setHighlighted( true );
			return;
		}
		index = index - numColors;
		int numSetters = setters.size();
		DrawingStyle style = canvas.getStyle();
		if (index < numSetters) {
			Method setter = (Method) setters.elementAt(index);
			try {
				setter.invoke(style, new Object[] {color});
				updateDrawing();
				updateViewer();
			} catch (Exception e) {
				System.out.println("Exception " + e + " when invoking " + setter.getName());
			}
			canvas.setStyle(style);
			return;
		}
		Figure[] figures = canvas.getSelections();
		for (int i = 0; i < figures.length; i++) {
			figures[i].setStyle(style);
		}
		//canvas.repaint();
		return;
	}
	/**
	 * Add an applier to the receiver, labeling it appropriately.
	 * 
	 * @param label the label to associate with the applier
	 */
	public void addApply(String label) {
		Button button = new Button(label);
		button.addActionListener(this);
		GridBagConstraints constraints = new GridBagConstraints();
		constraints.gridx = 0;
		constraints.gridy = ((colors.size() - ( colors.size() & 1 )) / 2) + setters.size();
		constraints.gridwidth = 2;
		constraints.fill = GridBagConstraints.BOTH;
		constraints.weightx = 1.0;
		constraints.weighty = 1.0;
		constraints.insets = new Insets( 2, 2, 2, 0 );
		add(button, constraints);
	}
	/**
	 * Add a <code>Color</code> to the receiver, with
	 * the given label. Basically, this creates a new
	 * <code>ColorButton</code> with the given <code>Color</code>
	 * and label.
	 * 
	 * @param color the color to add
	 * @param label the label to associate with the color
	 */
	public void addColor(Color color, String label) {
		colors.addElement(color);
		ColorButton button = new ColorButton(label, color);
		button.addActionListener(this);
		GridBagConstraints constraints = new GridBagConstraints();
		int colorIndex;
		if ( color == null ) colorIndex = colors.size() - 1;
		else colorIndex = colors.indexOf( color );
		constraints.gridx = colorIndex & 1;
		constraints.gridy = (colorIndex - (colorIndex & 1)) / 2;
		constraints.fill = GridBagConstraints.BOTH;
		constraints.insets = new Insets( 2, 2, 2, 2 );
		constraints.weightx = 1;
		constraints.weighty = 1;
		add(button, constraints);
		if ( ! areButtons ) {
			currentColorButton = button;
			button.setHighlighted( true );
			areButtons = true;
		}
	}
	/**
	 * Add a setter to the receiver, labeling it appropriately.
	 * 
	 * @param setter the setter to add
	 * @param label the label to associate with the setter
	 */
	public void addSetter(String setter, String label) {
		Method method = null;
		try {
			method = canvas.getStyle().getClass().getMethod(setter, new Class[] {Class.forName("java.awt.Color")});
			setters.addElement(method);
		} catch (Exception e) {System.out.println("can't find method " + setter + " or class Color... this should never happen!");}
		Button button = new Button(label);
		button.addActionListener(this);
		GridBagConstraints constraints = new GridBagConstraints();
		constraints.gridx = 0;
		constraints.gridy = ((colors.size() - ( colors.size() & 1 )) / 2) + setters.indexOf( method );
		constraints.gridwidth = 2;
		constraints.fill = GridBagConstraints.BOTH;
		constraints.weightx = 1.0;
		constraints.weighty = 1.0;
		constraints.insets = new Insets( 2, 2, 2, 0 );
		add(button, constraints);
	}
	/**
	 * Add a viewer to the palette to indicate the current style.
	 */
	public void addStyleViewer() {
		drawing = new SimpleDrawing();
		drawing.setStyle(canvas.getStyle());
		drawing.addFigure(new RectangleShape(5,5,25,15));
		drawing.addFigure(new Line(0,0,15,10));
		Figure label = new TextLabel("text");
		label.move(5,5);
		drawing.addFigure(label);
		updateDrawing();
		viewer = new PaintableViewer((Paintable)drawing);
		viewer.setBackground(canvas.getStyle().getBackgroundColor());
		GridBagConstraints constraints = new GridBagConstraints();
		constraints.gridx = 0;
		constraints.gridwidth = 2;
		constraints.insets = new Insets(4,4,4,4);
		constraints.fill = GridBagConstraints.BOTH;
		constraints.weightx = 1.0;
		constraints.weighty = 1.0;
		add(viewer,constraints);
	}
	/**
	 * Make sure the drawing indicates the current style, if we have one.
	 */
	public void updateDrawing() {
		if (drawing == null)
			return;
		drawing.setStyle(canvas.getStyle());
		for (FigureEnumeration e = drawing.figures(); e.hasMoreElements(); ) 
			e.nextElement().setStyle(canvas.getStyle());
	}
	/**
	 * Make sure the viewer shows the current style, if we are showing one.
	 */
	public void updateViewer() {
		if (viewer == null)
			return;
		viewer.setBackground(canvas.getStyle().getBackgroundColor());
		viewer.repaint();
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)ToolBar.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * Represents a set of Tools, associated with a
 * <code>DrawingCanvas</code> which they act on.
 */
 
public class ToolBar extends CanvasPalette {
	/**
	 * The list of tools that correspond to the setter buttons.
	 */
	protected Vector buttons = new Vector(4);
	/** 
	 * Creates a new <code>ToolBar</code> and assocaties it
	 * with the given <code>DrawingCanvas</code>.
	 *
	 * @param canvas the <code>DrawingCanvas</code> to
	 * associate with the <code>ToolBar</code>.
	 */
	public ToolBar(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
	/**
	 * Called if an action occurs in the toolbar.
	 *
	 * @param evt the event
	 */
	public void actionPerformed(java.awt.event.ActionEvent evt) {
		int index = indexOf((Component) evt.getSource());
		int numButtons = buttons.size();
		if (index < numButtons) {
			Method button = (Method) buttons.elementAt(index);
			try {
				button.invoke( canvas, new Class[] {} );
			} catch (Exception e) {
				System.out.println("Exception " + e + " when invoking " + button.getName());
			}
			return;
		}
	}
	/**
	 * Add a tool to the receiver, associating it with the given <code>Image</code>.
	 * 
	 * @param tool the tool to add
	 * @param image the <code>Image</code> to associate with the tool
	 */
	public void addButton(String tool, Image image) {
		Method method = null;
		try {
			method = canvas.getClass().getMethod(tool, new Class[0]);
			buttons.addElement(method);
		} catch (Exception e) {System.out.println("can't find method " + tool + " or the parameters");}
		ImageButton button = new ImageButton(image);
		button.addActionListener(this);
		GridBagConstraints constraints = new GridBagConstraints();
		constraints.gridx = buttons.indexOf( method ) + 1;
		constraints.gridy = 1;
		constraints.fill = GridBagConstraints.NONE;
		constraints.weightx = 1.0;
		constraints.weighty = 1.0;
		constraints.anchor = GridBagConstraints.WEST;
		constraints.insets = new Insets( 2, 4, 2, 0 );
		add(button, constraints);
	}
}
package com.rolemodelsoft.drawlet.awt;

/**
 * @(#)ToolPalette.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Enumeration;

/**
 * Although there are multiple ways a DrawingCanvas can get its tool(s), a typical
 * approach is providing a palette from which to choose them.  This class provides
 * a very simple version of one.  There is obviously room for improvement (e.g.
 * tool tips, etc.).
 * NOTE: Right now, the layout managers are kind of mixed up. They should all be
 * standardized (probably to GridBag) and tested to ensure that icons and labels can
 * be interchanged and still look good.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class ToolPalette extends CanvasPalette {

	/**
	 * The list of tools that correspond to the buttons.
	 */
	protected Vector tools = new Vector(5);

	/**
	 * The last button invoked on this ToolPalette
	 */
	protected Component lastButton;
	/** 
	 * Creates a new <code>ToolPalette</code> and associates
	 * it with the given <code>DrawingCanvas<c/ode>.
	 *
	 * @param canvas the canvas on which to apply tools.
	 */
	public ToolPalette(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
	/**
	 * Called if an action occurs in the palette.
	 * Set the tool of the canvas as appropriate.
	 *
	 * @param evt the event
	 */
	public void actionPerformed(ActionEvent evt) {
		if ( lastButton instanceof ImageButton )
			((ImageButton)lastButton).setHighlight( false );

		if ( evt.getSource() instanceof ImageButton )
			((ImageButton)evt.getSource()).setHighlight( true );

		lastButton = (Component)evt.getSource();
		canvas.setTool((InputEventHandler) tools.elementAt(indexOf((Component) evt.getSource())));
	}
	/**
	 * Add a tool to the receiver, labeling it appropriately.
	 * 
	 * @param tool the tool to add
	 * @param label the label to associate with the tool
	 */
	public void addTool(InputEventHandler tool, String label) {
		addToolButton(tool, label);
	}
	/**
	 * Add a tool to the receiver, associating it with
	 * the given label and <code>Image</code>.
	 *
	 * @param tool the EventHandler to add
	 * @param label the label to associate with the tool
	 * @param image the <code>Image</code> to associate with the tool
	 */
	public void addTool(InputEventHandler tool, String label, Image image) {
		tools.addElement(tool);
		ImageButton b = new ImageButton(label, image);
		b.addActionListener( this );
		GridBagConstraints constraints = new GridBagConstraints();
		constraints.gridx = 1;
		constraints.weighty = 1.0;
		constraints.insets = new Insets( 0, 2, 2, 2 );
		add(b, constraints);
	}
	/**
	 * Add a tool to the receiver in Button form, labeling it appropriately.
	 * 
	 * @param tool the tool to add
	 * @param label the label to associate with the tool
	 */
	protected Button addToolButton(InputEventHandler tool, String label) {
		tools.addElement(tool);
		return addButton(label);
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)AbstractFigure.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import com.rolemodelsoft.drawlet.util.Duplicatable;

/**
 * This provides basic default functionality for Figures that are assumed to be
 * movable but not necessarily reshapable, and that have observers that want to
 * know when their location changes.  It provides very basic functionality for most operations
 * and forces concrete subclasses to define, at a minimum:
 *	paint(Graphics);
 *	getBounds();
 *	basicTranslate(int,int);
 *
 * @version 	1.1.6, 12/28/98
 */
 
public abstract class AbstractFigure extends AbstractPaintable implements Figure {
	/**
	 * The <code>Figure's</code> listeners.
	 */
	protected transient Vector listeners;

	/**
	 * The <code>Figure's</code> location listeners.
	 */
	protected Vector locationListeners;
	/**
	 * Add a PropertyChangeListener to the listener list.
	 *
	 * @param listener  The PropertyChangeListener to be added
	 */

	public synchronized void addPropertyChangeListener(PropertyChangeListener listener) {
		if (listeners == null) {
		    listeners = new Vector();
		}
		if ( ! listeners.contains( listener ) ) {
			listeners.addElement(listener);
		}
	}
	/**
	 * Add a RelatedLocationListener to the listener list.
	 *
	 * @param listener  The RelatedLocationListener to be added
	 */

	public synchronized void addRelatedLocationListener(RelatedLocationListener listener) {
		if (locationListeners == null) {
		    locationListeners = new Vector();
		}
		if ( ! locationListeners.contains( listener ) ) {
			locationListeners.addElement(listener);
		}
	}
	/** 
	 * Moves the Figure to a new location. The x and y coordinates
	 * are in the parent's coordinate space.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @see #location
	 * @see #reshape
	 */
	protected void basicMove(int x, int y) {
		basicTranslate(x - getLeft(), y - getTop());
	}
	/** 
	 * Moves the Figure in the x and y direction.
	 * Subclasses should probably provide synchronized versions if they're 
	 * modifying attributes of the receiver.
	 *
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 * @see #location
	 * @see #translate
	 */
	protected abstract void basicTranslate(int x, int y);
	/**
	 * Denote that location changed.
	 *
	 * @param point the old location.
	 */
	protected void changedLocation(Point oldPoint) {
		Point newPoint = getLocation();
		if (oldPoint != null && oldPoint.equals(newPoint)) {
		    return;
		}
		PropertyChangeEvent evt = new PropertyChangeEvent(this,
						    LOCATION_PROPERTY, oldPoint, newPoint);
		fireLocationChange(evt);
		firePropertyChange(evt);
	}
	/**
	 * Denote that the shape changed. Assume that
	 * the shape has changed even if the bounds haven't.
	 * 
	 * @param oldBounds the old bounds.
	 */
	protected void changedShape(Rectangle oldBounds) {
		Rectangle newBounds = getBounds();
		PropertyChangeEvent evt = new PropertyChangeEvent(this,
						    SHAPE_PROPERTY, oldBounds, newBounds);
		fireShapeChange(evt);
		firePropertyChange(evt);
	}
	/**
	 * Denote that size changed.
	 * @param oldDimension the old dimensions.
	 */
	protected void changedSize(Dimension oldDimension) {
		Dimension newDimension = getSize();
		if (oldDimension != null && oldDimension.equals(newDimension)) {
		    return;
		}
		PropertyChangeEvent evt = new PropertyChangeEvent(this,
						    SIZE_PROPERTY, oldDimension, newDimension);
		fireSizeChange(evt);
		firePropertyChange(evt);
	}
	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * Figure. <code>x</code> and <code>y</code> are defined to be relative to the 
	 * coordinate system of this figure.
	 * By default, just check if it is within the receiver's bounds.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified x,y
	 * location is "inside" this Figure;
	 * 			<code>false</code> otherwise.
	 * @see #isWithin
	 */
	public boolean contains(int x, int y) {
		Rectangle bigBounds = new Rectangle(getBounds().x, getBounds().y, getBounds().width + 1, getBounds().height + 1);
		return bigBounds.contains(x, y);
	}
	/**  
	 * Checks whether the specified <code>Figure</code> is "inside" this
	 * <code>Figure</code>. The <code>Figures</code> are assumed to share
	 * the same coordinate system.
	 * By default, just check if it is within the receiver's bounds.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param figure the Figure to test for inclusion
	 * @return	boolean value of <code>true</code> if the specified Figure
	 * is completely "inside" this Figure;
	 * 			<code>false</code> otherwise.
	 * @see #isWithin
	 */
	public boolean contains(Figure figure) {
		return contains(figure.getBounds());
	}
	/**  
	 * Checks whether a specified Rectangle is "inside" this Figure, 
	 * where the Rectangle and this Figure are in the same coordinate system  
	 * By default, just check if its topLeft and bottomRight is inside the receiver.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param box the rectangle to test for inclusion
	 * @return	boolean value of <code>true</code> if the specified Rectangle
	 * is "inside" this Figure;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(Rectangle box) {
	Rectangle bigBounds = new Rectangle(getBounds().x, getBounds().y, getBounds().width + 1, getBounds().height + 1);
	return bigBounds.contains(box.x,box.y) && bigBounds.contains(box.x + box.width, box.y + box.height);
	}
	/**
	 * Deletes all location listeners.
	 * Though this is public, caution should be used before anything other
	 * than "this" does.
	 */
	protected synchronized void deleteLocationListeners() {
		locationListeners = null;
	}
	/**
	 * Called to allow the <code>Figure</code> to respond to being disconnected.
	 * Should clean up as appropriate if the figure is no longer valid.
	 */
	public void disconnect() {
		fireRelationChange(new PropertyChangeEvent(this,RELATION_PROPERTY,null,null));
		deleteLocationListeners();
	}
	/**
	 * Duplicates the receiver.  Copy non-transient observers... let postDuplicate resolve
	 * whether observers have also been copied.
	 * 
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicate() {
		Hashtable duplicates = new Hashtable(estimatedDuplicateSize());
		Duplicatable dup = (Duplicatable)duplicateIn(duplicates);
		dup.postDuplicate(duplicates);
		return dup;
	}
	/**
	 * Duplicates the receiver into the given <code>Hashtable</code>.
	 * Copy non-transient observers... let postDuplicate resolve
	 * whether observers have also been copied.
	 * 
	 * @param duplicates the Hashtable to put the new duplicate in
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		try { 
		    AbstractFigure clone = (AbstractFigure)clone();
			clone.deleteLocationListeners();
		    if (locationListeners != null) {
		    	for (Enumeration e = relatedLocationListeners(); e.hasMoreElements(); ) 
					clone.addRelatedLocationListener((RelatedLocationListener)e.nextElement());
		    }
		    duplicates.put(this,clone);
	    	return clone;
		} catch (CloneNotSupportedException e) { 
		    // this shouldn't happen, since we are Cloneable
		    throw new InternalError();
		}
	}
	/** 
	 * Answers a Handle that will provide 
	 * editing capabilities on the receiver, or null.
	 * By default, answer null.
	 * Subclasses may wish to provide something more meaningful.
	 *
	 * @param x the x coordinate to potentially begin editing
	 * @param y the y coordinate to potentially begin editing
	 * @return	a Handle that will provide default capabilities
	 * on the receiver, or null if there is no default
	 */
	public Handle editTool(int x, int y)  {
		return null;
	}
	/**
	 * Answers the expected number of significant duplicates when duplicating the receiver.
	 * Subclasses may wish to override.
	 * 
	 * @returns an integer estimate of the number of objects to be duplicated 
	 */
	protected int estimatedDuplicateSize() {
		return 1;
	}
	/**
	 * Answer the figure, if any, associated with the locator.
	 * This can be a useful utility to determine whether or not a particular
	 * locator is tied to some figure.
	 *
	 * @param aLocator the Locator to mine for figures
	 * @return	the Figure associated with the locator, or null if none
	 */
	public static Figure figureFromLocator(Locator aLocator) {
		if (aLocator instanceof FigureHolder) {
			return ((FigureHolder)aLocator).getFigure();
		}
		if (aLocator instanceof RelativeLocator) {
			return figureFromLocator(((RelativeLocator)aLocator).getBase());
		}
		return null;
	}
	/**
	 * Denote that location changed.
	 *
	 * @param event
	 */
	protected void fireLocationChange(PropertyChangeEvent event) {
		java.util.Vector targets;
		synchronized (this) {
			if (locationListeners == null) {
				return;
			}
			targets = (java.util.Vector) locationListeners.clone();
		}
		for (int i = 0; i < targets.size(); i++) {
			RelatedLocationListener target = (RelatedLocationListener) targets.elementAt(i);
			target.locationChanged(event);
		}
	}
	/**
	 * Report a property update to any registered listeners.
	 *
	 * @param event the <code>PropertyChangeEvent</code> to report.
	 */
	protected void firePropertyChange(PropertyChangeEvent event) {
		java.util.Vector targets;
		synchronized (this) {
			if (listeners == null) {
				return;
			}
			targets = (java.util.Vector) listeners.clone();
		}
		for (int i = 0; i < targets.size(); i++) {
			PropertyChangeListener target = (PropertyChangeListener) targets.elementAt(i);
			target.propertyChange(event);
		}
	}
	/**
	 * Report a property update to any registered listeners.
	 * No event is fired if old and new are equal and non-null.
	 *
	 * @param propertyName  The programmatic name of the property
	 *		that was changed.
	 * @param oldValue  The old value of the property.
	 * @param newValue  The new value of the property.
	 */
	protected void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
		if (oldValue != null && oldValue.equals(newValue)) {
			return;
		}
		java.util.Vector targets;
		synchronized (this) {
			if (listeners == null) {
				return;
			}
			targets = (java.util.Vector) listeners.clone();
		}
		PropertyChangeEvent evt = new PropertyChangeEvent(this, propertyName, oldValue, newValue);
		for (int i = 0; i < targets.size(); i++) {
			PropertyChangeListener target = (PropertyChangeListener) targets.elementAt(i);
			target.propertyChange(evt);
		}
	}
	/**
	 * Report that the relation has changed.
	 *
	 * @param event the actual <code>PropertyChangeEvent</code> to report.
	 */
	protected void fireRelationChange(PropertyChangeEvent event) {
		java.util.Vector targets;
		synchronized (this) {
			if (locationListeners == null) {
				return;
			}
			targets = (java.util.Vector) locationListeners.clone();
		}
		for (int i = 0; i < targets.size(); i++) {
			RelatedLocationListener target = (RelatedLocationListener) targets.elementAt(i);
			target.relationChanged(event);
		}
	}
	/**
	 * Report that the shape changed.
	 *
	 * @param event the actual <code>PropertyChangeEvent</code> to report.
	 */
	protected void fireShapeChange(PropertyChangeEvent event) {
		java.util.Vector targets;
		synchronized (this) {
			if (locationListeners == null) {
				return;
			}
			targets = (java.util.Vector) locationListeners.clone();
		}
		for (int i = 0; i < targets.size(); i++) {
			RelatedLocationListener target = (RelatedLocationListener) targets.elementAt(i);
			target.shapeChanged(event);
		}
	}
	/**
	 * Report that the size changed.
	 *
	 * @param event the actual <code>PropertyChangeEvent</code> to report.
	 */
	protected void fireSizeChange(PropertyChangeEvent event) {
		java.util.Vector targets;
		synchronized (this) {
			if (locationListeners == null) {
				return;
			}
			targets = (java.util.Vector) locationListeners.clone();
		}
		for (int i = 0; i < targets.size(); i++) {
			RelatedLocationListener target = (RelatedLocationListener) targets.elementAt(i);
			target.sizeChanged(event);
		}
	}
	/** 
	 * Returns the current rectangular area covered by this figure.
	 * 
	 * @return	a Rectangle representing the current rectangular area
	 * covered by this figure
	 */
	public abstract Rectangle getBounds();
	/** 
	 * Answer the handles associated with the receiver.
	 * By default, there are none.
	 * Subclasses may wish to provide other handles that may allow for editing
	 * the receiver in some way.
	 * 
	 * @return	an array of the Handles associated with the receiver
	 */
	public Handle[] getHandles() {
		return new Handle[0];
	}
	/** 
	 * Answer a point indicating the location of the receiver... typically the topLeft.
	 * NOTE: This may not correspond to the point indicated by getLocator() as this method
	 * is often used to determine the position before a Locator has already been affected.
	 * 
	 * @return	a Point indicating the location of the receiver
	 */
	protected Point getLocation() {
		Rectangle myBounds = getBounds();
		return new java.awt.Point(myBounds.x,myBounds.y);
	}
	/** 
	 * Returns the current locator of this figure.
	 * This may or may not represent the top left of the receiver's area.
	 * By default, it does.
	 * Subclasses may wish to provide something more meaningful.
	 * 
	 * @return	the current Locator of this figure
	 */
	public Locator getLocator() {
		return new DrawingPoint(getLeft(), getTop());
	}
	/** 
	 * Answer the style which defines how to paint the figure.
	 * 
	 * @return	the DrawingStyle which defines how to paint the figure
	 */
	public DrawingStyle getStyle() {
		return new SimpleDrawingStyle();
	}
	/** 
	 * Answers whether the receiver intersects another figure.
	 * By default, just check if the bounds intersect.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param anotherFigure the figure the receiver is potentially intersecting.
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * specified figure;
	 * 			<code>false</code> otherwise.
	 * @see #bounds
	 */
	public boolean intersects(Figure anotherFigure) {
	return intersects(anotherFigure.getBounds());
	}
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 * By default, just check if the bounds intersects.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * specified Rectangular area;
	 * 			<code>false</code> otherwise.
	 * @see #bounds
	 */
	public boolean intersects(Rectangle box) {
		Rectangle bigBounds = new Rectangle(getBounds().x, getBounds().y, getBounds().width + 1, getBounds().height + 1);
		Rectangle bigBox = new Rectangle(box.x, box.y, box.width + 1, box.height + 1);
		return bigBounds.intersects(bigBox);
	}
	/** 
	 * Answers whether the receiver is obsolete
	 * True if some event has happened that makes this a meaningless object.
	 * 
	 * @return	boolean value of <code>true</code> if the receiver is obsolete;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isObsolete() {
	return false;
	}
	/** 
	 * Answers whether the receiver is fully within another Figure.
	 * By default, we ask the other figure if the receiver is inside it.  
	 * Subclasses may want to do something more sophisticated.
	 *
	 * @param anotherFigure the figure the receiver is potentially inside.
	 * @return	boolean value of <code>true</code> if the receiver is fully within
	 * the Figure specified;
	 * 			<code>false</code> otherwise.
	 * @see #inside
	 */
	public boolean isWithin(Figure anotherFigure) {
	return anotherFigure.contains(this);
	}
	/** 
	 * Answers whether the receiver is fully within a Rectangular area.
	 * By default, just check if the topLeft and bottomRight are inside the area.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver is fully within
	 * the Rectangle specified;
	 * 			<code>false</code> otherwise.
	 * @see #left
	 * @see #top
	 * @see #right
	 * @see #bottom
	 */
	public boolean isWithin(Rectangle box) {
		return (box.contains(getLeft(), getTop())) && (box.contains(getRight(), getBottom()));
	}
	/** 
	 * Answers a locator corresponding to a significant point on the receiver.
	 * By default, answer a point with the same relative position as the x and y
	 * are at the time of the request.
	 *
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 * @return	a Locator corresponding to a significant point on the receiver
	 */
	public Locator locatorAt(int x, int y) {
		double relativeX = ((double)x - (double)getLeft()) / (double)getWidth();
		double relativeY = ((double)y - (double)getTop()) / (double)getHeight();
		return new FigureRelativePoint(this, relativeX, relativeY);
	}
	/** 
	 * Moves the Figure to a new location. The x and y coordinates
	 * are in the parent's coordinate space.
	 * Let observers know what changed.
	 * This is a TemplateMethod with hooks:
	 * 	resetLocationCache()
	 * 	basicMove()
	 *  changedLocation()
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @see #location
	 */
	public void move(int x, int y)  {
		Point oldLocation = getLocation();
		basicMove(x,y);
		resetLocationCache();
		changedLocation(oldLocation);
	}
	/** 
	 * Moves the Figure to a new location.
	 * Note: Subclasses may wish to update dependencies based on this new location
	 *
	 * @param locator the Locator which identifies the desired x, y coordinates.
	 * @see #getLocator
	 */
	public void move(Locator locator) {
		move(locator.x(),locator.y());
	}
	/** 
	 * Paints the figure.
	 *
	 * @param g the specified Graphics window
	 */
	public abstract void paint(Graphics g);
	/**
	 * After a series of Figures are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes they might like to reconcile.  In this case,
	 * get rid of observers and replace them with any duplicates available.
	 *
	 * @param duplicates a Hashtable where originals as keys and duplicates as elements
	 */
	public void postDuplicate(Hashtable duplicates) {
		Enumeration e = relatedLocationListeners();
		deleteLocationListeners();
		while ( e.hasMoreElements() ) {
			RelatedLocationListener duplicate = (RelatedLocationListener)duplicates.get(e.nextElement());
			if (duplicate != null)
				addRelatedLocationListener(duplicate);
		}
	}
	/**
	 * Answer with an enumeration over the RelatedLocationListeners.
	 */

	public Enumeration relatedLocationListeners() {
		if (locationListeners == null)
			return (new Vector()).elements();
		return locationListeners.elements();
	}
	/**
	 * Remove the PropertyChangeListener from the listener list.
	 *
	 * @param listener  The PropertyChangeListener to be removed
	 */

	public synchronized void removePropertyChangeListener(PropertyChangeListener listener) {
		if (listeners == null) {
		    return;
		}
		listeners.removeElement(listener);
	}
	/**
	 * Remove the RelatedLocationListener from the listener list.
	 *
	 * @param listener  The RelatedLocationListener to be removed
	 */

	public synchronized void removeRelatedLocationListener(RelatedLocationListener listener) {
		if (locationListeners == null) {
			return;
		}
		locationListeners.removeElement(listener);
	}
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the other figure.
	 * By default, make it the middle of the receiver.
	 * Subclasses may wish to do something more meaningful.
	 *
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 */
	public Locator requestConnection(Figure requestor, int x, int y) {
		if (requestor == this)
			return null;
		return new FigureRelativePoint(this,0.5,0.5);
	}
	/**
	 * Flush caches with respect to determining location.  This is a hook method.
	 * Subclasses may wish to override.
	 */
	protected void resetLocationCache() {
	}
	/** 
	 * Reshapes the Figure to the specified bounding box.
	 * By default, ignore width and height.  Subclasses may wish to override.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 * @see #move
	 */
	public void setBounds(int x, int y, int width, int height) {
		move(x,y);
	}
	/**
	 * Resizes the receiver to the specified width and height.
	 * By default, do nothing.  Subclasses may wish to override.
	 *
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	public void setSize(int width, int height)  {
	}
	/** 
	 * Resizes the receiver to the specified dimension.
	 *
	 * @param d the figure dimension
	 * @see #getSize
	 * @see #setBounds
	 */
	public void setSize(Dimension d)  {
		setSize(d.width, d.height);
	}
	/** 
	 * Set the style defining how to paint the receiver.
	 * The default is to ignore the style.
	 * Most subclasses will want to do something more useful.
	 *
	 * @param style the specified DrawingStyle
	 */
	public void setStyle(DrawingStyle style) {
		firePropertyChange(STYLE_PROPERTY,this.getStyle(),style);
	}
	/** 
	 * Moves the Figure in the x and y direction.
	 * Let observers know what changed.
	 * This is a TemplateMethod with hooks:
	 * 	resetLocationCache()
	 * 	basicTranslate()
	 *  changedLocation()
	 *
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 * @see #location
	 */
	public void translate(int x, int y) {
		Point oldLocation = getLocation();
		basicTranslate(x, y);
		resetLocationCache();
		changedLocation(oldLocation);
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)AbstractInputEventHandler.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import java.awt.event.*;

/**
 * Provides a default implementation of <code>InputEventHandler</code>.
 * @version 	1.1.6, 12/28/98
 */
public class AbstractInputEventHandler implements InputEventHandler {
	/**
	 * Create a new AbstractInputEventHandler.
	 */
	public AbstractInputEventHandler() {
		super();
	}
	/**
	 * Called when a key is pressed.
	 * 
	 * @param e the KeyEvent to handle
	 */
	public void keyPressed(KeyEvent e) {
	}
	/**
	 * Called when a key is released.
	 * 
	 * @param e the KeyEvent to handle
	 */
	public void keyReleased(KeyEvent e) {
	}
	/**
	 * Called when a key is typed.
	 * 
	 * @param e the KeyEvent to handle
	 */
	public void keyTyped(KeyEvent e) {
	}
	/**
	 * Called when the mouse is clicked.
	 * 
	 * @param e the MouseEvent to handle
	 */
	public void mouseClicked(MouseEvent e) {
		if (e.getClickCount() == 2)
			mouseDoubleClicked(e);
		else
			mouseSingleClicked(e);
	}
	/**
	 * Called when the mouse is double-clicked.
	 * 
	 * @param e the MouseEvent to handle
	 */
	protected void mouseDoubleClicked(MouseEvent e) {
	}
	/**
	 * Called when the mouse is dragged.
	 * 
	 * @param e the MouseEvent to handle
	 */
	public void mouseDragged(MouseEvent e) {
	}
	/**
	 * Called when the mouse has entered.
	 * 
	 * @param e the MouseEvent to handle
	 */
	public void mouseEntered(MouseEvent e) {
	}
	/**
	 * Called when the mouse has exited.
	 * 
	 * @param e the MouseEvent to handle
	 */
	public void mouseExited(MouseEvent e) {
	}
	/**
	 * Called when the mouse is moved.
	 * 
	 * @param e the MouseEvent to handle
	 */
	public void mouseMoved(MouseEvent e) {
	}
	/**
	 * Called when the mouse is pressed.
	 * 
	 * @param e the MouseEvent to handle
	 */
	public void mousePressed(MouseEvent e) {
	}
	/**
	 * Called when the mouse is released.
	 * 
	 * @param e the MouseEvent to handle
	 */
	public void mouseReleased(MouseEvent e) {
	}
	/**
	 * Called when the mouse is single-clicked.
	 * 
	 * @param e the MouseEvent to handle
	 */
	protected void mouseSingleClicked(MouseEvent e) {
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)AbstractLocator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import java.util.Hashtable;
/**
 * Provides a default implementation of <code>Locator</code>.
 */
public abstract class AbstractLocator implements Locator {
/**
 * Creates a new AbstractLocator.
 */
public AbstractLocator() {
	super();
}
	/**
	 * Duplicates the receiver.
	 * 
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicate() {
		try { 
		    return clone();
		} catch (CloneNotSupportedException e) { 
		    // this shouldn't happen, since we are Cloneable
		    throw new InternalError();
		}
	}
	/**
	 * Duplicates the receiver into the given <code>Hashtable</code>.
	 * 
	 * @param duplicates the Hashtable to put the new duplicate in
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		Object dup = duplicate();
		duplicates.put(this,dup);
		return dup;
	}
	/**
	 * After a series of Objects are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes it might like to reconcile.
	 * In this case, do nothing.
	 *
	 * @param duplicates a Hashtable with originals as keys and duplicates
	 * as elements
	 */
	public void postDuplicate(Hashtable duplicates) {
	}
	/** 
	 * Answer the radius of the Locator (as a PolarCoordinate).
	 * 
	 * @return	an integer representing the radius
	 */
	public int r() {
		int myX = x();
		int myY = y();
		return (int)Math.sqrt((myX * myX) + (myY * myY));
	}
	/** 
	 * Answer the angle in radians of the Locator (as a PolarCoordinate).
	 * 
	 * @return double representing theta
	 */
	public double theta() {
		double theta = Math.atan((double)y()/(double)x());
		if (x() < 0)
			theta = theta + Math.PI;
		return theta;
	}
/**
 * Returns a String that represents the value of this object.
 *
 * @return a string representation of the receiver
 */
public String toString() {
	return getClass().getName() + " [x=" + x() + ",y=" + y() + ";r=" + r() + ",theta=" + theta() + "]";
}
	/** 
	 * Answer the x coordinate.
	 * 
	 * @return	an integer representing the x coordinate
	 */
	public abstract int x();
	/** 
	 * Answer the y coordinate.
	 * 
	 * @return	an integer representing the y coordinate
	 */
	public abstract int y();
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)AbstractPaintable.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import java.awt.*;

/**
 * Provides a default implementation of <code>Paintable</code>.
 */

public class AbstractPaintable implements Paintable {
/**
 * Creates a new AbstractPaintable.
 */
public AbstractPaintable() {
	super();
}
	/** 
	 * Returns the bottommost coordinate of the receiver.
	 * 
	 * @return	an integer representing the bottommost coordinate of the receiver
	 */
	public int getBottom() {
		return getTop() + getHeight();
	}
/**
 * Answers the rectangular space covered by the receiver.
 *
 * @return a <code>Rectangle</code> representing the bounds.
 */
public java.awt.Rectangle getBounds() {
	return null;
}
	/** 
	 * Returns the height of the receiver.
	 * 
	 * @return	an integer representing the height of the receiver.
	 */
	public int getHeight() {
		return getSize().height;
	}
	/** 
	 * Returns the leftmost coordinate of the receiver.
	 * 
	 * @return	an integer representing the leftmost coordinate of the receiver
	 */
	public int getLeft() {
		return getBounds().x;
	}
	/** 
	 * Returns the rightmost coordinate of the receiver.
	 * 
	 * @return	an integer representing the rightmost coordinate of the receiver
	 */
	public int getRight() {
		return getLeft() + getWidth();
	}
	/** 
	 * Returns the current size of the receiver.
	 *
	 * @return	a Dimension corresponding to the current size of the receiver.
	 * @see #getBounds
	 */ 
	public Dimension getSize() {
		Rectangle myBounds = getBounds();
		return new Dimension(myBounds.width,myBounds.height);
	}
	/** 
	 * Returns the topmost coordinate of the receiver.
	 * 
	 * @return	an integer representing the topmost coordinate of the receiver.
	 */
	public int getTop() {
		return getBounds().y;
	}
	/** 
	 * Returns the width of the receiver.
	 * 
	 * @return	an integer representing the width of the receiver.
	 */
	public int getWidth() {
		return getSize().width;
	}
/**
 * Tells the receiver to paint itself.
 *
 * @param g the <code>Graphics</code> object in which to paint.
 */
public void paint(java.awt.Graphics g) {
}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)BoxSelectionHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Enumeration;

/**
 * A handle one can use to select figures on a canvas within a box.
 * This would typically be used by some sort of SelectionTool, but there is
 * nothing in the code that assumes that to be the case.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class BoxSelectionHandle extends CanvasHandle {
	/**
	 * The Rectangular area to search for figures.
	 */
	protected Rectangle box;

	/**
	 * The x coordinate at which the box is anchored.
	 */
	protected int anchorX;

	/**
	 * The y coordinate at which the box is anchored.
	 */
	protected int anchorY;

	/**
	 * The figures that are surrounded.
	 */
	protected Vector surrounded = new Vector();

	/** 
	 * Constructs and initializes a new instance with a box derived from two points.
	 * NOTE: They do NOT have to be top-left and bottom-right
	 *
	 * @param startX the x coordinate of the anchor corner
	 * @param startY the y coordinate of the anchor corner
	 * @param endX the x coordinate of the floating corner
	 * @param endY the y coordinate of the floating corner
	 */
	public BoxSelectionHandle(int startX, int startY, int endX, int endY) {
	this.anchorX = startX;
	this.anchorY = startY;
	this.updateBox(endX,endY);
	}
	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * handle, where x and y are defined to be relative to the 
	 * coordinate system of this handle.  
	 * 
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified
	 * x,y position is "inside this handle;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(int x, int y)  {
	return box.contains(x,y);
	}
	/** 
	 * Returns the current bounds of this handle.
	 * 
	 * @return	a Rectangle representing the current bounds of this handle
	 */
	public Rectangle getBounds()  {
		return box;
	}
	/**
	 * Answer the Color to use when highlighting.
	 * 
	 * @return	the Color to use when highlighting
	 */
	protected Color getHighlightColor() {
		return canvas.getStyle().getHighlightColor();
	}
/**
 * Resize the selection area and make sure the ones inside are selected,
 * and those newly outside are not.
 *
 * @param evt the event
 */
public void mouseDragged(MouseEvent evt) {
	int x = getX(evt);
	int y = getY(evt);
	Rectangle bounds = new Rectangle(box.x, box.y, box.width, box.height);
	updateBox(x, y);
	if (bounds.contains(x, y)) {
		for (Enumeration e = surrounded.elements(); e.hasMoreElements();) {
			Figure figure = (Figure) e.nextElement();
			if (!figure.isWithin(box)) {
				surrounded.removeElement(figure);
				canvas.removeSelection(figure);
			}
		}
	} else {
		for (FigureEnumeration e = canvas.figures(); e.hasMoreElements();) {
			Figure figure = e.nextElement();
			if (!surrounded.contains(figure) && figure.isWithin(box)) {
				surrounded.addElement(figure);
				canvas.addSelection(figure);
			}
		}
	}
	bounds = bounds.union(box);
	canvas.repaint(bounds);
	evt.consume();
}
	/** 
	 * Paints the handle.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		Color oldColor = g.getColor();
		g.setColor(getHighlightColor());
		g.drawRect(box.x, box.y, box.width, box.height);
		g.setColor(oldColor);
	}
	/**
	 * Release control of the canvas and clean up if necessary.
	 * Since this is a public method,
	 * don't assume the receiver actually has control.
	 * 
	 * @param canvas the canvas which the receiver is to release control
	 */
	public void releaseControl(DrawingCanvas canvas) {
	canvas.removeHandle(this);
	super.releaseControl(canvas);
	}
	/**  
	 * Make the handle be the event handler for the canvas.
	 * Note, once it takes control, it is obligated to return 
	 * at a future point in time.  
	 * 
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @see #locate
	 */
	public void takeControl(DrawingCanvas canvas) {
	super.takeControl(canvas);
	canvas.addHandle(this);
	}
	/**  
	 * Change the box of the receiver based on its anchor and a new x,y.  
	 * 
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 */
	protected void updateBox(int x, int y) {
	box = new Rectangle(Math.min(anchorX,x), Math.min(anchorY,y), Math.abs(anchorX-x), Math.abs(anchorY-y));
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)CanvasHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This abstract class offers a simple base for handles to be used on a
 * canvas.  Subclasses need to provide, at a minimum, implementations for:
 *	<ul><li>bounds(),</li>
 *	<li>paint(Graphics)</li></ul>
 * Typically, these types of handles wait for some outside object to ask them
 * to takeControl() (e.g. mouse down on top of it), then they do something in
 * response to mouse events, and finally they releaseControl() (e.g. mouse up).
 *
 * @version 	1.1.6, 12/28/98
 */
 
public abstract class CanvasHandle extends AbstractInputEventHandler implements Handle {

	/**
	 * The general size of handles (assuming same size in x and y direction).
	 */
	protected static int HANDLE_SIZE = defaultHandleSize();

	/**
	 * Half the general width of handles.
	 */
	protected static int halfWidth = HANDLE_SIZE / 2;

	/**
	 * The canvas upon which to "operate"
	 */
	protected DrawingCanvas canvas;

	/**
	 * The tool that was in control before this one took over.
	 */
	protected InputEventHandler previousTool;

	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * handle. x and y are defined to be relative to the 
	 * coordinate system of this handle.  
	 * 
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified x,y
	 * location is "inside" this handle;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(int x, int y) {
		return getBounds().contains(x, y);
	}
	/** 
	 * Answers the default value of HANDLE_SIZE;
	 * 
	 * @return	an integer representing the default value of HANDLE_SIZE
	 */
	public static int defaultHandleSize()  {
	return 6;
	}
	/**
	 * The handle has completed its task... clean up as appropriate.
	 * Don't assume the canvas will cleanly ask us to releaseControl when we
	 * reset our tool.  This is the way we release control from within.
	 */
	protected void finished() {
	canvas.setTool(previousTool);
	releaseControl(canvas);
	}
	/** 
	 * Returns the current bounds of this handle.
	 * 
	 * @return	a Rectangle representing the current bounds
	 * of this handle
	 */
	public abstract Rectangle getBounds();
	/** 
	 * Answers the height of the handle;
	 * 
	 * @return	an integer representing the height of the handle
	 */
	protected int getHandleHeight()  {
		return HANDLE_SIZE;
	}
	/** 
	 * Answer the value of HANDLE_SIZE.  This is the default size of handles
	 * in the system (assume same x and y size).
	 * 
	 * @return	an integer representing the value of HANDLE_SIZE
	 */
	public static int getHandleSize()  {
		return HANDLE_SIZE;
	}
	/** 
	 * Answers the width of the handle;
	 * 
	 * @return	an integer representing the width of the handle
	 */
	protected int getHandleWidth()  {
		return HANDLE_SIZE;
	}
	/** 
	 * Returns the proper x value for the given event.
	 *
	 * @param evt the MouseEvent to get the corrected x for.
	 * @return	an integer representing the proper x coordinate.
	 */
	protected int getX( MouseEvent evt ) {
		return canvas.getLocator( evt.getX(), evt.getY() ).x();
	}
	/** 
	 * Returns the proper y value for the given event.
	 * 
	 * @param evt the MouseEvent to get the corrected y for.
	 * @return	an integer representing the proper y coordinate.
	 */
	protected int getY( MouseEvent evt ) {
		return canvas.getLocator( evt.getX(), evt.getY() ).y();
	}
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 * 
	 * @param box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects
	 * the specified Rectangular area;
	 * 			<code>false</code> otherwise.
	 */
	public boolean intersects(Rectangle box) {
		return getBounds().intersects(box);
	}
	/**
	 * By default, we are finished with our task when the mouse goes up, so we
	 * clean up.
	 * Subclasses may wish to override.
	 *
	 * @param e the event
	 */
	public void mouseReleased(MouseEvent e) {
		finished();
		e.consume();
	}
	/** 
	 * Paints the handle.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g) {
		Rectangle myBounds = getBounds();
		g.fillRect(myBounds.x, myBounds.y, myBounds.width, myBounds.height);
	}
	/**
	 * Release control of the canvas and clean up if necessary.
	 * Since this is a public method,
	 * don't assume the receiver actually has control.
	 * 
	 * @param canvas the canvas which the receiver is to release control
	 */
	public void releaseControl(DrawingCanvas canvas) {
		if (canvas.getTool() == this)
			canvas.toolTaskCompleted(this);
	}
	/** 
	 * Set the value of HANDLE_SIZE.  This is the default size of handles
	 * in the system (assume same x and y size).
	 *
	 * @param size the new size of handles.
	 */
	public static void setHandleSize(int size)  {
		HANDLE_SIZE = size;
		halfWidth = size / 2;
	}
	/**  
	 * Make the handle be the event handler for the canvas.
	 * Note, once it takes control, it is obligated to return 
	 * at a future point in time.  
	 * 
	 * @param canvas the canvas to be the event handler for
	 * @see #locate
	 */
	public void takeControl(DrawingCanvas canvas) {
		previousTool = canvas.getTool();
		this.canvas = canvas;
		canvas.setTool(this);
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)CanvasTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.event.*;

/**
 * This abstract class offers a simple base for Tools attached to a DrawingCanvas
 * Subclasses would typically provide a constructor that takes a DrawingCanvas
 * as an argument.
 *
 * @version 	1.1.6, 12/28/98
 */
public abstract class CanvasTool extends AbstractInputEventHandler {

	/**
	 * The canvas upon which to "operate"
	 */
	protected DrawingCanvas canvas;
	/** 
	 * Returns the proper x value for the given event.
	 *
	 * @param evt the MouseEvent to get the corrected x for.
	 * @return	an integer representing the proper x coordinate.
	 */
	protected int getX( MouseEvent evt ) {
		return canvas.getLocator( evt.getX(), evt.getY() ).x();
	}
	/** 
	 * Returns the proper y value for the given event.
	 * 
	 * @param evt the MouseEvent to get the corrected y for.
	 * @return	an integer representing the proper y coordinate.
	 */
	protected int getY( MouseEvent evt ) {
		return canvas.getLocator( evt.getX(), evt.getY() ).y();
	}
	/**
	 * Called if the mouse is released.  Default is to tell the canvas that
	 * the receiver's work is through.  Consume the event since we handle it.
	 * 
	 * @param e the MouseEvent
	 */
	public void mouseReleased(MouseEvent e) {
		canvas.toolTaskCompleted(this);
		e.consume();
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)ConstructionTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.event.*;

/**
 * This abstract class offers a simple base for Tools which create new Figures
 * and place them on a drawing canvas.
 * Concrete subclasses need to supply, at minmum:
 * 	basicNewFigure(int,int);
 *	some means of initializing canvas (e.g. Constructor)
 *
 * @version 	1.1.6, 12/28/98
 * @author 	Ken Auer
 */
 
public abstract class ConstructionTool extends CanvasTool {

	/**
	 * The figure the receiver is currently constructing
	 * (or null if the tool is not active).
	 */
	protected Figure figure;

	/**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a new Figure
	 */
	protected abstract Figure basicNewFigure(int x, int y);
	/**
	 * Called if the mouse is dragged (the mouse button is down and mouse is moving).  
	 * By default, move the figure we created, if we have one.  Consume the event
	 * since we handle it.
	 * 
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		figure.move( getX(e), getY(e) );
		e.consume();
	}
	/**
	 * Called if the mouse is pressed.  By default, create a new figure and add it
	 * to the canvas.  Subclasses may wish to consider newFigure() instead of
	 * this method.  Consume the event since we handle it.
	 *
	 * @param e the event 
	 * @see #newFigure
	 */
	public void mousePressed(MouseEvent e) {
		if ( e.getX() > canvas.getBounds().width || e.getY() > canvas.getBounds().height ) return;
		figure = newFigure(e.getX(), e.getY());
		canvas.addFigure(figure);
		e.consume();
	}
	/**
	 * Called if the mouse is released.  By default, we're done constructing and
	 * manipulating the figure, so forget about the figure and prepare to
	 * create a new one, in addition to inherited behavior.
	 * 
	 * @param e the event
	 */
	public void mouseReleased(MouseEvent e) {
		figure = null;
		super.mouseReleased(e);
	}
	/**
	 * Create and answer a new Figure.  This is a TemplateMethod/FactoryMethod.
	 * hooks are basicNewFigure() and setProperties().
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a newly created Figure
	 * @see #basicNewFigure
	 * @see #setProperties
	 */
	protected Figure newFigure(int x, int y) {
		Figure newFigure = basicNewFigure(x,y);
		setProperties(newFigure);
		return newFigure;
	}
	/**
	 * Set the properties of aFigure.  By default, use the canvas' style to
	 * set that of aFigure.
	 *
	 * @param aFigure the Figure to set properties.
	 */
	protected void setProperties(Figure aFigure) {
		aFigure.setStyle(canvas.getStyle());
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)DrawingPoint.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.util.Hashtable;
 
/**
 * This provides basic default functionality for MovableLocators.
 * Basically, it's just java.awt.Point with the proper interface to be
 * interchangeable with other Locators
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class DrawingPoint extends AbstractLocator implements MovableLocator {
	/**
	 * The serialization id.
	 */
	static final long serialVersionUID = 2517599265720696312L;

	/**
	 * The x coordinate of this DrawingPoint.
	 */
	protected int x;

	/**
	 * The y coordinate of this DrawingPoint.
	 */
	protected int y;
	/**
	 * Constructs and initializes a Point from the specified x and y 
	 * coordinates.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 */
	public DrawingPoint(int x, int y) {
		this.x = x;
		this.y = y;
	}
	/** 
	 * Moves the receiver to the x and y coordinates
	 * 
	 * @param x the new x coordinate
	 * @param y the new x coordinate
	 */
	public void move(int x, int y) {
		this.x = x;
		this.y = y;
	}
	/** 
	 * Moves the receiver in the x and y direction.
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	public void translate(int x, int y) {
		this.x += x;
		this.y += y;
	}
	/** 
	 * Answer the x coordinate.
	 * 
	 * @return	an integer representing the x coordinate
	 */
	public int x()  {
	return x;
	}
	/** 
	 * Set the x coordinate.
	 * 
	 * @param x its new desired x coordinate.
	 */
	public void x(int x)  {
	this.x = x;
	}
	/** 
	 * Answer the y coordinate.
	 * 
	 * @return	an integer representing the y coordinate
	 */
	public int y()  {
	return y;
	}
	/** 
	 * Set the y coordinate.
	 * 
	 * @param y its new desired y coordinate.
	 */
	public void y(int y)  {
	this.y = y;
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)EdgeLocator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
/**
 * This class implements Locator by providing x and y coordinates that are
 * at the intersection of the figure and the line defined by the 2 locators.
 */
public class EdgeLocator extends AbstractLocator implements RelativeLocator {
	/**
	 * The locator that is assumed to be inside a figure.
	 */
	protected Locator inside;

	/**
	 * The locator that is assumed to be outside a figure.
	 */
	protected Locator outside;
	
	/**
	 * The figure to which the line is connected.
	 */	
	protected Figure figure;
	protected int x, y, startX, startY, endX, endY;
/**
 * Creates a new EdgeLocator initialized with the given values.
 *
 * @param inside a Locator representing the end of a line inside a figure.
 * @param outside a Locator representing the end of a line outside a figure.
 * @param figure the figure at the inside Locator.
 */
public EdgeLocator(Locator inside, Locator outside) {
	this.inside = inside;
	this.outside = outside;
	this.figure = figureFromLocator(inside);
}
	/**
	 * 
	 */
	protected void computeIntersection() {

		//do a binary search.
		//check to see if the second point is contained in the figure
		//if it is then create a point between the end and middle and check
		//to see if it is contained.
		//if it is then it is the new end point repeat the step above
		//if it is not then set it as the new start point and create a middle point and check it.
		//continue until start and end are the same.

		//start points
		int startX = inside.x();
		int startY = inside.y();

		//end points
		int endX = outside.x();
		int endY = outside.y();

		if (this.startX == startX && this.startY == startY && this.endX == endX && this.endY == endY)
			return;
		else {
			this.startX = startX;
			this.startY = startY;
			this.endX = endX;
			this.endY = endY;
		}
		
		computeMiddlePoint(startX, startY, endX, endY);

		while (!differenceIsOneOrNone(startX, endX) || !differenceIsOneOrNone(startY, endY)) {
			if (!figure.contains(x, y)) {
				endX = x;
				endY = y;
			} else {
				startX = x;
				startY = y;
			}
			computeMiddlePoint(startX, startY, endX, endY);
		}
	}
/**
 * 
 * @return com.rolemodelsoft.drawlet.basics.DrawingPoint
 */
protected void computeMiddlePoint(int startX, int startY, int endX, int endY) {
	x = (startX + endX)/2;
	y = (startY + endY)/2;
}
/**
 * 
 */
protected boolean differenceIsOneOrNone(int x, int y) {
	return (x==y || x+1==y || x-1==y);
}
	/**
	 * Answer the figure, if any, associated with the locator.
	 * This can be a useful utility to determine whether or not a particular
	 * locator is tied to some figure.
	 *
	 * @param aLocator the Locator to mine for figures
	 * @return	the Figure associated with the locator, or null if none
	 */
	public static Figure figureFromLocator(Locator aLocator) {
		if (aLocator instanceof FigureHolder) {
			return ((FigureHolder)aLocator).getFigure();
		}
		if (aLocator instanceof RelativeLocator) {
			return figureFromLocator(((RelativeLocator)aLocator).getBase());
		}
		return null;
	}
	/** 
	 * Answer the base concrete locator of the receiver.
	 * 
	 * @return	the base concrete Locator of the receiver
	 */
	public Locator getBase() {
		return inside;
	}
/** 
 * Answer the x coordinate.
 * 
 * @return	an integer representing the x coordinate
 */
public int x() {
	computeIntersection();
	return x;
}
/** 
 * Answer the y coordinate.
 * 
 * @return	an integer representing the y coordinate
 */
public int y() {
	computeIntersection();
	return y;
}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)FigureRelativePoint.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.util.Hashtable;

/**
 * This class implements Locator by providing x and y coordinates that are
 * relative to a Figure.  Although it is assumed that the normal use will have
 * these locators within the bounds of the Figure, that is not necessary.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class FigureRelativePoint extends AbstractLocator implements FigureHolder {
	static final long serialVersionUID = -3811906450260337242L;
	
	/**
	 * The figure from which we derive our coordinates
	 */
	protected Figure figure;

	/**
	 * The relative x position from the top-left to bottom-right from
	 * which to derive our coordinates.  These will most likely be
	 * between 0.0 and 1.0, but do not have to be.
	 */
	protected double relativeX = defaultRelativeX();

	/**
	 * The relative y position from the top-left to bottom-right from
	 * which to derive our coordinates.  These will most likely be
	 * between 0.0 and 1.0, but do not have to be.
	 */
	protected double relativeY = defaultRelativeY();

	/**
	 * The x offset from the purely relative position.
	 */
	protected int offsetX = defaultOffsetX();

	/**
	 * The y offset from the purely relative position.
	 */
	protected int offsetY = defaultOffsetY();

	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a figure
	 *
	 * @param figure the figure to which the instance will be relative
	 * @see #defaultRelativeX
	 * @see #defaultRelativeY
	 * @see #defaultOffsetX
	 * @see #defaultOffsetY
	 */
	public FigureRelativePoint(Figure figure) {
		this.figure = figure;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a figure
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param relativeX the percentage of the width of the figure to determine X
	 * @param relativeY the percentage of the height of the figure to determine Y
	 * @see #defaultOffsetX
	 * @see #defaultOffsetY
	 */
	public FigureRelativePoint(Figure figure, double relativeX, double relativeY) {
	this(figure);
	this.relativeX = relativeX;
	this.relativeY = relativeY;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a figure
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param relativeX the percentage of the width of the figure to determine X
	 * @param relativeY the percentage of the height of the figure to determine Y
	 * @param offsetX the offset (added to relative X) in the x direction
	 * @param offsetY the offset (added to relative Y) in the y direction
	 */
	public FigureRelativePoint(Figure figure, double relativeX, double relativeY, int offsetX, int offsetY) {
	this(figure,relativeX,relativeY);
	this.offsetX = offsetX;
	this.offsetY = offsetY;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a figure
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param offsetX the offset (added to relative X) in the x direction
	 * @param offsetY the offset (added to relative Y) in the y direction
	 * @see #defaultRelativeX
	 * @see #defaultRelativeY
	 */
	public FigureRelativePoint(Figure figure, int offsetX, int offsetY) {
	this(figure);
	this.offsetX = offsetX;
	this.offsetY = offsetY;
	}
	/** 
	 * Answer the default/initial value of offsetX.
	 * Subclasses may wish to consider changing this from 0
	 * 
	 * @return	an integer representing the default/initial
	 * value of offsetX
	 */
	protected int defaultOffsetX() {
	return 0;
	}
	/** 
	 * Answer the default/initial value of offsetY.
	 * Subclasses may wish to consider changing this from 0
	 * 
	 * @return	an integer representing the default/initial
	 * value of offsetY
	 */
	protected int defaultOffsetY() {
	return 0;
	}
	/** 
	 * Answer the default/initial value of relativeX.
	 * Subclasses may wish to consider changing this from 0.0
	 * 
	 * @return	a double representing the default/initial
	 * value of relativeX
	 */
	protected double defaultRelativeX() {
	return 0.0;
	}
	/** 
	 * Answer the default/initial value of relativeY.
	 * Subclasses may wish to consider changing this from 0.0
	 * 
	 * @return	a double representing the default/initial
	 * value of relativeY
	 */
	protected double defaultRelativeY() {
	return 0.0;
	}
	/**
	 * Duplicates the receiver.
	 * 
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicate() {
	try { 
	    return clone();
	} catch (CloneNotSupportedException e) { 
	    // this shouldn't happen, since we are Cloneable
	    throw new InternalError();
	}
	}
	/**
	 * Duplicates the receiver in the given <code>Hashtable</code>.
	 * 
	 * @param duplicates the Hashtable to put the new duplicate in
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		Object dup = duplicate();
		duplicates.put(this,dup);
		return dup;
	}
	/** 
	 * Answer the figure to which the receiver is relative.
	 * 
	 * @return	the Figure to which the receiver is relative
	 */
	public Figure getFigure()  {
		return figure;
	}
	/**
	 * After a series of Objects are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes it might like to reconcile.
	 * In this case, get the duplicate of the current figure if there is one.
	 * Keep the original if not.
	 *
	 * @param duplicates a Hashtable where originals as keys and duplicates as elements
	 */
	public void postDuplicate(Hashtable duplicates) {
		Figure duplicate = (Figure)duplicates.get(figure);
		if (duplicate != null)
			setFigure(duplicate);
	}
	/** 
	 * Set the figure the receiver is holding.
	 *
	 * @param figure the Figure to hold
	 */
	public void setFigure(Figure figure) {
		this.figure = figure;
	}
	/** 
	 * Answer the x coordinate.
	 * 
	 * @return	an integer representing the x coordinate
	 */
	public int x()  {
		Rectangle bounds = figure.getBounds();
		return (int)(bounds.width * relativeX) + offsetX + bounds.x;
	}
	/** 
	 * Answer the y coordinate.
	 * 
	 * @return	an integer representing the y coordinate
	 */
	public int y()  {
		Rectangle bounds = figure.getBounds();
		return (int)(bounds.height * relativeY) + offsetY + bounds.y;
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)FigureTransfer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.datatransfer.*;
import com.rolemodelsoft.drawlet.*;
import java.util.Vector;
import java.io.*;

/**
 * Used to transfer objects from the <code>DrawingCanvas</code> to the clipboard.
 * @version 	1.1.6, 12/28/98
 */
 
public class FigureTransfer implements ClipboardOwner, Transferable {
	/**
	 * The figures that this <code>FigureTransfer</code> represents.
	 */
	protected Vector figures;

	/**
	 * The <code>DataFlavor</code> that <code>FigureTransfers</code> represent.
	 */
	public static DataFlavor figuresFlavor = new DataFlavor(Vector.class,"figures");
	/**
	 * Create a new <code>FigureTransfer</code>, and initialize it with
	 * the given <code>Vector</code>, which is assumed to contain
	 * <code>Figures</code>.
	 *
	 * @param figures a vector of the figures to be
	 * associated with this FigureTransfer
	 */
	public FigureTransfer(Vector figures) {
		this.figures = figures;
	}
	/**
	 * Return the data that the receiver holds, in the given flavor.
	 * 
	 * @param flavor the <code>DataFlavor</code> that the data should be returned as.
	 * @return	an Object that is the data.
	 */
	public Object getTransferData(DataFlavor flavor) throws IOException, UnsupportedFlavorException {
		if (flavor.equals(figuresFlavor)) return figures;
		else if (isSingleItemFlavorSupported(flavor)) {
			Figure figure = (Figure)figures.firstElement();
			String figureString;
			if (figures.firstElement() instanceof StringHolder) figureString = ((StringHolder)figure).getString();
			else figureString = figure.toString();
			
			if (flavor.equals(DataFlavor.stringFlavor)) return figureString;
			else if (flavor.equals(DataFlavor.plainTextFlavor)) return new StringReader(figureString);
			else throw new UnsupportedFlavorException(flavor);
		}
		else throw new UnsupportedFlavorException(flavor);
	}
	/**
	 * Answer the <code>DataFlavors</code> currently held.
	 * @return	an array of the <code>DataFlavors</code> currently
	 * contained in this <code>FigureTransfer</code>.
	 */
	public java.awt.datatransfer.DataFlavor[] getTransferDataFlavors() {
		if (figures.size() == 1) return new DataFlavor[] { figuresFlavor, DataFlavor.stringFlavor, DataFlavor.plainTextFlavor };
		else return new DataFlavor[] { figuresFlavor };
	}
	/**
	 * Answer whether or not the given flavor is supported.
	 * 
	 * @param flavor the flavor to check
	 * @return	boolean value of <code>true</code> if the flavor is supported;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		return (flavor.equals(figuresFlavor) || isSingleItemFlavorSupported(flavor));
	}
	/**
	 * @param flavor the flavor to check
	 * @return	boolean value of <code>true</code> if the flavor is supported;
	 * 			<code>false</code> otherwise.
	 */
	boolean isSingleItemFlavorSupported(DataFlavor flavor) {
		return 
			(figures.size() == 1 &&
						(flavor.equals(DataFlavor.stringFlavor) ||
						flavor.equals(DataFlavor.plainTextFlavor)));
	}
	/**
	 * Called when the receiver no longer hold ownership of the clipboard.
	 *
	 * @param clipboard the <code>Clipboard</code> whose ownership was lost.
	 * @param contents the contents of the <code>Clipboard</code> when
	 * ownership was lost.
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)FigureVectorEnumerator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import java.util.*;

/**
 * @version 	1.1.6, 12/28/98
 */

 public class FigureVectorEnumerator implements FigureEnumeration {
	/**
	 * The vector to enumerate over.
	 */
	Vector vector;

	/**
	 * The current index into the vector.
	 */
	int count;
	/**
	 * Create a FigureEnumeration over the Vector of Figures
	 * 
	 * @param v the vector
	 */
	public FigureVectorEnumerator(Vector v) {
		vector = v;
		count = 0;
	}
	/**
	 * Tests if this enumeration contains more figures.
	 *
	 * @return  boolean value of <code>true</code> if this
	 * enumeration contains more figures;
	 *          <code>false</code> otherwise.
	 */
	public boolean hasMoreElements() {
		return count < vector.size();
	}
	/**
	 * Returns the next figure of this enumeration.
	 *
	 * @return     the next Figure of this enumeration. 
	 * @exception  NoSuchElementException  if no more elements exist.
	 */
	public Figure nextElement() {
		synchronized (vector) {
		    if (count < vector.size()) {
				return (Figure)vector.elementAt(count++);
		    }
		}
		throw new NoSuchElementException("VectorEnumerator");
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)LocatorConnectionHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This class provides a handle that allows the corresponding figure to be
 * connected to (or disconnected from) another figure (as a slave to its location).
 * For example, this could be used to attach a label to another figure.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class LocatorConnectionHandle extends CanvasHandle implements FigureHolder {

	/**
	 * The figure we may connect/disconnect.
	 */
	protected Figure figure;

	/**
	 * The locator at which to visibly place the handle.
	 */
	protected Locator locator;

	/**
	 * The locator at which to show the current or pending connection 
	 * (or null while there is none to be displayed).
	 */
	protected Locator connection;

	/**
	 * The x coordinate at which the handle was first pressed.
	 */
	protected int anchorX;

	/**
	 * The y coordinate at which the handle was first pressed.
	 */
	protected int anchorY;

	/** 
	 * Constructs and initializes a new instance of a handle which can connect
	 * a Figure's location to another figure
	 *
	 * @param figure the figure which we may wish to connect/disconnect/reconnect
	 */
	public LocatorConnectionHandle(Figure figure) {
	this.setFigure(figure);
	}
	/** 
	 * Answer the default Color to use to draw the connection.
	 * 
	 * @return	the default Color to use to draw the connection
	 */
	protected Color defaultConnectionColor() {
		return Color.gray;
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure to return the default locator for
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,0.0,0.5);
	}
	/** 
	 * Returns the current bounds of this handle.
	 * 
	 * @return	a Rectangle representing the current bounds of this handle
	 */
	public Rectangle getBounds()  {
		Rectangle myBounds = getLocatorBounds();
		if (connection != null) 
			myBounds = myBounds.union(getConnectionBounds());
		myBounds.grow(1,1);  // fudge factor
		return myBounds;
	}
	/** 
	 * Returns the current bounds of the part of this handle that identifies
	 * to what the figure is connected.
	 * 
	 * @return	a Rectangle representing the current bounds of the part of
	 * this handle that identifies to what the figure is connected
	 */
	protected Rectangle getConnectionBounds()  {
		Locator center;
		if (connection == null)
			center = locator;
		else
			center = connection;
		int myRadius = getHandleWidth() / 2;
		return new Rectangle(center.x() - myRadius, center.y() - myRadius, getHandleWidth(), getHandleHeight());
	}
	/** 
	 * Answer the Color to use to draw the connection.
	 * 
	 * @return	the Color to use to draw the connection
	 */
	protected Color getConnectionColor() {
		if (canvas != null)
			return canvas.getStyle().getHighlightColor();
		return defaultConnectionColor();
	}
	/** 
	 * Returns the figure associated with this handle.
	 * 
	 * @return	the Figure associated with this handle
	 */
	public Figure getFigure()  {
		return figure;
	}
	/** 
	 * Returns the current bounds of the part of this handle that identifies
	 * the point of connection to the figure.
	 * 
	 * @return	a Rectangle representing the current bounds of the part of
	 * this handle that identifies the point of connection to the figure
	 */
	protected Rectangle getLocatorBounds()  {
		int myRadius = getHandleWidth() / 2;
		return new Rectangle(locator.x() - myRadius, locator.y() - myRadius, getHandleWidth(), getHandleHeight());
	}
	/** 
	 * Answer whether or not the figure is actually connected to something
	 * 
	 * @return	boolean value of <code>true</code> if the figure is actually
	 * connected to something;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isConnected()  {
		return AbstractFigure.figureFromLocator(figure.getLocator()) != null;
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Ask for a connection if over a figure.  Use it, or the coordinates
	 * to display the other side of the pending connection.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle oldBounds = getBounds();
		int x = getX(evt);
		int y = getY(evt);
		Figure connectFigure = canvas.otherFigureAt(figure, x, y);
		if (connectFigure != null)
			connection = connectFigure.requestConnection(figure, x, y);
		else
			connection = new DrawingPoint(x, y);
		canvas.repaint(getBounds().union(oldBounds));
		evt.consume();
	}
	/**
	 * Called if the mouse is down.  Show the connection point, if there is one
	 * and record where the mouse went down.
	 *
	 * @param evt the event 
	 */
	public void mousePressed(MouseEvent evt) {
		anchorX = getX(evt);
		anchorY = getY(evt);
		Rectangle oldBounds = getBounds();
		if (isConnected())
			connection = ((RelativeLocator) figure.getLocator()).getBase();
		// move the cursor to connection... beats me how to do it.
		canvas.repaint(getBounds().union(oldBounds));
		evt.consume();
	}
	/**
	 * Called if the mouse is released.  If it goes up over a figure that provides
	 * a valid connection, use it as the base of the new relative locator for 
	 * the figure and make sure the figure is in front of the figure to which
	 * we are connecting it.  Otherwise, (if the mouse did not go up in the 
	 * same place it went down) change the figure's locator to one that is not
	 * relative to any others.
	 * 
	 * @param evt the event
	 */
	public void mouseReleased(MouseEvent evt) {
		Locator oldLocator = figure.getLocator();
		Rectangle damage = getBounds();
		int x = getX(evt);
		int y = getY(evt);
		if (anchorX != x || anchorY != y) {
			Figure target = canvas.otherFigureAt(figure, x, y);
			if (target == null)
				figure.move(new DrawingPoint(oldLocator.x(), oldLocator.y()));
			else {
				Locator newLocator = target.requestConnection(figure, x, y);
				if (newLocator != null) {
					connection = new RelativePoint(newLocator, oldLocator.x() - newLocator.x(), oldLocator.y() - newLocator.y());
					figure.move(connection);
					canvas.moveFigureInFront(figure, target);
				} else
					figure.move(new DrawingPoint(oldLocator.x(), oldLocator.y()));
			}
		}
		damage = damage.union(getBounds());
		damage.grow(1, 1);
		canvas.repaint(damage);
		connection = null;
		super.mouseReleased(evt);
	}
	/** 
	 * Paints the handle.  
	 * Filled in if connected, hollow if not.
	 * Draw the otherwise invisible connection if actively editing.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		Rectangle myLocatorBounds = getLocatorBounds();
		if ((connection == null && isConnected()) || (connection != null && AbstractFigure.figureFromLocator(connection) != null) ) {
			Rectangle myConnectionBounds = getConnectionBounds();
			g.fillOval(myConnectionBounds.x, myConnectionBounds.y, myConnectionBounds.width, myConnectionBounds.height);	
		}
		g.drawOval(myLocatorBounds.x, myLocatorBounds.y, myLocatorBounds.width, myLocatorBounds.height);	
		if (connection != null) {
			g.setColor(getConnectionColor());
			g.drawLine(locator.x(), locator.y(), connection.x(), connection.y());
		}
	}
	/** 
	 * Set the figure associated with this handle.  Reset the locator.
	 *
	 * @param figure the Figure to hold
	 */
	public void setFigure(Figure figure)  {
	this.figure = figure;
	locator = defaultLocator(figure);
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)DrawingPoint.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.util.Hashtable;
 
/**
 * This provides basic default functionality for MovableLocators.
 * Basically, it's just java.awt.Point with the proper interface to be
 * interchangeable with other Locators
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class PolarCoordinate extends AbstractLocator {
	/**
	 * The serialization id.
	 */
	static final long serialVersionUID = 2517599265720696312L;

	/**
	 * The x coordinate of this DrawingPoint.
	 */
	protected int r;

	/**
	 * The y coordinate of this DrawingPoint.
	 */
	protected double theta;
	/**
	 * Constructs and initializes a PolarCoordinate from the specified r and theta 
	 * values.
	 * 
	 * @param r the radius.
	 * @param theta the theta
	 */
	public PolarCoordinate(int r, double theta) {
		this.r = r;
		this.theta = theta;
	}
	/** 
	 * Answer the radius of the Locator (as a PolarCoordinate).
	 * 
	 * @return	an integer representing the radius
	 */
	public int r() {
		return r;
	}
	/** 
	 * Answer the angle in radians of the Locator (as a PolarCoordinate).
	 * 
	 * @return double representing theta
	 */
	public double theta() {
		return theta;
	}
	/** 
	 * Answer the x coordinate.
	 * 
	 * @return	an integer representing the x coordinate
	 */
	public int x()  {
		return (int)( r * Math.cos( theta ) );
	}
	/** 
	 * Answer the y coordinate.
	 * 
	 * @return	an integer representing the y coordinate
	 */
	public int y()  {
		return (int)( r * Math.sin( theta ) );
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)PrototypeConstructionTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;

/**
 * This tool create new Figures based on a prototypical Figure.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class PrototypeConstructionTool extends ConstructionTool {

	/**
	 * The figure we duplicate to create new ones.
	 */
	protected Figure prototype;

	/** 
	 * Constructs and initializes a new instance of a tool which constructs new
	 * figures on a DrawingCanvas based on a prototypical figure
	 *
	 * @param canvas the canvas to which we add new figures.
	 * @param prototype the figure which we duplicate to create new ones.
	 */
	public PrototypeConstructionTool(DrawingCanvas canvas, Figure prototype) {
	this.canvas = canvas;
	this.prototype = prototype;
	}
   /**
	 * Create and answer a new Figure which is a duplicate of the prototype.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a new Figure which is a duplicate of the prototype
	 */
	protected Figure basicNewFigure(int x, int y)  {
	Figure newFigure = (Figure)prototype.duplicate();
	newFigure.move(x,y);
	return newFigure;
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)RelativePoint.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.util.Hashtable;

/**
 * This class implements RelativeLocator and MovableLocator by providing 
 * movable x and y offsets to some base locator.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class RelativePoint extends AbstractLocator implements MovableLocator, RelativeLocator {
	static final long serialVersionUID = 1110101860794525149L;
	
	/**
	 * The locator to which this is relative.
	 */
	protected Locator base;

	/**
	 * The x offset from the base locator.
	 */
	protected int offsetX = defaultOffsetX();

	/**
	 * The y offset from the base locator.
	 */
	protected int offsetY = defaultOffsetY();

	/**
	 * Constructs and initializes a RelativePoint with the specified parameters.
	 * 
	 * @param base the Locator to which the constructed point is relative.
	 * @param x the x coordinate of the offset
	 * @param y the y coordinate of the offset
	 */
	public RelativePoint(Locator base, int x, int y) {
	this.base = base;
	this.offsetX = x;
	this.offsetY = y;
	}
	/** 
	 * Answer the default/initial value of offsetX.
	 * Subclasses may wish to consider changing this from 0
	 * 
	 * @return	an integer representing the default/initial value of offsetX
	 */
	protected int defaultOffsetX() {
	return 0;
	}
	/** 
	 * Answer the default/initial value of offsetY.
	 * Subclasses may wish to consider changing this from 0
	 * 
	 * @return	an integer representing the default/initial value of offsetY
	 */
	protected int defaultOffsetY() {
	return 0;
	}
	/**
	 * Duplicates the receiver.  This means duplicating the base.
	 * 
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicate() {
	try {
	    RelativePoint duplicate = (RelativePoint)clone();
	    duplicate.base = (Locator)base.duplicate();
	    return duplicate;
	} catch (CloneNotSupportedException e) { 
	    // this shouldn't happen, since we are Cloneable
	    throw new InternalError();
	}
	}
	/**
	 * Duplicates the receiver in the given <code>Hashtable</code>.
	 * 
	 * @param duplicates the Hashtable to put the new duplicate in
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		try {
		    RelativePoint duplicate = (RelativePoint)clone();
		    duplicates.put(this,duplicate);
		    duplicate.base = (Locator)base.duplicateIn(duplicates);
		    return duplicate;
		} catch (CloneNotSupportedException e) { 
		    // this shouldn't happen, since we are Cloneable
		    throw new InternalError();
		}
	}
	/** 
	 * Answer the base concrete locator of the receiver.
	 * This is a recursive operation to get to the real base in a potential
	 * chain of RelativeLocators
	 * 
	 * @return	the base concrete Locator of the receiver
	 */
	public Locator getBase()  {
		if (base instanceof RelativeLocator)
			return ((RelativeLocator)base).getBase();
		return base;
	}
	/** 
	 * Moves the receiver to the x and y coordinates
	 * 
	 * @param x the new x coordinate
	 * @param y the new x coordinate
	 */
	public synchronized void move(int x, int y)  {
	x(x);
	y(y);
	}
	/**
	 * After a series of Objects are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes it might like to reconcile.
	 * In this case, get the duplicate of the current figure if there is one.
	 * Keep the original if not.
	 *
	 * @param duplicates a Hashtable where originals as keys and duplicates as elements
	 */
	public void postDuplicate(Hashtable duplicates) {
		Figure figure = AbstractFigure.figureFromLocator(base);
		if (figure != null) {
			if (!duplicates.containsKey(figure))
				base = new DrawingPoint(base.x(),base.y());
		}
		base.postDuplicate(duplicates);
	}
	/** 
	 * Moves the receiver in the x and y direction.
	 * 
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	public synchronized void translate(int x, int y)  {
	offsetX += x;
	offsetY += y;
	}
	/** 
	 * Answer the x coordinate.
	 * 
	 * @return	an integer representing the x coordinate
	 */
	public int x()  {
	return base.x() + offsetX;
	}
	/** 
	 * Set the x coordinate.
	 * 
	 * @param x its new desired x coordinate.
	 */
	public void x(int x)  {
	offsetX = x - base.x();
	}
	/** 
	 * Answer the y coordinate.
	 * 
	 * @return	an integer representing the y coordinate
	 */
	public int y()  {
	return base.y() + offsetY;
	}
	/** 
	 * Set the y coordinate.
	 * 
	 * @param y its new desired y coordinate.
	 */
	public void y(int y)  {
	offsetY = y - base.y();
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)SelectionTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.event.*;
import java.awt.*;
import java.util.Vector;
import java.util.Enumeration;

/**
 * This provides basic functionality necessary to select a Figure(s) on a DrawingCanvas
 * and/or pass control onto Handles.  There could be other ways to provide this
 * functionality, but this seems like a safe generic way to do it.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class SelectionTool extends CanvasTool {

	/**
	 * The x coordinate where the mouse went down.
	 * This is used to determine what to do when other events occur.
	 */
	protected int referenceX;

	/**
	 * The y coordinate where the mouse went down.
	 * This is used to determine what to do when other events occur.
	 */
	protected int referenceY;

	/**
	 * The figure where the mouse went down.
	 * This is used to determine what to do when other events occur.
	 */
	protected Figure referenceFigure;

	/** 
	 * Constructs and initializes a new instance of a tool to select figures
	 * on a DrawingCanvas
	 *
	 * @param canvas the canvas from which to select figures.
	 */
	public SelectionTool(DrawingCanvas canvas) {
	this.canvas = canvas;
	}
	/**
	 * Select based on a box.
	 *
	 * @param e the event caused this to happen
	 * @param x the x coordinate of the mouse
	 * @param y the y coordinate of the mouse
	 * @see #mouseDragged
	 */
	public void boxSelect(MouseEvent e, int x, int y) {
		(new BoxSelectionHandle(referenceX, referenceY, x, y)).takeControl(canvas);
		e.consume();
	}
	/**
	 * Called if the mouse is double-clicked.  Try passing it on to a handle if
	 * there is one present at the point.  Otherwise, see if there is a figure
	 * who might want to give control to an editTool.
	 *
	 * @param e the event 
	 * @see #mouseClicked
	 */
	protected void mouseDoubleClicked(MouseEvent e) {
		int x = e.getX();
		int y = e.getY();
		Handle handle = canvas.handleAt(x, y);
		if (handle != null) {
			handle.takeControl(canvas);
			handle.mouseClicked(e);
			return;
		}
		Figure figure = canvas.figureAt(x, y);
		if (figure != null) {
			referenceFigure = figure;
			Handle editTool = figure.editTool(x, y);
			if (editTool != null)
				editTool.takeControl(canvas);
			e.consume();
			return;
		}
		referenceFigure = null;
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * If we didn't put it down on top of a figure, do a box select,
	 * otherwise, move the selected figures (and handles);
	 *
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		int x = e.getX();
		int y = e.getY();
		if (referenceFigure == null || canvas.getSelections().length == 0) {
			boxSelect(e, x, y);
			return;
		}

		Rectangle selectionBounds = canvas.getSelections()[0].getBounds();
		Figure selections[] = canvas.getSelections();
		for (int i = 1; i < selections.length; i++) {
			selectionBounds = selectionBounds.union(selections[i].getBounds());
		}

		int moveX = x - referenceX;
		int moveY = y - referenceY;

		Locator moveLocator = canvas.getLocator( selectionBounds.x + moveX, selectionBounds.y + moveY );
		moveX = moveLocator.x() - selectionBounds.x;
		moveY = moveLocator.y() - selectionBounds.y;
		
		moveLocator = canvas.getLocator( selectionBounds.x + selectionBounds.width + moveX, selectionBounds.y + selectionBounds.height + moveY );
		if ( selectionBounds.x + selectionBounds.width + moveX > canvas.getBounds().width - 1 ) {
			moveX = moveLocator.x() - selectionBounds.x - selectionBounds.width;
		}
		if ( selectionBounds.y + selectionBounds.height + moveY > canvas.getBounds().height - 1 ) {
			moveY = moveLocator.y() - selectionBounds.y - selectionBounds.height;
		}
		
		Rectangle bounds = ((Figure) canvas.getSelections()[0]).getBounds();
		Handle handles[] = canvas.getHandles();
		for (int i = 0; i < handles.length; i++) {
			bounds = bounds.union(handles[i].getBounds());
		}
		for (int i = 0; i < selections.length; i++) {
			Figure figure = selections[i];
			bounds = bounds.union(figure.getBounds());
			figure.translate(moveX, moveY);
			bounds = bounds.union(figure.getBounds());
		}
		for (int i = 0; i < handles.length; i++) {
			bounds = bounds.union(handles[i].getBounds());
		}
		canvas.repaint(bounds);
		if (moveX != 0)
			referenceX = x;
		if (moveY != 0)
			referenceY = y;
		e.consume();
	}
	/**
	 * Called if the mouse is down.
	 * Keep track of where the mouse went down and then look for handles to
	 * take control, or figures to select/deselect
	 *
	 * @param e the event 
	 */
	public void mousePressed(MouseEvent e) {
		int x = e.getX();
		int y = e.getY();
		referenceX = x;
		referenceY = y;
		Handle handle = canvas.handleAt(x, y);
		if (handle != null) {
			handle.takeControl(canvas);
			handle.mousePressed(e);
			return;
		}
		e.consume();
		Figure figure = canvas.figureAt(x, y);
		if (figure != null) {
			referenceFigure = figure;
			if (e.isShiftDown())
				canvas.addSelection(figure);
			else {
				if (e.isControlDown())
					canvas.toggleSelection(figure);
				else
					canvas.select(figure);
			}
		} else {
			referenceFigure = null;
			if (!e.isShiftDown() && !e.isControlDown())
				canvas.clearSelections();
		}
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)SimpleDrawing.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.util.*;
import java.awt.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;

/**
 * This provides basic functionality necessary to provide a meaningful
 * working version of a Drawing.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class SimpleDrawing extends AbstractPaintable implements Drawing {
	static final long serialVersionUID = -6724809038122913127L;

	/**
	 * The figures which appear on the canvas.
	 */
	protected Vector figures = defaultFigures();

	/**
	 * The width of the drawing.
	 */
	protected int width = defaultWidth();

	/**
	 * The height of the drawing.
	 */
	protected int height;

	/**
	 * The property change listeners.
	 */
	transient protected Vector listeners;

	/**
	 * The background color.
	 */
	protected Color backgroundColor = defaultBackgroundColor();

	/**
	 * Stores whether this drawing is to be sized dynamically or not.
	 */
	protected boolean dynamicSize = defaultDynamicSize();
/**
 * Create a new instance of a Drawing
 */
public SimpleDrawing() {
}
/**
 * Create a new instance of a Drawing
 */
public SimpleDrawing( int width, int height ) {
	this.width = width;
	this.height = height;
	dynamicSize = false;
}
	/** 
	 * Add the figure to the contents of the canvas.
	 *
	 * @param figure the figure to add
	 */
	public void addFigure(Figure figure) {
		figures.addElement(figure);
	}
	/** 
	 * Add the figure to the contents of the canvas, sticking it behind 
	 * an existingFigure which is already there.
	 * Reflect the change if it's visible.
	 * Become an observer on the figure.
	 *
	 * @param figure the figure to add
	 * @param existingFigure the figure to which the new figure should be behind
	 */
	public void addFigureBehind(Figure figure, Figure existingFigure) {
		int existingIndex = figures.indexOf(existingFigure);
		if (existingIndex == -1) figures.addElement(figure);
		else figures.insertElementAt(figure,existingIndex);
	}
	/**
	 * Add a PropertyChangeListener to the listener list.
	 *
	 * @param listener  The PropertyChangeListener to be added
	 */

	public synchronized void addPropertyChangeListener(PropertyChangeListener listener) {
		if (listeners == null) {
		    listeners = new Vector();
		}
		if ( ! listeners.contains( listener ) ) {
			listeners.addElement(listener);
		}
	}
	/**
	 * Denote that size changed.
	 * @param oldDimension the old dimensions.
	 */
	protected void changedSize(Dimension oldDimension) {
		Dimension newDimension = getSize();
		if (oldDimension != null && oldDimension.equals(newDimension)) {
		    return;
		}
		PropertyChangeEvent evt = new PropertyChangeEvent(this,
						    SIZE_PROPERTY, oldDimension, newDimension);
		firePropertyChange(evt);
	}
	/**
	 * Answer the default color for this Drawing's background.
	 *
	 * @return a Color representing this Drawing's background color.
	 */
	protected Color defaultBackgroundColor() {
		return SystemColor.window;
	}
	/** 
	 * Answer the default for whether this Drawing should be dynamically sized.
	 *
	 * @return a boolean representing whether this Drawing should be dynamically sized by default.
	 */
	protected boolean defaultDynamicSize() {
		return true;
	}
	/** 
	 * Answer the default Vector to contain this Drawing's figures.
	 *
	 * @return a Vector to contain this Drawing's figures.
	 */
	protected Vector defaultFigures() {
		return new Vector(10);
	}
	/** 
	 * Answer the default width for this Drawing.
	 *
	 * @return an integer representing this Drawing's default width.
	 */
	protected int defaultWidth() {
		return 0;
	}
	/** 
	 * Answer the figure at a given point
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return the Figure at x,y; null if none found
	 */
	public Figure figureAt(int x, int y) {
		for (Enumeration e = new ReverseVectorEnumerator(figures); e.hasMoreElements();) {
			Figure figure = (Figure)e.nextElement();
			if (figure.contains(x,y)) return figure;
		}
		return null;
	}
	/** 
	 * Answer a FigureEnumeration over the figures of the receiver.
	 * 
	 * @return	a FigureEnumeration over the figures of the receiver
	 */
	public FigureEnumeration figures() {
		return new FigureVectorEnumerator(figures);
	}
	/**
	 * Report a property update to any registered listeners.
	 *
	 * @param event
	 */
	protected void firePropertyChange(PropertyChangeEvent event) {
		java.util.Vector targets;
		synchronized (this) {
			if (listeners == null) {
				return;
			}
			targets = (java.util.Vector) listeners.clone();
		}
		for (int i = 0; i < targets.size(); i++) {
			PropertyChangeListener target = (PropertyChangeListener) targets.elementAt(i);
			target.propertyChange(event);
		}
	}
	/** 
	 * Returns the current rectangular area covered by the receiver.
	 * 
	 * @return	a Rectangle representing the current rectangular area.
	 */
	public Rectangle getBounds() {
		if ( isDynamicSize() ) {
			Rectangle encompass = new Rectangle( 0, 0, 0, 0 );
			for (FigureEnumeration e = figures() ; e.hasMoreElements() ;) {
	 		    encompass.add(e.nextElement().getBounds());
			}
			return encompass;
		}
		return new Rectangle( 0, 0, width, height );
	}
	/** 
	 * Returns the current size of the receiver.
	 * 
	 * @return	a Dimension representing the current size.
	 */
	public Dimension getSize() {
		return new Dimension( getBounds().width, getBounds().height );
	}
	/** 
	 * Get the style defining how to paint on the canvas.
	 *
	 * @return the receivers current DrawingStyle
	 */
	public DrawingStyle getStyle() {
		DrawingStyle style = new SimpleDrawingStyle();
		style.setBackgroundColor( backgroundColor );
		return style;
	}
	/**
	 * Answers whether this Drawing is currently dynamically sized or not.
	 *
	 * @return a boolean value of true if this drawing is dynamically sized;
	 * false otherwise.
	 */
	public boolean isDynamicSize() {
		return dynamicSize;
	}
	/** 
	 * Move the figure behind an existingFigure if it is not already there.
	 * 
	 * @param figure the figure to move
	 * @param existingFigure the figure to which the new figure should be behind
	 * @exception IllegalArgumentException if one or both figures are unknown to receiver.
	 */
	public void moveFigureBehind(Figure figure, Figure existingFigure) {
		int index = figures.indexOf(figure);
		int existingIndex = figures.indexOf(existingFigure);
		if (index == -1 || existingIndex == -1) {
			throw new IllegalArgumentException("At least one of the figures do not exist on this drawing"); 
		}
		if (index > existingIndex) {
			figures.removeElement(figure);
			figures.insertElementAt(figure,existingIndex);
		}
	}
	/** 
	 * Move the figure in front of an existingFigure if it is not already there.
	 * 
	 * @param figure the figure to move
	 * @param existingFigure the figure to which the new figure should be in front
	 * @exception IllegalArgumentException if one or both figures are unknown to receiver.
	 */
	public void moveFigureInFront(Figure figure, Figure existingFigure) {
		int index = figures.indexOf(figure);
		int existingIndex = figures.indexOf(existingFigure);
		if (index == -1 || existingIndex == -1) { 
			throw new IllegalArgumentException("At least one of the figures do not exist on this drawing"); 
		}
		if (index < existingIndex) {
			figures.removeElement(figure);
			figures.insertElementAt(figure,existingIndex);
		}
	}
	/** 
	 * Move the figure behind all other figures.
	 * 
	 * @param figure the figure to move
	 * @exception IllegalArgumentException if figure is unknown to receiver.
	 */
	public void moveFigureToBack(Figure figure) {
		if (figures.removeElement(figure)) 
			figures.insertElementAt(figure,0);
		else
			throw new IllegalArgumentException("The figure does not exist on this drawing"); 
	}
	/** 
	 * Move the figure in front of all other figures.
	 * 
	 * @param figure the figure to move
	 * @exception IllegalArgumentException if figure is unknown to receiver.
	 */
	public void moveFigureToFront(Figure figure) {
		if (figures.removeElement(figure)) 
			figures.addElement(figure);
		else
			throw new IllegalArgumentException("The figure does not exist on this drawing"); 
	}
	/** 
	 * Answer the figure at a given point excluding the identified figure
	 *
	 * @param figure the figure to exclude from the search
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return the Figure at the given point, excluding the specified figure;
	 * null if none found
	 */
	public Figure otherFigureAt(Figure excludedFigure, int x, int y) {
		for (Enumeration e = new ReverseVectorEnumerator(figures); e.hasMoreElements();) {
			Figure figure = (Figure)e.nextElement();
			if ((excludedFigure != figure) && figure.contains(x,y)) return figure;
		}
		return null;
	}
	/** 
	 * Paints the component.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g) {
		paintBackground(g);
		paintAll(g);
	}
	/** 
	 * Paints everything in the receiver.
	 * Don't bother asking figures to paint themselves if they
	 * are not in the clipped region.
	 *
	 * @param g the specified Graphics window
	 */
	protected void paintAll(Graphics g) {
		Rectangle clip = g.getClipBounds();
		if (clip == null || clip.height == -1 || clip.width == -1) {
			paintCompletely(g);
			return;
		}
		for (Enumeration e = figures.elements() ; e.hasMoreElements() ;) {
			Figure fig = (Figure)e.nextElement();
	    	if (fig.intersects(clip))
				fig.paint(g);
		}
	}
	/** 
	 * Paints the background of the receiver.
	 *
	 * @param g the specified Graphics window
	 */
	protected void paintBackground(Graphics g) {
		Rectangle myBounds = getBounds();
		g.setColor(backgroundColor);
		g.fillRect(myBounds.x, myBounds.y, myBounds.width, myBounds.height);
	}
	/** 
	 * Paints everything in the receiver, ignoring the clipping region.
	 * 
	 * @param g the specified Graphics window
	 */
	protected void paintCompletely(Graphics g) {
		for (Enumeration e = figures.elements() ; e.hasMoreElements() ;) {
 		    ((Figure)e.nextElement()).paint(g);
		}
	}
	/** 
	 * Remove the figure.
	 * We may want to also tell the figure to dispose of itself completely...
	 * currently this is left up to the sender to allow them to do other things
	 * before disposing.  This is somewhat arbitraty however and a different
	 * approach may be desired by different implementations.
	 * 
	 * @param figure the figure to remove
	 */
	public void removeFigure(Figure figure) {
		figures.removeElement(figure);
	}
	/**
	 * Remove a PropertyChangeListener from the listener list.
	 *
	 * @param listener  The PropertyChangeListener to be removed.
	 */

	public synchronized void removePropertyChangeListener(PropertyChangeListener listener) {
		if (listeners == null) {
		    return;
		}
		listeners.removeElement(listener);
	}
	/**
	 * Sets whether this Drawing is dynamically sized or not.
	 *
	 * @param dynamicSize a boolean specifying this drawing's new
	 * dynamic size state.
	 */
	public void setDynamicSize( boolean dynamicSize ) {
		this.dynamicSize = dynamicSize;
	}
	/** 
	 * Sets the current size covered by this drawing.
	 *
	 * @param width an integer representing the new width.
	 * @param height an integer representing the new height.
	 */
	public void setSize( int width, int height ) {
		Dimension oldDimension = getSize();
		this.width = width;
		this.height = height;
		changedSize( oldDimension );
	}
	/** 
	 * Sets the current size covered by this drawing.
	 *
	 * @param d a Dimension representing the new size.
	 */
	public void setSize( Dimension d ) {
		setSize( d.width, d.height );
	}
	/** 
	 * Set the style defining how to paint on the canvas.
	 *
	 * @param style the specified DrawingStyle
	 */
	public void setStyle(DrawingStyle style) {
		backgroundColor = style.getBackgroundColor();
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)SimpleDrawingCanvas.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996-1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.util.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.awt.datatransfer.*;
import java.beans.*;

/**
 * This provides basic functionality necessary to provide a meaningful
 * working version of a DrawingCanvas that can be tied to a Component in an AWT
 * Application.  It is expected that this would serve as the base for these sort
 * of Components, but not required.
 *
 * @version 	1.1.6, 12/28/98
 */

public class SimpleDrawingCanvas extends AbstractPaintable implements InputEventHandler, DrawingCanvas, ComponentHolder, PropertyChangeListener {

	/**
	 * A place to hold figures that are cut or copied.  
	 * Currently, this is a rather primitive approach.
	 */
	protected static Clipboard clipboard = new Clipboard( "" );

	/**
	 * The amount to add to drawings which are dynamically sized.
	 */
	protected static final int buffer = 200;
	
	/**
	 * The drawing we are displaying/manipulating.
	 */
	protected Drawing drawing;
	
	/**
	 * The Component we are associated with.
	 */
	protected Component component;
	
	/**
	 * The tool which is the first thing we look to in order to handle events.
	 */
	protected InputEventHandler tool = defaultTool();

	/**
	 * The figures which are currently selected for further operations.
	 */
	protected Vector selections = defaultSelections();

	/**
	 * The handles that may take control of events.  Typically these are
	 * attached to figures, but not always.
	 */
	protected Vector handles = defaultHandles();

	/**
	 * A mapping of figures to their associated handles.
	 * This is typically used to allow for proper clean up if figures
	 * are removed, but may be used for other purposes.
	 */
	protected Hashtable figureHandles = defaultFigureHandles();

	/**
	 * The style with which it is expected new figures will be drawn.
	 */
	protected DrawingStyle style = defaultStyle();

	/**
	 * The width of the canvas.
	 */
	protected int width = defaultWidth();

	/**
	 * The height of the canvas.
	 */
	protected int height = defaultHeight();
	/**
	 * Default constructor
	 */
	public SimpleDrawingCanvas() {
		this(defaultDrawing());
	}
	/**
	 * @param drawing the drawing to construct this canvas with
	 */
	public SimpleDrawingCanvas(Drawing drawing) {
		super();
		this.drawing = drawing;
		this.listenToDrawing();
	}
	/** 
	 * Add the figure to the contents of the canvas.
	 * Reflect the change if it's visible.
	 * Become an observer on the figure.
	 *
	 * @param figure the figure to add
	 */
	public void addFigure(Figure figure) {
		drawing.addFigure(figure);
		listenToFigure(figure);
		repaint(figure.getBounds());
	}
	/** 
	 * Add the figure to the contents of the canvas, sticking it behind 
	 * an existingFigure which is already there.
	 * Reflect the change if it's visible.
	 * Become an observer on the figure.
	 *
	 * @param figure the figure to add
	 * @param existingFigure the figure to which the new figure should be behind
	 */
	public void addFigureBehind(Figure figure, Figure existingFigure) {
		drawing.addFigureBehind(figure,existingFigure);
		listenToFigure(figure);
		repaint(figure.getBounds());
	}
	/**
	 * Add the handle to the receiver.
	 * Reflect the change if it's visible.
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection.
	 * 
	 * @param handle the handle to add
	 */
	public void addHandle(Handle handle) {
		this.handles.addElement(handle);
		repaint(handle.getBounds());
	}
	/**
	 * Add the handles to the receiver.
	 * Reflect the change if it's visible.
	 * 
	 * @param handles the array of handles to add
	 */
	protected void addHandles(Handle handles[]) {
		for (int i=0; i < handles.length; i++)
			this.handles.addElement(handles[i]);
		repaint();
	}
	/**
	 * Add the handles corresponding to the figure to the receiver.
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection, but some tools may wish to be more selective.
	 * 
	 * @param figure the figure for which handles should be added.
	 */
	public void addHandles(Figure figure) {
		Handle handles[] = figure.getHandles();
		figureHandles.put(figure, handles);
		addHandles(handles);
	}
	/**
	 * Add the handles to the receiver, associating them the given figure.
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection, but some tools may wish to be more selective.
	 * 
	 * @param figure the figure for which handles are associated.
	 * @param handles the handles to add.
	 */
	public void addHandles(Figure figure, Handle handles[]) {
		Handle oldHandles[] = (Handle[])figureHandles.get(figure);
		if (oldHandles == null)
			figureHandles.put(figure, handles);
		else {
			Handle newHandles[] = new Handle[oldHandles.length + handles.length];
			System.arraycopy(oldHandles,0,newHandles,0,oldHandles.length);
			System.arraycopy(handles,0,newHandles,oldHandles.length,handles.length);
			figureHandles.put(figure, newHandles);
		}
		addHandles(handles);
	}
	/**
	 * Add the figure to the selections.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the Figure to add
	 */
	public void addSelection(Figure figure) {
		if (!selections.contains(figure)) {
			selections.addElement(figure);
			addHandles(figure);
		}
	}
	/**
	 * Remove all the handles from the receiver.
	 * Reflect the change if it's visible.
	 */
	protected void clearHandles() {
		handles.removeAllElements();
		figureHandles.clear();
		repaint();
	}
	/**
	 * Remove all the selections of the receiver.
	 * Reflect the change if it's visible.
	 * Remove any obsolete figures which were selected.
	 */
	public void clearSelections() {
		for (Enumeration e = selections.elements(); e.hasMoreElements(); ) {
			Figure figure = (Figure)e.nextElement();
			if (figure.isObsolete())
				removeFigure(figure);
		}
		selections.removeAllElements();
		clearHandles();
	}
	/**
	 * Copy the selections.  Make sure the copies are in the same order
	 * as the originals appeared.
	 */
	public void copySelections()  {
		putToClipboard(duplicateFigures(validateOrder(selections)));
	}
	/**
	 * Cut the selections
	 */
	public void cutSelections()  {
		copySelections();
		deleteSelections();
	}
	/**
	 * Answer the default color for the Component background.
	 * 
	 * @return Color
	 */
	protected Color defaultComponentBackground() {
		if ( drawing != null && drawing.isDynamicSize() ) 
			return drawing.getStyle().getBackgroundColor() ;
		else 
			return Color.gray;
	}
	/**
	 * Answer the default/initial Drawing to use.
	 * 
	 * @return	the default/initial Drawing to use
	 */
	protected static Drawing defaultDrawing() {
		return new SimpleDrawing();
	}
	/**
	 * Answer the default/initial figureHandles to use.
	 * 
	 * @return Hashtable
	 */
	protected Hashtable defaultFigureHandles() {
		return new Hashtable();
	}
	/**
	 * Answer the default/initial handles to use.
	 * 
	 * @return Vector
	 */
	protected Vector defaultHandles() {
		return new Vector(4);
	}
	/**
	 * Answer the default/initial height to use.
	 * 
	 * @return	the default/initial height to use
	 */
	protected int defaultHeight() {
		return 200;
	}
	/**
	 * Answer the default/initial selections to use.
	 * 
	 * @return Vector
	 */
	protected Vector defaultSelections() {
		return new Vector();
	}
	/**
	 * Answer the default/initial DrawingStyle to use.
	 */
	protected DrawingStyle defaultStyle() {
		return new SimpleDrawingStyle();
	}
	/**
	 * Answer the default/initial tool to use.
	 * 
	 * @return	the default/initial InputEventHandler (tool) to use
	 */
	protected InputEventHandler defaultTool() {
		return new SelectionTool(this);
	}
	/**
	 * Answer the default/initial width to use.
	 * 
	 * @return	the default/initial width to use
	 */
	protected int defaultWidth() {
		return 200;
	}
	/**
	 * Delete the selections
	 */
	public void deleteSelections()  {
		Vector oldSelections = (Vector)selections.clone();
		clearSelections();
		for (Enumeration e = oldSelections.elements(); e.hasMoreElements();) {
			removeFigure((Figure)e.nextElement());
		}
	}
	/**
	 * Called when the Drawing's size changes.
	 */
	protected void drawingSizeChange()  {
		if ( drawing.isDynamicSize() ) {
			if ( drawing.getSize().width > getSize().width ) {
				setSize( Math.max( component.getSize().width, drawing.getSize().width + buffer ), getSize().height );
			}
			if ( drawing.getSize().height > getSize().height ) {
				setSize( getSize().height, Math.max( component.getSize().height, drawing.getSize().height + buffer ) );
			}
		} else {
			setSize( drawing.getSize() );
		}
	}
	/**
	 * Duplicate all of the figures and return a new Vector
	 *
	 * @param toCopy the vector of figures to copy
	 * @return	a new Vector containing all of the figures
	 */
	protected Vector duplicateFigures(Vector toCopy)  {
		Vector copy = new Vector(toCopy.size());
		Hashtable duplicates = new Hashtable();
		for (Enumeration e = toCopy.elements(); e.hasMoreElements();) {
			Figure original = (Figure)e.nextElement();
			Figure duplicate = (Figure)original.duplicateIn(duplicates);
			duplicates.put(original,duplicate);
			copy.addElement(duplicate);
		}
		for (Enumeration e = copy.elements(); e.hasMoreElements();) {
			Figure duplicate = (Figure)e.nextElement();
			duplicate.postDuplicate(duplicates);
		}
		return copy;
	}
	/** 
	 * Answer the figure at a given point
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return the Figure at the given point; null if none found
	 */
	public Figure figureAt(int x, int y) {
		return drawing.figureAt(x,y);
	}
	/**
	 * Called when a figure changes.
	 */
	protected void figureChange(PropertyChangeEvent evt)  {
		Figure figure = (Figure)evt.getSource();
		Rectangle area = figure.getBounds();
		repaint(area);
	}
	/**
	 * Called when a figure's location changes.
	 */
	protected void figureLocationChange(PropertyChangeEvent evt)  {
		Figure figure = (Figure)evt.getSource();
		Rectangle area = figure.getBounds();
		Object newValue = evt.getNewValue();
		Object oldValue = evt.getOldValue();
		if (oldValue != null) {
			Point oldPoint = (Point)oldValue;
			area.add(oldPoint);
			area.add(oldPoint.x + area.width, oldPoint.y + area.height);
		}
		Point newPoint = (Point)newValue;
		area = new Rectangle(area.x, area.y, area.width, area.height);
		area.add(newPoint);
		area.add(newPoint.x + area.width, newPoint.y + area.height);
		repaint(area);
	}
	/** 
	 * Answer a FiguresEnumeration over the figures of the receiver.
	 * 
	 * @return	a FiguresEnumeration over the figures of the receiver
	 */
	public FigureEnumeration figures() {
		return drawing.figures();
	}
	/**
	 * Called when a figure's shape changes.
	 */
	protected void figureShapeChange(PropertyChangeEvent evt)  {
		Figure figure = (Figure)evt.getSource();
		Rectangle area = figure.getBounds();
		Object newValue = evt.getNewValue();
		Object oldValue = evt.getOldValue();
		if (oldValue != null)
			area = area.union((Rectangle)oldValue);
		area = area.union((Rectangle)newValue);
		repaint(area);
	}
	/**
	 * Called when a figure's size changes.
	 */
	protected void figureSizeChange(PropertyChangeEvent evt)  {
		Figure figure = (Figure)evt.getSource();
		Rectangle area = figure.getBounds();
		Dimension newSize = (Dimension)evt.getNewValue();
		Dimension oldSize = (Dimension)evt.getOldValue();
		if (newSize.width > area.width) 
			area = new Rectangle(area.x, area.y, newSize.width, newSize.height);
		if (newSize.height > area.height)
			area = new Rectangle(area.x, area.y, area.width, oldSize.height);
		repaint(area);
	}
	/** 
	 * Returns the current bounds of the receiver.
	 * 
	 * @return	a Rectangle representing the current bounds
	 */
	public Rectangle getBounds() {
		if ( drawing.isDynamicSize() ) {
			if ( width < component.getSize().width ) width = component.getSize().width;
			if ( height < component.getSize().height ) height = component.getSize().height;
		}
		return new Rectangle( 0, 0, width, height );
	}
	/**
	 * Answers the Clipboard for this canvas.
	 *
	 * @return the Clipboard for this canvas.
	 */
	protected Clipboard getClipboard() {
		return clipboard;
	}
	/**
	 * Answers the Component we are drawing/displaying.
	 *
	 * @return the Component we are drawing/displaying.
	 */
	public Component getComponent() {
		return component;
	}
	/**
	 * Answer a Figure (or null) created based on the string (e.g. from the clipboard)
	 * 
	 * @param string the string to create the new figure from
	 * @return	a new Figure created from the string
	 */
	protected Figure getFigureFromString(String string)  {
		Figure newFigure = new com.rolemodelsoft.drawlet.text.TextLabel(string);
		newFigure.setStyle(this.getStyle());
		return newFigure;
	}
	/**
	 * Answer the stuff on the clipboard
	 * 
	 * @return	the clipboard as a Vector
	 */
	protected Vector getFromClipboard()  {
		return getFromClipboard(getClipboard());

	}
	/**
	 * Answer the stuff on the clipboard
	 *
	 * @param clipboard the Clipboard to get the data from.
	 * @return	the clipboard as a Vector
	 */
	protected Vector getFromClipboard(Clipboard clipboard)  {
		Transferable transfer = clipboard.getContents(this);
		if (transfer == null)  {
			System.out.println("nothing to paste");
			Toolkit.getDefaultToolkit().beep();
			return new Vector();
		}
		try {
			if (transfer.isDataFlavorSupported(FigureTransfer.figuresFlavor))
				return (Vector)transfer.getTransferData(FigureTransfer.figuresFlavor);
			else 
			 if (transfer.isDataFlavorSupported(DataFlavor.stringFlavor)) {
				String string = (String)transfer.getTransferData(DataFlavor.stringFlavor);
				Figure label = getFigureFromString(string);
				Vector container = new Vector(1);
				container.addElement(label);
				return container;
			}
		} catch (UnsupportedFlavorException e) {
			System.out.println("Problem pasting: " + e);
		} catch (java.io.IOException e) {
			System.out.println("Problem pasting: " + e);
		}
		Toolkit.getDefaultToolkit().beep();
		return new Vector();

	}
	/**
	 * Answer the stuff on the system clipboard
	 * 
	 * @return	the clipboard as a Vector
	 */
	protected Vector getFromSystemClipboard()  {
		return getFromClipboard(getSystemClipboard());
	}
	/**
	 * Answer the Color to use in displaying handles.
	 * 
	 * @return	the Color to use when displaying handles
	 */
	protected Color getHandleColor() {
		return style.getForegroundColor();
	}
	/** 
	 * Answer the handles of the receiver.  The returned array 
	 * and its contents should be treated as read-only.
	 * 
	 * @return	an array of Handles representing the handles of the receiver;
	 * should be treated as read-only
	 */
	public Handle[] getHandles() {
		Handle myHandles[] = new Handle[handles.size()];
		handles.copyInto(myHandles);
		return myHandles;
	}
	/** 
	 * Answer the proper locator to be used for the given coordinates.
	 * This method adjusts depending on whether the drawing is dynamically
	 * sized or not.
	 *
	 * @param x the x coordinate to return a Locator for.
	 * @param y the y coordinate to return a Locator for.
	 * @return a Locator corresponding to the given x and y coordinates.
	 */
	public Locator getLocator( int x, int y ) {
		if ( drawing.isDynamicSize() ) {
			return new DrawingPoint( x, y );
		}
		int locX = Math.min( x, width - 1 );
		locX = Math.max( locX, 0 );
		int locY = Math.min( y, height - 1 );
		locY = Math.max( locY, 0 );
		return new DrawingPoint( locX, locY );
	}
	/** 
	 * Answer the selections of the receiver.  The returned array 
	 * and its contents should be treated as read-only.
	 * 
	 * @return	an array of Figures representing the selections of the receiver;
	 * should be treated as read-only
	 */
	public Figure[] getSelections() {
		Figure mySelections[] = new Figure[selections.size()];
		selections.copyInto(mySelections);
		return mySelections;
	}
	/** 
	 * Returns the current size of the receiver.
	 * 
	 * @return	a Dimension representing the current size
	 */
	public Dimension getSize() {
		return new Dimension( getBounds().width, getBounds().height );
	}
	/** 
	 * Answer the style which defines how to paint on the canvas.
	 * 
	 * @return	the DrawingStyle which defines how to paint on the canvas
	 */
	public DrawingStyle getStyle() {
		return style;
	}
	/**
	 * Answer the system Clipboard.
	 *
	 * @return the Clipboard for the system. At this point (i.e. in Java 1.1) this is basically only good
	 * for strings.
	 */
	protected Clipboard getSystemClipboard() {
		return Toolkit.getDefaultToolkit().getSystemClipboard();
	}
	/**
	 * Answer the active tool
	 * 
	 * @return	the active InputEventHandler (tool)
	 */
	public InputEventHandler getTool() {
		if (tool == null) return defaultTool();
		return tool;
	}
	/** 
	 * Answer the handle at a given point
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return the Handle at the given point; null if none found
	 */
	public Handle handleAt(int x, int y) {
		for (Enumeration e = new ReverseVectorEnumerator(handles); e.hasMoreElements();) {
			Handle handle = (Handle)e.nextElement();
			if (handle.contains(x,y)) return handle;
		}
		return null;
	}
	/** 
	 * Stop listening to all of the figures of the drawing.
	 */
	protected void ignoreDrawing() {
		for (FigureEnumeration e = figures(); e.hasMoreElements();) {
			ignoreFigure(e.nextElement());
		}
		drawing.removePropertyChangeListener( this );
	}
	/** 
	 * Stop listening to the specified figure.
	 * 
	 * @param figure the figure to stop listening to
	 */
	protected void ignoreFigure(Figure figure) {
		figure.removePropertyChangeListener(this);
	}
	/**
	 * Invoked when a key has been pressed.
	 * 
	 * @param e the event
	 */
	public void keyPressed(KeyEvent e) {
		getTool().keyPressed(e);
	}
	/**
	 * Invoked when a key has been released.
	 * 
	 * @param e the event
	 */
	public void keyReleased(KeyEvent e) {
		getTool().keyReleased(e);
	}
	/**
	 * Invoked when a key has been typed.
	 * This event occurs when a key press is followed by a key release.
	 * 
	 * @param e the event
	 */
	public void keyTyped(KeyEvent e) {
		getTool().keyTyped(e);
		if (!e.isConsumed())
			processKey(e);
	}
	/** 
	 * Begin listening to all of the figures of the drawing.
	 */
	protected void listenToDrawing() {
		for (FigureEnumeration e = figures(); e.hasMoreElements();) {
			listenToFigure(e.nextElement());
		}
		drawing.addPropertyChangeListener( this );
	}
	/** 
	 * Begin listening to the specified figure.
	 * 
	 * @param figure the figure to begin listening to
	 */
	protected void listenToFigure(Figure figure) {
		figure.addPropertyChangeListener(this);
	}
	/** 
	 * Returns the minimum size of the receiver.
	 * 
	 * @return	a Dimension representing the minimum size.
	 */
	protected Dimension minimumSize() {
		Dimension compSize, size = null;
		compSize = component.getSize();
		size.width = Math.max( compSize.width, drawing.getSize().width );
		size.height = Math.max( compSize.height, drawing.getSize().height );		
		return size;
	}
	/**
	 * Invoked when the mouse has been clicked on a component.
	 * 
	 * @param e the event
	 */
	public void mouseClicked(MouseEvent e) {
		getTool().mouseClicked(e);
	}
	/**
	 * Invoked when a mouse button is pressed on a component and then 
	 * dragged.  Mouse drag events will continue to be delivered to
	 * the component where the first originated until the mouse button is
	 * released (regardless of whether the mouse position is within the
	 * bounds of the component).
	 * 
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		getTool().mouseDragged(e);
	}
	/**
	 * Invoked when the mouse enters a component.
	 * 
	 * @param e the event
	 */
	public void mouseEntered(MouseEvent e) {
		getTool().mouseEntered(e);
	}
	/**
	 * Invoked when the mouse exits a component.
	 * 
	 * @param e the event
	 */
	public void mouseExited(MouseEvent e) {
		getTool().mouseExited(e);
	}
	/**
	 * Invoked when the mouse button has been moved on a component
	 * (with no buttons no down).
	 * 
	 * @param e the event
	 */
	public void mouseMoved(MouseEvent e) {
		getTool().mouseMoved(e);
	}
	/**
	 * Invoked when a mouse button has been pressed on a component.
	 * 
	 * @param e the event
	 */
	public void mousePressed(MouseEvent e) {
		getComponent().requestFocus();
		getTool().mousePressed(e);
	}
	/**
	 * Invoked when a mouse button has been released on a component.
	 * 
	 * @param e the event
	 */
	public void mouseReleased(MouseEvent e) {
		getTool().mouseReleased(e);
	}
	/** 
	 * Move the figure behind an existingFigure if it is not already there.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to move
	 * @param existingFigure the figure to which the new figure should be behind
	 */
	public void moveFigureBehind(Figure figure, Figure existingFigure) {
		try {
			drawing.moveFigureBehind(figure,existingFigure);
			repaint(figure.getBounds());
		} catch (IllegalArgumentException e) {}; // ignore for now
	}
	/** 
	 * Move the figure in front of an existingFigure if it is not already there.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to move
	 * @param existingFigure the figure to which the new figure should be in front
	 */
	public void moveFigureInFront(Figure figure, Figure existingFigure) {
		try {
			drawing.moveFigureInFront(figure,existingFigure);
			repaint(figure.getBounds());
		} catch (IllegalArgumentException e) {}; // ignore for now
	}
	/** 
	 * Move the figure behind all other figures.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to move
	 */
	public void moveFigureToBack(Figure figure) {
		try {
			drawing.moveFigureToBack(figure);
			repaint(figure.getBounds());
		} catch (IllegalArgumentException e) {}; // ignore for now
	}
	/** 
	 * Move the figure in front of all other figures.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to move
	 */
	public void moveFigureToFront(Figure figure) {
		try {
			drawing.moveFigureToFront(figure);
			repaint(figure.getBounds());
		} catch (IllegalArgumentException e) {}; // ignore for now
	}
	/** 
	 * Move the selected figures behind all other figures.
	 * Reflect the change if it's visible.
	 */
	public void moveSelectionsToBack() {
		for (Enumeration e = new ReverseVectorEnumerator(selections); e.hasMoreElements(); )
			moveFigureToBack((Figure)e.nextElement());
	}
	/** 
	 * Move the selected figures in front of all other figures.
	 * Reflect the change if it's visible.
	 */
	public void moveSelectionsToFront() {
		for (Enumeration e = selections.elements(); e.hasMoreElements(); )
			moveFigureToFront((Figure)e.nextElement());
	}
	/** 
	 * Answer the figure at a given point excluding the identified figure
	 *
	 * @param figure the figure to exclude from the search
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return the Figure at the given point, excluding the identified figure;
	 * null if none found
	 */
	public Figure otherFigureAt(Figure excludedFigure, int x, int y) {
		return drawing.otherFigureAt(excludedFigure, x, y);
	}
	/** 
	 * Paints the canvas.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g) {
		g.setClip( getBounds() );
		paintBackground(g);
		paintForeground(g);
		paintHandles(g);
	}
	/** 
	 * Paints the background of the canvas.
	 *
	 * @param g the specified Graphics window
	 */
	protected void paintBackground(Graphics g) {
		g.setColor(getStyle().getBackgroundColor());
		g.fillRect(0,0,getSize().width,getSize().height);
	}
	/** 
	 * Paints the foreground of the canvas.
	 *
	 * @param g the specified Graphics window
	 */
	protected void paintForeground(Graphics g) {
		g.setColor(getStyle().getForegroundColor());
		drawing.paint(g);
	}
	/** 
	 * Paints the handles on the canvas.
	 * Don't bother asking handles to paint themselves if they
	 * are not in the clipped region.
	 *
	 * @param g the specified Graphics window
	 */
	protected void paintHandles(Graphics g) {
		Rectangle clip = g.getClipBounds();
		g.setColor(getHandleColor());
		if (clip == null || clip.height == -1 || clip.width == -1) {
			for (Enumeration e = handles.elements() ; e.hasMoreElements() ;) {
 	  	 		((Handle)e.nextElement()).paint(g);
			}
		} else {
			for (Enumeration e = handles.elements() ; e.hasMoreElements() ;) {
				Handle h = (Handle)e.nextElement();
		 		if (h.intersects(clip))
					h.paint(g);
			}
		}
	}
	/**
	 * Paste from the clipboard.
	 * NOTE: Currently multiple pastes of the same thing shows up in the same place.
	 * Need to review entire approach to copy/cut/paste based on 1.1 model.
	 */
	public void paste()  {
		clearSelections();
		Vector duplicates = duplicateFigures(getFromClipboard());
		for (Enumeration e = duplicates.elements(); e.hasMoreElements();) {
			Figure duplicate = (Figure)e.nextElement();
			duplicate.translate(5,5);
			addFigure(duplicate);
			addSelection(duplicate);
		}
	}
	/**
	 * Paste from the system clipboard.
	 * NOTE: Currently multiple pastes of the same thing shows up in the same place.
	 * Need to review entire approach to copy/cut/paste based on 1.1 model.
	 */
	public void pasteFromSystem()  {
		clearSelections();
		Vector duplicates = duplicateFigures(getFromSystemClipboard());
		for (Enumeration e = duplicates.elements(); e.hasMoreElements();) {
			Figure duplicate = (Figure)e.nextElement();
			duplicate.translate(5,5);
			addFigure(duplicate);
			addSelection(duplicate);
		}
	}
	/**
	 * Called if a character is typed.  Handle special keys to perform
	 * certain operations.
	 * 
	 * @param evt the event
	 * @see #keyTyped
	 */
	protected void processKey(KeyEvent evt) {
		char key = evt.getKeyChar();
		switch (key) {
			case '\b' :
			case 127 : // delete key
				deleteSelections();
				evt.consume();
				return;
			case 2 : // Ctrl-b
				moveSelectionsToBack();
				evt.consume();
				return;
			case 3 : // Ctrl-c
				copySelections();
				evt.consume();
				return;
			case 6 : // Ctrl-f
				moveSelectionsToFront();
				evt.consume();
				return;
			case 24 : // Ctrl-x
				cutSelections();
				evt.consume();
				return;
			case 22 : // Ctrl-v
				paste();
				evt.consume();
				return;
		}
		return;
	}
	/**
	 * Called when a property changes.
	 * 
	 * @param evt the event
	 */
	public void propertyChange(PropertyChangeEvent evt) {
		if (evt.getSource() instanceof Figure) {
			String propertyName = evt.getPropertyName();
			if (propertyName == Figure.SHAPE_PROPERTY) {
				figureShapeChange(evt);
			} else if (propertyName == Figure.LOCATION_PROPERTY) {
				figureLocationChange(evt);
			} else if (propertyName == Figure.SIZE_PROPERTY) {
				figureSizeChange(evt);
			} else {
				figureChange(evt);
			}
			if ( drawing.isDynamicSize() ) drawingSizeChange();
		} else if ( evt.getSource().equals( drawing ) ) {
			if ( evt.getPropertyName() == Drawing.SIZE_PROPERTY ) {
				drawingSizeChange();
			}
		}
	}
	/**
	 * Add the stuff to the clipboard
	 * 
	 * @param stuff the vector to add to the clipboard
	 */
	protected void putToClipboard(Vector stuff)  {
		// clipboard = stuff;

		Clipboard clipboard = getClipboard();
		FigureTransfer transfer = new FigureTransfer(stuff);
		clipboard.setContents(transfer,transfer);

		// If it has a string, put it to the System clipboard, too.
		if ( stuff.size() == 1 && stuff.firstElement() instanceof StringHolder ) {
			clipboard = getSystemClipboard();
			StringSelection stringSelection = new StringSelection( ((StringHolder)stuff.firstElement()).getString() );
			clipboard.setContents( stringSelection, stringSelection );
		}
	}
	/** 
	 * Remove the figure.
	 * Reflect the change if it's visible.
	 * Remove the receiver as an Observer of
	 * the figure and disconnect the figure completely.
	 * 
	 * @param figure the figure to remove
	 */
	public void removeFigure(Figure figure) {
		drawing.removeFigure(figure);
		ignoreFigure(figure);
		repaint(figure.getBounds());
		figure.disconnect(); // any other clean up that needs to happen
	}
	/**
	 * Remove the handle from the receiver.
	 * Reflect the change if it's visible.
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection.  Since it is public, don't assume
	 * the handle asked for is actually present.
	 * 
	 * @param handle the handle to remove
	 */
	public void removeHandle(Handle handle) {
		if (this.handles.removeElement(handle))
			repaint(handle.getBounds());
	}
	/**
	 * Remove the handles to the receiver.
	 * Reflect the change if it's visible.
	 * 
	 * @param handles the array of handles to be removed
	 */
	protected void removeHandles(Handle handles[]) {
		for (int i=0; i < handles.length; i++)
			this.handles.removeElement(handles[i]);
		repaint();
	}
	/**
	 * Remove the handles corresponding to the figure to the receiver
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection, but some tools may wish to be more selective.
	 * 
	 * @param figure the figure for which handles should be removed.
	 */
	public void removeHandles(Figure figure) {
		Object handles = figureHandles.remove(figure);
		if (handles != null)
			removeHandles((Handle[])handles);
	}
	/**
	 * Remove the figure from the selections.
	 * Reflect the change if it's visible.
	 * If a figure is obsolete, remove it.
	 *
	 * @param figure the figure being deselected
	 */
	public void removeSelection(Figure figure) {
		removeHandles(figure);
		selections.removeElement(figure);
		if (figure.isObsolete())
			removeFigure(figure);
	}
	/** 
	 * Repaint the canvas.
	 */
	protected void repaint() {
		component.repaint();
	}
	/** 
	 * Repaints part of the canvas. This will result in a
	 * call to update as soon as possible.
	 * 
	 * @param rectangle is the region to be repainted
	 * @see #repaint
	 */
	public void repaint(Rectangle rectangle) {
		/*
		 * Note that we're fudging by a pixel in every direction to avoid
		 * differences in the way various drawing primitives determine 
		 * where to start/stop drawing
		 */
		component.repaint(rectangle.x - 1, rectangle.y - 1, rectangle.width + 2, rectangle.height + 2);
	}
	/**
	 * Make the figure the only selection
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure being deselected
	 */
	public void select(Figure figure) {
		clearSelections();
		addSelection(figure);
	}
	/**
	 * Set the Component to draw/display.
	 *
	 * @param component the component to draw/display.
	 */
	public void setComponent(Component component) {
		this.component = component;
		drawingSizeChange();
	}
	/** 
	 * Set the drawing associated with this canvas.
	 * Reflect the change if it's visible.
	 * Become an observer on the drawing.
	 *
	 * @param newDrawing the drawing to associate with
	 */
	public void setDrawing(Drawing newDrawing) {
		if (this.drawing == newDrawing) 
			return;
		if (this.drawing != null) {
			clearSelections();
			ignoreDrawing();
		}
		this.drawing = newDrawing;
		if (this.drawing != null) {
			listenToDrawing();
			style.setBackgroundColor(newDrawing.getStyle().getBackgroundColor());
			component.setBackground(defaultComponentBackground());
		}
		repaint();
	}
	/** 
	 * Set the size of the receiver.
	 *
	 * @param width the new width of the receiver.
	 * @param width the new height of the receiver.
	 */
	public void setSize( int width, int height ) {
		this.width = width;
		this.height = height;
	}
	/** 
	 * Set the size of the receiver.
	 *
	 * @param size the new size of the receiver.
	 */
	public void setSize( Dimension size ) {
		setSize( size.width, size.height );
	}
	/** 
	 * Set the style defining how to paint on the canvas.
	 * Change the foreground and background of the component immediately.
	 *
	 * @param style the specified DrawingStyle
	 */
	public void setStyle(DrawingStyle style) {
		this.style = style;
		drawing.setStyle(style);
		component.setForeground(style.getForegroundColor());
		component.setBackground(defaultComponentBackground());
	}
	/**
	 * Set the active tool
	 * 
	 * @param newTool the tool to make active
	 */
	public void setTool(InputEventHandler newTool) {
		if (tool instanceof Handle)
			((Handle)tool).releaseControl(this);
		tool = newTool;
	}
	/**
	 * Toggle whether or not the figure is selected
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure of interest
	 */
	public void toggleSelection(Figure figure) {
		if (selections.contains(figure)) removeSelection(figure);
		else addSelection(figure);
	}
	/**
	 * Take appropriate action when the tool has completed its task.
	 * Default behavior is to do nothing.  Subclasses may wish to give
	 * control back to another tool.
	 * 
	 * @param tool the tool which completed its task
	 */
	public void toolTaskCompleted(InputEventHandler tool) {
	}
	/**
	 * Answer a Vector containing the figures in the same order they appear in
	 * the receiver.
	 *
	 * @param unordered the Vector of figures that need to be ordered
	 * @return	a Vector containing the figures in the same order they appear
	 * in the receiver
	 */
	protected Vector validateOrder(Vector unordered)  {
		Vector copy = new Vector(unordered.size());
		int positions[] = new int[unordered.size()];
		/*
		 * Simple sort since we probably aren't dealing with a lot of elements.
		 * For each element: 
		 *	get position in receiver,
		 *	determine how many we've already seen belong in front of it,
		 *	stick it in the appropriate place
		 */
		int i = 0;
		for (Enumeration e = unordered.elements(); e. hasMoreElements(); i++) {
			Object element = e.nextElement();
			int j = 0;
			for (FigureEnumeration f = figures(); f.hasMoreElements(); j++) {
				if (f.nextElement() == element) {
					positions[i] = j;
					break;
				}
			}
			int count = 0;
			for (j=0; j < i; j++)
				if (positions[i] > positions[j]) count++;
			copy.insertElementAt(element,count);
		}
		return copy;
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)SimpleDrawingStyle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import java.awt.*;
import java.util.Hashtable;

/**
 * This provides basic functionality necessary to provide a meaningful
 * working version of a DrawingStyle. 
 *
 * @version 	1.1.6, 12/28/98
 */
 
public class SimpleDrawingStyle implements DrawingStyle {

	/**
	 * Holds this style's background color.
	 */
	protected Color backgroundColor = defaultBackgroundColor();

	/**
	 * Holds this style's foreground color.
	 */
	protected Color foregroundColor = defaultForegroundColor();

	/**
	 * Holds this style's font.
	 */
	protected Font font = defaultFont();

	/**
	 * Holds this style's fill color.
	 */
	protected Color fillColor = defaultFillColor(); 

	/**
	 * Holds this style's line color.
	 */
	protected Color lineColor; // = defaultLineColor();  Let it be the same as foregroundColor

	/**
	 * Holds this style's text color.
	 */
	protected Color textColor; // = defaultTextColor();  Let it be the same as foregroundColor

	/**
	 * Holds this style's selection background color.
	 */
	protected Color selectionBackgroundColor = defaultSelectionBackgroundColor();

	/**
	 * Holds this style's selection foreground color.
	 */
	protected Color selectionForegroundColor = defaultSelectionForegroundColor();

	/**
	 * Holds this style's highlight color.
	 */
	protected Color highlightColor = defaultHighlightColor();

	/**
	 * Answer the default/initial value for backgroundColor
	 * 
	 * @return	a Color representing the default/initial value for backgroundColor
	 */
	protected Color defaultBackgroundColor() {
		return Color.white;
	}
	/**
	 * Answer the default/initial value for fillColor
	 * 
	 * @return	a Color representing the default/initial value for fillColor
	 */
	protected Color defaultFillColor() {
		return Color.lightGray;
	}
	/**
	 * Answer the default/initial value for the font
	 * 
	 * @return	a Font representing the default/initial value for the font
	 */
	protected Font defaultFont()  {
		Font newFont; // = Font.getFont("font.default");
	//	if (newFont == null)
			newFont = new Font("TimesRoman",Font.PLAIN,12);
		return newFont;
	}
	/**
	 * Answer the default/initial value for foregroundColor
	 * 
	 * @return	a Color representing the default/initial value for foregroundColor
	 */
	protected Color defaultForegroundColor() {
	return Color.black;
	}
	/**
	 * Answer the default/initial value for selectionBackgroundColor
	 * 
	 * @return	a Color representing the default/initial value for
	 * selectionBackgroundColor
	 */
	protected Color defaultHighlightColor() {
		return Color.gray;
	}
	/**
	 * Answer the default/initial value for lineColor
	 * 
	 * @return	a Color representing the default/initial value for lineColor
	 */
	protected Color defaultLineColor() {
		return getForegroundColor();
	}
	/**
	 * Answer the default/initial value for selectionBackgroundColor
	 * 
	 * @return	a Color representing the default/initial value for
	 * selectionBackgroundColor
	 */
	protected Color defaultSelectionBackgroundColor() {
		return getForegroundColor();
	}
	/**
	 * Answer the default/initial value for selectionForegroundColor
	 * 
	 * @return	a Color representing the default/initial value for
	 * selectionForegroundColor
	 */
	protected Color defaultSelectionForegroundColor() {
		return getBackgroundColor();
	}
	/**
	 * Answer the default/initial value for textColor
	 * 
	 * @return	a Color representing the default/initial value for textColor
	 */
	protected Color defaultTextColor() {
		return getForegroundColor();
	}
	/**
	 * Duplicates the receiver.
	 * 
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicate() {
		try { 
		    return clone();
		} catch (CloneNotSupportedException e) { 
		    // this shouldn't happen, since we are Cloneable
		    throw new InternalError();
		}
	}
	/**
	 * Duplicates the receiver in the given <code>Hashtable</code>.
	 * 
	 * @param duplicates the Hashtable to put the new duplicate in
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		if (duplicates.containsKey(this))
			return duplicates.get(this);
		else {
			Object dup = this.duplicate();
			duplicates.put(this,dup);
			return dup;
		}
	}
	/**
	 * Answer this style's background Color.
	 * 
	 * @return	the Color to use for the background
	 */
	public Color getBackgroundColor() {
		return backgroundColor;
	}
	/**
	 * Answer this style's fill Color.
	 * 
	 * @return	the Color to use for filling.
	 */
	public Color getFillColor() {
		return fillColor;
	}
	/** 
	 * Answer this style's font.
	 * 
	 * @return	the font to use.
	 */
	public Font getFont()  {
		return font;
	}
	/**
	 * Answer this style's foreground Color.
	 * 
	 * @return	the Color to use for the foreground.
	 */
	public Color getForegroundColor() {
		return foregroundColor;
	}
	/**
	 * Answer this style's highlight Color.
	 * 
	 * @return	the Color to use for the highlight.
	 */
	public Color getHighlightColor()  {
		if (highlightColor == null)
			return defaultHighlightColor();
		return highlightColor;
	}
	/**
	 * Answer this style's line Color.
	 * 
	 * @return	the Color to use for the line.
	 */
	public Color getLineColor() {
		if (lineColor == null)
			return defaultLineColor();
		return lineColor;
	}
	/**
	 * Answer this style's selection background Color.
	 * 
	 * @return	the Color to use for the selection's background
	 */
	public Color getSelectionBackgroundColor()  {
		if (selectionBackgroundColor == null)
			return defaultSelectionBackgroundColor();
		return selectionBackgroundColor;
	}
	/**
	 * Answer this style's selection foreground Color.
	 * 
	 * @return	the Color to use for the selection's foreground.
	 */
	public Color getSelectionForegroundColor() {
		if (selectionForegroundColor == null)
			return defaultSelectionForegroundColor();
		return selectionForegroundColor;
	}
	/**
	 * Answer this style's text Color.
	 * 
	 * @return	the Color to use for the text.
	 */
	public Color getTextColor() {
		if (textColor == null)
			return defaultTextColor();
		return textColor;
	}
	/**
	 * After a series of objects are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes it might like to reconcile.  
	 * For example, replacing observers with their duplicates, if available.
	 * 
	 * @param duplicates a Hashtable with originals as keys and duplicats as elements
	 */
	public void postDuplicate(Hashtable duplicates) {
	}
	/**
	 * Set this style's background Color.
	 * 
	 * @param color the Color to use for the background.
	 */
	public void setBackgroundColor(Color color) {
		backgroundColor = color;
	}
	/**
	 * Set this style's fill Color.
	 * 
	 * @param color the Color to use for filling.
	 */
	public void setFillColor(Color color) {
		fillColor = color;
	}
	/** 
	 * Set this style's font.
	 * 
	 * @param newFont the font to use.
	 */
	public void setFont(Font newFont)  {
		font = newFont;
	}
	/**
	 * Set this style's foreground Color.
	 * 
	 * @param color the Color to use for the foreground.
	 */
	public void setForegroundColor(Color color) {
		foregroundColor = color;
	}
	/**
	 * Set this style's highlight Color.
	 * 
	 * @param color the Color to use for the highlight.
	 */
	public void setHighlightColor(Color color) {
		highlightColor = color;
	}
	/**
	 * Set this style's line Color.
	 * 
	 * @param color the Color to use for the line.
	 */
	public void setLineColor(Color color) {
		lineColor = color;
	}
	/**
	 * Set this style's selection background Color.
	 * 
	 * @param color the Color to use for the selection's background.
	 */
	public void setSelectionBackgroundColor(Color color) {
		selectionBackgroundColor = color;
	}
	/**
	 * Set this style's selection foreground Color.
	 * 
	 * @param color the Color to use for the selection's foreground.
	 */
	public void setSelectionForegroundColor(Color color) {
		selectionForegroundColor = color;
	}
	/**
	 * Set this style's text Color.
	 * 
	 * @param color the Color to use for the text.
	 */
	public void setTextColor(Color color) {
		textColor = color;
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)SquareCanvasHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import com.rolemodelsoft.drawlet.*;
import java.awt.*;

/**
 * Since many handles will be simple small squares (probably appearing on a 
 * DrawingCanvas), this abstract class offers a simple base for handles which
 * appear this way.
 * All subclasses need to provide, at a minimum, are the coordinates for the center
 * of the square via:
 *	centerX()
 *	centerY()
 *
 * @version 	1.1.6, 12/28/98
 */
public abstract class SquareCanvasHandle extends CanvasHandle {
	/**  
	 * Answer the x coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the x coordinate in the center of the
	 * handle
	 */
	protected abstract int centerX();
	/**  
	 * Answer the y coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the y coordinate in the center of the
	 * handle
	 */
	protected abstract int centerY();
	/** 
	 * Returns the current bounds of this handle.
	 * 
	 * @return	a Rectangle representing the current bounds of this handle
	 */
	public Rectangle getBounds()  {
		return new Rectangle(centerX() - halfWidth, centerY() - halfWidth, halfWidth * 2, halfWidth * 2);
	}
	/** 
	 * Paints the handle.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		g.fillRect(centerX() - halfWidth, centerY() - halfWidth, 2*halfWidth, 2*halfWidth);
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_AbstractFigure.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.text.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import java.awt.*;
import junit.framework.*;


public class TC_AbstractFigure extends TestCase {
	protected AbstractFigure figure;
	protected PropertyChangeRevealer propertyRevealer;
	protected RelatedLocationRevealer locationRevealer;
/**
 * BasicStringRendererTest constructor comment.
 * @param name java.lang.String
 */
public TC_AbstractFigure(String name) {
	super(name);
}
/**
 * Sets up the fixture, for example, open a network connection.
 * This method is called before a test is executed.
 */
protected void setUp() 
{
	figure = new RectangleShape();
	figure.setBounds( 5, 5, 10, 10 );
	propertyRevealer = new PropertyChangeRevealer();
	locationRevealer = new RelatedLocationRevealer();
}
/**
 * Test to make sure PropertyChangeListeners are being added
 * correctly.
 */
public void testAddPropertyChangeListener() 
{
	figure.addPropertyChangeListener( propertyRevealer );
	figure.translate( 5, 5 );
	assert( "No PropertyChangeEvent propagated", propertyRevealer.getEventCount() == 1 );

	propertyRevealer.clearEventCount();
	figure.addPropertyChangeListener( propertyRevealer );
	figure.translate( 5, 5 );
	assert( "Two PropertyChangeEvents propagated", propertyRevealer.getEventCount() == 1 );
}
/**
 * Test to make sure RelatedLocationListeners are being added
 * correctly.
 */
public void testAddRelatedLocationListener() 
{
	figure.addRelatedLocationListener( locationRevealer );
	figure.translate( 5, 5 );
	assert( "No RelatedLocatorEvent propagated", locationRevealer.getEventCount() == 1 );

	locationRevealer.clearEventCount();
	figure.addRelatedLocationListener( locationRevealer );
	figure.translate( 5, 5 );
	assert( "Two RelatedLocatorEvents propagated", locationRevealer.getEventCount() == 1 );
}
/**
 * Test to make sure the contains( Figure ) method is
 * functioning properly.
 */
public void testContainsFigure() 
{
	Figure testFigure = new RectangleShape( 6, 6, 8, 8 );
	assert( "Figure does not contain the figure", figure.contains( testFigure ) );
	testFigure.setBounds( 4, 4, 12, 12 );
	assert( "Figure contains the figure", ! figure.contains( testFigure ) );
}
/**
 * Test to make sure the contains( int, int ) method is
 * functioning properly.
 */
public void testContainsIntInt() 
{
	assert( "Figure does not contain the point", figure.contains( 6, 6 ) );
	assert( "Figure contains the point", ! figure.contains( 20, 20 ) );
}
/**
 * Test to make sure the contains( Rectangle ) method is
 * functioning properly.
 */
public void testContainsRectangle() 
{
	Rectangle testRect = new Rectangle( 6, 6, 8, 8 );
	assert( "Figure does not contain the rectangle", figure.contains( testRect ) );
	testRect.setBounds( 4, 4, 12, 12 );
	assert( "Figure contains the rectangle", ! figure.contains( testRect ) );
}
/**
 * Test to make sure that (a) the figure is firing the
 * disconnect event, and (b) that it then doesn't fire
 * anything else ( at least until someone else registers).
 */
public void testDisconnect() 
{
	figure.addRelatedLocationListener( locationRevealer );
	figure.disconnect();
	assert( "No disconnect event was fired", locationRevealer.getEventCount() == 1 );
	locationRevealer.clearEventCount();
	figure.translate( 5, 5 );
	assert( "A RelatedLocation event was fired", locationRevealer.getEventCount() == 0 );
}
/**
 * Test to make sure that the edit tool returned is null.
 * Subclasses should override if they return expect a
 * different result.
 */
public void testEditTool() 
{
	assert( "The edit tool returned was not null. You probably need to override testEditTool.", figure.editTool( 0, 0 ) == null );
}
/**
 * Test to make sure the bottom is properly returned.
 */
public void testGetBottom() 
{
	assertEquals( "The int returned does not correspond to the expected bottommost coordinate of the figure", 15, figure.getBottom() );
}
/**
 * Test to make sure the handles are properly returned.
 */
public void testGetHandles() 
{
	assert( "There were no handles returned.", figure.getHandles().length > 0 );
}
/**
 * Test to make sure the height of the figure is properly
 * returned.
 */
public void testGetHeight() 
{
	assertEquals( "The height was not what was expected", 10, figure.getHeight() );
}
/**
 * Test to make sure the leftmost coordinate of the figure
 * is properly returned.
 */
public void testGetLeft() 
{
	assertEquals( "The int returned does not correspond to the expected leftmost coordinate of the figure", 5, figure.getLeft() );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testGetLocator() 
{
	assert( "There was no locator returned.", figure.getLocator() != null );
	assert( "The locator returned does not correspond to the top left corner of this figure. If that is OK, you need to override this test method.", figure.getLocator().x() == 5 && figure.getLocator().y() == 5 );
}
/**
 * Test to make sure the rightmost coordinate of the figure
 * is properly returned.
 */
public void testGetRight() 
{
	assertEquals( "The int returned does not correspond to the expected rightmost coordinate of the figure", 15, figure.getRight() );
}
/**
 * Test to make sure the size of the figure is properly
 * returned.
 */
public void testGetSize() 
{
	assert( "The size was not what was expected; it was " + figure.getSize().toString() + "instead of 10 by 10.", figure.getSize().width == 10 && figure.getSize().height == 10 );
}
/**
 * Test to make sure the style returned is correct.
 */
public void testGetStyle() 
{
	assert( "The style returned was null. If this is allowable, you need to override this test method.", figure.getStyle() != null );
	assert( "The style returned was not a SimpleDrawingStyle. If this is allowable, you need to override this test method.", figure.getStyle() instanceof SimpleDrawingStyle );
}
/**
 * Test to make sure the top is properly returned.
 */
public void testGetTop() 
{
	assertEquals( "The int returned does not correspond to the expected topmost coordinate of the figure", 5, figure.getTop() );
}
/**
 * Test to make sure the width of the figure is properly
 * returned.
 */
public void testGetWidth() 
{
	assertEquals( "The width was not what was expected.", 10, figure.getWidth() );
}
/**
 * Test to make sure the intersects( Figure ) method is
 * functioning properly.
 */
public void testIntersectsFigure() 
{
	Figure testFigure = new RectangleShape( 6, 6, 2, 2 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 2, 2, 5, 5 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 16, 16, 6, 6 );
	assert( "Figure intersects the figure", ! figure.intersects( testFigure ) );
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectsRectangle() 
{
	Rectangle testRect = new Rectangle( 6, 6, 2, 2 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 2, 2, 5, 5 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 16, 16, 6, 6 );
	assert( "Figure intersects the rectangle", ! figure.intersects( testRect ) );
}
/**
 * Test to make sure the isObsolete() is functioning
 * properly.
 */
public void testIsObsolete() 
{
	assert( "The figure was obselete; if that is a valid state, you should override this test method.", ! figure.isObsolete() );
}
/**
 * Test to make sure the isWithin( Figure ) method is
 * functioning properly.
 */
public void testIsWithinFigure() 
{
	Figure testFigure = new RectangleShape( 4, 4, 12, 12 );
	assert( "Figure isn't within the figure", figure.isWithin( testFigure ) );
	testFigure.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the figure", ! figure.isWithin( testFigure ) );
}
/**
 * Test to make sure the isWithin( Rectangle ) method is
 * functioning properly.
 */
public void testIsWithinRectangle() 
{
	Rectangle testRect = new Rectangle( 4, 4, 12, 12 );
	assert( "Figure isn't within the rectangle", figure.isWithin( testRect ) );
	testRect.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the rectangle", ! figure.isWithin( testRect ) );
}
/**
 * Test to make sure locatorAt returns the proper value.
 */
public void testLocatorAt() 
{
	Locator locator = figure.locatorAt( 0, 0 );
	if( ! ( locator instanceof FigureRelativePoint ) ) {
		assert( "The locator returned isn't a FigureRelativePoint", false );
		return;
	}
	FigureRelativePoint relativeLocator = (FigureRelativePoint) locator;
	assert( "The relativeLocator returned doesn't have the figure as its owner", relativeLocator.getFigure() == figure );
	assert( "The relativeLocator has an improper x", relativeLocator.x() == 0 );
	assert( "The relativeLocator has an improper y", relativeLocator.y() == 0 );
	figure.translate( 1, 1 );
	assert( "The relativeLocator improperly adjusted its x", relativeLocator.x() == 1 );
	assert( "The relativeLocator improperly adjusted its y", relativeLocator.y() == 1 );
}
/**
 * Test to make sure move( int, int ) works correctly.
 */
public void testMoveIntInt() 
{
	figure.move( 10, 10 );
	Locator loc = figure.getLocator();
	assertEquals( "The figure did not move properly", 10, loc.x() );
	assertEquals( "The figure did not move properly", 10, loc.y() );
}
/**
 * Test to make sure move( Locator ) works correctly.
 */
public void testMoveLocator()
{
	figure.move( new DrawingPoint( 10, 10 ) );
	Locator loc = figure.getLocator();
	assertEquals( "The figure did not move properly", 10, loc.x() );
	assertEquals( "The figure did not move properly", 10, loc.y() );
}
/**
 * Test to make sure relatedLocationListeners
 * works correctly.
 */
public void testRelatedLocationListeners()
{
	assert( "The figure already had location listeners", ! figure.relatedLocationListeners().hasMoreElements() );
	figure.addRelatedLocationListener( locationRevealer );
	assert( "The figure did not have any location listeners", figure.relatedLocationListeners().hasMoreElements() );
	assert( "The figure did not properly add the location listener", ( (RelatedLocationRevealer) figure.relatedLocationListeners().nextElement() ) == locationRevealer );
}
/**
 * Test to make sure PropertyChangeListeners are being
 * removed correctly.
 */
public void testRemovePropertyChangeListener() 
{
	figure.addPropertyChangeListener( propertyRevealer );
	figure.removePropertyChangeListener( propertyRevealer );
	figure.translate( 5, 5 );
	assert( "A PropertyChangeEvent propagated", propertyRevealer.getEventCount() == 0 );
}
/**
 * Test to make sure RelatedLocationListeners are being
 * removed correctly.
 */
public void testRemoveRelatedLocationListener() 
{
	figure.addRelatedLocationListener( locationRevealer );
	figure.removeRelatedLocationListener( locationRevealer );
	figure.translate( 5, 5 );
	assert( "RelatedLocatorEvent propagated", locationRevealer.getEventCount() == 0 );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testRequestConnection() 
{
	Locator locator = figure.requestConnection( new RectangleShape(), 0, 0 );
	if( locator == null ) {
		assert( "There was no locator returned.", false );
		return;
	}
	assert( "The locator returned does not correspond to the center of this figure. If that is OK, you need to override this test method.", locator.x() == 10 && locator.y() == 10 );
	if ( locator instanceof FigureRelativePoint ) {
		FigureRelativePoint relativeLocator = (FigureRelativePoint) locator;
		assert( "The locator returned does not have the figure as its owner.", relativeLocator.getFigure() == figure );
		figure.translate( 1, 1 );
		assert( "The locator did not update when the figure moved.", relativeLocator.x() == 11 && relativeLocator.y() == 11 );
	} else {
		assert( "The locator returned is not a FigureRelativePoint. If that is OK, you need to override this test method.", false );
	}
}
/**
 * Test to make sure the bounds are properly set.
 */
public void testSetBounds()
{
	assert( "The figure's left side is incorrect.", figure.getLeft() == 5 );
	assert( "The figure's right side is incorrect.", figure.getRight() == 15 );
	assert( "The figure's top side is incorrect.", figure.getTop() == 5 );
	assert( "The figure's bottom side is incorrect.", figure.getBottom() == 15 );
	
}
/**
 * Test to make sure the setSize( Dimension ) method is
 * working properly.
 */
public void testSetSizeDimension()
{
	figure.setSize( new Dimension( 100, 100 ) );
	assert( "The figure's width is incorrect.", figure.getWidth() == 100 );
	assert( "The figure's height is incorrect.", figure.getHeight() == 100 );
}
/**
 * Test to make sure the setSize( int, int ) method is
 * working properly.
 */
public void testSetSizeIntInt()
{
	figure.setSize( 100, 100 );
	assert( "The figure's width is incorrect.", figure.getWidth() == 100 );
	assert( "The figure's height is incorrect.", figure.getHeight() == 100 );
}
/**
 * Test to make sure the setStyle method is working properly.
 */
public void testSetStyle()
{
	figure.addPropertyChangeListener( propertyRevealer );
	DrawingStyle oldStyle = figure.getStyle();
	DrawingStyle newStyle = new SimpleDrawingStyle();
	figure.setStyle( newStyle );
	assert( "The figure did not properly fire a property change event.", propertyRevealer.getEventCount() > 0 );
	assert( "The event's type is incorrect; it should be " + Figure.STYLE_PROPERTY + ", but it is " + propertyRevealer.getPropertyChangeEvent().getPropertyName() + " instead.", propertyRevealer.getPropertyChangeEvent().getPropertyName() == Figure.STYLE_PROPERTY );
	assert( "the event's newValue is incorrect", propertyRevealer.getPropertyChangeEvent().getNewValue() == newStyle );
}
/**
 * Test to make sure translate works correctly.
 */
public void testTranslate() 
{
	translateByExpecting( 5, 5 , 10, 10 );
	translateByExpecting( -5, -5, 5, 5);
}
/**
 * Test to make sure translate works correctly.
 */
protected void translateByExpecting(int startX, int startY, int expectedX, int expectedY) 
{
	figure.translate( startX, startY );
	assertEquals( "The figure did not translate properly", new Point( expectedX, expectedY ), new Point( figure.getLeft(), figure.getTop() ) );
}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_CanvasTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import junit.framework.*;
import java.awt.*;
import java.awt.event.*;

/**
 * 
 */
public class TC_CanvasTool extends TestCase{
	SimpleDrawingCanvas canvas;
	AbstractInputEventHandler tool;
	MouseEvent event;
/**
 * @param name java.lang.String
 */
public TC_CanvasTool(String name) {
	super(name);
}
/**
 */
public void setUp() {
	canvas = new SimpleDrawingCanvas();
	new DrawingCanvasComponent( canvas );
	tool = new SelectionTool( canvas );
	event = new MouseEvent( canvas.getComponent(), MouseEvent.MOUSE_RELEASED, System.currentTimeMillis(), 0, 0, 0, 1, false );
}
/**
 * Test to make sure that the event is consumed when the mouse is released.
 */
public void testMouseReleased() {
	tool.mouseReleased( event );
	assert( "The event was not consumed.", event.isConsumed() );
}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_DrawingPoint.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import junit.framework.*;
import junit.ui.*;
import java.util.*;
/**
 * 
 */
public class TC_DrawingPoint extends TestCase {
	DrawingPoint loc1, loc2, loc3;
	/**
	 * @param name java.lang.String
	 */
	public TC_DrawingPoint(String name) {
		super(name);
	}

	public void setUp() {
		loc1 = new DrawingPoint( 0, 0 );
		loc2 = new DrawingPoint( 10, 10 );
		loc3 = new DrawingPoint( -10, -10 );
	}
	/**
	 *
	 */
	public void testDuplicate() {
		Object obj = loc1.duplicate();
		assert( "The object wasn't a drawingpoint", obj instanceof DrawingPoint );
		DrawingPoint loc = (DrawingPoint)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}
	/**
	 *
	 */
	public void testDuplicateIn() {
		Hashtable hash = new Hashtable();
		loc1.duplicateIn( hash );
		Object obj = hash.get( loc1 );
		assert( "The object wasn't a drawingpoint", obj instanceof DrawingPoint );
		DrawingPoint loc = (DrawingPoint)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}
	/**
	 *
	 */
	public void testMoveIntInt() {
		loc1.move( 5, 5 );
		assertEquals( 5, loc1.x() );
		assertEquals( 5, loc1.y() );
		
	}
	/**
	 *
	 */
	public void testPostDuplicate() {
	}
	/**
	 *
	 */
	public void testR() {
		assertEquals( 0, loc1.r() );
		assertEquals( 14, loc2.r() );
		assertEquals( 14, loc3.r() );
		
	}
	/**
	 *
	 */
	public void testTheta() {
		assertEquals( new Double( Double.NaN ), new Double( loc1.theta() ) );
		assertEquals( new Double( 0.7853981633974483 ), new Double( loc2.theta() ) );
		assertEquals( new Double( 3.9269908169872414 ), new Double( loc3.theta() ) );
		
	}
	/**
	 *
	 */
	public void testToString() {
		assertEquals( "com.rolemodelsoft.drawlet.basics.DrawingPoint [x=0,y=0;r=0,theta=NaN]", loc1.toString() );
		assertEquals( "com.rolemodelsoft.drawlet.basics.DrawingPoint [x=10,y=10;r=14,theta=0.7853981633974483]", loc2.toString() );
	}
	/**
	 *
	 */
	public void testTranslateIntInt() {
		loc2.translate( 5, 5 );
		assertEquals( 15, loc2.x() );
		assertEquals( 15, loc2.y() );
		
	}
	/**
	 *
	 */
	public void testX() {
		assertEquals( 0, loc1.x() );
		assertEquals( 10, loc2.x() );
		assertEquals( -10, loc3.x() );
	}
	/**
	 *
	 */
	public void testXInt() {
		loc1.x( 5 );
		assertEquals( 5, loc1.x() );
	}
	/**
	 *
	 */
	public void testY() {
		assertEquals( 0, loc1.y() );
		assertEquals( 10, loc2.y() );
		assertEquals( -10, loc3.y() );
		
	}
	/**
	 *
	 */
	public void testYInt() {
		loc1.y( 5 );
		assertEquals( 5, loc1.y() );
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_EdgeLocator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import junit.framework.*;
import junit.ui.*;
import java.util.*;
/**
 * 
 */
public class TC_EdgeLocator extends TestCase {
	protected Figure figure;
	protected Locator loc1, loc2, loc3;

	public TC_EdgeLocator(String name) {
		super(name);
	}

	public void setUp() {
		figure = new RectangleShape(10, 10, 20, 20);
		loc1 = new DrawingPoint(0,0);
		loc2 = new FigureRelativePoint(figure, 0.5, 0.5);
		loc3 = new EdgeLocator(loc2, loc1);
		
	}

	public void testDuplicate() {
		Object obj = loc1.duplicate();
		assert( "The object wasn't a EdgeLocator", obj instanceof EdgeLocator );
		EdgeLocator loc = (EdgeLocator)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}

	public void testDuplicateIn() {
		Hashtable hash = new Hashtable();
		loc1.duplicateIn( hash );
		Object obj = hash.get( loc1 );
		assert( "The object wasn't a EdgeLocator", obj instanceof EdgeLocator );
		EdgeLocator loc = (EdgeLocator)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}

	public void testPostDuplicate() {
	}

	public void testR() {
		assertEquals( 14, loc3.r() );
		
	}

	public void testTheta() {
		assertEquals( new Double( 0.7853981633974483 ), new Double( loc3.theta() ) );
		
	}

	public void testToString() {
		assertEquals( "com.rolemodelsoft.drawlet.basics.EdgeLocator [x=10,y=10;r=14,theta=0.7853981633974483]", loc3.toString() );
	}

	public void testX() {
		assertEquals( 10, loc3.x() );
	}

	public void testY() {
		assertEquals( 10, loc3.y() );
		
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_FigureRelativePoint.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import junit.framework.*;
import junit.ui.*;
import java.util.*;
/**
 * 
 */
public class TC_FigureRelativePoint extends TestCase {
	protected Figure figure;
	protected Locator loc1, loc2, loc3;

	public TC_FigureRelativePoint(String name) {
		super(name);
	}

	public void setUp() {
		figure = new RectangleShape(0, 0, 20, 20);
		loc1 = new FigureRelativePoint(figure, 0.0, 0.0);
		loc2 = new FigureRelativePoint(figure, 0.5, 0.5);
		loc3 = new FigureRelativePoint(figure, 1.0, 1.0, -10, -10);
	}

	public void testDuplicate() {
		Object obj = loc1.duplicate();
		assert( "The object wasn't a FigureRelativePoint", obj instanceof FigureRelativePoint );
		FigureRelativePoint loc = (FigureRelativePoint)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}

	public void testDuplicateIn() {
		Hashtable hash = new Hashtable();
		loc1.duplicateIn( hash );
		Object obj = hash.get( loc1 );
		assert( "The object wasn't a FigureRelativePoint", obj instanceof FigureRelativePoint );
		FigureRelativePoint loc = (FigureRelativePoint)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}

	public void testPostDuplicate() {
	}

	public void testR() {
		assertEquals( 0, loc1.r() );
		assertEquals( 14, loc2.r() );
		assertEquals( 14, loc3.r() );
		
	}

	public void testTheta() {
		assertEquals( new Double( Double.NaN ), new Double( loc1.theta() ) );
		assertEquals( new Double( 0.7853981633974483 ), new Double( loc2.theta() ) );
		assertEquals( new Double( 0.7853981633974483 ), new Double( loc3.theta() ) );
		
	}

	public void testToString() {
		assertEquals( "com.rolemodelsoft.drawlet.basics.FigureRelativePoint [x=0,y=0;r=0,theta=NaN]", loc1.toString() );
		assertEquals( "com.rolemodelsoft.drawlet.basics.FigureRelativePoint [x=10,y=10;r=14,theta=0.7853981633974483]", loc2.toString() );
	}

	public void testX() {
		assertEquals( 0, loc1.x() );
		assertEquals( 10, loc2.x() );
		assertEquals( 10, loc3.x() );
	}

	public void testY() {
		assertEquals( 0, loc1.y() );
		assertEquals( 10, loc2.y() );
		assertEquals( 10, loc3.y() );
		
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_PolarCoordinate.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import junit.framework.*;
import junit.ui.*;
import java.util.*;
/**
 * 
 */
public class TC_PolarCoordinate extends TestCase {
	PolarCoordinate loc1, loc2, loc3;

	public TC_PolarCoordinate(String name) {
		super(name);
	}

	public void setUp() {
		loc1 = new PolarCoordinate( 0, 0.0 );
		loc2 = new PolarCoordinate( 10, 0.5 );
		loc3 = new PolarCoordinate( 40, -0.5 );
	}

	public void testDuplicate() {
		Object obj = loc1.duplicate();
		assert( "The object wasn't a drawingpoint", obj instanceof PolarCoordinate );
		PolarCoordinate loc = (PolarCoordinate)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}

	public void testDuplicateIn() {
		Hashtable hash = new Hashtable();
		loc1.duplicateIn( hash );
		Object obj = hash.get( loc1 );
		assert( "The object wasn't a drawingpoint", obj instanceof PolarCoordinate );
		PolarCoordinate loc = (PolarCoordinate)obj;
		assertEquals( loc1.x(), loc.x() );
		assertEquals( loc1.y(), loc.y() );
	}

	public void testPostDuplicate() {
	}

	public void testR() {
		assertEquals( 0, loc1.r() );
		assertEquals( 10, loc2.r() );
		assertEquals( 40, loc3.r() );
		
	}

	public void testTheta() {
		assertEquals( new Double( 0.0 ), new Double( loc1.theta() ) );
		assertEquals( new Double( 0.5 ), new Double( loc2.theta() ) );
		assertEquals( new Double( -0.5 ), new Double( loc3.theta() ) );
		
	}

	public void testToString() {
		assertEquals( "com.rolemodelsoft.drawlet.basics.PolarCoordinate [x=0,y=0;r=0,theta=0.0]", loc1.toString() );
		assertEquals( "com.rolemodelsoft.drawlet.basics.PolarCoordinate [x=8,y=4;r=10,theta=0.5]", loc2.toString() );
	}

	public void testX() {
		assertEquals( 0, loc1.x() );
		assertEquals( 8, loc2.x() );
		assertEquals( 35, loc3.x() );
	}

	public void testY() {
		assertEquals( 0, loc1.y() );
		assertEquals( 4, loc2.y() );
		assertEquals( -19, loc3.y() );
		
	}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_SimpleDrawing.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import junit.framework.*;
import junit.ui.*;
import java.awt.*;
/**
 * 
 */
public class TC_SimpleDrawing extends TestCase {
	protected SimpleDrawing drawing;
	protected Figure figure1, figure2, figure3;
	protected PropertyChangeRevealer propertyRevealer;
/**
 * SimpleDrawingTest constructor comment.
 * @param name java.lang.String
 */
public TC_SimpleDrawing(String name) {
	super(name);
}
/**
 * Sets up this test.
 */
public void setUp() {
	drawing = new SimpleDrawing( 200, 200 );
	figure1 = new RectangleShape( 0, 0, 10, 10 );
	figure2 = new RectangleShape( 0, 0, 10, 10 );
	figure3 = new RectangleShape( 0, 0, 10, 10 );

	drawing.addFigure( figure1 );

	propertyRevealer = new PropertyChangeRevealer();
}

public void testAddFigure() {
	FigureEnumeration enum = drawing.figures();
	assert( "The figure was not added.", enum.hasMoreElements() );
	enum.nextElement();
	assert( "More than one figure was added.", ! enum.hasMoreElements() );
}

public void testAddFigureBehind() {
	drawing.addFigureBehind( figure2, figure1 );
	FigureEnumeration enum = drawing.figures();
	assertEquals( "The figure was not added behind.", figure2, enum.nextElement() );
	assert( "The second figure was not added.", enum.hasMoreElements() );
}
/**
 * Test to make sure PropertyChangeListeners are being added
 * correctly.
 */
public void testAddPropertyChangeListener() 
{
	drawing.addPropertyChangeListener( propertyRevealer );
	drawing.setSize( 100, 100 );
	assert( "No PropertyChangeEvent propagated", propertyRevealer.getEventCount() == 1 );

	propertyRevealer.clearEventCount();
	drawing.addPropertyChangeListener( propertyRevealer );
	drawing.setSize( 200, 200 );
	assert( "Two PropertyChangeEvents propagated", propertyRevealer.getEventCount() == 1 );
}

public void testFigureAt() {
	drawing.addFigureBehind( figure2, figure1 );
	FigureEnumeration enum = drawing.figures();
	assertEquals( "The correct figure was not returned.", figure1, drawing.figureAt( 5, 5 ) );
}

public void testFigures() {
	FigureEnumeration enum = drawing.figures();
	assert( "The enumeration is null.", enum != null );
	assert( "The enumeration is empty.", enum.hasMoreElements() );
}

public void testGetBounds() {
	drawing.setSize( 10, 10 );
	assertEquals( "The bounds returned were incorrect.", new Rectangle( 0, 0, 10, 10 ), drawing.getBounds() );
	drawing.setDynamicSize( true );
	Rectangle rec = new Rectangle( 0, 0, 0, 0 );
	rec.add( figure1.getBounds() );
	assertEquals( "The bounds returned were incorrect.", rec, drawing.getBounds() );
	drawing.removeFigure( figure1 );
	assertEquals( "The bounds were not empty.", new Rectangle( 0, 0, 0, 0 ), drawing.getBounds() );
}
/**
 * Tests getSize().
 */
public void testGetSize() {
	drawing.setSize( 10, 10 );
	assertEquals( "The size returned were incorrect.", new Dimension( 10, 10 ), drawing.getSize() );
	drawing.setDynamicSize( true );
	Rectangle rec = new Rectangle( 0, 0, 0, 0 );
	rec.add( figure1.getBounds() );
	assertEquals( "The size was incorrect.", rec.getSize(), drawing.getSize() );
	drawing.removeFigure( figure1 );
	assertEquals( "The size was incorrect.", (new Rectangle( 0, 0, 0, 0 )).getSize(), drawing.getSize() );
}
/**
 * Tests getStyle().
 */
public void testGetStyle() {
	assertEquals( "The style returned was incorrect.", SystemColor.window, drawing.getStyle().getBackgroundColor() );
}

public void testIsDynamicSize() {
	assert( "The size was dynamic.", ! drawing.isDynamicSize() );
}

public void testMoveFigureBehind() {
	drawing.addFigure( figure2 );
	assertEquals( "Figure 2 was not in the front", figure2, drawing.figureAt( 5, 5 ) );
	drawing.moveFigureBehind( figure2, figure1 );
	assertEquals( "Figure 1 was not in the front", figure1, drawing.figureAt( 5, 5 ) );
}

public void testMoveFigureInFront() {
	drawing.addFigure( figure2 );
	assertEquals( "Figure 2 was not in the front", figure2, drawing.figureAt( 5, 5 ) );
	drawing.moveFigureInFront( figure1, figure2 );
	assertEquals( "Figure 1 was not in the front", figure1, drawing.figureAt( 5, 5 ) );
}

public void testMoveFigureToBack() {
	drawing.addFigure( figure2 );
	drawing.addFigure( figure3 );
	assertEquals( "Figure 3 was not in the front", figure3, drawing.figureAt( 5, 5 ) );
	drawing.moveFigureToBack( figure3 );
	assertEquals( "Figure 3 was not in the back", figure3, drawing.figures().nextElement() );
}

public void testMoveFigureToFront() {
	drawing.addFigure( figure2 );
	drawing.addFigure( figure3 );
	assertEquals( "Figure 3 was not in the front", figure3, drawing.figureAt( 5, 5 ) );
	drawing.moveFigureToFront( figure1 );
	assertEquals( "Figure 1 was not in the front", figure1, drawing.figureAt( 5, 5 ) );
}

public void testOtherFigureAt() {
	drawing.addFigure( figure2 );
	assertEquals( "Figure 2 was not in the front", figure2, drawing.figureAt( 5, 5 ) );
	assertEquals( "Figure 1 was not returned", figure1, drawing.otherFigureAt( figure2, 5, 5 ) );
}

public void testRemoveFigure() {
	drawing.removeFigure( figure1 );
	assert( "Figure 1 was not removed", ! drawing.figures().hasMoreElements() );
}
/**
 * Test to make sure PropertyChangeListeners are being
 * removed correctly.
 */
public void testRemovePropertyChangeListener() 
{
	drawing.addPropertyChangeListener( propertyRevealer );
	drawing.removePropertyChangeListener( propertyRevealer );
	drawing.setSize( 100, 100 );
	assert( "A PropertyChangeEvent propagated", propertyRevealer.getEventCount() == 0 );
}
/**
 * Tests setDynamicSize().
 */
public void testSetDynamicSize() {
	drawing.setDynamicSize( true );
	assert( "The size wasn't dynamic.", drawing.isDynamicSize() );
}

public void testSetSizeDimension() {
	drawing.setSize( new Dimension( 10, 10 ) );
	assertEquals( "The size returned was incorrect.", new Dimension( 10, 10 ), drawing.getSize() );
	drawing.setDynamicSize( true );
	drawing.setSize( 500, 500 );
	Rectangle rec = new Rectangle( 0, 0, 0, 0 );
	rec.add( figure1.getBounds() );
	assertEquals( "The size returned was incorrect.", rec.getSize(), drawing.getSize() );
}

public void testSetSizeIntInt() {
	drawing.setSize( 10, 10 );
	assertEquals( "The size returned was incorrect.", new Dimension( 10, 10 ), drawing.getSize() );
	drawing.setDynamicSize( true );
	drawing.setSize( 500, 500 );
	Rectangle rec = new Rectangle( 0, 0, 0, 0 );
	rec.add( figure1.getBounds() );
	assertEquals( "The size returned was incorrect.", rec.getSize(), drawing.getSize() );
}
/**
 * Tests setStyle().
 */
public void testSetStyle() {
	DrawingStyle style = new SimpleDrawingStyle();
	style.setBackgroundColor( Color.red );
	drawing.setStyle( style );
	assertEquals( "The style returned was incorrect.", Color.red, drawing.getStyle().getBackgroundColor() );
}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TC_SimpleDrawingCanvas.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.shapes.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.text.*;
import junit.framework.*;
import junit.ui.*;
import java.awt.*;
import java.awt.datatransfer.*;
/**
 * 
 */
public class TC_SimpleDrawingCanvas extends TestCase {
	DrawingCanvas canvas, staticCanvas;
	Drawing drawing;
	SimpleDrawingCanvas simpleCanvas;
	DrawingCanvasComponent component, staticComponent;
	Figure figure1, figure2, figure3;
	Handle handle;
/**
 * SimpleDrawingCanvasTest constructor comment.
 * @param name java.lang.String
 */
public TC_SimpleDrawingCanvas(String name) {
	super(name);
}
/**
 * Sets up this test
 */
public void setUp() {
	canvas = new SimpleDrawingCanvas( drawing = new SimpleDrawing() );
	simpleCanvas = (SimpleDrawingCanvas) canvas;
	component = new DrawingCanvasComponent(canvas);
	figure1 = new RectangleShape( 0, 0, 10, 10 );
	figure2 = new RectangleShape( 0, 0, 10, 10 );
	figure3 = new RectangleShape( 0, 0, 10, 10 );
	handle = new BoxSelectionHandle( 5, 5, 10, 10 );
	canvas.addFigure( figure1 );

	staticCanvas = new SimpleDrawingCanvas( new SimpleDrawing( 50, 50 ) );
	staticComponent = new DrawingCanvasComponent( staticCanvas );
	staticComponent.setSize( 10, 10 );
}
/**
 * Test addFigure.
 */
public void testAddFigure() {
	assert( "The figure was not added.", canvas.figures().hasMoreElements() );
	assertEquals( "The figure was not added properly.", figure1, canvas.figures().nextElement() );
}
/**
 * Test addFigureBehind. Most of the difference is actually
 * drawing, so not much is different.
 */
public void testAddFigureBehind() {
	canvas.addFigureBehind( figure2, figure1 );
	assert( "The figures were not added.", canvas.figures().hasMoreElements() );
	assertEquals( "The second figure was not added properly.", figure2, canvas.figures().nextElement() );
}
/**
 * Test addFigureBehind. Most of the difference is actually
 * drawing, so not much is different.
 */
public void testAddHandle() {
	assert( "There were already handles on the canvas.", canvas.getHandles().length == 0 );
	canvas.addHandle( handle );
	assert( "There was not a handle added.", canvas.getHandles().length == 1 );
	assertEquals( "The handle was not added properly.", handle, canvas.getHandles()[0] );
}
/**
 * Test addFigureBehind. Most of the difference is actually
 * drawing, so not much is different.
 */
public void testAddHandlesFigure() {
	assert( "There were already handles on the canvas.", canvas.getHandles().length == 0 );
	canvas.addHandles( figure1 );
	assert( "The wrong number of handles were added; was " + String.valueOf( canvas.getHandles().length ) + " instead of 8.", canvas.getHandles().length == 8 );
	assert( "The handles were not added properly.", figure1.getHandles()[0] instanceof TopLeftHandle );
}

public void testAddHandlesFigureHandle() {
	assert( "There were already handles on the canvas.", canvas.getHandles().length == 0 );
	Handle handles[] = { handle };
	canvas.addHandles( figure1, handles );
	assert( "The wrong number of handles were added; was " + String.valueOf( canvas.getHandles().length ) + " instead of 1.", canvas.getHandles().length == 1 );
	assertEquals( "The handles were not added properly.", handle, canvas.getHandles()[0] );
}
/**
 * Test addSelection.
 */
public void testAddSelection() {
	assert( "There was already a selection.", canvas.getSelections().length == 0 );
	canvas.addSelection( figure1 );
	assertEquals( "The figure was not added correctly.", figure1, canvas.getSelections()[0] );
}

public void testClearSelections() {
	canvas.addSelection( figure1 );
	canvas.addSelection( figure2 );
	assert( "The figures were not added.", canvas.getSelections().length == 2 );
	canvas.clearSelections();
	assert( "The figures were not removed.", canvas.getSelections().length == 0 );
}

public void testCopySelections() {
	canvas.addSelection( new TextLabel( "This is test text" ) );
	assertEquals( "The text label was not added.", 1, canvas.getSelections().length );
	((SimpleDrawingCanvas)canvas).copySelections();
	assert( "The text flavor was not on the clipboard", Toolkit.getDefaultToolkit().getSystemClipboard().getContents( this ).isDataFlavorSupported( DataFlavor.stringFlavor ) );
	try {
		assertEquals( "The text was not put on the system clipboard.", "This is test text", (String) Toolkit.getDefaultToolkit().getSystemClipboard().getContents( this ).getTransferData( DataFlavor.stringFlavor ) );
	} catch ( Exception e ) {
		assert( "An exception was thrown", false );
	}
}
/**
 * Test cutSelections.
 */
public void testCutSelections() {
	canvas.removeFigure( figure1 );
	TextLabel label = new TextLabel( "This is test text" );
	canvas.addFigure( label );
	canvas.addSelection( label );
	assertEquals( "The text label was not added.", 1, canvas.getSelections().length );
	((SimpleDrawingCanvas)canvas).cutSelections();
	assert( "The text flavor was not on the clipboard", Toolkit.getDefaultToolkit().getSystemClipboard().getContents( this ).isDataFlavorSupported( DataFlavor.stringFlavor ) );
	try {
		assertEquals( "The text was not put on the system clipboard.", "This is test text", (String) Toolkit.getDefaultToolkit().getSystemClipboard().getContents( this ).getTransferData( DataFlavor.stringFlavor ) );
	} catch ( Exception e ) {
		assert( "An exception was thrown", false );
	}
	assert( "There were still figures left", ! canvas.figures().hasMoreElements() );
}

public void testDeleteSelections() {
	canvas.addSelection( figure1 );
	((SimpleDrawingCanvas)canvas).deleteSelections();
	assert( "There were still some figures left", ! canvas.figures().hasMoreElements() );
}

public void testFigureAt() {
	canvas.addFigureBehind( figure2, figure1 );
	FigureEnumeration enum = canvas.figures();
	assertEquals( "The correct figure was not returned.", figure1, canvas.figureAt( 5, 5 ) );
}

public void testFigures() {
	assert( "The FigureEnumeration was null.", canvas.figures() != null );
	assert( "The FigureEnumeration was empty.", canvas.figures().hasMoreElements() );
}
/**
 * Test getBounds.
 */
public void testGetBounds() {
	component.setSize( 300, 300 );
	assertEquals( "The bounds were incorrect", new Rectangle( 0, 0, 300, 300 ), canvas.getBounds() );
	drawing.setDynamicSize( false );
	component.setSize( 400, 400 );
	assertEquals( "The bounds were incorrect", new Rectangle( 0, 0, 300, 300 ), canvas.getBounds() );
}
/**
 * Test getComponent.
 */
public void testGetComponent() {
	assert( "The component was null.", simpleCanvas.getComponent() != null );
}

public void testGetHandles() {
	assert( "The handle array returned was the wrong length.", canvas.getHandles().length == 0 );
	canvas.addHandle( handle );
	assert( "The handle array returned was the wrong length.", canvas.getHandles().length == 1 );
}
/**
 * Test getLocator.
 */
public void testGetLocator() {
	assert( "The locator returned was incorrect (dynamic).", canvas.getLocator( 60, 60 ).x() == 60 &&  canvas.getLocator( 60, 60 ).y() == 60 );
	assert( "The locator returned was incorrect (static); x was " + String.valueOf( staticCanvas.getLocator( 60, 60 ).x() ) + ".", staticCanvas.getLocator( 60, 60 ).x() == 49 &&  staticCanvas.getLocator( 60, 60 ).y() == 49 );
}

public void testGetSelections() {
	assert( "There were already selections.", canvas.getSelections().length == 0 );
	canvas.addSelection( figure1 );
	assert( "The selection array returned was the wrong length.", canvas.getSelections().length == 1 );
}
/**
 * Test getSize.
 */
public void testGetSize() {
	assertEquals( "The size was incorrect", new Dimension( 200, 200 ), canvas.getSize() );
}

public void testGetStyle() {
	assert( "The style was null.", canvas.getStyle() != null );
}

public void testGetTool() {
	assert( "The tool was null.", canvas.getTool() != null );
	assert( "The tool wasn't a selection tool", canvas.getTool() instanceof SelectionTool );
}

public void testHandleAt() {
	canvas.addHandle( handle );
	assertEquals( "The correct handle was not returned.", handle, canvas.handleAt( 6, 6 ) );
}

public void testMoveFigureBehind() {
	canvas.addFigure( figure2 );
	assertEquals( "Figure 2 was not in the front", figure2, canvas.figureAt( 5, 5 ) );
	canvas.moveFigureBehind( figure2, figure1 );
	assertEquals( "Figure 1 was not in the front", figure1, canvas.figureAt( 5, 5 ) );
}

public void testMoveFigureInFront() {
	canvas.addFigure( figure2 );
	assertEquals( "Figure 2 was not in the front", figure2, canvas.figureAt( 5, 5 ) );
	canvas.moveFigureInFront( figure1, figure2 );
	assertEquals( "Figure 1 was not in the front", figure1, canvas.figureAt( 5, 5 ) );
}

public void testMoveFigureToBack() {
	canvas.addFigure( figure2 );
	canvas.addFigure( figure3 );
	assertEquals( "Figure 3 was not in the front", figure3, canvas.figureAt( 5, 5 ) );
	canvas.moveFigureToBack( figure3 );
	assertEquals( "Figure 3 was not in the back", figure3, canvas.figures().nextElement() );
}

public void testMoveFigureToFront() {
	canvas.addFigure( figure2 );
	canvas.addFigure( figure3 );
	assertEquals( "Figure 3 was not in the front", figure3, canvas.figureAt( 5, 5 ) );
	canvas.moveFigureToFront( figure1 );
	assertEquals( "Figure 1 was not in the front", figure1, canvas.figureAt( 5, 5 ) );
}

public void testMoveSelectionsToBack() {
	canvas.addFigure( figure2 );
	canvas.addFigure( figure3 );
	assertEquals( "Figure 3 was not in the front", figure3, canvas.figureAt( 5, 5 ) );
	canvas.addSelection( figure3 );
	simpleCanvas.moveSelectionsToBack();
	assertEquals( "Figure 3 was not in the back", figure3, canvas.figures().nextElement() );
}

public void testMoveSelectionsToFront() {
	canvas.addFigure( figure2 );
	canvas.addFigure( figure3 );
	assertEquals( "Figure 3 was not in the front", figure3, canvas.figureAt( 5, 5 ) );
	canvas.addSelection( figure1 );
	simpleCanvas.moveSelectionsToFront();
	assertEquals( "Figure 1 was not in the front", figure1, canvas.figureAt( 5, 5 ) );
}

public void testOtherFigureAt() {
	canvas.addFigure( figure2 );
	assertEquals( "Figure 2 was not in the front", figure2, canvas.figureAt( 5, 5 ) );
	assertEquals( "Figure 1 was not returned", figure1, canvas.otherFigureAt( figure2, 5, 5 ) );
}

public void testPaste() {
	canvas.addSelection( figure1 );
	canvas.addFigure( figure2 );
	canvas.addSelection( figure2 );
	assertEquals( "The figures were not added.", 2, canvas.getSelections().length );
	((SimpleDrawingCanvas)canvas).copySelections();
	((SimpleDrawingCanvas)canvas).paste();
	FigureEnumeration enum = canvas.figures();
	for ( int i = 0; i < 4; i++, enum.nextElement() ) {
		assert( "There were not enough elements on the canvas", enum.hasMoreElements() );
	}
	assert( "There were too many elements on the canvas", ! enum.hasMoreElements() );
}

public void testPasteFromSystem() {
	canvas.removeFigure( figure1 );
	canvas.addSelection( new TextLabel( "This is test text" ) );
	((SimpleDrawingCanvas)canvas).copySelections();
	canvas.clearSelections();
	((SimpleDrawingCanvas)canvas).pasteFromSystem();
	assert( "The canvas was empty", canvas.figures().hasMoreElements() );
	assert( "A text label was not added to the canvas", canvas.figures().nextElement() instanceof TextLabel );
	assertEquals( "The TextLabel has the wrong string", "This is test text", ((StringHolder)canvas.figures().nextElement()).getString() );
}

public void testRemoveFigure() {
	canvas.removeFigure( figure1 );
	assert( "Figure 1 was not removed", ! canvas.figures().hasMoreElements() );
}

public void testRemoveHandle() {
	canvas.addHandle( handle );
	assert( "There was not a handle added.", canvas.getHandles().length == 1 );
	canvas.removeHandle( handle );
	assert( "The handle was not removed from the canvas.", canvas.getHandles().length == 0 );
}

public void testRemoveHandlesFigure() {
	canvas.addHandles( figure1 );
	assertEquals( "There were not handles added.", 8, canvas.getHandles().length );
	canvas.removeHandles( figure1 );
	assertEquals( "The handles were not removed from the canvas.", 0, canvas.getHandles().length );
}

public void testRemoveSelection() {
	canvas.addSelection( figure1 );
	assertEquals( "The figure was not added correctly.", figure1, canvas.getSelections()[0] );
	canvas.removeSelection( figure1 );
	assert( "The figure was not removed properly.", canvas.getSelections().length == 0 );
}

public void testSelect() {
	canvas.addSelection( figure1 );
	canvas.addSelection( figure2 );
	assert( "The figures were not added correctly.", canvas.getSelections().length == 2 );
	canvas.select( figure3 );
	assert( "The figure was not selected correctly.", canvas.getSelections().length == 1 );
	assertEquals( "The figure was not selected correctly.", figure3, canvas.getSelections()[0] );
}

public void testSetComponent() {
	Component comp = new Canvas();
	simpleCanvas.setComponent( comp );
	assertEquals( "The component was not set correctly.", comp, simpleCanvas.getComponent() );
}
/**
 * Test setDrawing.
 */
public void testSetDrawing() {
	canvas.select( figure1 );
	SimpleDrawing drawing2 = new SimpleDrawing();
	((SimpleDrawingCanvas)canvas).setDrawing( drawing2 );
	assertEquals( "The figure is still selected", 0, canvas.getSelections().length );
	drawing2.setDynamicSize( false );
	drawing2.setSize( 10, 10 );
	assertEquals( "The canvas is not receiving update events from the new drawing", new Dimension( 10, 10 ), canvas.getSize() );
}
/**
 * Test setSize( <code>Dimension</code> ).
 */
public void testSetSizeDimension() {
	canvas.setSize( new Dimension( 10, 10 ) );
	assertEquals( "The size was not set properly.", new Dimension( 10, 10 ), canvas.getSize() );
}
/**
 * Test setSize( int, int ).
 */
public void testSetSizeIntInt() {
	canvas.setSize( 10, 10 );
	assertEquals( "The size was not set properly.", new Dimension( 10, 10 ), canvas.getSize() );
}

public void testSetStyle() {
	DrawingStyle style = new SimpleDrawingStyle();
	canvas.setStyle( style );
	assertEquals( "The style was not set properly.", style, canvas.getStyle() );
}

public void testSetTool() {
	InputEventHandler tool = new RectangleTool( canvas );
	canvas.setTool( tool );
	assertEquals( "The tool wasn't set properly", tool, canvas.getTool() );
}

public void testToggleSelection() {
	canvas.toggleSelection( figure1 );
	assertEquals( "The figure was not selected correctly.", figure1, canvas.getSelections()[0] );
	canvas.toggleSelection( figure1 );
	assert( "The figure was not deselected correctly.", canvas.getSelections().length == 0 );
}

public void testToolTaskCompleted() {
	// Method is a no-op
}
}
package com.rolemodelsoft.drawlet.basics;

/**
 * @(#)TS_InputEventHandlerTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import junit.framework.*;
import com.rolemodelsoft.test.*;
/**
 * Runs all tests in com.rolemodelsoft.drawlet.basics
 */
public class TS_InputEventHandlerTests extends junit.framework.TestCase {

public TS_InputEventHandlerTests(String name) {
	super(name);
}

public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}

public static TestSuite suite() {
	TestSuite suite = new TestSuite();
	
	suite.addTest(new TestSuite(TC_AbstractFigure.class));
	suite.addTest(new TestSuite(TC_CanvasTool.class));
	suite.addTest(new TestSuite(TC_DrawingPoint.class));
	suite.addTest(new TestSuite(TC_EdgeLocator.class));
	suite.addTest(new TestSuite(TC_FigureRelativePoint.class));
	suite.addTest(new TestSuite(TC_PolarCoordinate.class));
	suite.addTest(new TestSuite(TC_SimpleDrawing.class));
	suite.addTest(new TestSuite(TC_SimpleDrawingCanvas.class));
	
	return suite;
}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)Component.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.Component;

/**
 * @version 	1.1.6, 12/28/98
 */

public interface ComponentHolder {
	/**
	 * Answers the held component.
	 *
	 * @return the Component associated with this holder
	 */
	Component getComponent();
	/**
	 * Associates the given component with the receiver.
	 *
	 * @param component the component to associate with this holder
	 */
	void setComponent(Component component);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)Drawing.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;
import com.rolemodelsoft.drawlet.util.Duplicatable;
import java.io.Serializable;
import java.beans.PropertyChangeListener;

/**
 * This interface defines a generic Drawing which holds a SequenceOfFigures
 * which can be painted and potentially manipulated.  It is expected that this
 * will be the fundamental unit to store and retrieve diagrams, pictures, etc.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface Drawing extends SequenceOfFigures, Paintable {
	/** 
	 * The size property selector
	 */
	public static String SIZE_PROPERTY = "size";

	/** 
	 * The style property selector
	 */
	public static String STYLE_PROPERTY = "style";
	/** 
	 * Add a PropertyChangeListener.
	 * 
	 * @param listener the listener to add.
	 */
	public abstract void addPropertyChangeListener( PropertyChangeListener listener );
	/** 
	 * Returns the size of this drawing.
	 * 
	 * @return a Dimension representing the size of this drawing.
	 */
	public abstract Dimension getSize();
	/** 
	 * Answer the style which defines how to paint on the canvas.
	 *
	 * @return the DrawingStyle which defines how to paint on the canvas
	 */
	public abstract DrawingStyle getStyle();
	/** 
	 * Returns whether this drawing is sized dynamically or statically.
	 */
	public abstract boolean isDynamicSize();
	/** 
	 * Paints all of the parts of the drawing.
	 * 
	 * @param g the specified Graphics window
	 */
	public abstract void paint(Graphics g);
	/** 
	 * Remove the specified PropertyChangeListener.
	 * 
	 * @param listener the listener to remove.
	 */
	public abstract void removePropertyChangeListener( PropertyChangeListener listener );
	/** 
	 * Sets whether this drawing is sized dynamically or statically.
	 *
	 * @param dynamicSize boolean specifying the dynamicSize state the drawing is to have.
	 */
	public abstract void setDynamicSize( boolean dynamicSize );
	/** 
	 * Sets the size of this drawing.
	 * 
	 * @param width the width the drawing should be set to.
	 * @param height the height the drawing should be set to.
	 */
	public abstract void setSize( int width, int height );
	/** 
	 * Resizes the receiver to the specified dimension.
	 *
	 * @param d the new dimension
	 */
	public abstract void setSize(Dimension d);
	/** 
	 * Set the style defining how to paint the receiver.
	 *
	 * @param style the specified DrawingStyle
	 */
	public abstract void setStyle(DrawingStyle style);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)DrawingCanvas.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * A generic DrawingCanvas on which Figures can be drawn and manipulated.
 * It is expected that this will typically implemented by some sort of graphical
 * Component but that is not necessary.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface DrawingCanvas extends SequenceOfFigures, Paintable {

	/**
	 * Add the handle to the receiver.
	 * Reflect the change if it's visible.
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection.
	 *
	 * @param handle the handle to add
	 */
	public abstract void addHandle(Handle handle);
	/**
	 * Add the handles corresponding to the figure to the receiver.
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection, but some tools may wish to be more selective.
	 *
	 * @param figure the figure for which handles should be added.
	 */
	public abstract void addHandles(Figure figure);
	/**
	 * Add the handles corresponding to the figure to the receiver. 
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection, but some tools may wish to be more selective.
	 *
	 * @param figure the figure for which handles are associated.
	 * @param handles the handles to add.
	 */
	public abstract void addHandles(Figure figure, Handle handles[]);
	/**
	 * Add the figure to the selections.
	 * Reflect the change if it's visible.
	 *
	 * @param figure the Figure to add
	 */
	public abstract void addSelection(Figure figure);
	/**
	 * Clear all the selections of the receiver.
	 * Reflect the change if it's visible.
	 */
	public abstract void clearSelections();
	/** 
	 * Answer the handles of the receiver.  The returned array 
	 * and its contents should be treated as read-only.
	 *
	 * @return the handles of the receiver in an array; should be treated as read-only
	 */
	public Handle[] getHandles();
	/**
	 * Answer the locator that should be used for the given coordinates
	 *
	 * @param x the horizontal coordinate
	 * @param y the vertical coordinate
	 * @return a Locator representing the proper posititioning for the specified x, y
	 */
	public abstract Locator getLocator( int x, int y );
	/** 
	 * Answer the selections of the receiver.  The returned array 
	 * and its contents should be treated as read-only.
	 *
	 * @return an array of Figures that represent the selections
	 * of the receiver; should be treated as read-only
	 */
	public abstract Figure[] getSelections();
	/** 
	 * Answer the style which defines how to paint on the canvas.
	 *
	 * @return the DrawingStyle which defines how to paint on the canvas
	 */
	public abstract DrawingStyle getStyle();
	/**
	 * Answer the active tool
	 *
	 * @return the InputEventHandler that is the active tool
	 */
	public abstract InputEventHandler getTool();
	/** 
	 * Answer the handle at a given point
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return the Handle at the given point
	 */
	public abstract Handle handleAt(int x, int y);
	/**
	 * Remove the handle from the receiver.
	 * Reflect the change if it's visible.
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection.  Since it is public, don't assume
	 * the handle asked for is actually present.
	 *
	 * @param handle the handle to remove
	 */
	public abstract void removeHandle(Handle handle);
	/**
	 * Remove the handles corresponding to the figure to the receiver. 
	 * NOTE: Although this is public, it is assumed that
	 * most handles will be added/removed automatically through
	 * the process of selection, but some tools may wish to be more selective.
	 *
	 * @param figure the figure for which handles should be removed.
	 */
	public abstract void removeHandles(Figure figure);
	/**
	 * Remove the figure from the selections.
	 * Reflect the change if it's visible.
	 *
	 * @param figure the figure being deselected
	 */
	public abstract void removeSelection(Figure figure);
	/** 
	 * Repaints part of the canvas.
	 *
	 * @param rectangle is the region to be repainted
	 */
	public abstract void repaint(Rectangle rectangle);
	/**
	 * Make the figure the only selection
	 * Reflect the change if it's visible.
	 *
	 * @param figure the figure being deselected
	 */
	public abstract void select(Figure figure);
	/** 
	 * Set the size of the receiver.
	 *
	 * @param width the horizontal size
	 * @param height the vertical size
	 */
	public abstract void setSize( int width, int height );
	/** 
	 * Set the size of the receiver.
	 *
	 * @param size the new size
	 */
	public abstract void setSize( Dimension size );
	/** 
	 * Set the style defining how to paint on the canvas.
	 *
	 * @param style the specified DrawingStyle
	 */
	public abstract void setStyle(DrawingStyle style);
	/**
	 * Set the active tool
	 *
	 * @param tool the InputEventHandler to set as the active tool
	 */
	public abstract void setTool(InputEventHandler tool);
	/**
	 * Toggle whether or not the figure is selected. 
	 * Reflect the change if it's visible.
	 *
	 * @param figure the figure of interest
	 */
	public abstract void toggleSelection(Figure figure);
	/**
	 * Take appropriate action when the tool has completed its task.
	 *
	 * @param tool the tool which completed its task
	 */
	public abstract void toolTaskCompleted(InputEventHandler tool);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)DrawingStyle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import com.rolemodelsoft.drawlet.util.Duplicatable;
import java.awt.*;

/**
 * Defines a set of getters and setters for drawing styles.
 * DrawingStyles are used by Figure and DrawingCanvas to define various attributes.
 * @version 	1.1.6, 12/28/98
 */
 
public interface DrawingStyle extends Duplicatable {
	/**
	 * Answer the Color to use for the background.
	 * @return Color the Color to use for the background
	 */
	public abstract Color getBackgroundColor();
	/**
	 * Answer the Color to use when filling the figure.
	 *
	 * @return the Color to use when filling the figure
	 */
	public abstract Color getFillColor();
	/** 
	 * Answer the Font with which to paint text.
	 *
	 * @return the Font with which to paint text
	 */
	public abstract Font getFont();
	/**
	 * Answer the Color to use for the foreground.
	 *
	 * @return the Color to use for the foreground
	 */
	public abstract Color getForegroundColor();
	/**
	 * Answer the Color to use to highlight areas of the canvas.
	 *
	 * @return the Color to use to highlight areas of the canvas
	 */
	public abstract Color getHighlightColor();
	/**
	 * Answer the Color to use when drawing lines.
	 *
	 * @return the Color to use when drawing lines
	 */
	public abstract Color getLineColor();
	/**
	 * Answer the Color to use for the background when indicating selection.
	 *
	 * @return the Color to use for the background when indicating selection
	 */
	public abstract Color getSelectionBackgroundColor();
	/**
	 * Answer the Color to use for the foreground when indicating selection.
	 *
	 * @return the Color to use for the foreground when indicating selection
	 */
	public abstract Color getSelectionForegroundColor();
	/**
	 * Answer the Color to use when drawing text.
	 *
	 * @return the Color to use when drawing text
	 */
	public abstract Color getTextColor();
	/**
	 * Set the Color to use for the background.
	 *
	 * @param color the color
	 */
	public abstract void setBackgroundColor(Color color);
	/**
	 * Set the Color to use when filling the figure.
	 *
	 * @param color the color
	 */
	public abstract void setFillColor(Color color);
	/** 
	 * Set the Font with which to paint text
	 *
	 * @param newFont the Font to use for text
	 */
	public abstract void setFont(Font newFont);
	/**
	 * Set the Color to use for the foreground.
	 *
	 * @param color the color
	 */
	public abstract void setForegroundColor(Color color);
	/**
	 * Set the Color to use to highlight areas of the canvas.
	 *
	 * @param color the color
	 */
	public abstract void setHighlightColor(Color color);
	/**
	 * Set the Color to use when drawing lines.
	 *
	 * @param color the color
	 */
	public abstract void setLineColor(Color color);
	/**
	 * Set the Color to use for the background during selection.
	 *
	 * @param color the color
	 */
	public abstract void setSelectionBackgroundColor(Color color);
	/**
	 * Set the Color to use for the foreground during selection.
	 *
	 * @param color the color
	 */
	public abstract void setSelectionForegroundColor(Color color);
	/**
	 * Set the Color to use when drawing text.
	 *
	 * @param color the color
	 */
	public abstract void setTextColor(Color color);
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)ExitingFrame.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.awt.event.*;
public class ExitingFrame extends Frame implements WindowListener {
/**
 * ExitingFrame constructor comment.
 */
public ExitingFrame() {
	super();
	addWindowListener(this);
}
/**
 * ExitingFrame constructor comment.
 * @param title java.lang.String
 */
public ExitingFrame(String title) {
	super(title);
	addWindowListener(this);
}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry.
	 */
	public static void main(java.lang.String[] args) {
		try {
			java.awt.Frame frame = new ExitingFrame();
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of ExitingFrame");
			exception.printStackTrace(System.out);
		}
	}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowActivated(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowClosed(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	System.exit(0);
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowClosing(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	this.dispose();
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowDeactivated(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowDeiconified(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowIconified(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowOpened(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)FullApplet.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996-1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.lines.DrawLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.*;
import java.net.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic DrawingCanvas as an Applet.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class FullApplet extends SimpleApplet {

	/**
	 * Applet info.
	 *
	 * @return a String containing information on this applet
	 */
	public String getAppletInfo() {
		return "Drawlets Example v1.1.6 (30 Mar 1999), by RoleModel Software.";
	}
	/**
	 * @return the ToolPalette.
	 */
	public ToolPalette getToolPalette() {
		if ( toolPalette == null ) {
			Image triangleImage = getImage("greygrey/triangle.gif");
			Image pentagonImage = getImage("greygrey/pentagon.gif");
			Image freeHandImage = getImage("greygrey/freehand.gif");
			
			// Create the tool palette
			toolPalette = super.getToolPalette();
			toolPalette.addTool(new PolygonTool(canvas,3), "Triangle", triangleImage);
			toolPalette.addTool(new PolygonTool(canvas,5), "Pentagon", pentagonImage);
			toolPalette.addTool(new DrawLineTool(canvas), "Smooth Line", freeHandImage);
		}
		return toolPalette;
	}
	/**
	 * Initializes the applet.
	 * You never need to call this directly, it is called automatically
	 * by the system once the applet is created.
	 * @see #start
	 * @see #stop
	 * @see #destroy
	 */
	public void init() {
		super.init();

		GridBagConstraints stylePaletteConstraints = new GridBagConstraints();
		stylePaletteConstraints.gridx = 3;
		stylePaletteConstraints.gridy = 2;
		stylePaletteConstraints.anchor = GridBagConstraints.NORTH;
		add(getStylePalette(), stylePaletteConstraints);

	}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)FullPanel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import com.rolemodelsoft.drawlet.awt.*;
import java.awt.*;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.lines.DrawLineTool;

public class FullPanel extends SimplePanel {
/**
 * FullPanelExample constructor comment.
 */
public FullPanel() {
	super();
}
/**
 * FullPanelExample constructor comment.
 * @param layout java.awt.LayoutManager
 */
public FullPanel(java.awt.LayoutManager layout) {
	super(layout);
}
	/**
	 * @return the ToolPalette.
	 */
	public ToolPalette getToolPalette() {
		if ( toolPalette == null ) {
			Image triangleImage = getImage("triangle.gif");
			Image pentagonImage = getImage("pentagon.gif");
			Image freeHandImage = getImage("freehand.gif");
			
			// Create the tool palette
			toolPalette = super.getToolPalette();
			toolPalette.addTool(new PolygonTool(canvas,3), "Triangle", triangleImage);
			toolPalette.addTool(new PolygonTool(canvas,5), "Pentagon", pentagonImage);
			toolPalette.addTool(new DrawLineTool(canvas), "Smooth Line", freeHandImage);
		}
		return toolPalette;
	}
	/**
	 * Initialize class
	 */
	protected void initialize() {
		super.initialize();

		GridBagConstraints stylePaletteConstraints = new GridBagConstraints();
		stylePaletteConstraints.gridx = 3;
		stylePaletteConstraints.gridy = 2;
		stylePaletteConstraints.anchor = GridBagConstraints.NORTH;
		add(getStylePalette(), stylePaletteConstraints);
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry.
	 */
	public static void main(java.lang.String[] args) {
		try {
			java.awt.Frame frame = new ExitingFrame( "Drawlet Panel Example" );
			FullPanel aTestCanvas;
			aTestCanvas = new FullPanel();
			frame.add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
	}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)FullScrollingApplet.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import java.awt.*;
public class FullScrollingApplet extends FullApplet {
	/**
	 * @return the canvas.
	 */
	protected Component getCanvasComponent() {
		canvas = new SimpleDrawingCanvas(new SimpleDrawing(500,500));
		DrawingCanvasComponent canvasComponent = new BufferedDrawingCanvasComponent( canvas );
		ScrollPane component = new ScrollPane();
		component.add( canvasComponent );
		return component;
	}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)FullScrollingPanel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

 import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;

public class FullScrollingPanel extends FullPanel {
/**
 * FullScrollingPanel constructor comment.
 */
public FullScrollingPanel() {
	super();
}
/**
 * FullScrollingPanel constructor comment.
 * @param layout java.awt.LayoutManager
 */
public FullScrollingPanel(java.awt.LayoutManager layout) {
	super(layout);
}
	/**
	 * @return the canvas.
	 */
	protected Component getCanvasComponent() {
		canvas = new SimpleDrawingCanvas(new SimpleDrawing(500,500));
		DrawingCanvasComponent canvasComponent = new BufferedDrawingCanvasComponent( canvas );
		ScrollPane component = new ScrollPane();
		component.add( canvasComponent );
		return component;
	}
/**
 * Starts the application.
 * @param args an array of command-line arguments
 */
public static void main(java.lang.String[] args) {
		try {
			java.awt.Frame frame = new ExitingFrame( "Drawlet Panel Example" );
			FullScrollingPanel aTestCanvas;
			aTestCanvas = new FullScrollingPanel();
			frame.add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)SimpleApplet.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996-1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.*;
import java.net.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic DrawingCanvas as an Applet.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimpleApplet extends java.applet.Applet {
	/**
	 * The canvas to use.
	 */
	protected DrawingCanvas canvas;

	/**
	 * The tool palette for this applet.
	 */
	protected ToolPalette toolPalette;

	/**
	 * The style palette for this applet.
	 */
	protected StylePalette stylePalette;

	/**
	 * The tool bar for actions.
	 */
	protected ToolBar toolBar;

	/**
	 * Applet info.
	 *
	 * @return a String containing information on this applet
	 */
	public String getAppletInfo() {
		return "Simple Drawlets Example v1.1.6 (30 Mar 1999), by RoleModel Software.";
	}
	/**
	 * @return Component the component holding the canvas.
	 */
	protected Component getCanvasComponent() {
		canvas = new SimpleDrawingCanvas();
		DrawingCanvasComponent component = new BufferedDrawingCanvasComponent( canvas );
		return component;
	}
	/**
	 * Get an image.
	 *
	 * @param name the name of the Image to get.
	 * @return the ToolPalette.
	 */
	public Image getImage( String name ) {
		Image image = null;
		try {
			InputStream in = getClass().getResourceAsStream( "/" + name );
			if ( in == null ) {
				System.err.println( "Image not found" );
				return image;
			}

			byte[] buffer = new byte[ in.available() ];
			in.read( buffer );
			image = Toolkit.getDefaultToolkit().createImage( buffer );
		} catch ( IOException e ) {
			System.err.println( "Unable to read image" );
			e.printStackTrace();
		}
		return image;
	}
	
	/**
	 * @return the StylePalette
	 */
	public StylePalette getStylePalette() {
		if ( stylePalette == null ) {
			// Create the color palette
			stylePalette = new StylePalette(canvas);
			stylePalette.setLayout(new GridBagLayout());
			stylePalette.setBackground( Color.lightGray );
			stylePalette.addColor(Color.blue, "blue");
			stylePalette.addColor(Color.green, "green");
			stylePalette.addColor(Color.cyan, "cyan");
			stylePalette.addColor(Color.magenta, "magenta");
			stylePalette.addColor(Color.red, "red");
			stylePalette.addColor(Color.pink, "pink");
			stylePalette.addColor(Color.orange, "orange");
			stylePalette.addColor(Color.yellow, "yellow");
			stylePalette.addColor(Color.black, "black");
			stylePalette.addColor(Color.darkGray, "dark gray");
			stylePalette.addColor(Color.gray, "gray");
			stylePalette.addColor(Color.lightGray, "light gray");
			stylePalette.addColor(Color.white, "white");
			stylePalette.addColor(null, "NO");
			stylePalette.addSetter("setFillColor", "Fill");
			stylePalette.addSetter("setLineColor", "Line");
			stylePalette.addSetter("setTextColor", "Text");
			stylePalette.addSetter("setBackgroundColor", "Back");
			stylePalette.addSetter("setForegroundColor", "Fore");
			stylePalette.addApply("Apply");
			stylePalette.addStyleViewer();
		}
		return stylePalette;
	}
	/**
	 * @return the ToolPalette.
	 */
	public ToolBar getToolBar() {
		if ( toolBar == null ) {
			Image cutImage = getImage("greygrey/cut.gif");
			Image copyImage = getImage("greygrey/copy.gif");
			Image pasteImage = getImage("greygrey/paste.gif");
			Image systemPasteImage = getImage("greygrey/systempaste.gif");
			
			// Create the tool palette
			toolBar = new ToolBar( canvas );
			toolBar.setLayout(new GridBagLayout());
			toolBar.setBackground( Color.lightGray );
			toolBar.addButton( "cutSelections", cutImage );
			toolBar.addButton( "copySelections", copyImage );
			toolBar.addButton( "paste", pasteImage );
			toolBar.addButton( "pasteFromSystem", systemPasteImage );
		}
		return toolBar;
	}
	/**
	 * @return the ToolPalette.
	 */
	public ToolPalette getToolPalette() {
		if ( toolPalette == null ) {
			Image selectionImage = getImage("greygrey/select.gif");
			Image labelImage = getImage("greygrey/label.gif");
			Image lineImage = getImage("greygrey/line.gif");
			Image rectangleImage = getImage("greygrey/box.gif");
			Image roundedRectangleImage = getImage("greygrey/rrect.gif");
			Image ellipseImage = getImage("greygrey/ellipse.gif");
			Image nGonImage = getImage("greygrey/nsided.gif");
			
			// Create the tool palette
			toolPalette = new ToolPalette(canvas);
			toolPalette.setLayout(new GridBagLayout());
			toolPalette.setBackground( Color.lightGray );
			toolPalette.addTool(new SelectionTool(canvas), "Select", selectionImage);
			toolPalette.addTool(new LabelTool(canvas), "Label", labelImage);
			toolPalette.addTool(new ConnectingLineTool(canvas), "Line", lineImage);
			toolPalette.addTool(new RectangleTool(canvas), "Box", rectangleImage);
			toolPalette.addTool(new RectangularCreationTool(canvas,RoundedRectangleShape.class), "Rounded", roundedRectangleImage);
			toolPalette.addTool(new EllipseTool(canvas), "Ellipse", ellipseImage);
			toolPalette.addTool(new AnySidedPolygonTool(canvas), "N-gon", nGonImage);

			// This is something of a hack to make sure that the selection tool
			// (which is the default tool for SimpleDrawingCanvas is selected
			toolPalette.actionPerformed( new ActionEvent( toolPalette.getComponent(0), ActionEvent.ACTION_PERFORMED, "SelectionTool" ) );
		}
		return toolPalette;
	}
	/**
	 * Initializes the applet.
	 * You never need to call this directly, it is called automatically
	 * by the system once the applet is created.
	 * @see #start
	 * @see #stop
	 * @see #destroy
	 */
	public void init() {
		setBackground( Color.lightGray );
		
		setLayout(new GridBagLayout());

		// Create the canvas
		Component canvasComponent = getCanvasComponent();
		GridBagConstraints canvasConstraints = new GridBagConstraints();
		canvasConstraints.gridx = 2;
		canvasConstraints.gridy = 2;
		canvasConstraints.weightx = 1.0;
		canvasConstraints.weighty = 1.0;
		canvasConstraints.fill = GridBagConstraints.BOTH;
		add(canvasComponent, canvasConstraints);

		// Set the drawing style
		DrawingStyle style = new SimpleDrawingStyle();
		style.setBackgroundColor(Color.blue);
		style.setForegroundColor(Color.white);
		canvas.setStyle(style);

		GridBagConstraints toolPaletteConstraints = new GridBagConstraints();
		toolPaletteConstraints.gridx = 1;
		toolPaletteConstraints.gridy = 2;
		toolPaletteConstraints.anchor = GridBagConstraints.NORTH;
		add(getToolPalette(), toolPaletteConstraints);
		
		GridBagConstraints toolBarConstraints = new GridBagConstraints();
		toolBarConstraints.gridx = 1;
		toolBarConstraints.gridy = 1;
		toolBarConstraints.gridwidth = 3;
		toolBarConstraints.anchor = GridBagConstraints.WEST;
		toolBarConstraints.fill = GridBagConstraints.HORIZONTAL;
		Panel panel = new Panel();
		panel.setLayout( new BorderLayout() );
		panel.setBackground( Color.lightGray );
		panel.add( getToolBar(), BorderLayout.WEST );
		add(panel, toolBarConstraints);
	}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)SimpleModelExample.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.examples.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.LabelTool;
import com.rolemodelsoft.drawlet.util.ValueAdapter;
import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic drawing tool as a Panel with a simple underlying model.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimpleModelPanel extends SimplePanel implements ActionListener {

	/**
	 * The model to use for this applet
	 */
	protected SingleDrawingModel model;
	
	/**
	 * The temporary file name to save to and restore from
	 */
	protected static String TempFileName = "temp.rmd";

	/**
	 * The command panel
	 */
	protected Panel commandPanel;
	/**
	 * Default constructor.
	 */
	public SimpleModelPanel() {
		super();
	}
	/**
	 * SimplePanelExample constructor comment.
	 * 
	 * @param layout the layout manager to use with this application
	 */
	public SimpleModelPanel(java.awt.LayoutManager layout) {
		super(layout);
	}
	/**
	 * @param e the event
	 */
	public void actionPerformed (java.awt.event.ActionEvent e)
	{
		// Hack to get commands working
		String commandString = e.getActionCommand();
		if (commandString.equals("Clear")) getModel().clearDrawing();
		else if (commandString.equals("Save")) getModel().saveDrawing(TempFileName);
		else if (commandString.equals("Restore")) getModel().restoreDrawing(TempFileName);
		else if (commandString.equals("Print")) getModel().printDrawing(getFrame( this ));
	}
	/**
	 * @return Component the component holding the canvas.
	 */
	protected Component getCanvasComponent() {
		canvas = new SimpleDrawingCanvas(getModel().getDrawing());
		Component component = new BufferedDrawingCanvasComponent(canvas);
		ValueAdapter adapter = new ValueAdapter(model,"getDrawing",canvas,"setDrawing");
		component.setSize(300,240);
		return component;
	}
	/**
	 * @return the Frame.
	 */
	public Panel getCommandPanel () {
		if ( commandPanel == null ) {
			// add commands, quick and dirty
			commandPanel = new Panel();
			commandPanel.setLayout(new GridLayout(1,3,2,2));
			String[] commands = {"Clear", "Save", "Restore", "Print"};
			for (int i=0; i<commands.length; i++) {
				Button button = new Button(commands[i]);
				button.setActionCommand(commands[i]);
				button.addActionListener(this);
				commandPanel.add(button);
			}
		}
		return commandPanel;
	}
	/**
	 * @return the Frame.
	 */
	public Frame getFrame ( Component c ) {
		while ( ( c = c.getParent() ) != null ) {
			if ( c instanceof Frame ) return (Frame) c;
		}
		return null;
	}
	/**
	 * @return the model
	 */
	protected SingleDrawingModel getModel() {
		if ( model == null ) {
			model = new SingleDrawingModel();
		}
		return model;
	}
	/**
	 */
	public StylePalette getStylePalette() {
		if ( stylePalette == null ) {
			// Create the color palette
			stylePalette = new StylePalette(canvas);
			stylePalette.setLayout(new GridBagLayout());
			stylePalette.addColor(Color.blue, "blue");
			stylePalette.addColor(Color.green, "green");
			stylePalette.addColor(Color.cyan, "cyan");
			stylePalette.addColor(Color.magenta, "magenta");
			stylePalette.addColor(Color.red, "red");
			stylePalette.addColor(Color.pink, "pink");
			stylePalette.addColor(Color.orange, "orange");
			stylePalette.addColor(Color.yellow, "yellow");
			stylePalette.addColor(Color.black, "black");
			stylePalette.addColor(Color.darkGray, "dark gray");
			stylePalette.addColor(Color.gray, "gray");
			stylePalette.addColor(Color.lightGray, "light gray");
			stylePalette.addColor(Color.white, "white");
			stylePalette.addColor(null, "NO");
			stylePalette.addSetter("setFillColor", "Fill");
			stylePalette.addSetter("setLineColor", "Line");
			stylePalette.addSetter("setTextColor", "Text");
			stylePalette.addSetter("setBackgroundColor", "Back");
			stylePalette.addSetter("setForegroundColor", "Fore");
			stylePalette.addApply("Apply");
			stylePalette.addStyleViewer();
		}
		return stylePalette;
	}
	/**
	 * Initialize class
	 */
	protected void initialize() {
		super.initialize();

		GridBagConstraints commandPanelConstraints = new GridBagConstraints();
		commandPanelConstraints.gridx = 2;
		commandPanelConstraints.gridy = 3;
		commandPanelConstraints.anchor = GridBagConstraints.SOUTH;
		add(getCommandPanel(), commandPanelConstraints);
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry
	 */
	public static void main(java.lang.String[] args) {
		try {
			java.awt.Frame frame = new ExitingFrame( "Simple Model Example" );
			SimpleModelPanel aTestCanvas;
			aTestCanvas = new SimpleModelPanel();
			frame.add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
	}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)SimplePanel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.LabelTool;
import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic DrawingTool as an Applet.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimplePanel extends Panel {
	
	/**
	 * The canvas this panel will use
	 */
	protected DrawingCanvas canvas;

	/**
	 * The tool palette for this panel.
	 */
	protected ToolPalette toolPalette;

	/**
	 * The style palette for this panel.
	 */
	protected StylePalette stylePalette;
	
	/**
	 * The tool bar for actions.
	 */
	protected ToolBar toolBar;
	/**
	 * Default constructor.
	 */
	public SimplePanel() {
		super();
		initialize();
	}
	/**
	 * @param layout the layout manager to use with this application.
	 */
	public SimplePanel(java.awt.LayoutManager layout) {
		super(layout);
	}
	/**
	 * @return Component the component holding the canvas.
	 */
	protected Component getCanvasComponent() {
		canvas = new SimpleDrawingCanvas();
		DrawingCanvasComponent component = new BufferedDrawingCanvasComponent( canvas );
		return component;
	}
	/**
	 * Get an image.
	 *
	 * @param name the name of the Image to get.
	 * @return the ToolPalette.
	 */
	public Image getImage( String name ) {
		Image image = null;
		try {
			InputStream in = getClass().getResourceAsStream( "/" + name );
			if ( in == null ) {
				System.err.println( "Image not found" );
				return image;
			}

			byte[] buffer = new byte[ in.available() ];
			in.read( buffer );
			image = Toolkit.getDefaultToolkit().createImage( buffer );
		} catch ( IOException e ) {
			System.err.println( "Unable to read image" );
			e.printStackTrace();
		}
		return image;
	}
	
	/**
	 */
	public StylePalette getStylePalette() {
		if ( stylePalette == null ) {
			// Create the color palette
			stylePalette = new StylePalette(canvas);
			stylePalette.setLayout(new GridBagLayout());
			stylePalette.setBackground( Color.white );
			stylePalette.addColor(Color.blue, "blue");
			stylePalette.addColor(Color.green, "green");
			stylePalette.addColor(Color.cyan, "cyan");
			stylePalette.addColor(Color.magenta, "magenta");
			stylePalette.addColor(Color.red, "red");
			stylePalette.addColor(Color.pink, "pink");
			stylePalette.addColor(Color.orange, "orange");
			stylePalette.addColor(Color.yellow, "yellow");
			stylePalette.addColor(Color.black, "black");
			stylePalette.addColor(Color.darkGray, "dark gray");
			stylePalette.addColor(Color.gray, "gray");
			stylePalette.addColor(Color.lightGray, "light gray");
			stylePalette.addColor(Color.white, "white");
			stylePalette.addColor(null, "NO");
			stylePalette.addSetter("setFillColor", "Fill");
			stylePalette.addSetter("setLineColor", "Line");
			stylePalette.addSetter("setTextColor", "Text");
			stylePalette.addSetter("setBackgroundColor", "Back");
			stylePalette.addSetter("setForegroundColor", "Fore");
			stylePalette.addApply("Apply");
			stylePalette.addStyleViewer();
		}
		return stylePalette;
	}
	/**
	 * @return the ToolPalette.
	 */
	public ToolBar getToolBar() {
		if ( toolBar == null ) {
			Image cutImage = getImage("cut.gif");
			Image copyImage = getImage("copy.gif");
			Image pasteImage = getImage("paste.gif");
			Image systemPasteImage = getImage("systempaste.gif");
			
			// Create the tool palette
			toolBar = new ToolBar( canvas );
			toolBar.setLayout(new GridBagLayout());
			toolBar.setBackground( Color.white );
			toolBar.addButton( "cutSelections", cutImage );
			toolBar.addButton( "copySelections", copyImage );
			toolBar.addButton( "paste", pasteImage );
			toolBar.addButton( "pasteFromSystem", systemPasteImage );
		}
		return toolBar;
	}
	/**
	 * @return the ToolPalette.
	 */
	public ToolPalette getToolPalette() {
		if ( toolPalette == null ) {
			Image selectionImage = getImage("select.gif");
			Image labelImage = getImage("label.gif");
			Image lineImage = getImage("line.gif");
			Image rectangleImage = getImage("box.gif");
			Image roundedRectangleImage = getImage("rrect.gif");
			Image ellipseImage = getImage("ellipse.gif");
			Image nGonImage = getImage("nsided.gif");
			
			// Create the tool palette
			toolPalette = new ToolPalette(canvas);
			toolPalette.setLayout(new GridBagLayout());
			toolPalette.addTool(new SelectionTool(canvas), "Select", selectionImage);
			toolPalette.addTool(new LabelTool(canvas), "Label", labelImage);
			toolPalette.addTool(new ConnectingLineTool(canvas), "Line", lineImage);
			toolPalette.addTool(new RectangleTool(canvas), "Box", rectangleImage);
			toolPalette.addTool(new RectangularCreationTool(canvas,RoundedRectangleShape.class), "Rounded", roundedRectangleImage);
			toolPalette.addTool(new EllipseTool(canvas), "Ellipse", ellipseImage);
			toolPalette.addTool(new AnySidedPolygonTool(canvas), "N-gon", nGonImage);
		}
		return toolPalette;
	}
	/**
	 * Initialize class
	 */
	protected void initialize() {
		setName("Drawlets");
		setSize(420, 300);
		setLayout(new GridBagLayout());
		setBackground( Color.white );

		// Create the canvas
		Component canvasComponent = getCanvasComponent();
		GridBagConstraints canvasConstraints = new GridBagConstraints();
		canvasConstraints.gridx = 2;
		canvasConstraints.gridy = 2;
		canvasConstraints.weightx = 1.0;
		canvasConstraints.weighty = 1.0;
		canvasConstraints.fill = GridBagConstraints.BOTH;
		add(canvasComponent, canvasConstraints);

		// Set the drawing style
		DrawingStyle style = new SimpleDrawingStyle();
		style.setBackgroundColor(Color.blue);
		style.setForegroundColor(Color.white);
		canvas.setStyle(style);

		GridBagConstraints toolPaletteConstraints = new GridBagConstraints();
		toolPaletteConstraints.gridx = 1;
		toolPaletteConstraints.gridy = 2;
		toolPaletteConstraints.anchor = GridBagConstraints.NORTH;
		add(getToolPalette(), toolPaletteConstraints);

		GridBagConstraints toolBarConstraints = new GridBagConstraints();
		toolBarConstraints.gridx = 1;
		toolBarConstraints.gridy = 1;
		toolBarConstraints.gridwidth = 3;
		toolBarConstraints.anchor = GridBagConstraints.WEST;
		toolBarConstraints.fill = GridBagConstraints.HORIZONTAL;
		Panel panel = new Panel();
		panel.setLayout( new BorderLayout() );
		panel.setBackground( Color.white );
		panel.add( getToolBar(), BorderLayout.WEST );
		add(panel, toolBarConstraints);
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry.
	 */
	public static void main(java.lang.String[] args) {
		try {
			java.awt.Frame frame = new ExitingFrame( "Drawlets Panel Example" );
			SimplePanel aTestCanvas;
			aTestCanvas = new SimplePanel();
			frame.add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
	}
	/**
	 * Set the drawing associated with the receiver to the specified drawing
	 * 
	 * @param newDrawing the drawing
	 */
	public void setDrawing(Drawing newDrawing) {
		((SimpleDrawingCanvas)canvas).setDrawing(newDrawing);
	}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)SimplestApplet.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996-1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.lines.AdornedLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.LabelTool;
import java.awt.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic DrawingTool as an Applet.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimplestApplet extends java.applet.Applet {

	/**
	 * The canvas to use.
	 */
	protected DrawingCanvas canvas;

	/**
	 * The tool palette for this applet.
	 */
	protected ToolPalette toolPalette;
	/**
	 * Applet info.
	 */
	public String getAppletInfo() {
			return "Simple Drawlets Example v1.1.6 (30 Dec 1998), by RoleModel Software.";
	}
	/**
	 * Initializes the applet.
	 * You never need to call this directly, it is called automatically
	 * by the system once the applet is created.
	 * @see #start
	 * @see #stop
	 * @see #destroy
	 */
	public void init() {
		setLayout(new BorderLayout());

		// Create the canvas
		canvas = new SimpleDrawingCanvas();
		Component canvasComponent = new BufferedDrawingCanvasComponent(canvas);
		canvasComponent.setSize(200,200);
		add("Center", canvasComponent);

		// Set the drawing style
		DrawingStyle style = new SimpleDrawingStyle();
		style.setBackgroundColor(Color.blue);
		style.setForegroundColor(Color.white);
		canvas.setStyle(style);

		// Create the tool palette
		toolPalette = new ToolPalette(canvas);
		toolPalette.setLayout(new GridLayout(8,1,2,2));
		toolPalette.addTool(new SelectionTool(canvas), "Select");
		toolPalette.addTool(new LabelTool(canvas), "Label");
		toolPalette.addTool(new ConnectingLineTool(canvas), "Line");
		toolPalette.addTool(new AdornedLineTool(canvas), "Line w/ Arrow");
		toolPalette.addTool(new RectangleTool(canvas), "Box");
		toolPalette.addTool(new RectangularCreationTool(canvas,RoundedRectangleShape.class), "Rounded");
		toolPalette.addTool(new EllipseTool(canvas), "Ellipse");
		toolPalette.addTool(new AnySidedPolygonTool(canvas), "N-Sided");
		add("West",toolPalette);
	}
}
package com.rolemodelsoft.drawlet.examples.awt;

/**
 * @(#)SimplePanel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.lines.AdornedLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.LabelTool;
import java.awt.*;
import java.awt.event.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic DrawingTool as an Applet.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimplestPanel extends Panel {
	
	/**
	 * The canvas this applet will use
	 */
	protected DrawingCanvas canvas;
	/**
	 * Default constructor.
	 */
	public SimplestPanel() {
		super();
		initialize();
	}
	/**
	 * @param layout the layout manager to use with this application.
	 */
	public SimplestPanel(java.awt.LayoutManager layout) {
		super(layout);
	}
	/**
	 * Initialize class
	 */
	protected void initialize() {
		setName("Drawlets");
		setSize(320, 180);
		//setFont(new java.awt.Font("timesroman", 0, 12));
		setLayout(new BorderLayout());
		
		// Create the canvas
		canvas = new SimpleDrawingCanvas();
		Component canvasComponent = new BufferedDrawingCanvasComponent(canvas);
		canvasComponent.setSize(300,240);
		add("Center", canvasComponent);

		// Set the drawing style
		DrawingStyle style = new SimpleDrawingStyle();
		style.setBackgroundColor(Color.blue);
		style.setForegroundColor(Color.white);
		canvas.setStyle(style);

		// Create the tool palette
		ToolPalette toolPalette = new ToolPalette(canvas);
		toolPalette.setLayout(new GridLayout(8,1,2,2));
		toolPalette.addTool(new SelectionTool(canvas), "Select");
		toolPalette.addTool(new LabelTool(canvas), "Label");
		toolPalette.addTool(new ConnectingLineTool(canvas), "Line");
		toolPalette.addTool(new AdornedLineTool(canvas), "Line w/ Arrow");
		toolPalette.addTool(new RectangleTool(canvas), "Box");
		toolPalette.addTool(new RectangularCreationTool(canvas,RoundedRectangleShape.class), "Rounded");
		toolPalette.addTool(new EllipseTool(canvas), "Ellipse");
		toolPalette.addTool(new AnySidedPolygonTool(canvas), "N-Sided");
		add("West",toolPalette);
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry.
	 */
	public static void main(java.lang.String[] args) {
		try {
			java.awt.Frame frame = new ExitingFrame( "Simplest Drawlet Panel Example" );
			SimplestPanel aTestCanvas;
			aTestCanvas = new SimplestPanel();
			frame.add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
	}
	/**
	 * Set the drawing associated with the receiver to the specified drawing
	 * 
	 * @param newDrawing the drawing
	 */
	public void setDrawing(Drawing newDrawing) {
		((SimpleDrawingCanvas)canvas).setDrawing(newDrawing);
	}
}
package com.rolemodelsoft.drawlet.examples.graphnode;

/**
 * @(#)ArrowGraphNode.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.lines.*;

/**
 * Here is an example of a <code>GraphNode</code> with <code>Arrow</code>s.
 */
public class ArrowGraphNode extends GraphNode {
	static final long serialVersionUID = -451845920984253071L;
	/**
	 * Answer a new line creation handle
	 */
	protected ConnectedLineCreationHandle basicNewLineCreationHandle() {
		return new AdornedLineCreationHandle(this);
	}
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the other figure.
	 * By default, make it the middle of the receiver.
	 * Subclasses may wish to do something more meaningful.
	 *
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 */
	public Locator requestConnection(Figure requestor, int x, int y) {
	if (requestor == this)
		return null;
	return new FigureRelativePoint(this,0.5,0.5);
	}
}
package com.rolemodelsoft.drawlet.examples.graphnode;

/**
 * @(#)GraphNodeApplet.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import java.awt.*;

/**
 * Here is a pathetically naive example of a simple Nodes and Arcs
 * Application.  It is meant to illustrate how simple it is to add your own
 * figures, and connect them.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class ArrowGraphNodeApplet extends java.applet.Applet {

	/**
	 * the canvas for this applet.
	 */
	DrawingCanvas canvas;
	/**
	 * Applet info.
	 */
	public String getAppletInfo() {
		return "ArrowGraphNode Drawlet Example v1.1.6 (19 Feb 1999), by RoleModel Software.";
	}
	/**
	 * Initializes the applet.
	 * You never need to call this directly, it is called automatically
	 * by the system once the applet is created.
	 * @see #start
	 * @see #stop
	 * @see #destroy
	 */
	public void init() {
		setLayout(new BorderLayout());
		
		// Create the canvas
		DrawingCanvasComponent canvasComponent = new BufferedDrawingCanvasComponent();
		canvas = canvasComponent.getCanvas();
		canvas.setTool(new ArrowGraphNodeTool(canvas));
		canvasComponent.setSize(getSize());
		add("Center",canvasComponent);
	}
}
package com.rolemodelsoft.drawlet.examples.graphnode;

/**
 * @(#)ArrowGraphNodeTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;

/**
 * Here is a simple tool that extends the <code>SelectionTool</code>
 * to create <code>ArrowGraphNodes</code> when a doubleClick event occurs.
 */
public class ArrowGraphNodeTool extends GraphNodeTool {
/**
 * ArrowGraphNodeTool constructor comment.
 * @param canvas com.rolemodelsoft.drawlet.DrawingCanvas
 */
public ArrowGraphNodeTool(DrawingCanvas canvas) {
	super(canvas);
}
	/**
	 * Answer a new graph node
	 */
	protected GraphNode basicNewGraphNode() {
		return new ArrowGraphNode();
	}
}
package com.rolemodelsoft.drawlet.examples.graphnode;

/**
 * @(#)GraphNode.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.text.*;
import com.rolemodelsoft.drawlet.shapes.lines.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.util.*;
import java.awt.*;

/**
 * Here is a simple example of a graph node figure.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class GraphNode extends RectangleShape implements LabelHolder {
	 static final long serialVersionUID = -5689584309732244534L;
	
	/**
	 * the label associated with this graphnode.
	 */
	protected String string = defaultString();

	/**
	 * the color of the text in the graphnode.
	 */
	protected Color textColor = defaultTextColor();

	/**
	 * the renderer responsible for drawing the label.
	 */
	transient protected StringRenderer renderer;
	/**
	 * Answer a new line creation handle
	 */
	protected ConnectedLineCreationHandle basicNewLineCreationHandle() {
		return new ConnectedLineCreationHandle(this);
	}
	/**
	 * Denote that the shape changed.
	 * 
	 * @param rect the old bounds.
	 */
	protected void changedShape(Rectangle rect) {
		renderer = null;
		super.changedShape(rect);
	}
	/**
	 * Denote that size changed.
	 * 
	 * @param dimension the old dimensions.
	 */
	protected void changedSize(Dimension dimension) {
		renderer = null;
		super.changedSize(dimension);
	}
	/**
	 * Answer the default/initial value for height
	 * 
	 * @return	an integer representing the default/initial value for height
	 */
	protected int defaultHeight() {
		return 30;
	}
	/**
	 * Answer the default/initial value for string
	 * 
	 * @return	a String representing the default/inital value for string
	 */
	protected String defaultString() {
		return "label";
	}
	/**
	 * Answer the default/initial value for textColor
	 * 
	 * @return	a Color representing the default/initial value for textColor
	 */
	protected Color defaultTextColor() {
		return Color.black;
	}
	/**
	 * Answer the default/initial value for width
	 * 
	 * @return	an integer representing the default/initial value for width
	 */
	protected int defaultWidth() {
		return 60;
	}
	/** 
	 * Answers a Handle that will provide 
	 * editing capabilities on the receiver.
	 *
	 * @param x the x coordinate to potentially begin editing
	 * @param y the y coordinate to potentially begin editing
	 * @return	a Handle that will provide editing capabilities on
	 * the receiver
	 */
	public Handle editTool(int x, int y)  {
		return new LabelEditHandle(this);
	}
	/** 
	 * Answer the handles associated with the receiver.
	 * 
	 * @return	an array of the Handles associated with the receiver
	 */
	public Handle[] getHandles() {
		Handle superHandles[] = super.getHandles();
		Handle handles[] = new Handle[superHandles.length + 1];
		System.arraycopy(superHandles,0,handles,0,superHandles.length);
		handles[superHandles.length] = basicNewLineCreationHandle();
		return handles;
	}
	/** 
	 * Returns the current bounds of the label.
	 * 
	 * @return	a Rectangle representing the current bounds of the label
	 */
	public Rectangle getLabelBounds()  {
		return new Rectangle(x + 5, y + 2, width - 10, height - 4);
	}
	/** 
	 * Answer the renderer to paint the label.
	 * 
	 * @return	the StringRender to paint the label
	 */
	protected StringRenderer getRenderer()  {
		if (renderer == null) {
			Rectangle myLabelBounds = getLabelBounds();
			renderer = new BasicStringComposer(string, myLabelBounds.width, myLabelBounds.height);
		}
		return renderer;
	}
	/** 
	 * Answer the string the figure paints.
	 * 
	 * @return	the String the figure paints
	 */
	public String getString()  {
		return string;
	}
	/** 
	 * Answer the style which defines how to paint the figure.
	 * 
	 * @return	the DrawingStyle which defines how to paint the figure
	 */
	public DrawingStyle getStyle()  {
		DrawingStyle style = super.getStyle();
		style.setTextColor(textColor);
		return style;
	}
	/**
	 * Answer the Color to use when drawing text.
	 * 
	 * @return	the Color to use when drawing text
	 */
	public Color getTextColor() {
		return textColor;
	}
	/** 
	 * Paints the figure.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		super.paint(g);
		//g.drawRect(getLabelBounds().x, getLabelBounds().y, getLabelBounds().width, getLabelBounds().height);
		paintText(g);
	}
	/** 
	 * Paints the text.
	 *
	 * @param g the specified Graphics window
	 */
	public void paintText(Graphics original)  {
		Graphics g = original.create();
		g.setColor(textColor);
		Rectangle myLabelBounds = getLabelBounds();
		g.clipRect(myLabelBounds.x, myLabelBounds.y, myLabelBounds.width, myLabelBounds.height);
		getRenderer().paint(g, myLabelBounds.x, myLabelBounds.y);
	}
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the other figure.
	 * By default, make it the middle of the receiver.
	 * Subclasses may wish to do something more meaningful.
	 *
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 */
	public Locator requestConnection(Figure requestor, int x, int y) {
	if (requestor == this)
		return null;
	return new FigureRelativePoint(this,0.5,0.5);
	}
	/**
	 * Flush caches with respect to determining size.  This is a hook method.
	 * Subclasses may wish to override.
	 */
	protected void resetSizeCache() {
		renderer = null;
	}
	/** 
	 * Set the string the figure paints.
	 *
	 * @param newString the string to paint
	 */
	public void setString(String newString) {
		String oldString = string;
		string = newString;
		renderer = null;
		firePropertyChange(STRING_PROPERTY, oldString, newString);
	}
	/** 
	 * Set the style defining how to paint the figure.
	 * 
	 * @param style the specified DrawingStyle
	 */
	public void setStyle(DrawingStyle style)  {
		super.setStyle(style);
		if (style != null) {
			setTextColor(style.getTextColor());
		}
	}
	/**
	 * Set the Color to use when drawing text.
	 * 
	 * @param color the color
	 */
	public void setTextColor(Color color) {
		Color oldColor = textColor;
		textColor = color;
		firePropertyChange(TEXT_COLOR_PROPERTY, oldColor, color);
	}
}
package com.rolemodelsoft.drawlet.examples.graphnode;

/**
 * @(#)GraphNodeApplet.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.awt.*;
import java.awt.*;

/**
 * Here is a pathetically naive example of a simple Nodes and Arcs
 * Application.  It is meant to illustrate how simple it is to add your own
 * figures, and connect them.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class GraphNodeApplet extends java.applet.Applet {

	/**
	 * the canvas for this applet.
	 */
	DrawingCanvas canvas;
	/**
	 * Applet info.
	 */
	public String getAppletInfo() {
		return "GraphNode Drawlets Example v1.1.6 (30 Dec 1998), by RoleModel Software.";
	}
	/**
	 * Initializes the applet.
	 * You never need to call this directly, it is called automatically
	 * by the system once the applet is created.
	 * @see #start
	 * @see #stop
	 * @see #destroy
	 */
	public void init() {
		setLayout(new BorderLayout());
		
		// Create the canvas
		DrawingCanvasComponent canvasComponent = new BufferedDrawingCanvasComponent();
		canvas = canvasComponent.getCanvas();
		canvas.setTool(new GraphNodeTool(canvas));
		canvasComponent.setSize(getSize());
		add("Center",canvasComponent);
	}
}
package com.rolemodelsoft.drawlet.examples.graphnode;

/**
 * @(#)GraphNodeTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.*;

/**
 * Here is a simple tool that extends the SelectionTool to create GraphNodes
 * when a doubleClick event occurs.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class GraphNodeTool extends SelectionTool {
	/**
	 * @param canvas the canvas this tool will draw on/manipulate
	 */

	public GraphNodeTool(DrawingCanvas canvas) {
		super(canvas);
	}
	/**
	 * Answer a new graph node
	 */
	protected GraphNode basicNewGraphNode() {
		return new GraphNode();
	}
	/**
	 * Called if the mouse is double-clicked.
	 *
	 * @param e the event 
	 * @see #mouseClicked
	 */
	protected void mouseDoubleClicked(MouseEvent e) {
		int x = e.getX();
		int y = e.getY();
		super.mouseDoubleClicked(e);
		if (e.isConsumed())
			return;
		Figure newFigure = basicNewGraphNode();
		newFigure.setStyle(canvas.getStyle());
		newFigure.move(e.getX(), e.getY());
		canvas.addFigure(newFigure);
		e.consume();
	}
}
package com.rolemodelsoft.drawlet.examples.jfc;

/**
 * @(#)ExitingFrame.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

public class ExitingFrame extends JFrame implements WindowListener {
/**
 * ExitingFrame constructor comment.
 */
public ExitingFrame() {
	super();
	addWindowListener(this);
}
/**
 * ExitingFrame constructor comment.
 * @param title java.lang.String
 */
public ExitingFrame(String title) {
	super(title);
	addWindowListener(this);
}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry.
	 */
	public static void main(java.lang.String[] args) {
		try {
			java.awt.Frame frame = new ExitingFrame();
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of ExitingFrame");
			exception.printStackTrace(System.out);
		}
	}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowActivated(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowClosed(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	System.exit(0);
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowClosing(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	this.dispose();
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowDeactivated(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowDeiconified(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowIconified(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
/**
 * Method to handle events for the WindowListener interface.
 * @param e java.awt.event.WindowEvent
 */
public void windowOpened(WindowEvent e) {
	// user code begin {1}
	// user code end
	// user code begin {2}
	// user code end
}
}
package com.rolemodelsoft.drawlet.examples.jfc;

/**
 * @(#)SimpleModelExample.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.jfc.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.LabelTool;
import com.rolemodelsoft.drawlet.examples.*;
import com.rolemodelsoft.drawlet.util.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This is a trivial example of how one might separate some underlying model 
 * from presentation to the user.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimpleModelPanel extends SimplePanel {

	/**
	 * the model for this application.
	 */
	protected SingleDrawingModel model;

	/**
	 * the temporary file name that will be saved to/restore from.
	 */
	protected static String TempFileName = "temp.rmd";
	/**
	 * Default constructor.
	 */
	public SimpleModelPanel() {
		super();
	}
	/**
	 * @param layout the layout this application should use.
	 */
	public SimpleModelPanel(java.awt.LayoutManager layout) {
		super(layout);
	}
	/**
	 * SimplePanelExample constructor comment.
	 * @param layout the layout this application should use.
	 * @param isDoubleBuffered determines whether the application will be
	 * double buffered or not.
	 */
	public SimpleModelPanel(java.awt.LayoutManager layout, boolean isDoubleBuffered) {
		super(layout, isDoubleBuffered);
	}
	/**
	 * @param isDoubleBuffered  determines whether the application will be
	 * double buffered or not.
	 */
	public SimpleModelPanel(boolean isDoubleBuffered) {
		super(isDoubleBuffered);
	}
	/**
	 * @return Component the component holding the canvas.
	 */
	protected JComponent getCanvasComponent() {
		canvas = new SimpleDrawingCanvas(getModel().getDrawing());
		JComponent component = new JDrawingCanvasComponent(canvas);
		ValueAdapter adapter = new ValueAdapter(getModel(),"getDrawing",canvas,"setDrawing");
		return component;
	}
	/**
	 * @return the model
	 */
	protected SingleDrawingModel getModel() {
		if ( model == null ) {
			model = new SingleDrawingModel();
		}
		return model;
	}
	/**
	 * Initialize class
	 */
	protected void initialize() {
		super.initialize();
		toolBar.addSeparator();

		class ClearAction extends AbstractAction {
			SingleDrawingModel model;
			public ClearAction(String name, SingleDrawingModel model) {
				super(name);
				this.model = model;
			}
			public void actionPerformed(ActionEvent e) {
				model.clearDrawing();
			}
		}
		class SaveAction extends AbstractAction {
			SingleDrawingModel model;
			String fileName;
			public SaveAction(String name, SingleDrawingModel model, String fileName) {
				super(name);
				this.model = model;
				this.fileName = fileName;
			}
			public void actionPerformed(ActionEvent e) {
				model.saveDrawing(fileName);
			}
		}
		class RestoreAction extends AbstractAction {
			SingleDrawingModel model;
			String fileName;
			public RestoreAction(String name, SingleDrawingModel model, String fileName) {
				super(name);
				this.model = model;
				this.fileName = fileName;
			}
			public void actionPerformed(ActionEvent e) {
				model.restoreDrawing(fileName);
			}
		}
		toolBar.add(new ClearAction("Clear", getModel()));
		toolBar.add(new SaveAction("Save",getModel(),TempFileName));
		toolBar.add(new RestoreAction("Restore",getModel(),TempFileName));
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry.
	 */
	public static void main(java.lang.String[] args) {
		try {
			javax.swing.JFrame frame = new ExitingFrame( "Drawlets" );
			SimpleModelPanel aDrawletsExample;
			aDrawletsExample = new SimpleModelPanel();
			frame.getContentPane().add("Center", aDrawletsExample);
			frame.setSize(aDrawletsExample.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of com.rolemodelsoft.drawlet.examples.jfc.SimpleModelExample");
			exception.printStackTrace(System.out);
		}
	}
}
package com.rolemodelsoft.drawlet.examples.jfc;

/**
 * @(#)SimplePanelExample.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.jfc.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.shapes.lines.DrawLineTool;
import com.rolemodelsoft.drawlet.text.LabelTool;
import javax.swing.*;
import java.awt.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic DrawingTool.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimplePanel extends JPanel {
	protected JToolBar toolPalette, toolBar;
	protected DrawingCanvas canvas;
	/**
	 * Default constructor.
	 */
	public SimplePanel() {
		super();
		initialize();
	}
	/**
	 * @param layout the layout this application should use.
	 */
	public SimplePanel(java.awt.LayoutManager layout) {
		super(layout);
		initialize();
	}
	/**
	 * @param layout the layout this application should use.
	 * @param isDoubleBuffered determines whether the application will be
	 * double buffered or not.
	 */
	public SimplePanel(java.awt.LayoutManager layout, boolean isDoubleBuffered) {
		super(layout, isDoubleBuffered);
		initialize();
	}
	/**
	 * @param isDoubleBuffered  determines whether the application will be
	 * double buffered or not.
	 */
	public SimplePanel(boolean isDoubleBuffered) {
		super(isDoubleBuffered);
		initialize();
	}
	/**
	 * @return Component the component holding the canvas.
	 */
	protected JComponent getCanvasComponent() {
		JDrawingCanvasComponent component = new JDrawingCanvasComponent();
		canvas = component.getCanvas();
		return component;
	}
	/**
	 * @return the ToolPalette.
	 */
	public JToolBar getToolBar() {
		if ( toolBar == null ) {
			toolBar = new JToolBar();
			toolBar.setFloatable( false );
			toolBar.add(new CanvasAction("", new ImageIcon("greygrey/copy.gif", "Copy"), canvas, "copySelections"));
			toolBar.add(new CanvasAction("", new ImageIcon("greygrey/cut.gif", "Cut"), canvas, "cutSelections"));
			toolBar.add(new CanvasAction("", new ImageIcon("greygrey/paste.gif", "Paste"), canvas, "paste"));
			toolBar.add(new CanvasAction("", new ImageIcon("greygrey/systempaste.gif", "Paste From System"), canvas, "pasteFromSystem"));
		}
		return toolBar;
	}
	/**
	 * @return the ToolPalette.
	 */
	public JToolBar getToolPalette() {
		if ( toolPalette == null ) {
			toolPalette = new JToolBar();
			toolPalette.setLayout( new BoxLayout( toolPalette, BoxLayout.Y_AXIS ) );
			toolPalette.setFloatable( false );
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/select.gif", "Select"), canvas, (InputEventHandler)new SelectionTool(canvas)));
			toolPalette.addSeparator();
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/label.gif", "Label"), canvas, (InputEventHandler)new LabelTool(canvas)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/line.gif", "Line"), canvas, (InputEventHandler)new ConnectingLineTool(canvas)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/box.gif", "Box"), canvas, (InputEventHandler)new RectangleTool(canvas)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/rrect.gif", "Rounded"), canvas, (InputEventHandler)new RectangularCreationTool(canvas,RoundedRectangleShape.class)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/ellipse.gif", "Ellipse"), canvas, (InputEventHandler)new EllipseTool(canvas)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/triangle.gif", "Triangle"), canvas, (InputEventHandler)new PolygonTool(canvas,3)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/pentagon.gif", "Pentagon"), canvas, (InputEventHandler)new PolygonTool(canvas,5)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/nsided.gif", "N-Sided"), canvas, (InputEventHandler)new AnySidedPolygonTool(canvas)));
			toolPalette.add(new CanvasToolAction("", new ImageIcon("greygrey/freehand.gif", "Smooth Line"), canvas, (InputEventHandler)new DrawLineTool(canvas)));
		}
		return toolPalette;
	}
	/**
	 * Initialize class
	 */
	protected void initialize() {
		setName("Drawlets");
		setSize(426, 240);
		//setFont(new java.awt.Font("timesroman", 0, 12));
		setLayout(new BorderLayout());
		
		// Create the canvas
		add(getCanvasComponent());

		// Set the drawing style
		DrawingStyle style = new SimpleDrawingStyle();
		style.setBackgroundColor(Color.blue);
		style.setForegroundColor(Color.white);
		canvas.setStyle(style);

		// Create the tool palette
		add(getToolPalette(),BorderLayout.WEST);
		add(getToolBar(),BorderLayout.NORTH);
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry
	 */
	public static void main(java.lang.String[] args) {
		try {
			javax.swing.JFrame frame = new ExitingFrame( "Drawlets" );
			SimplePanel aTestCanvas;
			aTestCanvas = new SimplePanel();
			frame.getContentPane().add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
	}
}
package com.rolemodelsoft.drawlet.examples.jfc;

/**
 * @(#)SimpleScrollingPanel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.jfc.*;
import javax.swing.*;
import java.awt.*;
/**
 * This is a beginning attempt at a scrolling implemenation with JFC. There are obvious problems,
 * possibly dealing with the ViewPort, at least in VisualAge (it has not yet been tested outside).
 * Hopefully it will at least give you an idea for how to implement it, and the problems may be fixed
 * either outside of VisualAge or in Java 2.
 */
public class SimpleScrollingPanel extends SimplePanel {
/**
 * SimpleScrollingPanel constructor comment.
 */
public SimpleScrollingPanel() {
	super();
}
/**
 * SimpleScrollingPanel constructor comment.
 * @param layout java.awt.LayoutManager
 */
public SimpleScrollingPanel(java.awt.LayoutManager layout) {
	super(layout);
}
/**
 * SimpleScrollingPanel constructor comment.
 * @param layout java.awt.LayoutManager
 * @param isDoubleBuffered boolean
 */
public SimpleScrollingPanel(java.awt.LayoutManager layout, boolean isDoubleBuffered) {
	super(layout, isDoubleBuffered);
}
/**
 * SimpleScrollingPanel constructor comment.
 * @param isDoubleBuffered boolean
 */
public SimpleScrollingPanel(boolean isDoubleBuffered) {
	super(isDoubleBuffered);
}
	/**
	 * @return JComponent the component holding the canvas.
	 */
	protected JComponent getCanvasComponent() {
		canvas = new SimpleDrawingCanvas( new SimpleDrawing( 500, 500 ) );
		JDrawingCanvasComponent component = new JDrawingCanvasComponent( canvas );
		JScrollPane scroller = new JScrollPane( component );
		return scroller;
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry
	 */
	public static void main(java.lang.String[] args) {
		try {
			javax.swing.JFrame frame = new ExitingFrame( "Drawlets" );
			SimpleScrollingPanel aTestCanvas;
			aTestCanvas = new SimpleScrollingPanel();
			frame.getContentPane().add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
	}
}
package com.rolemodelsoft.drawlet.examples.jfc;

/**
 * @(#)SimplestPanel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.jfc.*;
import com.rolemodelsoft.drawlet.shapes.lines.ConnectingLineTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RectangleTool;
import com.rolemodelsoft.drawlet.shapes.RectangularCreationTool;
import com.rolemodelsoft.drawlet.shapes.rectangles.RoundedRectangleShape;
import com.rolemodelsoft.drawlet.shapes.ellipses.EllipseTool;
import com.rolemodelsoft.drawlet.shapes.polygons.PolygonTool;
import com.rolemodelsoft.drawlet.shapes.polygons.AnySidedPolygonTool;
import com.rolemodelsoft.drawlet.text.LabelTool;
import javax.swing.*;
import java.awt.*;

/**
 * Although there are plenty of ways to use the drawlet framework, it may not
 * be apparent without some examples.  Here is a very simple one which provides
 * a basic DrawingTool.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class SimplestPanel extends JPanel {
	/**
	 * Default constructor.
	 */
	public SimplestPanel() {
		super();
		initialize();
	}
	/**
	 * @param layout the layout this application should use.
	 */
	public SimplestPanel(java.awt.LayoutManager layout) {
		super(layout);
	}
	/**
	 * @param layout the layout this application should use.
	 * @param isDoubleBuffered determines whether the application will be
	 * double buffered or not.
	 */
	public SimplestPanel(java.awt.LayoutManager layout, boolean isDoubleBuffered) {
		super(layout, isDoubleBuffered);
	}
	/**
	 * @param isDoubleBuffered  determines whether the application will be
	 * double buffered or not.
	 */
	public SimplestPanel(boolean isDoubleBuffered) {
		super(isDoubleBuffered);
	}
	/**
	 * Initialize class
	 */
	private void initialize() {
		setName("Drawlets");
		setSize(426, 240);
		//setFont(new java.awt.Font("timesroman", 0, 12));
		setLayout(new BorderLayout());
		
		// Create the canvas
		JDrawingCanvasComponent canvasComponent = new JDrawingCanvasComponent();
		DrawingCanvas canvas = canvasComponent.getCanvas();
	//	canvasComponent.setSize(300,240);
		add(canvasComponent);

		// Set the drawing style
		DrawingStyle style = new SimpleDrawingStyle();
		style.setBackgroundColor(Color.blue);
		style.setForegroundColor(Color.white);
		canvas.setStyle(style);

		// Create the tool palette
		JToolBar toolPalette = new JToolBar();
		toolPalette.add(new CanvasToolAction("Select", canvas, (InputEventHandler)new SelectionTool(canvas)));
		toolPalette.addSeparator();
		toolPalette.add(new CanvasToolAction("Label", canvas, (InputEventHandler)new LabelTool(canvas)));
		toolPalette.add(new CanvasToolAction("Line", canvas, (InputEventHandler)new ConnectingLineTool(canvas)));
		toolPalette.add(new CanvasToolAction("Line w/Arrow", canvas, (InputEventHandler)new com.rolemodelsoft.drawlet.shapes.lines.AdornedLineTool(canvas)));
		toolPalette.add(new CanvasToolAction("Box", canvas, (InputEventHandler)new RectangleTool(canvas)));
		toolPalette.add(new CanvasToolAction("Rounded", canvas, (InputEventHandler)new RectangularCreationTool(canvas,RoundedRectangleShape.class)));
		toolPalette.add(new CanvasToolAction("Ellipse", canvas, (InputEventHandler)new EllipseTool(canvas)));
		toolPalette.add(new CanvasToolAction("N-Sided", canvas, (InputEventHandler)new AnySidedPolygonTool(canvas)));
		add(toolPalette,BorderLayout.NORTH);
	}
	/**
	 * main entrypoint - starts the part when it is run as an application
	 * 
	 * @param args the arguments passed to the application on entry
	 */
	public static void main(java.lang.String[] args) {
		try {
			JFrame frame = new ExitingFrame( "Drawlets" );
			SimplestPanel aTestCanvas;
			aTestCanvas = new SimplestPanel();
			frame.getContentPane().add("Center", aTestCanvas);
			frame.setSize(aTestCanvas.getSize());
			frame.setVisible(true);
		} catch (Throwable exception) {
			System.err.println("Exception occurred in main() of TestCanvas");
			exception.printStackTrace(System.out);
		}
	}
}
package com.rolemodelsoft.drawlet.examples;

/**
 * @(#)SingleDrawingModel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.util.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.io.*;
import java.awt.*;

/**
 * @version 	1.1.6, 12/29/98
 */
 
public class SingleDrawingModel extends BasicObservable {
	/**
	 * The drawing
	 */
	Drawing drawing = new SimpleDrawing();
/**
 * Default constructor.
 */
public SingleDrawingModel() {
}
/**
 * Clears the drawing.
 */
public void clearDrawing() {
	Drawing newDrawing = new SimpleDrawing();
	newDrawing.setStyle( drawing.getStyle() );
	setDrawing(newDrawing);
}
/**
 * Gets the drawing.
 *
 * @return the Drawing currently associated with the receiver
 */
public Drawing getDrawing() {
	return drawing;
}
	/**
	 * Print the drawing.
	 */
	public void printDrawing( Frame f ) {
		Toolkit tk = Toolkit.getDefaultToolkit();
		java.util.Properties props = new java.util.Properties();

		if( tk != null ) {
			PrintJob pj = tk.getPrintJob( f, "Drawlets print job", props );

			if ( pj != null ) {
				Graphics pg = pj.getGraphics();

				if( pg != null ) {
					try {
						drawing.paint( pg );
					}
					finally {
						pg.dispose();
					}
				}
				pj.end();
			}
			System.out.println( props );
		}
	}
/**
 * Restore the drawing from the given file name
 * 
 * @param fileName the file name
 */
public void restoreDrawing(String fileName) {
	try {
		FileInputStream fileIn = new FileInputStream(fileName);
		ObjectInputStream in = new ObjectInputStream(fileIn);
		setDrawing((Drawing)in.readObject());
		in.close();
	} catch (Throwable e) {System.out.println("Cannot read drawing from file " + fileName);}
}
/**
 * Save the drawing to the given file name.
 *
 * @param fileName the file name
 */
public void saveDrawing(String fileName) {
	try {
		FileOutputStream fileOut = new FileOutputStream(fileName);
		ObjectOutputStream objectOut = new ObjectOutputStream(fileOut);
		objectOut.writeObject(getDrawing());
		objectOut.close();
		fileOut.close();			
	} catch (Throwable e) {
		System.out.println("Cannot write drawing to file " + fileName + "\nException string: " + e.toString() + "\nException message: " + e.getMessage());
		e.printStackTrace();
	}
}
/**
 * Set the drawing associated with the receiver to the given drawing,
 * and tell observers about it.
 *
 * @param newDrawing the drawing
 */
public void setDrawing(Drawing newDrawing) {
//	newDrawing.setStyle( drawing.getStyle() );
	drawing = newDrawing;
	notifyObservers("getDrawing");
}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)Figure.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;
import java.util.Hashtable;
import java.util.Enumeration;
import com.rolemodelsoft.drawlet.util.*;
import java.io.Serializable;
import java.beans.PropertyChangeListener;

/**
 * This interface defines a generic Figure that can be drawn on and 
 * potentially manipulated on a DrawingCanvas (or elsewhere).
 *
 * @version 	1.1.6, 12/28/98
 */
 
public interface Figure extends Paintable, Duplicatable, Serializable {
	/**
	 * The shape property selector.
	 */
	public static String SHAPE_PROPERTY = "shape";

	/**
	 * The size property selector.
	 */
	public static String SIZE_PROPERTY = "size";

	/**
	 * The location property selector.
	 */
	public static String LOCATION_PROPERTY = "location";

	/**
	 * The style property selector.
	 */
	public static String STYLE_PROPERTY = "style";

	/**
	 * The fill color property selector.
	 */
	public static String FILL_COLOR_PROPERTY = "fillColor";

	/**
	 * The line color property selector.
	 */
	public static String LINE_COLOR_PROPERTY = "lineColor";

	/**
	 * The text color property selector.
	 */
	public static String TEXT_COLOR_PROPERTY = "textColor";

	/**
	 * The string property selector.
	 */
	public static String STRING_PROPERTY = "string";

	/**
	 * The relation property selector.
	 */
	public static String RELATION_PROPERTY = "relation";
	/**
	 * Add a PropertyChangeListener to the listener list.
	 *
	 * @param listener  The PropertyChangeListener to be added
	 */

	public void addPropertyChangeListener(PropertyChangeListener listener);
	/**
	 * Add a RelatedLocationListener to the listener list.
	 *
	 * @param listener  The RelatedLocationListener to be added
	 */

	public void addRelatedLocationListener(RelatedLocationListener listener);
	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * Figure, where x and y are defined to be relative to the 
	 * coordinate system of this figure.  
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return  boolean value of <code>true</code> if x and y are "inside" this Figure;
	 *          <code>false</code> otherwise.
	 */
	public abstract boolean contains(int x, int y);
	/**  
	 * Checks whether a specified Figure is completely "inside" this
	 * Figure, where the figures share the same coordinate system.
	 * It is intended that objects that are smart enough to use something
	 * besides bounds will implement this intelligently, and isWithin will
	 * turn the parameter and receiver around to invoke this.
	 *
	 * @param figure the Figure to test for inclusion
	 * @return boolean value of <code>true</code> if the specified Figure is completely "inside" this Figure
	 *         <code>false</code> otherwise.
	 */
	public abstract boolean contains(Figure figure);
	/**  
	 * Checks whether a specified Rectangle is "inside" this
	 * Figure, where the Rectangle and this Figure are in the same coordinate system  
	 *
	 * @param box the rectangle to test for inclusion
	 * @return boolean value of <code>true</code> if the specified Rectangle is "inside" this Figure;
	 *         <code>false</code> otherwise.
	 */
	public abstract boolean contains(Rectangle box);
	/** 
	 * Clean up as appropriate if the figure is no longer connected to others.
	 */
	public abstract void disconnect();
	/** 
	 * Answers a Handle that will provide 
	 * editing capabilities on the receiver, or null.
	 *
	 * @param x the x coordinate to potentially begin editing
	 * @param y the y coordinate to potentially begin editing
	 * @return a Handle that will provide editing capabilities on the receiver, or null
	 */
	public abstract Handle editTool(int x, int y);
	/** 
	 * Answer the handles associated with the receiver.
	 *
	 * @return an array of the Handles associated with the receiver
	 */
	public abstract Handle[] getHandles();
	/** 
	 * Returns the current locator of this figure.
	 * This may or may not represent the top left of the receiver's area.
	 *
	 * @return the current Locator of this figure
	 */
	public abstract Locator getLocator();
	/** 
	 * Answer the style which defines how to paint the figure.
	 * NOTE: It may be valid to return null if the figure just doesn't care
	 *
	 * @return the DrawingStyle which defines how to paint the figure
	 */
	public abstract DrawingStyle getStyle();
	/** 
	 * Answers whether the receiver intersects another figure.
	 *
	 * @param anotherFigure the figure the receiver is potentially intersecting.
	 * @return boolean value of <code>true</code> if the receiver intersects another figure;
	 *         <code>false</code> otherwise.
	 */
	public abstract boolean intersects(Figure anotherFigure);
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 *
	 * @param box the Rectangular area
	 * @return boolean value of <code>true</code> if the receiver intersects the Rectangular area;
	 *         <code>false</code> otherwise.
	 */
	public abstract boolean intersects(Rectangle box);
	/** 
	 * Answers whether the receiver is obsolete. 
	 * True if some event has happened that makes this a meaningless object.
	 *
	 * @return boolean value of <code>true</code> if some event has happened
	 * that makes this a meaningless object;
	 *         <code>false</code> otherwise.
	 */
	public abstract boolean isObsolete();
	/** 
	 * Answers whether the receiver is fully within another Figure.
	 *
	 * @param anotherFigure the figure the receiver is potentially inside.
	 * @return boolean value of <code>true</code> if the receiver is fully within another Figure;
	 *         <code>false</code> otherwise.
	 */
	public abstract boolean isWithin(Figure anotherFigure);
	/** 
	 * Answers whether the receiver is fully within a Rectangular area.
	 *
	 * @param box the Rectangular area
	 * @return boolean value of <code>true</code> if the receiver is fully within a Rectangular area;
	 *         <code>false</code> otherwise.
	 */
	public abstract boolean isWithin(Rectangle box);
	/** 
	 * Answers a locator corresponding to a significant point on the receiver.
	 * It is up to the receiver whether to answer an absolute or relative locator
	 * or even whether to give a non-null answer or ignore the x, y coordinates.
	 *
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 * @return a Locator corresponding to a significant point on the receiver
	 */
	public abstract Locator locatorAt(int x, int y);
	/** 
	 * Moves the Figure to a new location. The x and y coordinates
	 * are in the parent's coordinate space.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @see #location
	 * @see #reshape
	 */
	public abstract void move(int x, int y);
	/** 
	 * Moves the Figure to a new location.
	 * Note: Subclasses may wish to update dependencies based on this new location
	 *
	 * @param locator the Locator which identifies the desired x, y coordinates.
	 * @see #location
	 */
	public abstract void move(Locator locator);
	/**
	 * Answer with an Enumeration over the RelatedLocationListeners.
	 *
	 * @return an Enumeration over the RelatedLocationListeners
	 */

	public Enumeration relatedLocationListeners();
	/**
	 * Remove a PropertyChangeListener from the listener list.
	 *
	 * @param listener  The PropertyChangeListener to be removed
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener);
	/**
	 * Remove a RelatedLocationListener from the listener list.
	 *
	 * @param listener  The RelatedLocationListener to be removed
	 */
	public void removeRelatedLocationListener(RelatedLocationListener listener);
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the other Figure.
	 * It is up to the receiver whether to answer a relative locator
	 * or even whether to give a non-null answer or ignore the x, y coordinates.
	 *
	 * @param requestor the Figure requesting a connection
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 * @return a Locator corresponding to a significant point on the
	 * receiver that will serve as a connection to the other Figure
	 */
	public abstract Locator requestConnection(Figure requestor, int x, int y);
	/** 
	 * Reshapes the Figure to the specified bounding box.
	 * Observable objects may wish to notify dependents
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 * @see #bounds
	 * @see #move
	 * @see #resize
	 */
	public abstract void setBounds(int x, int y, int width, int height);
	/**
	 * Resizes the Figure to the specified width and height.
	 *
	 * @param width the width of the figure
	 * @param height the height of the figure
	 * @see #size
	 * @see #reshape
	 */
	public abstract void setSize(int width, int height);
	/** 
	 * Resizes the Figure to the specified dimension.
	 *
	 * @param d the figure dimension
	 * @see #size
	 * @see #reshape
	 */
	public abstract void setSize(Dimension d);
	/** 
	 * Set the style defining how to paint the Figure.
	 *
	 * @param style the specified DrawingStyle
	 */
	public abstract void setStyle(DrawingStyle style);
	/** 
	 * Moves the Figure in the x and y direction.  
	 * Observable objects may wish to notify dependents
	 *
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 * @see #location
	 * @see #reshape
	 */
	public abstract void translate(int x, int y);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)FigureEnumeration.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

/**
 * An object that implements the FigureEnumeration interface generates a
 * series of Figures, one at a time. Successive calls to the 
 * <code>nextFigure</code> method return successive Figures of the 
 * series.  This is just like a standard Enumeration, except we know we are dealing
 * with Figures and therefore users don't have to do any casting.
 * <p>
 * For example, to print all Figures of a vector <i>v</i>:
 * <blockquote><pre>
 *     for (FigureEnumeration e = MyFigureEnumerator(v); e.hasMoreElements() ;) {
 *         System.out.println(e.nextElement());<br>
 *     }
 * </pre></blockquote>
 * <p>
 *
 * @see     java.util.Enumeration
 *
 * @version 	1.1.6, 12/28/98
 * @since   JDK1.1
 */
 
public interface FigureEnumeration {
	/**
	 * Tests if this enumeration contains more figures.
	 *
	 * @return  boolean value of <code>true</code> if this enumeration contains more figures;
	 *          <code>false</code> otherwise.
	 * @since   JDK1.1
	 */
	boolean hasMoreElements();
	/**
	 * Returns the next Figure of this enumeration.
	 *
	 * @return     the Figure which is the next element of this enumeration. 
	 * @exception  NoSuchElementException  if no more elements exist.
	 * @since      JDK1.1
	 */
	Figure nextElement();
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)FigureHolder.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
/**
 * When implemented, this interface allows implementing classes to serve a Figure.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public interface FigureHolder {

	/** 
	 * Answer the figure the receiver is holding.
	 *
	 * @return the Figure the receiver is holding
	 */
	public Figure getFigure();
	/** 
	 * Set the figure the receiver is holding.
	 *
	 * @param figure the Figure to hold
	 */
	public void setFigure(Figure figure);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)Handle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * This interface defines protocols for Handles
 * which are event handlers that may visibly appear on a DrawingCanvas.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface Handle extends InputEventHandler {

	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * handle, where x and y are defined to be relative to the 
	 * coordinate system of this handle.  
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified
	 * x,y location is "inside" this handle;
	 * 			<code>false</code> otherwise.
	 */
	public abstract boolean contains(int x, int y);
	/** 
	 * Returns the current bounds of this handle.
	 *
	 * @return a Rectangle representing the current bounds of this handle
	 */
	public abstract Rectangle getBounds();
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 *
	 * @param box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * rectangular area;
	 * 			<code>false</code> otherwise.
	 */
	public abstract boolean intersects(Rectangle box);
	/** 
	 * Paints the handle.
	 *
	 * @param g the specified Graphics window
	 */
	public abstract void paint(Graphics g);
	/**
	 * Release control of the canvas and clean up if necessary.
	 * Since this is a public method,
	 * don't assume the receiver actually has control.
	 *
	 * @param canvas the canvas which the receiver is to release control
	 */
	public abstract void releaseControl(DrawingCanvas canvas);
	/**  
	 * Make the handle be the event handler for the canvas.
	 * Note, once it takes control, it is obligated to return 
	 * at a future point in time.  
	 *
	 * @param canvas the canvas which the receiver is to control
	 */
	public abstract void takeControl(DrawingCanvas canvas);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)EventHandler.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;
import java.awt.event.*;

/**
 * This interface defines the basic protocol to handle Events.
 * Implementing classes can be used as tools for DrawingCanvas, etc.
 * It is expected that most implementers will use the various event methods (i.e.
 * actionPerformed, keyPressed, MouseReleased, etc.) for handling and dispatching
 * events, but it should not be assumed that this will be the only InputEventHandler
 * implementation.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface InputEventHandler extends KeyListener, MouseListener, MouseMotionListener {

}
package com.rolemodelsoft.drawlet.jfc;

/**
 * @(#)CanvasToolAction.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import javax.swing.*;

/**
 * Basically this class is used to invoke actions on the canvas (ex. copy, paste etc...).
 * @version 	1.1.6, 12/29/98
 */

public class CanvasAction extends AbstractAction {

	/**
	 * The canvas for which the tool is to be set.
	 */
	DrawingCanvas canvas;

	/**
	 * The action to be invoked on the canvas.
	 */
	String action;
	/**
	 * Creates a new, default CanvasToolAction.
	 */
	public CanvasAction() {
		super();
	}
/**
 * Creates a new CanvasToolAction and initializes it with the given name.
 *
 * @param name a String representing the name.
 */
public CanvasAction(String name) {
	super(name);
}
	/**
	 * Create a CanvasToolAction which will set the canvases tool when performed.
	 * 
	 * @param name the name for the action
	 * @param canvas the canvas to set the tool for
	 * @param tool the tool to have the canvas use
	 */
	public CanvasAction(String name, DrawingCanvas canvas, String action) {
		super(name);
		this.canvas = canvas;
		this.action = action;
	}
	/**
	 * Creates a new CanvasToolAction and initializes it with the given name and icon.
	 *
	 * @param name the name for this action
	 * @param icon the icon
	 */
	public CanvasAction(String name, Icon icon) {
		super(name, icon);
	}
	/**
	 * Create a CanvasToolAction which will set the canvases tool when performed.
	 * 
	 * @param name the name for this action
	 * @param icon the icon
	 * @param canvas the canvas to set the tool for
	 * @param tool the tool to have the canvas use
	 */
	public CanvasAction(String name, Icon icon, DrawingCanvas canvas, String action) {
		super(name, icon);
		this.canvas = canvas;
		this.action = action;
	}
	/**
	 * Set the tool of the canvas.
	 * 
	 * @param e the event
	 */
	public void actionPerformed(java.awt.event.ActionEvent e) {
		try {
			canvas.getClass().getMethod(action, new Class[] {}).invoke( canvas, new Class[] {} );
		} catch ( Exception exception ) {
		}
	}
}
package com.rolemodelsoft.drawlet.jfc;

/**
 * @(#)CanvasToolAction.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import javax.swing.*;

/**
 * This class is used to create tools that will be used on 
 * the canvas (ex figure creation, figure selection etc...).
 * @version 	1.1.6, 12/29/98
 */

public class CanvasToolAction extends AbstractAction {

	/**
	 * The canvas for which the tool is to be set.
	 */
	DrawingCanvas canvas;

	/**
	 * The tool the canvas is to be set to use.
	 */
	InputEventHandler tool;
	/**
	 * Creates a new, default CanvasToolAction.
	 */
	public CanvasToolAction() {
		super();
	}
/**
 * Creates a new CanvasToolAction and initializes it with the given name.
 *
 * @param name a String representing the name.
 */
public CanvasToolAction(String name) {
	super(name);
}
	/**
	 * Create a CanvasToolAction which will set the canvases tool when performed.
	 * 
	 * @param name the name for the action
	 * @param canvas the canvas to set the tool for
	 * @param tool the tool to have the canvas use
	 */
	public CanvasToolAction(String name, DrawingCanvas canvas, InputEventHandler tool) {
		super(name);
		this.canvas = canvas;
		this.tool = tool;
	}
	/**
	 * Creates a new CanvasToolAction and initializes it with the given name and icon.
	 *
	 * @param name the name for this action
	 * @param icon the icon
	 */
	public CanvasToolAction(String name, Icon icon) {
		super(name, icon);
	}
	/**
	 * Create a CanvasToolAction which will set the canvases tool when performed.
	 * 
	 * @param name the name for this action
	 * @param icon the icon
	 * @param canvas the canvas to set the tool for
	 * @param tool the tool to have the canvas use
	 */
	public CanvasToolAction(String name, Icon icon, DrawingCanvas canvas, InputEventHandler tool) {
		super(name, icon);
		this.canvas = canvas;
		this.tool = tool;
	}
	/**
	 * Set the tool of the canvas.
	 * 
	 * @param e the event
	 */
	public void actionPerformed(java.awt.event.ActionEvent e) {
			canvas.setTool(tool);
	}
}
package com.rolemodelsoft.drawlet.jfc;

/**
 * @(#)JDrawingCanvasComponent.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.util.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.awt.datatransfer.*;
import javax.swing.JComponent;

import java.beans.*;

/**
 * This is a Swing component which owns and acts on a DrawingCanvas.
 *
 * @version 	1.1.6, 12/29/98
 */ 

 public class JDrawingCanvasComponent extends JComponent {
	
	/**
	 * The drawing canvas we are displaying/manipulating.
	 */
	protected DrawingCanvas canvas;
	
	/**
	 * Creates a new, default JDrawingCanvas Component.
	 */
	public JDrawingCanvasComponent() {
		this(new SimpleDrawingCanvas());
	}
	/**
	 * Creates a new JDrawingCanvasComponent and initializes it with the given DrawingCanvas.
	 *
	 * @param canvas the canvas we are displaying/manipulating
	 */
	public JDrawingCanvasComponent(DrawingCanvas canvas) {
		super();
		this.setCanvas(canvas);
		if (canvas instanceof MouseListener)
			this.addMouseListener((MouseListener)canvas);
		if (canvas instanceof MouseMotionListener)
			this.addMouseMotionListener((MouseMotionListener)canvas);
		if (canvas instanceof KeyListener)
			this.addKeyListener((KeyListener)canvas);
	}
	/**
	 * Answers the receiver's canvas.
	 *
	 * @return the DrawingCanvas we are displaying/manipulating
	 */
	public DrawingCanvas getCanvas() {
		return canvas;
	}
	/**
	 * @return the value of the preferredSize property.
	 */
	public Dimension getPreferredSize() {
		return getCanvas().getBounds().getSize();
	}
	/**
	 * Tell the awt that this object is able to handle the focus
	 *
	 * @return boolean value of <code>true</code> if the receiver
	 * is focus traversable;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isFocusTraversable() {
		return true;
	}
	/** 
	 * Paints the component.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g) {
		super.paint(g);
		canvas.paint(g);
	}
	/** 
	 * Repaints part of the component. This will result in a
	 * call to update as soon as possible.
	 * 
	 * @param rectangle is the region to be repainted
	 */
	public void repaint(Rectangle rectangle) {
	/*
	 * Note that we're fudging by a pixel to avoid
	 * differences in the way various drawing primitives determine 
	 * where to start/stop drawing (draw vs. fill)
	 */
		repaint(rectangle.x, rectangle.y, rectangle.width + 1, rectangle.height + 1);
	}
	/**
	 * Sets the canvas we are associated with.
	 *
	 * @param canvas the canvas we are to draw/manipulate
	 */
	protected void setCanvas(DrawingCanvas canvas) {
		if (canvas instanceof ComponentHolder)
			((ComponentHolder)canvas).setComponent(this);
		this.canvas = canvas;
	}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)LabelHolder.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * When implemented, this interface allows implementing classes to serve a 
 * String to appear in a particular Rectangular region
 *
 * @version 	1.1.6, 12/28/98
 */
public interface LabelHolder extends StringHolder {

	/** 
	 * Returns the current bounds of the label.
	 *
	 * @return	a Rectangle representing the current bounds of the label
	 */
	public abstract Rectangle getLabelBounds();
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)LineFigure.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * This interface defines protocols for Figures
 * that are essentially a line or series of line segments.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface LineFigure extends Figure {

	/**
	 * Add the locator at the specified position.
	 * @param locator the new Locator to add.
	 * @param index the index of the locator desired.
	 */
	public abstract void addLocator(int index, Locator locator);
	/**
	 * Add the locator at the end.
	 * @param locator the new Locator to add.
	 */
	public abstract void addLocator(Locator locator);
	/**
	 * Answer the indexth locator.
	 * 
	 * @param index the index of the locator desired.
	 * @return	the Locator at the desired index
	 */
	public abstract Locator getLocator(int index);
	/**
	 * Answer the number of points which define the receiver.
	 * 
	 * @return	an integer representing the number of points
	 * which define the receiver
	 */
	public abstract int getNumberOfPoints();
	/**
	 * Remove the locator at the specified position.
	 *
	 * @param index the index of the locator desired.
	 */
	public abstract void removeLocator(int index);
	/**
	 * Set the locator at the specifed position.
	 * @param locator the new Locator.
	 * @param index the index of the locator desired.
	 */
	public abstract void setLocator(int index, Locator locator);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)Locator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import com.rolemodelsoft.drawlet.util.Duplicatable;
import java.io.Serializable;

/**
 * This interface defines protocols for objects that provide
 * 2-D coordinates.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface Locator extends Duplicatable, Serializable {

	/** 
	 * Answer the radius of the Locator (as a PolarCoordinate).
	 * 
	 * @return	an integer representing the radius
	 */
	public abstract int r();
	/** 
	 * Answer the angle in radians of the Locator (as a PolarCoordinate).
	 * 
	 * @return double representing theta
	 */
	public abstract double theta();
	/** 
	 * Answer the x coordinate.
	 * 
	 * @return	an integer representing the x coordinate
	 */
	public abstract int x();
	/** 
	 * Answer the y coordinate.
	 * 
	 * @return	an integer representing the y coordinate
	 */
	public abstract int y();
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)MovableLocator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
/**
 * This interface defines protocols for objects that provide
 * 2-D coordinates which can be changed/manipulated
 *
 * @version 	1.1.6, 12/28/98
 */
 
public interface MovableLocator extends Locator {

	/** 
	 * Moves the receiver to the x and y coordinates
	 * 
	 * @param x the new x coordinate
	 * @param y the new x coordinate
	 */
	public abstract void move(int x, int y);
	/** 
	 * Moves the receiver in the x and y direction.
	 * 
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	public abstract void translate(int x, int y);
	/** 
	 * Set the x coordinate.
	 * 
	 * @param x its new desired x coordinate.
	 */
	public abstract void x(int x);
	/** 
	 * Set the y coordinate.
	 * 
	 * @param y its new desired y coordinate.
	 */
	public abstract void y(int y);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)Paintable.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;

/**
 * @version 	1.1.6, 12/28/98
 */

 public interface Paintable {
	/** 
	 * Returns the bottommost coordinate of the receiver.
	 *
	 * @return an integer representing the bottommost coordinate of the receiver
	 */
	public abstract int getBottom();
	/** 
	 * Returns the current bounds of the receiver.
	 * 
	 * @return	a Rectangle representing the current bounds
	 */
	public abstract Rectangle getBounds();
	/** 
	 * Returns the height of the receiver.
	 *
	 * @return an integer representing the height of the receiver
	 */
	public abstract int getHeight();
	/** 
	 * Returns the leftmost coordinate of the receiver.
	 *
	 * @return an integer representing the leftmost coordinate of the receiver
	 */
	public abstract int getLeft();
	/** 
	 * Returns the rightmost coordinate of the receiver.
	 *
	 * @return an integer representing the rightmost coordinate of the receiver
	 */
	public abstract int getRight();
	/** 
	 * Returns the current size of the receiver.
	 * @see #setSize
	 *
	 * @return a Dimension representing the current size of the receiver
	 */
	public abstract Dimension getSize();
	/** 
	 * Returns the topmost coordinate of the receiver.
	 *
	 * @return an integer representing the topmost coordinate of the receiver
	 */
	public abstract int getTop();
	/** 
	 * Returns the width of the receiver.
	 *
	 * @return an integer representing the width of the receiver
	 */
	public abstract int getWidth();
	/** 
	 * Paints the receiver.
	 * 
	 * @param g the Graphics object to use for painting
	 */
	public abstract void paint(Graphics g);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)PolygonFigure.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * This interface defines protocols for Figures
 * whose boundaries are defined by a polygon.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface PolygonFigure extends Figure {

	/**  
	 * Answer a Polygon associated with the figure
	 * 
	 * @return	the Polygon associated with the figure
	 */
	public abstract Polygon getPolygon();
	/**  
	 * Set the Polygon associated with the figure
	 * 
	 * @param polygon the Polygon
	 */
	public abstract void setPolygon(Polygon polygon);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)PropertyChangeRevealer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.beans.*;

/**
 * A utility class to allow tests to access property change events.
 *
 * @version 	1.1.6, 12/30/98
 */

public class PropertyChangeRevealer implements PropertyChangeListener {
	/**
	 * the most recent property change event generated by the subject
	 */
	PropertyChangeEvent propertyChangeEvent = null;

	int eventCount = 0;
	/**
	 * Set the event count to 0.
	 */
	public void clearEventCount()  {
		eventCount = 0;
	}
	/**
	 * Set the event count to 0.
	 */
	public int getEventCount()  {
		return eventCount;
	}
	/**
	 * @return the PropertyChangeEvent last heard.
	 */
	public PropertyChangeEvent getPropertyChangeEvent() {
		return propertyChangeEvent;
	}
	/**
	 * Catch PropertyChangeEvent's.
	 *
	 * evt the event
	 */
	public void propertyChange(PropertyChangeEvent evt)  {
		eventCount++;
		propertyChangeEvent = evt ;
	}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)RelatedLocationListener.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.beans.PropertyChangeEvent;

/**
 * This interface defines the methods that an object desiring to relate itself to the location
 * of another object (through listening to that object) should answer.
 *
 * @version 	1.1.6, 12/28/98
 */

 public interface RelatedLocationListener {
	/**
	 * Called by subjects when their location changes.
	 *
	 * @param event PropertyChangeEvent
	 */
	void locationChanged(PropertyChangeEvent event);
	/**
	 * Called by subjects when their relationship to the receiver has changed.
	 *
	 * @param event PropertyChangeEvent
	 */
	void relationChanged(PropertyChangeEvent event);
	/**
	 * Called by subjects when their shape changes.
	 *
	 * @param event PropertyChangeEvent
	 */
	void shapeChanged(PropertyChangeEvent event);
	/**
	 * Called by subjects when their size changes.
	 *
	 * @param event PropertyChangeEvent
	 */
	void sizeChanged(PropertyChangeEvent event);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)PropertyChangeRevealer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.beans.*;
import com.rolemodelsoft.drawlet.*;

/**
 * A utility class to allow tests to access property change events.
 *
 * @version 	1.1.6, 12/30/98
 */

public class RelatedLocationRevealer implements RelatedLocationListener {
	/**
	 * the most recent property change event generated by the subject
	 */
	PropertyChangeEvent relatedLocationEvent = null;

	int eventCount = 0;
/**
 * Set the event count to 0.
 */
public void clearEventCount() {
	eventCount = 0;
}
/**
 * Get the event count.
 *
 * @return an integer representing the number of events that have occured since
 * the receiver was created or cleared.
 */
public int getEventCount() {
	return eventCount;
}
/**
 * Called when an object the receiver is registered with as a RelatedLocationListener
 * changes its location.
 * 
 * @param evt the event describing the change in location, represented by a
 * PropertyChangeEvent.
 */
public void locationChanged( PropertyChangeEvent evt ) {
	eventCount++;
	relatedLocationEvent = evt;
}
/**
 * Called when an object the receiver is registered with as a RelatedLocationListener
 * changes its relationship.
 * 
 * @param evt the event describing the change in relationship, represented by a
 * PropertyChangeEvent.
 */
public void relationChanged( PropertyChangeEvent evt ) {
	eventCount++;
	relatedLocationEvent = evt;
}
/**
 * Called when an object the receiver is registered with as a RelatedLocationListener
 * changes its shape.
 * 
 * @param evt the event describing the change in shape, represented by a
 * PropertyChangeEvent.
 */
public void shapeChanged( PropertyChangeEvent evt ) {
	eventCount++;
	relatedLocationEvent = evt;
}
/**
 * Called when an object the receiver is registered with as a RelatedLocationListener
 * changes its size.
 * 
 * @param evt the event describing the change in size, represented by a
 * PropertyChangeEvent.
 */
public void sizeChanged( PropertyChangeEvent evt ) {
	eventCount++;
	relatedLocationEvent = evt;
}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)RelativeLocator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
/**
 * This interface defines protocols for objects that provide
 * 2-D coordinates that are relative to some other base locator.
 *
 * @version 	1.1.6, 12/28/98
 */

public interface RelativeLocator extends Locator {

	/** 
	 * Answer the base concrete locator of the receiver.
	 * 
	 * @return	the base concrete Locator of the receiver
	 */
	public abstract Locator getBase();
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)SequenceOfFigures.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;
import java.io.Serializable;

/**
 * A generic SequenceOfFigures from which Figures can be inserted, moved, and/or deleted.
 *
 * @version 	1.1.6, 12/28/98
 */
public interface SequenceOfFigures extends Serializable {

	/** 
	 * Add the figure to the receiver.
	 * 
	 * @param figure the figure to add
	 */
	public abstract void addFigure(Figure figure);
	/** 
	 * Add the figure to the receiver, sticking it behind 
	 * an existingFigure which is already there.
	 * 
	 * @param figure the figure to add
	 * @param existingFigure the figure to which the new figure should be behind
	 */
	public abstract void addFigureBehind(Figure figure, Figure existingFigure);
	/** 
	 * Answer the figure at a given point
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	the Figure at the given point
	 */
	public abstract Figure figureAt(int x, int y);
	/** 
	 * Answer a FigureEnumeration over the figures of the receiver.
	 * 
	 * @return	a FigureEnumeration over the figures of the receiver
	 */
	public abstract FigureEnumeration figures();
	/** 
	 * Move the figure behind an existingFigure if it is not already there.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to move
	 * @param existingFigure the figure to which the new figure should be behind
	 */
	public abstract void moveFigureBehind(Figure figure, Figure existingFigure);
	/** 
	 * Move the figure in front of an existingFigure if it is not already there.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to move
	 * @param existingFigure the figure to which the new figure should be in front
	 */
	public abstract void moveFigureInFront(Figure figure, Figure existingFigure);
	/** 
	 * Move the figure behind all other figures.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to add
	 */
	public abstract void moveFigureToBack(Figure figure);
	/** 
	 * Move the figure in front of all other figures.
	 * Reflect the change if it's visible.
	 * 
	 * @param figure the figure to add
	 */
	public abstract void moveFigureToFront(Figure figure);
	/** 
	 * Answer the figure at a given point excluding the identified figure
	 * 
	 * @param figure the figure to exclude from the search
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	the Figure at the given point excluding the identified Figure
	 */
	public abstract Figure otherFigureAt(Figure excludedFigure, int x, int y);
	/** 
	 * Remove the figure.
	 * Implementors may choose to tell the figure to dispose of itself completely...
	 * It is recommended that this occur unless implementors have a reason for using
	 * a different strategy.
	 * 
	 * @param figure the figure to remove
	 */
	public abstract void removeFigure(Figure figure);
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)AbstractRectangleShape.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;

/**
 * This provides a basic abstract implementation of rectangular bounded Figures.
 * Although this is an abstract class, it is
 * acknowledged that there are other implementations (e.g. ones that 
 * uses Locators) which are more flexible.  Therefore, one may wish to consider
 * subclassing the superclass instead.
 *
 * This provides most of its functionality based on a simple x, y, width, height attributes,
 * and forces concrete subclasses to define, at a minimum:
 *	paintStrokes(Graphics);
 *	paintFilled(Graphics);
 *
 */
public abstract class AbstractRectangleShape extends FilledShape {
	/**
	 * The x coordinate for the shape
	 */
	protected int x = defaultX();

	/**
	 * The y coordinate for the shape
	 */
	protected int y = defaultY();

	/**
	 * The width of the shape
	 */
	protected int width = defaultWidth();
	
	
	/**
	 * The height of the shape
	 */
	protected int height = defaultHeight();
/**
 * Creates a new, default AbstractRectangleShape.
 */
public AbstractRectangleShape() {
}
/**
 * Creates a new AbstractRectangeShape initialized with the given values
 * for x, y, width and height.
 *
 * @param x an integer representing the x for this AbstractRectangeShape.
 * @param y an integer representing the y for this AbstractRectangeShape.
 * @param width an integer representing the width for this AbstractRectangeShape.
 * @param height an integer representing the height for this AbstractRectangeShape.
 */
public AbstractRectangleShape(int x, int y, int width, int height) {
	this.x = x;
	this.y = y;
	this.width = width;
	this.height = height;
}
/**
 * Creates a new AbstractRectangeShape initialized with the given Rectangle.
 *
 * @param rectangle a Rectangle representing the bounds to be used for initialization.
 */
public AbstractRectangleShape(Rectangle rectangle) {
	this(rectangle.x,rectangle.y,rectangle.width,rectangle.height);
}
	/** 
	 * Moves the receiver to a new location. The x and y coordinates
	 * are in the parent's coordinate space.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 */
	protected synchronized void basicMove(int x, int y)  {
		this.x = x;
		this.y = y;
	}
	/** 
	 * Reshapes the receiver to the specified bounding box.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	protected synchronized void basicReshape(int x, int y, int width, int height)  {
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
	}
	/** 
	 * Resizes the receiver to the specified width and height.
	 * 
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	protected synchronized void basicResize(int width, int height)  {
		this.width = width;
		this.height = height;
	}
	/** 
	 * Moves the receiver in the x and y direction.
	 * 
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	protected synchronized void basicTranslate(int x, int y) {
		this.x += x;
		this.y += y;
	}
	/**
	 * Denote that the receiver has changed. Assume that
	 * the shape has changed even if the bounds haven't.
	 * 
	 * @param oldBounds the old bounds.
	 */
	protected void changedShape(Rectangle oldBounds) {
		if (oldBounds != null && oldBounds.equals(getBounds())) {
		    return;
		}
		super.changedShape( oldBounds );
	}
	/**
	 * Answer the default/initial value for height.
	 * 
	 * @return	an integer representing the default/initial value for height.
	 */
	protected int defaultHeight() {
		return 10;
	}
	/**
	 * Answer the default/initial value for width.
	 * 
	 * @return	an integer representing the default/initial value for width.
	 */
	protected int defaultWidth() {
		return 10;
	}
	/**
	 * Answer the default/initial value for x.
	 * 
	 * @return	an integer representing the default/initial value for x.
	 */
	protected int defaultX() {
		return 0;
	}
	/**
	 * Answer the default/initial value for y.
	 * 
	 * @return	an integer representing the default/initial value for y.
	 */
	protected int defaultY() {
		return 0;
	}
	/** 
	 * Returns the current bounds of the receiver.
	 * 
	 * @return	a Rectangle representing the current bounds of the receiver.
	 */
	public Rectangle getBounds()  {
		return new Rectangle(x,y,width,height);
	}
	/** 
	 * Returns the current size of the receiver.
	 * 
	 * @return	a Dimension representing the current size of the receiver.
	 */
	public Dimension getSize()  {
		return new Dimension(width,height);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)AbstractShape.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.beans.PropertyChangeEvent;

/**
 * This provides basic default functionality for Figures that are assumed to be
 * movable and reshapable with observers that want to know when their locations
 * or shapes change.  It provides most of its functionality based on its bounds(),
 * and forces concrete subclasses to define, at a minimum:
 *	paint(Graphics);
 *	getBounds();
 *	basicTranslate(int,int);
 *	basicReshape(int,int,int,int);
 *
 * @version 	1.1.6, 12/29/98
 */
 
public abstract class AbstractShape extends AbstractFigure {

	/** 
	 * Reshapes the receiver to the specified bounding box.
	 * Subclasses should probably provide synchronized versions if they're 
	 * modifying attributes of the receiver.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	protected abstract void basicReshape(int x, int y, int width, int height);
	/**
	 * Resizes the receiver to the specified width and height.
	 *
	 * @param width the new width.
	 * @param height the new height.
	 * @see #basicReshape
	 */
	protected void basicResize(int width, int height)  {
		basicReshape(getLeft(),getTop(),width,height);
	}
	/** 
	 * Answer the handles associated with the receiver.
	 * A better way to do this would be with a Strategy... maybe next release.
	 * 
	 * @return	an array containing the Handles associated with the receiver
	 */
	public Handle[] getHandles() {
		Handle handles[] = new Handle[8];
		handles[0] = new TopLeftHandle(this);
		handles[1] = new TopRightHandle(this);
		handles[2] = new BottomRightHandle(this);
		handles[3] = new BottomLeftHandle(this);
		handles[4] = new TopHandle(this);
		handles[5] = new RightHandle(this);
		handles[6] = new BottomHandle(this);
		handles[7] = new LeftHandle(this);
		return handles;
	}
	/**
	 * Flush caches with respect to determining bounds.  This is a hook method.
	 * Subclasses may wish to override.
	 */
	protected void resetBoundsCache() {
		resetLocationCache();
		resetSizeCache();
	}
	/**
	 * Flush caches with respect to determining size.  This is a hook method.
	 * Subclasses may wish to override.
	 */
	protected void resetSizeCache() {
	}
	/** 
	 * Answer a new version of the given polygon reshaped to the specified 
	 * bounding box.
	 * This is a useful utility.
	 *
	 * @param polygon the polygon
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 * @return	a Polygon which is a new version of the given polygon reshaped to
	 * the specified bounding box
	 */
	public static Polygon reshapedPolygon(Polygon polygon, int x, int y, int width, int height) {
		int npoints = polygon.npoints;
		int xpoints[] = new int[npoints], ypoints[] = new int[npoints];
		Rectangle polyBounds = polygon.getBounds();
		double xScale = (double)width / (double)polyBounds.width;
		double yScale = (double)height / (double)polyBounds.height;
		for (int i=0; i < npoints; i++) {
			xpoints[i] = x + (int)((polygon.xpoints[i] - polyBounds.x) * xScale);
			ypoints[i] = y + (int)((polygon.ypoints[i] - polyBounds.y) * yScale);
		}
		return new Polygon(xpoints, ypoints, npoints);
	}
	/** 
	 * Answer a new version of the given polygon reshaped to the specified 
	 * bounding box.
	 * This is a useful utility.
	 *
	 * @param polygon the polygon
	 * @param bounds its new bounding box
	 * @return	a Polygon which is a new version of the given polygon reshaped to
	 * the specified bounding box
	 */
	public static Polygon reshapedPolygon(Polygon polygon, Rectangle bounds) {
		return reshapedPolygon(polygon, bounds.x, bounds.y, bounds.width, bounds.height);
	}
	/** 
	 * Reshapes the receiver to the specified bounding box.
	 * Let observers know what changed.
	 * This is a TemplateMethod with hooks:
	 * 	resetBoundsCache(),
	 * 	basicReshape(),
	 *  changedShape()
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	public void setBounds(int x, int y, int width, int height) {
		Rectangle oldBounds = getBounds();
		basicReshape(x,y,width,height);
		resetBoundsCache();
		changedShape(oldBounds);
	}
	/**
	 * Resizes the receiver to the specified width and height.
	 * Let observers know what changed.
	 * This is a TemplateMethod with hooks:
	 * 	resetSizeCache(),
	 * 	basicResize(),
	 *  changedSize()
	 *
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	public void setSize(int width, int height)  {
		Dimension oldSize = getSize();
		basicResize(width,height);
		resetSizeCache();
		changedSize(oldSize);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)BottomHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the bottom of a figure
 * and resizes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */

public class BottomHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its bottom.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public BottomHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,0.5,1.0);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Resize the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();
		resize(bounds.width, Math.max(getY(evt) - bounds.y, 1));
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)BottomLeftHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the bottom left of a figure
 * and reshapes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class BottomLeftHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its bottom-left.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public BottomLeftHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,0.0,1.0);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Reshape the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();
		int legitX = Math.min(getX(evt), figure.getRight() - 1);
		reshape(legitX, bounds.y, bounds.width - (legitX - bounds.x), Math.max(getY(evt) - bounds.y, 1));
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)BottomRightHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the bottom right of a figure
 * and resizes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class BottomRightHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its bottom-right.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public BottomRightHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,1.0,1.0);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Resize the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();
		resize(Math.max(getX(evt) - bounds.x, 1), Math.max(getY(evt) - bounds.y, 1));
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)BoundsHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.*;

/**
 * Although there are plenty of ways to add Handles to figures, a common thing to
 * do is put them on the corners or sides of some rectangular area (e.g. the bounds).
 * This abstract class serves as a base for handles that do just that.
 * Subclasses need to implement, at a minimum:
 * 	Their own constructors, modeled after this one's
 *	defaultLocator(Figure);
 * and should probably define a mouseDrag(Event,int,int) method to invoke the
 * provided resize(int,int) or reshape(int,int,int,int) method if the handle
 * is there to allow reshaping of the underlying figure.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public abstract class BoundsHandle extends SquareCanvasHandle implements FigureHolder {

	/**
	 * The figure whose bounds the handle may modify.
	 */
	protected Figure figure;

	/**
	 * The locator defining where to place the handle.
	 */
	protected Locator locator;

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds in some way.  Subclasses should use a similar
	 * constructor, invoking super.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public BoundsHandle(Figure figure) {
		this.setFigure(figure);
	}
	/**  
	 * Answer the x coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the x coordinate at the center of the handle
	 */
	public int centerX() {
		return locator.x();
	}
	/**  
	 * Answer the y coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the y coordinate at the center of the handle
	 */
	public int centerY() {
		return locator.y();
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected abstract Locator defaultLocator(Figure figure);
	/** 
	 * Returns the figure associated with this handle.
	 * 
	 * @return	the Figure associated with this handle
	 */
	public Figure getFigure()  {
		return figure;
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Resize/reshape the figure as appropriate.  Consume the event.
	 * Subclasses should provide their own behavior to resize/reshape the figure.
	 *
	 * @param evt the event
	 * @see #resize
	 * @see #reshape
	 */
	public void mouseDragged(MouseEvent evt) {
		evt.consume();
	}
	/**
	 * Reshape the figure and cleanly repaint the canvas.
	 * 
	 * @param x the new x.
	 * @param y the new y.
	 * @param width the new width.
	 * @param height the new height.
	 */
	public void reshape(int x, int y, int width, int height)  {
		Rectangle bounds = figure.getBounds();
		figure.setBounds(x, y, width, height);
		bounds = bounds.union(figure.getBounds());
		bounds.grow(halfWidth, halfWidth);
		canvas.repaint(bounds);
	}
	/**
	 * Resize the figure and cleanly repaint the canvas.
	 * Note that we're assuming that at least some of the other handles must be repainted.
	 *
	 * @param width the new width.
	 * @param height the new height.
	 */
	public void resize(int width, int height)  {
		Rectangle bounds = figure.getBounds();
		figure.setSize(width, height);
		bounds = bounds.union(figure.getBounds());
		bounds.grow(halfWidth, halfWidth);
		canvas.repaint(bounds);
	}
	/** 
	 * Set the figure associated with this handle.  Reset the locator.
	 *
	 * @param figure the Figure to hold.
	 */
	public void setFigure(Figure figure)  {
		this.figure = figure;
		locator = defaultLocator(figure);
	}
}
package com.rolemodelsoft.drawlet.shapes.ellipses;

/**
 * @(#)Ellipse.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;

/**
 * This provides a basic concrete implementation of Ellipses that are 
 * assumed to be movable and reshapable with observers that want to know when 
 * their locations or shapes change.
 * NOTE: intersection algorithms are currently incomplete and only determine if
 * the Rectangular bounds intersect.
 */

 public class Ellipse extends AbstractRectangleShape {
	static final long serialVersionUID = 6345684636936094954L;
/**
 * Creates a new, default Ellipse.
 */
public Ellipse() {
}
/**
 * Creates a new Ellipse initialized to the given values.
 * 
 * @param x an integer representing the x coordinate for the new Ellipse.
 * @param y an integer representing the y coordinate for the new Ellipse.
 * @param width an integer representing the width for the new Ellipse.
 * @param height an integer representing the height for the new Ellipse.
 */
public Ellipse(int x, int y, int width, int height) {
	this.x = x;
	this.y = y;
	this.width = width;
	this.height = height;
}
/**
 * Creates a new Ellipse initialized to the given value.
 * 
 * @param rectangle a Rectangle representing the bounds for the new Ellipse.
 */
public Ellipse(Rectangle rectangle) {
	this(rectangle.x,rectangle.y,rectangle.width,rectangle.height);
}
	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * Figure. <code>x</code> and <code>y</code> are defined to be relative to the 
	 * coordinate system of this figure.  This is not guaranteed to be 100% accurate around the edges
	 * due to rounding errors, but shouldn't be off by more than a single pixel.
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified x,y
	 * location is "inside" this Figure;
	 * 			<code>false</code> otherwise.
	 * @see #isWithin
	 */
	public boolean contains(int x, int y) {
		if ( ! super.contains( x, y ) )
			return false;
		/*
		 * using standard geometric formula for ellipse
		 *   x^2   y^2
		 *   --- + --- = 1
		 *   a^2   b^2
		 * where a and b are x and y coordinates where y=0 and x=0 respectively...
		 * multiplying by a^2*b^2 and using <= to determine points on or inside yields 
		 *   b^2*x^2 + a^2*y^2 <= a^2*b^2
		 */
		int a = ( getWidth() + 1 ) / 2;  // be conservative in finding points by rounding up
		int b = ( getHeight() + 1 ) / 2;
		int aSquared = a * a;
		int bSquared = b * b;
		int centerX = getLeft() + a;
		int normalizedX = x - centerX;
		int centerY = getTop() + b;
		int normalizedY = y - centerY;
		return ( ( normalizedX * normalizedX ) * bSquared ) + ( ( normalizedY * normalizedY ) * aSquared ) <= aSquared * bSquared;
	}
	/**  
	 * Checks whether a specified Rectangle is "inside" this Figure, 
	 * where the Rectangle and this Figure are in the same coordinate system  
	 * In addition to checking topLeft and bottomRight, check topRight and bottomLeft.
	 * If all four corners are inside, everything is inside.
	 *
	 * @param box the rectangle to test for inclusion
	 * @return	boolean value of <code>true</code> if the specified Rectangle
	 * is "inside" this Figure;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(Rectangle box) {
		return super.contains(box) &&
			contains(box.x + box.width,box.y) && contains(box.x, box.y + box.height)
			&& contains(box.x, box.y) && contains(box.x + box.width, box.y + box.height);
	}
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 * By default, just check if the bounds intersects.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * specified Rectangular area;
	 * 			<code>false</code> otherwise.
	 * @see #bounds
	 */
	public boolean intersects(Rectangle box) {
		// if bounding boxes don't intersect, figures don't intersect
		if (! super.intersects(box))
			return false;
			
		// if the ellipse contains one of the corners of the box, they intersect
		if (contains(box.x, box.y))
			return true;
		if (contains(box.x + box.width, box.y))
			return true;
		if (contains(box.x, box.y + box.height))
			return true;
		if (contains(box.x + box.width, box.y + box.height))
			return true;
			
		// if the box contains any of the four edges of the ellipse, they intersect
		Rectangle bigBox = new Rectangle(box.x, box.y, box.width + 1, box.height + 1);
		if (bigBox.contains(x + (width/2), y))
			return true;
		if (bigBox.contains(x, y + (height/2)))
			return true;
		if (bigBox.contains(x + (width/2), y + height))
			return true;
		if (bigBox.contains(x + width, y + (height/2)))
			return true;
			
		// otherwise they don't intersect
		return false;
	}
	/**
	 * Paint the receiver, filling all the contained area.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paintFilled(Graphics g)  {
		super.paintFilled(g);
		// fudge by 1 since filling primitives are different
		// than drawing primitive
		g.fillOval(x,y,width+1,height+1);
	}
	/**
	 * Paint the outline of the receiver.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paintStrokes(Graphics g)  {
		super.paintStrokes(g);
		g.drawOval(x,y,width,height);
	}
}
package com.rolemodelsoft.drawlet.shapes.ellipses;

/**
 * @(#)EllipseTool.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This tool produces Ellipses.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class EllipseTool extends ShapeTool {
	/** 
	 * Constructs and initializes a new instance of a tool to create 
	 * Ellipses on a DrawingCanvas
	 *
	 * @param canvas the canvas on which to place Ellipses.
	 */
	public EllipseTool(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
   /**
	 * Create and answer a new Ellipse at the given coordinates.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a newly created Ellipse
	 */
	protected Figure basicNewFigure(int x, int y)  {
		Figure newFigure = new Ellipse();
		newFigure.move(x,y);
		return newFigure;
	}
	/**
	 * Called if the mouse is dragged (when the mouse button is down).
	 *
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		Rectangle bounds = figure.getBounds();

		int figureX = Math.min(anchorX, getX(e));
		int figureY = Math.min(anchorY, getY(e));
		int figureWidth = Math.abs(anchorX - getX(e));
		int figureHeight = Math.abs(anchorY - getY(e));

		figure.setBounds( figureX, figureY, figureWidth, figureHeight );
		bounds = bounds.union(figure.getBounds());
		canvas.repaint(bounds);
		e.consume();
	}
}
package com.rolemodelsoft.drawlet.shapes.ellipses;

/**
 * @(#)TC_Ellipse.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.ellipses.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import java.awt.*;
import junit.framework.*;
import junit.ui.*;

public class TC_Ellipse extends TC_AbstractFigure {
	/**
	 */
	public TC_Ellipse(String name) {
		super(name);
	}
	/**
	 * Sets up the fixture, for example, open a network connection.
	 * This method is called before a test is executed.
	 */
	protected void setUp() 
	{
		figure = new Ellipse(5,5,10,10);
		propertyRevealer = new PropertyChangeRevealer();
		locationRevealer = new RelatedLocationRevealer();
	}
/**
 * Test to make sure the contains( Figure ) method is
 * functioning properly.
 */
public void testContainsFigure() 
{
	Figure testFigure = new RectangleShape( 9, 9, 2, 2 );
	assert( "Figure does not contain the figure", figure.contains( testFigure ) );
	testFigure.setBounds( 6, 6, 13, 13 );
	assert( "Figure contains the figure", ! figure.contains( testFigure ) );
}
/**
 * Test to make sure the contains( int, int ) method is
 * functioning properly.
 */
public void testContainsIntInt() 
{
	assert( "Figure does not contain the point 10, 10", figure.contains( 10, 10 ) );
	assert( "Figure contains the point 20, 20", ! figure.contains( 20, 20 ) );

	// check the corners
	assert( "Figure contains the point 7, 7", ! figure.contains( 5, 6 ) );
	
	assert( "Figure contains the point 13, 7", ! figure.contains( 15, 6 ) );
	
	assert( "Figure contains the point 13, 13", ! figure.contains( 15, 14 ) );
	
	assert( "Figure contains the point 7, 13", ! figure.contains( 5, 14 ) );

	assert( "Figure does not contain the point 6, 6", ! figure.contains( 6, 6 ) );

	assert( "Figure contains the point 8, 8", figure.contains( 8, 8) );

	assert( "Figure does contain the point 7, 7", figure.contains( 7, 7) );
}
	/**
	 * Test to make sure the contains( Rectangle ) method is
	 * functioning properly.
	 */
	public void testContainsRectangle() 
	{
		Rectangle testRect = new Rectangle( 9, 9, 2, 2 );
		assert( "Figure does not contain the rectangle", figure.contains( testRect ) );
		testRect.setBounds( 7, 7, 13, 13 );
		assert( "Figure contains the rectangle", ! figure.contains( testRect ) );
		testRect.setBounds( 6, 10, 5, 4 );
		assert( "Figure contains the rectangle", ! figure.contains( testRect ) );
	}
/**
 * Test to make sure the intersects( Figure ) method is
 * functioning properly.
 */
public void testIntersectsFigure() 
{
	Figure testFigure = new RectangleShape( 6, 6, 2, 2 );
	assert( "Ellipse does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 2, 2, 4, 4 );
	assert( "Ellipse does intersect the figure", ! figure.intersects( testFigure ) );
	testFigure.setBounds( 16, 16, 6, 6 );
	assert( "Ellipse intersects the figure", ! figure.intersects( testFigure ) );
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectsRectangle() 
{
	Rectangle testRect = new Rectangle( 8, 8, 2, 2 );
	assert( "Figure should intersect the rectangle", figure.intersects( testRect ) );

	// check rectangles near each of the four corners that would intersect if it weren't for rounded corners
	testRect.setBounds( 4, 4, 2, 2 );
	assert( "Figure should not intersect the rectangle (though bounds overlaps)", !figure.intersects( testRect ) );
	testRect.setBounds( 14, 14, 6, 6 );
	assert( "Figure should not intersect the rectangle", ! figure.intersects( testRect ) );
	testRect.setBounds( 14, 0, 6, 6 );
	assert( "Figure should not intersect the rectangle", ! figure.intersects( testRect ) );
	testRect.setBounds( 0, 14, 6, 6 );
	assert( "Figure should not intersect the rectangle", ! figure.intersects( testRect ) );

	testRect.setBounds( 2, 2, 10, 10 );
	assert( "Figure should intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 16, 16, 6, 6 );
	assert( "Figure should not intersect the rectangle", ! figure.intersects( testRect ) );
	testRect.setBounds( 6, 6, 3, 3);
	assert( "Figure should intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 5, 5, 10, 10);
	assert( "Ellipse should intersect its bounding rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 5, 5, 11, 11);
	assert( "Ellipse should intersect its bounding rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 4, 4, 12, 12);
	assert( "Ellipse should intersect its bounding rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 5, 5, 9, 9);
	assert( "Ellipse should intersect its bounding rectangle", figure.intersects( testRect ) );
}
}
package com.rolemodelsoft.drawlet.shapes.ellipses;

/**
 * @(#)TS_EllipseTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

public class TS_EllipseTests extends junit.framework.TestCase {

public TS_EllipseTests(String name) {
	super(name);
}

public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}

public static TestSuite suite() {
	TestSuite suite = new TestSuite();
	
	suite.addTest(new TestSuite(TC_Ellipse.class));
	
	return suite;
}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)FilledShape.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.Graphics;
import java.awt.Color;
import java.awt.Rectangle;

/**
 * This provides basic default functionality for Figures that draw shapes which
 * may be opaque.  It provides most of its functionality based on 
 * its getBounds(), and forces concrete subclasses to define, at a minimum:
 *	paintStrokes(Graphics);
 *	paintFilled(Graphics);
 *	getBounds();
 *	basicTranslate(int,int);
 *	basicReshape(int,int,int,int);
 *
 * @version 	1.1.6, 12/29/98
 */
 
public abstract class FilledShape extends AbstractShape {

	/**
	 * The line color for drawing the figure is cached here.
	 */
	protected Color lineColor = defaultLineColor();

	/**
	 * The fill color for drawing the figure is cached here.
	 */
	protected Color fillColor = defaultFillColor();

	/**
	 * Reshapes the receiver to the specified bounding box.
	 * Subclasses should probably provide synchronized versions if they're 
	 * modifying attributes of the receiver.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	protected void basicReshape(int x, int y, int width, int height)  {
	}
	/**
	 * Moves the receiver in the x and y direction.
	 * 
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	protected void basicTranslate(int x, int y)  {
	}
	/**
	 * Answer the default/initial value for fillColor
	 * 
	 * @return	the default/initial value for fillColor
	 */
	protected Color defaultFillColor() {
		return Color.gray;
	}
	/**
	 * Answer the default/initial value for lineColor
	 * 
	 * @return	the default/initial value for lineColor
	 */
	protected Color defaultLineColor() {
		return Color.black;
	}
	/** 
	 * Returns the current bounds of the receiver.
	 * 
	 * @return	a Rectangle representing the current bounds of the receiver.
	 */
	public abstract Rectangle getBounds();
	/**
	 * Answer the Color to use when filling the shape.
	 * 
	 * @return	the Color to use when filling the shape
	 */
	public Color getFillColor() {
		return fillColor;
	}
	/**
	 * Answer the Color to use when drawing lines.
	 * 
	 * @return	the Color to use when drawing lines
	 */
	public Color getLineColor() {
		return lineColor;
	}
	/** 
	 * Answer the DrawingStyle which defines how to paint the figure.
	 * 
	 * @return	the DrawingStyle which defines how to paint the figure.
	 */
	public DrawingStyle getStyle()  {
		DrawingStyle style = super.getStyle();
		style.setLineColor(getLineColor());
		style.setFillColor(getFillColor());
		return style;
	}
	/**
	 * Answer whether the receiver fills in its inside 
	 * (obscuring things underneath) or not.
	 * 
	 * @return	boolean value of <code>true</code> if the receiver
	 * fills in its inside;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isOpaque() {
		return fillColor != null;
	}
	/**
	 * Answer whether the receiver draws its lines
	 * (obscuring things underneath) or not.
	 * 
	 * @return boolean value of <code>true</code> if the receiver draws its lines;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isStroked() {
		return (getLineColor() != null);
	}
	/** 
	 * Paints the figure.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		if (isOpaque()) paintFilled(g);
		if (isStroked()) paintStrokes(g);
	}
	/**
	 * Paint the shape, filling all contained area
	 *
	 * @param g the specified Graphics window
	 */
	public void paintFilled(Graphics g)  {
		g.setColor(getFillColor());
		// fill figure
	}
	/**
	 * Paint the outline of the shape
	 *
	 * @param g the specified Graphics window
	 */
	public void paintStrokes(Graphics g)  {
		g.setColor(getLineColor());
		// draw figure
	}
	/**
	 * Set the Color to use when filling the shape.
	 * 
	 * @param color the color
	 */
	public void setFillColor(Color color) {
		Color oldColor = fillColor;
		fillColor = color;
		firePropertyChange(FILL_COLOR_PROPERTY,oldColor,color);
	}
	/**
	 * Set the Color to use when drawing lines.
	 * 
	 * @param color the color
	 */
	public void setLineColor(Color color) {
		Color oldColor = lineColor;
		lineColor = color;
		firePropertyChange(LINE_COLOR_PROPERTY, oldColor, color);
	}
	/** 
	 * Set the DrawingStyle defining how to paint the figure.
	 * 
	 * @param style the specified DrawingStyle.
	 */
	public void setStyle(DrawingStyle style) {
		DrawingStyle oldStyle = getStyle();
		if (style != null) {
			setLineColor(style.getLineColor());
			setFillColor(style.getFillColor());
		}
		firePropertyChange(STYLE_PROPERTY, oldStyle, style);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)LeftHandle.java	1.0 97/02/25 Ken Auer
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the left of a figure
 * and reshapes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class LeftHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its left.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public LeftHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,0.0,0.5);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Reshape the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();
		int legitX = Math.min(getX(evt), figure.getRight() - 1);
		reshape(legitX, bounds.y, bounds.width - (legitX - bounds.x), bounds.height);
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)AdornedLine.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.util.*;
import java.beans.PropertyChangeEvent;

/**
 * This provides a basic implementation of Lines that can have adornments.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class AdornedLine extends ConnectingLine {
	static final long serialVersionUID = -5836590406891850934L;

	/**
	 * The adornments associated with this AdornedLine.
	 */
	protected Vector adornments = new Vector();

	/**
	 * The figure the adornment rests next to.
	 */
	protected Figure figure;
/**
 * Creates a new AdornedLine initialized with the given values.
 *
 * @param beginX an integer representing the x coordinate the line should start at.
 * @param beginY an integer representing the y coordinate the line should start at.
 * @param endX an integer representing the x coordinate the line should end at.
 * @param endY an integer representing the y coordinate the line should end at.
 */
public AdornedLine(int beginX, int beginY, int endX, int endY) {
	super(beginX, beginY, endX, endY);
}
	/**
	 * Creates a new AdornedLine initialized with the given values.
	 *
	 * @param beginX an integer representing the x coordinate the line should start at.
	 * @param beginY an integer representing the y coordinate the line should start at.
	 * @param endX an integer representing the x coordinate the line should end at.
	 * @param endY an integer representing the y coordinate the line should end at.
	 * @param mustConnect specifies whether this line has to be connected in order to exist.
	 */
	public AdornedLine(int beginX, int beginY, int endX, int endY, boolean mustConnect) {
		super(beginX, beginY, endX, endY, mustConnect);
	}
	/**
	 * Creates a new AdornedLine initialized with the given values.
	 *
	 * @param begin a Locator representing the location the line should start at.
	 * @param end a Locator representing the location the line should end at.
	 */
	public AdornedLine(Locator begin, Locator end) {
		super(begin, end);
	}
	/**
	 * Creates a new AdornedLine initialized with the given values.
	 *
	 * @param begin a Locator representing the location the line should start at.
	 * @param end a Locator representing the location the line should end at.
	 * @param mustConnect specifies whether this line has to be connected in order to exist.
	 */
	public AdornedLine( Locator begin, Locator end, boolean mustConnect ) {
		super(begin, end, mustConnect);
	}
	/**
	 * Adds the given figure to the line as an adornment.
	 *
	 * @param adornment the Figure to add as an adornment.
	 */
	public void addAdornment( Figure adornment) {
		if ( ! adornments.contains( adornment ) ) {
			adornments.addElement(adornment);
		}
	}
	/** 
	 * Answers an enumeration of the adornments.
	 * 
	 * @return a FigureEnumeration over the adornments.
	 */
	protected FigureEnumeration adornments()  {
		return new FigureVectorEnumerator(adornments);
	}
	/** 
	 * Reshapes the receiver to the specified bounding box.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	protected synchronized void basicReshape(int x, int y, int width, int height)  {
		int npoints = points.length;
		int xpoints[] = new int[npoints], ypoints[] = new int[npoints];
		Rectangle bounds = getLineBounds();
		double xScale = (double)width / (double)bounds.width;
		double yScale = (double)height / (double)bounds.height;
		for (int i=0; i < npoints; i++) {
			if (points[i] instanceof MovableLocator) {
				((MovableLocator)points[i]).x(x + (int)((points[i].x() - bounds.x) * xScale));
				((MovableLocator)points[i]).y(y + (int)((points[i].y() - bounds.y) * yScale));
			}
		}
	}
	/**
	 * Set the locator at the appropriate position.
	 * 
	 * @param index the index of the locator desired.
	 * @param locator the new Locator.
	 */
	protected synchronized void basicSetLocator(int index, Locator locator)  {
		Figure figure = figureFromLocator(points[index]);
		if (figure != null) 
			figure.removeRelatedLocationListener(this);
		points[index] = locator;
		figure = figureFromLocator(locator);
		if (figure != null) {
			figure.addRelatedLocationListener(this);
			if (index == points.length - 1)
				points[index] = new EdgeLocator(locator, points[points.length - 2]);
		}
	}
	/**  
	 * Checks whether a specified x,y location is "inside" the
	 * receiver, where x and y are defined to be relative to the 
	 * coordinate system of the receiver.
	 * Checks the adornments as well.
	 * 
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified x,y location is
	 * inside this Figure;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(int x, int y)  {
		for (int i=1; i < points.length; i++) {
			if (insideSegment(x,y,points[i-1].x(),points[i-1].y(),points[i].x(),points[i].y())) 
				return true;
		}
		FigureEnumeration adornmentsE = adornments();
		while ( adornmentsE.hasMoreElements() ) {
			if ( adornmentsE.nextElement().contains( x, y ) ) {
				return true;
			}
		}
		return false;
	}
	/** 
	 * Returns the current bounds of the receiver.
	 * Adds in the adornments.
	 * 
	 * @return	a Rectangle representing the current bounds of the receiver.
	 */
	public Rectangle getBounds()  {
		Rectangle bounds = getLineBounds(); // make a copy, otherwise we'll mess with the polygon's bounds attribute
		FigureEnumeration adornmentsE = adornments();
		while ( adornmentsE.hasMoreElements() ) {
			bounds.add( adornmentsE.nextElement().getBounds() );
		}
		return bounds;
	}
	/** 
	 * Returns the current bounds of the line only (excludes adornments).
	 * 
	 * @return	a Rectangle representing the current bounds of the line.
	 */
	protected Rectangle getLineBounds()  {
		return new Rectangle(getPolygon().getBounds()); // make a copy, otherwise we'll mess with the polygon's bounds attribute
	}
	/** 
	 * Answer a point indicating the location of the receiver... typically the topLeft.
	 * NOTE: This may not correspond to the point indicated by getLocator() as this method
	 * is often used to determine the position before a Locator has already been affected.
	 * 
	 * @return	a Point indicating the location of the receiver
	 */
	protected Point getLocation() {
		Rectangle myBounds = getLineBounds();
		return new java.awt.Point(myBounds.x,myBounds.y);
	}
	/** 
	 * Paints the receiver.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		super.paint(g);
		paintAdornments(g);
	}
	/** 
	 * Paints the adornments of the receiver.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paintAdornments(Graphics g)  {
		FigureEnumeration adornmentsE = adornments();
		while ( adornmentsE.hasMoreElements() ) {
			adornmentsE.nextElement().paint(g);
		}
	}
	/**
	 * Remove the specified adornment.
	 *
	 * @param adornment the adornment to remove.
	 */
	public void removeAdornment( Figure adornment) {
		adornments.removeElement(adornment);
	}
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the other figure.
	 * 
	 * @param requestor the figure requesting the connection
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 * @return	a Locator corresponding to a significant point on the receiver
	 */
	public Locator requestConnection(Figure requestor, int x, int y) {
		// make sure we aren't already connected to the locator 
		// which is trying to connect to us 
		if (isListening(requestor))
			return null;
		return locatorAt(x,y);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)AdornedLineCreationHandle.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;

/**
 * This class provides a handle that creates adorned lines from one figure
 * to another.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class AdornedLineCreationHandle extends ConnectedLineCreationHandle {
	/**
	 * Creates a new AdornedLineCreationHandle initialized with the given Figure.
	 *
	 * @param figure the figure to which the handle is attached.
	 */
	public AdornedLineCreationHandle(Figure figure) {
		super(figure);
	}
	/**
	 * AdornedLineCreationHandle constructor comment.
	 *
	 * @param figure the figure to which the handle is attached.
	 * @param locator the locator at which to locate the handle and beginning of any lines.
	 */
	public AdornedLineCreationHandle(Figure figure, Locator locator) {
		super(figure, locator);
	}
	/**  
	 * Return a ConnectingLine of the proper type.
	 * 
	 * @return	a ConnectingLine of the proper type.
	 */
	protected ConnectingLine basicNewLine( Locator point )  {
		AdornedLine aLine = new AdornedLine( (Locator) locator.duplicate(), point, true );
		aLine.addAdornment( new Arrow( aLine ) );
		return aLine;
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)AdornedLineTool.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;

/**
 * This tool produces AdornedLines
 * @version 	1.1.6, 03/22/99
 */

public class AdornedLineTool extends ConnectingLineTool {
/**
 * Creates a new AdornedLineTool initialized with the given DrawingCanvas.
 *
 * @param canvas the DrawingCanvas to create AdornedLines on.
 */
public AdornedLineTool(DrawingCanvas canvas) {
	super(canvas);
}
	/**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 */
	protected Figure basicNewFigure(int x, int y)  {
		AdornedLine line = new AdornedLine(x,y,x,y);
		line.addAdornment( new Arrow( line ) );
		return line;
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)Arrow.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.beans.*;
import java.awt.*;

/**
 * Arrows are a basic adornment for AdornedLines.
 *
 * @version 	1.1.6, 03/22/99
 */

public class Arrow extends AbstractShape implements RelatedLocationListener {
	static final long serialVersionUID = 7418562820276513057L;

	/**
	 * The LineFigure we are attached to.
	 */
	protected LineFigure line;

	/**
	 * Defines which end of the line we are attached to.
	 */
	public static final int FORWARD = -1;

	/**
	 * Defines which end of the line we are attached to.
	 */
	public static final int REVERSE = 1;
	
	/**
	 * The direction of the arrow
	 */
	protected int direction = defaultDirection();
	
	/**
	 * The polygon defining the arrow head
	 */
	transient protected Polygon polygon;

	/**
	 * The ArrowStyle which defines how to draw this Arrow.
	 */
	protected ArrowStyle arrowStyle = defaultArrowStyle();

	protected PropertyChangeEvent lastEvent;
	/**
	 * Construct a new arrow initialized with the given LineFigure.
	 *
	 * @param line the LineFigure to attach to.
	 */
	public Arrow( LineFigure line ) {
		this.line = line;
		line.addRelatedLocationListener( this );
	}
	/**
	 * Construct a new arrow initialized with the given LineFigure and direction.
	 *
	 * @param line the LineFigure to attach to.
	 * @param the direction for the Arrow.
	 */
	public Arrow( LineFigure line, int direction ) {
		this.line = line;
		line.addRelatedLocationListener( this );
		this.direction = direction;
	}
/** 
 * Reshapes the Arrow to the specified bounding box. Does nothing by default.
 *
 * @param x the x coordinate
 * @param y the y coordinate
 * @param width the width of the figure
 * @param height the height of the figure
 * @see #getBounds
 */
protected synchronized void basicReshape(int x, int y, int width, int height) {
}
	/**
	 * Shifts the receiver by the specified values.
	 * Don't think this is legal if we are attached to a line.
	 *
	 * @param x the amount to shift horizontally.
	 * @param y the amount to shift vertically.
	 */
	protected void basicTranslate(int x, int y) {
		// don't think this is legal if we are attached to a line
		polygon.translate(x, y);
	}
	/**
	 * Answer the default shape for the arrow.
	 *
	 * @return a Polygon representing the default shape for the arrow.
	 */
	protected ArrowStyle defaultArrowStyle() {
		return new ArrowStyle();
	}
	/**
	 * Answer the default direction for the arrow.
	 *
	 * @return an integer representing the default direction for the arrow.
	 */
	protected int defaultDirection() {
		return FORWARD;
	}
	/**
	 * Answer the default shape for the arrow.
	 *
	 * @return a Polygon representing the default shape for the arrow.
	 */
	protected Polygon defaultPolygon() {
		Locator[] locs = getArrowStyle().getLocators();
		int xArray[] = new int[locs.length];
		int yArray[] = new int[locs.length];
		for ( int i = 0; i < locs.length; i++ ) {
			xArray[i] = getArrowLocator( locs[i] ).x();
			yArray[i] = getArrowLocator( locs[i] ).y();
		}
		return new Polygon(
			xArray,
			yArray,
			locs.length);
	}
	/**
	 * Answer the proper Locator for the given values.
	 *
	 * @param source the first Locator to calculate from.
	 * @param destination the second Locator to calculate from.
	 * @param offset the amount the Locator should be offset.
	 * @return a Locator calculated from the given values.
	 */
	protected Locator getArrowLocator( Locator loc ) {
		Locator dest = getDestinationLocator();
		Locator source = getSourceLocator();
		Locator relative = new DrawingPoint(dest.x()-source.x(), dest.y()-source.y());
		PolarCoordinate coord = new PolarCoordinate( loc.r(), loc.theta() + relative.theta() );
		return new DrawingPoint( dest.x() + coord.x(), dest.y() + coord.y() );
	}
	/**
	 * Answer the ArrowStyle that defines how to draw the arrow.
	 *
	 * @return the ArrowStyle defining how to draw the arrow.
	 */
	public ArrowStyle getArrowStyle() {
		return arrowStyle;
	}
	/**
	 * Answer the bounds of the receiver.
	 *
	 * @return a Rectangle representing the receiver's bounds.
	 */
	public Rectangle getBounds() {
		return getPolygon().getBounds();
	}
	/**
	 * Answers the index of the destination, depending on the direction of the receiver.
	 *
	 * @return an integer representing the index of the destination.
	 */
	protected int getDestinationIndex() {
		if (getDirection() == FORWARD)
			return line.getNumberOfPoints() - 1;
		else
			return 0;
	}
	/**
	 * Answers the Locator of the destination, depending on the direction of the receiver.
	 *
	 * @return the Locator of the destination.
	 */
	protected Locator getDestinationLocator() {
		return line.getLocator(getDestinationIndex());
	}
	/**
	 * Answers the direction of the receiver.
	 *
	 * @return the integer representing the direction (+1 or -1).
	 */
	protected int getDirection() {
		return direction;
	}
	/** 
	 * Answer the handles associated with the receiver.
	 * A better way to do this would be with a Strategy... maybe next release.
	 * 
	 * @return	an array containing the Handles associated with the receiver.
	 */
	public Handle[] getHandles() {
		return new Handle[0];
	}
	/**
	 * Answer the polygon that defines the arrow.
	 *
	 * @return the Polygon defining the arrow.
	 */
	protected Polygon getPolygon() {
		if (polygon == null)
			polygon = defaultPolygon();
		return polygon;
	}
	/**
	 * Answers the index of the source, depending on the direction of the receiver.
	 *
	 * @return an integer representing the index of the source.
	 */
	protected int getSourceIndex() {
		return getDestinationIndex() + getDirection();
	}
	/**
	 * Answers the Locator of the source, depending on the direction of the receiver.
	 *
	 * @return the Locator of the source.
	 */
	protected Locator getSourceLocator() {
		return line.getLocator(getSourceIndex());
	}
	/**
	 * Update because the location of my line has changed.
	 *
	 * @evt the event.
	 */
	public void locationChanged( PropertyChangeEvent evt ) {
		updateShape();
	}
	/**
	 * Paints the Arrow.
	 */
	public void paint(java.awt.Graphics g) {
		if ( arrowStyle.isOpaque() )
			g.fillPolygon(getPolygon());
		else
			g.drawPolygon(getPolygon());
	}
	/**
	 * Update because the relation of my line has changed.
	 *
	 * @evt the event.
	 */
	public void relationChanged( PropertyChangeEvent event ) {
		updateShape();
	}
	/**
	 * Flush caches with respect to determining location.  This is a hook method.
	 * Subclasses may wish to override.
	 */
	protected void resetLocationCache() {
		polygon = null;
	}
	/**
	 * Update because the shape of my line has changed.
	 *
	 * @evt the event.
	 */
	public void shapeChanged( PropertyChangeEvent evt ) {
		lastEvent = evt;
		updateShape();
		lastEvent = null;
	}
	/**
	 * Update because the size of my line has changed.
	 *
	 * @evt the event.
	 */
	public void sizeChanged( PropertyChangeEvent evt ) {
		updateShape();
	}
	/**
	 * The line has notified the receiver of a change.
	 * Assume our location has changed due to some movement/reshaping
	 * of the line.
	 */
	protected void updateShape() {
		Rectangle oldShape = getBounds();
		resetBoundsCache();
		changedShape(oldShape);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)ArrowStyle.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.util.*;
import java.awt.*;

/**
 * Defines how an arrow will be drawn.
 * @version 	1.1.6, 03/22/99
 */

public class ArrowStyle extends SimpleDrawingStyle {
	/**
	 * A boolean designating whether this arrow is opaque or not.
	 */
	protected boolean isOpaque = defaultIsOpaque();
	/**
	 * The locators which define the arrow head
	 */
	protected Locator[] locators;

	/**
	 * The default locators for ArrowStyles.
	 */
	protected static Locator[] defaultLocators;
/**
 * 
 */
public ArrowStyle() {
	this(getDefaultLocators());
}
/**
 * 
 */
public ArrowStyle(Locator[] locators) {
	this.locators = locators;
}
/**
 * 
 */
public ArrowStyle(Locator[] locators, boolean isOpaque) {
	this.locators = locators;
	this.isOpaque = isOpaque;
}
	/**
	 * getFillColor method comment.
	 */
	protected boolean defaultIsOpaque() {
		return true;
	}
/**
 * 
 */
protected static Locator[] getDefaultLocators() {
	if ( defaultLocators == null ) {
		defaultLocators = new Locator[] {
			new PolarCoordinate( 10, 0.5 + Math.PI ),
			new PolarCoordinate( 0, 0.0 ),
			new PolarCoordinate( 10, -0.5 + Math.PI )
		};
	}
	return defaultLocators;
}
/**
 * getLocators method comment.
 */
public Locator[] getLocators() {
	return locators;
}
/**
 * isFilled method comment.
 */
public boolean isOpaque() {
	return isOpaque;
}
/**
 * 
 */
public static void setDefaultLocators( Locator[] locators ) {
	defaultLocators = locators;
}
/**
 * setFilled method comment.
 */
public void setOpaque(boolean opaque) {
	this.isOpaque = opaque;
}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)ConnectedLineCreationHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that creates connected lines from one figure
 * to another.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class ConnectedLineCreationHandle extends SquareCanvasHandle implements FigureHolder {

	/**
	 * The figure to which the handle is attached.
	 */
	protected Figure figure;

	/**
	 * The location at which the handle is displayed which will also
	 * serve as a prototype of the starting point of any line.
	 * This will typically be a FigureRelativePoint, but not necessarily.
	 */
	protected Locator locator;

	/**
	 * The line being constructed.
	 */
	protected ConnectingLine line;

	/**
	 * Construct and initialize a handle that will create connecting lines
	 * from a figure to some other figure.
	 *
	 * @param figure the figure to which the handle is attached.
	 */
	public ConnectedLineCreationHandle(Figure figure) {
		this.setFigure(figure);
	}
	/**
	 * Construct and initialize a handle that will create connecting lines
	 * from a figure to some other figure.
	 *
	 * @param figure the figure to which the handle is attached.
	 * @param locator the locator at which to locate the handle and beginning of any lines.
	 */
	public ConnectedLineCreationHandle(Figure figure, Locator locator) {
		this(figure);
		this.locator = locator;
	}
	/**  
	 * Answers a new ConnectingLine. This is a template method.  
	 *
	 * @param point the point at which the new line should start.
	 * @return	a new ConnectingLine.
	 */
	protected ConnectingLine basicNewLine( Locator point )  {
		return new ConnectingLine( (Locator) locator.duplicate(), point, true );
	}
	/**  
	 * Answer the x coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the x coordinate at the center
	 * of the handle
	 */
	protected int centerX()  {
		return locator.x();
	}
	/**  
	 * Answer the y coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the y coordinate at the center
	 * of the handle
	 */
	protected int centerY()  {
		return locator.y();
	}
	/** 
	 * Answer the default/initial locator.
	 *
	 * @param figure the Figure the Locator is to be relative to.
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,0.5,0.5);
	}
	/** 
	 * Returns the figure associated with this handle.
	 * 
	 * @return	the Figure assocatiated with this handle
	 */
	public Figure getFigure()  {
		return figure;
	}
	/**
	 * Called if the mouse is double-clicked.
	 * 
	 * @param evt the event 
	 */
	protected void mouseDoubleClicked(MouseEvent evt) {
		//just absorb it
		evt.consume();
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * If in the midst of constructing a line, move its latest point.
	 *
	 * @param evt the event
	 * @see #movePoint
	 */
	public void mouseDragged(MouseEvent evt) {
		if (line == null)
			return;
		movePoint(line.getNumberOfPoints() - 1, getX(evt), getY(evt));
		evt.consume();
	}
	/**
	 * Called if the mouse is moved (the mouse button is up).
	 * If in the midst of constructing a line, move its latest point.
	 *
	 * @param evt the event
	 * @see #movePoint
	 */
	public void mouseMoved(MouseEvent evt) {
		if (line == null)
			return;
		movePoint(line.getNumberOfPoints() - 1, getX(evt), getY(evt));
		evt.consume();
	}
	/**
	 * Called when the mouse goes down.
	 * If we're not already in the midst of constructing a line, start one.
	 *
	 * @param evt the event 
	 */
	public void mousePressed(MouseEvent evt) {
		if (line == null) {
			line = basicNewLine( new DrawingPoint(getX(evt), getY(evt)));
			canvas.addFigure(line);
		}
		evt.consume();
	}
	/**
	 * Called if the mouse goes up.
	 * Connect the latest point to any underlying figure if possible.
	 * If the shift key is down, add a point and keep constructing.
	 * If not, we're done.
	 *
	 * @param evt the event
	 */
	public void mouseReleased(MouseEvent evt) {
		if (line == null)
			return;
		int x = getX(evt);
		int y = getY(evt);
		Figure target = canvas.otherFigureAt(line, x, y);
		if (!evt.isShiftDown() && (target == null || target == figure))
			canvas.removeFigure(line);
		else {
			Locator newLocator = target.requestConnection(figure, x, y);
			if (newLocator != null) {
				line.setLocator(line.getNumberOfPoints() - 1, newLocator);
				canvas.moveFigureBehind(line, figure);
				canvas.moveFigureBehind(line, target);
			}
		}
		if (!evt.isShiftDown())
			super.mouseReleased(evt);
		else
			line.addLocator(new DrawingPoint(x, y));
		evt.consume();
	}
	/**
	 * Move the identified point of the line.  
	 * Assumes that pointIndex is a valid point in the current line being edited.
	 * 
	 * @param pointIndex the index of the point to move.
	 * @param x the x coordinate
	 * @param y the y coordinate
	 */
	protected void movePoint(int pointIndex, int x, int y) {
		Rectangle bounds = line.getBounds();
		line.setLocator(pointIndex,new DrawingPoint(x,y));
		bounds = bounds.union(line.getBounds());
		canvas.repaint(bounds);
	}
	/** 
	 * Paints the handle.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		g.fillRect(centerX() - halfWidth, centerY() - halfWidth, 2, 2*halfWidth);
		g.drawLine(centerX() - halfWidth, centerY(), centerX() + halfWidth, centerY());
		g.fillRect(centerX() + halfWidth - 1, centerY() - halfWidth, 2, 2*halfWidth);
	}
	/**
	 * Release control of the canvas and clean up if necessary.
	 * Since this is a public method,
	 * don't assume the receiver actually has control.
	 * Since we're obviously done constructing the line, see if it's valid
	 * If not, delete it.
	 * Prepare to create a new one next time we get control.
	 *
	 * @param canvas the canvas which the receiver is to release control of.
	 */
	public void releaseControl(DrawingCanvas canvas) {
	if (line != null) {
		if (line.isObsolete()) 
			canvas.removeFigure(line);
		line = null;
	}
	super.releaseControl(canvas);
	}
	/** 
	 * Set the figure associated with this handle.  Reset the locator.
	 *
	 * @param figure the Figure to associate with.
	 */
	public void setFigure(Figure figure)  {
		this.figure = figure;
		locator = defaultLocator(figure);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)ConnectingLine.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.beans.PropertyChangeEvent;

/**
 * This provides a basic implementation of Lines that can connect to other figures.
 * Note that there can be any number of Locators that make up a Line and
 * that those Locators may be tied to Figures.  When they are, the Line
 * will become a dependent of the Figure and will assume it is moved when
 * the Figure is moved/reshaped in any way.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class ConnectingLine extends Line implements RelatedLocationListener {
	static final long serialVersionUID = -5097190759387410089L;

	/**
	 * A boolean defining whether this line must be connected.
	 */
	protected boolean mustConnect = defaultConnectState();
	/** 
	 * Constructs and initializes a new instance of a connecting line.
	 *
	 * @param beginX the x coordinate for the first point defining the line
	 * @param beginY the y coordinate for the first point defining the line
	 * @param endX the x coordinate for the second point defining the line
	 * @param endY the y coordinate for the second point defining the line
	 */
	public ConnectingLine(int beginX, int beginY, int endX, int endY) {
		super(beginX, beginY, endX, endY);
	}
	/** 
	 * Constructs and initializes a new instance of a connecting line.
	 *
	 * @param beginX the x coordinate for the first point defining the line
	 * @param beginY the y coordinate for the first point defining the line
	 * @param endX the x coordinate for the second point defining the line
	 * @param endY the y coordinate for the second point defining the line
	 * @param mustConnect specifies whether this line has to be connected in order to exist
	 */
	public ConnectingLine(int beginX, int beginY, int endX, int endY, boolean mustConnect) {
		super(beginX, beginY, endX, endY);
		this.mustConnect = mustConnect;
	}
	/** 
	 * Constructs and initializes a new instance of a connecting line.
	 *
	 * @param begin the locator which is the first point defining the line
	 * @param end the locator which is the second point defining the line
	 */
	public ConnectingLine(Locator begin, Locator end) {
		super(begin, end);
	}
	/** 
	 * Constructs and initializes a new instance of a connecting line.
	 *
	 * @param begin the locator which is the first point defining the line
	 * @param end the locator which is the second point defining the line
	 * @param mustConnect specifies whether this line has to be connected in order to exist
	 */
	public ConnectingLine( Locator begin, Locator end, boolean mustConnect ) {
		super(begin, end);
		this.mustConnect = mustConnect;
	}
	/**
	 * Set the locator at the appropriate position.
	 * 
	 * @param index the index of the locator desired.
	 * @param locator the new Locator.
	 */
	protected synchronized void basicSetLocator(int index, Locator locator)  {
		Figure figure = figureFromLocator(points[index]);
		if (figure != null) 
			figure.removeRelatedLocationListener(this);
		points[index] = locator;
		figure = figureFromLocator(locator);
		if (figure != null) 
			figure.addRelatedLocationListener(this);
	}
	/**
	 * Answer the default state for connectedness.
	 *
	 * @return a boolean representing whether this line must be connected to
	 * exist.
	 */
	protected boolean defaultConnectState() {
		return false;
	}
	/** 
	 * Clean up as appropriate if the receiver is no longer valid.
	 */
	public void disconnect() {
		for (int i = 0; i < points.length; i++) {
			Figure figure = figureFromLocator(points[i]);
			if (figure != null)
				figure.removeRelatedLocationListener(this);
		}
		super.disconnect();
	}
	/**
	 * Remove any dependence on the figure.
	 * 
	 * @param figure the Figure to disassociate from.
	 */
	protected synchronized void freeFromFigure(Figure figure)  {
		for (int i = 0; i < points.length; i++) {
			if (figureFromLocator(points[i]) == figure) 
				points[i] = new DrawingPoint(points[i].x(),points[i].y());
		}
	}
	/** 
	 * Answer the handles associated with the receiver.
	 * 
	 * @return	an array of the Handles associated with the receiver
	 */
	public Handle[] getHandles() {
		int numberOfHandles = getNumberOfPoints();
		Handle handles[] = new Handle[numberOfHandles];
		for (int i=0; i < numberOfHandles; i++)
			handles[i] = new ConnectingLinePointHandle(this,i);
		return handles;
	}
	/** 
	 * Answers whether the receiver is listening to the figure directly 
	 * or indirectly (via chain of listeners)
	 * 
	 * @param figure the Figure to test
	 * @return	boolean value of <code>true</code> if the receiver is listening
	 * to the figure directly or indirectly;
	 * 			<code>false</code> otherwise.
	 */
	protected boolean isListening(Figure figure) {
		for (Enumeration e = figure.relatedLocationListeners(); e.hasMoreElements();) {
			RelatedLocationListener listener = (RelatedLocationListener) e.nextElement();
			if (listener == this)
				return true;
			else
				if (listener instanceof Figure)
					if (isListening((Figure) listener))
						return true;
		}
		return false;
	}
	/** 
	 * Answers whether the receiver is obsolete.
	 * 
	 * @return	boolean value of <code>true</code> if both ends
	 * are not connected to a figure;
	 * 			<code>false</code> otherwise.
	 */
	public boolean isObsolete() {
		return mustConnect ? ((figureFromLocator(getLocator(0)) == null) || (figureFromLocator(getLocator(getNumberOfPoints() - 1)) == null)) : false;
	}
	/**
	 * Update because the location of something we depend on has changed.
	 *
	 * @evt the event.
	 */
	public void locationChanged(PropertyChangeEvent event) {
		updateShape();
	}
	/**
	 * Update because the relationship of something we depend on has changed.
	 *
	 * @evt the event.
	 */
	public void relationChanged(PropertyChangeEvent event) {
		freeFromFigure((Figure)event.getSource());
	}
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the other figure.
	 * 
	 * @param requestor the figure requesting the connection
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 * @return	a Locator corresponding to a significant point on the receiver
	 */
	public Locator requestConnection(Figure requestor, int x, int y) {
		// make sure we aren't already connected to the locator 
		// which is trying to connect to us 
		if (isListening(requestor))
			return null;
		return locatorAt(x,y);
	}
	/**
	 * Update because the shape of something we depend on has changed.
	 *
	 * @evt the event.
	 */
	public void shapeChanged(PropertyChangeEvent event) {
		updateShape();
	}
	/**
	 * Update because the size of something we depend on has changed.
	 *
	 * @evt the event.
	 */
	public void sizeChanged(PropertyChangeEvent event) {
		updateShape();
	}
	/**
	 * Assume our location has changed due to some movement/reshaping
	 * of something we depend on.
	 */
	protected void updateShape() {
		Rectangle oldBounds = getBounds();
		resetBoundsCache();
		changedShape(oldBounds);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)ConnectingLinePointHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at a particular Locator on a line
 * and reshapes the figure accordingly as it is dragged, potentially turning
 * the specific locator into one relative to another figure.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class ConnectingLinePointHandle extends LinePointHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the the position of a particular point defining a line and
	 * connect that point to a figure if appropriate.
	 *
	 * @param figure the figure whose point (vertex) we may wish to modify.
	 * @param index the identifying index of the point of interest.
	 */
	public ConnectingLinePointHandle(LineFigure figure, int index) {
		super(figure,index);
	}
	/** 
	 * Answer whether the handle's locator is connected to a figure (or about
	 * to request it).
	 * 
	 * @return	boolean value of <code>true</code> if the handle's locator is
	 * connected to a figure (or about to request it);
	 * 			<code>false</code> otherwise.
	 */
	protected boolean isConnected()  {
		Locator locator = figure.getLocator(index);
		return (locator instanceof FigureHolder) || (canvas != null && canvas.otherFigureAt(figure, locator.x(), locator.y()) != null);
	}
	/**
	 * Called if the mouse goes up.
	 * If there is a figure at the given point, connect to it if possible before
	 * giving up control.
	 *
	 * @param evt the event
	 */
	public void mouseReleased(MouseEvent evt) {
		int x = getX(evt);
		int y = getY(evt);
		Figure target = canvas.otherFigureAt(figure, x, y);
		if (target != null) {
			Locator newLocator = target.requestConnection(figure, x, y);
			if (newLocator != null) {
				Rectangle bounds = figure.getBounds();
				figure.setLocator(index, newLocator);
				canvas.moveFigureBehind(figure, target);
				bounds = bounds.union(figure.getBounds());
				bounds.grow(halfWidth, halfWidth);
				canvas.repaint(bounds);
			}
		}
		super.mouseReleased(evt);
	}
	/** 
	 * Paints the handle.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		if (isConnected())
			g.fillRect(centerX() - halfWidth, centerY() - halfWidth, 2*halfWidth, 2*halfWidth);
		else
			g.drawRect(centerX() - halfWidth, centerY() - halfWidth, 2*halfWidth, 2*halfWidth);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)ConnectingLineTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.event.*;
import java.awt.*;

/**
 * This tool produces ConnectingLines.
 * @version 	1.1.6, 12/29/98
 */

public class ConnectingLineTool extends LineTool {

	/** 
	 * Constructs and initializes a new instance of a tool to create connecting 
	 * lines on a DrawingCanvas.
	 *
	 * @param canvas the canvas on which to place lines.
	 */
	public ConnectingLineTool(DrawingCanvas canvas) {
		super(canvas);
	}
	/**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 */
	protected Figure basicNewFigure(int x, int y)  {
		return new ConnectingLine(x,y,x,y);
	}
	/**
	 * Called if the mouse goes down.
	 * When creating the new line, see if the first point is on top of another
	 * figure, and connect it if possible.
	 *
	 * @param e the event 
	 */
	public void mousePressed(MouseEvent e) {
		super.mousePressed(e);
		// see if the initial point was on top of a figure
		LineFigure myShape = (LineFigure) figure;
		if (myShape.getNumberOfPoints() != 2)
			return;
		int x = e.getX();
		int y = e.getY();
		Figure target = canvas.otherFigureAt(figure, x, y);
		if (target != null) {
			Locator newLocator = target.requestConnection(myShape, x, y);
			if (newLocator != null) {
				myShape.setLocator(0, newLocator);
				canvas.moveFigureBehind(myShape, target);
				canvas.repaint(myShape.getBounds());
			}
		}
	}
	/**
	 * Called if the mouse is released.
	 * If we are constructing a line, finalize the
	 * latest point by seeing if it is one that could connect to an underlying
	 * figure and, if so connect it.  We're done constructing this 
	 * one unless the shift key is down in which case we add another point
	 * and keep going.
	 * Otherwise, get ready to construct the next one.
	 *
	 * @param e the event 
	 */
	public void mouseReleased(MouseEvent e) {
		if (figure == null)
			return;
		LineFigure myShape = (LineFigure) figure;
		Rectangle bounds = myShape.getBounds();
		int x = getX(e);
		int y = getY(e);
		Figure target = canvas.otherFigureAt(myShape, x, y);
		if (target != null) {
			Locator newLocator = target.requestConnection(myShape, x, y);
			if (newLocator != null) {
				myShape.setLocator(myShape.getNumberOfPoints() - 1, newLocator);
				canvas.moveFigureBehind(myShape, target);
				bounds = bounds.union(myShape.getBounds());
			}
		}

		Locator nextLast, last;
		LineFigure line = (LineFigure)figure;
		nextLast = line.getLocator( line.getNumberOfPoints() - 2 );
		last = line.getLocator( line.getNumberOfPoints() - 1 );
		if ( last.x() == nextLast.x() && last.y() == nextLast.y() ) {
			if ( line.getNumberOfPoints() == 2 )
				canvas.removeFigure( figure );
			else
				return;
		}
		
		if (!e.isShiftDown()) {
			figure = null;
			canvas.toolTaskCompleted(this);
		} else {
			myShape.addLocator(new DrawingPoint(x, y));
			bounds = bounds.union(myShape.getBounds());
		}
		canvas.repaint(bounds);
		e.consume();
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/*
 * @(#)DrawLineTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This tool produces lines.
 * Currently, one must hold the shift key down in order to add more than
 * a single segment.
 *
 * @version 	1.1.6, 12/29/98
 */
public class DrawLineTool extends ShapeTool {
	/** 
	 * Constructs and initializes a new instance of a tool to create lines
	 * on a DrawingCanvas
	 *
	 * @param canvas the canvas on which to place lines.
	 */
	public DrawLineTool(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
   /**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a newly created Figure
	 */
	protected Figure basicNewFigure(int x, int y)  {
		return new Line(x,y,x,y);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down and mouse is moving).  
	 * If we are constructing a line, move its last locator as indicated.
	 *
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		LineFigure myShape = (LineFigure) figure;
		Rectangle bounds = myShape.getBounds();
		myShape.addLocator(canvas.getLocator( e.getX(), e.getY() ));
		bounds = bounds.union(myShape.getBounds());
		canvas.repaint(bounds);
		e.consume();
	}
	/**
	 * Called if the mouse moves (when the mouse button is up).
	 * If we are constructing a line, move its last locator as indicated.
	 * When done, consume the event.
	 *
	 * @param e the event
	 */
	public void mouseMoved(MouseEvent e) {
		if (figure == null)
			return;
		movePoint(((LineFigure) figure).getNumberOfPoints() - 1, getX(e), getY(e));
		e.consume();
	}
	/**
	 * Called if the mouse is released.
	 * If we are constructing a line, and the shift key is down, add another
	 * point.  Otherwise, we're done constructing this one... get ready for the
	 * next one.
	 *
	 * @param e the event 
	 */
	public void mouseReleased(MouseEvent e) {
		if (figure == null)
			return;
		LineFigure myShape = (LineFigure) figure;
		Rectangle bounds = myShape.getBounds();

		Locator nextLast, last;
		LineFigure line = (LineFigure)figure;
		nextLast = line.getLocator( line.getNumberOfPoints() - 2 );
		last = line.getLocator( line.getNumberOfPoints() - 1 );
		if ( last.x() == nextLast.x() && last.y() == nextLast.y() ) {
			if ( line.getNumberOfPoints() == 2 )
				canvas.removeFigure( figure );
			else
				return;
		}
		
		if (!e.isShiftDown()) {
			figure = null;
			canvas.toolTaskCompleted(this);
		} else {
			myShape.addLocator(new DrawingPoint(getX(e), getY(e)));
			bounds = bounds.union(myShape.getBounds());
		}
		canvas.repaint(bounds);
		e.consume();
	}
	/**
	 * Move the identified point of the line.  
	 * Assumes that pointIndex is a valid point in the current line being edited.
	 * 
	 * @param pointIndex the index of the point to move.
	 * @param x the x coordinate.
	 * @param y the y coordinate.
	 */
	protected void movePoint(int pointIndex, int x, int y) {
		LineFigure myShape = (LineFigure)figure;
		Rectangle bounds = myShape.getBounds();
		myShape.setLocator(pointIndex,new DrawingPoint(x,y));
		bounds = bounds.union(myShape.getBounds());
		canvas.repaint(bounds);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

import java.beans.*;
/**
 * @(#)Line.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import com.rolemodelsoft.drawlet.util.Duplicatable;
import java.awt.*;
import java.util.Hashtable;

/**
 * This provides a basic implementation of Lines.
 * It is not expected that the locators that make up this line be connected to
 * other figures.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class Line extends LinearShape {
	static final long serialVersionUID = 4513748053006007765L;
	
	/**
	 * The locators which define the points of the line.
	 */
	protected Locator points[] = new Locator[0];

	/**
	 * A cache to keep the bounds for efficient reference.
	 */
	transient protected Polygon polygon;

	/** 
	 * Constructs and initializes a new instance of a line.
	 *
	 * @param locators the locators which define the points of the line
	 * @exception IllegalArgumentException If their are not at least 2 locators.
	 */
	public Line(Locator locators[]) {
		if (locators.length < 2)
			throw new IllegalArgumentException("Must supply at least two locators"); 
		this.points = locators;
	}
	/** 
	 * Constructs and initializes a new instance of a line.
	 *
	 * @param beginX the x coordinate for the first point defining the line
	 * @param beginY the y coordinate for the first point defining the line
	 * @param endX the x coordinate for the second point defining the line
	 * @param endY the y coordinate for the second point defining the line
	 */
	public Line(int beginX, int beginY, int endX, int endY) {
		this(new DrawingPoint(beginX,beginY),new DrawingPoint(endX,endY));
	}
	/** 
	 * Constructs and initializes a new instance of a line.
	 *
	 * @param begin the locator which is the first point defining the line
	 * @param end the locator which is the second point defining the line
	 */
	public Line(Locator begin, Locator end) {
		this.points = new Locator[2];
		this.basicSetLocator(0,begin);
		this.basicSetLocator(1,end);
	}
	/**
	 * Add the locator at the given position.
	 * 
	 * @param index the index of the locator desired.
	 * @param locator the new Locator.
	 */
	protected synchronized void basicAddLocator(int index, Locator locator)  {
		Locator newPoints[] = new Locator[points.length + 1];
		System.arraycopy(points,0,newPoints,0,index);
		System.arraycopy(points,index,newPoints,index+1, points.length - index);
		points = newPoints;
		basicSetLocator(index,locator);
	}
	/**
	 * Remove the locator at the given position.
	 * 
	 * @param index the index of the locator no longer desired.
	 */
	protected synchronized void basicRemoveLocator(int index)  {
		Locator newPoints[] = new Locator[points.length - 1];
		System.arraycopy(points,0,newPoints,0,index - 1);
		System.arraycopy(points,index + 1,newPoints,index, points.length - index);
		points = newPoints;
	}
	/** 
	 * Reshapes the receiver to the specified bounding box.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the receiver
	 * @param height the height of the receiver
	 * @see #getBounds
	 */
	protected synchronized void basicReshape(int x, int y, int width, int height)  {
		int npoints = points.length;
		int xpoints[] = new int[npoints], ypoints[] = new int[npoints];
		Rectangle bounds = getBounds();
		double xScale = (double)width / (double)bounds.width;
		double yScale = (double)height / (double)bounds.height;
		for (int i=0; i < npoints; i++) {
			if (points[i] instanceof MovableLocator) {
				((MovableLocator)points[i]).x(x + (int)((points[i].x() - bounds.x) * xScale));
				((MovableLocator)points[i]).y(y + (int)((points[i].y() - bounds.y) * yScale));
			}
		}
	}
	/**
	 * Set the locator at the given position.
	 * 
	 * @param index the index of the locator desired.
	 * @param locator the new Locator.
	 */
	protected synchronized void basicSetLocator(int index, Locator locator)  {
		points[index] = locator;
	}
	/** 
	 * Moves the receiver in the x and y direction.
	 * 
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	protected synchronized void basicTranslate(int x, int y)  {
		for (int i=0; i < points.length; i++) {
			if (points[i] instanceof MovableLocator) {
				((MovableLocator)points[i]).translate(x,y);
			}
		}
	}
	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * Figure, where x and y are defined to be relative to the 
	 * coordinate system of this figure.  
	 * 
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * 
	 * @return	boolean value of <code>true</code> if the specified x,y location is
	 * "inside this Figure;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(int x, int y)  {
		for (int i=1; i < points.length; i++) {
			if (insideSegment(x,y,points[i-1].x(),points[i-1].y(),points[i].x(),points[i].y())) 
				return true;
		}
		return false;
	}
	/**
	 * Duplicates the receiver in the given Hashtable. 
	 * 
	 * @param duplicates the Hashtable to put the new duplicate in
	 * @return	an Object which is a duplicate of the receiver
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		Line duplicate = (Line)super.duplicateIn(duplicates);
		duplicate.points = new Locator[points.length];
		for (int i=0; i < points.length; i++) 
			duplicate.points[i] = (Locator)points[i].duplicateIn(duplicates);
		duplicate.resetBoundsCache();
		return duplicate;
	}
	/**
	 * Answers the expected number of significant duplicates when duplicating
	 * the receiver.
	 * Subclasses may wish to override.
	 * 
	 * @returns an integer representing an estimate of the number of objects
	 * to be duplicated
	 */
	protected int estimatedDuplicateSize() {
		return super.estimatedDuplicateSize() + points.length;
	}
	/** 
	 * Returns the current bounds of the receiver.
	 * 
	 * @return	a Rectangle representing the current bounds of the receiver.
	 */
	public Rectangle getBounds()  {
		return new Rectangle(getPolygon().getBounds()); // make a copy, otherwise we'll mess with the polygon's bounds attribute
	}
	/**
	 * Answer the indexth locator.
	 * 
	 * @param index the index of the locator desired.
	 * @return	the Locator at the given index
	 */
	public Locator getLocator(int index)  {
		return points[index];
	}
	/**
	 * Answer the number of points which define the receiver.
	 * 
	 * @return	an integer representing the number of points which define the
	 * receiver
	 */
	public int getNumberOfPoints()  {
		return points.length;
	}
	/**  
	 * Answer the Polygon associated with the receiver.
	 * 
	 * @return	the Polygon associated with the receiver.
	 */
	protected Polygon getPolygon() {
	if (polygon == null) {
		int npoints = points.length;
		int xpoints[] = new int[npoints];
		int ypoints[] = new int[npoints];
		for (int i=0; i < npoints; i++) {
			xpoints[i] = points[i].x();
			ypoints[i] = points[i].y();
		}
		polygon = new Polygon(xpoints,ypoints,npoints);
	}
	return polygon;
	}
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 * 
	 * @param bounds the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * given rectangular area;
	 * 			<code>false</code> otherwise.
	 */
	public boolean intersects(Rectangle bounds) {
		return getBounds().intersects(bounds);
	}
	/** 
	 * Paints the receiver.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		super.paint(g);
		int npoints = points.length;
		int xpoints[] = new int[npoints];
		int ypoints[] = new int[npoints];
		for (int i=0; i < npoints; i++) {
			xpoints[i] = points[i].x();
			ypoints[i] = points[i].y();
		}
		g.drawPolyline(xpoints, ypoints, npoints);
	}
	/**
	 * After a series of Figures are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes it might like to reconcile.
	 * 
	 * In this case, remove any dependency on any figures defining original 
	 * points.  If there is an available duplicate corresponding to the original, 
	 * use it as the original was used.  If not, convert it to a non-Figure-
	 * dependent point.
	 *
	 * @param duplicates a Hashtable where originals as keys and duplicates as elements
	 */
	public void postDuplicate(Hashtable duplicates) {
		super.postDuplicate(duplicates);
		for (int i=0; i < points.length; i++) {
			Figure figure = figureFromLocator(points[i]);
			if (figure != null) {
				if (!duplicates.containsKey(figure)) {
					points[i] = new DrawingPoint(points[i].x(),points[i].y());
				}
			}
		}
		for (int i=0; i < points.length; i++) 
			points[i].postDuplicate(duplicates);
	}
	/**
	 * Flush caches with respect to determining bounds.
	 */
	protected void resetBoundsCache() {
		polygon = null;
	}
	/**
	 * Flush caches with respect to determining location.
	 */
	protected void resetLocationCache() {
		polygon = null;
	}
	/**
	 * Flush caches with respect to determining size.
	 */
	protected void resetSizeCache() {
		polygon = null;
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/*
 * @(#)LinearShape.java	
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.Color;

/**
 * This provides basic default functionality for LineFigures that are assumed 
 * to be moveable and reshapeable with observers that want to know when their 
 * locations or shapes change.  It provides most of its functionality based on 
 * its bounds, plus some additional facilities which assume the line is made
 * up of multiple locators, and forces concrete subclasses to define, 
 * at a minimum:
 *	paint(Graphics);
 *	getLocator(int);
 *	getNumberOfPoints();
 *	basicAddLocator(Locator,int);
 *	basicSetLocator(Locator,int);
 *	basicRemoveLocator(int);
 *
 * @version 	1.1.6, 12/29/98
 */
 
public abstract class LinearShape extends AbstractShape implements LineFigure {

	/**
	 * The color with which to paint the line.
	 */
	protected Color lineColor = defaultLineColor();

	/**
	 * Add the locator at the appropriate position.
	 * This is a TemplateMethod with hooks:
	 * 	resetBoundsCache()
	 * 	basicAddLocator()
	 *  changedShape()
	 *
	 * @param locator the new Locator to add.
	 * @param index the index of the locator desired.
	 */
	public void addLocator(int index, Locator locator) {
		Rectangle oldBounds = getBounds();
		if (!oldBounds.contains(locator.x(), locator.y()))
			resetBoundsCache();
		basicAddLocator(index, locator);
		changedShape(oldBounds);
	}
	/**
	 * Add the locator at the end.
	 * 
	 * @param locator the new Locator to add.
	 */
	public void addLocator(Locator locator) {
		addLocator(getNumberOfPoints(), locator);
	}
	/**
	 * Add the locator at the given position.
	 * 
	 * @param index the index to add the Locator at.
	 * @param locator the new Locator.
	 */
	protected abstract void basicAddLocator(int index, Locator locator);
	/**
	 * Remove the locator at the given position.
	 * 
	 * @param index the index of the locator no longer desired.
	 */
	protected abstract void basicRemoveLocator(int index);
	/** 
	 * Reshapes the receiver to the specified bounding box.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 * @see #getBounds
	 */
	protected synchronized void basicReshape(int x, int y, int width, int height)  {
		int npoints = getNumberOfPoints();
		Rectangle bounds = getBounds();
		double xScale = (double)width / (double)bounds.width;
		double yScale = (double)height / (double)bounds.height;
		for (int i=0; i < npoints; i++) {
			Locator locator = getLocator(i);
			if (locator instanceof MovableLocator) {
				((MovableLocator)locator).x(x + (int)((locator.x() - bounds.x) * xScale));
				((MovableLocator)locator).y(y + (int)((locator.y() - bounds.y) * yScale));
			}
		}
	}
	/**
	 * Set the locator at the given position.
	 * 
	 * @param index the index of the locator to be set.
	 * @param locator the new Locator.
	 */
	protected abstract void basicSetLocator(int index, Locator locator);
	/** 
	 * Moves the receiver in the x and y direction.
	 * 
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	protected synchronized void basicTranslate(int x, int y)  {
		int npoints = getNumberOfPoints();
		for (int i=0; i < npoints; i++) {
			Locator locator = getLocator(i);
			if (locator instanceof MovableLocator) 
				((MovableLocator)locator).translate(x,y);
		}
	}
	/**  
	 * Checks whether a specified x,y location is "inside" the
	 * receiver, where x and y are defined to be relative to the 
	 * coordinate system of the receiver.  
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified x,y
	 * location is "inside" this figure;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(int x, int y)  {
		int size = getNumberOfPoints();
		for (int i=1; i < size; i++) {
			Locator previous = getLocator(i-1);
			Locator current = getLocator(i);
			if (insideSegment(x,y,previous.x(),previous.y(),current.x(),current.y())) 
				return true;
		}
		return false;
	}
	/**
	 * Answer the default/initial value for lineColor
	 * 
	 * @return	a Color representing the default/initial value for lineColor
	 */
	protected Color defaultLineColor() {
		return Color.black;
	}
	/** 
	 * Returns the current bounds of the receiver.
	 * 
	 * @return	a Rectangle representing the current bounds of the receiver.
	 */
	public Rectangle getBounds()  {
		int npoints = getNumberOfPoints();
		Locator locator = getLocator(0);
		Rectangle bounds = new Rectangle(locator.x(),locator.y(),1,1);
		for (int i=1; i < npoints; i++) {
			locator = getLocator(i);
			bounds.add(locator.x(),locator.y());
		}
		return bounds;
	}
	/** 
	 * Answer the handles associated with the receiver.
	 * 
	 * @return	an array containing the Handles associated with the
	 * receiver
	 */
	public Handle[] getHandles() {
		int numberOfHandles = getNumberOfPoints();
		Handle handles[] = new Handle[numberOfHandles];
		for (int i=0; i < numberOfHandles; i++)
			handles[i] = new LinePointHandle(this,i);
		return handles;
	}
	/**
	 * Answer the Color to use when drawing lines.
	 * 
	 * @return	the Color to use when drawing lines
	 */
	public Color getLineColor() {
		return lineColor;
	}
	/**
	 * Answer the indexth locator.
	 * 
	 * @param index the index of the locator desired.
	 * @return	the Locator at the indexth position
	 */
	public abstract Locator getLocator(int index);
	/**
	 * Answer the number of points which define the receiver.
	 * 
	 * @return	an integer representing the number of points which
	 * define the receiver
	 */
	public abstract int getNumberOfPoints();
	/** 
	 * Answer the style which defines how to paint the figure.
	 * 
	 * @return	the DrawingStyle which defines how to paint the figure
	 */
	public DrawingStyle getStyle()  {
		DrawingStyle style = super.getStyle();
		style.setLineColor(getLineColor());
		return style;
	}
	/**  
	 * Answer whether the point specified is inside the specified line segment.
	 * 
	 * @param x the x coordinate of the point being tested.
	 * @param y the y coordinate of the point being tested.
	 * @param x0 the x coordinate of the beginning of the line segment.
	 * @param y0 the y coordinate of the beginning of the line segment.
	 * @param x1 the x coordinate of the end of the line segment.
	 * @param y2 the y coordinate of the end of the line segment.
	 * @return	boolean value of <code>true</code> if the point specified is
	 * inside the line segment specified;
	 * 			<code>false</code> otherwise.
	 */
	protected boolean insideSegment(int x, int y, int x0, int y0, int x1, int y1) {
		/*
		 * first check if we have a vertical segment and act accordingly
		 */
		if (x1 == x0) return (x == x0) && (y >= Math.min(y0,y1)) && (y <= Math.max(y0,y1));
		/*
		 * before going on, verify that x,y is inside the bounding box,
		 * if not, it can't possible be on the line segment.
		 */
		Rectangle bounds = new Rectangle(x0,y0,1,1);
		int fudge = (int)Math.round(insideTolerance());
		bounds.add(x1,y1);
		bounds.grow(fudge,fudge);
		if (!bounds.contains(x,y)) return false;
		/*
		 * Now check the basic "y (or f(x)) = mx + b" definition of the line.
		 * Since we know it's within the bounds of the segment, 
		 * if it's on the line, it's on the segment.
		 */
		double slope = (double)(y1-y0) / (double)(x1-x0);
		double fx = (slope * (double)(x-x0)) + (double)y0;
		return Math.abs(fx - (double)y) < insideTolerance(slope);
	}
	/**  
	 * Answer a number to define how close we have to be to the geometric line 
	 * to be considered inside it.
	 * 
	 * @return	a double representing a number to define how close we have to
	 * be to be considered inside it
	 */
	protected double insideTolerance() {
		return 2.5;
	}
	/**  
	 * Answer a number to define how close we have to be to the geometric line 
	 * to be considered inside it, given a particular slope.
	 *
	 * @return	a double representing a number to define how close we have to
	 * the geometric line defined by the given slope in order to be considered
	 * inside it
	 */
	protected double insideTolerance(double slope) {
		return Math.max(insideTolerance(),Math.abs(slope));
	}
	/** 
	 * Answers whether the receiver intersects a rectangular area.
	 * 
	 * @param bounds the rectangular area.
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * specified rectangular area;
	 * 			<code>false</code> otherwise.
	 */
	public boolean intersects(Rectangle bounds) {
		return getBounds().intersects(bounds);
	}
	/** 
	 * Answers a locator corresponding to a significant point on the receiver.
	 * By default, answer a point with the same relative position as x and y
	 * are at the time of the request.
	 * 
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 * @return	a Locator corresponding to a significant point on the receiver
	 */
	public Locator locatorAt(int x, int y) {
		int numberOfSegments = getNumberOfPoints();
		int segment = 1;
		Locator previous = getLocator(0);
		Locator current = getLocator(1);
		for (; segment < numberOfSegments; segment++) {
			previous = getLocator(segment-1);
			current = getLocator(segment);
			if (insideSegment(x,y,previous.x(),previous.y(),current.x(),current.y())) 
				break;
		}
		if (segment == numberOfSegments)
			return null;
		int currentX = current.x();
		int currentY = current.y();
		int previousX = previous.x();
		int previousY = previous.y();

	//	double relativeX = 0.5;
	//	double relativeY = 0.5;
		double relativeX = 0.0;
		double relativeY = 0.0;
		if (currentX != previousX)
			relativeX = Math.abs(((double)(x - previousX)) / (double)(currentX - previousX));
		if (currentY != previousY)
			relativeY = Math.abs(((double)(y - previousY)) / (double)(currentY - previousY));
		return new LineFigureRelativePoint(this, segment, relativeX, relativeY);
	}
	/** 
	 * Paints the receiver.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g) {
		g.setColor(getLineColor());
		// draw figure
	}
	/**
	 * Remove the locator at the given position.
	 * This is a TemplateMethod with hooks:
	 * 	resetBoundsCache()
	 * 	basicRemoveLocator()
	 *  changedShape()
	 *
	 * @param index the index of the locator to be removed.
	 */
	public void removeLocator(int index) {
		Rectangle oldBounds = getBounds();
		resetBoundsCache();
		basicRemoveLocator(index);
		changedShape(oldBounds);
	}
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the other figure.
	 * By default, make it the middle of the receiver.
	 * Subclasses may wish to do something more meaningful.
	 *
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 * @return	a Locator corresponding to a significant point on the
	 * receiver that will serve as a connection to the given figure
	 */
	public Locator requestConnection(Figure requestor, int x, int y) {
		if (requestor == this)
			return null;
		return locatorAt(x,y);
	}
	/**
	 * Set the Color to use when drawing lines.
	 * 
	 * @param color the color
	 */
	public void setLineColor(Color color) {
		Color oldColor = lineColor;
		lineColor = color;
		firePropertyChange(LINE_COLOR_PROPERTY, oldColor, color);
	}
	/**
	 * Set the locator at the appropriate position.
	 * This is a TemplateMethod with hooks:
	 * 	resetBoundsCache()
	 * 	basicSetLocator()
	 *  changedShape()
	 *
	 * @param index the index of the locator desired.
	 * @param locator the new Locator.
	 */
	public void setLocator(int index, Locator locator) {
		Rectangle oldBounds = getBounds();
		resetBoundsCache();
		basicSetLocator(index, locator);
		changedShape(oldBounds);
	}
	/** 
	 * Set the DrawingStyle defining how to paint the receuver.
	 * 
	 * @param style the specified DrawingStyle.
	 */
	public void setStyle(DrawingStyle style) {
		DrawingStyle oldStyle = getStyle();
		if (style != null) {
			setLineColor(style.getLineColor());
		}
		firePropertyChange(STYLE_PROPERTY, oldStyle, style);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)LineFigureRelativePoint.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.util.Hashtable;

/**
 * This class implements Locator by providing x and y coordinates that are
 * relative to a point on a segment of LineFigure.  
 * Although it is assumed that the normal use will have
 * these locators on the LineFigure, that is not necessary.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class LineFigureRelativePoint extends FigureRelativePoint {
	static final long serialVersionUID = 8456398037474191179L;
	
	/**
	 * The 1-based index of the segment within the figure from which 
	 * we will define our relative position.
	 */
	protected int segment = 1;
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a point on a line
	 *
	 * @param figure the figure to which the instance will be relative
	 */
	public LineFigureRelativePoint(LineFigure figure) {
		super(figure);
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a point on a line
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param relativeX the percentage of the length of the default segment at which to determine X
	 * @param relativeY the percentage of the length of the default segment at which to determine Y
	 */
	public LineFigureRelativePoint(LineFigure figure, double relativeX, double relativeY) {
		this(figure);
		this.relativeX = relativeX;
		this.relativeY = relativeY;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a point on a line
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param relativeX the percentage of the length of the default segment at which to determine X
	 * @param relativeY the percentage of the length of the default segment at which to determine Y
	 * @param offsetX the offset (added to relative X) in the x direction
	 * @param offsetY the offset (added to relative Y) in the y direction
	 */
	public LineFigureRelativePoint(LineFigure figure, double relativeX, double relativeY, int offsetX, int offsetY) {
		this(figure,relativeX,relativeY);
		this.offsetX = offsetX;
		this.offsetY = offsetY;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a point on a line
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param segment the 1-based index of the specific segment which will be the basis of our relative location
	 * @param relativeX the percentage of the length of the segment at which to determine X
	 * @param relativeY the percentage of the length of the segment at which to determine Y
	 */
	public LineFigureRelativePoint(LineFigure figure, int segment, double relativeX, double relativeY) {
		this(figure,relativeX,relativeY);
		this.segment = segment;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a point on a line
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param segment the 1-based index of the specific segment which will be the basis of our relative location
	 * @param relativeX the percentage of the length of the segment at which to determine X
	 * @param relativeY the percentage of the length of the segment at which to determine Y
	 * @param offsetX the offset (added to relative X) in the x direction
	 * @param offsetY the offset (added to relative Y) in the y direction
	 */
	public LineFigureRelativePoint(LineFigure figure, int segment, double relativeX, double relativeY, int offsetX, int offsetY) {
		this(figure,relativeX,relativeY,offsetX,offsetY);
		this.segment = segment;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a point on a line
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param offsetX the offset (added to default relative X) in the x direction
	 * @param offsetY the offset (added to default relative Y) in the y direction
	 */
	public LineFigureRelativePoint(LineFigure figure, int offsetX, int offsetY) {
		this(figure);
		this.offsetX = offsetX;
		this.offsetY = offsetY;
	}
	/** 
	 * Constructs and initializes a new instance of a locator which is
	 * relative to a point on a line
	 *
	 * @param figure the figure to which the instance will be relative
	 * @param segment the 1-based index of the specific segment which will be the basis of our relative location
	 * @param offsetX the offset (added to default relative X) in the x direction
	 * @param offsetY the offset (added to default relative Y) in the y direction
	 */
	public LineFigureRelativePoint(LineFigure figure, int segment, int offsetX, int offsetY) {
		this(figure,offsetX,offsetY);
		this.segment = segment;
	}
	/** 
	 * Answer the default/initial value of relativeX.
	 * 
	 * @return	a double representing the default/initial value of relativeX
	 */
	protected double defaultRelativeX() {
		return 0.5;
	}
	/** 
	 * Answer the default/initial value of relativeY.
	 * 
	 * @return	a double representing the default/initial value of relativeY
	 */
	protected double defaultRelativeY() {
		return 0.5;
	}
	/** 
	 * Answer the x coordinate.
	 * 
	 * @return	an integer representing the x coordinate
	 */
	public int x()  {
		LineFigure myFigure = (LineFigure)figure;
		Locator begin = myFigure.getLocator(segment - 1);
		Locator end = myFigure.getLocator(segment);
		return (int)Math.rint((double)(end.x() - begin.x()) * relativeX) + begin.x() + offsetX;
	}
	/** 
	 * Answer the y coordinate.
	 * 
	 * @return	an integer representing the y coordinate
	 */
	public int y()  {
		LineFigure myFigure = (LineFigure)figure;
		Locator begin = myFigure.getLocator(segment - 1);
		Locator end = myFigure.getLocator(segment);
		return (int)Math.rint((double)(end.y() - begin.y()) * relativeY) + begin.y() + offsetY;
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)LinePointHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at a particular Locator on a line
 * and reshapes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
public class LinePointHandle extends SquareCanvasHandle implements FigureHolder {

	/**
	 * The figure to which we are attached.
	 */
	protected LineFigure figure;

	/**
	 * The index of the point (vertex) on the line we manipulate.
	 */
	protected int index;

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the the position of a particular point defining a line.
	 *
	 * @param figure the figure whose point (vertex) we may wish to move.
	 * @param index the identifying index of the point of interest.
	 */
	public LinePointHandle(LineFigure figure, int index) {
		this.figure = figure;
		this.index = index;
	}
	/**  
	 * Answer the x coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the x coordinate at the center of the handle
	 */
	protected int centerX()  {
		return figure.getLocator(index).x();
	}
	/**  
	 * Answer the y coordinate at the center of the handle.  
	 * 
	 * @return	an integer representing the x coordinate at the center of the handle
	 */
	protected int centerY()  {
		return figure.getLocator(index).y();
	}
	/** 
	 * Answers the figure associated with this handle.
	 * 
	 * @return	the Figure associated with this handle
	 */
	public Figure getFigure()  {
		return figure;
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 *
	 * @param evt the event
	 * @see #move
	 */
	public void mouseDragged(MouseEvent evt) {
		move(getX(evt), getY(evt));
		evt.consume();
	}
	/**
	 * Called if the mouse is released (the mouse button goes up).
	 *
	 * @param evt the event
	 */
	public void mouseReleased(MouseEvent evt) {
		if ( figure.isObsolete() )
			canvas.removeFigure( figure );
			canvas.removeHandles( figure );
		super.mouseReleased( evt );
	}
	/**
	 * Reshape the figure by moving the corresponding point
	 * and cleanly repaint the canvas.
	 *
	 * @param x the x coordinate for the new location of the point
	 * @param y the y coordinate for the new location of the point
	 */
	public void move(int x, int y)  {
		Rectangle bounds = figure.getBounds();
		figure.setLocator(index, new DrawingPoint(x,y));
		bounds = bounds.union(figure.getBounds());
		bounds.grow(halfWidth, halfWidth);
		canvas.repaint(bounds);
	}
	/** 
	 * Set the figure the receiver is holding.
	 * Note: in this case, it must be a LineFigure
	 *
	 * @param newFigure the new LineFigure to affect.
	 * @exception IllegalArgumentException If the figure is not a LineFigure.
	 */
	public void setFigure(Figure newFigure)  {
		if (!(newFigure instanceof LineFigure))
			throw new IllegalArgumentException("Figure must be a LineFigure"); 
		figure = (LineFigure)newFigure;
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/*
 * @(#)LineTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This tool produces Lines.
 * Currently, one must hold the shift key down in order to add more than
 * a single segment.
 *
 * @version 	1.1.6, 12/29/98
 */
public class LineTool extends ShapeTool {
	/** 
	 * Constructs and initializes a new instance of a tool to create lines
	 * on a DrawingCanvas
	 *
	 * @param canvas the canvas on which to place lines.
	 */
	public LineTool(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
   /**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a newly created Figure
	 */
	protected Figure basicNewFigure(int x, int y)  {
		return new Line(x,y,x,y);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down and mouse is moving).  
	 * If we are constructing a line, move its last locator as indicated.
	 *
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		movePoint(((LineFigure) figure).getNumberOfPoints() - 1, getX(e), getY(e) );
		e.consume();
	}
	/**
	 * Called if the mouse moves (when the mouse button is up).
	 * If we are constructing a line, move its last locator as indicated.
	 * When done, consume the event.
	 *
	 * @param e the event
	 */
	public void mouseMoved(MouseEvent e) {
		if (figure == null)
			return;
		movePoint(((LineFigure) figure).getNumberOfPoints() - 1, getX(e), getY(e));
		e.consume();
	}
	/**
	 * Called if the mouse is released.
	 * If we are constructing a line, and the shift key is down, add another
	 * point.  Otherwise, we're done constructing this one... get ready for the
	 * next one.
	 *
	 * @param e the event 
	 */
	public void mouseReleased(MouseEvent e) {
		if (figure == null)
			return;
		LineFigure myShape = (LineFigure) figure;
		Rectangle bounds = myShape.getBounds();

		Locator nextLast, last;
		LineFigure line = (LineFigure)figure;
		nextLast = line.getLocator( line.getNumberOfPoints() - 2 );
		last = line.getLocator( line.getNumberOfPoints() - 1 );
		if ( last.x() == nextLast.x() && last.y() == nextLast.y() ) {
			if ( line.getNumberOfPoints() == 2 )
				canvas.removeFigure( figure );
			else
				return;
		}
		
		if (!e.isShiftDown()) {
			figure = null;
			canvas.toolTaskCompleted(this);
		} else {
			myShape.addLocator(new DrawingPoint(getX(e), getY(e)));
			bounds = bounds.union(myShape.getBounds());
		}
		canvas.repaint(bounds);
		e.consume();
	}
	/**
	 * Move the identified point of the line.  
	 * Assumes that pointIndex is a valid point in the current line being edited.
	 * 
	 * @param pointIndex the index of the point to move.
	 * @param x the x coordinate.
	 * @param y the y coordinate.
	 */
	protected void movePoint(int pointIndex, int x, int y) {
		LineFigure myShape = (LineFigure)figure;
		Rectangle bounds = myShape.getBounds();
		myShape.setLocator(pointIndex,new DrawingPoint(x,y));
		bounds = bounds.union(myShape.getBounds());
		canvas.repaint(bounds);
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)TC_AdornedLine.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.lines.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import java.awt.*;
import junit.framework.*;
import junit.ui.*;

public class TC_AdornedLine extends TC_AbstractFigure {
	AdornedLine adornedLine;
	Arrow adornment1;
/**
 * AdornedLineTest constructor comment.
 * @param name java.lang.String
 */
public TC_AdornedLine(String name) {
	super(name);
}
/**
 * Sets up the fixture, for example, open a network connection.
 * This method is called before a test is executed.
 */
public void setUp() 
{
	adornedLine = new AdornedLine( 0, 0, 50, 50 );
	figure = (AbstractFigure) adornedLine;
	adornment1 = new Arrow( adornedLine );
	adornedLine.addAdornment( adornment1 );
	propertyRevealer = new PropertyChangeRevealer();
	locationRevealer = new RelatedLocationRevealer();
}
/**
 * Test to make sure ardornments are being added properly.
 */
public void testAddAdornment() 
{
	assert( "The adornment was not properly added.", figure.contains( 43, 48 ) );
}
/**
 * Test to make sure the contains( Figure ) method is
 * functioning properly.
 */
public void testContainsFigure() 
{
	Figure testFigure = new RectangleShape( 9, 9, 2, 2 );
	assert( "Figure does not contain the figure", figure.contains( testFigure ) );
	testFigure.setBounds( 51, 51, 2, 2 );
	assert( "Figure contains the figure", ! figure.contains( testFigure ) );
}
/**
 * Test to make sure the contains( int, int ) method is
 * functioning properly.
 */
public void testContainsIntInt() 
{
	assert( "Figure does not contain the point", figure.contains( 6, 6 ) );
	assert( "Figure contains the point", ! figure.contains( 5, 2 ) );
}
/**
 * Test to make sure the contains( Rectangle ) method is
 * functioning properly.
 */
public void testContainsRectangle() 
{
	Rectangle testRect = new Rectangle( 9, 9, 2, 2 );
	assert( "Figure does not contain the rectangle", figure.contains( testRect ) );
	testRect.setBounds( 51, 51, 2, 2 );
	assert( "Figure contains the rectangle", ! figure.contains( testRect ) );
}
/**
 * Test to make sure the bottom is properly returned.
 */
public void testGetBottom() 
{
	assertEquals( "The int returned does not correspond to the expected bottommost coordinate of the figure", 50, figure.getBottom() );
}
/**
 * Test to make sure the bottom is properly returned.
 */
public void testGetBounds() 
{
	assertEquals( "The bounds were incorrect", new Rectangle( 0, 0, 50, 50 ), figure.getBounds() );
}
/**
 * Test to make sure the height of the figure is properly
 * returned.
 */
public void testGetHeight() 
{
	assertEquals( "The height was not what was expected", 50, figure.getHeight() );
}
/**
 * Test to make sure the leftmost coordinate of the figure
 * is properly returned.
 */
public void testGetLeft() 
{
	assertEquals( "The int returned does not correspond to the expected leftmost coordinate of the figure", 0, figure.getLeft() );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testGetLocator() 
{
	assert( "There was no locator returned.", figure.getLocator() != null );
	assert( "The locator returned does not correspond to the top left corner of this figure. If that is OK, you need to override this test method.", figure.getLocator().x() == 0 && figure.getLocator().y() == 0 );
}
/**
 * Test to make sure the rightmost coordinate of the figure
 * is properly returned.
 */
public void testGetRight() 
{
	assertEquals( "The int returned does not correspond to the expected rightmost coordinate of the figure", 50, figure.getRight() );
}
/**
 * Test to make sure the size of the figure is properly
 * returned.
 */
public void testGetSize() 
{
	assertEquals( "The size was not what was expected", new Dimension( 50, 50 ), figure.getSize() );
}
/**
 * Test to make sure the top is properly returned.
 */
public void testGetTop() 
{
	assert( "The int returned does not correspond to the expected topmost coordinate of the figure", figure.getTop() == 0 );
}
/**
 * Test to make sure the width of the figure is properly
 * returned.
 */
public void testGetWidth() 
{
	assertEquals( "The width was not what was expected", 50, figure.getWidth() );
}
/**
 * Test to make sure the intersects( Figure ) method is
 * functioning properly.
 */
public void testIntersectsFigure() 
{
	Figure testFigure = new RectangleShape( 6, 6, 2, 2 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 2, 2, 5, 5 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 52, 52, 6, 6 );
	assert( "Figure intersects the figure", ! figure.intersects( testFigure ) );
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectsRectangle() 
{
	Rectangle testRect = new Rectangle( 6, 6, 2, 2 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 2, 2, 5, 5 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 52, 52, 6, 6 );
	assert( "Figure intersects the rectangle", ! figure.intersects( testRect ) );
}
/**
 * Test to make sure the isWithin( Figure ) method is
 * functioning properly.
 */
public void testIsWithinFigure() 
{
	Figure testFigure = new RectangleShape( -1, -1, 52, 52 );
	assert( "Figure isn't within the figure", figure.isWithin( testFigure ) );
	testFigure.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the figure", ! figure.isWithin( testFigure ) );
}
/**
 * Test to make sure the isWithin( Rectangle ) method is
 * functioning properly.
 */
public void testIsWithinRectangle() 
{
	Rectangle testRect = new Rectangle( -1, -1, 52, 52 );
	assert( "Figure isn't within the rectangle", figure.isWithin( testRect ) );
	testRect.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the rectangle", ! figure.isWithin( testRect ) );
}
/**
 * Test to make sure relatedLocationListeners
 * works correctly.
 */
public void testRelatedLocationListeners()
{
	assert( "The figure did not already have a location listener", figure.relatedLocationListeners().hasMoreElements() );
	figure.addRelatedLocationListener( locationRevealer );
	assert( "The figure did not have any location listeners", figure.relatedLocationListeners().hasMoreElements() );
}
/**
 * Test to make sure ardornments are being removed properly.
 */
public void testRemoveAdornment() 
{
	adornedLine.removeAdornment( adornment1 );
	assert( "The adornment was not properly removed.", ! figure.contains( 1, 5 ) );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testRequestConnection() 
{
	Locator locator = figure.requestConnection( new RectangleShape(), 0, 0 );
	if( locator == null ) {
		assert( "There was no locator returned.", false );
		return;
	}
	assert( "The locator returned does not correspond to the center of this figure. It was " + String.valueOf( locator.x() ) + ", " + String.valueOf( locator.y() ) + ". If that is OK, you need to override this test method.", locator.x() == 0 && locator.y() == 0 );
	if ( locator instanceof FigureRelativePoint ) {
		FigureRelativePoint relativeLocator = (FigureRelativePoint) locator;
		assert( "The locator returned does not have the figure as its owner.", relativeLocator.getFigure() == figure );
		figure.translate( 1, 1 );
		assert( "The locator did not update when the figure moved.", relativeLocator.x() == 1 && relativeLocator.y() == 1 );
	} else {
		assert( "The locator returned is not a FigureRelativePoint. If that is OK, you need to override this test method.", false );
	}
}
/**
 * Test to make sure the bounds are properly set.
 */
public void testSetBounds()
{
	figure.setBounds( 5, 5, 100, 100 );
	assertEquals( "The figure's left side is incorrect.", 5, figure.getLeft() );
	assertEquals( "The figure's right side is incorrect", 105, figure.getRight() );
	assertEquals( "The figure's top side is incorrect.", 5, figure.getTop() );
	assertEquals( "The figure's bottom side is incorrect.", 105, figure.getBottom() );
	
}
/**
 * Test to make sure the setSize( Dimension ) method is
 * working properly.
 */
public void testSetSizeDimension()
{
	figure.setSize( new Dimension( 100, 100 ) );
	assertEquals( "The figure's width is incorrect.", 100, figure.getWidth() );
	assertEquals( "The figure's height is incorrect.", 100, figure.getHeight() );
}
/**
 * Test to make sure the setSize( int, int ) method is
 * working properly.
 */
public void testSetSizeIntInt()
{
	figure.setSize( 100, 100 );
	assertEquals( "The figure's width is incorrect", 100, figure.getWidth() );
	assertEquals( "The figure's height is incorrect", 100, figure.getHeight() );
}
/**
 * Test to make sure translate works correctly.
 */
public void testTranslate() 
{
	translateByExpecting( 5, 5 , 5, 5 );
	translateByExpecting( -5, -5, 0, 0);
}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)TC_Arrow.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import com.rolemodelsoft.drawlet.shapes.lines.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import java.awt.*;
import junit.framework.*;

public class TC_Arrow extends TC_AbstractFigure {
	AbstractFigure forwardFigure, reverseFigure;
	AdornedLine lineFigure;
/**
 * ArrowTest constructor comment.
 * @param name java.lang.String
 */
public TC_Arrow(String name) {
	super(name);
}
/**
 * ArrowTest constructor comment.
 * @param name java.lang.String
 */
public void setUp() {
	lineFigure = new AdornedLine( 0, 0, 50, 50 );
	forwardFigure = new Arrow( lineFigure, Arrow.FORWARD );
	reverseFigure = new Arrow( lineFigure, Arrow.REVERSE );
	figure = forwardFigure;
	lineFigure.addAdornment( (Arrow)figure );
	propertyRevealer = new PropertyChangeRevealer();
	locationRevealer = new RelatedLocationRevealer();
	ArrowStyle.setDefaultLocators( new Locator[] {
										new PolarCoordinate( 10, 0.0 + Math.PI ),
										new PolarCoordinate( 20, 0.5 + Math.PI ),
										new PolarCoordinate( 0, 0.0 ),
										new PolarCoordinate( 20, -0.5 + Math.PI )
									}
	);
}
/**
 * Test to make sure PropertyChangeListeners are being added
 * correctly.
 */
public void testAddPropertyChangeListener() 
{
	figure.addPropertyChangeListener( propertyRevealer );
	figure.setStyle( new SimpleDrawingStyle() );
	assert( "No PropertyChangeEvent propagated", propertyRevealer.getEventCount() == 1 );

	propertyRevealer.clearEventCount();
	figure.addPropertyChangeListener( propertyRevealer );
	figure.setStyle( new SimpleDrawingStyle() );
	assert( "Two PropertyChangeEvents propagated", propertyRevealer.getEventCount() == 1 );
}
/**
 * Test to make sure RelatedLocationListeners are being added
 * correctly.
 */
public void testAddRelatedLocationListener() 
{
	figure.addRelatedLocationListener( locationRevealer );
	lineFigure.translate( 5, 5 );
	assertEquals( "RelatedLocatorEvent not propagated.", 1, locationRevealer.getEventCount() );
}
/**
 * Test to make sure the contains( Figure ) method is
 * functioning properly.
 */
public void testContainsFigure() 
{
	Figure testFigure = new RectangleShape( 43, 43, 5, 5);
	assert( "Figure does not contain the figure", forwardFigure.contains( testFigure ) );
	testFigure.setBounds( 9, 9, 12, 12 );
	assert( "Figure contains the figure", ! forwardFigure.contains( testFigure ) );
}
/**
 * Test to make sure the contains( int, int ) method is
 * functioning properly.
 */
public void testContainsIntInt() 
{
	assert( "Figure does not contain the point", forwardFigure.contains( 49, 49 ) );
	assert( "Figure contains the point", ! forwardFigure.contains( 20, 20 ) );
}
/**
 * Test to make sure the contains( Rectangle ) method is
 * functioning properly.
 */
public void testContainsRectangle() 
{
	Rectangle testRect = new Rectangle( 43, 43, 5, 5);
	assert( "Figure does not contain the rectangle", forwardFigure.contains( testRect ) );
	testRect.setBounds( 9, 9, 12, 12 );
	assert( "Figure contains the rectangle", ! forwardFigure.contains( testRect ) );
}
/**
 * Test to make sure the ArrowStyle is properly returned.
 */
public void testGetArrowStyle() 
{
	ArrowStyle style = ((Arrow)figure).getArrowStyle();
	Locator[] styleLocs, locs;
	styleLocs = style.getLocators();
	locs = new Locator[] { new PolarCoordinate( 10, 0.0 + Math.PI ), new PolarCoordinate( 20, 0.5 + Math.PI ), new PolarCoordinate( 0, 0.0 ), new PolarCoordinate( 20, -0.5 + Math.PI ) };
	for ( int i = 0; i < styleLocs.length; i++ ) {
		assertEquals( "The ArrowStyle's locators were incorrect", locs[i].x(), styleLocs[i].x() );
		assertEquals( "The ArrowStyle's locators were incorrect", locs[i].y(), styleLocs[i].y() );
	}
	assert( "The style wasn't opaque", style.isOpaque() );
}
/**
 * Test to make sure the bottom is properly returned.
 */
public void testGetBottom() 
{
	assertEquals( "The int returned does not correspond to the expected bottommost coordinate of the figure", 50, figure.getBottom() );
}
/**
 * Test to make sure the bounds are returned properly.
 */
public void testGetBounds() 
{
	assertEquals( "The bounds were incorrect.", new Rectangle( 31, 31, 19, 19 ), forwardFigure.getBounds() );
}
/**
 * Test to make sure the handles are properly returned.
 */
public void testGetHandles() 
{
	assert( "There were handles returned.", forwardFigure.getHandles().length == 0 );
}
/**
 * Test to make sure the height of the figure is properly
 * returned.
 */
public void testGetHeight() 
{
	assertEquals( "The height was not what was expected", 19, forwardFigure.getHeight() );
}
/**
 * Test to make sure the leftmost coordinate of the figure
 * is properly returned.
 */
public void testGetLeft() 
{
	assertEquals( "The int returned does not correspond to the expected leftmost coordinate of the figure", 31, forwardFigure.getLeft() );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testGetLocator() 
{
	Locator loc = forwardFigure.getLocator();
	assert( "There was no locator returned.", loc != null );
	assertEquals( "The locator returned does not correspond to the top left corner of this figure. If that is OK, you need to override this test method.", 31, loc.x() );
	assertEquals( "The locator returned does not correspond to the top left corner of this figure. If that is OK, you need to override this test method.", 31, loc.y() );
}
/**
 * Test to make sure the rightmost coordinate of the figure
 * is properly returned.
 */
public void testGetRight() 
{
	assertEquals( "The int returned does not correspond to the expected rightmost coordinate of the figure", 50, figure.getRight() );
}
/**
 * Test to make sure the size of the figure is properly
 * returned.
 */
public void testGetSize() 
{
	assertEquals( "The size was not what was expected", new Dimension( 19, 19 ), forwardFigure.getSize() );
}
/**
 * Test to make sure the top is properly returned.
 */
public void testGetTop() 
{
	assertEquals( "The int returned does not correspond to the expected topmost coordinate of the figure", 31, forwardFigure.getTop() );
}
/**
 * Test to make sure the width of the figure is properly
 * returned.
 */
public void testGetWidth() 
{
	assertEquals( "The width was not what was expected", 19, forwardFigure.getWidth() );
}
/**
 * Test to make sure the intersects( Figure ) method is
 * functioning properly.
 */
public void testIntersectsFigure() 
{
	Figure testFigure = new RectangleShape( 43, 43, 2, 2 );
	assert( "Figure does not intersect the figure", forwardFigure.intersects( testFigure ) );
	testFigure.setBounds( 29, 29, 11, 11 );
	assert( "Figure does not intersect the figure", forwardFigure.intersects( testFigure ) );
	testFigure.setBounds( 16, 16, 6, 6 );
	assert( "Figure intersects the figure", ! forwardFigure.intersects( testFigure ) );
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectsRectangle() 
{
	Rectangle testRect = new Rectangle( 43, 43, 2, 2 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 29, 29, 11, 11 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 16, 16, 6, 6 );
	assert( "Figure intersects the rectangle", ! figure.intersects( testRect ) );
}
/**
 * Test to make sure the isWithin( Figure ) method is
 * functioning properly.
 */
public void testIsWithinFigure() 
{
	Figure testFigure = new RectangleShape( 30, 30, 21, 21 );
	assert( "Figure isn't within the figure", figure.isWithin( testFigure ) );
	testFigure.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the figure", ! figure.isWithin( testFigure ) );
}
/**
 * Test to make sure the isWithin( Rectangle ) method is
 * functioning properly.
 */
public void testIsWithinRectangle() 
{
	Rectangle testRect = new Rectangle( 30, 30, 21, 21 );
	assert( "Figure isn't within the rectangle", figure.isWithin( testRect ) );
	testRect.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the rectangle", ! figure.isWithin( testRect ) );
}
/**
 * ArrowTest constructor comment.
 * @param name java.lang.String
 */
public void testLineMoved() {
	assertEquals( "Arrows original bounds are not correct", new Rectangle( 31, 31, 19, 19 ), figure.getBounds() );
	lineFigure.translate( 5, 5 );
	assertEquals( "Arrows bounds are not correct", new Rectangle( 36, 36, 19, 19 ), figure.getBounds() );
}
/**
 * Test to make sure locatorAt returns the proper value.
 */
public void testLocatorAt() 
{
	Locator locator = figure.locatorAt( 0, 0 );
	if( ! ( locator instanceof FigureRelativePoint ) ) {
		assert( "The locator returned isn't a FigureRelativePoint", false );
		return;
	}
	FigureRelativePoint relativeLocator = (FigureRelativePoint) locator;
	assert( "The relativeLocator returned doesn't have the figure as its owner", relativeLocator.getFigure() == figure );
	assert( "The relativeLocator has an improper x", relativeLocator.x() == 0 );
	assert( "The relativeLocator has an improper y", relativeLocator.y() == 0 );
	lineFigure.translate( 1, 1 );
	assert( "The relativeLocator improperly adjusted its x", relativeLocator.x() == 1 );
	assert( "The relativeLocator improperly adjusted its y", relativeLocator.y() == 1 );
}
/**
 * Test to make sure move( int, int ) works correctly.
 */
public void testMoveIntInt() 
{
	figure.move( 20, 20 );
	Locator loc = figure.getLocator();
	assertEquals( "The figure did not move properly", 31, loc.x() );
	assertEquals( "The figure did not move properly", 31, loc.y() );
}
/**
 * Test to make sure move( Locator ) works correctly.
 */
public void testMoveLocator()
{
	figure.move( new DrawingPoint( 20, 20 ) );
	Locator loc = figure.getLocator();
	assertEquals( "The figure did not move properly", 31, loc.x() );
	assertEquals( "The figure did not move properly", 31, loc.y() );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testRequestConnection() 
{
	Locator locator = figure.requestConnection( new RectangleShape(), 0, 0 );
	assert( "There was no locator returned.", locator != null );
	assertEquals( "The locator returned does not correspond to the center of this figure. If that is OK, you need to override this test method.", 40, locator.x() );
	assertEquals( "The locator returned does not correspond to the center of this figure. If that is OK, you need to override this test method.", 40, locator.y() );
	assert( "The locator returned is not a FigureRelativePoint. If that is OK, you need to override this test method.", locator instanceof FigureRelativePoint );
	FigureRelativePoint relativeLocator = (FigureRelativePoint) locator;
	assertEquals( "The locator returned does not have the figure as its owner.", figure, relativeLocator.getFigure() );
	lineFigure.translate( 1, 1 );
	assertEquals( "The locator did not update when the figure moved.", 41, relativeLocator.x() );
	assertEquals( "The locator did not update when the figure moved.", 41, relativeLocator.y() );
}
/**
 * Test to make sure the bounds are properly set.
 */
public void testSetBounds()
{
	figure.setBounds( 50, 50, 50, 50 );
	assertEquals( "The figure's bounds changed", new Rectangle( 31, 31, 19, 19 ), figure.getBounds() );
	
}
/**
 * Test to make sure the setSize( Dimension ) method is
 * working properly.
 */
public void testSetSizeDimension()
{
	figure.setSize( new Dimension( 100, 100 ) );
	assertEquals( "The figure's size changed", new Dimension( 19, 19 ), figure.getSize() );
}
/**
 * Test to make sure the setSize( int, int ) method is
 * working properly.
 */
public void testSetSizeIntInt()
{
	figure.setSize( 100, 100 );
	assertEquals( "The figure's size changed", new Dimension( 19, 19 ), figure.getSize() );
}
/**
 * Test to make sure translate works correctly.
 */
public void testTranslate() 
{
	translateByExpecting( 5, 5 , 31, 31 );
	translateByExpecting( -5, -5, 31, 31);
}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)TC_ArrowStyle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.lines.*;
import junit.framework.*;
import junit.ui.*;

public class TC_ArrowStyle extends TestCase {
	Locator[] locs;
/**
 * ArrowStyleTest constructor comment.
 * @param name java.lang.String
 */
public TC_ArrowStyle(String name) {
	super(name);
}
	/**
	 */
	public void setUp() {
		locs = new Locator[] {
			new PolarCoordinate( 10, 0.0 + Math.PI ),
			new PolarCoordinate( 20, 0.5 + Math.PI ),
			new PolarCoordinate( 0, 0.0 ),
			new PolarCoordinate( 20, -0.5 + Math.PI )
		};

		ArrowStyle.setDefaultLocators( locs );
	}
	/**
	 */
	public void testGetLocators() {
		ArrowStyle style = new ArrowStyle();
		for ( int i = 0; i < locs.length; i++ ) {
			assertEquals( "The locators returned were incorrect", (int)locs[i].theta(), (int)style.getLocators()[i].theta() );
			assertEquals( "The locators returned were incorrect", locs[i].r(), style.getLocators()[i].r() );
		}
	}
	/**
	 */
	public void testIsOpaque() {
		assert( "The opacity was incorrect", (new ArrowStyle()).isOpaque() );
	}
	/**
	 */
	public void testSetDefaultLocators() {
		Locator[] locs = new Locator[] {
			new PolarCoordinate( 10, 0.5 ),
			new PolarCoordinate( 0, 0 ),
			new PolarCoordinate( 10, -0.5 )
		};

		ArrowStyle style = new ArrowStyle();
		
		assert( "The right locators were already in place", (int)locs[0].theta() != (int)style.getLocators()[0].theta() );

		ArrowStyle.setDefaultLocators( locs );
		assert( "The right locators were propagated.",  (int)locs[0].theta() != (int)style.getLocators()[0].theta() );
		style = new ArrowStyle();
		assertEquals( "The wrong locators were returned", locs, style.getLocators() );
	}
	/**
	 */
	public void testSetOpaque() {
		ArrowStyle style = new ArrowStyle();
		style.setOpaque( false );
		assert( "The opacity was not set.", ! style.isOpaque() );
	}
}
package com.rolemodelsoft.drawlet.shapes.lines;

/**
 * @(#)TS_LineTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

public class TS_LineTests extends junit.framework.TestCase {

public TS_LineTests(String name) {
	super(name);
}

public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}

public static TestSuite suite() {
	TestSuite suite = new TestSuite();
	
	suite.addTest(new TestSuite(TC_AdornedLine.class));
	suite.addTest(new TestSuite(TC_ArrowStyle.class));
	suite.addTest(new TestSuite(TC_Arrow.class));
	
	return suite;
}
}
package com.rolemodelsoft.drawlet.shapes.polygons;

/*
 * @(#)AnySidedPolygonTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This tool produces PolygonShapes with a variable number of sides.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class AnySidedPolygonTool extends ShapeTool {
	/** 
	 * Constructs and initializes a new instance of a tool to create polygons
	 * on a DrawingCanvas
	 *
	 * @param canvas the canvas on which to place PolygonFigures.
	 */
	public AnySidedPolygonTool(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
   /**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a newly created Figure
	 */
	protected Figure basicNewFigure(int x, int y)  {
		PolygonFigure newShape = new PolygonShape();
		Polygon polygon = new Polygon();
		polygon.addPoint(x,y);
		newShape.setPolygon(polygon);
		return newShape;
	}
	/**
	 * Answer the default/initial number of sides.
	 * 
	 * @return	an integer representing the default/initial number of sides.
	 */
	protected int defaultNumberOfSides() {
		return 3;
	}
	/**
	 * Called when the mouse is double-clicked.
	 * 
	 * @param e the MouseEvent to handle.
	 */
	protected void mouseDoubleClicked( MouseEvent e ) {
		if (figure == null)
			return;
		PolygonFigure myShape = (PolygonFigure) figure;
		Rectangle bounds = myShape.getBounds();
		Polygon polygon = myShape.getPolygon();
		polygon = new Polygon(polygon.xpoints,polygon.ypoints,polygon.npoints-1);
		if (polygon.npoints > 2) {
			myShape.setPolygon(polygon);
			polygon.addPoint(polygon.xpoints[0], polygon.ypoints[0]);
			canvas.repaint(bounds.union(myShape.getBounds()));
		} else
			canvas.removeFigure(figure);
		figure = null;
		canvas.toolTaskCompleted(this);
		e.consume();
	}
	/**
	 * Called if the mouse is dragged (moved with the mouse button down).
	 * If we are constructing a polygon, move its last point as indicated, 
	 * and consume the event.
	 *
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		movePoint(((PolygonFigure) figure).getPolygon().npoints - 1, getX(e), getY(e) );
		e.consume();
	}
	/**
	 * Called if the mouse moves (when the mouse button is up).
	 * If we are constructing a polygon, move its last point as indicated.
	 *
	 * @param e the event
	 */
	public void mouseMoved(MouseEvent e) {
		if (figure == null)
			return;
		movePoint(((PolygonFigure) figure).getPolygon().npoints - 1, getX(e), getY(e) );
		e.consume();
	}
	/**
	 * Called if the mouse goes up.
	 *
	 * @param e the event 
	 */
	public void mouseReleased(MouseEvent e) {
		if (figure == null || e.getClickCount() > 1)
			return;
		PolygonFigure myShape = (PolygonFigure) figure;
		Rectangle bounds = myShape.getBounds();
		Polygon polygon = myShape.getPolygon();
		polygon = new Polygon(polygon.xpoints,polygon.ypoints,polygon.npoints);
		polygon.addPoint( getX(e), getY(e) );
		myShape.setPolygon(polygon);
		canvas.repaint(bounds.union(myShape.getBounds()));
		e.consume();
	}
	/**
	 * Move the identified point of the polygon.  
	 * Assumes that pointIndex is a valid point in the current polygon being edited
	 * 
	 * @param pointIndex the index of the point to move.
	 * @param x the x coordinate.
	 * @param y the y coordinate.
	 */
	protected void movePoint(int pointIndex, int x, int y) {
		PolygonFigure myShape = (PolygonFigure) figure;
		Rectangle bounds = myShape.getBounds();
		Polygon polygon = myShape.getPolygon();
		polygon = new Polygon(polygon.xpoints, polygon.ypoints, polygon.npoints);
		polygon.xpoints[pointIndex] = x;
		polygon.ypoints[pointIndex] = y;
		myShape.setPolygon(polygon);
		bounds = bounds.union(myShape.getBounds());
		canvas.repaint(bounds);
	}
}
package com.rolemodelsoft.drawlet.shapes.polygons;

/**
 * @(#)PolygonPointHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import com.rolemodelsoft.drawlet.basics.SquareCanvasHandle;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at a particular Locator on a 
 * PolygonFigure and reshapes the figure accordingly as it is dragged.  
 * If the first and last point of the polygon are the same, it moves them both.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class PolygonPointHandle extends SquareCanvasHandle implements FigureHolder {

	/**
	 * The figure to which the handle is attached.
	 */
	protected PolygonFigure figure;

	/**
	 * The index of the point (vertex) we manipulate.
	 */
	protected int index = 0;

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the the position of a particular point defining a polygon.
	 *
	 * @param figure the figure whose point (vertex) we may wish to move.
	 * @param index the identifying index of the point of interest
	 */
	public PolygonPointHandle(PolygonFigure figure, int index) {
		this.figure = figure;
		this.index = index;
	}
	/**  
	 * Answer the x coordinate at the center of the handle.
	 *
	 * @return an integer representing the x coordinate at the center of the handle.
	 */
	public int centerX()  {
		return figure.getPolygon().xpoints[index];
	}
	/**  
	 * Answer the y coordinate at the center of the handle.  
	 *
	 * @return an integer representing the y coordinate at the center of the handle.
	 */
	public int centerY()  {
		return figure.getPolygon().ypoints[index];
	}
	/** 
	 * Returns the Figure associated with this handle.
	 *
	 * @return the Figure associated with this handle.
	 */
	public Figure getFigure()  {
		return figure;
	}
	/**
	 * Called if the mouse is dragged (moved with the mouse button down).
	 * Move the particular point of the polygon accordingly.
	 *
	 * @param evt the event
	 * @see #move
	 */
	public void mouseDragged(MouseEvent evt) {
		move(getX(evt), getY(evt));
		evt.consume();
	}
	/**
	 * Reshape the figure by moving the corresponding point
	 * and cleanly repaint the canvas.
	 *
	 * @param x an integer representing the x coordinate to move to.
	 * @param y an integer representing the y coordinate to move to.
	 */
	public void move(int x, int y) {
		Rectangle bounds = figure.getBounds();
		
		// There should be a cleaner way to copy a polygon and all its points
		Polygon polygon = figure.getPolygon();
		polygon = new Polygon(polygon.xpoints, polygon.ypoints, polygon.npoints);
		/* 
		 *if closed polygon and moving the starting/ending point, 
		 * we need to move both 
		 */
		if (index == 0) {
			int lastIndex = polygon.npoints - 1;
			if ((polygon.xpoints[index] == polygon.xpoints[lastIndex]) && (polygon.ypoints[index] == polygon.ypoints[lastIndex])) {
				polygon.xpoints[polygon.npoints - 1] = x;
				polygon.ypoints[polygon.npoints - 1] = y;
			}
		}
		polygon.xpoints[index] = x;
		polygon.ypoints[index] = y;
		figure.setPolygon(polygon);
		bounds = bounds.union(figure.getBounds());
		bounds.grow(halfWidth, halfWidth);
		canvas.repaint(bounds);
	}
	/** 
	 * Set the figure the receiver is holding.
	 * Note: in this case, it must be a PolygonFigure.
	 *
	 * @param newFigure the new PolygonFigure to affect.
	 * @exception IllegalArgumentException If the figure is not a PolygonFigure.
	 */
	public void setFigure(Figure newFigure)  {
		if (!(newFigure instanceof PolygonFigure))
			throw new IllegalArgumentException("Figure must be a PolygonFigure"); 
		figure = (PolygonFigure)newFigure;
	}
}
package com.rolemodelsoft.drawlet.shapes.polygons;

/*
 * @(#)PolygonShape.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.util.GraphicsGeometry;
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.util.Hashtable;

/**
 * This provides a basic concrete implementation of PolygonFigures that are 
 * assumed to be movable and reshapable with observers that want to know when 
 * their locations or shapes change.  Although this is a concrete class, it is
 * acknowledged that there are probably other implementations (e.g. one that 
 * uses Locators) which are more flexible.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class PolygonShape extends FilledShape implements PolygonFigure {
	static final long serialVersionUID = 8026851545884863918L;
	
	/**
	 * The underlying, defining polygon.
	 */
	protected Polygon polygon = defaultPolygon();

	/** 
	 * Constructs a new instance of the receiver.
	 */
	public PolygonShape() {
	}
	/** 
	 * Constructs and initializes a new instance of the receiver based on
	 * a defining polygon.
	 *
	 * @param polygon the polygon which defines the basic shape of the receiver.
	 */
	public PolygonShape(Polygon polygon) {
	this.polygon = polygon;
	}
	/** 
	 * Reshapes the Figure to the specified bounding box.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width of the figure
	 * @param height the height of the figure
	 */
	protected synchronized void basicReshape(int x, int y, int width, int height) {
		basicSetPolygon(reshapedPolygon(polygon, x, y, width, height));
	}
	/**  
	 * Set the Polygon associated with the Figure.
	 *
	 * @param polygon the Polygon.
	 */
	protected synchronized void basicSetPolygon(Polygon polygon) {
		this.polygon = polygon;
	}
	/** 
	 * Moves the Figure in the x and y direction.
	 *
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 */
	protected synchronized void basicTranslate(int x, int y) {
		polygon.translate(x, y);
	}
	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * Figure, where x and y are defined to be relative to the 
	 * coordinate system of this figure.  
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 */
	public boolean contains(int x, int y) {
		return polygon.contains(x, y);
	}
	/**
	 * Answer the default/initial value for polygon
	 */
	protected Polygon defaultPolygon() {
		return new Polygon();
	}
	/**
	 * Duplicates the receiver into a Hashtable.
	 *
	 * @param duplicates the Hashtable to place the duplicate in.
	 * @return the duplicate.
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		PolygonShape duplicate = (PolygonShape)super.duplicateIn(duplicates);
		duplicate.polygon = new Polygon(polygon.xpoints,polygon.ypoints,polygon.npoints);
		return duplicate;
	}
	/** 
	 * Returns the current bounds of the receiver.
	 *
	 * @return a Rectangle representing the current bounds of the receiver.
	 */
	public Rectangle getBounds()  {
		return new Rectangle(polygon.getBounds()); // make a copy, otherwise we'll mess with the polygon's bounds attribute
	}
	/** 
	 * Answer the handles associated with the receiver.
	 *
	 * @return an array of Handles.
	 */
	public Handle[] getHandles() {
		int numberOfHandles = polygon.npoints;
		int lastIndex = polygon.npoints - 1;
		/* 
		 * if the first and last points are the same, 
		 * we only need one handle 
		 */
		if ((polygon.xpoints[0] == polygon.xpoints[lastIndex]) &&
			(polygon.ypoints[0] == polygon.ypoints[lastIndex]))
			numberOfHandles--;

		Handle handles[] = new Handle[numberOfHandles];
		for (int i=0; i < numberOfHandles; i++)
			handles[i] = new PolygonPointHandle(this,i);
		return handles;
	}
	/**  
	 * Answer a Polygon associated with the receiver.
	 *
	 * @return a copy of the Polygon which defines the receiver.
	 */
	public Polygon getPolygon() {
		return new Polygon(polygon.xpoints,polygon.ypoints,polygon.npoints);
	}
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 * By default, just check if the bounds intersects.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * specified Rectangular area;
	 * 			<code>false</code> otherwise.
	 * @see #bounds
	 */
public boolean intersects(int x1, int y1, int x2, int y2) {
	
	if(polygon.npoints < 2) return false;
	
	// check lines between consecutive points
	for(int i = 0; i < polygon.npoints-1; i++){
		if( GraphicsGeometry.segmentIntersectsSegment(x1, y1, x2, y2, polygon.xpoints[i], polygon.ypoints[i], polygon.xpoints[i+1], polygon.ypoints[i+1]) )
			return true;
	}
	// check connecting line from last point to first point
	if( GraphicsGeometry.segmentIntersectsSegment(x1, y1, x2, y2, polygon.xpoints[polygon.npoints-1], polygon.ypoints[polygon.npoints-1], polygon.xpoints[0], polygon.ypoints[0]) )
		return true;
		
	return false;
}
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 *
	 * @param Box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * specified Rectangular area;
	 * 			<code>false</code> otherwise.
	 * @see #bounds
	 */
public boolean intersects(Rectangle Box) {
	// because the width and height of the bounds is one less than what is drawn on screen,
	// to check bounds must expand by one
	Rectangle bigBox = new Rectangle(Box.x, Box.y, Box.width+1, Box.height+1);
	Rectangle bigBounds = new Rectangle(getBounds().x, getBounds().y, getBounds().width+1, getBounds().height+1);
	
	//if bounds don't intersect, box and poly don't intersect
	if (! bigBounds.intersects(bigBox))	
		return false;
	
	//check if rectangle completely inside polygon
	if( contains(Box.x, Box.y) )
		return true;

	//check if polygon completely inside rectangle
	if( bigBox.contains(polygon.xpoints[0], polygon.ypoints[0]) ) 
		return true;
	
	//check top side of rectangle
	if( intersects(Box.x, Box.y, Box.x + Box.width, Box.y) )
		return true;

	//check right side of rectangle
	if( intersects(Box.x + Box.width, Box.y, Box.x + Box.width, Box.y + Box.height) )
		return true;

	//check bottom side of rectangle
	if( intersects(Box.x + Box.width, Box.y + Box.height, Box.x, Box.y + Box.height) )
		return true;

	//check left side of rectangle
	if( intersects(Box.x, Box.y + Box.height, Box.x, Box.y) )
		return true;
		
	return false;
}
	/**
	 * Paint the shape, filling all contained area.
	 *
	 * @param g the specified Graphics window
	 */
	public void paintFilled(Graphics g)  {
		super.paintFilled(g);
		g.fillPolygon(polygon);
	}
	/**
	 * Paint the outline of the shape.
	 *
	 * @param g the specified Graphics window
	 */
	public void paintStrokes(Graphics g)  {
		super.paintStrokes(g);
		g.drawPolygon(polygon);
	}
	/**  
	 * Set the Polygon associated with the figure.
	 * Let observers know what changed.
	 * This is a TemplateMethod with hooks:
	 * 	resetBoundsCache();
	 * 	basicSetPolygon();
	 *  changedShape();
	 *
	 * @param polygon the Polygon
	 */
	public void setPolygon(Polygon polygon) {
		Rectangle oldShape = getBounds();
		resetBoundsCache();
		basicSetPolygon(polygon);
		changedShape(oldShape);
	}
}
package com.rolemodelsoft.drawlet.shapes.polygons;

/**
 * @(#)PolygonTestVisualizer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.examples.awt.*;
import java.awt.*;

public class PolygonTestVisualizer extends ExitingFrame {
	protected Polygon polygon;
	protected Rectangle[] rectangles;
/**
 * PolygonTestVisualizer constructor comment.
 */
public PolygonTestVisualizer() {
	super();
}
/**
 * PolygonTestVisualizer constructor comment.
 * @param title java.lang.String
 */
public PolygonTestVisualizer(Polygon polygon, Rectangle[] rectangles) {
	super();
	this.polygon = polygon;
	this.rectangles = rectangles;
}
/**
 * PolygonTestVisualizer constructor comment.
 * @param title java.lang.String
 */
public PolygonTestVisualizer(String title) {
	super(title);
}
/**
 * 
 * @param g java.awt.Graphics
 */
public void paint(Graphics g) {
	g.setColor(Color.black);
	for (int i=0; i<rectangles.length; i++) {
		Rectangle rect = rectangles[i];
		g.drawRect(rect.x,rect.y,rect.width,rect.height);
	}
	g.setColor(Color.blue);
	g.drawPolygon(polygon);	
}
}
package com.rolemodelsoft.drawlet.shapes.polygons;

/*
 * @(#)PolygonTool.java	1.0 96/12/18 Ken Auer
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This tool produces PolygonShapes with a given number of sides.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class PolygonTool extends ShapeTool {

	/**
	 * The number of sides of the polygon we are to create.
	 */
	int numberOfSides = defaultNumberOfSides();

	/** 
	 * Constructs and initializes a new instance of a tool to create polygons
	 * on a DrawingCanvas
	 *
	 * @param canvas the canvas on which to place PolygonFigures.
	 */
	public PolygonTool(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
	/** 
	 * Constructs and initializes a new instance of a tool to create polygons
	 * on a DrawingCanvas with a given number of sides.
	 *
	 * @param canvas the canvas on which to place PolygonFigures.
	 * @param numberOfSides the number of sides of the polygons we create.
	 */
	public PolygonTool(DrawingCanvas canvas, int numberOfSides) {
		this(canvas);
		this.numberOfSides = numberOfSides;
	}
   /**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate.
	 * @param y the y coordinate.
	 * @return	a newly created Figure.
	 */
	protected Figure basicNewFigure(int x, int y)  {
		PolygonFigure newShape = new PolygonShape();
		Polygon polygon = new Polygon();
		polygon.addPoint(x,y);
		newShape.setPolygon(polygon);
		return newShape;
	}
	/**
	 * Answer the default/initial number of sides.
	 * 
	 * @return	an integer representing the default/initial number of sides.
	 */
	protected int defaultNumberOfSides() {
		return 3;
	}
	/**
	 * Called if the mouse is dragged (moved with the mouse button down).
	 * If we are constructing a polygon, move its last point as indicated,
	 * and consume the event.
	 *
	 * @param e the event.
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		movePoint(Math.min(((PolygonFigure) figure).getPolygon().npoints, numberOfSides) - 1, getX(e), getY(e) );
		e.consume();
	}
	/**
	 * Called if the mouse moves (when the mouse button is up).
	 * If we are constructing a polygon, move its last point as indicated.
	 *
	 * @param e the event.
	 */
	public void mouseMoved(MouseEvent e) {
		if (figure == null)
			return;
		movePoint(Math.min(((PolygonFigure) figure).getPolygon().npoints, numberOfSides) - 1, getX( e ), getY( e ));
		e.consume();
	}
	/**
	 * Called if the mouse goes up.
	 * If we are constructing a polygon, and we don't have the right number of
	 * sides yet, add another point.  Otherwise, we're done constructing this 
	 * one... get ready for the next one.
	 *
	 * @param e the event.
	 */
	public void mouseReleased(MouseEvent e) {
		if (figure == null)
			return;
		PolygonFigure myShape = (PolygonFigure) figure;
		Rectangle bounds = myShape.getBounds();
		Polygon polygon = myShape.getPolygon();
		polygon = new Polygon(polygon.xpoints,polygon.ypoints,polygon.npoints);
		if (polygon.npoints > numberOfSides) {
			figure = null;
			canvas.toolTaskCompleted(this);
		} else {
			polygon.addPoint( getX(e), getY(e) );
			if (polygon.npoints == numberOfSides)
				polygon.addPoint(polygon.xpoints[0], polygon.ypoints[0]);
		}
		myShape.setPolygon(polygon);
		canvas.repaint(bounds.union(myShape.getBounds()));
		e.consume();
	}
	/**
	 * Move the identified point of the polygon.  
	 * Assumes that pointIndex is a valid point in the current polygon being edited.
	 * 
	 * @param pointIndex the index of the point to move.
	 * @param x the x coordinate
	 * @param y the y coordinate
	 */
	protected void movePoint(int pointIndex, int x, int y) {
		PolygonFigure myShape = (PolygonFigure) figure;
		Rectangle bounds = myShape.getBounds();
		Polygon polygon = myShape.getPolygon();
		polygon = new Polygon(polygon.xpoints, polygon.ypoints, polygon.npoints);
		polygon.xpoints[pointIndex] = x;
		polygon.ypoints[pointIndex] = y;
		myShape.setPolygon(polygon);
		bounds = bounds.union(myShape.getBounds());
		canvas.repaint(bounds);
	}
}
package com.rolemodelsoft.drawlet.shapes.polygons;

/**
 * @(#)TC_Polygon.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.ellipses.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.shapes.polygons.*;
import java.awt.*;
import junit.framework.*;
import junit.ui.*;

public class TC_Polygon extends TC_AbstractFigure {
	protected static boolean run;
/**
 * PolygonTest constructor comment.
 */
public TC_Polygon(String name) {
	super(name);
}
/**
 * Sets up the fixture, for example, open a network connection.
 * This method is called before a test is executed.
 */
protected void setUp() 
{
	int[] intx = {5, 15, 15, 5};
	int[] inty = {5, 5, 15, 15};
	Polygon p = new Polygon(intx, inty, 4);
	figure = new PolygonShape(p);
	propertyRevealer = new PropertyChangeRevealer();
	locationRevealer = new RelatedLocationRevealer();
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectRectPolygon1() 
{
	Polygon p1 = new Polygon(new int[] {50, 100, 100, 50}, new int[] {100, 100, 200, 200}, 4);
	PolygonShape poly1 = new PolygonShape(p1);
	Rectangle rects1[] = new Rectangle[] {
		// sides and corners
		new Rectangle(40,90,10,10),
		new Rectangle(100,90,10,10),
		new Rectangle(100,200,10,10),
		new Rectangle(40,200,10,10),
		new Rectangle(70,90,10,10),
		new Rectangle(40,145,10,10),
		new Rectangle(100,145,10,10),
		new Rectangle(70,200,10,10),

		// other stuff
		new Rectangle(60,95,30,10),
		new Rectangle(95,110,10,80),
		new Rectangle(60,195,30,10),
		new Rectangle(45,110,10,80),
		new Rectangle(70,120,10,60),
		new Rectangle(30,80,90,140),
		new Rectangle(45,95,10,10),
		new Rectangle(95,95,10,10),
		new Rectangle(95,195,10,10),
		new Rectangle(45,195,10,10),
		new Rectangle(43,93,64,14),
		new Rectangle(43,193,64,14),
		new Rectangle(41,91,16,118),
		new Rectangle(93,91,16,118),
		new Rectangle(50,100,50,100)
	};
	visualize(p1,rects1,"Polygon1 Intersection");
	for(int i = 0; i < rects1.length; i++) {
		if (!poly1.intersects( rects1[i] ) )
			assert( "Perpendicular polygon should intersect the rectangle", poly1.intersects( rects1[i] ) );
	}
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectRectPolygon2() 
{
	int[] intx2 = {150, 200, 250, 300, 350, 175};
	int[] inty2 = {250, 200, 150, 110, 300, 120};
	Polygon p2 = new Polygon(intx2, inty2, 6);	
	PolygonShape poly2 = new PolygonShape(p2);
	Rectangle rects2[] = new Rectangle[] {
		 //corner and sides
		new Rectangle(300,100,10,10),
		new Rectangle(290,100,10,10),
		new Rectangle(350,300,10,10),
		new Rectangle(345,300,10,10),
		new Rectangle(140,245,10,10),
		new Rectangle(140,250,10,10),
		new Rectangle(170,110,10,10),
		
		new Rectangle(145,245,15,10),
		new Rectangle(170,115,15,10),
		new Rectangle(340,200,25,115),
		new Rectangle(140,180,15,90),
		new Rectangle(185,200,25,25),
		new Rectangle(235,200,25,25),
		new Rectangle(195,130,20,20),
		new Rectangle(270,105,20,20),
		new Rectangle(270,145,20,20),
		new Rectangle(130,100,240,225)
	};
	visualize(p2,rects2,"Polygon2 Intersection");
	for(int i = 0; i < rects2.length; i++) {
		assert( "Skewed polygon should intersect the rectangle", poly2.intersects( rects2[i] ) );
	}
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectRectPolygon3() 
{
	int[] intx3 = {100, 300, 300, 100, 200, 200};
	int[] inty3 = {100, 100, 200, 200, 50, 250};
	Polygon p3 = new Polygon(intx3, inty3, 6);
	PolygonShape poly3 = new PolygonShape(p3);
	Rectangle rects3[] = new Rectangle[] {
		new Rectangle(145,155,20,20),
		new Rectangle(145,125,20,20),
		new Rectangle(173,95,9,20),
		new Rectangle(173,185,9,20),
		new Rectangle(185,125,20,9),
		new Rectangle(185,100,9,20),
		new Rectangle(185,180,9,20),
		new Rectangle(180,137,20,9),
	};
	visualize(p3,rects3,"Polygon3 Intersection");
	for(int i = 0; i < rects3.length; i++) {
		assert( "Hollow polygon should intersect the rectangle", poly3.intersects( rects3[i] ) );
	}
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectsRectangle() 
{
	testIntersectRectPolygon1();
	testIntersectRectPolygon2();
	testIntersectRectPolygon3();
	testNonIntersectRectPolygon1();
	testNonIntersectRectPolygon2();
	testNonIntersectRectPolygon3();
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testNonIntersectRectPolygon1() 
{
	Polygon p1 = new Polygon(new int[] {50, 100, 100, 50}, new int[] {100, 100, 200, 200}, 4);
	PolygonShape poly1 = new PolygonShape(p1);
	Rectangle rects1[] = new Rectangle[] {
		new Rectangle(39,90,10,10),
		new Rectangle(100,89,10,10),
		new Rectangle(100,201,10,10),
		new Rectangle(39,200,10,10),
		new Rectangle(70,89,10,10),
		new Rectangle(39,145,10,10),
		new Rectangle(101,145,10,10),
		new Rectangle(70,201,10,10),
	};
	visualize(p1,rects1,"Polygon1 Non-Intersection");
	for(int i = 0; i < rects1.length; i++) {
		assert( "Perpendicular polygon should not intersect the rectangle", !poly1.intersects( rects1[i] ) );
	}
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testNonIntersectRectPolygon2() 
{
	int[] intx2 = {150 , 200 , 250 , 300, 350, 175};
	int[] inty2 = {250 , 200 , 150 , 110, 300, 120};
	Polygon p2 = new Polygon(intx2, inty2, 6);
	PolygonShape poly2 = new PolygonShape(p2);
	Rectangle rects2[] = new Rectangle[] {
		new Rectangle(225,150,10,10),
		new Rectangle(225,185,10,10),
		new Rectangle(158,130,10,10),
		new Rectangle(310,130,10,10),
		new Rectangle(180,222,10,10),
		new Rectangle(260,222,10,10),
		new Rectangle(170,109,10,10),
		new Rectangle(139,245,10,10),
		new Rectangle(351,300,10,10),
	};
	visualize(p2,rects2,"Polygon2 Non-Intersection");
	for(int i = 0; i < rects2.length; i++) {
		assert( "Skewed polygon should not intersect the rectangle", ! poly2.intersects( rects2[i] ) );
	}
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testNonIntersectRectPolygon3() 
{
	int[] intx3 = {100, 300, 300, 100, 200, 200};
	int[] inty3 = {100, 100, 200, 200, 50, 250};
	Polygon p3 = new Polygon(intx3, inty3, 6);
	PolygonShape poly3 = new PolygonShape(p3);
	Rectangle rects3[] = new Rectangle[] {
		new Rectangle(155,140,20,20),
		new Rectangle(173,101,20,20),
		new Rectangle(173,179,20,20),
		new Rectangle(179,125,20,20)
	};
	visualize(p3,rects3,"Polygon3 Non-Intersection");
	for(int i = 0; i < rects3.length; i++) {
		assert( "Hollow polygon should not intersect the rectangle", !poly3.intersects( rects3[i] ) );
	}
}
/**
 * Paint the polygon and rectangles to see what we are trying to test... visually verifying
 * This method is basically used to see why the unit tests may be failing.
 * To run them modify the run field to equal true.
 */
protected static void visualize(Polygon p1, Rectangle[] rects, String title) {
	if (!run)
		return;
	PolygonTestVisualizer visualizer = new PolygonTestVisualizer(p1,rects);
	visualizer.setBounds(0,75,400,350);
	visualizer.setTitle(title);
	visualizer.show();
	visualizer.toFront();
}
}
package com.rolemodelsoft.drawlet.shapes.polygons;

/**
 * @(#)TS_PolygonTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

public class TS_PolygonTests extends junit.framework.TestCase {

public TS_PolygonTests(String name) {
	super(name);
}

public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}

public static TestSuite suite() {
	TestSuite suite = new TestSuite();
	
	suite.addTest(new TestSuite(TC_Polygon.class));
	
	return suite;
}
}
package com.rolemodelsoft.drawlet.shapes.rectangles;

/**
 * @(#)RectangleShape.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;

/**
 * This provides a basic concrete implementation of BoundedFigures in the form
 * of "rectilinear" boxes that are assumed to be movable and reshapable with
 * observers that want to know when their locations or shapes change.
 * Although this is a concrete class, it is acknowledged that there are
 * probably other implementations (e.g. one that uses Locators) which are more flexible.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class RectangleShape extends AbstractRectangleShape {
	static final long serialVersionUID = 5365603316454725622L;
	/**
	 * Constructs a new, default instance of the receiver.
	 */
	public RectangleShape() {
	}
	/**
	 * Constructs a new instance of the receiver initialized with the
	 * given values for x, y, width and height.
	 * 
	 * @param x an integer representing the x coordinate.
	 * @param y an integer representing the y coordinate.
	 * @param width an integer representing the width.
	 * @param height an integer representing the height.
	 */
	public RectangleShape(int x, int y, int width, int height) {
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
	}
	/**
	 * Constructs a new instance of the receiver initialized with the
	 * given Rectangle.
	 * 
	 * @param rectangle a Rectangle representing the bounding box.
	 */
	public RectangleShape(Rectangle rectangle) {
		this(rectangle.x,rectangle.y,rectangle.width,rectangle.height);
	}
	/**
	 * Paint the shape, filling all contained area.
	 * 
	 * @param g the specified Graphics window.
	 */
	public void paintFilled(Graphics g)  {
		super.paintFilled(g);
		g.fillRect(x,y,width,height);
	}
	/**
	 * Paint the outline of the shape.
	 * 
	 * @param g the specified Graphics window
	 */
	public void paintStrokes(Graphics g)  {
		super.paintStrokes(g);
		g.drawRect(x,y,width,height);
	}
}
package com.rolemodelsoft.drawlet.shapes.rectangles;

/**
 * @(#)RectangleTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This tool produces RectangleShapes.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class RectangleTool extends ShapeTool {
	/** 
	 * Constructs and initializes a new instance of a tool to create 
	 * RectangleShapes on a DrawingCanvas
	 *
	 * @param canvas the canvas on which to place RectangleShapes.
	 */
	public RectangleTool(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
   /**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a newly created Figure
	 */
	protected Figure basicNewFigure(int x, int y)  {
		Figure newFigure = new RectangleShape();
		newFigure.move(x,y);
		return newFigure;
	}
	/**
	 * Called if the mouse is dragged (moved with the mouse button down).
	 *
	 * @param e the event.
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		Rectangle bounds = figure.getBounds();

		int figureX = Math.min(anchorX, getX(e));
		int figureY = Math.min(anchorY, getY(e));
		int figureWidth = Math.abs(anchorX - getX(e));
		int figureHeight = Math.abs(anchorY - getY(e));

		figure.setBounds( figureX, figureY, figureWidth, figureHeight );
		bounds = bounds.union(figure.getBounds());
		canvas.repaint(bounds);
		e.consume();
	}
}
package com.rolemodelsoft.drawlet.shapes.rectangles;

/**
 * @(#)RoundedRectangleShape.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;

/**
 * This provides an implementation of a RectangleShape whose corners are rounded.
 * NOTE: intersection algorithms are currently incomplete and only determine if
 * the Rectangular bounds intersect.
 */
 
public class RoundedRectangleShape extends AbstractRectangleShape {
	static final long serialVersionUID = 6428757790918919037L;

	/**
	 * The width of the corner arc.
	 */
	protected int arcWidth = defaultArcWidth();

	/**
	 * The height of the corner arc.
	 */
	protected int arcHeight = defaultArcHeight();
	/**
	 * Constructs a new, default instance of the receiver.
	 */
	public RoundedRectangleShape() {
	}
	/**
	 * Constructs a new instance of the receiver initialized with the
	 * given values for x, y, width and height.
	 * 
	 * @param x an integer representing the x coordinate.
	 * @param y an integer representing the y coordinate.
	 * @param width an integer representing the width.
	 * @param height an integer representing the height.
	 */
	public RoundedRectangleShape(int x, int y, int width, int height) {
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
	}
	/**
	 * Constructs a new instance of the receiver initialized with the
	 * given values for x, y, width, height, arcWidth, and arcHeight.
	 * 
	 * @param x an integer representing the x coordinate.
	 * @param y an integer representing the y coordinate.
	 * @param width an integer representing the width.
	 * @param height an integer representing the height.
	 * @param arcWidth an integer representing the width of the corners.
	 * @param arcHeight an integer representing the height of the corners.
	 */
	public RoundedRectangleShape(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
		this.arcWidth = arcWidth;
		this.arcHeight = arcHeight;
	}
	/**
	 * Constructs a new instance of the receiver initialized with the
	 * given Rectangle.
	 * 
	 * @param rectangle a Rectangle representing the bounding box.
	 */
	public RoundedRectangleShape(Rectangle rectangle) {
		this(rectangle.x,rectangle.y,rectangle.width,rectangle.height);
	}
	/**
	 * Constructs a new instance of the receiver initialized with the
	 * given Rectangle, arcWidth, and arcHeight.
	 * 
	 * @param rectangle a Rectangle representing the bounding box.
	 * @param arcWidth an integer representing the width of the corners.
	 * @param arcHeight an integer representing the height of the corners.
	 */
	public RoundedRectangleShape(Rectangle rectangle, int arcWidth, int arcHeight) {
		this(rectangle.x,rectangle.y,rectangle.width,rectangle.height, arcWidth, arcHeight);
	}
	/**  
	 * Checks whether a specified x,y location is "inside" this
	 * Figure. <code>x</code> and <code>y</code> are defined to be relative to the 
	 * coordinate system of this figure.  This is not guaranteed to be 100% accurate around the edges
	 * due to rounding errors, but shouldn't be off by more than a single pixel.
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 * @return	boolean value of <code>true</code> if the specified x,y
	 * location is "inside" this Figure;
	 * 			<code>false</code> otherwise.
	 * @see #isWithin
	 */
	public boolean contains(int x, int y) {
		if (!super.contains(x,y))
			return false;
		/*
		 * if it's in one of the corners, verify it is not outside the rounded area (ellipse)
		 */
		int a = (arcWidth + 1) / 2; // get half the width of the arc, being overly conservative
		int b = (arcHeight + 1) / 2; // get half the height of the arc, being overly conservative
		return
			!(
			(cornerExcludes(new Rectangle( getLeft(), getTop(), a, b ), getLeft() + a, getTop() + b, x, y)) ||
			(cornerExcludes(new Rectangle( getLeft(), getBottom()-b, a, b ), getLeft() + a, getBottom() - b, x, y)) ||
			(cornerExcludes(new Rectangle( getRight()-a, getTop(), a, b ), getRight() - a, getTop() + b, x, y)) ||
			(cornerExcludes(new Rectangle( getRight()-a, getBottom()-b, a, b ), getRight() - a, getBottom() - b, x, y)));
	}
	/**  
	 * Checks whether a specified Rectangle is "inside" this Figure, 
	 * where the Rectangle and this Figure are in the same coordinate system  
	 * In addition to checking topLeft and bottomRight, check topRight and bottomLeft.
	 * If all four corners are inside, everything is inside.
	 *
	 * @param box the rectangle to test for inclusion
	 * @return	boolean value of <code>true</code> if the specified Rectangle
	 * is "inside" this Figure;
	 * 			<code>false</code> otherwise.
	 */
	public boolean contains(Rectangle box) {
		return super.contains(box) &&
			contains(box.x, box.y) && contains(box.x + box.width, box.y + box.height)
			&& contains(box.x + box.width, box.y) && contains(box.x, box.y + box.height);
	}
	/**  
	 * Checks whether a specified x,y location is "inside" the specified corner but outside the arc
	 * in that corner. <code>x</code> and <code>y</code> are defined to be relative to the 
	 * coordinate system of this figure.
	 *
	 * @param corner the corner in which the rounded edge we are interested in appears. 
	 * @param centerX the x coordinate of the center of the arc
	 * @param centerY the y coordinate of the center of the arc
	 * @param x the x coordinate of the location we are examining
	 * @param y the y coordinate of the location we are examining
	 * @return	boolean value of <code>true</code> if the specified x,y
	 * location is "inside" the corner but outside the arc;
	 * 			<code>false</code> otherwise.
	 */
	protected boolean cornerExcludes(Rectangle corner, int centerX, int centerY, int x, int y) {
		Rectangle bigCorner = new Rectangle (corner.x, corner.y, corner.width + 1, corner.height + 1);
		if ( bigCorner.contains( x, y ) ) {
			/*
			 * using standard geometric formula for ellipse
			 *   x^2   y^2
			 *   --- + --- = 1
			 *   a^2   b^2
			 * where a and b are x and y coordinates where y=0 and x=0 respectively...
			 * multiplying by a^2*b^2 and using > to determine points outside yields 
			 *   b^2*x^2 + a^2*y^2 <= a^2*b^2
			 */
			int a = corner.width;
			int b = corner.height;
			int aSquared = a * a;
			int bSquared = b * b;
			int normalizedX = x - centerX;
			int normalizedY = y - centerY;
			return (((normalizedX * normalizedX) * bSquared) + ((normalizedY * normalizedY) * aSquared)) > (aSquared * bSquared);
		}

		return false;
	}
	/**
	 * Answer the default/initial value for arc height.
	 * 
	 * @return	an integer representing the default/initial value for arc height.
	 */
	protected int defaultArcHeight() {
		return 20;
	}
	/**
	 * Answer the default/initial value for arc width.
	 * 
	 * @return	an integer representing the default/initial value for arc width.
	 */
	protected int defaultArcWidth() {
		return 20;
	}
	/** 
	 * Answers whether the receiver intersects a Rectangular area.
	 * By default, just check if the bounds intersects.
	 * Subclasses may wish to do something more sophisticated.
	 *
	 * @param box the Rectangular area
	 * @return	boolean value of <code>true</code> if the receiver intersects the
	 * specified Rectangular area;
	 * 			<code>false</code> otherwise.
	 * @see #bounds
	 */
	public boolean intersects(Rectangle box) {
		// if bounding boxes don't intersect, figures don't intersect
		if (! super.intersects(box))
			return false;
			
		// if the receiver contains one of the corners of the box, they intersect
		if (contains(box.x, box.y))
			return true;
		if (contains(box.x + box.width, box.y))
			return true;
		if (contains(box.x, box.y + box.height))
			return true;
		if (contains(box.x + box.width, box.y + box.height))
			return true;
			
		// if the box contains any of the sides of the receiver, they intersect
		Rectangle bigBox = new Rectangle(box.x, box.y, box.width + 1, box.height + 1);
		if (bigBox.contains(x + (width/2), y))
			return true;
		if (bigBox.contains(x, y + (height/2)))
			return true;
		if (bigBox.contains(x + (width/2), y + height))
			return true;
		if (bigBox.contains(x + width, y + (height/2)))
			return true;
			
		// otherwise they don't intersect
		return false;

	}
	/**
	 * Paint the shape, filling all contained area.
	 * 
	 * @param g the specified Graphics window.
	 */
	public void paintFilled(Graphics g)  {
		super.paintFilled(g);
		// fudge by 1 since filling primitives are different
		// than drawing primitive
		g.fillRoundRect(x,y,width+1,height+1,arcWidth,arcHeight);
	}
	/**
	 * Paint the outline of the shape.
	 * 
	 * @param g the specified Graphics window.
	 */
	public void paintStrokes(Graphics g)  {
		super.paintStrokes(g);
		g.drawRoundRect(x,y,width,height,arcWidth,arcHeight);
	}
}
package com.rolemodelsoft.drawlet.shapes.rectangles;

/**
 * @(#)TC_RectangelTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.awt.*;
import java.awt.*;
import java.awt.event.*;
import junit.framework.*;
import junit.ui.*;

public class TC_RectangleTool extends TestCase {
	protected DrawingCanvas staticCanvas, dynamicCanvas;
	protected Component staticComponent, dynamicComponent, component;
	protected RectangleTool staticTool, dynamicTool;
	protected MouseEvent pressedEventLoc1, pressedEventLoc2, pressedEventLoc3, pressedEventLoc4,
							draggedEventLoc1, draggedEventLoc2, draggedEventLoc3, draggedEventLoc4,
							releasedEventLoc1, releasedEventLoc2, releasedEventLoc3, releasedEventLoc4;
/**
 * 
 * @param name java.lang.String
 */
public TC_RectangleTool( String name ) {
	super( name );
}
/**
 */
public void setUp() {
	staticComponent = new DrawingCanvasComponent( staticCanvas = new SimpleDrawingCanvas( new SimpleDrawing( 40, 40 ) ) );
	staticTool = new RectangleTool( staticCanvas );
	dynamicComponent = new DrawingCanvasComponent( dynamicCanvas = new SimpleDrawingCanvas( new SimpleDrawing() ) );
	dynamicTool = new RectangleTool( dynamicCanvas );
	component = new Canvas();

	pressedEventLoc1 = new MouseEvent( component, MouseEvent.MOUSE_PRESSED, System.currentTimeMillis(), 0, 0, 0, 1, false );
	pressedEventLoc2 = new MouseEvent( component, MouseEvent.MOUSE_PRESSED, System.currentTimeMillis(), 0, 10, 10, 1, false );
	pressedEventLoc3 = new MouseEvent( component, MouseEvent.MOUSE_PRESSED, System.currentTimeMillis(), 0, 20, 20, 1, false );
	pressedEventLoc4 = new MouseEvent( component, MouseEvent.MOUSE_PRESSED, System.currentTimeMillis(), 0, 50, 50, 1, false );

	draggedEventLoc1 = new MouseEvent( component, MouseEvent.MOUSE_DRAGGED, System.currentTimeMillis(), 0, 0, 0, 0, false );
	draggedEventLoc2 = new MouseEvent( component, MouseEvent.MOUSE_DRAGGED, System.currentTimeMillis(), 0, 10, 10, 0, false );
	draggedEventLoc3 = new MouseEvent( component, MouseEvent.MOUSE_DRAGGED, System.currentTimeMillis(), 0, 20, 20, 0, false );
	draggedEventLoc4 = new MouseEvent( component, MouseEvent.MOUSE_DRAGGED, System.currentTimeMillis(), 0, 50, 50, 0, false );

	releasedEventLoc1 = new MouseEvent( component, MouseEvent.MOUSE_RELEASED, System.currentTimeMillis(), 0, 0, 0, 0, false );
	releasedEventLoc2 = new MouseEvent( component, MouseEvent.MOUSE_RELEASED, System.currentTimeMillis(), 0, 10, 10, 0, false );
	releasedEventLoc3 = new MouseEvent( component, MouseEvent.MOUSE_RELEASED, System.currentTimeMillis(), 0, 20, 20, 0, false );
	releasedEventLoc4 = new MouseEvent( component, MouseEvent.MOUSE_RELEASED, System.currentTimeMillis(), 0, 50, 50, 0, false );

	dynamicTool.mousePressed( pressedEventLoc2 );
	staticTool.mousePressed( pressedEventLoc2 );
}
	public void testMouseDragged() {
		// Dynamic test
		assertEquals( "The figures bounds were incorrect.", new Rectangle( 10, 10, 10, 10 ), dynamicCanvas.figures().nextElement().getBounds() );
		dynamicTool.mouseDragged( draggedEventLoc4 );
		assertEquals( "The figure did not resize correctly.", new Rectangle( 10, 10, 40, 40 ), dynamicCanvas.figures().nextElement().getBounds() );
		// Static test
		assertEquals( "The figures bounds were incorrect.", new Rectangle( 10, 10, 10, 10 ), staticCanvas.figures().nextElement().getBounds() );
		staticTool.mouseDragged( draggedEventLoc4 );
		assertEquals( "The figure did not resize correctly.", new Rectangle( 10, 10, 29, 29 ), staticCanvas.figures().nextElement().getBounds() );
	}
	public void testMousePressed() {
		// Dynamic tests
		assert( "The canvas did not have the figure already.", dynamicCanvas.figures().hasMoreElements() );
	}
	public void testMouseReleased() {
		dynamicTool.mouseReleased( releasedEventLoc2 );
		assert( "The canvas had the figure.", ! dynamicCanvas.figures().hasMoreElements() );
		dynamicTool.mousePressed( pressedEventLoc2 );
		dynamicTool.mouseReleased( releasedEventLoc3 );
		assert( "The canvas did not have the figure.", dynamicCanvas.figures().hasMoreElements() );
		dynamicCanvas.removeFigure( dynamicCanvas.figures().nextElement() );
		assert( "The figure was not removed.", ! dynamicCanvas.figures().hasMoreElements() );
		dynamicTool.mousePressed( pressedEventLoc2 );
		dynamicTool.mouseReleased( releasedEventLoc1 );
		assert( "The canvas did not have the figure.", dynamicCanvas.figures().hasMoreElements() );
	}
}
package com.rolemodelsoft.drawlet.shapes.rectangles;

/**
 * @(#)TC_RoundedRectangle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import junit.framework.*;
import java.awt.Rectangle;

public class TC_RoundedRectangle extends TC_AbstractFigure {
/**
 * RoundedRectangleTest constructor comment.
 * @param name java.lang.String
 */
public TC_RoundedRectangle(String name) {
	super(name);
}
	/**
	 * Sets up the fixture, for example, open a network connection.
	 * This method is called before a test is executed.
	 */
	protected void setUp() 
	{
		figure = new RoundedRectangleShape(5,5,10,10,4,6);
		propertyRevealer = new PropertyChangeRevealer();
		locationRevealer = new RelatedLocationRevealer();
	}
/**
 * Test to make sure the contains( Figure ) method is
 * functioning properly.
 */
public void testContainsFigure() 
{
	Figure testFigure = new RectangleShape( 6, 6, 8, 8 );
	assert( "Figure does not contain the figure", figure.contains( testFigure ) );
	testFigure.setBounds( 4, 4, 12, 12 );
	assert( "Figure contains the figure", ! figure.contains( testFigure ) );
	testFigure.setBounds( 5, 5, 9, 9 );
	assert( "Figure contains the figure", ! figure.contains( testFigure ) );
	testFigure.setBounds( 5, 10, 5, 4 );
	assert( "Figure contains the figure", ! figure.contains( testFigure ) );
}
/**
 * Test to make sure the contains( int, int ) method is
 * functioning properly.
 */
public void testContainsIntInt() 
{
	assert( "Figure does not contain the point 10, 10", figure.contains( 10, 10 ) );
	assert( "Figure contains the point 20, 20", ! figure.contains( 20, 20 ) );

	// check the corners
	assert( "Figure contains the point 5, 6", ! figure.contains( 5, 6 ) );
	assert( "Figure does not contain the point 6, 6", figure.contains( 6, 6 ) );
	
	assert( "Figure contains the point 15, 6", ! figure.contains( 15, 6 ) );
	assert( "Figure does not contain the point 14, 6", figure.contains( 14, 6 ) );
	
	assert( "Figure contains the point 15, 14", ! figure.contains( 15, 14 ) );
	assert( "Figure does not contain the point 14, 14", figure.contains( 14, 14 ) );
	
	assert( "Figure contains the point 5, 14", ! figure.contains( 5, 14 ) );
	assert( "Figure does not contain the point 6, 14", figure.contains( 6, 14 ) );

	assert( "Figure contains the point 10, 5", figure.contains( 10, 5 ) );
}
/**
 * Test to make sure the contains( Rectangle ) method is
 * functioning properly.
 */
public void testContainsRectangle() 
{
	Rectangle testRect = new Rectangle( 6, 6, 8, 8 );
	assert( "Figure does not contain the rectangle", figure.contains( testRect ) );
	testRect.setBounds( 4, 4, 12, 12 );
	assert( "Figure contains the rectangle", ! figure.contains( testRect ) );
	testRect.setBounds( 5, 5, 9, 9 );
	assert( "Figure contains the figure", ! figure.contains( testRect ) );
	testRect.setBounds( 5, 10, 5, 4 );
	assert( "Figure contains the figure", ! figure.contains( testRect ) );
}
/**
 * Test to make sure the intersects( Figure ) method is
 * functioning properly.
 */
public void testIntersectsFigure() 
{
	Figure testFigure = new RectangleShape( 6, 6, 2, 2 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 2, 2, 5, 5 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 16, 16, 6, 6 );
	assert( "Figure intersects the figure", ! figure.intersects( testFigure ) );
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectsRectangle() 
{
	Rectangle testRect = new Rectangle( 6, 6, 2, 2 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 2, 2, 5, 5 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 16, 16, 6, 6 );
	assert( "Figure intersects the rectangle", ! figure.intersects( testRect ) );
	testRect.setBounds( 4, 4, 1, 2 );
	assert( "RR does not intersect the rectangle", ! figure.intersects( testRect ) );
	testRect.setBounds( 4, 4, 2, 2 );
	assert( "RR does intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 5, 5, 10, 10 );
	assert( "RR does intersect its bounds", figure.intersects( testRect ) );
}
/**
 * Test to make sure the isWithin( Figure ) method is
 * functioning properly.
 */
public void testIsWithinFigure() 
{
	Figure testFigure = new RectangleShape( 4, 4, 12, 12 );
	assert( "Figure isn't within the figure", figure.isWithin( testFigure ) );
	testFigure.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the figure", ! figure.isWithin( testFigure ) );
}
/**
 * Test to make sure the isWithin( Rectangle ) method is
 * functioning properly.
 */
public void testIsWithinRectangle() 
{
	Rectangle testRect = new Rectangle( 4, 4, 12, 12 );
	assert( "Figure isn't within the rectangle", figure.isWithin( testRect ) );
	testRect.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the rectangle", ! figure.isWithin( testRect ) );
}
}
package com.rolemodelsoft.drawlet.shapes.rectangles;

/**
 * @(#)TS_RectangleTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

public class TS_RectangleTests extends junit.framework.TestCase {

public TS_RectangleTests(String name) {
	super(name);
}

public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}

public static TestSuite suite() {
	TestSuite suite = new TestSuite();
	
	suite.addTest(new TestSuite(TC_RectangleTool.class));
	suite.addTest(new TestSuite(TC_RoundedRectangle.class));
	
	return suite;
}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)RectangularCreationTool.java
 *
 * Copyright (c) 1999-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.shapes.*;
import java.awt.*;
import java.awt.event.*;

/**
 * This tool produces Shapes by creating an instance and then moving
 * their topLeft and bottomRight.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class RectangularCreationTool extends ShapeTool {

	/**
	 * The ShapeClass must be something that can be created with
	 * a default Constructor and resized by its bottom right.
	 * Consider subclasses of AbstractRectangleShape for example.
	 */
	protected Class shapeClass;
	/** 
	 * Constructs and initializes a new instance of a tool to create 
	 * Shapes of the specified shapeClass on a DrawingCanvas
	 *
	 * The shapeClass must be something that can be created with
	 * a default Constructor and resized by its bottom right.
	 * Consider subclasses of AbstractRectangleShape for example.
	 *
	 * @param canvas the canvas on which to place shapes.
	 * @param shapeClass the class of shape to create.
	 */
	public RectangularCreationTool(DrawingCanvas canvas, Class shapeClass) {
		this.canvas = canvas;
		this.shapeClass = shapeClass;
	}
   /**
	 * Create and answer a new Figure.
	 * 
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @return	a newly created Figure
	 */
	protected Figure basicNewFigure(int x, int y)  {
		Figure newFigure;
		try {
			newFigure = (Figure)shapeClass.newInstance();
		} catch (Throwable e) {
			System.out.println(shapeClass + " could not be instantiated via a default constructor.");
			return null;
		}
		newFigure.move(x,y);
		return newFigure;
	}
	/**
	 * Called if the mouse is dragged (when the mouse button is down).
	 *
	 * @param e the event
	 */
	public void mouseDragged(MouseEvent e) {
		if (figure == null)
			return;
		Rectangle bounds = figure.getBounds();

		int figureX = Math.min(anchorX, getX(e));
		int figureY = Math.min(anchorY, getY(e));
		int figureWidth = Math.abs(anchorX - getX(e));
		int figureHeight = Math.abs(anchorY - getY(e));

		figure.setBounds( figureX, figureY, figureWidth, figureHeight );
		bounds = bounds.union(figure.getBounds());
		canvas.repaint(bounds);
		e.consume();
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)RightHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the right of a figure
 * and resizes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class RightHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its right.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public RightHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,1.0,0.5);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Resize the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();	
		resize(Math.max(getX(evt) - bounds.x, 1), bounds.height);
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)ShapeTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.event.*;

/**
 * This provides basic functionality for a tool used to create concrete Figures.
 * Concrete subclasses must provide, at a minimum:
 * 	newShape(int,int)
 * But may also wish to consider mouseDragged(MouseEvent) or other events that
 * dynamically alter/build the Figure.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public abstract class ShapeTool extends ConstructionTool {
	/**
	 * The x coordinate where the mouse first went down.
	 */
	protected int anchorX;

	/**
	 * The y coordinate where the mouse first went down.
	 */
	protected int anchorY;
	/**
	 * Called if the mouse is pressed.  By default, create a new figure and add it.
	 * Keep track of where the mouse went down for future operations in
	 * addition to standard creation of new shape if we are not in the midst
	 * of constructing one.
	 * Consume the event since we handle it.
	 *
	 * @param e the event 
	 */
	public void mousePressed(MouseEvent e) {
		if ( e.getX() > canvas.getBounds().width || e.getY() > canvas.getBounds().height ) {
			e.consume();
			return;
		}
		if (figure == null) {
			anchorX = e.getX();
			anchorY = e.getY();
			super.mousePressed(e);
		} else
			e.consume();
	}
	/**
	 * Called if the mouse is released.
	 * By default, if the mouse went up the same place it went down, 
	 * remove the figure as it is probably bogus.
	 * In addition, call the superclass to get inherited behavior.
	 *
	 * @param e the event 
	 */
	public void mouseReleased(MouseEvent e) {
		if ((getX(e) == anchorX) && (getY(e) == anchorY) && (figure != null))
			canvas.removeFigure(figure);
		super.mouseReleased(e);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)TopHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the top of a figure
 * and reshapes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class TopHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its top.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public TopHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,0.5,0.0);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Reshape the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();
		int legitY = Math.min(getY(evt), figure.getBottom() - 1);
		reshape(bounds.x, legitY, bounds.width, bounds.height - (legitY - bounds.y));
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)TopLeftHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the top left of a figure
 * and reshapes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class TopLeftHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its top-left.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public TopLeftHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,0.0,0.0);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Reshape the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();
		int legitX = Math.min(getX(evt), figure.getRight() - 1);
		int legitY = Math.min(getY(evt), figure.getBottom() - 1);
		reshape(legitX, legitY, bounds.width - (legitX - bounds.x), bounds.height - (legitY - bounds.y));
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet.shapes;

/**
 * @(#)TopRightHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.MouseEvent;

/**
 * This class provides a handle that shows up at the top right of a figure
 * and reshapes the figure accordingly as it is dragged.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class TopRightHandle extends BoundsHandle {

	/** 
	 * Constructs and initializes a new instance of a handle which can 
	 * affect the figure's bounds by changing its top-right.
	 *
	 * @param figure the figure whose bounds we may wish to change.
	 */
	public TopRightHandle(Figure figure) {
		super(figure);
	}
	/** 
	 * Answer the default/initial locator.
	 * 
	 * @param figure the figure
	 * @return	the default/initial Locator
	 */
	protected Locator defaultLocator(Figure figure) {
		return new FigureRelativePoint(figure,1.0,0.0);
	}
	/**
	 * Called if the mouse is dragged (the mouse button is down).
	 * Reshape the figure as appropriate.
	 *
	 * @param evt the event
	 */
	public void mouseDragged(MouseEvent evt) {
		Rectangle bounds = figure.getBounds();
		int legitY = Math.min(getY(evt), figure.getBottom() - 1);
		reshape(bounds.x, legitY, Math.max(getX(evt) - bounds.x, 1), bounds.height - (legitY - bounds.y));
		super.mouseDragged(evt);
	}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)StringHolder.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
/**
 * When implemented, this interface allows implementing classes to serve a String.
 *
 * @version 	1.1.6, 12/28/98
 */
 
public interface StringHolder {

	/** 
	 * Answer the string the receiver is holding.
	 * 
	 * @return	the String the receiver is holding
	 */
	public String getString();
	/** 
	 * Set the String the receiver is holding.
	 * 
	 * @param string the String to hold
	 */
	public void setString(String string);
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)TC_DrawletFunctionalTest.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.awt.*;
import java.beans.*;
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.shapes.polygons.*;
import com.rolemodelsoft.drawlet.shapes.lines.*;
import com.rolemodelsoft.drawlet.text.*;

public class TC_DrawletFunctionalTest extends junit.framework.TestCase {
	//DrawingCanvas canvas;
	Figure rectangle;
	final Rectangle rectangleStart = new Rectangle(80, 200, 87, 34);

	Figure triangle;
	final int triangleX[] = {5, 5, 20};
	final int triangleY[] = {5, 20, 13};
	final Point triangleLocation = new Point(5, 5);

	Figure pentagon;
	final int pentagonX[] = {100, 150, 220, 300, 160};
	final int pentagonY[] = {50, 75, 68, 49, 25};
	final Point pentagonLocation = new Point(100, 25);

	LineFigure line1;
	LineFigure line2;

	TextLabel text;


public TC_DrawletFunctionalTest(String name) {
	super(name);
}
	/**
	 * Sets up the tests.
	 */
	public void setUp() {
		// Create the triangle
		triangle = new PolygonShape( new Polygon( triangleX, triangleY, triangleX.length ) );

		// Create the rectangle
		rectangle = new RectangleShape();
		rectangle.setBounds( rectangleStart.x, rectangleStart.y, rectangleStart.width, rectangleStart.height );
		
		// Create the polygon
		pentagon = new PolygonShape( new Polygon( pentagonX, pentagonY, pentagonX.length ) );

		// Get the shape locators, for attaching the lines
		Locator triangleLoc = triangle.requestConnection( null, 0, 0 );
		Locator rectangleLoc = rectangle.requestConnection( null, 0, 0 );
		Locator pentagonLoc = pentagon.requestConnection( null, 0, 0 );

		// Create the first line, from the triangle to the rectangle
		line1 = new ConnectingLine( triangleLoc, rectangleLoc );

		// Get the locators for the new line, to attach the second line
		Locator lineLoc1 = line1.getLocator(0);
		Locator lineLoc2 = line1.getLocator(1);

		// Get the midpoint of the first line
		Point linePoint = new Point( ( Math.abs( lineLoc1.x() - lineLoc2.x() ) / 2 ) + Math.min( lineLoc1.x(), lineLoc2.x() ),
						( Math.abs( lineLoc1.y() - lineLoc2.y() ) / 2 ) + Math.min( lineLoc1.y(), lineLoc2.y() ) );

		// Create the second line, b/t the pentagon and the midpoint of the first line
		line2 = new ConnectingLine( 0, 0, 1, 1 );
		line2.setLocator( 0, line1.requestConnection( line2, linePoint.x, linePoint.y ) );
		line2.setLocator( 1, pentagonLoc );

		// Get the locators for the second line, to attach the text label
		lineLoc1 = line2.getLocator(0);
		lineLoc2 = line2.getLocator(1);

		// Get the midpoint of the second line
		linePoint.x = ( Math.abs( lineLoc1.x() - lineLoc2.x() ) / 2 ) + Math.min( lineLoc1.x(), lineLoc2.x() );
		linePoint.y = ( Math.abs( lineLoc1.y() - lineLoc2.y() ) / 2 ) + Math.min( lineLoc1.y(), lineLoc2.y() );

		// Attach a text label to the second line
		text = new TextLabel( "Test Label" );
		text.move( line2.requestConnection( text, linePoint.x, linePoint.y ) );
	}
	/**
	 * Tests to make sure line creation is working properly.
	 */
	public void testLineCreation() {
		line2 = null;

		Locator Loc1 = line1.getLocator(0);
		Locator Loc2 = line1.getLocator(1);
		Locator pentagonLoc = pentagon.getLocator();

		line2 = new ConnectingLine( 0, 0, 1, 1 );
		assert( "line2 is null", line2 != null );
		assert( "line2 is not a ConnectingLine", line2 instanceof ConnectingLine );

		assert( "line1 is null", line1 != null );

		Point linePoint = new Point( ( Math.abs( Loc1.x() - Loc2.x() ) / 2) + Math.min( Loc1.x(), Loc2.x() ),
						( Math.abs( Loc1.y() - Loc2.y() ) / 2 ) + Math.min( Loc1.y(), Loc2.y() ) );
		assert( "linePoint is null", linePoint != null );
		assert( "linePoint is not within line1;\n\tlinePoint: " + linePoint.toString() +
			"\n\tLoc1: [x=" + Integer.toString( Loc1.x() ) + ", y=" + Integer.toString( Loc1.y() ) + "]" +
			"\n\tLoc2: [x=" + Integer.toString( Loc2.x() ) + ", y=" + Integer.toString( Loc2.y() ) + "]",
				line1.contains( linePoint.x, linePoint.y ) );

		Locator lineLoc = line1.requestConnection( line2, linePoint.x, linePoint.y );
		assert( "lineLoc is null; " + linePoint.toString(), lineLoc != null );

		try {
			line2.addLocator( 0, pentagonLoc );
		}
		catch ( Exception e ) {
			assert( "Adding locator 0 to line2 failed; Exception: " + e.toString(), false );
		}

		try {
			line2.addLocator( 1, lineLoc );
		}
		catch ( Exception e ) {
			assert( "Adding locator 1 to line2 failed; Exception: " + e.toString(), false );
		}
	}
	/**
	 * Tests to see if the proper events are propagated when the triangle is
	 * moved.
	 */
	public void testMoveTriangle() {
	PropertyChangeRevealer triangleListener = new PropertyChangeRevealer();

	triangle.addPropertyChangeListener( triangleListener );

	triangle.translate( 5, 5 );

	assert( "There was no PropertyChangeEvent propagated", triangleListener.getPropertyChangeEvent() != null );
	}
	/**
	 * Tests to see if the pentagon is in the proper location.
	 */
	public void testPentagonLocation() {
	Point actualPentagonLocation = new Point( pentagon.getLocator().x(), pentagon.getLocator().y() );
	assertEquals( actualPentagonLocation, pentagonLocation );
	}
	/**
	 * Tests to see if the rectangle is in the proper location.
	 */
	public void testRectangleLocation() {
	Point actualRectangleLocation = new Point( rectangle.getLocator().x(), rectangle.getLocator().y() );
	Point assumedRectangleLocation = new Point( rectangleStart.x, rectangleStart.y );
	assertEquals( actualRectangleLocation, assumedRectangleLocation );
	}
	/**
	 * Tests to see if the triangle is in the proper location.
	 */
	public void testTriangleLocation() {
	Point actualTriangleLocation = new Point( triangle.getLocator().x(), triangle.getLocator().y() );
	assertEquals( actualTriangleLocation, triangleLocation );
	}
}
package com.rolemodelsoft.drawlet.text;

/**
 * @(#)LabelEditHandle.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;
import java.awt.event.*;

/**
 * A handle one can use to edit labels.
 * Currently this is rather primitive as it does not allow for insert and 
 * replacing of a subset of text... this should be a temporary limitation.
 * This would typically be invoked as an editTool (see Figure), but could
 * be used in other contexts.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class LabelEditHandle extends CanvasHandle {

	/**
	 * The figure whose label we are editing.  
	 * Note the figure must also implement LabelHolder.
	 */
	protected Figure figure;

	/**
	 * The local copy of the string being edited.
	 */
	protected String string = "";

	/**
	 * Answer a new instance prepared to edit the figure.
	 *
	 * @param figure the figure whose label is to be edited.
	 * @exception IllegalArgumentException If the figure does not implement LabelHolder interface.
	 */
	public LabelEditHandle(Figure figure) {
	if (!(figure instanceof LabelHolder))
		throw new IllegalArgumentException("Figure must also implement LabelHolder interface"); 
	this.figure = figure;
	}
	/** 
	 * Returns the current bounds of this handle.
	 *
	 * @return a Rectangle representing the current bounds of this handle.
	 */
	public Rectangle getBounds()  {
		Rectangle figureBounds = figure.getBounds();
		return new Rectangle(figureBounds.x - 3, figureBounds.y - 3, figureBounds.width + 6, figureBounds.height + 6);
	}
	/** 
	 * Returns the current bounds of the label.
	 *
	 * @return a Rectangle representing the current bounds of the label.
	 */
	public Rectangle getLabelBounds()  {
		return ((LabelHolder)figure).getLabelBounds();
	}
	/**
	 * Answer the DrawingStyle to use when displaying.
	 *
	 * @return the DrawingStyle to use when displaying.
	 */
	protected DrawingStyle getStyle() {
		if (canvas == null)
			return figure.getStyle();
		return canvas.getStyle();
	}
	/**
	 * Called if a character is typed.
	 * Edit the text.  For now this just means adding to what has been typed
	 * so far, or backspacing over it.  Should allow text insertion/replacement
	 * in future.
	 *
	 * @param evt the event
	 */
	public void keyTyped(KeyEvent evt) {
		// Filter out Ctrl or Alt and Function keys
		if (evt.isControlDown() || evt.isMetaDown() || evt.isAltDown() || evt.isActionKey()) {
			return;
		}
		Rectangle oldBounds = getBounds();
		char key = evt.getKeyChar();
		switch (key) {
			case '\b' :
				int length = string.length();
				if (length < 2)
					string = "";
				else
					string = string.substring(0, length - 1);
				break;
			case 127 : // delete key
			//			deleteSelections();
				break;
			case 10 : // Enter key
				if (!evt.isShiftDown()) {
					finished();
					evt.consume();
					return;
				}
			default :
				string = string + key;
		}
	 	((StringHolder) figure).setString(string);
		canvas.repaint(getBounds().union(oldBounds));
		evt.consume();
	}
	/**
	 * Called if the mouse is pressed.
	 * If outside our bounds, give up control and pass event on to underlying tool.
	 * Inside, do nothing.  In the future this should probably denote a type
	 * of "cursor" placement for specific editing.
	 *
	 * @param evt the event 
	 */
	public void mousePressed(MouseEvent evt) {
		if (!getBounds().contains(getX(evt), getY(evt))) {
			finished();
			canvas.getTool().mousePressed(evt);
			return;
		}
		// move the cursor
		evt.consume();
	}
	/**
	 * Called if the mouse goes up.
	 * Currently, do nothing.  In the future this should probably denote a
	 * block of text as selected for edit.
	 *
	 * @param evt the event.
	 */
	public void mouseReleased(MouseEvent evt) {
		evt.consume();
	}
	/** 
	 * Paints the receiver.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g)  {
		Color oldColor = g.getColor();
		DrawingStyle myStyle = getStyle();
		DrawingStyle figureStyle = figure.getStyle();
		DrawingStyle myFigureStyle = (DrawingStyle)figureStyle.duplicate();
		g.setColor(myStyle.getHighlightColor());
		Rectangle myBounds = getBounds();
		g.fillRect(myBounds.x, myBounds.y, myBounds.width, myBounds.height);
		Rectangle innerBounds = getLabelBounds();
		if (((StringHolder)figure).getString().equals(string)) {
			// clearRect() doesn't work as expected on Netscape... 
			// fill it explicitly instead
		//	g.clearRect(innerBounds.x, innerBounds.y, innerBounds.width, innerBounds.height);
			g.setColor(myStyle.getBackgroundColor());
			g.fillRect(innerBounds.x, innerBounds.y, innerBounds.width, innerBounds.height);
			g.setColor(oldColor);
		} else {
			g.setColor(myStyle.getSelectionBackgroundColor());
			g.fillRect(innerBounds.x, innerBounds.y, innerBounds.width, innerBounds.height);
			g.setColor(myStyle.getSelectionForegroundColor());
			myFigureStyle.setBackgroundColor(myStyle.getSelectionBackgroundColor());
			myFigureStyle.setForegroundColor(myStyle.getSelectionForegroundColor());
			myFigureStyle.setTextColor(myStyle.getSelectionForegroundColor());
		}
		figure.setStyle(myFigureStyle);
		figure.paint(g);
		figure.setStyle(figureStyle);
	}
	/**
	 * Release control of the canvas and clean up if necessary.
	 * Since this is a public method,
	 * don't assume the receiver actually has control.
	 *
	 * @param canvas the canvas which the receiver is to release control
	 */
	public void releaseControl(DrawingCanvas canvas) {
		canvas.removeHandle(this);
		super.releaseControl(canvas);
	}
	/**  
	 * Make the handle be the event handler for the canvas.
	 * Note, once it takes control, it is obligated to return 
	 * at a future point in time.
	 *
	 * @param x the x coordinate 
	 * @param y the y coordinate
	 */
	public void takeControl(DrawingCanvas canvas) {
		super.takeControl(canvas);
		canvas.addHandle(this);
	}
}
package com.rolemodelsoft.drawlet.text;

/**
 * @(#)LabelTool.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import java.awt.*;

/**
 * This tool produces text labels.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class LabelTool extends ConstructionTool {

	/** 
	 * Constructs and initializes a new instance of a tool to create labels
	 * on a DrawingCanvas
	 *
	 * @param canvas the canvas on which to place labels.
	 */
	public LabelTool(DrawingCanvas canvas) {
		this.canvas = canvas;
	}
   /**
	 * Create and answer a new Figure.
	 *
	 * @param x the x coordinate.
	 * @param y the y coordinate.
	 * @return a new Figure.
	 */
	protected Figure basicNewFigure(int x, int y)  {
		Figure newFigure = new TextLabel(new Font("Courier",Font.BOLD,24));
		newFigure.move(x,y);
		return newFigure;
	}
}
package com.rolemodelsoft.drawlet.text;

/**
 * @(#)TS_TextLabel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.text.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.util.*;
import junit.ui.*;
import junit.framework.*;
import java.awt.*;

public class TC_TextLabel extends TC_AbstractFigure {
	protected TextLabel label;
	protected BasicStringRenderer renderer;
/**
 * TextLabelTest constructor comment.
 * @param name java.lang.String
 */
public TC_TextLabel(String name) {
	super(name);
}
/**
 * Sets up the fixture, for example, open a network connection.
 * This method is called before a test is executed.
 */
public void setUp() 
{
	figure = new TextLabel();
	label = (TextLabel) figure;
	figure.setBounds( 10, 10, 10, 10 );
	propertyRevealer = new PropertyChangeRevealer();
	locationRevealer = new RelatedLocationRevealer();
	renderer = new BasicStringRenderer( label.getString(), label.getFont() );
}
/**
 * Test to make sure the contains( Figure ) method is
 * functioning properly.
 */
public void testContainsFigure() 
{
	Figure testFigure = new RectangleShape( 11, 11, 8, 8 );
	assert( "Figure does not contain the figure", figure.contains( testFigure ) );
	testFigure.setBounds( 4, 4, 12, 12 );
	assert( "Figure contains the figure", ! figure.contains( testFigure ) );
}
/**
 * Test to make sure the contains( int, int ) method is
 * functioning properly.
 */
public void testContainsIntInt() 
{
	assert( "Figure does not contain the point", figure.contains( 11, 11 ) );
	assert( "Figure contains the point", ! figure.contains( 30, 30 ) );
}
/**
 * Test to make sure the contains( Rectangle ) method is
 * functioning properly.
 */
public void testContainsRectangle() 
{
	Rectangle testRect = new Rectangle( 11, 11, 8, 8 );
	assert( "Figure does not contain the rectangle", figure.contains( testRect ) );
	testRect.setBounds( 4, 4, 12, 12 );
	assert( "Figure contains the rectangle", ! figure.contains( testRect ) );
}
/**
 * Test to make sure that the edit tool returned is null.
 * Subclasses should override if they return expect a
 * different result.
 */
public void testEditTool() 
{
	assert( "The edit tool returned was null.", figure.editTool( 0, 0 ) != null );
	assert( "The edit tool was not a LabelEditHandle.", figure.editTool( 0, 0 ) instanceof LabelEditHandle );
}
/**
 * Test to make sure the bottom is properly returned.
 */
public void testGetBottom() 
{
	assertEquals( "The int returned does not correspond to the expected bottommost coordinate of the figure", renderer.getStringHeight() + 10, figure.getBottom() );
}
/**
 * Test to make sure the height of the figure is properly
 * returned.
 */
public void testGetHeight() 
{
	assertEquals( "The height was not what was expected.", renderer.getStringHeight(), figure.getHeight() );
}
/**
 * Test to make sure the leftmost coordinate of the figure
 * is properly returned.
 */
public void testGetLeft() 
{
	assertEquals( "The int returned does not correspond to the expected leftmost coordinate of the figure", 10, figure.getLeft() );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testGetLocator() 
{
	assert( "There was no locator returned.", figure.getLocator() != null );
	assertEquals( "The x of the locator returned does not correspond to the left side of this label.", 13, figure.getLocator().x() );
	assertEquals( "The y of the locator returned does not correspond to the top of this label.", 10, figure.getLocator().y() );
}
/**
 * Test to make sure the rightmost coordinate of the figure
 * is properly returned.
 */
public void testGetRight() 
{
	assertEquals( "The int returned does not correspond to the expected rightmost coordinate of the figure", renderer.getStringWidth() + 6 + 10, figure.getRight() );
}
/**
 * Test to make sure the size of the figure is properly
 * returned.
 */
public void testGetSize() 
{
	assertEquals( "The size was not what was expected.", new Dimension( renderer.getStringWidth() + 6, renderer.getStringHeight() ), figure.getSize() );
}
/**
 * Test to make sure the top is properly returned.
 */
public void testGetTop() 
{
	assertEquals( "The int returned does not correspond to the expected topmost coordinate of the figure", 10, figure.getTop() );
}
/**
 * Test to make sure the width of the figure is properly
 * returned.
 */
public void testGetWidth() 
{
	assertEquals( "The width was not what was expected.", renderer.getStringWidth() + 6, figure.getWidth() );
}
/**
 * Test to make sure the intersects( Figure ) method is
 * functioning properly.
 */
public void testIntersectsFigure() 
{
	Figure testFigure = new RectangleShape( 11, 11, 2, 2 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 7, 7, 5, 5 );
	assert( "Figure does not intersect the figure", figure.intersects( testFigure ) );
	testFigure.setBounds( 40, 40, 6, 6 );
	assert( "Figure intersects the figure", ! figure.intersects( testFigure ) );
}
/**
 * Test to make sure the intersects( Rectangle ) method is
 * functioning properly.
 */
public void testIntersectsRectangle() 
{
	Rectangle testRect = new Rectangle( 11, 11, 2, 2 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 7, 7, 5, 5 );
	assert( "Figure does not intersect the rectangle", figure.intersects( testRect ) );
	testRect.setBounds( 40, 40, 6, 6 );
	assert( "Figure intersects the rectangle", ! figure.intersects( testRect ) );
}
/**
 * Test to make sure the isWithin( Figure ) method is
 * functioning properly.
 */
public void testIsWithinFigure() 
{
	Figure testFigure = new RectangleShape( 9, 9, 30, 30 );
	assert( "Figure isn't within the figure", figure.isWithin( testFigure ) );
	testFigure.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the figure", ! figure.isWithin( testFigure ) );
}
/**
 * Test to make sure the isWithin( Rectangle ) method is
 * functioning properly.
 */
public void testIsWithinRectangle() 
{
	Rectangle testRect = new Rectangle( 9, 9, 30, 30 );
	assert( "Figure isn't within the rectangle", figure.isWithin( testRect ) );
	testRect.setBounds( 5, 5, 2, 2 );
	assert( "Figure is within the rectangle", ! figure.isWithin( testRect ) );
}
/**
 * Test to make sure move( int, int ) works correctly.
 */
public void testMoveIntInt() 
{
	figure.move( 20, 20 );
	assert( "The figure did not move properly", figure.getLeft() == 20 && figure.getTop() == 20 );
}
/**
 * Test to make sure the locator is properly returned.
 */
public void testRequestConnection() 
{
	Locator locator = figure.requestConnection( new RectangleShape(), 0, 0 );
	assert( "There was no locator returned.", locator != null );
	assert( "The locator is incorrect.", locator.x() == 0 && locator.y() == 0 );
	if ( locator instanceof FigureRelativePoint ) {
		FigureRelativePoint relativeLocator = (FigureRelativePoint) locator;
		assert( "The locator returned does not have the figure as its owner.", relativeLocator.getFigure() == figure );
		figure.translate( 1, 1 );
		assert( "The locator did not update when the figure moved.", relativeLocator.x() == 1 && relativeLocator.y() == 1 );
	} else {
		assert( "The locator returned is not a FigureRelativePoint. If that is OK, you need to override this test method.", false );
	}
}
/**
 * Test to make sure the bounds are properly set.
 */
public void testSetBounds()
{
	label.setBounds( 20, 20, 100, 100 );
	assertEquals( "The figure's left side is incorrect.", 20, figure.getLeft() );
	assertEquals( "The figure's right side is incorrect.", 20 + renderer.getStringWidth() + 6, figure.getRight() );
	assertEquals( "The figure's top side is incorrect.", 20, figure.getTop() );
	assertEquals( "The figure's bottom side is incorrect.", 20 + renderer.getStringHeight(), figure.getBottom() );
	
}
/**
 * Test to make sure the setSize( Dimension ) method is
 * working properly.
 */
public void testSetSizeDimension()
{
	figure.setSize( new Dimension( 100, 100 ) );
	assertEquals( "The figure's width is incorrect.", renderer.getStringWidth() + 6, figure.getWidth() );
	assertEquals( "The figure's height is incorrect.", renderer.getStringHeight(), figure.getHeight() );
}
/**
 * Test to make sure the setSize( int, int ) method is
 * working properly.
 */
public void testSetSizeIntInt()
{
	figure.setSize( 100, 100 );
	assertEquals( "The figure's width is incorrect.", renderer.getStringWidth() + 6, figure.getWidth() );
	assertEquals( "The figure's height is incorrect.", renderer.getStringHeight(), figure.getHeight() );
}
/**
 * Test to make sure translate works correctly.
 */
public void testTranslate() 
{
	translateByExpecting( 5, 5 , 15, 15 );
	translateByExpecting( -5, -5, 10, 10 );
}
}
package com.rolemodelsoft.drawlet.text;

/**
 * @(#)TextLabel.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996-1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.util.*;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * This provides a basic implementation of a figure which displays simple text.
 * It is assumed that this figure may be connected to another figure
 * via its locator, but it can be used as just raw text also.
 *
 * @version 	1.1.6, 12/29/98
 */
 
public class TextLabel extends AbstractFigure implements LabelHolder, RelatedLocationListener {
	static final long serialVersionUID = 2853476878632285634L;
	
	/**
	 * The string which serves as the label.
	 */
	protected String string = defaultString();

	/**
	 * The font with which to paint the label.
	 */
	protected Font font = defaultFont();

	/**
	 * The color with which to paint the label.
	 */
	protected Color textColor = defaultTextColor();

	/**
	 * The location of the label, the topLeft of where the string shows up.
	 */
	protected MovableLocator locator = defaultLocator();

	/**
	 * The renderer used to actually display the label.
	 */
	protected transient StringRenderer renderer;

	/**
	 * Since ints are not objects and zero could be valid for any cache, 
	 * we need to come up with "unset" values for each.  For width and 
	 * height, -1 would be fine, but those might be valid for top/left,
	 * as could basically any other integer.  However, Integer.MIN_VALUE 
	 * seems to be as unlikely as anything to be practical.  So, we use it.
	 */
	protected static int unset = Integer.MIN_VALUE;

	/**
	 * Used to cache the width value to calc bounds faster.
	 */
	protected int width = unset;

	/**
	 * Used to cache the height value to calc bounds faster.
	 */
	protected int height = unset;

	/**
	 * Used to cache the left value to calc bounds faster.
	 */
	protected int left = unset;

	/**
	 * Used to cache the top value to calc bounds faster.
	 */
	protected int top = unset;

	/** 
	 * Constructs a new instance of a label.
	 */
	public TextLabel() {
	}
	/** 
	 * Constructs and initializes a new instance of a label with the given Color.
	 *
	 * @param color the color with which to display the label
	 */
	public TextLabel(Color color) {
		setTextColor(color);
	}
	/** 
	 * Constructs and initializes a new instance of a label with the given Font.
	 *
	 * @param font the font with which to display the label.
	 */
	public TextLabel(Font font) {
		setFont(font);
	}
	/** 
	 * Constructs and initializes a new instance of a label with
	 * the given Font and Color.
	 *
	 * @param font the Font with which to display the label.
	 * @param color the Color with which to display the label.
	 */
	public TextLabel(Font font, Color color) {
		this(font);
		setTextColor(color);
	}
	/** 
	 * Constructs and initializes a new instance of a label with the
	 * given String.
	 *
	 * @param string the String to display as the label.
	 */
	public TextLabel(String string) {
		setString(string);
	}
	/** 
	 * Constructs and initializes a new instance of a label with the given
	 * String and Font.
	 *
	 * @param string the String to display as the label.
	 * @param font the Font with which to display the label.
	 */
	public TextLabel(String string, Font font) {
		this(string);
		setFont(font);
	}
	/** 
	 * Constructs and initializes a new instance of a label with the given
	 * String, Font, and Color.
	 *
	 * @param string the String to display as the label.
	 * @param font the Font with which to display the label.
	 * @param color the Color with which to display the label.
	 */
	public TextLabel(String string, Font font, Color color) {
		this(string, font);
		setTextColor(color);
	}
	/** 
	 * Moves the receiver to a new location.
	 *
	 * @param newLocator the Locator to base the figure's position
	 * @see #getLocator
	 * @see #move
	 */
	protected synchronized void basicMove(Locator newLocator) {
		Figure oldFigure = figureFromLocator(locator);
		Figure newFigure = figureFromLocator(newLocator);
		if (oldFigure != newFigure) {
			if (oldFigure != null)
				oldFigure.removeRelatedLocationListener(this);
			if (newFigure != null)
				newFigure.addRelatedLocationListener(this);
		}
		if (newLocator instanceof MovableLocator)
			locator = (MovableLocator) newLocator;
		else
			locator = new RelativePoint(newLocator, 0, 0);
	}
	/** 
	 * Moves the receiver in the x and y direction.
	 * Subclasses should probably provide synchronized versions if they're 
	 * modifying attributes of the receiver.
	 *
	 * @param x amount to move in the x direction
	 * @param y amount to move in the y direction
	 * @see #getLocator
	 */
	protected void basicTranslate(int x, int y)  {
		locator.translate(x,y);
	}
	/**
	 * Answer the default/initial value for the Font.
	 *
	 * @return the default/initial Font.
	 */
	protected Font defaultFont()  {
		// it would be nice if there was a simple way to get the system font
		Font newFont; // = Font.getFont("font.default");
		//	if (newFont == null)
		newFont = new Font("TimesRoman",Font.PLAIN,12);
		return newFont;
	}
	/**
	 * Answer the default/initial value for the Locator.
	 *
	 * @return the default/initial MovableLocator.
	 */
	protected MovableLocator defaultLocator() {
		return new DrawingPoint(0,0);
	}
	/**
	 * Answer the default/initial value for the String.
	 *
	 * @return the default/initial String.
	 */
	protected String defaultString() {
		return "label";
	}
	/**
	 * Answer the default/initial value for the color to paint the text.
	 *
	 * @return the default/initial Color for painting text.
	 */
	protected Color defaultTextColor() {
		return Color.black;
	}
	/**
	 * Duplicates the receiver and places the duplicate in the given Hashtable.
	 *
	 * @param duplicates the Hashtable to put the duplicate in.
	 * @return the duplicate.
	 */
	public synchronized Object duplicateIn(Hashtable duplicates) {
		TextLabel myDuplicate = (TextLabel)super.duplicateIn(duplicates);
		myDuplicate.locator = (MovableLocator)locator.duplicateIn(duplicates);
		myDuplicate.renderer = null;
		myDuplicate.resetBoundsCache();
		return myDuplicate;
	}
	/** 
	 * Answers a Handle that will provide editing capabilities on the receiver.
	 *
	 * @param x the x coordinate to potentially begin editing.
	 * @param y the y coordinate to potentially begin editing.
	 */
	public Handle editTool(int x, int y)  {
		return new LabelEditHandle(this);
	}
	/**
	 * Remove any dependence on the Figure.
	 *
	 * @param figure the Figure to disassociate.
	 */
	protected synchronized void freeFromFigure(Figure figure)  {
		if (figureFromLocator(locator) == figure) 
			locator = new DrawingPoint(locator.x(),locator.y());
	}
	/** 
	 * Returns the current bounds of the receiver.
	 *
	 * @return a Rectangle representing the current bounds.
	 */
	public Rectangle getBounds()  {
		return new Rectangle(getLeft(), getTop(), getWidth(), getHeight());
	}
	/** 
	 * Answer the Font with which the receiver paints.
	 *
	 * @return the Font in use.
	 */
	public Font getFont()  {
		return font;
	}
	/** 
	 * Answer the Handles associated with the receiver.
	 *
	 * @return an array of Handles.
	 */
	public Handle[] getHandles() {
		Handle handles[] = new Handle[1];
		handles[0] = new LocatorConnectionHandle(this);
		return handles;
	}
	/** 
	 * Returns the height of the receiver.
	 *
	 * @return an integer representing the height of the receiver.
	 */
	public int getHeight() {
		if (height == unset)
			height = getRenderer().getStringHeight();
		return height;
	}
	/** 
	 * Returns the current bounds of the label.
	 *
	 * @return an integer representing the bounds of the label.
	 */
	public Rectangle getLabelBounds()  {
		return new Rectangle(getLabelLeft(), getLabelTop(), getLabelWidth(), getLabelHeight());
	}
	/** 
	 * Returns the height of the label of this figure.
	 *
	 * @return an integer representing the height of the label.
	 */
	protected int getLabelHeight() {
		return getHeight();
	}
	/** 
	 * Returns the leftmost coordinate of the label of this figure.
	 *
	 * @return an integer representing the leftmost coordinate of the label.
	 */
	protected int getLabelLeft() {
		return getLeft() + xMargin();
	}
	/** 
	 * Returns the topmost coordinate of the label of this figure.
	 *
	 * @return an integer representing the topmost coordinate of the label.
	 */
	protected int getLabelTop() {
		return getTop();
	}
	/** 
	 * Returns the width of the label of this figure.
	 *
	 * @return an integer representing the width of the label.
	 */
	protected int getLabelWidth() {
		return getWidth() - (2 * xMargin());
	}
	/** 
	 * Returns the leftmost coordinate of the receiver.
	 *
	 * @return an integer representing the leftmost coordinate of the receiver.
	 */
	public int getLeft() {
		if (left == unset)
			left = locator.x() - xMargin();
		return left;
	}
	/** 
	 * Returns the current locator of the receiver.
	 * Answer a duplicate for security.
	 *
	 * @return a duplicate of the current Locator of the receiver.
	 */
	public Locator getLocator() {
		return (Locator)locator.duplicate();
	}
	/**
	 * Answer the renderer to use to display the label.
	 *
	 * @return the StringRenderer used to display the label.
	 */
	protected StringRenderer getRenderer() {
		if (renderer == null) {
			renderer = new BasicStringRenderer(getString(), getFont());
		}
		return renderer;
	}
	/** 
	 * Answer the String the receiver paints.
	 *
	 * @return the String the receiver paints.
	 */
	public String getString()  {
		return string;
	}
	/** 
	 * Answer the DrawingStyle which defines how to paint the receiver.
	 *
	 * @return the DrawingStyle defining how to paint the receiver.
	 */
	public DrawingStyle getStyle()  {
		DrawingStyle style = super.getStyle();
		style.setTextColor(getTextColor());
		style.setFont(getFont());
		return style;
	}
	/**
	 * Answer the Color to use when drawing text.
	 *
	 * @return the Color to use when drawing text.
	 */
	public Color getTextColor() {
		return textColor;
	}
	/** 
	 * Returns the topmost coordinate of the receiver.
	 *
	 * @return an integer representing the topmost coordinate of the receiver.
	 */
	public int getTop() {
		if (top == unset)
			top = locator.y();
		return top;
	}
	/** 
	 * Returns the width of the receiver.
	 *
	 * @return an integer representing the width of the receiver.
	 */
	public int getWidth() {
		if (width == unset)
			width = getRenderer().getStringWidth() + (2 * xMargin());
		return width;
	}
	/** 
	 * Answers whether the receiver is listening to the figure directly 
	 * or indirectly (via chain of listeners).
	 *
	 * @param figure the Figure to test.
	 * @return a boolean value of true if we are listening; false otherwise.
	 */
	protected boolean isListening(Figure figure) {
		for (Enumeration e = figure.relatedLocationListeners(); e.hasMoreElements();) {
			RelatedLocationListener listener = (RelatedLocationListener) e.nextElement();
			if (listener == this)
				return true;
			else
				if (listener instanceof Figure)
					if (isListening((Figure) listener))
						return true;
		}
		return false;
	}
	/** 
	 * Answers whether the receiver is obsolete
	 * True if some event has happened that makes this a meaningless object.
	 *
	 * @return boolean value of true if the receiver is obselete; false otherwise.
	 */
	public boolean isObsolete() {
		return string.length() == 0;
	}
	/**
	 * Called when the location of something the receiver is listening to has changed.
	 *
	 * @param event the event describing the change.
	 */
	public void locationChanged(PropertyChangeEvent event) {
		updateLocation();
	}
	/** 
	 * Moves the receiver to a new location.
	 * This is a TemplateMethod with hooks:
	 * 	resetLocationCache()
	 * 	basicMove()
	 *  changedLocation()
	 *
	 * @param locator the Locator which identifies the desired x, y coordinates.
	 * @see #getLocator
	 * @see #basicMove
	 */
	public void move(Locator locator) {
		Point oldLocation = getLocation();
		resetLocationCache();
		basicMove(locator);
		changedLocation(oldLocation);
	}
	/** 
	 * Paints the receiver.
	 *
	 * @param g the specified Graphics window.
	 */
	public void paint(Graphics g)  {
		g.setColor(textColor);
		g.setFont(font);
		getRenderer().paint(g, getLabelLeft(), getLabelTop());
	}
	/**
	 * After a series of Figures are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes it might like to reconcile.
	 * 
	 * In this case, remove any dependency on any figures defining original 
	 * points.  If there is an available duplicate corresponding to the original, 
	 * use it as the original was used.  If not, convert it to a non-Figure-
	 * dependent point.  Copy remaining points as they are.
	 *
	 * @param duplicates a Hashtable with originals as keys and duplicates as elements.
	 */
	public void postDuplicate(Hashtable duplicates) {
		super.postDuplicate(duplicates);
		Figure myFigure = figureFromLocator(locator);
		if (myFigure != null) {
			if (!duplicates.containsKey(myFigure))
				locator = new DrawingPoint(locator.x(),locator.y());
		}
		locator.postDuplicate(duplicates);
	}
	/**
	 * Called when the relation of something the receiver is listening to has changed.
	 *
	 * @param event the event describing the change.
	 */
	public void relationChanged(PropertyChangeEvent event) {
		freeFromFigure((Figure)event.getSource());
	}
	/** 
	 * Answers a Locator corresponding to a significant point on the receiver 
	 * that will serve as a connection to the requesting Figure.
	 *
	 * @param x the x coordinate of the requested locator
	 * @param y the y coordinate of the requested locator
	 */
	public Locator requestConnection(Figure requestor, int x, int y) {
		// make sure we aren't already connected to the locator 
		// which is trying to connect to us 
		if (isListening(requestor))
			return null;
		return locatorAt(x,y);
	}
	/**
	 * Flush caches with respect to determining bounds. 
	 */
	protected void resetBoundsCache() {
		resetLocationCache();
		resetSizeCache();
	}
	/**
	 * Flush caches with respect to determining location
	 */
	protected void resetLocationCache() {
		left = unset;
		top = unset;
	}
	/**
	 * Flush caches with respect to determining size
	 */
	protected void resetSizeCache() {
		width = unset;
		height = unset;
	}
	/** 
	 * Set the Font with which to paint text
	 * This is a TemplateMethod with hooks:
	 * 	resetBoundsCache()
	 *  changedShape()
	 *
	 * @param newFont the Font to use for text.
	 */
	public void setFont(Font newFont)  {
		Rectangle oldBounds = getBounds();
		resetBoundsCache();
		font = newFont;
		getRenderer().setFont(font);
		changedShape(oldBounds);
	}
	/** 
	 * Set the string the figure paints.
	 * This is a TemplateMethod with hooks:
	 * 	resetSizeCache()
	 *  changedSize()
	 *
	 * @param newString the string to paint.
	 */
	public void setString(String newString)  {
		Dimension oldSize = getSize();
		resetSizeCache();
		string = newString;
		getRenderer().setString(string);
		changedSize(oldSize);
	}
	/** 
	 * Set the style defining how to paint the figure.
	 *
	 * @param style the specified DrawingStyle.
	 */
	public void setStyle(DrawingStyle style) {
		DrawingStyle oldStyle = getStyle();
		if (style != null) {
			setFont(style.getFont());
			setTextColor(style.getTextColor());
		}
		firePropertyChange(STYLE_PROPERTY, oldStyle, style);
	}
	/**
	 * Set the Color to use when drawing text.
	 *
	 * @param color the Color.
	 */
	public void setTextColor(Color color) {
		Color oldColor = textColor;
		textColor = color;
		firePropertyChange(TEXT_COLOR_PROPERTY, oldColor, color);
	}
	/**
	 * Called when the shape of something the receiver is listening to has changed.
	 *
	 * @param event the event describing the change.
	 */
	public void shapeChanged(PropertyChangeEvent event) {
		updateLocation();
	}
	/**
	 * Called when the size of something the receiver is listening to has changed.
	 *
	 * @param event the event describing the change.
	 */
	public void sizeChanged(PropertyChangeEvent event) {
		updateLocation();
	}
	/**
	 * The Figure has notified the receiver of a change.
	 * Assume our location has changed due to some movement/reshaping
	 * of the Figure.
	 */
	protected void updateLocation() {
		Point oldLocation = getLocation();
		resetLocationCache();
		changedLocation(oldLocation);
	}
	/**
	 * Answer the margin to use in the x direction.
	 *
	 * @return an integer representing the margin to use in the x direction.
	 */
	protected int xMargin()  {
		return 3;
	}
}
package com.rolemodelsoft.drawlet.text;

/**
 * @(#)TS_TextTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

public class TS_TextTests extends junit.framework.TestCase {

public TS_TextTests(String name) {
	super(name);
}

public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}

public static TestSuite suite() {
	TestSuite suite = new TestSuite();
	
	suite.addTest(new TestSuite(TC_TextLabel.class));
	
	return suite;
}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)TS_AllDrawletTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

/**
 *This suite runs all the Drawlet tests.
 * @version 	1.1.6, 12/29/98
 */
 
public class TS_AllDrawletTests extends TestCase {
	/**
	 * @param test the test.
	 */
	public TS_AllDrawletTests( String test ) {
		super( test );
	}
/**
 * Starts the application.
 * @param args an array of command-line arguments
 */
public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}
	/**
	 * @return a Test representation of a suite of tests.
	 */
	public static Test suite() {
		TestSuite suite = new TestSuite();

		suite.addTest( TS_DrawletFunctionalTest.suite() );
		suite.addTest( TS_AllDrawletUnitTests.suite() );

		return suite;
	}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)TS_AllDrawletUnitTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.util.*;
import com.rolemodelsoft.drawlet.basics.*;
import com.rolemodelsoft.drawlet.shapes.*;
import com.rolemodelsoft.drawlet.shapes.rectangles.*;
import com.rolemodelsoft.drawlet.shapes.polygons.*;
import com.rolemodelsoft.drawlet.shapes.lines.*;
import com.rolemodelsoft.drawlet.text.*;
import com.rolemodelsoft.drawlet.shapes.ellipses.*;
import junit.framework.*;
import com.rolemodelsoft.test.*;

/**
 * @version 	1.1.6, 12/29/98
 */
 
 public class TS_AllDrawletUnitTests extends TestCase {
	/**
	 * @param test the test.
	 */
	public TS_AllDrawletUnitTests( String test ) {
		super( test );
	}
/**
 * Starts the application.
 * @param args an array of command-line arguments
 */
public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}
	/**
	 * @return a Test representation of a suite of tests.
	 */
	public static Test suite() {
		TestSuite suite = new TestSuite();

		suite.addTest(TS_InputEventHandlerTests.suite());
		suite.addTest(TS_EllipseTests.suite());
		suite.addTest(TS_LineTests.suite());
		suite.addTest(TS_PolygonTests.suite());
		suite.addTest(TS_RectangleTests.suite());
		suite.addTest(TS_TextTests.suite());
		suite.addTest(TS_UtilityTests.suite());
		
		return suite;
	}
}
package com.rolemodelsoft.drawlet;

/**
 * @(#)TS_DrawletFunctionalTest.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

/**
 * @version 1.1.6, 12/29/98
 */

public class TS_DrawletFunctionalTest extends TestCase {
	/**
	 * @param test the test.
	 */
	public TS_DrawletFunctionalTest( String test ) {
		super( test );
	}
/**
 * Starts the application.
 * @param args an array of command-line arguments
 */
public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}
	/**
	 * @return a Test representation of a suite of tests.
	 */
	public static Test suite() {
		TestSuite suite = new TestSuite();

		suite.addTest(new TestSuite(TC_DrawletFunctionalTest.class));

		return suite;
	}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)BasicObservable.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.util.Enumeration;
import java.util.Vector;
import java.io.Serializable;

/**
 * This provides basic default functionality for Observables that maintain and notify their
 * Observers.  It is an abstract class as there is nothing to observe until subclasses provide
 * some content.  Note that the list of observers is serializable.
 *
 * @version 	1.1.6, 12/30/98
 */
public abstract class BasicObservable implements Observable, Serializable {

	/**
	 * The list of Observers.  Could be null, a single Observer, or a
	 * Vector of Observers.
	 */
	protected Object observerList;

	/**
	 * Adds an observer to the observer list.
	 *
	 * @param observer the observer to be added.
	 */
	public synchronized void addObserver(Observer observer) {
		if (observerList != null) {
			if (observerList instanceof Vector) {
				if (!((Vector) observerList).contains(observer)) {
					((Vector) observerList).addElement(observer);
				}
			} else
				if (observerList != observer) {
					Vector tmp = new Vector();
					tmp.addElement(observerList);
					tmp.addElement(observer);
					observerList = tmp;
				}
		} else {
			observerList = observer;
		}
	}
	/**
	 * Deletes an observer from the observer list.
	 *
	 * @param observer the observer to be deleted
	 */
	public synchronized void deleteObserver(Observer observer) {
		if (observerList == observer) {
			observerList = null;
		} else
			if (observerList != null && observerList instanceof Vector) {
				((Vector) observerList).removeElement(observer);
			}
	}
	/**
	 * Deletes all observers from the observer list.
	 * Though this is public, caution should be used before anything other
	 * than "this" invokes it.
	 */
	public synchronized void deleteObservers() {
		observerList = null;
	}
	/** 
	 * Answers the array of current Observers.
	 *
	 * @return an array of Observers.
	 */
	public Observer[] getObservers() {
		Observer observers[] = new Observer[0];
		if (observerList != null) {
			if (observerList instanceof Vector) {
				Vector v = (Vector)observerList;
				observers = new Observer[v.size()];
				v.copyInto(observers);
	 	   	} 
			else {
				observers = new Observer[1];
				observers[0] = (Observer)observerList;
			}
		}
		return observers;
	}
	/**
	 * Notifies a particular observer that a change in the receiver occurred.
	 *
	 * @param observer the particular observer to be notified.
	 * @param arg info to pass along to the observer being notified.
	 */
	protected void notifyObserver(Object observer, Object arg)  {
		((Observer)observer).update(this, arg);
	}
	/**
	 * Notifies all observers that an observable change occurred.
	 * Though this is public, caution should be used before anything other
	 * than "this" invokes it.
	 */
	public void notifyObservers()  {
		notifyObservers(null);
	}
	/**
	 * Notifies all observers that an observable change occurs.
	 * Though this is public, caution should be used before anything other
	 * than "this" invokes it.
	 *
	 * @param arg info to pass along to those being notified.
	 */
	public synchronized void notifyObservers(Object arg) {
		if (observerList != null) {
			if (observerList instanceof Vector)
				for (Enumeration e = new ReverseVectorEnumerator((Vector) observerList); e.hasMoreElements();)
					notifyObserver(e.nextElement(), arg);
				else
					notifyObserver(observerList, arg);
		}
	}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)BasicStringComposer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * This provides basic default functionality for StringRenderers that provide
 * wrapping.  The first version doesn't handle all possibilities, but works for
 * most purposes.
 * NOTE: it is possible that individual line widths could be greater than the 
 * maxWidth as widths include calculation of ending whitespace in case someone
 * wants to do something with them.  However, no non-END_ABSORB characters 
 * should show up past the maxWidth.
 *
 * @version 	1.1.6, 12/30/98
 */
public class BasicStringComposer extends BasicStringRenderer {
	/**
	 * The maximum width to allow non-END_ABSORBing (whitespace) characters
	 * before wrapping.
	 */
	protected int maxWidth = defaultMaxWidth();

	/**
	 * The maximum height to compose. ALL_VERTICAL means continue to end
	 * NOTE: If this is set to something other than the default no characters beyond
	 * those needed to get to that height will be taken into consideration.
	 */
	protected int maxHeight = defaultMaxHeight();

	/**
	 * This keeps track of the last known x coordinate that we know to
	 * to be good (i.e. not past the maxWidth) for the last character we
	 * know we can display on the current line being composed.
	 */
	protected int goodWidth;

	/** 
	 * Answer an instance prepared to render a String.
	 *
	 * @param string the string to render
	 */
	public BasicStringComposer(String string) {
		super(string);
	}
	/** 
	 * Answer an instance prepared to render a string,
	 * wrapping after a given width.
	 *
	 * @param string the string to render
	 * @param maxWidth the width at which to wrap text to a new line
	 */
	public BasicStringComposer(String string, int maxWidth) {
		this(string);
		this.maxWidth = maxWidth;
	}
	/** 
	 * Answer an instance prepared to render a string,
	 * wrapping after a given width, and ignoring text past a given height.
	 *
	 * @param string the string to render
	 * @param maxWidth the width at which to wrap text to a new line
	 * @param maxHeight the height at which to ignore other text
	 */
	public BasicStringComposer(String string, int maxWidth, int maxHeight) {
		this(string, maxWidth);
		this.maxHeight = maxHeight;
	}
	/** 
	 * Answer an instance prepared to render a string using a particular font
	 *
	 * @param string the string to render
	 * @param font the font to use when rendering
	 */
	public BasicStringComposer(String string, Font font) {
		super(string, font);
	}
	/** 
	 * Answer an instance prepared to render a string using a particular font,
	 * wrapping after a given width.
	 *
	 * @param string the string to render
	 * @param font the font to use when rendering
	 * @param maxWidth the width at which to wrap text to a new line
	 */
	public BasicStringComposer(String string, Font font, int maxWidth) {
		this(string, font);
		this.maxWidth = maxWidth;
	}
	/** 
	 * Answer an instance prepared to render a string using a particular font,
	 * wrapping after a given width, and ignoring text past a given height.
	 *
	 * @param string the string to render
	 * @param font the font to use when rendering
	 * @param maxWidth the width at which to wrap text to a new line
	 * @param maxHeight the height at which to ignore other text
	 */
	public BasicStringComposer(String string, Font font, int maxWidth, int maxHeight) {
		this(string, font, maxWidth);
		this.maxHeight = maxHeight;
	}
	/** 
	 * Record the current settings as the end of a line.
	 * Leave the running counters alone so others may use them before preparing
	 * to start the next line.
	 * Used during composition.
	 * @see beginNextLine
	 */
	protected void addLine() {
		// everything else below should be same as super unless otherwise noted
		int newEnds[] = new int[ends.length + 1];
		int newBegins[] = new int[begins.length + 1];
		int newManuals[][] = new int[manuals.length + 1][];
		int newWidths[] = new int[widths.length + 1];
		System.arraycopy(ends, 0, newEnds, 0, ends.length);
		System.arraycopy(begins, 0, newBegins, 0, begins.length);
		System.arraycopy(manuals, 0, newManuals, 0, manuals.length);
		System.arraycopy(widths, 0, newWidths, 0, widths.length);
		newEnds[ends.length] = end;
		//this is the only line that is different
		newWidths[ends.length] = goodWidth;
		ends = newEnds;
		begins = newBegins;
		manuals = newManuals;
		widths = newWidths;
		height += getLineHeight();
	}
	/** 
	 * Set up to compose the next line.
	 * Reset the running counters and buffers which may be needed.
	 * Used during composition.  Often after addLine().
	 * @see addLine()
	 */
	protected void beginNextLine() {
		super.beginNextLine();
		goodWidth = width;
	}
	/** 
	 * Make the last/pending line into a "real one".
	 */
	protected void closeLastLine() {
		if (end == 0)
			addLine();
		else if (end == last && !isFullyComposed()) {
			// if the last character is a BREAK_BEFORE,
			// include its width
			if ((getSpecialFlags(string, end - 1) & BREAK_BEFORE) != 0) {
				adjustWidthForSpecial(string, end - 1);
				goodWidth = width;
			}
			newLine();
		}
	}
	/** 
	 * Compose the text in such a way as to produce all lines necessary
	 * to display text properly.
	 */
	protected void compose() {
		compose(maxHeight);
	}
	/** 
	 * Compose the text in such a way as to produce the lines necessary
	 * to display text between the given vertical coordinates.
	 *
	 * @param yBegin the vertical point at which we need to begin composing.
	 * @param yEnd the vertical point at which we'll stop composing.
	 */
	protected void compose(int yBegin, int yEnd) {
		composeVerticalArea(yBegin, yEnd);
		closeLastLine();
	}
	/** 
	 * Compose the text in such a way as to produce the lines necessary
	 * to display text between the given vertical coordinates.  However, 
	 * don't assume that the last character scanned is the end of a real line.
	 * It is just the end of the current line as known so far.
	 *
	 * @param yBegin the vertical point at which we need to begin composing.
	 * @param yEnd the vertical point at which we'll stop composing.
	 */
	protected void composeVerticalArea(int yBegin, int yEnd) {
		/*
		 * since we are wrapping there is no way to predict exactly what goes
		 * on the first lines, so just keep composing until we've reached the 
		 * last line needed.
		 */
		boolean breakBefore = false;
		while (end < last && (yEnd == ALL_VERTICAL || height < yEnd)) {
			if (breakBefore) //ended on BREAK_BEFORE... add width now before continuing
				adjustWidthForSpecial(string, end - 1);
			int special = scanToSpecial(string, end, last);
			if (width > maxWidth) {
				if (breakBefore) {
					end = end - 1;
					breakBefore = false;
				}
				if (end == begin) {
					/*
					 * by the first special character, we're too wide.
					 * Recursively back up from the last character 
					 * scanned and start over, making sure at least 
					 * one character gets on the line.
					 */
					int newLast = (special == NO_INDEX) ? last - 1 : special;
					if (newLast <= begin) {
						end = begin + 1;
						goodWidth = width;
						next = end;
						newLine();
					} else {
						int oldLast = last;
						width = 0;
						last = newLast;
						composeVerticalArea(yBegin, yEnd);
						last = oldLast;
					}
				} else {
					/*
					 * Since we are now past the end of a legal line,
					 * the previous scan should be the end of a line.
					 * Make it so.  The next time through the loop,
					 * We will rescan what we just scanned
					 * as the beginning of the next line.
					 */
					next = end;
					newLine();
				}
			} else {
				breakBefore = false;
				if (special == NO_INDEX) {
					goodWidth = width;
					end = last;
				} else {
					boolean newLine = handleSpecial(string, special);
					goodWidth = width;
					if (newLine)
						newLine();
					else {
						if ((getSpecialFlags(string, special) & BREAK_BEFORE) != 0)
							breakBefore = true;
						end = special + 1;
					}
				}
			}
		}
	}
	/**
	 * Answer the default/initial value for maxHeight.
	 *
	 * @return an integer representing the default/initial maxHeight.
	 */
	protected int defaultMaxHeight() {
		return ALL_VERTICAL;
	}
	/**
	 * Answer the default/initial value for maxWidth.
	 *
	 * @return an integer representing the default/initial maxWidth.
	 */
	protected int defaultMaxWidth() {
		return Integer.MAX_VALUE;
	}
	/**
	 * Answer the default/initial value for the array where characters can be marked as special.
	 * By default, we assume ASCII.  Subclasses can easily override this but may wish to address special
	 * characters in a more compact way (perhaps as a different implementer of the base interface).
	 *
	 * @return an array of ints.
	 */
	protected int[] defaultSpecials() {
		int specialChars[] = new int[256];
		specialChars['\n'] = NEW_LINE | NO_FONT;
		specialChars['\r'] = NEW_LINE | NO_FONT;
		specialChars['\f'] = NEW_LINE | NO_FONT;
		specialChars['\t'] = NO_FONT | END_ABSORB | BREAK_AFTER;
		specialChars[' '] = END_ABSORB | BREAK_AFTER;
		//	specialChars['\b'] = ???;
		specialChars['-'] = BREAK_AFTER;
		specialChars['!'] = BREAK_AFTER;
		specialChars['@'] = BREAK_AFTER;
		specialChars['%'] = BREAK_AFTER;
		specialChars['&'] = BREAK_AFTER;
		specialChars['*'] = BREAK_AFTER;
		specialChars['_'] = BREAK_AFTER;
		specialChars['+'] = BREAK_AFTER;
		specialChars['='] = BREAK_AFTER;
		specialChars['|'] = BREAK_AFTER;
		specialChars['}'] = BREAK_AFTER;
		specialChars[']'] = BREAK_AFTER;
		specialChars[')'] = BREAK_AFTER;
		specialChars[':'] = BREAK_AFTER;
		specialChars[';'] = BREAK_AFTER;
		specialChars['?'] = BREAK_AFTER;
		specialChars['/'] = BREAK_AFTER;
		specialChars['\\'] = BREAK_AFTER;
		specialChars['>'] = BREAK_AFTER;
		specialChars[','] = BREAK_AFTER;
		specialChars['.'] = BREAK_AFTER;
		specialChars['~'] = BREAK_BEFORE;
		specialChars['#'] = BREAK_BEFORE;
		specialChars['$'] = BREAK_BEFORE;
		specialChars['^'] = BREAK_BEFORE;
		specialChars['<'] = BREAK_BEFORE;
		specialChars['('] = BREAK_BEFORE;
		specialChars['['] = BREAK_BEFORE;
		specialChars['{'] = BREAK_BEFORE;
		return specialChars;
	}
	/** 
	 * Handles the special character at index during composition.  
	 * Answer true if a new line is necessary.
	 *
	 * @param string the string in which to find the special character
	 * @param index index of special character
	 * @return boolean whether or not a new line is necessary.
	 */
	protected boolean handleSpecial(String string, int index) {
		if (index == NO_INDEX)
			return true;
		int flags = getSpecialFlags(string, index);
		int oldWidth = width;
		if ((flags & END_DELAY) != 0)
			adjustWidthForSpecial(string, index);
		if ((flags & NEW_LINE) != 0) {
			end = index;
			next = index + 1;
			return true;
		} else {
			// Assumes END_ABSORB will also be a BREAK, if not we could end up with problems?
			if (((flags & BREAK) != 0) && (width > maxWidth)) {
				if ((flags & BREAK_BEFORE) != 0) {
					width = oldWidth;
					end = index;
				} else
					// BREAK_AFTER ???
					end = index + 1;
				next = end;
				return true;
			}
		}
		// Someday need to handle VERTICAL_MOVE ???
		
		if ((flags & BREAK_BEFORE) != 0)
			width = oldWidth;
		if ((flags & NO_FONT) != 0)
			handleManually(index);
		return false;
	}
	/** 
	 * Answer whether the string has been fully composed.
	 *
	 * @return a boolean value of true if the line is fully composed; false otherwise.
	 */
	protected boolean isFullyComposed() {
		if ((maxHeight != ALL_VERTICAL) && (maxHeight <= height))
			return true;
		return super.isFullyComposed();
	}
	/** 
	 * Paints the string.
	 *
	 * @param g the specified Graphics window.
	 */
	public void paint(Graphics g) {
		verifyMetrics(g);
		
		/*
		 * Determine what lines we really need to display
		 */
		int minY, maxY;
		Rectangle clip = g.getClipBounds();
		minY = startY;
		maxY = (maxHeight == -1) ? -1 : startY + getLineHeight() + maxHeight;
		if (clip.height != -1) {
			minY = Math.max(minY, clip.y - getLineHeight());
			int maxClipY = clip.y + clip.height + getLineHeight();
			maxY = (maxHeight == -1) ? maxClipY : Math.min(maxClipY, maxY);
		}
		
		paintTextBetween(g, minY, maxY);
	}
	/**
	 * Returns the String representation of the receiver's values.
	 *
	 * @return a String representing the receiver's values.
	 */
	public String toString() {
		return getClass().getName() + "[begin=" + begin + ",end=" + end + ",next=" + next + ",last=" + last + ",goodWidth=" + goodWidth + ",width=" + width + ",height=" + height + ",lines=" + ends.length + "]";
	}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)BasicStringRenderer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * This provides basic default functionality for StringRenderer.
 * It doesn't do any wrapping.  However, it handles tabs and new-lines and
 * provides hooks for a lot of subclasses which might want to handle more
 * complicated StringRendering.
 * NOTE: If a font is not supplied when constructed, it will be determined
 * by default or by the current font the first time it is asked to paint,
 * whichever comes first.
 *
 * There are basically two main responsibilities of this class:
 *	1. compose the characters of a string into lines and information about those lines
 *    that will be meaningful before, during, and after displaying them.
 *  2. paint the parts of the composed string that should be visible when given a
 *    given area and medium on which to display.
 *
 * These two responsibilities are intertwined for efficiency.  Nothing will be composed
 * before it is necessary to either paint or respond to some query.  It is very possible that
 * only a small portion of the string will ever be composed, e.g. if only asked to paint the first
 * few lines of a long string with no other action that would change the viewing area to display more.
 * On the other hand, just because something wasn't ever completely viewed, doesn't mean it will
 * not be totally composed (e.g. if asked for width, all lines must be composed in order to figure
 * out what line is widest).  Subclasses may provide for constraints that will make various scenarios
 * more or less efficient as they change the amount of services dependent on composition. 
 *
 * @version 	1.1.6, 12/30/98
 */
public class BasicStringRenderer implements StringRenderer {
	protected String string;
	/**
	 * NOTE: We use lazy initialization for metrics to avoid problems of
	 *  1. changing metrics mid-composition (as line height etc. is affected by the metrics),
	 *  2. performance problems of calculating default when one was not explicitly set.
	 *  3. delaying identification of metrics (and resetting of all composition-related attributes)
	 *		until composition actually begins.
	 */
	protected FontMetrics metrics;

	/**
	 * This determines the starting x location to begin rendering the string;
	 */
	protected int startX = defaultStartX();

	/**
	 * This determines the starting y location to begin rendering the string;
	 */
	protected int startY = defaultStartY();
	
	/**
	 * These hold information necessary to determine the beginning, end, and width of each line
	 * along with which indexes of characters within that line need manual intervention when displaying
	 * (e.g. tabs are usually not displayed as characters, but rather an amount of space determined based
	 * on current placement must be determined... other special characters must also need special handling).
	 * All integers in these arrays (except widths) indicate indexes of characters within the entire string;
	 */
	protected int begins[];
	protected int ends[];
	protected int widths[];
	protected int manuals[][]; // first dimension: line number; second dimension: array of 0..N index of specials.
	
	/**
	 * Provides a map of character values to types of special handling (see static attributes of StringRenderer)
	 * which are bitORed together.
	 */
	protected int specials[] = defaultSpecials();

	/**
	 * The following are all used to keep track of running counters when composing lines.
	 * NOTE: "end", "next", "last" are often manipulated to a point that clouds their literal meaning.
	 * For example, "end" (of line) is set to be the last character of the string until it is determined that
	 * this assumption is invalid.
	 */
	protected int begin, end, next, last, width, height;

	/**
	 * Symbolic name for result of not finding an index when searched.
	 */
	protected static int NO_INDEX = -1;
	
	/**
	 * Symbolic name for identifying all vertical space to be composed or painted.
	 */
	protected static int ALL_VERTICAL = -1;
 
/** 
 * Answer an instance prepared to render a String.
 *
 * @param string the string to render
 */
public BasicStringRenderer(String string) {
	this.setString(string);
}
	/** 
	 * Answer an instance prepared to render the given string using a particular font.
	 *
	 * @param string the string to render.
	 * @param font the font to use when rendering.
	 */
	public BasicStringRenderer(String string, Font font) {
		this.string = string;
		this.setFont(font);
	}
	/** 
	 * Adjusts the theoretical end of the line to absorb whitespace in order to speed
	 * up display of the lines later.  Width of characters are still accounted for.
	 */
	protected void absorbEnd() {
		// peek back for other dead end space
		int back = end - 1;
		while (begin < back && ((getSpecialFlags(string, back) & END_ABSORB) != 0)) {
			end = back;
			back = end - 1;
		}
		// peek ahead for other dead end space that does not initiate vertical movement
		if (((getSpecialFlags(string, next-1)) & VERTICAL) == 0) {
			while ((next < last) && (((getSpecialFlags(string, next)) & (VERTICAL | END_ABSORB)) == END_ABSORB)) {
				adjustWidthForSpecial(string, next);
				next = next + 1;
			}
		}
	}
	/** 
	 * Record the current settings as the end of a line.
	 * Leave the running counters alone so others may use them before preparing
	 * to start the next line.
	 * Used during composition.
	 *
	 * @see beginNextLine
	 */
	protected void addLine() {
		int newEnds[] = new int[ends.length + 1];
		int newBegins[] = new int[begins.length + 1];
		int newManuals[][] = new int[manuals.length + 1][];
		int newWidths[] = new int[widths.length + 1];
		System.arraycopy(ends, 0, newEnds, 0, ends.length);
		System.arraycopy(begins, 0, newBegins, 0, begins.length);
		System.arraycopy(manuals, 0, newManuals, 0, manuals.length);
		System.arraycopy(widths, 0, newWidths, 0, widths.length);
		newEnds[ends.length] = end;
		newWidths[ends.length] = width;
		ends = newEnds;
		begins = newBegins;
		manuals = newManuals;
		widths = newWidths;
		height += getLineHeight();
	}
	/** 
	 * Adjust the width to account for the special character at the specified index.
	 * Used during composition.
	 *
	 * @param string string to check
	 * @param index the index of the special character
	 */
	protected void adjustWidthForSpecial(String string, int index) {
		if (index == NO_INDEX)
			return;
		char c = string.charAt(index);
		int flags = getSpecialFlags(c);
		if ((flags & NO_FONT) == 0) {
			width += metrics.charWidth(c);
			return;
		}
		switch (c) {
			case '\t' :
				width = nextTabStop(width);
				break;
			default : // what's left, new lines?? if so, do nothing
		}
	}
	/** 
	 * Answer whether all of the characters in the string between start (inclusive) and stop (exclusive)
	 * are END_ABSORBed characters.
	 *
	 * @param start an integer representing the starting index, inclusive.
	 * @param stop an integer representing the stopping index, exclusive.
	 * @return boolean
	 */
	protected boolean areAllAbsorbed(int start, int stop) {
		for (int i=start; i < stop; i++) {
			if ((getSpecialFlags(string,i) & END_ABSORB) == 0)
				return false;
		}
		return true;
	}
	/** 
	 * Set up to compose the next line.
	 * Reset the running counters and buffers which may be needed.
	 * Used during composition.  Often after addLine().
	 *
	 * @see addLine()
	 */
	protected void beginNextLine() {
		begin = next;
		end = begin;
		begins[begins.length - 1] = begin;
		manuals[manuals.length - 1] = new int[0];
		width = 0;
		next = last;
	}
	/** 
	 * Make the last/pending line into a "real one" if indicated.
	 * This will be only when the string is empty.
	 */
	protected void closeLastLine() {
		if (end == last && last == 0 && !isFullyComposed())
			addLine();
	}
	/** 
	 * Compose the text in such a way as to produce all lines necessary
	 * to display text properly.
	 */
	protected void compose() {
		compose(ALL_VERTICAL);
	}
	/** 
	 * Compose the text in such a way as to produce the lines necessary
	 * to display text up to the given vertical coordinate.
	 *
	 * @param ySize the vertical coordinate at which we'll stop composing.
	 */
	protected void compose(int ySize) {
		compose(0, ySize);
	}
	/** 
	 * Compose the text in such a way as to produce the lines necessary
	 * to display text between the given vertical coordinates.
	 *
	 * @param yBegin the vertical coordinate at which we need to begin composing.
	 * @param yEnd the vertical coordinate at which we'll stop composing.
	 */
	protected void compose(int yBegin, int yEnd) {
		while (end < last && (yEnd == ALL_VERTICAL || height < yEnd)) {
			int special = scanToSpecial(string, end, last);
			if (special == NO_INDEX)
				end = last;
			else {
				if (handleSpecial(string, special))
					newLine();
				else
					end = special + 1;
			}
			if (end == last)  // this will close the last line
				newLine();
		}
		closeLastLine();  // this is really only relevant in the special case of an empty string
	}
	/** 
	 * Assume something has changed that makes the current composition bogus.
	 * Do it again.
	 */
	protected void composeAll() {
		reset();
		compose();
	}
	/** 
	 * If no composing has happened yet, make it happen.
	 * If something has changed since the last time the string was composed,
	 * do it again.
	 * No matter what, make sure the entire string is composed.
	 */
	protected void composeIfNecessary() {
		if (!isFullyComposed())
			compose();
	}
	/**
	 * Answer the default/initial value for the Font.
	 *
	 * @return the default/initial value for the Font.
	 */
	protected Font defaultFont() {
		String fontName = Toolkit.getDefaultToolkit().getFontList()[0];
		return new Font(fontName, Font.PLAIN, 12);
	}
	/**
	 * Answer the default/initial value for metrics.
	 *
	 * @return the default/initial value for metrics.
	 */
	protected FontMetrics defaultMetrics() {
		return Toolkit.getDefaultToolkit().getFontMetrics(defaultFont());
	}
	/**
	 * Answer the default/initial value for the array where characters can be marked as special.
	 * By default, we assume ASCII.  Subclasses can easily override this but may wish to address special
	 * characters in a more compact way (perhaps as a different implementer of the base interface).
	 *
	 * @return an array of ints.
	 */
	protected int[] defaultSpecials() {
		int specialChars[] = new int[256];
		specialChars['\n'] = NEW_LINE | NO_FONT;
		specialChars['\r'] = NEW_LINE | NO_FONT;
		specialChars['\f'] = NEW_LINE | NO_FONT;
		specialChars['\t'] = NO_FONT | END_ABSORB;
		specialChars[' '] = END_ABSORB;
		return specialChars;
	}
	/**
	 * Answer the default/initial value for the x coordinate.
	 *
	 * @return the default/initial value for the x coordinate.
	 */
	protected int defaultStartX() {
		return 0;
	}
	/**
	 * Answer the default/initial value for the y coordinate.
	 *
	 * @return the default/initial value for the y coordinate.
	 */
	protected int defaultStartY() {
		return 0;
	}
	/** 
	 * Answer the baseline (coordinates in y direction from top) to display text.
	 * NOTE: Assumes metrics have been set.
	 * Used during painting.
	 *
	 * @return an integer representing the baseline from which to display text.
	 */
	protected int getBaseline() {
		return metrics.getMaxAscent();
	}
	/**
	 * Answer the Font the receiver is using.
	 *
	 * @return the Font the receiver is using.
	 */
	public Font getFont() {
		if (metrics == null)
			return defaultFont();
		return metrics.getFont();
	}
	/** 
	 * Answer the height of a line.
	 * Used during composition.
	 *
	 * @return an integer representing the height of a line.
	 */
	protected int getLineHeight() {
		return getMetrics().getHeight();
	}
	/**
	 * Answer the font metrics the receiver is using.
	 *
	 * @return the FontMetrics the receiver is using.
	 */
	protected FontMetrics getMetrics() {
		if (metrics == null)
			metrics = defaultMetrics();
		return metrics;
	}
	/** 
	 * Answers an array of the substrings, one for each line.
	 * Don't strip off any ending white space.
	 * NOTE: This will probably be changed to return an Enumeration or Iterator.
	 *
	 * @return an array of Strings.
	 */
	public String[] getRawStringLines() {
		composeIfNecessary();
		String array[] = new String[ends.length];
		for (int i = 0; i < array.length; i++)
			array[i] = string.substring(begins[i], begins[i + 1]);
		return array;
	}
	/** 
	 * Answer the flags associated with a particular character.
	 *
	 * @param c the character.
	 * @return the flags associated with the given character, encoded in an integer.
	 */
	protected int getSpecialFlags(char c) {
		return specials[c];
	}
	/** 
	 * Answer the flags associated with a particular character.
	 *
	 * @param string string to check.
	 * @param index point to look at.
	 * @return the flags associated with the given character, encoded in an integer.
	 */
	protected int getSpecialFlags(String string, int index) {
		return specials[string.charAt(index)];
	}
	/**
	 * Answer the String associated with the receiver.
	 *
	 * @return the String associated with the receiver.
	 */
	public String getString() {
		return string;
	}
	/** 
	 * Answer the height of the composed string.
	 *
	 * @return an integer representing the height of the composed string.
	 */
	public int getStringHeight() {
		composeIfNecessary();
		return height;
	}
	/** 
	 * Answers an array of the substrings, one for each line.
	 * This will strip off any ending characters identified as END_ABSORB.
	 * NOTE: This will probably be changed to return an Enumeration or Iterator.
	 *
	 * @return an array of Strings.
	 */
	public String[] getStringLines() {
		composeIfNecessary();
		String array[] = new String[ends.length];
		for (int i = 0; i < array.length; i++)
			array[i] = string.substring(begins[i], ends[i]);
		return array;
	}
	/** 
	 * Answer the width of the composed string.
	 *
	 * @return an integer representing the width of the composed string.
	 */
	public int getStringWidth() {
		composeIfNecessary();
		int max = 0;
		for (int i = 0; i < widths.length; i++)
			max = Math.max(max, widths[i]);
		return max;
	}
	/** 
	 * Flags the special character at index as one to display manually.
	 * Used during composition.  
	 * Assumes the special character is part of current/last line composed.
	 *
	 * @param index index of special character to flag.
	 */
	protected void handleManually(int index) {
		int previous[] = manuals[manuals.length - 1];
		int current[] = new int[previous.length + 1];
		System.arraycopy(previous, 0, current, 0, previous.length);
		current[previous.length] = index;
		manuals[manuals.length - 1] = current;
	}
	/** 
	 * Handles the special character at index during composition.  
	 * Answer true if a new line is necessary.
	 *
	 * @param string the string in which to find the special character.
	 * @param index index of special character.
	 * @return boolean whether or not a new line is necessary.
	 */
	protected boolean handleSpecial(String string, int index) {
		if (index == NO_INDEX) // end of string
			return true;
		int flags = getSpecialFlags(string, index);
		if ((flags & END_DELAY) != 0)
			adjustWidthForSpecial(string, index);
		if ((flags & NEW_LINE) != 0) {
			end = index;
			next = index + 1;
			return true;
		}
		if ((flags & NO_FONT) != 0)
			handleManually(index);
		return false;
	}
	/** 
	 * Answers the index of the first special character in the specified range which is not
	 * END_ABSORB only, or -1 if none.
	 * Used during composition.
	 *
	 * @param string string to check.
	 * @param start starting point to look.
	 * @param ending point to look.
	 * @return an integer representing the index of the first special character.
	 */
	protected int indexOfSpecial(String string, int start, int stop) {
		for (int i = start; i < stop; i++) {
			int flags = getSpecialFlags(string, i);
			if ((flags != 0) && (flags != END_ABSORB))
				return i;
		}
		return NO_INDEX;
	}
	/** 
	 * Answer whether the string has been fully composed.
	 *
	 * @return a boolean value of true if the string is fully composed; false otherwise.
	 */
	protected boolean isFullyComposed() {
		int numberOfLines = ends.length;
		/*
		 * The following is an attempt to verify that we've composed everything... 
		 * it seems too complicated, but other alternatives I can come up with right now
		 * seem to offer their own ugliness, so we'll live with this for now.
		 * Here is a description of the logic below:
		 *  Even an empty string should take up one line after being composed... no lines, not composed.
		 *  If the last character of last line is last character of string, we've composed everything.
		 *  If not, we might be in a situation where the last characters of the string are absorbed, i.e. not
		 *     worth displaying.  If this is the case with the last line, we've composed everything
		 */
		return ((numberOfLines != 0) &&
			(ends[numberOfLines - 1] == string.length() || // 
				(begins[numberOfLines] == string.length() && (areAllAbsorbed(ends[numberOfLines - 1],string.length())))));
	}
	/** 
	 * Record the current settings as the end of a line, 
	 * and set up to scan the next line.
	 * Used during composition.
	 */
	protected void newLine() {
		absorbEnd();
		addLine();
		beginNextLine();
	}
	/**
	 * Answer the next tab stop after the x position.
	 *
	 * @param x the x position from which to tab.
	 * @return an integer representing the next tab stop.
	 */
	protected int nextTabStop(int x) {
		int tabSpace = 40;
		return ((x / tabSpace) + 1) * tabSpace;
	}
	/** 
	 * Paints the string.
	 *
	 * @param g the specified Graphics window
	 */
	public void paint(Graphics g) {
		verifyMetrics(g);
		/*
		 * Determine what lines we really need to display
		 */
		int minY, maxY;
		Rectangle clip = g.getClipBounds();
		if (clip == null || clip.height == -1) {
			minY = startY;
			maxY = ALL_VERTICAL;
		} else {
			minY = clip.y - getLineHeight();
			maxY = clip.y + clip.height + getLineHeight();
		}
		
		paintTextBetween(g, minY, maxY);
	}
	/** 
	 * Paints the string starting at the top left specified.
	 *
	 * @param g the specified Graphics window.
	 * @param x the X coordinate at which to start.
	 * @param y the Y coordinate at which to start.
	 */
	public void paint(Graphics g, int x, int y) {
		startX = x;
		startY = y;
		paint(g);
	}
	/** 
	 * Paints the special character at the specified index at the appropriate
	 * place.  Answer the new x coordinate.
	 *
	 * @param g the specified Graphics window.
	 * @param index the index of the special character.
	 * @param x the x coordinate.
	 * @param y the y coordinate.
	 * @returns an integer representing the new x coordinate after acting on the special character.
	 */
	protected int paintSpecial(Graphics g, int index, int x, int y) {
		// for now, we only handle tabs
		if (string.charAt(index) != '\t')
			throw new IllegalArgumentException("Only tabs are supported for manual display");
		// just move the x coordinate, we don't need to display anything
		int currentWidth = x - startX;
		int nextTab = nextTabStop(currentWidth);
		return startX + nextTab;
	}
	/** 
	 * Paints the string between the given horizontal window.
	 *
	 * @param g the specified Graphics window.
	 * @param topY int identifying the top of the area in which we would like to paint.
	 * @param bottomY int identifying the bottom of the area in which we would like to paint.
	 */
	protected void paintTextBetween(Graphics g, int topY, int bottomY) {
		/*
		 * Make sure the lines we need to display are composed.
		 * For now, assume that if the metrics are set, 
		 * we are using the right font.
		 */
		compose(topY - startY, (bottomY == -1) ? -1 : bottomY - startY);
		/*
		 * Display the lines of text
		 */
		int x = startX;
		int y = startY + getBaseline();
		int maxY = (bottomY == -1) ? Integer.MAX_VALUE : bottomY;
		for (int i = 0; y < maxY && i < ends.length; i++) {
			if (y > topY) {
				int special[] = manuals[i];
				if (special.length == 0)
					g.drawString(string.substring(begins[i], ends[i]), x, y);
				else {
					int startIndex = begins[i];
					int x2 = x;
					int stopIndex;
					for (int j = 0; j < special.length; j++) {
						stopIndex = special[j];
						if (stopIndex > startIndex) {
							String sub = string.substring(startIndex, stopIndex);
							g.drawString(sub, x2, y);
							x2 += metrics.stringWidth(sub);
						}
						x2 = paintSpecial(g, stopIndex, x2, y);
						startIndex = stopIndex + 1;
					}
					stopIndex = ends[i];
					if (stopIndex > startIndex)
						g.drawString(string.substring(startIndex, stopIndex), x2, y);
				}
			}
			y += getLineHeight();
		}
	}
	/**
	 * Reset the receiver to start composing.
	 */
	protected void reset() {
		if (ends == null || (ends.length != 0)) {
			ends = new int[0];
			begins = new int[1];
			manuals = new int[1][0];
			widths = new int[0];
			height = 0;
		}
		// the following are all used to compose lines
		next = 0;
		last = string.length();
		beginNextLine();
	}
	/** 
	 * Scan the characters from start to stop, adding to the width until
	 * a special character is found.
	 * Answers the index of the first special character, or NO_INDEX if none.
	 *
	 * @param string string to check
	 * @param start starting point to look, inclusive
	 * @param stop stopping point to look, exclusive
	 * @return an integer representing the first special character.
	 */
	protected int scanToSpecial(String string, int start, int stop) {
		int special = indexOfSpecial(string, start, stop);
		int actualStop = special;
		if (special == -1) {
			special = NO_INDEX;
			actualStop = stop;
		}
		if (actualStop > start)
			width += getMetrics().stringWidth(string.substring(start, actualStop));
		if (special == NO_INDEX)
			return special;
		/*
		 * delay adjusting the width for special characters that won't get put
		 * at the end of the line until we've determined if we are there or not
		 */
		int flags = getSpecialFlags(string, special);
		if ((flags & END_DELAY) == 0)
			adjustWidthForSpecial(string, special);
		return special;
	}
	/**
	 * Set the receiver up to compose and display based on the given font.
	 *
	 * @param font the font to use as a base.
	 */
	public void setFont(Font font) {
		if (font != getFont()) {
			metrics = Toolkit.getDefaultToolkit().getFontMetrics(font);
			reset();
		}
	}
	/**
	 * Set the receiver up to display the string.
	 *
	 * @param string the string to use as a base.
	 */
	public void setString(String newString) {
		if (string != newString) {
			string = newString;
			reset();
		}
	}
	/**
	 * Returns the String representation of the receiver's values.
	 *
	 * @return a String representation of the receiver's values.
	 */
	public String toString() {
		return getClass().getName() + "[begin=" + begin + ",end=" + end + ",next=" + next + ",last=" + last + ",width=" + width + ",height=" + height + ",lines=" + ends.length + "]";
	}
	/** 
	 * Verify that the metrics used to compose the text is 
	 * consistent with the font/metrics about to be used to render
	 * the string.
	 *
	 * @param g the specified Graphics window.
	 */
	protected void verifyMetrics(Graphics g) {
		/*
		 * Make sure we are using the right font.  If we are, assume the metrics are the same.
		 */
		if (metrics == null || (getFont().equals(g.getFontMetrics().getFont()))) {
			metrics = g.getFontMetrics();
			reset();
		}
	}
} 
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)Duplicatable.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.util.Hashtable;

/**
 * This interface defines a simple interface for objects that can be duplicated.
 * This would typically be used for copy/cut/paste type operations.  
 * The intended use is:
 *    1. One or more objects are sent duplicate().
 *    2. A Hashtable is built with each of the original objects as keys and their
 *       duplicates as corresponding values.
 *    3. Each of the duplicates are sent postDuplicate(Hashtable), giving them
 *       the opportunity to resolve any changes they might want to make.
 * E.g. Object A points to Object B, Object A' (the result of duplicate())
 * still points to Object B.  When Object A' receives postDuplicate(), it could
 * make a change to point to Object B', or some other object if B' is not available.
 *
 * @version 	1.1.6, 12/30/98
 */
public interface Duplicatable extends Cloneable {

	/**
	 * Answers a duplicate of this object.
	 * This is intended for copy/cut/paste operations, hence may be different
	 * than what you would like to do for other "cloning" type operations.
	 *
	 * @param Object the duplicate.
	 */
	public abstract Object duplicate();
	/**
	 * Answers a duplicate of this object.  While doing so, place the original
	 * as a key and the duplicate as a value into the duplicates HashTable.
	 * Implementers may also wish to place duplicates of components in the HashTable
	 * if they may be significant when resolving pointers after a group of objects
	 * have been duplicated.
	 * This is intended for copy/cut/paste operations, hence may be different
	 * than what you would like to do for other "cloning" type operations.
	 *
	 * @param duplicates the table which will be used to map the originals to the duplicates.
	 * @param Object the duplicate.
	 * @see #postDuplicate
	 */
	public abstract Object duplicateIn(Hashtable duplicates);
	/**
	 * After a series of objects are duplicated, this can be sent to each of the
	 * duplicates to resolve any changes it might like to reconcile.  
	 * For example, replacing observers with their duplicates, if available.
	 * 
	 * @param duplicates a Hashtable with originals as keys and duplicates as elements.
	 */
	public abstract void postDuplicate(Hashtable duplicates);
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)GraphicsGeometry.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.*;
import com.rolemodelsoft.drawlet.basics.*;

/**
 * A collection of static geometric utility methods.
 */
public class GraphicsGeometry {
/**
 * Answer whether the two intervals overlap.
 * @return boolean
 * @param start1 int
 * @param stop1 int
 * @param start2 int
 * @param stop2 int
 */
public static boolean intervalsOverlap(int start1, int stop1, int start2, int stop2) {
	return (
		isBetween(start1,start2,stop2) ||
		isBetween(stop1,start2,stop2) ||
		isBetween(start2,start1,stop1)
	);
}
/**
 * 
 * @return boolean
 * @param seg1Constant int
 * @param seg1Boundary1 int
 * @param seg1Boundary2 int
 * @param seg2Constant int
 * @param seg2Boundary1 int
 * @param seg2Boundary2 int
 */

//ONLY for pairs of VERTICAL or HORIZONTAL lines, not diagonals 
protected static boolean isBetween(double potential, double first, double last) {
	if (first <= last)
		return (first <= potential) && (potential <= last);
	else
		return (last <= potential) && (potential <= first);
}
/**
 * Answer whether the first segment on a line overlaps the second segment on the same line.
 */
protected static boolean sameLineSegmentsOverlap(int segment1x1, int segment1y1, int segment1x2, int segment1y2, int segment2x1, int segment2y1, int segment2x2, int segment2y2){
	if (segment1x1 == segment1x2)
		return intervalsOverlap(segment1y1, segment1y2, segment2y1, segment2y2);
	return intervalsOverlap(segment1x1, segment1x2, segment2x1, segment2x2);
}
/**
 * Answer whether the first segment intersects the line through the second segment.
 */
public static boolean segmentIntersectsLine(int segment1x1, int segment1y1, int segment1x2, int segment1y2, int segment2x1, int segment2y1, int segment2x2, int segment2y2){
	if (segment2x2 == segment2x1)
		return (isBetween(segment2x1,segment1x1,segment1x2));
	double slope = (double)(segment2y2 - segment2y1) / (double)(segment2x2 - segment2x1);
	double yIntercept = (double)segment2y1 - (double)(slope * segment2x1);
	return segmentIntersectsNonVerticalLine(segment1x1, segment1y1, segment1x2, segment1y2, slope, yIntercept);
}
/**
 * Answer whether a segment intersects a non-vertical line.
 * @return boolean
 * @param segmentX1 int
 * @param segmentY1 int
 * @param segmentX2 int
 * @param segmentY2 int
 * @param slope double
 * @param yIntercept double
 */
public static boolean segmentIntersectsNonVerticalLine(int segmentX1, int segmentY1, int segmentX2, int segmentY2, double slope, double yIntercept) {
	double y1 = slope * segmentX1 + yIntercept;
	double y2 = slope * segmentX2 + yIntercept;
	/*
	 * if the two endpoints of the segment are on different sides
	 * of the line or at least one is on it, the segment intersects the line
	 */
	return ( ((segmentY1 >= y1) && (segmentY2 <= y2)) || ((segmentY1 <= y1) && (segmentY2 >= y2)) );
}
/**
 * Answer whether the two segments intersect.
 * based on Paul Bourke's algorithm at http://www.mhri.edu.au/~pdb/geometry/lineline2d
 */
public static boolean segmentIntersectsSegment(int segment1x1, int segment1y1, int segment1x2, int segment1y2, int segment2x1, int segment2y1, int segment2x2, int segment2y2){
	double ud, uan, ubn, ua, ub, crossx, crossy;
	ud = (segment2y2 - segment2y1) * (segment1x2 - segment1x1) - (segment2x2 - segment2x1) * (segment1y2 - segment1y1);
	uan = (segment2x2 - segment2x1) * (segment1y1 - segment2y1) - (segment2y2 - segment2y1) * (segment1x1 - segment2x1);
	ubn = (segment1x2 - segment1x1) * (segment1y1 - segment2y1) - (segment1y2 - segment1y1) * (segment1x1 - segment2x1);
	if (ud == 0){
		if (uan == 0)
			return sameLineSegmentsOverlap(segment1x1, segment1y1, segment1x2, segment1y2, segment2x1, segment2y1, segment2x2, segment2y2);
		return false;
	}
	ua = uan / ud;
	ub = ubn / ud;
	return isBetween(ua, 0, 1) && isBetween(ub, 0, 1);
}
/**
 * Answer whether the first segment overlaps the second segment.
 */
public static boolean segmentOverlapsSegment(int segment1x1, int segment1y1, int segment1x2, int segment1y2, int segment2x1, int segment2y1, int segment2x2, int segment2y2){
	return (segmentsOnSameLine(segment1x1, segment1y1, segment1x2, segment1y2, segment2x1, segment2y1, segment2x2, segment2y2))
		&& sameLineSegmentsOverlap(segment1x1, segment1y1, segment1x2, segment1y2, segment2x1, segment2y1, segment2x2, segment2y2);
}
/**
 * Answer whether the first segment and second segment are on the same line.
 */
protected static boolean segmentsOnSameLine(int segment1x1, int segment1y1, int segment1x2, int segment1y2, int segment2x1, int segment2y1, int segment2x2, int segment2y2){
	if (segment1x1 == segment1x2)
		return (segment2x1 == segment2x2) && (segment2x1 == segment1x1);
	if (segment2x1 == segment2x2)
		return false;
	double slope1 = (double)(segment1y2 - segment1y1) / (double)(segment1x2 - segment1x1);
	double slope2 = (double)(segment2y2 - segment2y1) / (double)(segment2x2 - segment2x1);
	double yIntercept1 = (double)segment1y1 - (double)(slope1 * segment1x1);
	double yIntercept2 = (double)segment2y1 - (double)(slope2 * segment2x1);
	return ((slope1 == slope2) && (yIntercept1 == yIntercept2));
}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)Observable.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
/**
 * This interface defines a generic Observable interface.
 * This is desirable as the java.util.Observable is a class which
 * could force unnatural hierarchies.  It also allows public access
 * to a list of Observers as an array which may not be the actual
 * collection of observers, allowing implementers to let others know
 * what they are observing without allowing them to modify that list
 * through "illegal" channels.
 *
 * @version 	1.1.6, 12/30/98
 */
public interface Observable {
	/**
	 * Adds an observer to the observer list.
	 *
	 * @param observer the observer to be added.
	 */
	public void addObserver(Observer observer);
	/**
	 * Deletes an observer from the observer list.
	 *
	 * @param observer the observer to be deleted.
	 */
	public void deleteObserver(Observer observer);
	/**
	 * Deletes all observers from the observer list.
	 * Though this is public, caution should be used before anything other
	 * than "this" invokes it.
	 */
	public void deleteObservers();
	/** 
	 * Answers a collection of observers.
	 *
	 * @return an array of Observers.
	 */
	public Observer[] getObservers();
	/**
	 * Notifies all observers that an observable change occurs.
	 * Though this is public, caution should be used before anything other
	 * than "this" does.
	 */
	public void notifyObservers();
	/**
	 * Notifies all observers that an observable change occurred.
	 * Though this is public, caution should be used before anything other
	 * than "this" invokes it.
	 *
	 * @param arg info to pass along to those being notified.
	 */
	public void notifyObservers(Object arg);
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)Observer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
/**
 * This interface defines a generic Observer interface
 * This is desirable as the java.util.Observable is a class which
 * could force unnatural hierarchies.  This looks for the Observable
 * interface instead of the class to allow more flexibility in implementing
 * concrete classes.
 *
 * @version 	1.1.6, 12/30/98
 */

public interface Observer {
	/**
	 * The subject has notified the receiver of a change.
	 *
	 * @param subject the observed subject.
	 * @param arg the argument being notified.
	 */
	void update(Observable subject, Object arg);
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)ReverseVectorEnumerator.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1996 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.util.Enumeration;
import java.util.Vector;
import java.util.NoSuchElementException;

/**
 * This provides an Enumeration of a Vector that goes through
 * the Vector in reverse order.
 *
 * @version 	1.1.6, 12/30/98
 */
public class ReverseVectorEnumerator implements Enumeration {
	/**
	 * The Vector to enumerate over.
	 */
	Vector vector;

	/**
	 * Stores the current index into the Vector.
	 */
	int count;
	/** 
	 * Answer an instance prepared to iterate over a Vector.
	 *
	 * @param v the Vector over which to enumerate.
	 */
	public ReverseVectorEnumerator(Vector v) {
		vector = v;
		count = v.size() - 1;
	}
	/**
	 * Answers whether there are more elements in the vector.
	 *
	 * @return a boolean value of true if the enumeration
	 * contains more elements; false if it is empty.
	 */
	public boolean hasMoreElements() {
		return count >= 0;
	}
	/**
	 * Returns the next element of the enumeration. Calls to this
	 * method will enumerate successive elements.
	 *
	 * @exception NoSuchElementException If no more elements exist.
	 * @returns the next object in the Vector.
	 */
	public Object nextElement() {
		synchronized (vector) {
			if (count >= 0) {
				return vector.elementAt(count--);
			}
		}
		throw new NoSuchElementException("ReverseVectorEnumerator");
	}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)SegmentIntersectionTestVisualizer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.examples.awt.*;
import java.awt.*;

public class SegmentIntersectionTestVisualizer extends ExitingFrame {
	protected int[] Line;
	protected int[][] TestLine;
/**
 * PolygonTestVisualizer constructor comment.
 */
public SegmentIntersectionTestVisualizer() {
	super();
}
/**
 * PolygonTestVisualizer constructor comment.
 * @param title java.lang.String
 */
public SegmentIntersectionTestVisualizer(int[] Line, int[][] TestLine) {
	super();
	this.Line = Line;
	this.TestLine = TestLine;
}
/**
 * PolygonTestVisualizer constructor comment.
 * @param title java.lang.String
 */
public SegmentIntersectionTestVisualizer(String title) {
	super(title);
}
/**
 * 
 * @param g java.awt.Graphics
 */
public void paint(Graphics g) {
	g.setColor(Color.black);
	for(int i = 0; i < TestLine.length; i++)
		g.drawLine(TestLine[i][0], TestLine[i][1], TestLine[i][2], TestLine[i][3]);
	g.setColor(Color.red);
	g.drawLine(Line[0], Line[1], Line[2], Line[3]);
}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)StringRenderer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 * Copyright (c) 1997 Knowledge Systems Corporation (KSC). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS AND KSC MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. NEITHER RMS NOR KSC SHALL BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
import java.awt.*;

/**
 * This interface defines a generic interface for objects that can measure
 * and display strings.  It is expected that implementers will determine
 * what to do with special characters, wrapping, etc.
 * This should be used when wanting a bit more intelligence than 
 * FontMetrics.stringWidth() and Graphics.drawString() offer.
 * In addition to some basic protocols, a variety of flags are offered which
 * may somehow be associated with special characters to give implementers
 * help in determining what to do when those characters are discovered and
 * offer implementers the opportunity to avoid hard coding based on particular
 * characters in a "standard" way.
 *
 * @version 	1.1.6, 12/30/98
 */
public interface StringRenderer {

	/**
	 * Encourage new lines after these characters if later non-special 
	 * characters would otherwise end up past some maximum x coordinate.
	 */
	public static int BREAK_AFTER = 1;

	/**
	 * Encourage new lines before these characters if later non-special 
	 * characters would otherwise end up past some maximum x coordinate.
	 */
	public static int BREAK_BEFORE = 2;

	/**
	 * Encourage new lines if later non-special 
	 * characters would otherwise end up past some maximum x coordinate.
	 */
	public static int BREAK = BREAK_AFTER | BREAK_BEFORE;

	/**
	 * Ignore (with respect to wrapping) these characters at the end 
	 * of lines and avoid using them at the beginning of a line unless 
	 * preceded by a NEW_LINE.
	 */
	public static int END_ABSORB = 4;

	/**
	 * These characters are not displayed based on what the font says, 
	 * but rather some sort of special handling.
		 */
	public static int NO_FONT = 8;

	/**
	 * These characters force the character which immediately follows 
	 * them to appear on a new line.
	 */
	public static int NEW_LINE = 16;

	/**
	 * These characters force some manual vertical movement more 
	 * complicated than a simple NEW_LINE.
	 */
	public static int VERTICAL_MOVE = 32;

	/**
	 * These characters force some vertical movement.
	 */
	public static int VERTICAL = NEW_LINE | VERTICAL_MOVE;

	/**
	 * These characters force some vertical movement.
	 */
	public static int END_DELAY = END_ABSORB | BREAK_BEFORE;
	/**
	 * Answer the font the receiver is using.
	 *
	 * @return the Font the receiver is using.
	 */
	public Font getFont();
	/** 
	 * Answers an array of the substrings, one for each line.
	 * Don't strip off any ending white space.
	 * NOTE: This will probably be changed to return an Enumeration or Iterator.
	 *
	 * @returns an array of Strings.
	 */
	public String[] getRawStringLines();
	/**
	 * Answer the String the receiver is associated with.
	 *
	 * @returns the String the receiver is associated with.
	 */
	public String getString();
	/** 
	 * Answer the height of the composed string.
	 *
	 * @returns an integer representing the height of the composed string.
	 */
	public int getStringHeight();
	/** 
	 * Answers an array of the substrings, one for each line.
	 * NOTE: This will probably be changed to return an Enumeration or Iterator.
	 *
	 * @returns an array of Strings.
	 */
	public String[] getStringLines();
	/** 
	 * Answer the width of the composed string.
	 *
	 * @returns an integer representing the width of the composed string.
	 */
	public int getStringWidth();
	/** 
	 * Paints the string.
	 *
	 * @param g the specified Graphics window.
	 */
	public void paint(Graphics g);
	/** 
	 * Paints the string starting at the top left specified.
	 *
	 * @param g the specified Graphics window.
	 * @param x the x coordinate to display the leftmost point.
	 * @param y the y coordinate to display the topmost point.
	 */
	public void paint(Graphics g, int x, int y);
	/**
	 * Set the receiver up to compose and display based on the given font.
	 *
	 * @param font the font to use as a base.
	 */
	public void setFont(Font font);
	/**
	 * Set the receiver up to display the string.
	 *
	 * @param string the string to use as a base.
	 */
	public void setString(String newString);
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)TC_BasicStringComposer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.util.BasicStringComposer;
import com.rolemodelsoft.drawlet.util.StringRenderer;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Toolkit;
import junit.framework.*;

public class TC_BasicStringComposer extends TestCase {
	protected StringRenderer composer;
	protected StringRenderer bigComposer;
	protected StringRenderer multilineComposer;
	protected StringRenderer narrowComposer;
	protected StringRenderer shortNarrowComposer;
	protected StringRenderer nothingComposer;
	protected StringRenderer emptyLineComposer;
	protected StringRenderer noWidthComposer;
	protected StringRenderer breakBeforeComposer;
	protected StringRenderer breakAfterComposer;
/**
 * BasicStringComposerTest constructor comment.
 * @param name java.lang.String
 */
public TC_BasicStringComposer(String name) {
	super(name);
}
/**
 * Sets up the fixture, for example, open a network connection.
 * This method is called before a test is executed.
 */
protected void setUp() 
{
	composer = new BasicStringComposer("this ");
	bigComposer = new BasicStringComposer("this is a test of the emergency HotDraw system",new Font("SansSerif",Font.BOLD,18));
	multilineComposer = new BasicStringComposer("this is a\n\tindented and more \ncomplex test of the emergency HotDraw system",new Font("Serif",Font.PLAIN,12));
	narrowComposer = new BasicStringComposer("this is a\n\tindented and more \ncomplex test of the emergency HotDraw system",new Font("Serif",Font.PLAIN,12), 100);
	shortNarrowComposer = new BasicStringComposer("this is a\n\tindented and more \ncomplex test of the emergency HotDraw system",new Font("Serif",Font.PLAIN,12), 100, 36);
	nothingComposer = new BasicStringComposer("");
	emptyLineComposer = new BasicStringComposer("\n");
	noWidthComposer = new BasicStringComposer("thw",0);
	breakBeforeComposer = new BasicStringComposer("splitthis(here)<\n>",70);
	breakAfterComposer = new BasicStringComposer("split?here?",50); //this is breaking
}
/**
 * Test for accurate String lines.
 */
public void testRawStringLines() 
{
	String[] lines1 = composer.getRawStringLines();
	String[] lines2 = bigComposer.getRawStringLines();
	String[] lines3 = multilineComposer.getRawStringLines();
	String[] lines4 = narrowComposer.getRawStringLines();
	String[] lines5 = shortNarrowComposer.getRawStringLines();
	String[] lines6 = nothingComposer.getRawStringLines();
	String[] lines7 = emptyLineComposer.getRawStringLines();
	String[] lines8 = noWidthComposer.getRawStringLines();
	String[] lines9 = breakBeforeComposer.getRawStringLines();
	String[] lines10 = breakAfterComposer.getRawStringLines();
	assertEquals(1,lines1.length);
	assertEquals(1,lines2.length);
	assertEquals(3,lines3.length);
	assertEquals(6,lines4.length);
	assertEquals(3,lines5.length);
	assertEquals(1,lines6.length);
	assertEquals(2,lines7.length);
	assertEquals(3,lines8.length);
	assertEquals(3,lines9.length);
	assertEquals(2,lines10.length);
	assertEquals(composer.getString(),lines1[0]);
	assertEquals(bigComposer.getString(),lines2[0]);
	assertEquals("this is a\n",lines3[0]);
	assertEquals("\tindented and more \n",lines3[1]);
	assertEquals("complex test of the emergency HotDraw system",lines3[2]);
	assertEquals("this is a\n",lines4[0]);
	assertEquals("\tindented ",lines4[1]);
	assertEquals("and more \n",lines4[2]);
	assertEquals("complex test of the ",lines4[3]);
	assertEquals("emergency HotDraw ",lines4[4]);
	assertEquals("system",lines4[5]);
	assertEquals("this is a\n",lines5[0]);
	assertEquals("\tindented ",lines5[1]);
	assertEquals("and more \n",lines5[2]);
	assertEquals("",lines6[0]);
	assertEquals("\n",lines7[0]);
	assertEquals("",lines7[1]);
	assertEquals("t",lines8[0]);
	assertEquals("h",lines8[1]);
	assertEquals("w",lines8[2]);
	assertEquals("splitthis",lines9[0]);
	assertEquals("(here)<\n",lines9[1]);
	assertEquals(">",lines9[2]);
	assertEquals("split?",lines10[0]);
	assertEquals("here?",lines10[1]);
}
/**
 * Test to make sure we properly recompose after changing the fonts.
 */
public void testSetFont() 
{
	testStringWidth();
	testStringHeight();
	composer.setFont(new Font("SansSerif",Font.BOLD,18));
	bigComposer.setFont(new Font("Serif",Font.BOLD,14));
	multilineComposer.setFont(new Font("Monospace",Font.BOLD,18));
	narrowComposer.setFont(new Font("SansSerif",Font.BOLD,10));
	shortNarrowComposer.setFont(new Font("SansSerif",Font.BOLD,4));  //this is an issue that needs to be fixed
	nothingComposer.setFont(new Font("Monospace",Font.BOLD,18));
	emptyLineComposer.setFont(new Font("SansSerif",Font.BOLD,10));
	noWidthComposer.setFont(new Font("Monospace",Font.BOLD,18));
	testStringWidth();
	testStringHeight();
}
/**
 * Test to make sure we properly recompose after changing the strings.
 */
public void testSetString() 
{
	testStringWidth();
	testStringHeight();
	composer.setString("Changed text");
	bigComposer.setString("Changed text");
	testStringWidth();
	testStringHeight();
}
/**
 * Test for accurate String height.
 */
public void testStringHeight() 
{
	assertEquals(Toolkit.getDefaultToolkit().getFontMetrics(composer.getFont()).getHeight(),composer.getStringHeight());
	assertEquals(Toolkit.getDefaultToolkit().getFontMetrics(bigComposer.getFont()).getHeight(),bigComposer.getStringHeight());
	assertEquals(3 * Toolkit.getDefaultToolkit().getFontMetrics(multilineComposer.getFont()).getHeight(),multilineComposer.getStringHeight());
	assertEquals(6 * Toolkit.getDefaultToolkit().getFontMetrics(narrowComposer.getFont()).getHeight(),narrowComposer.getStringHeight());
	assertEquals(3 * Toolkit.getDefaultToolkit().getFontMetrics(shortNarrowComposer.getFont()).getHeight(),shortNarrowComposer.getStringHeight());
	assertEquals(1 * Toolkit.getDefaultToolkit().getFontMetrics(nothingComposer.getFont()).getHeight(),nothingComposer.getStringHeight());
	assertEquals(2 * Toolkit.getDefaultToolkit().getFontMetrics(emptyLineComposer.getFont()).getHeight(),emptyLineComposer.getStringHeight());
	assertEquals(3 * Toolkit.getDefaultToolkit().getFontMetrics(noWidthComposer.getFont()).getHeight(),noWidthComposer.getStringHeight());
}
/**
 * Test for accurate String lines.
 */
public void testStringLines() 
{
	String[] lines1 = composer.getStringLines();
	String[] lines2 = bigComposer.getStringLines();
	String[] lines3 = multilineComposer.getStringLines();
	String[] lines4 = narrowComposer.getStringLines();
	String[] lines5 = shortNarrowComposer.getStringLines();
	String[] lines6 = nothingComposer.getStringLines();
	String[] lines7 = emptyLineComposer.getStringLines();
	String[] lines8 = noWidthComposer.getStringLines();
	String[] lines9 = breakBeforeComposer.getStringLines();
	String[] lines10 = breakAfterComposer.getStringLines();
	assertEquals(1,lines1.length);
	assertEquals(1,lines2.length);
	assertEquals(3,lines3.length);
	assertEquals(6,lines4.length);
	assertEquals(3,lines5.length);
	assertEquals(1,lines6.length);
	assertEquals(2,lines7.length);
	assertEquals(3,lines8.length);
	assertEquals(3,lines9.length);
	assertEquals(2,lines10.length);
	assertEquals(1,lines1.length);
	assertEquals(1,lines2.length);
	assertEquals(3,lines3.length);
	assertEquals(6,lines4.length);
	assertEquals(3,lines5.length);
	assertEquals(1,lines6.length);
	assertEquals(2,lines7.length);
	assertEquals(3,lines8.length);
	assertEquals(composer.getString().trim(),lines1[0]);
	assertEquals(bigComposer.getString(),lines2[0]);
	assertEquals("this is a",lines3[0]);
	assertEquals("\tindented and more",lines3[1]);
	assertEquals("complex test of the emergency HotDraw system",lines3[2]);
	assertEquals("this is a",lines4[0]);
	assertEquals("\tindented",lines4[1]);
	assertEquals("and more",lines4[2]);
	assertEquals("complex test of the",lines4[3]);
	assertEquals("emergency HotDraw",lines4[4]);
	assertEquals("system",lines4[5]);
	assertEquals("this is a",lines5[0]);
	assertEquals("\tindented",lines5[1]);
	assertEquals("and more",lines5[2]);
	assertEquals("",lines6[0]);
	assertEquals("",lines7[0]);
	assertEquals("",lines7[1]);
	assertEquals("t",lines8[0]);
	assertEquals("h",lines8[1]);
	assertEquals("w",lines8[2]);
	assertEquals("splitthis",lines9[0]);
	assertEquals("(here)<",lines9[1]);
	assertEquals(">",lines9[2]);
	assertEquals("split?",lines10[0]);
	assertEquals("here?",lines10[1]);
}
/**
 * Test for accurate String widths.
 */
public void testStringWidth() 
{
	FontMetrics metrics = Toolkit.getDefaultToolkit().getFontMetrics(composer.getFont());
	assertEquals("composer" + composer.getFont() + " String size different than sum of words",metrics.stringWidth("A simple test"),metrics.stringWidth("A ") + metrics.stringWidth("simple ") + metrics.stringWidth("test"));
	assertEquals(metrics.stringWidth(composer.getString()),composer.getStringWidth());
	metrics = Toolkit.getDefaultToolkit().getFontMetrics(bigComposer.getFont());
	assertEquals("bigComposer" + bigComposer.getFont().getPeer() + " String size different than sum of words",metrics.stringWidth("A simple test"),metrics.stringWidth("A ") + metrics.stringWidth("simple ") + metrics.stringWidth("test"));
	assertEquals(metrics.stringWidth(bigComposer.getString()),bigComposer.getStringWidth());
	metrics = Toolkit.getDefaultToolkit().getFontMetrics(multilineComposer.getFont());
	String[] lines = multilineComposer.getStringLines();
	assertEquals(metrics.stringWidth(lines[2]),multilineComposer.getStringWidth());
	assert("Expected wider than max width - 20",80 < narrowComposer.getStringWidth());
	assert("Expected narrower than max width + 10 due to whitespace",110 > narrowComposer.getStringWidth());
	assert("Expected wider than max width - 20",80 < shortNarrowComposer.getStringWidth());
	assert("Expected narrower than max width + 10 due to whitespace",110 > shortNarrowComposer.getStringWidth());
	assertEquals(0,nothingComposer.getStringWidth());
	assertEquals(0,emptyLineComposer.getStringWidth());
	metrics = Toolkit.getDefaultToolkit().getFontMetrics(noWidthComposer.getFont());
	assertEquals("noWidthComposer" + noWidthComposer.getFont() + " String size different than sum of words",metrics.stringWidth("A simple test"),metrics.stringWidth("A ") + metrics.stringWidth("simple ") + metrics.stringWidth("test"));
	assertEquals(metrics.stringWidth("w"),noWidthComposer.getStringWidth());
	metrics = Toolkit.getDefaultToolkit().getFontMetrics(breakBeforeComposer.getFont());
	assertEquals("breakBeforeComposer" + breakBeforeComposer.getFont() + " String size different than sum of words",metrics.stringWidth("A simple test"),metrics.stringWidth("A ") + metrics.stringWidth("simple ") + metrics.stringWidth("test"));
	assertEquals(metrics.stringWidth("splitthis"),breakBeforeComposer.getStringWidth());
	metrics = Toolkit.getDefaultToolkit().getFontMetrics(breakAfterComposer.getFont());
	assertEquals("breakAfterComposer" + breakAfterComposer.getFont() + " String size different than sum of words",metrics.stringWidth("A simple test"),metrics.stringWidth("A ") + metrics.stringWidth("simple ") + metrics.stringWidth("test"));
	assertEquals(Math.max(metrics.stringWidth("split?"),metrics.stringWidth("here?")),breakAfterComposer.getStringWidth());
}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)TC_BasicStringRenderer.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.util.BasicStringRenderer;
import com.rolemodelsoft.drawlet.util.StringRenderer;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Toolkit;
import junit.framework.*;

public class TC_BasicStringRenderer extends TestCase {
	protected StringRenderer renderer;
	protected StringRenderer bigRenderer;
	protected StringRenderer multilineRenderer;
	protected StringRenderer bigIndentRenderer;
	protected StringRenderer nothingRenderer;
	protected StringRenderer emptyLineRenderer;
/**
 * BasicStringRendererTest constructor comment.
 * @param name java.lang.String
 */
public TC_BasicStringRenderer(String name) {
	super(name);
}
/**
 * Sets up the fixture, for example, open a network connection.
 * This method is called before a test is executed.
 */
protected void setUp() 
{
	renderer = new BasicStringRenderer("this is a test of the emergency HotDraw system ");
	bigRenderer = new BasicStringRenderer("this is a test of the emergency HotDraw system",new Font("SansSerif",Font.BOLD,18));
	multilineRenderer = new BasicStringRenderer("this is a\n\tindented and more \ncomplex test of the emergency HotDraw system",new Font("Serif",Font.PLAIN,12));
	bigIndentRenderer = new BasicStringRenderer("this is a\n\t\t\t\t\t\treally indented and more \ncomplex test of the emergency HotDraw system",new Font("Serif",Font.PLAIN,12));
	nothingRenderer = new BasicStringRenderer("");
	emptyLineRenderer = new BasicStringRenderer("\n");
}
/**
 * Test for accurate String lines.
 */
public void testRawStringLines() 
{
	String[] lines1 = renderer.getRawStringLines();
	String[] lines2 = bigRenderer.getRawStringLines();
	String[] lines3 = multilineRenderer.getRawStringLines();
	String[] lines4 = nothingRenderer.getRawStringLines();
	String[] lines5 = emptyLineRenderer.getRawStringLines();
	assertEquals(1,lines1.length);
	assertEquals(1,lines2.length);
	assertEquals(3,lines3.length);
	assertEquals(1,lines4.length);
	assertEquals(2,lines5.length);
	assertEquals(renderer.getString(),lines1[0]);
	assertEquals(bigRenderer.getString(),lines2[0]);
	assertEquals("this is a\n",lines3[0]);
	assertEquals("\tindented and more \n",lines3[1]);
	assertEquals("complex test of the emergency HotDraw system",lines3[2]);
	assertEquals("",lines4[0]);
	assertEquals("\n",lines5[0]);
	assertEquals("",lines5[1]);
}
/**
 * Test to make sure we properly recompose after changing the fonts.
 */
public void testSetFont() 
{
	testStringWidth();
	testStringHeight();
	renderer.setFont(new Font("SansSerif",Font.BOLD,18));
	bigRenderer.setFont(new Font("Serif",Font.BOLD,14));
	multilineRenderer.setFont(new Font("Monospace",Font.BOLD,18));
	bigIndentRenderer.setFont(new Font("SansSerif",Font.BOLD,10));
	nothingRenderer.setFont(new Font("Monospace",Font.BOLD,18));
	emptyLineRenderer.setFont(new Font("SansSerif",Font.BOLD,10));
	testStringWidth();
	testStringHeight();
}
/**
 * Test to make sure we properly recompose after changing the strings.
 */
public void testSetString() 
{
	testStringWidth();
	testStringHeight();
	renderer.setString("Changed text");
	bigRenderer.setString("Changed text");
	testStringWidth();
	testStringHeight();
}
/**
 * Test for accurate String height.
 */
public void testStringHeight() 
{
	assertEquals(Toolkit.getDefaultToolkit().getFontMetrics(renderer.getFont()).getHeight(),renderer.getStringHeight());
	assertEquals(Toolkit.getDefaultToolkit().getFontMetrics(bigRenderer.getFont()).getHeight(),bigRenderer.getStringHeight());
	assertEquals(3 * Toolkit.getDefaultToolkit().getFontMetrics(multilineRenderer.getFont()).getHeight(),multilineRenderer.getStringHeight());
	assertEquals(Toolkit.getDefaultToolkit().getFontMetrics(nothingRenderer.getFont()).getHeight(),nothingRenderer.getStringHeight());
	assertEquals(2 * Toolkit.getDefaultToolkit().getFontMetrics(emptyLineRenderer.getFont()).getHeight(),emptyLineRenderer.getStringHeight());
}
/**
 * Test for accurate String lines.
 */
public void testStringLines() 
{
	String[] lines1 = renderer.getStringLines();
	String[] lines2 = bigRenderer.getStringLines();
	String[] lines3 = multilineRenderer.getStringLines();
	String[] lines4 = nothingRenderer.getStringLines();
	String[] lines5 = emptyLineRenderer.getStringLines();
	assertEquals(1,lines1.length);
	assertEquals(1,lines2.length);
	assertEquals(3,lines3.length);
	assertEquals(1,lines4.length);
	assertEquals(2,lines5.length);
	assertEquals(renderer.getString().trim(),lines1[0]);
	assertEquals(bigRenderer.getString(),lines2[0]);
	assertEquals("this is a",lines3[0]);
	assertEquals("\tindented and more",lines3[1]);
	assertEquals("complex test of the emergency HotDraw system",lines3[2]);
	assertEquals("",lines4[0]);
	assertEquals("",lines5[0]);
	assertEquals("",lines5[1]);
}
/**
 * Test for accurate String widths.
 */
public void testStringWidth() 
{
	assertEquals(Toolkit.getDefaultToolkit().getFontMetrics(renderer.getFont()).stringWidth(renderer.getString()),renderer.getStringWidth());
	assertEquals(Toolkit.getDefaultToolkit().getFontMetrics(bigRenderer.getFont()).stringWidth(bigRenderer.getString()),bigRenderer.getStringWidth());
	FontMetrics metrics = Toolkit.getDefaultToolkit().getFontMetrics(multilineRenderer.getFont());
	String[] lines = multilineRenderer.getStringLines();
	assertEquals(metrics.stringWidth(lines[2]),multilineRenderer.getStringWidth());
	FontMetrics indentMetrics = Toolkit.getDefaultToolkit().getFontMetrics(bigIndentRenderer.getFont());
	String[] indentLines = bigIndentRenderer.getStringLines();
	assert("Expected wider than last line",indentMetrics.stringWidth(indentLines[2]) < bigIndentRenderer.getStringWidth());
	assert("Expected wider than raw 2nd line",indentMetrics.stringWidth(indentLines[1]) < bigIndentRenderer.getStringWidth());
	assertEquals(0,nothingRenderer.getStringWidth());
	assertEquals(0,emptyLineRenderer.getStringWidth());
}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)TS_SegmentIntersection.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import com.rolemodelsoft.drawlet.util.GraphicsGeometry;
import java.awt.Toolkit;
import junit.framework.*;
import junit.ui.*;
import java.awt.*;
import junit.framework.*;



public class TC_SegmentIntersection extends TestCase {
	protected static boolean run = false;
/**
 * BasicStringRendererTest constructor comment.
 * @param name java.lang.String
 */
public TC_SegmentIntersection(String name) {
	super(name);
}
/**
 * Sets up the fixture, for example, open a network connection.
 * This method is called before a test is executed.
 */
protected void setUp() 
{

}
/**
 * 
 */
public void testCollinearIntersection() {
int segment1x1 = 50;
int segment1y1 = 50;
int segment1x2 = 50;
int segment1y2 = 100;
int segment2x1 = 50;
int segment2y1 = 30;
int segment2x2 = 50;
int segment2y2 = 70;
	
	assert( "Collinear lines should intersect", GraphicsGeometry.segmentIntersectsSegment(segment1x1, segment1y1, segment1x2, segment1y2, segment2x1, segment2y1, segment2x2, segment2y2) );
	
	}
/**
 * 
 */
public void testIntersectsLine1() {
	int[] line1 = {50, 50, 50, 100};
	int[][] cros1 = {	{50, 30, 50, 50},
						{50, 30, 50, 51},				
						{50, 100, 50, 130},
						{50, 99, 50, 130},
						{25, 30, 50, 50},
						{25, 30, 50, 51},
						{75, 30, 50, 50},
						{75, 30, 50, 51},
						{25, 130, 50, 100},
						{25, 130, 50, 99},
						{75, 130, 50, 100},
						{75, 130, 50, 99},
						{25, 50, 50, 50},
						{25, 50, 51, 50},
						{25, 50, 50, 51},
						{25, 50, 51, 51},
						{75, 50, 50, 50},
						{75, 50, 49, 50},
						{75, 50, 50, 51},
						{75, 50, 49, 51},
						{25, 100, 50, 100},
						{25, 100, 51, 100},
						{25, 100, 50, 99},
						{25, 100, 51, 99},
						{75, 100, 50, 100},
						{75, 100, 49, 100},
						{75, 100, 50, 99},
						{75, 100, 49, 99},
						{49, 50, 51, 100},
						{51, 50, 49, 100},
						{25, 50, 75, 100},
						{75, 50, 25, 100},
						{25, 60, 75, 90},
						{75, 60, 25, 90},
						{25, 40, 75, 110},
						{75, 40, 25, 110},
						{25, 50, 50, 75},
						{75, 50, 50, 75},
						{25, 100, 50, 75},
						{75, 100, 50, 75},
						{50, 50, 50, 100},
						{50, 30, 50, 130},
						{50, 60, 50, 90},
						{25, 75, 50, 75},
						{25, 75, 51, 75},
						{75, 75, 50, 75},
						{75, 75, 49, 75}
						};
	visualize(line1, cros1, "Line 1 Intersection");
	for(int i = 0; i < cros1.length; i++){
		assert( "Lines should intersect vertical", GraphicsGeometry.segmentIntersectsSegment(line1[0], line1[1], line1[2], line1[3], cros1[i][0], cros1[i][1], cros1[i][2], cros1[i][3]) );
	}
}
/**
 * 
 */
public void testIntersectsLine2() {
	int[] line2 = {100, 100, 200, 100};
	int[][] cros2 = {	{80, 100, 100, 100},
						{80, 100, 101, 100},
						{220, 100, 200, 100},
						{220, 100, 199, 100},
						{80, 80, 100, 100},
						{80, 80, 101, 100},
						{80, 120, 100, 100},
						{80, 120, 101, 100},
						{100, 80, 100, 100},
						{100, 80, 100, 101},
						{100, 80, 101, 100},
						{100, 80, 101, 101},						
						{100, 120, 100, 100},
						{100, 120, 100, 99},
						{100, 120, 101, 100},
						{100, 120, 101, 99},
						{200, 80, 200, 100},
						{200, 80, 200, 101},
						{200, 80, 199, 100},
						{200, 80, 199, 101},						
						{200, 120, 200, 100},
						{200, 120, 200, 99},
						{200, 120, 199, 100},
						{200, 120, 199, 99},
						{220, 80, 200, 100},
						{220, 80, 199, 100},
						{220, 120, 200, 100},
						{220, 120, 199, 100},
						{100, 100, 200, 100},
						{80, 100, 220, 100},
						{120, 100, 180, 100},
						{150, 80, 150, 100},
						{150, 80, 150, 101},
						{150, 80, 151, 100},
						{150, 80, 151, 101},
						{150, 120, 150, 100},
						{150, 120, 150, 99},
						{150, 120, 151, 100},
						{150, 120, 151, 99},
						{100, 99, 200, 101},
						{100, 101, 200, 99},
						{100, 80, 150, 100},
						{100, 80, 149, 101},
						{100, 120, 150, 100},
						{100, 120, 151, 99},
						{200, 80, 150, 100},
						{200, 80, 149, 101},
						{200, 120, 150, 100},
						{200, 120, 151, 99},
	};
	visualize(line2, cros2, "Line 2 Intersection");
	for(int i = 0; i < cros2.length; i++){
		assert( "Lines should intersect horizontal", GraphicsGeometry.segmentIntersectsSegment(line2[0], line2[1], line2[2], line2[3], cros2[i][0], cros2[i][1], cros2[i][2], cros2[i][3]) );
	}
}
/**
 * 
 */
public void testIntersectsLine3() {
	int[] line3 = {100, 50, 200, 45};
	int[][] cros3 = {	{80, 50, 100, 50},
						{80, 50, 101, 50},
						{80, 40, 100, 50},
						{80, 40, 101, 50},
						{80, 60, 100, 50},
						{80, 60, 110, 45},
						{100, 51, 200, 44},
						{100, 49, 200, 46},
						{100, 50, 200, 45},
						{120, 49, 140, 48},
						{80, 51, 100, 50},
						{80, 51, 120, 49},
						{80, 51, 220, 44},
						{220, 44, 200, 45},
						{220, 44, 180, 46},
						{220, 24, 200, 45},
						{220, 64, 200, 45},
						{220, 24, 199, 46},
						{220, 64, 199, 45},
						{100, 30, 100, 50},
						{100, 30, 100, 51},
						{100, 70, 100, 50},
						{100, 70, 100, 49},
						{100, 30, 101, 50},
						{100, 70, 101, 49},
						{200, 25, 200, 45},
						{200, 25, 200, 46},
						{200, 25, 199, 46},
						{200, 65, 200, 45},
						{200, 65, 199, 45},
						{140, 25, 140, 48},
						{140, 25, 140, 50},
						{140, 25, 120, 49},
						{140, 25, 160, 47},
						{140, 75, 140, 48},
						{140, 75, 140, 45},
						{140, 75, 120, 49},
						{140, 75, 160, 47},
						{139, 25, 141, 75},
						{141, 25, 139, 75}
	};
	visualize(line3, cros3, "Line 3 Intersection");
	for(int i = 0; i < cros3.length; i++){
		assert( "Lines should intersect small positive slope", GraphicsGeometry.segmentIntersectsSegment(line3[0], line3[1], line3[2], line3[3], cros3[i][0], cros3[i][1], cros3[i][2], cros3[i][3]) );
	}
}
/**
 * 
 */
public void testIntersectsLine4() {
	int[] line4 = {250, 45, 350, 50};
	int[][] cros4 = {	{230, 45, 250, 45},
						{230, 45, 251, 45},
						{230, 25, 250, 45},
						{230, 65, 250, 45},
						{230, 25, 251, 46},
						{230, 65, 251, 45},
						{250, 25, 250, 45},
						{250, 65, 250, 45},
						{250, 25, 250, 46},
						{250, 65, 250, 44},
						{250, 45, 350, 50},
						{230, 44, 370, 51},
						{270, 46, 290, 47},
						{380, 50, 350, 50},
						{380, 50, 349, 50},
						{380, 30, 350, 50},
						{380, 70, 350, 50},
						{380, 30, 348, 51},
						{380, 70, 348, 49},
						{250, 44, 350, 51},
						{250, 46, 350, 49},
						{290, 25, 290, 47},
						{290, 25, 290, 48},
						{289, 25, 290, 47},
						{290, 25, 270, 46},
						{290, 25, 310, 48},
						{310, 35, 309, 55}
	};
	visualize(line4, cros4, "Line 4 Intersection");
	for(int i = 0; i < cros4.length; i++){
		assert( "Lines should intersect negative small slope", GraphicsGeometry.segmentIntersectsSegment(line4[0], line4[1], line4[2], line4[3], cros4[i][0], cros4[i][1], cros4[i][2], cros4[i][3]) );
	}
}
/**
 * 
 */
public void testIntersectsLine5() {
	int[] line5 = {50, 200, 55, 300};
	int[][] cros5 = {	{50, 175, 50, 200},
						{50, 175, 50, 201},
						{45, 175, 50, 200},
						{30, 175, 51, 202},
						{50, 200, 55, 300},
						{49, 180, 56, 320},
						{51, 220, 52, 240},
						{49, 200, 56, 300},
						{51, 200, 55, 300},
						{30, 240, 52, 240},
						{30, 220, 52, 240},
						{30, 260, 52, 240},
						{30, 240, 51, 220},
						{30, 240, 53, 260},
						{30, 200, 70, 200},
						{30, 300, 70, 300}	
	};		
	visualize(line5, cros5, "Line 5 Intersection");
	for(int i = 0; i < cros5.length; i++){
		assert( "Lines should intersect large negative slope", GraphicsGeometry.segmentIntersectsSegment(line5[0], line5[1], line5[2], line5[3], cros5[i][0], cros5[i][1], cros5[i][2], cros5[i][3]) );
	}
}
/**
 * 
 */
public void testIntersectsLine6() {
	int[] line6 = {105, 200, 100, 300};
	int[][] cros6 = {	{105, 180, 105, 200},
						{105, 180, 105, 201},
						{105, 200, 100, 300},
						{106, 180, 99, 320},
						{104, 220, 103, 240},
						{106, 200, 99, 300},
						{104, 200, 101, 300},
						{80, 250, 120, 250},
						{90, 250, 110, 251},
						{90, 250, 110, 249},
						{80, 240, 103, 240},
						{80, 240, 104, 220},
						{80, 240, 102, 260},
						{80, 200, 120, 200},
						{80, 300, 120, 300},
						{120, 240, 103, 240},
						{120, 240, 104, 220},
						{120, 240, 102, 260},
						{80, 180, 105, 200},
						{120, 180, 105, 200}			
	};	
	visualize(line6, cros6, "Line 6 Intersection");
	for(int i = 0; i < cros6.length; i++){
		assert( "Lines should intersect large positive slope", GraphicsGeometry.segmentIntersectsSegment(line6[0], line6[1], line6[2], line6[3], cros6[i][0], cros6[i][1], cros6[i][2], cros6[i][3]) );
	}
}
/**
 * 
 */
public void testIntersectsLine7() {
	int[] line7 = {200, 200, 300, 300};
	int[][] cros7 = {	{200, 200, 300, 300},
						{180, 180, 320, 320},
						{220, 220, 280, 280},
						{280, 300, 320, 300},
						{180, 200, 220, 200},
						{200, 300, 300, 200},
						{250, 220, 250, 270},
						{250, 220, 249, 249},
						{250, 220, 251, 251},
						{199, 200, 301, 300},
						{201, 200, 299, 300},
						{230, 250, 251, 251},
						{270, 250, 249, 249}	
	};
	visualize(line7, cros7, "Line 7 Intersection");
	for(int i = 0; i < cros7.length; i++){
		assert( "Lines should intersect regular positive slope", GraphicsGeometry.segmentIntersectsSegment(line7[0], line7[1], line7[2], line7[3], cros7[i][0], cros7[i][1], cros7[i][2], cros7[i][3]) );
	}
}
/**
 * 
 */
public void testNoIntersectsLine1() {
	int[] line1 = {50, 50, 50, 100};
	int[][] nocros1 = {	{50, 30, 50, 49},
						{50, 30, 49, 50},
						{50, 30, 51, 50},
						{30, 30, 49, 50},
						{70, 30, 51, 50},
						{49, 50, 49, 100},
						{51, 50, 51, 100},
						{50, 120, 50, 101},
						{30, 100, 49, 100},
						{70, 100, 51, 100},
						{30, 100, 49, 50},
						{70, 100, 51, 50},
						{48, 47, 52, 51}
	};
	visualize(line1, nocros1, "Line 1 Non-Intersection");
	for(int i = 0; i < nocros1.length; i++){
		assert( "Lines should not intersect vertical", !GraphicsGeometry.segmentIntersectsSegment(line1[0], line1[1], line1[2], line1[3], nocros1[i][0], nocros1[i][1], nocros1[i][2], nocros1[i][3]) );
	}
}
/**
 * 
 */
public void testNoIntersectsLine2() {
	int[] line2 = {100, 100, 200, 100};
	int[][] nocros2 = {	{100, 99, 200, 99},
						{100, 101, 200, 101},
						{80, 100, 99, 100},
						{201, 100, 220, 100},
						{99, 80, 99, 120},
						{201, 80, 201, 120},
						{150, 80, 150, 99},
						{150, 120, 150, 101},
						{150, 80, 149, 99},
						{150, 80, 151, 99},
						{150, 80, 130, 99},
						{150, 80, 170, 99},
						{150, 120, 130, 101},
						{150, 120, 170, 101},
						{100, 98, 200, 99},
						{100, 99, 200, 98},
						{100, 101, 200, 102},
						{100, 102, 200, 101},
						{89, 90, 100, 101},
						{89, 110, 100, 99},
	};
	visualize(line2, nocros2, "Line 2 Non-Intersection");
	for(int i = 0; i < nocros2.length; i++){
		assert( "Lines should not intersect horizontal", !GraphicsGeometry.segmentIntersectsSegment(line2[0], line2[1], line2[2], line2[3], nocros2[i][0], nocros2[i][1], nocros2[i][2], nocros2[i][3]) );
	}
}
/**
 * 
 */
public void testNoIntersectsLine3() {
	int[] line3 = {100, 50, 200, 45};
	int[][] nocros3 = {	{100, 49, 200, 44},
						{100, 51, 200, 46},
						{80, 50, 100, 49},
						{80, 50, 100, 51},
						{80, 50, 99, 50},
						{140, 30, 139, 48},
						{140, 70, 141, 48},
						{201, 30, 201, 70},
						{220, 30, 199, 45},
						{220, 70, 200, 46} 
	};	
	visualize(line3, nocros3, "Line 3 Non-Intersection");
	for(int i = 0; i < nocros3.length; i++){
		assert( "Lines should not intersect the small positive slope", !GraphicsGeometry.segmentIntersectsSegment(line3[0], line3[1], line3[2], line3[3], nocros3[i][0], nocros3[i][1], nocros3[i][2], nocros3[i][3]) );
	}
}
/**
 * Paint the polygon and rectangles to see what we are trying to test... visually verifying
 * This method is basically used to see why the unit tests may be failing.
 * To run them modify the run field to equal true.
 */
protected static void visualize(int[] Line, int[][] TestLine, String title) {
	if (!run)
		return;
	SegmentIntersectionTestVisualizer visualizer = new SegmentIntersectionTestVisualizer(Line, TestLine);
	visualizer.setBounds(0,75,400,350);
	visualizer.setTitle(title);
	visualizer.show();
	visualizer.toFront();
}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)TS_UtilityTests.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import junit.framework.*;
import com.rolemodelsoft.test.*;

public class TS_UtilityTests extends junit.framework.TestCase {

public TS_UtilityTests(String name) {
	super(name);
}

public static void main(java.lang.String[] args) {
	TestRunnerHelper.run();
}

public static TestSuite suite() {
	TestSuite suite = new TestSuite();
	
	suite.addTest(new TestSuite(TC_BasicStringComposer.class));
	suite.addTest(new TestSuite(TC_BasicStringRenderer.class));
	suite.addTest(new TestSuite(TC_SegmentIntersection.class));
	
	return suite;
}
}
package com.rolemodelsoft.drawlet.util;

/**
 * @(#)ValueAdapter.java
 *
 * Copyright (c) 1998-2001 RoleModel Software, Inc. (RMS). All Rights Reserved.
 *
 * Permission to use, copy, demonstrate, or modify this software
 * and its documentation for NON-COMMERCIAL or NON-PRODUCTION USE ONLY and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies and all terms of license agreed to when downloading 
 * this software are strictly followed.
 *
 * RMS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. RMS SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

import java.lang.reflect.*;

/**
 * This is a quick and dirty version of the Adapter pattern which observes some model
 * for changes in some aspect, and updates a target with a one argument message based
 * on the new value of that aspect.  There are a lot of other and more flexible ways to
 * accomplish similar goals.  This was an attempt to get a minimal version up and prove
 * the concept.
 *
 * @version 	1.1.6, 12/30/98
 */
public class ValueAdapter implements Observer {
	/**
	 * The model.
	 */
	Object model;

	/**
	 * The aspect.
	 */
	String aspect;

	/**
	 * The target.
	 */
	Object target;

	/**
	 * The affect.
	 */
	String affect;
	/**
	 * Answer an instance that affects its target whenever the identified aspect of
	 * the identified model changes.
	 *
	 * @param model the subject being watched.
	 * @param aspect the aspect of the subject of interest... tied to a getter message.
	 * @param target the object who will be affected when the aspect changes.
	 * @param affect the message sent to the target when the aspect changes, with the value of the aspect as its argument.
	 */
	public ValueAdapter(Observable model, String aspect, Object target, String affect) {
		this.model = model;
		model.addObserver(this);
		this.aspect = aspect;
		this.target = target;
		this.affect = affect;
	}
	/**
	 * Answer an instance that affects its target whenever the identified aspect of
	 * the identified model changes.
	 *
	 * @param model the subject being watched.
	 * @param aspect the aspect of the subject of interest... tied to a getter message.
	 * @param target the object who will be affected when the aspect changes.
	 * @param affect the message sent to the target when the aspect changes, with the value of the aspect as its argument.
	 */
	public ValueAdapter(Object model, String aspect, Object target, String affect) {
		this.model = model;
		this.aspect = aspect;
		this.target = target;
		this.affect = affect;
	}
	/**
	 * Provides the method corresponding to the parameters or null if one can't be found.
	 *
	 * @param targetClass the class which theoretically holds the method we'd like to find.
	 * @param message the name of the method we'd like to find.
	 * @param targetParameterClasses the classes of the arguments of the method we'd like to find.
	 * @return the specified Method, or null if it can't be found.
	 */
	protected Method getMessage(Class targetClass, String message, Class[] targetParameterClasses) {
		try { 
			return targetClass.getMethod(affect, targetParameterClasses);
		} catch (NoSuchMethodException e) {
			return null;
		}
	}
	/**
	 * If being updated by the subject with regard to the aspect of interest, update the target.
	 *
	 * @param subject the Observable who notified us of the change.
	 * @param arg the identifier of the aspect which changed.
	 */
	public void update(Observable subject, Object arg) {
		if (arg == aspect && subject == model) {
			try {
				updateTarget();
			} catch (NoSuchMethodException e) { System.out.println(e);
			} catch (IllegalAccessException e) { System.out.println(e);
			} catch (InvocationTargetException e) { System.out.println(e);
			} 
		}
	}
	/**
	 * Send the one argument message to the target with the value as the argument.
	 *
	 * @exception NoSuchMethodException If we just can't find a method that seems to fit the criteria.
	 * @exception IllegalAccessException If access is not permitted to the method (needs to be public?).
	 * @exception InvocationTargetException If an error occurred after invoking the method.
	 */
	public void updateTarget() throws NoSuchMethodException, IllegalAccessException, InvocationTargetException {
		Method targetMessage;
		Object targetArgument = value();
		Class targetClass = target.getClass();
		Class targetParameterClass = targetArgument.getClass();
		Class targetParameterClasses[] = new Class[1];
		do {
			targetParameterClasses[0] = targetParameterClass;
			targetMessage = getMessage(targetClass, affect, targetParameterClasses);
			if (targetMessage == null) {
				Class interfaces[] = targetParameterClass.getInterfaces();
				for (int i = 0; i < interfaces.length; i++) {
					targetParameterClasses[0] = interfaces[i];
					targetMessage = getMessage(targetClass, affect, targetParameterClasses);
					if (targetMessage != null)
						break;
				}
				targetParameterClass = targetParameterClass.getSuperclass();
			}
		} while ((targetMessage == null) && (targetParameterClass != null));
		if (targetMessage == null)
			throw new NoSuchMethodException(affect + "not found in " + target.getClass().getName());
		Object targetMessageParameters[] = new Object[] {targetArgument};
		targetMessage.invoke(target, targetMessageParameters);
	}
	/**
	 * Get the value of interest from the subject.
	 *
	 * @return the value (an Object) of interest to the subject.
	 *
	 * @exception NoSuchMethodException If we just can't find a method that seems to fit the criteria.
	 * @exception IllegalAccessException If access is not permitted to the method (needs to be public?).
	 * @exception InvocationTargetException If an error occurred after invoking the method.
	 */
	public Object value() throws NoSuchMethodException, IllegalAccessException, InvocationTargetException {
		Class subjectClass = model.getClass();
		Method subjectMessage = subjectClass.getMethod(aspect, new Class[0]);
		return subjectMessage.invoke(model, new Object[0]);
	}
}
package com.rolemodelsoft.test;

import junit.ui.*;
import org.apache.oro.text.regex.*;
import java.util.*;
import java.io.*;

public class TestRunnerHelper {
protected TestRunnerHelper() {
	super();
}
protected static String getLineContainingTestSuiteName(String exceptionString) 
{

	StringTokenizer tokenizer = new StringTokenizer(exceptionString, System.getProperty("line.separator"));
	String line = "";
	while (tokenizer.hasMoreTokens())
	{
		line = (String)tokenizer.nextToken();
		if (line.indexOf("TS_") == -1)
			line = "";
		else
			break;
	}
	
	if (line.equals(""))
		throw new RuntimeException("could not find the test suite to run.");

	
	return line;
}
protected static String getTestSuiteClassName(String exceptionStack) {
	Perl5Matcher matcher = new Perl5Matcher();
	Perl5Compiler compiler = new Perl5Compiler();
	Pattern beginningCompiledPattern, endCompiledPattern = null;

	String beginningPattern = "^\\s*\\w+\\s+";
	String endPattern = "\\.main.*$";
	try
	{
		beginningCompiledPattern = compiler.compile(beginningPattern);
		endCompiledPattern = compiler.compile(endPattern);
	}
	catch (MalformedPatternException e1)
	{
		throw new RuntimeException("Unsupported pattern: " + beginningPattern + "\nOr: " + endPattern);
	}

	String line = getLineContainingTestSuiteName(exceptionStack);
	line = Util.substitute(matcher, beginningCompiledPattern, new StringSubstitution(""), line);
	line = Util.substitute(matcher, endCompiledPattern, new StringSubstitution(""), line);
	line = line.trim();
	line = line.replace('/','.');
	
	return line;
}
public static void run() {
	RuntimeException e = new RuntimeException();
	e.fillInStackTrace();
	String exceptionText;
	try {
		throw e;
	} catch ( RuntimeException e2 ) { 
		StringWriter writer = new StringWriter();
		e2.printStackTrace(new PrintWriter(writer));
		exceptionText = writer.toString();
	}
	new TestRunner().main(new String[] {getTestSuiteClassName(exceptionText)});
}
}
package junit.framework;

/**
 * A set of assert methods.
 */

public class Assert {
	public Assert() {
	}
	/**
	 * Asserts that a condition is true. If it isn't it throws
	 * an AssertionFailedError with the given message.
	 */
	public void assert(String message, boolean condition) {
		onAssertion();
		if (!condition)
			fail(message);
	}
	/**
	 * Asserts that a condition is true. If it isn't it throws
	 * an AssertionFailedError.
	 */
	public void assert(boolean condition) {
		assert(null, condition);
	}
	/**
	 * Asserts that two doubles are equal.
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 * @param delta tolerated delta
	 */
	public void assertEquals(double expected, double actual, double delta) {
	    assertEquals(null, expected, actual, delta);
	}
	/**
	 * Asserts that two longs are equal.
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 */
	public void assertEquals(long expected, long actual) {
	    assertEquals(null, expected, actual);
	}
	/**
	 * Asserts that two objects are equal. If they are not
	 * an AssertionFailedError is thrown.
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 */
	public void assertEquals(Object expected, Object actual) {
	    assertEquals(null, expected, actual);
	}
	/**
	 * Asserts that two doubles are equal.
	 * @param message the detail message for this assertion
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 * @param delta tolerated delta
	 */
	public void assertEquals(String message, double expected, double actual, double delta) {
		onAssertion();
	    if (Math.abs(expected-actual) > delta)
			failNotEquals(message, new Double(expected), new Double(actual));
	}
	/**
	 * Asserts that two longs are equal.
	 * @param message the detail message for this assertion
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 */
	public void assertEquals(String message, long expected, long actual) {
	    assertEquals(message, new Long(expected), new Long(actual));
	}
	/**
	 * Asserts that two objects are equal. If they are not
	 * an AssertionFailedError is thrown.
	 * @param message the detail message for this assertion
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 */
	public void assertEquals(String message, Object expected, Object actual) {
		onAssertion();
		if (expected == null && actual == null)
			return;
		if (expected != null && expected.equals(actual))
			return;
		failNotEquals(message, expected, actual);
	}
	/**
	 * Asserts that an object isn't null.
	 */
	public void assertNotNull(Object object) {
		assertNotNull(null, object);
	}
	/**
	 * Asserts that an object isn't null.
	 */
	public void assertNotNull(String message, Object object) {
		assert(message, object != null); 
	}
	/**
	 * Asserts that an object is null.
	 */
	public void assertNull(Object object) {
		assertNull(null, object);
	}
	/**
	 * Asserts that an object is null.
	 */
	public void assertNull(String message, Object object) {
		assert(message, object == null); 
	}
	/**
	 * Asserts that two objects refer to the same object. If they are not
	 * the same an AssertionFailedError is thrown.
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 */
	public void assertSame(Object expected, Object actual) {
	    assertSame(null, expected, actual);
	}
	/**
	 * Asserts that two objects refer to the same object. If they are not
	 * an AssertionFailedError is thrown.
	 * @param message the detail message for this assertion
	 * @param expected the expected value of an object
	 * @param actual the actual value of an object
	 */
	public void assertSame(String message, Object expected, Object actual) {
		onAssertion();
		if (expected == actual)
			return;
		failNotSame(message, expected, actual);
	}
	/**
	 * Fails a test with no message. 
	 */
	public void fail() {
		fail(null);
	}
	/**
	 * Fails a test with the given message. 
	 */
	public void fail(String message) {
		throw new AssertionFailedError(message);
	}
	protected void failNotEquals(String message, Object expected, Object actual) {
		String formatted= "";
		if (message != null)
			formatted= message+" ";
		fail(formatted+"expected:<"+expected+"> but was:<"+actual+">");
	}
	protected void failNotSame(String message, Object expected, Object actual) {
		String formatted= "";
		if (message != null)
			formatted= message+" ";
		fail(formatted+"expected same");
	}
/**
 * This method is called every time an assertion is made (this does not include direct fails).
 */
protected void onAssertion()
{
}
}
package junit.framework;

/**
 * Thrown when an assertion failed.
 */
public class AssertionFailedError extends Error {

	public AssertionFailedError () {
	}
	public AssertionFailedError (String message) {
		super (message);
	}
}
package junit.framework;

/**
 * A <em>Protectable</em> can be run and can throw a Throwable.
 *
 * @see TestResult
 */
public interface Protectable {

	/**
	 * Run the the following method protected.
	 */
	public abstract void protect() throws Throwable;
}
package junit.framework;

/**
 * A <em>Test</em> can be run and collect its results.
 *
 * @see TestResult
 */
public interface Test {
	/**
	 * Counts the number of test cases that will be run by this test.
	 */
	public abstract int countTestCases();
	/**
	 * Runs a test and collects its result in a TestResult instance.
	 */
	public abstract void run(TestResult result);
}
package junit.framework;

import java.lang.reflect.*;

/**
 * A test case defines the fixture to run multiple tests. To define a test case<br>
 * 1) implement a subclass of TestCase<br>
 * 2) define instance variables that store the state of the fixture<br>
 * 3) initialize the fixture state by overriding <code>setUp</code><br>
 * 4) clean-up after a test by overriding <code>tearDown</code>.<br>
 * Each test runs in its own fixture so there
 * can be no side effects among test runs.
 * Here is an example:
 * <pre>
 * public class MathTest extends TestCase {
 *     protected double fValue1;
 *     protected double fValue2;
 *
 *     public MathTest(String name) {
 *         super(name);
 *     }
 *
 *    protected void setUp() {
 *         fValue1= 2.0;
 *         fValue2= 3.0;
 *     }
 * }
 * </pre>
 *
 * For each test implement a method which interacts
 * with the fixture. Verify the expected results with assertions specified
 * by calling <code>assert</code> with a boolean.
 * <pre>
 *    protected void testAdd() {
 *        double result= fValue1 + fValue2;
 *        assert(result == 5.0);
 *    }
 * </pre>
 * Once the methods are defined you can run them. The framework supports
 * both a static type safe and more dynamic way to run a test.
 * In the static way you override the runTest method and define the method to
 * be invoked. A convenient way to do so is with an anonymous inner class.
 * <pre>
 * Test test= new MathTest("add") {
 *        public void runTest() {
 *            testAdd();
 *        }
 * };
 * test.run();
 * </pre>
 * The dynamic way uses reflection to implement <code>runTest</code>. It dynamically finds
 * and invokes a method.
 * In this case the name of the test case has to correspond to the test method
 * to be run.
 * <pre>
 * Test= new MathTest("testAdd");
 * test.run();
 * </pre>
 * The tests to be run can be collected into a TestSuite. JUnit provides
 * different <i>test runners</i> which can run a test suite and collect the results.
 * A test runner either expects a static method <code>suite</code> as the entry
 * point to get a test to run or it will extract the suite automatically.
 * <pre>
 * public static Test suite() {
 *      suite.addTest(new MathTest("testAdd"));
 *      suite.addTest(new MathTest("testDivideByZero"));
 *      return suite;
 *  }
 * </pre>
 * @see TestResult
 * @see TestSuite
 */

public abstract class TestCase extends Assert implements Test {
	/**
	 * the name of the test case
	 */
	private final String fName;

	protected TestResult result;
	/**
	 * Constructs a test case with the given name.
	 */
	public TestCase(String name) {
		fName= name;
	}
	/**
	 * Counts the number of test cases executed by run(TestResult result).
	 */
	public int countTestCases() {
		return 1;
	}
	/**
	 * Creates a default TestResult object
	 *
	 * @see TestResult
	 */
	protected TestResult createResult() {
	    return new TestResult();
	}
	/**
	 * Gets the name of the test case.
	 */
	public String name() {
		return fName;
	}
protected void onAssertion()
{
	result.assertionMade();
}
	/**
	 * A convenience method to run this test, collecting the results with a
	 * default TestResult object.
	 *
	 * @see TestResult
	 */
	public TestResult run() {
		TestResult result= createResult();
		run(result);
		return result;
	}
	/**
	 * Runs the test case and collects the results in TestResult.
	 */
	public void run(TestResult result) {
		this.result = result;
		result.run(this);
	}
	/**
	 * Runs the bare test sequence.
	 * @exception Throwable if any exception is thrown
	 */
	public void runBare() throws Throwable {
		setUp();
		try {
			runTest();
		}
		finally {
			tearDown();
		}
	}
	/**
	 * Override to run the test and assert its state.
	 * @exception Throwable if any exception is thrown
	 */
	protected void runTest() throws Throwable {
		Method runMethod= null;
		try {
			// use getMethod to get all public inherited
			// methods. getDeclaredMethods returns all
			// methods of this class but excludes the
			// inherited ones.
			runMethod= getClass().getMethod(fName, new Class[0]);
		} catch (NoSuchMethodException e) {
			fail("Method \""+fName+"\" not found");
		}
		if (runMethod != null && !Modifier.isPublic(runMethod.getModifiers())) {
			fail("Method \""+fName+"\" should be public");
		}

		try {
			runMethod.invoke(this, new Class[0]);
		}
		catch (InvocationTargetException e) {
			e.fillInStackTrace();
			throw e.getTargetException();
		}
		catch (IllegalAccessException e) {
			e.fillInStackTrace();
			throw e;
		}
	}
	/**
	 * Sets up the fixture, for example, open a network connection.
	 * This method is called before a test is executed.
	 */
	protected void setUp() throws Exception {
	}
	/**
	 * Tears down the fixture, for example, close a network connection.
	 * This method is called after a test is executed.
	 */
	protected void tearDown() throws Exception {
	}
	/**
	 * Returns a string representation of the test case
	 */
	public String toString() {
	    return name()+"("+getClass().getName()+")";
	}
}
package junit.framework;

/**
 * A <code>TestFailure</code> collects a failed test together with
 * the caught exception.
 * @see TestResult
 */
public class TestFailure extends Object {
	protected Test fFailedTest;
	protected Throwable fThrownException;

	/**
	 * Constructs a TestFailure with the given test and exception.
	 */
	public TestFailure(Test failedTest, Throwable thrownException) {
		fFailedTest= failedTest;
		fThrownException= thrownException;
	}
	/**
	 * Gets the failed test.
	 */
	public Test failedTest() {
	    return fFailedTest;
	}
	/**
	 * Gets the thrown exception.
	 */
	public Throwable thrownException() {
	    return fThrownException;
	}
	/**
	 * Returns a short description of the failure.
	 */
	public String toString() {
	    StringBuffer buffer= new StringBuffer();
	    buffer.append(fFailedTest+": "+fThrownException.getMessage());
	    return buffer.toString();
	}
}
package junit.framework;

/**
 * A Listener for test progress
 */
public interface TestListener {
   /**
 	* An error occurred.
 	*/
	public void addError(Test test, Throwable t);
   /**
 	* A failure occurred.
 	*/
 	public void addFailure(Test test, Throwable t);
   /**
 	* A test ended.
 	*/
 	public void endTest(Test test);
   /**
 	* A test started.
 	*/
	public void startTest(Test test);
}
package junit.framework;

import java.util.Vector;
import java.util.Enumeration;

/**
 * A <code>TestResult</code> collects the results of executing
 * a test case. It is an instance of the Collecting Parameter pattern.
 * The test framework distinguishes between <i>failures</i> and <i>errors</i>.
 * A failure is anticipated and checked for with assertions. Errors are
 * unanticipated problems like an <code>ArrayIndexOutOfBoundsException</code>.
 *
 * @see Test
 */
public class TestResult extends Object {
	protected Vector fFailures;
	protected Vector fErrors;
	protected Vector fListeners;
	protected int fRunTests, fAssertions;
	private boolean fStop;
	
	public TestResult() {
		fFailures= new Vector();
		fErrors= new Vector();
		fListeners= new Vector();
		fRunTests= 0;
		fStop= false;
	}
	/**
	 * Adds an error to the list of errors. The passed in exception
	 * caused the error.
	 */
	public void addError(Test test, Throwable t) {
		fErrors.addElement(new TestFailure(test, t));
		for (Enumeration e= cloneListeners().elements(); e.hasMoreElements(); ) {
			((TestListener)e.nextElement()).addError(test, t);
		}
	}
	/**
	 * Adds a failure to the list of failures. The passed in exception
	 * caused the failure.
	 */
	public synchronized void addFailure(Test test, AssertionFailedError t) {
		fFailures.addElement(new TestFailure(test, t));
		for (Enumeration e= cloneListeners().elements(); e.hasMoreElements(); ) {
			((TestListener)e.nextElement()).addFailure(test, t);
		}
	}
	/**
	 * Registers a TestListener
	 */
	public synchronized void addListener(TestListener listener) {
		fListeners.addElement(listener);
	}
	/**
	 * Gets the number of detected failures.
	 */
	public synchronized int assertionCount() {
		return fAssertions;
	}
	/**
	 * Adds a failure to the list of failures. The passed in exception
	 * caused the failure.
	 */
	public synchronized void assertionMade() {
		fAssertions++;
	}
	/**
	 * Returns a copy of the listeners.
	 */
	private synchronized Vector cloneListeners() {
		return (Vector)fListeners.clone();
	}
	/**
	 * Informs the result that a test was completed.
	 */
	public void endTest(Test test) {
		for (Enumeration e= cloneListeners().elements(); e.hasMoreElements(); ) {
			((TestListener)e.nextElement()).endTest(test);
		}
	}
	/**
	 * Gets the number of detected errors.
	 */
	public synchronized int errorCount() {
		return fErrors.size();
	}
	/**
	 * Returns an Enumeration for the errors
	 */
	public synchronized Enumeration errors() {
		return fErrors.elements();
	}
	/**
	 * Gets the number of detected failures.
	 */
	public synchronized int failureCount() {
		return fFailures.size();
	}
	/**
	 * Returns an Enumeration for the failures
	 */
	public synchronized Enumeration failures() {
		return fFailures.elements();
	}
	/**
	 * Runs a TestCase.
	 */
	protected void run(final TestCase test) {
		startTest(test);
		Protectable p= new Protectable() {
			public void protect() throws Throwable {
				test.runBare();
			}
		};
		runProtected(test, p);
		endTest(test);
	}
	/**
	 * Gets the number of run tests.
	 */
	public synchronized int runCount() {
		return fRunTests;
	}
	/**
	 * Runs a TestCase.
	 */
	public void runProtected(final Test test, Protectable p) {
		try {
			p.protect();
		} 
		catch (AssertionFailedError e) {
			addFailure(test, e);
		}
		catch (Throwable e) {
			addError(test, e);
		}
	}
	/**
	 * Gets the number of run tests.
	 * @deprecated use <code>runCount</code> instead
	 */
	public synchronized int runTests() {
		return runCount();
	}
	/**
	 * Checks whether the test run should stop
	 */
	public synchronized boolean shouldStop() {
		return fStop;
	}
	/**
	 * Informs the result that a test will be started.
	 */
	public void startTest(Test test) {
		synchronized(this) {
			fRunTests++;
		}
		for (Enumeration e= cloneListeners().elements(); e.hasMoreElements(); ) {
			((TestListener)e.nextElement()).startTest(test);
		}
	}
	/**
	 * Marks that the test run should stop.
	 */
	public synchronized void stop() {
		fStop= true;
	}
	/**
	 * Gets the number of detected errors.
	 * @deprecated use <code>errorCount</code> instead
	 */
	public synchronized int testErrors() {
		return errorCount();
	}
	/**
	 * Gets the number of detected failures.
	 * @deprecated use <code>failureCount</code> instead
	 */
	public synchronized int testFailures() {
		return failureCount();
	}
	/**
	 * Returns whether the entire test was successful or not.
	 */
	public synchronized boolean wasSuccessful() {
		return testFailures() == 0 && testErrors() == 0;
	}
}
package junit.framework;

import java.util.Vector;
import java.util.Enumeration;
import java.lang.reflect.*;

/**
 * A <code>TestSuite</code> is a <code>Composite</code> of Tests.
 * It runs a collection of test cases. Here is an example using
 * the dynamic test definition.
 * <pre>
 * TestSuite suite= new TestSuite();
 * suite.addTest(new MathTest("testAdd"));
 * suite.addTest(new MathTest("testDivideByZero"));
 * </pre>
 * Alternatively, a TestSuite can extract the tests to be run automatically.
 * To do so you pass the class of your TestCase class to the
 * TestSuite constructor.
 * <pre>
 * TestSuite suite= new TestSuite(MathTest.class);
 * </pre>
 * This constructor creates a suite with all the methods
 * starting with "test" that take no arguments.
 *
 * @see Test
 */
public class TestSuite implements Test {

	private Vector fTests= new Vector(10);
	private String fName;

   /**
	 * Constructs an empty TestSuite.
	 */
	public TestSuite() {
	}
	/**
	 * Constructs a TestSuite from the given class. Adds all the methods
	 * starting with "test" as test cases to the suite.
	 * Parts of this method was written at 2337 meters in the Hffihtte,
	 * Kanton Uri
	 */

	 public TestSuite(final Class theClass) {
		fName= theClass.getName();	
		Constructor constructor= getConstructor(theClass);
		if (!Modifier.isPublic(theClass.getModifiers())) {
			addTest(warning("Class "+theClass.getName()+" is not public"));
			return;

		}
		if (constructor == null) {
			addTest(warning("Class "+theClass.getName()+" has no public constructor TestCase(String name)"));
			return;
		}

		Class superClass= theClass;
		Vector names= new Vector();
		while (Test.class.isAssignableFrom(superClass)) {
			Method[] methods= superClass.getDeclaredMethods();
			for (int i= 0; i < methods.length; i++) {
				addTestMethod(methods[i], names, constructor);
			}
			superClass= superClass.getSuperclass();
		}
		if (fTests.size() == 0)
			addTest(warning("No tests found in "+theClass.getName()));
	}
   /**
	 * Constructs an empty TestSuite.
	 */
	public TestSuite(String name) {
		fName= name;
	}
	/**
	 * Adds a test to the suite.
	 */
	public void addTest(Test test) {
		fTests.addElement(test);
	}
	private void addTestMethod(Method m, Vector names, Constructor constructor) {
		String name= m.getName();
		if (names.contains(name)) 
			return;
		if (isPublicTestMethod(m)) {
			names.addElement(name);

			Object[] args= new Object[]{name};
			try {
				addTest((Test)constructor.newInstance(args));
			} catch (Exception t) {
				addTest(warning("Cannot instantiate test case: "+name));
			}
		} else { // almost a test method
			if (isTestMethod(m)) 
				addTest(warning("Test method isn't public: "+m.getName()));
		}
	}
	/**
	 * Counts the number of test cases that will be run by this test.
	 */
	public int countTestCases() {
		int count= 0;
		for (Enumeration e= tests(); e.hasMoreElements(); ) {
			Test test= (Test)e.nextElement();
			count= count + test.countTestCases();
		}
		return count;
	}
	/**
	 * Gets a constructor which takes a single String as
	 * its argument.
	 */
	private Constructor getConstructor(Class theClass) {
		Class[] args= { String.class };
		Constructor c= null;
		try {
			c= theClass.getConstructor(args);
		} catch(Exception e) {
		}
		return c;
	}
	/**
	 */
	private boolean isPublicTestMethod(Method m) {
		return isTestMethod(m) && Modifier.isPublic(m.getModifiers());
	 }
	/**
	 */
	private boolean isTestMethod(Method m) {
		String name= m.getName();
		Class[] parameters= m.getParameterTypes();
		Class returnType= m.getReturnType();
		return parameters.length == 0 && name.startsWith("test") && returnType.equals(Void.TYPE);
	 }
	/**
	 * Runs the tests and collects their result in a TestResult.
	 */
	public void run(TestResult result) {
		while (fTests.size() > 0) {
	  		if (result.shouldStop() )
	  			break;
			((Test)fTests.elementAt(0)).run(result);
			fTests.removeElementAt(0);
		}
	}
	/**
	 * Returns the test at the given index 
	 */
	public Test testAt(int index) {
		return (Test)fTests.elementAt(index);
	}
	/**
	 * Returns the number of tests in this suite 
	 */
	public int testCount() {
		return fTests.size();
	}
	/**
	 * Returns the tests as an enumeration
	 */
	public Enumeration tests() {
		return fTests.elements();
	}
	/**
	 */
	public String toString() {
		if (fName != null)
			return fName;
		return super.toString();
	 }
	/**
	 * Returns a test which will fail and log a warning message.
	 */
	 private Test warning(final String message) {
		return new TestCase("warning") {
			protected void runTest() {
				fail(message);
			}
		};		
	}
}
package junit.ui;

import java.awt.*;
import java.awt.event.*;

import junit.util.Version;

class AboutDialog extends Dialog {
	public AboutDialog(Frame parent) {
		super(parent);
		
		setResizable(false);
		setLayout(new GridBagLayout());
		setSize(296, 138);
		setTitle("About");
		
		Button button= new Button("Close");
		button.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					dispose();
				}
			}
		);
		
		Label label1= new Label("JUnit");
		label1.setFont(new Font("dialog", Font.PLAIN, 36));
		
		Label label2= new Label("JUnit "+Version.id()+ " by Kent Beck and Erich Gamma");
		label2.setFont(new Font("dialog", Font.PLAIN, 14));
		
		Logo logo= new Logo();

		GridBagConstraints constraintsLabel1= new GridBagConstraints();
		constraintsLabel1.gridx = 3; constraintsLabel1.gridy = 0;
		constraintsLabel1.gridwidth = 1; constraintsLabel1.gridheight = 1;
		constraintsLabel1.anchor = GridBagConstraints.CENTER;
		constraintsLabel1.weightx = 0.0;
		constraintsLabel1.weighty = 0.0;
		add(label1, constraintsLabel1);

		GridBagConstraints constraintsLabel2= new GridBagConstraints();
		constraintsLabel2.gridx = 2; constraintsLabel2.gridy = 1;
		constraintsLabel2.gridwidth = 2; constraintsLabel2.gridheight = 1;
		constraintsLabel2.anchor = GridBagConstraints.CENTER;
		constraintsLabel2.weightx = 0.0;
		constraintsLabel2.weighty = 0.0;
		add(label2, constraintsLabel2);

		GridBagConstraints constraintsButton1= new GridBagConstraints();
		constraintsButton1.gridx = 2; constraintsButton1.gridy = 2;
		constraintsButton1.gridwidth = 2; constraintsButton1.gridheight = 1;
		constraintsButton1.anchor = GridBagConstraints.CENTER;
		constraintsButton1.weightx = 0.0;
		constraintsButton1.weighty = 0.0;
		constraintsButton1.insets= new Insets(8, 0, 8, 0);
		add(button, constraintsButton1);

		GridBagConstraints constraintsLogo1= new GridBagConstraints();
		constraintsLogo1.gridx = 2; constraintsLogo1.gridy = 0;
		constraintsLogo1.gridwidth = 1; constraintsLogo1.gridheight = 1;
		constraintsLogo1.anchor = GridBagConstraints.CENTER;
		constraintsLogo1.weightx = 0.0;
		constraintsLogo1.weighty = 0.0;
		add(logo, constraintsLogo1);

		addWindowListener(
			new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					dispose();
				}
			}
		);
	}
}
package junit.ui;

import java.awt.*;
import java.awt.image.*;

class Logo extends Canvas {
	private Image fImage;
	private int fWidth;
	private int fHeight;
	public Logo() {
		fImage= loadImage("logo.gif");
		MediaTracker tracker= new MediaTracker(this);
	  	tracker.addImage(fImage, 0);
		try {
			tracker.waitForAll();
		} catch (Exception e) {
		}

		if (fImage != null) {
			fWidth= fImage.getWidth(this);
			fHeight= fImage.getHeight(this);
		} else {
			fWidth= 20;
			fHeight= 20;
		}
		setSize(fWidth, fHeight);
	}
	public Image loadImage(String name) {
		Toolkit toolkit= Toolkit.getDefaultToolkit();
		try {
			java.net.URL url= getClass().getResource(name);
			return toolkit.createImage((ImageProducer) url.getContent());
		} catch (Exception ex) {
		}
		return null;
	}
	public void paint(Graphics g) {
		paintBackground(g);
		if (fImage != null)
			g.drawImage(fImage, 0, 0, fWidth, fHeight, this);
	}
	public void paintBackground( java.awt.Graphics g) {
		g.setColor(SystemColor.control);
		g.fillRect(0, 0, getBounds().width, getBounds().height);
	}
}
package junit.ui;

import java.awt.*;

class ProgressBar extends Canvas {
	public boolean fError= false;
	public int fTotal= 0;
	public int fProgress= 0;
	public int fProgressX= 0;
	public ProgressBar() {
		super();
		setSize(20, 30);
	}
	private Color getStatusColor() {
		if (fError)
			return Color.red;
		return Color.green;
	}
	public void paint(Graphics g) {
		paintBackground(g);
		paintStatus(g);
	}
	public void paintBackground(Graphics g) {
		g.setColor(SystemColor.control);
		g.fillRect(0, 0, getBounds().width, getBounds().height);
		g.setColor(Color.darkGray);
		g.drawLine(0, 0, getBounds().width-1, 0);
		g.drawLine(0, 0, 0, getBounds().height-1);
		g.setColor(Color.white);
		g.drawLine(getBounds().width-1, 0, getBounds().width-1, getBounds().height-1);
		g.drawLine(0, getBounds().height-1, getBounds().width-1, getBounds().height-1);
	}
	public void paintStatus(Graphics g) {
		g.setColor(getStatusColor());
		Rectangle r= new Rectangle(0, 0, fProgressX, getBounds().height);
		g.fillRect(1, 1, r.width-1, r.height-2);
	}
	private void paintStep(int startX, int endX) {
		repaint(startX, 1, endX-startX, getBounds().height-2);
	}
	public void reset() {
		fProgressX= 1;
		fProgress= 0;
		fError= false;
		paint(getGraphics());
	}
	public int scale(int value) {
		if (fTotal > 0)
			return Math.max(1, value*(getBounds().width-1)/fTotal);
		return value; 
	}
	public void setBounds(int x, int y, int w, int h) {
		super.setBounds(x, y, w, h);
		fProgressX= scale(fProgress);
	}
	public void start(int total) {
		fTotal= total;
		reset();
	}
	public void step(boolean successful) {
		fProgress++;
		int x= fProgressX;

		fProgressX= scale(fProgress);

		if (!fError && !successful) {
			fError= true;
			x= 1;
		}
		paintStep(x, fProgressX);
	}
}
package junit.ui;

import junit.framework.*;
import junit.util.*;

import java.util.Stack;
import java.util.Vector;
import java.lang.reflect.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
 
/**
 * A simple user interface to run tests.
 * Enter the name of a class with a suite method which should return
 * the tests to be run.
 * <pre>
 * Synopsis: java java.ui.TestRunner [TestCase]
 * </pre>
 * TestRunner takes as an optional argument the name of the testcase class to be run.
 */
public class TestRunner extends Object implements TestListener {
	// the following were added to capture times of individual tests
	protected Stack startTimes = new Stack();
	protected long tooLong = 1000;
	protected Checkbox fCheckPerformance;
	protected TextField fPerformanceField;

	protected Frame fFrame;
	protected Vector fExceptions;
	protected Vector fFailedTests;
	private Thread fRunner;
	private TestResult fTestResult;
	protected TestSuiteLoader fTestLoader;
	
	private TraceFrame fTraceFrame;

	private TextField fSuiteField;
	private Button fRun;
	private ProgressBar fProgressIndicator;
	protected List fFailureList;
	private Logo fLogo;
	private Label fNumberOfErrors;
	private Label fNumberOfFailures;
	private Label fNumberOfRuns;
	private Label fNumberOfAsserts;
	private Button fQuitButton;
	private Button fShowErrorButton;
	private Button fRerunButton;
	private TextField fStatusLine;
	private Panel fPanel;
	
	private static Font PLAIN_FONT= new Font("dialog", Font.PLAIN, 12);
	private static final int GAP= 4;
	private static final String SUITE_METHODNAME= "suite";
	public TestRunner() {
	}
	private void about() {
		AboutDialog about= new AboutDialog(fFrame);
		about.setModal(true);
		about.setLocation(300, 300);
		about.setVisible(true);
	}
	public void addError(Test test, Throwable t) {
		fNumberOfErrors.setText(Integer.toString(fTestResult.errorCount()));
		appendFailure("Error", test, t);
	}
	public void addFailure(Test test, Throwable t) {
		fNumberOfFailures.setText(Integer.toString(fTestResult.failureCount()));
		appendFailure("Failure", test, t);
	}
	private void addGrid(Panel p, Component co, int x, int y, int w, int fill, double wx, int anchor) {
		GridBagConstraints c= new GridBagConstraints();
		c.gridx= x; c.gridy= y;
		c.gridwidth= w;
		c.anchor= anchor;
		c.weightx= wx;
		c.fill= fill;
		if (fill == GridBagConstraints.BOTH || fill == GridBagConstraints.VERTICAL)
			c.weighty= 1.0;
		c.insets= new Insets(y == 0 ? GAP : 0, x == 0 ? GAP : 0, GAP, GAP);
		p.add(co, c);
	}
	protected void appendFailure(String kind, Test test, Throwable t) {
		kind+= ": " + test;
		String msg= t.getMessage();
		if (msg != null) {
			kind+= ":" + StringUtil.truncate(msg, 100); 
		}
		fFailureList.add(kind);
		fExceptions.addElement(t);
		fFailedTests.addElement(test);
	}
protected void appendPerformance(Test test, long milliseconds) {
	if (!showPerformance())
		return;
	String explanation = "Slow: " + milliseconds + ": " + test;
	fFailureList.add(explanation);
	fExceptions.addElement(new RuntimeException("Test ran greater than " + tooLong));
	fFailedTests.addElement(test);
}
	/**
	 * Creates the JUnit menu. Clients override this
	 * method to add additional menu items.
	 */
	protected Menu createJUnitMenu() {
		Menu menu = new Menu("JUnit");
		MenuItem mi = new MenuItem("About...");
		mi.addActionListener(
		    new ActionListener() {
		        public void actionPerformed(ActionEvent event) {
		            about();
		        }
		    }
		);
		menu.add(mi);
		
		menu.addSeparator();
		mi = new MenuItem("Exit");
		mi.addActionListener(
		    new ActionListener() {
		        public void actionPerformed(ActionEvent event) {
		            System.exit(0);
		        }
		    }
		);
		menu.add(mi);

		return menu;
	}
	protected void createMenus(MenuBar mb) {
		mb.add(createJUnitMenu());
	}
	protected TestResult createTestResult() {
		return new TestResult();
	}
	protected Frame createUI(String suiteName) {	
		Frame frame= new Frame("Run Test Suite");
		Image icon= loadFrameIcon();	
		if (icon != null)
			frame.setIconImage(icon);

		frame.setLayout(new BorderLayout(0, 0));
		frame.setBackground(SystemColor.control);
		final Frame finalFrame= frame;
		
		frame.addWindowListener(
			new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					finalFrame.dispose();
					System.exit(0);
				}
			}
		); 

		MenuBar mb = new MenuBar();
		createMenus(mb);
		frame.setMenuBar(mb);
		
		//---- first section
		Label suiteLabel= new Label("Enter the name of the TestCase class:");

		fSuiteField= new TextField(suiteName != null ? suiteName : "");
		fSuiteField.selectAll();
		fSuiteField.requestFocus();
		fSuiteField.setFont(PLAIN_FONT);
		fSuiteField.setColumns(40);
		fSuiteField.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runSuite();
				}
			}
		);
		fSuiteField.addTextListener(
			new TextListener() {
				public void textValueChanged(TextEvent e) {
					fRun.setEnabled(fSuiteField.getText().length() > 0);
					fStatusLine.setText("");
				}
			}
		);
		
		Label suiteLabel2= new Label(".suite()");
		
		fRun= new Button("Run");
		fRun.setEnabled(false);
		fRun.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					runSuite();
				}
			}
		);
		//----- inserted the stuff below
		fCheckPerformance= new Checkbox("Performance",false);
		fPerformanceField= new TextField(String.valueOf(tooLong));
		fPerformanceField.selectAll();
		fPerformanceField.requestFocus();
		fPerformanceField.setFont(PLAIN_FONT);
		fPerformanceField.setColumns(8);
		fPerformanceField.addTextListener(
			new TextListener() {
				public void textValueChanged(TextEvent evt) {
					String forcedPrefix = "**";
					if (!fPerformanceField.getText().startsWith(forcedPrefix)) {
						try {
							tooLong = (new Long(fPerformanceField.getText())).longValue();
						} catch (NumberFormatException e) {
							tooLong = 1000;
							fPerformanceField.setText(forcedPrefix + tooLong);
							fPerformanceField.selectAll();
							fPerformanceField.requestFocus();
						}
					}
				}
			}
		);
		//----- inserted the stuff above
	

		//---- second section
		Label progressLabel= new Label("Progress:");
		fProgressIndicator= new ProgressBar();	

		//---- third section
		fNumberOfErrors= new Label("0000", Label.RIGHT);
		fNumberOfErrors.setText("0");
		fNumberOfErrors.setFont(PLAIN_FONT);
	
		fNumberOfFailures= new Label("0000", Label.RIGHT);
		fNumberOfFailures.setText("0");
		fNumberOfFailures.setFont(PLAIN_FONT);
	
		fNumberOfRuns= new Label("0000", Label.RIGHT) {
			public Dimension getPreferredSize() {
				return new Dimension( 50, javax.swing.text.StyleContext.getDefaultStyleContext ().getFontMetrics(PLAIN_FONT).getHeight() );
			}
		};
		
		fNumberOfRuns.setText("0");
		fNumberOfRuns.setFont(PLAIN_FONT);
	
		fNumberOfAsserts= new Label("0000", Label.RIGHT) {
			public Dimension getPreferredSize() {
				return new Dimension( 50, javax.swing.text.StyleContext.getDefaultStyleContext ().getFontMetrics(PLAIN_FONT).getHeight() );
			}
		};
		
		fNumberOfAsserts.setText("0");
		fNumberOfAsserts.setFont(PLAIN_FONT);

		Panel numbersPanel= new Panel(new FlowLayout());
		numbersPanel.add(new Label("Runs:"));			numbersPanel.add(fNumberOfRuns);
		numbersPanel.add(new Label("   Assertions:"));	numbersPanel.add(fNumberOfAsserts);
		numbersPanel.add(new Label("   Errors:"));		numbersPanel.add(fNumberOfErrors);
		numbersPanel.add(new Label("   Failures:"));	numbersPanel.add(fNumberOfFailures);

	
		//---- fourth section
		Label failureLabel= new Label("Errors and Failures:");
		
		fFailureList= new List(6);
		fFailureList.addMouseListener(
			new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					if (e.getClickCount() >= 2)
						showErrorTrace();
				}
			}
		);
		fFailureList.addItemListener(
			new ItemListener() {
				public void itemStateChanged(ItemEvent e) {
					fShowErrorButton.setEnabled(isErrorSelected());
					fRerunButton.setEnabled(isErrorSelected());
				}
			}
		);

		fShowErrorButton= new Button("Show...");
		fShowErrorButton.setEnabled(false);
		fShowErrorButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					showErrorTrace();
				}
			}
		);

		fRerunButton= new Button("Run");
		fRerunButton.setEnabled(false);
		fRerunButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					rerun();
				}
			}
		);

		Panel failedPanel= new Panel(new GridLayout(0, 1, 0, 2));
		failedPanel.add(fShowErrorButton);
		failedPanel.add(fRerunButton);
		
		//---- fifth section
		fStatusLine= new TextField();
		fStatusLine.setFont(PLAIN_FONT);
		fStatusLine.setEditable(false);
		fStatusLine.setForeground(Color.red);

		fQuitButton= new Button("Exit");
		fQuitButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					System.exit(0);
				}
			}
		);
	
		// ---------
		fLogo= new Logo();
	
		//---- overall layout
		Panel panel= new Panel(new GridBagLayout());
		fPanel= panel;
	
		addGrid(panel, suiteLabel,		 0, 0, 2, GridBagConstraints.HORIZONTAL, 	1.0, GridBagConstraints.WEST);
		
		addGrid(panel, fSuiteField, 	 0, 1, 1, GridBagConstraints.HORIZONTAL, 	1.0, GridBagConstraints.WEST);
		addGrid(panel, suiteLabel2, 	 1, 1, 1, GridBagConstraints.NONE, 			0.0, GridBagConstraints.WEST);
		addGrid(panel, fRun, 			 2, 1, 1, GridBagConstraints.HORIZONTAL, 	0.0, GridBagConstraints.CENTER);
		// added these two lines
		addGrid(panel, fCheckPerformance, 2, 2, 1, GridBagConstraints.HORIZONTAL, 	0.0, GridBagConstraints.CENTER);
		addGrid(panel, fPerformanceField, 2, 3, 1, GridBagConstraints.HORIZONTAL, 	0.0, GridBagConstraints.CENTER);

		addGrid(panel, progressLabel, 	 0, 2, 2, GridBagConstraints.HORIZONTAL, 	1.0, GridBagConstraints.WEST);
		addGrid(panel, fProgressIndicator, 0, 3, 2, GridBagConstraints.HORIZONTAL, 	1.0, GridBagConstraints.WEST);
		// changed from "2, 3, 1" to "2, 4, 1"
		addGrid(panel, fLogo, 			 2, 4, 1, GridBagConstraints.NONE, 			0.0, GridBagConstraints.NORTH);

		addGrid(panel, numbersPanel,	 0, 4, 2, GridBagConstraints.NONE, 			0.0, GridBagConstraints.CENTER);

		addGrid(panel, failureLabel, 	 0, 5, 2, GridBagConstraints.HORIZONTAL, 	1.0, GridBagConstraints.WEST);
		addGrid(panel, fFailureList, 	 0, 6, 2, GridBagConstraints.BOTH, 			1.0, GridBagConstraints.WEST);
		addGrid(panel, failedPanel, 2, 6, 1, GridBagConstraints.HORIZONTAL, 	0.0, GridBagConstraints.CENTER);
		
		addGrid(panel, fStatusLine, 	 0, 7, 2, GridBagConstraints.HORIZONTAL, 	1.0, GridBagConstraints.CENTER);
		addGrid(panel, fQuitButton, 	 2, 7, 1, GridBagConstraints.HORIZONTAL, 	0.0, GridBagConstraints.CENTER);
		
		frame.add(panel);
		
		frame.pack();
		return frame;
	}
	public void endTest(Test test) {
		//---- added this section
		long endTime = System.currentTimeMillis();
		long startTime = ((Long)startTimes.pop()).longValue();
		long runTime = endTime - startTime;
		if (runTime > tooLong)
			appendPerformance(test,runTime);
		//--- end of additions
		
		setLabelValue(fNumberOfRuns, fTestResult.runCount());
		setLabelValue(fNumberOfAsserts, fTestResult.assertionCount());
		fProgressIndicator.step(fTestResult.wasSuccessful());
	}
	private Test getTest(String suiteClassName) {
		if (suiteClassName.length() <= 0) {
			fStatusLine.setText("");
			return null;
		}
		
		Class testClass= null;
		try {
			testClass= loadSuiteClass(suiteClassName);
		} catch(Exception e) {
			runFailed("Class \""+suiteClassName+"\" not found");
			return null;
		}
			
		Method suiteMethod= null;
		try {
			suiteMethod= testClass.getMethod(SUITE_METHODNAME, new Class[0]);
	 	} catch(Exception e) {
	 		// try to extract a test suite automatically
			fStatusLine.setText("");			
			return new TestSuite(testClass);
		}
	
		Test test= null;
		try {
			test= (Test)suiteMethod.invoke(null, new Class[0]); // static method
			if (test == null)
				return test;
		} catch(Exception e) {
			runFailed("Could not invoke the suite() method " + e);
			return null;
		}
		fStatusLine.setText("");
		return test;
	}
	private boolean isErrorSelected() {
		return fFailureList.getSelectedIndex() != -1;
	}
	private Image loadFrameIcon() {
		Toolkit toolkit= Toolkit.getDefaultToolkit();
		try {
			java.net.URL url= getClass().getResource("smalllogo.gif");
			return toolkit.createImage((ImageProducer) url.getContent());
		} catch (Exception ex) {
		}
		return null;
	}
	protected Class loadSuiteClass(String suiteClassName) throws ClassNotFoundException {
		return fTestLoader.load(suiteClassName);
	}
	/**
	 * main entrypoint
	 */
	public static void main(String[] args) {
		new TestRunner().start(args, new StandardTestSuiteLoader());
	}
	protected void rerun() {
		int index= fFailureList.getSelectedIndex();
		if (index == -1)
			return;
	
		Test test= (Test)fFailedTests.elementAt(index);
		if (!(test instanceof TestCase)) {
			showInfo("Could not reload "+ test.toString());
			return;
		}
		Test reloadedTest= null;
		try {
			Class reloadedTestClass= fTestLoader.reload(test.getClass());
			Class[] classArgs= { String.class };
			Constructor constructor= reloadedTestClass.getConstructor(classArgs);
			Object[] args= new Object[]{((TestCase)test).name()};
			reloadedTest=(Test)constructor.newInstance(args);
		} catch(Exception e) {
			showInfo("Could not reload "+ test.toString());
			return;
		}
		TestResult result= new TestResult();
		reloadedTest.run(result);
		
		String message= reloadedTest.toString();
		if(result.wasSuccessful())
			showInfo(message+" was successful");
		else if (result.errorCount() == 1)
			showStatus(message+" had an error");
		else
			showStatus(message+" had a failure");
	}
	protected void reset() {
		setLabelValue(fNumberOfErrors, 0);
		setLabelValue(fNumberOfFailures, 0);
		setLabelValue(fNumberOfRuns, 0);
		fProgressIndicator.reset();
		fShowErrorButton.setEnabled(false);
		fRerunButton.setEnabled(false);
		fFailureList.removeAll();
		fExceptions= new Vector(10);
		fFailedTests= new Vector(10);
	}
	/**
	 * runs a suite.
	 * @deprecated use runSuite() instead
	 */
	public void run() {
		runSuite();
	}
	private void runFailed(String message) {
		showStatus(message);
		fRun.setLabel("Run");
		fRunner= null;
	}
	synchronized public void runSuite() {
		if (fRunner != null) {
			fTestResult.stop();
		} else {
			fRun.setLabel("Stop");
			showInfo("Initializing...");
			reset();
			
			showInfo("Load Test Case...");
			final Test testSuite= getTest(fSuiteField.getText());
			if (testSuite != null) {
				fRunner= new Thread() {
					public void run() {
						fTestResult= createTestResult();
						fTestResult.addListener(TestRunner.this);
						fProgressIndicator.start(testSuite.countTestCases());
						showInfo("Running...");
					
						long startTime= System.currentTimeMillis();
						testSuite.run(fTestResult);
						
						if (fTestResult.shouldStop()) {
							showStatus("Stopped");
						} else {
							long endTime= System.currentTimeMillis();
							long runTime= endTime-startTime;
							showInfo("Finished: " + StringUtil.elapsedTimeAsString(runTime) + " seconds");
						}
						fTestResult= null;
						fRun.setLabel("Run");
						fRunner= null;
					}
				};
				fRunner.start();
			}
		}
	}
	private void setLabelValue(Label label, int value) {
		label.setText(Integer.toString(value));
	}
	public void setSuiteName(String suite) {
		fSuiteField.setText(suite);
	}
	private void showErrorTrace() {
		int index= fFailureList.getSelectedIndex();
		if (index == -1)
			return;
	
		Throwable t= (Throwable) fExceptions.elementAt(index);
		if (fTraceFrame == null) {
			fTraceFrame= new TraceFrame();
			fTraceFrame.setBounds(20, 20, 850, 650);
	   	}
		fTraceFrame.showTrace(t);
		fTraceFrame.setVisible(true);
	}
	protected void showInfo(String message) {
		fStatusLine.setFont(PLAIN_FONT);
		fStatusLine.setForeground(Color.black);
		fStatusLine.setText(message);
	}
/**
 * Returns whether this TestRunner should show potential performance problems.
 */
protected boolean showPerformance() {
	return fCheckPerformance.getState();
}
	protected void showStatus(String status) {
		fStatusLine.setFont(PLAIN_FONT);
		fStatusLine.setForeground(Color.red);
		fStatusLine.setText(status);
	}
	/**
	 * Starts the TestRunner
	 */
	public void start(String[] args, TestSuiteLoader loader) {
		fTestLoader= loader;
		String suiteName= null;
		if (args.length == 1) 
			suiteName= args[0];
		else if (args.length == 2 && args[0].equals("-c")) 
			suiteName= StringUtil.extractClassName(args[1]);
			
		fFrame= createUI(suiteName);
		fFrame.setBounds(10, 10, 900, 700);
		fFrame.setVisible(true);
	
		if (suiteName != null) {
			setSuiteName(suiteName);
			runSuite();
		}
	}
	public void startTest(Test test) {
		startTimes.push(new Long(System.currentTimeMillis()));
		showInfo("Running: "+test);
	}
}
package junit.ui;

import java.io.*;
import java.awt.*;
import java.awt.event.*;

import junit.framework.*;
import junit.util.*;

class TraceFrame extends Frame {
	private Button fButton= null;
	private TextArea fTextArea= null;
	public TraceFrame() {
		setLayout(new GridBagLayout());
		setBackground(SystemColor.control);
		setSize(403, 236);
		setTitle("Stack Trace");

		fTextArea= new TextArea();
		fTextArea.setRows(10);
		fTextArea.setColumns(60);

		fButton= new Button("Close");
		fButton.addActionListener(
			new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					dispose();
				}
			}
		);

		GridBagConstraints constraintsStackTextArea = new GridBagConstraints();
		constraintsStackTextArea.gridx = 0; constraintsStackTextArea.gridy = 0;
		constraintsStackTextArea.gridwidth = 1; constraintsStackTextArea.gridheight = 1;
		constraintsStackTextArea.fill = GridBagConstraints.BOTH;
		constraintsStackTextArea.anchor = GridBagConstraints.CENTER;
		constraintsStackTextArea.weightx = 1.0;
		constraintsStackTextArea.weighty = 1.0;
		constraintsStackTextArea.insets = new Insets(8, 8, 8, 8);
		add(fTextArea, constraintsStackTextArea);

		GridBagConstraints constraintsCloseButton = new GridBagConstraints();
		constraintsCloseButton.gridx = 0; constraintsCloseButton.gridy = 1;
		constraintsCloseButton.gridwidth = 1; constraintsCloseButton.gridheight = 1;
		constraintsCloseButton.anchor = java.awt.GridBagConstraints.EAST;
		constraintsCloseButton.weightx = 0.0;
		constraintsCloseButton.weighty = 0.0;
		constraintsCloseButton.insets = new Insets(0, 8, 8, 8);
		add(fButton, constraintsCloseButton);
		
		addWindowListener(
			new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					dispose();
				}
			}
		);
	}
	/**
	 * Shows the stack trace of the passed in throwable
	 */
	public void showTrace(Throwable t) {
		StringWriter stringWriter= new StringWriter();
		PrintWriter writer= new PrintWriter(stringWriter);
		t.printStackTrace(writer);
		StringBuffer buffer= stringWriter.getBuffer();
		fTextArea.setText(StringUtil.truncate(buffer.toString(), 5000));
	}
}
package junit.util;

/**
 * The standard test suite loader. It can only load the same class once.
 */
public class StandardTestSuiteLoader implements TestSuiteLoader {
	/**
	 * Uses the system class loader to load the test class
	 */
	public Class load(String suiteClassName) throws ClassNotFoundException {
		return Class.forName(suiteClassName);
	}
	/**
	 * Uses the system class loader to load the test class
	 */
	public Class reload(Class aClass) throws ClassNotFoundException {
		return aClass;
	}
}
package junit.util;

import java.text.NumberFormat;

/**
 * A class with junit related String utilities.
 */
public class StringUtil {
	protected StringUtil() {
	}
	public static String elapsedTimeAsString(long runTime) {
		return NumberFormat.getInstance().format((double)runTime/1000);
	}
	public static String extractClassName(String className) {
		if(className.startsWith("Default package for")) 
			return className.substring(className.lastIndexOf(".")+1);
		return className;
	}
	static public String truncate(String s, int length) {
		if (s.length() > length)
			s= s.substring(0, length)+"...";
		return s;
	}
}
package junit.util;

/**
 * An interface to define how a test suite should be loaded.
 */
public interface TestSuiteLoader {
	abstract public Class load(String suiteClassName) throws ClassNotFoundException;
	abstract public Class reload(Class aClass) throws ClassNotFoundException;
}
package junit.util;

/**
 * This class defines the current version of JUnit
 */
public class Version {
	private Version() {
		// don't instantiate
	}
	public static String id() {
		return "3.2";
	}
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The CharStringPointer class is used to facilitate traversal of a char[]
 * in the manner pointer traversals of strings are performed in C/C++.
 * It is expected that the compiler will inline all the functions.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: CharStringPointer.java,v 1.2 2000/07/23 23:25:24 jon Exp $

 */
final class CharStringPointer {
  static final char _END_OF_STRING = Character.MAX_VALUE;
  int _offset;
  char[] _array;

  CharStringPointer(char[] charArray) {
    this(charArray, 0);
  }
  CharStringPointer(char[] charArray, int offset) {
    _array  = charArray;
    _offset = offset;
  }
  char _decrement() { return _decrement(1); }
  char _decrement(int inc) {
    _offset-=inc; 
    if(_offset < 0)
      _offset = 0;

    return _array[_offset];
  }
  int _getLength() { return _array.length; }
  int _getOffset() { return _offset; }
  char _getValue()  {
    return _getValue(_offset);
  }
  char _getValue(int offset) {
    if(offset < _array.length && offset >= 0)
      return _array[offset];
    return _END_OF_STRING;
  }
  char _getValueRelative(int offset) {
    return _getValue(_offset + offset);
  }
  char _increment() { return _increment(1); }
  char _increment(int inc) {
    _offset+=inc;
    if(_isAtEnd()) {
      _offset = _array.length;
      return _END_OF_STRING;
    }

    return _array[_offset];
  }
  boolean _isAtEnd() {
    return (_offset >= _array.length);
  }
  char _postDecrement() {
    char ret;
    ret = _getValue();
    _decrement();
    return ret;
  }
  char _postIncrement() {
    char ret;
    ret = _getValue();
    _increment();
    return ret;
  }
  void _setOffset(int offset) { _offset = offset; }
  String _toString(int offset) {
    return new String(_array, offset, _array.length - offset);
  }
  public String toString() {
    return _toString(0);
  }
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

import java.lang.*;

/**
 * A class used to signify the occurrence of a syntax error in a
 * regular expression that is being compiled.  The class is not
 * declared final so that it may be subclassed for identifying
 * more specific pattern comilation errors.  However, at this point
 * in time, this package does not subclass MalformedPatternException
 * for any purpose.  This does not preclude users and third party
 * implementors of the interfaces of this package from subclassing it
 * for their own purposes.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: MalformedPatternException.java,v 1.2 2000/07/23 23:25:24 jon Exp $

 * @see PatternCompiler
 */
public class MalformedPatternException extends Exception {

  /**
   * Simply calls the corresponding constructor of its superclass.
   */
  public MalformedPatternException() {
    super();
  }
  /**
   * Simply calls the corresponding constructor of its superclass.
   * <p>
   * @param message  A message indicating the nature of the parse error.
   */
  public MalformedPatternException(String message) {
    super(message);
  }
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The MatchResult interface allows PatternMatcher implementors to return
 * results storing match information in whatever format they like, while
 * presenting a consistent way of accessing that information.  However,
 * MatchResult implementations should strictly follow the behavior
 * described for the interface methods.
 * <p>
 *
 * A MatchResult instance contains a pattern match and its saved groups.
 * You can access the entire match directly using the
 * {@link #group(int)} method with an argument of 0,
 * or by the {@link #toString()} method which is
 * defined to return the same thing.  It is also possible to obtain
 * the beginning and ending offsets of a match relative to the input
 * producing the match by using the 
 * {@link #beginOffset(int)} and {@link #endOffset(int)} methods.  The
 * {@link #begin(int)} and {@link #end(int)} are useful in some
 * circumstances and return the begin and end offsets of the subgroups
 * of a match relative to the beginning of the match.
 * <p>
 *
 * You might use a MatchResult as follows:
 * <blockquote><pre>
 * int groups;
 * PatternMatcher matcher;
 * PatternCompiler compiler;
 * Pattern pattern;
 * PatternMatcherInput input;
 * MatchResult result;
 *
 * compiler = new Perl5Compiler();
 * matcher  = new Perl5Matcher();
 *
 * try {
 *   pattern = compiler.compile(somePatternString);
 * } catch(MalformedPatternException e) {
 *   System.out.println("Bad pattern.");
 *   System.out.println(e.getMessage());
 *   return;
 * }
 *
 * input   = new PatternMatcherInput(someStringInput);
 *
 * while(matcher.contains(input, pattern)) {
 *   result = matcher.getMatch();  
 *   // Perform whatever processing on the result you want.
 *   // Here we just print out all its elements to show how its
 *   // methods are used.
 * 
 *   System.out.println("Match: " + result.toString());
 *   System.out.println("Length: " + result.length());
 *   groups = result.groups();
 *   System.out.println("Groups: " + groups);
 *   System.out.println("Begin offset: " + result.beginOffset(0));
 *   System.out.println("End offset: " + result.endOffset(0));
 *   System.out.println("Saved Groups: ");
 *
 *   // Start at 1 because we just printed out group 0
 *   for(int group = 1; group < groups; group++) {
 *	 System.out.println(group + ": " + result.group(group));
 *	 System.out.println("Begin: " + result.begin(group));
 *	 System.out.println("End: " + result.end(group));
 *   }
 * }
 * </pre></blockquote>

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: MatchResult.java,v 1.2 2000/07/23 23:25:24 jon Exp $

 * @see PatternMatcher
 */

public interface MatchResult {
  /**
   * @param group The pattern subgroup.
   * @return The offset into group 0 of the first token in the indicated
   *         pattern subgroup.  If a group was never matched or does
   *         not exist, returns -1.  Be aware that a group that matches
   *         the null string at the end of a match will have an offset
   *         equal to the length of the string, so you shouldn't blindly
   *         use the offset to index an array or String.
   */
  public int begin(int group);
  /**
   * Returns an offset marking the beginning of the pattern match
   * relative to the beginning of the input from which the match
   * was extracted.
   * <p>
   * @param group The pattern subgroup.
   * @return The offset of the first token in the indicated
   *         pattern subgroup.  If a group was never matched or does
   *         not exist, returns -1.
   */
  public int beginOffset(int group);
  /**
   * @param group The pattern subgroup.
   * @return Returns one plus the offset into group 0 of the last token in
   *         the indicated pattern subgroup.  If a group was never matched
   *         or does not exist, returns -1.  A group matching the null
   *         string will return its start offset.
   */
  public int end(int group);
  /**
   * Returns an offset marking the end of the pattern match
   * relative to the beginning of the input from which the match was
   * extracted.
   * <p>
   * @param group The pattern subgroup.
   * @return Returns one plus the offset of the last token in
   *         the indicated pattern subgroup.  If a group was never matched
   *         or does not exist, returns -1.  A group matching the null
   *         string will return its start offset.
   */
  public int endOffset(int group);
  /**
   * Returns the contents of the parenthesized subgroups of a match,
   * counting parentheses from left to right and starting from 1.
   * Group 0 always refers to the entire match.  For example, if the
   * pattern <code> foo(\d+) </code> is used to extract a match
   * from the input <code> abfoo123 </code>, then <code> group(0) </code>
   * will return <code> foo123 </code> and <code> group(1) </code> will return
   * <code> 123 </code>.  <code> group(2) </code> will return
   * <code> null </code> because there is only one subgroup in the original
   * pattern.
   * <p>
   * @param group The pattern subgroup to return.
   * @return A string containing the indicated pattern subgroup.  Group
   *         0 always refers to the entire match.  If a group was never
   *         matched, it returns null.  This is not to be confused with
   *         a group matching the null string, which will return a String
   *         of length 0.
   */
  public String group(int group);
  /**
   * @return The number of groups contained in the result.  This number
   *         includes the 0th group.  In other words, the result refers
   *         to the number of parenthesized subgroups plus the entire match
   *         itself.
   */
  public int groups();
  /**
   * A convenience method returning the length of the entire match.
   * If you want to get the length of a particular subgroup you should
   * use the {@link #group(int)} method to get
   * the string and then access its length() method as follows:
   * <p>
   * <blockquote><pre>
   * int length = -1; // Use -1 to indicate group doesn't exist
   * MatchResult result;
   * String subgroup;
   * 
   * // Initialization of result omitted
   *
   * subgroup = result.group(1);
   * if(subgroup != null)
   *   length = subgroup.length();
   *
   * </pre></blockquote>
   * <p>
   *
   * The length() method serves as a more a more efficient way to do:
   * <p>
   * <blockquote><pre>
   * length = result.group(0).length();
   * </pre></blockquote>
   * <p>
   *
   * @return The length of the match.
   */
  public int length();
  /**
   * Returns the same as group(0).
   *
   * @return A string containing the entire match.
   */
  public String toString();
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The OpCode class should not be instantiated.  It is a holder of various
 * constants and static methods pertaining to the manipulation of the 
 * op-codes used in a compiled regular expression.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: OpCode.java,v 1.2 2000/07/23 23:25:24 jon Exp $
 */
final class OpCode {

  // Names, values, and descriptions of operators correspond to those of
  // Perl regex bytecodes and for compatibility purposes are drawn from
  // regcomp.h in the Perl source tree by Larry Wall.
  static final char  // Has Operand   Meaning
	 _END     = 0,   // no       End of program.
	 _BOL     = 1,   // no       Match "" at beginning of line.
	 _MBOL    = 2,   // no       Same, assuming multiline.
	 _SBOL    = 3,   // no       Same, assuming singleline.
	 _EOL     = 4,   // no       Match "" at end of line.
	 _MEOL    = 5,   // no       Same, assuming multiline.
	 _SEOL    = 6,   // no       Same, assuming singleline.
	 _ANY     = 7,   // no       Match any one character (except newline).
	 _SANY    = 8,   // no       Match any one character.
	 _ANYOF   = 9,   // yes      Match character in (or not in) this class.
	 _CURLY   = 10,  // yes      Match this simple thing {n,m} times.
	 _CURLYX  = 11,  // yes      Match this complex thing {n,m} times.
	 _BRANCH  = 12,  // yes      Match this alternative, or the next...
	 _BACK    = 13,  // no       Match "", "next" ptr points backward.
	 _EXACTLY = 14,  // yes      Match this string (preceded by length).
	 _NOTHING = 15,  // no       Match empty string.
	 _STAR    = 16,  // yes      Match this (simple) thing 0 or more times.
	 _PLUS    = 17,  // yes      Match this (simple) thing 1 or more times.
	 _ALNUM   = 18,  // no       Match any alphanumeric character
	 _NALNUM  = 19,  // no       Match any non-alphanumeric character
	 _BOUND   = 20,  // no       Match "" at any word boundary
	 _NBOUND  = 21,  // no       Match "" at any word non-boundary
	 _SPACE   = 22,  // no       Match any whitespace character
	 _NSPACE  = 23,  // no       Match any non-whitespace character
	 _DIGIT   = 24,  // no       Match any numeric character
	 _NDIGIT  = 25,  // no       Match any non-numeric character
	 _REF     = 26,  // yes      Match some already matched string
	 _OPEN    = 27,  // yes      Mark this point in input as start of #n.
	 _CLOSE   = 28,  // yes      Analogous to OPEN.
	 _MINMOD  = 29,  // no       Next operator is not greedy.
	 _GBOL    = 30,  // no       Matches where last m//g left off.
	 _IFMATCH = 31,  // no       Succeeds if the following matches.
	 _UNLESSM = 32,  // no       Fails if the following matches.
	 _SUCCEED = 33,  // no       Return from a subroutine, basically.
	 _WHILEM  = 34;  // no       Do curly processing and see if rest matches.

  // Lengths of the various operands.
  static final int _operandLength[] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0
  };

  static final char _opType[] = {
	_END, _BOL, _BOL, _BOL, _EOL, _EOL, _EOL, _ANY, _ANY, _ANYOF, _CURLY,
	_CURLY, _BRANCH, _BACK, _EXACTLY, _NOTHING, _STAR, _PLUS, _ALNUM,
	_NALNUM, _BOUND, _NBOUND, _SPACE, _NSPACE, _DIGIT, _NDIGIT, _REF,
	_OPEN, _CLOSE, _MINMOD,	_BOL, _BRANCH, _BRANCH, _END, _WHILEM
  };

  static final char _opLengthVaries[] = {
	_BRANCH, _BACK, _STAR, _PLUS, _CURLY, _CURLYX, _REF, _WHILEM
  };

  static final char _opLengthOne[] = {
	_ANY, _SANY, _ANYOF, _ALNUM, _NALNUM, _SPACE, _NSPACE, _DIGIT, _NDIGIT
  };

  static final int  _NULL_OFFSET  = -1;
  static final char _NULL_POINTER =  0;

  private OpCode() { }
  static final char _getArg1(char[] program, int offset) {
    return program[offset + 2]; 
  }
  static final char _getArg2(char[] program, int offset) {
    return program[offset + 3]; 
  }
  static final int _getNext(char[] program, int offset) {
    int offs;

    if(program == null)
      return _NULL_OFFSET;


    offs = _getNextOffset(program, offset);
    if(offs == _NULL_POINTER)
      return _NULL_OFFSET;

    if(program[offset] == OpCode._BACK)
      return (offset - offs);

    return (offset + offs);
  }
  static final int _getNextOffset(char[] program, int offset) {
    return ((int)program[offset + 1]); 
  }
  static final int _getNextOperator(int offset) { return (offset + 2); }
  static final int _getOperand(int offset) {
    return (offset + 2);
  }
  static final int _getPrevOperator(int offset) { return (offset - 2); }
  static final boolean _isInArray(char ch, char[] array, int start) {
    while(start < array.length)
      if(ch == array[start++])
	return true;
    return false;
  }
  // doesn't really belong in this class, but we want Perl5Matcher not to
  // depend on Perl5Compiler
  static final boolean _isWordCharacter(char token) {
    return ((token >= 'a' && token <= 'z') ||
            (token >= 'A' && token <= 'Z') ||
            (token >= '0' && token <= '9') ||
            (token == '_'));
  }
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The Pattern interface allows multiple representations of a regular
 * expression to be defined.  In general, different regular expression
 * compilers will produce different types of pattern representations.
 * Some will produce state transition tables derived from syntax trees,
 * others will produce byte code representations of an NFA, etc.  The
 * Pattern interface does not impose any specific internal pattern
 * representation, and consequently, Pattern implementations are not meant
 * to be interchangeable among differing PatternCompiler and PatternMatcher
 * implementations.  The documentation accompanying a specific implementation
 * will define what other classes a Pattern can interact with.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: Pattern.java,v 1.2 2000/07/23 23:25:24 jon Exp $

 * @see PatternCompiler
 * @see PatternMatcher
 */
public interface Pattern {

  /**
   * This method returns an integer containing the compilation options used
   * to compile this pattern.
   * <p>
   * @return The compilation options used to compile the pattern.
   */
  public int getOptions();
  /**
   * This method returns the string representation of the pattern.  Its
   * purpose is to allow a pattern to be reconstructed after compilation.
   * In other words, when you compile a pattern, the resulting data 
   * structures bear no relation to the string defining the pattern.
   * It is often useful to be able to access the string defining a pattern 
   * after it has been compiled.
   * <p>
   * @return The original string representation of the regular expression
   *         pattern.
   */
  public String getPattern();
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The PatternCompiler interface defines the operations a regular
 * expression compiler must implement.  However, the types of
 * regular expressions recognized by a compiler and the Pattern 
 * implementations produced as a result of compilation are not
 * restricted.
 * <p>
 * A PatternCompiler instance is used to compile the string representation
 * (either as a String or char[]) of a regular expression into a Pattern
 * instance.  The Pattern can then be used in conjunction with the appropriate
 * PatternMatcher instance to perform pattern searches.  A form
 * of use might be:
 * <p>
 * <blockquote><pre>
 * PatternCompiler compiler;
 * PatternMatcher matcher;
 * Pattern pattern;
 * String input;
 *
 * // Initialization of compiler, matcher, and input omitted;
 *
 * try {
 *   pattern = compiler.compile("\\d+");
 * } catch(MalformedPatternException e) {
 *   System.out.println("Bad pattern.");
 *   System.out.println(e.getMessage());
 *   System.exit(1);
 * }
 * 
 *
 * if(matcher.matches(input, pattern))
 *    System.out.println(input + " is a number");
 * else
 *    System.out.println(input + " is not a number");
 *
 * </pre></blockquote>
 * <p>
 * Specific PatternCompiler implementations such as Perl5Compiler may have
 * variations of the compile() methods that take extra options affecting
 * the compilation of a pattern.  However, the PatternCompiler method
 * implementations should provide the default behavior of the class.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: PatternCompiler.java,v 1.2 2000/07/23 23:25:24 jon Exp $

 * @see Pattern
 * @see PatternMatcher
 * @see MalformedPatternException
 */
public interface PatternCompiler {
  /**
   * Compiles a regular expression into a data structure that can be used
   * by a PatternMatcher implementation to perform pattern matching.
   * <p>
   * @param pattern  A regular expression to compile.
   * @return A Pattern instance constituting the compiled regular expression.
   * @exception MalformedPatternException  If the compiled expression
   *  does not conform to the grammar understood by the PatternCompiler or
   *  if some other error in the expression is encountered.
   */
  public Pattern compile(char[] pattern) throws MalformedPatternException;
  /**
   * Compiles a regular expression into a data structure that can be
   * used by a PatternMatcher implementation to perform pattern matching.
   * Additional regular expression syntax specific options can be passed
   * as a bitmask of options.
   * <p>
   * @param pattern  A regular expression to compile.
   * @param options  A set of flags giving the compiler instructions on
   *                 how to treat the regular expression.  The flags
   *                 are a logical OR of any number of the allowable
   *                 constants permitted by the PatternCompiler
   *                 implementation.
   * @return A Pattern instance constituting the compiled regular expression.
   * @exception MalformedPatternException  If the compiled expression
   *  does not conform to the grammar understood by the PatternCompiler or
   *  if some other error in the expression is encountered.
   */
  public Pattern compile(char[] pattern, int options)
       throws MalformedPatternException;
  /**
   * Compiles a regular expression into a data structure that can be used
   * by a PatternMatcher implementation to perform pattern matching.
   * <p>
   * @param pattern  A regular expression to compile.
   * @return A Pattern instance constituting the compiled regular expression.
   * @exception MalformedPatternException  If the compiled expression
   *  does not conform to the grammar understood by the PatternCompiler or
   *  if some other error in the expression is encountered.
   */
  public Pattern compile(String pattern) throws MalformedPatternException;
  /**
   * Compiles a regular expression into a data structure that can be
   * used by a PatternMatcher implementation to perform pattern matching.
   * Additional regular expression syntax specific options can be passed
   * as a bitmask of options.
   * <p>
   * @param pattern  A regular expression to compile.
   * @param options  A set of flags giving the compiler instructions on
   *                 how to treat the regular expression.  The flags
   *                 are a logical OR of any number of the allowable
   *                 constants permitted by the PatternCompiler
   *                 implementation.
   * @return A Pattern instance constituting the compiled regular expression.
   * @exception MalformedPatternException  If the compiled expression
   *  does not conform to the grammar understood by the PatternCompiler or
   *  if some other error in the expression is encountered.
   */
  public Pattern compile(String pattern, int options)
       throws MalformedPatternException;
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The PatternMatcher interface defines the operations a regular
 * expression matcher must implement.  However, the types of the Pattern 
 * implementations recognized by a matcher are not restricted.  Typically
 * PatternMatcher instances will only recognize a specific type of Pattern.
 * For example, the Perl5Matcher only recognizes Perl5Pattern instances.
 * However, none of the PatternMatcher methods are required to throw an
 * exception in case of the use of an invalid pattern.  This is done for
 * efficiency reasons, although usually a CastClassException will be
 * thrown by the Java runtime system if you use the wrong Pattern
 * implementation.  It is the responsibility of the programmer to make
 * sure he uses the correct Pattern instance with a given PatternMatcher
 * instance.  The current version of this package only contains the Perl5
 * suite of pattern matching classes, but future ones for other regular
 * expression grammars may be added and users may also create their own
 * implementations of the provided interfaces. Therefore the programmer
 * should be careful not to mismatch classes.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: PatternMatcher.java,v 1.2 2000/07/23 23:25:25 jon Exp $

 * @see Pattern
 * @see PatternCompiler
 * @see MatchResult
 */
public interface PatternMatcher {

  /**
   * Determines if a string (represented as a char[]) contains a pattern.
   * If the pattern is matched by some substring of the input, a MatchResult
   * instance representing the <b>first</b> such match is made acessible via 
   * {@link #getMatch()}.  If you want to access
   * subsequent matches you should either use a PatternMatcherInput object
   * or use the offset information in the MatchResult to create a substring
   * representing the remaining input.  Using the MatchResult offset 
   * information is the recommended method of obtaining the parts of the
   * string preceeding the match and following the match.
   * <p>
   * @param input  The String to test for a match.
   * @param pattern  The Pattern to be matched.
   * @return True if the input contains a pattern match, false otherwise.
   */
  public boolean contains(char[] input, Pattern pattern);
  /**
   * Determines if a string contains a pattern.  If the pattern is
   * matched by some substring of the input, a MatchResult instance
   * representing the <b> first </b> such match is made acessible via 
   * {@link #getMatch()}.  If you want to access
   * subsequent matches you should either use a PatternMatcherInput object
   * or use the offset information in the MatchResult to create a substring
   * representing the remaining input.  Using the MatchResult offset 
   * information is the recommended method of obtaining the parts of the
   * string preceeding the match and following the match.
   * <p>
   * @param input  The String to test for a match.
   * @param pattern  The Pattern to be matched.
   * @return True if the input contains a pattern match, false otherwise.
   */
  public boolean contains(String input, Pattern pattern);
  /**
   * Determines if the contents of a PatternMatcherInput, starting from the
   * current offset of the input contains a pattern.
   * If a pattern match is found, a MatchResult
   * instance representing the <b>first</b> such match is made acessible via 
   * {@link #getMatch()}.  The current offset of the
   * PatternMatcherInput is set to the offset corresponding to the end
   * of the match, so that a subsequent call to this method will continue
   * searching where the last call left off.  You should remember that the
   * region between the begin and end offsets of the PatternMatcherInput are
   * considered the input to be searched, and that the current offset
   * of the PatternMatcherInput reflects where a search will start from.
   * Matches extending beyond the end offset of the PatternMatcherInput
   * will not be matched.  In other words, a match must occur entirely
   * between the begin and end offsets of the input.  See
   * {@link PatternMatcherInput} for more details.
   * <p>
   * This method is usually used in a loop as follows:
   * <blockquote><pre>
   * PatternMatcher matcher;
   * PatternCompiler compiler;
   * Pattern pattern;
   * PatternMatcherInput input;
   * MatchResult result;
   *
   * compiler = new Perl5Compiler();
   * matcher  = new Perl5Matcher();
   *
   * try {
   *   pattern = compiler.compile(somePatternString);
   * } catch(MalformedPatternException e) {
   *   System.out.println("Bad pattern.");
   *   System.out.println(e.getMessage());
   *   return;
   * }
   *
   * input   = new PatternMatcherInput(someStringInput);
   *
   * while(matcher.contains(input, pattern)) {
   *   result = matcher.getMatch();  
   *   // Perform whatever processing on the result you want.
   * }
   *
   * </pre></blockquote>
   * <p>
   * @param input  The PatternMatcherInput to test for a match.
   * @param pattern  The Pattern to be matched.
   * @return True if the input contains a pattern match, false otherwise.
   */
  public boolean contains(PatternMatcherInput input, Pattern pattern);
  /**
   * Fetches the last match found by a call to a matches() or contains()
   * method.
   * <p>
   * @return A MatchResult instance containing the pattern match found
   *         by the last call to any one of the matches() or contains()
   *         methods.  If no match was found by the last call,
   *         returns null.
   */
  public MatchResult getMatch();
  /**
   * Determines if a string (represented as a char[]) exactly matches
   * a given pattern.  If there is an exact match, a MatchResult
   * instance representing the match is made accesible via
   * {@link #getMatch()}.
   * <p>
   * @param input  The char[] to test for a match.
   * @param pattern  The Pattern to be matched.
   * @return True if input matches pattern, false otherwise.
   */
  public boolean matches(char[] input, Pattern pattern);
  /**
   * Determines if a string exactly matches a given pattern.  If
   * there is an exact match, a MatchResult instance
   * representing the match is made accesible via
   * {@link #getMatch()}.
   * <p>
   * @param input  The String to test for an exact match.
   * @param pattern  The Pattern to be matched.
   * @return True if input matches pattern, false otherwise.
   */
  public boolean matches(String input, Pattern pattern);
  /**
   * Determines if the contents of a PatternMatcherInput instance
   * exactly matches a given pattern.  If
   * there is an exact match, a MatchResult instance
   * representing the match is made accesible via
   * {@link #getMatch()}.  Unlike the
   * {@link #contains(PatternMatcherInput, Pattern)}
   * method, the current offset of the PatternMatcherInput argument
   * is not updated.  You should remember that the region between
   * the begin and end offsets of the PatternMatcherInput will be
   * tested for an exact match.
   * <p>
   * @param input  The PatternMatcherInput to test for a match.
   * @param pattern  The Pattern to be matched.
   * @return True if input matches pattern, false otherwise.
   */
  public boolean matches(PatternMatcherInput input, Pattern pattern);
  /**
   * Determines if a prefix of a string (represented as a char[])
   * matches a given pattern.
   * If a prefix of the string matches the pattern, a MatchResult instance
   * representing the match is made accesible via
   * {@link #getMatch()}.
   * <p>
   * This method is useful for certain common token identification tasks
   * that are made more difficult without this functionality.
   * <p>
   * @param input  The char[] to test for a prefix match.
   * @param pattern  The Pattern to be matched.
   * @return True if input matches pattern, false otherwise.
   */
  public boolean matchesPrefix(char[] input, Pattern pattern);
  /**
   * Determines if a prefix of a string (represented as a char[])
   * matches a given pattern, starting from a given offset into the string.
   * If a prefix of the string matches the pattern, a MatchResult instance
   * representing the match is made accesible via
   * {@link #getMatch()}.
   * <p>
   * This method is useful for certain common token identification tasks
   * that are made more difficult without this functionality.
   * <p>
   * @param input  The char[] to test for a prefix match.
   * @param pattern  The Pattern to be matched.
   * @param offset The offset at which to start searching for the prefix.
   * @return True if input matches pattern, false otherwise.
   */
  public boolean matchesPrefix(char[] input, Pattern pattern, int offset);
  /**
   * Determines if a prefix of a string matches a given pattern.
   * If a prefix of the string matches the pattern, a MatchResult instance
   * representing the match is made accesible via
   * {@link #getMatch()}.
   * <p>
   * This method is useful for certain common token identification tasks
   * that are made more difficult without this functionality.
   * <p>
   * @param input  The String to test for a prefix match.
   * @param pattern  The Pattern to be matched.
   * @return True if input matches pattern, false otherwise.
   */
  public boolean matchesPrefix(String input, Pattern pattern);
  /**
   * Determines if a prefix of a PatternMatcherInput instance
   * matches a given pattern.  If there is a match, a MatchResult instance
   * representing the match is made accesible via
   * {@link #getMatch()}.  Unlike the
   * {@link #contains(PatternMatcherInput, Pattern)}
   * method, the current offset of the PatternMatcherInput argument
   * is not updated.  You should remember that the region starting
   * from the begin offset of the PatternMatcherInput will be
   * tested for a prefix match.
   * <p>
   * This method is useful for certain common token identification tasks
   * that are made more difficult without this functionality.
   * <p>
   * @param input  The PatternMatcherInput to test for a prefix match.
   * @param pattern  The Pattern to be matched.
   * @return True if input matches pattern, false otherwise.
   */
  public boolean matchesPrefix(PatternMatcherInput input, Pattern pattern);
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The PatternMatcherInput class is used to preserve state across
 * calls to the <code>contains()</code> methods of PatternMatcher instances.
 * It is also used to specify that only a subregion of a string
 * should be used as input when looking for a pattern match.  All that
 * is meant by preserving state is that the end offset of the last match
 * is remembered, so that the next match is performed from that point
 * where the last match left off.  This offset can be accessed from
 * the {@link #getCurrentOffset()} method and can be set with the
 * {@link #setCurrentOffset(int)} method.
 * <p>
 * You would use a PatternMatcherInput object when you want to search for
 * more than just the first occurrence of a pattern in a string, or when
 * you only want to search a subregion of the string for a match.  An
 * example of its most common use is:
 * <blockquote><pre>
 * PatternMatcher matcher;
 * PatternCompiler compiler;
 * Pattern pattern;
 * PatternMatcherInput input;
 * MatchResult result;
 *
 * compiler = new Perl5Compiler();
 * matcher  = new Perl5Matcher();
 *
 * try {
 *   pattern = compiler.compile(somePatternString);
 * } catch(MalformedPatternException e) {
 *   System.out.println("Bad pattern.");
 *   System.out.println(e.getMessage());
 *   return;
 * }
 *
 * input   = new PatternMatcherInput(someStringInput);
 *
 * while(matcher.contains(input, pattern)) {
 *   result = matcher.getMatch();  
 *   // Perform whatever processing on the result you want.
 * }
 * // Suppose we want to start searching from the beginning again with
 * // a different pattern.
 * // Just set the current offset to the begin offset.
 * input.setCurrentOffset(input.getBeginOffset());
 *
 * // Second search omitted
 *
 * // Suppose we're done with this input, but want to search another string.
 * // There's no need to create another PatternMatcherInput instance.
 * // We can just use the setInput() method.
 * input.setInput(aNewInputString);
 *
 * </pre></blockquote>
 *
 *

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: PatternMatcherInput.java,v 1.2 2000/07/23 23:25:25 jon Exp $

 * @see PatternMatcher
 */
public final class PatternMatcherInput {
  String _originalStringInput;
  char[] _originalCharInput, _originalBuffer, _toLowerBuffer;
  int _beginOffset, _endOffset, _currentOffset;
  int _matchBeginOffset = -1, _matchEndOffset = -1;

  /**
   * Like calling:
   * <blockquote><pre>
   * PatternMatcherInput(input, 0, input.length);
   * </pre></blockquote>
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   */
  public PatternMatcherInput(char[] input) {
    this(input, 0, input.length);
  }
  /**
   * Creates a PatternMatcherInput object, associating a region of a string
   * (represented as a char[]) as input
   * to be used for pattern matching by PatternMatcher objects.
   * A copy of the string is not made, therefore you should not modify
   * the string unless you know what you are doing.
   * The current offset of the PatternMatcherInput is set to the begin
   * offset of the region.
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   * @param begin  The offset into the char[] to use as the beginning of
   *               the input.
   * @param length The length of the reegion starting from the begin offset
   *               to use as the input for pattern matching purposes.
   */
  public PatternMatcherInput(char[] input, int begin, int length) {
    setInput(input, begin, length);
  }
  /**
   * Like calling
   * <blockquote><pre>
   * PatternMatcherInput(input, 0, input.length());
   * </pre></blockquote>
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   */
  public PatternMatcherInput(String input) {
    this(input, 0, input.length());
  }
  /**
   * Creates a PatternMatcherInput object, associating a region of a String
   * as input to be used for pattern matching by PatternMatcher objects.
   * A copy of the string is not made, therefore you should not modify
   * the string unless you know what you are doing.
   * The current offset of the PatternMatcherInput is set to the begin
   * offset of the region.
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   * @param begin  The offset into the char[] to use as the beginning of
   *               the input.
   * @param length The length of the reegion starting from the begin offset
   *               to use as the input for pattern matching purposes.
   */
  public PatternMatcherInput(String input, int begin, int length) {
    setInput(input, begin, length);
  }
  /**
   * Returns the character at a particular offset relative to the begin
   * offset of the input.
   * <p>
   * @param offset  The offset at which to fetch a character (relative to
   *                the beginning offset.
   * @return The character at a particular offset.
   * @exception ArrayIndexOutOfBoundsException If the offset does not occur
   *            within the bounds of the input.
   */
  public char charAt(int offset) {
    return _originalBuffer[_beginOffset + offset];
  }
  /**
   * Returns whether or not the end of the input has been reached.
   * <p>
   * @return True if the current offset is greater than or equal to the
   *         end offset.
   */
  public boolean endOfInput(){ return (_currentOffset >= _endOffset); }
  /**
   * @return The offset of the input that should be considered the start
   *         of the region to be considered as input by PatternMatcher
   *         methods.
   */
  public int getBeginOffset()   { return _beginOffset; }
  /**
   * Retrieves the char[] buffer to be used used as input by PatternMatcher
   * implementations to look for matches.  This array should be treated
   * as read only by the programmer.
   * <p>
   * @return The char[] buffer to be used as input by PatternMatcher
   *         implementations.
   */
  public char[] getBuffer() { return _originalBuffer;  }
  /**
   * @return The offset of the input that should be considered the current
   *         offset where PatternMatcher methods should start looking for
   *         matches.
   */
  public int getCurrentOffset() { return _currentOffset; }
  /**
   * @return The offset of the input that should be considered the end
   *         of the region to be considered as input by PatternMatcher
   *         methods.  This offset is actually 1 plus the last offset
   *         that is part of the input region.
   */
  public int getEndOffset()     { return _endOffset;  }
  /**
   * Retrieves the original input used to initialize the PatternMatcherInput
   * instance.  If a String was used, the String instance will be returned.
   * If a char[] was used, a char instance will be returned.  This violates
   * data encapsulation and hiding principles, but it is a great convenience
   * for the programmer.
   * <p>
   * @return The String or char[] input used to initialize the
   *         PatternMatcherInput instance.
   */
  public Object getInput(){
    if(_originalStringInput == null)
      return _originalCharInput;
    return _originalStringInput;
  }
  /**
   * Returns the offset marking the beginning of the match found by
   * contains().
   * <p>
   * @return The begin offset of a contains() match.
   */
  public int getMatchBeginOffset()    { return _matchBeginOffset; }
  /**
   * Returns the offset marking the end of the match found by contains().
   * <p>
   * @return The end offset of a contains() match.
   */
  public int getMatchEndOffset()      { return _matchEndOffset; }
  /**
   * @return The length of the region to be considered input for pattern
   *         matching purposes.  Essentially this is then end offset minus
   *         the begin offset.
   */
  public int length()        {
    return (_endOffset - _beginOffset);
    //return _originalBuffer.length;
  }
  /**
   * A convenience method returning the part of the input corresponding
   * to the last match found by a call to a Perl5Matcher
   * {@link Perl5Matcher#contains contains} method.
   * The method is not called getMatch() so as not to confuse it
   * with Perl5Matcher's getMatch() which returns a MatchResult instance
   * and also for consistency with preMatch() and postMatch().
   * <p>
   * @return The input consisting of the match found by contains().
   */
  public String match() {
    return new String(_originalBuffer, _matchBeginOffset,
		      _matchEndOffset - _matchBeginOffset);
  }
  /**
   * A convenience method returning the part of the input occurring after
   * the last match found by a call to a Perl5Matcher
   * {@link Perl5Matcher#contains contains} method.
   * <p>
   * @return The input succeeding a contains() match.
   */
  public String postMatch() {
    return new String(_originalBuffer, _matchEndOffset,
		      _endOffset - _matchEndOffset);
  }
  /**
   * A convenience method returning the part of the input occurring before
   * the last match found by a call to a Perl5Matcher
   * {@link Perl5Matcher#contains contains} method.
   * <p>
   * @return The input preceeding a match.
   */
  public String preMatch() {
    return new String(_originalBuffer, _beginOffset,
		      _matchBeginOffset - _beginOffset);
  }
  /**
   * Sets the offset of the input that should be considered the start
   * of the region to be considered as input by PatternMatcher
   * methods.  In other words, everything before this offset is ignored
   * by a PatternMatcher.
   * <p>
   * @param offset  The offset to use as the beginning of the input.
   */
  public void setBeginOffset(int offset)   { _beginOffset = offset; }
  /**
   * Sets the offset of the input that should be considered the current
   * offset where PatternMatcher methods should start looking for
   * matches.  Also resets all match offset information to -1.  By calling
   * this method, you invalidate all previous match information.  Therefore
   * a PatternMatcher implementation must call this method before setting
   * match offset information.
   * <p>
   * @param offset  The offset to use as the current offset.
   */
  public void setCurrentOffset(int offset) {
    _currentOffset    = offset;
    setMatchOffsets(-1, -1);
  }
  /**
   * Sets the offset of the input that should be considered the end
   * of the region to be considered as input by PatternMatcher
   * methods.  This offset is actually 1 plus the last offset
   * that is part of the input region.
   * <p>
   * @param offset  The offset to use as the end of the input.
   */
  public void setEndOffset(int offset)     { _endOffset = offset; }
  /**
   * This method is identical to calling:
   * <blockquote><pre>
   * setInput(input, 0, input.length);
   * </pre></blockquote>
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   */
  public void setInput(char[] input) {
    setInput(input, 0, input.length);
  }
  /**
   * Associates a region of a string (represented as a char[]) as input
   * to be used for pattern matching by PatternMatcher objects.
   * A copy of the string is not made, therefore you should not modify
   * the string unless you know what you are doing.
   * The current offset of the PatternMatcherInput is set to the begin
   * offset of the region.
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   * @param begin  The offset into the char[] to use as the beginning of
   *               the input.
   * @param length The length of the reegion starting from the begin offset
   *               to use as the input for pattern matching purposes.
   */
  public void setInput(char[] input, int begin, int length) {
    _originalStringInput = null;
    _toLowerBuffer  = null;
    _originalBuffer = _originalCharInput = input;
    setCurrentOffset(begin);
    setBeginOffset(begin);
    setEndOffset(_beginOffset + length);
  }
  /**
   * This method is identical to calling:
   * <blockquote><pre>
   * setInput(input, 0, input.length());
   * </pre></blockquote>
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   */
  public void setInput(String input) {
    setInput(input, 0, input.length());
  }
  /**
   * Associates a region of a String as input
   * to be used for pattern matching by PatternMatcher objects.
   * The current offset of the PatternMatcherInput is set to the begin
   * offset of the region.
   * <p>
   * @param input  The input to associate with the PatternMatcherInput.
   * @param begin  The offset into the String to use as the beginning of
   *               the input.
   * @param length The length of the reegion starting from the begin offset
   *               to use as the input for pattern matching purposes.
   */
  public void setInput(String input, int begin, int length) {
    _originalStringInput = input;
    _originalCharInput = null;
    _toLowerBuffer = null;
    _originalBuffer = input.toCharArray();
    setCurrentOffset(begin);
    setBeginOffset(begin);
    setEndOffset(_beginOffset + length);
  }
  /**
   * This method is intended for use by PatternMatcher implementations.
   * It is necessary to record the location of the previous match so that
   * consecutive contains() matches involving null string matches are
   * properly handled.  If you are not implementing a PatternMatcher, forget
   * this method exists.  If you use it outside of its intended context, you
   * will only disrupt the stored state.
   * <p>
   * As a note, the preMatch(), postMatch(), and match() methods are provided
   * as conveniences because PatternMatcherInput must store match offset
   * information to completely preserve state for consecutive PatternMatcher
   * contains() matches.
   * <p>
   * @param matchBeginOffset  The begin offset of a match found by contains().
   * @param matchEndOffset    The end offset of a match found by contains().
   */
  public void setMatchOffsets(int matchBeginOffset, int matchEndOffset) {
    _matchBeginOffset    = matchBeginOffset;
    _matchEndOffset      = matchEndOffset;
  }
  /**
   * Returns a new string that is a substring of the PatternMatcherInput
   * instance. The substring begins at the specified beginOffset relative
   * to the begin offset and extends to the end offset of the
   * PatternMatcherInput.
   * <p>
   * @param beginOffset  The offset relative to the begin offset of the
   *        PatternMatcherInput at which to start the substring.
   * @return The specified substring.
   * @exception ArrayIndexOutOfBoundsException If the offset does not occur
   *            within the bounds of the input.
   */
  public String substring(int beginOffset) {
    beginOffset+=_beginOffset;
    return new String(_originalBuffer, beginOffset, _endOffset - beginOffset);
  }
  /**
   * Returns a new string that is a substring of the PatternMatcherInput
   * instance. The substring begins at the specified beginOffset relative
   * to the begin offset and extends to the specified endOffset - 1
   * relative to the begin offset of the PatternMatcherInput instance.
   * <p>
   * @param beginOffset  The offset relative to the begin offset of the
   *        PatternMatcherInput at which to start the substring (inclusive).
   * @param endOffset  The offset relative to the begin offset of the
   *        PatternMatcherInput at which to end the substring (exclusive).
   * @return The specified substring.
   * @exception ArrayIndexOutOfBoundsException If one of the offsets does
   *        not occur within the bounds of the input.
   */
  public String substring(int beginOffset, int endOffset) {
    return new String(_originalBuffer, _beginOffset+beginOffset,
		      endOffset - beginOffset);
  }
  /**
   * Returns the string representation of the input, where the input is
   * considered to start from the begin offset and end at the end offset.
   * <p>
   * @return The string representation of the input.
   */
  public String toString() {
    return new String(_originalBuffer, _beginOffset, length());
  }
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

/**
 * The Perl5Compiler class is used to create compiled regular expressions
 * conforming to the Perl5 regular expression syntax.  It generates
 * Perl5Pattern instances upon compilation to be used in conjunction
 * with a Perl5Matcher instance.  Please see the user's guide for more 
 * information about Perl5 regular expressions.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: Perl5Compiler.java,v 1.2 2000/07/23 23:25:26 jon Exp $

 * @see PatternCompiler
 * @see MalformedPatternException
 * @see Perl5Pattern
 * @see Perl5Matcher
 */

public final class Perl5Compiler implements PatternCompiler {
  private static final int __WORSTCASE = 0, __NONNULL = 0x1, __SIMPLE = 0x2,
						   __SPSTART = 0x4, __TRYAGAIN = 0x8;

  private static final char
	__CASE_INSENSITIVE = 0x0001,
	__GLOBAL           = 0x0002,
	__KEEP             = 0x0004,
	__MULTILINE        = 0x0008,
	__SINGLELINE       = 0x0010,
	__EXTENDED         = 0x0020,
	__READ_ONLY        = 0x8000;

  private static final String __META_CHARS = "^$.[()|?+*\\";
  private static final String __HEX_DIGIT =
  "0123456789abcdef0123456789ABCDEFx";
  private CharStringPointer __input;
  private boolean __sawBackreference;
  private char[] __modifierFlags = { 0 };

  // IMPORTANT: __numParentheses starts out equal to 1 during compilation.
  // It is always one greater than the number of parentheses encountered
  // so far in the regex.  That is because it refers to the number of groups
  // to save, and the entire match is always saved (group 0)
  private int __numParentheses, __programSize, __cost;

  // When doing the second pass and actually generating code, __programSize
  // keeps track of the current offset.
  private char[] __program;

  /**
   * The default mask for the {@link #compile compile} methods.
   * It is equal to 0.
   * The default behavior is for a regular expression to be case sensitive
   * and to not specify if it is multiline or singleline.  When MULITLINE_MASK
   * and SINGLINE_MASK are not defined, the <b>^</b>, <b>$</b>, and <b>.</b>
   * metacharacters are
   * interpreted according to the value of isMultiline() in Perl5Matcher.
   * The default behavior of Perl5Matcher is to treat the Perl5Pattern
   * as though MULTILINE_MASK were enabled.  If isMultiline() returns false,
   * then the pattern is treated as though SINGLINE_MASK were set.  However,
   * compiling a pattern with the MULTILINE_MASK or SINGLELINE_MASK masks
   * will ALWAYS override whatever behavior is specified by the setMultiline()
   * in Perl5Matcher.
   */
  public static final int DEFAULT_MASK          = 0;

  /**
   * A mask passed as an option to the {@link #compile compile} methods
   * to indicate a compiled regular expression should be case insensitive.
   */
  public static final int CASE_INSENSITIVE_MASK = __CASE_INSENSITIVE;

  /**
   * A mask passed as an option to the  {@link #compile compile} methods
   * to indicate a compiled regular expression should treat input as having
   * multiple lines.  This option affects the interpretation of
   * the <b>^</b> and <b>$</b> metacharacters.  When this mask is used,
   * the <b>^</b> metacharacter matches at the beginning of every line,
   * and the <b>$</b> metacharacter matches at the end of every line.
   * Additionally the <b> . </b> metacharacter will not match newlines when
   * an expression is compiled with <b> MULTILINE_MASK </b>, which is its
   * default behavior.
   * The <b>SINGLELINE_MASK</b> and <b>MULTILINE_MASK</b> should not be
   * used together.
   */
  public static final int MULTILINE_MASK        = __MULTILINE;

  /**
   * A mask passed as an option to the {@link #compile compile} methods
   * to indicate a compiled regular expression should treat input as being
   * a single line.  This option affects the interpretation of
   * the <b>^</b> and <b>$</b> metacharacters.  When this mask is used,
   * the <b>^</b> metacharacter matches at the beginning of the input,
   * and the <b>$</b> metacharacter matches at the end of the input.
   * The <b>^</b> and <b>$</b> metacharacters will not match at the beginning
   * and end of lines occurring between the begnning and end of the input.
   * Additionally, the <b> . </b> metacharacter will match newlines when
   * an expression is compiled with <b> SINGLELINE_MASK </b>, unlike its
   * default behavior.
   * The <b>SINGLELINE_MASK</b> and <b>MULTILINE_MASK</b> should not be
   * used together.
   */
  public static final int SINGLELINE_MASK       = __SINGLELINE;

  /**
   * A mask passed as an option to the {@link #compile compile} methods
   * to indicate a compiled regular expression should be treated as a Perl5
   * extended pattern (i.e., a pattern using the <b>/x</b> modifier).  This 
   * option tells the compiler to ignore whitespace that is not backslashed or
   * within a character class.  It also tells the compiler to treat the
   * <b>#</b> character as a metacharacter introducing a comment as in
   * Perl.  In other words, the <b>#</b> character will comment out any
   * text in the regular expression between it and the next newline.
   * The intent of this option is to allow you to divide your patterns
   * into more readable parts.  It is provided to maintain compatibility
   * with Perl5 regular expressions, although it will not often
   * make sense to use it in Java.
   */
  public static final int EXTENDED_MASK         = __EXTENDED;

  /**
   * A mask passed as an option to the {@link #compile compile} methods
   * to indicate that the resulting Perl5Pattern should be treated as a
   * read only data structure by Perl5Matcher, making it safe to share
   * a single Perl5Pattern instance among multiple threads without needing
   * synchronization.  Without this option, Perl5Matcher reserves the right
   * to store heuristic or other information in Perl5Pattern that might
   * accelerate future matches.  When you use this option, Perl5Matcher will
   * not store or modify any information in a Perl5Pattern.  Use this option
   * when you want to share a Perl5Pattern instance among multiple threads
   * using different Perl5Matcher instances.
   */
  public static final int READ_ONLY_MASK        = __READ_ONLY;

  // Emit an operator with arguments.
  // Return an offset into the __programarray as a pointer to node.
  private int __emitArgNode(char operator, char arg) {
    int offset;

    offset = __programSize;

    if(__program== null)
      __programSize+=3;
    else {
      __program[__programSize++] = operator;
      __program[__programSize++] = OpCode._NULL_POINTER;
      __program[__programSize++] = arg;
    }

    return offset;
  }
  // Emit a specific character code.
  private void __emitCode(char code) {

    if(__program != null)
      __program[__programSize] = code;

    ++__programSize;
  }
  // Emit an operator with no arguments.
  // Return an offset into the __program array as a pointer to node.
  private int __emitNode(char operator) {
    int offset;

    offset = __programSize;

    if(__program == null)
      __programSize+=2;
    else {
      __program[__programSize++] = operator;
      __program[__programSize++] = OpCode._NULL_POINTER;
    }

    return offset;
  }
  private char __getNextChar() {
    char ret, value;

    ret = __input._postIncrement();

    while(true) {
      value = __input._getValue();

      if(value == '(' && __input._getValueRelative(1) == '?' &&
	 __input._getValueRelative(2) == '#') {
	// Skip comments
	while(value != CharStringPointer._END_OF_STRING && value != ')')
	  value = __input._increment();
	__input._increment();
	continue;
      }

      if((__modifierFlags[0] & __EXTENDED) != 0) {
	if(Character.isWhitespace(value)) {
	  __input._increment();
	  continue;
	} else if(value == '#') {
	  while(value != CharStringPointer._END_OF_STRING && value != '\n')
	    value = __input._increment();
	  __input._increment();
	  continue;
	}
      }

      // System.err.println("next: " + ret + " last: " + __input._getValue()); // debug


      return ret;
    }

  }
  private static boolean __isComplexRepetitionOp(char[] ch, int offset) {
    if(offset < ch.length && offset >= 0)
       return (ch[offset] == '*' || ch[offset] == '+' || ch[offset] == '?'
	       || (ch[offset] == '{' && __parseRepetition(ch, offset)));
    return false;
  }
  private static boolean __isSimpleRepetitionOp(char ch) {
    return (ch == '*' || ch == '+' || ch == '?');
  }
  private int __parseAlternation(int[] retFlags)
    throws MalformedPatternException 
  {
    int chain, offset, latest;
    int flags = 0;
    char value;

    retFlags[0] = __WORSTCASE;

    offset = __emitNode(OpCode._BRANCH);

    chain  = OpCode._NULL_OFFSET;

    if(__input._getOffset() == 0) {
      __input._setOffset(-1);
      __getNextChar();
    } else {
      __input._decrement();
      __getNextChar();
    }

    value = __input._getValue();

    while(value != CharStringPointer._END_OF_STRING &&
	  value != '|' && value != ')') {
      flags &= ~__TRYAGAIN;
      latest = __parseBranch(retFlags);

      if(latest == OpCode._NULL_OFFSET) {
	if((flags & __TRYAGAIN) != 0){
	  value = __input._getValue();
	  continue;
	}
	return OpCode._NULL_OFFSET;
      }

      retFlags[0] |= (flags & __NONNULL);

      if(chain == OpCode._NULL_OFFSET)
	retFlags[0] |= (flags & __SPSTART);
      else {
	++__cost;
	__programAddTail(chain, latest);
      }
      chain = latest;
      value = __input._getValue();
    }

    // If loop was never entered.
    if(chain == OpCode._NULL_OFFSET)
      __emitNode(OpCode._NOTHING);

    return offset;
  }
  private int __parseAtom(int[] retFlags) throws MalformedPatternException {
    boolean doDefault;
    char value;
    int offset, flags[] = { 0 };
    
    
    retFlags[0] = __WORSTCASE;
    doDefault = false;
    offset = OpCode._NULL_OFFSET;

  tryAgain:
    while(true) {

      value = __input._getValue();

      switch(value) {
      case '^' :
	__getNextChar();
	// The order here is important in order to support /ms.
	// /m takes precedence over /s for ^ and $, but not for .
	if((__modifierFlags[0] & __MULTILINE) != 0)
	  offset = __emitNode(OpCode._MBOL);
	else if((__modifierFlags[0] & __SINGLELINE) != 0)
	  offset = __emitNode(OpCode._SBOL);
	else
	  offset = __emitNode(OpCode._BOL);
	break tryAgain;

      case '$':
	__getNextChar();
	// The order here is important in order to support /ms.
	// /m takes precedence over /s for ^ and $, but not for .
	if((__modifierFlags[0] & __MULTILINE) != 0)
	  offset = __emitNode(OpCode._MEOL);
	else if((__modifierFlags[0] & __SINGLELINE) != 0)
	  offset = __emitNode(OpCode._SEOL);
	else
	  offset = __emitNode(OpCode._EOL);
	break tryAgain;

      case '.':
	__getNextChar();
	// The order here is important in order to support /ms.
	// /m takes precedence over /s for ^ and $, but not for .
	if((__modifierFlags[0] & __SINGLELINE) != 0)
	  offset = __emitNode(OpCode._SANY);
	else
	  offset = __emitNode(OpCode._ANY);
	++__cost;
	retFlags[0] |= (__NONNULL | __SIMPLE);
	break tryAgain;

      case '[':
	__input._increment();
	offset = __parseCharacterClass();
	retFlags[0] |= (__NONNULL | __SIMPLE);
	break tryAgain;

      case '(':
	__getNextChar();
	offset = __parseExpression(true, flags);
	if(offset == OpCode._NULL_OFFSET) {
	  if((flags[0] & __TRYAGAIN) != 0)
	    continue tryAgain;
	  return OpCode._NULL_OFFSET;
	}
	retFlags[0] |= (flags[0] & (__NONNULL | __SPSTART));
	break tryAgain;

      case '|':
      case ')':
	if((flags[0] & __TRYAGAIN) != 0) {
	  retFlags[0] |= __TRYAGAIN;
	  return OpCode._NULL_OFFSET;
	}

	throw new MalformedPatternException("Error in expression at " +
				   __input._toString(__input._getOffset()));
	//break tryAgain;

      case '?':
      case '+':
      case '*':
	throw new MalformedPatternException(
                 "?+* follows nothing in expression");
	//break tryAgain;

      case '\\':
	value = __input._increment();

	switch(value) {
	case 'A' :
	  offset = __emitNode(OpCode._SBOL);
	  retFlags[0] |= __SIMPLE;
	  __getNextChar();
	  break;
	case 'G':
	  offset = __emitNode(OpCode._GBOL);
	  retFlags[0] |= __SIMPLE;
	  __getNextChar();
	  break;
	case 'Z':
	  offset = __emitNode(OpCode._SEOL);
	  retFlags[0] |= __SIMPLE;
	  __getNextChar();
	  break;
	case 'w':
	  offset = __emitNode(OpCode._ALNUM);
	  retFlags[0] |= (__NONNULL | __SIMPLE);
	  __getNextChar();
	  break;
	case 'W':
	  offset = __emitNode(OpCode._NALNUM);
	  retFlags[0] |= (__NONNULL | __SIMPLE);
	  __getNextChar();
	  break;
	case 'b':
	  offset = __emitNode(OpCode._BOUND);
	  retFlags[0] |= __SIMPLE;
	  __getNextChar();
	  break;
	case 'B':
	  offset = __emitNode(OpCode._NBOUND);
	  retFlags[0] |= __SIMPLE;
	  __getNextChar();
	  break;
	case 's':
	  offset = __emitNode(OpCode._SPACE);
	  retFlags[0] |= (__NONNULL | __SIMPLE);
	  __getNextChar();
	  break;
	case 'S':
	  offset = __emitNode(OpCode._NSPACE);
	  retFlags[0] |= (__NONNULL | __SIMPLE);
	  __getNextChar();
	  break;
	case 'd':
	  offset = __emitNode(OpCode._DIGIT);
	  retFlags[0] |= (__NONNULL | __SIMPLE);
	  __getNextChar();
	  break;
	case 'D':
	  offset = __emitNode(OpCode._NDIGIT);
	  retFlags[0] |= (__NONNULL | __SIMPLE);
	  __getNextChar();
	  break;
	case 'n': case 'r': case 't': case 'f': case 'e': case 'a': case 'x':
	case 'c': case '0':
	  doDefault = true;
	  break tryAgain;
	case '1': case '2': case '3': case '4': case '5': case '6': case '7':
	case '8': case '9':
	  int num;
	  StringBuffer buffer = new StringBuffer(10);

	  num = 0;
	  value = __input._getValueRelative(num);

	  while(Character.isDigit(value)) {
	    buffer.append(value);
	    ++num;
	    value = __input._getValueRelative(num);
	  }

	  try {
	    num = Integer.parseInt(buffer.toString());
	  } catch(NumberFormatException e) {
	    throw new MalformedPatternException(
	   "Unexpected number format exception.  Please report this bug." +
	   "NumberFormatException message: " + e.getMessage());
	  }

	  if(num > 9 && num >= __numParentheses) {
	    doDefault = true;
	    break tryAgain;
	  } else {
	    // A backreference may only occur AFTER its group
	    if(num >= __numParentheses)
	      throw new MalformedPatternException("Invalid backreference: \\" +
						  num);
	    __sawBackreference = true;
	    offset = __emitArgNode(OpCode._REF, (char)num);
	    retFlags[0] |= __NONNULL;

	    value = __input._getValue();
	    while(Character.isDigit(value))
	      value = __input._increment();

	    __input._decrement();
	    __getNextChar();
	  }
	  break;
	case '\0':
	case CharStringPointer._END_OF_STRING:
	  if(__input._isAtEnd())
	    throw new
	      MalformedPatternException("Trailing \\ in expression.");
	  // fall through to default
	default:
	  doDefault = true;
	  break tryAgain;
	}
	break tryAgain;

      case '#':
	// skip over comments
	if((__modifierFlags[0] & __EXTENDED) != 0) {
	  while(!__input._isAtEnd() && __input._getValue() != '\n')
	    __input._increment();
	  if(!__input._isAtEnd())
	    continue tryAgain;
	}
	// fall through to default
      default:
	__input._increment();
	doDefault = true;
	break tryAgain;
      }// end master switch
    } // end tryAgain


    if(doDefault) {
      char ender;
      int length, pOffset, maxOffset, lastOffset, numLength[];

      offset = __emitNode(OpCode._EXACTLY);
      // Not sure that it's ok to use 0 to mark end.
      //__emitCode((char)0);
      __emitCode((char)CharStringPointer._END_OF_STRING);

    forLoop:
      for(length = 0, pOffset = __input._getOffset() - 1,
	    maxOffset = __input._getLength();
	  length < 127 && pOffset < maxOffset; ++length) {

	lastOffset = pOffset;
	value = __input._getValue(pOffset);

	switch(value) {
	case '^': case '$': case '.': case '[': case '(': case ')':
	case '|':
	  break forLoop;
	case '\\':
	  value = __input._getValue(++pOffset);

	  switch(value) {
	  case 'A': case 'G': case 'Z': case 'w': case 'W': case 'b':
	  case 'B': case 's': case 'S': case 'd': case 'D':
	    --pOffset;
	    break forLoop;
	  case 'n':
	    ender = '\n';
	    ++pOffset;
	    break;
	  case 'r':
	    ender = '\r';
	    ++pOffset;
	    break;
	  case 't':
	    ender = '\t';
	    ++pOffset;
	    break;
	  case 'f':
	    ender = '\f';
	    ++pOffset;
	    break;
	  case 'e':
	    ender = '\033';
	    ++pOffset;
	    break;
	  case 'a':
	    ender = '\007';
	    ++pOffset;
	    break;
	  case 'x':
	    numLength = new int[1];
	    ender = (char)__parseHex(__input._array, ++pOffset, 2, numLength);
	    pOffset+=numLength[0];
	    break;
	  case 'c':
	    ++pOffset;
	    ender = __input._getValue(pOffset++);
	    if(Character.isLowerCase(ender))
	      ender = Character.toUpperCase(ender);
	    ender ^= 64;
	    break;
	  case '0': case '1': case '2': case'3': case '4': case '5':
	  case '6': case '7': case '8': case '9':
	    boolean doOctal = false;
	    value = __input._getValue(pOffset);

	    if(value == '0')
	      doOctal = true;
	    value = __input._getValue(pOffset + 1);

	    if(Character.isDigit(value)) {
	      int num;
	      StringBuffer buffer = new StringBuffer(10);

	      num = pOffset;
	      value = __input._getValue(num);

	      while(Character.isDigit(value)){
		buffer.append(value);
		++num;
		value = __input._getValue(num);
	      }

	      try {
		num = Integer.parseInt(buffer.toString());
	      } catch(NumberFormatException e) {
		throw new MalformedPatternException(
	     "Unexpected number format exception.  Please report this bug." +
	     "NumberFormatException message: " + e.getMessage());
	      }

	      if(!doOctal)
		doOctal = (num >= __numParentheses);
	    }

	    if(doOctal) {
	      numLength = new int[1];
	      ender = (char)__parseOctal(__input._array, pOffset, 3, numLength);
	      pOffset+=numLength[0];
	    } else {
	      --pOffset;
	      break forLoop;
	    }
	    break;

	  case CharStringPointer._END_OF_STRING:
	  case '\0':
	    if(pOffset >= maxOffset)
	      throw new
		MalformedPatternException("Trailing \\ in expression.");
	    // fall through to default
	  default:
	    ender = __input._getValue(pOffset++);
	    break;
	  } // end backslash switch
	  break;

	case '#':
	  if((__modifierFlags[0] & __EXTENDED) != 0) {
	    while(pOffset < maxOffset && __input._getValue(pOffset) != '\n')
	      ++pOffset;
	  }
	  // fall through to whitespace handling
	case ' ': case '\t': case '\n': case '\r': case '\f': case '\013':
	  if((__modifierFlags[0] & __EXTENDED) != 0) {
	    ++pOffset;
	    --length;
	    continue;
	  }
	  // fall through to default
	default:
	  ender = __input._getValue(pOffset++);
	  break;

	}   // end master switch

	if((__modifierFlags[0] & __CASE_INSENSITIVE) != 0 &&
	   Character.isUpperCase(ender))
	  ender = Character.toLowerCase(ender);

	if(pOffset < maxOffset && __isComplexRepetitionOp(__input._array, pOffset)) {
	  if(length > 0)
	    pOffset = lastOffset;
	  else {
	    ++length;
	    __emitCode(ender);
	  }
	  break;
	}

	__emitCode(ender);


      } // end for loop


      __input._setOffset(pOffset - 1);
      __getNextChar();

      if(length < 0)
	throw new MalformedPatternException(
         "Unexpected compilation failure.  Please report this bug!");
      if(length > 0)
	retFlags[0] |= __NONNULL;
      if(length == 1)
	retFlags[0] |= __SIMPLE;
      if(__program!= null)
	__program[OpCode._getOperand(offset)] = (char)length;
      //__emitCode('\0'); // debug
      __emitCode(CharStringPointer._END_OF_STRING);
    }

    return offset;
  }
  private int __parseBranch(int[] retFlags) throws MalformedPatternException {
    boolean nestCheck = false, handleRepetition = false;
    int offset, next, min, max, flags[] = { 0 };
    char operator, value;

    min = 0;
    max = Character.MAX_VALUE;
    offset = __parseAtom(flags);

    if(offset == OpCode._NULL_OFFSET) {
      if((flags[0] & __TRYAGAIN) != 0)
	retFlags[0] |= __TRYAGAIN;
      return OpCode._NULL_OFFSET;
    }

    operator = __input._getValue();

    if(operator == '(' && __input._getValueRelative(1) == '?' &&
       __input._getValueRelative(2) == '#') {
      while(operator != CharStringPointer._END_OF_STRING && operator != ')')
	operator = __input._increment();

      if(operator != CharStringPointer._END_OF_STRING) {
	__getNextChar();
	operator = __input._getValue();
      }
    }

    if(operator == '{' &&
       __parseRepetition(__input._array, __input._getOffset())) {
      int maxOffset, pos;

      next = __input._getOffset() + 1;
      pos = maxOffset = __input._getLength();

      value = __input._getValue(next);

      while(Character.isDigit(value) || value == ',') {
	if(value == ',') {
	  if(pos != maxOffset)
	    break;
	  else
	    pos = next;
	}
	++next;
	value = __input._getValue(next);
      }

      if(value == '}') {
	int num;
	StringBuffer buffer = new StringBuffer(10);

	if(pos == maxOffset)
	  pos = next;
	__input._increment();

	num = __input._getOffset();
	value = __input._getValue(num);

	while(Character.isDigit(value)) {
	  buffer.append(value);
	  ++num;
	  value = __input._getValue(num);
	}

	try {
	  min = Integer.parseInt(buffer.toString());
	} catch(NumberFormatException e) {
	  throw new MalformedPatternException(
	 "Unexpected number format exception.  Please report this bug." +
	   "NumberFormatException message: " + e.getMessage());
	}

	value = __input._getValue(pos);
	if(value == ',')
	  ++pos;
	else
	  pos = __input._getOffset();

	num = pos;
	buffer = new StringBuffer(10);

	value = __input._getValue(num);

	while(Character.isDigit(value)){
	  buffer.append(value);
	  ++num;
	  value = __input._getValue(num);
	}

	try {
	  if(num != pos)
	    max = Integer.parseInt(buffer.toString());
	} catch(NumberFormatException e) {
	  throw new MalformedPatternException(
	 "Unexpected number format exception.  Please report this bug." +
	   "NumberFormatException message: " + e.getMessage());
	}

	//System.err.println("min: " + min + " max: " + max); //debug

	if(max == 0 && __input._getValue(pos) != '0')
	  max = Character.MAX_VALUE;
	__input._setOffset(next);
	__getNextChar();

	//System.err.println("min: " + min + " max: " + max); //debug

	nestCheck = true;
	handleRepetition = true;
      }
    }

    if(!nestCheck) {
      handleRepetition = false;

      if(!__isSimpleRepetitionOp(operator)) {
	retFlags[0] = flags[0];
	return offset;
      }

      __getNextChar();

      retFlags[0] = ((operator != '+') ?
		  (__WORSTCASE | __SPSTART) : (__WORSTCASE | __NONNULL));

      if(operator == '*' && ((flags[0] & __SIMPLE) != 0)) {
	__programInsertOperator(OpCode._STAR, offset);
	__cost+=4;
      } else if(operator == '*') {
	min = 0;
	handleRepetition = true;
      } else if(operator == '+' && (flags[0] & __SIMPLE) != 0) {
	__programInsertOperator(OpCode._PLUS, offset);
	__cost+=3;
      } else if(operator == '+') {
	min = 1;
	handleRepetition = true;
      } else if(operator == '?') {
	min = 0;
	max = 1;
	handleRepetition = true;
      }
    }

    if(handleRepetition) {

      // handle repetition
      if((flags[0] & __SIMPLE) != 0){
	__cost+= ((2 + __cost) / 2);
	__programInsertOperator(OpCode._CURLY, offset);
      } else {
	__cost += (4 + __cost);
	__programAddTail(offset, __emitNode(OpCode._WHILEM));
	__programInsertOperator(OpCode._CURLYX, offset);
	__programAddTail(offset, __emitNode(OpCode._NOTHING));
      }

      if(min > 0)
	retFlags[0] = (__WORSTCASE | __NONNULL);

      if(max != 0 && max < min)
	throw new MalformedPatternException(
       "Invalid interval {" + min + "," + max + "}");

      if(__program!= null) {
	__program[offset + 2] = (char)min;
	__program[offset + 3] = (char)max;
      }
    }


    if(__input._getValue() == '?') {
      __getNextChar();
      __programInsertOperator(OpCode._MINMOD, offset);
      __programAddTail(offset, offset + 2);
    }

    if(__isComplexRepetitionOp(__input._array, __input._getOffset()))
      throw new MalformedPatternException(
        "Nested repetitions *?+ in expression");

    return offset;
  }
  private int __parseCharacterClass() throws MalformedPatternException {
    boolean range = false, skipTest;
    char clss, deflt, lastclss = Character.MAX_VALUE;
    int offset, bits, numLength[] = { 0 };

    offset = __emitNode(OpCode._ANYOF);

    if(__input._getValue() == '^') {
      ++__cost;
      __input._increment();
      deflt = 0;
    } else {
      deflt = 0xffff;
    }

    bits = __programSize;
    for(clss = 0; clss < 16; clss++)
      __emitCode(deflt);

    clss = __input._getValue();

    if(clss == ']' || clss == '-')
      skipTest = true;
    else
      skipTest = false;

    while((!__input._isAtEnd() && (clss = __input._getValue()) != ']')
	  || skipTest) {
      // It sucks, but we have to make this assignment every time
      skipTest = false;
      __input._increment();
      if(clss == '\\') {
	clss = __input._postIncrement();

	switch(clss){
	case 'w':
	  for(clss = 0; clss < 256; clss++)
	    if(OpCode._isWordCharacter(clss))
	      __setCharacterClassBits(__program, bits, deflt, clss);
	  lastclss = Character.MAX_VALUE;
	  continue;
	case 'W':
	  for(clss = 0; clss < 256; clss++)
	    if(!OpCode._isWordCharacter(clss))
	      __setCharacterClassBits(__program, bits, deflt, clss);
	  lastclss = Character.MAX_VALUE;
	  continue;
	case 's':
	  for(clss = 0; clss < 256; clss++)
	    if(Character.isWhitespace(clss))
	      __setCharacterClassBits(__program, bits, deflt, clss);
	  lastclss = Character.MAX_VALUE;
	  continue;
	case 'S':
	  for(clss = 0; clss < 256; clss++)
	    if(!Character.isWhitespace(clss))
	      __setCharacterClassBits(__program, bits, deflt, clss);
	  lastclss = Character.MAX_VALUE;
	  continue;
	case 'd':
	  for(clss = '0'; clss <= '9'; clss++)
	    __setCharacterClassBits(__program, bits, deflt, clss);
	  lastclss = Character.MAX_VALUE;
	  continue;
	case 'D':
	  for(clss = 0; clss < '0'; clss++)
	    __setCharacterClassBits(__program, bits, deflt, clss);
	  for(clss = (char)('9' + 1); clss < 256; clss++)
	    __setCharacterClassBits(__program, bits, deflt, clss);
	  lastclss = Character.MAX_VALUE;
	  continue;
	case 'n':
	  clss = '\n';
	  break;
	case 'r':
	  clss = '\r';
	  break;
	case 't':
	  clss = '\t';
	  break;
	case 'f':
	  clss = '\f';
	  break;
	case 'b':
	  clss = '\b';
	  break;
	case 'e':
	  clss = '\033';
	  break;
	case 'a':
	  clss = '\007';
	  break;
	case 'x':
	  clss = (char)__parseHex(__input._array, __input._getOffset(), 2,
				  numLength);
	  __input._increment(numLength[0]);
	  break;
	case 'c':
	  clss = __input._postIncrement();
	  if(Character.isLowerCase(clss))
	    clss = Character.toUpperCase(clss);
	  clss ^= 64;
	  break;
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	  clss = (char)__parseOctal(__input._array, __input._getOffset() - 1,
				    3, numLength);
	  __input._increment(numLength[0] - 1);
	  break;
	}
      }

      if(range) {
	if(lastclss > clss)
	  throw new MalformedPatternException(
			 "Invalid [] range in expression.");
	range = false;
      } else {
	lastclss = clss;

	if(__input._getValue() == '-' &&
	   __input._getOffset() + 1 < __input._getLength() &&
	   __input._getValueRelative(1) != ']') {
	  __input._increment();
	  range = true;
	  continue;
	}
      }

      while(lastclss <= clss) {
	__setCharacterClassBits(__program, bits, deflt, lastclss);
	if((__modifierFlags[0] & __CASE_INSENSITIVE) != 0 &&
	   Character.isUpperCase(lastclss))
	  __setCharacterClassBits(__program, bits, deflt,
				 Character.toLowerCase(lastclss));

	++lastclss;
      }

      lastclss = clss;
    }

    if(__input._getValue() != ']')
      throw new MalformedPatternException("Unmatched [] in expression.");

    __getNextChar();

    return offset;
  }
  private int __parseExpression(boolean isParenthesized, int[] hintFlags)
    throws MalformedPatternException {
    char value, paren;
    int nodeOffset = OpCode._NULL_OFFSET, parenthesisNum = 0, br, ender;
    int[] flags = { 0 };
    String modifiers = "iogmsx";


    // Initially we assume expression doesn't match null string.
    hintFlags[0] = __NONNULL;

    if (isParenthesized) {
      paren = 1;
      if(__input._getValue() == '?') {
	__input._increment();
	paren = value = __input._postIncrement();

	switch(value) {
	case ':' :
	case '=' :
	case '!' : break;
	case '#' :
	  value = __input._getValue();
	  while(value != CharStringPointer._END_OF_STRING && value != ')')
	    value = __input._increment();
	  if(value != ')')
	    throw new MalformedPatternException(
	       "Sequence (?#... not terminated");
	  __getNextChar();
	  hintFlags[0] = __TRYAGAIN;
	  return OpCode._NULL_OFFSET;
	default :
	  __input._decrement();
	  value = __input._getValue();
	  while(value != CharStringPointer._END_OF_STRING &&
		modifiers.indexOf(value) != -1) {
	    __setModifierFlag(__modifierFlags, value);
	    value = __input._increment();
	  }
	  if(value != ')')
	    throw new MalformedPatternException(
	       "Sequence (?" + value + "...) not recognized");
	  __getNextChar();
	  hintFlags[0] = __TRYAGAIN;
	  return OpCode._NULL_OFFSET;
	}
      } else {
	parenthesisNum = __numParentheses;
	++__numParentheses;
	nodeOffset = __emitArgNode(OpCode._OPEN, (char)parenthesisNum);
      }
    } else 
      paren = 0;

    br = __parseAlternation(flags);

    if(br == OpCode._NULL_OFFSET)
      return OpCode._NULL_OFFSET;

    if(nodeOffset != OpCode._NULL_OFFSET)
      __programAddTail(nodeOffset, br);
    else
      nodeOffset = br;

    if((flags[0] & __NONNULL) == 0)
      hintFlags[0] &= ~__NONNULL;

    hintFlags[0] |= (flags[0] & __SPSTART);

    while(__input._getValue() == '|') {
      __getNextChar();
      br = __parseAlternation(flags);

      if(br == OpCode._NULL_OFFSET)
	return OpCode._NULL_OFFSET;

      __programAddTail(nodeOffset, br);

      if((flags[0] & __NONNULL) == 0)
	hintFlags[0] &= ~__NONNULL;

      hintFlags[0] |= (flags[0] & __SPSTART);
    }

    switch(paren) {
    case ':' :
      ender = __emitNode(OpCode._NOTHING);
      break;
    case 1:
      ender = __emitArgNode(OpCode._CLOSE, (char)parenthesisNum);
      break;
    case '=':
    case '!':
      ender = __emitNode(OpCode._SUCCEED);
      hintFlags[0] &= ~__NONNULL;
      break;
    case 0  :
    default :
      ender = __emitNode(OpCode._END);
      break;
    }

    __programAddTail(nodeOffset, ender);

    for(br = nodeOffset; br != OpCode._NULL_OFFSET;
	br = OpCode._getNext(__program, br))
      __programAddOperatorTail(br, ender);

    if(paren == '=') {
      __programInsertOperator(OpCode._IFMATCH, nodeOffset);
      __programAddTail(nodeOffset, __emitNode(OpCode._NOTHING));
    } else if(paren == '!') {
      __programInsertOperator(OpCode._UNLESSM, nodeOffset);
      __programAddTail(nodeOffset, __emitNode(OpCode._NOTHING));
    }

    if(paren != 0 && (__input._isAtEnd() || __getNextChar() != ')')) {
      throw new MalformedPatternException("Unmatched parentheses.");
    } else if(paren == 0 && !__input._isAtEnd()) { 
      if(__input._getValue() == ')')
	throw new MalformedPatternException("Unmatched parentheses.");
      else
	// Should never happen.
	throw new MalformedPatternException(
       "Unreached characters at end of expression.  Please report this bug!");
    }


    return nodeOffset;
  }
  private static int __parseHex(char[] str, int offset, int maxLength,
				int[] scanned)
  {
    int val = 0, index;

    scanned[0] = 0;
    while(offset < str.length && maxLength-- > 0 &&
	  (index = __HEX_DIGIT.indexOf(str[offset])) != -1) {
      val <<= 4;
      val |= (index & 15);
      ++offset;
      ++scanned[0];
    }

    return val;
  }
  private static int __parseOctal(char[] str, int offset, int maxLength,
				 int[] scanned)
  {
    int val = 0, index;

    scanned[0] = 0;
    while(offset < str.length && 
	  maxLength > 0 && str[offset] >= '0' && str[offset] <= '7') {
      val <<= 3;
      val |= (str[offset] - '0');
      --maxLength;
      ++offset;
      ++scanned[0];
    }

    return val;
  }
  // determines if {\d+,\d*} is the next part of the string
  private static boolean __parseRepetition(char[] str, int offset) {
    if(str[offset] != '{')
      return false;
    ++offset;

    if(offset >= str.length || !Character.isDigit(str[offset]))
      return false;

    while(offset < str.length && Character.isDigit(str[offset]))
      ++offset;

    if(offset < str.length && str[offset] == ',')
      ++offset;

    while(offset < str.length && Character.isDigit(str[offset]))
      ++offset;

    if(offset >= str.length || str[offset] != '}')
      return false;

    return true;
  }
  private void __programAddOperatorTail(int current, int value) {
    if(__program== null || current == OpCode._NULL_OFFSET ||
       OpCode._opType[__program[current]] != OpCode._BRANCH)
      return;
    __programAddTail(OpCode._getNextOperator(current), value);
  }
  private void __programAddTail(int current, int value) {
    int scan, temp, offset;

    if(__program== null || current == OpCode._NULL_OFFSET)
      return;

    scan = current;

    while(true) {
      temp = OpCode._getNext(__program, scan);
      if(temp == OpCode._NULL_OFFSET)
	break;
      scan = temp;
    }

    if(__program[scan] == OpCode._BACK)
      offset = scan - value;
    else
      offset = value - scan;

    __program[scan + 1] = (char)offset;
  }
  // Insert an operator at a given offset.
  private void __programInsertOperator(char operator, int operand) {
    int src, dest, offset;

    offset = (OpCode._opType[operator] == OpCode._CURLY ? 2 : 0);


    if(__program== null) {
      __programSize+=(2 + offset);
      return;
    }

    src = __programSize;
    __programSize+=(2 + offset);
    dest = __programSize;

    while(src > operand) {
      --src;
      --dest;
      __program[dest] = __program[src];
    }

    __program[operand++] = operator;
    __program[operand++] = OpCode._NULL_POINTER;

    while(offset-- > 0)
      __program[operand++] = OpCode._NULL_POINTER;

  }
  // Set the bits in a character class.  Only recognizes ascii.
  private void __setCharacterClassBits(char[] bits, int offset, char deflt,
				       char ch)
  {
    if(__program== null || ch >= 256)
      return;
    ch &= 0xffff;

    if(deflt == 0) {
      bits[offset + (ch >> 4)] |= (1 << (ch & 0xf));
    } else {
      bits[offset + (ch >> 4)] &= ~(1 << (ch & 0xf));
    }
  }
  private static void __setModifierFlag(char[] flags, char ch) {
    switch(ch) {
    case 'i' : flags[0] |= __CASE_INSENSITIVE; return;
    case 'g' : flags[0] |= __GLOBAL; return;
    case 'o' : flags[0] |= __KEEP; return;
    case 'm' : flags[0] |= __MULTILINE; return;
    case 's' : flags[0] |= __SINGLELINE; return;
    case 'x' : flags[0] |= __EXTENDED; return;
    }
  }
  /**
   * Same as calling <b>compile(pattern, Perl5Compiler.DEFAULT_MASK);</b>
   * <p>
   * @param pattern  A regular expression to compile.
   * @return A Pattern instance constituting the compiled regular expression.
   *         This instance will always be a Perl5Pattern and can be reliably
   *         casted to a Perl5Pattern.
   * @exception MalformedPatternException  If the compiled expression
   *  is not a valid Perl5 regular expression.
   */
  public Pattern compile(char[] pattern) throws MalformedPatternException {
	 return compile(pattern, DEFAULT_MASK);
  }
  /**
   * Compiles a Perl5 regular expression into a Perl5Pattern instance that
   * can be used by a Perl5Matcher object to perform pattern matching.
   * Please see the user's guide for more information about Perl5 regular
   * expressions.
   * <p>
   * @param pattern  A Perl5 regular expression to compile.
   * @param options  A set of flags giving the compiler instructions on
   *                 how to treat the regular expression.  The flags
   *                 are a logical OR of any number of the five <b>MASK</b>
   *                 constants.  For example:
   *                 <pre>
   * regex =
   *   compiler.compile(pattern, Perl5Compiler.
   *                    CASE_INSENSITIVE_MASK |
   *                    Perl5Compiler.MULTILINE_MASK);
   *                 </pre>
   *                  This says to compile the pattern so that it treats
   *                  input as consisting of multiple lines and to perform
   *                  matches in a case insensitive manner.
   * @return A Pattern instance constituting the compiled regular expression.
   *         This instance will always be a Perl5Pattern and can be reliably
   *         casted to a Perl5Pattern.
   * @exception MalformedPatternException  If the compiled expression
   *  is not a valid Perl5 regular expression.
   */
  public Pattern compile(char[] pattern, int options)
       throws MalformedPatternException {
    int[] flags = { 0 };
    int caseInsensitive, scan;
    Perl5Pattern regexp;
    String mustString, startString;

    int first;
    boolean sawOpen = false, sawPlus = false;

    StringBuffer lastLongest, longest;
    int length, minLength = 0, curBack, back, backmost;


    __input = new CharStringPointer(pattern);

    caseInsensitive    = options & __CASE_INSENSITIVE;
    __modifierFlags[0] = (char)options;
    __sawBackreference = false;
    __numParentheses   = 1;
    __programSize      = 0;
    __cost             = 0;
    __program= null;

    __emitCode((char)0);
    if(__parseExpression(false, flags) == OpCode._NULL_OFFSET) {
      //System.err.println("null -- Size: " + __programSize); // debug
      // return null;
      throw new MalformedPatternException("Unknown compilation error.");
    }

    //System.err.println("First Pass Size: " + __programSize); //debug

    if(__programSize >= Character.MAX_VALUE - 1)
      throw new MalformedPatternException("Expression is too large.");


    __program= new char[__programSize];
    regexp = new Perl5Pattern();

    regexp._program    = __program;
    regexp._expression = new String(pattern);

    __input._setOffset(0);

    __numParentheses   = 1;
    __programSize      = 0;
    __cost             = 0;

    __emitCode((char)0);
    if(__parseExpression(false, flags) == OpCode._NULL_OFFSET) {
      //System.err.println("null -- Size: " + __programSize); //debug 
      //return null;
      throw new MalformedPatternException("Unknown compilation error.");
    }

    //System.err.println("Second Pass Size: " + __programSize); //debug

    caseInsensitive = __modifierFlags[0] & __CASE_INSENSITIVE;

    regexp._isExpensive      = (__cost >= 10);
    regexp._startClassOffset = OpCode._NULL_OFFSET;
    regexp._anchor           = 0;
    regexp._back             = -1;
    regexp._options          = options;
    regexp._startString      = null;
    regexp._mustString       = null;
    mustString               = null;
    startString              = null;

    scan = 1;
    if(__program[OpCode._getNext(__program, scan)] == OpCode._END){
      boolean doItAgain;  // bad variables names!
      char op;

      first = scan = OpCode._getNextOperator(scan);
      op = __program[first];

      while((op == OpCode._OPEN && (sawOpen = true)) ||
	    (op == OpCode._BRANCH &&
	     __program[OpCode._getNext(__program, first)] != OpCode._BRANCH) ||
	    op == OpCode._PLUS || op == OpCode._MINMOD ||
	    (OpCode._opType[op] == OpCode._CURLY && 
	     OpCode._getArg1(__program, first) > 0)) {
	if(op == OpCode._PLUS)
	  sawPlus = true;
	else
	  first+=OpCode._operandLength[op];

	first = OpCode._getNextOperator(first);
	op = __program[first];
      }

      doItAgain = true;

      while(doItAgain) {
	doItAgain = false;
	op = __program[first];

	if(op == OpCode._EXACTLY) {
	  startString =
	    new String(__program, OpCode._getOperand(first + 1),
		       __program[OpCode._getOperand(first)]);

	} else if(OpCode._isInArray(op, OpCode._opLengthOne, 2))
	  regexp._startClassOffset = first;
	else if(op == OpCode._BOUND || op == OpCode._NBOUND)
	  regexp._startClassOffset = first;
	else if(OpCode._opType[op] == OpCode._BOL) {
	  regexp._anchor = Perl5Pattern._OPT_ANCH;
	  first = OpCode._getNextOperator(first);
	  doItAgain = true;
	  continue;
	} else if(op == OpCode._STAR &&
		  OpCode._opType[__program[OpCode._getNextOperator(first)]] == 
		  OpCode._ANY && (regexp._anchor & Perl5Pattern._OPT_ANCH) != 0)
	  {
	    regexp._anchor = Perl5Pattern._OPT_ANCH | Perl5Pattern._OPT_IMPLICIT;
	    first = OpCode._getNextOperator(first);
	    doItAgain = true;
	    continue;
	}
      } // end while do it again

      if(sawPlus && (!sawOpen || !__sawBackreference))
	regexp._anchor |= Perl5Pattern._OPT_SKIP;


      //length = OpCode._getNextOperator(first); //debug
      // System.err.println("first: " + first + "nextoper: " + length);
      //System.err.print("first " + (int)op + " next "); // debug
      //if(length >= 0 && length < _program.length) //debug
      //System.err.print((int)(__program[length])); //debug
      //else  //debug
      //System.err.print("out of range"); //debug
      //System.err.println(" offset " + (int)(first - scan)); // debug

      lastLongest   = new StringBuffer();
      longest   = new StringBuffer();
      length    = 0;
      minLength = 0;
      curBack   = 0;
      back   = 0;
      backmost   = 0;

      while(scan > 0 && (op = __program[scan]) != OpCode._END) {

	if(op == OpCode._BRANCH) {
	  if(__program[OpCode._getNext(__program, scan)] == OpCode._BRANCH) {
	    curBack = -30000;
	    while(__program[scan] == OpCode._BRANCH)
	      scan = OpCode._getNext(__program, scan);
	  } else
	    scan = OpCode._getNextOperator(scan);
	  continue;
	}

	if(op == OpCode._UNLESSM) {
	  curBack = -30000;
	  scan = OpCode._getNext(__program, scan);
	  continue;
	}

	if(op == OpCode._EXACTLY) {
	  int temp;

	  first = scan;
	  while(__program[(temp = OpCode._getNext(__program, scan))] == 
		OpCode._CLOSE)
	    scan = temp;

	  minLength += __program[OpCode._getOperand(first)];

	  temp = __program[OpCode._getOperand(first)];

	  if(curBack - back == length) {
	    lastLongest.append(new String(__program, OpCode._getOperand(first) + 1,
				      temp));
	    length  += temp;
	    curBack += temp;
	    first = OpCode._getNext(__program, scan);
	  } else if(temp >= (length + (curBack >= 0 ? 1 : 0))) {
	    length = temp;
	    lastLongest =
	      new StringBuffer(new String(__program,
					  OpCode._getOperand(first) + 1, temp));
	    back = curBack;
	    curBack += length;
	    first = OpCode._getNext(__program, scan);
	  } else
	    curBack += temp;
	} else if(OpCode._isInArray(op, OpCode._opLengthVaries, 0)) {
	  curBack = -30000;
	  length = 0;

	  if(lastLongest.length() > longest.length()) {
	    longest = lastLongest;
	    backmost = back;
	  }

	  lastLongest = new StringBuffer();

	  if(op == OpCode._PLUS && 
	     OpCode._isInArray(__program[OpCode._getNextOperator(scan)],
			    OpCode._opLengthOne, 0))
	    ++minLength;
	  else if(OpCode._opType[op] == OpCode._CURLY &&
		  OpCode._isInArray(__program[OpCode._getNextOperator(scan) + 2],
				 OpCode._opLengthOne, 0))
	    minLength += OpCode._getArg1(__program, scan);
	} else if(OpCode._isInArray(op, OpCode._opLengthOne, 0)) {
	  ++curBack;
	  ++minLength;
	  length = 0;
	  if(lastLongest.length() > longest.length()) {
	    longest = lastLongest;
	    backmost = back;
	  }
	  lastLongest = new StringBuffer();
	}

	scan = OpCode._getNext(__program, scan);
      } // end while

      if(lastLongest.length() +
	 ((OpCode._opType[__program[first]] == OpCode._EOL) ? 1 : 0) >
	 longest.length()) {
	longest = lastLongest;
	backmost = back;
      } else
	lastLongest = new StringBuffer();

      if(longest.length() > 0 && startString == null) {
	mustString = longest.toString();
	if(backmost < 0)
	  backmost = -1;
	regexp._back = backmost;

	/*

	  if(longest.length() > 
	  (((caseInsensitive & __CASE_INSENSITIVE) != 0 ||
	  OpCode._opType[__program[first]] == OpCode._EOL)
	  ? 1 : 0))
	  */	    
      } else
	longest = null;
    } // end if


    regexp._isCaseInsensitive = ((caseInsensitive & __CASE_INSENSITIVE) != 0);
    regexp._numParentheses  = __numParentheses - 1;
    regexp._minLength       = minLength;

    if(mustString != null) {
      regexp._mustString = mustString.toCharArray();
      regexp._mustUtility = 100;
    }

    if(startString != null)
      regexp._startString = startString.toCharArray();

    return regexp;
  }
  /**
   * Same as calling <b>compile(pattern, Perl5Compiler.DEFAULT_MASK);</b>
   * <p>
   * @param pattern  A regular expression to compile.
   * @return A Pattern instance constituting the compiled regular expression.
   *         This instance will always be a Perl5Pattern and can be reliably
   *         casted to a Perl5Pattern.
   * @exception MalformedPatternException  If the compiled expression
   *  is not a valid Perl5 regular expression.
   */
  public Pattern compile(String pattern) throws MalformedPatternException {
	 return compile(pattern.toCharArray(), DEFAULT_MASK);
  }
  /**
   * Compiles a Perl5 regular expression into a Perl5Pattern instance that
   * can be used by a Perl5Matcher object to perform pattern matching.
   * Please see the user's guide for more information about Perl5 regular
   * expressions.
   * <p>
   * @param pattern  A Perl5 regular expression to compile.
   * @param options  A set of flags giving the compiler instructions on
   *                 how to treat the regular expression.  The flags
   *                 are a logical OR of any number of the five <b>MASK</b>
   *                 constants.  For example:
   *                 <pre>
   * regex =
   *   compiler.compile("^\\w+\\d+$",
   *                    Perl5Compiler.CASE_INSENSITIVE_MASK |
   *                    Perl5Compiler.MULTILINE_MASK);
   *                 </pre>
   *                  This says to compile the pattern so that it treats
   *                  input as consisting of multiple lines and to perform
   *                  matches in a case insensitive manner.
   * @return A Pattern instance constituting the compiled regular expression.
   *         This instance will always be a Perl5Pattern and can be reliably
   *         casted to a Perl5Pattern.
   * @exception MalformedPatternException  If the compiled expression
   *  is not a valid Perl5 regular expression.
   */
  public Pattern compile(String pattern, int options)
       throws MalformedPatternException {
	 return compile(pattern.toCharArray(), options);
  }
  /**
   * Given a character string, returns a Perl5 expression that interprets
   * each character of the original string literally.  In other words, all
   * special metacharacters are quoted/escaped.  This method is useful for
   * converting user input meant for literal interpretation into a safe
   * regular expression representing the literal input.
   * <p>
   * In effect, this method is the analog of the Perl5 quotemeta() builtin
   * method.
   * <p>
   * @param expression The expression to convert.
   * @return A String containing a Perl5 regular expression corresponding to
   *         a literal interpretation of the pattern.
   */
  public static final String quotemeta(char[] expression) {
    int ch;
    StringBuffer buffer;

    buffer = new StringBuffer(2*expression.length);
    for(ch = 0; ch < expression.length; ch++) {
      if(!OpCode._isWordCharacter(expression[ch]))
	buffer.append('\\');
      buffer.append(expression[ch]);
    }

    return buffer.toString();
  }
  /**
   * Given a character string, returns a Perl5 expression that interprets
   * each character of the original string literally.  In other words, all
   * special metacharacters are quoted/escaped.  This method is useful for
   * converting user input meant for literal interpretation into a safe
   * regular expression representing the literal input.
   * <p>
   * In effect, this method is the analog of the Perl5 quotemeta() builtin
   * method.
   * <p>
   * @param pattern The pattern to convert.
   * @return A String containing a Perl5 regular expression corresponding to
   *         a literal interpretation of the pattern.
   */
  public static final String quotemeta(String expression) {
    return quotemeta(expression.toCharArray());
  }
}
package org.apache.oro.text.regex;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation", "Jakarta-Oro" 
 *    must not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" 
 *    or "Jakarta-Oro", nor may "Apache" or "Jakarta-Oro" appear in their 
 *    name, without prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon software originally written 
 * by Daniel F. Savarese. We appreciate his contributions.
 */

import java.io.IOException;
import java.util.*;

/**
 * The Perl5Matcher class is used to match regular expressions
 * (conforming to the Perl5 regular expression syntax) generated by
 * Perl5Compiler.

 @author <a href="mailto:dfs@savarese.org">Daniel F. Savarese</a>
 @version $Id: Perl5Matcher.java,v 1.2 2000/07/23 23:25:26 jon Exp $

 * @see PatternMatcher
 * @see Perl5Compiler
 */
public final class Perl5Matcher implements PatternMatcher {
  private static final char __EOS = Character.MAX_VALUE;
  private static final int __INITIAL_NUM_OFFSETS = 20;

  private boolean __multiline = false, __lastSuccess = false;
  private char __previousChar, __input[], __originalInput[];
  private Perl5Repetition __currentRep;
  private int __numParentheses, __bol, __eol, __currentOffset, __endOffset;

  private char[] __program;
  private int __expSize, __inputOffset, __lastParen;
  private int[] __beginMatchOffsets, __endMatchOffsets;
  private Stack __stack = new Stack();
  private Perl5MatchResult __lastMatchResult = null;

  private static final int __DEFAULT_LAST_MATCH_END_OFFSET = -100;
  private int __lastMatchInputEndOffset = __DEFAULT_LAST_MATCH_END_OFFSET;
  private static boolean
    __compare(char[] s1, int s1Offs, char[] s2, int s2Offs, int n)
  {
    int cnt;

    for(cnt = 0; cnt < n; cnt++, s1Offs++, s2Offs++) {
      if(s1Offs >= s1.length)
	return false;
      if(s2Offs >= s2.length)
	return false;
      if(s1[s1Offs] != s2[s2Offs])
	return false;
    }

    return true;
  }
  private static int __findFirst(char[] input, int current, int endOffset,
				 char[] mustString)
  {
    int count, saveCurrent;
    char ch;


    if(input.length == 0)
      return endOffset;

    ch = mustString[0];
    // Find the offset of the first character of the must string
    while(current < endOffset) {
      if(ch == input[current]){
	saveCurrent = current;
	count = 0;

	while(current < endOffset && count < mustString.length) {
	  if(mustString[count] != input[current])
	    break;
	  ++count;
	  ++current;
	}

	current = saveCurrent;

	if(count >= mustString.length)
	  break;
      }
      ++current;
    }

    return current;
  }
  // Initialize globals needed before calling __tryExpression for first time
  private void __initInterpreterGlobals(Perl5Pattern expression, char[] input,
					int beginOffset, int endOffset) {
    __input                      = input;
    __endOffset                  = endOffset;
    __currentRep                 = new Perl5Repetition();
    __currentRep._numInstances   = 0;
    __currentRep._lastRepetition = null;
    __program                    = expression._program;
    __stack.setSize(0);

    if(beginOffset == 0)
      __previousChar = '\n';
    else {
      __previousChar = input[beginOffset - 1];
      if(!__multiline && __previousChar == '\n')
	__previousChar = '\0';
    }

    __numParentheses    = expression._numParentheses;
    __currentOffset     = beginOffset;

    __bol = beginOffset;
    __eol = endOffset;

    // Ok, here we're using endOffset as a temporary variable.
    endOffset = __numParentheses + 1;
    if(__beginMatchOffsets == null || endOffset > __beginMatchOffsets.length) {
      if(endOffset < __INITIAL_NUM_OFFSETS)
	endOffset = __INITIAL_NUM_OFFSETS;
      __beginMatchOffsets = new int[endOffset];
      __endMatchOffsets   = new int[endOffset];      
    }
  }
  // Expects to receive a valid regular expression program.  No checking
  // is done to ensure validity.
  // __originalInput must be set before calling this method for
  // __lastMatchResult to be set correctly.
  private boolean __interpret(Perl5Pattern expression, char[] input,
			      int beginOffset, int endOffset)
  {
    boolean success;
    int minLength = 0, dontTry = 0, offset;
    char ch, mustString[];

    __initInterpreterGlobals(expression, input, beginOffset, endOffset);

    success = false;
    mustString = expression._mustString;

  _mainLoop:
    while(true) {

      if(mustString != null &&
	 ((expression._anchor & Perl5Pattern._OPT_ANCH) == 0 ||
	  (__multiline && expression._back >= 0))) {

	__currentOffset =
	  __findFirst(__input, __currentOffset, endOffset, mustString);
	
	if(__currentOffset >= endOffset) {
	  if((expression._options & Perl5Compiler.READ_ONLY_MASK) == 0)
	    expression._mustUtility++;
	  success = false;
	  break _mainLoop;
	} else if(expression._back >= 0) {
	  __currentOffset-=expression._back;
	  if(__currentOffset < beginOffset)
	    __currentOffset = beginOffset;
	  minLength = expression._back + mustString.length;
	} else if(!expression._isExpensive &&
		  (expression._options & Perl5Compiler.READ_ONLY_MASK) == 0 &&
		  (--expression._mustUtility < 0)) {
	  // Be careful!  The preceding logical expression is constructed
	  // so that mustUtility is only decremented if the expression is
	  // compiled without READ_ONLY_MASK.
	  mustString = expression._mustString = null;
	  __currentOffset = beginOffset;
	} else {
	  __currentOffset = beginOffset;
	  minLength = mustString.length;
	}
      }

      if((expression._anchor & Perl5Pattern._OPT_ANCH) != 0) {
	if(__tryExpression(expression, beginOffset)) {
	  success = true;
	  break _mainLoop;
	} else if(__multiline ||
		  (expression._anchor & Perl5Pattern._OPT_IMPLICIT) != 0) {

	  if(minLength > 0)
	    dontTry = minLength - 1;
	  endOffset-=dontTry;

	  if(__currentOffset > beginOffset)
	    --__currentOffset;

	  while(__currentOffset < endOffset) {
	    if(__input[__currentOffset++] == '\n') {
	      if(__currentOffset < endOffset &&
		 __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      }
	    }
	  }
	}

	break _mainLoop;
      }


      if(expression._startString != null) {
	mustString = expression._startString;
	if((expression._anchor & Perl5Pattern._OPT_SKIP) != 0) {
	  ch = mustString[0];

	  while(__currentOffset < endOffset) {
	    if(ch == __input[__currentOffset]) {
	      if(__tryExpression(expression, __currentOffset)){
		success = true;
		break _mainLoop;
	      }
	      ++__currentOffset;
	      while(__currentOffset < endOffset &&
		    __input[__currentOffset] == ch)
		    ++__currentOffset;
	    }
	    ++__currentOffset;
	  }
	} else {

	  while((__currentOffset =
		 __findFirst(__input, __currentOffset, endOffset, mustString))
		< endOffset){
	    if(__tryExpression(expression, __currentOffset)) {
	      success = true;
	      break _mainLoop;
	    }
	    ++__currentOffset;
	  }
	}

	break _mainLoop;
      }

      if((offset = expression._startClassOffset) != OpCode._NULL_OFFSET) {
	boolean doEvery, tmp;

	doEvery = ((expression._anchor & Perl5Pattern._OPT_SKIP) == 0);

	if(minLength > 0)
	  dontTry = minLength - 1;
	endOffset -= dontTry;
	tmp = true;

	switch(__program[offset]) {
	case OpCode._ANYOF:
	  offset = OpCode._getOperand(offset);
	  while(__currentOffset < endOffset) {
	    ch = __input[__currentOffset];

	    if(ch < 256 &&
	       (__program[offset + (ch >> 4)] & (1 << (ch & 0xf))) == 0) {
	      if(tmp && __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      } else
		tmp = doEvery;
	    } else
	      tmp = true;
	    ++__currentOffset;
	  }

	  break;

	case OpCode._BOUND:
	  if(minLength > 0) {
	    ++dontTry;
	    --endOffset;
	  }

	  if(__currentOffset != beginOffset) {
	    ch = __input[__currentOffset - 1];
	    tmp = OpCode._isWordCharacter(ch);
	  } else
	    tmp = OpCode._isWordCharacter(__previousChar);

	  while(__currentOffset < endOffset) {
	    ch = __input[__currentOffset];
	    if(tmp != OpCode._isWordCharacter(ch)){
	      tmp = !tmp;
	      if(__tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      }
	    }
	    ++__currentOffset;
	  }

	  if((minLength > 0 || tmp) && 
	     __tryExpression(expression, __currentOffset)) {
	    success = true;
	    break _mainLoop;
	  }
	  break;

	case OpCode._NBOUND:
	  if(minLength > 0) {
	    ++dontTry;
	    --endOffset;
	  }

	  if(__currentOffset != beginOffset) {
	    ch = __input[__currentOffset - 1];
	    tmp = OpCode._isWordCharacter(ch);
	  } else
	    tmp = OpCode._isWordCharacter(__previousChar);

	  while(__currentOffset < endOffset) {
	    ch = __input[__currentOffset];
	    if(tmp != OpCode._isWordCharacter(ch))
	      tmp = !tmp;
	    else if(__tryExpression(expression, __currentOffset)) {
	      success = true;
	      break _mainLoop;
	    }

	    ++__currentOffset;
	  }

	  if((minLength > 0 || !tmp) &&
	     __tryExpression(expression, __currentOffset)) {
	    success = true;
	    break _mainLoop;
	  }
	  break;

	case OpCode._ALNUM:
	  while(__currentOffset < endOffset) {
	    ch = __input[__currentOffset];
	    if(OpCode._isWordCharacter(ch)) {
	      if(tmp && __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      } else
		tmp = doEvery;
	    } else
	      tmp = true;
	    ++__currentOffset;
	  }
	  break;

	case OpCode._NALNUM:
	  while(__currentOffset < endOffset) {
	    ch = __input[__currentOffset];
	    if(!OpCode._isWordCharacter(ch)) {
	      if(tmp && __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      } else
		tmp = doEvery;
	    } else
	      tmp = true;
	    ++__currentOffset;
	  }
	  break;

	case OpCode._SPACE:
	  while(__currentOffset < endOffset) {
	    if(Character.isWhitespace(__input[__currentOffset])) {
	      if(tmp && __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      } else
		tmp = doEvery;
	    } else
	      tmp = true;
	    ++__currentOffset;
	  }
	  break;

	case OpCode._NSPACE:
	  while(__currentOffset < endOffset) {
	    if(!Character.isWhitespace(__input[__currentOffset])) {
	      if(tmp && __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      } else
		tmp = doEvery;
	    } else
	      tmp = true;
	    ++__currentOffset;
	  }
	  break;

	case OpCode._DIGIT:
	  while(__currentOffset < endOffset) {
	    if(Character.isDigit(__input[__currentOffset])) {
	      if(tmp && __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      } else
		tmp = doEvery;
	    } else
	      tmp = true;
	    ++__currentOffset;
	  }
	  break;


	case OpCode._NDIGIT:
	  while(__currentOffset < endOffset) {
	    if(!Character.isDigit(__input[__currentOffset])) {
	      if(tmp && __tryExpression(expression, __currentOffset)) {
		success = true;
		break _mainLoop;
	      } else
		tmp = doEvery;
	    } else
	      tmp = true;
	    ++__currentOffset;
	  }
	  break;
	} // end switch

      } else {
	if(minLength > 0)
	  dontTry = minLength - 1;
	endOffset-=dontTry;

	do {
	  if(__tryExpression(expression, __currentOffset)) {
	    success = true;
	    break _mainLoop;
	  }
	} while(__currentOffset++ < endOffset);

      }


      break _mainLoop;
    } // end while

    __lastSuccess = success;
    __lastMatchResult = null;

    return success;
  }
  private boolean __match(int offset) {
    char nextChar, op;
    int scan, next, input, maxScan, current, line, arg;
    boolean inputRemains = true, minMod = false;
    Perl5Repetition rep;


    input    = __inputOffset;
    inputRemains = (input < __endOffset);
    nextChar = (inputRemains ? __input[input] : __EOS);

    scan     = offset;
    maxScan  = __program.length;

    while(scan < maxScan /*&& scan > 0*/){
      next = OpCode._getNext(__program, scan);

      switch(op = __program[scan]) {

      case OpCode._BOL:
	if(input == __bol ? __previousChar == '\n' :
	   (__multiline && (inputRemains || input < __eol) && 
	    __input[input - 1] == '\n'))
	  break;
	return false;

      case OpCode._MBOL:
	if(input == __bol ? __previousChar == '\n' :
	   ((inputRemains || input < __eol) && __input[input - 1] == '\n'))
	  break;
	return false;

      case OpCode._SBOL:
	if(input == __bol && __previousChar == '\n')
	  break;
	return false;

      case OpCode._GBOL:
	if(input == __bol)
	  break;
	return true;

      case OpCode._EOL :
	if((inputRemains || input < __eol) && nextChar != '\n')
	  return false;
	if(!__multiline && __eol - input > 1)
	  return false;
	break;

      case OpCode._MEOL:
	if((inputRemains || input < __eol) && nextChar != '\n')
	  return false;
	break;

      case OpCode._SEOL:
	if((inputRemains || input < __eol) && nextChar != '\n')
	  return false;
	if(__eol - input > 1)
	  return false;
	break;

      case OpCode._SANY:
	if(!inputRemains && input >= __eol)
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._ANY:
	if((!inputRemains && input >= __eol) || nextChar == '\n')
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._EXACTLY:
	current = OpCode._getOperand(scan);
	line = __program[current++];

	if(__program[current] != nextChar)
	  return false;
	if(__eol - input < line)
	  return false;

	if(line > 1 && !__compare(__program, current, __input, input, line))
	  return false;

	input+=line;
	inputRemains = (input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._ANYOF:
	current = OpCode._getOperand(scan);

	if(nextChar == __EOS && inputRemains)
	  nextChar = __input[input];

	if(nextChar >= 256 || (__program[current + (nextChar >> 4)] &
	    (1 << (nextChar & 0xf))) != 0)
	  return false;

	if(!inputRemains && input >= __eol)
	  return false;

	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._ALNUM:
	if(!inputRemains)
	  return false;
	if(!OpCode._isWordCharacter(nextChar))
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._NALNUM:
	if(!inputRemains && input >= __eol)
	  return false;
	if(OpCode._isWordCharacter(nextChar))
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;


      case OpCode._NBOUND:
      case OpCode._BOUND:
	boolean a, b;

	if(input == __bol)
	  a = OpCode._isWordCharacter(__previousChar);
	else
	  a = OpCode._isWordCharacter(__input[input - 1]);

	b = OpCode._isWordCharacter(nextChar);

	if((a == b) == (__program[scan] == OpCode._BOUND))
	  return false;
	break;

      case OpCode._SPACE:
	if(!inputRemains && input >= __eol)
	  return false;
	if(!Character.isWhitespace(nextChar))
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;


      case OpCode._NSPACE:
	if(!inputRemains)
	  return false;
	if(Character.isWhitespace(nextChar))
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._DIGIT:
	if(!Character.isDigit(nextChar))
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._NDIGIT:
	if(!inputRemains && input >= __eol)
	  return false;
	if(Character.isDigit(nextChar))
	  return false;
	inputRemains = (++input < __endOffset);
	nextChar = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._REF:
	arg = OpCode._getArg1(__program, scan);
	current = __beginMatchOffsets[arg];

	if(current == OpCode._NULL_OFFSET)
	  return false;

	if(__endMatchOffsets[arg] == OpCode._NULL_OFFSET)
	  return false;

	if(current == __endMatchOffsets[arg])
	  break;

	if(__input[current] != nextChar)
	  return false;

	line = __endMatchOffsets[arg] - current;

	if(input + line > __eol)
	  return false;

	if(line > 1 && !__compare(__input, current, __input, input, line))
	  return false;

	input+=line;
	inputRemains = (input < __endOffset);
	nextChar     = (inputRemains ? __input[input] : __EOS);
	break;

      case OpCode._NOTHING:
	break;

      case OpCode._BACK:
	break;

      case OpCode._OPEN:
	arg = OpCode._getArg1(__program, scan);
	__beginMatchOffsets[arg] = input;

	if(arg > __expSize)
	  __expSize = arg;
	break;

      case OpCode._CLOSE:
	arg = OpCode._getArg1(__program, scan);
	__endMatchOffsets[arg] = input;

	if(arg > __lastParen)
	  __lastParen = arg;
	break;

      case OpCode._CURLYX:
	rep = new Perl5Repetition();
	rep._lastRepetition = __currentRep;
	__currentRep = rep;

	rep._parenFloor = __lastParen;
	rep._numInstances = -1;
	rep._min    = OpCode._getArg1(__program, scan);
	rep._max    = OpCode._getArg2(__program, scan);
	rep._scan   = OpCode._getNextOperator(scan) + 2;
	rep._next   = next;
	rep._minMod = minMod;
	// Must initialize to -1 because if we initialize to 0 and are
	// at the beginning of the input the OpCode._WHILEM case will
	// not work right.
	rep._lastLocation = -1;
	__inputOffset = input;

	// use minMod as temporary
	minMod = __match(OpCode._getPrevOperator(next));

	// leave scope call not pertinent?
	__currentRep = rep._lastRepetition;
	return minMod;

      case OpCode._WHILEM:
	rep = __currentRep;

	arg = rep._numInstances + 1;
	__inputOffset = input;

	if(input == rep._lastLocation) {
	  __currentRep = rep._lastRepetition;
	  line = __currentRep._numInstances;
	  if(__match(rep._next))
	    return true;
	  __currentRep._numInstances = line;
	  __currentRep = rep;
	  return false;
	}

	if(arg < rep._min) {
	  rep._numInstances = arg;
	  rep._lastLocation = input;
	  if(__match(rep._scan))
	    return true;
	  rep._numInstances = arg - 1;
	  return false;
	}

	if(rep._minMod) {
	  __currentRep = rep._lastRepetition;
	  line = __currentRep._numInstances;
	  if(__match(rep._next))
	    return true;
	  __currentRep._numInstances = line;
	  __currentRep = rep;

	  if(arg >= rep._max)
	    return false;

	  __inputOffset = input;
	  rep._numInstances = arg;
	  rep._lastLocation = input;

	  if(__match(rep._scan))
	    return true;

	  rep._numInstances = arg - 1;
	  return false;
	}

	if(arg < rep._max) {
	  __pushState(rep._parenFloor);
	  rep._numInstances = arg;
	  rep._lastLocation = input;
	  if(__match(rep._scan))
	    return true;
	  __popState();
	  __inputOffset = input;
	}

	__currentRep = rep._lastRepetition;
	line = __currentRep._numInstances;
	if(__match(rep._next))
	  return true;

	rep._numInstances = line;
	__currentRep = rep;
	rep._numInstances = arg - 1;
	return false;

      case OpCode._BRANCH:
	if(__program[next] != OpCode._BRANCH)
	  next = OpCode._getNextOperator(scan);
	else {
	  int lastParen;

	  lastParen = __lastParen;

	  do {

	    __inputOffset = input;

	    if(__match(OpCode._getNextOperator(scan)))
	      return true;

	    for(arg = __lastParen; arg > lastParen; --arg)
	      //__endMatchOffsets[arg] = 0;
	      __endMatchOffsets[arg] = OpCode._NULL_OFFSET;
	    __lastParen = arg;

	    scan = OpCode._getNext(__program, scan);
	  } while(scan != OpCode._NULL_OFFSET &&
		  __program[scan] == OpCode._BRANCH);
	  return false;
	}

	break;

      case OpCode._MINMOD:
	minMod = true;
	break;


      case OpCode._CURLY:
      case OpCode._STAR:
      case OpCode._PLUS:
	if(op == OpCode._CURLY) {
	  line = OpCode._getArg1(__program, scan);
	  arg  = OpCode._getArg2(__program, scan);
	  scan = OpCode._getNextOperator(scan) + 2;
	} else if(op == OpCode._STAR) {
	  line = 0;
	  arg  = Character.MAX_VALUE;
	  scan = OpCode._getNextOperator(scan);
	} else {
	  line = 1;
	  arg  = Character.MAX_VALUE;
	  scan = OpCode._getNextOperator(scan);
	}

	if(__program[next] == OpCode._EXACTLY) {
	  nextChar = __program[OpCode._getOperand(next) + 1];
	  current  = 0;
	} else {
	  nextChar = __EOS;
	  current  = -1000;
	}
	__inputOffset = input;

	if(minMod) {
	  minMod = false;

	  if(line > 0 && __repeat(scan, line) < line)
	    return false;


	  while(arg >= line || (arg == Character.MAX_VALUE && line > 0)) {
	    // there may be a bug here with respect to
	    // __inputOffset >= __input.length, but it seems to be right for
	    // now.  the issue is with __inputOffset being reset later.
	    // is this test really supposed to happen here?
	    if(current == -1000 || __inputOffset >= __endOffset ||
	       __input[__inputOffset] == nextChar) {
	      if(__match(next))
		return true;
	    }

	    __inputOffset = input + line;

	    if(__repeat(scan, 1) != 0) {
	      ++line;
	      __inputOffset = input + line;
	    } else
	      return false;
	  }

	} else {
	  arg = __repeat(scan, arg);

	  if(line < arg && OpCode._opType[__program[next]] == OpCode._EOL &&
	     (!__multiline || __program[next] == OpCode._SEOL))
	    line = arg;

	  while(arg >= line) {
	    // there may be a bug here with respect to
	    // __inputOffset >= __input.length, but it seems to be right for
	    // now.  the issue is with __inputOffset being reset later.
	    // is this test really supposed to happen here?
	    if(current == -1000 || __inputOffset >= __endOffset ||
	       __input[__inputOffset] == nextChar) {
	      if(__match(next))
		return true;
	    }

	    --arg;
	    __inputOffset = input + arg;
	  }
	}

	return false;

      case OpCode._SUCCEED:
      case OpCode._END:
	__inputOffset = input;
	// This enforces the rule that two consecutive matches cannot have
	// the same end offset.
	if(__inputOffset == __lastMatchInputEndOffset)
	  return false;
	return true;

      case OpCode._IFMATCH:
	__inputOffset = input;
	scan = OpCode._getNextOperator(scan);
	if(!__match(scan))
	  return false;
	break;

      case OpCode._UNLESSM:
	__inputOffset = input;
	scan = OpCode._getNextOperator(scan);
	if(__match(scan))
	  return false;
	break;


      default:
	// todo: Need to throw an exception here.

      } // end switch

      //scan = (next > 0 ? next : 0);
      scan = next;
    } // end while scan



    return false;
  }
  private void __popState() {
    int[] state;
    int entry, paren;

    state = (int[])__stack.pop();

    __expSize     = state[0];
    __lastParen   = state[1];
    __inputOffset = state[2];

    for(entry = 3; entry < state.length; entry+=3) {
      paren = state[entry + 2];
      __beginMatchOffsets[paren] = state[entry + 1];

      if(paren <= __lastParen)
	__endMatchOffsets[paren] = state[entry];
    }

    for(paren = __lastParen + 1; paren <= __numParentheses; paren++) {
      if(paren > __expSize)
	__beginMatchOffsets[paren] = OpCode._NULL_OFFSET;
      __endMatchOffsets[paren] = OpCode._NULL_OFFSET;
    }
  }
  private void __pushState(int parenFloor) {
    int[] state;
    int stateEntries, paren;

    stateEntries = 3*(__expSize - parenFloor);
    if(stateEntries <= 0)
      state = new int[3];
    else
      state = new int[stateEntries + 3];

    state[0] = __expSize;
    state[1] = __lastParen;
    state[2] = __inputOffset;

    for(paren = __expSize; paren > parenFloor; paren-=3, stateEntries-=3) {
      state[stateEntries]     = __endMatchOffsets[paren];
      state[stateEntries + 1] = __beginMatchOffsets[paren];
      state[stateEntries + 2] = paren;
    }

    __stack.push(state);
  }
  private int __repeat(int offset, int max) {
    int scan, eol, operand, ret;
    char ch;

    scan = __inputOffset;
    eol  = __eol;

    if(max != Character.MAX_VALUE && max < eol - scan)
      eol = scan + max;

    operand = OpCode._getOperand(offset);

    switch(__program[offset]) {

    case OpCode._ANY:
      while(scan < eol && __input[scan] != '\n')
	++scan;
      break;

    case OpCode._SANY:
      scan = eol;
      break;

    case OpCode._EXACTLY:
      ++operand;
      while(scan < eol && __program[operand] == __input[scan])
	++scan;
      break;

    case OpCode._ANYOF:
      if(scan < eol && (ch = __input[scan]) < 256) {
	while((__program[operand + (ch >> 4)] & (1 << (ch & 0xf))) == 0) {
	  if(++scan < eol)
	    ch = __input[scan];
	  else
	    break;
	}
      }
      break;

    case OpCode._ALNUM:
      while(scan < eol && OpCode._isWordCharacter(__input[scan]))
	++scan;
      break;

    case OpCode._NALNUM:
      while(scan < eol && !OpCode._isWordCharacter(__input[scan]))
	++scan;
      break;

    case OpCode._SPACE:
      while(scan < eol && Character.isWhitespace(__input[scan]))
	++scan;
      break;

    case OpCode._NSPACE:
      while(scan < eol && !Character.isWhitespace(__input[scan]))
	++scan;
      break;

    case OpCode._DIGIT:
      while(scan < eol && Character.isDigit(__input[scan]))
	++scan;
      break;

    case OpCode._NDIGIT:
      while(scan < eol && !Character.isDigit(__input[scan]))
	++scan;
      break;

    default:
      break;

    }

    ret = scan - __inputOffset;
    __inputOffset = scan;

    return ret;
  }
  // Set the match result information.  Only call this if we successfully
  // matched.
  private void __setLastMatchResult() {
    int offs;

    //endOffset+=dontTry;

    __lastMatchResult = new Perl5MatchResult(__numParentheses + 1);

    // This can happen when using Perl5StreamInput
    if(__endMatchOffsets[0] > __originalInput.length)
      throw new ArrayIndexOutOfBoundsException();

    __lastMatchResult._match =
      new String(__originalInput, __beginMatchOffsets[0],
		 __endMatchOffsets[0] - __beginMatchOffsets[0]);

    __lastMatchResult._matchBeginOffset = __beginMatchOffsets[0];

    while(__numParentheses >= 0) {
      offs = __beginMatchOffsets[__numParentheses];

      if(offs >= 0)
	__lastMatchResult._beginGroupOffset[__numParentheses]
	  = offs - __lastMatchResult._matchBeginOffset;
      else
	__lastMatchResult._beginGroupOffset[__numParentheses] =
	  OpCode._NULL_OFFSET;

      offs = __endMatchOffsets[__numParentheses];

      if(offs >= 0)
	__lastMatchResult._endGroupOffset[__numParentheses]
	  = offs - __lastMatchResult._matchBeginOffset;
      else
	__lastMatchResult._endGroupOffset[__numParentheses] =
	  OpCode._NULL_OFFSET;

      --__numParentheses;
    }

    // Free up for garbage collection
    __originalInput = null;
  }
  private boolean __tryExpression(Perl5Pattern expression, int offset) {
    int count;

    __inputOffset = offset;
    __lastParen   = 0;
    __expSize     = 0;

    if(__numParentheses > 0) {
      for(count=0; count <= __numParentheses; count++) {
	__beginMatchOffsets[count] = OpCode._NULL_OFFSET;
	__endMatchOffsets[count]   = OpCode._NULL_OFFSET;
      }
    }

    if(__match(1)){
      __beginMatchOffsets[0] = offset;
      __endMatchOffsets[0]   = __inputOffset;
      return true;
    }

    return false;
  }
  char[] _toLower(char[] input) {
    int current;
    char[] inp;
    // todo:
    // Certainly not the best way to do case insensitive matching.
    // Must definitely change this in some way, but for now we
    // do what Perl does and make a copy of the input, converting
    // it all to lowercase.  This is truly better handled in the
    // compilation phase.
    inp = new char[input.length];
    System.arraycopy(input, 0, inp, 0, input.length);
    input = inp;

    // todo: Need to inline toLowerCase()
    for(current = 0; current < input.length; current++)
      if(Character.isUpperCase(input[current]))
	input[current] = Character.toLowerCase(input[current]);

    return input;
  }
  /**
   * Determines if a string (represented as a char[]) contains a pattern.
   * If the pattern is
   * matched by some substring of the input, a MatchResult instance
   * representing the <b> first </b> such match is made acessible via 
   * {@link #getMatch()}.  If you want to access
   * subsequent matches you should either use a PatternMatcherInput object
   * or use the offset information in the MatchResult to create a substring
   * representing the remaining input.  Using the MatchResult offset 
   * information is the recommended method of obtaining the parts of the
   * string preceeding the match and following the match.
   * <p>
   * The pattern must be a Perl5Pattern instance, otherwise a
   * ClassCastException will be thrown.  You are not required to, and 
   * indeed should NOT try to (for performance reasons), catch a
   * ClassCastException because it will never be thrown as long as you use
   * a Perl5Pattern 