//=============================================================================
//
// Adventure Game Studio (AGS)
//
// Copyright (C) 1999-2011 Chris Jones and 2011-20xx others
// The full list of copyright holders can be found in the Copyright.txt
// file, which is part of this source code distribution.
//
// The AGS source code is provided under the Artistic License 2.0.
// A copy of this license can be found in the file License.txt and at
// http://www.opensource.org/licenses/artistic-license-2.0.php
//
//=============================================================================
//
// Managed script object wrapping std::map<String, String> and
// unordered_map<String, String>.
//
// TODO: support wrapping non-owned Dictionary, passed by the reference, -
// that would let expose internal engine's dicts using same interface.
// TODO: maybe optimize key lookup operations further by not creating a String
// object from const char*. It seems, C++14 standard allows to use convertible
// types as keys; need to research what perfomance impact that would make.
//
//=============================================================================
#ifndef __AC_SCRIPTDICT_H
#define __AC_SCRIPTDICT_H

#include <map>
#include <unordered_map>
#include <string.h>
#include "ac/dynobj/cc_agsdynamicobject.h"
#include "util/string.h"
#include "util/string_types.h"

using namespace AGS::Common;

class ScriptDictBase : public AGSCCDynamicObject
{
public:
    int Dispose(const char *address, bool force) override;
    const char *GetType() override;
    int Serialize(const char *address, char *buffer, int bufsize) override;
    void Unserialize(int index, const char *serializedData, int dataSize) override;

    virtual bool IsCaseSensitive() const = 0;
    virtual bool IsSorted() const = 0;

    virtual void Clear() = 0;
    virtual bool Contains(const char *key) = 0;
    virtual const char *Get(const char *key) = 0;
    virtual bool Remove(const char *key) = 0;
    virtual bool Set(const char *key, const char *value) = 0;
    virtual int GetItemCount() = 0;
    virtual void GetKeys(std::vector<const char*> &buf) const = 0;
    virtual void GetValues(std::vector<const char*> &buf) const = 0;

private:
    virtual size_t CalcSerializeSize() = 0;
    virtual void SerializeContainer() = 0;
    virtual void UnserializeContainer(const char *serializedData) = 0;
};

template <typename TDict, bool is_sorted, bool is_casesensitive>
class ScriptDictImpl final : public ScriptDictBase
{
public:
    typedef typename TDict::const_iterator ConstIterator;

    ScriptDictImpl() = default;

    bool IsCaseSensitive() const override { return is_casesensitive; }
    bool IsSorted() const override { return is_sorted; }

    void Clear() override
    {
        for (auto it = _dic.begin(); it != _dic.end(); ++it)
            DeleteItem(it);
        _dic.clear();
    }
    bool Contains(const char *key) override { return _dic.count(String::Wrapper(key)) != 0; }
    const char *Get(const char *key) override
    {
        auto it = _dic.find(String::Wrapper(key));
        if (it == _dic.end()) return nullptr;
        return it->second.GetCStr();
    }
    bool Remove(const char *key) override
    {
        auto it = _dic.find(String::Wrapper(key));
        if (it == _dic.end()) return false;
        DeleteItem(it);
        _dic.erase(it);
        return true;
    }
    bool Set(const char *key, const char *value) override
    {
        if (!key) return false;
        if (!value)
        { // remove keys with null value
            Remove(key);
            return true;
        }
        size_t key_len = strlen(key);
        size_t value_len = strlen(value);
        return TryAddItem(key, key_len, value, value_len);
    }
    int GetItemCount() override { return _dic.size(); }
    void GetKeys(std::vector<const char*> &buf) const override
    {
        for (auto it = _dic.begin(); it != _dic.end(); ++it)
            buf.push_back(it->first.GetCStr());
    }
    void GetValues(std::vector<const char*> &buf) const override
    {
        for (auto it = _dic.begin(); it != _dic.end(); ++it)
            buf.push_back(it->second.GetCStr());
    }

private:
    bool TryAddItem(const char *key, size_t key_len, const char *value, size_t value_len)
    {
        String elem_key(key, key_len);
        String elem_value;
        elem_value.SetString(value, value_len);
        _dic[elem_key] = elem_value;
        return true;
    }
    void DeleteItem(ConstIterator it) { /* do nothing */ }

    size_t CalcSerializeSize() override
    {
        size_t total_sz = sizeof(int32_t);
        for (auto it = _dic.begin(); it != _dic.end(); ++it)
        {
            total_sz += sizeof(int32_t) + it->first.GetLength();
            total_sz += sizeof(int32_t) + it->second.GetLength();
        }
        return total_sz;
    }

    void SerializeContainer() override
    {
        SerializeInt((int)_dic.size());
        for (auto it = _dic.begin(); it != _dic.end(); ++it)
        {
            SerializeInt((int)it->first.GetLength());
            memcpy(&serbuffer[bytesSoFar], it->first.GetCStr(), it->first.GetLength());
            bytesSoFar += it->first.GetLength();
            SerializeInt((int)it->second.GetLength());
            memcpy(&serbuffer[bytesSoFar], it->second.GetCStr(), it->second.GetLength());
            bytesSoFar += it->second.GetLength();
        }
    }

    void UnserializeContainer(const char *serializedData) override
    {
        size_t item_count = (size_t)UnserializeInt();
        for (size_t i = 0; i < item_count; ++i)
        {
            size_t key_len = UnserializeInt();
            int key_pos = bytesSoFar; bytesSoFar += key_len;
            size_t value_len = UnserializeInt();
            if (value_len != (size_t)-1) // do not restore keys with null value (old format)
            {
                int value_pos = bytesSoFar; bytesSoFar += value_len;
                TryAddItem(&serializedData[key_pos], key_len, &serializedData[value_pos], value_len);
            }
        }
    }

    TDict _dic;
};

typedef ScriptDictImpl< std::map<String, String>, true, true > ScriptDict;
typedef ScriptDictImpl< std::map<String, String, StrLessNoCase>, true, false > ScriptDictCI;
typedef ScriptDictImpl< std::unordered_map<String, String>, false, true > ScriptHashDict;
typedef ScriptDictImpl< std::unordered_map<String, String, HashStrNoCase, StrEqNoCase>, false, false > ScriptHashDictCI;

#endif // __AC_SCRIPTDICT_H
