/*
 *  demorse: An application to decode Morse signals to text
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef COMMON_H
#define COMMON_H	1

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>

/************ CUSTOMIZE THESE DEFINITIONS FOR YOUR NEEDS *************/

#define SOUND_CARD           "hw:1" /* Your dsp device */
#define CAPTURE_SRC          "Line" /* Signal recording source */
#define CAPTURE_VOL			 "Line" /* Input level control device */
#define CAPTURE_LEV              50 /* Default level of line and igain */
#define NUM_CHANNELS              2 /* Stereo/Mono: 1=Mono, 2=Stereo   */
#define LEFT_CHAN				  0
#define RIGHT_CHAN				  1
#define USE_CHANNEL       LEFT_CHAN /* Channel for capture LEFT|RIGHT */

/* Definitions for Morse code conversion to ascii */
#define NUMBER_OF_CHAR	55 /* Number of chars in ASCII table */

/* Initial receiving speed in w.p.m */
#define INITIAL_WPM		20

/* Definitions for word wrap control */
#define LINE_LENGTH		80 /* Maximum line length if word wrap enabled */

/***************** END OF CUSTOMIZABLE SECTION ***********************/

/* General definitions for signal processing */
#define DSP_RATE              48000 /* dsp sampling speed in samples/sec */
#define PERIOD_SIZE			   2048 /* Period size of dsp in frames */
#define NUM_PERIODS				  2 /* Number of periods */

/* Signal Detector parameter definitions */
#define AUDIO_FREQUENCY         500 /* Nominal detector frequency (Hz)  */
#define CYCLES_PER_FRAG			  2	/* Cycles of signal/signal fragment  */
#define MAX_UNIT_LEN             30 /* Max length of Morse unit, fragments */
#define MIN_UNIT_LEN             10 /* Max length of Morse unit, fragments */
#define STEP_THRESHOLD			105 /* % diff between successive sig levels */
#define DET_THRESHOLD			 60 /* Mark/Space detector threshold */

/* Definitions of flags for various actions */
#define MARK_TONE				0x0080 /* Input signal is mark tone (key down)*/
#define ADAPT_SPEED				0x0100 /* Enable speed tracking */
#define WORD_WRAP				0x0200 /* Enable word wrapping  */
#define DSP_IO_ERROR			0x0400 /* Error reading from dsp */
#define SETUP_LEVEL				0x0800 /* Enable audio level setup */

/* Return values */
#define ERROR			1	 /* Error condition */
#define SUCCESS			0	 /* No error condition */

/*-------------------------------------------------------------------*/

typedef struct
{
  char
	snd_card[32], /* Sound card name */
	cap_src[32],  /* Capture source  */
	cap_vol[32];  /* Capture volume  */

  int
	channel,	/* ALSA "channel" for use in mixer setup */
	num_chn,	/* Number of audio channels (2=stereo, 1=mono) */
	cap_lev,	/* Recording/Capture level */
	dsp_rate,	/* DSP rate (speed) samples/sec */
	spd_wpm,	/* Morse speed WPM */
	wrd_wrp,	/* Maximum line length if word wrap enabled   */
	det_thr,	/* Signal threshold for Mark/Space separation */
	unit_elem,	/* Length of a Morse unit element (a dit) */
	init_perd;	/* Initial Morse speed at start up */

  /* Signal/dsp samples buffer */
  short *buffer;
  int
	buffer_idx,	 /* Index to signal samples buffer */
	buffer_size; /* Buffer size according to stereo/mono mode */

} rc_data_t;

/*-------------------------------------------------------------------*/

/* Function prototypes produced by cproto */
/* decode.c */
char Get_Character(void);
void Adapt_Decoder(void);
char Hex_to_Ascii(int *hex_code);
/* demorse.c */
int main(int argc, char *argv[]);
void Usage(void);
int isFlagSet(int flag);
int isFlagClear(int flag);
void Set_Flag(int flag);
void Clear_Flag(int flag);
void Toggle_Flag(int flag);
/* detect.c */
int Get_Fragment(void);
/* sound.c */
void Setup_Sound_Card(void);
void Setup_Signal_Input(void);
void Close_Mixer_Handle(void);
int Xrun_Recovery(int error);
char Get_Signal_Sample(void);

#endif

