/*  detect.c
 *
 *  Signal detection functions of xdemorse application
 */

/*
 *  xdemorse: An application to decode Morse code signals to text
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "detect.h"
#include "shared.h"
#include <math.h>

/*------------------------------------------------------------------------*/

/*  Get_Fragment()
 *
 *  Detects the cw beat frequency signal o/p from the radio
 *  receiver and determines the status, (mark or space) of a
 *  'fragment' (a small fraction, ~1/8) of a Morse code element.
 *  Signal detection is done by using a Goertzel algorithm.
 */

  int
Get_Fragment( void )
{
  static int
	/* Fragment length in DSP samples */
	frag_len		 = (DSP_RATE*CYCLES_PER_FRAG)/AUDIO_FREQUENCY,
	last_block		 = 0, /* Previous value of Goertzel block size  */
	samples_buff_len = (CYCLES_PER_FRAG*MAX_UNIT_LEN*DSP_RATE)/AUDIO_FREQUENCY,
	samples_buff_idx = 0; /* Sample buffer index */

  int
	frag_lev,    /* Level of the Morse signal 'fragment'     */
	frag_timer,  /* Counter for timing duration of fragment  */
	block_size,  /* Block size (N) of the Goertzel algorithm */
	up_steps,	 /* Num of consecutive increasing signal levels */
	dn_steps,	 /* Num of consecutive decreasing signal levels */
	idx;

  /* Circular signal samples buffer for Goertzel detector */
  static char
	samples_buff[(CYCLES_PER_FRAG*MAX_UNIT_LEN*DSP_RATE)/AUDIO_FREQUENCY];

  /* Circular signal level buffer and index for edge detector */
  static int
	sig_level_idx = 0,
	sig_level_buff[ MAX_UNIT_LEN ];

  /* Variables for the Goertzel algorithm */
  static double coeff;
  double q0, q1, q2;


  /* Goertzel block size depends on Morse speed */
  block_size = frag_len * rc_data.unit_elem;

  /* Recalculate Goertzel parameters in block size changes */
  if( block_size != last_block )
  {
	double w;

	w = 2.0 * M_PI * (double)AUDIO_FREQUENCY / (double)DSP_RATE;
	coeff = 2.0 * cos(w);
	last_block = block_size;
  }

  /* Buffer dsp samples of input signal for a fragment */
  for(frag_timer = 0; frag_timer < frag_len; frag_timer++ )
  {
	/* Get next signal sample from buffer, abort on error */
	samples_buff[samples_buff_idx] = Get_Signal_Sample();
	if( isFlagSet(DSP_IO_ERROR) )
	  return(-1);

	/* Advance/Reset circular buffers' index */
	if( ++samples_buff_idx >= samples_buff_len )
	  samples_buff_idx = 0;

  } /* for( frag_timer = 0; frag_timer < frag_len ... */

  /** Calculate signal fragment level over a block **/
  /* Backstep buffer index for use of samples */
  samples_buff_idx -= block_size;
  if( samples_buff_idx < 0 )
	samples_buff_idx += samples_buff_len;

  /* Calculate fragment level using Goertzel algorithm */
  q1 = q2 = 0.0;
  for( idx = 0; idx < block_size; idx++ )
  {
	q0 = coeff * q1 - q2 + (double)samples_buff[samples_buff_idx];
	q2 = q1;
	q1 = q0;

	/* Reset circular buffers' index */
	if( ++samples_buff_idx >= samples_buff_len )
	  samples_buff_idx = 0;
  }

  /* Scalar magnitude of input signal scaled by block size */
  q1 /= (double)block_size;
  q2 /= (double)block_size;
  frag_lev = (int)(q1*q1 + q2*q2 - q1*q2*coeff);

  /* Save signal power level to circular buffer */
  sig_level_buff[sig_level_idx] = frag_lev;
  if( ++sig_level_idx >= MAX_UNIT_LEN )
	sig_level_idx = 0;

  /* Backstep buffer index for use of fragment levels */
  sig_level_idx -= rc_data.unit_elem;
  if( sig_level_idx < 0 )
	sig_level_idx += MAX_UNIT_LEN;

  /* Count the number of "steps" in the signal's edge that are
   * in the same direction (increasing or decreasing amplitude) */
  up_steps = dn_steps = 0;
  for( idx = 1; idx < rc_data.unit_elem; idx++ )
  {
	int tmp1, tmp2;

	/* Compare successive signal levels */
	tmp1 = sig_level_buff[sig_level_idx];
	if( ++sig_level_idx >= MAX_UNIT_LEN )
	  sig_level_idx = 0;
	tmp2 = sig_level_buff[sig_level_idx];

	/* Successive levels are compared for more
	 * than STEP_THRESHOLD difference up or down */
	if( STEP_THRESHOLD*tmp1 < 100*tmp2 )
	  up_steps++;
	else if( 100*tmp1 > STEP_THRESHOLD*tmp2 )
	  dn_steps++;
  }
  if( ++sig_level_idx >= MAX_UNIT_LEN )
	sig_level_idx = 0;

  /* Set tone status. */
  if( 100*up_steps > rc_data.det_thr*(rc_data.unit_elem-1) )
	Set_Flag( MARK_TONE );
  else
	if( 100*dn_steps > rc_data.det_thr*(rc_data.unit_elem-1) )
	  Clear_Flag( MARK_TONE );

  return( frag_lev );

} /* End of Get_Fragment() */

/*------------------------------------------------------------------------*/

