/*  demorse.c
 *
 *  Main body of demorse application
 */

/*
 *  demorse: An application to decode Morse signals to text
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"
#include "shared.h"

/* Signal handler */
static void sig_handler( int signal );

/*------------------------------------------------------------------------*/

  int
main( int argc, char *argv[] )
{
  int
	option,    /* Command line option returned by getopt() */
	line_idx;  /* Number of characters printed on line     */

  /* Decoded Morse character */
  char dec_char;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;

  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  /*** Process command line options ***/

  /* Initialize defaults */
  Set_Flag( ADAPT_SPEED );
  line_idx = rc_data.wrd_wrp = 0;

  strncpy( rc_data.snd_card, SOUND_CARD, 31 ); /* Sound card name (hw:0 etc) */
  strncpy( rc_data.cap_src, CAPTURE_SRC, 31 ); /* Recording/capture source   */
  strncpy( rc_data.cap_vol, CAPTURE_VOL, 31 ); /* Input level contol name    */

  rc_data.num_chn  = NUM_CHANNELS;	/* Number of channels used (mono=1, stereo=2) */
  rc_data.channel  = USE_CHANNEL;	/* Channel in use: 0=left, 1=right */
  rc_data.cap_lev  = CAPTURE_LEV;	/* Recording/Capture level */
  rc_data.spd_wpm  = INITIAL_WPM;	/* Morse speed WPM */
  rc_data.wrd_wrp  = LINE_LENGTH;	/* Maximum line length if word wrap enabled   */
  rc_data.det_thr  = DET_THRESHOLD; /* Signal threshold for Mark/Space separation */
  rc_data.dsp_rate = DSP_RATE;		/* DSP sampling rate */
  /* Initial unit element as fragments/element */
  rc_data.unit_elem = (60*AUDIO_FREQUENCY)/(50*CYCLES_PER_FRAG*INITIAL_WPM);
  /* Initial Morse speed at start up as fragments/element */
  rc_data.init_perd = (60*AUDIO_FREQUENCY)/(50*CYCLES_PER_FRAG*INITIAL_WPM);

  /* Allocate memory to recv samples buffer */
  rc_data.buffer_size = PERIOD_SIZE * rc_data.num_chn;
  rc_data.buffer_idx  = rc_data.buffer_size;
  rc_data.buffer = (short *)malloc(
	  (size_t)rc_data.buffer_size * sizeof(short) );
  if( rc_data.buffer == NULL )
  {
	fprintf( stderr, "demorse: Memory allocation for buffer failed\n" );
	exit( -1 );
  }
  memset( rc_data.buffer, 0,
	  (size_t)rc_data.buffer_size * sizeof(short) );

  /* Process command line options */
  while( (option = getopt(argc, argv, "c:d:f:l:p:r:s:u:w:hiv") ) != -1 )
	switch( option )
	{
	  case 'c': /* Capture level */
		rc_data.cap_lev = atoi( optarg );
		if( (rc_data.cap_lev < 0) || (rc_data.cap_lev > 100) )
		{
		  fprintf( stderr, "demorse: capture level out of range" );
		  Usage();
		  exit( -1 );
		}
		printf("demorse: setting capture level to %d\n", rc_data.cap_lev );
		break;

	  case 'd': /* Detector threshold */
		rc_data.det_thr = atoi( optarg );
		if( (rc_data.det_thr < 0) || (rc_data.det_thr > 100) )
		{
		  fprintf( stderr, "demorse: detector threshold out of range" );
		  Usage();
		  exit( -1 );
		}
		printf("demorse: setting detector threshold to %d\n", rc_data.det_thr );
		break;

	  case 'f': /* Decode at fixed speed */
		rc_data.unit_elem = atoi( optarg );
		Clear_Flag( ADAPT_SPEED );

		/* Speed limit 10-30 wpm */
		if( (rc_data.unit_elem > MAX_UNIT_LEN) ||
			(rc_data.unit_elem < MIN_UNIT_LEN) )
		{
		  fprintf( stderr, "demorse: Morse code speed out of range" );
		  Usage();
		  exit( -1 );
		}

		/* Convert speed to frags per element */
		printf( "demorse: fixing Morse decoding speed to %d wpm\n",
			rc_data.unit_elem );
		rc_data.unit_elem = (60*AUDIO_FREQUENCY)/
		  (50*CYCLES_PER_FRAG*rc_data.unit_elem);
		break;

	  case 'l': /* Recording level device */
		strncpy( rc_data.cap_vol, optarg, 31 );
		printf( "demorse: setting recording level device to %s\n", rc_data.cap_vol );
		break;

	  case 'p': /* Sound card (hw:0 etc) */
		strncpy( rc_data.snd_card, optarg, 31 );
		printf( "demorse: setting dsp device to %s\n", rc_data.snd_card );
		break;

	  case 'r': /* Capture source */
		strncpy( rc_data.cap_src, optarg, 31 );
		printf( "demorse: setting recording source to %s\n", rc_data.cap_src );
		break;

	  case 's': /* Specify Morse code starting speed WPM */
		rc_data.init_perd = atoi( optarg );
		Set_Flag( ADAPT_SPEED );

		/* Speed limit 10-30 wpm */
		if( (rc_data.init_perd > MAX_UNIT_LEN) ||
			(rc_data.init_perd < MIN_UNIT_LEN) )
		{
		  fprintf( stderr, "%s", "demorse: Morse code speed out of range" );
		  Usage();
		  exit( -1 );
		}

		/* Convert speed to frags per element */
		printf( "demorse: setting initial Morse decoding speed to %d wpm\n",
			rc_data.init_perd );
		rc_data.unit_elem = rc_data.init_perd =
		  (60*AUDIO_FREQUENCY)/(50*CYCLES_PER_FRAG*rc_data.init_perd);
		break;

	  case 'u': /* Channel to use left/right */
		if( strcmp(optarg, "left") == 0 )
		  rc_data.channel = LEFT_CHAN;
		else if( strcmp(optarg, "right") == 0 )
		  rc_data.channel = RIGHT_CHAN;
		else
		{
		  fprintf( stderr, "demorse: unrecognized option: %s\n", optarg );
		  Usage();
		  exit( -1 );
		}
		printf( "demorse: setting recording source to %s\n", optarg );
		break;

	  case 'w':	/* Specify word wrap line length */
		rc_data.wrd_wrp = atoi( optarg );

		/* Check word wrap line length */
		if( (rc_data.wrd_wrp > LINE_LENGTH) || (rc_data.wrd_wrp <= 0) )
		{
		  fprintf( stderr, "%s", "demorse: word wrap line length out of range" );
		  Usage();
		  exit( -1 );
		}

		Set_Flag( WORD_WRAP );
		printf( "demorse: setting word wrap to %d columns\n", rc_data.wrd_wrp );
		break;

	  case 'h': /* Print usage and exit */
		Usage();
		exit( 0 );

	  case 'i': /* Display information for signal input setup */
		Set_Flag( SETUP_LEVEL );
		break;

	  case 'v': /* Print version */
		printf( "%s\n", VERSION );
		exit( 0 );

	  default: /* Print usage and exit */
		Usage();
		exit( -1 );

	} /* End of switch( option ) */

  /* Prepare sound card */
  Setup_Sound_Card();

  /* Start audio level setup loop */
  if( isFlagSet(SETUP_LEVEL) )
	Setup_Signal_Input();

  /* Decode and print Morse signals until stopped */
  while( 1 )
  {
	/* If word wrap is enabled, wrap on 'space' char */
	if( isFlagSet(WORD_WRAP) )
	{
	  dec_char = Get_Character();
	  if( (dec_char == ' ') && (line_idx >= rc_data.wrd_wrp) )
	  {
		line_idx = 0;
		printf( "%c", '\n' );
		fflush( stdout );
	  }
	  else
	  {
		line_idx++;
		printf( "%c", dec_char );
		fflush( stdout );
	  } /* if( ((dec_char = Get_Character()) == ' ') && */

	} /* if( isFlagSet(WORD_WRAP) ) */
	else
	{
	  /* Just print chars if no word wrap */
	  printf( "%c", Get_Character() );
	  fflush( stdout );
	}

  } /* while( 1 ) */

} /* End of  main() */

/*------------------------------------------------------------------------*/

/*  Usage()
 *
 *  Prints usage information
 */

  void
Usage( void )
{
  fprintf( stderr, "%s\n\n",
	  "Usage: demorse [-c<capture level>][-d<detector threshold>][-f<fixed speed>]\n"
	  "               [-l<level device>][-m<mixer device>][-p<dsp device>]\n"
	  "               [-r<recording source>][-s<initial speed>][-u<channel>]\n"
	  "               [-w<line length>][-hiv]" );

  fprintf( stderr, "%s\n\n",
	  "       -c<capture level>: Capture/Recording source level.\n"
	  "       Value should be 0-100, default is 50." );

  fprintf( stderr, "%s\n\n",
	  "       -d<detector threshold>: Tone/No-tone detector threshold.\n"
	  "       Value should be 0-100, default is 60." );

  fprintf( stderr, "%s\n\n",
	  "       -f<fixed-speed>: Fixed speed in w.p.m at which\n"
	  "       to expect and to decode a Morse coded signal.\n"
	  "       Valid range is 10-30 words per minute." );

  fprintf( stderr, "%s\n\n",
	  "       -l<level device>: Input level device. Default is \"igain\"." );

  fprintf( stderr, "%s\n\n",
	  "       -p<dsp device>: DSP device. Default is \"/dev/dsp\"." );

  fprintf( stderr, "%s\n\n",
	  "       -r<recording source>: Recording/Capture source. Default is \"line\"." );

  fprintf( stderr, "%s\n\n",
	  "       -s<initial speed>: Initial speed in w.p.m at start-up,\n"
	  "       at which to expect and to decode a Morse code signal.\n"
	  "       Actual decoding speed then adapts to the incoming Morse\n"
	  "       signal, limited to the range 10-30 words per minute." );

  fprintf( stderr, "%s\n\n",
	  "       -u<channel>: Channel to use. Value should be left|right.\n"
	  "       Default is \"right\"." );

  fprintf( stderr, "%s\n\n",
	  "       -w<line length>: Line length in columns beyond which\n"
	  "       word wrapping is performed. Maximum is LINE_LENGTH columns." );

  fprintf( stderr, "%s\n\n",
	  "       -h: Print this usage information and exit.");

  fprintf( stderr, "%s\n\n",
	  "       -i: Display information for setting up signal input.");

  fprintf( stderr, "%s\n\n",
	  "       -v: Print version number and exit.");

} /* End of Usage() */

/*------------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  /* Wrap up and quit */
  if( dsp_fd > 0 ) close( dsp_fd );
  if( mix_fd > 0 ) close( mix_fd );

  fprintf( stderr, "\n" );
  switch( signal )
  {
	case SIGINT :
	  fprintf( stderr, "%s\n",  "demorse: Exiting via User Interrupt" );
	  exit( signal );

	case SIGSEGV :
	  fprintf( stderr, "%s\n",  "demorse: Segmentation Fault" );
	  exit( signal );

	case SIGFPE :
	  fprintf( stderr, "%s\n",  "demorse: Floating Point Exception" );
	  exit( signal );

	case SIGABRT :
	  fprintf( stderr, "%s\n",  "demorse: Abort Signal received" );
	  exit( signal );

	case SIGTERM :
	  fprintf( stderr, "%s\n",  "demorse: Termination Request received" );
	  exit( signal );
  }

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flow control flags
 *
 *  See demorse.h for definition of flow control flags
 */

/* An int variable holding the single-bit flags */
static int Flags = 0;

  int
isFlagSet( int flag )
{
  return( (Flags & flag) == flag );
}

  int
isFlagClear( int flag )
{
  return( (~Flags & flag) == flag );
}

  void
Set_Flag( int flag )
{
  Flags |= flag;
}

  void
Clear_Flag( int flag )
{
  Flags &= ~flag;
}

  void
Toggle_Flag( int flag )
{
  Flags ^= flag;
}

/*------------------------------------------------------------------------*/
