/*  sound.c
 *
 *  Soundcard handling functions of xdemorse application
 */

/*
 *  xdemorse: An application to decode and display
 *  Morse code signals using a computer's sound card
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "sound.h"
#include "shared.h"

/* ALSA pcm capture and mixer handles */
static snd_pcm_t *capture_handle = NULL;
static snd_mixer_t *mixer_handle = NULL;
static snd_pcm_hw_params_t *hw_params = NULL;

/*------------------------------------------------------------------------*/

/*  Setup_Sound_Card()
 *
 *  Sets up mixer and DSP devices
 */
  void
Setup_Sound_Card( void )
{
  snd_mixer_elem_t *elem;
  snd_mixer_selem_id_t *sid;
  long cmin, cmax;
  char mesg[128];
  int error;


  /*** Set up pcm parameters ***/
  /* Open pcm for capture */
  error = snd_pcm_open(
	  &capture_handle, rc_data.snd_card, SND_PCM_STREAM_CAPTURE, 0 );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot open audio device %s", rc_data.snd_card );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Allocate memory to hardware parameters structure */
  error = snd_pcm_hw_params_malloc( &hw_params );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot allocate hardware parameter struct" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Initialize hardware parameter structure */
  error = snd_pcm_hw_params_any( capture_handle, hw_params );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot initialize hardware parameter struct" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Set access type */
  error = snd_pcm_hw_params_set_access(
	  capture_handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot set access type (RW_INTERLEAVED)" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Set sample format */
  error = snd_pcm_hw_params_set_format(
	  capture_handle, hw_params, SND_PCM_FORMAT_S16_LE );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot set sample format (S16_LE)" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Set sample rate */
  error = snd_pcm_hw_params_set_rate(
	  capture_handle, hw_params, (unsigned int )rc_data.dsp_rate, 0 );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot set sample rate to %d", rc_data.dsp_rate );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Set channel count */
  error = snd_pcm_hw_params_set_channels(
	  capture_handle, hw_params, (unsigned int )rc_data.num_chn );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot set channel count to %d", rc_data.num_chn );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Set number of periods */
  error = snd_pcm_hw_params_set_periods(
	  capture_handle, hw_params, NUM_PERIODS, 0 );
  if( error < 0)
  {
	snprintf( mesg, 127,
		"demorse: Cannot set number periods" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Set period size */
  error = snd_pcm_hw_params_set_period_size(
	  capture_handle, hw_params, PERIOD_SIZE, 0 );
  if( error < 0)
  {
	snprintf( mesg, 127,
		"demorse: Cannot set period size" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Set parameters */
  error = snd_pcm_hw_params( capture_handle, hw_params );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot set capture parameters" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }
  snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  /*** Set up mixer (capture source and volume) ***/
  /* Open mixer */
  error = snd_mixer_open( &mixer_handle, 0 );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot open mixer" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Attach mixer */
  error = snd_mixer_attach( mixer_handle, rc_data.snd_card );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot attach mixer to %s", rc_data.snd_card );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Register mixer */
  error = snd_mixer_selem_register( mixer_handle, NULL, NULL );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot register mixer" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Load mixer */
  error = snd_mixer_load( mixer_handle );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot load mixer" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Allocate selem_id structure */
  error = snd_mixer_selem_id_malloc( &sid );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot allocate selem_id struct" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  /* Find capture selem */
  snd_mixer_selem_id_set_index( sid, 0 );
  snd_mixer_selem_id_set_name( sid, rc_data.cap_src );
  elem = snd_mixer_find_selem( mixer_handle, sid );
  if( !elem )
  {
	snprintf( mesg, 127,
		"demorse: Cannot find capture element %s", rc_data.cap_src );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	snd_mixer_selem_id_free(sid);
	exit( -1 );
  }

  /* Set capture switch for cap source */
  if( snd_mixer_selem_has_capture_switch(elem) )
  {
	error = snd_mixer_selem_set_capture_switch(
		elem, rc_data.channel, 1 );
	if( error < 0 )
	{
	  snprintf( mesg, 127,
		  "demorse: Cannot set capture source switch %s", rc_data.cap_src );
	  fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	  snd_mixer_selem_id_free(sid);
	  exit( -1 );
	}
  }

  /* Find capture volume selem */
  if( strcmp(rc_data.cap_vol, "--") != 0 )
  {
	snd_mixer_selem_id_set_index( sid, 0 );
	snd_mixer_selem_id_set_name( sid, rc_data.cap_vol );
	elem = snd_mixer_find_selem( mixer_handle, sid );
	if( !elem )
	{
	  snprintf( mesg, 127,
		  "demorse: Cannot find volume element %s", rc_data.cap_vol );
	  fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	  exit( -1 );
	}

	/* Set capture volume */
	if( elem && snd_mixer_selem_has_capture_volume(elem) )
	{
	  /* Change from % volume to sound card value */
	  long lev;
	  snd_mixer_selem_get_capture_volume_range( elem, &cmin, &cmax );
	  lev = cmin + ((cmax - cmin) * rc_data.cap_lev) / 100;

	  /* Set capture volume */
	  error = snd_mixer_selem_set_capture_volume(
		  elem, rc_data.channel, lev );
	  if( error < 0 )
	  {
		snprintf( mesg, 127,
			"demorse: Cannot set capture volume to %d", (int)lev );
		fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
		exit( -1 );
	  }
	}
  } /* if( strcmp(rc_data.cap_vol, "--") != 0 ) */

  snd_mixer_selem_id_free(sid);
  Close_Mixer_Handle();

  /* Prepare audio interface for use */
  error = snd_pcm_prepare( capture_handle );
  if( error < 0 )
  {
	snprintf( mesg, 127,
		"demorse: Cannot prepare audio interface" );
	fprintf( stderr, "%s\n%s\n", mesg, snd_strerror(error) );
	exit( -1 );
  }

  return;
} /* End of Setup_Sound_Card() */

/*------------------------------------------------------------------------*/

/*
 * Close the sound card handles
 */
  void
Close_Mixer_Handle( void )
{
  if( mixer_handle != NULL )
	snd_mixer_close( mixer_handle );
  mixer_handle = NULL;
}

/*------------------------------------------------------------------------*/

/*  Get_Signal_Sample()
 *
 *  Gets the next DSP sample of the signal input
 */

  char
Get_Signal_Sample( void )
{
  snd_pcm_sframes_t error;
  char sample;


  /* Refill dsp samples buffer when needed */
  if( rc_data.buffer_idx >= rc_data.buffer_size )
  {
	/* Start buffer index according to channel */
	rc_data.buffer_idx = rc_data.channel;

	/* Read audio samples from DSP, abort on error */
	error = snd_pcm_readi(
		capture_handle, rc_data.buffer, PERIOD_SIZE );
	if( error != PERIOD_SIZE )
	{
	  char mesg[128];
	  snprintf( mesg, 127,
		  "Read from audio interface failed\n"
		  "Error: %s\n", snd_strerror((int)error) );
	  fprintf( stderr, "%s", mesg );

	  /* Try to recover if broken pipe or suspend */
	  if( !Xrun_Recovery((int)error) )
	  {
		return( 0 );
	  }
	}
  } /* End of if( buffer_idx >= buffer_size ) */

  /* Get a sample from the buffer in int form, reduce to 8 bits.
   * The slope detector gives false results if 16-bit sampling is used */
  sample = (char)(rc_data.buffer[rc_data.buffer_idx] / 256);

  /* Increment according to mono/stereo mode */
  rc_data.buffer_idx += rc_data.num_chn;

  return( sample );
} /* Get_Signal_Sample() */

/*------------------------------------------------------------------------*/

/* Xrun_Recovery()
 *
 * Recover from underrun (broken pipe) and suspend
 */
  int
Xrun_Recovery( int error )
{
  if( error == -EPIPE )
  {
	error = snd_pcm_prepare( capture_handle );
	if( error < 0 )
	{
	  char mesg[128];
	  snprintf( mesg, 127,
		  "Cannot recover from underrun, prepare failed\n"
		  "Error: %s\n", snd_strerror(error) );
	  return( 0 );
	}
  }
  else if( error == -ESTRPIPE )
  {
	while( (error = snd_pcm_resume(capture_handle)) == -EAGAIN )
	  sleep(1);
	if( error < 0 )
	{
	  error = snd_pcm_prepare( capture_handle );
	  if( error < 0 )
	  {
		char mesg[128];
		snprintf( mesg, 127,
			"Cannot recover from suspend, prepare failed\n"
			"Error: %s\n", snd_strerror(error) );
		return( 0 );
	  }
	}
  }

  return( 1 );
} /* Xrun_Recovery() */

/*------------------------------------------------------------------------*/

/*  Setup_Signal_Input()
 *
 *  Displays information needed for the setup of
 *  the Morse sound signals fed to the sound card
 */

  void
Setup_Signal_Input( void )
{
  int
	/* Size of samples buffer depends on stereo/mono mode */
	buffer_size = rc_data.num_chn * PERIOD_SIZE * (int)sizeof( short ),
	sample_lev,  /* Normalized DSP sound sample level */
	sample_max,  /* Max value of DSP sound samples    */
	idx;         /* Index for loops etc */

  /* Signal/DSP samples buffer */
  short buffer[buffer_size];

  /* Buffer for graph plotting */
  char graph[ 46 ];
  snd_pcm_sframes_t error;

  while( 1 )
  {
	sample_max = 0;

	/* Read audio samples from DSP, abort on error */
	error = snd_pcm_readi(
		capture_handle, buffer, PERIOD_SIZE );
	if( error != PERIOD_SIZE )
	{
	  char mesg[128];
	  snprintf( mesg, 127,
		  "demorse: Read from DSP failed\n"
		  "Error: %s\n", snd_strerror((int)error) );
	  fprintf( stderr, "%s", mesg );
	}

	/*** Find max sample level (average of 4 samples) ***/
	/* Start buffer index according to stereo/mono mode */
	if( rc_data.num_chn == 1 ) /* Mono */
	  idx = 0;
	else
	  idx = rc_data.channel;

	for( ; idx < PERIOD_SIZE; idx += rc_data.num_chn )
	{
	  sample_lev = buffer[idx]/256;
	  if( sample_max < sample_lev )
		sample_max = ( (sample_max * 3) + sample_lev ) / 4;
	}
	sample_max += 2;

	/* Make a graph of detector o/p level */
	for( idx = 0; idx < sample_max / 3; idx++ )
	  graph[ idx ] = '=';
	graph[ idx++ ] = '>';
	for( ; idx < 44; idx++ )
	  graph[ idx ] = '.';
	graph[ idx ] = '<';
	graph[ 45 ]  = '\0';

	fprintf( stderr, "%03d %s\n", sample_max, graph );

  } /* while( 1 ) */

} /* End of Setup_Signal_Input() */

/*------------------------------------------------------------------------*/
