// -*- c++ -*-

/*
 *  Author: Arvin Schnell
 */


#ifndef Trans_h
#define Trans_h


/*
 *  Classes to preform transformations between two coordinate systems.
 */


template <typename TA, typename TB>
class Trans1D
{

public:

    Trans1D ();

    void setA (TA, TA);
    void setB (TB, TB);

    void AtoB (TA, TB &) const;
    void BtoA (TB, TA &) const;

    TB AtoB (TA) const;
    TA BtoA (TB) const;

    bool inA (TA) const;
    bool inB (TB) const;

private:

    void calc ();

    TA A1, A2;
    TB B1, B2;

    double Am, Ac;
    double Bm, Bc;

    bool flagA, flagB;

};


template <typename TA, typename TB>
class Trans2D
{

public:

    Trans2D ();

    void setA (TA, TA, TA, TA);
    void setB (TB, TB, TB, TB);

    void AtoB (TA, TA, TB &, TB &) const;
    void BtoA (TB, TB, TA &, TA &) const;

    bool inA (TA, TA) const;
    bool inB (TB, TB) const;

    Trans1D <TA, TB> X, Y;

};


template <typename TA, typename TB>
Trans1D <TA, TB>::Trans1D ()
    : flagA (false),
      flagB (false)
{
}


template <typename TA, typename TB> void
Trans1D <TA, TB>::setA (TA A1, TA A2)
{
    Trans1D::A1 = A1;
    Trans1D::A2 = A2;

    flagA = true;
    if (flagB)
	calc ();
}


template <typename TA, typename TB> void
Trans1D <TA, TB>::setB (TB B1, TB B2)
{
    Trans1D::B1 = B1;
    Trans1D::B2 = B2;

    flagB = true;
    if (flagA)
	calc ();
}


template <typename TA, typename TB> inline void
Trans1D <TA, TB>::AtoB (TA A, TB & B) const
{
    B = (TB) (Bm * A + Bc);
}


template <typename TA, typename TB> inline TB
Trans1D <TA, TB>::AtoB (TA A) const
{
    return (TB) (Bm * A + Bc);
}


template <typename TA, typename TB> inline void
Trans1D <TA, TB>::BtoA (TB B, TA & A) const
{
    A = (TA) (Am * B + Ac);
}


template <typename TA, typename TB> inline TA
Trans1D <TA, TB>::BtoA (TB B) const
{
    return (TA) (Am * B + Ac);
}


template <typename TA, typename TB> bool
Trans1D <TA, TB>::inA (TA A) const
{
    return A >= (A1 < A2 ? A1 : A2) && A <= (A1 > A2 ? A1 : A2);
}


template <typename TA, typename TB> bool
Trans1D <TA, TB>::inB (TB B) const
{
    return B >= (B1 < B2 ? B1 : B2) && B <= (B1 > B2 ? B1 : B2);
}


template <typename TA, typename TB> void
Trans1D<TA, TB>::calc ()
{
    Am = (double) (A2 - A1) / (double) (B2 - B1);
    Ac = A1 - B1 * Am;

    Bm = (double) (B2 - B1) / (double) (A2 - A1);
    Bc = B1 - A1 * Bm;
}


template <typename TA, typename TB>
Trans2D <TA, TB>::Trans2D ()
{
}


template <typename TA, typename TB> void
Trans2D <TA, TB>::setA (TA A1x, TA A1y, TA A2x, TA A2y)
{
    X.setA (A1x, A2x);
    Y.setA (A1y, A2y);
}


template <typename TA, typename TB> void
Trans2D <TA, TB>::setB (TB B1x, TB B1y, TB B2x, TB B2y)
{
    X.setB (B1x, B2x);
    Y.setB (B1y, B2y);
}


template <typename TA, typename TB> void
Trans2D <TA, TB>::AtoB (TA Ax, TA Ay, TB & Bx, TB & By) const
{
    X.AtoB (Ax, Bx);
    Y.AtoB (Ay, By);
}


template <typename TA, typename TB> void
Trans2D <TA, TB>::BtoA (TB Bx, TB By, TA & Ax, TA & Ay) const
{
    X.BtoA (Bx, Ax);
    Y.BtoA (By, Ay);
}


template <typename TA, typename TB> bool
Trans2D <TA, TB>::inA (TA Ax, TA Ay) const
{
    return X.inA (Ax) && Y.inA (Ay);
}


template <typename TA, typename TB> bool
Trans2D <TA, TB>::inB (TB Bx, TB By) const
{
    return X.inB (Bx) && Y.inB (By);
}


#endif
