/*
 * ===========================
 * VDK Visual Development Kit
 * Version 0.4
 * Revision 0.2
 * October 1998
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef MENU_H
#define MENU_H
#include <vdk/widcontain.h>
#include <vdk/dlist.h>
#include <vdk/vdktypes.h>
#include <vdk/rawpixmap.h>
#include <gdk/gdkkeysyms.h>
class VDKForms;
class VDKMenubar;
class VDKOptionMenu;

/*!
  \class VDKMenu
  \brief Provides a menu items container
  \par PROGRAMMING TIPS
  Making a menu bar should be easier than the "hard way" in GTK+. 
  Here an example:
  \code
  MyForm::Setup()
  {
  // makes a new menu bar
  menubar = new VDKMenubar(this); 
  // makes a menu to be attached as first menu to menu bar
  menu1 = new VDKMenuItem(menubar,"_File..",mini_ofolder_xpm);
  // attach help menu to menu bar (rightmost)
  menu2 = new VDKMenuItem(menubar,"_Help",book_xpm,r_justify);
  // makes menu1 submenu
  VDKMenu *menu = new VDKMenu(this);
  menu11 = new VDKMenuItem(menu,"_Open");
  menu12 = new VDKMenuItem(menu,"_Close");
  menu->Separator();
  menu13 = new VDKMenuItem(menu,"_Quit..");
  // attach menu to menu1
  menu1->Add(menu);
  // add menubar and an empty labelto form
  Add(menubar);
  } 
  \endcode
 */
class VDKMenu: public VDKObjectContainer
{
 protected:
  GtkAccelGroup *accel_group; 
 public:
  /*!
    Constructor
    \param owner
   */
  VDKMenu(VDKForm* owner);
  /*!
    Destructor
  */
  virtual ~VDKMenu();
  /*!
    Provides a menu separator
   */
  void Separator();
  virtual void SetFont(VDKFont* font);
  /*!
    Pops menu
   */
  void Popup(guint button = 0,  guint32 activate_time = 0);
  virtual void Add(VDKObject* wid, int justify = l_justify,
      int expand = true, int fill = true , int padding = false);
  /*!
    Return accelerator group assigned to menu
   */
  GtkAccelGroup *AccelGroup() { return accel_group; }
};

/*!
  \class VDKMenuItem
  \brief Provides a menu item
  \par SIGNALS
  \arg \b activate_signal Received whenever user selects an item
  \par TIP
  Menu item caption line is parsed and an acceleartor is installed if an
  underscore is found. By default installs Alt + underscored key
  \code
  VDKMenuItem *item = new VDKMenuItem(menu,"_Accelerated");
  // alt+a triggers activate_signal
  \endcode
  If accelerated menu item does not have submenus using accelerated key 
  makes activate_signal to be emitted, otherwise the submenu branch will 
  be displayed
  \par EXAMPLES
  In ./testvdk/testvdk.cc
 */
class VDKMenuItem: public VDKObject
{
 protected: 
  VDKObjectSignal s_activated;
  VDKMenu* menu;
  GtkWidget *box,*lbl,*pixmapWidget, *tickWidget;
  GdkPixmap* pixmap,*tickPixmap;
  bool ticked;
  guint accelerator_key;
  guint8 modkey;
public:
  /*!
    Sets/gets menu item marked state
   */
  __rwproperty(VDKMenuItem,bool) Checked;
  /*!
    Sets/gets menu item caption
   */
  __rwproperty(VDKMenuItem,const char*) Caption;
  /*!
    Constructor, makes a menu item to be assigned to a menu
    \param menu owner menu
    \param prompt menu caption
    \param pixmap pixmap data
    \param key accelerator key
    \param modkey accelerator key modifier
    \param accel wheter should contain an accelerator or not

    A pixmap will be packed at start if <pixmap> isn't NULL.
    Leaving <key> to default value makes <label> to be parsed
    for accelerators. For instance label == "_Quit" makes the
    accelerator as alt+q. Setting <key> other then default overrides
    label parsing, setting <modkey> to other than default overrides
    <alt> modifier.
  */
  VDKMenuItem(VDKMenu* menu ,
	      const char* prompt = NULL, 
	      char** pixmap = NULL,
	      guint key = GDK_VoidSymbol,
	      guint8 modkey = GDK_MOD1_MASK,
	      bool accel = true);
  /*!
    Constructor, makes a menu item to be assigned to a menu bar
    \param bar owner menu bar
    \param prompt menu caption
    \param pixmap pixmap data
    \param align label alignement
    \param key accelerator key
    \param modkey accelerator key modifier
    \param accel wheter should contain an accelerator or not
  */
  VDKMenuItem(VDKMenubar* bar, 
	      const char* prompt = NULL, char** pixmap = NULL,
	      int align = l_justify,
	      guint key = GDK_VoidSymbol,
	      guint8 modkey = GDK_MOD1_MASK,
	      bool accel = true);
  /*!
    \internal
  */
  VDKMenuItem(VDKForm* owner, 
	      const char* prompt = NULL, char** pixmap = NULL,
	      int align = l_justify,
	      guint key = GDK_VoidSymbol,
	      guint8 modkey = GDK_MOD1_MASK,
	      bool accel = true);
  /*!
    Destructor
  */
  virtual ~VDKMenuItem();

  void SetCaption(const char* str);
  /*!
    Returns accelerated key
   */
  guint AccKey() { return accelerator_key; }
  /*!
    Returns accelerated key modifier
   */
  guint8 ModKey() { return modkey; }

  const char* GetCaption ();
  /*!
    Adds a submenus to this menu
    \par TIP
    Allows submenus to be attached in cascade
   */
  void Add(VDKMenu* submenu);
  void Tick(bool flag);
  virtual void SetFont(VDKFont* font);
  void SetPixmap(VDKRawPixmap* newpix);
}; 

/*!
  \class VDKMenubar
  \brief Provides a gtkmenubar wrapper
 */
class VDKMenubar: public VDKObjectContainer
{
 protected:
  GtkAccelGroup *accel_group; 
  void SetShadow(int shadow)
    { 
      // gtk_menu_bar_set_shadow_type(GTK_MENU_BAR(widget),(GtkShadowType) shadow); 
      // gtk_object_set(GTK_OBJECT(widget), "shadow_type", shadow, NULL);
    }
public:
  /*!
    Sets/gets menubar shadow type(default shadow_out)
   */
  __rwproperty(VDKMenubar, int) Shadow;
  /*!
    Constructor
    \param owner
  */
  VDKMenubar(VDKForm* owner);
  /*!
    Destructor
   */
  ~VDKMenubar();
  /*!
    Sets menubar font
    \par TIP
    Applying a font to a menubar will spread the font to all
    menus and submenus contained
   */
  virtual void SetFont(VDKFont* font);
  /*!
    Add a menu item to menubar
    \par TIP
    <menu_item> must be a VDKMenuItem or derived class otherwise a
    waring message will be generated and nothing will be added.
   */
  virtual void Add(VDKObject* menu_item, int justify = l_justify,
      int expand = true, int fill = true , int padding = false);
  /*!
    Returns assigned accelerator group 
   */
  GtkAccelGroup *AccelGroup() { return accel_group; }
};

/*!
  Provides a gtkoptionmenu wrapper
 */
class VDKOptionMenu: public VDKObject
{
public:
  VDKOptionMenu(VDKForm* owner);
  ~VDKOptionMenu();
  /*!
    Adds a menu
   */
  void Add(VDKMenu* menu);
  virtual void SetFont(VDKFont* ) {}
};


#endif

