/*
 * ===========================
 * VDK Visual Development Kit
 * Version 2.0.0
 * December 2000
 * ===========================
 *
 * Copyright (C) 1998,199,2000,2001 Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <vdk/vdktextview.h>
#include <vdk/colors.h>
#include <cstdio>
#include <sys/stat.h>
#include <vdk/vdkfont.h>
#include <string.h>
/*
---------------------------------------------
             VDKTextBuffer class
---------------------------------------------
*/
/*
*/
void 
VDKTextBuffer::Ref() 
{ 
  ref++;
}
/*
*/
void 
VDKTextBuffer::Unref() 
{
  ref--;
  if(ref == (unsigned int) 0 )
    delete this;
}
/*
*/
VDKTextBuffer::VDKTextBuffer():ref(0),
    Pointer("Pointer",this,0,&VDKTextBuffer::SetPointer,
        &VDKTextBuffer::GetPointer),
    Column("Column",this,0,&VDKTextBuffer::SetColumn,
            &VDKTextBuffer::GetColumn),
    Line("Line",this,0,&VDKTextBuffer::SetLine,
            &VDKTextBuffer::GetLine),
    Length("Length",this,0,&VDKTextBuffer::GetLength),
    Changed("Changed",this,false,&VDKTextBuffer::SetChanged,
            &VDKTextBuffer::GetChanged)
{
  buffer = gtk_text_buffer_new (NULL);
}

VDKTextBuffer::VDKTextBuffer(char* filename):ref(0),
  Pointer("Pointer",this,0,&VDKTextBuffer::SetPointer,&VDKTextBuffer::GetPointer),
  Column("Column",this,0,&VDKTextBuffer::SetColumn,&VDKTextBuffer::GetColumn),
  Line("Line",this,0,&VDKTextBuffer::SetLine, &VDKTextBuffer::GetLine),
  Length("Length",this,0,&VDKTextBuffer::GetLength),
  Changed("Changed",this,false,&VDKTextBuffer::SetChanged, &VDKTextBuffer::GetChanged)

{
  buffer = gtk_text_buffer_new (NULL);
  LoadFromFile(filename);
}
/*
*/
VDKTextBuffer::~VDKTextBuffer()
{
#if VERBOSE
  printf("\ndebug: deleting textbuffer:%p",this);
  fflush(stdout);
#endif
}

static gboolean
read_loop (GtkTextBuffer *buffer, 
	   const char    *filename,
	   GIOChannel    *io,
	   GError       **error)
{
	GIOStatus status;
	//GtkWidget *widget;
	GtkTextIter end;
	gchar *str = NULL;
	gsize size = 0;
	*error = NULL;
	gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (buffer),
					      &end);
	if ((status=g_io_channel_read_line (io, &str, &size, NULL, error)) == 
      G_IO_STATUS_NORMAL && size)
      {
		  gtk_text_buffer_insert (GTK_TEXT_BUFFER (buffer),
						&end, str, size);
		  g_free(str);
		  return TRUE;
	    }
	else if(!*error && (status=g_io_channel_read_to_end (io, &str, &size, error)) == G_IO_STATUS_NORMAL && size)	
  {
		gtk_text_buffer_insert (GTK_TEXT_BUFFER (buffer),
					&end, str, size);
		g_free(str);
		return TRUE;
	}
	if (status == G_IO_STATUS_EOF && !*error)
		return FALSE;
	if (!*error)
		return FALSE;
		/* because of error in input we clear already loaded text */
	//gtk_text_buffer_set_text(buffer, "", 0);
	return FALSE;
}
/*
*/
GError s_error;
GError* p_error = &s_error;
bool 
VDKTextBuffer::LoadFromFile(const char* filename)
{ 
  GIOChannel *io;  
	GError *error = NULL;
	g_return_val_if_fail (filename != NULL, false);
	io = g_io_channel_new_file (filename, "r", &error);
	if (!io) 
	 return false;
	if (g_io_channel_set_encoding (io, NULL, &error) != 
      G_IO_STATUS_NORMAL)
   return false;
  while (!error && 
          read_loop (GTK_TEXT_BUFFER(buffer), filename, io, &error) )
          ;	
  g_io_channel_unref (io);
	if (error)
	 {
      g_print("%s\n",error->message);
  		return FALSE;
    }
	gtk_text_buffer_set_modified (GTK_TEXT_BUFFER (buffer), FALSE);
	return true;
}
/*
*/
bool 
VDKTextBuffer::SaveToFile(const char* filename)
{
  GIOChannel *io;  
	GError *error = NULL;
	g_return_val_if_fail (filename != NULL, false);
	GtkTextIter iter, iterend;
	gchar *buf;
	gboolean more = FALSE;
	gsize length=0;
	io=g_io_channel_new_file(filename, "w+", &error);
	if (!io) return false;
	if (g_io_channel_set_encoding (io, "UTF-8" , &error) 
    != G_IO_STATUS_NORMAL)
      return false;
  gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER (buffer), &iter);
	iterend=iter;
	do
    {
		more = gtk_text_iter_forward_line(&iterend);
		buf = gtk_text_iter_get_text(&iter, &iterend);
		if (g_io_channel_write_chars(io, buf, -1, &length, &error) 
      != G_IO_STATUS_NORMAL)
      {
        g_io_channel_unref (io); 
        return false;
      }
    g_free (buf);
		iter = iterend;
	} while (more);
	
  if (g_io_channel_flush(io, &error) != G_IO_STATUS_NORMAL)
	 {
      g_io_channel_unref (io);
      return false;
    }
 	g_io_channel_unref (io);
	gtk_text_buffer_set_modified (GTK_TEXT_BUFFER (buffer), FALSE);
  return true;
}
/*
*/
void 
VDKTextBuffer::TextInsert(const char* txt, int nchar)
{
  gtk_text_buffer_insert_at_cursor (buffer,txt,nchar);
}
/*
*/
void
VDKTextBuffer::Clear()
{
  GtkTextIter start, end;
  gtk_text_buffer_get_bounds (buffer, &start, &end);
  gtk_text_buffer_delete(buffer,&start,&end);
}
/*
*/
void 
VDKTextBuffer::SetPointer(int char_offset)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_offset (buffer,&iter,char_offset);
  gtk_text_buffer_place_cursor(buffer,&iter);
}
/*
*/
int
VDKTextBuffer::GetPointer()
{
  int offset = -1;
  GtkTextIter  iter;
  GtkTextMark* mark = gtk_text_buffer_get_mark(buffer,INSERT_MARK);
  if(mark)
    {
      gtk_text_buffer_get_iter_at_mark(buffer,&iter,mark);
      offset = gtk_text_iter_get_offset(&iter);
    }
  return offset;
}
/*
*/
void 
VDKTextBuffer::SetLine(int row)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_line(buffer,&iter,row);
  gtk_text_buffer_place_cursor(buffer,&iter);
}
/*
*/
int
VDKTextBuffer::GetLine()
{
  int line = -1;
  GtkTextIter  iter;
  GtkTextMark* mark = gtk_text_buffer_get_mark(buffer,INSERT_MARK);
  if(mark)
    {
      gtk_text_buffer_get_iter_at_mark(buffer,&iter,mark);
      line  = gtk_text_iter_get_line(&iter);
    }
  return line;
}
/*
*/
int
VDKTextBuffer::GetColumn()
{
  int lineoffset = -1;
  GtkTextIter  iter;
  GtkTextMark* mark = gtk_text_buffer_get_mark(buffer,INSERT_MARK);
  if(mark)
    {
      gtk_text_buffer_get_iter_at_mark(buffer,&iter,mark);
      lineoffset = gtk_text_iter_get_line_offset (&iter);
    }
  return lineoffset;
}
/*
*/
void 
VDKTextBuffer::SetColumn(int col)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_line_offset(buffer,&iter,Line,col);
  gtk_text_buffer_place_cursor(buffer,&iter);
}
/*
*/
gchar* 
VDKTextBuffer::GetChars(int start, int end)
{
  GtkTextIter first,last;
  gtk_text_buffer_get_iter_at_offset(buffer,&first,start);
  if(end >= 0)
    gtk_text_buffer_get_iter_at_offset(buffer,&last,end);
  else

    gtk_text_buffer_get_end_iter(buffer,&last);
  // returned address should be gfree()'d by user or it will leak.
  return gtk_text_buffer_get_slice(buffer,&first,&last,FALSE);
}

/*
Forward delete chars from insertion point
*/
void  
VDKTextBuffer::ForwardDelete(int nchars)
{
  GtkTextIter  start, end;
  GtkTextMark* mark = gtk_text_buffer_get_mark(buffer,INSERT_MARK);
  if(mark)
    {
      int offset = (int) Pointer + nchars;
      int len = Length;
      gtk_text_buffer_get_iter_at_mark(buffer,&start,mark);
      if(offset < len)
        gtk_text_buffer_get_iter_at_offset(buffer,&end,offset);
      else
        gtk_text_buffer_get_end_iter(buffer,&end);
      gtk_text_buffer_delete(buffer,&start,&end);
    }
}

  /*
  */
void  
VDKTextBuffer::BackwardDelete(int nchars)
{
  GtkTextIter  start, end;
  GtkTextMark* mark = gtk_text_buffer_get_mark(buffer,INSERT_MARK);
  if(mark)
  { 
      int offset = (int) Pointer - nchars;
      offset = offset < 0 ? 0 : offset;
      gtk_text_buffer_get_iter_at_mark(buffer,&end,mark);
      gtk_text_buffer_get_iter_at_offset(buffer,&start,offset);
      gtk_text_buffer_delete(buffer,&start,&end);
  }
}
/*
*/
int 
VDKTextBuffer::GetLineAtOffset(int offset)
{
  int line = -1;
  GtkTextIter  iter;
  gtk_text_buffer_get_iter_at_offset(buffer,&iter,offset);
  line  = gtk_text_iter_get_line(&iter);
  return line; 
}
/*
---------------------------------------------
             VDKTextView class
---------------------------------------------
*/
/*
 */
VDKTextView::VDKTextView(VDKForm* owner, VDKTextBuffer* buff, int left_border):
  VDKObjectContainer(owner),buffer(buff),left_border(left_border),
  Pointer("Pointer",this,0,&VDKTextView::SetPointer,
	    &VDKTextView::GetPointer),
  Column("Column",this,0,&VDKTextView::SetColumn,&VDKTextView::GetColumn),
  Line("Line",this,0,&VDKTextView::SetLine,&VDKTextView::GetLine),
  Length("Length",this,0,&VDKTextView::GetLength),
  Editable("Editable",this,true,&VDKTextView::SetEditable,&VDKTextView::GetEditable),
  MaxUndo("MaxUndo",this,0),
  LineAutoSelect("LineAutoSelect",this,false),
  ShowLineNumbers("ShowLineNumbers",this,false,&VDKTextView::SetShowLineNumbers),
  FirstVisibleLine("FirstVisibleLine",this,0,&VDKTextView::GetFirstVisibleLine),
  LastVisibleLine("LastVisibleLine",this,0,&VDKTextView::GetLastVisibleLine),
  Changed("Changed",this,false,&VDKTextView::SetChanged,&VDKTextView::GetChanged)
{
  widget = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (widget),
  				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
  if(!buffer)
    {
      //GtkTextTagTable *table = gtk_text_tag_table_new();
      buffer = new VDKTextBuffer();
      //g_object_ref(table);
      //GTK_TEXT_BUFFER(buffer->buffer)->tag_table = table;
    }
  else
    buffer->Ref();
  view = gtk_text_view_new_with_buffer (buffer->buffer);
  sigwid = view;
  if(left_border)
    TextBorder(left_border,TVB_LEFT);
  gtk_container_add (GTK_CONTAINER (widget), view);
  gtk_widget_show(view);
  // ConnectDefaultSignals();
  ConnectSignals();
 }
/*
*/
void
VDKTextView::HandleRealize(GtkWidget* widget, gpointer gp)
{
/*
  VDKTextView* text = reinterpret_cast<VDKTextView*>(gp);
  GtkTextView* wid = GTK_TEXT_VIEW(text->WrappedWidget());
  GdkWindow* win = gtk_text_view_get_window (wid,GTK_TEXT_WINDOW_LEFT);
  VDKColor * color = new VDKColor(text->Owner(),"black");
  if( win )
    gdk_window_set_background(win,color->Color());
*/
  VDKTextView* text = reinterpret_cast<VDKTextView*>(gp);
  if(text)
    {
      text->SignalEmit(realize_signal);
      text->SignalEmit("realize");
    }
}
/*
*/
void 
VDKTextView::ConnectSignals()
{
  gtk_signal_connect(GTK_OBJECT(view),"realize",
			  GTK_SIGNAL_FUNC(VDKTextView::HandleRealize),
			  (gpointer) this);
}
/*
 */
VDKTextView::~VDKTextView()
{
  if(buffer)
    buffer->Unref();
}
/*
*/
VDKTextBuffer* 
VDKTextView::Buffer(VDKTextBuffer* buff)
{
  if(buff && (buff != buffer))
  {
    if(buffer)
      buffer->Unref();
    buffer = buff;
    buffer->Ref();
    gtk_text_view_set_buffer(GTK_TEXT_VIEW(view),buffer->buffer);
   }
  return buffer;
}
/*
 */
void VDKTextView::SetBackground(VDKRgb rgb, GtkStateType state)
{
  /*
  VDKColor *color = new VDKColor(Owner(),rgb.red,rgb.green,rgb.blue);
  GdkWindow* win = gtk_text_view_get_window (GTK_TEXT_VIEW(view),GTK_TEXT_WINDOW_TEXT);
  if(win)
    {
      gdk_window_set_background (win,color->Color());
    }
  */
  VDKColor *color = new VDKColor(Owner(),rgb.red,rgb.green,rgb.blue);
  gtk_widget_modify_base (GTK_WIDGET(view),state, color->Color());
}

void VDKTextView::SetForeground(VDKRgb rgb, GtkStateType state)
{
  VDKColor *color = new VDKColor(Owner(),rgb.red,rgb.green,rgb.blue);
  gtk_widget_modify_text (GTK_WIDGET(view), state, color->Color());
}

void 
VDKTextView::SetFont(VDKFont* font)
{
  VDKObject::_setFont_(sigwid,font);
}

void 
VDKTextView::TextBorder(int size, int which)
{
  int mask = which & TVB_TYPEMASK;
  if((which == TVB_ALL) || (mask == TVB_LEFT))
    gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (view),
                                        GTK_TEXT_WINDOW_LEFT,
                                        size);
  if((which == TVB_ALL) || (mask == TVB_TOP))
    gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (view),
                                        GTK_TEXT_WINDOW_TOP,
                                        size);
  if((which == TVB_ALL) || (mask == TVB_RIGHT))
    gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (view),
                                        GTK_TEXT_WINDOW_RIGHT,
                                        size);
  if((which == TVB_ALL) || (mask == TVB_BOTTOM))
    gtk_text_view_set_border_window_size (GTK_TEXT_VIEW (view),
                                        GTK_TEXT_WINDOW_BOTTOM,
                                        size);
}

/*
  Scrolls to a pointer pos or (default) to current
  pointer position, leaving <margin> pixels free
*/
void 
VDKTextView::ScrollToPos (int pointer, int margin)
{
  if(pointer >= 0)
    buffer->Pointer = pointer;
  GtkTextMark* mark = gtk_text_buffer_get_mark(buffer->Buffer(),
                            INSERT_MARK);
  if(mark)
    // more args on new gtk+ snapshoots
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view),mark,margin,0,0,0);
}
/*
  Scrolls to a line,column leaving <margin> pixels free
*/
void 
VDKTextView::ScrollToLine(int line, int col, int margin)
{
  buffer->Line = line;
  buffer->Column = col;
  GtkTextMark* mark = gtk_text_buffer_get_mark(buffer->Buffer(),
                            INSERT_MARK);
  if(mark)
    // more args on new gtk+ snapshoots
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view),mark,margin,0,0,0);
}

/*
 gets lines
*/
static void
get_lines (GtkTextView  *text_view,
           gint          first_y,
           gint          last_y,
           GArray       *buffer_coords,
           GArray       *numbers,
           gint         *countp)
{
  GtkTextIter iter;
  gint count;
  gint size;  

  g_array_set_size (buffer_coords, 0);
  g_array_set_size (numbers, 0);
  
  /* Get iter at first y */
  gtk_text_view_get_line_at_y (text_view, &iter, first_y, NULL);

  /* For each iter, get its location and add it to the arrays.
   * Stop when we pass last_y
   */
  count = 0;
  size = 0;

  while (!gtk_text_iter_is_end (&iter))
    {
      gint y, height;
      gint line_num;
      
      gtk_text_view_get_line_yrange (text_view, &iter, &y, &height);

      g_array_append_val (buffer_coords, y);
      line_num = gtk_text_iter_get_line (&iter);
      g_array_append_val (numbers, line_num);
      
      ++count;

      if ((y + height) >= last_y)
        break;
      
      gtk_text_iter_forward_line (&iter);
    }

  *countp = count;
}

/*
  handle expose on text view showing line numbers
  (shameless stolen from testext.c example in gtk+ distribution)
*/
static gint
line_numbers_expose (GtkWidget      *widget,
                     GdkEventExpose *event,
                     gpointer        user_data)
{
  gint count;
  GArray *numbers;
  GArray *pixels;
  gint first_y;
  gint last_y;
  gint i;
  GdkWindow *left_win;
  PangoLayout *layout;
  GtkTextView *text_view;
  GtkTextWindowType type;
  GdkDrawable *target;
  VDKTextView* text = reinterpret_cast<VDKTextView*>(user_data);
  if(! text || !(text->ShowLineNumbers))
    return FALSE;
  else
    text_view = GTK_TEXT_VIEW (widget);
  /* See if this expose is on the line numbers window */
  left_win = gtk_text_view_get_window (text_view,
                                       GTK_TEXT_WINDOW_LEFT);
  if (event->window == left_win)
    {
      type = GTK_TEXT_WINDOW_LEFT;
      target = left_win;
    }
  else
    return FALSE;
  
  first_y = event->area.y;
  last_y = first_y + event->area.height;

  gtk_text_view_window_to_buffer_coords (text_view,
                                         type,
                                         0,
                                         first_y,
                                         NULL,
                                         &first_y);
  gtk_text_view_window_to_buffer_coords (text_view,
                                         type,
                                         0,
                                         last_y,
                                         NULL,
                                         &last_y);

  numbers = g_array_new (FALSE, FALSE, sizeof (gint));
  pixels = g_array_new (FALSE, FALSE, sizeof (gint));
  get_lines (text_view,
             first_y,
             last_y,
             pixels,
             numbers,
             &count);
  /* Draw fully internationalized numbers! */
  layout = gtk_widget_create_pango_layout (widget, "");
  i = 0;
  while (i < count)
    {
      gint pos;
      gchar *str;
      gtk_text_view_buffer_to_window_coords (text_view,
                                             type,
                                             0,
                                             g_array_index (pixels, gint, i),
                                             NULL,
                                             &pos);

      str = g_strdup_printf ("%5d:", g_array_index (numbers, gint, i)+1);
      pango_layout_set_text (layout, str, -1);
      gdk_draw_layout (target,
                       widget->style->fg_gc [widget->state],
                       /* 2 is just a random padding */
                       2, pos + 2,
                       layout);

      g_free (str);
      ++i;
    }
  g_array_free (pixels, TRUE);
  g_array_free (numbers, TRUE);
  g_object_unref (G_OBJECT (layout));
  return TRUE;
}

/*
*/
static int expose_connection = 0;
void 
VDKTextView::SetShowLineNumbers(bool show)
{
 GdkWindow* left_win = gtk_text_view_get_window (GTK_TEXT_VIEW(view),
                                       GTK_TEXT_WINDOW_LEFT);
  if(left_win)
   {
    if(show)
      {
        expose_connection = gtk_signal_connect (GTK_OBJECT (view),
                      "expose_event",
                      GTK_SIGNAL_FUNC (line_numbers_expose),
                      this);
        gtk_widget_queue_draw(view);
      }
     else
      {
      gtk_signal_disconnect(GTK_OBJECT (view),expose_connection);
      TextBorder(0, TVB_LEFT); 
      }
   }
}
/*
*/
int
VDKTextView::GetFirstVisibleLine()
{
    GdkRectangle  visible_rect;
    GtkTextIter iter;
    int line_top;
    gtk_text_view_get_visible_rect(GTK_TEXT_VIEW(view),&visible_rect);
    gtk_text_view_get_line_at_y(GTK_TEXT_VIEW(view),&iter,visible_rect.y,NULL);
    line_top = gtk_text_iter_get_line(&iter);
    return line_top;
}
/*
*/
int
VDKTextView::GetLastVisibleLine()
{
    GdkRectangle  visible_rect;
    GtkTextIter iter;
    int line_top;
    gtk_text_view_get_visible_rect(GTK_TEXT_VIEW(view),&visible_rect);
    gtk_text_view_get_line_at_y(GTK_TEXT_VIEW(view),&iter,
        visible_rect.y+visible_rect.height,NULL);
    line_top = gtk_text_iter_get_line(&iter);
    return line_top;
}
/*
*/
