/*  wmbday
 *  Copyright (C) 2003-2005 astratis <steffen@x-berg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "main.h"
#include "ring.h"
#include "date.h"

#define WHITESPACE    " \t\r\n"

extern struct settings mySettings;


int _LoadDatafile_Simple(FILE *fp) {
  char buffer[128];
  char* buf;
  int day, month, year;
  int count = 0;
  int line = 0;
  char* ptrchar;

  while (fgets(buffer, 101, fp) != NULL) {
    ++line;

    buf = buffer;

    /* ignore whitespace at the beginning of line and lines that are comments */
    buf += strspn(buf, WHITESPACE);
    if(*buf == '#')
      continue;

    /* first character of line has to be a number */
    if(!isdigit(*buf)) {
      fprintf(stderr, "data file: line %d does not begin with a number\n", line);
      continue;
    }

    /* read the date */
    sscanf(buf, "%2d.%2d.%4d", &day, &month, &year);

    /* check date */
    if(is_date(day, month, year) == 0) {
      fprintf(stderr, "data file: invalid date in line %d. (Out of range or in future)\n", line);
      continue;
    }

    /* jump behind the date */
    buf += strcspn(buf, WHITESPACE);
    buf += strspn(buf, WHITESPACE);

    /* handle names in "" */
    if(*buf == '\"') {
      ++buf;
      ptrchar = buf + strcspn(buf, "\"");
    }
    else
      ptrchar = buf + strcspn(buf, "\n\r");

    *ptrchar = '\0';

    /* check if there is a name in this line */
    if(!strlen(buf)) {
      fprintf(stderr, "data file: no name given in line %d.\n", line);
      continue;
    }

    ring_add(buf, day, month, year);

    ++count;
  }

  return count;
}


int _LoadDatafile_vCard(FILE *fp) {
  char buffer[128];
  char* ptr;
  int day, month, year;
  char name[96];
  int count = 0;

  int name_found = 0, bday_found = 0;

  while (fgets(buffer, 100, fp) != NULL) {
    if (!strncmp(buffer, "FN:", 3)) {
      strncpy(name, buffer+3, sizeof(name));
      ptr = strrchr(name, '\n');
      *ptr = '\0';
      name_found = 1;
      continue;
    }
    if (!strncmp(buffer, "BDAY:", 5)) {
      sscanf(buffer+5, "%4d-%2d-%2d", &year, &month, &day);
      bday_found = 1;
      continue;
    }
    if (!strncmp(buffer, "END:VCARD", 9)) {
      if (name_found && bday_found) {
        ring_add(name, day, month, year);
        ++count;
      }
      name_found = bday_found = 0;
    }
  }

  return count;
}


void LoadDatafile(void) {
  FILE *fp;
  char buffer[16];
  int count;

  /* if there is a ring, we clear it here first */
  ring_clear();

  /* open data file */
  if((fp = fopen(mySettings.datafile, "r")) == NULL) {
    fprintf(stderr, "Couldn't open datafile: %s\n", mySettings.datafile);
    exit(1);
    return;
  }

  /* check data file format (simple or vcard) and call appropriate parser */
  fgets(buffer, 12, fp);
  rewind(fp);

  if (!strcmp(buffer, "BEGIN:VCARD"))
    count = _LoadDatafile_vCard(fp);
  else
    count = _LoadDatafile_Simple(fp);

  /* don't start, if no birthdays could be loaded */
  if(count == 0) {
    printf("No birthdays could be loaded, exiting.\n");
    exit(0);
  }
  else
    printf("Loaded %d birthdays\n", count);

  /*close file and free memory */
  fclose(fp);

  return;
}
