/*
 * Copyright (c) 1994, 2014, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package glue.java.lang;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Properties;
import java.util.PropertyPermission;

import blues.Log;
import blues.LogOutputStream;

/**
 * The <code>System</code> class contains several useful class fields and
 * methods. It cannot be instantiated.
 *
 * <p>
 * Among the facilities provided by the <code>System</code> class are standard
 * input, standard output, and error output streams; access to externally
 * defined properties and environment variables; a means of loading files and
 * libraries; and a utility method for quickly copying a portion of an array.
 *
 * @author unascribed
 * @since JDK1.0
 */
public final class System {

    /**
     * The "standard" input stream. This stream is already open and ready to
     * supply input data. Typically this stream corresponds to keyboard input or
     * another input source specified by the host environment or user.
     */
    private static InputStream field_in = nullInputStream();

    public static InputStream get_in() {
        return field_in;
    }

    /**
     * The "standard" output stream. This stream is already open and ready to
     * accept output data. Typically this stream corresponds to display output
     * or another output destination specified by the host environment or user.
     * <p>
     * For simple stand-alone Java applications, a typical way to write a line
     * of output data is: <blockquote>
     * 
     * <pre>
     * System.out.println(data)
     * </pre>
     * 
     * </blockquote>
     * <p>
     * See the <code>println</code> methods in class <code>PrintStream</code>.
     *
     * @see java.io.PrintStream#println()
     * @see java.io.PrintStream#println(boolean)
     * @see java.io.PrintStream#println(char)
     * @see java.io.PrintStream#println(char[])
     * @see java.io.PrintStream#println(double)
     * @see java.io.PrintStream#println(float)
     * @see java.io.PrintStream#println(int)
     * @see java.io.PrintStream#println(long)
     * @see java.io.PrintStream#println(java.lang.Object)
     * @see java.io.PrintStream#println(java.lang.String)
     */
    private static PrintStream field_out = logPrintStream(Log.LOG_OUT);

    public static PrintStream get_out() {
        return field_out;
    }

    /**
     * The "standard" error output stream. This stream is already open and ready
     * to accept output data.
     * <p>
     * Typically this stream corresponds to display output or another output
     * destination specified by the host environment or user. By convention,
     * this output stream is used to display error messages or other information
     * that should come to the immediate attention of a user even if the
     * principal output stream, the value of the variable <code>out</code>, has
     * been redirected to a file or other destination that is typically not
     * continuously monitored.
     */
    private static PrintStream field_err = logPrintStream(Log.LOG_ERR);

    public static PrintStream get_err() {
        return field_err;
    }

    /**
     * Reassigns the "standard" input stream.
     *
     * <p>
     * First, if there is a security manager, its <code>checkPermission</code>
     * method is called with a <code>RuntimePermission("setIO")</code>
     * permission to see if it's ok to reassign the "standard" input stream.
     * <p>
     *
     * @param in
     *            the new standard input stream.
     *
     * @throws SecurityException
     *             if a security manager exists and its
     *             <code>checkPermission</code> method doesn't allow reassigning
     *             of the standard input stream.
     *
     * @see SecurityManager#checkPermission
     * @see java.lang.RuntimePermission
     *
     * @since JDK1.1
     */
    public static void setIn(InputStream in) {
        checkIO();
        field_in = in;
    }

    /**
     * Reassigns the "standard" output stream.
     *
     * <p>
     * First, if there is a security manager, its <code>checkPermission</code>
     * method is called with a <code>RuntimePermission("setIO")</code>
     * permission to see if it's ok to reassign the "standard" output stream.
     *
     * @param out
     *            the new standard output stream
     *
     * @throws SecurityException
     *             if a security manager exists and its
     *             <code>checkPermission</code> method doesn't allow reassigning
     *             of the standard output stream.
     *
     * @see SecurityManager#checkPermission
     * @see java.lang.RuntimePermission
     *
     * @since JDK1.1
     */
    public static void setOut(PrintStream out) {
        checkIO();
        field_out = out;
    }

    /**
     * Reassigns the "standard" error output stream.
     *
     * <p>
     * First, if there is a security manager, its <code>checkPermission</code>
     * method is called with a <code>RuntimePermission("setIO")</code>
     * permission to see if it's ok to reassign the "standard" error output
     * stream.
     *
     * @param err
     *            the new standard error output stream.
     *
     * @throws SecurityException
     *             if a security manager exists and its
     *             <code>checkPermission</code> method doesn't allow reassigning
     *             of the standard error output stream.
     *
     * @see SecurityManager#checkPermission
     * @see java.lang.RuntimePermission
     *
     * @since JDK1.1
     */
    public static void setErr(PrintStream err) {
        checkIO();
        field_err = err;
    }

    private static void checkIO() {
        java.lang.SecurityManager sm = java.lang.System.getSecurityManager();
        if (sm != null) {
            sm.checkPermission(new RuntimePermission("setIO"));
        }
    }

    /**
     * Sets the System security.
     *
     * <p>
     * If there is a security manager already installed, this method first calls
     * the security manager's <code>checkPermission</code> method with a
     * <code>RuntimePermission("setSecurityManager")</code> permission to ensure
     * it's ok to replace the existing security manager. This may result in
     * throwing a <code>SecurityException</code>.
     *
     * <p>
     * Otherwise, the argument is established as the current security manager.
     * If the argument is <code>null</code> and no security manager has been
     * established, then no action is taken and the method simply returns.
     *
     * @param s
     *            the security manager.
     * @exception SecurityException
     *                if the security manager has already been set and its
     *                <code>checkPermission</code> method doesn't allow it to be
     *                replaced.
     * @see #getSecurityManager
     * @see SecurityManager#checkPermission
     * @see java.lang.RuntimePermission
     */
    public static void setSecurityManager(final java.lang.SecurityManager s) {
        try {
            s.checkPackageAccess("java.lang");
        } catch (Exception e) {
            // no-op
        }
        setSecurityManager0(s);
    }

    private static synchronized void setSecurityManager0(final java.lang.SecurityManager s) {
        java.lang.SecurityManager sm = java.lang.System.getSecurityManager();
        if (sm != null) {
            // ask the currently installed security manager if we
            // can replace it.
            sm.checkPermission(new RuntimePermission("setSecurityManager"));
        }
        java.lang.System.setSecurityManager(s);
    }

    /**
     * Returns the current time in milliseconds. Note that while the unit of
     * time of the return value is a millisecond, the granularity of the value
     * depends on the underlying operating system and may be larger. For
     * example, many operating systems measure time in units of tens of
     * milliseconds.
     *
     * <p>
     * See the description of the class <code>Date</code> for a discussion of
     * slight discrepancies that may arise between "computer time" and
     * coordinated universal time (UTC).
     *
     * @return the difference, measured in milliseconds, between the current
     *         time and midnight, January 1, 1970 UTC.
     * @see java.util.Date
     */
    public static long currentTimeMillis() {
        return java.lang.System.currentTimeMillis();
    }

    /**
     * System properties. The following properties are guaranteed to be defined:
     * <dl>
     * <dt>java.version
     * <dd>Java version number
     * <dt>java.vendor
     * <dd>Java vendor specific string
     * <dt>java.vendor.url
     * <dd>Java vendor URL
     * <dt>java.home
     * <dd>Java installation directory
     * <dt>java.class.version
     * <dd>Java class version number
     * <dt>java.class.path
     * <dd>Java classpath
     * <dt>os.name
     * <dd>Operating System Name
     * <dt>os.arch
     * <dd>Operating System Architecture
     * <dt>os.version
     * <dd>Operating System Version
     * <dt>file.separator
     * <dd>File separator ("/" on Unix)
     * <dt>path.separator
     * <dd>Path separator (":" on Unix)
     * <dt>line.separator
     * <dd>Line separator ("\n" on Unix)
     * <dt>user.name
     * <dd>User account name
     * <dt>user.home
     * <dd>User home directory
     * <dt>user.dir
     * <dd>User's current working directory
     * </dl>
     */

    private static Properties props = new Properties();

    /**
     * Determines the current system properties.
     * <p>
     * First, if there is a security manager, its
     * <code>checkPropertiesAccess</code> method is called with no arguments.
     * This may result in a security exception.
     * <p>
     * The current set of system properties for use by the
     * {@link #getProperty(String)} method is returned as a
     * <code>Properties</code> object. If there is no current set of system
     * properties, a set of system properties is first created and initialized.
     * This set of system properties always includes values for the following
     * keys:
     * <table summary="Shows property keys and associated values">
     * <tr>
     * <th>Key</th>
     * <th>Description of Associated Value</th>
     * </tr>
     * <tr>
     * <td><code>java.version</code></td>
     * <td>Java Runtime Environment version</td>
     * </tr>
     * <tr>
     * <td><code>java.vendor</code></td>
     * <td>Java Runtime Environment vendor</td></tr
     * <tr>
     * <td><code>java.vendor.url</code></td>
     * <td>Java vendor URL</td>
     * </tr>
     * <tr>
     * <td><code>java.home</code></td>
     * <td>Java installation directory</td>
     * </tr>
     * <tr>
     * <td><code>java.vm.specification.version</code></td>
     * <td>Java Virtual Machine specification version</td>
     * </tr>
     * <tr>
     * <td><code>java.vm.specification.vendor</code></td>
     * <td>Java Virtual Machine specification vendor</td>
     * </tr>
     * <tr>
     * <td><code>java.vm.specification.name</code></td>
     * <td>Java Virtual Machine specification name</td>
     * </tr>
     * <tr>
     * <td><code>java.vm.version</code></td>
     * <td>Java Virtual Machine implementation version</td>
     * </tr>
     * <tr>
     * <td><code>java.vm.vendor</code></td>
     * <td>Java Virtual Machine implementation vendor</td>
     * </tr>
     * <tr>
     * <td><code>java.vm.name</code></td>
     * <td>Java Virtual Machine implementation name</td>
     * </tr>
     * <tr>
     * <td><code>java.specification.version</code></td>
     * <td>Java Runtime Environment specification version</td>
     * </tr>
     * <tr>
     * <td><code>java.specification.vendor</code></td>
     * <td>Java Runtime Environment specification vendor</td>
     * </tr>
     * <tr>
     * <td><code>java.specification.name</code></td>
     * <td>Java Runtime Environment specification name</td>
     * </tr>
     * <tr>
     * <td><code>java.class.version</code></td>
     * <td>Java class format version number</td>
     * </tr>
     * <tr>
     * <td><code>java.class.path</code></td>
     * <td>Java class path</td>
     * </tr>
     * <tr>
     * <td><code>java.library.path</code></td>
     * <td>List of paths to search when loading libraries</td>
     * </tr>
     * <tr>
     * <td><code>java.io.tmpdir</code></td>
     * <td>Default temp file path</td>
     * </tr>
     * <tr>
     * <td><code>java.compiler</code></td>
     * <td>Name of JIT compiler to use</td>
     * </tr>
     * <tr>
     * <td><code>java.ext.dirs</code></td>
     * <td>Path of extension directory or directories</td>
     * </tr>
     * <tr>
     * <td><code>os.name</code></td>
     * <td>Operating system name</td>
     * </tr>
     * <tr>
     * <td><code>os.arch</code></td>
     * <td>Operating system architecture</td>
     * </tr>
     * <tr>
     * <td><code>os.version</code></td>
     * <td>Operating system version</td>
     * </tr>
     * <tr>
     * <td><code>file.separator</code></td>
     * <td>File separator ("/" on UNIX)</td>
     * </tr>
     * <tr>
     * <td><code>path.separator</code></td>
     * <td>Path separator (":" on UNIX)</td>
     * </tr>
     * <tr>
     * <td><code>line.separator</code></td>
     * <td>Line separator ("\n" on UNIX)</td>
     * </tr>
     * <tr>
     * <td><code>user.name</code></td>
     * <td>User's account name</td>
     * </tr>
     * <tr>
     * <td><code>user.home</code></td>
     * <td>User's home directory</td>
     * </tr>
     * <tr>
     * <td><code>user.dir</code></td>
     * <td>User's current working directory</td>
     * </tr>
     * </table>
     * <p>
     * Multiple paths in a system property value are separated by the path
     * separator character of the platform.
     * <p>
     * Note that even if the security manager does not permit the
     * <code>getProperties</code> operation, it may choose to permit the
     * {@link #getProperty(String)} operation.
     *
     * @return the system properties
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkPropertiesAccess</code> method doesn't allow
     *                access to the system properties.
     * @see #setProperties
     * @see java.lang.SecurityException
     * @see java.lang.SecurityManager#checkPropertiesAccess()
     * @see java.util.Properties
     */
    public static Properties getProperties() {
        java.lang.SecurityManager sm = java.lang.System.getSecurityManager();
        if (sm != null) {
            sm.checkPropertiesAccess();
        }

        Log.log(1, Log.LOG_SYS, "System.getProperties");

        return props;
    }

    public static Properties properties() {
        return props;
    }

    /**
     * Sets the system properties to the <code>Properties</code> argument.
     * <p>
     * First, if there is a security manager, its
     * <code>checkPropertiesAccess</code> method is called with no arguments.
     * This may result in a security exception.
     * <p>
     * The argument becomes the current set of system properties for use by the
     * {@link #getProperty(String)} method. If the argument is <code>null</code>
     * , then the current set of system properties is forgotten.
     *
     * @param props
     *            the new system properties.
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkPropertiesAccess</code> method doesn't allow
     *                access to the system properties.
     * @see #getProperties
     * @see java.util.Properties
     * @see java.lang.SecurityException
     * @see java.lang.SecurityManager#checkPropertiesAccess()
     */
    public static void setProperties(Properties props) {
        java.lang.SecurityManager sm = java.lang.System.getSecurityManager();
        if (sm != null) {
            sm.checkPropertiesAccess();
        }
        /*
         * if (props == null) { props = new Properties(); initProperties(props);
         * } System.props = props;
         */
        Log.log(1, Log.LOG_SYS, "System.setProperties");
    }

    /**
     * Gets the system property indicated by the specified key.
     * <p>
     * First, if there is a security manager, its
     * <code>checkPropertyAccess</code> method is called with the key as its
     * argument. This may result in a SecurityException.
     * <p>
     * If there is no current set of system properties, a set of system
     * properties is first created and initialized in the same manner as for the
     * <code>getProperties</code> method.
     *
     * @param key
     *            the name of the system property.
     * @return the string value of the system property, or <code>null</code> if
     *         there is no property with that key.
     *
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkPropertyAccess</code> method doesn't allow
     *                access to the specified system property.
     * @exception NullPointerException
     *                if <code>key</code> is <code>null</code>.
     * @exception IllegalArgumentException
     *                if <code>key</code> is empty.
     * @see #setProperty
     * @see java.lang.SecurityException
     * @see java.lang.SecurityManager#checkPropertyAccess(java.lang.String)
     * @see java.lang.System#getProperties()
     */
    public static String getProperty(String key) {
        checkKey(key);
        java.lang.SecurityManager sm = java.lang.System.getSecurityManager();
        if (sm != null) {
            sm.checkPropertyAccess(key);
        }

        String r = props.getProperty(key);
        Log.log(1, Log.LOG_SYS, "System.getProperty(\"", key, "\")=", r);
        return r;
    }

    /**
     * Gets the system property indicated by the specified key.
     * <p>
     * First, if there is a security manager, its
     * <code>checkPropertyAccess</code> method is called with the
     * <code>key</code> as its argument.
     * <p>
     * If there is no current set of system properties, a set of system
     * properties is first created and initialized in the same manner as for the
     * <code>getProperties</code> method.
     *
     * @param key
     *            the name of the system property.
     * @param def
     *            a default value.
     * @return the string value of the system property, or the default value if
     *         there is no property with that key.
     *
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkPropertyAccess</code> method doesn't allow
     *                access to the specified system property.
     * @exception NullPointerException
     *                if <code>key</code> is <code>null</code>.
     * @exception IllegalArgumentException
     *                if <code>key</code> is empty.
     * @see #setProperty
     * @see java.lang.SecurityManager#checkPropertyAccess(java.lang.String)
     * @see java.lang.System#getProperties()
     */
    public static String getProperty(String key, String def) {
        checkKey(key);
        java.lang.SecurityManager sm = java.lang.System.getSecurityManager();
        if (sm != null) {
            sm.checkPropertyAccess(key);
        }

        String r = props.getProperty(key, def);
        Log.log(1, Log.LOG_SYS, "System.getProperty(\"", key, "\")=", r);
        return r;
    }

    /**
     * Sets the system property indicated by the specified key.
     * <p>
     * First, if a security manager exists, its
     * <code>SecurityManager.checkPermission</code> method is called with a
     * <code>PropertyPermission(key, "write")</code> permission. This may result
     * in a SecurityException being thrown. If no exception is thrown, the
     * specified property is set to the given value.
     * <p>
     *
     * @param key
     *            the name of the system property.
     * @param value
     *            the value of the system property.
     * @return the previous value of the system property, or <code>null</code>
     *         if it did not have one.
     *
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkPermission</code> method doesn't allow setting
     *                of the specified property.
     * @exception NullPointerException
     *                if <code>key</code> or <code>value</code> is
     *                <code>null</code>.
     * @exception IllegalArgumentException
     *                if <code>key</code> is empty.
     * @see #getProperty
     * @see java.lang.System#getProperty(java.lang.String)
     * @see java.lang.System#getProperty(java.lang.String, java.lang.String)
     * @see java.util.PropertyPermission
     * @see SecurityManager#checkPermission
     * @since 1.2
     */
    public static String setProperty(String key, String value) {
        checkKey(key);
        java.lang.SecurityManager sm = java.lang.System.getSecurityManager();
        if (sm != null) {
            sm.checkPermission(new PropertyPermission(key, "write"));
        }

        Log.log(1, Log.LOG_SYS, "System.setProperty(\"", key, "\")=", value);
        return (String) props.setProperty(key, value);
    }

    private static void checkKey(String key) {
        if (key == null) {
            throw new NullPointerException("key can't be null");
        }
        if (key.equals("")) {
            throw new IllegalArgumentException("key can't be empty");
        }
    }

    /**
     * Terminates the currently running Java Virtual Machine. The argument
     * serves as a status code; by convention, a nonzero status code indicates
     * abnormal termination.
     * <p>
     * This method calls the <code>exit</code> method in class
     * <code>Runtime</code>. This method never returns normally.
     * <p>
     * The call <code>System.exit(n)</code> is effectively equivalent to the
     * call: <blockquote>
     * 
     * <pre>
     * Runtime.getRuntime().exit(n)
     * </pre>
     * 
     * </blockquote>
     *
     * @param status
     *            exit status.
     * @throws SecurityException
     *             if a security manager exists and its <code>checkExit</code>
     *             method doesn't allow exit with the specified status.
     * @see java.lang.Runtime#exit(int)
     */
    public static void exit(int status) {
        jail.java.lang.Runtime.getRuntime().exit(status);
    }

    /**
     * Runs the finalization methods of any objects pending finalization.
     * <p>
     * Calling this method suggests that the Java Virtual Machine expend effort
     * toward running the <code>finalize</code> methods of objects that have
     * been found to be discarded but whose <code>finalize</code> methods have
     * not yet been run. When control returns from the method call, the Java
     * Virtual Machine has made a best effort to complete all outstanding
     * finalizations.
     * <p>
     * The call <code>System.runFinalization()</code> is effectively equivalent
     * to the call: <blockquote>
     * 
     * <pre>
     * Runtime.getRuntime().runFinalization()
     * </pre>
     * 
     * </blockquote>
     *
     * @see java.lang.Runtime#runFinalization()
     */
    public static void runFinalization() {
        jail.java.lang.Runtime.getRuntime().runFinalization();
    }

    /**
     * Loads a code file with the specified filename from the local file system
     * as a dynamic library. The filename argument must be a complete path name.
     * <p>
     * The call <code>System.load(name)</code> is effectively equivalent to the
     * call: <blockquote>
     * 
     * <pre>
     * Runtime.getRuntime().load(name)
     * </pre>
     * 
     * </blockquote>
     *
     * @param filename
     *            the file to load.
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkLink</code> method doesn't allow loading of the
     *                specified dynamic library
     * @exception UnsatisfiedLinkError
     *                if the file does not exist.
     * @exception NullPointerException
     *                if <code>filename</code> is <code>null</code>
     * @see java.lang.Runtime#load(java.lang.String)
     * @see java.lang.SecurityManager#checkLink(java.lang.String)
     */
    public static void load(String filename) {
        jail.java.lang.Runtime.getRuntime().load(filename);
    }

    /**
     * Loads the system library specified by the <code>libname</code> argument.
     * The manner in which a library name is mapped to the actual system library
     * is system dependent.
     * <p>
     * The call <code>System.loadLibrary(name)</code> is effectively equivalent
     * to the call <blockquote>
     * 
     * <pre>
     * Runtime.getRuntime().loadLibrary(name)
     * </pre>
     * 
     * </blockquote>
     *
     * @param libname
     *            the name of the library.
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkLink</code> method doesn't allow loading of the
     *                specified dynamic library
     * @exception UnsatisfiedLinkError
     *                if the library does not exist.
     * @exception NullPointerException
     *                if <code>libname</code> is <code>null</code>
     * @see java.lang.Runtime#loadLibrary(java.lang.String)
     * @see java.lang.SecurityManager#checkLink(java.lang.String)
     */
    public static void loadLibrary(String libname) {
        jail.java.lang.Runtime.getRuntime().loadLibrary(libname);
    }

    /**
     * Maps a library name into a platform-specific string representing a native
     * library.
     *
     * @param libname
     *            the name of the library.
     * @return a platform-dependent native library name.
     * @exception NullPointerException
     *                if <code>libname</code> is <code>null</code>
     * @see java.lang.System#loadLibrary(java.lang.String)
     * @see java.lang.ClassLoader#findLibrary(java.lang.String)
     * @since 1.2
     */
    public static String mapLibraryName(String libname) {
        return libname;
    }

    /**
     * The following two methods exist because in, out, and err must be
     * initialized to null. The compiler, however, cannot be permitted to inline
     * access to them, since they are later set to more sensible values by
     * initializeSystemClass().
     */
    private static InputStream nullInputStream() throws NullPointerException {
        return new NullInputStream();
    }

    private static PrintStream nullPrintStream() throws NullPointerException {
        return new PrintStream(new NullOutputStream());
    }

    private static PrintStream logPrintStream(int source) {
        return new PrintStream(new LogOutputStream(source));
    }

    private static class NullInputStream extends InputStream {
        @Override
        public int read() throws IOException {
            return -1;
        }
    }

    private static class NullOutputStream extends OutputStream {
        @Override
        public void write(int b) throws IOException {
        }
    }
}
