/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2019 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package impl.java.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.security.AccessController;
import java.security.PrivilegedAction;

import blues.Log;

public class LocalFile extends FsFile {

    private static File tempDir;
    private static long counter = 0;

    private File location;
    private RandomAccessFile raf;

    public LocalFile(LocalDir parent, String name) {
        setPath(parent, name, null);
    }

    private static File createTempDir() {
        File tmp;
        try {
            tmp = File.createTempFile("blueslocal", null, null);
            tmp.delete();
            tmp.mkdirs();
            tmp.deleteOnExit();
        } catch (IOException e) {
            tmp = null;
        }
        return tmp;
    }

    private static void initTempDir() {
        if (tempDir == null) {
            tempDir = (File) AccessController.doPrivileged(new PrivilegedAction(){
                public Object run() {
                    return createTempDir();
                }
            });
        }
    }

    public static String getTempDir() {
        initTempDir();
        return tempDir.getAbsolutePath();
    }

    public static synchronized File getTempName(String name) {
        initTempDir();
        if (tempDir == null) {
            return null;
        }

        String suffix = ".tmp";
        int ldot = name.lastIndexOf('.');
        if (ldot > 0) {
            suffix = name.substring(ldot);
        }
        final String tname = Long.toHexString(counter++);
        StringBuilder sb = new StringBuilder(8 + tname.length() + 4);
        if (tname.length() < 8) {
            for (int i = 0; i < (8 - tname.length()); i++) {
                sb.append('0');
            }
        }
        sb.append(tname);
        sb.append(suffix);

        File f = new File(tempDir, sb.toString());
        f.deleteOnExit();
        return f;
    }

    private boolean open() {
        if (raf != null)
            return true;
        if (location == null) {
            location = getTempName(getName());
            if (location == null) {
                return false;
            }
        }
        raf = (RandomAccessFile) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                try {
                    return new RandomAccessFile(location, "rw");
                } catch (FileNotFoundException e) {
                    return null;
                }
            }
        });

        return (raf != null);
    }

    @Override
    public int readBytes(long offset, byte[] b, int off, int len) {
        if (open() == false)
            return -1;
        int r;
        try {
            raf.seek(offset);
            r = raf.read(b, off, len);
        } catch (IOException e) {
            return -2;
        }
        return r;
    }

    @Override
    public int writeBytes(long offset, byte[] b, int off, int len) {
        if (open() == false)
            return -1;
        try {
            raf.seek(offset);
            raf.write(b, off, len);
            length = raf.length();
        } catch (IOException e) {
            return -2;
        }
        return len;
    }

    @Override
    protected void lastReferenceClosed() {
        if (raf != null) {
            try {
                raf.close();
            } catch (IOException e) {
            }
            raf = null;
        }
    }

    @Override
    protected void populateAttributes() {
        accessFlags = ACCESS_READ | ACCESS_WRITE;
        lastModifiedTime = glue.java.lang.System.currentTimeMillis();
    }

    public String getLocalPath() {
        return location.getPath();
    }

    @Override
    public void cleanup() {
        lastReferenceClosed();
        if (location!=null) {
            location.delete();
        }
    }

    public static void cleanupTempDir() {
        if (tempDir==null) return;
        Log.log(Log.LOG_FILE, "Cleaning up local temp dir ",tempDir.getAbsolutePath());

        File[] files = tempDir.listFiles();
        if (files!=null) {
            for (File f : files) {
                f = f.getAbsoluteFile();
                if (f.delete()==false) {
                    Log.log(Log.LOG_FILE,"Unable to delete local file ",f.getAbsolutePath());
                }
            }
        }
        if (tempDir.delete()==false) {
            Log.log(Log.LOG_FILE,"Unable to delete local temp dir ",tempDir.getAbsolutePath());
        }
        tempDir = null;
    }

}
