/* Copyright (c) 2006 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

/* $Header: /code/cbmfs/cbmdisk.hpp,v 1.5 2006/07/21 14:46:04 doj Exp $ */

#ifndef CBMDISK__HPP
#define CBMDISK__HPP

#include "cbmfs.hpp"

class CBMdisk
{
public:

#pragma pack(push, 1)

  struct TS {
    BYTE track;
    BYTE sector;
    TS(BYTE t, BYTE s) : track(t), sector(s) { }
    bool operator == (const TS& r) const { return track==r.track && sector==r.sector; }
    void clear() { track=sector=0; }
  };
  BOOST_STATIC_ASSERT(sizeof(TS) == 2);

  struct direntry {
    TS next;
    BYTE FileType;
    TS file;
    BYTE filename[16];
    TS REL;
    union {
      BYTE RELlength;
      BYTE geos_file_structure;
    };
    BYTE geos_filetype;
    BYTE geos_year;
    BYTE geos_month;
    BYTE geos_day;
    BYTE geos_hour;
    BYTE geos_minute;
    BYTE totalSectorsLo;
    BYTE totalSectorsHi;
  };
  BOOST_STATIC_ASSERT(sizeof(direntry) == 32);

  struct dirsector {
    direntry entry[8];
  };
  BOOST_STATIC_ASSERT(sizeof(dirsector) == SECTOR_SIZE);

  struct bamentry {
    BYTE freeblocks;
    BYTE bm0, bm1, bm2;
  };
  BOOST_STATIC_ASSERT(sizeof(bamentry) == 4);

  struct filesector {
    TS next;
    BYTE data[SECTOR_DATA_SIZE];
  };
  BOOST_STATIC_ASSERT(sizeof(filesector) == SECTOR_SIZE);

#pragma pack(pop)

  CBMdisk(BYTE *disk_);
  virtual TS dir() = 0;
  filesector* getsector(TS);
  filesector* newsector();
  const int fileLength(const TS&);
  direntry* get_direntry(const char *filename);
  direntry* new_direntry();
  int filenum();
  int freeblocks();
  virtual int maxblocks() = 0;
  virtual void bam_set(const BYTE t, const BYTE s, const bool b) = 0;
  virtual void printbam() = 0;
  void printdir();
  void printsectors();
  virtual int track_max() = 0;

  filesector* getsector(const BYTE t, const BYTE s) { return getsector(TS(t,s)); }
  dirsector* getdirsector(const TS& ts) { return reinterpret_cast<dirsector*>(getsector(ts)); }
  void bam_set(TS& ts, const bool b) { this->bam_set(ts.track, ts.sector, b); }

protected:
  virtual const int* sectorsPerTrack() = 0;
  virtual bool bam_get(const BYTE t, const BYTE s) = 0;
  bool bam_get(const TS& ts) { return this->bam_get(ts.track, ts.sector); }
  virtual void fix(bamentry*, const BYTE t) = 0;
  virtual bamentry* getbamentry(const BYTE t) = 0;
  virtual bool reserved_track(const BYTE t) = 0;
  void printbamblocks();

  BYTE *disk;
};

extern CBMdisk *disk;

#endif
