/* Copyright (c) 2006 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

/* $Header: /code/cbmfs/d80.cpp,v 1.3 2006/07/21 15:05:58 doj Exp $ */

#include "d80.hpp"
#include <stdio.h>

D80::D80(BYTE *disk_) :
  CBMdisk(disk_)
{
  this->bam_set(38,0,0);
  this->bam_set(38,3,0);
  this->bam_set(39,0,0);
  this->bam_set(39,1,0);
}

CBMdisk::TS D80::dir() { return TS(39,1); }
int D80::maxblocks() { return 2083; }
int D80::track_max() { return 77; }

const int* D80::sectorsPerTrack()
{
  static const int t[] = {
    -1,

    29,29,29,29,29,29,29,29,29,29,
    29,29,29,29,29,29,29,29,29,29,
    29,29,29,29,29,29,29,29,29,29,
    29,29,29,29,29,29,29,29,29,
    27,27,27,27,27,27,27,27,27,27,
    27,27,27,27,
    25,25,25,25,25,25,25,25,25,25,25,
    23,23,23,23,23,23,23,23,23,23,
    23,23,23,

    29,29,29,29,29,29,29,29,29,29,
    29,29,29,29,29,29,29,29,29,29,
    29,29,29,29,29,29,29,29,29,29,
    29,29,29,29,29,29,29,29,29,
    27,27,27,27,27,27,27,27,27,27,
    27,27,27,27,
    25,25,25,25,25,25,25,25,25,25,25,
    23,23,23,23,23,23,23,23,23,23,
    23,23,23,
  };
  return t;
}

void D80::printbam()
{
  printf("BAM: D80 77 track image\n");
  this->printbaminfo();
  this->printbamblocks();
}

void D80::printbaminfo()
{
  const header *HEADER=reinterpret_cast<header*>(getsector(39,0));
  printf("bam=%i,%i "
	 "DOSformat=%c "
	 "reserved=%02X "
	 "unused1=0x%02X,0x%02X "
	 "unused2=0x%02X,0x%02X "
	 "unused3=0x%02X "
	 "DOSversion=%c%c "
	 "unused4=0x%02X,0x%02X,0x%02X,0x%02X,\n",
	 HEADER->bam.track, HEADER->bam.sector,
	 HEADER->DOSformat,
	 HEADER->reserved,
	 HEADER->unused1[0],HEADER->unused1[1],
	 HEADER->unused2[0],HEADER->unused2[1],
	 HEADER->unused3,
	 HEADER->DOSversion[0],HEADER->DOSversion[1],
	 HEADER->unused4[0],HEADER->unused4[1],HEADER->unused4[2],HEADER->unused4[3]
	 );
}

D80::bamentry_80* D80::getbamentry_80(const BYTE t)
{
  if(t==0)
    return NULL;
  if(t>this->track_max())
    return NULL;

  if(t<=50)
    {
      bam_80 *BAM=reinterpret_cast<bam_80*>(getsector(38,0));
      if(!BAM) return NULL;
      return &(BAM->entry[t-1]);
    }
  if(t<=100)
    {
      bam_80 *BAM=reinterpret_cast<bam_80*>(getsector(38,3));
      if(!BAM) return NULL;
      return &(BAM->entry[t-51]);
    }
  if(t<=150)
    {
      bam_80 *BAM=reinterpret_cast<bam_80*>(getsector(38,6));
      if(!BAM) return NULL;
      return &(BAM->entry[t-101]);
    }
  if(t<=200)
    {
      bam_80 *BAM=reinterpret_cast<bam_80*>(getsector(38,9));
      if(!BAM) return NULL;
      return &(BAM->entry[t-151]);
    }

  return NULL;
}

bool D80::bam_get(const BYTE t, const BYTE s)
{
  const bamentry_80 *e=this->getbamentry_80(t);
  if(!e) return false;

  if(s<8)
    return get_bit(e->bm0, s);
  if(s<16)
    return get_bit(e->bm1, s-8);
  if(s<24)
    return get_bit(e->bm2, s-16);
  if(s<32)
    return get_bit(e->bm3, s-24);

  return false;
}

void D80::bam_set(const BYTE t, const BYTE s, const bool b)
{
  bamentry_80 *e=this->getbamentry_80(t);
  if(!e)
    return;

  if(s<8)
    set_bit(e->bm0, s, b);
  else if(s<16)
    set_bit(e->bm1, s-8, b);
  else if(s<24)
    set_bit(e->bm2, s-16, b);
  else if(s<32)
    set_bit(e->bm2, s-24, b);
  else
    return;

  this->fix(e, t);
}

void D80::fix(bamentry_80 *e, const BYTE t)
{
  fuse_assert(e);
  if(t==0 || t>this->track_max())
    return;

  const int sectors=this->sectorsPerTrack()[t];

  e->freeblocks=0;
  const uint32_t b=e->bm0 | (e->bm1 << 8) | (e->bm2 << 16) | (e->bm3 << 24);
  for(int i=0; i<sectors; i++)
    if(b&(1<<i))
      e->freeblocks++;
}

bool D80::reserved_track(const BYTE t) { return t==39; }

void D80::fix(bamentry*, const BYTE t)
{
  syslog(LOG_ERR, "D80::fix(): this function should not be called.\n");
}

CBMdisk::bamentry* D80::getbamentry(const BYTE t)
{
  return reinterpret_cast<bamentry*>(getbamentry_80(t));
}
