/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.runtime;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.connector.Connector;
import org.apache.kafka.connect.connector.ConnectorContext;
import org.apache.kafka.connect.connector.Task;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.runtime.ConnectorConfig;
import org.apache.kafka.connect.runtime.ConnectorStatus;
import org.apache.kafka.connect.runtime.SourceTaskOffsetCommitter;
import org.apache.kafka.connect.runtime.TargetState;
import org.apache.kafka.connect.runtime.TaskConfig;
import org.apache.kafka.connect.runtime.TaskStatus;
import org.apache.kafka.connect.runtime.TransformationChain;
import org.apache.kafka.connect.runtime.WorkerConfig;
import org.apache.kafka.connect.runtime.WorkerConnector;
import org.apache.kafka.connect.runtime.WorkerSinkTask;
import org.apache.kafka.connect.runtime.WorkerSourceTask;
import org.apache.kafka.connect.runtime.WorkerTask;
import org.apache.kafka.connect.runtime.isolation.Plugins;
import org.apache.kafka.connect.sink.SinkRecord;
import org.apache.kafka.connect.sink.SinkTask;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.apache.kafka.connect.storage.Converter;
import org.apache.kafka.connect.storage.OffsetBackingStore;
import org.apache.kafka.connect.storage.OffsetStorageReaderImpl;
import org.apache.kafka.connect.storage.OffsetStorageWriter;
import org.apache.kafka.connect.util.ConnectorTaskId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Worker {
    private static final Logger log = LoggerFactory.getLogger(Worker.class);
    private final ExecutorService executor;
    private final Time time;
    private final String workerId;
    private final Plugins plugins;
    private final WorkerConfig config;
    private final Converter defaultKeyConverter;
    private final Converter defaultValueConverter;
    private final Converter internalKeyConverter;
    private final Converter internalValueConverter;
    private final OffsetBackingStore offsetBackingStore;
    private final Map<String, Object> producerProps;
    private final ConcurrentMap<String, WorkerConnector> connectors = new ConcurrentHashMap<String, WorkerConnector>();
    private final ConcurrentMap<ConnectorTaskId, WorkerTask> tasks = new ConcurrentHashMap<ConnectorTaskId, WorkerTask>();
    private SourceTaskOffsetCommitter sourceTaskOffsetCommitter;

    public Worker(String workerId, Time time, Plugins plugins, WorkerConfig config, OffsetBackingStore offsetBackingStore) {
        this.executor = Executors.newCachedThreadPool();
        this.workerId = workerId;
        this.time = time;
        this.plugins = plugins;
        this.config = config;
        this.defaultKeyConverter = plugins.newConverter(config.getClass("key.converter").getName(), config);
        this.defaultKeyConverter.configure(config.originalsWithPrefix("key.converter."), true);
        this.defaultValueConverter = plugins.newConverter(config.getClass("value.converter").getName(), config);
        this.defaultValueConverter.configure(config.originalsWithPrefix("value.converter."), false);
        this.internalKeyConverter = plugins.newConverter(config.getClass("internal.key.converter").getName(), config);
        this.internalKeyConverter.configure(config.originalsWithPrefix("internal.key.converter."), true);
        this.internalValueConverter = plugins.newConverter(config.getClass("internal.value.converter").getName(), config);
        this.internalValueConverter.configure(config.originalsWithPrefix("internal.value.converter."), false);
        this.offsetBackingStore = offsetBackingStore;
        this.offsetBackingStore.configure(config);
        this.producerProps = new HashMap<String, Object>();
        this.producerProps.put("bootstrap.servers", Utils.join((Collection)config.getList("bootstrap.servers"), (String)","));
        this.producerProps.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
        this.producerProps.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
        this.producerProps.put("request.timeout.ms", Integer.toString(Integer.MAX_VALUE));
        this.producerProps.put("retries", Integer.toString(Integer.MAX_VALUE));
        this.producerProps.put("max.block.ms", Long.toString(Long.MAX_VALUE));
        this.producerProps.put("acks", "all");
        this.producerProps.put("max.in.flight.requests.per.connection", "1");
        this.producerProps.putAll(config.originalsWithPrefix("producer."));
    }

    public void start() {
        log.info("Worker starting");
        this.offsetBackingStore.start();
        this.sourceTaskOffsetCommitter = new SourceTaskOffsetCommitter(this.config);
        log.info("Worker started");
    }

    public void stop() {
        log.info("Worker stopping");
        long started = this.time.milliseconds();
        long limit = started + this.config.getLong("task.shutdown.graceful.timeout.ms");
        if (!this.connectors.isEmpty()) {
            log.warn("Shutting down connectors {} uncleanly; herder should have shut down connectors before the Worker is stopped", this.connectors.keySet());
            this.stopConnectors();
        }
        if (!this.tasks.isEmpty()) {
            log.warn("Shutting down tasks {} uncleanly; herder should have shut down tasks before the Worker is stopped", this.tasks.keySet());
            this.stopAndAwaitTasks();
        }
        long timeoutMs = limit - this.time.milliseconds();
        this.sourceTaskOffsetCommitter.close(timeoutMs);
        this.offsetBackingStore.stop();
        log.info("Worker stopped");
    }

    public boolean startConnector(String connName, Map<String, String> connProps, ConnectorContext ctx, ConnectorStatus.Listener statusListener, TargetState initialState) {
        WorkerConnector workerConnector;
        if (this.connectors.containsKey(connName)) {
            throw new ConnectException("Connector with name " + connName + " already exists");
        }
        ClassLoader savedLoader = this.plugins.currentThreadLoader();
        try {
            ConnectorConfig connConfig = new ConnectorConfig(this.plugins, connProps);
            String connClass = connConfig.getString("connector.class");
            log.info("Creating connector {} of type {}", (Object)connName, (Object)connClass);
            Connector connector = this.plugins.newConnector(connClass);
            workerConnector = new WorkerConnector(connName, connector, ctx, statusListener);
            log.info("Instantiated connector {} with version {} of type {}", new Object[]{connName, connector.version(), connector.getClass()});
            savedLoader = this.plugins.compareAndSwapLoaders(connector);
            workerConnector.initialize(connConfig);
            workerConnector.transitionTo(initialState);
            Plugins.compareAndSwapLoaders(savedLoader);
        }
        catch (Throwable t) {
            log.error("Failed to start connector {}", (Object)connName, (Object)t);
            Plugins.compareAndSwapLoaders(savedLoader);
            statusListener.onFailure(connName, t);
            return false;
        }
        WorkerConnector existing = this.connectors.putIfAbsent(connName, workerConnector);
        if (existing != null) {
            throw new ConnectException("Connector with name " + connName + " already exists");
        }
        log.info("Finished creating connector {}", (Object)connName);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isSinkConnector(String connName) {
        WorkerConnector workerConnector = (WorkerConnector)this.connectors.get(connName);
        if (workerConnector == null) {
            throw new ConnectException("Connector " + connName + " not found in this worker.");
        }
        ClassLoader savedLoader = this.plugins.currentThreadLoader();
        try {
            savedLoader = this.plugins.compareAndSwapLoaders(workerConnector.connector());
            boolean bl = workerConnector.isSinkConnector();
            return bl;
        }
        finally {
            Plugins.compareAndSwapLoaders(savedLoader);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<Map<String, String>> connectorTaskConfigs(String connName, int maxTasks, List<String> sinkTopics) {
        log.trace("Reconfiguring connector tasks for {}", (Object)connName);
        WorkerConnector workerConnector = (WorkerConnector)this.connectors.get(connName);
        if (workerConnector == null) {
            throw new ConnectException("Connector " + connName + " not found in this worker.");
        }
        Connector connector = workerConnector.connector();
        ArrayList<Map<String, String>> result = new ArrayList<Map<String, String>>();
        ClassLoader savedLoader = this.plugins.currentThreadLoader();
        try {
            savedLoader = this.plugins.compareAndSwapLoaders(connector);
            String taskClassName = connector.taskClass().getName();
            for (Map taskProps : connector.taskConfigs(maxTasks)) {
                HashMap<String, String> taskConfig = new HashMap<String, String>(taskProps);
                taskConfig.put("task.class", taskClassName);
                if (sinkTopics != null) {
                    taskConfig.put("topics", Utils.join(sinkTopics, (String)","));
                }
                result.add(taskConfig);
            }
        }
        finally {
            Plugins.compareAndSwapLoaders(savedLoader);
        }
        return result;
    }

    private void stopConnectors() {
        for (String connector : this.connectors.keySet()) {
            this.stopConnector(connector);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean stopConnector(String connName) {
        log.info("Stopping connector {}", (Object)connName);
        WorkerConnector workerConnector = (WorkerConnector)this.connectors.remove(connName);
        if (workerConnector == null) {
            log.warn("Ignoring stop request for unowned connector {}", (Object)connName);
            return false;
        }
        ClassLoader savedLoader = this.plugins.currentThreadLoader();
        try {
            savedLoader = this.plugins.compareAndSwapLoaders(workerConnector.connector());
            workerConnector.shutdown();
        }
        finally {
            Plugins.compareAndSwapLoaders(savedLoader);
        }
        log.info("Stopped connector {}", (Object)connName);
        return true;
    }

    public Set<String> connectorNames() {
        return this.connectors.keySet();
    }

    public boolean isRunning(String connName) {
        WorkerConnector workerConnector = (WorkerConnector)this.connectors.get(connName);
        return workerConnector != null && workerConnector.isRunning();
    }

    public boolean startTask(ConnectorTaskId id, Map<String, String> connProps, Map<String, String> taskProps, TaskStatus.Listener statusListener, TargetState initialState) {
        WorkerTask workerTask;
        log.info("Creating task {}", (Object)id);
        if (this.tasks.containsKey(id)) {
            throw new ConnectException("Task already exists in this worker: " + id);
        }
        ClassLoader savedLoader = this.plugins.currentThreadLoader();
        try {
            ConnectorConfig connConfig = new ConnectorConfig(this.plugins, connProps);
            String connType = connConfig.getString("connector.class");
            ClassLoader connectorLoader = this.plugins.delegatingLoader().connectorLoader(connType);
            savedLoader = Plugins.compareAndSwapLoaders(connectorLoader);
            TaskConfig taskConfig = new TaskConfig(taskProps);
            Class<Task> taskClass = taskConfig.getClass("task.class").asSubclass(Task.class);
            Task task = this.plugins.newTask(taskClass);
            log.info("Instantiated task {} with version {} of type {}", new Object[]{id, task.version(), taskClass.getName()});
            Converter keyConverter = (Converter)connConfig.getConfiguredInstance("key.converter", Converter.class);
            if (keyConverter != null) {
                keyConverter.configure(connConfig.originalsWithPrefix("key.converter."), true);
            } else {
                keyConverter = this.defaultKeyConverter;
            }
            Converter valueConverter = (Converter)connConfig.getConfiguredInstance("value.converter", Converter.class);
            if (valueConverter != null) {
                valueConverter.configure(connConfig.originalsWithPrefix("value.converter."), false);
            } else {
                valueConverter = this.defaultValueConverter;
            }
            workerTask = this.buildWorkerTask(connConfig, id, task, statusListener, initialState, keyConverter, valueConverter, connectorLoader);
            workerTask.initialize(taskConfig);
            Plugins.compareAndSwapLoaders(savedLoader);
        }
        catch (Throwable t) {
            log.error("Failed to start task {}", (Object)id, (Object)t);
            Plugins.compareAndSwapLoaders(savedLoader);
            statusListener.onFailure(id, t);
            return false;
        }
        WorkerTask existing = this.tasks.putIfAbsent(id, workerTask);
        if (existing != null) {
            throw new ConnectException("Task already exists in this worker: " + id);
        }
        this.executor.submit(workerTask);
        if (workerTask instanceof WorkerSourceTask) {
            this.sourceTaskOffsetCommitter.schedule(id, (WorkerSourceTask)workerTask);
        }
        return true;
    }

    private WorkerTask buildWorkerTask(ConnectorConfig connConfig, ConnectorTaskId id, Task task, TaskStatus.Listener statusListener, TargetState initialState, Converter keyConverter, Converter valueConverter, ClassLoader loader) {
        if (task instanceof SourceTask) {
            TransformationChain<SourceRecord> transformationChain = new TransformationChain<SourceRecord>(connConfig.transformations());
            OffsetStorageReaderImpl offsetReader = new OffsetStorageReaderImpl(this.offsetBackingStore, id.connector(), this.internalKeyConverter, this.internalValueConverter);
            OffsetStorageWriter offsetWriter = new OffsetStorageWriter(this.offsetBackingStore, id.connector(), this.internalKeyConverter, this.internalValueConverter);
            KafkaProducer producer = new KafkaProducer(this.producerProps);
            return new WorkerSourceTask(id, (SourceTask)task, statusListener, initialState, keyConverter, valueConverter, transformationChain, (KafkaProducer<byte[], byte[]>)producer, offsetReader, offsetWriter, this.config, loader, this.time);
        }
        if (task instanceof SinkTask) {
            TransformationChain<SinkRecord> transformationChain = new TransformationChain<SinkRecord>(connConfig.transformations());
            return new WorkerSinkTask(id, (SinkTask)task, statusListener, initialState, this.config, keyConverter, valueConverter, transformationChain, loader, this.time);
        }
        log.error("Tasks must be a subclass of either SourceTask or SinkTask", (Object)task);
        throw new ConnectException("Tasks must be a subclass of either SourceTask or SinkTask");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void stopTask(ConnectorTaskId taskId) {
        WorkerTask task = (WorkerTask)this.tasks.get(taskId);
        if (task == null) {
            log.warn("Ignoring stop request for unowned task {}", (Object)taskId);
            return;
        }
        log.info("Stopping task {}", (Object)task.id());
        if (task instanceof WorkerSourceTask) {
            this.sourceTaskOffsetCommitter.remove(task.id());
        }
        ClassLoader savedLoader = this.plugins.currentThreadLoader();
        try {
            savedLoader = Plugins.compareAndSwapLoaders(task.loader());
            task.stop();
        }
        finally {
            Plugins.compareAndSwapLoaders(savedLoader);
        }
    }

    private void stopTasks(Collection<ConnectorTaskId> ids) {
        for (ConnectorTaskId taskId : ids) {
            this.stopTask(taskId);
        }
    }

    private void awaitStopTask(ConnectorTaskId taskId, long timeout) {
        WorkerTask task = (WorkerTask)this.tasks.remove(taskId);
        if (task == null) {
            log.warn("Ignoring await stop request for non-present task {}", (Object)taskId);
            return;
        }
        if (!task.awaitStop(timeout)) {
            log.error("Graceful stop of task {} failed.", (Object)task.id());
            task.cancel();
        }
    }

    private void awaitStopTasks(Collection<ConnectorTaskId> ids) {
        long now = this.time.milliseconds();
        long deadline = now + this.config.getLong("task.shutdown.graceful.timeout.ms");
        for (ConnectorTaskId id : ids) {
            long remaining = Math.max(0L, deadline - this.time.milliseconds());
            this.awaitStopTask(id, remaining);
        }
    }

    public void stopAndAwaitTasks() {
        this.stopAndAwaitTasks(new ArrayList<ConnectorTaskId>(this.tasks.keySet()));
    }

    public void stopAndAwaitTasks(Collection<ConnectorTaskId> ids) {
        this.stopTasks(ids);
        this.awaitStopTasks(ids);
    }

    public void stopAndAwaitTask(ConnectorTaskId taskId) {
        this.stopTask(taskId);
        this.awaitStopTasks(Collections.singletonList(taskId));
    }

    public Set<ConnectorTaskId> taskIds() {
        return this.tasks.keySet();
    }

    public Converter getInternalKeyConverter() {
        return this.internalKeyConverter;
    }

    public Converter getInternalValueConverter() {
        return this.internalValueConverter;
    }

    public Plugins getPlugins() {
        return this.plugins;
    }

    public String workerId() {
        return this.workerId;
    }

    public void setTargetState(String connName, TargetState state) {
        log.info("Setting connector {} state to {}", (Object)connName, (Object)state);
        WorkerConnector workerConnector = (WorkerConnector)this.connectors.get(connName);
        if (workerConnector != null) {
            ClassLoader connectorLoader = this.plugins.delegatingLoader().connectorLoader(workerConnector.connector());
            this.transitionTo(workerConnector, state, connectorLoader);
        }
        for (Map.Entry taskEntry : this.tasks.entrySet()) {
            if (!((ConnectorTaskId)taskEntry.getKey()).connector().equals(connName)) continue;
            WorkerTask workerTask = (WorkerTask)taskEntry.getValue();
            this.transitionTo(workerTask, state, workerTask.loader());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void transitionTo(Object connectorOrTask, TargetState state, ClassLoader loader) {
        block5: {
            ClassLoader savedLoader = this.plugins.currentThreadLoader();
            try {
                savedLoader = Plugins.compareAndSwapLoaders(loader);
                if (connectorOrTask instanceof WorkerConnector) {
                    ((WorkerConnector)connectorOrTask).transitionTo(state);
                    break block5;
                }
                if (connectorOrTask instanceof WorkerTask) {
                    ((WorkerTask)connectorOrTask).transitionTo(state);
                    break block5;
                }
                throw new ConnectException("Request for state transition on an object that is neither a WorkerConnector nor a WorkerTask: " + connectorOrTask.getClass());
            }
            finally {
                Plugins.compareAndSwapLoaders(savedLoader);
            }
        }
    }
}

