/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tools;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import net.sourceforge.argparse4j.ArgumentParsers;
import net.sourceforge.argparse4j.impl.Arguments;
import net.sourceforge.argparse4j.inf.ArgumentAction;
import net.sourceforge.argparse4j.inf.ArgumentParser;
import net.sourceforge.argparse4j.inf.Namespace;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.OutOfOrderSequenceException;
import org.apache.kafka.common.errors.ProducerFencedException;

public class TransactionalMessageCopier {
    private static ArgumentParser argParser() {
        ArgumentParser parser = ArgumentParsers.newArgumentParser((String)"transactional-message-copier").defaultHelp(true).description("This tool copies messages transactionally from an input partition to an output topic, committing the consumed offsets along with the output messages");
        parser.addArgument(new String[]{"--input-topic"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"INPUT-TOPIC"}).dest("inputTopic").help("Consume messages from this topic");
        parser.addArgument(new String[]{"--input-partition"}).action((ArgumentAction)Arguments.store()).required(true).type(Integer.class).metavar(new String[]{"INPUT-PARTITION"}).dest("inputPartition").help("Consume messages from this partition of the input topic.");
        parser.addArgument(new String[]{"--output-topic"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"OUTPUT-TOPIC"}).dest("outputTopic").help("Produce messages to this topic");
        parser.addArgument(new String[]{"--broker-list"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"HOST1:PORT1[,HOST2:PORT2[...]]"}).dest("brokerList").help("Comma-separated list of Kafka brokers in the form HOST1:PORT1,HOST2:PORT2,...");
        parser.addArgument(new String[]{"--max-messages"}).action((ArgumentAction)Arguments.store()).required(false).setDefault((Object)-1).type(Integer.class).metavar(new String[]{"MAX-MESSAGES"}).dest("maxMessages").help("Process these many messages upto the end offset at the time this program was launched. If set to -1 we will just read to the end offset of the input partition (as of the time the program was launched).");
        parser.addArgument(new String[]{"--consumer-group"}).action((ArgumentAction)Arguments.store()).required(false).setDefault((Object)-1).type(String.class).metavar(new String[]{"CONSUMER-GROUP"}).dest("consumerGroup").help("The consumer group id to use for storing the consumer offsets.");
        parser.addArgument(new String[]{"--transaction-size"}).action((ArgumentAction)Arguments.store()).required(false).setDefault((Object)200).type(Integer.class).metavar(new String[]{"TRANSACTION-SIZE"}).dest("messagesPerTransaction").help("The number of messages to put in each transaction. Default is 200.");
        parser.addArgument(new String[]{"--transactional-id"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"TRANSACTIONAL-ID"}).dest("transactionalId").help("The transactionalId to assign to the producer");
        parser.addArgument(new String[]{"--enable-random-aborts"}).action((ArgumentAction)Arguments.storeTrue()).type(Boolean.class).metavar(new String[]{"ENABLE-RANDOM-ABORTS"}).dest("enableRandomAborts").help("Whether or not to enable random transaction aborts (for system testing)");
        return parser;
    }

    private static KafkaProducer<String, String> createProducer(Namespace parsedArgs) {
        String transactionalId = parsedArgs.getString("transactionalId");
        String brokerList = parsedArgs.getString("brokerList");
        Properties props = new Properties();
        props.put("bootstrap.servers", brokerList);
        props.put("transactional.id", transactionalId);
        props.put("key.serializer", "org.apache.kafka.common.serialization.StringSerializer");
        props.put("value.serializer", "org.apache.kafka.common.serialization.StringSerializer");
        return new KafkaProducer(props);
    }

    private static KafkaConsumer<String, String> createConsumer(Namespace parsedArgs) {
        String consumerGroup = parsedArgs.getString("consumerGroup");
        String brokerList = parsedArgs.getString("brokerList");
        Integer numMessagesPerTransaction = parsedArgs.getInt("messagesPerTransaction");
        Properties props = new Properties();
        props.put("group.id", consumerGroup);
        props.put("bootstrap.servers", brokerList);
        props.put("isolation.level", "read_committed");
        props.put("max.poll.records", numMessagesPerTransaction.toString());
        props.put("enable.auto.commit", "false");
        props.put("session.timeout.ms", "10000");
        props.put("heartbeat.interval.ms", "3000");
        props.put("auto.offset.reset", "earliest");
        props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        return new KafkaConsumer(props);
    }

    private static ProducerRecord<String, String> producerRecordFromConsumerRecord(String topic, ConsumerRecord<String, String> record) {
        return new ProducerRecord(topic, record.key(), record.value());
    }

    private static Map<TopicPartition, OffsetAndMetadata> consumerPositions(KafkaConsumer<String, String> consumer) {
        HashMap<TopicPartition, OffsetAndMetadata> positions = new HashMap<TopicPartition, OffsetAndMetadata>();
        for (TopicPartition topicPartition : consumer.assignment()) {
            positions.put(topicPartition, new OffsetAndMetadata(consumer.position(topicPartition), null));
        }
        return positions;
    }

    private static void resetToLastCommittedPositions(KafkaConsumer<String, String> consumer) {
        for (TopicPartition topicPartition : consumer.assignment()) {
            OffsetAndMetadata offsetAndMetadata = consumer.committed(topicPartition);
            if (offsetAndMetadata != null) {
                consumer.seek(topicPartition, offsetAndMetadata.offset());
                continue;
            }
            consumer.seekToBeginning(Collections.singleton(topicPartition));
        }
    }

    private static long messagesRemaining(KafkaConsumer<String, String> consumer, TopicPartition partition) {
        long currentPosition = consumer.position(partition);
        Map endOffsets = consumer.endOffsets(Collections.singleton(partition));
        if (endOffsets.containsKey(partition)) {
            return (Long)endOffsets.get(partition) - currentPosition;
        }
        return 0L;
    }

    private static String toJsonString(Map<String, Object> data) {
        String json;
        try {
            ObjectMapper mapper = new ObjectMapper();
            json = mapper.writeValueAsString(data);
        }
        catch (JsonProcessingException e) {
            json = "Bad data can't be written as json: " + e.getMessage();
        }
        return json;
    }

    private static String statusAsJson(long consumed, long remaining, String transactionalId) {
        HashMap<String, Object> statusData = new HashMap<String, Object>();
        statusData.put("progress", transactionalId);
        statusData.put("consumed", consumed);
        statusData.put("remaining", remaining);
        return TransactionalMessageCopier.toJsonString(statusData);
    }

    private static String shutDownString(long consumed, long remaining, String transactionalId) {
        HashMap<String, Object> shutdownData = new HashMap<String, Object>();
        shutdownData.put("remaining", remaining);
        shutdownData.put("consumed", consumed);
        shutdownData.put("shutdown_complete", transactionalId);
        return TransactionalMessageCopier.toJsonString(shutdownData);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws IOException {
        Namespace parsedArgs = TransactionalMessageCopier.argParser().parseArgsOrFail(args);
        Integer numMessagesPerTransaction = parsedArgs.getInt("messagesPerTransaction");
        final String transactionalId = parsedArgs.getString("transactionalId");
        String outputTopic = parsedArgs.getString("outputTopic");
        String consumerGroup = parsedArgs.getString("consumerGroup");
        TopicPartition inputPartition = new TopicPartition(parsedArgs.getString("inputTopic"), parsedArgs.getInt("inputPartition").intValue());
        final KafkaProducer<String, String> producer = TransactionalMessageCopier.createProducer(parsedArgs);
        final KafkaConsumer<String, String> consumer = TransactionalMessageCopier.createConsumer(parsedArgs);
        consumer.assign(Collections.singleton(inputPartition));
        long maxMessages = parsedArgs.getInt("maxMessages") == -1 ? Long.MAX_VALUE : (long)parsedArgs.getInt("maxMessages").intValue();
        maxMessages = Math.min(TransactionalMessageCopier.messagesRemaining(consumer, inputPartition), maxMessages);
        boolean enableRandomAborts = parsedArgs.getBoolean("enableRandomAborts");
        producer.initTransactions();
        final AtomicBoolean isShuttingDown = new AtomicBoolean(false);
        final AtomicLong remainingMessages = new AtomicLong(maxMessages);
        final AtomicLong numMessagesProcessed = new AtomicLong(0L);
        Runtime.getRuntime().addShutdownHook(new Thread(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                isShuttingDown.set(true);
                producer.close();
                KafkaConsumer kafkaConsumer = consumer;
                synchronized (kafkaConsumer) {
                    consumer.close();
                }
                System.out.println(TransactionalMessageCopier.shutDownString(numMessagesProcessed.get(), remainingMessages.get(), transactionalId));
            }
        });
        try {
            Random random = new Random();
            while (0L < remainingMessages.get()) {
                System.out.println(TransactionalMessageCopier.statusAsJson(numMessagesProcessed.get(), remainingMessages.get(), transactionalId));
                if (isShuttingDown.get()) {
                    break;
                }
                int messagesInCurrentTransaction = 0;
                long numMessagesForNextTransaction = Math.min((long)numMessagesPerTransaction.intValue(), remainingMessages.get());
                try {
                    producer.beginTransaction();
                    while ((long)messagesInCurrentTransaction < numMessagesForNextTransaction) {
                        ConsumerRecords records = consumer.poll(200L);
                        for (ConsumerRecord record : records) {
                            producer.send(TransactionalMessageCopier.producerRecordFromConsumerRecord(outputTopic, (ConsumerRecord<String, String>)record));
                            ++messagesInCurrentTransaction;
                        }
                    }
                    producer.sendOffsetsToTransaction(TransactionalMessageCopier.consumerPositions(consumer), consumerGroup);
                    if (enableRandomAborts && random.nextInt() % 3 == 0) {
                        throw new KafkaException("Aborting transaction");
                    }
                    producer.commitTransaction();
                    remainingMessages.set(maxMessages - numMessagesProcessed.addAndGet(messagesInCurrentTransaction));
                }
                catch (OutOfOrderSequenceException | ProducerFencedException e) {
                    throw e;
                }
                catch (KafkaException e) {
                    producer.abortTransaction();
                    TransactionalMessageCopier.resetToLastCommittedPositions(consumer);
                }
            }
        }
        finally {
            producer.close();
            KafkaConsumer<String, String> kafkaConsumer = consumer;
            synchronized (kafkaConsumer) {
                consumer.close();
            }
        }
        System.exit(0);
    }
}

