/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.state.internals;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.SimpleTimeZone;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.kafka.streams.errors.InvalidStateStoreException;
import org.apache.kafka.streams.errors.ProcessorStateException;
import org.apache.kafka.streams.processor.ProcessorContext;
import org.apache.kafka.streams.state.internals.Segment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Segments {
    private static final Logger log = LoggerFactory.getLogger(Segments.class);
    static final long MIN_SEGMENT_INTERVAL = 60000L;
    private final ConcurrentHashMap<Long, Segment> segments = new ConcurrentHashMap();
    private final String name;
    private final int numSegments;
    private final long segmentInterval;
    private final SimpleDateFormat formatter;
    private long minSegmentId = Long.MAX_VALUE;
    private long maxSegmentId = -1L;

    static long segmentInterval(long retentionPeriod, int numSegments) {
        return Math.max(retentionPeriod / (long)(numSegments - 1), 60000L);
    }

    Segments(String name, long retentionPeriod, int numSegments) {
        this.name = name;
        this.numSegments = numSegments;
        this.segmentInterval = Segments.segmentInterval(retentionPeriod, numSegments);
        this.formatter = new SimpleDateFormat("yyyyMMddHHmm");
        this.formatter.setTimeZone(new SimpleTimeZone(0, "UTC"));
    }

    long segmentId(long timestamp) {
        return timestamp / this.segmentInterval;
    }

    String segmentName(long segmentId) {
        return this.name + "-" + this.formatter.format(new Date(segmentId * this.segmentInterval));
    }

    Segment getSegmentForTimestamp(long timestamp) {
        return this.getSegment(this.segmentId(timestamp));
    }

    Segment getOrCreateSegment(long segmentId, ProcessorContext context) {
        if (segmentId > this.maxSegmentId - (long)this.numSegments) {
            long key = segmentId % (long)this.numSegments;
            Segment segment = this.segments.get(key);
            if (!this.isSegment(segment, segmentId)) {
                this.cleanup(segmentId);
            }
            Segment newSegment = new Segment(this.segmentName(segmentId), this.name, segmentId);
            Segment previousSegment = this.segments.putIfAbsent(key, newSegment);
            if (previousSegment == null) {
                newSegment.openDB(context);
                long l = this.maxSegmentId = segmentId > this.maxSegmentId ? segmentId : this.maxSegmentId;
                if (this.minSegmentId == Long.MAX_VALUE) {
                    this.minSegmentId = this.maxSegmentId;
                }
            }
            return previousSegment == null ? newSegment : previousSegment;
        }
        return null;
    }

    void openExisting(ProcessorContext context) {
        try {
            File dir = new File(context.stateDir(), this.name);
            if (dir.exists()) {
                String[] list = dir.list();
                if (list != null) {
                    long[] segmentIds = new long[list.length];
                    for (int i = 0; i < list.length; ++i) {
                        segmentIds[i] = this.segmentIdFromSegmentName(list[i]);
                    }
                    Arrays.sort(segmentIds);
                    for (long segmentId : segmentIds) {
                        if (segmentId < 0L) continue;
                        this.getOrCreateSegment(segmentId, context);
                    }
                }
            } else if (!dir.mkdir()) {
                throw new ProcessorStateException(String.format("dir %s doesn't exist and cannot be created for segments %s", dir, this.name));
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    List<Segment> segments(long timeFrom, long timeTo) {
        long segFrom = Math.max(this.minSegmentId, this.segmentId(Math.max(0L, timeFrom)));
        long segTo = Math.min(this.maxSegmentId, this.segmentId(Math.min(this.maxSegmentId * this.segmentInterval, Math.max(0L, timeTo))));
        ArrayList<Segment> segments = new ArrayList<Segment>();
        for (long segmentId = segFrom; segmentId <= segTo; ++segmentId) {
            Segment segment = this.getSegment(segmentId);
            if (segment == null || !segment.isOpen()) continue;
            try {
                segments.add(segment);
                continue;
            }
            catch (InvalidStateStoreException invalidStateStoreException) {
                // empty catch block
            }
        }
        return segments;
    }

    void flush() {
        for (Segment segment : this.segments.values()) {
            segment.flush();
        }
    }

    public void close() {
        for (Segment segment : this.segments.values()) {
            segment.close();
        }
    }

    private Segment getSegment(long segmentId) {
        Segment segment = this.segments.get(segmentId % (long)this.numSegments);
        if (!this.isSegment(segment, segmentId)) {
            return null;
        }
        return segment;
    }

    private boolean isSegment(Segment store, long segmentId) {
        return store != null && store.id == segmentId;
    }

    private void cleanup(long segmentId) {
        long oldestSegmentId = this.maxSegmentId < segmentId ? segmentId - (long)this.numSegments : this.maxSegmentId - (long)this.numSegments;
        for (Map.Entry<Long, Segment> segmentEntry : this.segments.entrySet()) {
            Segment segment = segmentEntry.getValue();
            if (segment == null || segment.id > oldestSegmentId) continue;
            this.segments.remove(segmentEntry.getKey());
            segment.close();
            try {
                segment.destroy();
            }
            catch (IOException e) {
                log.error("Error destroying {}", (Object)segment, (Object)e);
            }
        }
        if (oldestSegmentId > this.minSegmentId) {
            this.minSegmentId = oldestSegmentId + 1L;
        }
    }

    private long segmentIdFromSegmentName(String segmentName) {
        try {
            Date date = this.formatter.parse(segmentName.substring(this.name.length() + 1));
            return date.getTime() / this.segmentInterval;
        }
        catch (Exception ex) {
            return -1L;
        }
    }
}

