#!/bin/bash

# sbopkglint test, must be sourced by sbopkglint (not run standalone).

# PKG, PRGNAM, VERSION, ARCH are set by sbopkglint. also the current
# directory is the root of the installed package tree.

########################################################################
# for noarch packages, do nothing.
# for everything else, make sure any ELF binaries/libraries match the
# ARCH, and that libs are in the correct directory (lib vs. lib64).

# warnings:
# if an i?86 package has any 64-bit ELF objects (libs or bins)
# if an x86_64 package has any 32-bit ELF objects (libs or bins)
# if an i?86 package has lib64 or usr/lib64 at all
# if an x86_64 package has 64-bit libs in lib or usr/lib

# note: sometimes files in /lib/firmware are ELF, and would cause
# false "wrong directory" warnings, so we exclude that dir from the
# search.

case "$ARCH" in
	noarch) ;; # ok, do nothing.
	i?86) WRONGDIR="lib64"; CPU="80386" ;;
	x86_64) WRONGDIR="lib"; CPU="x86-64" ;;
	*) warn "ARCH isn't noarch, i?86, or x86_64. don't know how to check binaries." ;;
esac

INWRONGDIR=""
WRONGARCH=""
NOTSTRIPPED=""
if [ -n "$WRONGDIR" ]; then
	find * -type f -print0 | \
		xargs -0 file -m /etc/file/magic/elf | \
		grep 'ELF.*\(executable\|shared object\)' > .tmp.$$

		while read line; do
			file="$( echo $line | cut -d: -f1 )"
			filetype="$( echo $line | cut -d: -f2 )"
			case "$file" in
				# 20220414 bkw: only check for libs directly in the dir.
				# this avoids e.g. lib/udev/<executable> and usr/lib/prgnam/plugins/*.so.
				# had to relax this check; it was too strict.
				$WRONGDIR/*/*|usr/$WRONGDIR/*/*) continue ;;
				$WRONGDIR/*|usr/$WRONGDIR/*)
					INWRONGDIR+="$file " ;;
			esac

			# 64-bit packages can contain 2 types of 32-bit binaries:
			# - statically linked.
			# - statified. very few of these exist, and we can't make
			#   them on 15.0 (statifier can't handle modern kernel/glibc
			#   and the author hasn't updated it).
			if [ "$ARCH" = "x86_64" ]; then
				echo "$filetype" | grep -q 'statically linked' && continue
				grep -q DL_RO_DYN_TEMP_CNT "$file" && continue
			fi

			# "no machine" ELF objects are allowed, but since they still come
			# in 64-bit and 32-bit varieties, they must be in the correct
			# directory.
			if ! echo "$filetype" | grep -q -e "$CPU" -e 'no machine'; then
				WRONGARCH+="$file "
			fi

			# don't check "no machine" ELF objects for being stripped.
			# our strip command doesn't know how to strip them!
			if ! echo "$filetype" | grep -q 'no machine'; then
				if echo "$filetype" | grep -q "not stripped"; then
					NOTSTRIPPED+="$file "
				fi
			fi
		done < .tmp.$$
		rm -f .tmp.$$
fi

[ -n "$INWRONGDIR" ] && warn "shared lib(s) in wrong dir for ARCH:" && ls -l $INWRONGDIR
[ -n "$WRONGARCH" ] && warn "ELF object(s) with wrong arch (should be $CPU):" && ls -l $WRONGARCH
[ -n "$NOTSTRIPPED" ] && warn "ELF object(s) not stripped:" && ls -l $NOTSTRIPPED
