/*
 * Copyright (C) 2004-2014
 *	Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2015/11/26  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/01/07  Tommy Scheunemann <net@arrishq.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Functions needed for producing a Equinox directory hierarchy
 * configuration
 *
 * In this case the rootmenu is the top level directory
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <strings.h>

#include "wmconfig.h"
#include "prototypes.h"
#include "package.h"

#if (defined(__unix__) || defined(unix)) && !defined(USG)
#include <limits.h>
#include <sys/param.h>
#endif

#define EDE_DIR		".ede/programs/wmconfig"
#define EDE_MENU_ICON	"gnome-folder.png"

extern const char *root_menu_name;
extern unsigned int flags;

static void create_desktop(const char *file, struct package *app)
{
    int fd;
    FILE *f;

    if ((app == (struct package *)NULL) || (app->restart)) {
	return;
    }

    /* Remove the ending & */
    if (app->exec != NULL) {
	char *p;
	p = strrchr(app->exec, '&');
	if (p != NULL)
	    *p = '\0';
    }

    /*  At the very least we need a filename and command to execute */
    if (file == NULL || app->exec == NULL) {
	return;
    }

    fd = open (file, O_CREAT | O_EXCL | O_RDWR, 0600);

    f = fdopen(fd, "wx");
    if (f == (FILE *) NULL) {
	fprintf(stderr, gettext("Could not create file %s\n"), file);
	perror("Error code");
	return;
    }

    /*  Write the .desktop file:  */
    fprintf(f, "[Desktop Entry]\n");
    if (app->name != NULL) {
	fprintf(f, "Name=%s\n", app->name);
    }
    if (app->icon != NULL) {
	fprintf(f, "Icon=%s\n", app->icon);
    }
    fprintf(f, "Exec=%s\n", app->exec);
    if (app->terminal != NULL) {
	fprintf(f, "Terminal=1\n");
    }
    fclose(f);
}

/*  Like mkdir -p  */
static void make_directory_path(const char *path)
{
    char buf[PATH_MAX];
    int pi=0;
    int bi=0;
    while( path[pi] != '\0' ) {
        buf[bi] = path[pi];
        bi++ ; pi++ ;
        buf[bi] = '\0';
        if ( path[pi] == '\0' || path[pi] == '/' ) {
            mkdir(buf, 0700);
        }
    }
}

static void create_ede_menu_directory(const char *directory, const char *name)
{
    char dot_directory[PATH_MAX+1];
    int fd;
    FILE *f;

    make_directory_path(directory);
    snprintf(dot_directory, sizeof(dot_directory), "%s/.directory", directory);
    fd = open(dot_directory, O_CREAT | O_EXCL | O_RDWR, 0600);
    f = fdopen (fd, "wx");
    if (f == (FILE *) NULL) {
        fprintf(stderr, gettext("Could not create file %s\n"), dot_directory);
        perror("Error code");
        return;
    }
    fprintf(f, "[Desktop Entry]\n");
    fprintf(f, "Icon=%s\n", EDE_MENU_ICON);
    fprintf(f, "Name=%s\n", name);
    fclose(f);
}

static void make_dir(struct group *root, int level, const char *dir_name)
{
    struct item *item;
    char c_tmp[PATH_MAX];

    if (root == (struct group *)NULL) {
	return;
    }

    item = root->items;
    while (item->type != 0) {
	if (item->type == ITEM_MENU) {
	    struct group *tmp;

	    tmp = (struct group *)item->data;
	    snprintf(c_tmp, sizeof(c_tmp), "%s/%s", single_string(dir_name), single_string(tmp->name) );
            create_ede_menu_directory(c_tmp, tmp->name);
	} else if (item->type == ITEM_APP) {
	    struct package *app;

	    app = (struct package *)item->data;
	    if (app->name && app->exec) {
		snprintf(c_tmp, sizeof(c_tmp), "%s/%s.desktop", single_string(dir_name), single_string(app->name) );
                create_desktop(c_tmp, app);
	    }
	}
	item++;
    }
    /* second pass, recursive output... */
    item = root->items;
    while (item->type != 0) {
	if (item->type == ITEM_MENU) {
            struct group *tmp;

	    tmp = (struct group *)item->data;
	    snprintf(c_tmp, sizeof(c_tmp), "%s/%s", dir_name, tmp->name);
	    make_dir(item->data, level+1, c_tmp);
	}
	item++;
    }
}

static void clean_dir(const char *dir_name)
{
    struct dirent **namelist;
    int nr_files;
    int i;

    if (dir_name == NULL)
	return;

    /* read the whole directory in one pass */
    nr_files = scandir(dir_name, &namelist, NULL, alphasort);
    for (i = 0; i < nr_files; i++) {
	if ( (strcmp(namelist[i]->d_name, ".") != 0) && (strcmp(namelist[i]->d_name, "..") != 0) ) {
	    struct stat st;
	    char tmp[PATH_MAX];
	    int retval;

	    snprintf(tmp, sizeof(tmp), "%s/%s", dir_name, namelist[i]->d_name);
	    if (stat(tmp, &st) != 0) {
		fprintf(stderr, gettext ("Could not stat %s\n"), tmp);
		return;
	    }

	    if (S_ISDIR(st.st_mode)) {
		clean_dir(tmp);
		retval = rmdir(tmp);
	    } else {
		retval = unlink(tmp);
	    }

	    if (retval != 0) {
		perror("Could not remove");
		return;
	    }
	}
	free(namelist[i]);
    }
    if (nr_files && namelist) {
	free(namelist);
    }
}

/* Main function */
void output_ede(struct group *root)
{
    struct stat st;
    char root_dir[PATH_MAX];
    struct passwd *pw = NULL;

    pw = getpwuid(getuid());
    if (pw == (struct passwd *)NULL) {
	fprintf(stderr, gettext ("Could not find out who you are (getpwnam failed)!\n"));
	return;
    }
    snprintf(root_dir, sizeof(root_dir), "%s/%s", pw->pw_dir, EDE_DIR);

    /* First, check if the root_dir already exists and remove it */
    if (stat(root_dir, &st) == 0) {

	/* that one exists */
	if (!S_ISDIR(st.st_mode)) {
	    fprintf(stderr, gettext ("Error: %s exists but it is not a directory\n"), root_dir);
	    return;
	}
	clean_dir(root_dir);
	rmdir(root_dir);
    }
    create_ede_menu_directory(root_dir, DEFAULT_ROOT_NAME);
    make_dir(root, 0, root_dir);
}
