/*
 * Copyright (C) 2005-2015 Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2015/11/26  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2015/01/01  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/01/07  Tommy Scheunemann <net@arrishq.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <strings.h>

#include "wmconfig.h"
#include "prototypes.h"
#include "package.h"

#if (defined(__unix__) || defined(unix)) && !defined(USG)
#include <limits.h>
#include <sys/param.h>
#endif

/* define default output dirs
 if the directory exists and there are
 contents, they will be replaced with
 Wmconfig entries
*/

/* Item Dir for Wmconfig entries */
#define APPLICATIONS_DIR ".local/share/applications"

/* Where to write the menu file */
#define OUTPUTDIR ".config/menus/applications-merged"

/* The menu file itself */
#define INFO_FILE "wmconfig.menu"

extern const char *root_menu_name;
extern const char *output_directory;
extern unsigned int flags;

/* Fill the configuration file with contents the <INCLUDE> function is used to
 let the xml file read by the panel load the files */
static void fill_infofile(struct group *root, int level, const char *file, const char *itemdir)
{
    int i;
    int fd;
    FILE *f;
    struct item *item;
    if (root == (struct group *)NULL) {
	return;
    }

    /* Open / create the file and append the menu info at the end */
    fd = open(file, O_APPEND | O_RDWR, 0600);

    f = fdopen(fd, "wx");
    if (level>=0) {
	for (i=0; i<level; i++) {
	    fprintf(f," ");
	}
	fprintf(f,"<Menu><Name>%s</Name>\n", root->name);
    }

    item = root->items;

    while (item->type != 0) {
	if (item->type == ITEM_MENU) {
	    fclose(f);
	    fill_infofile(item->data, level+1, file, itemdir);
	} else if (item->type == ITEM_APP) {
	    struct package *app;
	    app = (struct package *)item->data;
	    if (app != NULL) {
		for (i=0; i<level; i++) {
		    fprintf(f," ");
		}

/* We don't write bash.desktop, we write Wmconfig_bash.desktop in case the user
 already have such an entry - easier for the difference what is written by
 Wmconfig.
 The string Wmconfig can be replaced giving the root_menu option
*/
		if (! app->restart) {
		    fprintf(f," <Include>\n");
		    fprintf(f,"   <Filename>%s_%s.desktop</Filename>\n", single_string(root_menu_name), single_string(app->name));
		    fprintf(f,"  </Include>\n");
		}
	    }
	}
	item++;
    }
    if (level>=0) {
	for (i=0; i<level; i++) {
	    fprintf(f," ");
	}
	fprintf(f,"</Menu>\n");
    }
}

/* This creates the basic xml configuration file and fills it with required contents - header */
static void create_infofile (const char *file, const char *itemdir, struct group *root)
{
    int fd;
    FILE *f;

    /* Open and create the menu xml file */
    fd = open(file, O_CREAT | O_EXCL | O_RDWR, 0600);

    f = fdopen(fd, "wx");
    if (f == (FILE *) NULL) {
	fprintf(stderr, gettext ("Could not create file %s\n"), file);
	perror("Error code");
	return;
    }
    fprintf (f,"<?xml version=\"1.0\"?>\n");
    fprintf (f,"<Menu>\n");
    fprintf (f," <Name>%s</Name>\n", root->name);
    fprintf (f," <MergeFile type=\"parent\"/>");
    fclose (f);
    fill_infofile (root, 0, file, itemdir);
}

/* Create the .desktop files for proggies */
static void create_desktop_entry(const char *file, struct package *app, const int freedesktop_type)
{
    int fd;
    FILE *f;
    if (app == (struct package *)NULL) {
	return;
    }

    /* Remove the & at the end to prevent background starting */
    if (app->exec != NULL) {
	char *p;
	p = strrchr(app->exec, '&');
	if (p != NULL) {
	    *p = '\0';
	}
    }

    /* At the very least we need a filename and command to execute */
    if (file == NULL || app->exec == NULL || app->restart) {
	return;
    }

    fd = open(file, O_CREAT | O_EXCL | O_RDWR, 0600);

    f = fdopen(fd, "wx");
    if (f == (FILE *) NULL) {
	fprintf(stderr, gettext ("Could not create file %s\n"), file);
	perror("Error code");
	return;
    }

    /* Write the .desktop file */
    fprintf(f, "[Desktop Entry]\n");
    if (app->name != NULL) {
	fprintf(f, "Name=%s\n", app->name);
    }

    if ((app->icon != NULL) && (!is_set(NO_ICONS))) {
        fprintf(f, "Icon=%s\n", app->icon);
    }

    if (app->description != NULL) {
	fprintf(f, "Comment=%s\n", app->description);
    }
    fprintf(f, "Type=Application\n");

    switch (freedesktop_type) {

	/* Generic freedesktop output requested
	   Should work with everything except Gnome, KDE, LXDE, XFCE
	   which we handle seperate */
	case 0:
	    if (app->terminal != NULL) {
		fprintf(f, "Terminal=true\n");
	    }
	    fprintf(f, "Exec=%s\n", app->exec);
	    fprintf(f, "NotShowIn=GNOME;KDE;LXDE;XFCE;\n");
	break;

	/* KDE output requested */
	case 1:
	    if (app->terminal != NULL) {
		fprintf(f, "Terminal=true\n");
	    }
	    fprintf(f, "Exec=%s\n", app->exec);
	    fprintf(f, "OnlyShowIn=KDE\n");
	    fprintf(f, "NotShowIn=GNOME;LXDE;ROX;Unity;XFCE;Old;\n");
	break;

	/* Gnome output requested */
	case 2:
	    if (app->terminal != NULL) {
		fprintf(f, "Terminal=true\n");
	    }
	    fprintf(f, "Exec=%s\n", app->exec);
	    fprintf(f, "OnlyShowIn=GNOME\n");
	    fprintf(f, "NotShowIn=KDE;LXDE;ROX;Unity;XFCE;Old;\n");
	break;

	/* Xfce output requested */
	case 3:
	    if (app->terminal != NULL) {
		fprintf(f, "Terminal=true\n");
	    }
	    fprintf(f, "Exec=%s\n", app->exec);
	    fprintf(f, "OnlyShowIn=XFCE\n");
	    fprintf(f, "NotShowIn=GNOME;KDE;LXDE;ROX;Unity;Old;\n");
	break;

	/* LXDE output requested */
	case 4:
	    if (app->terminal != NULL) {
		fprintf(f, "Terminal=true\n");
	    }
	    fprintf(f, "Exec=%s\n", app->exec);
	    fprintf(f, "OnlyShowIn=LXDE\n");
	    fprintf(f, "NotShowIn=GNOME;KDE;ROX;Unity;XFCE;Old;\n");
	break;

	/* E17 Output requested
	   E17 doesn't seem to support yet OnlyShowIn / NotShowIn */
	case 5:
	    if (app->terminal != NULL) {
		fprintf(f, "Terminal=true\n");
	    }
	    fprintf(f, "Exec=%s\n", app->exec);
	break;
    }

    /* Mimetype Support */
    if (app->mimetype != NULL) {
	fprintf(f, "MimeType=%s;\n", app->mimetype);
    }
    fclose(f);
}

/*  Like mkdir -p  */
static void make_directory_path(const char *path)
{
    char buf[PATH_MAX];
    int pi=0;
    int bi=0;
    while( path[pi] != '\0' ) {
	buf[bi] = path[pi];
	bi++ ; pi++ ;
	buf[bi] = '\0';
	if ( path[pi] == '\0' || path[pi] == '/' ) {
	    mkdir(buf, 0700);
	}
    }
}

static void make_dir(struct group *root, const char *dir_name, const int freedesktop_type)
{
    struct item *item;
    struct stat st;
    char c_tmp[PATH_MAX];

    if (root == (struct group *)NULL) {
	return;
    }
    item = root->items;
    while (item->type != 0) {
	if (item->type == ITEM_APP) {
	    struct package *app;

	    app = (struct package *)item->data;
	    if (app->name && app->exec) {
		snprintf(c_tmp, sizeof(c_tmp), "%s/%s_%s.desktop", single_string(dir_name), single_string(root_menu_name), single_string(app->name));

		/* Delete the file if it exists already */
		if (stat(c_tmp, &st) == 0) {
		    unlink(c_tmp);
		}
                create_desktop_entry(c_tmp, app, freedesktop_type);
	    }
	}
	item++;
    }

    /* second pass, recursive output... */
    item = root->items;
    while (item->type != 0) {
	if (item->type == ITEM_MENU) {
	    struct group *tmp;
	    tmp = (struct group *)item->data;
	    snprintf(c_tmp, sizeof(c_tmp), "%s/%s", dir_name, tmp->name);
	    make_dir(item->data, dir_name, freedesktop_type);
	}
	item++;
    }
}

/* Basic output function - launchers and the configuration file */
void output_freedesktop(struct group *root, const int freedesktop_type)
{
    struct stat st;
    char root_dir[PATH_MAX];
    char config_file[PATH_MAX];
    char outputdir[PATH_MAX];
    struct passwd *pw = NULL;
    pw = getpwuid(getuid());
    if (pw == (struct passwd *)NULL) {
	fprintf(stderr, gettext ("Could not find out who you are (getpwnam failed)!\n"));
	return;
    }

    /* Check if the outputdir option is given or not */
    if (output_directory == NULL) {
	snprintf(outputdir, sizeof(root_dir), "%s/%s", pw->pw_dir, OUTPUTDIR);
    } else {
	snprintf(outputdir, sizeof(root_dir), "%s", output_directory);
    }
    snprintf(config_file, sizeof(config_file), "%s/%s", outputdir, INFO_FILE);
    snprintf(root_dir, sizeof(root_dir), "%s/%s", pw->pw_dir, APPLICATIONS_DIR);

    /* Check if the directory for the desktop entries exist */
    if (stat(root_dir, &st) == 0) {
	if (!S_ISDIR(st.st_mode)) {
	    fprintf(stderr, gettext ("Error: %s exists but it is not a directory\n"), root_dir);
	    return;
	}
    }

    /* Check if the directory for the xml file exists */
    if (stat(outputdir, &st) == 0) {
	if (!S_ISDIR(st.st_mode)) {
	    fprintf(stderr, gettext ("Error: %s exists but it is not a directory\n"), outputdir);
	    return;
	}
    }

    /* Delete the xml file if it exists */
    if (stat(config_file, &st) == 0) {
	unlink(config_file);
    }

    /* Create the required directories */
    make_directory_path(root_dir);
    make_directory_path(outputdir);

    /* Fill the menu directory */
    make_dir(root, root_dir, freedesktop_type);

    /* Create xml configuration file */
    if (output_directory == NULL) {
	snprintf(root_dir, sizeof(root_dir), "%s/%s", pw->pw_dir, OUTPUTDIR);
    } else {
	snprintf(root_dir, sizeof(root_dir), "%s", output_directory);
    }

    create_infofile(config_file, root_dir, root);
}
