/*
 * Copyright (C) 1997 Red Hat Software, Inc.
 *	Cristian Gafton <gafton@redhat.com>
 *
 * Modify:  2014/01/07  Tommy Scheunemann <net@arrishq.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* WM config - functions to read and parse the config files */
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <limits.h>
#include <fcntl.h>

#include "wmconfig.h"

/* This reads & parses a system configuration file */
int read_file(const char *filename)
{
    int f;
    FILE *fd;
    char buffer[BUFSIZ];
    char *lang, *LANG;
    if (!filename) {
	return -1;
    }

    LANG = getenv("LANG");

    f = open(filename, O_RDONLY);
    fd = fdopen (f, "r");

    if (fd == (FILE *)NULL) {
	fprintf(stderr, gettext ("Could not read settings from %s\n"), filename);
	return -4;
    }

    while (fgets(buffer, BUFSIZ, fd) != NULL) {
	int  slen;
	char *package, *tag, *string;
	int ret;

	slen = strlen(buffer);
	
	if (buffer[slen-1] != '\n') {
	    /* this line is possible invalid... bail out */
	    continue;
	} else {
	    buffer[slen--] = '\0';
	}

	ret = parse_line(buffer, &package, &tag, &lang, &string);
	if (ret != 0) {
	    memset(buffer, 0, sizeof(buffer));
	    continue;
	}

	if (lang != NULL) {
	    if (LANG == NULL) {
		continue;
	    } else if (strcmp(lang,LANG) != 0) {
		continue;
	    }
	}

#ifdef STRICT_PACKAGES
	if (strcmp(package, filename) != 0) {
	    /* for the system file the package name must be the same
	     * as with the file name... Otherwise ignore it .. */
	    fprintf(stderr, gettext ("Ignoring package '%s' line in file \"%s\"\n"), package, filename);
	    memset(buffer, 0, sizeof(buffer));
	    continue;
	}
#endif /* STRICT_PACKAGES */
	ret = update_package(package, tag, string);
	if (ret != 0) {
	    fprintf(stderr, gettext ("Warning: %s updating (<%s>, <%s>, <%s>) failed!\n"), filename, package, tag, string);
	}
	memset(buffer, 0, sizeof(buffer));
    }
    fclose(fd);
    return 0;
}

/*
 * This function parses the system configuration from
 * a confdir and fills in the structures needed by
 * this program
 */
int parse_files(const char *dir_from)
{
    struct stat st;
    int ret;
    DIR *sys_dir;
    struct dirent **namelist;
    int nr_files;
    int i;

    if (dir_from == NULL) {
	return PARSE_OK;
    }

    ret = stat(dir_from, &st);
    if (ret != 0) {
	fprintf(stderr, "%s: %s\n", __FUNCTION__, strerror(errno));
	return PARSE_ABORT;
    }

    /* okay, it exist... do some more tests */
    if (!S_ISDIR(st.st_mode)) {
	fprintf(stderr, gettext ("%s: %s is not a directory\n"), __FUNCTION__, dir_from);
	return PARSE_ABORT;
    };

    /* ...and is a directory */
    sys_dir = opendir(dir_from);
    if (sys_dir == (DIR *)NULL) {
	fprintf(stderr, gettext ("%s: can not read system config files from %s\n"), __FUNCTION__, dir_from);
	return PARSE_ABORT;
    };
    closedir(sys_dir);

    /* read the whole directory in one pass */
    nr_files = scandir(dir_from, &namelist, NULL, alphasort);
    for (i = 0; i < nr_files; i++) {
	if ( (strcmp(namelist[i]->d_name, ".") != 0) &&
	     (strcmp(namelist[i]->d_name, "..") != 0) ) {
	    char filename[PATH_MAX];
	    int ret_val;
	    snprintf(filename, sizeof(filename), "%s/%s", dir_from, namelist[i]->d_name);
	    ret_val = read_file(filename);
	    if (ret_val != 0) {
		fprintf(stderr, gettext ("\tReading file \"%s\" failed !\n"), filename);
	    }
	}
	free(namelist[i]);
    }
    if (nr_files && namelist) {
	free(namelist);
    }
    return PARSE_OK;
}
