/* viewcommandlogop.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2021 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "view_command_log_op.hh"
#include "listermode.h"
#include "worker_locale.h"
#include "worker.h"
#include "fileentry.hh"
#include "argclass.hh"
#include "nwc_path.hh"
#include "aguix/aguix.h"
#include "aguix/awindow.h"
#include "aguix/text.h"
#include "aguix/fieldlistview.h"
#include "aguix/button.h"
#include <algorithm>
#include <functional>
#include "aguix/textview.h"
#include "worker_time.hh"

const char *ViewCommandLogOp::name = "ViewCommandLogOp";

ViewCommandLogOp::ViewCommandLogOp() : FunctionProto()
{
}

ViewCommandLogOp::~ViewCommandLogOp()
{
}

ViewCommandLogOp *ViewCommandLogOp::duplicate() const
{
    ViewCommandLogOp *ta = new ViewCommandLogOp();
    return ta;
}

bool ViewCommandLogOp::isName(const char *str)
{
    if ( strcmp( str, name ) == 0 ) {
        return true;
    } else {
        return false;
    }
}

const char *ViewCommandLogOp::getName()
{
    return name;
}

int ViewCommandLogOp::run( std::shared_ptr< WPUContext > wpu, ActionMessage *msg )
{
    Lister *l1;
    ListerMode *lm1;
    
    l1 = msg->getWorker()->getActiveLister();
    if ( l1 == NULL )
        return 1;

    lm1 = l1->getActiveMode();

    msg->getWorker()->setWaitCursor();

    auto log = msg->getWorker()->getCommandLog();

    int res = ui( *Worker::getAGUIX(), log );
    if ( res == 0 ) {
        if ( lm1 && ! m_go_to_entry.empty() ) {
            std::list< RefCount< ArgClass > > args;

            args.push_back( new StringArg( m_go_to_entry ) );
            lm1->runCommand( "show_entry", args );
        }
    } else if ( res == 1 ) {
        msg->getWorker()->clearCommandLog();
    }

    msg->getWorker()->unsetWaitCursor();
    return 0;
}

const char *ViewCommandLogOp::getDescription()
{
    return catalog.getLocale( 1409 );
}

int ViewCommandLogOp::ui( AGUIX &aguix, const std::list< FileCommandLog::Entry > &log )
{
    int res = -1;

    auto win = std::shared_ptr<AWindow>( new AWindow( &aguix,
                                                      10, 10,
                                                      500, 400,
                                                      catalog.getLocale( 1410 ),
                                                      AWindow::AWINDOW_DIALOG ) );
    win->create();

    AContainer *cont0 = win->setContainer( new AContainer( win.get(), 1, 3 ), true );
    cont0->setMaxSpace( 5 );

    cont0->addWidget( new Text( &aguix, 0, 0, catalog.getLocale( 1411 ) ),
                      0, 0, AContainer::CO_INCWNR );

    auto lv = cont0->addWidget( new FieldListView( &aguix, 0, 0, 
                                                   200, 200, 0 ),
                                0, 1, AContainer::CO_MIN );
    lv->setNrOfFields( 3 );
    lv->setShowHeader( true );
    lv->setFieldText( 0, catalog.getLocale( 1415 ) );
    lv->setFieldText( 1, catalog.getLocale( 1319 ) );
    lv->setFieldText( 2, catalog.getLocale( 1412 ) );
    lv->setHBarState( 2 );
    lv->setVBarState( 2 );
    lv->setAcceptFocus( true );
    lv->setDisplayFocus( true );
    lv->setGlobalFieldSpace( 5 );
    lv->setDefaultColorMode( FieldListView::PRECOLOR_ONLYACTIVE );

    AContainer *cont10 = cont0->add( new AContainer( win.get(), 3, 1 ), 0, 2 );
    cont10->setBorderWidth( 0 );
    cont10->setMinSpace( 5 );
    cont10->setMaxSpace( -1 );
    
    auto gob = cont10->addWidget( new Button( &aguix, 0, 0,
                                              catalog.getLocale( 1413 ),
                                              0 ),
                                  0, 0, AContainer::CO_FIX );
    auto clearb = cont10->addWidget( new Button( &aguix, 0, 0,
                                                 catalog.getLocale( 1414 ),
                                              0 ),
                                     1, 0, AContainer::CO_FIX );
    auto closeb = cont10->addWidget( new Button( &aguix, 0, 0,
                                                 catalog.getLocale( 633 ),
                                                 0 ),
                                     2, 0, AContainer::CO_FIX );
   
    win->contMaximize( true );
    win->setDoTabCycling( true );

    time_t now = time( NULL );

    for ( auto &e : log ) {
        int row = lv->addRow();

        lv->setText( row, 0, e.m_command );
        lv->setText( row, 1, e.m_description );
        lv->setText( row, 2, WorkerTime::convert_time_diff_to_string( now - e.m_time ) );
    }

    maximizeWin( aguix, win, lv, cont0 );
    win->show();

    lv->takeFocus();

    AGMessage *msg;
    int endmode = 0;

    for ( ; endmode == 0; ) {
        msg = aguix.WaitMessage( NULL );
        if ( msg != NULL ) {
            switch ( msg->type ) {
                case AG_CLOSEWINDOW:
                    endmode = -1;
                    break;
                case AG_BUTTONCLICKED:
                    if ( msg->button.button == gob ) {
                        endmode = 1;
                    } else if ( msg->button.button == closeb ) {
                        endmode = -1;
                    } else if ( msg->button.button == clearb ) {
                        endmode = -2;
                    }
                    break;
                case AG_KEYPRESSED:
                    if ( msg->key.key == XK_Return ) {
                        endmode = 1;
                    } else if ( msg->key.key == XK_Escape ) {
                        endmode = -1;
                    }
                    break;
                case AG_FIELDLV_DOUBLECLICK:
                    if ( msg->fieldlv.lv == lv ) {
                        // double click in lv, actual element is unimportant here
                        endmode = 1;
                    }
                    break;
            }
            aguix.ReplyMessage( msg );
        }
    }

    if ( endmode == 1 ) {
        int row = lv->getActiveRow();
        if ( lv->isValidRow( row ) == true ) {
            auto it = log.begin();
            std::advance( it, row );
            if ( it != log.end() ) {
                m_go_to_entry = it->m_path;
            }
        }
        res = 0;
    } else if ( endmode == -2 ) {
        res = 1;
    }

    return res;
}

void ViewCommandLogOp::maximizeWin( AGUIX &aguix,
                                    std::shared_ptr< AWindow > win,
                                    FieldListView *lv,
                                    AContainer *cont )
{
    int old_w = lv->getWidth();
    int old_h = lv->getHeight();

    int new_w = lv->getMaximumWidth();
    int new_h = lv->getMaximumHeight();

    if ( new_w <= old_w &&
         new_h <= old_h ) {
        cont->rearrange();
    } else {
        int my_w = new_w + 10;
        int my_h = new_h + 10;

        if ( my_w < 400 ) my_w = 400;
        if ( my_h < 300 ) my_h = 300;

        int rx, ry, rw, rh;

        aguix.getLargestDimensionOfCurrentScreen( &rx, &ry,
                                                  &rw, &rh );

        int mw = rw * 80 / 100;
        int mh = rh * 80 / 100;

        if ( mw < my_w ) {
            my_w = mw;
        }
        if ( mh < my_h ) {
            my_h = mh;
        }

        if ( my_w < new_w ) {
            cont->setMinWidth( my_w, 0, 1 );
        } else {
            cont->setMinWidth( new_w, 0, 1 );
        }
        if ( my_h < new_h ) {
            cont->setMinHeight( my_h, 0, 1 );
        } else {
            cont->setMinHeight( new_h, 0, 1 );
        }
        win->contMaximize( true );
    }
}
