//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2011-2012 Mikhail Fedotov <anyremote@mail.ru>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <dirent.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/sendfile.h>

#define SERVER     "webserver/1.1"
#define PROTOCOL   "HTTP/1.1"
#define RFC1123FMT "%a, %d %b %Y %H:%M:%S GMT"

#include "str.h"
#include "utils.h"
#include "conf.h"
#include "cmds.h"
#include "queue.h"
#include "mutex.h"
#include "thread.h"
#include "list.h"
#include "var.h"
#include "sys_util.h"
#include "pr_web.h"
#include "security.h"
#include "state.h"
#include "gen_html.h"
#include "gen_xml.h"

extern char tmp[MAXMAXLEN];
extern int  gotExitSignal;

static int visits = 0;          // counts client connections
    
typedef struct _WebConnection_ {
    int fileDescriptor;    
} _WebConnection;

typedef struct _WebClientConnection_ {
    int serverPort;    
    int connDescriptor;    
    int runMode;   // SERVER_WEB or SERVER_CMXML 
} _WebClientConnection;

long m_cookie = 0;
int  m_secure = NO_COOKIE;

boolean_t runWeb = BOOL_YES;
boolean_t answerReady = BOOL_NO;

char* confDir = NULL;

int refreshPage = -1;  // refrech time in seconds, -1 meand no refresh

// Global data

#define HTTP_ACTION    "action="
#define HTTP_EDITFIELD "http_editfield="

#define IVIEW_HEAD1 "<gui>\n  <properties>\n    <project>anyRemote GUI</project>\n    <designer>anyRemote</designer>\n    <controlsystem>\n      <hostname>"
#define IVIEW_HEAD2 "</hostname>\n      <commandPort>"
#define IVIEW_HEAD3 "</commandPort>\n    </controlsystem>\n    <size>\n      <portrait width=\"320\" height=\"480\" />\n      <landscape width=\"480\" height=\"320\" />\n    </size>\n    <imagefolder>\n    </imagefolder>\n    <cache images=\"0\" />\n    <debug loaderrors=\"0\" />\n  </properties>"
#define IVIEW_TAIL  "</gui>"

SingleList * clientSockets = NULL;

string_t* serverIP = NULL;

static const char* i2string(int data,char* buf)
{
    sprintf(buf,"%d",data);
    return buf;
}

void freeWMessage(void *ptr)
{
    wMessage *wm = (wMessage *) ptr;
    if (wm->string) {
        stringFree(wm->string,  BOOL_YES);
        wm->string = NULL;
    }
    free(wm);
}

static void sendToWebServer(wMessage *buf)
{
    if (queueExists(Q_WEB) == RC_OK && buf) {
        mutexLock(M_WEB);
        DEBUG2("send to web server %d %s", buf->button, (buf->string ? buf->string->str : "no text"));
        queuePush(Q_WEB, buf);
        mutexUnlock(M_WEB);
    }
}

static const char *get_mime_type(const char *name)
{
    char *ext = strrchr(name, '.');
    if (!ext) {
        return NULL;
    }

    if (strcmp(ext, ".html") == 0 || strcmp(ext, ".htm") == 0) {
        return "text/html";
    }
    if (strcmp(ext, ".jpg") == 0 || strcmp(ext, ".jpeg") == 0) {
        return "image/jpeg";
    }
    if (strcmp(ext, ".gif") == 0) {
        return "image/gif";
    }
    if (strcmp(ext, ".png") == 0) {
        return "image/png";
    }
    if (strcmp(ext, ".css") == 0) {
        return "text/css";
    }
    if (strcmp(ext, ".au") == 0) {
        return "audio/basic";
    }
    if (strcmp(ext, ".wav") == 0) {
        return "audio/wav";
    }
    if (strcmp(ext, ".avi") == 0) {
        return "video/x-msvideo";
    }
    if (strcmp(ext, ".mpeg") == 0 ||  strcmp(ext, ".mpg") == 0) {
        return "video/mpeg";
    }
    if (strcmp(ext, ".mp3") == 0) {
        return "audio/mpeg";
    }
    return NULL;
}

static int sendData(int fd, const char *s)
{
    //printf("%s\n",s);
    int bytes_total = strlen(s);
    
    int bytes_sent = send(fd,s,strlen(s),MSG_NOSIGNAL);
    
    if (bytes_sent != bytes_total) {
        ERROR2("[WS]: Error on send data: sent %d from %d", bytes_sent, bytes_total);
        return -1;
    }
    return 0;
}

void sendCookie(int fd)
{
    if (m_secure == NO_COOKIE) {
        return;
    }

    char b[32];
    sendData(fd, "Set-Cookie: anyremote_id=");
    sprintf(b,"%ld",m_cookie);
    sendData(fd, b);
    sendData(fd, "\r\n");

    INFO2("[WS]: sendCookie %s",b);
}

static void sendHeaders(int fd, int status,
                        const char *title, const char *extra, const char *mime,
                        int length, time_t date)
{
    char f[4096];
    //INFO2("[WS]: sendHeaders %s",title);

    sprintf(f, "%s %d %s\r\n", PROTOCOL, status, title);
    sendData(fd, f);

    sprintf(f, "Server: %s\r\n", SERVER);
    sendData(fd, f);

    time_t now = time(NULL);
    char timebuf[128];
    strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&now));

    sprintf(f, "Date: %s\r\n", timebuf);
    sendData(fd, f);

    if (extra) {
        sprintf(f, "%s\r\n", extra);
        sendData(fd, f);
    }
    if (mime) {
        sprintf(f, "Content-Type: %s\r\n", mime);
        sendData(fd, f);
    }

    // Generated images can be saved to the file with the same name
    //sprintf(f, "Cache-Control: public, max-age=36000\r\n");
    sprintf(f, "Cache-Control: no-cache, must-revalidate\r\n");
    sendData(fd, f);

    if (length >= 0) {
        sprintf(f, "Content-Length: %d\r\n", length);
        sendData(fd, f);
    }
    sendCookie(fd);

    if (date != -1) {
        strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&date));
        sprintf(f, "Last-Modified: %s\r\n", timebuf);
        sendData(fd, f);
    }

    sprintf(f, "Connection: close\r\n\r\n");
    sendData(fd, f);
}

static void sendError(int fd, int status, char *title, char *extra, char *text)
{
    char f[4096];

    INFO2("[WS]: sendError %d", status);
    sendHeaders(fd, status, title, extra, "text/html", -1, -1);

    sprintf(f, "<HTML><HEAD><TITLE>%d %s</TITLE></HEAD>\r\n", status, title);
    sendData(fd, f);

    if (status != 304) { // 304  - Not Modified

        sprintf(f, "<BODY><H4>%d %s</H4>\r\n", status, title);
        sendData(fd, f);

        sprintf(f, "%s\r\n", text);
        sendData(fd, f);

        sprintf(f, "</BODY></HTML>");
        sendData(fd, f);

    } else {
        time_t now = time(NULL);
        char timebuf[128];
        strftime(timebuf, sizeof(timebuf), RFC1123FMT, gmtime(&now));

        sprintf(f, "Date: %s\r\n", timebuf);
        sendData(fd, f);
    }


    sprintf(f, "\r\n");
    sendData(fd, f);
}

static void sendFile(int fd, char *path, struct stat *statbuf)
{
    //char data[4096];
    //int n;
    int bytes_sent;

    INFO2("[WS]: sendFile %s",path);

    /*FILE *file = fopen(path, "r");
    if (!file) {
        ERROR2("[WS]: Access denied: %s", path);
        sendError(fd, 403, "Forbidden", NULL, "Access denied.");
    */
    
    int fdout = open(path, O_RDONLY);
    if (fdout < 0) {
    
        ERROR2("[WS]: Access denied: %s", path);
        sendError(fd, 403, "Forbidden", NULL, "Access denied.");

    } else {

        int length = S_ISREG(statbuf->st_mode) ? statbuf->st_size : -1;
        sendHeaders(fd, 200, "OK", NULL, get_mime_type(path), length, statbuf->st_mtime);

        bytes_sent = sendfile(fd,fdout,NULL,length);
	
        /*while ((n = fread(data, 1, sizeof(data), file)) > 0) {
            //INFO2("read %d bytes from file",n);

            bytes_sent = send(fd,data,n,0);
            if (n != bytes_sent) {
                ERROR2("[WS]: Error on send file %s", path);
                break;
            }
        }*/
	
        if (length != bytes_sent) {
            ERROR2("[WS]: Error on send file %s", path);
        }

        close(fdout);
    }
    sendData(fd, "\r\n");
}

static void sendGeneratedImage(int fd, char *path)
{
    INFO2("[WS]: sendGeneratedImage %s", path);

    const char* home = getenv("HOME");
    if (!home) {
        return;
    }

    struct stat statbuf;

    string_t* img = stringNew(home);
    stringAppend(img,"/.anyRemote/");
    stringAppend(img,path);

    INFO2("[WS]: sendGeneratedImage full name %s", img->str);

    if (stat(img->str, &statbuf) >= 0) {
        sendFile(fd, img->str, &statbuf);
    } else {
        sendError(fd, 404, "Not Found", NULL, "File not found.");
    }
    stringFree(img, BOOL_YES);
}

static void sendIcon(int fd, char *path)
{
    INFO2("[WS]: sendIcon %s", path);

    struct stat statbuf;

    string_t* icon = stringNew(confDir ? confDir : ".");
    if (getIViewer()) {
   	    stringAppend(icon,"/Utils/iViewer");
	    stringAppend(icon,path);
    } else {
        stringAppend(icon,"/Icons/");

        char b[32];
        stringAppend(icon,i2string(iconSize(),b));
        if (path[0] != '/') {
            stringAppend(icon,"/");
        }
        stringAppend(icon,path);
    }

    INFO2("[WS]: sendIcon full name %s", icon->str);

    if (stat(icon->str, &statbuf) >= 0) {
        sendFile(fd, icon->str, &statbuf);
    } else {
        sendError(fd, 404, "Not Found", NULL, "File not found.");
    }
    stringFree(icon, BOOL_YES);
}

static void sendFavicon(int fd)
{
    struct stat statbuf;

    string_t* icon = stringNew(confDir ? confDir : ".");
    stringAppend(icon,"/Icons/anyRemote.png");

    if (stat(icon->str, &statbuf) >= 0) {
        sendFile(fd, icon->str, &statbuf);
    } else {
        //ERROR2("[WS]: sendFavicon can not open favicon file");
	sendError(fd, 404, "Not Found", NULL, "File not found.");
    }
    stringFree(icon, BOOL_YES);
}

static void sendIViewerGui(int fd) 
{
    INFO2("[WS]: sendIViewerGui");
	
    sendData(fd, IVIEW_HEAD1);
    
    if (!serverIP) {
	    ERROR2("[WS]: sendIViewerGui can not determine TCP address");
		return;
	}
    sendData(fd, serverIP->str);
    
    sendData(fd, IVIEW_HEAD2);
     
    int tcpPort = getIViewerTcpPort();
    if (tcpPort < 0) {
    	ERROR2("[WS]: sendIViewerGui can not determine TCP port");
    	return;
    }
    char num[16];
    sprintf(num,"%d", tcpPort);
    
    sendData(fd, num);
    sendData(fd, IVIEW_HEAD3);

    string_t* guiFile = stringNew(confDir ? confDir : ".");
    stringAppend(guiFile,"/Utils/anyremote.gui");
    
    struct stat statbuf;
    if (stat(guiFile->str, &statbuf) >= 0) {
        sendFile(fd, guiFile->str, &statbuf);
    } else {
        ERROR2("[WS]: sendIViewerGui can not open GUI file");
    }
    sendData(fd, IVIEW_TAIL);
}

void addLink(string_t* addTo, int port, boolean_t trailSlash)
{
    stringAppend(addTo, "http://");
    stringAppend(addTo, serverIP->str);
    stringAppend(addTo, ":");

    char num[16];
    sprintf(num,"%d", port);

    stringAppend(addTo, num);
    if (trailSlash) {
        stringAppend(addTo, "/");
    }
}

/*void sendFormOld(_WebClientConnection* cc, string_t* httpPageH, string_t* httpPageT, boolean_t cookie)
{
    
    string_t* header = stringNew("HTTP/1.1 200 OK\r\n");

    if (cc->runMode == SERVER_CMXML) {
        stringAppend(header, "Content-type: text/xml; charset=UTF-8\r\nConnection: close\r\n");
    } else {
        stringAppend(header, "Content-type: text/html\r\nCache-Control: no-cache, must-revalidate\r\n");
    }

    if (refreshPage > 0) {
        stringAppend(header, "Refresh: ");

        char num[16];
        sprintf(num,"%d", refreshPage);
        stringAppend(header, num);

        stringAppend(header, "; url=");
        addLink(header, cc->port, BOOL_YES);

        stringAppend(header, "\r\n");
    } else {
        if (runMode == SERVER_CMXML) {
            stringAppend(header, "Expires: -1\r\n");
        }
    }

    //printf("HEADER: %s",header->str);
    sendData(cc->connDescriptor, header->str);

    if (cookie) {
        sendCookie(cc->connDescriptor);
    }

    sendData(cc->connDescriptor, "\r\n");

    sendData(fd, httpPageH->str);
    //printf("%s",httpPageH->str);

    //if refresh < 0:
    //	rStr = "%s" % (uploadTmout)
    //	string_t* httpRate = stringNew('HTTP-EQUIV=REFRESH CONTENT=\"' + rStr + ';URL=/\"');
    //sendData(cc->connDescriptor, httpRate->str);
    //stringFree(httpRate,  BOOL_YES);

    sendData(cc->connDescriptor, httpPageT->str);
    //printf("%s",httpPageT->str);
}*/

void sendForm(_WebClientConnection* cc, string_t* content, boolean_t cookie)
{
    string_t* header = stringNew("HTTP/1.1 200 OK\r\n");

    if (cc->runMode == SERVER_CMXML) {
        stringAppend(header, "Content-type: text/xml; charset=UTF-8\r\nConnection: close\r\n");
    } else {
        stringAppend(header, "Content-type: text/html\r\nCache-Control: no-cache, must-revalidate\r\n");
    }

    if (refreshPage > 0) {
        stringAppend(header, "Refresh: ");

        char num[16];
        sprintf(num,"%d", refreshPage);
        stringAppend(header, num);

        stringAppend(header, "; url=");
        addLink(header, cc->serverPort, BOOL_YES);

        stringAppend(header, "\r\n");
    } else {
        if (cc->runMode == SERVER_CMXML) {
            stringAppend(header, "Expires: -1\r\n");
        }
    }

    sendData(cc->connDescriptor, header->str);

    if (cookie) {
        sendCookie(cc->connDescriptor);
    }

    sendData(cc->connDescriptor, "\r\n");

    sendData(cc->connDescriptor, content->str);
    //printf("%s",content->str);
}

long parseCookie(char* buffer)
{
    //DEBUG2("[WS]: parseCookie line: %s", buffer);
    if (!buffer) {
        return 0;
    }

    char* start = buffer;
    char* startc = NULL;

    while ((start = strstr(start, "anyremote_id="))) {
        startc = start;
        start += 13;  // size("anyremote_id=") == 12

    }
    if (!startc) {
        return 0;
    }
    startc += 13;

    //DEBUG2("[WS]: parseCookie start: %s", startc);

    start = startc;
    while (startc && isdigit(*startc)) {
        startc++;
    }
    *startc = '\0';
    //DEBUG2("[WS]: parseCookie number: %s", start);

    return atol(start);
}

boolean_t handleMenuOnList(int midx, int lidx, int tvisits)
{
    int iLen = lfSize();
    int mLen = menuSize();

    if (lidx < iLen &&             //  negative list index means empty list
            midx >= 0 && midx < mLen) {

        wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = stringNew("Msg:");

        SingleList *menuItem = menuNth(midx);
        stringAppend(wm->string, ((string_t*) menuItem->data)->str);

        stringAppend(wm->string, "(");
        char b[32];
        stringAppend(wm->string, i2string(lidx + 1,b));
        stringAppend(wm->string, ",");

        if (lidx >= 0) {
            SingleList *ptr = lfListNth(lidx);
            stringAppend(wm->string, ((ListItem*) ptr->data)->string->str);
            INFO2("[WS]: choosed list item (%s)", ((ListItem*) ptr->data)->string->str);
        } else { 		// list was empty
            INFO2("[WS]: choosed list item in empty list");
        }

        stringAppend(wm->string, ")");

        sendToWebServer(wm);
        return BOOL_YES;
    } else {
        INFO2("[WS]: (%d) Wrong selected index in list, ignore.", tvisits);
    }
    return BOOL_NO;
} 

boolean_t securityCheck(_WebClientConnection* cc, int tvisits, long c, char* path)
{
   boolean_t passed = BOOL_YES;
   
   if (getUsePassword()) {
   
         INFO2("[WS]: (%d) secure mode", tvisits);
	 
         if (m_secure == COOKIE_SENT) {
            
	    INFO2("[WS]: (%d) secure mode, test cookie", tvisits);

            if (c != m_cookie) {
        	INFO2("[WS]: (%d) wrong cookie (wait for %ld)", tvisits, m_cookie);
		passed = BOOL_NO;
            } else {
        	INFO2("[WS]: (%d) cookie OK (%ld)", tvisits, m_cookie);
            }
	} else if (m_secure == NO_COOKIE) {
	    
	    INFO2("[WS]: (%d) secure mode, ask for pass", tvisits);
	    passed = BOOL_NO;
	
	} else { // m_secure == NEED_SEND_COOKIE
	    
	    INFO2("[WS]: (%d) secure mode, retrieve password", tvisits);
        
	    char* p = NULL;

            if ((p = strstr(path, HTTP_EDITFIELD))) {  // edit field

        	char * item = strstr(path, HTTP_ACTION);

        	if (!item) {
                    INFO2("[WS]: (%d) No data in edit field, ignore", tvisits);
		    passed = BOOL_NO;
        	} else {

                    item += strlen(HTTP_ACTION);
	            INFO2("[WS]: (%d) edit field %s", tvisits, item);

                    char* plus = NULL;
                    while((plus = strstr(item,"+"))) {
                	*plus = ' ';  // replace "+" back to spaces
                    }

                    char * amp = strstr(p,"&");
                    *amp = '\0';
                    char* index = p + strlen(HTTP_EDITFIELD);

		    if (strcmp("Ok", item) == 0 || strcmp("Select", item) == 0) {              // WEB edit field OK
                    	string_t* cmd = stringNew("Msg:_PASSWORD_");
        	    	stringAppend(cmd, "(,");
                    	stringAppend(cmd, index);
                    	stringAppend(cmd, ")");

		    	if (checkPassword(cmd->str)) {
			    // hack to send response with current form
			    *path = '/';
			    *(path+1) = '\0';
			    
			    srandom((unsigned int) time(NULL));
                	    m_cookie = random();
			} else {
			    INFO2("[WS]: (%d) secure mode, wrong password", tvisits);
                    	    passed = BOOL_NO;
		    	}
		    	stringFree(cmd, BOOL_YES);
		    } else {                                    // WEB edit field Cancel
		        INFO2("[WS]: (%d) secure mode, cancel pressed", tvisits);
		        passed = BOOL_NO;
		    }
        	}
		
            } else if ((p = strstr(path,XML_EFIELD_SUBMIT))) {  // CMXML edit field OK

        	char * item = p+strlen(XML_EFIELD_SUBMIT);
        	INFO2("[WS]: (%d) Entered value %s", tvisits, item);

        	string_t* cmd = stringNew("Msg:_PASSWORD_");
        	stringAppend(cmd, "(,");
                stringAppend(cmd, item);
                stringAppend(cmd, ")");
		
		if (checkPassword(cmd->str)) {
		    // hack to send response with current form
		    *path = '/';
		    *(path+1) = '\0';
		    
		    srandom((unsigned int) time(NULL));
                    m_cookie = random();
		} else {
                    passed = BOOL_NO;
		}
		stringFree(cmd, BOOL_YES);

            } else if ((p = strstr(path,XML_EFIELD_CANCEL))) {  // CMXML edit field Cancel
		INFO2("[WS]: (%d) secure mode, cancel pressed XML", tvisits);
		passed = BOOL_NO;
            } else {
		INFO2("[WS]: (%d) secure mode, request password", tvisits);
		passed = BOOL_NO;
	    }
	}
	
	if (passed == BOOL_NO) {
	    
	    string_t* content = (cc->runMode == SERVER_CMXML ? renderPassXMLForm(cc->serverPort) : renderPassHTMLForm(cc->serverPort));
	    
	    if(content) {
	    
	       m_secure    = NEED_SEND_COOKIE;
	       sendForm(cc, content, BOOL_NO);
	       stringFree(content, BOOL_YES);
	    }
	} else {
	    m_secure = COOKIE_SENT;
	}
    }
    
    return passed;
}

void sendButtonPress(int button)
{
    wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
    wm->button = button;
    wm->string = (void*) NULL;
    
    sendToWebServer(wm);
}
  
boolean_t processHtmlButtonPress(int tvisits, char* p)
{  
    *p = '\0';
    p--;

    while (isdigit(*p)) {
        p--;
    }
    int button = ((++p == '\0') ? -1 : atoi(p)+1);

    INFO2("[WS]: (%d) Got button %d", tvisits, button);
    
    sendButtonPress(button);
    
    return BOOL_YES;
}

boolean_t processXmlButtonPress(int tvisits, char* p)
{ 
    p += strlen(XML_BUTTON_PRESS);

    int button = atoi(p);

    INFO2("[WS]: (%d) Got button XML %d", tvisits, button);

    sendButtonPress(button);
    
    return BOOL_YES;
}

boolean_t processHtmlMenu(int tvisits, char* p)
{ 
    *p = '\0';
    p--;

    while (*p != '/') {
        p--;
    }

    INFO2("[WS]: (%d) Got menu item %s", tvisits, ++p);

    wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
    wm->button = -1;
    wm->string = stringNew("");
    char * ptr = p;
    char * p2 =  NULL;

    while ((p2 = strstr(ptr,"%20"))) {   // replace %20 to space
        *p2 = '\0';
        stringAppend(wm->string, ptr);
        stringAppend(wm->string, " ");

        ptr = (p2 + 3);
    }
    stringAppend(wm->string, ptr);

    sendToWebServer(wm);
    return BOOL_YES;
}

boolean_t processXmlShortMenu(int tvisits, char* p)
{ 
    boolean_t wait = BOOL_NO;
    
    p += strlen(XML_SHORT_MENU);

    INFO2("[WS]: (%d) Got menu item %s", tvisits, p);

    int idx = atoi(p) - 1;
    int iLen = menuSize();

    if (idx >= 0 && idx < iLen) {
        SingleList *menuItem = menuNth(idx);
        if (menuItem) {

            wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
            wm->button = -1;
            wm->string = stringNew("");
            stringAppend(wm->string, ((string_t*) menuItem->data)->str);

            sendToWebServer(wm);
            wait = BOOL_YES;
        } else {
            ERROR2("[WS]: Can not get list item #%d from list having %d items", idx, iLen);
        }
    } else {
        INFO2("[WS]: (%d) Wrong selected index in list, ignore..", tvisits);
    }
    
    return wait;
}

boolean_t processXmlListMenu(int tvisits, char* p)
{ 
    p += strlen(XML_LIST_MENU);
    char * q = strstr(p,"?");
    *q = '\0';
    q++;

    INFO2("[WS]: (%d) Got list menu item (%s,%s)", tvisits, p, q);

    int menu_idx = atoi(p) - 1;
    int list_idx = atoi(q);

    return handleMenuOnList(menu_idx, list_idx, tvisits);
}

boolean_t processXmlListMenu2(int tvisits, char* p)
{ 
    boolean_t wait = BOOL_NO;
    
    p += strlen(XML_LIST_MENU2);

    INFO2("[WS]: (%d) Got list menu item %s", tvisits, p);

    char * item = strstr(p,",");
    if (!item) {
        INFO2("[WS]: (%d) Improper item in list, ignore", tvisits);
    } else {

        (*item) = '\0';
        item++;

        int lidx = atoi(p);
        int midx = atoi(item) - 1;

        if (handleMenuOnList(midx, lidx, tvisits)) {
            wait = BOOL_YES;
        }
    }
    return wait;
}

void processXmlLongMenu(_WebClientConnection* cc, int tvisits, char* p)
{ 
    p += strlen(XML_LONG_MENU);

    INFO2("[WS]: (%d) Got menu wrapper %s", tvisits, p);

    string_t* content = sendXMLMenu(atoi(p), cc->serverPort, -1);
    if (content) {
        sendForm(cc, content, BOOL_YES);
	stringFree(content, BOOL_YES);
    }
}

void processXmlExtMenu(_WebClientConnection* cc, int tvisits, char* p)
{ 
    p += strlen(XML_LIST_MENU_EXT);
    p++;

    INFO2("[WS]: (%d) Got list item %s", tvisits, p);

    int listItem = atoi(p);

    string_t* content = sendXMLMenu(LI, cc->serverPort, listItem);
    if (content) {
        sendForm(cc, content, BOOL_YES);
	stringFree(content, BOOL_YES);
    }
}

boolean_t processHtmlListMenu(int tvisits, char* p, char* path)
{ 
    boolean_t wait = BOOL_NO;
    
    char * item = strstr(path, HTTP_ACTION);

    if (!item) {
        INFO2("[WS]: (%d) No selected item in list, ignore", tvisits);
    } else {

        if (item) {
            item += strlen(HTTP_ACTION);
        }

        char* plus = NULL;
        while((plus = strstr(item,"+"))) {
            *plus = ' ';  // replace "+" back to spaces
        }

        char * amp = strstr(p,"&");
        *amp = '\0';
        char* index = p + 5;  // size("list=") == 5

        int idx = atoi(index);
        int iLen = lfSize();

        if (idx >= 0 && idx < iLen) {

            wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
            wm->button = -1;
            wm->string = stringNew("Msg:");
            stringAppend(wm->string, item);
            stringAppend(wm->string, "(");
            char b[32];
            stringAppend(wm->string, i2string(idx+1,b));
            stringAppend(wm->string, ",");

            SingleList *ptr = lfListNth(idx);
            stringAppend(wm->string, ((ListItem*) ptr->data)->string->str);
            INFO2("[WS]: choosed list item (%s)", ((ListItem*) ptr->data)->string->str);

            stringAppend(wm->string, ")");

            sendToWebServer(wm);
            wait = BOOL_YES;
        } else {
            INFO2("[WS]: (%d) Wrong selected index in list, ignore...", tvisits);
        }
    }
    
    return wait;
}
       
boolean_t processHtmlEfield(int tvisits, char* p, char* path)
{ 
    boolean_t wait = BOOL_NO;
    
    char * item = strstr(path, HTTP_ACTION);
    if (!item) {
        INFO2("[WS]: (%d) No data in edit field, ignore", tvisits);
    } else {

        item += strlen(HTTP_ACTION);
        INFO2("[WS]: (%d) edit field %s", tvisits, item);

        char* plus = NULL;
        while((plus = strstr(item,"+"))) {
            *plus = ' ';  // replace "+" back to spaces
        }

        char * amp = strstr(p,"&");
        *amp = '\0';
        char* index = p + strlen(HTTP_EDITFIELD);

        wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        
	if (efPassword()) {  // password was asked from cfg.file so do not handle it internally
	    if (strcmp("Ok", item) == 0) {
        	wm->string = stringNew("Msg:_PASSWORD_(,");
        	stringAppend(wm->string, index);
        	stringAppend(wm->string, ")");
		
		setEfPassword(BOOL_NO);
	    }
        } else {
            wm->string = stringNew("Msg:");
            stringAppend(wm->string, item);
	    
            stringAppend(wm->string, "(,");
            stringAppend(wm->string, index);
            stringAppend(wm->string, ")");
        }
        sendToWebServer(wm);
        wait = BOOL_YES;
    }
    return wait;
}

boolean_t processXmlEfieldSubmit(int tvisits, char* p)
{ 
    char * item = p+strlen(XML_EFIELD_SUBMIT);
    INFO2("[WS]: (%d) Entered value %s", tvisits, item);

    wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
    wm->button = -1;
    if (efPassword()) {
        wm->string = stringNew("Msg:_PASSWORD_");
        setEfPassword(BOOL_NO);
    } else {
        wm->string = stringNew("Msg:Ok");
    }
    stringAppend(wm->string, "(,");
    stringAppend(wm->string, item);
    stringAppend(wm->string, ")");

    sendToWebServer(wm);
    return  BOOL_YES;
          
}
boolean_t processXmlEfieldCancel()
{           
    wMessage* wm = (wMessage*) malloc(sizeof(wMessage));
    wm->button = -1;
    wm->string = stringNew("Msg:Back(,)");
    sendToWebServer(wm);
    
    return BOOL_YES;
} 

static struct {
    string_t* (*hooks[6]) (int) ; // CF,TX,LI,FM,WM,EF
} _renderHooks[] = {
    //                  CF                  TX                  LI                  FM    WM                  EF 
    /* SERVER_WEB  */ {{renderCtrlHTMLForm, renderTextHTMLForm, renderListHTMLForm, NULL, renderWmanHTMLForm, renderEditHTMLForm}},
    /* SERVER_CMXML*/ {{renderCtrlXMLForm,  renderTextXMLForm,  renderListXMLForm,  NULL, renderWmanXMLForm,  renderEditXMLForm }}
};


static void renderForm(int tvisits, _WebClientConnection* cc)
{
    int f   = curForm() - 1;
    int idx = (cc->runMode == SERVER_CMXML ? 1 : 0);
    
    INFO2("[WS]: (%d) renderForm %d", tvisits, f);
    
    
    if (_renderHooks[idx].hooks[f]) {
        string_t* content = _renderHooks[idx].hooks[f](cc->serverPort);
	if (content) {
	     sendForm(cc, content, BOOL_YES);
	     stringFree(content, BOOL_YES);
	}
    }
}


pointer_t serveRequest(pointer_t data)
{
    char buf[4096];
    char *method = NULL;
    char *path   = NULL;
    struct stat statbuf;

    _WebClientConnection* cc = (_WebClientConnection*) data;

    int tvisits;

    INFO2("[WS]: -------------------------------------------------------");

    mutexLock(M_WEB);
    tvisits = ++visits;
    mutexUnlock(M_WEB);
    
    INFO2("[WS]: (%d) process request", tvisits);


    /*if (answerReady == BOOL_NO) {
        INFO2("[WS]: (%d) drop request", tvisits);
        sendError(fd, 304, "Please wait", NULL, "Please wait.");
        return (void*) -1;
    }*/

    int wasRead = read(cc->connDescriptor, buf, sizeof(buf)-1);
    if (wasRead<0) {
        free(data);
        return (void*) -1;
    }
    buf[wasRead] = '\0';

    INFO2("[WS]: (%d) URL: %s", tvisits, buf);

    char * cookieLine = NULL;

    char* firstLine = strtok(buf, "\r");

    int rq = RQ_UNKNOWN;
    if (!firstLine) {
        INFO2("[WS]: non-valid request >%s<",buf);
        free(data);
        return (void*) -1;
    }

    if (strstr(firstLine,"POST ")) {
        rq = RQ_POST;
    } else if (strstr(firstLine,"GET ")) {
        rq = RQ_GET;
    }

    char * line       = NULL;
    char * screenDef  = NULL;
    while ((line = strtok(NULL, "\r"))) {
        if (strstr(line,"Cookie:")) {
            cookieLine = line;
        }
        if (rq == RQ_POST) {
            path = line;  // it will be last one
        }

        if (cc->runMode == SERVER_CMXML && strstr(line,"x-CiscoIPPhoneDisplay:")) {
            screenDef = line;
        }
    }

    //parse path for GET request" GET /path HTTP/1.1
    method = strtok(firstLine, " ");
    if (rq == RQ_GET) {
        //path   = strtok(NULL, " ");
        path = firstLine + 4; // "GET "
        char *ptr = strstr(path," HTTP");
        if (ptr) {
            *ptr = '\0';
        }
    }

    long c = parseCookie(cookieLine);

    INFO2("[WS]: (%d) Path: >%s<",  tvisits, path);
    INFO2("[WS]: (%d) Cookie: %ld", tvisits, c);

    if (screenDef) {
        parseScreenDef(screenDef);
    }

    char* p = NULL;

    boolean_t wait    = BOOL_NO;

    boolean_t allowed = securityCheck(cc, tvisits, c, path);  // if fail, generates edit field to ask pass inside

    if (allowed) {

        if (rq == RQ_UNKNOWN) {

            ERROR2("[WS]: (%d) Method %s is not supported", tvisits, (method?method:"UNKNOWN"));
            sendError(cc->connDescriptor, 501, "Not supported", NULL, "Method is not supported.");

        } else if (strcmp(path,"/")  == 0 || strcmp(path,"\n") == 0) {

            renderForm(tvisits, cc);

        } else if (strcmp(path,"/favicon.ico") == 0) {

            sendFavicon(cc->connDescriptor);

        } else if (strstr((path+1),"/")  == NULL && strstr(path,".png")) {	// icon

            sendIcon(cc->connDescriptor, path);
            
        } else if ((p = strstr(path,".key"))) {  // button pressed, web server

            wait = processHtmlButtonPress(tvisits,p);

        } else if ((p = strstr(path,XML_BUTTON_PRESS))) {  // button pressed, cm-xml

            wait = processXmlButtonPress(tvisits,p);

        } else if ((p = strstr(path,".menu"))) {  // menu in text screen

            wait = processHtmlMenu(tvisits, p);

        } else if ((p = strstr(path,XML_SHORT_MENU))) {  // menu (shorter than XML_SOFTKEY_NUM items) in CMXML

	    wait = processXmlShortMenu(tvisits, p);

        } else if ((p = strstr(path,XML_LIST_MENU))) {  // list short menu handling in CMXML
             
	    wait = processXmlListMenu(tvisits, p);
	    
        } else if ((p = strstr(path,XML_LIST_MENU2))) {  // list menu handling (2nd step) in CMXML

	    wait = processXmlListMenu2(tvisits, p);

        } else if ((p = strstr(path, XML_LONG_MENU))) {  // menu (more than XML_SOFTKEY_NUM items) in CMXML

            processXmlLongMenu(cc, tvisits, p);
	    
        } else if ((p = strstr(path, XML_LIST_MENU_EXT))) {  // list menu extension in CMXML

            processXmlExtMenu(cc, tvisits, p);

        } else if ((p = strstr(path,"list="))) {  // menu in list screen, list item choosed

	    wait = processHtmlListMenu(tvisits, p, path);
	    
        } else if ((p = strstr(path, HTTP_EDITFIELD))) {  // edit field

            wait = processHtmlEfield(tvisits, p, path);
	    
        } else if ((p = strstr(path,XML_EFIELD_SUBMIT))) {  // CMXML edit field OK

            wait = processXmlEfieldSubmit(tvisits, p);

        } else if ((p = strstr(path,XML_EFIELD_CANCEL))) {  // CMXML edit field Cancel

            wait = processXmlEfieldCancel();

        } else if ((p = strstr(path, HTTP_ACTION))) {  // menu in list screen, no list item choosed
            
	    // or key press
            p += strlen(HTTP_ACTION);

            char* btn = p;
            while (isdigit(*btn)) {
                btn++;
            }

            int button = -1;
            if (btn != p) {
                *btn = '\0';
                button = atoi(p);
            }

            wMessage* wm = NULL;

            if (button > 0) {

                INFO2("[WS]: (%d) Got button %d", tvisits, button);

                wm = (wMessage*) malloc(sizeof(wMessage));
                wm->button = button;
                wm->string = (void*) NULL;

            } else {
                char* plus = NULL;
                while((plus = strstr(p,"+"))) {
                    *plus = ' ';  // replace "+" back to spaces
                }

                wm = (wMessage*) malloc(sizeof(wMessage));
                wm->button = -1;
                wm->string = stringNew("Msg:");
                stringAppend(wm->string, p);
                stringAppend(wm->string, "(,)");
            }

            if (wm) {
                sendToWebServer(wm);
                wait = BOOL_YES;
            }

        } else if (strstr(path,"xml_layout")) {  // ControlForm as image in CMXML

            sendGeneratedImage(cc->connDescriptor, "layout_xml.png");

        } else if (strstr(path,"xml_image")) {   // WindowManager screen as image in CMXML

            renderXMLImage();
            sendGeneratedImage(cc->connDescriptor, "image_xml.png");

        } else if (strstr(path,"anyremote.gui")) {   // iViewer GUI file reguest
        
            sendIViewerGui(cc->connDescriptor);
       
        } else if (strstr(path,"generated_cover")) {	// cover for iViewer
            
            sendGeneratedImage(cc->connDescriptor, "generated_cover");
     
        } else { // treat as file/dir name

            string_t* fpath = stringNew("");

            if (getIViewer()) {   // search images in $(CfgData)/Icons/iViewer

                stringAppend(fpath, confDir);
                stringAppend(fpath,"/Utils/iViewer/");
            }

            stringAppend(fpath, path);

            if (stat(fpath->str, &statbuf) < 0) {
                ERROR2("[WS]: (%d) File %s not found", tvisits, fpath->str);
                sendError(cc->connDescriptor, 404, "Not Found", NULL, "File not found.");

                /*} else if (S_ISDIR(statbuf.st_mode)) {

                	//INFO2("[WS]: (%d)  Directory listing requested", tvisits);

                	len = fpath->len;
                	if (len == 0 || fpath->str[len - 1] != '/') {

                	    snprintf(pathbuf, sizeof(pathbuf), "Location: %s/", fpath->str);
                	    sendError(fd, 302, "Found", pathbuf, "Directories must end with a slash.");

                	} else {
                	    snprintf(pathbuf, sizeof(pathbuf), "%sindex.html", path);
                	    if (stat(pathbuf, &statbuf) >= 0) {

                        	sendFile(fd, pathbuf, &statbuf);

                	    } else {
                        	DIR *dir;
                        	struct dirent *de;

                        	sendHeaders(fd, 200, "OK", NULL, "text/html", -1, statbuf.st_mtime);
                        	sprintf(f, "<HTML><HEAD><TITLE>Index of %s</TITLE></HEAD>\r\n<BODY>", fpath->str);
                    	bytes_sent = send(fd,f,strlen(f),0);

                        	sprintf(f, "<H4>Index of %s</H4>\r\n<PRE>\n", path);
                    	sendData(fd, f);

                    	sendData(fd, "Name Last Modified Size\r\n");
                    	sendData(fd, "<HR>\r\n");

                        	//if (len > 1) {
                    	//    sprintf(f, "<A HREF=\"..\">..</A>\r\n");
                        	//    sendData(fd, f);
                    	//}
                	//INFO2("[Thread] Parse %s\n", path);

                        	dir = opendir(fpath->str);
                        	while ((de = readdir(dir)) != NULL) {
                        	    char timebuf[32];
                        	    struct tm *tm;
                        	     strcpy(pathbuf, path);
                        	    strcat(pathbuf, de->d_name);

                        	    stat(pathbuf, &statbuf);
                        	    tm = gmtime(&statbuf.st_mtime);
                        	    strftime(timebuf, sizeof(timebuf), "%d-%b-%Y %H:%M:%S", tm);

                        	    sprintf(f, "<A HREF=\"%s%s\">", de->d_name, S_ISDIR(statbuf.st_mode) ? "/" : "");
                        	    sendData(fd, f);

                	    sprintf(f, "%s%s", de->d_name, S_ISDIR(statbuf.st_mode) ? "/</A>" : "</A> ");
                        	    sendData(fd, f);

                        	    if (_D_EXACT_NAMLEN(de) < 32) {
                        	sprintf(f, "%*s", (int) (32 - _D_EXACT_NAMLEN(de)), "");
                                	sendData(fd, f);
                        	    }
                        	    if (S_ISDIR(statbuf.st_mode)) {
                                	sprintf(f, "%s\r\n", timebuf);
                                	sendData(fd, f);
                        	    } else {
                                	sprintf(f, "%s %10d\r\n", timebuf, (int) statbuf.st_size);
                                	sendData(fd, f);
                        	    }
                        	}
                        	closedir(dir);

                        	sprintf(f, "</PRE>\r\n<HR>\r\n<ADDRESS>%s</ADDRESS>\r\n</BODY></HTML>\r\n", SERVER);
                	    }
                	}*/

            } else {

                INFO2("[WS] (%d) Send file", tvisits);
                sendFile(cc->connDescriptor, fpath->str, &statbuf);
            }

            stringFree(fpath, BOOL_YES);
        }
    }

    if (wait) {
        mutexLock(M_WEB);
        answerReady = BOOL_NO;
        mutexUnlock(M_WEB);

        int waitingTime = 0;

        while (!answerReady && waitingTime < 1500) {	// 1/50*1500 = 30 sec
            // Main loop timer (1/50 of second)
            usleep(20000);
            waitingTime++;
        }

        INFO2("[WS]: (%d) Wait answer for %d (1/50 sec)", tvisits, waitingTime);

        renderForm(tvisits, cc);
    }

    close(cc->connDescriptor);

    mutexLock(M_WEB);
    
    //INFO2("[WS]: (%d) clientSockets before remove #%d", tvisits, listSingleLength(clientSockets));
    clientSockets = listSingleRemove(clientSockets, CAST_INT_TO_POINTER(cc->connDescriptor));
    //INFO2("[WS]: (%d) clientSockets after remove #%d", tvisits, listSingleLength(clientSockets));
    
    free(data);
    
    mutexUnlock(M_WEB);

    INFO2("[WS]: (%d) Request processed", tvisits);
    threadExit(T_MAX);

    return (void*)0;
}

static pointer_t webServerMain(pointer_t data)
{
    ConnectInfo* conn = (ConnectInfo*) data;
    _WebConnection* cn = (_WebConnection*) conn->connectionData;
    
    queueNew(Q_WEB);

    confDir = getCfgDir();
    
    if (conn->mode == SERVER_WEB) {
        initHtmlGenerator();
    }
    
    char* v2 = dupVarValue("RefreshPage");
    if (v2 != NULL) {
        refreshPage = atoi(v2);
        if (refreshPage <= 0) {
            refreshPage = -1;
        }
        free(v2);
    }

    INFO2("[WS]: $(RefreshPage) = %d", refreshPage);

    //if (runMode == SERVER_CMXML) {  iViewer also use this
        char* v3 = dupVarValue("IpAddr");
        if (v3 != NULL) {
            INFO2("[WS]: $(IpAddr) = %s", v3);
            serverIP = stringNew(v3);
            free(v3);
        } else { // try to get it ourselves
            string_t* cmd = stringNew("/sbin/ifconfig|grep 'inet addr'|cut -f 2 -d ':'|cut -f 1 -d ' '|grep -v 127.0.0.1|head -1");

            size_t sz = 0;
            char* buf = executeCommandPipe(cmd->str, &sz);
            if (buf) {
                if (*(buf+sz-1) == '\n') {
                    *(buf+sz-1) = '\0';   // strip new-line
                }
                serverIP = stringNew(buf);

                free(buf);
            }
        }
        INFO2("[WS]: set Server IP as = %s", (serverIP ? serverIP->str : "not determined"));
    //}

    struct sockaddr_in sin;
    memset((void *) &sin, 0, sizeof(sin));

    cn->fileDescriptor = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);

    if (cn->fileDescriptor < 0) {
        logger(L_ERR, "[WS]: webServerMain: Failed to create a socket");

        free(conn->connectionData);
	conn->connectionData = NULL;
	
        threadExit(T_WEB);
        return (void*)0;
    }

    int optval = 1;
    setsockopt(cn->fileDescriptor, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval));

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = INADDR_ANY;
    sin.sin_port = htons(conn->port);
    int ret = bind(cn->fileDescriptor, (struct sockaddr *) &sin, sizeof(sin));
    if (ret < 0) {
        logger(L_ERR, "[WS]: webServerMain: Failed to bind a socket");
        printf("ERROR: on binding %d->%s\n", errno, strerror(errno));

        gotExitSignal = 1;

	if (cn->fileDescriptor != -1) {
            close(cn->fileDescriptor);
            cn->fileDescriptor = -1;
	}

        free(conn->connectionData);
	conn->connectionData = NULL;

        threadExit(T_WEB);
        return (void*)0;
    }

    listen(cn->fileDescriptor, 100);
    INFO2("[WS]: webServerMain: HTTP server listening on port %d", conn->port);

    while (runWeb) {

        logger(L_INF, "[WS]: webServerMain: HTTP server ready to accept connection");

        int s = accept(cn->fileDescriptor, NULL, NULL);
        if (s < 0) {
            errnoDebug("[WS]: accept() ",errno);  // testing debug
            if (errno == EAGAIN) {
                continue;
            } else {
                break;
            }
        }

	char buf[INET6_ADDRSTRLEN];
	peerName(s,buf,INET6_ADDRSTRLEN);
 
        INFO2("[WS]: webServerMain: HTTP server accepts connection from %s", buf);
	if (isAllowed(buf)) {
            INFO2("[WS]: webServerMain: connection accepted");
            
	    _WebClientConnection* ptr = (_WebClientConnection*) malloc(sizeof(_WebClientConnection));
            ptr->serverPort     = conn->port;
            ptr->connDescriptor = s;
            ptr->runMode        = conn->mode;

            mutexLock(M_WEB);
            clientSockets = listSingleAppend(clientSockets, CAST_INT_TO_POINTER(s));
            mutexUnlock(M_WEB);

            int rq = threadNew(T_MAX, serveRequest, (void *) ptr, DETACHED);
            if (rq != RC_OK) {
        	logger(L_ERR, "[WS]: webServerMain: Can not run processing thread");
            }
	} else {
            INFO2("[WS]: webServerMain: host %s is not in the list of accepted host, skip connection", buf);
	    close(s);
	}
    }

    logger(L_ERR, "[WS]: webServerMain: exit thread");
    if (cn->fileDescriptor != -1) {
        close(cn->fileDescriptor);
        cn->fileDescriptor = -1;
    }
    
    free(conn->connectionData);
    conn->connectionData = NULL;

    threadExit(T_WEB);
    return (void*)0;
}

static int openWebInternal(ConnectInfo* conn)
{
    if (threadExists(T_WEB) == RC_OK) {
        INFO2("[WS]: openWebInternal, skip initialization");
        return 0;
    }
 
    if (conn->connectionData) {
        free(conn->connectionData);
    }
    
    conn->connectionData = (_WebConnection*) malloc(sizeof(_WebConnection));
    _WebConnection* cn = (_WebConnection*) conn->connectionData;
    cn->fileDescriptor = -1;

    mutexNew(M_WEB);
    
    initState();

    threadNew(T_WEB,webServerMain,conn,JOINABLE);

    return 0;
}

int openWeb(ConnectInfo* conn)
{
    DEBUG2("[DS]: Web Server mode. Use port %d", conn->port);
    if (openWebInternal(conn) < 0) {
        conn->connected = BOOL_NO;
        return EXIT_NOK;
    }
    
     
    conn->connected = BOOL_YES;
    return EXIT_OK;
}

int checkWebPort(char* buf, int capacity)
{
    //logger(L_DBG,"[DS]: checkWebPort");

    if (queueExists(Q_WEB) != RC_OK) {
        return 0;
    }

    //logger(L_DBG,"[WS]: checkWebPort queueExists");

    int l = 0;

    // Verify commands from queue (timeout about 1/2 sec)
    wMessage* wm = (wMessage*) queuePop(Q_WEB);
    if (wm != NULL) {
        logger(L_DBG, "[DS]: Got event");

        if (wm->button > 0) {
            INFO2("[DS]: Button pressed %d", wm->button);
            char bbuf[16];
            if (wm->button == 10) {
                sprintf(bbuf,"Msg:%c",'*');
            } else if (wm->button == 11) {
                sprintf(bbuf,"Msg:%c",'0');
            } else if (wm->button == 12) {
                sprintf(bbuf,"Msg:%c",'#');
            } else {
                sprintf(bbuf,"Msg:%d",wm->button);
            }
            strncpy(buf,bbuf,capacity);
            l = strlen(buf);
        } else if (wm->string) {
            INFO2("[DS]: Data %s", wm->string->str);

            strncpy(buf,wm->string->str,capacity);
            l = strlen(wm->string->str);
        }
        freeWMessage(wm);
    }
    return l;
}


void closeWebPort(ConnectInfo* conn, int final)
{
    INFO2("[WS]: closeWebPort %d", final);
    _WebConnection* cn = (_WebConnection*) conn->connectionData;

    // close all sockets
    if (cn->fileDescriptor != -1) {
        close(cn->fileDescriptor);
        cn->fileDescriptor = -1;
    }

    mutexLock(M_WEB);

    //INFO2("[WS]: clientSockets before cleanup #%d", listSingleLength(clientSockets));

    SingleList* list = clientSockets;
    while (list) {

        int clientFD = CAST_POINTER_TO_INT(list->data);
        if (clientFD > 0) {
            close(clientFD);
        }
        list = listSingleNext(list);
    }

    listSingleFree(clientSockets);
    clientSockets = NULL;

    mutexUnlock(M_WEB);

    if (final) {
        runWeb = BOOL_NO;
	
	freeState();

        if (serverIP) {
            stringFree(serverIP, BOOL_YES);
        }

        if (confDir) {
            free(confDir);
        }

        mutexRemove(M_WEB);
        queueRemove(Q_WEB, freeWMessage);
    }

    return;
}

static void stripCommandEnding(char *s)
{
    if (s && strlen(s) >= 2) {
        s[strlen(s)-2] = '\0';   // strip );
    }
}

//
// user can add Get(password) inside cfg.file, so handle it also
//
static void getPass()
{
    logger(L_INF, "[WS]: getPass");

    m_secure    = NEED_SEND_COOKIE;
    
    srandom((unsigned int) time(NULL));
    m_cookie = random();
    
    setPassField();
}

int writeWebConnStr(const char* value, int mode)
{
    DEBUG2("[WS]: writeWebConnStr %d", (value ? (int) strlen(value) : -1));

    wMessage* wm  = NULL;
    wMessage* wm2 = NULL;

    char * cmd = strdup(value);

    //boolean_t skipSpaces = BOOL_NO;
    //INFO2("[WS]: parse lenght %d", (cmd ? (int) strlen(cmd) : 0));

    stripCommandEnding(cmd);

    if (strlen(cmd) < MAXMAXLEN) {
        INFO2("[WS]: parse %d %s", curForm(), cmd);
    } else {
        INFO2("[WS]: parse %d ... command too long ...", curForm());
    }

    char* token = strtok(cmd,",");
    while (isspace(*token)) {
        ++token;
    }

    mutexLock(M_WEB);

    if (strncmp(token,"Set(status",10) == 0) {
        setCfStatus(cmd+strlen("Set(status,"));
    } else if (strncmp(token,"Set(title",9) == 0) {
        setCfTitle(cmd+strlen("Set(title,"));
    } else if (strncmp(token,"Set(icons",9) == 0) {
        setIcons();
        xmlSetLayoutOk(BOOL_NO);  // Need to regenerate
    } else if (strncmp(token,"Set(font",8) == 0) {
        setFont(CF);
    } else if (strncmp(token,"Set(fg",6) == 0) {
        setFgBg(CF,BOOL_YES);
    } else if (strncmp(token,"Set(bg",6) == 0) {
        setFgBg(CF,BOOL_NO);
    } else if (strncmp(token,"Set(volume",10) == 0) {
        char* sz = strtok(NULL,",");
        setCfVolume(sz);
    } else if (strncmp(token,"Set(layout",10) == 0 ||
               strncmp(token,"Set(skin",8) == 0) {
        setSkin();
	xmlSetLayoutOk(BOOL_NO);  // Need to regenerate
    } else if (strncmp(token,"Set(cover",9) == 0) {
        ERROR2("[WS]: Improperly formed command Set(cover,...)");
    } else if (strncmp(token,"Set(list",8) == 0) {
        setList(BOOL_NO);
    } else if (strncmp(token,"Set(iconlist",12) == 0) {
        setList(BOOL_YES);
    } else if (strncmp(token,"Set(text",8) == 0) {
        setText(BOOL_NO);
    } else if (strncmp(token,"Set(menu",8) == 0) {
        setMenu();
    } else if (strncmp(token,"Set(editfield",13) == 0) {
        setEditfield();
    } else if (strncmp(token,"Set(image",9) == 0) {
        ERROR2("[WS]: Improperly formed command Set(image,...)");
    } else if (strncmp(token,"Set(popup",9) == 0) {
        // ignore
    } else if (strncmp(token,"Set(disconnect",14) == 0) {
        // ignore
    } else if (strncmp(token,"Set(fullscreen",16) == 0) {
        // ignore
    } else if (strncmp(token,"Set(vibrate",11) == 0) {
        // ignore
    } else if (strncmp(token,"Set(parameter",13) == 0) {
        setParams();
    } else if (strncmp(token,"Get(password",12) == 0) {
        getPass();
    } else if (strncmp(token,"Get(screen_size",16) == 0) {

        int w = htmlScreenWidth ();
        int h = htmlScreenHeight();
        if  (mode == SERVER_CMXML) {
            w = xmlScreenWidth ();
            h = xmlScreenHeight();
	}
        
	char b[32];

        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = stringNew("SizeX(");
	
        const char *buf = i2string(w,b);
        stringAppend(wm->string,buf);
	
        stringAppend(wm->string,",)");

        wm2 = (wMessage*) malloc(sizeof(wMessage));
        wm2->button = -1;
	
        wm2->string = stringNew("SizeY(");
	
	buf = i2string(h,b);
        stringAppend(wm2->string,buf);
	
        stringAppend(wm2->string,",)");

    } else if (strncmp(token,"Get(icon_padding",16) == 0) {

        char b[32];
        const char *buf = i2string(iconPadding(),b);

        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = stringNew("IconPadding(");
        stringAppend(wm->string,buf);
        stringAppend(wm->string,",)");

    } else if (strncmp(token,"Get(model",9) == 0) {

        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = stringNew("WebInterface");

    } else if (strncmp(token,"Get(is_exists",13) == 0) {

        char* sz = strtok(NULL,",");
        char* icon = strtok(NULL,",");
        if (sz && icon) {
            while (isspace(*sz)) {
                ++sz;
            }
            while (isspace(*icon)) {
                ++icon;
            }

            wm = (wMessage*) malloc(sizeof(wMessage));
            wm->button = -1;
            wm->string = stringNew("IconExists(");
            stringAppend(wm->string,sz);
            stringAppend(wm->string,",");
            stringAppend(wm->string,icon);
        }

    } else if (strncmp(token,"Get(cover_size",14) == 0) {

        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = stringNew("CoverSize(");
        char b[32];

        stringAppend(wm->string,
                     i2string((mode == SERVER_CMXML ? xmlScreenHeight() : (htmlScreenHeight()*8)/10),b));

        stringAppend(wm->string,",)");

    } else if (strncmp(token,"Get(version",11) == 0) {

        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = stringNew("Version(,web_interface)");

    } else if (strncmp(token,"Get(cursor",10) == 0) {
        // ignore
    } else if (strncmp(token,"Get(ping",8) == 0) {

        wm = (wMessage*) malloc(sizeof(wMessage));
        wm->button = -1;
        wm->string = stringNew("Msg:Ping");

    } else if (strncmp(token,"End(",4) == 0) {
        answerReady = BOOL_YES;
    } else {
        ERROR2("[WS]: Unknown command %s", token);
    }

    mutexUnlock(M_WEB);

    if (wm) {
        sendToWebServer(wm);
    }
    if (wm2) {
        sendToWebServer(wm2);
    }

    free(cmd);
    INFO2("[WS]: parsed %d", curForm());
    return EXIT_OK;
}

int writeWebConnFile(const dMessage* dm)
{
    //INFO2("[WS]: writeWebConnFile: %s", (const char*) dm->value);

    mutexLock(M_WEB);

    if (strncmp(dm->value,"Set(cover",9) == 0) {
        setCfCover(dm->file);
        xmlSetLayoutOk(BOOL_NO);  // Need to regenerate
    } else if (strncmp(dm->value,"Set(image",9) == 0) {
        setImage((dm->value + 10), dm->file);  // 10 = 9 + ","
    } else {
        ERROR2("[WS]: Unknown command file:%s", (const char*) dm->value);
    }

    mutexUnlock(M_WEB);

    INFO2("[WS]: parsed %d", curForm());
    return EXIT_OK;
}

int writeWebConn(const dMessage* dm, int mode)
{
    INFO2("[WS]: writeWebConn (%d)", dm->size);
    if (dm->type == DM_SET) {

        return writeWebConnStr(dm->value, mode);

    } else if (dm->type == DM_SETFILE) {

        return writeWebConnFile(dm);

    } else {
        ERROR2("[WS]: Not supported");
        return EXIT_OK;
    }
}

//
// Special case for Web/CMXML servers
//
int writeWeb(ConnectInfo* peer, const dMessage* dm)
{
    // suppose peer->mode == SERVER_WEB || peer->mode == SERVER_CMXML
    if (dm->type == DM_SENDB || dm->type == DM_SENDS) {
        return EXIT_NOK;  // skip that
    }
    return writeWebConn(dm, peer->mode);
}

