open! Dune_engine
open Import

let add_diff sctx loc alias ~dir ~input ~output =
  let open Action_builder.O in
  let action = Action.Chdir (Path.build dir, Action.diff input output) in
  Super_context.add_alias_action sctx alias ~dir ~loc:(Some loc)
    (Action_builder.paths [ input; Path.build output ]
    >>> Action_builder.return (Action.Full.make action))

let rec subdirs_until_root dir =
  match Path.parent dir with
  | None -> [ dir ]
  | Some d -> dir :: subdirs_until_root d

let depend_on_files ~named dir =
  subdirs_until_root dir
  |> List.concat_map ~f:(fun dir -> List.map named ~f:(Path.relative dir))
  |> Action_builder.paths_existing

let formatted_dir_basename = ".formatted"

let gen_rules_output sctx (config : Format_config.t) ~version ~dialects
    ~expander ~output_dir =
  assert (formatted_dir_basename = Path.Build.basename output_dir);
  let loc = Format_config.loc config in
  let dir = Path.Build.parent_exn output_dir in
  let source_dir = Path.Build.drop_build_context_exn dir in
  let alias_formatted = Alias.fmt ~dir:output_dir in
  let depend_on_files named = depend_on_files ~named (Path.build dir) in
  let setup_formatting file =
    let input_basename = Path.Source.basename file in
    let input = Path.Build.relative dir input_basename in
    let output = Path.Build.relative output_dir input_basename in
    let formatter =
      let input = Path.build input in
      match Path.Source.basename file with
      | "dune" when Format_config.includes config Dune ->
        Option.some
        @@ Action_builder.with_file_targets ~file_targets:[ output ]
        @@
        let open Action_builder.O in
        let+ () = Action_builder.path input in
        Action.Full.make (Action.format_dune_file ~version input output)
      | _ ->
        let ext = Path.Source.extension file in
        let open Option.O in
        let* dialect, kind = Dialect.DB.find_by_extension dialects ext in
        let* () =
          Option.some_if
            (Format_config.includes config (Dialect (Dialect.name dialect)))
            ()
        in
        let+ loc, action, extra_deps =
          match Dialect.format dialect kind with
          | Some _ as action -> action
          | None -> (
            match Dialect.preprocess dialect kind with
            | None -> Dialect.format Dialect.ocaml kind
            | Some _ -> None)
        in
        let src = Path.as_in_build_dir_exn input in
        let extra_deps =
          match extra_deps with
          | [] -> Action_builder.return ()
          | extra_deps -> depend_on_files extra_deps
        in
        let open Action_builder.With_targets.O in
        Action_builder.with_no_targets extra_deps
        >>> Preprocessing.action_for_pp_with_target ~loc ~expander ~action ~src
              ~target:output
    in
    Memo.Build.Option.iter formatter ~f:(fun action ->
        let open Memo.Build.O in
        Super_context.add_rule sctx ~mode:Standard ~loc ~dir action
        >>> add_diff sctx loc alias_formatted ~dir ~input:(Path.build input)
              ~output)
  in
  let open Memo.Build.O in
  let* () =
    Source_tree.files_of source_dir
    >>= Memo.Build.parallel_iter_set
          (module Path.Source.Set)
          ~f:setup_formatting
  in
  Rules.Produce.Alias.add_deps alias_formatted (Action_builder.return ())

let gen_rules sctx ~output_dir =
  let open Memo.Build.O in
  let dir = Path.Build.parent_exn output_dir in
  let* config = Super_context.format_config sctx ~dir in
  Memo.Build.when_
    (not (Format_config.is_empty config))
    (fun () ->
      let* expander = Super_context.expander sctx ~dir in
      let scope = Super_context.find_scope_by_dir sctx output_dir in
      let project = Scope.project scope in
      let dialects = Dune_project.dialects project in
      let version = Dune_project.dune_version project in
      gen_rules_output sctx config ~version ~dialects ~expander ~output_dir)

let setup_alias sctx ~dir =
  let open Memo.Build.O in
  let* config = Super_context.format_config sctx ~dir in
  Memo.Build.when_
    (not (Format_config.is_empty config))
    (fun () ->
      let output_dir = Path.Build.relative dir formatted_dir_basename in
      let alias = Alias.fmt ~dir in
      let alias_formatted = Alias.fmt ~dir:output_dir in
      Rules.Produce.Alias.add_deps alias
        (Action_builder.dep (Dep.alias alias_formatted)))
