#include "unalf.h"
#include "addrs.h"
#include "sanity.h"
#include <time.h>
#include <utime.h>

int bad_checksum, bad_checksum_count = 0;
int new_file = 0;
unsigned int bytes_written = 0;
char *out_filename;

void dpoke(int addr, u16 value) {
	mem[addr] = value & 0xff;
	mem[addr + 1] = value >> 8;
}

u16 dpeek(int addr) {
	return mem[addr] | (mem[addr + 1] << 8);
}

void fix_filename(void) {
	char *p;

	if(!out_filename) return;

	if(strlen(out_filename) > 12) {
		fprintf(stderr, "%s: filename in ALF header not null-terminated, fixing.\n", self);
		out_filename[12] = '\0';
	}

	sanity_check_filename(out_filename);

	if(opts.lowercase) {
		for(p = out_filename; *p; p++)
			*p = tolower(*p);
	}

	if(!opts.keepdot) {
		for(p = out_filename; *p; p++)
			if(p[0] == '.' && !p[1])
				*p = '\0';
	}
}

void set_datetime() {
	u16 t, d;
	struct tm tm;
	time_t time;
	struct utimbuf utb;

	t = dpeek(alf_hdr_time0);

	/* don't set the date/time at all, if it's zero */
	if(!t) {
		fprintf(stderr, "%s: warning: date/time for %s is null, using current date/time\n", self, out_filename);
		return;
	}

	d = dpeek(alf_hdr_date0);

	tm.tm_isdst = -1;
	tm.tm_sec = (t & 0x1f) << 1;
	tm.tm_hour = t >> 11;
	if(tm.tm_hour == 24) tm.tm_hour = 0;
	tm.tm_min = (t >> 5) & 0x3f;

	tm.tm_year = (d >> 9) + 1980 - 1900;
	tm.tm_mon = ((d >> 5) & 0x0f) - 1;
	tm.tm_mday = (d & 0x1f);

	time = mktime(&tm);

	if(time != (time_t) -1) {
		utb.actime = utb.modtime = time;
		utime(out_filename, &utb); /* ignore errors */
	}
}

void make_backup(void) {
	/* up to 12-char FILENAME.EXT, plus a ~, plus null terminator = 14 */
	char backup[14];

	strncpy(backup, out_filename, 13);
	strncat(backup, "~", 13);

	/* silently ignore errors! */
	rename(out_filename, backup);
}

void extract_alf(void) {
	/* get ready to call fake 6502 stuff. set up memory like the Atari. */
	dpoke(MEMTOP, 0xbc1f);

	while(read_alf_header()) {
		out_filename = (char *)(mem + alf_hdr_filename);

		fix_filename();

		if(!file_wanted(out_filename)) {
			if(fseek(in_file, getquad(alf_hdr_compsize0), SEEK_CUR) != 0) {
				fprintf(stderr, "%s: fatal: seek failed on input!\n", self);
				exit(1);
			}
			out_filename = 0;
			continue;
		}

		if(!opts.quiet) {
			printf("%s %s\n", opts.testonly ? "Testing" : "Uncrunching", out_filename);
		}

		if(opts.extract_to_stdout) {
			out_file = stdout;
		} else {
			char *realname = out_filename;

			if(opts.testonly) {
				out_filename = "/dev/null";
			} else if(!opts.overwrite) {
				make_backup();
			}

			if(!(out_file = fopen(out_filename, "wb"))) {
				fprintf(stderr, "%s: fatal:  ", self);
				perror(out_filename);
				exit(1);
			}

			out_filename = realname;
		}

		bad_checksum = bytes_written = 0;
		new_file = 1;
		uncrunch_file();
		if(bad_checksum) bad_checksum_count++;
		if(bytes_written != getquad(alf_hdr_origsize0))
			fprintf(stderr, "%s: %s should be %u bytes, but extracted to %u.\n",
					self, out_filename, getquad(alf_hdr_origsize0), bytes_written);

		if(!opts.extract_to_stdout) {
			fclose(out_file);
			if(!opts.ignore_datetime)
				set_datetime();
		}
		out_filename = 0;
	}

	if(opts.testonly && !opts.quiet) {
		if(bad_checksum_count)
			printf("%d file%s with bad checksum!\n",
					bad_checksum_count,
					bad_checksum_count == 1 ? "" : "s");
		else
			printf("All files OK.\n");
	}
}

void chksum_err(void) {
	bad_checksum = 1;
	fprintf(stderr, "%s: checksum error on %s\n", self, out_filename);
}

void truncated_err(void){
	fprintf(stderr, "%s: fatal: compressed data is truncated, EOF before end marker\n", self);
	exit(1);
}

void stack_underrun(void){
	fprintf(stderr, "%s: fatal: stack underrun\n", self);
	exit(1);
}

void stack_overrun(void){
	fprintf(stderr, "%s: fatal: stack overrun\n", self);
	exit(1);
}
