/*****************************************************************\
*       64-bit BBC BASIC Interpreter                              *
*       Copyright (c) 2021 R.T.Russell http://www.rtrussell.co.uk *
*                                                                 *
*       bbasmb_x86_64.c: x86-64 assembler                         *
*       Version 1.23a, 05-Jun-2021                                *
\*****************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <limits.h>
#include "BBC.h"

#ifndef __WINDOWS__
#define stricmp strcasecmp
#define strnicmp strncasecmp
#endif

#if defined(__x86_64__) || defined(__aarch64__)
#define OC ((unsigned int) stavar[15] + (void *)((long long) stavar[13] << 32)) 
#define PC ((unsigned int) stavar[16] + (void *)((long long) stavar[17] << 32)) 
#else
#define OC (void *) stavar[15]
#define PC (void *) stavar[16]
#endif

// External routines:
void newlin (void) ; 
void *getput (unsigned char *) ; 
void error (int, const char *) ; 
void token (signed char) ; 
void text (const char*) ; 
void crlf (void) ; 
void comma (void) ; 
void spaces (int) ; 
int range0 (char) ; 
signed char nxt (void) ; 

long long expri (void) ; 
VAR expr (void) ; 
VAR exprs (void) ; 
VAR loadn (void *, unsigned char) ; 
void storen (VAR, void *, unsigned char) ; 

// Routines in bbcmos.c:
void *sysadr (char *) ; 
unsigned char osrdch (void) ; 
void oswrch (unsigned char) ; 
int oskey (int) ; 
void osline (char *) ; 
int osopen (int, char *) ; 
void osshut (int) ; 
unsigned char osbget (int, int *) ; 
void osbput (int, unsigned char) ; 
long long getptr (int) ; 
void setptr (int, long long) ; 
long long getext (int) ; 
void oscli (char *) ;
int osbyte (int, int) ;
void osword (int, void*) ;

// instruction mnemonics (alphabetically ordered):
// No privileged ops, no SSSE3, no SSE4, no AVX

#define CMPPD 0x3C
#define CMPPS 0x3D
#define CMPSD 0x3F
#define CMPSS 0x41
#define CMPXCHG16B 0x44
#define FPLO 0x72
#define FILD 0x93
#define FISTP 0x98
#define FLD 0x9C
#define FSTP 0xBD
#define FPHI 0xCB // <= FXCH
#define LOCK 0xFF
#define OPT 0x141
#define REPLO 0x196
#define REPHI 0x19A
#define XCHG 0x1ED

static char *mnemonics[] = {
	"ADC","ADD","ADDPD","ADDPS","ADDSD","ADDSS","ADDSUBPD","ADDSUBPS",
	"AND","ANDNPD","ANDNPS","ANDPD","ANDPS","BSF","BSR","BSWAP",
	"BT","BTC","BTR","BTS","CALL","CBW","CDQ","CDQE",
	"CLC","CLD","CLFLUSH","CLI","CMC","CMOVA","CMOVAE","CMOVB",
	"CMOVBE","CMOVC","CMOVE","CMOVG","CMOVGE","CMOVL","CMOVLE","CMOVNA",
	"CMOVNAE","CMOVNB","CMOVNBE","CMOVNC","CMOVNE","CMOVNG","CMOVNGE","CMOVNL",
	"CMOVNLE","CMOVNO","CMOVNP","CMOVNS","CMOVNZ","CMOVO","CMOVP","CMOVPE",
	"CMOVPO","CMOVS","CMOVZ","CMP","CMPPD","CMPPS","CMPSB","CMPSD",
// 040
	"CMPSQ","CMPSS","CMPSW","CMPXCHG","CMPXCHG16B","CMPXCHG8B","COMISD","COMISS",
	"CPUID","CQO","CVTDQ2PD","CVTDQ2PS","CVTPD2DQ","CVTPD2PI","CVTPD2PS","CVTPI2PD",
	"CVTPI2PS","CVTPS2DQ","CVTPS2PD","CVTPS2PI","CVTSD2SI","CVTSD2SS","CVTSI2SD","CVTSI2SS",
	"CVTSS2SD","CVTSS2SI","CVTTPD2DQ","CVTTPD2PI","CVTTPS2DQ","CVTTPS2PI","CVTTSD2SI","CVTTSS2SI",
	"CWD","CWDE","DB","DD","DEC","DIV","DIVPD","DIVPS",
	"DIVSD","DIVSS","DQ","DW","EMMS","ENTER","EQUB","EQUD",
	"EQUQ","EQUW","F2XM1","FABS","FADD","FADDP","FBLD","FBSTP",
	"FCHS","FCLEX","FCMOVB","FCMOVBE","FCMOVE","FCMOVNB","FCMOVNBE","FCMOVNE",
// 080
	"FCMOVNU","FCMOVU","FCOM","FCOMI","FCOMIP","FCOMP","FCOMPP","FCOS",
	"FDECSTP","FDIV","FDIVP","FDIVR","FDIVRP","FFREE","FIADD","FICOM",
	"FICOMP","FIDIV","FIDIVR","FILD","FIMUL","FINCSTP","FINIT","FIST",
	"FISTP","FISTTP","FISUB","FISUBR","FLD","FLD1","FLDCW","FLDENV",
	"FLDL2E","FLDL2T","FLDLG2","FLDLN2","FLDPI","FLDZ","FMUL","FMULP",
	"FNCLEX","FNINIT","FNOP","FNSAVE","FNSTCW","FNSTENV","FNSTSW","FPATAN",
	"FPREM","FPREM1","FPTAN","FRNDINT","FRSTOR","FSAVE","FSCALE","FSIN",
	"FSINCOS","FSQRT","FST","FSTCW","FSTENV","FSTP","FSTSW","FSUB",
// 0C0
	"FSUBP","FSUBR","FSUBRP","FTST","FUCOM","FUCOMI","FUCOMIP","FUCOMP",
	"FUCOMPP","FWAIT","FXAM","FXCH","FXRSTOR","FXSAVE","FXTRACT","FYL2X",
	"FYL2XP1","HADDPD","HADDPS","HSUBPD","HSUBPS","IDIV","IMUL","INC",
	"JA","JAE","JB","JBE","JC","JE","JECXZ","JG",
	"JGE","JL","JLE","JMP","JNA","JNAE","JNB","JNBE",
	"JNC","JNE","JNG","JNGE","JNL","JNLE","JNO","JNP",
	"JNS","JNZ","JO","JP","JPE","JPO","JRCXZ","JS",
	"JZ","LAHF","LDDQU","LDMXCSR","LEA","LEAVE","LFENCE","LOCK",
// 100
	"LODSB","LODSD","LODSQ","LODSW","LOOP","LOOPE","LOOPNE","LOOPNZ",
	"LOOPZ","LSL","MASKMOVDQU","MASKMOVQ","MAXPD","MAXPS","MAXSD","MAXSS",
	"MFENCE","MINPD","MINPS","MINSD","MINSS","MOV","MOVAPD","MOVAPS",
	"MOVBE","MOVD","MOVDDUP","MOVDQ2Q","MOVDQA","MOVDQU","MOVHLPS","MOVHPD",
	"MOVHPS","MOVLHPS","MOVLPD","MOVLPS","MOVMSKPD","MOVMSKPS","MOVNTDQ","MOVNTI",
	"MOVNTPD","MOVNTPS","MOVNTQ","MOVQ","MOVQ2DQ","MOVSB","MOVSD","MOVSHDUP",
	"MOVSLDUP","MOVSQ","MOVSS","MOVSW","MOVSX","MOVSXD","MOVUPD","MOVUPS",
	"MOVZX","MUL","MULPD","MULPS","MULSD","MULSS","NEG","NOP",
// 140
	"NOT","OPT","OR","ORPD","ORPS","PACKSSDW","PACKSSWB","PACKUSWB",
	"PADDB","PADDD","PADDQ","PADDSB","PADDSW","PADDUSB","PADDUSW","PADDW",
	"PAND","PANDN","PAUSE","PAVGB","PAVGW","PCMPEQB","PCMPEQD","PCMPEQW",
	"PCMPGTB","PCMPGTD","PCMPGTW","PEXTRW","PINSRW","PMADDWD","PMAXSW","PMAXUB",
	"PMINSW","PMINUB","PMOVMSKB","PMULHUW","PMULHW","PMULLW","PMULUDQ","POP",
	"POPCNT","POPF","POPFQ","POR","PSADBW","PSHUFD","PSHUFHW","PSHUFLW",
	"PSHUFW","PSLLD","PSLLDQ","PSLLQ","PSLLW","PSRAD","PSRAW","PSRLD",
	"PSRLDQ","PSRLQ","PSRLW","PSUBB","PSUBD","PSUBQ","PSUBSB","PSUBSW",
// 180
	"PSUBUSB","PSUBUSW","PSUBW","PUNPCKHBW","PUNPCKHDQ","PUNPCKHQDQ","PUNPCKHWD","PUNPCKLBW",
	"PUNPCKLDQ","PUNPCKLQDQ","PUNPCKLWD","PUSH","PUSHF","PUSHFQ","PXOR","RCL",
	"RCPPS","RCPSS","RCR","RDPMC","RDTSC","RDTSCP","REP","REPE",
	"REPNE","REPNZ","REPZ","RET","RETF","RETN","ROL","ROR",
	"RSQRTPS","RSQRTSS","SAHF","SAL","SAR","SBB","SCASB","SCASD",
	"SCASQ","SCASW","SETA","SETAE","SETB","SETBE","SETC","SETE",
	"SETG","SETGE","SETL","SETLE","SETNA","SETNAE","SETNB","SETNBE",
	"SETNC","SETNE","SETNG","SETNGE","SETNL","SETNLE","SETNO","SETNP",
// 1C0
	"SETNS","SETNZ","SETO","SETP","SETPE","SETPO","SETS","SETZ",
	"SFENCE","SGDT","SHL","SHLD","SHR","SHRD","SHUFPD","SHUFPS",
	"SMSW","SQRTPD","SQRTPS","SQRTSD","SQRTSS","STC","STD","STI",
	"STMXCSR","STOSB","STOSD","STOSQ","STOSW","SUB","SUBPD","SUBPS",
	"SUBSD","SUBSS","TEST","UCOMISD","UCOMISS","UNPCKHPD","UNPCKHPS","UNPCKLPD",
	"UNPCKLPS","VERR","VERW","WAIT","XADD","XCHG","XLATB","XOR",
	"XORPD","XORPS","XRSTOR","XSAVE" } ;

static char *specials[] = {
	"CMPEQPD","CMPLTPD","CMPLEPD","CMPUNORDPD","CMPNEQPD","CMPNLTPD","CMPNLEPD","CMPORDPD",
	"CMPEQPS","CMPLTPS","CMPLEPS","CMPUNORDPS","CMPNEQPS","CMPNLTPS","CMPNLEPS","CMPORDPS",
	"CMPEQSD","CMPLTSD","CMPLESD","CMPUNORDSD","CMPNEQSD","CMPNLTSD","CMPNLESD","CMPORDSD",
	"CMPEQSS","CMPLTSS","CMPLESS","CMPUNORDSS","CMPNEQSS","CMPNLTSS","CMPNLESS","CMPORDSS" } ;

static int special[4] = {CMPPD, CMPPS, CMPSD, CMPSS} ;
		
// operands:
// Note that 16-bit and 32-bit indirect operands are omitted because they
// cannot address memory in a 64-bit flat addressing space such as BBCSDL's
// 0x7C = [mem], 0x7D = [rip{+disp}], 0x7E = immediate, 0x7F = no operand
static char *operands[] = {
	"AL","CL","DL","BL","AH","CH","DH","BH", 		// 0x : 8-bit registers
	"R8B","R9B","R10B","R11B","R12B","R13B","R14B","R15B",
	"AX","CX","DX","BX","SP","BP","SI","DI",		// 1x : 16-bit registers
	"R8W","R9W","R10W","R11W","R12W","R13W","R14W","R15W",
	"RAX","RCX","RDX","RBX","RSP","RBP","RSI","RDI",	// 2x : 64-bit registers
	"R8","R9","R10","R11","R12","R13","R14","R15",
	"EAX","ECX","EDX","EBX","ESP","EBP","ESI","EDI",	// 3x : 32-bit registers
	"R8D","R9D","R10D","R11D","R12D","R13D","R14D","R15D",
	"[RAX","[RCX","[RDX","[RBX","[RSP","[RBP","[RSI","[RDI",// 4x : 64-bit register indirect
	"[R8","[R9","[R10","[R11","[R12","[R13","[R14","[R15",
	"XMM0","XMM1","XMM2","XMM3","XMM4","XMM5","XMM6","XMM7",// 5x : XMM registers (128 bit)
	"XMM8","XMM9","XMM10","XMM11","XMM12","XMM13","XMM14","XMM15",
	"ST0","ST1","ST2","ST3","ST4","ST5","ST6","ST7",	// 6x : floating-point stack...
	"MM0","MM1","MM2","MM3","MM4","MM5","MM6","MM7",	//      ..shared with MMX regs 
	"ES","CS","SS","DS","FS","GS","","",			// 7x : segment regs and extras
	"SPL","BPL","SIL","DIL","[REL","[RIP","","",
	"BYTE","WORD","DWORD","QWORD","TBYTE","SHORT","NEAR" } ;// 8x : overrides (internal)

// List of instructions:
static unsigned char instructions[] = {
	0x00,0x15,0xC4,0xFD,0x30,0x00, // adc rax,imm
	0x00,0x11,0x34,0x7E,0x3F,0x00, // adc reg,reg
	0x00,0x13,0x14,0x9E,0x3F,0x00, // adc reg,[r/m]
	0x10,0x81,0xFC,0xFD,0x3F,0x00, // adc reg,imm
	0x00,0x11,0x34,0x7E,0x4F,0x00, // adc [r/m],reg
	0x10,0x81,0xFC,0xFD,0x4F,0x00, // adc [r/m],imm
	0x00,0x05,0xC4,0xFD,0xB0,0x00, // add rax,imm
	0x00,0x01,0x34,0x7E,0xBF,0x00, // add reg,reg
	0x00,0x03,0x14,0x9E,0xBF,0x00, // add reg,[r/m]
	0x00,0x81,0xFC,0xFD,0xBF,0x00, // add reg,imm
	0x00,0x01,0x34,0x7E,0xCF,0x00, // add [r/m],reg
	0x00,0x81,0xFC,0xFD,0xCF,0x00, // add [r/m],imm
	0x00,0x58,0x13,0x9E,0x5F,0x01, // addpd xmm,[r/m]
	0x00,0x58,0x13,0xBE,0x5F,0x01, // addpd xmm,xmm
	0x00,0x58,0x12,0x9E,0xDF,0x01, // addps xmm,[r/m]
	0x00,0x58,0x12,0xBE,0xDF,0x01, // addps xmm,xmm
	0xF2,0x58,0x13,0x9E,0x5F,0x02, // addsd xmm,[r/m]
	0xF2,0x58,0x13,0xBE,0x5F,0x02, // addsd xmm,xmm
	0xF3,0x58,0x13,0x9E,0xDF,0x02, // addss xmm,[r/m]
	0xF3,0x58,0x13,0xBE,0xDF,0x02, // addss xmm,xmm
	0x00,0xD0,0x13,0x9E,0x5F,0x03, // addsubpd xmm,[r/m]
	0x00,0xD0,0x13,0xBE,0x5F,0x03, // addsubpd xmm,xmm
	0xF2,0xD0,0x13,0x9E,0xDF,0x03, // addsubps xmm,[r/m]
	0xF2,0xD0,0x13,0xBE,0xDF,0x03, // addsubps xmm,xmm
	0x00,0x25,0xC4,0xFD,0x30,0x04, // and rax,imm
	0x00,0x21,0x34,0x7E,0x3F,0x04, // and reg,reg
	0x00,0x23,0x14,0x9E,0x3F,0x04, // and reg,[r/m]
	0x20,0x81,0xFC,0xFD,0x3F,0x04, // and reg,imm
	0x00,0x21,0x34,0x7E,0x4F,0x04, // and [r/m],reg
	0x20,0x81,0xFC,0xFD,0x4F,0x04, // and [r/m],imm
	0x00,0x55,0x13,0x9E,0xDF,0x04, // andnpd xmm,[r/m]
	0x00,0x55,0x13,0xBE,0xDF,0x04, // andnpd xmm,xmm
	0x00,0x55,0x12,0x9E,0x5F,0x05, // andnps xmm,[r/m]
	0x00,0x55,0x12,0xBE,0x5F,0x05, // andnps xmm,xmm
	0x00,0x54,0x13,0x9E,0xDF,0x05, // andpd xmm,[r/m]
	0x00,0x54,0x13,0xBE,0xDF,0x05, // andpd xmm,xmm
	0x00,0x54,0x12,0x9E,0x5F,0x06, // andps xmm,[r/m]
	0x00,0x54,0x12,0xBE,0x5F,0x06, // andps xmm,xmm
	0x00,0xBC,0x12,0x7E,0xBF,0x06, // bsf reg,reg
	0x00,0xBC,0x12,0x9E,0xBF,0x06, // bsf reg,[r/m]
	0x00,0xBD,0x12,0x7E,0x3F,0x07, // bsr reg,reg
	0x00,0xBD,0x12,0x9E,0x3F,0x07, // bsr reg,[r/m]
	0x00,0xC8,0x22,0xFE,0xBF,0x07, // bswap reg
	0x00,0xA3,0x32,0x7E,0x3F,0x08, // bt reg,reg
	0x20,0xBA,0xB2,0xFD,0x3F,0x08, // bt reg,imm
	0x00,0xA3,0x32,0x7E,0x4F,0x08, // bt [r/m],reg
	0x20,0xBA,0xB2,0xFD,0x4F,0x08, // bt [r/m],imm
	0x00,0xBB,0x32,0x7E,0xBF,0x08, // btc reg,reg
	0x38,0xBA,0xB2,0xFD,0xBF,0x08, // btc reg,imm
	0x00,0xBB,0x32,0x7E,0xCF,0x08, // btc [r/m],reg
	0x38,0xBA,0xB2,0xFD,0xCF,0x08, // btc [r/m],imm
	0x00,0xB3,0x32,0x7E,0x3F,0x09, // btr reg,reg
	0x30,0xBA,0xB2,0xFD,0x3F,0x09, // btr reg,imm
	0x00,0xB3,0x32,0x7E,0x4F,0x09, // btr [r/m],reg
	0x30,0xBA,0xB2,0xFD,0x4F,0x09, // btr [r/m],imm
	0x00,0xAB,0x32,0x7E,0xBF,0x09, // bts reg,reg
	0x28,0xBA,0xB2,0xFD,0xBF,0x09, // bts reg,imm
	0x00,0xAB,0x32,0x7E,0xCF,0x09, // bts [r/m],reg
	0x28,0xBA,0xB2,0xFD,0xCF,0x09, // bts [r/m],imm
	0x10,0xFF,0x30,0xFE,0x2F,0x0A, // call reg64
	0x10,0xFF,0x30,0xFE,0x4F,0x0A, // call [r/m]
	0x00,0xE8,0x40,0xFF,0x7E,0x0A, // call imm
	0x00,0x98,0x01,0xFE,0xFF,0x0A, // cbw
	0x00,0x99,0x00,0xFE,0x7F,0x0B, // cdq
	0x48,0x98,0x01,0xFE,0xFF,0x0B, // cdqe
	0x00,0xF8,0x00,0xFE,0x7F,0x0C, // clc
	0x00,0xFC,0x00,0xFE,0xFF,0x0C, // cld
	0x38,0xAE,0x32,0xFE,0x4F,0x0D, // clflush [r/m]
	0x00,0xFA,0x00,0xFE,0xFF,0x0D, // cli
	0x00,0xF5,0x00,0xFE,0x7F,0x0E, // cmc
	0x00,0x47,0x12,0x7E,0xBF,0x0E, // cmova reg,reg
	0x00,0x47,0x12,0x9E,0xBF,0x0E, // cmova reg,[r/m]
	0x00,0x43,0x12,0x7E,0x3F,0x0F, // cmovae reg,reg
	0x00,0x43,0x12,0x9E,0x3F,0x0F, // cmovae reg,[r/m]
	0x00,0x42,0x12,0x7E,0xBF,0x0F, // cmovb reg,reg
	0x00,0x42,0x12,0x9E,0xBF,0x0F, // cmovb reg,[r/m]
	0x00,0x46,0x12,0x7E,0x3F,0x10, // cmovbe reg,reg
	0x00,0x46,0x12,0x9E,0x3F,0x10, // cmovbe reg,[r/m]
	0x00,0x42,0x12,0x7E,0xBF,0x10, // cmovc reg,reg
	0x00,0x42,0x12,0x9E,0xBF,0x10, // cmovc reg,[r/m]
	0x00,0x44,0x12,0x7E,0x3F,0x11, // cmove reg,reg
	0x00,0x44,0x12,0x9E,0x3F,0x11, // cmove reg,[r/m]
	0x00,0x4F,0x12,0x7E,0xBF,0x11, // cmovg reg,reg
	0x00,0x4F,0x12,0x9E,0xBF,0x11, // cmovg reg,[r/m]
	0x00,0x4D,0x12,0x7E,0x3F,0x12, // cmovge reg,reg
	0x00,0x4D,0x12,0x9E,0x3F,0x12, // cmovge reg,[r/m]
	0x00,0x4C,0x12,0x7E,0xBF,0x12, // cmovl reg,reg
	0x00,0x4C,0x12,0x9E,0xBF,0x12, // cmovl reg,[r/m]
	0x00,0x4E,0x12,0x7E,0x3F,0x13, // cmovle reg,reg
	0x00,0x4E,0x12,0x9E,0x3F,0x13, // cmovle reg,[r/m]
	0x00,0x46,0x12,0x7E,0xBF,0x13, // cmovna reg,reg
	0x00,0x46,0x12,0x9E,0xBF,0x13, // cmovna reg,[r/m]
	0x00,0x42,0x12,0x7E,0x3F,0x14, // cmovnae reg,reg
	0x00,0x42,0x12,0x9E,0x3F,0x14, // cmovnae reg,[r/m]
	0x00,0x43,0x12,0x7E,0xBF,0x14, // cmovnb reg,reg
	0x00,0x43,0x12,0x9E,0xBF,0x14, // cmovnb reg,[r/m]
	0x00,0x47,0x12,0x7E,0x3F,0x15, // cmovnbe reg,reg
	0x00,0x47,0x12,0x9E,0x3F,0x15, // cmovnbe reg,[r/m]
	0x00,0x43,0x12,0x7E,0xBF,0x15, // cmovnc reg,reg
	0x00,0x43,0x12,0x9E,0xBF,0x15, // cmovnc reg,[r/m]
	0x00,0x45,0x12,0x7E,0x3F,0x16, // cmovne reg,reg
	0x00,0x45,0x12,0x9E,0x3F,0x16, // cmovne reg,[r/m]
	0x00,0x4E,0x12,0x7E,0xBF,0x16, // cmovng reg,reg
	0x00,0x4E,0x12,0x9E,0xBF,0x16, // cmovng reg,[r/m]
	0x00,0x4C,0x12,0x7E,0x3F,0x17, // cmovnge reg,reg
	0x00,0x4C,0x12,0x9E,0x3F,0x17, // cmovnge reg,[r/m]
	0x00,0x4D,0x12,0x7E,0xBF,0x17, // cmovnl reg,reg
	0x00,0x4D,0x12,0x9E,0xBF,0x17, // cmovnl reg,[r/m]
	0x00,0x4F,0x12,0x7E,0x3F,0x18, // cmovnle reg,reg
	0x00,0x4F,0x12,0x9E,0x3F,0x18, // cmovnle reg,[r/m]
	0x00,0x41,0x12,0x7E,0xBF,0x18, // cmovno reg,reg
	0x00,0x41,0x12,0x9E,0xBF,0x18, // cmovno reg,[r/m]
	0x00,0x4B,0x12,0x7E,0x3F,0x19, // cmovnp reg,reg
	0x00,0x4B,0x12,0x9E,0x3F,0x19, // cmovnp reg,[r/m]
	0x00,0x49,0x12,0x7E,0xBF,0x19, // cmovns reg,reg
	0x00,0x49,0x12,0x9E,0xBF,0x19, // cmovns reg,[r/m]
	0x00,0x45,0x12,0x7E,0x3F,0x1A, // cmovnz reg,reg
	0x00,0x45,0x12,0x9E,0x3F,0x1A, // cmovnz reg,[r/m]
	0x00,0x40,0x12,0x7E,0xBF,0x1A, // cmovo reg,reg
	0x00,0x40,0x12,0x9E,0xBF,0x1A, // cmovo reg,[r/m]
	0x00,0x4A,0x12,0x7E,0x3F,0x1B, // cmovp reg,reg
	0x00,0x4A,0x12,0x9E,0x3F,0x1B, // cmovp reg,[r/m]
	0x00,0x4A,0x12,0x7E,0xBF,0x1B, // cmovpe reg,reg
	0x00,0x4A,0x12,0x9E,0xBF,0x1B, // cmovpe reg,[r/m]
	0x00,0x4B,0x12,0x7E,0x3F,0x1C, // cmovpo reg,reg
	0x00,0x4B,0x12,0x9E,0x3F,0x1C, // cmovpo reg,[r/m]
	0x00,0x48,0x12,0x7E,0xBF,0x1C, // cmovs reg,reg
	0x00,0x48,0x12,0x9E,0xBF,0x1C, // cmovs reg,[r/m]
	0x00,0x44,0x12,0x7E,0x3F,0x1D, // cmovz reg,reg
	0x00,0x44,0x12,0x9E,0x3F,0x1D, // cmovz reg,[r/m]
	0x00,0x3D,0xC4,0xFD,0xB0,0x1D, // cmp rax,imm
	0x00,0x39,0x34,0x7E,0xBF,0x1D, // cmp reg,reg
	0x00,0x3B,0x14,0x9E,0xBF,0x1D, // cmp reg,[r/m]
	0x38,0x81,0xFC,0xFD,0xBF,0x1D, // cmp reg,imm
	0x00,0x39,0x34,0x7E,0xCF,0x1D, // cmp [r/m],reg
	0x38,0x81,0xFC,0xFD,0xCF,0x1D, // cmp [r/m],imm
	0x00,0xC2,0x93,0x9F,0x5F,0x1E, // cmppd xmm,[r/m],imm
	0x00,0xC2,0x93,0xBF,0x5F,0x1E, // cmppd xmm,xmm,imm
	0x00,0xC2,0x92,0x9F,0xDF,0x1E, // cmpps xmm,[r/m],imm
	0x00,0xC2,0x92,0xBF,0xDF,0x1E, // cmpps xmm,xmm,imm
	0x00,0xA6,0x04,0xFE,0x7F,0x1F, // cmpsb
	0xF2,0xC2,0x93,0x9F,0xDF,0x1F, // cmpsd xmm,[r/m],imm
	0xF2,0xC2,0x93,0xBF,0xDF,0x1F, // cmpsd xmm,xmm,imm
	0x00,0xA7,0x00,0xFE,0xFF,0x1F, // cmpsd
	0x48,0xA7,0x01,0xFE,0x7F,0x20, // cmpsq
	0xF3,0xC2,0x93,0x9F,0xDF,0x20, // cmpss xmm,[r/m],imm
	0xF3,0xC2,0x93,0xBF,0xDF,0x20, // cmpss xmm,xmm,imm
	0x00,0xA7,0x01,0xFE,0x7F,0x21, // cmpsw
	0x00,0xB1,0x36,0x7E,0xBF,0x21, // cmpxchg reg,reg
	0x00,0xB1,0x36,0x7E,0xCF,0x21, // cmpxchg [r/m],reg
	0x08,0xC7,0x32,0xFE,0x4F,0x22, // cmpxchg16b [r/m]
	0x08,0xC7,0x32,0xFE,0xCF,0x22, // cmpxchg8b [r/m]
	0x00,0x2F,0x13,0x9E,0x5F,0x23, // comisd xmm,[r/m]
	0x00,0x2F,0x13,0xBE,0x5F,0x23, // comisd xmm,xmm
	0x00,0x2F,0x12,0x9E,0xDF,0x23, // comiss xmm,[r/m]
	0x00,0x2F,0x12,0xBE,0xDF,0x23, // comiss xmm,xmm
	0x00,0xA2,0x02,0xFE,0x7F,0x24, // cpuid
	0x48,0x99,0x01,0xFE,0xFF,0x24, // cqo
	0xF3,0xE6,0x13,0x9E,0x5F,0x25, // cvtdq2pd xmm,[r/m]
	0xF3,0xE6,0x13,0xBE,0x5F,0x25, // cvtdq2pd xmm,xmm
	0x00,0x5B,0x12,0x9E,0xDF,0x25, // cvtdq2ps xmm,[r/m]
	0x00,0x5B,0x12,0xBE,0xDF,0x25, // cvtdq2ps xmm,xmm
	0xF2,0xE6,0x13,0x9E,0x5F,0x26, // cvtpd2dq xmm,[r/m]
	0xF2,0xE6,0x13,0xBE,0x5F,0x26, // cvtpd2dq xmm,xmm
	0x00,0x2D,0x13,0x9E,0xEF,0x26, // cvtpd2pi mmx,[r/m]
	0x00,0x2D,0x13,0xBE,0xEF,0x26, // cvtpd2pi mmx,xmm
	0x00,0x5A,0x13,0x9E,0x5F,0x27, // cvtpd2ps xmm,[r/m]
	0x00,0x5A,0x13,0xBE,0x5F,0x27, // cvtpd2ps xmm,xmm
	0x00,0x2A,0x13,0x9E,0xDF,0x27, // cvtpi2pd xmm,[r/m]
	0x00,0x2A,0x13,0xDE,0xDF,0x27, // cvtpi2pd xmm,mmx
	0x00,0x2A,0x12,0x9E,0x5F,0x28, // cvtpi2ps xmm,[r/m]
	0x00,0x2A,0x12,0xDE,0x5F,0x28, // cvtpi2ps xmm,mmx
	0x00,0x5B,0x13,0x9E,0xDF,0x28, // cvtps2dq xmm,[r/m]
	0x00,0x5B,0x13,0xBE,0xDF,0x28, // cvtps2dq xmm,xmm
	0x00,0x5A,0x12,0x9E,0x5F,0x29, // cvtps2pd xmm,[r/m]
	0x00,0x5A,0x12,0xBE,0x5F,0x29, // cvtps2pd xmm,xmm
	0x00,0x2D,0x12,0x9E,0xEF,0x29, // cvtps2pi mmx,[r/m]
	0x00,0x2D,0x12,0xBE,0xEF,0x29, // cvtps2pi mmx,xmm
	0xF2,0x2D,0x13,0x9E,0x3F,0x2A, // cvtsd2si reg,[r/m]
	0xF2,0x2D,0x13,0xBE,0x3F,0x2A, // cvtsd2si reg,xmm
	0xF2,0x5A,0x13,0x9E,0xDF,0x2A, // cvtsd2ss xmm,[r/m]
	0xF2,0x5A,0x13,0xBE,0xDF,0x2A, // cvtsd2ss xmm,xmm
	0xF2,0x2A,0x13,0x7E,0x5F,0x2B, // cvtsi2sd xmm,reg
	0xF2,0x2A,0x13,0x9E,0x5F,0x2B, // cvtsi2sd xmm,[r/m]
	0xF3,0x2A,0x13,0x7E,0xDF,0x2B, // cvtsi2ss xmm,reg
	0xF3,0x2A,0x13,0x9E,0xDF,0x2B, // cvtsi2ss xmm,[r/m]
	0xF3,0x5A,0x13,0x9E,0x5F,0x2C, // cvtss2sd xmm,[r/m]
	0xF3,0x5A,0x13,0xBE,0x5F,0x2C, // cvtss2sd xmm,xmm
	0xF3,0x2D,0x13,0x9E,0xBF,0x2C, // cvtss2si reg,[r/m]
	0xF3,0x2D,0x13,0xBE,0xBF,0x2C, // cvtss2si reg,xmm
	0x00,0xE6,0x13,0x9E,0x5F,0x2D, // cvttpd2dq xmm,[r/m]
	0x00,0xE6,0x13,0xBE,0x5F,0x2D, // cvttpd2dq xmm,xmm
	0x00,0x2C,0x13,0x9E,0xEF,0x2D, // cvttpd2pi mmx,[r/m]
	0x00,0x2C,0x13,0xBE,0xEF,0x2D, // cvttpd2pi mmx,xmm
	0xF3,0x5B,0x13,0x9E,0x5F,0x2E, // cvttps2dq xmm,[r/m]
	0xF3,0x5B,0x13,0xBE,0x5F,0x2E, // cvttps2dq xmm,xmm
	0x00,0x2C,0x12,0x9E,0xEF,0x2E, // cvttps2pi mmx,[r/m]
	0x00,0x2C,0x12,0xBE,0xEF,0x2E, // cvttps2pi mmx,xmm
	0xF2,0x2C,0x13,0x9E,0x3F,0x2F, // cvttsd2si reg,[r/m]
	0xF2,0x2C,0x13,0xBE,0x3F,0x2F, // cvttsd2si reg,xmm
	0xF3,0x2C,0x13,0x9E,0xBF,0x2F, // cvttss2si reg,[r/m]
	0xF3,0x2C,0x13,0xBE,0xBF,0x2F, // cvttss2si reg,xmm
	0x00,0x99,0x01,0xFE,0x7F,0x30, // cwd
	0x00,0x98,0x00,0xFE,0xFF,0x30, // cwde
	0xFF,0xFF,0x80,0xFF,0x7E,0x31, // db imm
	0xFF,0xFF,0x40,0xFE,0xFE,0x31, // dd imm
	0x08,0xFF,0x34,0xFE,0x3F,0x32, // dec reg
	0x08,0xFF,0x34,0xFE,0x4F,0x32, // dec [r/m]
	0x30,0xF7,0x34,0xFE,0xBF,0x32, // div reg
	0x30,0xF7,0x34,0xFE,0xCF,0x32, // div [r/m]
	0x00,0x5E,0x13,0x9E,0x5F,0x33, // divpd xmm,[r/m]
	0x00,0x5E,0x13,0xBE,0x5F,0x33, // divpd xmm,xmm
	0x00,0x5E,0x12,0x9E,0xDF,0x33, // divps xmm,[r/m]
	0x00,0x5E,0x12,0xBE,0xDF,0x33, // divps xmm,xmm
	0xF2,0x5E,0x13,0x9E,0x5F,0x34, // divsd xmm,[r/m]
	0xF2,0x5E,0x13,0xBE,0x5F,0x34, // divsd xmm,xmm
	0xF3,0x5E,0x13,0x9E,0xDF,0x34, // divss xmm,[r/m]
	0xF3,0x5E,0x13,0xBE,0xDF,0x34, // divss xmm,xmm
	0xFF,0xFF,0xC0,0xFE,0x7E,0x35, // dq imm
	0xFF,0xFF,0x80,0xFE,0xFE,0x35, // dw imm
	0x00,0x77,0x02,0xFE,0x7F,0x36, // emms
	0x00,0xC8,0x80,0xFD,0xFE,0x36, // enter imm,imm
	0xFF,0xFF,0x80,0xFF,0x7E,0x37, // equb imm
	0xFF,0xFF,0x40,0xFE,0xFE,0x37, // equd imm
	0xFF,0xFF,0xC0,0xFE,0x7E,0x38, // equq imm
	0xFF,0xFF,0x80,0xFE,0xFE,0x38, // equw imm
	0xF0,0xD9,0x00,0xFE,0x7F,0x39, // f2xm1
	0xE1,0xD9,0x00,0xFE,0xFF,0x39, // fabs
	0x00,0xD8,0x3C,0xFE,0x4F,0x3A, // fadd [r/m]
	0x00,0xD8,0x1C,0xCE,0x60,0x3A, // fadd st0,stn
	0x00,0xDC,0x30,0xC0,0x67,0x3A, // fadd stn,st0
	0x00,0xDE,0x30,0xC0,0xE7,0x3A, // faddp stn,st0
	0xC1,0xDE,0x00,0xFE,0xFF,0x3A, // faddp
	0x20,0xDF,0x30,0xFE,0x4F,0x3B, // fbld [r/m]
	0x30,0xDF,0x30,0xFE,0xCF,0x3B, // fbstp [r/m]
	0xE0,0xD9,0x00,0xFE,0x7F,0x3C, // fchs
	0xE2,0xDB,0x02,0xFE,0xFF,0x3C, // fclex
	0x00,0xDA,0x10,0xCE,0x60,0x3D, // fcmovb st0,stn
	0x10,0xDA,0x10,0xCE,0xE0,0x3D, // fcmovbe st0,stn
	0x08,0xDA,0x10,0xCE,0x60,0x3E, // fcmove st0,stn
	0x00,0xDB,0x10,0xCE,0xE0,0x3E, // fcmovnb st0,stn
	0x10,0xDB,0x10,0xCE,0x60,0x3F, // fcmovnbe st0,stn
	0x08,0xDB,0x10,0xCE,0xE0,0x3F, // fcmovne st0,stn
	0x18,0xDB,0x10,0xCE,0x60,0x40, // fcmovnu st0,stn
	0x18,0xDA,0x10,0xCE,0xE0,0x40, // fcmovu st0,stn
	0x10,0xD8,0x3C,0xFE,0x4F,0x41, // fcom [r/m]
	0x10,0xD8,0x3C,0xFE,0x67,0x41, // fcom stn
	0xD1,0xD8,0x00,0xFE,0x7F,0x41, // fcom
	0x30,0xDB,0x10,0xCE,0xE0,0x41, // fcomi st0,stn
	0x30,0xDF,0x10,0xCE,0x60,0x42, // fcomip st0,stn
	0x18,0xD8,0x3C,0xFE,0xCF,0x42, // fcomp [r/m]
	0x18,0xD8,0x3C,0xFE,0xE7,0x42, // fcomp stn
	0xD9,0xD8,0x00,0xFE,0xFF,0x42, // fcomp
	0xD9,0xDE,0x00,0xFE,0x7F,0x43, // fcompp
	0xFF,0xD9,0x00,0xFE,0xFF,0x43, // fcos
	0xF6,0xD9,0x00,0xFE,0x7F,0x44, // fdecstp
	0x30,0xD8,0x3C,0xFE,0xCF,0x44, // fdiv [r/m]
	0x30,0xD8,0x1C,0xCE,0xE0,0x44, // fdiv st0,stn
	0x38,0xDC,0x30,0xC0,0xE7,0x44, // fdiv stn,st0
	0x38,0xDE,0x30,0xC0,0x67,0x45, // fdivp stn,st0
	0xF9,0xDE,0x00,0xFE,0x7F,0x45, // fdivp
	0x38,0xD8,0x3C,0xFE,0xCF,0x45, // fdivr [r/m]
	0x38,0xD8,0x1C,0xCE,0xE0,0x45, // fdivr st0,stn
	0x30,0xDC,0x30,0xC0,0xE7,0x45, // fdivr stn,st0
	0x30,0xDE,0x30,0xC0,0x67,0x46, // fdivrp stn,st0
	0xF1,0xDE,0x00,0xFE,0x7F,0x46, // fdivrp
	0x00,0xDD,0x30,0xFE,0xE7,0x46, // ffree stn
	0x00,0xDA,0x3C,0xFE,0x4F,0x47, // fiadd [r/m]
	0x10,0xDA,0x3C,0xFE,0xCF,0x47, // ficom [r/m]
	0x18,0xDA,0x3C,0xFE,0x4F,0x48, // ficomp [r/m]
	0x30,0xDA,0x3C,0xFE,0xCF,0x48, // fidiv [r/m]
	0x38,0xDA,0x3C,0xFE,0x4F,0x49, // fidivr [r/m]
	0x00,0xDB,0x3C,0xFE,0xCF,0x49, // fild [r/m]
	0x08,0xDA,0x3C,0xFE,0x4F,0x4A, // fimul [r/m]
	0xF7,0xD9,0x00,0xFE,0xFF,0x4A, // fincstp
	0xE3,0xDB,0x02,0xFE,0x7F,0x4B, // finit
	0x10,0xDB,0x3C,0xFE,0xCF,0x4B, // fist [r/m]
	0x18,0xDB,0x3C,0xFE,0x4F,0x4C, // fistp [r/m]
	0x08,0xDB,0x3C,0xFE,0xCF,0x4C, // fisttp [r/m]
	0x20,0xDA,0x3C,0xFE,0x4F,0x4D, // fisub [r/m]
	0x28,0xDA,0x3C,0xFE,0xCF,0x4D, // fisubr [r/m]
	0x00,0xD9,0x3C,0xFE,0x4F,0x4E, // fld [r/m]
	0x00,0xD9,0x3C,0xFE,0x67,0x4E, // fld stn
	0xE8,0xD9,0x00,0xFE,0xFF,0x4E, // fld1
	0x28,0xD9,0x30,0xFE,0x4F,0x4F, // fldcw [r/m]
	0x20,0xD9,0x30,0xFE,0xCF,0x4F, // fldenv [r/m]
	0xEA,0xD9,0x00,0xFE,0x7F,0x50, // fldl2e
	0xE9,0xD9,0x00,0xFE,0xFF,0x50, // fldl2t
	0xEC,0xD9,0x00,0xFE,0x7F,0x51, // fldlg2
	0xED,0xD9,0x00,0xFE,0xFF,0x51, // fldln2
	0xEB,0xD9,0x00,0xFE,0x7F,0x52, // fldpi
	0xEE,0xD9,0x00,0xFE,0xFF,0x52, // fldz
	0x08,0xD8,0x3C,0xFE,0x4F,0x53, // fmul [r/m]
	0x08,0xD8,0x1C,0xCE,0x60,0x53, // fmul st0,stn
	0x08,0xDC,0x30,0xC0,0x67,0x53, // fmul stn,st0
	0x08,0xDE,0x30,0xC0,0xE7,0x53, // fmulp stn,st0
	0xC9,0xDE,0x00,0xFE,0xFF,0x53, // fmulp
	0xE2,0xDB,0x00,0xFE,0x7F,0x54, // fnclex
	0xE3,0xDB,0x00,0xFE,0xFF,0x54, // fninit
	0xD0,0xD9,0x00,0xFE,0x7F,0x55, // fnop
	0x30,0xDD,0x30,0xFE,0xCF,0x55, // fnsave [r/m]
	0x38,0xD9,0x30,0xFE,0x4F,0x56, // fnstcw [r/m]
	0x30,0xD9,0x30,0xFE,0xCF,0x56, // fnstenv [r/m]
	0xE0,0xDF,0x00,0xFE,0x10,0x57, // fnstsw ax
	0x38,0xDD,0x30,0xFE,0x4F,0x57, // fnstsw [r/m]
	0xF3,0xD9,0x00,0xFE,0xFF,0x57, // fpatan
	0xF8,0xD9,0x00,0xFE,0x7F,0x58, // fprem
	0xF5,0xD9,0x00,0xFE,0xFF,0x58, // fprem1
	0xF2,0xD9,0x00,0xFE,0x7F,0x59, // fptan
	0xFC,0xD9,0x00,0xFE,0xFF,0x59, // frndint
	0x20,0xDD,0x30,0xFE,0x4F,0x5A, // frstor [r/m]
	0x30,0xDD,0x32,0xFE,0xCF,0x5A, // fsave [r/m]
	0xFD,0xD9,0x00,0xFE,0x7F,0x5B, // fscale
	0xFE,0xD9,0x00,0xFE,0xFF,0x5B, // fsin
	0xFB,0xD9,0x00,0xFE,0x7F,0x5C, // fsincos
	0xFA,0xD9,0x00,0xFE,0xFF,0x5C, // fsqrt
	0x10,0xD9,0x3C,0xFE,0x4F,0x5D, // fst [r/m]
	0x10,0xDD,0x30,0xFE,0x67,0x5D, // fst stn
	0x38,0xD9,0x32,0xFE,0xCF,0x5D, // fstcw [r/m]
	0x30,0xD9,0x32,0xFE,0x4F,0x5E, // fstenv [r/m]
	0x18,0xD9,0x3C,0xFE,0xCF,0x5E, // fstp [r/m]
	0x18,0xDD,0x30,0xFE,0xE7,0x5E, // fstp stn
	0xE0,0xDF,0x02,0xFE,0x10,0x5F, // fstsw ax
	0x38,0xDD,0x32,0xFE,0x4F,0x5F, // fstsw [r/m]
	0x20,0xD8,0x3C,0xFE,0xCF,0x5F, // fsub [r/m]
	0x20,0xD8,0x1C,0xCE,0xE0,0x5F, // fsub st0,stn
	0x28,0xDC,0x30,0xC0,0xE7,0x5F, // fsub stn,st0
	0x28,0xDE,0x30,0xC0,0x67,0x60, // fsubp stn,st0
	0xE9,0xDE,0x00,0xFE,0x7F,0x60, // fsubp
	0x28,0xD8,0x3C,0xFE,0xCF,0x60, // fsubr [r/m]
	0x28,0xD8,0x1C,0xCE,0xE0,0x60, // fsubr st0,stn
	0x20,0xDC,0x30,0xC0,0xE7,0x60, // fsubr stn,st0
	0x20,0xDE,0x30,0xC0,0x67,0x61, // fsubrp stn,st0
	0xE1,0xDE,0x00,0xFE,0x7F,0x61, // fsubrp
	0xE4,0xD9,0x00,0xFE,0xFF,0x61, // ftst
	0x20,0xDD,0x30,0xFE,0x67,0x62, // fucom stn
	0xE1,0xDD,0x00,0xFE,0x7F,0x62, // fucom
	0x28,0xDB,0x10,0xCE,0xE0,0x62, // fucomi st0,stn
	0x28,0xDF,0x10,0xCE,0x60,0x63, // fucomip st0,stn
	0x28,0xDD,0x30,0xFE,0xE7,0x63, // fucomp stn
	0xE9,0xDD,0x00,0xFE,0xFF,0x63, // fucomp
	0xE9,0xDA,0x00,0xFE,0x7F,0x64, // fucompp
	0x00,0x9B,0x00,0xFE,0xFF,0x64, // fwait
	0xE5,0xD9,0x00,0xFE,0x7F,0x65, // fxam
	0x08,0xD9,0x3C,0xFE,0xE7,0x65, // fxch stn
	0xC9,0xD9,0x00,0xFE,0xFF,0x65, // fxch
	0x08,0xAE,0x32,0xFE,0x4F,0x66, // fxrstor [r/m]
	0x00,0xAE,0x32,0xFE,0xCF,0x66, // fxsave [r/m]
	0xF4,0xD9,0x00,0xFE,0x7F,0x67, // fxtract
	0xF1,0xD9,0x00,0xFE,0xFF,0x67, // fyl2x
	0xF9,0xD9,0x00,0xFE,0x7F,0x68, // fyl2xp1
	0x00,0x7C,0x13,0x9E,0xDF,0x68, // haddpd xmm,[r/m]
	0x00,0x7C,0x13,0xBE,0xDF,0x68, // haddpd xmm,xmm
	0xF2,0x7C,0x13,0x9E,0x5F,0x69, // haddps xmm,[r/m]
	0xF2,0x7C,0x13,0xBE,0x5F,0x69, // haddps xmm,xmm
	0x00,0x7D,0x13,0x9E,0xDF,0x69, // hsubpd xmm,[r/m]
	0x00,0x7D,0x13,0xBE,0xDF,0x69, // hsubpd xmm,xmm
	0xF2,0x7D,0x13,0x9E,0x5F,0x6A, // hsubps xmm,[r/m]
	0xF2,0x7D,0x13,0xBE,0x5F,0x6A, // hsubps xmm,xmm
	0x38,0xF7,0x34,0xFE,0xBF,0x6A, // idiv reg
	0x38,0xF7,0x34,0xFE,0xCF,0x6A, // idiv [r/m]
	0x00,0xAF,0x16,0x7E,0x3F,0x6B, // imul reg,reg
	0x00,0x69,0xD8,0x7F,0x3F,0x6B, // imul reg,reg,imm
	0x00,0xAF,0x16,0x9E,0x3F,0x6B, // imul reg,[r/m]
	0x00,0x69,0xD8,0x9F,0x3F,0x6B, // imul reg,[r/m],imm
	0x28,0xF7,0x34,0xFE,0x3F,0x6B, // imul reg
	0x28,0xF7,0x34,0xFE,0x4F,0x6B, // imul [r/m]
	0x00,0xFF,0x34,0xFE,0xBF,0x6B, // inc reg
	0x00,0xFF,0x34,0xFE,0xCF,0x6B, // inc [r/m]
	0x00,0x77,0x00,0xFF,0x7E,0x6C, // ja imm
	0x00,0x73,0x00,0xFF,0xFE,0x6C, // jae imm
	0x00,0x72,0x00,0xFF,0x7E,0x6D, // jb imm
	0x00,0x76,0x00,0xFF,0xFE,0x6D, // jbe imm
	0x00,0x72,0x00,0xFF,0x7E,0x6E, // jc imm
	0x00,0x74,0x00,0xFF,0xFE,0x6E, // je imm
	0x67,0xE3,0x01,0xFF,0x7E,0x6F, // jecxz imm
	0x00,0x7F,0x00,0xFF,0xFE,0x6F, // jg imm
	0x00,0x7D,0x00,0xFF,0x7E,0x70, // jge imm
	0x00,0x7C,0x00,0xFF,0xFE,0x70, // jl imm
	0x00,0x7E,0x00,0xFF,0x7E,0x71, // jle imm
	0x20,0xFF,0x30,0xFE,0xAF,0x71, // jmp reg64
	0x20,0xFF,0x30,0xFE,0xCF,0x71, // jmp [r/m]
	0x00,0xEB,0x00,0xFF,0xFE,0x71, // jmp imm
	0x00,0x76,0x00,0xFF,0x7E,0x72, // jna imm
	0x00,0x72,0x00,0xFF,0xFE,0x72, // jnae imm
	0x00,0x73,0x00,0xFF,0x7E,0x73, // jnb imm
	0x00,0x77,0x00,0xFF,0xFE,0x73, // jnbe imm
	0x00,0x73,0x00,0xFF,0x7E,0x74, // jnc imm
	0x00,0x75,0x00,0xFF,0xFE,0x74, // jne imm
	0x00,0x7E,0x00,0xFF,0x7E,0x75, // jng imm
	0x00,0x7C,0x00,0xFF,0xFE,0x75, // jnge imm
	0x00,0x7D,0x00,0xFF,0x7E,0x76, // jnl imm
	0x00,0x7F,0x00,0xFF,0xFE,0x76, // jnle imm
	0x00,0x71,0x00,0xFF,0x7E,0x77, // jno imm
	0x00,0x7B,0x00,0xFF,0xFE,0x77, // jnp imm
	0x00,0x79,0x00,0xFF,0x7E,0x78, // jns imm
	0x00,0x75,0x00,0xFF,0xFE,0x78, // jnz imm
	0x00,0x70,0x00,0xFF,0x7E,0x79, // jo imm
	0x00,0x7A,0x00,0xFF,0xFE,0x79, // jp imm
	0x00,0x7A,0x00,0xFF,0x7E,0x7A, // jpe imm
	0x00,0x7B,0x00,0xFF,0xFE,0x7A, // jpo imm
	0x00,0xE3,0x00,0xFF,0x7E,0x7B, // jrcxz imm
	0x00,0x78,0x00,0xFF,0xFE,0x7B, // js imm
	0x00,0x74,0x00,0xFF,0x7E,0x7C, // jz imm
	0x00,0x9F,0x00,0xFE,0xFF,0x7C, // lahf
	0xF2,0xF0,0x13,0x9E,0x5F,0x7D, // lddqu xmm,[r/m]
	0x10,0xAE,0x32,0xFE,0xCF,0x7D, // ldmxcsr [r/m]
	0x00,0x8D,0x10,0x9E,0x3F,0x7E, // lea reg,[r/m]
	0x00,0xC9,0x00,0xFE,0xFF,0x7E, // leave
	0xE8,0xAE,0x02,0xFE,0x7F,0x7F, // lfence
	0x00,0xF0,0x00,0xFE,0xFF,0x7F, // lock
	0x00,0xAC,0x04,0xFE,0x7F,0x80, // lodsb
	0x00,0xAD,0x00,0xFE,0xFF,0x80, // lodsd
	0x48,0xAD,0x01,0xFE,0x7F,0x81, // lodsq
	0x00,0xAD,0x01,0xFE,0xFF,0x81, // lodsw
	0x00,0xE2,0x00,0xFF,0x7E,0x82, // loop imm
	0x00,0xE1,0x00,0xFF,0xFE,0x82, // loope imm
	0x00,0xE0,0x00,0xFF,0x7E,0x83, // loopne imm
	0x00,0xE0,0x00,0xFF,0xFE,0x83, // loopnz imm
	0x00,0xE1,0x00,0xFF,0x7E,0x84, // loopz imm
	0x00,0x03,0x12,0x7E,0xBF,0x84, // lsl reg,reg
	0x00,0x03,0x12,0x9E,0xBF,0x84, // lsl reg,[r/m]
	0x00,0xF7,0x13,0xBE,0x5F,0x85, // maskmovdqu xmm,xmm
	0x00,0xF7,0x12,0xDE,0xEF,0x85, // maskmovq mmx,mmx
	0x00,0x5F,0x13,0x9E,0x5F,0x86, // maxpd xmm,[r/m]
	0x00,0x5F,0x13,0xBE,0x5F,0x86, // maxpd xmm,xmm
	0x00,0x5F,0x12,0x9E,0xDF,0x86, // maxps xmm,[r/m]
	0x00,0x5F,0x12,0xBE,0xDF,0x86, // maxps xmm,xmm
	0xF2,0x5F,0x13,0x9E,0x5F,0x87, // maxsd xmm,[r/m]
	0xF2,0x5F,0x13,0xBE,0x5F,0x87, // maxsd xmm,xmm
	0xF3,0x5F,0x13,0x9E,0xDF,0x87, // maxss xmm,[r/m]
	0xF3,0x5F,0x13,0xBE,0xDF,0x87, // maxss xmm,xmm
	0xF0,0xAE,0x02,0xFE,0x7F,0x88, // mfence
	0x00,0x5D,0x13,0x9E,0xDF,0x88, // minpd xmm,[r/m]
	0x00,0x5D,0x13,0xBE,0xDF,0x88, // minpd xmm,xmm
	0x00,0x5D,0x12,0x9E,0x5F,0x89, // minps xmm,[r/m]
	0x00,0x5D,0x12,0xBE,0x5F,0x89, // minps xmm,xmm
	0xF2,0x5D,0x13,0x9E,0xDF,0x89, // minsd xmm,[r/m]
	0xF2,0x5D,0x13,0xBE,0xDF,0x89, // minsd xmm,xmm
	0xF3,0x5D,0x13,0x9E,0x5F,0x8A, // minss xmm,[r/m]
	0xF3,0x5D,0x13,0xBE,0x5F,0x8A, // minss xmm,xmm
	0x00,0xB0,0xA0,0xFD,0x8F,0x8A, // mov reg8,imm
	0x00,0xA1,0xC4,0xF8,0xB0,0x8A, // mov rax,[mem]
	0x00,0x89,0x34,0x7E,0xBF,0x8A, // mov reg,reg
	0x00,0x8B,0x14,0x9E,0xBF,0x8A, // mov reg,[r/m]
	0x00,0x8C,0x30,0xEE,0xBF,0x8A, // mov reg,sreg
	0x00,0xB8,0xE0,0xFD,0xBF,0x8A, // mov reg,imm
	0x00,0x89,0x34,0x7E,0xCF,0x8A, // mov [r/m],reg
	0x00,0x8C,0x30,0xEE,0xCF,0x8A, // mov [r/m],sreg
	0x00,0xC7,0xF4,0xFD,0xCF,0x8A, // mov [r/m],imm
	0x00,0x8E,0x10,0x3E,0xF7,0x8A, // mov sreg,reg16
	0x00,0x8E,0x10,0x9E,0xF7,0x8A, // mov sreg,[r/m]
	0x00,0xA3,0xC4,0x60,0xFC,0x8A, // mov [mem],rax
	0x00,0x29,0x33,0xBE,0x4F,0x8B, // movapd [r/m],xmm
	0x00,0x28,0x13,0x9E,0x5F,0x8B, // movapd xmm,[r/m]
	0x00,0x28,0x13,0xBE,0x5F,0x8B, // movapd xmm,xmm
	0x00,0x29,0x32,0xBE,0xCF,0x8B, // movaps [r/m],xmm
	0x00,0x28,0x12,0x9E,0xDF,0x8B, // movaps xmm,[r/m]
	0x00,0x28,0x12,0xBE,0xDF,0x8B, // movaps xmm,xmm
	0xF0,0x38,0x12,0x9E,0x3F,0x8C, // movbe reg,[r/m]
	0xF1,0x38,0x32,0x7E,0x4F,0x8C, // movbe [r/m],reg
	0x00,0x7E,0x33,0xBE,0xBF,0x8C, // movd reg,xmm
	0x00,0x7E,0x32,0xDE,0xBF,0x8C, // movd reg,mmx
	0x00,0x7E,0x33,0xBE,0xCF,0x8C, // movd [r/m],xmm
	0x00,0x7E,0x32,0xDE,0xCF,0x8C, // movd [r/m],mmx
	0x00,0x6E,0x13,0x7E,0xDF,0x8C, // movd xmm,reg
	0x00,0x6E,0x13,0x9E,0xDF,0x8C, // movd xmm,[r/m]
	0x00,0x6E,0x12,0x7E,0xEF,0x8C, // movd mmx,reg
	0x00,0x6E,0x12,0x9E,0xEF,0x8C, // movd mmx,[r/m]
	0xF2,0x12,0x13,0x9E,0x5F,0x8D, // movddup xmm,[r/m]
	0xF2,0x12,0x13,0xBE,0x5F,0x8D, // movddup xmm,xmm
	0xF2,0xD6,0x13,0xBE,0xEF,0x8D, // movdq2q mmx,xmm
	0x00,0x7F,0x33,0xBE,0x4F,0x8E, // movdqa [r/m],xmm
	0x00,0x6F,0x13,0x9E,0x5F,0x8E, // movdqa xmm,[r/m]
	0x00,0x6F,0x13,0xBE,0x5F,0x8E, // movdqa xmm,xmm
	0xF3,0x7F,0x33,0xBE,0xCF,0x8E, // movdqu [r/m],xmm
	0xF3,0x6F,0x13,0x9E,0xDF,0x8E, // movdqu xmm,[r/m]
	0xF3,0x6F,0x13,0xBE,0xDF,0x8E, // movdqu xmm,xmm
	0x00,0x12,0x12,0xBE,0x5F,0x8F, // movhlps xmm,xmm
	0x00,0x17,0x33,0xBE,0xCF,0x8F, // movhpd [r/m],xmm
	0x00,0x16,0x13,0x9E,0xDF,0x8F, // movhpd xmm,[r/m]
	0x00,0x17,0x32,0xBE,0x4F,0x90, // movhps [r/m],xmm
	0x00,0x16,0x12,0x9E,0x5F,0x90, // movhps xmm,[r/m]
	0x00,0x16,0x12,0xBE,0xDF,0x90, // movlhps xmm,xmm
	0x00,0x13,0x33,0xBE,0x4F,0x91, // movlpd [r/m],xmm
	0x00,0x12,0x13,0x9E,0x5F,0x91, // movlpd xmm,[r/m]
	0x00,0x13,0x32,0xBE,0xCF,0x91, // movlps [r/m],xmm
	0x00,0x12,0x12,0x9E,0xDF,0x91, // movlps xmm,[r/m]
	0x00,0x50,0x13,0xBE,0x3F,0x92, // movmskpd reg,xmm
	0x00,0x50,0x12,0xBE,0xBF,0x92, // movmskps reg,xmm
	0x00,0xE7,0x33,0xBE,0x4F,0x93, // movntdq [r/m],xmm
	0x00,0xC3,0x32,0x7E,0xCF,0x93, // movnti [r/m],reg
	0x00,0x2B,0x33,0xBE,0x4F,0x94, // movntpd [r/m],xmm
	0x00,0x2B,0x32,0xBE,0xCF,0x94, // movntps [r/m],xmm
	0x00,0xE7,0x32,0xDE,0x4F,0x95, // movntq [r/m],mmx
	0x00,0xD6,0x33,0xBE,0xCF,0x95, // movq [r/m],xmm
	0x00,0x7F,0x32,0xDE,0xCF,0x95, // movq [r/m],mmx
	0xF3,0x7E,0x13,0x9E,0xDF,0x95, // movq xmm,[r/m]
	0x00,0xD6,0x33,0xBE,0xDF,0x95, // movq xmm,xmm
	0x00,0x6F,0x12,0x9E,0xEF,0x95, // movq mmx,[r/m]
	0x00,0x6F,0x12,0xDE,0xEF,0x95, // movq mmx,mmx
	0xF3,0xD6,0x13,0xDE,0x5F,0x96, // movq2dq xmm,mmx
	0x00,0xA4,0x04,0xFE,0xFF,0x96, // movsb
	0xF2,0x11,0x33,0xBE,0x4F,0x97, // movsd [r/m],xmm
	0xF2,0x10,0x13,0x9E,0x5F,0x97, // movsd xmm,[r/m]
	0xF2,0x10,0x13,0xBE,0x5F,0x97, // movsd xmm,xmm
	0x00,0xA5,0x00,0xFE,0x7F,0x97, // movsd
	0xF3,0x16,0x13,0x9E,0xDF,0x97, // movshdup xmm,[r/m]
	0xF3,0x16,0x13,0xBE,0xDF,0x97, // movshdup xmm,xmm
	0xF3,0x12,0x13,0x9E,0x5F,0x98, // movsldup xmm,[r/m]
	0xF3,0x12,0x13,0xBE,0x5F,0x98, // movsldup xmm,xmm
	0x48,0xA5,0x01,0xFE,0xFF,0x98, // movsq
	0xF3,0x11,0x33,0xBE,0x4F,0x99, // movss [r/m],xmm
	0xF3,0x10,0x13,0x9E,0x5F,0x99, // movss xmm,[r/m]
	0xF3,0x10,0x13,0xBE,0x5F,0x99, // movss xmm,xmm
	0x00,0xA5,0x01,0xFE,0xFF,0x99, // movsw
	0x00,0xBF,0x16,0x3E,0x3F,0x9A, // movsx reg,reg16
	0x00,0xBF,0x16,0x9E,0x3F,0x9A, // movsx reg,[r/m]
	0x00,0x63,0x10,0x7E,0xBF,0x9A, // movsxd reg,reg
	0x00,0x63,0x10,0x9E,0xBF,0x9A, // movsxd reg,[r/m]
	0x00,0x11,0x33,0xBE,0x4F,0x9B, // movupd [r/m],xmm
	0x00,0x10,0x13,0x9E,0x5F,0x9B, // movupd xmm,[r/m]
	0x00,0x10,0x13,0xBE,0x5F,0x9B, // movupd xmm,xmm
	0x00,0x11,0x32,0xBE,0xCF,0x9B, // movups [r/m],xmm
	0x00,0x10,0x12,0x9E,0xDF,0x9B, // movups xmm,[r/m]
	0x00,0x10,0x12,0xBE,0xDF,0x9B, // movups xmm,xmm
	0x00,0xB7,0x16,0x3E,0x3F,0x9C, // movzx reg,reg16
	0x00,0xB7,0x16,0x9E,0x3F,0x9C, // movzx reg,[r/m]
	0x20,0xF7,0x34,0xFE,0xBF,0x9C, // mul reg
	0x20,0xF7,0x34,0xFE,0xCF,0x9C, // mul [r/m]
	0x00,0x59,0x13,0x9E,0x5F,0x9D, // mulpd xmm,[r/m]
	0x00,0x59,0x13,0xBE,0x5F,0x9D, // mulpd xmm,xmm
	0x00,0x59,0x12,0x9E,0xDF,0x9D, // mulps xmm,[r/m]
	0x00,0x59,0x12,0xBE,0xDF,0x9D, // mulps xmm,xmm
	0xF2,0x59,0x13,0x9E,0x5F,0x9E, // mulsd xmm,[r/m]
	0xF2,0x59,0x13,0xBE,0x5F,0x9E, // mulsd xmm,xmm
	0xF3,0x59,0x13,0x9E,0xDF,0x9E, // mulss xmm,[r/m]
	0xF3,0x59,0x13,0xBE,0xDF,0x9E, // mulss xmm,xmm
	0x18,0xF7,0x34,0xFE,0x3F,0x9F, // neg reg
	0x18,0xF7,0x34,0xFE,0x4F,0x9F, // neg [r/m]
	0x00,0x1F,0x32,0xFE,0xAF,0x9F, // nop reg64
	0x00,0x1F,0x32,0xFE,0xCF,0x9F, // nop [r/m]
	0x00,0x90,0x00,0xFE,0xFF,0x9F, // nop
	0x10,0xF7,0x34,0xFE,0x3F,0xA0, // not reg
	0x10,0xF7,0x34,0xFE,0x4F,0xA0, // not [r/m]
	0x00,0x0D,0xC4,0xFD,0x30,0xA1, // or rax,imm
	0x00,0x09,0x34,0x7E,0x3F,0xA1, // or reg,reg
	0x00,0x0B,0x14,0x9E,0x3F,0xA1, // or reg,[r/m]
	0x08,0x81,0xFC,0xFD,0x3F,0xA1, // or reg,imm
	0x00,0x09,0x34,0x7E,0x4F,0xA1, // or [r/m],reg
	0x08,0x81,0xFC,0xFD,0x4F,0xA1, // or [r/m],imm
	0x00,0x56,0x13,0x9E,0xDF,0xA1, // orpd xmm,[r/m]
	0x00,0x56,0x13,0xBE,0xDF,0xA1, // orpd xmm,xmm
	0x00,0x56,0x12,0x9E,0x5F,0xA2, // orps xmm,[r/m]
	0x00,0x56,0x12,0xBE,0x5F,0xA2, // orps xmm,xmm
	0x00,0x6B,0x13,0x9E,0xDF,0xA2, // packssdw xmm,[r/m]
	0x00,0x6B,0x13,0xBE,0xDF,0xA2, // packssdw xmm,xmm
	0x00,0x6B,0x12,0x9E,0xEF,0xA2, // packssdw mmx,[r/m]
	0x00,0x6B,0x12,0xDE,0xEF,0xA2, // packssdw mmx,mmx
	0x00,0x63,0x13,0x9E,0x5F,0xA3, // packsswb xmm,[r/m]
	0x00,0x63,0x13,0xBE,0x5F,0xA3, // packsswb xmm,xmm
	0x00,0x63,0x12,0x9E,0x6F,0xA3, // packsswb mmx,[r/m]
	0x00,0x63,0x12,0xDE,0x6F,0xA3, // packsswb mmx,mmx
	0x00,0x67,0x13,0x9E,0xDF,0xA3, // packuswb xmm,[r/m]
	0x00,0x67,0x13,0xBE,0xDF,0xA3, // packuswb xmm,xmm
	0x00,0x67,0x12,0x9E,0xEF,0xA3, // packuswb mmx,[r/m]
	0x00,0x67,0x12,0xDE,0xEF,0xA3, // packuswb mmx,mmx
	0x00,0xFC,0x13,0x9E,0x5F,0xA4, // paddb xmm,[r/m]
	0x00,0xFC,0x13,0xBE,0x5F,0xA4, // paddb xmm,xmm
	0x00,0xFC,0x12,0x9E,0x6F,0xA4, // paddb mmx,[r/m]
	0x00,0xFC,0x12,0xDE,0x6F,0xA4, // paddb mmx,mmx
	0x00,0xFE,0x13,0x9E,0xDF,0xA4, // paddd xmm,[r/m]
	0x00,0xFE,0x13,0xBE,0xDF,0xA4, // paddd xmm,xmm
	0x00,0xFE,0x12,0x9E,0xEF,0xA4, // paddd mmx,[r/m]
	0x00,0xFE,0x12,0xDE,0xEF,0xA4, // paddd mmx,mmx
	0x00,0xD4,0x13,0x9E,0x5F,0xA5, // paddq xmm,[r/m]
	0x00,0xD4,0x13,0xBE,0x5F,0xA5, // paddq xmm,xmm
	0x00,0xD4,0x12,0x9E,0x6F,0xA5, // paddq mmx,[r/m]
	0x00,0xD4,0x12,0xDE,0x6F,0xA5, // paddq mmx,mmx
	0x00,0xEC,0x13,0x9E,0xDF,0xA5, // paddsb xmm,[r/m]
	0x00,0xEC,0x13,0xBE,0xDF,0xA5, // paddsb xmm,xmm
	0x00,0xEC,0x12,0x9E,0xEF,0xA5, // paddsb mmx,[r/m]
	0x00,0xEC,0x12,0xDE,0xEF,0xA5, // paddsb mmx,mmx
	0x00,0xED,0x13,0x9E,0x5F,0xA6, // paddsw xmm,[r/m]
	0x00,0xED,0x13,0xBE,0x5F,0xA6, // paddsw xmm,xmm
	0x00,0xED,0x12,0x9E,0x6F,0xA6, // paddsw mmx,[r/m]
	0x00,0xED,0x12,0xDE,0x6F,0xA6, // paddsw mmx,mmx
	0x00,0xDC,0x13,0x9E,0xDF,0xA6, // paddusb xmm,[r/m]
	0x00,0xDC,0x13,0xBE,0xDF,0xA6, // paddusb xmm,xmm
	0x00,0xDC,0x12,0x9E,0xEF,0xA6, // paddusb mmx,[r/m]
	0x00,0xDC,0x12,0xDE,0xEF,0xA6, // paddusb mmx,mmx
	0x00,0xDD,0x13,0x9E,0x5F,0xA7, // paddusw xmm,[r/m]
	0x00,0xDD,0x13,0xBE,0x5F,0xA7, // paddusw xmm,xmm
	0x00,0xDD,0x12,0x9E,0x6F,0xA7, // paddusw mmx,[r/m]
	0x00,0xDD,0x12,0xDE,0x6F,0xA7, // paddusw mmx,mmx
	0x00,0xFD,0x13,0x9E,0xDF,0xA7, // paddw xmm,[r/m]
	0x00,0xFD,0x13,0xBE,0xDF,0xA7, // paddw xmm,xmm
	0x00,0xFD,0x12,0x9E,0xEF,0xA7, // paddw mmx,[r/m]
	0x00,0xFD,0x12,0xDE,0xEF,0xA7, // paddw mmx,mmx
	0x00,0xDB,0x13,0x9E,0x5F,0xA8, // pand xmm,[r/m]
	0x00,0xDB,0x13,0xBE,0x5F,0xA8, // pand xmm,xmm
	0x00,0xDB,0x12,0x9E,0x6F,0xA8, // pand mmx,[r/m]
	0x00,0xDB,0x12,0xDE,0x6F,0xA8, // pand mmx,mmx
	0x00,0xDF,0x13,0x9E,0xDF,0xA8, // pandn xmm,[r/m]
	0x00,0xDF,0x13,0xBE,0xDF,0xA8, // pandn xmm,xmm
	0x00,0xDF,0x12,0x9E,0xEF,0xA8, // pandn mmx,[r/m]
	0x00,0xDF,0x12,0xDE,0xEF,0xA8, // pandn mmx,mmx
	0xF3,0x90,0x01,0xFE,0x7F,0xA9, // pause
	0x00,0xE0,0x13,0x9E,0xDF,0xA9, // pavgb xmm,[r/m]
	0x00,0xE0,0x13,0xBE,0xDF,0xA9, // pavgb xmm,xmm
	0x00,0xE0,0x12,0x9E,0xEF,0xA9, // pavgb mmx,[r/m]
	0x00,0xE0,0x12,0xDE,0xEF,0xA9, // pavgb mmx,mmx
	0x00,0xE3,0x13,0x9E,0x5F,0xAA, // pavgw xmm,[r/m]
	0x00,0xE3,0x13,0xBE,0x5F,0xAA, // pavgw xmm,xmm
	0x00,0xE3,0x12,0x9E,0x6F,0xAA, // pavgw mmx,[r/m]
	0x00,0xE3,0x12,0xDE,0x6F,0xAA, // pavgw mmx,mmx
	0x00,0x74,0x13,0x9E,0xDF,0xAA, // pcmpeqb xmm,[r/m]
	0x00,0x74,0x13,0xBE,0xDF,0xAA, // pcmpeqb xmm,xmm
	0x00,0x74,0x12,0x9E,0xEF,0xAA, // pcmpeqb mmx,[r/m]
	0x00,0x74,0x12,0xDE,0xEF,0xAA, // pcmpeqb mmx,mmx
	0x00,0x76,0x13,0x9E,0x5F,0xAB, // pcmpeqd xmm,[r/m]
	0x00,0x76,0x13,0xBE,0x5F,0xAB, // pcmpeqd xmm,xmm
	0x00,0x76,0x12,0x9E,0x6F,0xAB, // pcmpeqd mmx,[r/m]
	0x00,0x76,0x12,0xDE,0x6F,0xAB, // pcmpeqd mmx,mmx
	0x00,0x75,0x13,0x9E,0xDF,0xAB, // pcmpeqw xmm,[r/m]
	0x00,0x75,0x13,0xBE,0xDF,0xAB, // pcmpeqw xmm,xmm
	0x00,0x75,0x12,0x9E,0xEF,0xAB, // pcmpeqw mmx,[r/m]
	0x00,0x75,0x12,0xDE,0xEF,0xAB, // pcmpeqw mmx,mmx
	0x00,0x64,0x13,0x9E,0x5F,0xAC, // pcmpgtb xmm,[r/m]
	0x00,0x64,0x13,0xBE,0x5F,0xAC, // pcmpgtb xmm,xmm
	0x00,0x64,0x12,0x9E,0x6F,0xAC, // pcmpgtb mmx,[r/m]
	0x00,0x64,0x12,0xDE,0x6F,0xAC, // pcmpgtb mmx,mmx
	0x00,0x66,0x13,0x9E,0xDF,0xAC, // pcmpgtd xmm,[r/m]
	0x00,0x66,0x13,0xBE,0xDF,0xAC, // pcmpgtd xmm,xmm
	0x00,0x66,0x12,0x9E,0xEF,0xAC, // pcmpgtd mmx,[r/m]
	0x00,0x66,0x12,0xDE,0xEF,0xAC, // pcmpgtd mmx,mmx
	0x00,0x65,0x13,0x9E,0x5F,0xAD, // pcmpgtw xmm,[r/m]
	0x00,0x65,0x13,0xBE,0x5F,0xAD, // pcmpgtw xmm,xmm
	0x00,0x65,0x12,0x9E,0x6F,0xAD, // pcmpgtw mmx,[r/m]
	0x00,0x65,0x12,0xDE,0x6F,0xAD, // pcmpgtw mmx,mmx
	0x00,0xC5,0x93,0xBF,0xBF,0xAD, // pextrw reg,xmm,imm
	0x00,0xC5,0x92,0xDF,0xBF,0xAD, // pextrw reg,mmx,imm
	0x00,0xC4,0x93,0x7F,0x5F,0xAE, // pinsrw xmm,reg,imm
	0x00,0xC4,0x93,0x9F,0x5F,0xAE, // pinsrw xmm,[r/m],imm
	0x00,0xC4,0x92,0x7F,0x6F,0xAE, // pinsrw mmx,reg,imm
	0x00,0xC4,0x92,0x9F,0x6F,0xAE, // pinsrw mmx,[r/m],imm
	0x00,0xF5,0x13,0x9E,0xDF,0xAE, // pmaddwd xmm,[r/m]
	0x00,0xF5,0x13,0xBE,0xDF,0xAE, // pmaddwd xmm,xmm
	0x00,0xF5,0x12,0x9E,0xEF,0xAE, // pmaddwd mmx,[r/m]
	0x00,0xF5,0x12,0xDE,0xEF,0xAE, // pmaddwd mmx,mmx
	0x00,0xEE,0x13,0x9E,0x5F,0xAF, // pmaxsw xmm,[r/m]
	0x00,0xEE,0x13,0xBE,0x5F,0xAF, // pmaxsw xmm,xmm
	0x00,0xEE,0x12,0x9E,0x6F,0xAF, // pmaxsw mmx,[r/m]
	0x00,0xEE,0x12,0xDE,0x6F,0xAF, // pmaxsw mmx,mmx
	0x00,0xDE,0x13,0x9E,0xDF,0xAF, // pmaxub xmm,[r/m]
	0x00,0xDE,0x13,0xBE,0xDF,0xAF, // pmaxub xmm,xmm
	0x00,0xDE,0x12,0x9E,0xEF,0xAF, // pmaxub mmx,[r/m]
	0x00,0xDE,0x12,0xDE,0xEF,0xAF, // pmaxub mmx,mmx
	0x00,0xEA,0x13,0x9E,0x5F,0xB0, // pminsw xmm,[r/m]
	0x00,0xEA,0x13,0xBE,0x5F,0xB0, // pminsw xmm,xmm
	0x00,0xEA,0x12,0x9E,0x6F,0xB0, // pminsw mmx,[r/m]
	0x00,0xEA,0x12,0xDE,0x6F,0xB0, // pminsw mmx,mmx
	0x00,0xDA,0x13,0x9E,0xDF,0xB0, // pminub xmm,[r/m]
	0x00,0xDA,0x13,0xBE,0xDF,0xB0, // pminub xmm,xmm
	0x00,0xDA,0x12,0x9E,0xEF,0xB0, // pminub mmx,[r/m]
	0x00,0xDA,0x12,0xDE,0xEF,0xB0, // pminub mmx,mmx
	0x00,0xD7,0x13,0xBE,0x3F,0xB1, // pmovmskb reg,xmm
	0x00,0xD7,0x12,0xDE,0x3F,0xB1, // pmovmskb reg,mmx
	0x00,0xE4,0x13,0x9E,0xDF,0xB1, // pmulhuw xmm,[r/m]
	0x00,0xE4,0x13,0xBE,0xDF,0xB1, // pmulhuw xmm,xmm
	0x00,0xE4,0x12,0x9E,0xEF,0xB1, // pmulhuw mmx,[r/m]
	0x00,0xE4,0x12,0xDE,0xEF,0xB1, // pmulhuw mmx,mmx
	0x00,0xE5,0x13,0x9E,0x5F,0xB2, // pmulhw xmm,[r/m]
	0x00,0xE5,0x13,0xBE,0x5F,0xB2, // pmulhw xmm,xmm
	0x00,0xE5,0x12,0x9E,0x6F,0xB2, // pmulhw mmx,[r/m]
	0x00,0xE5,0x12,0xDE,0x6F,0xB2, // pmulhw mmx,mmx
	0x00,0xD5,0x13,0x9E,0xDF,0xB2, // pmullw xmm,[r/m]
	0x00,0xD5,0x13,0xBE,0xDF,0xB2, // pmullw xmm,xmm
	0x00,0xD5,0x12,0x9E,0xEF,0xB2, // pmullw mmx,[r/m]
	0x00,0xD5,0x12,0xDE,0xEF,0xB2, // pmullw mmx,mmx
	0x00,0xF4,0x13,0x9E,0x5F,0xB3, // pmuludq xmm,[r/m]
	0x00,0xF4,0x13,0xBE,0x5F,0xB3, // pmuludq xmm,xmm
	0x00,0xF4,0x12,0x9E,0x6F,0xB3, // pmuludq mmx,[r/m]
	0x00,0xF4,0x12,0xDE,0x6F,0xB3, // pmuludq mmx,mmx
	0x00,0x58,0x20,0xFE,0xAF,0xB3, // pop reg64
	0x00,0x8F,0x34,0xFE,0xCF,0xB3, // pop [r/m]
	0xF3,0xB8,0x13,0x7E,0x3F,0xB4, // popcnt reg,reg
	0xF3,0xB8,0x13,0x9E,0x3F,0xB4, // popcnt reg,[r/m]
	0x00,0x9D,0x00,0xFE,0xFF,0xB4, // popf
	0x00,0x9D,0x00,0xFE,0x7F,0xB5, // popfq
	0x00,0xEB,0x13,0x9E,0xDF,0xB5, // por xmm,[r/m]
	0x00,0xEB,0x13,0xBE,0xDF,0xB5, // por xmm,xmm
	0x00,0xEB,0x12,0x9E,0xEF,0xB5, // por mmx,[r/m]
	0x00,0xEB,0x12,0xDE,0xEF,0xB5, // por mmx,mmx
	0x00,0xF6,0x13,0x9E,0x5F,0xB6, // psadbw xmm,[r/m]
	0x00,0xF6,0x13,0xBE,0x5F,0xB6, // psadbw xmm,xmm
	0x00,0xF6,0x12,0x9E,0x6F,0xB6, // psadbw mmx,[r/m]
	0x00,0xF6,0x12,0xDE,0x6F,0xB6, // psadbw mmx,mmx
	0x00,0x70,0x93,0x9F,0xDF,0xB6, // pshufd xmm,[r/m],imm
	0x00,0x70,0x93,0xBF,0xDF,0xB6, // pshufd xmm,xmm,imm
	0xF3,0x70,0x93,0x9F,0x5F,0xB7, // pshufhw xmm,[r/m],imm
	0xF3,0x70,0x93,0xBF,0x5F,0xB7, // pshufhw xmm,xmm,imm
	0xF2,0x70,0x93,0x9F,0xDF,0xB7, // pshuflw xmm,[r/m],imm
	0xF2,0x70,0x93,0xBF,0xDF,0xB7, // pshuflw xmm,xmm,imm
	0x00,0x70,0x92,0x9F,0x6F,0xB8, // pshufw mmx,[r/m],imm
	0x00,0x70,0x92,0xDF,0x6F,0xB8, // pshufw mmx,mmx,imm
	0x00,0xF2,0x13,0x9E,0xDF,0xB8, // pslld xmm,[r/m]
	0x00,0xF2,0x13,0xBE,0xDF,0xB8, // pslld xmm,xmm
	0x30,0x72,0xB3,0xFD,0xDF,0xB8, // pslld xmm,imm
	0x00,0xF2,0x12,0x9E,0xEF,0xB8, // pslld mmx,[r/m]
	0x00,0xF2,0x12,0xDE,0xEF,0xB8, // pslld mmx,mmx
	0x30,0x72,0xB2,0xFD,0xEF,0xB8, // pslld mmx,imm
	0x38,0x73,0xB3,0xFD,0x5F,0xB9, // pslldq xmm,imm
	0x00,0xF3,0x13,0x9E,0xDF,0xB9, // psllq xmm,[r/m]
	0x00,0xF3,0x13,0xBE,0xDF,0xB9, // psllq xmm,xmm
	0x30,0x73,0xB3,0xFD,0xDF,0xB9, // psllq xmm,imm
	0x00,0xF3,0x12,0x9E,0xEF,0xB9, // psllq mmx,[r/m]
	0x00,0xF3,0x12,0xDE,0xEF,0xB9, // psllq mmx,mmx
	0x30,0x73,0xB2,0xFD,0xEF,0xB9, // psllq mmx,imm
	0x00,0xF1,0x13,0x9E,0x5F,0xBA, // psllw xmm,[r/m]
	0x00,0xF1,0x13,0xBE,0x5F,0xBA, // psllw xmm,xmm
	0x30,0x71,0xB3,0xFD,0x5F,0xBA, // psllw xmm,imm
	0x00,0xF1,0x12,0x9E,0x6F,0xBA, // psllw mmx,[r/m]
	0x00,0xF1,0x12,0xDE,0x6F,0xBA, // psllw mmx,mmx
	0x30,0x71,0xB2,0xFD,0x6F,0xBA, // psllw mmx,imm
	0x00,0xE2,0x13,0x9E,0xDF,0xBA, // psrad xmm,[r/m]
	0x00,0xE2,0x13,0xBE,0xDF,0xBA, // psrad xmm,xmm
	0x20,0x72,0xB3,0xFD,0xDF,0xBA, // psrad xmm,imm
	0x00,0xE2,0x12,0x9E,0xEF,0xBA, // psrad mmx,[r/m]
	0x00,0xE2,0x12,0xDE,0xEF,0xBA, // psrad mmx,mmx
	0x20,0x72,0xB2,0xFD,0xEF,0xBA, // psrad mmx,imm
	0x00,0xE1,0x13,0x9E,0x5F,0xBB, // psraw xmm,[r/m]
	0x00,0xE1,0x13,0xBE,0x5F,0xBB, // psraw xmm,xmm
	0x20,0x71,0xB3,0xFD,0x5F,0xBB, // psraw xmm,imm
	0x00,0xE1,0x12,0x9E,0x6F,0xBB, // psraw mmx,[r/m]
	0x00,0xE1,0x12,0xDE,0x6F,0xBB, // psraw mmx,mmx
	0x20,0x71,0xB2,0xFD,0x6F,0xBB, // psraw mmx,imm
	0x00,0xD2,0x13,0x9E,0xDF,0xBB, // psrld xmm,[r/m]
	0x00,0xD2,0x13,0xBE,0xDF,0xBB, // psrld xmm,xmm
	0x10,0x72,0xB3,0xFD,0xDF,0xBB, // psrld xmm,imm
	0x00,0xD2,0x12,0x9E,0xEF,0xBB, // psrld mmx,[r/m]
	0x00,0xD2,0x12,0xDE,0xEF,0xBB, // psrld mmx,mmx
	0x10,0x72,0xB2,0xFD,0xEF,0xBB, // psrld mmx,imm
	0x18,0x73,0xB3,0xFD,0x5F,0xBC, // psrldq xmm,imm
	0x00,0xD3,0x13,0x9E,0xDF,0xBC, // psrlq xmm,[r/m]
	0x00,0xD3,0x13,0xBE,0xDF,0xBC, // psrlq xmm,xmm
	0x10,0x73,0xB3,0xFD,0xDF,0xBC, // psrlq xmm,imm
	0x00,0xD3,0x12,0x9E,0xEF,0xBC, // psrlq mmx,[r/m]
	0x00,0xD3,0x12,0xDE,0xEF,0xBC, // psrlq mmx,mmx
	0x10,0x73,0xB2,0xFD,0xEF,0xBC, // psrlq mmx,imm
	0x00,0xD1,0x13,0x9E,0x5F,0xBD, // psrlw xmm,[r/m]
	0x00,0xD1,0x13,0xBE,0x5F,0xBD, // psrlw xmm,xmm
	0x10,0x71,0xB3,0xFD,0x5F,0xBD, // psrlw xmm,imm
	0x00,0xD1,0x12,0x9E,0x6F,0xBD, // psrlw mmx,[r/m]
	0x00,0xD1,0x12,0xDE,0x6F,0xBD, // psrlw mmx,mmx
	0x10,0x71,0xB2,0xFD,0x6F,0xBD, // psrlw mmx,imm
	0x00,0xF8,0x13,0x9E,0xDF,0xBD, // psubb xmm,[r/m]
	0x00,0xF8,0x13,0xBE,0xDF,0xBD, // psubb xmm,xmm
	0x00,0xF8,0x12,0x9E,0xEF,0xBD, // psubb mmx,[r/m]
	0x00,0xF8,0x12,0xDE,0xEF,0xBD, // psubb mmx,mmx
	0x00,0xFA,0x13,0x9E,0x5F,0xBE, // psubd xmm,[r/m]
	0x00,0xFA,0x13,0xBE,0x5F,0xBE, // psubd xmm,xmm
	0x00,0xFA,0x12,0x9E,0x6F,0xBE, // psubd mmx,[r/m]
	0x00,0xFA,0x12,0xDE,0x6F,0xBE, // psubd mmx,mmx
	0x00,0xFB,0x13,0x9E,0xDF,0xBE, // psubq xmm,[r/m]
	0x00,0xFB,0x13,0xBE,0xDF,0xBE, // psubq xmm,xmm
	0x00,0xFB,0x12,0x9E,0xEF,0xBE, // psubq mmx,[r/m]
	0x00,0xFB,0x12,0xDE,0xEF,0xBE, // psubq mmx,mmx
	0x00,0xE8,0x13,0x9E,0x5F,0xBF, // psubsb xmm,[r/m]
	0x00,0xE8,0x13,0xBE,0x5F,0xBF, // psubsb xmm,xmm
	0x00,0xE8,0x12,0x9E,0x6F,0xBF, // psubsb mmx,[r/m]
	0x00,0xE8,0x12,0xDE,0x6F,0xBF, // psubsb mmx,mmx
	0x00,0xE9,0x13,0x9E,0xDF,0xBF, // psubsw xmm,[r/m]
	0x00,0xE9,0x13,0xBE,0xDF,0xBF, // psubsw xmm,xmm
	0x00,0xE9,0x12,0x9E,0xEF,0xBF, // psubsw mmx,[r/m]
	0x00,0xE9,0x12,0xDE,0xEF,0xBF, // psubsw mmx,mmx
	0x00,0xD8,0x13,0x9E,0x5F,0xC0, // psubusb xmm,[r/m]
	0x00,0xD8,0x13,0xBE,0x5F,0xC0, // psubusb xmm,xmm
	0x00,0xD8,0x12,0x9E,0x6F,0xC0, // psubusb mmx,[r/m]
	0x00,0xD8,0x12,0xDE,0x6F,0xC0, // psubusb mmx,mmx
	0x00,0xD9,0x13,0x9E,0xDF,0xC0, // psubusw xmm,[r/m]
	0x00,0xD9,0x13,0xBE,0xDF,0xC0, // psubusw xmm,xmm
	0x00,0xD9,0x12,0x9E,0xEF,0xC0, // psubusw mmx,[r/m]
	0x00,0xD9,0x12,0xDE,0xEF,0xC0, // psubusw mmx,mmx
	0x00,0xF9,0x13,0x9E,0x5F,0xC1, // psubw xmm,[r/m]
	0x00,0xF9,0x13,0xBE,0x5F,0xC1, // psubw xmm,xmm
	0x00,0xF9,0x12,0x9E,0x6F,0xC1, // psubw mmx,[r/m]
	0x00,0xF9,0x12,0xDE,0x6F,0xC1, // psubw mmx,mmx
	0x00,0x68,0x13,0x9E,0xDF,0xC1, // punpckhbw xmm,[r/m]
	0x00,0x68,0x13,0xBE,0xDF,0xC1, // punpckhbw xmm,xmm
	0x00,0x68,0x12,0x9E,0xEF,0xC1, // punpckhbw mmx,[r/m]
	0x00,0x68,0x12,0xDE,0xEF,0xC1, // punpckhbw mmx,mmx
	0x00,0x6A,0x13,0x9E,0x5F,0xC2, // punpckhdq xmm,[r/m]
	0x00,0x6A,0x13,0xBE,0x5F,0xC2, // punpckhdq xmm,xmm
	0x00,0x6A,0x12,0x9E,0x6F,0xC2, // punpckhdq mmx,[r/m]
	0x00,0x6A,0x12,0xDE,0x6F,0xC2, // punpckhdq mmx,mmx
	0x00,0x6D,0x13,0x9E,0xDF,0xC2, // punpckhqdq xmm,[r/m]
	0x00,0x6D,0x13,0xBE,0xDF,0xC2, // punpckhqdq xmm,xmm
	0x00,0x69,0x13,0x9E,0x5F,0xC3, // punpckhwd xmm,[r/m]
	0x00,0x69,0x13,0xBE,0x5F,0xC3, // punpckhwd xmm,xmm
	0x00,0x69,0x12,0x9E,0x6F,0xC3, // punpckhwd mmx,[r/m]
	0x00,0x69,0x12,0xDE,0x6F,0xC3, // punpckhwd mmx,mmx
	0x00,0x60,0x13,0x9E,0xDF,0xC3, // punpcklbw xmm,[r/m]
	0x00,0x60,0x13,0xBE,0xDF,0xC3, // punpcklbw xmm,xmm
	0x00,0x60,0x12,0x9E,0xEF,0xC3, // punpcklbw mmx,[r/m]
	0x00,0x60,0x12,0xDE,0xEF,0xC3, // punpcklbw mmx,mmx
	0x00,0x62,0x13,0x9E,0x5F,0xC4, // punpckldq xmm,[r/m]
	0x00,0x62,0x13,0xBE,0x5F,0xC4, // punpckldq xmm,xmm
	0x00,0x62,0x12,0x9E,0x6F,0xC4, // punpckldq mmx,[r/m]
	0x00,0x62,0x12,0xDE,0x6F,0xC4, // punpckldq mmx,mmx
	0x00,0x6C,0x13,0x9E,0xDF,0xC4, // punpcklqdq xmm,[r/m]
	0x00,0x6C,0x13,0xBE,0xDF,0xC4, // punpcklqdq xmm,xmm
	0x00,0x61,0x13,0x9E,0x5F,0xC5, // punpcklwd xmm,[r/m]
	0x00,0x61,0x13,0xBE,0x5F,0xC5, // punpcklwd xmm,xmm
	0x00,0x61,0x12,0x9E,0x6F,0xC5, // punpcklwd mmx,[r/m]
	0x00,0x61,0x12,0xDE,0x6F,0xC5, // punpcklwd mmx,mmx
	0x00,0x50,0x20,0xFE,0xAF,0xC5, // push reg64
	0x30,0xFF,0x30,0xFE,0xCF,0xC5, // push [r/m]
	0x00,0x68,0xC8,0xFF,0xFE,0xC5, // push imm
	0x00,0x9C,0x00,0xFE,0x7F,0xC6, // pushf
	0x00,0x9C,0x00,0xFE,0xFF,0xC6, // pushfq
	0x00,0xEF,0x13,0x9E,0x5F,0xC7, // pxor xmm,[r/m]
	0x00,0xEF,0x13,0xBE,0x5F,0xC7, // pxor xmm,xmm
	0x00,0xEF,0x12,0x9E,0x6F,0xC7, // pxor mmx,[r/m]
	0x00,0xEF,0x12,0xDE,0x6F,0xC7, // pxor mmx,mmx
	0x10,0xD3,0x34,0x02,0xBF,0xC7, // rcl reg,cl
	0x10,0xC1,0xB4,0xFD,0xBF,0xC7, // rcl reg,imm
	0x10,0xD3,0x34,0x02,0xCF,0xC7, // rcl [r/m],cl
	0x10,0xC1,0xB4,0xFD,0xCF,0xC7, // rcl [r/m],imm
	0x00,0x53,0x12,0x9E,0x5F,0xC8, // rcpps xmm,[r/m]
	0x00,0x53,0x12,0xBE,0x5F,0xC8, // rcpps xmm,xmm
	0xF3,0x53,0x13,0x9E,0xDF,0xC8, // rcpss xmm,[r/m]
	0xF3,0x53,0x13,0xBE,0xDF,0xC8, // rcpss xmm,xmm
	0x18,0xD3,0x34,0x02,0x3F,0xC9, // rcr reg,cl
	0x18,0xC1,0xB4,0xFD,0x3F,0xC9, // rcr reg,imm
	0x18,0xD3,0x34,0x02,0x4F,0xC9, // rcr [r/m],cl
	0x18,0xC1,0xB4,0xFD,0x4F,0xC9, // rcr [r/m],imm
	0x00,0x33,0x02,0xFE,0xFF,0xC9, // rdpmc
	0x00,0x31,0x02,0xFE,0x7F,0xCA, // rdtsc
	0xF9,0x01,0x02,0xFE,0xFF,0xCA, // rdtscp
	0x00,0xF3,0x00,0xFE,0x7F,0xCB, // rep
	0x00,0xF3,0x00,0xFE,0xFF,0xCB, // repe
	0x00,0xF2,0x00,0xFE,0x7F,0xCC, // repne
	0x00,0xF2,0x00,0xFE,0xFF,0xCC, // repnz
	0x00,0xF3,0x00,0xFE,0x7F,0xCD, // repz
	0x00,0xC2,0x80,0xFE,0xFE,0xCD, // ret imm
	0x00,0xC3,0x00,0xFE,0xFF,0xCD, // ret
	0x00,0xCA,0x80,0xFE,0x7E,0xCE, // retf imm
	0x00,0xCB,0x00,0xFE,0x7F,0xCE, // retf
	0x00,0xC2,0x80,0xFE,0xFE,0xCE, // retn imm
	0x00,0xC3,0x00,0xFE,0xFF,0xCE, // retn
	0x00,0xD3,0x34,0x02,0x3F,0xCF, // rol reg,cl
	0x00,0xC1,0xB4,0xFD,0x3F,0xCF, // rol reg,imm
	0x00,0xD3,0x34,0x02,0x4F,0xCF, // rol [r/m],cl
	0x00,0xC1,0xB4,0xFD,0x4F,0xCF, // rol [r/m],imm
	0x08,0xD3,0x34,0x02,0xBF,0xCF, // ror reg,cl
	0x08,0xC1,0xB4,0xFD,0xBF,0xCF, // ror reg,imm
	0x08,0xD3,0x34,0x02,0xCF,0xCF, // ror [r/m],cl
	0x08,0xC1,0xB4,0xFD,0xCF,0xCF, // ror [r/m],imm
	0x00,0x52,0x12,0x9E,0x5F,0xD0, // rsqrtps xmm,[r/m]
	0x00,0x52,0x12,0xBE,0x5F,0xD0, // rsqrtps xmm,xmm
	0xF3,0x52,0x13,0x9E,0xDF,0xD0, // rsqrtss xmm,[r/m]
	0xF3,0x52,0x13,0xBE,0xDF,0xD0, // rsqrtss xmm,xmm
	0x00,0x9E,0x00,0xFE,0x7F,0xD1, // sahf
	0x20,0xD3,0x34,0x02,0xBF,0xD1, // sal reg,cl
	0x20,0xC1,0xB4,0xFD,0xBF,0xD1, // sal reg,imm
	0x20,0xD3,0x34,0x02,0xCF,0xD1, // sal [r/m],cl
	0x20,0xC1,0xB4,0xFD,0xCF,0xD1, // sal [r/m],imm
	0x38,0xD3,0x34,0x02,0x3F,0xD2, // sar reg,cl
	0x38,0xC1,0xB4,0xFD,0x3F,0xD2, // sar reg,imm
	0x38,0xD3,0x34,0x02,0x4F,0xD2, // sar [r/m],cl
	0x38,0xC1,0xB4,0xFD,0x4F,0xD2, // sar [r/m],imm
	0x00,0x1D,0xC4,0xFD,0xB0,0xD2, // sbb rax,imm
	0x00,0x19,0x34,0x7E,0xBF,0xD2, // sbb reg,reg
	0x00,0x1B,0x14,0x9E,0xBF,0xD2, // sbb reg,[r/m]
	0x18,0x81,0xFC,0xFD,0xBF,0xD2, // sbb reg,imm
	0x00,0x19,0x34,0x7E,0xCF,0xD2, // sbb [r/m],reg
	0x18,0x81,0xFC,0xFD,0xCF,0xD2, // sbb [r/m],imm
	0x00,0xAE,0x04,0xFE,0x7F,0xD3, // scasb
	0x00,0xAF,0x00,0xFE,0xFF,0xD3, // scasd
	0x48,0xAF,0x01,0xFE,0x7F,0xD4, // scasq
	0x00,0xAF,0x01,0xFE,0xFF,0xD4, // scasw
	0x00,0x97,0x32,0xFE,0x0F,0xD5, // seta reg8
	0x00,0x97,0x32,0xFE,0x4F,0xD5, // seta [r/m]
	0x00,0x93,0x32,0xFE,0x8F,0xD5, // setae reg8
	0x00,0x93,0x32,0xFE,0xCF,0xD5, // setae [r/m]
	0x00,0x92,0x32,0xFE,0x0F,0xD6, // setb reg8
	0x00,0x92,0x32,0xFE,0x4F,0xD6, // setb [r/m]
	0x00,0x96,0x32,0xFE,0x8F,0xD6, // setbe reg8
	0x00,0x96,0x32,0xFE,0xCF,0xD6, // setbe [r/m]
	0x00,0x92,0x32,0xFE,0x0F,0xD7, // setc reg8
	0x00,0x92,0x32,0xFE,0x4F,0xD7, // setc [r/m]
	0x00,0x94,0x32,0xFE,0x8F,0xD7, // sete reg8
	0x00,0x94,0x32,0xFE,0xCF,0xD7, // sete [r/m]
	0x00,0x9F,0x32,0xFE,0x0F,0xD8, // setg reg8
	0x00,0x9F,0x32,0xFE,0x4F,0xD8, // setg [r/m]
	0x00,0x9D,0x32,0xFE,0x8F,0xD8, // setge reg8
	0x00,0x9D,0x32,0xFE,0xCF,0xD8, // setge [r/m]
	0x00,0x9C,0x32,0xFE,0x0F,0xD9, // setl reg8
	0x00,0x9C,0x32,0xFE,0x4F,0xD9, // setl [r/m]
	0x00,0x9E,0x32,0xFE,0x8F,0xD9, // setle reg8
	0x00,0x9E,0x32,0xFE,0xCF,0xD9, // setle [r/m]
	0x00,0x96,0x32,0xFE,0x0F,0xDA, // setna reg8
	0x00,0x96,0x32,0xFE,0x4F,0xDA, // setna [r/m]
	0x00,0x92,0x32,0xFE,0x8F,0xDA, // setnae reg8
	0x00,0x92,0x32,0xFE,0xCF,0xDA, // setnae [r/m]
	0x00,0x93,0x32,0xFE,0x0F,0xDB, // setnb reg8
	0x00,0x93,0x32,0xFE,0x4F,0xDB, // setnb [r/m]
	0x00,0x97,0x32,0xFE,0x8F,0xDB, // setnbe reg8
	0x00,0x97,0x32,0xFE,0xCF,0xDB, // setnbe [r/m]
	0x00,0x93,0x32,0xFE,0x0F,0xDC, // setnc reg8
	0x00,0x93,0x32,0xFE,0x4F,0xDC, // setnc [r/m]
	0x00,0x95,0x32,0xFE,0x8F,0xDC, // setne reg8
	0x00,0x95,0x32,0xFE,0xCF,0xDC, // setne [r/m]
	0x00,0x9E,0x32,0xFE,0x0F,0xDD, // setng reg8
	0x00,0x9E,0x32,0xFE,0x4F,0xDD, // setng [r/m]
	0x00,0x9C,0x32,0xFE,0x8F,0xDD, // setnge reg8
	0x00,0x9C,0x32,0xFE,0xCF,0xDD, // setnge [r/m]
	0x00,0x9D,0x32,0xFE,0x0F,0xDE, // setnl reg8
	0x00,0x9D,0x32,0xFE,0x4F,0xDE, // setnl [r/m]
	0x00,0x9F,0x32,0xFE,0x8F,0xDE, // setnle reg8
	0x00,0x9F,0x32,0xFE,0xCF,0xDE, // setnle [r/m]
	0x00,0x91,0x32,0xFE,0x0F,0xDF, // setno reg8
	0x00,0x91,0x32,0xFE,0x4F,0xDF, // setno [r/m]
	0x00,0x9B,0x32,0xFE,0x8F,0xDF, // setnp reg8
	0x00,0x9B,0x32,0xFE,0xCF,0xDF, // setnp [r/m]
	0x00,0x99,0x32,0xFE,0x0F,0xE0, // setns reg8
	0x00,0x99,0x32,0xFE,0x4F,0xE0, // setns [r/m]
	0x00,0x95,0x32,0xFE,0x8F,0xE0, // setnz reg8
	0x00,0x95,0x32,0xFE,0xCF,0xE0, // setnz [r/m]
	0x00,0x90,0x32,0xFE,0x0F,0xE1, // seto reg8
	0x00,0x90,0x32,0xFE,0x4F,0xE1, // seto [r/m]
	0x00,0x9A,0x32,0xFE,0x8F,0xE1, // setp reg8
	0x00,0x9A,0x32,0xFE,0xCF,0xE1, // setp [r/m]
	0x00,0x9A,0x32,0xFE,0x0F,0xE2, // setpe reg8
	0x00,0x9A,0x32,0xFE,0x4F,0xE2, // setpe [r/m]
	0x00,0x9B,0x32,0xFE,0x8F,0xE2, // setpo reg8
	0x00,0x9B,0x32,0xFE,0xCF,0xE2, // setpo [r/m]
	0x00,0x98,0x32,0xFE,0x0F,0xE3, // sets reg8
	0x00,0x98,0x32,0xFE,0x4F,0xE3, // sets [r/m]
	0x00,0x94,0x32,0xFE,0x8F,0xE3, // setz reg8
	0x00,0x94,0x32,0xFE,0xCF,0xE3, // setz [r/m]
	0xF8,0xAE,0x02,0xFE,0x7F,0xE4, // sfence
	0x00,0x01,0x32,0xFE,0xCF,0xE4, // sgdt [r/m]
	0x20,0xD3,0x34,0x02,0x3F,0xE5, // shl reg,cl
	0x20,0xC1,0xB4,0xFD,0x3F,0xE5, // shl reg,imm
	0x20,0xD3,0x34,0x02,0x4F,0xE5, // shl [r/m],cl
	0x20,0xC1,0xB4,0xFD,0x4F,0xE5, // shl [r/m],imm
	0x00,0xA5,0x32,0x7E,0xBF,0xE5, // shld reg,reg,cl
	0x00,0xA4,0xB2,0x7F,0xBF,0xE5, // shld reg,reg,imm
	0x00,0xA5,0x32,0x7E,0xCF,0xE5, // shld [r/m],reg,cl
	0x00,0xA4,0xB2,0x7F,0xCF,0xE5, // shld [r/m],reg,imm
	0x28,0xD3,0x34,0x02,0x3F,0xE6, // shr reg,cl
	0x28,0xC1,0xB4,0xFD,0x3F,0xE6, // shr reg,imm
	0x28,0xD3,0x34,0x02,0x4F,0xE6, // shr [r/m],cl
	0x28,0xC1,0xB4,0xFD,0x4F,0xE6, // shr [r/m],imm
	0x00,0xAD,0x32,0x7E,0xBF,0xE6, // shrd reg,reg,cl
	0x00,0xAC,0xB2,0x7F,0xBF,0xE6, // shrd reg,reg,imm
	0x00,0xAD,0x32,0x7E,0xCF,0xE6, // shrd [r/m],reg,cl
	0x00,0xAC,0xB2,0x7F,0xCF,0xE6, // shrd [r/m],reg,imm
	0x00,0xC6,0x93,0x9F,0x5F,0xE7, // shufpd xmm,[r/m],imm
	0x00,0xC6,0x93,0xBF,0x5F,0xE7, // shufpd xmm,xmm,imm
	0x00,0xC6,0x92,0x9F,0xDF,0xE7, // shufps xmm,[r/m],imm
	0x00,0xC6,0x92,0xBF,0xDF,0xE7, // shufps xmm,xmm,imm
	0x20,0x01,0x32,0xFE,0x3F,0xE8, // smsw reg
	0x20,0x01,0x32,0xFE,0x4F,0xE8, // smsw [r/m]
	0x00,0x51,0x13,0x9E,0xDF,0xE8, // sqrtpd xmm,[r/m]
	0x00,0x51,0x13,0xBE,0xDF,0xE8, // sqrtpd xmm,xmm
	0x00,0x51,0x12,0x9E,0x5F,0xE9, // sqrtps xmm,[r/m]
	0x00,0x51,0x12,0xBE,0x5F,0xE9, // sqrtps xmm,xmm
	0xF2,0x51,0x13,0x9E,0xDF,0xE9, // sqrtsd xmm,[r/m]
	0xF2,0x51,0x13,0xBE,0xDF,0xE9, // sqrtsd xmm,xmm
	0xF3,0x51,0x13,0x9E,0x5F,0xEA, // sqrtss xmm,[r/m]
	0xF3,0x51,0x13,0xBE,0x5F,0xEA, // sqrtss xmm,xmm
	0x00,0xF9,0x00,0xFE,0xFF,0xEA, // stc
	0x00,0xFD,0x00,0xFE,0x7F,0xEB, // std
	0x00,0xFB,0x00,0xFE,0xFF,0xEB, // sti
	0x18,0xAE,0x32,0xFE,0x4F,0xEC, // stmxcsr [r/m]
	0x00,0xAA,0x04,0xFE,0xFF,0xEC, // stosb
	0x00,0xAB,0x00,0xFE,0x7F,0xED, // stosd
	0x48,0xAB,0x01,0xFE,0xFF,0xED, // stosq
	0x00,0xAB,0x01,0xFE,0x7F,0xEE, // stosw
	0x00,0x2D,0xC4,0xFD,0xB0,0xEE, // sub rax,imm
	0x00,0x29,0x34,0x7E,0xBF,0xEE, // sub reg,reg
	0x00,0x2B,0x14,0x9E,0xBF,0xEE, // sub reg,[r/m]
	0x28,0x81,0xFC,0xFD,0xBF,0xEE, // sub reg,imm
	0x00,0x29,0x34,0x7E,0xCF,0xEE, // sub [r/m],reg
	0x28,0x81,0xFC,0xFD,0xCF,0xEE, // sub [r/m],imm
	0x00,0x5C,0x13,0x9E,0x5F,0xEF, // subpd xmm,[r/m]
	0x00,0x5C,0x13,0xBE,0x5F,0xEF, // subpd xmm,xmm
	0x00,0x5C,0x12,0x9E,0xDF,0xEF, // subps xmm,[r/m]
	0x00,0x5C,0x12,0xBE,0xDF,0xEF, // subps xmm,xmm
	0xF2,0x5C,0x13,0x9E,0x5F,0xF0, // subsd xmm,[r/m]
	0xF2,0x5C,0x13,0xBE,0x5F,0xF0, // subsd xmm,xmm
	0xF3,0x5C,0x13,0x9E,0xDF,0xF0, // subss xmm,[r/m]
	0xF3,0x5C,0x13,0xBE,0xDF,0xF0, // subss xmm,xmm
	0x00,0xA9,0xC4,0xFD,0x30,0xF1, // test rax,imm
	0x00,0x85,0x34,0x7E,0x3F,0xF1, // test reg,reg
	0x00,0xF7,0xF4,0xFD,0x3F,0xF1, // test reg,imm
	0x00,0x85,0x34,0x7E,0x4F,0xF1, // test [r/m],reg
	0x00,0xF7,0xF4,0xFD,0x4F,0xF1, // test [r/m],imm
	0x00,0x2E,0x13,0x9E,0xDF,0xF1, // ucomisd xmm,[r/m]
	0x00,0x2E,0x13,0xBE,0xDF,0xF1, // ucomisd xmm,xmm
	0x00,0x2E,0x12,0x9E,0x5F,0xF2, // ucomiss xmm,[r/m]
	0x00,0x2E,0x12,0xBE,0x5F,0xF2, // ucomiss xmm,xmm
	0x00,0x15,0x13,0x9E,0xDF,0xF2, // unpckhpd xmm,[r/m]
	0x00,0x15,0x13,0xBE,0xDF,0xF2, // unpckhpd xmm,xmm
	0x00,0x15,0x12,0x9E,0x5F,0xF3, // unpckhps xmm,[r/m]
	0x00,0x15,0x12,0xBE,0x5F,0xF3, // unpckhps xmm,xmm
	0x00,0x14,0x13,0x9E,0xDF,0xF3, // unpcklpd xmm,[r/m]
	0x00,0x14,0x13,0xBE,0xDF,0xF3, // unpcklpd xmm,xmm
	0x00,0x14,0x12,0x9E,0x5F,0xF4, // unpcklps xmm,[r/m]
	0x00,0x14,0x12,0xBE,0x5F,0xF4, // unpcklps xmm,xmm
	0x20,0x00,0x32,0xFE,0x9F,0xF4, // verr reg16
	0x20,0x00,0x32,0xFE,0xCF,0xF4, // verr [r/m]
	0x28,0x00,0x32,0xFE,0x1F,0xF5, // verw reg16
	0x28,0x00,0x32,0xFE,0x4F,0xF5, // verw [r/m]
	0x00,0x9B,0x00,0xFE,0xFF,0xF5, // wait
	0x00,0xC1,0x36,0x7E,0x3F,0xF6, // xadd reg,reg
	0x00,0xC1,0x36,0x7E,0x4F,0xF6, // xadd [r/m],reg
	0x00,0x87,0x14,0x7E,0xBF,0xF6, // xchg reg,reg
	0x00,0x87,0x14,0x9E,0xBF,0xF6, // xchg reg,[r/m]
	0x00,0x87,0x34,0x7E,0xCF,0xF6, // xchg [r/m],reg
	0x00,0xD7,0x00,0xFE,0x7F,0xF7, // xlatb
	0x00,0x35,0xC4,0xFD,0xB0,0xF7, // xor rax,imm
	0x00,0x31,0x34,0x7E,0xBF,0xF7, // xor reg,reg
	0x00,0x33,0x14,0x9E,0xBF,0xF7, // xor reg,[r/m]
	0x30,0x81,0xFC,0xFD,0xBF,0xF7, // xor reg,imm
	0x00,0x31,0x34,0x7E,0xCF,0xF7, // xor [r/m],reg
	0x30,0x81,0xFC,0xFD,0xCF,0xF7, // xor [r/m],imm
	0x00,0x57,0x13,0x9E,0x5F,0xF8, // xorpd xmm,[r/m]
	0x00,0x57,0x13,0xBE,0x5F,0xF8, // xorpd xmm,xmm
	0x00,0x57,0x12,0x9E,0xDF,0xF8, // xorps xmm,[r/m]
	0x00,0x57,0x12,0xBE,0xDF,0xF8, // xorps xmm,xmm
	0x28,0xAE,0x32,0xFE,0x4F,0xF9, // xrstor [r/m]
	0x20,0xAE,0x32,0xFE,0xCF,0xF9 } ; // xsave [r/m]

static unsigned char segpfx[] = {0x26, 0x2E, 0x36, 0x3E, 0x64, 0x65} ;

static char *oslist[] = {
		"osrdch", "oswrch", "oskey", "osline", "oscli", "osopen", "osbyte", "osword",
		"osshut", "osbget", "osbput", "getptr", "setptr", "getext" } ; 

static void *osfunc[] = {
		osrdch, oswrch, oskey, osline, oscli, osopen, osbyte, osword,
		osshut, osbget, osbput, getptr, setptr, getext } ;

// String comparison (q is NUL-terminated):
static int cmps (int n, char *p, char *q)
{
	int c = strnicmp (p, q, n) ;
	if (c != 0)
		return c ;
	return -(n < strlen(q)) ;
} 

// Scan for string in unordered table, e.g. operands[]
static int scan (char **arr, int num)
{
	int i, n = 0 ;
	while (((*(esi + n) >= 'A') && (*(esi + n) != ']')) || ((*(esi + n) >= '0') && (*(esi + n) <= '9')))
		n++ ;
	if (n == 0)
		return -1 ;
	for (i = 0;  i < num;  i++)
		if (cmps (n, (char *)esi, *(arr + i)) == 0)
			break ; 
	if (i >= num)
		return -1 ; 
	esi += n ; 
	return i ; 
}

// Find string in ordered table, e.g. mnemonics[]:
static int schop (char **arr, int num)
{
	int b = 0, c = 0 , h = 1, n = 0 ;
	while ((*(esi + n) >= 'A') || ((*(esi + n) >= '0') && (*(esi + n) <= '9')))
		n++ ; 
	while (h < num) { h *= 2 ; }
	while (h)
	    {
		int i ;
		h /= 2 ;
		i = b + h ;
		if (i < num)
			if ((c = cmps (n, (char *)esi, *(arr + i))) >= 0)
				b += h ;
	    }
	if (c != 0)
		return -1 ;
	esi += n ;
	return b ; 			
}

// Find 24-bit integer in ordered table, e.g. instructions[]
static int nchop (unsigned char *arr, unsigned int v, int num)
{
	int b = 0, h = 1 ;
	while (h < num) { h *= 2 ; }
	while (h)
	    {
		int i ;
		h /= 2 ;
		i = b + h ;
		if (i < num)
			if (v >= (0xFFFFFF00 & *(unsigned int *)(arr + 6 * i)))
				b += h ;
	    }
	if (v != (0xFFFFFF00 & *(unsigned int *)(arr + 6 * b)))
		return -1 ;
	return b ; 			
}

// Lookup function address
static void *function (char *addr, int len)
{
	int i, n = sizeof(oslist) / sizeof(oslist[0]) ;
	for (i = 0;  i < n;  i++)
		if (cmps (len, addr, *(oslist + i)) == 0)
			break ; 
	if (i < n)
		return *(osfunc + i) ;

	*(addr + len) = 0 ; // NUL-terminate (n.b. addr = accs)
	return sysadr (addr) ;
}

static void tabit (int x)
{
	if (vcount == x) 
		return ; 
	if (vcount > x)
		crlf () ; 
	spaces (x - vcount) ; 
}

static char *poke (void *p, int n) 
{
	char *d ;
	int tmp ;

	d = PC ;
	stavar[16] += n ;
	tmp = stavar[16] ;
	if (liston & BIT6)
	    {
		d = OC ;
		stavar[15] += n ;
		tmp = stavar[15] ;
	    }

	if ((d < (char *) 0x10000) || ((liston & BIT7) && (tmp >= stavar[12])))
		error (8, NULL) ; // 'Address out of range'

	memcpy (d, p, n) ;
	return d ;
}

// Process 64-bit indexed operands: [reg{*n} {+{n*}reg{*n}} {+disp}]
static long long sibeval(int type, int *pdis)
{
	char c ;
	int p = 1 ;
	long long sibinfo = 0 ;

	while (1)
	    {
		type &= 15 ;
		c = nxt () ;
		if (c == '*')
		    {
			esi++ ;
			c = nxt () ;
			if ((c < '1') || (c > '9'))
				error (16, NULL) ; // 'Syntax error'
			p *= (c & 0xF) ;
			esi++ ;
			c = nxt () ;
		    }

		sibinfo += ((long long) p << (type * 4)) ;
		p = 1 ;

		if (c == '-')
		    {
			*pdis = expri () ;
			return sibinfo ;			
		    }

		if (c != '+')
			break ;

		esi++ ;
		c = nxt () ;
		if ((c >= '1') && (c <= '9'))
		    {
			esi++ ;
			if (nxt () == '*')
			    {
				p *= (c & 0xF) ;
				while (*++esi == ' ') ;
			    }
			else
				esi-- ;
		    }
		type = scan (operands, sizeof(operands) / sizeof(operands[0])) ;
		if ((type < 0) && (p == 1))
		    {
			*pdis = expri () ;
			return sibinfo ;			
		    }
		if ((type < 0x20) || (type > 0x2F))
			error (16, NULL) ; // 'Syntax error'
	    }
	return sibinfo ;
}

// Parse and evaluate an operand.  The 'displacement' and 'immediate' values
// are updated only if the operand if of the relevant types (0x7C, 0x7E).
// If no operand (or size override etc.) is present then 0x7F is returned.
static int operand (int *pdis, long long *pimm, int *povr, long long *psib)
{
	char c ;
	int type ;
	VAR v ;

	c = nxt () ;
	if ((c == 0x0D) || (c == ':') || (c == ';') || (c == TREM))
		return 0x7F ;
	type = scan (operands, sizeof(operands) / sizeof(operands[0])) ;

	if (type >= 0x80) // size override
	    {
		*povr |= 1 << (type - 0x80) ;
		return operand (pdis, pimm, povr, psib) ;
	    }

	if ((type >= 0x70) && (type < 0x76) && (*esi == ':')) // segment override
	    {
		esi++ ;
		*povr |= 1 << (type - 0x68) ;
		return operand (pdis, pimm, povr, psib) ;
	    }

	if (type == 0x7C) // [rel dest]
	    {
		*pdis = expri () - (long long) PC ;
		if (nxt() != ']')
			error (16, NULL) ; // 'Syntax error'
		esi++ ;
		*psib = 0 ;
		return 0x45 ;
	    }

	if (type == 0x7D) // [rip + disp]
	    {
		if (nxt() != ']')
			*pdis = expri () ;
		if (nxt() != ']')
			error (16, NULL) ; // 'Syntax error'
		esi++ ;
		*psib = 0 ;
		return 0x45 ; // [rip coded as [rbp when SIB = 0
	    }

	if ((type >= 0x40) && (type <= 0x4F)) // indirect addressing
	    {
		if ((nxt () != ']') || (type == 0x44) || (type == 0x45) || (type == 0x4C) || (type == 0x4D))
			*psib = sibeval (type, pdis) ;
		if (nxt () != ']')
			error (16, NULL) ; // 'Syntax error'
		esi++ ;
	    }

	if (type >= 0)
		return type ;

	if (c == '[') // direct address
	    {
		esi++ ;
		*pimm = expri () ;
		if (nxt () != ']')
			error (16, NULL) ; // 'Syntax error'
		esi++ ;
		return 0x7C ;
	    }

	v = expr () ;
	if (v.s.t == -1)
	    {
		if (v.s.l > 255)
			error (19, NULL) ; // 'String too long'
		memcpy (accs, (const char *) (v.s.p + zero), v.s.l) ;
		*pimm = v.s.l ;
		*povr |= 0b10000000 ; // flag string
	    }
	else
	    {
		if (v.i.t)
			v.i.n = v.f ;
		if (nxt () == '[')
		    {
			*pdis = v.i.n ;
			return operand (pdis, pimm, povr, psib) ;
		    }
		*pimm = v.i.n ;
	    }
	return 0x7E ;
}

// Lookup combination of mnemonic and operands in instructions table
static int lookup (int mnemonic, int operand1, int operand2, int operand3)
{
	int instruction = 0 ;
	int promote ;
	int found ;

	if ((operand1 >= 0x78) && (operand1 <= 0x7B)) // SPL etc.
		operand1 -= 0x74 ;
	if ((operand2 >= 0x78) && (operand2 <= 0x7B)) // SPL etc.
		operand2 -= 0x74 ;

	for (promote = 0; promote < 180; promote++)
	    {
		instruction = (mnemonic << 23) | (operand1 << 16) | (operand2 << 9) ;
		if ((operand1 == 0x7E) || (operand2 == 0x7E) || (operand3 == 0x7E))
			instruction |= 0b00000000000000000000000100000000 ;
		if (promote / 90)
			instruction &= 0b11111111111111111111111011111111 ;

		  switch (promote % 10) // vertical promotion (do first for al,ax,eax special cases)
		    {
			case 9: instruction |= 0b00000000000100000010000000000000 ; // 0x->1x regb,regb
			case 8:
			     if (instruction & 0b00000000001000000100000000000000)
					continue ;
				instruction += 0b00000000000100000010000000000000 ; // 1x->2x regw,regw
			case 7: instruction |= 0b00000000000100000010000000000000 ; // 2x->3x regq,regq
			     if (instruction & 0b00000000010000001000000000000000)
					continue ;
				break ;

			case 6: instruction |= 0b00000000000000000010000000000000 ; // 0x->1x nc,regb
			case 5:
			     if (instruction & 0b00000000000000000100000000000000)
					continue ;
				instruction += 0b00000000000000000010000000000000 ; // 1x->2x nc,regw
			case 4: instruction |= 0b00000000000000000010000000000000 ; // 2x->3x nc,regq
			     if (instruction & 0b00000000000000001000000000000000)
					continue ;
				break ;

			case 3: instruction |= 0b00000000000100000000000000000000 ; // 0x->1x regb,nc
			case 2:
			     if (instruction & 0b00000000001000000000000000000000)
					continue ;
				instruction += 0b00000000000100000000000000000000 ; // 1x->2x regw,nc
			case 1: instruction |= 0b00000000000100000000000000000000 ; // 2x->3x regq,nc
			     if (instruction & 0b00000000010000000000000000000000)
					continue ;
		    }
		switch ((promote / 10) % 3) // horizontal promotion of operand 1
		    {
			case 2: instruction |= 0b00000000000011110000000000000000 ; //        regd,[mem]
			case 1: instruction |= 0b00000000000001110000000000000000 ; //     STn/MMn,[mem]
		    }
		switch ((promote / 30) % 3) // horizontal promotion of operand 2
		    {
			case 2: instruction |= 0b00000000000000000001111000000000 ; //      [mem],regd
			case 1: instruction |= 0b00000000000000000000111000000000 ; //      [mem],STn/MMn
		    }
		found = nchop (instructions + 2, instruction, sizeof(instructions) / 6) ;
		if (found >= 0)
			return found ;
	    }
	return -1 ;
}

// Encode a single assembly-language instruction:
static char *encode (int mnemonic, int operand1, int operand2, int operand3, int override, 
		     int displacement, long long sibinfo, long long immediate)
{
	int found = -1 ;
	unsigned char opcode1, opcode2, modrm = 0 ;
	unsigned char scale = 0, index = 0, base = 0 ;
	short flags ;
	char *pdisp = NULL ;

	if ((operand1 == 0x30) && (operand2 == 0x7E) && (immediate >= -128) && (immediate <= 127))
		found = lookup (mnemonic, 0x3F, operand2, operand3) ;

// 32-bit addressing is not useful in a 64-bit flat addressing space.
//	if ((found < 0) && (operand1 == 0x7C))
//		found = lookup (mnemonic, 0x4F, operand2, operand3) ;
//	if ((found < 0) && (operand2 == 0x7C))
//		found = lookup (mnemonic, operand1, 0x4F, operand3) ;

	if (found < 0)
		found = lookup (mnemonic, operand1, operand2, operand3) ;
	if (found < 0)
		error (16, NULL) ; // 'Syntax error'
	flags = *(short *)(instructions + found*6 + 2) ;
	opcode1 = *(instructions + found*6 + 1) ;
	opcode2 = *(instructions + found*6) ;

	// We now have everything we need to start encoding:
	//  mnemonic (e.g. to determine if it's a FP inst)
	//  operand1 type 00-7F
	//  operand2 type 00-7F
	//  operand3 type 00-7F
	//  immediate data, memory poimter or string length
	//  displacement, if any
	//  sibinfo, if SIB byte needed (16 4-bit fields)
	//  override (size and segment overrides, string flag)
	//  opcode1 byte (or none if opcode1=&FF and opcode2=&FF)
	//  opcode2 byte (or skeleton 'ModR/M' or prefix)
	//  flags (9 bits determining encoding rules etc.):
	// Flag bits:
	//  bit 0:    48, 66, F2 or F3 prefix present (depending on value of so) +
	//  bit 1:    add 0F prefix (Floating Point: 9B prefix)
	//  bit 2:    update word-size flag W (Floating Point: F flag)
	//  bit 3:    update sign-extend flag S (Floating Point: M flag)
	//  bits 5-4: 00 = normal *
	//            01 = Mod R/M present: reg from LH, r/m from RH operand
	//            10 = No Mod R/M: update LS 3-bits of opcode from LH operand
	//            11 = Mod R/M present: reg from RH, r/m from LH operand
	//  bits 8-6: 000 = no extra data bytes
	//            001 = 32-bit immediate data
	//            010 = 16-bit immediate data
	//            011 = 64-bit immediate data
	//            100 = 8-bit signed PC relative offset
	//            101 = 32-bit signed PC relative offset
	//            110 = 8-bit immediate data
	//            111 = 16-bit, 32-bit or 64-bit immediate data (8-bit ?SW)
	//
	// + if bit 0 is one  so& is prefix byte if 48, F2 or F3, otherwise prefix is 66
	// * if bit 4 is zero so& is second opcode byte (e.g for FP instructions)
	//   if bit 4 is one  so& is skeleton Mod R/M byte (with optional 3-bit opcode extension)

	if (override & 0b1111111 & ((override &0b1111111) - 1))
		error (16, NULL) ; // 'Syntax error'

	if (((opcode1 & 0xF6) != 0xB6) && (opcode1 != 0x63) && (opcode1 != 0xD3))
		if ((operand1 <= 0x3F) && (operand2 <= 0x3F) && ((operand1 ^ operand2) & 0x30))
			error (53, NULL) ; // 'Size mismatch'

	if (((opcode1 & 0xF6) == 0xB6) && (operand2 >= 0x40) && (operand2 <= 0x4F))
		if ((override & 0b11) == 0)
			error (2, "Size needed") ; // 'Size needed'

	// Output segment prefix, if any:
	if (override & 0x3F00)
	    {
		int i ;
		for (i = 0; i < 6; i++)
			if (override & (1 << (i + 8)))
				poke (&segpfx[i], 1) ;
	    }

	// Output 66 prefix if 16-bit register or WORD override:
	if ((opcode1 != 0x00) && (((operand1 >= 0x10) && (operand1 <= 0x1F))
	   || ((((operand2 >= 0x10) && (operand2 <= 0x1F)) || (override & 0b00000010))
	   && ((opcode1 & 0xF6) != 0xB6) && ((opcode1 != 0x01) || (opcode2 != 0x20)))))

		if ((mnemonic < FPLO) || (mnemonic > FPHI))
			if ((operand1 < 0x70) || (operand1 > 0x75))
			    {
				unsigned char prefix = 0x66 ;
				poke (&prefix, 1) ;
			    }

	// Output 48, 66, 67, F2 or F3 prefix, if any:
	if (flags & 0b000000001)
	    {
		unsigned char prefix = 0x66 ;
		if ((opcode2 == 0xF2) || (opcode2 == 0xF3) || (opcode2 == 0x48) || (opcode2 == 0x67))
		    {
			prefix = opcode2 ;
			opcode2 = 0 ;
		    }
		poke (&prefix, 1) ;
	    }

	// Convert 64-bit sibinfo to scale, index and base:
	if (sibinfo != 0)
	    {
		int i, s ;

		for (s = 3; s >= 0; s--)
			for (i = 0; i < 16; i++)
				if ((sibinfo >> (i * 4)) & (1 << s))
				    {
					if (s)
					    {
						if (index)
							error (16, NULL) ; // 'Syntax error'
						index = i | 0x80 ;
						scale = s ;
					    }
					else if (base)
					    {
						if (index)
							error (16, NULL) ; // 'Syntax error'
						index = i | 0x80 ;
						scale = 0 ;
					    }
					else
						base = i | 0x80 ;
				    }

		// Choose shorter encodings when there are alternatives:
		if ((scale == 0) && (index == 0x84)) // if index is RSP swap base and index
		    {
			index = base ;
			base = 0x84 ;
		    }

		if ((scale == 0) && (index != 0) && ((base == 0x85) || (base == 0x8D)))
		    {
			unsigned char tmp = index ;
			index = base ;
			base = tmp ;
		    }

		if ((scale == 1) && (base == 0)) // [reg+reg] has shorter encoding than [reg*2] 
		    {
			scale = 0 ;
			base = index ;
		    }
	    }

	// Output 9B prefix, if any:
	if ((flags & 0b000000010) && (mnemonic >= FPLO) && (mnemonic <= FPHI))
	    {
		unsigned char prefix = 0x9B ;
		poke (&prefix, 1) ;
	    }

	// Output REX prefix, if any:
	    {
		unsigned char rex = 0x40 ;

		if ((opcode1 != 0x8F) && ((opcode1 != 0xFF) || (opcode2 < 0x10))
		    && (((opcode1 != 0x50) && (opcode1 != 0x58)) || (flags & 3)))
		    {
			if (((operand1 >= 0x20) && (operand1 <= 0x2F)) || ((operand2 >= 0x20) && (operand2 <= 0x2F))
			   || ((override & 0b00001000) && ((mnemonic < FPLO) || (mnemonic > FPHI))) 
			   || (mnemonic == CMPXCHG16B))
					rex |= 0b1000 ; // REX.W bit
		    }
		else
		    {
			if (override & 0b00000101)
				error (16, "Invalid size override") ;

		    }

		if (flags & 0b10000)
		    {
			if ((operand1 < 0x60) && (operand1 & 8))
			    {
				if (flags & 0b100000)
					rex |= 0b0001 ; // REX.B
				else
					rex |= 0b0100 ; // REX.R
			    }
			if ((operand2 < 0x60) && (operand2 & 8))
			    {
				if (flags & 0b100000)
					rex |= 0b0100 ; // REX.R
				else
					rex |= 0b0001 ; // REX.B
			    }
		    }
		else
		    {
			if (((operand1 < 0x60) && (operand1 & 8))
			 || ((operand2 < 0x60) && (operand2 & 8)))
				rex |= 0b0001 ; // REX.B
		    }

		if (index || base)
			rex &= 0b11111100 ; // zero REX.B and REX.X
		if (index & 8)
			rex |= 0b0010 ; // REX.X
		if (base & 8)
			rex |= 0b0001 ; // REX.B

		if (((operand1 >= 0x78) && (operand1 <= 0x7B)) ||
		    ((operand2 >= 0x78) && (operand2 <= 0x7B)) ||
		    (rex != 0x40))
		    {
			if (((operand1 >= 0x04) && (operand1 <= 0x07)) ||
			    ((operand2 >= 0x04) && (operand2 <= 0x07)))
				error (16, "Invalid register combination") ;
			poke (&rex, 1) ;
		    }
	    }

	// Short / near jumps: n.b. the assembler must not attempt to determine whether
	// a 'short' or 'near' jump is suitable because that needs more than two passes!
	if ((flags & 0b111000000) == 0b100000000)
	    {
		if (opcode1 == 0xEB) // unconditional jump
		    {
			if ((override & 0b0100000) == 0) // not short
			    {
				flags |= 0b001000000 ;
				opcode1 -= 2 ;
			    }
		    }
		else if (opcode1 != 0xE8) // conditional jumps
		    {
			if ((override & 0b1000000) != 0) // near
			    {
				flags |= 0b001000010 ;
				opcode1 += 0x10 ;
			    }
		    }
 	    }

	// Output 0F prefix, if any:
	if ((flags & 0b000000010) && ((mnemonic < FPLO) || (mnemonic > FPHI)))
	    {
		unsigned char prefix = 0x0F ;
		poke (&prefix, 1) ;
	    }

	// Affect W and S bits according to operand and displacement:
	if (((operand1 >= 0x00) && (operand1 <= 0x0F)) || ((operand2 >= 0x00) && (operand2 <= 0x0F) && (opcode1 != 0xD3))
	 || ((operand1 >= 0x78) && (operand1 <= 0x7B)) || ((operand2 >= 0x78) && (operand2 <= 0x7B) && (opcode1 != 0xD3))
	 || (override & 0b00000001))
	    {
		if (flags & 4)  // W bit
			opcode1 &= 0xFE ;
	    }
	else if (flags & 8)
	    {
		if ((mnemonic < FPLO) || (mnemonic > FPHI)) // S bit
		    {
			if ((immediate < -128) || (immediate > 127) || (override & 1))
				opcode1 &= 0xFD ;
			else
			    {
				opcode1 |= 0x02 ;
				flags = (flags & 0b000111110) | 0b110000000 ;
			    }
		    }
		else  // MF bits
		    {
			if (override & 0b00001010)
				opcode1 |= 0x04 ;
			else
				opcode1 &= 0xFB ;
		    }
	    }

	// Reset W bit if BYTE override:
	if (flags & 4)
		if (((operand1 >= 0x40) && (operand1 <= 0x4F) && (operand2 >= 0x7D))
		 || ((operand2 >= 0x40) && (operand2 <= 0x4F) && (operand1 >= 0x7D)))
		    {
			if ((override & 0b00011111) == 0)
				error (2, "Size needed") ; // 'Size needed'
			if (override & 1)
				opcode1 &= 0xFE ;
		    }

	// Special treatment for FILD, FISTP:
	if (((mnemonic == FILD) || (mnemonic == FISTP)) && (override & 0b00001000))
		opcode2 |= 5 << 3 ;

	// Special treatment for FLD, FSTP:
	if (((mnemonic == FLD) || (mnemonic == FSTP)) && (override & 0b00010000))
	    {
		opcode1 |= 2 ;
		opcode2 |= 5 << 3 ;
	    }

	// Special treatment for shifts:
	if (((opcode1 == 0xC0) || (opcode1 == 0xC1)) && (immediate == 1))
	    {
		opcode1 += 0x10 ;
		flags &= 0b000111110 ;
		operand2 = 0 ;
	    }
	if (((opcode1 == 0xD2) || (opcode1 == 0xD3)) && (operand2 == 1))
		operand2 = 0 ;

	// Construct ModR/M byte (if any):
	if (flags & 0b10000)
	    {
		if (opcode2 < 0x80)
		    {
			modrm = opcode2 ; // use as skeleton
			opcode2 = 0 ;
		    }

		// Write Mod field:
		if ((operand1 == 0x7C) || (operand2 == 0x7C))
		    {
			modrm &= 0x3F ;
			base = 0x85 ;
		    }
		else if (((operand1 < 0x40) || (operand1 >= 0x50))
		      && ((operand2 < 0x40) || (operand2 >= 0x50)))
			modrm |= 0xC0 ;
		else
		    {
			modrm &= 0x3F ;
			if ((displacement < -128) || (displacement > 127))
				modrm |= 0x80 ;
			else if ((displacement) || (base == 0x85) || (base == 0x8D))
				modrm |= 0x40 ;
			if ((base == 0) && (index != 0))
				modrm &= 0x3F ;
			if ((base == 0) && ((operand1 == 0x45) || (operand2 == 0x45)))
				modrm &= 0x3F ;
		    }

		// Write R/M and/or reg fields:
		if (flags & 0b100000)
		    {  // reg from operand2, R/M from operand1
			if ((operand2 <= 0x7C) && !(modrm & 0b00111000))
			    {
				modrm |= ((operand2 & 7) << 3) ; // reg
				if (operand2 >= 0x78)
					modrm |= 4 << 3 ;
			    }
			if (operand1 <= 0x7C)
			    {
				modrm |= (operand1 & 7) ;        // R/M
				if (operand1 >= 0x78)
					modrm |= 4 ;
			    }
		    }
		else 
		    {  // reg from operand1, R/M from operand2
			if ((operand1 <= 0x7C) && !(modrm & 0b00111000))
			    {
				modrm |= ((operand1 & 7) << 3) ; // reg
				if (operand1 >= 0x78)
					modrm |= 4 << 3 ;
			    }
			if (operand2 <= 0x7C)
			    {
				modrm |= (operand2 & 7) ;        // R/M
				if (operand2 >= 0x78)
					modrm |= 4 ;
			    }
		    }

		// Override R/M field if SIB:
		if (index != 0)
			modrm = (modrm & 0b11111000) | 0b00000100 ;
	    }

	// Output opcode and ModR/M byte(s) (if any):
	if ((opcode1 != 0xFF) || (opcode2 != 0xFF))
	    {
		if ((flags & 0b110000) == 0b100000)
		    {
			opcode1 |= (operand1 & 7) ;
			if (operand1 >= 0x78)
				opcode1 |= 4 ;
		    }
		poke (&opcode1, 1) ;

		if (opcode2 != 0)
			poke (&opcode2, 1) ;

		if (flags & 0b10000)
			poke (&modrm, 1) ;
	    }

	// Output SIB byte (if any):
	if (((modrm & 0b11000000) != 0b11000000) && ((modrm & 0b00000111) == 0b00000100))
	    {
		if (index == 0x84)
			error (16, "Index cannot be RSP") ;
		unsigned char sib = (scale << 6) | ((index & 7) << 3) | (base & 7) ;
		if ((displacement == 0) && (base == 0))
			sib |= 5 ;
		if (index == 0)
			sib |= 0b00100000 ;
		poke (&sib, 1) ;
		if (sib == 0b00100101)
			poke (&displacement, 4) ; // absolute address
	    }

	// Output displacement (if any) depends ModR/M
	if ((operand1 == 0x7E) && (operand2 == 0x7E))
		poke (&displacement, 2) ; // for ENTER instruction
	else if ((modrm & 0b11000000) == 0b01000000)
		poke (&displacement, 1) ; // disp8
	else if ((modrm & 0b11000000) == 0b10000000)
		poke (&displacement, 4) ; // disp32
	else if ((index != 0) && (base == 0))
		poke (&displacement, 4) ; // disp32
	else if ((base == 0) && ((operand1 == 0x45) || (operand2 == 0x45)))
		pdisp = poke (&displacement, 4) ; // disp32

	// Output immediate data (if any):
	switch (flags & 0b111000000)
	    {
		case 0b000000000: // no data
			if ((operand1 == 0x7E) || (operand2 == 0x7E) || (operand3 == 0x7E))
				error (16, NULL) ; // ' Syntax error'
			break ;

		case 0b001000000: // 32-bit immediate data
			if (override & 0b10000000)
				error (6, NULL) ; // 'Type mismatch'
			poke (&immediate, 4) ;
			break ;

		case 0b010000000: // 16-bit immediate data
			if (override & 0b10000000)
				error (6, NULL) ; // 'Type mismatch'
			poke (&immediate, 2) ;
			break ;

		case 0b011000000: // 64-bit immediate data
			if (override & 0b10000000)
			    {
				immediate = (long long) function (accs, immediate) ;
				if (immediate == 0)
					error (51, NULL) ; // 'No such system call'
			    }
			poke (&immediate, 8) ;
			break ;

		case 0b100000000: // 8-bit PC-relative offset
			if (override & 0b10000000)
				error (6, NULL) ; // 'Type mismatch'
			immediate -= (long long) PC + 1 ;
			if (((immediate < -128) || (immediate > 127)) && ((liston & BIT5) != 0))
				error (1, NULL) ; // 'Jump out of range' 
			poke (&immediate, 1) ;
			break ;

		case 0b101000000: // 32-bit PC-relative offset
			if (override & 0b10000000)
				error (6, NULL) ; // 'Type mismatch'
			immediate -= (long long) PC + 4 ;
			if ((immediate < (long long) INT_MIN) || (immediate > (long long) INT_MAX))
				error (1, NULL) ; // 'Jump out of range' 
			poke (&immediate, 4) ;
			break ;

		case 0b110000000: // 8-bit immediate data or string
			if (override & 0b10000000)
				poke ((void *)accs, immediate) ;
			else
				poke (&immediate, 1) ;
			break ;

		case 0b111000000: // Variable-length data depending on promotions etc.
			if ((operand1 <= 0x0F) || (operand2 <= 0x0F) || (override & 0b00000001)
			 || ((operand1 >= 0x78) && (operand1 <= 0x7B))
			 || ((operand2 >= 0x78) && (operand2 <= 0x7B)))
				poke (&immediate, 1) ;
			else if (((operand1 >= 0x10) && (operand1 <= 0x1F)) || ((operand2 >= 0x10) && (operand2 <= 0x1F))
			  || (override & 0b00000010))
				poke (&immediate, 2) ;
			else if ((((operand1 >= 0x20) && (operand1 <= 0x2F)) || ((operand2 >= 0x20) && (operand2 <= 0x2F))
			  || (override & 0b00001000)) && ((opcode1 & 0xF8) == 0xB8))
			    {
				if (override & 0b10000000)
				    {
					immediate = (long long) function (accs, immediate) ;
					if (immediate == 0)
						error (51, NULL) ; // 'No such system call'
				    }
				poke (&immediate, 8) ;
			    }
			else
				poke (&immediate, 4) ;

	    }
	return pdisp ;
}

void assemble (void)
{
	signed char al ; 
	signed char *oldesi = esi ; 
	void *oldpc = PC ; 

	while (1)
	    {
		int mnemonic ;
		int operand1 ;
		int operand2 ;
		int operand3 ;
		int override = 0 ;
		long long sibinfo = 0 ;
		long long immediate = 0 ;
		int displacement = 0 ;
		char *pdisp ;

		al = nxt () ;
		esi++ ;

		switch (al) 
		    {
			case 0:
				esi-- ;
				liston = (liston & 0x0F) | 0x30 ; 
				return ;

			case ']':
				liston = (liston & 0x0F) | 0x30 ; 
				return ;

			case 0x0D:
			case ':':
				if (liston & BIT4)
				    {
					void *p ;
					int n = PC - oldpc ;
					if (liston & BIT6)
						p = OC - n ;
					else
						p = PC - n ;

					do
					    {
						char *t = accs ;
#if (defined (_WIN32)) && (__GNUC__ < 9)
						t += sprintf (t, "%016I64X ", (long long) (size_t) oldpc) ;
#else
						t += sprintf (t, "%016llX ", (long long) oldpc) ;
#endif
						while (n)
						    {
							t += sprintf (t, "%02X ", *(unsigned char *)p) ;
							oldpc++ ;
							p++ ;
							n-- ;
							if (strlen (accs) > 35)
								break ;
						    }
						text (accs) ;

						if (*oldesi == '.')
						    {
							tabit (38) ;
							do	
							    {
								token (*oldesi++ ) ;
							    }
							while (range0(*oldesi)) ;
							token (' ') ;
							while (*oldesi == ' ') oldesi++ ;
						    }
						tabit (49) ;
						while (oldesi < (esi-1)) 
							token (*oldesi++) ;
						crlf () ;
					    }
					while (n) ;
				    }
				if (*(esi-1) == 0x0D)
					newlin () ;
				nxt () ;
				oldesi = esi ;
				oldpc = PC ;
				break ;

			case ';':
			case TREM:
				while ((*esi != 0x0D) && (*esi != ':')) esi++ ;
				break ;

			case '.':
				{
				VAR v ;
				unsigned char type ;
				void *ptr = getput (&type) ;
				if (ptr == NULL)
					error (16, NULL) ; // 'Syntax error'
				if (type >= 128)
					error (6, NULL) ; // 'Type mismatch'
				if ((liston & BIT5) == 0)
				    {
					v = loadn (ptr, type) ;
					if (v.i.n)
						error (3, NULL) ; // 'Multiple label'
				    }
				v.i.t = 0 ;
				v.i.n = (intptr_t) PC ;
				storen (v, ptr, type) ;
				}
				break ;

			default:
				operand1 = 0x7F ;
				operand2 = 0x7F ;
				operand3 = 0x7F ; 
				esi-- ;
				mnemonic = schop (mnemonics, sizeof(mnemonics) / sizeof(mnemonics[0])) ;

				switch (mnemonic)
				    {
					case OPT:
						liston = (liston & 0x0F) | (expri () << 4) ;
						break ;

					case -1:
						mnemonic = scan (specials, sizeof(specials) / sizeof(specials[0])) ;
						if (mnemonic == -1)
							error (16, "Bad mnemonic") ; // 'Syntax error'
						immediate = mnemonic % 8 ;
						mnemonic = special[mnemonic / 8] ;
						operand3 = 0x7E ;

					default:
						operand1 = operand (&displacement, &immediate, &override, &sibinfo) ;
						if (nxt () == ',') esi++ ;
						if (operand1 == 0x7E)
							displacement = immediate ; // for ENTER
						operand2 = operand (&displacement, &immediate, &override, &sibinfo) ;
						if (nxt () == ',') esi++ ;
						if (operand3 == 0x7F)
						operand3 = operand (&displacement, &immediate, &override, &sibinfo) ;

					case LOCK:
					case REPLO ... REPHI:
						pdisp = encode (mnemonic, operand1, operand2, operand3, 
								override, displacement, sibinfo, immediate) ;
						if (pdisp != NULL)
							*(int *)pdisp -= PC - oldpc ;
				    }
		    }
	    } 

}
