/**************************************************************************************
 * Copyright (c) 2018-2022 ["Peking University Shenzhen Graduate School",
 *   "Peng Cheng Laboratory", and "Guangdong Bohua UHD Innovation Corporation"]
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the organizations (Peking University Shenzhen Graduate School,
 *    Peng Cheng Laboratory and Guangdong Bohua UHD Innovation Corporation) nor the
 *    names of its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For more information, contact us at rgwang@pkusz.edu.cn.
 **************************************************************************************/

#include "com_type.h"
#include "com_sys.h"
#include <math.h>

tab_s32 g_tbl_ai_tscpm_div[64] =
{
    65536, 32768, 21845, 16384, 13107, 10922, 9362, 8192, 7281, 6553, 5957, 5461, 5041, 4681, 4369, 4096, 3855, 3640,
    3449,  3276,  3120,  2978,  2849,  2730,  2621, 2520, 2427, 2340, 2259, 2184, 2114, 2048, 1985, 1927, 1872, 1820,
    1771,  1724,  1680,  1638,  1598,  1560,  1524, 1489, 1456, 1424, 1394, 1365, 1337, 1310, 1285, 1260, 1236, 1213,
    1191,  1170,  1149,  1129,  1110,  1092,  1074, 1057, 1040, 1024,
};

tab_s8 g_tbl_log2[257] =
{
    /* 0, 1 */
    -1, -1,
        /* 2, 3 */
        1, -1,
        /* 4 ~ 7 */
        2, -1, -1, -1,
        /* 8 ~ 15 */
        3, -1, -1, -1, -1, -1, -1, -1,
        /* 16 ~ 31 */
        4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        /* 31 ~ 63 */
        5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        /* 64 ~ 127 */
        6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        /* 128 ~ 255 */
        7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        /* 256 */
        8
    };

ALIGNED_32(s8* g_tbl_itrans[NUM_TRANS_TYPE][ITRANS_SIZE_TYPES]);
ALIGNED_32(s8 g_tbl_itrans_coeffs[ITRANS_COEFFS_SIZE]);

tab_s8 g_tbl_itrans_c4[4][4] = {
    { 34,    77,    79,     55, },
    { 58,    69,   -33,    -84, },
    { 72,    -7,   -75,     73, },
    { 81,   -75,    58,    -28, }
};

tab_s8 g_tbl_itrans_c8[4][4] = {
    { 123,   -32,    14,    8, },
    { -35,  -120,    25,    13, },
    { -8,    30,   123,    19, },
    { -3,    10,    -22,   126, },
};


tab_s8 g_tbl_mc_coeff_luma_hp[16][8] =
{
    {  0, 0,   0, 64,  0,   0,  0,  0 },
    {  0, 1,  -3, 63,  4,  -2,  1,  0 },
    { -1, 2,  -5, 62,  8,  -3,  1,  0 },
    { -1, 3,  -8, 60, 13,  -4,  1,  0 },
    { -1, 4, -10, 58, 17,  -5,  1,  0 },
    { -1, 4, -11, 52, 26,  -8,  3, -1 },
    { -1, 3,  -9, 47, 31, -10,  4, -1 },
    { -1, 4, -11, 45, 34, -10,  4, -1 },
    { -1, 4, -11, 40, 40, -11,  4, -1 },
    { -1, 4, -10, 34, 45, -11,  4, -1 },
    { -1, 4, -10, 31, 47,  -9,  3, -1 },
    { -1, 3,  -8, 26, 52, -11,  4, -1 },
    {  0, 1,  -5, 17, 58, -10,  4, -1 },
    {  0, 1,  -4, 13, 60,  -8,  3, -1 },
    {  0, 1,  -3,  8, 62,  -5,  2, -1 },
    {  0, 1,  -2,  4, 63,  -3,  1,  0 } 
};

tab_s8 g_tbl_mc_coeff_luma[4][8] =
{
    { 0,  0,  0, 64,  0,  0,  0,  0},
    { -1, 4, -10, 57, 19,  -7, 3, -1},
    { -1, 4, -11, 40, 40, -11, 4, -1},
    { -1, 3,  -7, 19, 57, -10, 4, -1}
};

tab_s8 g_tbl_mc_coeff_chroma_hp[32][4] =
{
    { 0, 64, 0, 0 },
    { -1, 63,  2,  0 },
    { -2, 62,  4,  0 },
    { -2, 60,  7,  -1 },
    { -2, 58, 10, -2 },
    { -3, 57, 12, -2 },
    { -4, 56, 14, -2 },
    { -4, 55, 15, -2 },
    { -4, 54, 16, -2 },
    { -5, 53, 18, -2 },
    { -6, 52, 20, -2 },
    { -6, 49, 24, -3 },
    { -6, 46, 28, -4 },
    { -5, 44, 29, -4 },
    { -4, 42, 30, -4 },
    { -4, 39, 33, -4 },
    { -4, 36, 36, -4 },
    { -4, 33, 39, -4 },
    { -4, 30, 42, -4 },
    { -4, 29, 44, -5 },
    { -4, 28, 46, -6 },
    { -3, 24, 49, -6 },
    { -2, 20, 52, -6 },
    { -2, 18, 53, -5 },
    { -2, 16, 54, -4 },
    { -2, 15, 55, -4 },
    { -2, 14, 56, -4 },
    { -2, 12, 57, -3 },
    { -2, 10, 58, -2 },
    { -1,  7, 60, -2 },
    { 0,  4, 62, -2 },
    { 0,  2, 63, -1 },
};

tab_s8 g_tbl_mc_coeff_chroma[8][4] =
{
    { 0, 64, 0, 0 },
    { -4, 62, 6, 0 },
    { -6, 56, 15, -1 },
    { -5, 47, 25, -3 },
    { -4, 36, 36, -4 },
    { -3, 25, 47, -5 },
    { -1, 15, 56, -6 },
    { 0, 6, 62, -4 }
};

tab_u8 g_tbl_qp_chroma_adjust[64] =
{
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9,
    10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
    20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
    30, 31, 32, 33, 34, 35, 36, 37, 38, 39,
    40, 41, 42, 42, 43, 43, 44, 44, 45, 45,
    46, 46, 47, 47, 48, 48, 48, 49, 49, 49,
    50, 50, 50, 51
};

void com_dct_coef_init()
{
    int i, blk_size = 2;
    const double PI = 3.14159265358979323846;
    s8 *coeffs = g_tbl_itrans_coeffs;

    for (i = 0; i < 6; i++) {
        const double s = sqrt((double)blk_size) * 32;

        for (int k = 0; k < blk_size; k++) {
            for (int n = 0; n < blk_size; n++) {
                double w0, v;
                
                // DCT2
                w0 = k == 0 ? sqrt(0.5) : 1;
                v = cos(PI * (n + 0.5) * k / blk_size) * w0 * sqrt(2.0 / blk_size);
                coeffs[DCT2 * blk_size * blk_size + k * blk_size + n] = (s8)(s * v + (v > 0 ? 0.5 : -0.5));
                
                // DCT8
                v = cos(PI*(k + 0.5)*(n + 0.5) / (blk_size + 0.5)) * sqrt(2.0 / (blk_size + 0.5));
                coeffs[DCT8*blk_size*blk_size + k * blk_size + n] = (s8)(s * v + (v > 0 ? 0.5 : -0.5));

                // DST7
                v = sin(PI*(k + 0.5)*(n + 1) / (blk_size + 0.5)) * sqrt(2.0 / (blk_size + 0.5));
                coeffs[DST7*blk_size*blk_size + k * blk_size + n] = (s8)(s * v + (v > 0 ? 0.5 : -0.5));

            }
        }
        
        g_tbl_itrans[DCT2][i] = coeffs;
        g_tbl_itrans[DCT8][i] = coeffs + DCT8*blk_size*blk_size;
        g_tbl_itrans[DST7][i] = coeffs + DST7*blk_size*blk_size;

        coeffs += NUM_TRANS_TYPE * blk_size*blk_size;

        blk_size <<= 1;
    }
}

tab_u32 g_tbl_wq_default_param[2][6] =
{
    { 67, 71, 71, 80, 80, 106 },
    { 64, 49, 53, 58, 58,  64 }
};

tab_u8 g_tbl_wq_default_matrix_4x4[16] =
{
    64, 64, 64, 68, 64, 64, 68, 72, 64, 68, 76, 80, 72, 76, 84, 96
};

tab_u8 g_tbl_wq_default_matrix_8x8[64] =
{
    64,  64,  64,  64,  68,  68,  72,  76,
    64,  64,  64,  68,  72,  76,  84,  92,
    64,  64,  68,  72,  76,  80,  88, 100,
    64,  68,  72,  80,  84,  92, 100, 112,
    68,  72,  80,  84,  92, 104, 112, 128,
    76,  80,  84,  92, 104, 116, 132, 152,
    96, 100, 104, 116, 124, 140, 164, 188,
    104, 108, 116, 128, 152, 172, 192, 216
};

tab_u8 g_tbl_wq_model_8x8[4][64] =
{
    {
        // Mode 0
        0, 0, 0, 4, 4, 4, 5, 5,
        0, 0, 3, 3, 3, 3, 5, 5,
        0, 3, 2, 2, 1, 1, 5, 5,
        4, 3, 2, 2, 1, 5, 5, 5,
        4, 3, 1, 1, 5, 5, 5, 5,
        4, 3, 1, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5
    },{
        // Mode 1
        0, 0, 0, 4, 4, 4, 5, 5,
        0, 0, 4, 4, 4, 4, 5, 5,
        0, 3, 2, 2, 2, 1, 5, 5,
        3, 3, 2, 2, 1, 5, 5, 5,
        3, 3, 2, 1, 5, 5, 5, 5,
        3, 3, 1, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5
    },{
        // Mode 2
        0, 0, 0, 4, 4, 3, 5, 5,
        0, 0, 4, 4, 3, 2, 5, 5,
        0, 4, 4, 3, 2, 1, 5, 5,
        4, 4, 3, 2, 1, 5, 5, 5,
        4, 3, 2, 1, 5, 5, 5, 5,
        3, 2, 1, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5
    },{
        // Mode 3
        0, 0, 0, 3, 2, 1, 5, 5,
        0, 0, 4, 3, 2, 1, 5, 5,
        0, 4, 4, 3, 2, 1, 5, 5,
        3, 3, 3, 3, 2, 5, 5, 5,
        2, 2, 2, 2, 5, 5, 5, 5,
        1, 1, 1, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5,
        5, 5, 5, 5, 5, 5, 5, 5
    }
};

tab_u8 g_tbl_wq_model_4x4[4][16] =
{
    {
        // Mode 0
        0, 4, 3, 5,
        4, 2, 1, 5,
        3, 1, 1, 5,
        5, 5, 5, 5
    },{
        // Mode 1
        0, 4, 4, 5,
        3, 2, 2, 5,
        3, 2, 1, 5,
        5, 5, 5, 5
    },{

        // Mode 2
        0, 4, 3, 5,
        4, 3, 2, 5,
        3, 2, 1, 5,
        5, 5, 5, 5
    },{
        // Mode 3
        0, 3, 1, 5,
        3, 4, 2, 5,
        1, 2, 2, 5,
        5, 5, 5, 5
    }
};

tab_u8 g_tbl_part_num[8] = { 
    1, // SIZE_2Nx2N 
    2, // SIZE_2NxnU 
    2, // SIZE_2NxnD 
    2, // SIZE_nLx2N 
    2, // SIZE_nRx2N 
    4, // SIZE_NxN   
    4, // SIZE_2NxhN 
    4  // SIZE_hNx2N 
};

const com_part_size_t g_tbl_tb_part[8] = {
    SIZE_2Nx2N, // SIZE_2Nx2N 
    SIZE_2NxhN, // SIZE_2NxnU 
    SIZE_2NxhN, // SIZE_2NxnD 
    SIZE_hNx2N, // SIZE_nLx2N 
    SIZE_hNx2N, // SIZE_nRx2N 
    SIZE_NxN,   // SIZE_NxN   
    SIZE_2NxhN, // SIZE_2NxhN 
    SIZE_hNx2N  // SIZE_hNx2N 
};

tab_u16 g_tbl_scan_blkpos[MAX_CU_LOG2][MAX_CU_LOG2] = {
    0, 4, 12, 28, 60, 124, 252, 
    508, 516, 532, 564, 628, 756, 1012, 
    1524, 1540, 1572, 1636, 1764, 2020, 2532, 
    3556, 3588, 3652, 3780, 4036, 4548, 5572, 
    7620, 7684, 7812, 8068, 8580, 9604, 11652, 
    15748, 15876, 16132, 16644, 17668, 19716, 23812, 
    32004, 32260, 32772, 33796, 35844, 39940, 48132
};

tab_u16 g_tbl_scan[64516] = {
    // 2 x 2:
    0, 1, 2, 3,
    // 2 x 4:
    0, 1, 4, 5, 2, 3, 6, 7,
    // 2 x 8:
    0, 1, 8, 9, 2, 3, 10, 11, 4, 5, 12, 13, 6, 7, 14, 15,
    // 2 x 16:
    0, 1, 16, 17, 2, 3, 18, 19, 4, 5, 20, 21, 6, 7, 22, 23, 8, 9, 24, 25, 10, 11, 26, 27, 12, 13, 28, 29, 14, 15, 30, 31,
    // 2 x 32:
    0, 1, 32, 33, 2, 3, 34, 35, 4, 5, 36, 37, 6, 7, 38, 39, 8, 9, 40, 41, 10, 11, 42, 43, 12, 13, 44, 45, 14, 15, 46, 47, 16, 17, 48, 49, 18, 19, 50, 51, 20, 21, 52, 53, 22, 23, 54, 55, 24, 25, 56, 57, 26, 27, 58, 59, 28, 29, 60, 61, 30, 31, 62, 63,
    // 2 x 64:
    0, 1, 64, 65, 2, 3, 66, 67, 4, 5, 68, 69, 6, 7, 70, 71, 8, 9, 72, 73, 10, 11, 74, 75, 12, 13, 76, 77, 14, 15, 78, 79, 16, 17, 80, 81, 18, 19, 82, 83, 20, 21, 84, 85, 22, 23, 86, 87, 24, 25, 88, 89, 26, 27, 90, 91, 28, 29, 92, 93, 30, 31, 94, 95, 32, 33, 96, 97, 34, 35, 98, 99, 36, 37, 100, 101, 38, 39, 102, 103, 40, 41, 104, 105, 42, 43, 106, 107, 44, 45, 108, 109, 46, 47, 110, 111, 48, 49, 112, 113, 50, 51, 114, 115, 52, 53, 116, 117, 54, 55, 118, 119, 56, 57, 120, 121, 58, 59, 122, 123, 60, 61, 124, 125, 62, 63, 126, 127,

    // 2 x 128:
    0, 1, 128, 129, 2, 3, 130, 131, 4, 5, 132, 133, 6, 7, 134, 135, 8, 9, 136, 137, 10, 11, 138, 139, 12, 13, 140, 141, 14, 15, 142, 143, 16, 17, 144, 145, 18, 19, 146, 147, 20, 21, 148, 149, 22, 23, 150, 151, 24, 25, 152, 153, 26, 27, 154, 155, 28, 29, 156, 157, 30, 31, 158, 159, 32, 33, 160, 161, 34, 35, 162, 163, 36, 37, 164, 165, 38, 39, 166, 167, 40, 41, 168, 169, 42, 43, 170, 171, 44, 45, 172, 173, 46, 47, 174, 175, 48, 49, 176, 177, 50, 51, 178, 179, 52, 53, 180, 181, 54, 55, 182, 183, 56, 57, 184, 185, 58, 59, 186, 187, 60, 61, 188, 189, 62, 63, 190, 191,
    64, 65, 192, 193, 66, 67, 194, 195, 68, 69, 196, 197, 70, 71, 198, 199, 72, 73, 200, 201, 74, 75, 202, 203, 76, 77, 204, 205, 78, 79, 206, 207, 80, 81, 208, 209, 82, 83, 210, 211, 84, 85, 212, 213, 86, 87, 214, 215, 88, 89, 216, 217, 90, 91, 218, 219, 92, 93, 220, 221, 94, 95, 222, 223, 96, 97, 224, 225, 98, 99, 226, 227, 100, 101, 228, 229, 102, 103, 230, 231, 104, 105, 232, 233, 106, 107, 234, 235, 108, 109, 236, 237, 110, 111, 238, 239, 112, 113, 240, 241, 114, 115, 242, 243, 116, 117, 244, 245, 118, 119, 246, 247, 120, 121, 248, 249, 122, 123, 250, 251, 124, 125, 252, 253, 126, 127, 254, 255,

    // 4 x 2:
    0, 1, 2, 4, 3, 5, 6, 7,
    // 4 x 4:
    0, 1, 4, 8, 5, 2, 3, 6, 9, 12, 13, 10, 7, 11, 14, 15,
    // 4 x 8:
    0, 1, 8, 16, 9, 2, 3, 10, 17, 24, 25, 18, 11, 4, 5, 12, 19, 26, 27, 20, 13, 6, 7, 14, 21, 28, 29, 22, 15, 23, 30, 31,
    // 4 x 16:
    0, 1, 16, 32, 17, 2, 3, 18, 33, 48, 49, 34, 19, 4, 5, 20, 35, 50, 51, 36, 21, 6, 7, 22, 37, 52, 53, 38, 23, 8, 9, 24, 39, 54, 55, 40, 25, 10, 11, 26, 41, 56, 57, 42, 27, 12, 13, 28, 43, 58, 59, 44, 29, 14, 15, 30, 45, 60, 61, 46, 31, 47, 62, 63,
    // 4 x 32:
    0, 1, 32, 64, 33, 2, 3, 34, 65, 96, 97, 66, 35, 4, 5, 36, 67, 98, 99, 68, 37, 6, 7, 38, 69, 100, 101, 70, 39, 8, 9, 40, 71, 102, 103, 72, 41, 10, 11, 42, 73, 104, 105, 74, 43, 12, 13, 44, 75, 106, 107, 76, 45, 14, 15, 46, 77, 108, 109, 78, 47, 16, 17, 48, 79, 110, 111, 80, 49, 18, 19, 50, 81, 112, 113, 82, 51, 20, 21, 52, 83, 114, 115, 84, 53, 22, 23, 54, 85, 116, 117, 86, 55, 24, 25, 56, 87, 118, 119, 88, 57, 26, 27, 58, 89, 120, 121, 90, 59, 28, 29, 60, 91, 122, 123, 92, 61, 30, 31, 62, 93, 124, 125, 94, 63, 95, 126, 127,

    // 4 x 64:
    0, 1, 64, 128, 65, 2, 3, 66, 129, 192, 193, 130, 67, 4, 5, 68, 131, 194, 195, 132, 69, 6, 7, 70, 133, 196, 197, 134, 71, 8, 9, 72, 135, 198, 199, 136, 73, 10, 11, 74, 137, 200, 201, 138, 75, 12, 13, 76, 139, 202, 203, 140, 77, 14, 15, 78, 141, 204, 205, 142, 79, 16, 17, 80, 143, 206, 207, 144, 81, 18, 19, 82, 145, 208, 209, 146, 83, 20, 21, 84, 147, 210, 211, 148, 85, 22, 23, 86, 149, 212, 213, 150, 87, 24, 25, 88, 151, 214, 215, 152, 89, 26, 27, 90, 153, 216, 217, 154, 91, 28, 29, 92, 155, 218, 219, 156, 93, 30, 31, 94, 157, 220, 221, 158, 95, 32, 33, 96,
    159, 222, 223, 160, 97, 34, 35, 98, 161, 224, 225, 162, 99, 36, 37, 100, 163, 226, 227, 164, 101, 38, 39, 102, 165, 228, 229, 166, 103, 40, 41, 104, 167, 230, 231, 168, 105, 42, 43, 106, 169, 232, 233, 170, 107, 44, 45, 108, 171, 234, 235, 172, 109, 46, 47, 110, 173, 236, 237, 174, 111, 48, 49, 112, 175, 238, 239, 176, 113, 50, 51, 114, 177, 240, 241, 178, 115, 52, 53, 116, 179, 242, 243, 180, 117, 54, 55, 118, 181, 244, 245, 182, 119, 56, 57, 120, 183, 246, 247, 184, 121, 58, 59, 122, 185, 248, 249, 186, 123, 60, 61, 124, 187, 250, 251, 188, 125, 62, 63, 126, 189, 252, 253, 190, 127, 191, 254, 255,

    // 4 x 128:
    0, 1, 128, 256, 129, 2, 3, 130, 257, 384, 385, 258, 131, 4, 5, 132, 259, 386, 387, 260, 133, 6, 7, 134, 261, 388, 389, 262, 135, 8, 9, 136, 263, 390, 391, 264, 137, 10, 11, 138, 265, 392, 393, 266, 139, 12, 13, 140, 267, 394, 395, 268, 141, 14, 15, 142, 269, 396, 397, 270, 143, 16, 17, 144, 271, 398, 399, 272, 145, 18, 19, 146, 273, 400, 401, 274, 147, 20, 21, 148, 275, 402, 403, 276, 149, 22, 23, 150, 277, 404, 405, 278, 151, 24, 25, 152, 279, 406, 407, 280, 153, 26, 27, 154, 281, 408, 409, 282, 155, 28, 29, 156, 283, 410, 411, 284, 157, 30, 31, 158, 285, 412, 413, 286, 159, 32, 33, 160,
    287, 414, 415, 288, 161, 34, 35, 162, 289, 416, 417, 290, 163, 36, 37, 164, 291, 418, 419, 292, 165, 38, 39, 166, 293, 420, 421, 294, 167, 40, 41, 168, 295, 422, 423, 296, 169, 42, 43, 170, 297, 424, 425, 298, 171, 44, 45, 172, 299, 426, 427, 300, 173, 46, 47, 174, 301, 428, 429, 302, 175, 48, 49, 176, 303, 430, 431, 304, 177, 50, 51, 178, 305, 432, 433, 306, 179, 52, 53, 180, 307, 434, 435, 308, 181, 54, 55, 182, 309, 436, 437, 310, 183, 56, 57, 184, 311, 438, 439, 312, 185, 58, 59, 186, 313, 440, 441, 314, 187, 60, 61, 188, 315, 442, 443, 316, 189, 62, 63, 190, 317, 444, 445, 318, 191, 64, 65, 192,
    319, 446, 447, 320, 193, 66, 67, 194, 321, 448, 449, 322, 195, 68, 69, 196, 323, 450, 451, 324, 197, 70, 71, 198, 325, 452, 453, 326, 199, 72, 73, 200, 327, 454, 455, 328, 201, 74, 75, 202, 329, 456, 457, 330, 203, 76, 77, 204, 331, 458, 459, 332, 205, 78, 79, 206, 333, 460, 461, 334, 207, 80, 81, 208, 335, 462, 463, 336, 209, 82, 83, 210, 337, 464, 465, 338, 211, 84, 85, 212, 339, 466, 467, 340, 213, 86, 87, 214, 341, 468, 469, 342, 215, 88, 89, 216, 343, 470, 471, 344, 217, 90, 91, 218, 345, 472, 473, 346, 219, 92, 93, 220, 347, 474, 475, 348, 221, 94, 95, 222, 349, 476, 477, 350, 223, 96, 97, 224,
    351, 478, 479, 352, 225, 98, 99, 226, 353, 480, 481, 354, 227, 100, 101, 228, 355, 482, 483, 356, 229, 102, 103, 230, 357, 484, 485, 358, 231, 104, 105, 232, 359, 486, 487, 360, 233, 106, 107, 234, 361, 488, 489, 362, 235, 108, 109, 236, 363, 490, 491, 364, 237, 110, 111, 238, 365, 492, 493, 366, 239, 112, 113, 240, 367, 494, 495, 368, 241, 114, 115, 242, 369, 496, 497, 370, 243, 116, 117, 244, 371, 498, 499, 372, 245, 118, 119, 246, 373, 500, 501, 374, 247, 120, 121, 248, 375, 502, 503, 376, 249, 122, 123, 250, 377, 504, 505, 378, 251, 124, 125, 252, 379, 506, 507, 380, 253, 126, 127, 254, 381, 508, 509, 382, 255, 383, 510, 511,

    // 8 x 2:
    0, 1, 2, 4, 3, 5, 6, 8, 7, 9, 10, 12, 11, 13, 14, 15,
    // 8 x 4:
    0, 1, 4, 8, 5, 2, 3, 6, 9, 12, 16, 13, 10, 7, 11, 14, 17, 20, 24, 21, 18, 15, 19, 22, 25, 28, 29, 26, 23, 27, 30, 31,
    // 8 x 8:
    0, 1, 8, 16, 9, 2, 3, 10, 17, 24, 32, 25, 18, 11, 4, 5, 12, 19, 26, 33, 40, 48, 41, 34, 27, 20, 13, 6, 7, 14, 21, 28, 35, 42, 49, 56, 57, 50, 43, 36, 29, 22, 15, 23, 30, 37, 44, 51, 58, 59, 52, 45, 38, 31, 39, 46, 53, 60, 61, 54, 47, 55, 62, 63,
    // 8 x 16:
    0, 1, 16, 32, 17, 2, 3, 18, 33, 48, 64, 49, 34, 19, 4, 5, 20, 35, 50, 65, 80, 96, 81, 66, 51, 36, 21, 6, 7, 22, 37, 52, 67, 82, 97, 112, 113, 98, 83, 68, 53, 38, 23, 8, 9, 24, 39, 54, 69, 84, 99, 114, 115, 100, 85, 70, 55, 40, 25, 10, 11, 26, 41, 56, 71, 86, 101, 116, 117, 102, 87, 72, 57, 42, 27, 12, 13, 28, 43, 58, 73, 88, 103, 118, 119, 104, 89, 74, 59, 44, 29, 14, 15, 30, 45, 60, 75, 90, 105, 120, 121, 106, 91, 76, 61, 46, 31, 47, 62, 77, 92, 107, 122, 123, 108, 93, 78, 63, 79, 94, 109, 124, 125, 110, 95, 111, 126, 127,

    // 8 x 32:
    0, 1, 32, 64, 33, 2, 3, 34, 65, 96, 128, 97, 66, 35, 4, 5, 36, 67, 98, 129, 160, 192, 161, 130, 99, 68, 37, 6, 7, 38, 69, 100, 131, 162, 193, 224, 225, 194, 163, 132, 101, 70, 39, 8, 9, 40, 71, 102, 133, 164, 195, 226, 227, 196, 165, 134, 103, 72, 41, 10, 11, 42, 73, 104, 135, 166, 197, 228, 229, 198, 167, 136, 105, 74, 43, 12, 13, 44, 75, 106, 137, 168, 199, 230, 231, 200, 169, 138, 107, 76, 45, 14, 15, 46, 77, 108, 139, 170, 201, 232, 233, 202, 171, 140, 109, 78, 47, 16, 17, 48, 79, 110, 141, 172, 203, 234, 235, 204, 173, 142, 111, 80, 49, 18, 19, 50, 81, 112,
    143, 174, 205, 236, 237, 206, 175, 144, 113, 82, 51, 20, 21, 52, 83, 114, 145, 176, 207, 238, 239, 208, 177, 146, 115, 84, 53, 22, 23, 54, 85, 116, 147, 178, 209, 240, 241, 210, 179, 148, 117, 86, 55, 24, 25, 56, 87, 118, 149, 180, 211, 242, 243, 212, 181, 150, 119, 88, 57, 26, 27, 58, 89, 120, 151, 182, 213, 244, 245, 214, 183, 152, 121, 90, 59, 28, 29, 60, 91, 122, 153, 184, 215, 246, 247, 216, 185, 154, 123, 92, 61, 30, 31, 62, 93, 124, 155, 186, 217, 248, 249, 218, 187, 156, 125, 94, 63, 95, 126, 157, 188, 219, 250, 251, 220, 189, 158, 127, 159, 190, 221, 252, 253, 222, 191, 223, 254, 255,

    // 8 x 64:
    0, 1, 64, 128, 65, 2, 3, 66, 129, 192, 256, 193, 130, 67, 4, 5, 68, 131, 194, 257, 320, 384, 321, 258, 195, 132, 69, 6, 7, 70, 133, 196, 259, 322, 385, 448, 449, 386, 323, 260, 197, 134, 71, 8, 9, 72, 135, 198, 261, 324, 387, 450, 451, 388, 325, 262, 199, 136, 73, 10, 11, 74, 137, 200, 263, 326, 389, 452, 453, 390, 327, 264, 201, 138, 75, 12, 13, 76, 139, 202, 265, 328, 391, 454, 455, 392, 329, 266, 203, 140, 77, 14, 15, 78, 141, 204, 267, 330, 393, 456, 457, 394, 331, 268, 205, 142, 79, 16, 17, 80, 143, 206, 269, 332, 395, 458, 459, 396, 333, 270, 207, 144, 81, 18, 19, 82, 145, 208,
    271, 334, 397, 460, 461, 398, 335, 272, 209, 146, 83, 20, 21, 84, 147, 210, 273, 336, 399, 462, 463, 400, 337, 274, 211, 148, 85, 22, 23, 86, 149, 212, 275, 338, 401, 464, 465, 402, 339, 276, 213, 150, 87, 24, 25, 88, 151, 214, 277, 340, 403, 466, 467, 404, 341, 278, 215, 152, 89, 26, 27, 90, 153, 216, 279, 342, 405, 468, 469, 406, 343, 280, 217, 154, 91, 28, 29, 92, 155, 218, 281, 344, 407, 470, 471, 408, 345, 282, 219, 156, 93, 30, 31, 94, 157, 220, 283, 346, 409, 472, 473, 410, 347, 284, 221, 158, 95, 32, 33, 96, 159, 222, 285, 348, 411, 474, 475, 412, 349, 286, 223, 160, 97, 34, 35, 98, 161, 224,
    287, 350, 413, 476, 477, 414, 351, 288, 225, 162, 99, 36, 37, 100, 163, 226, 289, 352, 415, 478, 479, 416, 353, 290, 227, 164, 101, 38, 39, 102, 165, 228, 291, 354, 417, 480, 481, 418, 355, 292, 229, 166, 103, 40, 41, 104, 167, 230, 293, 356, 419, 482, 483, 420, 357, 294, 231, 168, 105, 42, 43, 106, 169, 232, 295, 358, 421, 484, 485, 422, 359, 296, 233, 170, 107, 44, 45, 108, 171, 234, 297, 360, 423, 486, 487, 424, 361, 298, 235, 172, 109, 46, 47, 110, 173, 236, 299, 362, 425, 488, 489, 426, 363, 300, 237, 174, 111, 48, 49, 112, 175, 238, 301, 364, 427, 490, 491, 428, 365, 302, 239, 176, 113, 50, 51, 114, 177, 240,
    303, 366, 429, 492, 493, 430, 367, 304, 241, 178, 115, 52, 53, 116, 179, 242, 305, 368, 431, 494, 495, 432, 369, 306, 243, 180, 117, 54, 55, 118, 181, 244, 307, 370, 433, 496, 497, 434, 371, 308, 245, 182, 119, 56, 57, 120, 183, 246, 309, 372, 435, 498, 499, 436, 373, 310, 247, 184, 121, 58, 59, 122, 185, 248, 311, 374, 437, 500, 501, 438, 375, 312, 249, 186, 123, 60, 61, 124, 187, 250, 313, 376, 439, 502, 503, 440, 377, 314, 251, 188, 125, 62, 63, 126, 189, 252, 315, 378, 441, 504, 505, 442, 379, 316, 253, 190, 127, 191, 254, 317, 380, 443, 506, 507, 444, 381, 318, 255, 319, 382, 445, 508, 509, 446, 383, 447, 510, 511,

    // 8 x 128:
    0, 1, 128, 256, 129, 2, 3, 130, 257, 384, 512, 385, 258, 131, 4, 5, 132, 259, 386, 513, 640, 768, 641, 514, 387, 260, 133, 6, 7, 134, 261, 388, 515, 642, 769, 896, 897, 770, 643, 516, 389, 262, 135, 8, 9, 136, 263, 390, 517, 644, 771, 898, 899, 772, 645, 518, 391, 264, 137, 10, 11, 138, 265, 392, 519, 646, 773, 900, 901, 774, 647, 520, 393, 266, 139, 12, 13, 140, 267, 394, 521, 648, 775, 902, 903, 776, 649, 522, 395, 268, 141, 14, 15, 142, 269, 396, 523, 650, 777, 904, 905, 778, 651, 524, 397, 270, 143, 16, 17, 144, 271, 398, 525, 652, 779, 906, 907, 780, 653, 526, 399, 272, 145, 18, 19, 146, 273, 400,
    527, 654, 781, 908, 909, 782, 655, 528, 401, 274, 147, 20, 21, 148, 275, 402, 529, 656, 783, 910, 911, 784, 657, 530, 403, 276, 149, 22, 23, 150, 277, 404, 531, 658, 785, 912, 913, 786, 659, 532, 405, 278, 151, 24, 25, 152, 279, 406, 533, 660, 787, 914, 915, 788, 661, 534, 407, 280, 153, 26, 27, 154, 281, 408, 535, 662, 789, 916, 917, 790, 663, 536, 409, 282, 155, 28, 29, 156, 283, 410, 537, 664, 791, 918, 919, 792, 665, 538, 411, 284, 157, 30, 31, 158, 285, 412, 539, 666, 793, 920, 921, 794, 667, 540, 413, 286, 159, 32, 33, 160, 287, 414, 541, 668, 795, 922, 923, 796, 669, 542, 415, 288, 161, 34, 35, 162, 289, 416,
    543, 670, 797, 924, 925, 798, 671, 544, 417, 290, 163, 36, 37, 164, 291, 418, 545, 672, 799, 926, 927, 800, 673, 546, 419, 292, 165, 38, 39, 166, 293, 420, 547, 674, 801, 928, 929, 802, 675, 548, 421, 294, 167, 40, 41, 168, 295, 422, 549, 676, 803, 930, 931, 804, 677, 550, 423, 296, 169, 42, 43, 170, 297, 424, 551, 678, 805, 932, 933, 806, 679, 552, 425, 298, 171, 44, 45, 172, 299, 426, 553, 680, 807, 934, 935, 808, 681, 554, 427, 300, 173, 46, 47, 174, 301, 428, 555, 682, 809, 936, 937, 810, 683, 556, 429, 302, 175, 48, 49, 176, 303, 430, 557, 684, 811, 938, 939, 812, 685, 558, 431, 304, 177, 50, 51, 178, 305, 432,
    559, 686, 813, 940, 941, 814, 687, 560, 433, 306, 179, 52, 53, 180, 307, 434, 561, 688, 815, 942, 943, 816, 689, 562, 435, 308, 181, 54, 55, 182, 309, 436, 563, 690, 817, 944, 945, 818, 691, 564, 437, 310, 183, 56, 57, 184, 311, 438, 565, 692, 819, 946, 947, 820, 693, 566, 439, 312, 185, 58, 59, 186, 313, 440, 567, 694, 821, 948, 949, 822, 695, 568, 441, 314, 187, 60, 61, 188, 315, 442, 569, 696, 823, 950, 951, 824, 697, 570, 443, 316, 189, 62, 63, 190, 317, 444, 571, 698, 825, 952, 953, 826, 699, 572, 445, 318, 191, 64, 65, 192, 319, 446, 573, 700, 827, 954, 955, 828, 701, 574, 447, 320, 193, 66, 67, 194, 321, 448,
    575, 702, 829, 956, 957, 830, 703, 576, 449, 322, 195, 68, 69, 196, 323, 450, 577, 704, 831, 958, 959, 832, 705, 578, 451, 324, 197, 70, 71, 198, 325, 452, 579, 706, 833, 960, 961, 834, 707, 580, 453, 326, 199, 72, 73, 200, 327, 454, 581, 708, 835, 962, 963, 836, 709, 582, 455, 328, 201, 74, 75, 202, 329, 456, 583, 710, 837, 964, 965, 838, 711, 584, 457, 330, 203, 76, 77, 204, 331, 458, 585, 712, 839, 966, 967, 840, 713, 586, 459, 332, 205, 78, 79, 206, 333, 460, 587, 714, 841, 968, 969, 842, 715, 588, 461, 334, 207, 80, 81, 208, 335, 462, 589, 716, 843, 970, 971, 844, 717, 590, 463, 336, 209, 82, 83, 210, 337, 464,
    591, 718, 845, 972, 973, 846, 719, 592, 465, 338, 211, 84, 85, 212, 339, 466, 593, 720, 847, 974, 975, 848, 721, 594, 467, 340, 213, 86, 87, 214, 341, 468, 595, 722, 849, 976, 977, 850, 723, 596, 469, 342, 215, 88, 89, 216, 343, 470, 597, 724, 851, 978, 979, 852, 725, 598, 471, 344, 217, 90, 91, 218, 345, 472, 599, 726, 853, 980, 981, 854, 727, 600, 473, 346, 219, 92, 93, 220, 347, 474, 601, 728, 855, 982, 983, 856, 729, 602, 475, 348, 221, 94, 95, 222, 349, 476, 603, 730, 857, 984, 985, 858, 731, 604, 477, 350, 223, 96, 97, 224, 351, 478, 605, 732, 859, 986, 987, 860, 733, 606, 479, 352, 225, 98, 99, 226, 353, 480,
    607, 734, 861, 988, 989, 862, 735, 608, 481, 354, 227, 100, 101, 228, 355, 482, 609, 736, 863, 990, 991, 864, 737, 610, 483, 356, 229, 102, 103, 230, 357, 484, 611, 738, 865, 992, 993, 866, 739, 612, 485, 358, 231, 104, 105, 232, 359, 486, 613, 740, 867, 994, 995, 868, 741, 614, 487, 360, 233, 106, 107, 234, 361, 488, 615, 742, 869, 996, 997, 870, 743, 616, 489, 362, 235, 108, 109, 236, 363, 490, 617, 744, 871, 998, 999, 872, 745, 618, 491, 364, 237, 110, 111, 238, 365, 492, 619, 746, 873, 1000, 1001, 874, 747, 620, 493, 366, 239, 112, 113, 240, 367, 494, 621, 748, 875, 1002, 1003, 876, 749, 622, 495, 368, 241, 114, 115, 242, 369, 496,
    623, 750, 877, 1004, 1005, 878, 751, 624, 497, 370, 243, 116, 117, 244, 371, 498, 625, 752, 879, 1006, 1007, 880, 753, 626, 499, 372, 245, 118, 119, 246, 373, 500, 627, 754, 881, 1008, 1009, 882, 755, 628, 501, 374, 247, 120, 121, 248, 375, 502, 629, 756, 883, 1010, 1011, 884, 757, 630, 503, 376, 249, 122, 123, 250, 377, 504, 631, 758, 885, 1012, 1013, 886, 759, 632, 505, 378, 251, 124, 125, 252, 379, 506, 633, 760, 887, 1014, 1015, 888, 761, 634, 507, 380, 253, 126, 127, 254, 381, 508, 635, 762, 889, 1016, 1017, 890, 763, 636, 509, 382, 255, 383, 510, 637, 764, 891, 1018, 1019, 892, 765, 638, 511, 639, 766, 893, 1020, 1021, 894, 767, 895, 1022, 1023,

    // 16 x 2:
    0, 1, 2, 4, 3, 5, 6, 8, 7, 9, 10, 12, 11, 13, 14, 16, 15, 17, 18, 20, 19, 21, 22, 24, 23, 25, 26, 28, 27, 29, 30, 31,
    // 16 x 4:
    0, 1, 4, 8, 5, 2, 3, 6, 9, 12, 16, 13, 10, 7, 11, 14, 17, 20, 24, 21, 18, 15, 19, 22, 25, 28, 32, 29, 26, 23, 27, 30, 33, 36, 40, 37, 34, 31, 35, 38, 41, 44, 48, 45, 42, 39, 43, 46, 49, 52, 56, 53, 50, 47, 51, 54, 57, 60, 61, 58, 55, 59, 62, 63,
    // 16 x 8:
    0, 1, 8, 16, 9, 2, 3, 10, 17, 24, 32, 25, 18, 11, 4, 5, 12, 19, 26, 33, 40, 48, 41, 34, 27, 20, 13, 6, 7, 14, 21, 28, 35, 42, 49, 56, 64, 57, 50, 43, 36, 29, 22, 15, 23, 30, 37, 44, 51, 58, 65, 72, 80, 73, 66, 59, 52, 45, 38, 31, 39, 46, 53, 60, 67, 74, 81, 88, 96, 89, 82, 75, 68, 61, 54, 47, 55, 62, 69, 76, 83, 90, 97, 104, 112, 105, 98, 91, 84, 77, 70, 63, 71, 78, 85, 92, 99, 106, 113, 120, 121, 114, 107, 100, 93, 86, 79, 87, 94, 101, 108, 115, 122, 123, 116, 109, 102, 95, 103, 110, 117, 124, 125, 118, 111, 119, 126, 127,

    // 16 x 16:
    0, 1, 16, 32, 17, 2, 3, 18, 33, 48, 64, 49, 34, 19, 4, 5, 20, 35, 50, 65, 80, 96, 81, 66, 51, 36, 21, 6, 7, 22, 37, 52, 67, 82, 97, 112, 128, 113, 98, 83, 68, 53, 38, 23, 8, 9, 24, 39, 54, 69, 84, 99, 114, 129, 144, 160, 145, 130, 115, 100, 85, 70, 55, 40, 25, 10, 11, 26, 41, 56, 71, 86, 101, 116, 131, 146, 161, 176, 192, 177, 162, 147, 132, 117, 102, 87, 72, 57, 42, 27, 12, 13, 28, 43, 58, 73, 88, 103, 118, 133, 148, 163, 178, 193, 208, 224, 209, 194, 179, 164, 149, 134, 119, 104, 89, 74, 59, 44, 29, 14, 15, 30, 45, 60, 75, 90, 105, 120,
    135, 150, 165, 180, 195, 210, 225, 240, 241, 226, 211, 196, 181, 166, 151, 136, 121, 106, 91, 76, 61, 46, 31, 47, 62, 77, 92, 107, 122, 137, 152, 167, 182, 197, 212, 227, 242, 243, 228, 213, 198, 183, 168, 153, 138, 123, 108, 93, 78, 63, 79, 94, 109, 124, 139, 154, 169, 184, 199, 214, 229, 244, 245, 230, 215, 200, 185, 170, 155, 140, 125, 110, 95, 111, 126, 141, 156, 171, 186, 201, 216, 231, 246, 247, 232, 217, 202, 187, 172, 157, 142, 127, 143, 158, 173, 188, 203, 218, 233, 248, 249, 234, 219, 204, 189, 174, 159, 175, 190, 205, 220, 235, 250, 251, 236, 221, 206, 191, 207, 222, 237, 252, 253, 238, 223, 239, 254, 255,

    // 16 x 32:
    0, 1, 32, 64, 33, 2, 3, 34, 65, 96, 128, 97, 66, 35, 4, 5, 36, 67, 98, 129, 160, 192, 161, 130, 99, 68, 37, 6, 7, 38, 69, 100, 131, 162, 193, 224, 256, 225, 194, 163, 132, 101, 70, 39, 8, 9, 40, 71, 102, 133, 164, 195, 226, 257, 288, 320, 289, 258, 227, 196, 165, 134, 103, 72, 41, 10, 11, 42, 73, 104, 135, 166, 197, 228, 259, 290, 321, 352, 384, 353, 322, 291, 260, 229, 198, 167, 136, 105, 74, 43, 12, 13, 44, 75, 106, 137, 168, 199, 230, 261, 292, 323, 354, 385, 416, 448, 417, 386, 355, 324, 293, 262, 231, 200, 169, 138, 107, 76, 45, 14, 15, 46, 77, 108, 139, 170, 201, 232,
    263, 294, 325, 356, 387, 418, 449, 480, 481, 450, 419, 388, 357, 326, 295, 264, 233, 202, 171, 140, 109, 78, 47, 16, 17, 48, 79, 110, 141, 172, 203, 234, 265, 296, 327, 358, 389, 420, 451, 482, 483, 452, 421, 390, 359, 328, 297, 266, 235, 204, 173, 142, 111, 80, 49, 18, 19, 50, 81, 112, 143, 174, 205, 236, 267, 298, 329, 360, 391, 422, 453, 484, 485, 454, 423, 392, 361, 330, 299, 268, 237, 206, 175, 144, 113, 82, 51, 20, 21, 52, 83, 114, 145, 176, 207, 238, 269, 300, 331, 362, 393, 424, 455, 486, 487, 456, 425, 394, 363, 332, 301, 270, 239, 208, 177, 146, 115, 84, 53, 22, 23, 54, 85, 116, 147, 178, 209, 240,
    271, 302, 333, 364, 395, 426, 457, 488, 489, 458, 427, 396, 365, 334, 303, 272, 241, 210, 179, 148, 117, 86, 55, 24, 25, 56, 87, 118, 149, 180, 211, 242, 273, 304, 335, 366, 397, 428, 459, 490, 491, 460, 429, 398, 367, 336, 305, 274, 243, 212, 181, 150, 119, 88, 57, 26, 27, 58, 89, 120, 151, 182, 213, 244, 275, 306, 337, 368, 399, 430, 461, 492, 493, 462, 431, 400, 369, 338, 307, 276, 245, 214, 183, 152, 121, 90, 59, 28, 29, 60, 91, 122, 153, 184, 215, 246, 277, 308, 339, 370, 401, 432, 463, 494, 495, 464, 433, 402, 371, 340, 309, 278, 247, 216, 185, 154, 123, 92, 61, 30, 31, 62, 93, 124, 155, 186, 217, 248,
    279, 310, 341, 372, 403, 434, 465, 496, 497, 466, 435, 404, 373, 342, 311, 280, 249, 218, 187, 156, 125, 94, 63, 95, 126, 157, 188, 219, 250, 281, 312, 343, 374, 405, 436, 467, 498, 499, 468, 437, 406, 375, 344, 313, 282, 251, 220, 189, 158, 127, 159, 190, 221, 252, 283, 314, 345, 376, 407, 438, 469, 500, 501, 470, 439, 408, 377, 346, 315, 284, 253, 222, 191, 223, 254, 285, 316, 347, 378, 409, 440, 471, 502, 503, 472, 441, 410, 379, 348, 317, 286, 255, 287, 318, 349, 380, 411, 442, 473, 504, 505, 474, 443, 412, 381, 350, 319, 351, 382, 413, 444, 475, 506, 507, 476, 445, 414, 383, 415, 446, 477, 508, 509, 478, 447, 479, 510, 511,

    // 16 x 64:
    0, 1, 64, 128, 65, 2, 3, 66, 129, 192, 256, 193, 130, 67, 4, 5, 68, 131, 194, 257, 320, 384, 321, 258, 195, 132, 69, 6, 7, 70, 133, 196, 259, 322, 385, 448, 512, 449, 386, 323, 260, 197, 134, 71, 8, 9, 72, 135, 198, 261, 324, 387, 450, 513, 576, 640, 577, 514, 451, 388, 325, 262, 199, 136, 73, 10, 11, 74, 137, 200, 263, 326, 389, 452, 515, 578, 641, 704, 768, 705, 642, 579, 516, 453, 390, 327, 264, 201, 138, 75, 12, 13, 76, 139, 202, 265, 328, 391, 454, 517, 580, 643, 706, 769, 832, 896, 833, 770, 707, 644, 581, 518, 455, 392, 329, 266, 203, 140, 77, 14, 15, 78, 141, 204, 267, 330, 393, 456,
    519, 582, 645, 708, 771, 834, 897, 960, 961, 898, 835, 772, 709, 646, 583, 520, 457, 394, 331, 268, 205, 142, 79, 16, 17, 80, 143, 206, 269, 332, 395, 458, 521, 584, 647, 710, 773, 836, 899, 962, 963, 900, 837, 774, 711, 648, 585, 522, 459, 396, 333, 270, 207, 144, 81, 18, 19, 82, 145, 208, 271, 334, 397, 460, 523, 586, 649, 712, 775, 838, 901, 964, 965, 902, 839, 776, 713, 650, 587, 524, 461, 398, 335, 272, 209, 146, 83, 20, 21, 84, 147, 210, 273, 336, 399, 462, 525, 588, 651, 714, 777, 840, 903, 966, 967, 904, 841, 778, 715, 652, 589, 526, 463, 400, 337, 274, 211, 148, 85, 22, 23, 86, 149, 212, 275, 338, 401, 464,
    527, 590, 653, 716, 779, 842, 905, 968, 969, 906, 843, 780, 717, 654, 591, 528, 465, 402, 339, 276, 213, 150, 87, 24, 25, 88, 151, 214, 277, 340, 403, 466, 529, 592, 655, 718, 781, 844, 907, 970, 971, 908, 845, 782, 719, 656, 593, 530, 467, 404, 341, 278, 215, 152, 89, 26, 27, 90, 153, 216, 279, 342, 405, 468, 531, 594, 657, 720, 783, 846, 909, 972, 973, 910, 847, 784, 721, 658, 595, 532, 469, 406, 343, 280, 217, 154, 91, 28, 29, 92, 155, 218, 281, 344, 407, 470, 533, 596, 659, 722, 785, 848, 911, 974, 975, 912, 849, 786, 723, 660, 597, 534, 471, 408, 345, 282, 219, 156, 93, 30, 31, 94, 157, 220, 283, 346, 409, 472,
    535, 598, 661, 724, 787, 850, 913, 976, 977, 914, 851, 788, 725, 662, 599, 536, 473, 410, 347, 284, 221, 158, 95, 32, 33, 96, 159, 222, 285, 348, 411, 474, 537, 600, 663, 726, 789, 852, 915, 978, 979, 916, 853, 790, 727, 664, 601, 538, 475, 412, 349, 286, 223, 160, 97, 34, 35, 98, 161, 224, 287, 350, 413, 476, 539, 602, 665, 728, 791, 854, 917, 980, 981, 918, 855, 792, 729, 666, 603, 540, 477, 414, 351, 288, 225, 162, 99, 36, 37, 100, 163, 226, 289, 352, 415, 478, 541, 604, 667, 730, 793, 856, 919, 982, 983, 920, 857, 794, 731, 668, 605, 542, 479, 416, 353, 290, 227, 164, 101, 38, 39, 102, 165, 228, 291, 354, 417, 480,
    543, 606, 669, 732, 795, 858, 921, 984, 985, 922, 859, 796, 733, 670, 607, 544, 481, 418, 355, 292, 229, 166, 103, 40, 41, 104, 167, 230, 293, 356, 419, 482, 545, 608, 671, 734, 797, 860, 923, 986, 987, 924, 861, 798, 735, 672, 609, 546, 483, 420, 357, 294, 231, 168, 105, 42, 43, 106, 169, 232, 295, 358, 421, 484, 547, 610, 673, 736, 799, 862, 925, 988, 989, 926, 863, 800, 737, 674, 611, 548, 485, 422, 359, 296, 233, 170, 107, 44, 45, 108, 171, 234, 297, 360, 423, 486, 549, 612, 675, 738, 801, 864, 927, 990, 991, 928, 865, 802, 739, 676, 613, 550, 487, 424, 361, 298, 235, 172, 109, 46, 47, 110, 173, 236, 299, 362, 425, 488,
    551, 614, 677, 740, 803, 866, 929, 992, 993, 930, 867, 804, 741, 678, 615, 552, 489, 426, 363, 300, 237, 174, 111, 48, 49, 112, 175, 238, 301, 364, 427, 490, 553, 616, 679, 742, 805, 868, 931, 994, 995, 932, 869, 806, 743, 680, 617, 554, 491, 428, 365, 302, 239, 176, 113, 50, 51, 114, 177, 240, 303, 366, 429, 492, 555, 618, 681, 744, 807, 870, 933, 996, 997, 934, 871, 808, 745, 682, 619, 556, 493, 430, 367, 304, 241, 178, 115, 52, 53, 116, 179, 242, 305, 368, 431, 494, 557, 620, 683, 746, 809, 872, 935, 998, 999, 936, 873, 810, 747, 684, 621, 558, 495, 432, 369, 306, 243, 180, 117, 54, 55, 118, 181, 244, 307, 370, 433, 496,
    559, 622, 685, 748, 811, 874, 937, 1000, 1001, 938, 875, 812, 749, 686, 623, 560, 497, 434, 371, 308, 245, 182, 119, 56, 57, 120, 183, 246, 309, 372, 435, 498, 561, 624, 687, 750, 813, 876, 939, 1002, 1003, 940, 877, 814, 751, 688, 625, 562, 499, 436, 373, 310, 247, 184, 121, 58, 59, 122, 185, 248, 311, 374, 437, 500, 563, 626, 689, 752, 815, 878, 941, 1004, 1005, 942, 879, 816, 753, 690, 627, 564, 501, 438, 375, 312, 249, 186, 123, 60, 61, 124, 187, 250, 313, 376, 439, 502, 565, 628, 691, 754, 817, 880, 943, 1006, 1007, 944, 881, 818, 755, 692, 629, 566, 503, 440, 377, 314, 251, 188, 125, 62, 63, 126, 189, 252, 315, 378, 441, 504,
    567, 630, 693, 756, 819, 882, 945, 1008, 1009, 946, 883, 820, 757, 694, 631, 568, 505, 442, 379, 316, 253, 190, 127, 191, 254, 317, 380, 443, 506, 569, 632, 695, 758, 821, 884, 947, 1010, 1011, 948, 885, 822, 759, 696, 633, 570, 507, 444, 381, 318, 255, 319, 382, 445, 508, 571, 634, 697, 760, 823, 886, 949, 1012, 1013, 950, 887, 824, 761, 698, 635, 572, 509, 446, 383, 447, 510, 573, 636, 699, 762, 825, 888, 951, 1014, 1015, 952, 889, 826, 763, 700, 637, 574, 511, 575, 638, 701, 764, 827, 890, 953, 1016, 1017, 954, 891, 828, 765, 702, 639, 703, 766, 829, 892, 955, 1018, 1019, 956, 893, 830, 767, 831, 894, 957, 1020, 1021, 958, 895, 959, 1022, 1023,

    // 16 x 128:
    0, 1, 128, 256, 129, 2, 3, 130, 257, 384, 512, 385, 258, 131, 4, 5, 132, 259, 386, 513, 640, 768, 641, 514, 387, 260, 133, 6, 7, 134, 261, 388, 515, 642, 769, 896, 1024, 897, 770, 643, 516, 389, 262, 135, 8, 9, 136, 263, 390, 517, 644, 771, 898, 1025, 1152, 1280, 1153, 1026, 899, 772, 645, 518, 391, 264, 137, 10, 11, 138, 265, 392, 519, 646, 773, 900, 1027, 1154, 1281, 1408, 1536, 1409, 1282, 1155, 1028, 901, 774, 647, 520, 393, 266, 139, 12, 13, 140, 267, 394, 521, 648, 775, 902, 1029, 1156, 1283, 1410, 1537, 1664, 1792, 1665, 1538, 1411, 1284, 1157, 1030, 903, 776, 649, 522, 395, 268, 141, 14, 15, 142, 269, 396, 523, 650, 777, 904,
    1031, 1158, 1285, 1412, 1539, 1666, 1793, 1920, 1921, 1794, 1667, 1540, 1413, 1286, 1159, 1032, 905, 778, 651, 524, 397, 270, 143, 16, 17, 144, 271, 398, 525, 652, 779, 906, 1033, 1160, 1287, 1414, 1541, 1668, 1795, 1922, 1923, 1796, 1669, 1542, 1415, 1288, 1161, 1034, 907, 780, 653, 526, 399, 272, 145, 18, 19, 146, 273, 400, 527, 654, 781, 908, 1035, 1162, 1289, 1416, 1543, 1670, 1797, 1924, 1925, 1798, 1671, 1544, 1417, 1290, 1163, 1036, 909, 782, 655, 528, 401, 274, 147, 20, 21, 148, 275, 402, 529, 656, 783, 910, 1037, 1164, 1291, 1418, 1545, 1672, 1799, 1926, 1927, 1800, 1673, 1546, 1419, 1292, 1165, 1038, 911, 784, 657, 530, 403, 276, 149, 22, 23, 150, 277, 404, 531, 658, 785, 912,
    1039, 1166, 1293, 1420, 1547, 1674, 1801, 1928, 1929, 1802, 1675, 1548, 1421, 1294, 1167, 1040, 913, 786, 659, 532, 405, 278, 151, 24, 25, 152, 279, 406, 533, 660, 787, 914, 1041, 1168, 1295, 1422, 1549, 1676, 1803, 1930, 1931, 1804, 1677, 1550, 1423, 1296, 1169, 1042, 915, 788, 661, 534, 407, 280, 153, 26, 27, 154, 281, 408, 535, 662, 789, 916, 1043, 1170, 1297, 1424, 1551, 1678, 1805, 1932, 1933, 1806, 1679, 1552, 1425, 1298, 1171, 1044, 917, 790, 663, 536, 409, 282, 155, 28, 29, 156, 283, 410, 537, 664, 791, 918, 1045, 1172, 1299, 1426, 1553, 1680, 1807, 1934, 1935, 1808, 1681, 1554, 1427, 1300, 1173, 1046, 919, 792, 665, 538, 411, 284, 157, 30, 31, 158, 285, 412, 539, 666, 793, 920,
    1047, 1174, 1301, 1428, 1555, 1682, 1809, 1936, 1937, 1810, 1683, 1556, 1429, 1302, 1175, 1048, 921, 794, 667, 540, 413, 286, 159, 32, 33, 160, 287, 414, 541, 668, 795, 922, 1049, 1176, 1303, 1430, 1557, 1684, 1811, 1938, 1939, 1812, 1685, 1558, 1431, 1304, 1177, 1050, 923, 796, 669, 542, 415, 288, 161, 34, 35, 162, 289, 416, 543, 670, 797, 924, 1051, 1178, 1305, 1432, 1559, 1686, 1813, 1940, 1941, 1814, 1687, 1560, 1433, 1306, 1179, 1052, 925, 798, 671, 544, 417, 290, 163, 36, 37, 164, 291, 418, 545, 672, 799, 926, 1053, 1180, 1307, 1434, 1561, 1688, 1815, 1942, 1943, 1816, 1689, 1562, 1435, 1308, 1181, 1054, 927, 800, 673, 546, 419, 292, 165, 38, 39, 166, 293, 420, 547, 674, 801, 928,
    1055, 1182, 1309, 1436, 1563, 1690, 1817, 1944, 1945, 1818, 1691, 1564, 1437, 1310, 1183, 1056, 929, 802, 675, 548, 421, 294, 167, 40, 41, 168, 295, 422, 549, 676, 803, 930, 1057, 1184, 1311, 1438, 1565, 1692, 1819, 1946, 1947, 1820, 1693, 1566, 1439, 1312, 1185, 1058, 931, 804, 677, 550, 423, 296, 169, 42, 43, 170, 297, 424, 551, 678, 805, 932, 1059, 1186, 1313, 1440, 1567, 1694, 1821, 1948, 1949, 1822, 1695, 1568, 1441, 1314, 1187, 1060, 933, 806, 679, 552, 425, 298, 171, 44, 45, 172, 299, 426, 553, 680, 807, 934, 1061, 1188, 1315, 1442, 1569, 1696, 1823, 1950, 1951, 1824, 1697, 1570, 1443, 1316, 1189, 1062, 935, 808, 681, 554, 427, 300, 173, 46, 47, 174, 301, 428, 555, 682, 809, 936,
    1063, 1190, 1317, 1444, 1571, 1698, 1825, 1952, 1953, 1826, 1699, 1572, 1445, 1318, 1191, 1064, 937, 810, 683, 556, 429, 302, 175, 48, 49, 176, 303, 430, 557, 684, 811, 938, 1065, 1192, 1319, 1446, 1573, 1700, 1827, 1954, 1955, 1828, 1701, 1574, 1447, 1320, 1193, 1066, 939, 812, 685, 558, 431, 304, 177, 50, 51, 178, 305, 432, 559, 686, 813, 940, 1067, 1194, 1321, 1448, 1575, 1702, 1829, 1956, 1957, 1830, 1703, 1576, 1449, 1322, 1195, 1068, 941, 814, 687, 560, 433, 306, 179, 52, 53, 180, 307, 434, 561, 688, 815, 942, 1069, 1196, 1323, 1450, 1577, 1704, 1831, 1958, 1959, 1832, 1705, 1578, 1451, 1324, 1197, 1070, 943, 816, 689, 562, 435, 308, 181, 54, 55, 182, 309, 436, 563, 690, 817, 944,
    1071, 1198, 1325, 1452, 1579, 1706, 1833, 1960, 1961, 1834, 1707, 1580, 1453, 1326, 1199, 1072, 945, 818, 691, 564, 437, 310, 183, 56, 57, 184, 311, 438, 565, 692, 819, 946, 1073, 1200, 1327, 1454, 1581, 1708, 1835, 1962, 1963, 1836, 1709, 1582, 1455, 1328, 1201, 1074, 947, 820, 693, 566, 439, 312, 185, 58, 59, 186, 313, 440, 567, 694, 821, 948, 1075, 1202, 1329, 1456, 1583, 1710, 1837, 1964, 1965, 1838, 1711, 1584, 1457, 1330, 1203, 1076, 949, 822, 695, 568, 441, 314, 187, 60, 61, 188, 315, 442, 569, 696, 823, 950, 1077, 1204, 1331, 1458, 1585, 1712, 1839, 1966, 1967, 1840, 1713, 1586, 1459, 1332, 1205, 1078, 951, 824, 697, 570, 443, 316, 189, 62, 63, 190, 317, 444, 571, 698, 825, 952,
    1079, 1206, 1333, 1460, 1587, 1714, 1841, 1968, 1969, 1842, 1715, 1588, 1461, 1334, 1207, 1080, 953, 826, 699, 572, 445, 318, 191, 64, 65, 192, 319, 446, 573, 700, 827, 954, 1081, 1208, 1335, 1462, 1589, 1716, 1843, 1970, 1971, 1844, 1717, 1590, 1463, 1336, 1209, 1082, 955, 828, 701, 574, 447, 320, 193, 66, 67, 194, 321, 448, 575, 702, 829, 956, 1083, 1210, 1337, 1464, 1591, 1718, 1845, 1972, 1973, 1846, 1719, 1592, 1465, 1338, 1211, 1084, 957, 830, 703, 576, 449, 322, 195, 68, 69, 196, 323, 450, 577, 704, 831, 958, 1085, 1212, 1339, 1466, 1593, 1720, 1847, 1974, 1975, 1848, 1721, 1594, 1467, 1340, 1213, 1086, 959, 832, 705, 578, 451, 324, 197, 70, 71, 198, 325, 452, 579, 706, 833, 960,
    1087, 1214, 1341, 1468, 1595, 1722, 1849, 1976, 1977, 1850, 1723, 1596, 1469, 1342, 1215, 1088, 961, 834, 707, 580, 453, 326, 199, 72, 73, 200, 327, 454, 581, 708, 835, 962, 1089, 1216, 1343, 1470, 1597, 1724, 1851, 1978, 1979, 1852, 1725, 1598, 1471, 1344, 1217, 1090, 963, 836, 709, 582, 455, 328, 201, 74, 75, 202, 329, 456, 583, 710, 837, 964, 1091, 1218, 1345, 1472, 1599, 1726, 1853, 1980, 1981, 1854, 1727, 1600, 1473, 1346, 1219, 1092, 965, 838, 711, 584, 457, 330, 203, 76, 77, 204, 331, 458, 585, 712, 839, 966, 1093, 1220, 1347, 1474, 1601, 1728, 1855, 1982, 1983, 1856, 1729, 1602, 1475, 1348, 1221, 1094, 967, 840, 713, 586, 459, 332, 205, 78, 79, 206, 333, 460, 587, 714, 841, 968,
    1095, 1222, 1349, 1476, 1603, 1730, 1857, 1984, 1985, 1858, 1731, 1604, 1477, 1350, 1223, 1096, 969, 842, 715, 588, 461, 334, 207, 80, 81, 208, 335, 462, 589, 716, 843, 970, 1097, 1224, 1351, 1478, 1605, 1732, 1859, 1986, 1987, 1860, 1733, 1606, 1479, 1352, 1225, 1098, 971, 844, 717, 590, 463, 336, 209, 82, 83, 210, 337, 464, 591, 718, 845, 972, 1099, 1226, 1353, 1480, 1607, 1734, 1861, 1988, 1989, 1862, 1735, 1608, 1481, 1354, 1227, 1100, 973, 846, 719, 592, 465, 338, 211, 84, 85, 212, 339, 466, 593, 720, 847, 974, 1101, 1228, 1355, 1482, 1609, 1736, 1863, 1990, 1991, 1864, 1737, 1610, 1483, 1356, 1229, 1102, 975, 848, 721, 594, 467, 340, 213, 86, 87, 214, 341, 468, 595, 722, 849, 976,
    1103, 1230, 1357, 1484, 1611, 1738, 1865, 1992, 1993, 1866, 1739, 1612, 1485, 1358, 1231, 1104, 977, 850, 723, 596, 469, 342, 215, 88, 89, 216, 343, 470, 597, 724, 851, 978, 1105, 1232, 1359, 1486, 1613, 1740, 1867, 1994, 1995, 1868, 1741, 1614, 1487, 1360, 1233, 1106, 979, 852, 725, 598, 471, 344, 217, 90, 91, 218, 345, 472, 599, 726, 853, 980, 1107, 1234, 1361, 1488, 1615, 1742, 1869, 1996, 1997, 1870, 1743, 1616, 1489, 1362, 1235, 1108, 981, 854, 727, 600, 473, 346, 219, 92, 93, 220, 347, 474, 601, 728, 855, 982, 1109, 1236, 1363, 1490, 1617, 1744, 1871, 1998, 1999, 1872, 1745, 1618, 1491, 1364, 1237, 1110, 983, 856, 729, 602, 475, 348, 221, 94, 95, 222, 349, 476, 603, 730, 857, 984,
    1111, 1238, 1365, 1492, 1619, 1746, 1873, 2000, 2001, 1874, 1747, 1620, 1493, 1366, 1239, 1112, 985, 858, 731, 604, 477, 350, 223, 96, 97, 224, 351, 478, 605, 732, 859, 986, 1113, 1240, 1367, 1494, 1621, 1748, 1875, 2002, 2003, 1876, 1749, 1622, 1495, 1368, 1241, 1114, 987, 860, 733, 606, 479, 352, 225, 98, 99, 226, 353, 480, 607, 734, 861, 988, 1115, 1242, 1369, 1496, 1623, 1750, 1877, 2004, 2005, 1878, 1751, 1624, 1497, 1370, 1243, 1116, 989, 862, 735, 608, 481, 354, 227, 100, 101, 228, 355, 482, 609, 736, 863, 990, 1117, 1244, 1371, 1498, 1625, 1752, 1879, 2006, 2007, 1880, 1753, 1626, 1499, 1372, 1245, 1118, 991, 864, 737, 610, 483, 356, 229, 102, 103, 230, 357, 484, 611, 738, 865, 992,
    1119, 1246, 1373, 1500, 1627, 1754, 1881, 2008, 2009, 1882, 1755, 1628, 1501, 1374, 1247, 1120, 993, 866, 739, 612, 485, 358, 231, 104, 105, 232, 359, 486, 613, 740, 867, 994, 1121, 1248, 1375, 1502, 1629, 1756, 1883, 2010, 2011, 1884, 1757, 1630, 1503, 1376, 1249, 1122, 995, 868, 741, 614, 487, 360, 233, 106, 107, 234, 361, 488, 615, 742, 869, 996, 1123, 1250, 1377, 1504, 1631, 1758, 1885, 2012, 2013, 1886, 1759, 1632, 1505, 1378, 1251, 1124, 997, 870, 743, 616, 489, 362, 235, 108, 109, 236, 363, 490, 617, 744, 871, 998, 1125, 1252, 1379, 1506, 1633, 1760, 1887, 2014, 2015, 1888, 1761, 1634, 1507, 1380, 1253, 1126, 999, 872, 745, 618, 491, 364, 237, 110, 111, 238, 365, 492, 619, 746, 873, 1000,
    1127, 1254, 1381, 1508, 1635, 1762, 1889, 2016, 2017, 1890, 1763, 1636, 1509, 1382, 1255, 1128, 1001, 874, 747, 620, 493, 366, 239, 112, 113, 240, 367, 494, 621, 748, 875, 1002, 1129, 1256, 1383, 1510, 1637, 1764, 1891, 2018, 2019, 1892, 1765, 1638, 1511, 1384, 1257, 1130, 1003, 876, 749, 622, 495, 368, 241, 114, 115, 242, 369, 496, 623, 750, 877, 1004, 1131, 1258, 1385, 1512, 1639, 1766, 1893, 2020, 2021, 1894, 1767, 1640, 1513, 1386, 1259, 1132, 1005, 878, 751, 624, 497, 370, 243, 116, 117, 244, 371, 498, 625, 752, 879, 1006, 1133, 1260, 1387, 1514, 1641, 1768, 1895, 2022, 2023, 1896, 1769, 1642, 1515, 1388, 1261, 1134, 1007, 880, 753, 626, 499, 372, 245, 118, 119, 246, 373, 500, 627, 754, 881, 1008,
    1135, 1262, 1389, 1516, 1643, 1770, 1897, 2024, 2025, 1898, 1771, 1644, 1517, 1390, 1263, 1136, 1009, 882, 755, 628, 501, 374, 247, 120, 121, 248, 375, 502, 629, 756, 883, 1010, 1137, 1264, 1391, 1518, 1645, 1772, 1899, 2026, 2027, 1900, 1773, 1646, 1519, 1392, 1265, 1138, 1011, 884, 757, 630, 503, 376, 249, 122, 123, 250, 377, 504, 631, 758, 885, 1012, 1139, 1266, 1393, 1520, 1647, 1774, 1901, 2028, 2029, 1902, 1775, 1648, 1521, 1394, 1267, 1140, 1013, 886, 759, 632, 505, 378, 251, 124, 125, 252, 379, 506, 633, 760, 887, 1014, 1141, 1268, 1395, 1522, 1649, 1776, 1903, 2030, 2031, 1904, 1777, 1650, 1523, 1396, 1269, 1142, 1015, 888, 761, 634, 507, 380, 253, 126, 127, 254, 381, 508, 635, 762, 889, 1016,
    1143, 1270, 1397, 1524, 1651, 1778, 1905, 2032, 2033, 1906, 1779, 1652, 1525, 1398, 1271, 1144, 1017, 890, 763, 636, 509, 382, 255, 383, 510, 637, 764, 891, 1018, 1145, 1272, 1399, 1526, 1653, 1780, 1907, 2034, 2035, 1908, 1781, 1654, 1527, 1400, 1273, 1146, 1019, 892, 765, 638, 511, 639, 766, 893, 1020, 1147, 1274, 1401, 1528, 1655, 1782, 1909, 2036, 2037, 1910, 1783, 1656, 1529, 1402, 1275, 1148, 1021, 894, 767, 895, 1022, 1149, 1276, 1403, 1530, 1657, 1784, 1911, 2038, 2039, 1912, 1785, 1658, 1531, 1404, 1277, 1150, 1023, 1151, 1278, 1405, 1532, 1659, 1786, 1913, 2040, 2041, 1914, 1787, 1660, 1533, 1406, 1279, 1407, 1534, 1661, 1788, 1915, 2042, 2043, 1916, 1789, 1662, 1535, 1663, 1790, 1917, 2044, 2045, 1918, 1791, 1919, 2046, 2047,

    // 32 x 2:
    0, 1, 2, 4, 3, 5, 6, 8, 7, 9, 10, 12, 11, 13, 14, 16, 15, 17, 18, 20, 19, 21, 22, 24, 23, 25, 26, 28, 27, 29, 30, 32, 31, 33, 34, 36, 35, 37, 38, 40, 39, 41, 42, 44, 43, 45, 46, 48, 47, 49, 50, 52, 51, 53, 54, 56, 55, 57, 58, 60, 59, 61, 62, 63,
    // 32 x 4:
    0, 1, 4, 8, 5, 2, 3, 6, 9, 12, 16, 13, 10, 7, 11, 14, 17, 20, 24, 21, 18, 15, 19, 22, 25, 28, 32, 29, 26, 23, 27, 30, 33, 36, 40, 37, 34, 31, 35, 38, 41, 44, 48, 45, 42, 39, 43, 46, 49, 52, 56, 53, 50, 47, 51, 54, 57, 60, 64, 61, 58, 55, 59, 62, 65, 68, 72, 69, 66, 63, 67, 70, 73, 76, 80, 77, 74, 71, 75, 78, 81, 84, 88, 85, 82, 79, 83, 86, 89, 92, 96, 93, 90, 87, 91, 94, 97, 100, 104, 101, 98, 95, 99, 102, 105, 108, 112, 109, 106, 103, 107, 110, 113, 116, 120, 117, 114, 111, 115, 118, 121, 124, 125, 122, 119, 123, 126, 127,

    // 32 x 8:
    0, 1, 8, 16, 9, 2, 3, 10, 17, 24, 32, 25, 18, 11, 4, 5, 12, 19, 26, 33, 40, 48, 41, 34, 27, 20, 13, 6, 7, 14, 21, 28, 35, 42, 49, 56, 64, 57, 50, 43, 36, 29, 22, 15, 23, 30, 37, 44, 51, 58, 65, 72, 80, 73, 66, 59, 52, 45, 38, 31, 39, 46, 53, 60, 67, 74, 81, 88, 96, 89, 82, 75, 68, 61, 54, 47, 55, 62, 69, 76, 83, 90, 97, 104, 112, 105, 98, 91, 84, 77, 70, 63, 71, 78, 85, 92, 99, 106, 113, 120, 128, 121, 114, 107, 100, 93, 86, 79, 87, 94, 101, 108, 115, 122, 129, 136, 144, 137, 130, 123, 116, 109, 102, 95, 103, 110, 117, 124,
    131, 138, 145, 152, 160, 153, 146, 139, 132, 125, 118, 111, 119, 126, 133, 140, 147, 154, 161, 168, 176, 169, 162, 155, 148, 141, 134, 127, 135, 142, 149, 156, 163, 170, 177, 184, 192, 185, 178, 171, 164, 157, 150, 143, 151, 158, 165, 172, 179, 186, 193, 200, 208, 201, 194, 187, 180, 173, 166, 159, 167, 174, 181, 188, 195, 202, 209, 216, 224, 217, 210, 203, 196, 189, 182, 175, 183, 190, 197, 204, 211, 218, 225, 232, 240, 233, 226, 219, 212, 205, 198, 191, 199, 206, 213, 220, 227, 234, 241, 248, 249, 242, 235, 228, 221, 214, 207, 215, 222, 229, 236, 243, 250, 251, 244, 237, 230, 223, 231, 238, 245, 252, 253, 246, 239, 247, 254, 255,

    // 32 x 16:
    0, 1, 16, 32, 17, 2, 3, 18, 33, 48, 64, 49, 34, 19, 4, 5, 20, 35, 50, 65, 80, 96, 81, 66, 51, 36, 21, 6, 7, 22, 37, 52, 67, 82, 97, 112, 128, 113, 98, 83, 68, 53, 38, 23, 8, 9, 24, 39, 54, 69, 84, 99, 114, 129, 144, 160, 145, 130, 115, 100, 85, 70, 55, 40, 25, 10, 11, 26, 41, 56, 71, 86, 101, 116, 131, 146, 161, 176, 192, 177, 162, 147, 132, 117, 102, 87, 72, 57, 42, 27, 12, 13, 28, 43, 58, 73, 88, 103, 118, 133, 148, 163, 178, 193, 208, 224, 209, 194, 179, 164, 149, 134, 119, 104, 89, 74, 59, 44, 29, 14, 15, 30, 45, 60, 75, 90, 105, 120,
    135, 150, 165, 180, 195, 210, 225, 240, 256, 241, 226, 211, 196, 181, 166, 151, 136, 121, 106, 91, 76, 61, 46, 31, 47, 62, 77, 92, 107, 122, 137, 152, 167, 182, 197, 212, 227, 242, 257, 272, 288, 273, 258, 243, 228, 213, 198, 183, 168, 153, 138, 123, 108, 93, 78, 63, 79, 94, 109, 124, 139, 154, 169, 184, 199, 214, 229, 244, 259, 274, 289, 304, 320, 305, 290, 275, 260, 245, 230, 215, 200, 185, 170, 155, 140, 125, 110, 95, 111, 126, 141, 156, 171, 186, 201, 216, 231, 246, 261, 276, 291, 306, 321, 336, 352, 337, 322, 307, 292, 277, 262, 247, 232, 217, 202, 187, 172, 157, 142, 127, 143, 158, 173, 188, 203, 218, 233, 248,
    263, 278, 293, 308, 323, 338, 353, 368, 384, 369, 354, 339, 324, 309, 294, 279, 264, 249, 234, 219, 204, 189, 174, 159, 175, 190, 205, 220, 235, 250, 265, 280, 295, 310, 325, 340, 355, 370, 385, 400, 416, 401, 386, 371, 356, 341, 326, 311, 296, 281, 266, 251, 236, 221, 206, 191, 207, 222, 237, 252, 267, 282, 297, 312, 327, 342, 357, 372, 387, 402, 417, 432, 448, 433, 418, 403, 388, 373, 358, 343, 328, 313, 298, 283, 268, 253, 238, 223, 239, 254, 269, 284, 299, 314, 329, 344, 359, 374, 389, 404, 419, 434, 449, 464, 480, 465, 450, 435, 420, 405, 390, 375, 360, 345, 330, 315, 300, 285, 270, 255, 271, 286, 301, 316, 331, 346, 361, 376,
    391, 406, 421, 436, 451, 466, 481, 496, 497, 482, 467, 452, 437, 422, 407, 392, 377, 362, 347, 332, 317, 302, 287, 303, 318, 333, 348, 363, 378, 393, 408, 423, 438, 453, 468, 483, 498, 499, 484, 469, 454, 439, 424, 409, 394, 379, 364, 349, 334, 319, 335, 350, 365, 380, 395, 410, 425, 440, 455, 470, 485, 500, 501, 486, 471, 456, 441, 426, 411, 396, 381, 366, 351, 367, 382, 397, 412, 427, 442, 457, 472, 487, 502, 503, 488, 473, 458, 443, 428, 413, 398, 383, 399, 414, 429, 444, 459, 474, 489, 504, 505, 490, 475, 460, 445, 430, 415, 431, 446, 461, 476, 491, 506, 507, 492, 477, 462, 447, 463, 478, 493, 508, 509, 494, 479, 495, 510, 511,

    // 32 x 32:
    0, 1, 32, 64, 33, 2, 3, 34, 65, 96, 128, 97, 66, 35, 4, 5, 36, 67, 98, 129, 160, 192, 161, 130, 99, 68, 37, 6, 7, 38, 69, 100, 131, 162, 193, 224, 256, 225, 194, 163, 132, 101, 70, 39, 8, 9, 40, 71, 102, 133, 164, 195, 226, 257, 288, 320, 289, 258, 227, 196, 165, 134, 103, 72, 41, 10, 11, 42, 73, 104, 135, 166, 197, 228, 259, 290, 321, 352, 384, 353, 322, 291, 260, 229, 198, 167, 136, 105, 74, 43, 12, 13, 44, 75, 106, 137, 168, 199, 230, 261, 292, 323, 354, 385, 416, 448, 417, 386, 355, 324, 293, 262, 231, 200, 169, 138, 107, 76, 45, 14, 15, 46, 77, 108, 139, 170, 201, 232,
    263, 294, 325, 356, 387, 418, 449, 480, 512, 481, 450, 419, 388, 357, 326, 295, 264, 233, 202, 171, 140, 109, 78, 47, 16, 17, 48, 79, 110, 141, 172, 203, 234, 265, 296, 327, 358, 389, 420, 451, 482, 513, 544, 576, 545, 514, 483, 452, 421, 390, 359, 328, 297, 266, 235, 204, 173, 142, 111, 80, 49, 18, 19, 50, 81, 112, 143, 174, 205, 236, 267, 298, 329, 360, 391, 422, 453, 484, 515, 546, 577, 608, 640, 609, 578, 547, 516, 485, 454, 423, 392, 361, 330, 299, 268, 237, 206, 175, 144, 113, 82, 51, 20, 21, 52, 83, 114, 145, 176, 207, 238, 269, 300, 331, 362, 393, 424, 455, 486, 517, 548, 579, 610, 641, 672, 704, 673, 642,
    611, 580, 549, 518, 487, 456, 425, 394, 363, 332, 301, 270, 239, 208, 177, 146, 115, 84, 53, 22, 23, 54, 85, 116, 147, 178, 209, 240, 271, 302, 333, 364, 395, 426, 457, 488, 519, 550, 581, 612, 643, 674, 705, 736, 768, 737, 706, 675, 644, 613, 582, 551, 520, 489, 458, 427, 396, 365, 334, 303, 272, 241, 210, 179, 148, 117, 86, 55, 24, 25, 56, 87, 118, 149, 180, 211, 242, 273, 304, 335, 366, 397, 428, 459, 490, 521, 552, 583, 614, 645, 676, 707, 738, 769, 800, 832, 801, 770, 739, 708, 677, 646, 615, 584, 553, 522, 491, 460, 429, 398, 367, 336, 305, 274, 243, 212, 181, 150, 119, 88, 57, 26, 27, 58, 89, 120, 151, 182,
    213, 244, 275, 306, 337, 368, 399, 430, 461, 492, 523, 554, 585, 616, 647, 678, 709, 740, 771, 802, 833, 864, 896, 865, 834, 803, 772, 741, 710, 679, 648, 617, 586, 555, 524, 493, 462, 431, 400, 369, 338, 307, 276, 245, 214, 183, 152, 121, 90, 59, 28, 29, 60, 91, 122, 153, 184, 215, 246, 277, 308, 339, 370, 401, 432, 463, 494, 525, 556, 587, 618, 649, 680, 711, 742, 773, 804, 835, 866, 897, 928, 960, 929, 898, 867, 836, 805, 774, 743, 712, 681, 650, 619, 588, 557, 526, 495, 464, 433, 402, 371, 340, 309, 278, 247, 216, 185, 154, 123, 92, 61, 30, 31, 62, 93, 124, 155, 186, 217, 248, 279, 310, 341, 372, 403, 434, 465, 496,
    527, 558, 589, 620, 651, 682, 713, 744, 775, 806, 837, 868, 899, 930, 961, 992, 993, 962, 931, 900, 869, 838, 807, 776, 745, 714, 683, 652, 621, 590, 559, 528, 497, 466, 435, 404, 373, 342, 311, 280, 249, 218, 187, 156, 125, 94, 63, 95, 126, 157, 188, 219, 250, 281, 312, 343, 374, 405, 436, 467, 498, 529, 560, 591, 622, 653, 684, 715, 746, 777, 808, 839, 870, 901, 932, 963, 994, 995, 964, 933, 902, 871, 840, 809, 778, 747, 716, 685, 654, 623, 592, 561, 530, 499, 468, 437, 406, 375, 344, 313, 282, 251, 220, 189, 158, 127, 159, 190, 221, 252, 283, 314, 345, 376, 407, 438, 469, 500, 531, 562, 593, 624, 655, 686, 717, 748, 779, 810,
    841, 872, 903, 934, 965, 996, 997, 966, 935, 904, 873, 842, 811, 780, 749, 718, 687, 656, 625, 594, 563, 532, 501, 470, 439, 408, 377, 346, 315, 284, 253, 222, 191, 223, 254, 285, 316, 347, 378, 409, 440, 471, 502, 533, 564, 595, 626, 657, 688, 719, 750, 781, 812, 843, 874, 905, 936, 967, 998, 999, 968, 937, 906, 875, 844, 813, 782, 751, 720, 689, 658, 627, 596, 565, 534, 503, 472, 441, 410, 379, 348, 317, 286, 255, 287, 318, 349, 380, 411, 442, 473, 504, 535, 566, 597, 628, 659, 690, 721, 752, 783, 814, 845, 876, 907, 938, 969, 1000, 1001, 970, 939, 908, 877, 846, 815, 784, 753, 722, 691, 660, 629, 598, 567, 536, 505, 474, 443, 412,
    381, 350, 319, 351, 382, 413, 444, 475, 506, 537, 568, 599, 630, 661, 692, 723, 754, 785, 816, 847, 878, 909, 940, 971, 1002, 1003, 972, 941, 910, 879, 848, 817, 786, 755, 724, 693, 662, 631, 600, 569, 538, 507, 476, 445, 414, 383, 415, 446, 477, 508, 539, 570, 601, 632, 663, 694, 725, 756, 787, 818, 849, 880, 911, 942, 973, 1004, 1005, 974, 943, 912, 881, 850, 819, 788, 757, 726, 695, 664, 633, 602, 571, 540, 509, 478, 447, 479, 510, 541, 572, 603, 634, 665, 696, 727, 758, 789, 820, 851, 882, 913, 944, 975, 1006, 1007, 976, 945, 914, 883, 852, 821, 790, 759, 728, 697, 666, 635, 604, 573, 542, 511, 543, 574, 605, 636, 667, 698, 729, 760,
    791, 822, 853, 884, 915, 946, 977, 1008, 1009, 978, 947, 916, 885, 854, 823, 792, 761, 730, 699, 668, 637, 606, 575, 607, 638, 669, 700, 731, 762, 793, 824, 855, 886, 917, 948, 979, 1010, 1011, 980, 949, 918, 887, 856, 825, 794, 763, 732, 701, 670, 639, 671, 702, 733, 764, 795, 826, 857, 888, 919, 950, 981, 1012, 1013, 982, 951, 920, 889, 858, 827, 796, 765, 734, 703, 735, 766, 797, 828, 859, 890, 921, 952, 983, 1014, 1015, 984, 953, 922, 891, 860, 829, 798, 767, 799, 830, 861, 892, 923, 954, 985, 1016, 1017, 986, 955, 924, 893, 862, 831, 863, 894, 925, 956, 987, 1018, 1019, 988, 957, 926, 895, 927, 958, 989, 1020, 1021, 990, 959, 991, 1022, 1023,

    // 32 x 64:
    0, 1, 64, 128, 65, 2, 3, 66, 129, 192, 256, 193, 130, 67, 4, 5, 68, 131, 194, 257, 320, 384, 321, 258, 195, 132, 69, 6, 7, 70, 133, 196, 259, 322, 385, 448, 512, 449, 386, 323, 260, 197, 134, 71, 8, 9, 72, 135, 198, 261, 324, 387, 450, 513, 576, 640, 577, 514, 451, 388, 325, 262, 199, 136, 73, 10, 11, 74, 137, 200, 263, 326, 389, 452, 515, 578, 641, 704, 768, 705, 642, 579, 516, 453, 390, 327, 264, 201, 138, 75, 12, 13, 76, 139, 202, 265, 328, 391, 454, 517, 580, 643, 706, 769, 832, 896, 833, 770, 707, 644, 581, 518, 455, 392, 329, 266, 203, 140, 77, 14, 15, 78, 141, 204, 267, 330, 393, 456,
    519, 582, 645, 708, 771, 834, 897, 960, 1024, 961, 898, 835, 772, 709, 646, 583, 520, 457, 394, 331, 268, 205, 142, 79, 16, 17, 80, 143, 206, 269, 332, 395, 458, 521, 584, 647, 710, 773, 836, 899, 962, 1025, 1088, 1152, 1089, 1026, 963, 900, 837, 774, 711, 648, 585, 522, 459, 396, 333, 270, 207, 144, 81, 18, 19, 82, 145, 208, 271, 334, 397, 460, 523, 586, 649, 712, 775, 838, 901, 964, 1027, 1090, 1153, 1216, 1280, 1217, 1154, 1091, 1028, 965, 902, 839, 776, 713, 650, 587, 524, 461, 398, 335, 272, 209, 146, 83, 20, 21, 84, 147, 210, 273, 336, 399, 462, 525, 588, 651, 714, 777, 840, 903, 966, 1029, 1092, 1155, 1218, 1281, 1344, 1408, 1345, 1282,
    1219, 1156, 1093, 1030, 967, 904, 841, 778, 715, 652, 589, 526, 463, 400, 337, 274, 211, 148, 85, 22, 23, 86, 149, 212, 275, 338, 401, 464, 527, 590, 653, 716, 779, 842, 905, 968, 1031, 1094, 1157, 1220, 1283, 1346, 1409, 1472, 1536, 1473, 1410, 1347, 1284, 1221, 1158, 1095, 1032, 969, 906, 843, 780, 717, 654, 591, 528, 465, 402, 339, 276, 213, 150, 87, 24, 25, 88, 151, 214, 277, 340, 403, 466, 529, 592, 655, 718, 781, 844, 907, 970, 1033, 1096, 1159, 1222, 1285, 1348, 1411, 1474, 1537, 1600, 1664, 1601, 1538, 1475, 1412, 1349, 1286, 1223, 1160, 1097, 1034, 971, 908, 845, 782, 719, 656, 593, 530, 467, 404, 341, 278, 215, 152, 89, 26, 27, 90, 153, 216, 279, 342,
    405, 468, 531, 594, 657, 720, 783, 846, 909, 972, 1035, 1098, 1161, 1224, 1287, 1350, 1413, 1476, 1539, 1602, 1665, 1728, 1792, 1729, 1666, 1603, 1540, 1477, 1414, 1351, 1288, 1225, 1162, 1099, 1036, 973, 910, 847, 784, 721, 658, 595, 532, 469, 406, 343, 280, 217, 154, 91, 28, 29, 92, 155, 218, 281, 344, 407, 470, 533, 596, 659, 722, 785, 848, 911, 974, 1037, 1100, 1163, 1226, 1289, 1352, 1415, 1478, 1541, 1604, 1667, 1730, 1793, 1856, 1920, 1857, 1794, 1731, 1668, 1605, 1542, 1479, 1416, 1353, 1290, 1227, 1164, 1101, 1038, 975, 912, 849, 786, 723, 660, 597, 534, 471, 408, 345, 282, 219, 156, 93, 30, 31, 94, 157, 220, 283, 346, 409, 472, 535, 598, 661, 724, 787, 850, 913, 976,
    1039, 1102, 1165, 1228, 1291, 1354, 1417, 1480, 1543, 1606, 1669, 1732, 1795, 1858, 1921, 1984, 1985, 1922, 1859, 1796, 1733, 1670, 1607, 1544, 1481, 1418, 1355, 1292, 1229, 1166, 1103, 1040, 977, 914, 851, 788, 725, 662, 599, 536, 473, 410, 347, 284, 221, 158, 95, 32, 33, 96, 159, 222, 285, 348, 411, 474, 537, 600, 663, 726, 789, 852, 915, 978, 1041, 1104, 1167, 1230, 1293, 1356, 1419, 1482, 1545, 1608, 1671, 1734, 1797, 1860, 1923, 1986, 1987, 1924, 1861, 1798, 1735, 1672, 1609, 1546, 1483, 1420, 1357, 1294, 1231, 1168, 1105, 1042, 979, 916, 853, 790, 727, 664, 601, 538, 475, 412, 349, 286, 223, 160, 97, 34, 35, 98, 161, 224, 287, 350, 413, 476, 539, 602, 665, 728, 791, 854, 917, 980,
    1043, 1106, 1169, 1232, 1295, 1358, 1421, 1484, 1547, 1610, 1673, 1736, 1799, 1862, 1925, 1988, 1989, 1926, 1863, 1800, 1737, 1674, 1611, 1548, 1485, 1422, 1359, 1296, 1233, 1170, 1107, 1044, 981, 918, 855, 792, 729, 666, 603, 540, 477, 414, 351, 288, 225, 162, 99, 36, 37, 100, 163, 226, 289, 352, 415, 478, 541, 604, 667, 730, 793, 856, 919, 982, 1045, 1108, 1171, 1234, 1297, 1360, 1423, 1486, 1549, 1612, 1675, 1738, 1801, 1864, 1927, 1990, 1991, 1928, 1865, 1802, 1739, 1676, 1613, 1550, 1487, 1424, 1361, 1298, 1235, 1172, 1109, 1046, 983, 920, 857, 794, 731, 668, 605, 542, 479, 416, 353, 290, 227, 164, 101, 38, 39, 102, 165, 228, 291, 354, 417, 480, 543, 606, 669, 732, 795, 858, 921, 984,
    1047, 1110, 1173, 1236, 1299, 1362, 1425, 1488, 1551, 1614, 1677, 1740, 1803, 1866, 1929, 1992, 1993, 1930, 1867, 1804, 1741, 1678, 1615, 1552, 1489, 1426, 1363, 1300, 1237, 1174, 1111, 1048, 985, 922, 859, 796, 733, 670, 607, 544, 481, 418, 355, 292, 229, 166, 103, 40, 41, 104, 167, 230, 293, 356, 419, 482, 545, 608, 671, 734, 797, 860, 923, 986, 1049, 1112, 1175, 1238, 1301, 1364, 1427, 1490, 1553, 1616, 1679, 1742, 1805, 1868, 1931, 1994, 1995, 1932, 1869, 1806, 1743, 1680, 1617, 1554, 1491, 1428, 1365, 1302, 1239, 1176, 1113, 1050, 987, 924, 861, 798, 735, 672, 609, 546, 483, 420, 357, 294, 231, 168, 105, 42, 43, 106, 169, 232, 295, 358, 421, 484, 547, 610, 673, 736, 799, 862, 925, 988,
    1051, 1114, 1177, 1240, 1303, 1366, 1429, 1492, 1555, 1618, 1681, 1744, 1807, 1870, 1933, 1996, 1997, 1934, 1871, 1808, 1745, 1682, 1619, 1556, 1493, 1430, 1367, 1304, 1241, 1178, 1115, 1052, 989, 926, 863, 800, 737, 674, 611, 548, 485, 422, 359, 296, 233, 170, 107, 44, 45, 108, 171, 234, 297, 360, 423, 486, 549, 612, 675, 738, 801, 864, 927, 990, 1053, 1116, 1179, 1242, 1305, 1368, 1431, 1494, 1557, 1620, 1683, 1746, 1809, 1872, 1935, 1998, 1999, 1936, 1873, 1810, 1747, 1684, 1621, 1558, 1495, 1432, 1369, 1306, 1243, 1180, 1117, 1054, 991, 928, 865, 802, 739, 676, 613, 550, 487, 424, 361, 298, 235, 172, 109, 46, 47, 110, 173, 236, 299, 362, 425, 488, 551, 614, 677, 740, 803, 866, 929, 992,
    1055, 1118, 1181, 1244, 1307, 1370, 1433, 1496, 1559, 1622, 1685, 1748, 1811, 1874, 1937, 2000, 2001, 1938, 1875, 1812, 1749, 1686, 1623, 1560, 1497, 1434, 1371, 1308, 1245, 1182, 1119, 1056, 993, 930, 867, 804, 741, 678, 615, 552, 489, 426, 363, 300, 237, 174, 111, 48, 49, 112, 175, 238, 301, 364, 427, 490, 553, 616, 679, 742, 805, 868, 931, 994, 1057, 1120, 1183, 1246, 1309, 1372, 1435, 1498, 1561, 1624, 1687, 1750, 1813, 1876, 1939, 2002, 2003, 1940, 1877, 1814, 1751, 1688, 1625, 1562, 1499, 1436, 1373, 1310, 1247, 1184, 1121, 1058, 995, 932, 869, 806, 743, 680, 617, 554, 491, 428, 365, 302, 239, 176, 113, 50, 51, 114, 177, 240, 303, 366, 429, 492, 555, 618, 681, 744, 807, 870, 933, 996,
    1059, 1122, 1185, 1248, 1311, 1374, 1437, 1500, 1563, 1626, 1689, 1752, 1815, 1878, 1941, 2004, 2005, 1942, 1879, 1816, 1753, 1690, 1627, 1564, 1501, 1438, 1375, 1312, 1249, 1186, 1123, 1060, 997, 934, 871, 808, 745, 682, 619, 556, 493, 430, 367, 304, 241, 178, 115, 52, 53, 116, 179, 242, 305, 368, 431, 494, 557, 620, 683, 746, 809, 872, 935, 998, 1061, 1124, 1187, 1250, 1313, 1376, 1439, 1502, 1565, 1628, 1691, 1754, 1817, 1880, 1943, 2006, 2007, 1944, 1881, 1818, 1755, 1692, 1629, 1566, 1503, 1440, 1377, 1314, 1251, 1188, 1125, 1062, 999, 936, 873, 810, 747, 684, 621, 558, 495, 432, 369, 306, 243, 180, 117, 54, 55, 118, 181, 244, 307, 370, 433, 496, 559, 622, 685, 748, 811, 874, 937, 1000,
    1063, 1126, 1189, 1252, 1315, 1378, 1441, 1504, 1567, 1630, 1693, 1756, 1819, 1882, 1945, 2008, 2009, 1946, 1883, 1820, 1757, 1694, 1631, 1568, 1505, 1442, 1379, 1316, 1253, 1190, 1127, 1064, 1001, 938, 875, 812, 749, 686, 623, 560, 497, 434, 371, 308, 245, 182, 119, 56, 57, 120, 183, 246, 309, 372, 435, 498, 561, 624, 687, 750, 813, 876, 939, 1002, 1065, 1128, 1191, 1254, 1317, 1380, 1443, 1506, 1569, 1632, 1695, 1758, 1821, 1884, 1947, 2010, 2011, 1948, 1885, 1822, 1759, 1696, 1633, 1570, 1507, 1444, 1381, 1318, 1255, 1192, 1129, 1066, 1003, 940, 877, 814, 751, 688, 625, 562, 499, 436, 373, 310, 247, 184, 121, 58, 59, 122, 185, 248, 311, 374, 437, 500, 563, 626, 689, 752, 815, 878, 941, 1004,
    1067, 1130, 1193, 1256, 1319, 1382, 1445, 1508, 1571, 1634, 1697, 1760, 1823, 1886, 1949, 2012, 2013, 1950, 1887, 1824, 1761, 1698, 1635, 1572, 1509, 1446, 1383, 1320, 1257, 1194, 1131, 1068, 1005, 942, 879, 816, 753, 690, 627, 564, 501, 438, 375, 312, 249, 186, 123, 60, 61, 124, 187, 250, 313, 376, 439, 502, 565, 628, 691, 754, 817, 880, 943, 1006, 1069, 1132, 1195, 1258, 1321, 1384, 1447, 1510, 1573, 1636, 1699, 1762, 1825, 1888, 1951, 2014, 2015, 1952, 1889, 1826, 1763, 1700, 1637, 1574, 1511, 1448, 1385, 1322, 1259, 1196, 1133, 1070, 1007, 944, 881, 818, 755, 692, 629, 566, 503, 440, 377, 314, 251, 188, 125, 62, 63, 126, 189, 252, 315, 378, 441, 504, 567, 630, 693, 756, 819, 882, 945, 1008,
    1071, 1134, 1197, 1260, 1323, 1386, 1449, 1512, 1575, 1638, 1701, 1764, 1827, 1890, 1953, 2016, 2017, 1954, 1891, 1828, 1765, 1702, 1639, 1576, 1513, 1450, 1387, 1324, 1261, 1198, 1135, 1072, 1009, 946, 883, 820, 757, 694, 631, 568, 505, 442, 379, 316, 253, 190, 127, 191, 254, 317, 380, 443, 506, 569, 632, 695, 758, 821, 884, 947, 1010, 1073, 1136, 1199, 1262, 1325, 1388, 1451, 1514, 1577, 1640, 1703, 1766, 1829, 1892, 1955, 2018, 2019, 1956, 1893, 1830, 1767, 1704, 1641, 1578, 1515, 1452, 1389, 1326, 1263, 1200, 1137, 1074, 1011, 948, 885, 822, 759, 696, 633, 570, 507, 444, 381, 318, 255, 319, 382, 445, 508, 571, 634, 697, 760, 823, 886, 949, 1012, 1075, 1138, 1201, 1264, 1327, 1390, 1453, 1516, 1579, 1642,
    1705, 1768, 1831, 1894, 1957, 2020, 2021, 1958, 1895, 1832, 1769, 1706, 1643, 1580, 1517, 1454, 1391, 1328, 1265, 1202, 1139, 1076, 1013, 950, 887, 824, 761, 698, 635, 572, 509, 446, 383, 447, 510, 573, 636, 699, 762, 825, 888, 951, 1014, 1077, 1140, 1203, 1266, 1329, 1392, 1455, 1518, 1581, 1644, 1707, 1770, 1833, 1896, 1959, 2022, 2023, 1960, 1897, 1834, 1771, 1708, 1645, 1582, 1519, 1456, 1393, 1330, 1267, 1204, 1141, 1078, 1015, 952, 889, 826, 763, 700, 637, 574, 511, 575, 638, 701, 764, 827, 890, 953, 1016, 1079, 1142, 1205, 1268, 1331, 1394, 1457, 1520, 1583, 1646, 1709, 1772, 1835, 1898, 1961, 2024, 2025, 1962, 1899, 1836, 1773, 1710, 1647, 1584, 1521, 1458, 1395, 1332, 1269, 1206, 1143, 1080, 1017, 954, 891, 828,
    765, 702, 639, 703, 766, 829, 892, 955, 1018, 1081, 1144, 1207, 1270, 1333, 1396, 1459, 1522, 1585, 1648, 1711, 1774, 1837, 1900, 1963, 2026, 2027, 1964, 1901, 1838, 1775, 1712, 1649, 1586, 1523, 1460, 1397, 1334, 1271, 1208, 1145, 1082, 1019, 956, 893, 830, 767, 831, 894, 957, 1020, 1083, 1146, 1209, 1272, 1335, 1398, 1461, 1524, 1587, 1650, 1713, 1776, 1839, 1902, 1965, 2028, 2029, 1966, 1903, 1840, 1777, 1714, 1651, 1588, 1525, 1462, 1399, 1336, 1273, 1210, 1147, 1084, 1021, 958, 895, 959, 1022, 1085, 1148, 1211, 1274, 1337, 1400, 1463, 1526, 1589, 1652, 1715, 1778, 1841, 1904, 1967, 2030, 2031, 1968, 1905, 1842, 1779, 1716, 1653, 1590, 1527, 1464, 1401, 1338, 1275, 1212, 1149, 1086, 1023, 1087, 1150, 1213, 1276, 1339, 1402, 1465, 1528,
    1591, 1654, 1717, 1780, 1843, 1906, 1969, 2032, 2033, 1970, 1907, 1844, 1781, 1718, 1655, 1592, 1529, 1466, 1403, 1340, 1277, 1214, 1151, 1215, 1278, 1341, 1404, 1467, 1530, 1593, 1656, 1719, 1782, 1845, 1908, 1971, 2034, 2035, 1972, 1909, 1846, 1783, 1720, 1657, 1594, 1531, 1468, 1405, 1342, 1279, 1343, 1406, 1469, 1532, 1595, 1658, 1721, 1784, 1847, 1910, 1973, 2036, 2037, 1974, 1911, 1848, 1785, 1722, 1659, 1596, 1533, 1470, 1407, 1471, 1534, 1597, 1660, 1723, 1786, 1849, 1912, 1975, 2038, 2039, 1976, 1913, 1850, 1787, 1724, 1661, 1598, 1535, 1599, 1662, 1725, 1788, 1851, 1914, 1977, 2040, 2041, 1978, 1915, 1852, 1789, 1726, 1663, 1727, 1790, 1853, 1916, 1979, 2042, 2043, 1980, 1917, 1854, 1791, 1855, 1918, 1981, 2044, 2045, 1982, 1919, 1983, 2046, 2047,

    // 32 x 128:
    0, 1, 128, 256, 129, 2, 3, 130, 257, 384, 512, 385, 258, 131, 4, 5, 132, 259, 386, 513, 640, 768, 641, 514, 387, 260, 133, 6, 7, 134, 261, 388, 515, 642, 769, 896, 1024, 897, 770, 643, 516, 389, 262, 135, 8, 9, 136, 263, 390, 517, 644, 771, 898, 1025, 1152, 1280, 1153, 1026, 899, 772, 645, 518, 391, 264, 137, 10, 11, 138, 265, 392, 519, 646, 773, 900, 1027, 1154, 1281, 1408, 1536, 1409, 1282, 1155, 1028, 901, 774, 647, 520, 393, 266, 139, 12, 13, 140, 267, 394, 521, 648, 775, 902, 1029, 1156, 1283, 1410, 1537, 1664, 1792, 1665, 1538, 1411, 1284, 1157, 1030, 903, 776, 649, 522, 395, 268, 141, 14, 15, 142, 269, 396, 523, 650, 777, 904,
    1031, 1158, 1285, 1412, 1539, 1666, 1793, 1920, 2048, 1921, 1794, 1667, 1540, 1413, 1286, 1159, 1032, 905, 778, 651, 524, 397, 270, 143, 16, 17, 144, 271, 398, 525, 652, 779, 906, 1033, 1160, 1287, 1414, 1541, 1668, 1795, 1922, 2049, 2176, 2304, 2177, 2050, 1923, 1796, 1669, 1542, 1415, 1288, 1161, 1034, 907, 780, 653, 526, 399, 272, 145, 18, 19, 146, 273, 400, 527, 654, 781, 908, 1035, 1162, 1289, 1416, 1543, 1670, 1797, 1924, 2051, 2178, 2305, 2432, 2560, 2433, 2306, 2179, 2052, 1925, 1798, 1671, 1544, 1417, 1290, 1163, 1036, 909, 782, 655, 528, 401, 274, 147, 20, 21, 148, 275, 402, 529, 656, 783, 910, 1037, 1164, 1291, 1418, 1545, 1672, 1799, 1926, 2053, 2180, 2307, 2434, 2561, 2688, 2816, 2689, 2562,
    2435, 2308, 2181, 2054, 1927, 1800, 1673, 1546, 1419, 1292, 1165, 1038, 911, 784, 657, 530, 403, 276, 149, 22, 23, 150, 277, 404, 531, 658, 785, 912, 1039, 1166, 1293, 1420, 1547, 1674, 1801, 1928, 2055, 2182, 2309, 2436, 2563, 2690, 2817, 2944, 3072, 2945, 2818, 2691, 2564, 2437, 2310, 2183, 2056, 1929, 1802, 1675, 1548, 1421, 1294, 1167, 1040, 913, 786, 659, 532, 405, 278, 151, 24, 25, 152, 279, 406, 533, 660, 787, 914, 1041, 1168, 1295, 1422, 1549, 1676, 1803, 1930, 2057, 2184, 2311, 2438, 2565, 2692, 2819, 2946, 3073, 3200, 3328, 3201, 3074, 2947, 2820, 2693, 2566, 2439, 2312, 2185, 2058, 1931, 1804, 1677, 1550, 1423, 1296, 1169, 1042, 915, 788, 661, 534, 407, 280, 153, 26, 27, 154, 281, 408, 535, 662,
    789, 916, 1043, 1170, 1297, 1424, 1551, 1678, 1805, 1932, 2059, 2186, 2313, 2440, 2567, 2694, 2821, 2948, 3075, 3202, 3329, 3456, 3584, 3457, 3330, 3203, 3076, 2949, 2822, 2695, 2568, 2441, 2314, 2187, 2060, 1933, 1806, 1679, 1552, 1425, 1298, 1171, 1044, 917, 790, 663, 536, 409, 282, 155, 28, 29, 156, 283, 410, 537, 664, 791, 918, 1045, 1172, 1299, 1426, 1553, 1680, 1807, 1934, 2061, 2188, 2315, 2442, 2569, 2696, 2823, 2950, 3077, 3204, 3331, 3458, 3585, 3712, 3840, 3713, 3586, 3459, 3332, 3205, 3078, 2951, 2824, 2697, 2570, 2443, 2316, 2189, 2062, 1935, 1808, 1681, 1554, 1427, 1300, 1173, 1046, 919, 792, 665, 538, 411, 284, 157, 30, 31, 158, 285, 412, 539, 666, 793, 920, 1047, 1174, 1301, 1428, 1555, 1682, 1809, 1936,
    2063, 2190, 2317, 2444, 2571, 2698, 2825, 2952, 3079, 3206, 3333, 3460, 3587, 3714, 3841, 3968, 3969, 3842, 3715, 3588, 3461, 3334, 3207, 3080, 2953, 2826, 2699, 2572, 2445, 2318, 2191, 2064, 1937, 1810, 1683, 1556, 1429, 1302, 1175, 1048, 921, 794, 667, 540, 413, 286, 159, 32, 33, 160, 287, 414, 541, 668, 795, 922, 1049, 1176, 1303, 1430, 1557, 1684, 1811, 1938, 2065, 2192, 2319, 2446, 2573, 2700, 2827, 2954, 3081, 3208, 3335, 3462, 3589, 3716, 3843, 3970, 3971, 3844, 3717, 3590, 3463, 3336, 3209, 3082, 2955, 2828, 2701, 2574, 2447, 2320, 2193, 2066, 1939, 1812, 1685, 1558, 1431, 1304, 1177, 1050, 923, 796, 669, 542, 415, 288, 161, 34, 35, 162, 289, 416, 543, 670, 797, 924, 1051, 1178, 1305, 1432, 1559, 1686, 1813, 1940,
    2067, 2194, 2321, 2448, 2575, 2702, 2829, 2956, 3083, 3210, 3337, 3464, 3591, 3718, 3845, 3972, 3973, 3846, 3719, 3592, 3465, 3338, 3211, 3084, 2957, 2830, 2703, 2576, 2449, 2322, 2195, 2068, 1941, 1814, 1687, 1560, 1433, 1306, 1179, 1052, 925, 798, 671, 544, 417, 290, 163, 36, 37, 164, 291, 418, 545, 672, 799, 926, 1053, 1180, 1307, 1434, 1561, 1688, 1815, 1942, 2069, 2196, 2323, 2450, 2577, 2704, 2831, 2958, 3085, 3212, 3339, 3466, 3593, 3720, 3847, 3974, 3975, 3848, 3721, 3594, 3467, 3340, 3213, 3086, 2959, 2832, 2705, 2578, 2451, 2324, 2197, 2070, 1943, 1816, 1689, 1562, 1435, 1308, 1181, 1054, 927, 800, 673, 546, 419, 292, 165, 38, 39, 166, 293, 420, 547, 674, 801, 928, 1055, 1182, 1309, 1436, 1563, 1690, 1817, 1944,
    2071, 2198, 2325, 2452, 2579, 2706, 2833, 2960, 3087, 3214, 3341, 3468, 3595, 3722, 3849, 3976, 3977, 3850, 3723, 3596, 3469, 3342, 3215, 3088, 2961, 2834, 2707, 2580, 2453, 2326, 2199, 2072, 1945, 1818, 1691, 1564, 1437, 1310, 1183, 1056, 929, 802, 675, 548, 421, 294, 167, 40, 41, 168, 295, 422, 549, 676, 803, 930, 1057, 1184, 1311, 1438, 1565, 1692, 1819, 1946, 2073, 2200, 2327, 2454, 2581, 2708, 2835, 2962, 3089, 3216, 3343, 3470, 3597, 3724, 3851, 3978, 3979, 3852, 3725, 3598, 3471, 3344, 3217, 3090, 2963, 2836, 2709, 2582, 2455, 2328, 2201, 2074, 1947, 1820, 1693, 1566, 1439, 1312, 1185, 1058, 931, 804, 677, 550, 423, 296, 169, 42, 43, 170, 297, 424, 551, 678, 805, 932, 1059, 1186, 1313, 1440, 1567, 1694, 1821, 1948,
    2075, 2202, 2329, 2456, 2583, 2710, 2837, 2964, 3091, 3218, 3345, 3472, 3599, 3726, 3853, 3980, 3981, 3854, 3727, 3600, 3473, 3346, 3219, 3092, 2965, 2838, 2711, 2584, 2457, 2330, 2203, 2076, 1949, 1822, 1695, 1568, 1441, 1314, 1187, 1060, 933, 806, 679, 552, 425, 298, 171, 44, 45, 172, 299, 426, 553, 680, 807, 934, 1061, 1188, 1315, 1442, 1569, 1696, 1823, 1950, 2077, 2204, 2331, 2458, 2585, 2712, 2839, 2966, 3093, 3220, 3347, 3474, 3601, 3728, 3855, 3982, 3983, 3856, 3729, 3602, 3475, 3348, 3221, 3094, 2967, 2840, 2713, 2586, 2459, 2332, 2205, 2078, 1951, 1824, 1697, 1570, 1443, 1316, 1189, 1062, 935, 808, 681, 554, 427, 300, 173, 46, 47, 174, 301, 428, 555, 682, 809, 936, 1063, 1190, 1317, 1444, 1571, 1698, 1825, 1952,
    2079, 2206, 2333, 2460, 2587, 2714, 2841, 2968, 3095, 3222, 3349, 3476, 3603, 3730, 3857, 3984, 3985, 3858, 3731, 3604, 3477, 3350, 3223, 3096, 2969, 2842, 2715, 2588, 2461, 2334, 2207, 2080, 1953, 1826, 1699, 1572, 1445, 1318, 1191, 1064, 937, 810, 683, 556, 429, 302, 175, 48, 49, 176, 303, 430, 557, 684, 811, 938, 1065, 1192, 1319, 1446, 1573, 1700, 1827, 1954, 2081, 2208, 2335, 2462, 2589, 2716, 2843, 2970, 3097, 3224, 3351, 3478, 3605, 3732, 3859, 3986, 3987, 3860, 3733, 3606, 3479, 3352, 3225, 3098, 2971, 2844, 2717, 2590, 2463, 2336, 2209, 2082, 1955, 1828, 1701, 1574, 1447, 1320, 1193, 1066, 939, 812, 685, 558, 431, 304, 177, 50, 51, 178, 305, 432, 559, 686, 813, 940, 1067, 1194, 1321, 1448, 1575, 1702, 1829, 1956,
    2083, 2210, 2337, 2464, 2591, 2718, 2845, 2972, 3099, 3226, 3353, 3480, 3607, 3734, 3861, 3988, 3989, 3862, 3735, 3608, 3481, 3354, 3227, 3100, 2973, 2846, 2719, 2592, 2465, 2338, 2211, 2084, 1957, 1830, 1703, 1576, 1449, 1322, 1195, 1068, 941, 814, 687, 560, 433, 306, 179, 52, 53, 180, 307, 434, 561, 688, 815, 942, 1069, 1196, 1323, 1450, 1577, 1704, 1831, 1958, 2085, 2212, 2339, 2466, 2593, 2720, 2847, 2974, 3101, 3228, 3355, 3482, 3609, 3736, 3863, 3990, 3991, 3864, 3737, 3610, 3483, 3356, 3229, 3102, 2975, 2848, 2721, 2594, 2467, 2340, 2213, 2086, 1959, 1832, 1705, 1578, 1451, 1324, 1197, 1070, 943, 816, 689, 562, 435, 308, 181, 54, 55, 182, 309, 436, 563, 690, 817, 944, 1071, 1198, 1325, 1452, 1579, 1706, 1833, 1960,
    2087, 2214, 2341, 2468, 2595, 2722, 2849, 2976, 3103, 3230, 3357, 3484, 3611, 3738, 3865, 3992, 3993, 3866, 3739, 3612, 3485, 3358, 3231, 3104, 2977, 2850, 2723, 2596, 2469, 2342, 2215, 2088, 1961, 1834, 1707, 1580, 1453, 1326, 1199, 1072, 945, 818, 691, 564, 437, 310, 183, 56, 57, 184, 311, 438, 565, 692, 819, 946, 1073, 1200, 1327, 1454, 1581, 1708, 1835, 1962, 2089, 2216, 2343, 2470, 2597, 2724, 2851, 2978, 3105, 3232, 3359, 3486, 3613, 3740, 3867, 3994, 3995, 3868, 3741, 3614, 3487, 3360, 3233, 3106, 2979, 2852, 2725, 2598, 2471, 2344, 2217, 2090, 1963, 1836, 1709, 1582, 1455, 1328, 1201, 1074, 947, 820, 693, 566, 439, 312, 185, 58, 59, 186, 313, 440, 567, 694, 821, 948, 1075, 1202, 1329, 1456, 1583, 1710, 1837, 1964,
    2091, 2218, 2345, 2472, 2599, 2726, 2853, 2980, 3107, 3234, 3361, 3488, 3615, 3742, 3869, 3996, 3997, 3870, 3743, 3616, 3489, 3362, 3235, 3108, 2981, 2854, 2727, 2600, 2473, 2346, 2219, 2092, 1965, 1838, 1711, 1584, 1457, 1330, 1203, 1076, 949, 822, 695, 568, 441, 314, 187, 60, 61, 188, 315, 442, 569, 696, 823, 950, 1077, 1204, 1331, 1458, 1585, 1712, 1839, 1966, 2093, 2220, 2347, 2474, 2601, 2728, 2855, 2982, 3109, 3236, 3363, 3490, 3617, 3744, 3871, 3998, 3999, 3872, 3745, 3618, 3491, 3364, 3237, 3110, 2983, 2856, 2729, 2602, 2475, 2348, 2221, 2094, 1967, 1840, 1713, 1586, 1459, 1332, 1205, 1078, 951, 824, 697, 570, 443, 316, 189, 62, 63, 190, 317, 444, 571, 698, 825, 952, 1079, 1206, 1333, 1460, 1587, 1714, 1841, 1968,
    2095, 2222, 2349, 2476, 2603, 2730, 2857, 2984, 3111, 3238, 3365, 3492, 3619, 3746, 3873, 4000, 4001, 3874, 3747, 3620, 3493, 3366, 3239, 3112, 2985, 2858, 2731, 2604, 2477, 2350, 2223, 2096, 1969, 1842, 1715, 1588, 1461, 1334, 1207, 1080, 953, 826, 699, 572, 445, 318, 191, 64, 65, 192, 319, 446, 573, 700, 827, 954, 1081, 1208, 1335, 1462, 1589, 1716, 1843, 1970, 2097, 2224, 2351, 2478, 2605, 2732, 2859, 2986, 3113, 3240, 3367, 3494, 3621, 3748, 3875, 4002, 4003, 3876, 3749, 3622, 3495, 3368, 3241, 3114, 2987, 2860, 2733, 2606, 2479, 2352, 2225, 2098, 1971, 1844, 1717, 1590, 1463, 1336, 1209, 1082, 955, 828, 701, 574, 447, 320, 193, 66, 67, 194, 321, 448, 575, 702, 829, 956, 1083, 1210, 1337, 1464, 1591, 1718, 1845, 1972,
    2099, 2226, 2353, 2480, 2607, 2734, 2861, 2988, 3115, 3242, 3369, 3496, 3623, 3750, 3877, 4004, 4005, 3878, 3751, 3624, 3497, 3370, 3243, 3116, 2989, 2862, 2735, 2608, 2481, 2354, 2227, 2100, 1973, 1846, 1719, 1592, 1465, 1338, 1211, 1084, 957, 830, 703, 576, 449, 322, 195, 68, 69, 196, 323, 450, 577, 704, 831, 958, 1085, 1212, 1339, 1466, 1593, 1720, 1847, 1974, 2101, 2228, 2355, 2482, 2609, 2736, 2863, 2990, 3117, 3244, 3371, 3498, 3625, 3752, 3879, 4006, 4007, 3880, 3753, 3626, 3499, 3372, 3245, 3118, 2991, 2864, 2737, 2610, 2483, 2356, 2229, 2102, 1975, 1848, 1721, 1594, 1467, 1340, 1213, 1086, 959, 832, 705, 578, 451, 324, 197, 70, 71, 198, 325, 452, 579, 706, 833, 960, 1087, 1214, 1341, 1468, 1595, 1722, 1849, 1976,
    2103, 2230, 2357, 2484, 2611, 2738, 2865, 2992, 3119, 3246, 3373, 3500, 3627, 3754, 3881, 4008, 4009, 3882, 3755, 3628, 3501, 3374, 3247, 3120, 2993, 2866, 2739, 2612, 2485, 2358, 2231, 2104, 1977, 1850, 1723, 1596, 1469, 1342, 1215, 1088, 961, 834, 707, 580, 453, 326, 199, 72, 73, 200, 327, 454, 581, 708, 835, 962, 1089, 1216, 1343, 1470, 1597, 1724, 1851, 1978, 2105, 2232, 2359, 2486, 2613, 2740, 2867, 2994, 3121, 3248, 3375, 3502, 3629, 3756, 3883, 4010, 4011, 3884, 3757, 3630, 3503, 3376, 3249, 3122, 2995, 2868, 2741, 2614, 2487, 2360, 2233, 2106, 1979, 1852, 1725, 1598, 1471, 1344, 1217, 1090, 963, 836, 709, 582, 455, 328, 201, 74, 75, 202, 329, 456, 583, 710, 837, 964, 1091, 1218, 1345, 1472, 1599, 1726, 1853, 1980,
    2107, 2234, 2361, 2488, 2615, 2742, 2869, 2996, 3123, 3250, 3377, 3504, 3631, 3758, 3885, 4012, 4013, 3886, 3759, 3632, 3505, 3378, 3251, 3124, 2997, 2870, 2743, 2616, 2489, 2362, 2235, 2108, 1981, 1854, 1727, 1600, 1473, 1346, 1219, 1092, 965, 838, 711, 584, 457, 330, 203, 76, 77, 204, 331, 458, 585, 712, 839, 966, 1093, 1220, 1347, 1474, 1601, 1728, 1855, 1982, 2109, 2236, 2363, 2490, 2617, 2744, 2871, 2998, 3125, 3252, 3379, 3506, 3633, 3760, 3887, 4014, 4015, 3888, 3761, 3634, 3507, 3380, 3253, 3126, 2999, 2872, 2745, 2618, 2491, 2364, 2237, 2110, 1983, 1856, 1729, 1602, 1475, 1348, 1221, 1094, 967, 840, 713, 586, 459, 332, 205, 78, 79, 206, 333, 460, 587, 714, 841, 968, 1095, 1222, 1349, 1476, 1603, 1730, 1857, 1984,
    2111, 2238, 2365, 2492, 2619, 2746, 2873, 3000, 3127, 3254, 3381, 3508, 3635, 3762, 3889, 4016, 4017, 3890, 3763, 3636, 3509, 3382, 3255, 3128, 3001, 2874, 2747, 2620, 2493, 2366, 2239, 2112, 1985, 1858, 1731, 1604, 1477, 1350, 1223, 1096, 969, 842, 715, 588, 461, 334, 207, 80, 81, 208, 335, 462, 589, 716, 843, 970, 1097, 1224, 1351, 1478, 1605, 1732, 1859, 1986, 2113, 2240, 2367, 2494, 2621, 2748, 2875, 3002, 3129, 3256, 3383, 3510, 3637, 3764, 3891, 4018, 4019, 3892, 3765, 3638, 3511, 3384, 3257, 3130, 3003, 2876, 2749, 2622, 2495, 2368, 2241, 2114, 1987, 1860, 1733, 1606, 1479, 1352, 1225, 1098, 971, 844, 717, 590, 463, 336, 209, 82, 83, 210, 337, 464, 591, 718, 845, 972, 1099, 1226, 1353, 1480, 1607, 1734, 1861, 1988,
    2115, 2242, 2369, 2496, 2623, 2750, 2877, 3004, 3131, 3258, 3385, 3512, 3639, 3766, 3893, 4020, 4021, 3894, 3767, 3640, 3513, 3386, 3259, 3132, 3005, 2878, 2751, 2624, 2497, 2370, 2243, 2116, 1989, 1862, 1735, 1608, 1481, 1354, 1227, 1100, 973, 846, 719, 592, 465, 338, 211, 84, 85, 212, 339, 466, 593, 720, 847, 974, 1101, 1228, 1355, 1482, 1609, 1736, 1863, 1990, 2117, 2244, 2371, 2498, 2625, 2752, 2879, 3006, 3133, 3260, 3387, 3514, 3641, 3768, 3895, 4022, 4023, 3896, 3769, 3642, 3515, 3388, 3261, 3134, 3007, 2880, 2753, 2626, 2499, 2372, 2245, 2118, 1991, 1864, 1737, 1610, 1483, 1356, 1229, 1102, 975, 848, 721, 594, 467, 340, 213, 86, 87, 214, 341, 468, 595, 722, 849, 976, 1103, 1230, 1357, 1484, 1611, 1738, 1865, 1992,
    2119, 2246, 2373, 2500, 2627, 2754, 2881, 3008, 3135, 3262, 3389, 3516, 3643, 3770, 3897, 4024, 4025, 3898, 3771, 3644, 3517, 3390, 3263, 3136, 3009, 2882, 2755, 2628, 2501, 2374, 2247, 2120, 1993, 1866, 1739, 1612, 1485, 1358, 1231, 1104, 977, 850, 723, 596, 469, 342, 215, 88, 89, 216, 343, 470, 597, 724, 851, 978, 1105, 1232, 1359, 1486, 1613, 1740, 1867, 1994, 2121, 2248, 2375, 2502, 2629, 2756, 2883, 3010, 3137, 3264, 3391, 3518, 3645, 3772, 3899, 4026, 4027, 3900, 3773, 3646, 3519, 3392, 3265, 3138, 3011, 2884, 2757, 2630, 2503, 2376, 2249, 2122, 1995, 1868, 1741, 1614, 1487, 1360, 1233, 1106, 979, 852, 725, 598, 471, 344, 217, 90, 91, 218, 345, 472, 599, 726, 853, 980, 1107, 1234, 1361, 1488, 1615, 1742, 1869, 1996,
    2123, 2250, 2377, 2504, 2631, 2758, 2885, 3012, 3139, 3266, 3393, 3520, 3647, 3774, 3901, 4028, 4029, 3902, 3775, 3648, 3521, 3394, 3267, 3140, 3013, 2886, 2759, 2632, 2505, 2378, 2251, 2124, 1997, 1870, 1743, 1616, 1489, 1362, 1235, 1108, 981, 854, 727, 600, 473, 346, 219, 92, 93, 220, 347, 474, 601, 728, 855, 982, 1109, 1236, 1363, 1490, 1617, 1744, 1871, 1998, 2125, 2252, 2379, 2506, 2633, 2760, 2887, 3014, 3141, 3268, 3395, 3522, 3649, 3776, 3903, 4030, 4031, 3904, 3777, 3650, 3523, 3396, 3269, 3142, 3015, 2888, 2761, 2634, 2507, 2380, 2253, 2126, 1999, 1872, 1745, 1618, 1491, 1364, 1237, 1110, 983, 856, 729, 602, 475, 348, 221, 94, 95, 222, 349, 476, 603, 730, 857, 984, 1111, 1238, 1365, 1492, 1619, 1746, 1873, 2000,
    2127, 2254, 2381, 2508, 2635, 2762, 2889, 3016, 3143, 3270, 3397, 3524, 3651, 3778, 3905, 4032, 4033, 3906, 3779, 3652, 3525, 3398, 3271, 3144, 3017, 2890, 2763, 2636, 2509, 2382, 2255, 2128, 2001, 1874, 1747, 1620, 1493, 1366, 1239, 1112, 985, 858, 731, 604, 477, 350, 223, 96, 97, 224, 351, 478, 605, 732, 859, 986, 1113, 1240, 1367, 1494, 1621, 1748, 1875, 2002, 2129, 2256, 2383, 2510, 2637, 2764, 2891, 3018, 3145, 3272, 3399, 3526, 3653, 3780, 3907, 4034, 4035, 3908, 3781, 3654, 3527, 3400, 3273, 3146, 3019, 2892, 2765, 2638, 2511, 2384, 2257, 2130, 2003, 1876, 1749, 1622, 1495, 1368, 1241, 1114, 987, 860, 733, 606, 479, 352, 225, 98, 99, 226, 353, 480, 607, 734, 861, 988, 1115, 1242, 1369, 1496, 1623, 1750, 1877, 2004,
    2131, 2258, 2385, 2512, 2639, 2766, 2893, 3020, 3147, 3274, 3401, 3528, 3655, 3782, 3909, 4036, 4037, 3910, 3783, 3656, 3529, 3402, 3275, 3148, 3021, 2894, 2767, 2640, 2513, 2386, 2259, 2132, 2005, 1878, 1751, 1624, 1497, 1370, 1243, 1116, 989, 862, 735, 608, 481, 354, 227, 100, 101, 228, 355, 482, 609, 736, 863, 990, 1117, 1244, 1371, 1498, 1625, 1752, 1879, 2006, 2133, 2260, 2387, 2514, 2641, 2768, 2895, 3022, 3149, 3276, 3403, 3530, 3657, 3784, 3911, 4038, 4039, 3912, 3785, 3658, 3531, 3404, 3277, 3150, 3023, 2896, 2769, 2642, 2515, 2388, 2261, 2134, 2007, 1880, 1753, 1626, 1499, 1372, 1245, 1118, 991, 864, 737, 610, 483, 356, 229, 102, 103, 230, 357, 484, 611, 738, 865, 992, 1119, 1246, 1373, 1500, 1627, 1754, 1881, 2008,
    2135, 2262, 2389, 2516, 2643, 2770, 2897, 3024, 3151, 3278, 3405, 3532, 3659, 3786, 3913, 4040, 4041, 3914, 3787, 3660, 3533, 3406, 3279, 3152, 3025, 2898, 2771, 2644, 2517, 2390, 2263, 2136, 2009, 1882, 1755, 1628, 1501, 1374, 1247, 1120, 993, 866, 739, 612, 485, 358, 231, 104, 105, 232, 359, 486, 613, 740, 867, 994, 1121, 1248, 1375, 1502, 1629, 1756, 1883, 2010, 2137, 2264, 2391, 2518, 2645, 2772, 2899, 3026, 3153, 3280, 3407, 3534, 3661, 3788, 3915, 4042, 4043, 3916, 3789, 3662, 3535, 3408, 3281, 3154, 3027, 2900, 2773, 2646, 2519, 2392, 2265, 2138, 2011, 1884, 1757, 1630, 1503, 1376, 1249, 1122, 995, 868, 741, 614, 487, 360, 233, 106, 107, 234, 361, 488, 615, 742, 869, 996, 1123, 1250, 1377, 1504, 1631, 1758, 1885, 2012,
    2139, 2266, 2393, 2520, 2647, 2774, 2901, 3028, 3155, 3282, 3409, 3536, 3663, 3790, 3917, 4044, 4045, 3918, 3791, 3664, 3537, 3410, 3283, 3156, 3029, 2902, 2775, 2648, 2521, 2394, 2267, 2140, 2013, 1886, 1759, 1632, 1505, 1378, 1251, 1124, 997, 870, 743, 616, 489, 362, 235, 108, 109, 236, 363, 490, 617, 744, 871, 998, 1125, 1252, 1379, 1506, 1633, 1760, 1887, 2014, 2141, 2268, 2395, 2522, 2649, 2776, 2903, 3030, 3157, 3284, 3411, 3538, 3665, 3792, 3919, 4046, 4047, 3920, 3793, 3666, 3539, 3412, 3285, 3158, 3031, 2904, 2777, 2650, 2523, 2396, 2269, 2142, 2015, 1888, 1761, 1634, 1507, 1380, 1253, 1126, 999, 872, 745, 618, 491, 364, 237, 110, 111, 238, 365, 492, 619, 746, 873, 1000, 1127, 1254, 1381, 1508, 1635, 1762, 1889, 2016,
    2143, 2270, 2397, 2524, 2651, 2778, 2905, 3032, 3159, 3286, 3413, 3540, 3667, 3794, 3921, 4048, 4049, 3922, 3795, 3668, 3541, 3414, 3287, 3160, 3033, 2906, 2779, 2652, 2525, 2398, 2271, 2144, 2017, 1890, 1763, 1636, 1509, 1382, 1255, 1128, 1001, 874, 747, 620, 493, 366, 239, 112, 113, 240, 367, 494, 621, 748, 875, 1002, 1129, 1256, 1383, 1510, 1637, 1764, 1891, 2018, 2145, 2272, 2399, 2526, 2653, 2780, 2907, 3034, 3161, 3288, 3415, 3542, 3669, 3796, 3923, 4050, 4051, 3924, 3797, 3670, 3543, 3416, 3289, 3162, 3035, 2908, 2781, 2654, 2527, 2400, 2273, 2146, 2019, 1892, 1765, 1638, 1511, 1384, 1257, 1130, 1003, 876, 749, 622, 495, 368, 241, 114, 115, 242, 369, 496, 623, 750, 877, 1004, 1131, 1258, 1385, 1512, 1639, 1766, 1893, 2020,
    2147, 2274, 2401, 2528, 2655, 2782, 2909, 3036, 3163, 3290, 3417, 3544, 3671, 3798, 3925, 4052, 4053, 3926, 3799, 3672, 3545, 3418, 3291, 3164, 3037, 2910, 2783, 2656, 2529, 2402, 2275, 2148, 2021, 1894, 1767, 1640, 1513, 1386, 1259, 1132, 1005, 878, 751, 624, 497, 370, 243, 116, 117, 244, 371, 498, 625, 752, 879, 1006, 1133, 1260, 1387, 1514, 1641, 1768, 1895, 2022, 2149, 2276, 2403, 2530, 2657, 2784, 2911, 3038, 3165, 3292, 3419, 3546, 3673, 3800, 3927, 4054, 4055, 3928, 3801, 3674, 3547, 3420, 3293, 3166, 3039, 2912, 2785, 2658, 2531, 2404, 2277, 2150, 2023, 1896, 1769, 1642, 1515, 1388, 1261, 1134, 1007, 880, 753, 626, 499, 372, 245, 118, 119, 246, 373, 500, 627, 754, 881, 1008, 1135, 1262, 1389, 1516, 1643, 1770, 1897, 2024,
    2151, 2278, 2405, 2532, 2659, 2786, 2913, 3040, 3167, 3294, 3421, 3548, 3675, 3802, 3929, 4056, 4057, 3930, 3803, 3676, 3549, 3422, 3295, 3168, 3041, 2914, 2787, 2660, 2533, 2406, 2279, 2152, 2025, 1898, 1771, 1644, 1517, 1390, 1263, 1136, 1009, 882, 755, 628, 501, 374, 247, 120, 121, 248, 375, 502, 629, 756, 883, 1010, 1137, 1264, 1391, 1518, 1645, 1772, 1899, 2026, 2153, 2280, 2407, 2534, 2661, 2788, 2915, 3042, 3169, 3296, 3423, 3550, 3677, 3804, 3931, 4058, 4059, 3932, 3805, 3678, 3551, 3424, 3297, 3170, 3043, 2916, 2789, 2662, 2535, 2408, 2281, 2154, 2027, 1900, 1773, 1646, 1519, 1392, 1265, 1138, 1011, 884, 757, 630, 503, 376, 249, 122, 123, 250, 377, 504, 631, 758, 885, 1012, 1139, 1266, 1393, 1520, 1647, 1774, 1901, 2028,
    2155, 2282, 2409, 2536, 2663, 2790, 2917, 3044, 3171, 3298, 3425, 3552, 3679, 3806, 3933, 4060, 4061, 3934, 3807, 3680, 3553, 3426, 3299, 3172, 3045, 2918, 2791, 2664, 2537, 2410, 2283, 2156, 2029, 1902, 1775, 1648, 1521, 1394, 1267, 1140, 1013, 886, 759, 632, 505, 378, 251, 124, 125, 252, 379, 506, 633, 760, 887, 1014, 1141, 1268, 1395, 1522, 1649, 1776, 1903, 2030, 2157, 2284, 2411, 2538, 2665, 2792, 2919, 3046, 3173, 3300, 3427, 3554, 3681, 3808, 3935, 4062, 4063, 3936, 3809, 3682, 3555, 3428, 3301, 3174, 3047, 2920, 2793, 2666, 2539, 2412, 2285, 2158, 2031, 1904, 1777, 1650, 1523, 1396, 1269, 1142, 1015, 888, 761, 634, 507, 380, 253, 126, 127, 254, 381, 508, 635, 762, 889, 1016, 1143, 1270, 1397, 1524, 1651, 1778, 1905, 2032,
    2159, 2286, 2413, 2540, 2667, 2794, 2921, 3048, 3175, 3302, 3429, 3556, 3683, 3810, 3937, 4064, 4065, 3938, 3811, 3684, 3557, 3430, 3303, 3176, 3049, 2922, 2795, 2668, 2541, 2414, 2287, 2160, 2033, 1906, 1779, 1652, 1525, 1398, 1271, 1144, 1017, 890, 763, 636, 509, 382, 255, 383, 510, 637, 764, 891, 1018, 1145, 1272, 1399, 1526, 1653, 1780, 1907, 2034, 2161, 2288, 2415, 2542, 2669, 2796, 2923, 3050, 3177, 3304, 3431, 3558, 3685, 3812, 3939, 4066, 4067, 3940, 3813, 3686, 3559, 3432, 3305, 3178, 3051, 2924, 2797, 2670, 2543, 2416, 2289, 2162, 2035, 1908, 1781, 1654, 1527, 1400, 1273, 1146, 1019, 892, 765, 638, 511, 639, 766, 893, 1020, 1147, 1274, 1401, 1528, 1655, 1782, 1909, 2036, 2163, 2290, 2417, 2544, 2671, 2798, 2925, 3052, 3179, 3306,
    3433, 3560, 3687, 3814, 3941, 4068, 4069, 3942, 3815, 3688, 3561, 3434, 3307, 3180, 3053, 2926, 2799, 2672, 2545, 2418, 2291, 2164, 2037, 1910, 1783, 1656, 1529, 1402, 1275, 1148, 1021, 894, 767, 895, 1022, 1149, 1276, 1403, 1530, 1657, 1784, 1911, 2038, 2165, 2292, 2419, 2546, 2673, 2800, 2927, 3054, 3181, 3308, 3435, 3562, 3689, 3816, 3943, 4070, 4071, 3944, 3817, 3690, 3563, 3436, 3309, 3182, 3055, 2928, 2801, 2674, 2547, 2420, 2293, 2166, 2039, 1912, 1785, 1658, 1531, 1404, 1277, 1150, 1023, 1151, 1278, 1405, 1532, 1659, 1786, 1913, 2040, 2167, 2294, 2421, 2548, 2675, 2802, 2929, 3056, 3183, 3310, 3437, 3564, 3691, 3818, 3945, 4072, 4073, 3946, 3819, 3692, 3565, 3438, 3311, 3184, 3057, 2930, 2803, 2676, 2549, 2422, 2295, 2168, 2041, 1914, 1787, 1660,
    1533, 1406, 1279, 1407, 1534, 1661, 1788, 1915, 2042, 2169, 2296, 2423, 2550, 2677, 2804, 2931, 3058, 3185, 3312, 3439, 3566, 3693, 3820, 3947, 4074, 4075, 3948, 3821, 3694, 3567, 3440, 3313, 3186, 3059, 2932, 2805, 2678, 2551, 2424, 2297, 2170, 2043, 1916, 1789, 1662, 1535, 1663, 1790, 1917, 2044, 2171, 2298, 2425, 2552, 2679, 2806, 2933, 3060, 3187, 3314, 3441, 3568, 3695, 3822, 3949, 4076, 4077, 3950, 3823, 3696, 3569, 3442, 3315, 3188, 3061, 2934, 2807, 2680, 2553, 2426, 2299, 2172, 2045, 1918, 1791, 1919, 2046, 2173, 2300, 2427, 2554, 2681, 2808, 2935, 3062, 3189, 3316, 3443, 3570, 3697, 3824, 3951, 4078, 4079, 3952, 3825, 3698, 3571, 3444, 3317, 3190, 3063, 2936, 2809, 2682, 2555, 2428, 2301, 2174, 2047, 2175, 2302, 2429, 2556, 2683, 2810, 2937, 3064,
    3191, 3318, 3445, 3572, 3699, 3826, 3953, 4080, 4081, 3954, 3827, 3700, 3573, 3446, 3319, 3192, 3065, 2938, 2811, 2684, 2557, 2430, 2303, 2431, 2558, 2685, 2812, 2939, 3066, 3193, 3320, 3447, 3574, 3701, 3828, 3955, 4082, 4083, 3956, 3829, 3702, 3575, 3448, 3321, 3194, 3067, 2940, 2813, 2686, 2559, 2687, 2814, 2941, 3068, 3195, 3322, 3449, 3576, 3703, 3830, 3957, 4084, 4085, 3958, 3831, 3704, 3577, 3450, 3323, 3196, 3069, 2942, 2815, 2943, 3070, 3197, 3324, 3451, 3578, 3705, 3832, 3959, 4086, 4087, 3960, 3833, 3706, 3579, 3452, 3325, 3198, 3071, 3199, 3326, 3453, 3580, 3707, 3834, 3961, 4088, 4089, 3962, 3835, 3708, 3581, 3454, 3327, 3455, 3582, 3709, 3836, 3963, 4090, 4091, 3964, 3837, 3710, 3583, 3711, 3838, 3965, 4092, 4093, 3966, 3839, 3967, 4094, 4095,

    // 64 x 2:
    0, 1, 2, 4, 3, 5, 6, 8, 7, 9, 10, 12, 11, 13, 14, 16, 15, 17, 18, 20, 19, 21, 22, 24, 23, 25, 26, 28, 27, 29, 30, 32, 31, 33, 34, 36, 35, 37, 38, 40, 39, 41, 42, 44, 43, 45, 46, 48, 47, 49, 50, 52, 51, 53, 54, 56, 55, 57, 58, 60, 59, 61, 62, 64, 63, 65, 66, 68, 67, 69, 70, 72, 71, 73, 74, 76, 75, 77, 78, 80, 79, 81, 82, 84, 83, 85, 86, 88, 87, 89, 90, 92, 91, 93, 94, 96, 95, 97, 98, 100, 99, 101, 102, 104, 103, 105, 106, 108, 107, 109, 110, 112, 111, 113, 114, 116, 115, 117, 118, 120, 119, 121, 122, 124, 123, 125, 126, 127,

    // 64 x 4:
    0, 1, 4, 8, 5, 2, 3, 6, 9, 12, 16, 13, 10, 7, 11, 14, 17, 20, 24, 21, 18, 15, 19, 22, 25, 28, 32, 29, 26, 23, 27, 30, 33, 36, 40, 37, 34, 31, 35, 38, 41, 44, 48, 45, 42, 39, 43, 46, 49, 52, 56, 53, 50, 47, 51, 54, 57, 60, 64, 61, 58, 55, 59, 62, 65, 68, 72, 69, 66, 63, 67, 70, 73, 76, 80, 77, 74, 71, 75, 78, 81, 84, 88, 85, 82, 79, 83, 86, 89, 92, 96, 93, 90, 87, 91, 94, 97, 100, 104, 101, 98, 95, 99, 102, 105, 108, 112, 109, 106, 103, 107, 110, 113, 116, 120, 117, 114, 111, 115, 118, 121, 124, 128, 125, 122, 119, 123, 126,
    129, 132, 136, 133, 130, 127, 131, 134, 137, 140, 144, 141, 138, 135, 139, 142, 145, 148, 152, 149, 146, 143, 147, 150, 153, 156, 160, 157, 154, 151, 155, 158, 161, 164, 168, 165, 162, 159, 163, 166, 169, 172, 176, 173, 170, 167, 171, 174, 177, 180, 184, 181, 178, 175, 179, 182, 185, 188, 192, 189, 186, 183, 187, 190, 193, 196, 200, 197, 194, 191, 195, 198, 201, 204, 208, 205, 202, 199, 203, 206, 209, 212, 216, 213, 210, 207, 211, 214, 217, 220, 224, 221, 218, 215, 219, 222, 225, 228, 232, 229, 226, 223, 227, 230, 233, 236, 240, 237, 234, 231, 235, 238, 241, 244, 248, 245, 242, 239, 243, 246, 249, 252, 253, 250, 247, 251, 254, 255,

    // 64 x 8:
    0, 1, 8, 16, 9, 2, 3, 10, 17, 24, 32, 25, 18, 11, 4, 5, 12, 19, 26, 33, 40, 48, 41, 34, 27, 20, 13, 6, 7, 14, 21, 28, 35, 42, 49, 56, 64, 57, 50, 43, 36, 29, 22, 15, 23, 30, 37, 44, 51, 58, 65, 72, 80, 73, 66, 59, 52, 45, 38, 31, 39, 46, 53, 60, 67, 74, 81, 88, 96, 89, 82, 75, 68, 61, 54, 47, 55, 62, 69, 76, 83, 90, 97, 104, 112, 105, 98, 91, 84, 77, 70, 63, 71, 78, 85, 92, 99, 106, 113, 120, 128, 121, 114, 107, 100, 93, 86, 79, 87, 94, 101, 108, 115, 122, 129, 136, 144, 137, 130, 123, 116, 109, 102, 95, 103, 110, 117, 124,
    131, 138, 145, 152, 160, 153, 146, 139, 132, 125, 118, 111, 119, 126, 133, 140, 147, 154, 161, 168, 176, 169, 162, 155, 148, 141, 134, 127, 135, 142, 149, 156, 163, 170, 177, 184, 192, 185, 178, 171, 164, 157, 150, 143, 151, 158, 165, 172, 179, 186, 193, 200, 208, 201, 194, 187, 180, 173, 166, 159, 167, 174, 181, 188, 195, 202, 209, 216, 224, 217, 210, 203, 196, 189, 182, 175, 183, 190, 197, 204, 211, 218, 225, 232, 240, 233, 226, 219, 212, 205, 198, 191, 199, 206, 213, 220, 227, 234, 241, 248, 256, 249, 242, 235, 228, 221, 214, 207, 215, 222, 229, 236, 243, 250, 257, 264, 272, 265, 258, 251, 244, 237, 230, 223, 231, 238, 245, 252,
    259, 266, 273, 280, 288, 281, 274, 267, 260, 253, 246, 239, 247, 254, 261, 268, 275, 282, 289, 296, 304, 297, 290, 283, 276, 269, 262, 255, 263, 270, 277, 284, 291, 298, 305, 312, 320, 313, 306, 299, 292, 285, 278, 271, 279, 286, 293, 300, 307, 314, 321, 328, 336, 329, 322, 315, 308, 301, 294, 287, 295, 302, 309, 316, 323, 330, 337, 344, 352, 345, 338, 331, 324, 317, 310, 303, 311, 318, 325, 332, 339, 346, 353, 360, 368, 361, 354, 347, 340, 333, 326, 319, 327, 334, 341, 348, 355, 362, 369, 376, 384, 377, 370, 363, 356, 349, 342, 335, 343, 350, 357, 364, 371, 378, 385, 392, 400, 393, 386, 379, 372, 365, 358, 351, 359, 366, 373, 380,
    387, 394, 401, 408, 416, 409, 402, 395, 388, 381, 374, 367, 375, 382, 389, 396, 403, 410, 417, 424, 432, 425, 418, 411, 404, 397, 390, 383, 391, 398, 405, 412, 419, 426, 433, 440, 448, 441, 434, 427, 420, 413, 406, 399, 407, 414, 421, 428, 435, 442, 449, 456, 464, 457, 450, 443, 436, 429, 422, 415, 423, 430, 437, 444, 451, 458, 465, 472, 480, 473, 466, 459, 452, 445, 438, 431, 439, 446, 453, 460, 467, 474, 481, 488, 496, 489, 482, 475, 468, 461, 454, 447, 455, 462, 469, 476, 483, 490, 497, 504, 505, 498, 491, 484, 477, 470, 463, 471, 478, 485, 492, 499, 506, 507, 500, 493, 486, 479, 487, 494, 501, 508, 509, 502, 495, 503, 510, 511,

    // 64 x 16:
    0, 1, 16, 32, 17, 2, 3, 18, 33, 48, 64, 49, 34, 19, 4, 5, 20, 35, 50, 65, 80, 96, 81, 66, 51, 36, 21, 6, 7, 22, 37, 52, 67, 82, 97, 112, 128, 113, 98, 83, 68, 53, 38, 23, 8, 9, 24, 39, 54, 69, 84, 99, 114, 129, 144, 160, 145, 130, 115, 100, 85, 70, 55, 40, 25, 10, 11, 26, 41, 56, 71, 86, 101, 116, 131, 146, 161, 176, 192, 177, 162, 147, 132, 117, 102, 87, 72, 57, 42, 27, 12, 13, 28, 43, 58, 73, 88, 103, 118, 133, 148, 163, 178, 193, 208, 224, 209, 194, 179, 164, 149, 134, 119, 104, 89, 74, 59, 44, 29, 14, 15, 30, 45, 60, 75, 90, 105, 120,
    135, 150, 165, 180, 195, 210, 225, 240, 256, 241, 226, 211, 196, 181, 166, 151, 136, 121, 106, 91, 76, 61, 46, 31, 47, 62, 77, 92, 107, 122, 137, 152, 167, 182, 197, 212, 227, 242, 257, 272, 288, 273, 258, 243, 228, 213, 198, 183, 168, 153, 138, 123, 108, 93, 78, 63, 79, 94, 109, 124, 139, 154, 169, 184, 199, 214, 229, 244, 259, 274, 289, 304, 320, 305, 290, 275, 260, 245, 230, 215, 200, 185, 170, 155, 140, 125, 110, 95, 111, 126, 141, 156, 171, 186, 201, 216, 231, 246, 261, 276, 291, 306, 321, 336, 352, 337, 322, 307, 292, 277, 262, 247, 232, 217, 202, 187, 172, 157, 142, 127, 143, 158, 173, 188, 203, 218, 233, 248,
    263, 278, 293, 308, 323, 338, 353, 368, 384, 369, 354, 339, 324, 309, 294, 279, 264, 249, 234, 219, 204, 189, 174, 159, 175, 190, 205, 220, 235, 250, 265, 280, 295, 310, 325, 340, 355, 370, 385, 400, 416, 401, 386, 371, 356, 341, 326, 311, 296, 281, 266, 251, 236, 221, 206, 191, 207, 222, 237, 252, 267, 282, 297, 312, 327, 342, 357, 372, 387, 402, 417, 432, 448, 433, 418, 403, 388, 373, 358, 343, 328, 313, 298, 283, 268, 253, 238, 223, 239, 254, 269, 284, 299, 314, 329, 344, 359, 374, 389, 404, 419, 434, 449, 464, 480, 465, 450, 435, 420, 405, 390, 375, 360, 345, 330, 315, 300, 285, 270, 255, 271, 286, 301, 316, 331, 346, 361, 376,
    391, 406, 421, 436, 451, 466, 481, 496, 512, 497, 482, 467, 452, 437, 422, 407, 392, 377, 362, 347, 332, 317, 302, 287, 303, 318, 333, 348, 363, 378, 393, 408, 423, 438, 453, 468, 483, 498, 513, 528, 544, 529, 514, 499, 484, 469, 454, 439, 424, 409, 394, 379, 364, 349, 334, 319, 335, 350, 365, 380, 395, 410, 425, 440, 455, 470, 485, 500, 515, 530, 545, 560, 576, 561, 546, 531, 516, 501, 486, 471, 456, 441, 426, 411, 396, 381, 366, 351, 367, 382, 397, 412, 427, 442, 457, 472, 487, 502, 517, 532, 547, 562, 577, 592, 608, 593, 578, 563, 548, 533, 518, 503, 488, 473, 458, 443, 428, 413, 398, 383, 399, 414, 429, 444, 459, 474, 489, 504,
    519, 534, 549, 564, 579, 594, 609, 624, 640, 625, 610, 595, 580, 565, 550, 535, 520, 505, 490, 475, 460, 445, 430, 415, 431, 446, 461, 476, 491, 506, 521, 536, 551, 566, 581, 596, 611, 626, 641, 656, 672, 657, 642, 627, 612, 597, 582, 567, 552, 537, 522, 507, 492, 477, 462, 447, 463, 478, 493, 508, 523, 538, 553, 568, 583, 598, 613, 628, 643, 658, 673, 688, 704, 689, 674, 659, 644, 629, 614, 599, 584, 569, 554, 539, 524, 509, 494, 479, 495, 510, 525, 540, 555, 570, 585, 600, 615, 630, 645, 660, 675, 690, 705, 720, 736, 721, 706, 691, 676, 661, 646, 631, 616, 601, 586, 571, 556, 541, 526, 511, 527, 542, 557, 572, 587, 602, 617, 632,
    647, 662, 677, 692, 707, 722, 737, 752, 768, 753, 738, 723, 708, 693, 678, 663, 648, 633, 618, 603, 588, 573, 558, 543, 559, 574, 589, 604, 619, 634, 649, 664, 679, 694, 709, 724, 739, 754, 769, 784, 800, 785, 770, 755, 740, 725, 710, 695, 680, 665, 650, 635, 620, 605, 590, 575, 591, 606, 621, 636, 651, 666, 681, 696, 711, 726, 741, 756, 771, 786, 801, 816, 832, 817, 802, 787, 772, 757, 742, 727, 712, 697, 682, 667, 652, 637, 622, 607, 623, 638, 653, 668, 683, 698, 713, 728, 743, 758, 773, 788, 803, 818, 833, 848, 864, 849, 834, 819, 804, 789, 774, 759, 744, 729, 714, 699, 684, 669, 654, 639, 655, 670, 685, 700, 715, 730, 745, 760,
    775, 790, 805, 820, 835, 850, 865, 880, 896, 881, 866, 851, 836, 821, 806, 791, 776, 761, 746, 731, 716, 701, 686, 671, 687, 702, 717, 732, 747, 762, 777, 792, 807, 822, 837, 852, 867, 882, 897, 912, 928, 913, 898, 883, 868, 853, 838, 823, 808, 793, 778, 763, 748, 733, 718, 703, 719, 734, 749, 764, 779, 794, 809, 824, 839, 854, 869, 884, 899, 914, 929, 944, 960, 945, 930, 915, 900, 885, 870, 855, 840, 825, 810, 795, 780, 765, 750, 735, 751, 766, 781, 796, 811, 826, 841, 856, 871, 886, 901, 916, 931, 946, 961, 976, 992, 977, 962, 947, 932, 917, 902, 887, 872, 857, 842, 827, 812, 797, 782, 767, 783, 798, 813, 828, 843, 858, 873, 888,
    903, 918, 933, 948, 963, 978, 993, 1008, 1009, 994, 979, 964, 949, 934, 919, 904, 889, 874, 859, 844, 829, 814, 799, 815, 830, 845, 860, 875, 890, 905, 920, 935, 950, 965, 980, 995, 1010, 1011, 996, 981, 966, 951, 936, 921, 906, 891, 876, 861, 846, 831, 847, 862, 877, 892, 907, 922, 937, 952, 967, 982, 997, 1012, 1013, 998, 983, 968, 953, 938, 923, 908, 893, 878, 863, 879, 894, 909, 924, 939, 954, 969, 984, 999, 1014, 1015, 1000, 985, 970, 955, 940, 925, 910, 895, 911, 926, 941, 956, 971, 986, 1001, 1016, 1017, 1002, 987, 972, 957, 942, 927, 943, 958, 973, 988, 1003, 1018, 1019, 1004, 989, 974, 959, 975, 990, 1005, 1020, 1021, 1006, 991, 1007, 1022, 1023,

    // 64 x 32:
    0, 1, 32, 64, 33, 2, 3, 34, 65, 96, 128, 97, 66, 35, 4, 5, 36, 67, 98, 129, 160, 192, 161, 130, 99, 68, 37, 6, 7, 38, 69, 100, 131, 162, 193, 224, 256, 225, 194, 163, 132, 101, 70, 39, 8, 9, 40, 71, 102, 133, 164, 195, 226, 257, 288, 320, 289, 258, 227, 196, 165, 134, 103, 72, 41, 10, 11, 42, 73, 104, 135, 166, 197, 228, 259, 290, 321, 352, 384, 353, 322, 291, 260, 229, 198, 167, 136, 105, 74, 43, 12, 13, 44, 75, 106, 137, 168, 199, 230, 261, 292, 323, 354, 385, 416, 448, 417, 386, 355, 324, 293, 262, 231, 200, 169, 138, 107, 76, 45, 14, 15, 46, 77, 108, 139, 170, 201, 232,
    263, 294, 325, 356, 387, 418, 449, 480, 512, 481, 450, 419, 388, 357, 326, 295, 264, 233, 202, 171, 140, 109, 78, 47, 16, 17, 48, 79, 110, 141, 172, 203, 234, 265, 296, 327, 358, 389, 420, 451, 482, 513, 544, 576, 545, 514, 483, 452, 421, 390, 359, 328, 297, 266, 235, 204, 173, 142, 111, 80, 49, 18, 19, 50, 81, 112, 143, 174, 205, 236, 267, 298, 329, 360, 391, 422, 453, 484, 515, 546, 577, 608, 640, 609, 578, 547, 516, 485, 454, 423, 392, 361, 330, 299, 268, 237, 206, 175, 144, 113, 82, 51, 20, 21, 52, 83, 114, 145, 176, 207, 238, 269, 300, 331, 362, 393, 424, 455, 486, 517, 548, 579, 610, 641, 672, 704, 673, 642,
    611, 580, 549, 518, 487, 456, 425, 394, 363, 332, 301, 270, 239, 208, 177, 146, 115, 84, 53, 22, 23, 54, 85, 116, 147, 178, 209, 240, 271, 302, 333, 364, 395, 426, 457, 488, 519, 550, 581, 612, 643, 674, 705, 736, 768, 737, 706, 675, 644, 613, 582, 551, 520, 489, 458, 427, 396, 365, 334, 303, 272, 241, 210, 179, 148, 117, 86, 55, 24, 25, 56, 87, 118, 149, 180, 211, 242, 273, 304, 335, 366, 397, 428, 459, 490, 521, 552, 583, 614, 645, 676, 707, 738, 769, 800, 832, 801, 770, 739, 708, 677, 646, 615, 584, 553, 522, 491, 460, 429, 398, 367, 336, 305, 274, 243, 212, 181, 150, 119, 88, 57, 26, 27, 58, 89, 120, 151, 182,
    213, 244, 275, 306, 337, 368, 399, 430, 461, 492, 523, 554, 585, 616, 647, 678, 709, 740, 771, 802, 833, 864, 896, 865, 834, 803, 772, 741, 710, 679, 648, 617, 586, 555, 524, 493, 462, 431, 400, 369, 338, 307, 276, 245, 214, 183, 152, 121, 90, 59, 28, 29, 60, 91, 122, 153, 184, 215, 246, 277, 308, 339, 370, 401, 432, 463, 494, 525, 556, 587, 618, 649, 680, 711, 742, 773, 804, 835, 866, 897, 928, 960, 929, 898, 867, 836, 805, 774, 743, 712, 681, 650, 619, 588, 557, 526, 495, 464, 433, 402, 371, 340, 309, 278, 247, 216, 185, 154, 123, 92, 61, 30, 31, 62, 93, 124, 155, 186, 217, 248, 279, 310, 341, 372, 403, 434, 465, 496,
    527, 558, 589, 620, 651, 682, 713, 744, 775, 806, 837, 868, 899, 930, 961, 992, 1024, 993, 962, 931, 900, 869, 838, 807, 776, 745, 714, 683, 652, 621, 590, 559, 528, 497, 466, 435, 404, 373, 342, 311, 280, 249, 218, 187, 156, 125, 94, 63, 95, 126, 157, 188, 219, 250, 281, 312, 343, 374, 405, 436, 467, 498, 529, 560, 591, 622, 653, 684, 715, 746, 777, 808, 839, 870, 901, 932, 963, 994, 1025, 1056, 1088, 1057, 1026, 995, 964, 933, 902, 871, 840, 809, 778, 747, 716, 685, 654, 623, 592, 561, 530, 499, 468, 437, 406, 375, 344, 313, 282, 251, 220, 189, 158, 127, 159, 190, 221, 252, 283, 314, 345, 376, 407, 438, 469, 500, 531, 562, 593, 624,
    655, 686, 717, 748, 779, 810, 841, 872, 903, 934, 965, 996, 1027, 1058, 1089, 1120, 1152, 1121, 1090, 1059, 1028, 997, 966, 935, 904, 873, 842, 811, 780, 749, 718, 687, 656, 625, 594, 563, 532, 501, 470, 439, 408, 377, 346, 315, 284, 253, 222, 191, 223, 254, 285, 316, 347, 378, 409, 440, 471, 502, 533, 564, 595, 626, 657, 688, 719, 750, 781, 812, 843, 874, 905, 936, 967, 998, 1029, 1060, 1091, 1122, 1153, 1184, 1216, 1185, 1154, 1123, 1092, 1061, 1030, 999, 968, 937, 906, 875, 844, 813, 782, 751, 720, 689, 658, 627, 596, 565, 534, 503, 472, 441, 410, 379, 348, 317, 286, 255, 287, 318, 349, 380, 411, 442, 473, 504, 535, 566, 597, 628, 659, 690, 721, 752,
    783, 814, 845, 876, 907, 938, 969, 1000, 1031, 1062, 1093, 1124, 1155, 1186, 1217, 1248, 1280, 1249, 1218, 1187, 1156, 1125, 1094, 1063, 1032, 1001, 970, 939, 908, 877, 846, 815, 784, 753, 722, 691, 660, 629, 598, 567, 536, 505, 474, 443, 412, 381, 350, 319, 351, 382, 413, 444, 475, 506, 537, 568, 599, 630, 661, 692, 723, 754, 785, 816, 847, 878, 909, 940, 971, 1002, 1033, 1064, 1095, 1126, 1157, 1188, 1219, 1250, 1281, 1312, 1344, 1313, 1282, 1251, 1220, 1189, 1158, 1127, 1096, 1065, 1034, 1003, 972, 941, 910, 879, 848, 817, 786, 755, 724, 693, 662, 631, 600, 569, 538, 507, 476, 445, 414, 383, 415, 446, 477, 508, 539, 570, 601, 632, 663, 694, 725, 756, 787, 818, 849, 880,
    911, 942, 973, 1004, 1035, 1066, 1097, 1128, 1159, 1190, 1221, 1252, 1283, 1314, 1345, 1376, 1408, 1377, 1346, 1315, 1284, 1253, 1222, 1191, 1160, 1129, 1098, 1067, 1036, 1005, 974, 943, 912, 881, 850, 819, 788, 757, 726, 695, 664, 633, 602, 571, 540, 509, 478, 447, 479, 510, 541, 572, 603, 634, 665, 696, 727, 758, 789, 820, 851, 882, 913, 944, 975, 1006, 1037, 1068, 1099, 1130, 1161, 1192, 1223, 1254, 1285, 1316, 1347, 1378, 1409, 1440, 1472, 1441, 1410, 1379, 1348, 1317, 1286, 1255, 1224, 1193, 1162, 1131, 1100, 1069, 1038, 1007, 976, 945, 914, 883, 852, 821, 790, 759, 728, 697, 666, 635, 604, 573, 542, 511, 543, 574, 605, 636, 667, 698, 729, 760, 791, 822, 853, 884, 915, 946, 977, 1008,
    1039, 1070, 1101, 1132, 1163, 1194, 1225, 1256, 1287, 1318, 1349, 1380, 1411, 1442, 1473, 1504, 1536, 1505, 1474, 1443, 1412, 1381, 1350, 1319, 1288, 1257, 1226, 1195, 1164, 1133, 1102, 1071, 1040, 1009, 978, 947, 916, 885, 854, 823, 792, 761, 730, 699, 668, 637, 606, 575, 607, 638, 669, 700, 731, 762, 793, 824, 855, 886, 917, 948, 979, 1010, 1041, 1072, 1103, 1134, 1165, 1196, 1227, 1258, 1289, 1320, 1351, 1382, 1413, 1444, 1475, 1506, 1537, 1568, 1600, 1569, 1538, 1507, 1476, 1445, 1414, 1383, 1352, 1321, 1290, 1259, 1228, 1197, 1166, 1135, 1104, 1073, 1042, 1011, 980, 949, 918, 887, 856, 825, 794, 763, 732, 701, 670, 639, 671, 702, 733, 764, 795, 826, 857, 888, 919, 950, 981, 1012, 1043, 1074, 1105, 1136,
    1167, 1198, 1229, 1260, 1291, 1322, 1353, 1384, 1415, 1446, 1477, 1508, 1539, 1570, 1601, 1632, 1664, 1633, 1602, 1571, 1540, 1509, 1478, 1447, 1416, 1385, 1354, 1323, 1292, 1261, 1230, 1199, 1168, 1137, 1106, 1075, 1044, 1013, 982, 951, 920, 889, 858, 827, 796, 765, 734, 703, 735, 766, 797, 828, 859, 890, 921, 952, 983, 1014, 1045, 1076, 1107, 1138, 1169, 1200, 1231, 1262, 1293, 1324, 1355, 1386, 1417, 1448, 1479, 1510, 1541, 1572, 1603, 1634, 1665, 1696, 1728, 1697, 1666, 1635, 1604, 1573, 1542, 1511, 1480, 1449, 1418, 1387, 1356, 1325, 1294, 1263, 1232, 1201, 1170, 1139, 1108, 1077, 1046, 1015, 984, 953, 922, 891, 860, 829, 798, 767, 799, 830, 861, 892, 923, 954, 985, 1016, 1047, 1078, 1109, 1140, 1171, 1202, 1233, 1264,
    1295, 1326, 1357, 1388, 1419, 1450, 1481, 1512, 1543, 1574, 1605, 1636, 1667, 1698, 1729, 1760, 1792, 1761, 1730, 1699, 1668, 1637, 1606, 1575, 1544, 1513, 1482, 1451, 1420, 1389, 1358, 1327, 1296, 1265, 1234, 1203, 1172, 1141, 1110, 1079, 1048, 1017, 986, 955, 924, 893, 862, 831, 863, 894, 925, 956, 987, 1018, 1049, 1080, 1111, 1142, 1173, 1204, 1235, 1266, 1297, 1328, 1359, 1390, 1421, 1452, 1483, 1514, 1545, 1576, 1607, 1638, 1669, 1700, 1731, 1762, 1793, 1824, 1856, 1825, 1794, 1763, 1732, 1701, 1670, 1639, 1608, 1577, 1546, 1515, 1484, 1453, 1422, 1391, 1360, 1329, 1298, 1267, 1236, 1205, 1174, 1143, 1112, 1081, 1050, 1019, 988, 957, 926, 895, 927, 958, 989, 1020, 1051, 1082, 1113, 1144, 1175, 1206, 1237, 1268, 1299, 1330, 1361, 1392,
    1423, 1454, 1485, 1516, 1547, 1578, 1609, 1640, 1671, 1702, 1733, 1764, 1795, 1826, 1857, 1888, 1920, 1889, 1858, 1827, 1796, 1765, 1734, 1703, 1672, 1641, 1610, 1579, 1548, 1517, 1486, 1455, 1424, 1393, 1362, 1331, 1300, 1269, 1238, 1207, 1176, 1145, 1114, 1083, 1052, 1021, 990, 959, 991, 1022, 1053, 1084, 1115, 1146, 1177, 1208, 1239, 1270, 1301, 1332, 1363, 1394, 1425, 1456, 1487, 1518, 1549, 1580, 1611, 1642, 1673, 1704, 1735, 1766, 1797, 1828, 1859, 1890, 1921, 1952, 1984, 1953, 1922, 1891, 1860, 1829, 1798, 1767, 1736, 1705, 1674, 1643, 1612, 1581, 1550, 1519, 1488, 1457, 1426, 1395, 1364, 1333, 1302, 1271, 1240, 1209, 1178, 1147, 1116, 1085, 1054, 1023, 1055, 1086, 1117, 1148, 1179, 1210, 1241, 1272, 1303, 1334, 1365, 1396, 1427, 1458, 1489, 1520,
    1551, 1582, 1613, 1644, 1675, 1706, 1737, 1768, 1799, 1830, 1861, 1892, 1923, 1954, 1985, 2016, 2017, 1986, 1955, 1924, 1893, 1862, 1831, 1800, 1769, 1738, 1707, 1676, 1645, 1614, 1583, 1552, 1521, 1490, 1459, 1428, 1397, 1366, 1335, 1304, 1273, 1242, 1211, 1180, 1149, 1118, 1087, 1119, 1150, 1181, 1212, 1243, 1274, 1305, 1336, 1367, 1398, 1429, 1460, 1491, 1522, 1553, 1584, 1615, 1646, 1677, 1708, 1739, 1770, 1801, 1832, 1863, 1894, 1925, 1956, 1987, 2018, 2019, 1988, 1957, 1926, 1895, 1864, 1833, 1802, 1771, 1740, 1709, 1678, 1647, 1616, 1585, 1554, 1523, 1492, 1461, 1430, 1399, 1368, 1337, 1306, 1275, 1244, 1213, 1182, 1151, 1183, 1214, 1245, 1276, 1307, 1338, 1369, 1400, 1431, 1462, 1493, 1524, 1555, 1586, 1617, 1648, 1679, 1710, 1741, 1772, 1803, 1834,
    1865, 1896, 1927, 1958, 1989, 2020, 2021, 1990, 1959, 1928, 1897, 1866, 1835, 1804, 1773, 1742, 1711, 1680, 1649, 1618, 1587, 1556, 1525, 1494, 1463, 1432, 1401, 1370, 1339, 1308, 1277, 1246, 1215, 1247, 1278, 1309, 1340, 1371, 1402, 1433, 1464, 1495, 1526, 1557, 1588, 1619, 1650, 1681, 1712, 1743, 1774, 1805, 1836, 1867, 1898, 1929, 1960, 1991, 2022, 2023, 1992, 1961, 1930, 1899, 1868, 1837, 1806, 1775, 1744, 1713, 1682, 1651, 1620, 1589, 1558, 1527, 1496, 1465, 1434, 1403, 1372, 1341, 1310, 1279, 1311, 1342, 1373, 1404, 1435, 1466, 1497, 1528, 1559, 1590, 1621, 1652, 1683, 1714, 1745, 1776, 1807, 1838, 1869, 1900, 1931, 1962, 1993, 2024, 2025, 1994, 1963, 1932, 1901, 1870, 1839, 1808, 1777, 1746, 1715, 1684, 1653, 1622, 1591, 1560, 1529, 1498, 1467, 1436,
    1405, 1374, 1343, 1375, 1406, 1437, 1468, 1499, 1530, 1561, 1592, 1623, 1654, 1685, 1716, 1747, 1778, 1809, 1840, 1871, 1902, 1933, 1964, 1995, 2026, 2027, 1996, 1965, 1934, 1903, 1872, 1841, 1810, 1779, 1748, 1717, 1686, 1655, 1624, 1593, 1562, 1531, 1500, 1469, 1438, 1407, 1439, 1470, 1501, 1532, 1563, 1594, 1625, 1656, 1687, 1718, 1749, 1780, 1811, 1842, 1873, 1904, 1935, 1966, 1997, 2028, 2029, 1998, 1967, 1936, 1905, 1874, 1843, 1812, 1781, 1750, 1719, 1688, 1657, 1626, 1595, 1564, 1533, 1502, 1471, 1503, 1534, 1565, 1596, 1627, 1658, 1689, 1720, 1751, 1782, 1813, 1844, 1875, 1906, 1937, 1968, 1999, 2030, 2031, 2000, 1969, 1938, 1907, 1876, 1845, 1814, 1783, 1752, 1721, 1690, 1659, 1628, 1597, 1566, 1535, 1567, 1598, 1629, 1660, 1691, 1722, 1753, 1784,
    1815, 1846, 1877, 1908, 1939, 1970, 2001, 2032, 2033, 2002, 1971, 1940, 1909, 1878, 1847, 1816, 1785, 1754, 1723, 1692, 1661, 1630, 1599, 1631, 1662, 1693, 1724, 1755, 1786, 1817, 1848, 1879, 1910, 1941, 1972, 2003, 2034, 2035, 2004, 1973, 1942, 1911, 1880, 1849, 1818, 1787, 1756, 1725, 1694, 1663, 1695, 1726, 1757, 1788, 1819, 1850, 1881, 1912, 1943, 1974, 2005, 2036, 2037, 2006, 1975, 1944, 1913, 1882, 1851, 1820, 1789, 1758, 1727, 1759, 1790, 1821, 1852, 1883, 1914, 1945, 1976, 2007, 2038, 2039, 2008, 1977, 1946, 1915, 1884, 1853, 1822, 1791, 1823, 1854, 1885, 1916, 1947, 1978, 2009, 2040, 2041, 2010, 1979, 1948, 1917, 1886, 1855, 1887, 1918, 1949, 1980, 2011, 2042, 2043, 2012, 1981, 1950, 1919, 1951, 1982, 2013, 2044, 2045, 2014, 1983, 2015, 2046, 2047,

    // 64 x 64:
    0, 1, 64, 128, 65, 2, 3, 66, 129, 192, 256, 193, 130, 67, 4, 5, 68, 131, 194, 257, 320, 384, 321, 258, 195, 132, 69, 6, 7, 70, 133, 196, 259, 322, 385, 448, 512, 449, 386, 323, 260, 197, 134, 71, 8, 9, 72, 135, 198, 261, 324, 387, 450, 513, 576, 640, 577, 514, 451, 388, 325, 262, 199, 136, 73, 10, 11, 74, 137, 200, 263, 326, 389, 452, 515, 578, 641, 704, 768, 705, 642, 579, 516, 453, 390, 327, 264, 201, 138, 75, 12, 13, 76, 139, 202, 265, 328, 391, 454, 517, 580, 643, 706, 769, 832, 896, 833, 770, 707, 644, 581, 518, 455, 392, 329, 266, 203, 140, 77, 14, 15, 78, 141, 204, 267, 330, 393, 456,
    519, 582, 645, 708, 771, 834, 897, 960, 1024, 961, 898, 835, 772, 709, 646, 583, 520, 457, 394, 331, 268, 205, 142, 79, 16, 17, 80, 143, 206, 269, 332, 395, 458, 521, 584, 647, 710, 773, 836, 899, 962, 1025, 1088, 1152, 1089, 1026, 963, 900, 837, 774, 711, 648, 585, 522, 459, 396, 333, 270, 207, 144, 81, 18, 19, 82, 145, 208, 271, 334, 397, 460, 523, 586, 649, 712, 775, 838, 901, 964, 1027, 1090, 1153, 1216, 1280, 1217, 1154, 1091, 1028, 965, 902, 839, 776, 713, 650, 587, 524, 461, 398, 335, 272, 209, 146, 83, 20, 21, 84, 147, 210, 273, 336, 399, 462, 525, 588, 651, 714, 777, 840, 903, 966, 1029, 1092, 1155, 1218, 1281, 1344, 1408, 1345, 1282,
    1219, 1156, 1093, 1030, 967, 904, 841, 778, 715, 652, 589, 526, 463, 400, 337, 274, 211, 148, 85, 22, 23, 86, 149, 212, 275, 338, 401, 464, 527, 590, 653, 716, 779, 842, 905, 968, 1031, 1094, 1157, 1220, 1283, 1346, 1409, 1472, 1536, 1473, 1410, 1347, 1284, 1221, 1158, 1095, 1032, 969, 906, 843, 780, 717, 654, 591, 528, 465, 402, 339, 276, 213, 150, 87, 24, 25, 88, 151, 214, 277, 340, 403, 466, 529, 592, 655, 718, 781, 844, 907, 970, 1033, 1096, 1159, 1222, 1285, 1348, 1411, 1474, 1537, 1600, 1664, 1601, 1538, 1475, 1412, 1349, 1286, 1223, 1160, 1097, 1034, 971, 908, 845, 782, 719, 656, 593, 530, 467, 404, 341, 278, 215, 152, 89, 26, 27, 90, 153, 216, 279, 342,
    405, 468, 531, 594, 657, 720, 783, 846, 909, 972, 1035, 1098, 1161, 1224, 1287, 1350, 1413, 1476, 1539, 1602, 1665, 1728, 1792, 1729, 1666, 1603, 1540, 1477, 1414, 1351, 1288, 1225, 1162, 1099, 1036, 973, 910, 847, 784, 721, 658, 595, 532, 469, 406, 343, 280, 217, 154, 91, 28, 29, 92, 155, 218, 281, 344, 407, 470, 533, 596, 659, 722, 785, 848, 911, 974, 1037, 1100, 1163, 1226, 1289, 1352, 1415, 1478, 1541, 1604, 1667, 1730, 1793, 1856, 1920, 1857, 1794, 1731, 1668, 1605, 1542, 1479, 1416, 1353, 1290, 1227, 1164, 1101, 1038, 975, 912, 849, 786, 723, 660, 597, 534, 471, 408, 345, 282, 219, 156, 93, 30, 31, 94, 157, 220, 283, 346, 409, 472, 535, 598, 661, 724, 787, 850, 913, 976,
    1039, 1102, 1165, 1228, 1291, 1354, 1417, 1480, 1543, 1606, 1669, 1732, 1795, 1858, 1921, 1984, 2048, 1985, 1922, 1859, 1796, 1733, 1670, 1607, 1544, 1481, 1418, 1355, 1292, 1229, 1166, 1103, 1040, 977, 914, 851, 788, 725, 662, 599, 536, 473, 410, 347, 284, 221, 158, 95, 32, 33, 96, 159, 222, 285, 348, 411, 474, 537, 600, 663, 726, 789, 852, 915, 978, 1041, 1104, 1167, 1230, 1293, 1356, 1419, 1482, 1545, 1608, 1671, 1734, 1797, 1860, 1923, 1986, 2049, 2112, 2176, 2113, 2050, 1987, 1924, 1861, 1798, 1735, 1672, 1609, 1546, 1483, 1420, 1357, 1294, 1231, 1168, 1105, 1042, 979, 916, 853, 790, 727, 664, 601, 538, 475, 412, 349, 286, 223, 160, 97, 34, 35, 98, 161, 224, 287, 350, 413, 476, 539, 602,
    665, 728, 791, 854, 917, 980, 1043, 1106, 1169, 1232, 1295, 1358, 1421, 1484, 1547, 1610, 1673, 1736, 1799, 1862, 1925, 1988, 2051, 2114, 2177, 2240, 2304, 2241, 2178, 2115, 2052, 1989, 1926, 1863, 1800, 1737, 1674, 1611, 1548, 1485, 1422, 1359, 1296, 1233, 1170, 1107, 1044, 981, 918, 855, 792, 729, 666, 603, 540, 477, 414, 351, 288, 225, 162, 99, 36, 37, 100, 163, 226, 289, 352, 415, 478, 541, 604, 667, 730, 793, 856, 919, 982, 1045, 1108, 1171, 1234, 1297, 1360, 1423, 1486, 1549, 1612, 1675, 1738, 1801, 1864, 1927, 1990, 2053, 2116, 2179, 2242, 2305, 2368, 2432, 2369, 2306, 2243, 2180, 2117, 2054, 1991, 1928, 1865, 1802, 1739, 1676, 1613, 1550, 1487, 1424, 1361, 1298, 1235, 1172, 1109, 1046, 983, 920, 857, 794,
    731, 668, 605, 542, 479, 416, 353, 290, 227, 164, 101, 38, 39, 102, 165, 228, 291, 354, 417, 480, 543, 606, 669, 732, 795, 858, 921, 984, 1047, 1110, 1173, 1236, 1299, 1362, 1425, 1488, 1551, 1614, 1677, 1740, 1803, 1866, 1929, 1992, 2055, 2118, 2181, 2244, 2307, 2370, 2433, 2496, 2560, 2497, 2434, 2371, 2308, 2245, 2182, 2119, 2056, 1993, 1930, 1867, 1804, 1741, 1678, 1615, 1552, 1489, 1426, 1363, 1300, 1237, 1174, 1111, 1048, 985, 922, 859, 796, 733, 670, 607, 544, 481, 418, 355, 292, 229, 166, 103, 40, 41, 104, 167, 230, 293, 356, 419, 482, 545, 608, 671, 734, 797, 860, 923, 986, 1049, 1112, 1175, 1238, 1301, 1364, 1427, 1490, 1553, 1616, 1679, 1742, 1805, 1868, 1931, 1994, 2057, 2120, 2183,
    2246, 2309, 2372, 2435, 2498, 2561, 2624, 2688, 2625, 2562, 2499, 2436, 2373, 2310, 2247, 2184, 2121, 2058, 1995, 1932, 1869, 1806, 1743, 1680, 1617, 1554, 1491, 1428, 1365, 1302, 1239, 1176, 1113, 1050, 987, 924, 861, 798, 735, 672, 609, 546, 483, 420, 357, 294, 231, 168, 105, 42, 43, 106, 169, 232, 295, 358, 421, 484, 547, 610, 673, 736, 799, 862, 925, 988, 1051, 1114, 1177, 1240, 1303, 1366, 1429, 1492, 1555, 1618, 1681, 1744, 1807, 1870, 1933, 1996, 2059, 2122, 2185, 2248, 2311, 2374, 2437, 2500, 2563, 2626, 2689, 2752, 2816, 2753, 2690, 2627, 2564, 2501, 2438, 2375, 2312, 2249, 2186, 2123, 2060, 1997, 1934, 1871, 1808, 1745, 1682, 1619, 1556, 1493, 1430, 1367, 1304, 1241, 1178, 1115, 1052, 989, 926, 863, 800, 737,
    674, 611, 548, 485, 422, 359, 296, 233, 170, 107, 44, 45, 108, 171, 234, 297, 360, 423, 486, 549, 612, 675, 738, 801, 864, 927, 990, 1053, 1116, 1179, 1242, 1305, 1368, 1431, 1494, 1557, 1620, 1683, 1746, 1809, 1872, 1935, 1998, 2061, 2124, 2187, 2250, 2313, 2376, 2439, 2502, 2565, 2628, 2691, 2754, 2817, 2880, 2944, 2881, 2818, 2755, 2692, 2629, 2566, 2503, 2440, 2377, 2314, 2251, 2188, 2125, 2062, 1999, 1936, 1873, 1810, 1747, 1684, 1621, 1558, 1495, 1432, 1369, 1306, 1243, 1180, 1117, 1054, 991, 928, 865, 802, 739, 676, 613, 550, 487, 424, 361, 298, 235, 172, 109, 46, 47, 110, 173, 236, 299, 362, 425, 488, 551, 614, 677, 740, 803, 866, 929, 992, 1055, 1118, 1181, 1244, 1307, 1370, 1433, 1496,
    1559, 1622, 1685, 1748, 1811, 1874, 1937, 2000, 2063, 2126, 2189, 2252, 2315, 2378, 2441, 2504, 2567, 2630, 2693, 2756, 2819, 2882, 2945, 3008, 3072, 3009, 2946, 2883, 2820, 2757, 2694, 2631, 2568, 2505, 2442, 2379, 2316, 2253, 2190, 2127, 2064, 2001, 1938, 1875, 1812, 1749, 1686, 1623, 1560, 1497, 1434, 1371, 1308, 1245, 1182, 1119, 1056, 993, 930, 867, 804, 741, 678, 615, 552, 489, 426, 363, 300, 237, 174, 111, 48, 49, 112, 175, 238, 301, 364, 427, 490, 553, 616, 679, 742, 805, 868, 931, 994, 1057, 1120, 1183, 1246, 1309, 1372, 1435, 1498, 1561, 1624, 1687, 1750, 1813, 1876, 1939, 2002, 2065, 2128, 2191, 2254, 2317, 2380, 2443, 2506, 2569, 2632, 2695, 2758, 2821, 2884, 2947, 3010, 3073, 3136, 3200, 3137, 3074, 3011, 2948,
    2885, 2822, 2759, 2696, 2633, 2570, 2507, 2444, 2381, 2318, 2255, 2192, 2129, 2066, 2003, 1940, 1877, 1814, 1751, 1688, 1625, 1562, 1499, 1436, 1373, 1310, 1247, 1184, 1121, 1058, 995, 932, 869, 806, 743, 680, 617, 554, 491, 428, 365, 302, 239, 176, 113, 50, 51, 114, 177, 240, 303, 366, 429, 492, 555, 618, 681, 744, 807, 870, 933, 996, 1059, 1122, 1185, 1248, 1311, 1374, 1437, 1500, 1563, 1626, 1689, 1752, 1815, 1878, 1941, 2004, 2067, 2130, 2193, 2256, 2319, 2382, 2445, 2508, 2571, 2634, 2697, 2760, 2823, 2886, 2949, 3012, 3075, 3138, 3201, 3264, 3328, 3265, 3202, 3139, 3076, 3013, 2950, 2887, 2824, 2761, 2698, 2635, 2572, 2509, 2446, 2383, 2320, 2257, 2194, 2131, 2068, 2005, 1942, 1879, 1816, 1753, 1690, 1627, 1564, 1501,
    1438, 1375, 1312, 1249, 1186, 1123, 1060, 997, 934, 871, 808, 745, 682, 619, 556, 493, 430, 367, 304, 241, 178, 115, 52, 53, 116, 179, 242, 305, 368, 431, 494, 557, 620, 683, 746, 809, 872, 935, 998, 1061, 1124, 1187, 1250, 1313, 1376, 1439, 1502, 1565, 1628, 1691, 1754, 1817, 1880, 1943, 2006, 2069, 2132, 2195, 2258, 2321, 2384, 2447, 2510, 2573, 2636, 2699, 2762, 2825, 2888, 2951, 3014, 3077, 3140, 3203, 3266, 3329, 3392, 3456, 3393, 3330, 3267, 3204, 3141, 3078, 3015, 2952, 2889, 2826, 2763, 2700, 2637, 2574, 2511, 2448, 2385, 2322, 2259, 2196, 2133, 2070, 2007, 1944, 1881, 1818, 1755, 1692, 1629, 1566, 1503, 1440, 1377, 1314, 1251, 1188, 1125, 1062, 999, 936, 873, 810, 747, 684, 621, 558, 495, 432, 369, 306,
    243, 180, 117, 54, 55, 118, 181, 244, 307, 370, 433, 496, 559, 622, 685, 748, 811, 874, 937, 1000, 1063, 1126, 1189, 1252, 1315, 1378, 1441, 1504, 1567, 1630, 1693, 1756, 1819, 1882, 1945, 2008, 2071, 2134, 2197, 2260, 2323, 2386, 2449, 2512, 2575, 2638, 2701, 2764, 2827, 2890, 2953, 3016, 3079, 3142, 3205, 3268, 3331, 3394, 3457, 3520, 3584, 3521, 3458, 3395, 3332, 3269, 3206, 3143, 3080, 3017, 2954, 2891, 2828, 2765, 2702, 2639, 2576, 2513, 2450, 2387, 2324, 2261, 2198, 2135, 2072, 2009, 1946, 1883, 1820, 1757, 1694, 1631, 1568, 1505, 1442, 1379, 1316, 1253, 1190, 1127, 1064, 1001, 938, 875, 812, 749, 686, 623, 560, 497, 434, 371, 308, 245, 182, 119, 56, 57, 120, 183, 246, 309, 372, 435, 498, 561, 624, 687,
    750, 813, 876, 939, 1002, 1065, 1128, 1191, 1254, 1317, 1380, 1443, 1506, 1569, 1632, 1695, 1758, 1821, 1884, 1947, 2010, 2073, 2136, 2199, 2262, 2325, 2388, 2451, 2514, 2577, 2640, 2703, 2766, 2829, 2892, 2955, 3018, 3081, 3144, 3207, 3270, 3333, 3396, 3459, 3522, 3585, 3648, 3712, 3649, 3586, 3523, 3460, 3397, 3334, 3271, 3208, 3145, 3082, 3019, 2956, 2893, 2830, 2767, 2704, 2641, 2578, 2515, 2452, 2389, 2326, 2263, 2200, 2137, 2074, 2011, 1948, 1885, 1822, 1759, 1696, 1633, 1570, 1507, 1444, 1381, 1318, 1255, 1192, 1129, 1066, 1003, 940, 877, 814, 751, 688, 625, 562, 499, 436, 373, 310, 247, 184, 121, 58, 59, 122, 185, 248, 311, 374, 437, 500, 563, 626, 689, 752, 815, 878, 941, 1004, 1067, 1130, 1193, 1256, 1319, 1382,
    1445, 1508, 1571, 1634, 1697, 1760, 1823, 1886, 1949, 2012, 2075, 2138, 2201, 2264, 2327, 2390, 2453, 2516, 2579, 2642, 2705, 2768, 2831, 2894, 2957, 3020, 3083, 3146, 3209, 3272, 3335, 3398, 3461, 3524, 3587, 3650, 3713, 3776, 3840, 3777, 3714, 3651, 3588, 3525, 3462, 3399, 3336, 3273, 3210, 3147, 3084, 3021, 2958, 2895, 2832, 2769, 2706, 2643, 2580, 2517, 2454, 2391, 2328, 2265, 2202, 2139, 2076, 2013, 1950, 1887, 1824, 1761, 1698, 1635, 1572, 1509, 1446, 1383, 1320, 1257, 1194, 1131, 1068, 1005, 942, 879, 816, 753, 690, 627, 564, 501, 438, 375, 312, 249, 186, 123, 60, 61, 124, 187, 250, 313, 376, 439, 502, 565, 628, 691, 754, 817, 880, 943, 1006, 1069, 1132, 1195, 1258, 1321, 1384, 1447, 1510, 1573, 1636, 1699, 1762, 1825,
    1888, 1951, 2014, 2077, 2140, 2203, 2266, 2329, 2392, 2455, 2518, 2581, 2644, 2707, 2770, 2833, 2896, 2959, 3022, 3085, 3148, 3211, 3274, 3337, 3400, 3463, 3526, 3589, 3652, 3715, 3778, 3841, 3904, 3968, 3905, 3842, 3779, 3716, 3653, 3590, 3527, 3464, 3401, 3338, 3275, 3212, 3149, 3086, 3023, 2960, 2897, 2834, 2771, 2708, 2645, 2582, 2519, 2456, 2393, 2330, 2267, 2204, 2141, 2078, 2015, 1952, 1889, 1826, 1763, 1700, 1637, 1574, 1511, 1448, 1385, 1322, 1259, 1196, 1133, 1070, 1007, 944, 881, 818, 755, 692, 629, 566, 503, 440, 377, 314, 251, 188, 125, 62, 63, 126, 189, 252, 315, 378, 441, 504, 567, 630, 693, 756, 819, 882, 945, 1008, 1071, 1134, 1197, 1260, 1323, 1386, 1449, 1512, 1575, 1638, 1701, 1764, 1827, 1890, 1953, 2016,
    2079, 2142, 2205, 2268, 2331, 2394, 2457, 2520, 2583, 2646, 2709, 2772, 2835, 2898, 2961, 3024, 3087, 3150, 3213, 3276, 3339, 3402, 3465, 3528, 3591, 3654, 3717, 3780, 3843, 3906, 3969, 4032, 4033, 3970, 3907, 3844, 3781, 3718, 3655, 3592, 3529, 3466, 3403, 3340, 3277, 3214, 3151, 3088, 3025, 2962, 2899, 2836, 2773, 2710, 2647, 2584, 2521, 2458, 2395, 2332, 2269, 2206, 2143, 2080, 2017, 1954, 1891, 1828, 1765, 1702, 1639, 1576, 1513, 1450, 1387, 1324, 1261, 1198, 1135, 1072, 1009, 946, 883, 820, 757, 694, 631, 568, 505, 442, 379, 316, 253, 190, 127, 191, 254, 317, 380, 443, 506, 569, 632, 695, 758, 821, 884, 947, 1010, 1073, 1136, 1199, 1262, 1325, 1388, 1451, 1514, 1577, 1640, 1703, 1766, 1829, 1892, 1955, 2018, 2081, 2144, 2207,
    2270, 2333, 2396, 2459, 2522, 2585, 2648, 2711, 2774, 2837, 2900, 2963, 3026, 3089, 3152, 3215, 3278, 3341, 3404, 3467, 3530, 3593, 3656, 3719, 3782, 3845, 3908, 3971, 4034, 4035, 3972, 3909, 3846, 3783, 3720, 3657, 3594, 3531, 3468, 3405, 3342, 3279, 3216, 3153, 3090, 3027, 2964, 2901, 2838, 2775, 2712, 2649, 2586, 2523, 2460, 2397, 2334, 2271, 2208, 2145, 2082, 2019, 1956, 1893, 1830, 1767, 1704, 1641, 1578, 1515, 1452, 1389, 1326, 1263, 1200, 1137, 1074, 1011, 948, 885, 822, 759, 696, 633, 570, 507, 444, 381, 318, 255, 319, 382, 445, 508, 571, 634, 697, 760, 823, 886, 949, 1012, 1075, 1138, 1201, 1264, 1327, 1390, 1453, 1516, 1579, 1642, 1705, 1768, 1831, 1894, 1957, 2020, 2083, 2146, 2209, 2272, 2335, 2398, 2461, 2524, 2587, 2650,
    2713, 2776, 2839, 2902, 2965, 3028, 3091, 3154, 3217, 3280, 3343, 3406, 3469, 3532, 3595, 3658, 3721, 3784, 3847, 3910, 3973, 4036, 4037, 3974, 3911, 3848, 3785, 3722, 3659, 3596, 3533, 3470, 3407, 3344, 3281, 3218, 3155, 3092, 3029, 2966, 2903, 2840, 2777, 2714, 2651, 2588, 2525, 2462, 2399, 2336, 2273, 2210, 2147, 2084, 2021, 1958, 1895, 1832, 1769, 1706, 1643, 1580, 1517, 1454, 1391, 1328, 1265, 1202, 1139, 1076, 1013, 950, 887, 824, 761, 698, 635, 572, 509, 446, 383, 447, 510, 573, 636, 699, 762, 825, 888, 951, 1014, 1077, 1140, 1203, 1266, 1329, 1392, 1455, 1518, 1581, 1644, 1707, 1770, 1833, 1896, 1959, 2022, 2085, 2148, 2211, 2274, 2337, 2400, 2463, 2526, 2589, 2652, 2715, 2778, 2841, 2904, 2967, 3030, 3093, 3156, 3219, 3282, 3345,
    3408, 3471, 3534, 3597, 3660, 3723, 3786, 3849, 3912, 3975, 4038, 4039, 3976, 3913, 3850, 3787, 3724, 3661, 3598, 3535, 3472, 3409, 3346, 3283, 3220, 3157, 3094, 3031, 2968, 2905, 2842, 2779, 2716, 2653, 2590, 2527, 2464, 2401, 2338, 2275, 2212, 2149, 2086, 2023, 1960, 1897, 1834, 1771, 1708, 1645, 1582, 1519, 1456, 1393, 1330, 1267, 1204, 1141, 1078, 1015, 952, 889, 826, 763, 700, 637, 574, 511, 575, 638, 701, 764, 827, 890, 953, 1016, 1079, 1142, 1205, 1268, 1331, 1394, 1457, 1520, 1583, 1646, 1709, 1772, 1835, 1898, 1961, 2024, 2087, 2150, 2213, 2276, 2339, 2402, 2465, 2528, 2591, 2654, 2717, 2780, 2843, 2906, 2969, 3032, 3095, 3158, 3221, 3284, 3347, 3410, 3473, 3536, 3599, 3662, 3725, 3788, 3851, 3914, 3977, 4040, 4041, 3978, 3915, 3852,
    3789, 3726, 3663, 3600, 3537, 3474, 3411, 3348, 3285, 3222, 3159, 3096, 3033, 2970, 2907, 2844, 2781, 2718, 2655, 2592, 2529, 2466, 2403, 2340, 2277, 2214, 2151, 2088, 2025, 1962, 1899, 1836, 1773, 1710, 1647, 1584, 1521, 1458, 1395, 1332, 1269, 1206, 1143, 1080, 1017, 954, 891, 828, 765, 702, 639, 703, 766, 829, 892, 955, 1018, 1081, 1144, 1207, 1270, 1333, 1396, 1459, 1522, 1585, 1648, 1711, 1774, 1837, 1900, 1963, 2026, 2089, 2152, 2215, 2278, 2341, 2404, 2467, 2530, 2593, 2656, 2719, 2782, 2845, 2908, 2971, 3034, 3097, 3160, 3223, 3286, 3349, 3412, 3475, 3538, 3601, 3664, 3727, 3790, 3853, 3916, 3979, 4042, 4043, 3980, 3917, 3854, 3791, 3728, 3665, 3602, 3539, 3476, 3413, 3350, 3287, 3224, 3161, 3098, 3035, 2972, 2909, 2846, 2783, 2720, 2657,
    2594, 2531, 2468, 2405, 2342, 2279, 2216, 2153, 2090, 2027, 1964, 1901, 1838, 1775, 1712, 1649, 1586, 1523, 1460, 1397, 1334, 1271, 1208, 1145, 1082, 1019, 956, 893, 830, 767, 831, 894, 957, 1020, 1083, 1146, 1209, 1272, 1335, 1398, 1461, 1524, 1587, 1650, 1713, 1776, 1839, 1902, 1965, 2028, 2091, 2154, 2217, 2280, 2343, 2406, 2469, 2532, 2595, 2658, 2721, 2784, 2847, 2910, 2973, 3036, 3099, 3162, 3225, 3288, 3351, 3414, 3477, 3540, 3603, 3666, 3729, 3792, 3855, 3918, 3981, 4044, 4045, 3982, 3919, 3856, 3793, 3730, 3667, 3604, 3541, 3478, 3415, 3352, 3289, 3226, 3163, 3100, 3037, 2974, 2911, 2848, 2785, 2722, 2659, 2596, 2533, 2470, 2407, 2344, 2281, 2218, 2155, 2092, 2029, 1966, 1903, 1840, 1777, 1714, 1651, 1588, 1525, 1462, 1399, 1336, 1273, 1210,
    1147, 1084, 1021, 958, 895, 959, 1022, 1085, 1148, 1211, 1274, 1337, 1400, 1463, 1526, 1589, 1652, 1715, 1778, 1841, 1904, 1967, 2030, 2093, 2156, 2219, 2282, 2345, 2408, 2471, 2534, 2597, 2660, 2723, 2786, 2849, 2912, 2975, 3038, 3101, 3164, 3227, 3290, 3353, 3416, 3479, 3542, 3605, 3668, 3731, 3794, 3857, 3920, 3983, 4046, 4047, 3984, 3921, 3858, 3795, 3732, 3669, 3606, 3543, 3480, 3417, 3354, 3291, 3228, 3165, 3102, 3039, 2976, 2913, 2850, 2787, 2724, 2661, 2598, 2535, 2472, 2409, 2346, 2283, 2220, 2157, 2094, 2031, 1968, 1905, 1842, 1779, 1716, 1653, 1590, 1527, 1464, 1401, 1338, 1275, 1212, 1149, 1086, 1023, 1087, 1150, 1213, 1276, 1339, 1402, 1465, 1528, 1591, 1654, 1717, 1780, 1843, 1906, 1969, 2032, 2095, 2158, 2221, 2284, 2347, 2410, 2473, 2536,
    2599, 2662, 2725, 2788, 2851, 2914, 2977, 3040, 3103, 3166, 3229, 3292, 3355, 3418, 3481, 3544, 3607, 3670, 3733, 3796, 3859, 3922, 3985, 4048, 4049, 3986, 3923, 3860, 3797, 3734, 3671, 3608, 3545, 3482, 3419, 3356, 3293, 3230, 3167, 3104, 3041, 2978, 2915, 2852, 2789, 2726, 2663, 2600, 2537, 2474, 2411, 2348, 2285, 2222, 2159, 2096, 2033, 1970, 1907, 1844, 1781, 1718, 1655, 1592, 1529, 1466, 1403, 1340, 1277, 1214, 1151, 1215, 1278, 1341, 1404, 1467, 1530, 1593, 1656, 1719, 1782, 1845, 1908, 1971, 2034, 2097, 2160, 2223, 2286, 2349, 2412, 2475, 2538, 2601, 2664, 2727, 2790, 2853, 2916, 2979, 3042, 3105, 3168, 3231, 3294, 3357, 3420, 3483, 3546, 3609, 3672, 3735, 3798, 3861, 3924, 3987, 4050, 4051, 3988, 3925, 3862, 3799, 3736, 3673, 3610, 3547, 3484, 3421,
    3358, 3295, 3232, 3169, 3106, 3043, 2980, 2917, 2854, 2791, 2728, 2665, 2602, 2539, 2476, 2413, 2350, 2287, 2224, 2161, 2098, 2035, 1972, 1909, 1846, 1783, 1720, 1657, 1594, 1531, 1468, 1405, 1342, 1279, 1343, 1406, 1469, 1532, 1595, 1658, 1721, 1784, 1847, 1910, 1973, 2036, 2099, 2162, 2225, 2288, 2351, 2414, 2477, 2540, 2603, 2666, 2729, 2792, 2855, 2918, 2981, 3044, 3107, 3170, 3233, 3296, 3359, 3422, 3485, 3548, 3611, 3674, 3737, 3800, 3863, 3926, 3989, 4052, 4053, 3990, 3927, 3864, 3801, 3738, 3675, 3612, 3549, 3486, 3423, 3360, 3297, 3234, 3171, 3108, 3045, 2982, 2919, 2856, 2793, 2730, 2667, 2604, 2541, 2478, 2415, 2352, 2289, 2226, 2163, 2100, 2037, 1974, 1911, 1848, 1785, 1722, 1659, 1596, 1533, 1470, 1407, 1471, 1534, 1597, 1660, 1723, 1786, 1849,
    1912, 1975, 2038, 2101, 2164, 2227, 2290, 2353, 2416, 2479, 2542, 2605, 2668, 2731, 2794, 2857, 2920, 2983, 3046, 3109, 3172, 3235, 3298, 3361, 3424, 3487, 3550, 3613, 3676, 3739, 3802, 3865, 3928, 3991, 4054, 4055, 3992, 3929, 3866, 3803, 3740, 3677, 3614, 3551, 3488, 3425, 3362, 3299, 3236, 3173, 3110, 3047, 2984, 2921, 2858, 2795, 2732, 2669, 2606, 2543, 2480, 2417, 2354, 2291, 2228, 2165, 2102, 2039, 1976, 1913, 1850, 1787, 1724, 1661, 1598, 1535, 1599, 1662, 1725, 1788, 1851, 1914, 1977, 2040, 2103, 2166, 2229, 2292, 2355, 2418, 2481, 2544, 2607, 2670, 2733, 2796, 2859, 2922, 2985, 3048, 3111, 3174, 3237, 3300, 3363, 3426, 3489, 3552, 3615, 3678, 3741, 3804, 3867, 3930, 3993, 4056, 4057, 3994, 3931, 3868, 3805, 3742, 3679, 3616, 3553, 3490, 3427, 3364,
    3301, 3238, 3175, 3112, 3049, 2986, 2923, 2860, 2797, 2734, 2671, 2608, 2545, 2482, 2419, 2356, 2293, 2230, 2167, 2104, 2041, 1978, 1915, 1852, 1789, 1726, 1663, 1727, 1790, 1853, 1916, 1979, 2042, 2105, 2168, 2231, 2294, 2357, 2420, 2483, 2546, 2609, 2672, 2735, 2798, 2861, 2924, 2987, 3050, 3113, 3176, 3239, 3302, 3365, 3428, 3491, 3554, 3617, 3680, 3743, 3806, 3869, 3932, 3995, 4058, 4059, 3996, 3933, 3870, 3807, 3744, 3681, 3618, 3555, 3492, 3429, 3366, 3303, 3240, 3177, 3114, 3051, 2988, 2925, 2862, 2799, 2736, 2673, 2610, 2547, 2484, 2421, 2358, 2295, 2232, 2169, 2106, 2043, 1980, 1917, 1854, 1791, 1855, 1918, 1981, 2044, 2107, 2170, 2233, 2296, 2359, 2422, 2485, 2548, 2611, 2674, 2737, 2800, 2863, 2926, 2989, 3052, 3115, 3178, 3241, 3304, 3367, 3430,
    3493, 3556, 3619, 3682, 3745, 3808, 3871, 3934, 3997, 4060, 4061, 3998, 3935, 3872, 3809, 3746, 3683, 3620, 3557, 3494, 3431, 3368, 3305, 3242, 3179, 3116, 3053, 2990, 2927, 2864, 2801, 2738, 2675, 2612, 2549, 2486, 2423, 2360, 2297, 2234, 2171, 2108, 2045, 1982, 1919, 1983, 2046, 2109, 2172, 2235, 2298, 2361, 2424, 2487, 2550, 2613, 2676, 2739, 2802, 2865, 2928, 2991, 3054, 3117, 3180, 3243, 3306, 3369, 3432, 3495, 3558, 3621, 3684, 3747, 3810, 3873, 3936, 3999, 4062, 4063, 4000, 3937, 3874, 3811, 3748, 3685, 3622, 3559, 3496, 3433, 3370, 3307, 3244, 3181, 3118, 3055, 2992, 2929, 2866, 2803, 2740, 2677, 2614, 2551, 2488, 2425, 2362, 2299, 2236, 2173, 2110, 2047, 2111, 2174, 2237, 2300, 2363, 2426, 2489, 2552, 2615, 2678, 2741, 2804, 2867, 2930, 2993, 3056,
    3119, 3182, 3245, 3308, 3371, 3434, 3497, 3560, 3623, 3686, 3749, 3812, 3875, 3938, 4001, 4064, 4065, 4002, 3939, 3876, 3813, 3750, 3687, 3624, 3561, 3498, 3435, 3372, 3309, 3246, 3183, 3120, 3057, 2994, 2931, 2868, 2805, 2742, 2679, 2616, 2553, 2490, 2427, 2364, 2301, 2238, 2175, 2239, 2302, 2365, 2428, 2491, 2554, 2617, 2680, 2743, 2806, 2869, 2932, 2995, 3058, 3121, 3184, 3247, 3310, 3373, 3436, 3499, 3562, 3625, 3688, 3751, 3814, 3877, 3940, 4003, 4066, 4067, 4004, 3941, 3878, 3815, 3752, 3689, 3626, 3563, 3500, 3437, 3374, 3311, 3248, 3185, 3122, 3059, 2996, 2933, 2870, 2807, 2744, 2681, 2618, 2555, 2492, 2429, 2366, 2303, 2367, 2430, 2493, 2556, 2619, 2682, 2745, 2808, 2871, 2934, 2997, 3060, 3123, 3186, 3249, 3312, 3375, 3438, 3501, 3564, 3627, 3690,
    3753, 3816, 3879, 3942, 4005, 4068, 4069, 4006, 3943, 3880, 3817, 3754, 3691, 3628, 3565, 3502, 3439, 3376, 3313, 3250, 3187, 3124, 3061, 2998, 2935, 2872, 2809, 2746, 2683, 2620, 2557, 2494, 2431, 2495, 2558, 2621, 2684, 2747, 2810, 2873, 2936, 2999, 3062, 3125, 3188, 3251, 3314, 3377, 3440, 3503, 3566, 3629, 3692, 3755, 3818, 3881, 3944, 4007, 4070, 4071, 4008, 3945, 3882, 3819, 3756, 3693, 3630, 3567, 3504, 3441, 3378, 3315, 3252, 3189, 3126, 3063, 3000, 2937, 2874, 2811, 2748, 2685, 2622, 2559, 2623, 2686, 2749, 2812, 2875, 2938, 3001, 3064, 3127, 3190, 3253, 3316, 3379, 3442, 3505, 3568, 3631, 3694, 3757, 3820, 3883, 3946, 4009, 4072, 4073, 4010, 3947, 3884, 3821, 3758, 3695, 3632, 3569, 3506, 3443, 3380, 3317, 3254, 3191, 3128, 3065, 3002, 2939, 2876,
    2813, 2750, 2687, 2751, 2814, 2877, 2940, 3003, 3066, 3129, 3192, 3255, 3318, 3381, 3444, 3507, 3570, 3633, 3696, 3759, 3822, 3885, 3948, 4011, 4074, 4075, 4012, 3949, 3886, 3823, 3760, 3697, 3634, 3571, 3508, 3445, 3382, 3319, 3256, 3193, 3130, 3067, 3004, 2941, 2878, 2815, 2879, 2942, 3005, 3068, 3131, 3194, 3257, 3320, 3383, 3446, 3509, 3572, 3635, 3698, 3761, 3824, 3887, 3950, 4013, 4076, 4077, 4014, 3951, 3888, 3825, 3762, 3699, 3636, 3573, 3510, 3447, 3384, 3321, 3258, 3195, 3132, 3069, 3006, 2943, 3007, 3070, 3133, 3196, 3259, 3322, 3385, 3448, 3511, 3574, 3637, 3700, 3763, 3826, 3889, 3952, 4015, 4078, 4079, 4016, 3953, 3890, 3827, 3764, 3701, 3638, 3575, 3512, 3449, 3386, 3323, 3260, 3197, 3134, 3071, 3135, 3198, 3261, 3324, 3387, 3450, 3513, 3576,
    3639, 3702, 3765, 3828, 3891, 3954, 4017, 4080, 4081, 4018, 3955, 3892, 3829, 3766, 3703, 3640, 3577, 3514, 3451, 3388, 3325, 3262, 3199, 3263, 3326, 3389, 3452, 3515, 3578, 3641, 3704, 3767, 3830, 3893, 3956, 4019, 4082, 4083, 4020, 3957, 3894, 3831, 3768, 3705, 3642, 3579, 3516, 3453, 3390, 3327, 3391, 3454, 3517, 3580, 3643, 3706, 3769, 3832, 3895, 3958, 4021, 4084, 4085, 4022, 3959, 3896, 3833, 3770, 3707, 3644, 3581, 3518, 3455, 3519, 3582, 3645, 3708, 3771, 3834, 3897, 3960, 4023, 4086, 4087, 4024, 3961, 3898, 3835, 3772, 3709, 3646, 3583, 3647, 3710, 3773, 3836, 3899, 3962, 4025, 4088, 4089, 4026, 3963, 3900, 3837, 3774, 3711, 3775, 3838, 3901, 3964, 4027, 4090, 4091, 4028, 3965, 3902, 3839, 3903, 3966, 4029, 4092, 4093, 4030, 3967, 4031, 4094, 4095,

    // 64 x 128:
    0, 1, 128, 256, 129, 2, 3, 130, 257, 384, 512, 385, 258, 131, 4, 5, 132, 259, 386, 513, 640, 768, 641, 514, 387, 260, 133, 6, 7, 134, 261, 388, 515, 642, 769, 896, 1024, 897, 770, 643, 516, 389, 262, 135, 8, 9, 136, 263, 390, 517, 644, 771, 898, 1025, 1152, 1280, 1153, 1026, 899, 772, 645, 518, 391, 264, 137, 10, 11, 138, 265, 392, 519, 646, 773, 900, 1027, 1154, 1281, 1408, 1536, 1409, 1282, 1155, 1028, 901, 774, 647, 520, 393, 266, 139, 12, 13, 140, 267, 394, 521, 648, 775, 902, 1029, 1156, 1283, 1410, 1537, 1664, 1792, 1665, 1538, 1411, 1284, 1157, 1030, 903, 776, 649, 522, 395, 268, 141, 14, 15, 142, 269, 396, 523, 650, 777, 904,
    1031, 1158, 1285, 1412, 1539, 1666, 1793, 1920, 2048, 1921, 1794, 1667, 1540, 1413, 1286, 1159, 1032, 905, 778, 651, 524, 397, 270, 143, 16, 17, 144, 271, 398, 525, 652, 779, 906, 1033, 1160, 1287, 1414, 1541, 1668, 1795, 1922, 2049, 2176, 2304, 2177, 2050, 1923, 1796, 1669, 1542, 1415, 1288, 1161, 1034, 907, 780, 653, 526, 399, 272, 145, 18, 19, 146, 273, 400, 527, 654, 781, 908, 1035, 1162, 1289, 1416, 1543, 1670, 1797, 1924, 2051, 2178, 2305, 2432, 2560, 2433, 2306, 2179, 2052, 1925, 1798, 1671, 1544, 1417, 1290, 1163, 1036, 909, 782, 655, 528, 401, 274, 147, 20, 21, 148, 275, 402, 529, 656, 783, 910, 1037, 1164, 1291, 1418, 1545, 1672, 1799, 1926, 2053, 2180, 2307, 2434, 2561, 2688, 2816, 2689, 2562,
    2435, 2308, 2181, 2054, 1927, 1800, 1673, 1546, 1419, 1292, 1165, 1038, 911, 784, 657, 530, 403, 276, 149, 22, 23, 150, 277, 404, 531, 658, 785, 912, 1039, 1166, 1293, 1420, 1547, 1674, 1801, 1928, 2055, 2182, 2309, 2436, 2563, 2690, 2817, 2944, 3072, 2945, 2818, 2691, 2564, 2437, 2310, 2183, 2056, 1929, 1802, 1675, 1548, 1421, 1294, 1167, 1040, 913, 786, 659, 532, 405, 278, 151, 24, 25, 152, 279, 406, 533, 660, 787, 914, 1041, 1168, 1295, 1422, 1549, 1676, 1803, 1930, 2057, 2184, 2311, 2438, 2565, 2692, 2819, 2946, 3073, 3200, 3328, 3201, 3074, 2947, 2820, 2693, 2566, 2439, 2312, 2185, 2058, 1931, 1804, 1677, 1550, 1423, 1296, 1169, 1042, 915, 788, 661, 534, 407, 280, 153, 26, 27, 154, 281, 408, 535, 662,
    789, 916, 1043, 1170, 1297, 1424, 1551, 1678, 1805, 1932, 2059, 2186, 2313, 2440, 2567, 2694, 2821, 2948, 3075, 3202, 3329, 3456, 3584, 3457, 3330, 3203, 3076, 2949, 2822, 2695, 2568, 2441, 2314, 2187, 2060, 1933, 1806, 1679, 1552, 1425, 1298, 1171, 1044, 917, 790, 663, 536, 409, 282, 155, 28, 29, 156, 283, 410, 537, 664, 791, 918, 1045, 1172, 1299, 1426, 1553, 1680, 1807, 1934, 2061, 2188, 2315, 2442, 2569, 2696, 2823, 2950, 3077, 3204, 3331, 3458, 3585, 3712, 3840, 3713, 3586, 3459, 3332, 3205, 3078, 2951, 2824, 2697, 2570, 2443, 2316, 2189, 2062, 1935, 1808, 1681, 1554, 1427, 1300, 1173, 1046, 919, 792, 665, 538, 411, 284, 157, 30, 31, 158, 285, 412, 539, 666, 793, 920, 1047, 1174, 1301, 1428, 1555, 1682, 1809, 1936,
    2063, 2190, 2317, 2444, 2571, 2698, 2825, 2952, 3079, 3206, 3333, 3460, 3587, 3714, 3841, 3968, 4096, 3969, 3842, 3715, 3588, 3461, 3334, 3207, 3080, 2953, 2826, 2699, 2572, 2445, 2318, 2191, 2064, 1937, 1810, 1683, 1556, 1429, 1302, 1175, 1048, 921, 794, 667, 540, 413, 286, 159, 32, 33, 160, 287, 414, 541, 668, 795, 922, 1049, 1176, 1303, 1430, 1557, 1684, 1811, 1938, 2065, 2192, 2319, 2446, 2573, 2700, 2827, 2954, 3081, 3208, 3335, 3462, 3589, 3716, 3843, 3970, 4097, 4224, 4352, 4225, 4098, 3971, 3844, 3717, 3590, 3463, 3336, 3209, 3082, 2955, 2828, 2701, 2574, 2447, 2320, 2193, 2066, 1939, 1812, 1685, 1558, 1431, 1304, 1177, 1050, 923, 796, 669, 542, 415, 288, 161, 34, 35, 162, 289, 416, 543, 670, 797, 924, 1051, 1178,
    1305, 1432, 1559, 1686, 1813, 1940, 2067, 2194, 2321, 2448, 2575, 2702, 2829, 2956, 3083, 3210, 3337, 3464, 3591, 3718, 3845, 3972, 4099, 4226, 4353, 4480, 4608, 4481, 4354, 4227, 4100, 3973, 3846, 3719, 3592, 3465, 3338, 3211, 3084, 2957, 2830, 2703, 2576, 2449, 2322, 2195, 2068, 1941, 1814, 1687, 1560, 1433, 1306, 1179, 1052, 925, 798, 671, 544, 417, 290, 163, 36, 37, 164, 291, 418, 545, 672, 799, 926, 1053, 1180, 1307, 1434, 1561, 1688, 1815, 1942, 2069, 2196, 2323, 2450, 2577, 2704, 2831, 2958, 3085, 3212, 3339, 3466, 3593, 3720, 3847, 3974, 4101, 4228, 4355, 4482, 4609, 4736, 4864, 4737, 4610, 4483, 4356, 4229, 4102, 3975, 3848, 3721, 3594, 3467, 3340, 3213, 3086, 2959, 2832, 2705, 2578, 2451, 2324, 2197, 2070, 1943, 1816, 1689, 1562,
    1435, 1308, 1181, 1054, 927, 800, 673, 546, 419, 292, 165, 38, 39, 166, 293, 420, 547, 674, 801, 928, 1055, 1182, 1309, 1436, 1563, 1690, 1817, 1944, 2071, 2198, 2325, 2452, 2579, 2706, 2833, 2960, 3087, 3214, 3341, 3468, 3595, 3722, 3849, 3976, 4103, 4230, 4357, 4484, 4611, 4738, 4865, 4992, 5120, 4993, 4866, 4739, 4612, 4485, 4358, 4231, 4104, 3977, 3850, 3723, 3596, 3469, 3342, 3215, 3088, 2961, 2834, 2707, 2580, 2453, 2326, 2199, 2072, 1945, 1818, 1691, 1564, 1437, 1310, 1183, 1056, 929, 802, 675, 548, 421, 294, 167, 40, 41, 168, 295, 422, 549, 676, 803, 930, 1057, 1184, 1311, 1438, 1565, 1692, 1819, 1946, 2073, 2200, 2327, 2454, 2581, 2708, 2835, 2962, 3089, 3216, 3343, 3470, 3597, 3724, 3851, 3978, 4105, 4232, 4359,
    4486, 4613, 4740, 4867, 4994, 5121, 5248, 5376, 5249, 5122, 4995, 4868, 4741, 4614, 4487, 4360, 4233, 4106, 3979, 3852, 3725, 3598, 3471, 3344, 3217, 3090, 2963, 2836, 2709, 2582, 2455, 2328, 2201, 2074, 1947, 1820, 1693, 1566, 1439, 1312, 1185, 1058, 931, 804, 677, 550, 423, 296, 169, 42, 43, 170, 297, 424, 551, 678, 805, 932, 1059, 1186, 1313, 1440, 1567, 1694, 1821, 1948, 2075, 2202, 2329, 2456, 2583, 2710, 2837, 2964, 3091, 3218, 3345, 3472, 3599, 3726, 3853, 3980, 4107, 4234, 4361, 4488, 4615, 4742, 4869, 4996, 5123, 5250, 5377, 5504, 5632, 5505, 5378, 5251, 5124, 4997, 4870, 4743, 4616, 4489, 4362, 4235, 4108, 3981, 3854, 3727, 3600, 3473, 3346, 3219, 3092, 2965, 2838, 2711, 2584, 2457, 2330, 2203, 2076, 1949, 1822, 1695, 1568, 1441,
    1314, 1187, 1060, 933, 806, 679, 552, 425, 298, 171, 44, 45, 172, 299, 426, 553, 680, 807, 934, 1061, 1188, 1315, 1442, 1569, 1696, 1823, 1950, 2077, 2204, 2331, 2458, 2585, 2712, 2839, 2966, 3093, 3220, 3347, 3474, 3601, 3728, 3855, 3982, 4109, 4236, 4363, 4490, 4617, 4744, 4871, 4998, 5125, 5252, 5379, 5506, 5633, 5760, 5888, 5761, 5634, 5507, 5380, 5253, 5126, 4999, 4872, 4745, 4618, 4491, 4364, 4237, 4110, 3983, 3856, 3729, 3602, 3475, 3348, 3221, 3094, 2967, 2840, 2713, 2586, 2459, 2332, 2205, 2078, 1951, 1824, 1697, 1570, 1443, 1316, 1189, 1062, 935, 808, 681, 554, 427, 300, 173, 46, 47, 174, 301, 428, 555, 682, 809, 936, 1063, 1190, 1317, 1444, 1571, 1698, 1825, 1952, 2079, 2206, 2333, 2460, 2587, 2714, 2841, 2968,
    3095, 3222, 3349, 3476, 3603, 3730, 3857, 3984, 4111, 4238, 4365, 4492, 4619, 4746, 4873, 5000, 5127, 5254, 5381, 5508, 5635, 5762, 5889, 6016, 6144, 6017, 5890, 5763, 5636, 5509, 5382, 5255, 5128, 5001, 4874, 4747, 4620, 4493, 4366, 4239, 4112, 3985, 3858, 3731, 3604, 3477, 3350, 3223, 3096, 2969, 2842, 2715, 2588, 2461, 2334, 2207, 2080, 1953, 1826, 1699, 1572, 1445, 1318, 1191, 1064, 937, 810, 683, 556, 429, 302, 175, 48, 49, 176, 303, 430, 557, 684, 811, 938, 1065, 1192, 1319, 1446, 1573, 1700, 1827, 1954, 2081, 2208, 2335, 2462, 2589, 2716, 2843, 2970, 3097, 3224, 3351, 3478, 3605, 3732, 3859, 3986, 4113, 4240, 4367, 4494, 4621, 4748, 4875, 5002, 5129, 5256, 5383, 5510, 5637, 5764, 5891, 6018, 6145, 6272, 6400, 6273, 6146, 6019, 5892,
    5765, 5638, 5511, 5384, 5257, 5130, 5003, 4876, 4749, 4622, 4495, 4368, 4241, 4114, 3987, 3860, 3733, 3606, 3479, 3352, 3225, 3098, 2971, 2844, 2717, 2590, 2463, 2336, 2209, 2082, 1955, 1828, 1701, 1574, 1447, 1320, 1193, 1066, 939, 812, 685, 558, 431, 304, 177, 50, 51, 178, 305, 432, 559, 686, 813, 940, 1067, 1194, 1321, 1448, 1575, 1702, 1829, 1956, 2083, 2210, 2337, 2464, 2591, 2718, 2845, 2972, 3099, 3226, 3353, 3480, 3607, 3734, 3861, 3988, 4115, 4242, 4369, 4496, 4623, 4750, 4877, 5004, 5131, 5258, 5385, 5512, 5639, 5766, 5893, 6020, 6147, 6274, 6401, 6528, 6656, 6529, 6402, 6275, 6148, 6021, 5894, 5767, 5640, 5513, 5386, 5259, 5132, 5005, 4878, 4751, 4624, 4497, 4370, 4243, 4116, 3989, 3862, 3735, 3608, 3481, 3354, 3227, 3100, 2973,
    2846, 2719, 2592, 2465, 2338, 2211, 2084, 1957, 1830, 1703, 1576, 1449, 1322, 1195, 1068, 941, 814, 687, 560, 433, 306, 179, 52, 53, 180, 307, 434, 561, 688, 815, 942, 1069, 1196, 1323, 1450, 1577, 1704, 1831, 1958, 2085, 2212, 2339, 2466, 2593, 2720, 2847, 2974, 3101, 3228, 3355, 3482, 3609, 3736, 3863, 3990, 4117, 4244, 4371, 4498, 4625, 4752, 4879, 5006, 5133, 5260, 5387, 5514, 5641, 5768, 5895, 6022, 6149, 6276, 6403, 6530, 6657, 6784, 6912, 6785, 6658, 6531, 6404, 6277, 6150, 6023, 5896, 5769, 5642, 5515, 5388, 5261, 5134, 5007, 4880, 4753, 4626, 4499, 4372, 4245, 4118, 3991, 3864, 3737, 3610, 3483, 3356, 3229, 3102, 2975, 2848, 2721, 2594, 2467, 2340, 2213, 2086, 1959, 1832, 1705, 1578, 1451, 1324, 1197, 1070, 943, 816, 689, 562,
    435, 308, 181, 54, 55, 182, 309, 436, 563, 690, 817, 944, 1071, 1198, 1325, 1452, 1579, 1706, 1833, 1960, 2087, 2214, 2341, 2468, 2595, 2722, 2849, 2976, 3103, 3230, 3357, 3484, 3611, 3738, 3865, 3992, 4119, 4246, 4373, 4500, 4627, 4754, 4881, 5008, 5135, 5262, 5389, 5516, 5643, 5770, 5897, 6024, 6151, 6278, 6405, 6532, 6659, 6786, 6913, 7040, 7168, 7041, 6914, 6787, 6660, 6533, 6406, 6279, 6152, 6025, 5898, 5771, 5644, 5517, 5390, 5263, 5136, 5009, 4882, 4755, 4628, 4501, 4374, 4247, 4120, 3993, 3866, 3739, 3612, 3485, 3358, 3231, 3104, 2977, 2850, 2723, 2596, 2469, 2342, 2215, 2088, 1961, 1834, 1707, 1580, 1453, 1326, 1199, 1072, 945, 818, 691, 564, 437, 310, 183, 56, 57, 184, 311, 438, 565, 692, 819, 946, 1073, 1200, 1327,
    1454, 1581, 1708, 1835, 1962, 2089, 2216, 2343, 2470, 2597, 2724, 2851, 2978, 3105, 3232, 3359, 3486, 3613, 3740, 3867, 3994, 4121, 4248, 4375, 4502, 4629, 4756, 4883, 5010, 5137, 5264, 5391, 5518, 5645, 5772, 5899, 6026, 6153, 6280, 6407, 6534, 6661, 6788, 6915, 7042, 7169, 7296, 7424, 7297, 7170, 7043, 6916, 6789, 6662, 6535, 6408, 6281, 6154, 6027, 5900, 5773, 5646, 5519, 5392, 5265, 5138, 5011, 4884, 4757, 4630, 4503, 4376, 4249, 4122, 3995, 3868, 3741, 3614, 3487, 3360, 3233, 3106, 2979, 2852, 2725, 2598, 2471, 2344, 2217, 2090, 1963, 1836, 1709, 1582, 1455, 1328, 1201, 1074, 947, 820, 693, 566, 439, 312, 185, 58, 59, 186, 313, 440, 567, 694, 821, 948, 1075, 1202, 1329, 1456, 1583, 1710, 1837, 1964, 2091, 2218, 2345, 2472, 2599, 2726,
    2853, 2980, 3107, 3234, 3361, 3488, 3615, 3742, 3869, 3996, 4123, 4250, 4377, 4504, 4631, 4758, 4885, 5012, 5139, 5266, 5393, 5520, 5647, 5774, 5901, 6028, 6155, 6282, 6409, 6536, 6663, 6790, 6917, 7044, 7171, 7298, 7425, 7552, 7680, 7553, 7426, 7299, 7172, 7045, 6918, 6791, 6664, 6537, 6410, 6283, 6156, 6029, 5902, 5775, 5648, 5521, 5394, 5267, 5140, 5013, 4886, 4759, 4632, 4505, 4378, 4251, 4124, 3997, 3870, 3743, 3616, 3489, 3362, 3235, 3108, 2981, 2854, 2727, 2600, 2473, 2346, 2219, 2092, 1965, 1838, 1711, 1584, 1457, 1330, 1203, 1076, 949, 822, 695, 568, 441, 314, 187, 60, 61, 188, 315, 442, 569, 696, 823, 950, 1077, 1204, 1331, 1458, 1585, 1712, 1839, 1966, 2093, 2220, 2347, 2474, 2601, 2728, 2855, 2982, 3109, 3236, 3363, 3490, 3617,
    3744, 3871, 3998, 4125, 4252, 4379, 4506, 4633, 4760, 4887, 5014, 5141, 5268, 5395, 5522, 5649, 5776, 5903, 6030, 6157, 6284, 6411, 6538, 6665, 6792, 6919, 7046, 7173, 7300, 7427, 7554, 7681, 7808, 7936, 7809, 7682, 7555, 7428, 7301, 7174, 7047, 6920, 6793, 6666, 6539, 6412, 6285, 6158, 6031, 5904, 5777, 5650, 5523, 5396, 5269, 5142, 5015, 4888, 4761, 4634, 4507, 4380, 4253, 4126, 3999, 3872, 3745, 3618, 3491, 3364, 3237, 3110, 2983, 2856, 2729, 2602, 2475, 2348, 2221, 2094, 1967, 1840, 1713, 1586, 1459, 1332, 1205, 1078, 951, 824, 697, 570, 443, 316, 189, 62, 63, 190, 317, 444, 571, 698, 825, 952, 1079, 1206, 1333, 1460, 1587, 1714, 1841, 1968, 2095, 2222, 2349, 2476, 2603, 2730, 2857, 2984, 3111, 3238, 3365, 3492, 3619, 3746, 3873, 4000,
    4127, 4254, 4381, 4508, 4635, 4762, 4889, 5016, 5143, 5270, 5397, 5524, 5651, 5778, 5905, 6032, 6159, 6286, 6413, 6540, 6667, 6794, 6921, 7048, 7175, 7302, 7429, 7556, 7683, 7810, 7937, 8064, 8065, 7938, 7811, 7684, 7557, 7430, 7303, 7176, 7049, 6922, 6795, 6668, 6541, 6414, 6287, 6160, 6033, 5906, 5779, 5652, 5525, 5398, 5271, 5144, 5017, 4890, 4763, 4636, 4509, 4382, 4255, 4128, 4001, 3874, 3747, 3620, 3493, 3366, 3239, 3112, 2985, 2858, 2731, 2604, 2477, 2350, 2223, 2096, 1969, 1842, 1715, 1588, 1461, 1334, 1207, 1080, 953, 826, 699, 572, 445, 318, 191, 64, 65, 192, 319, 446, 573, 700, 827, 954, 1081, 1208, 1335, 1462, 1589, 1716, 1843, 1970, 2097, 2224, 2351, 2478, 2605, 2732, 2859, 2986, 3113, 3240, 3367, 3494, 3621, 3748, 3875, 4002,
    4129, 4256, 4383, 4510, 4637, 4764, 4891, 5018, 5145, 5272, 5399, 5526, 5653, 5780, 5907, 6034, 6161, 6288, 6415, 6542, 6669, 6796, 6923, 7050, 7177, 7304, 7431, 7558, 7685, 7812, 7939, 8066, 8067, 7940, 7813, 7686, 7559, 7432, 7305, 7178, 7051, 6924, 6797, 6670, 6543, 6416, 6289, 6162, 6035, 5908, 5781, 5654, 5527, 5400, 5273, 5146, 5019, 4892, 4765, 4638, 4511, 4384, 4257, 4130, 4003, 3876, 3749, 3622, 3495, 3368, 3241, 3114, 2987, 2860, 2733, 2606, 2479, 2352, 2225, 2098, 1971, 1844, 1717, 1590, 1463, 1336, 1209, 1082, 955, 828, 701, 574, 447, 320, 193, 66, 67, 194, 321, 448, 575, 702, 829, 956, 1083, 1210, 1337, 1464, 1591, 1718, 1845, 1972, 2099, 2226, 2353, 2480, 2607, 2734, 2861, 2988, 3115, 3242, 3369, 3496, 3623, 3750, 3877, 4004,
    4131, 4258, 4385, 4512, 4639, 4766, 4893, 5020, 5147, 5274, 5401, 5528, 5655, 5782, 5909, 6036, 6163, 6290, 6417, 6544, 6671, 6798, 6925, 7052, 7179, 7306, 7433, 7560, 7687, 7814, 7941, 8068, 8069, 7942, 7815, 7688, 7561, 7434, 7307, 7180, 7053, 6926, 6799, 6672, 6545, 6418, 6291, 6164, 6037, 5910, 5783, 5656, 5529, 5402, 5275, 5148, 5021, 4894, 4767, 4640, 4513, 4386, 4259, 4132, 4005, 3878, 3751, 3624, 3497, 3370, 3243, 3116, 2989, 2862, 2735, 2608, 2481, 2354, 2227, 2100, 1973, 1846, 1719, 1592, 1465, 1338, 1211, 1084, 957, 830, 703, 576, 449, 322, 195, 68, 69, 196, 323, 450, 577, 704, 831, 958, 1085, 1212, 1339, 1466, 1593, 1720, 1847, 1974, 2101, 2228, 2355, 2482, 2609, 2736, 2863, 2990, 3117, 3244, 3371, 3498, 3625, 3752, 3879, 4006,
    4133, 4260, 4387, 4514, 4641, 4768, 4895, 5022, 5149, 5276, 5403, 5530, 5657, 5784, 5911, 6038, 6165, 6292, 6419, 6546, 6673, 6800, 6927, 7054, 7181, 7308, 7435, 7562, 7689, 7816, 7943, 8070, 8071, 7944, 7817, 7690, 7563, 7436, 7309, 7182, 7055, 6928, 6801, 6674, 6547, 6420, 6293, 6166, 6039, 5912, 5785, 5658, 5531, 5404, 5277, 5150, 5023, 4896, 4769, 4642, 4515, 4388, 4261, 4134, 4007, 3880, 3753, 3626, 3499, 3372, 3245, 3118, 2991, 2864, 2737, 2610, 2483, 2356, 2229, 2102, 1975, 1848, 1721, 1594, 1467, 1340, 1213, 1086, 959, 832, 705, 578, 451, 324, 197, 70, 71, 198, 325, 452, 579, 706, 833, 960, 1087, 1214, 1341, 1468, 1595, 1722, 1849, 1976, 2103, 2230, 2357, 2484, 2611, 2738, 2865, 2992, 3119, 3246, 3373, 3500, 3627, 3754, 3881, 4008,
    4135, 4262, 4389, 4516, 4643, 4770, 4897, 5024, 5151, 5278, 5405, 5532, 5659, 5786, 5913, 6040, 6167, 6294, 6421, 6548, 6675, 6802, 6929, 7056, 7183, 7310, 7437, 7564, 7691, 7818, 7945, 8072, 8073, 7946, 7819, 7692, 7565, 7438, 7311, 7184, 7057, 6930, 6803, 6676, 6549, 6422, 6295, 6168, 6041, 5914, 5787, 5660, 5533, 5406, 5279, 5152, 5025, 4898, 4771, 4644, 4517, 4390, 4263, 4136, 4009, 3882, 3755, 3628, 3501, 3374, 3247, 3120, 2993, 2866, 2739, 2612, 2485, 2358, 2231, 2104, 1977, 1850, 1723, 1596, 1469, 1342, 1215, 1088, 961, 834, 707, 580, 453, 326, 199, 72, 73, 200, 327, 454, 581, 708, 835, 962, 1089, 1216, 1343, 1470, 1597, 1724, 1851, 1978, 2105, 2232, 2359, 2486, 2613, 2740, 2867, 2994, 3121, 3248, 3375, 3502, 3629, 3756, 3883, 4010,
    4137, 4264, 4391, 4518, 4645, 4772, 4899, 5026, 5153, 5280, 5407, 5534, 5661, 5788, 5915, 6042, 6169, 6296, 6423, 6550, 6677, 6804, 6931, 7058, 7185, 7312, 7439, 7566, 7693, 7820, 7947, 8074, 8075, 7948, 7821, 7694, 7567, 7440, 7313, 7186, 7059, 6932, 6805, 6678, 6551, 6424, 6297, 6170, 6043, 5916, 5789, 5662, 5535, 5408, 5281, 5154, 5027, 4900, 4773, 4646, 4519, 4392, 4265, 4138, 4011, 3884, 3757, 3630, 3503, 3376, 3249, 3122, 2995, 2868, 2741, 2614, 2487, 2360, 2233, 2106, 1979, 1852, 1725, 1598, 1471, 1344, 1217, 1090, 963, 836, 709, 582, 455, 328, 201, 74, 75, 202, 329, 456, 583, 710, 837, 964, 1091, 1218, 1345, 1472, 1599, 1726, 1853, 1980, 2107, 2234, 2361, 2488, 2615, 2742, 2869, 2996, 3123, 3250, 3377, 3504, 3631, 3758, 3885, 4012,
    4139, 4266, 4393, 4520, 4647, 4774, 4901, 5028, 5155, 5282, 5409, 5536, 5663, 5790, 5917, 6044, 6171, 6298, 6425, 6552, 6679, 6806, 6933, 7060, 7187, 7314, 7441, 7568, 7695, 7822, 7949, 8076, 8077, 7950, 7823, 7696, 7569, 7442, 7315, 7188, 7061, 6934, 6807, 6680, 6553, 6426, 6299, 6172, 6045, 5918, 5791, 5664, 5537, 5410, 5283, 5156, 5029, 4902, 4775, 4648, 4521, 4394, 4267, 4140, 4013, 3886, 3759, 3632, 3505, 3378, 3251, 3124, 2997, 2870, 2743, 2616, 2489, 2362, 2235, 2108, 1981, 1854, 1727, 1600, 1473, 1346, 1219, 1092, 965, 838, 711, 584, 457, 330, 203, 76, 77, 204, 331, 458, 585, 712, 839, 966, 1093, 1220, 1347, 1474, 1601, 1728, 1855, 1982, 2109, 2236, 2363, 2490, 2617, 2744, 2871, 2998, 3125, 3252, 3379, 3506, 3633, 3760, 3887, 4014,
    4141, 4268, 4395, 4522, 4649, 4776, 4903, 5030, 5157, 5284, 5411, 5538, 5665, 5792, 5919, 6046, 6173, 6300, 6427, 6554, 6681, 6808, 6935, 7062, 7189, 7316, 7443, 7570, 7697, 7824, 7951, 8078, 8079, 7952, 7825, 7698, 7571, 7444, 7317, 7190, 7063, 6936, 6809, 6682, 6555, 6428, 6301, 6174, 6047, 5920, 5793, 5666, 5539, 5412, 5285, 5158, 5031, 4904, 4777, 4650, 4523, 4396, 4269, 4142, 4015, 3888, 3761, 3634, 3507, 3380, 3253, 3126, 2999, 2872, 2745, 2618, 2491, 2364, 2237, 2110, 1983, 1856, 1729, 1602, 1475, 1348, 1221, 1094, 967, 840, 713, 586, 459, 332, 205, 78, 79, 206, 333, 460, 587, 714, 841, 968, 1095, 1222, 1349, 1476, 1603, 1730, 1857, 1984, 2111, 2238, 2365, 2492, 2619, 2746, 2873, 3000, 3127, 3254, 3381, 3508, 3635, 3762, 3889, 4016,
    4143, 4270, 4397, 4524, 4651, 4778, 4905, 5032, 5159, 5286, 5413, 5540, 5667, 5794, 5921, 6048, 6175, 6302, 6429, 6556, 6683, 6810, 6937, 7064, 7191, 7318, 7445, 7572, 7699, 7826, 7953, 8080, 8081, 7954, 7827, 7700, 7573, 7446, 7319, 7192, 7065, 6938, 6811, 6684, 6557, 6430, 6303, 6176, 6049, 5922, 5795, 5668, 5541, 5414, 5287, 5160, 5033, 4906, 4779, 4652, 4525, 4398, 4271, 4144, 4017, 3890, 3763, 3636, 3509, 3382, 3255, 3128, 3001, 2874, 2747, 2620, 2493, 2366, 2239, 2112, 1985, 1858, 1731, 1604, 1477, 1350, 1223, 1096, 969, 842, 715, 588, 461, 334, 207, 80, 81, 208, 335, 462, 589, 716, 843, 970, 1097, 1224, 1351, 1478, 1605, 1732, 1859, 1986, 2113, 2240, 2367, 2494, 2621, 2748, 2875, 3002, 3129, 3256, 3383, 3510, 3637, 3764, 3891, 4018,
    4145, 4272, 4399, 4526, 4653, 4780, 4907, 5034, 5161, 5288, 5415, 5542, 5669, 5796, 5923, 6050, 6177, 6304, 6431, 6558, 6685, 6812, 6939, 7066, 7193, 7320, 7447, 7574, 7701, 7828, 7955, 8082, 8083, 7956, 7829, 7702, 7575, 7448, 7321, 7194, 7067, 6940, 6813, 6686, 6559, 6432, 6305, 6178, 6051, 5924, 5797, 5670, 5543, 5416, 5289, 5162, 5035, 4908, 4781, 4654, 4527, 4400, 4273, 4146, 4019, 3892, 3765, 3638, 3511, 3384, 3257, 3130, 3003, 2876, 2749, 2622, 2495, 2368, 2241, 2114, 1987, 1860, 1733, 1606, 1479, 1352, 1225, 1098, 971, 844, 717, 590, 463, 336, 209, 82, 83, 210, 337, 464, 591, 718, 845, 972, 1099, 1226, 1353, 1480, 1607, 1734, 1861, 1988, 2115, 2242, 2369, 2496, 2623, 2750, 2877, 3004, 3131, 3258, 3385, 3512, 3639, 3766, 3893, 4020,
    4147, 4274, 4401, 4528, 4655, 4782, 4909, 5036, 5163, 5290, 5417, 5544, 5671, 5798, 5925, 6052, 6179, 6306, 6433, 6560, 6687, 6814, 6941, 7068, 7195, 7322, 7449, 7576, 7703, 7830, 7957, 8084, 8085, 7958, 7831, 7704, 7577, 7450, 7323, 7196, 7069, 6942, 6815, 6688, 6561, 6434, 6307, 6180, 6053, 5926, 5799, 5672, 5545, 5418, 5291, 5164, 5037, 4910, 4783, 4656, 4529, 4402, 4275, 4148, 4021, 3894, 3767, 3640, 3513, 3386, 3259, 3132, 3005, 2878, 2751, 2624, 2497, 2370, 2243, 2116, 1989, 1862, 1735, 1608, 1481, 1354, 1227, 1100, 973, 846, 719, 592, 465, 338, 211, 84, 85, 212, 339, 466, 593, 720, 847, 974, 1101, 1228, 1355, 1482, 1609, 1736, 1863, 1990, 2117, 2244, 2371, 2498, 2625, 2752, 2879, 3006, 3133, 3260, 3387, 3514, 3641, 3768, 3895, 4022,
    4149, 4276, 4403, 4530, 4657, 4784, 4911, 5038, 5165, 5292, 5419, 5546, 5673, 5800, 5927, 6054, 6181, 6308, 6435, 6562, 6689, 6816, 6943, 7070, 7197, 7324, 7451, 7578, 7705, 7832, 7959, 8086, 8087, 7960, 7833, 7706, 7579, 7452, 7325, 7198, 7071, 6944, 6817, 6690, 6563, 6436, 6309, 6182, 6055, 5928, 5801, 5674, 5547, 5420, 5293, 5166, 5039, 4912, 4785, 4658, 4531, 4404, 4277, 4150, 4023, 3896, 3769, 3642, 3515, 3388, 3261, 3134, 3007, 2880, 2753, 2626, 2499, 2372, 2245, 2118, 1991, 1864, 1737, 1610, 1483, 1356, 1229, 1102, 975, 848, 721, 594, 467, 340, 213, 86, 87, 214, 341, 468, 595, 722, 849, 976, 1103, 1230, 1357, 1484, 1611, 1738, 1865, 1992, 2119, 2246, 2373, 2500, 2627, 2754, 2881, 3008, 3135, 3262, 3389, 3516, 3643, 3770, 3897, 4024,
    4151, 4278, 4405, 4532, 4659, 4786, 4913, 5040, 5167, 5294, 5421, 5548, 5675, 5802, 5929, 6056, 6183, 6310, 6437, 6564, 6691, 6818, 6945, 7072, 7199, 7326, 7453, 7580, 7707, 7834, 7961, 8088, 8089, 7962, 7835, 7708, 7581, 7454, 7327, 7200, 7073, 6946, 6819, 6692, 6565, 6438, 6311, 6184, 6057, 5930, 5803, 5676, 5549, 5422, 5295, 5168, 5041, 4914, 4787, 4660, 4533, 4406, 4279, 4152, 4025, 3898, 3771, 3644, 3517, 3390, 3263, 3136, 3009, 2882, 2755, 2628, 2501, 2374, 2247, 2120, 1993, 1866, 1739, 1612, 1485, 1358, 1231, 1104, 977, 850, 723, 596, 469, 342, 215, 88, 89, 216, 343, 470, 597, 724, 851, 978, 1105, 1232, 1359, 1486, 1613, 1740, 1867, 1994, 2121, 2248, 2375, 2502, 2629, 2756, 2883, 3010, 3137, 3264, 3391, 3518, 3645, 3772, 3899, 4026,
    4153, 4280, 4407, 4534, 4661, 4788, 4915, 5042, 5169, 5296, 5423, 5550, 5677, 5804, 5931, 6058, 6185, 6312, 6439, 6566, 6693, 6820, 6947, 7074, 7201, 7328, 7455, 7582, 7709, 7836, 7963, 8090, 8091, 7964, 7837, 7710, 7583, 7456, 7329, 7202, 7075, 6948, 6821, 6694, 6567, 6440, 6313, 6186, 6059, 5932, 5805, 5678, 5551, 5424, 5297, 5170, 5043, 4916, 4789, 4662, 4535, 4408, 4281, 4154, 4027, 3900, 3773, 3646, 3519, 3392, 3265, 3138, 3011, 2884, 2757, 2630, 2503, 2376, 2249, 2122, 1995, 1868, 1741, 1614, 1487, 1360, 1233, 1106, 979, 852, 725, 598, 471, 344, 217, 90, 91, 218, 345, 472, 599, 726, 853, 980, 1107, 1234, 1361, 1488, 1615, 1742, 1869, 1996, 2123, 2250, 2377, 2504, 2631, 2758, 2885, 3012, 3139, 3266, 3393, 3520, 3647, 3774, 3901, 4028,
    4155, 4282, 4409, 4536, 4663, 4790, 4917, 5044, 5171, 5298, 5425, 5552, 5679, 5806, 5933, 6060, 6187, 6314, 6441, 6568, 6695, 6822, 6949, 7076, 7203, 7330, 7457, 7584, 7711, 7838, 7965, 8092, 8093, 7966, 7839, 7712, 7585, 7458, 7331, 7204, 7077, 6950, 6823, 6696, 6569, 6442, 6315, 6188, 6061, 5934, 5807, 5680, 5553, 5426, 5299, 5172, 5045, 4918, 4791, 4664, 4537, 4410, 4283, 4156, 4029, 3902, 3775, 3648, 3521, 3394, 3267, 3140, 3013, 2886, 2759, 2632, 2505, 2378, 2251, 2124, 1997, 1870, 1743, 1616, 1489, 1362, 1235, 1108, 981, 854, 727, 600, 473, 346, 219, 92, 93, 220, 347, 474, 601, 728, 855, 982, 1109, 1236, 1363, 1490, 1617, 1744, 1871, 1998, 2125, 2252, 2379, 2506, 2633, 2760, 2887, 3014, 3141, 3268, 3395, 3522, 3649, 3776, 3903, 4030,
    4157, 4284, 4411, 4538, 4665, 4792, 4919, 5046, 5173, 5300, 5427, 5554, 5681, 5808, 5935, 6062, 6189, 6316, 6443, 6570, 6697, 6824, 6951, 7078, 7205, 7332, 7459, 7586, 7713, 7840, 7967, 8094, 8095, 7968, 7841, 7714, 7587, 7460, 7333, 7206, 7079, 6952, 6825, 6698, 6571, 6444, 6317, 6190, 6063, 5936, 5809, 5682, 5555, 5428, 5301, 5174, 5047, 4920, 4793, 4666, 4539, 4412, 4285, 4158, 4031, 3904, 3777, 3650, 3523, 3396, 3269, 3142, 3015, 2888, 2761, 2634, 2507, 2380, 2253, 2126, 1999, 1872, 1745, 1618, 1491, 1364, 1237, 1110, 983, 856, 729, 602, 475, 348, 221, 94, 95, 222, 349, 476, 603, 730, 857, 984, 1111, 1238, 1365, 1492, 1619, 1746, 1873, 2000, 2127, 2254, 2381, 2508, 2635, 2762, 2889, 3016, 3143, 3270, 3397, 3524, 3651, 3778, 3905, 4032,
    4159, 4286, 4413, 4540, 4667, 4794, 4921, 5048, 5175, 5302, 5429, 5556, 5683, 5810, 5937, 6064, 6191, 6318, 6445, 6572, 6699, 6826, 6953, 7080, 7207, 7334, 7461, 7588, 7715, 7842, 7969, 8096, 8097, 7970, 7843, 7716, 7589, 7462, 7335, 7208, 7081, 6954, 6827, 6700, 6573, 6446, 6319, 6192, 6065, 5938, 5811, 5684, 5557, 5430, 5303, 5176, 5049, 4922, 4795, 4668, 4541, 4414, 4287, 4160, 4033, 3906, 3779, 3652, 3525, 3398, 3271, 3144, 3017, 2890, 2763, 2636, 2509, 2382, 2255, 2128, 2001, 1874, 1747, 1620, 1493, 1366, 1239, 1112, 985, 858, 731, 604, 477, 350, 223, 96, 97, 224, 351, 478, 605, 732, 859, 986, 1113, 1240, 1367, 1494, 1621, 1748, 1875, 2002, 2129, 2256, 2383, 2510, 2637, 2764, 2891, 3018, 3145, 3272, 3399, 3526, 3653, 3780, 3907, 4034,
    4161, 4288, 4415, 4542, 4669, 4796, 4923, 5050, 5177, 5304, 5431, 5558, 5685, 5812, 5939, 6066, 6193, 6320, 6447, 6574, 6701, 6828, 6955, 7082, 7209, 7336, 7463, 7590, 7717, 7844, 7971, 8098, 8099, 7972, 7845, 7718, 7591, 7464, 7337, 7210, 7083, 6956, 6829, 6702, 6575, 6448, 6321, 6194, 6067, 5940, 5813, 5686, 5559, 5432, 5305, 5178, 5051, 4924, 4797, 4670, 4543, 4416, 4289, 4162, 4035, 3908, 3781, 3654, 3527, 3400, 3273, 3146, 3019, 2892, 2765, 2638, 2511, 2384, 2257, 2130, 2003, 1876, 1749, 1622, 1495, 1368, 1241, 1114, 987, 860, 733, 606, 479, 352, 225, 98, 99, 226, 353, 480, 607, 734, 861, 988, 1115, 1242, 1369, 1496, 1623, 1750, 1877, 2004, 2131, 2258, 2385, 2512, 2639, 2766, 2893, 3020, 3147, 3274, 3401, 3528, 3655, 3782, 3909, 4036,
    4163, 4290, 4417, 4544, 4671, 4798, 4925, 5052, 5179, 5306, 5433, 5560, 5687, 5814, 5941, 6068, 6195, 6322, 6449, 6576, 6703, 6830, 6957, 7084, 7211, 7338, 7465, 7592, 7719, 7846, 7973, 8100, 8101, 7974, 7847, 7720, 7593, 7466, 7339, 7212, 7085, 6958, 6831, 6704, 6577, 6450, 6323, 6196, 6069, 5942, 5815, 5688, 5561, 5434, 5307, 5180, 5053, 4926, 4799, 4672, 4545, 4418, 4291, 4164, 4037, 3910, 3783, 3656, 3529, 3402, 3275, 3148, 3021, 2894, 2767, 2640, 2513, 2386, 2259, 2132, 2005, 1878, 1751, 1624, 1497, 1370, 1243, 1116, 989, 862, 735, 608, 481, 354, 227, 100, 101, 228, 355, 482, 609, 736, 863, 990, 1117, 1244, 1371, 1498, 1625, 1752, 1879, 2006, 2133, 2260, 2387, 2514, 2641, 2768, 2895, 3022, 3149, 3276, 3403, 3530, 3657, 3784, 3911, 4038,
    4165, 4292, 4419, 4546, 4673, 4800, 4927, 5054, 5181, 5308, 5435, 5562, 5689, 5816, 5943, 6070, 6197, 6324, 6451, 6578, 6705, 6832, 6959, 7086, 7213, 7340, 7467, 7594, 7721, 7848, 7975, 8102, 8103, 7976, 7849, 7722, 7595, 7468, 7341, 7214, 7087, 6960, 6833, 6706, 6579, 6452, 6325, 6198, 6071, 5944, 5817, 5690, 5563, 5436, 5309, 5182, 5055, 4928, 4801, 4674, 4547, 4420, 4293, 4166, 4039, 3912, 3785, 3658, 3531, 3404, 3277, 3150, 3023, 2896, 2769, 2642, 2515, 2388, 2261, 2134, 2007, 1880, 1753, 1626, 1499, 1372, 1245, 1118, 991, 864, 737, 610, 483, 356, 229, 102, 103, 230, 357, 484, 611, 738, 865, 992, 1119, 1246, 1373, 1500, 1627, 1754, 1881, 2008, 2135, 2262, 2389, 2516, 2643, 2770, 2897, 3024, 3151, 3278, 3405, 3532, 3659, 3786, 3913, 4040,
    4167, 4294, 4421, 4548, 4675, 4802, 4929, 5056, 5183, 5310, 5437, 5564, 5691, 5818, 5945, 6072, 6199, 6326, 6453, 6580, 6707, 6834, 6961, 7088, 7215, 7342, 7469, 7596, 7723, 7850, 7977, 8104, 8105, 7978, 7851, 7724, 7597, 7470, 7343, 7216, 7089, 6962, 6835, 6708, 6581, 6454, 6327, 6200, 6073, 5946, 5819, 5692, 5565, 5438, 5311, 5184, 5057, 4930, 4803, 4676, 4549, 4422, 4295, 4168, 4041, 3914, 3787, 3660, 3533, 3406, 3279, 3152, 3025, 2898, 2771, 2644, 2517, 2390, 2263, 2136, 2009, 1882, 1755, 1628, 1501, 1374, 1247, 1120, 993, 866, 739, 612, 485, 358, 231, 104, 105, 232, 359, 486, 613, 740, 867, 994, 1121, 1248, 1375, 1502, 1629, 1756, 1883, 2010, 2137, 2264, 2391, 2518, 2645, 2772, 2899, 3026, 3153, 3280, 3407, 3534, 3661, 3788, 3915, 4042,
    4169, 4296, 4423, 4550, 4677, 4804, 4931, 5058, 5185, 5312, 5439, 5566, 5693, 5820, 5947, 6074, 6201, 6328, 6455, 6582, 6709, 6836, 6963, 7090, 7217, 7344, 7471, 7598, 7725, 7852, 7979, 8106, 8107, 7980, 7853, 7726, 7599, 7472, 7345, 7218, 7091, 6964, 6837, 6710, 6583, 6456, 6329, 6202, 6075, 5948, 5821, 5694, 5567, 5440, 5313, 5186, 5059, 4932, 4805, 4678, 4551, 4424, 4297, 4170, 4043, 3916, 3789, 3662, 3535, 3408, 3281, 3154, 3027, 2900, 2773, 2646, 2519, 2392, 2265, 2138, 2011, 1884, 1757, 1630, 1503, 1376, 1249, 1122, 995, 868, 741, 614, 487, 360, 233, 106, 107, 234, 361, 488, 615, 742, 869, 996, 1123, 1250, 1377, 1504, 1631, 1758, 1885, 2012, 2139, 2266, 2393, 2520, 2647, 2774, 2901, 3028, 3155, 3282, 3409, 3536, 3663, 3790, 3917, 4044,
    4171, 4298, 4425, 4552, 4679, 4806, 4933, 5060, 5187, 5314, 5441, 5568, 5695, 5822, 5949, 6076, 6203, 6330, 6457, 6584, 6711, 6838, 6965, 7092, 7219, 7346, 7473, 7600, 7727, 7854, 7981, 8108, 8109, 7982, 7855, 7728, 7601, 7474, 7347, 7220, 7093, 6966, 6839, 6712, 6585, 6458, 6331, 6204, 6077, 5950, 5823, 5696, 5569, 5442, 5315, 5188, 5061, 4934, 4807, 4680, 4553, 4426, 4299, 4172, 4045, 3918, 3791, 3664, 3537, 3410, 3283, 3156, 3029, 2902, 2775, 2648, 2521, 2394, 2267, 2140, 2013, 1886, 1759, 1632, 1505, 1378, 1251, 1124, 997, 870, 743, 616, 489, 362, 235, 108, 109, 236, 363, 490, 617, 744, 871, 998, 1125, 1252, 1379, 1506, 1633, 1760, 1887, 2014, 2141, 2268, 2395, 2522, 2649, 2776, 2903, 3030, 3157, 3284, 3411, 3538, 3665, 3792, 3919, 4046,
    4173, 4300, 4427, 4554, 4681, 4808, 4935, 5062, 5189, 5316, 5443, 5570, 5697, 5824, 5951, 6078, 6205, 6332, 6459, 6586, 6713, 6840, 6967, 7094, 7221, 7348, 7475, 7602, 7729, 7856, 7983, 8110, 8111, 7984, 7857, 7730, 7603, 7476, 7349, 7222, 7095, 6968, 6841, 6714, 6587, 6460, 6333, 6206, 6079, 5952, 5825, 5698, 5571, 5444, 5317, 5190, 5063, 4936, 4809, 4682, 4555, 4428, 4301, 4174, 4047, 3920, 3793, 3666, 3539, 3412, 3285, 3158, 3031, 2904, 2777, 2650, 2523, 2396, 2269, 2142, 2015, 1888, 1761, 1634, 1507, 1380, 1253, 1126, 999, 872, 745, 618, 491, 364, 237, 110, 111, 238, 365, 492, 619, 746, 873, 1000, 1127, 1254, 1381, 1508, 1635, 1762, 1889, 2016, 2143, 2270, 2397, 2524, 2651, 2778, 2905, 3032, 3159, 3286, 3413, 3540, 3667, 3794, 3921, 4048,
    4175, 4302, 4429, 4556, 4683, 4810, 4937, 5064, 5191, 5318, 5445, 5572, 5699, 5826, 5953, 6080, 6207, 6334, 6461, 6588, 6715, 6842, 6969, 7096, 7223, 7350, 7477, 7604, 7731, 7858, 7985, 8112, 8113, 7986, 7859, 7732, 7605, 7478, 7351, 7224, 7097, 6970, 6843, 6716, 6589, 6462, 6335, 6208, 6081, 5954, 5827, 5700, 5573, 5446, 5319, 5192, 5065, 4938, 4811, 4684, 4557, 4430, 4303, 4176, 4049, 3922, 3795, 3668, 3541, 3414, 3287, 3160, 3033, 2906, 2779, 2652, 2525, 2398, 2271, 2144, 2017, 1890, 1763, 1636, 1509, 1382, 1255, 1128, 1001, 874, 747, 620, 493, 366, 239, 112, 113, 240, 367, 494, 621, 748, 875, 1002, 1129, 1256, 1383, 1510, 1637, 1764, 1891, 2018, 2145, 2272, 2399, 2526, 2653, 2780, 2907, 3034, 3161, 3288, 3415, 3542, 3669, 3796, 3923, 4050,
    4177, 4304, 4431, 4558, 4685, 4812, 4939, 5066, 5193, 5320, 5447, 5574, 5701, 5828, 5955, 6082, 6209, 6336, 6463, 6590, 6717, 6844, 6971, 7098, 7225, 7352, 7479, 7606, 7733, 7860, 7987, 8114, 8115, 7988, 7861, 7734, 7607, 7480, 7353, 7226, 7099, 6972, 6845, 6718, 6591, 6464, 6337, 6210, 6083, 5956, 5829, 5702, 5575, 5448, 5321, 5194, 5067, 4940, 4813, 4686, 4559, 4432, 4305, 4178, 4051, 3924, 3797, 3670, 3543, 3416, 3289, 3162, 3035, 2908, 2781, 2654, 2527, 2400, 2273, 2146, 2019, 1892, 1765, 1638, 1511, 1384, 1257, 1130, 1003, 876, 749, 622, 495, 368, 241, 114, 115, 242, 369, 496, 623, 750, 877, 1004, 1131, 1258, 1385, 1512, 1639, 1766, 1893, 2020, 2147, 2274, 2401, 2528, 2655, 2782, 2909, 3036, 3163, 3290, 3417, 3544, 3671, 3798, 3925, 4052,
    4179, 4306, 4433, 4560, 4687, 4814, 4941, 5068, 5195, 5322, 5449, 5576, 5703, 5830, 5957, 6084, 6211, 6338, 6465, 6592, 6719, 6846, 6973, 7100, 7227, 7354, 7481, 7608, 7735, 7862, 7989, 8116, 8117, 7990, 7863, 7736, 7609, 7482, 7355, 7228, 7101, 6974, 6847, 6720, 6593, 6466, 6339, 6212, 6085, 5958, 5831, 5704, 5577, 5450, 5323, 5196, 5069, 4942, 4815, 4688, 4561, 4434, 4307, 4180, 4053, 3926, 3799, 3672, 3545, 3418, 3291, 3164, 3037, 2910, 2783, 2656, 2529, 2402, 2275, 2148, 2021, 1894, 1767, 1640, 1513, 1386, 1259, 1132, 1005, 878, 751, 624, 497, 370, 243, 116, 117, 244, 371, 498, 625, 752, 879, 1006, 1133, 1260, 1387, 1514, 1641, 1768, 1895, 2022, 2149, 2276, 2403, 2530, 2657, 2784, 2911, 3038, 3165, 3292, 3419, 3546, 3673, 3800, 3927, 4054,
    4181, 4308, 4435, 4562, 4689, 4816, 4943, 5070, 5197, 5324, 5451, 5578, 5705, 5832, 5959, 6086, 6213, 6340, 6467, 6594, 6721, 6848, 6975, 7102, 7229, 7356, 7483, 7610, 7737, 7864, 7991, 8118, 8119, 7992, 7865, 7738, 7611, 7484, 7357, 7230, 7103, 6976, 6849, 6722, 6595, 6468, 6341, 6214, 6087, 5960, 5833, 5706, 5579, 5452, 5325, 5198, 5071, 4944, 4817, 4690, 4563, 4436, 4309, 4182, 4055, 3928, 3801, 3674, 3547, 3420, 3293, 3166, 3039, 2912, 2785, 2658, 2531, 2404, 2277, 2150, 2023, 1896, 1769, 1642, 1515, 1388, 1261, 1134, 1007, 880, 753, 626, 499, 372, 245, 118, 119, 246, 373, 500, 627, 754, 881, 1008, 1135, 1262, 1389, 1516, 1643, 1770, 1897, 2024, 2151, 2278, 2405, 2532, 2659, 2786, 2913, 3040, 3167, 3294, 3421, 3548, 3675, 3802, 3929, 4056,
    4183, 4310, 4437, 4564, 4691, 4818, 4945, 5072, 5199, 5326, 5453, 5580, 5707, 5834, 5961, 6088, 6215, 6342, 6469, 6596, 6723, 6850, 6977, 7104, 7231, 7358, 7485, 7612, 7739, 7866, 7993, 8120, 8121, 7994, 7867, 7740, 7613, 7486, 7359, 7232, 7105, 6978, 6851, 6724, 6597, 6470, 6343, 6216, 6089, 5962, 5835, 5708, 5581, 5454, 5327, 5200, 5073, 4946, 4819, 4692, 4565, 4438, 4311, 4184, 4057, 3930, 3803, 3676, 3549, 3422, 3295, 3168, 3041, 2914, 2787, 2660, 2533, 2406, 2279, 2152, 2025, 1898, 1771, 1644, 1517, 1390, 1263, 1136, 1009, 882, 755, 628, 501, 374, 247, 120, 121, 248, 375, 502, 629, 756, 883, 1010, 1137, 1264, 1391, 1518, 1645, 1772, 1899, 2026, 2153, 2280, 2407, 2534, 2661, 2788, 2915, 3042, 3169, 3296, 3423, 3550, 3677, 3804, 3931, 4058,
    4185, 4312, 4439, 4566, 4693, 4820, 4947, 5074, 5201, 5328, 5455, 5582, 5709, 5836, 5963, 6090, 6217, 6344, 6471, 6598, 6725, 6852, 6979, 7106, 7233, 7360, 7487, 7614, 7741, 7868, 7995, 8122, 8123, 7996, 7869, 7742, 7615, 7488, 7361, 7234, 7107, 6980, 6853, 6726, 6599, 6472, 6345, 6218, 6091, 5964, 5837, 5710, 5583, 5456, 5329, 5202, 5075, 4948, 4821, 4694, 4567, 4440, 4313, 4186, 4059, 3932, 3805, 3678, 3551, 3424, 3297, 3170, 3043, 2916, 2789, 2662, 2535, 2408, 2281, 2154, 2027, 1900, 1773, 1646, 1519, 1392, 1265, 1138, 1011, 884, 757, 630, 503, 376, 249, 122, 123, 250, 377, 504, 631, 758, 885, 1012, 1139, 1266, 1393, 1520, 1647, 1774, 1901, 2028, 2155, 2282, 2409, 2536, 2663, 2790, 2917, 3044, 3171, 3298, 3425, 3552, 3679, 3806, 3933, 4060,
    4187, 4314, 4441, 4568, 4695, 4822, 4949, 5076, 5203, 5330, 5457, 5584, 5711, 5838, 5965, 6092, 6219, 6346, 6473, 6600, 6727, 6854, 6981, 7108, 7235, 7362, 7489, 7616, 7743, 7870, 7997, 8124, 8125, 7998, 7871, 7744, 7617, 7490, 7363, 7236, 7109, 6982, 6855, 6728, 6601, 6474, 6347, 6220, 6093, 5966, 5839, 5712, 5585, 5458, 5331, 5204, 5077, 4950, 4823, 4696, 4569, 4442, 4315, 4188, 4061, 3934, 3807, 3680, 3553, 3426, 3299, 3172, 3045, 2918, 2791, 2664, 2537, 2410, 2283, 2156, 2029, 1902, 1775, 1648, 1521, 1394, 1267, 1140, 1013, 886, 759, 632, 505, 378, 251, 124, 125, 252, 379, 506, 633, 760, 887, 1014, 1141, 1268, 1395, 1522, 1649, 1776, 1903, 2030, 2157, 2284, 2411, 2538, 2665, 2792, 2919, 3046, 3173, 3300, 3427, 3554, 3681, 3808, 3935, 4062,
    4189, 4316, 4443, 4570, 4697, 4824, 4951, 5078, 5205, 5332, 5459, 5586, 5713, 5840, 5967, 6094, 6221, 6348, 6475, 6602, 6729, 6856, 6983, 7110, 7237, 7364, 7491, 7618, 7745, 7872, 7999, 8126, 8127, 8000, 7873, 7746, 7619, 7492, 7365, 7238, 7111, 6984, 6857, 6730, 6603, 6476, 6349, 6222, 6095, 5968, 5841, 5714, 5587, 5460, 5333, 5206, 5079, 4952, 4825, 4698, 4571, 4444, 4317, 4190, 4063, 3936, 3809, 3682, 3555, 3428, 3301, 3174, 3047, 2920, 2793, 2666, 2539, 2412, 2285, 2158, 2031, 1904, 1777, 1650, 1523, 1396, 1269, 1142, 1015, 888, 761, 634, 507, 380, 253, 126, 127, 254, 381, 508, 635, 762, 889, 1016, 1143, 1270, 1397, 1524, 1651, 1778, 1905, 2032, 2159, 2286, 2413, 2540, 2667, 2794, 2921, 3048, 3175, 3302, 3429, 3556, 3683, 3810, 3937, 4064,
    4191, 4318, 4445, 4572, 4699, 4826, 4953, 5080, 5207, 5334, 5461, 5588, 5715, 5842, 5969, 6096, 6223, 6350, 6477, 6604, 6731, 6858, 6985, 7112, 7239, 7366, 7493, 7620, 7747, 7874, 8001, 8128, 8129, 8002, 7875, 7748, 7621, 7494, 7367, 7240, 7113, 6986, 6859, 6732, 6605, 6478, 6351, 6224, 6097, 5970, 5843, 5716, 5589, 5462, 5335, 5208, 5081, 4954, 4827, 4700, 4573, 4446, 4319, 4192, 4065, 3938, 3811, 3684, 3557, 3430, 3303, 3176, 3049, 2922, 2795, 2668, 2541, 2414, 2287, 2160, 2033, 1906, 1779, 1652, 1525, 1398, 1271, 1144, 1017, 890, 763, 636, 509, 382, 255, 383, 510, 637, 764, 891, 1018, 1145, 1272, 1399, 1526, 1653, 1780, 1907, 2034, 2161, 2288, 2415, 2542, 2669, 2796, 2923, 3050, 3177, 3304, 3431, 3558, 3685, 3812, 3939, 4066, 4193, 4320, 4447,
    4574, 4701, 4828, 4955, 5082, 5209, 5336, 5463, 5590, 5717, 5844, 5971, 6098, 6225, 6352, 6479, 6606, 6733, 6860, 6987, 7114, 7241, 7368, 7495, 7622, 7749, 7876, 8003, 8130, 8131, 8004, 7877, 7750, 7623, 7496, 7369, 7242, 7115, 6988, 6861, 6734, 6607, 6480, 6353, 6226, 6099, 5972, 5845, 5718, 5591, 5464, 5337, 5210, 5083, 4956, 4829, 4702, 4575, 4448, 4321, 4194, 4067, 3940, 3813, 3686, 3559, 3432, 3305, 3178, 3051, 2924, 2797, 2670, 2543, 2416, 2289, 2162, 2035, 1908, 1781, 1654, 1527, 1400, 1273, 1146, 1019, 892, 765, 638, 511, 639, 766, 893, 1020, 1147, 1274, 1401, 1528, 1655, 1782, 1909, 2036, 2163, 2290, 2417, 2544, 2671, 2798, 2925, 3052, 3179, 3306, 3433, 3560, 3687, 3814, 3941, 4068, 4195, 4322, 4449, 4576, 4703, 4830, 4957, 5084, 5211, 5338,
    5465, 5592, 5719, 5846, 5973, 6100, 6227, 6354, 6481, 6608, 6735, 6862, 6989, 7116, 7243, 7370, 7497, 7624, 7751, 7878, 8005, 8132, 8133, 8006, 7879, 7752, 7625, 7498, 7371, 7244, 7117, 6990, 6863, 6736, 6609, 6482, 6355, 6228, 6101, 5974, 5847, 5720, 5593, 5466, 5339, 5212, 5085, 4958, 4831, 4704, 4577, 4450, 4323, 4196, 4069, 3942, 3815, 3688, 3561, 3434, 3307, 3180, 3053, 2926, 2799, 2672, 2545, 2418, 2291, 2164, 2037, 1910, 1783, 1656, 1529, 1402, 1275, 1148, 1021, 894, 767, 895, 1022, 1149, 1276, 1403, 1530, 1657, 1784, 1911, 2038, 2165, 2292, 2419, 2546, 2673, 2800, 2927, 3054, 3181, 3308, 3435, 3562, 3689, 3816, 3943, 4070, 4197, 4324, 4451, 4578, 4705, 4832, 4959, 5086, 5213, 5340, 5467, 5594, 5721, 5848, 5975, 6102, 6229, 6356, 6483, 6610, 6737,
    6864, 6991, 7118, 7245, 7372, 7499, 7626, 7753, 7880, 8007, 8134, 8135, 8008, 7881, 7754, 7627, 7500, 7373, 7246, 7119, 6992, 6865, 6738, 6611, 6484, 6357, 6230, 6103, 5976, 5849, 5722, 5595, 5468, 5341, 5214, 5087, 4960, 4833, 4706, 4579, 4452, 4325, 4198, 4071, 3944, 3817, 3690, 3563, 3436, 3309, 3182, 3055, 2928, 2801, 2674, 2547, 2420, 2293, 2166, 2039, 1912, 1785, 1658, 1531, 1404, 1277, 1150, 1023, 1151, 1278, 1405, 1532, 1659, 1786, 1913, 2040, 2167, 2294, 2421, 2548, 2675, 2802, 2929, 3056, 3183, 3310, 3437, 3564, 3691, 3818, 3945, 4072, 4199, 4326, 4453, 4580, 4707, 4834, 4961, 5088, 5215, 5342, 5469, 5596, 5723, 5850, 5977, 6104, 6231, 6358, 6485, 6612, 6739, 6866, 6993, 7120, 7247, 7374, 7501, 7628, 7755, 7882, 8009, 8136, 8137, 8010, 7883, 7756,
    7629, 7502, 7375, 7248, 7121, 6994, 6867, 6740, 6613, 6486, 6359, 6232, 6105, 5978, 5851, 5724, 5597, 5470, 5343, 5216, 5089, 4962, 4835, 4708, 4581, 4454, 4327, 4200, 4073, 3946, 3819, 3692, 3565, 3438, 3311, 3184, 3057, 2930, 2803, 2676, 2549, 2422, 2295, 2168, 2041, 1914, 1787, 1660, 1533, 1406, 1279, 1407, 1534, 1661, 1788, 1915, 2042, 2169, 2296, 2423, 2550, 2677, 2804, 2931, 3058, 3185, 3312, 3439, 3566, 3693, 3820, 3947, 4074, 4201, 4328, 4455, 4582, 4709, 4836, 4963, 5090, 5217, 5344, 5471, 5598, 5725, 5852, 5979, 6106, 6233, 6360, 6487, 6614, 6741, 6868, 6995, 7122, 7249, 7376, 7503, 7630, 7757, 7884, 8011, 8138, 8139, 8012, 7885, 7758, 7631, 7504, 7377, 7250, 7123, 6996, 6869, 6742, 6615, 6488, 6361, 6234, 6107, 5980, 5853, 5726, 5599, 5472, 5345,
    5218, 5091, 4964, 4837, 4710, 4583, 4456, 4329, 4202, 4075, 3948, 3821, 3694, 3567, 3440, 3313, 3186, 3059, 2932, 2805, 2678, 2551, 2424, 2297, 2170, 2043, 1916, 1789, 1662, 1535, 1663, 1790, 1917, 2044, 2171, 2298, 2425, 2552, 2679, 2806, 2933, 3060, 3187, 3314, 3441, 3568, 3695, 3822, 3949, 4076, 4203, 4330, 4457, 4584, 4711, 4838, 4965, 5092, 5219, 5346, 5473, 5600, 5727, 5854, 5981, 6108, 6235, 6362, 6489, 6616, 6743, 6870, 6997, 7124, 7251, 7378, 7505, 7632, 7759, 7886, 8013, 8140, 8141, 8014, 7887, 7760, 7633, 7506, 7379, 7252, 7125, 6998, 6871, 6744, 6617, 6490, 6363, 6236, 6109, 5982, 5855, 5728, 5601, 5474, 5347, 5220, 5093, 4966, 4839, 4712, 4585, 4458, 4331, 4204, 4077, 3950, 3823, 3696, 3569, 3442, 3315, 3188, 3061, 2934, 2807, 2680, 2553, 2426,
    2299, 2172, 2045, 1918, 1791, 1919, 2046, 2173, 2300, 2427, 2554, 2681, 2808, 2935, 3062, 3189, 3316, 3443, 3570, 3697, 3824, 3951, 4078, 4205, 4332, 4459, 4586, 4713, 4840, 4967, 5094, 5221, 5348, 5475, 5602, 5729, 5856, 5983, 6110, 6237, 6364, 6491, 6618, 6745, 6872, 6999, 7126, 7253, 7380, 7507, 7634, 7761, 7888, 8015, 8142, 8143, 8016, 7889, 7762, 7635, 7508, 7381, 7254, 7127, 7000, 6873, 6746, 6619, 6492, 6365, 6238, 6111, 5984, 5857, 5730, 5603, 5476, 5349, 5222, 5095, 4968, 4841, 4714, 4587, 4460, 4333, 4206, 4079, 3952, 3825, 3698, 3571, 3444, 3317, 3190, 3063, 2936, 2809, 2682, 2555, 2428, 2301, 2174, 2047, 2175, 2302, 2429, 2556, 2683, 2810, 2937, 3064, 3191, 3318, 3445, 3572, 3699, 3826, 3953, 4080, 4207, 4334, 4461, 4588, 4715, 4842, 4969, 5096,
    5223, 5350, 5477, 5604, 5731, 5858, 5985, 6112, 6239, 6366, 6493, 6620, 6747, 6874, 7001, 7128, 7255, 7382, 7509, 7636, 7763, 7890, 8017, 8144, 8145, 8018, 7891, 7764, 7637, 7510, 7383, 7256, 7129, 7002, 6875, 6748, 6621, 6494, 6367, 6240, 6113, 5986, 5859, 5732, 5605, 5478, 5351, 5224, 5097, 4970, 4843, 4716, 4589, 4462, 4335, 4208, 4081, 3954, 3827, 3700, 3573, 3446, 3319, 3192, 3065, 2938, 2811, 2684, 2557, 2430, 2303, 2431, 2558, 2685, 2812, 2939, 3066, 3193, 3320, 3447, 3574, 3701, 3828, 3955, 4082, 4209, 4336, 4463, 4590, 4717, 4844, 4971, 5098, 5225, 5352, 5479, 5606, 5733, 5860, 5987, 6114, 6241, 6368, 6495, 6622, 6749, 6876, 7003, 7130, 7257, 7384, 7511, 7638, 7765, 7892, 8019, 8146, 8147, 8020, 7893, 7766, 7639, 7512, 7385, 7258, 7131, 7004, 6877,
    6750, 6623, 6496, 6369, 6242, 6115, 5988, 5861, 5734, 5607, 5480, 5353, 5226, 5099, 4972, 4845, 4718, 4591, 4464, 4337, 4210, 4083, 3956, 3829, 3702, 3575, 3448, 3321, 3194, 3067, 2940, 2813, 2686, 2559, 2687, 2814, 2941, 3068, 3195, 3322, 3449, 3576, 3703, 3830, 3957, 4084, 4211, 4338, 4465, 4592, 4719, 4846, 4973, 5100, 5227, 5354, 5481, 5608, 5735, 5862, 5989, 6116, 6243, 6370, 6497, 6624, 6751, 6878, 7005, 7132, 7259, 7386, 7513, 7640, 7767, 7894, 8021, 8148, 8149, 8022, 7895, 7768, 7641, 7514, 7387, 7260, 7133, 7006, 6879, 6752, 6625, 6498, 6371, 6244, 6117, 5990, 5863, 5736, 5609, 5482, 5355, 5228, 5101, 4974, 4847, 4720, 4593, 4466, 4339, 4212, 4085, 3958, 3831, 3704, 3577, 3450, 3323, 3196, 3069, 2942, 2815, 2943, 3070, 3197, 3324, 3451, 3578, 3705,
    3832, 3959, 4086, 4213, 4340, 4467, 4594, 4721, 4848, 4975, 5102, 5229, 5356, 5483, 5610, 5737, 5864, 5991, 6118, 6245, 6372, 6499, 6626, 6753, 6880, 7007, 7134, 7261, 7388, 7515, 7642, 7769, 7896, 8023, 8150, 8151, 8024, 7897, 7770, 7643, 7516, 7389, 7262, 7135, 7008, 6881, 6754, 6627, 6500, 6373, 6246, 6119, 5992, 5865, 5738, 5611, 5484, 5357, 5230, 5103, 4976, 4849, 4722, 4595, 4468, 4341, 4214, 4087, 3960, 3833, 3706, 3579, 3452, 3325, 3198, 3071, 3199, 3326, 3453, 3580, 3707, 3834, 3961, 4088, 4215, 4342, 4469, 4596, 4723, 4850, 4977, 5104, 5231, 5358, 5485, 5612, 5739, 5866, 5993, 6120, 6247, 6374, 6501, 6628, 6755, 6882, 7009, 7136, 7263, 7390, 7517, 7644, 7771, 7898, 8025, 8152, 8153, 8026, 7899, 7772, 7645, 7518, 7391, 7264, 7137, 7010, 6883, 6756,
    6629, 6502, 6375, 6248, 6121, 5994, 5867, 5740, 5613, 5486, 5359, 5232, 5105, 4978, 4851, 4724, 4597, 4470, 4343, 4216, 4089, 3962, 3835, 3708, 3581, 3454, 3327, 3455, 3582, 3709, 3836, 3963, 4090, 4217, 4344, 4471, 4598, 4725, 4852, 4979, 5106, 5233, 5360, 5487, 5614, 5741, 5868, 5995, 6122, 6249, 6376, 6503, 6630, 6757, 6884, 7011, 7138, 7265, 7392, 7519, 7646, 7773, 7900, 8027, 8154, 8155, 8028, 7901, 7774, 7647, 7520, 7393, 7266, 7139, 7012, 6885, 6758, 6631, 6504, 6377, 6250, 6123, 5996, 5869, 5742, 5615, 5488, 5361, 5234, 5107, 4980, 4853, 4726, 4599, 4472, 4345, 4218, 4091, 3964, 3837, 3710, 3583, 3711, 3838, 3965, 4092, 4219, 4346, 4473, 4600, 4727, 4854, 4981, 5108, 5235, 5362, 5489, 5616, 5743, 5870, 5997, 6124, 6251, 6378, 6505, 6632, 6759, 6886,
    7013, 7140, 7267, 7394, 7521, 7648, 7775, 7902, 8029, 8156, 8157, 8030, 7903, 7776, 7649, 7522, 7395, 7268, 7141, 7014, 6887, 6760, 6633, 6506, 6379, 6252, 6125, 5998, 5871, 5744, 5617, 5490, 5363, 5236, 5109, 4982, 4855, 4728, 4601, 4474, 4347, 4220, 4093, 3966, 3839, 3967, 4094, 4221, 4348, 4475, 4602, 4729, 4856, 4983, 5110, 5237, 5364, 5491, 5618, 5745, 5872, 5999, 6126, 6253, 6380, 6507, 6634, 6761, 6888, 7015, 7142, 7269, 7396, 7523, 7650, 7777, 7904, 8031, 8158, 8159, 8032, 7905, 7778, 7651, 7524, 7397, 7270, 7143, 7016, 6889, 6762, 6635, 6508, 6381, 6254, 6127, 6000, 5873, 5746, 5619, 5492, 5365, 5238, 5111, 4984, 4857, 4730, 4603, 4476, 4349, 4222, 4095, 4223, 4350, 4477, 4604, 4731, 4858, 4985, 5112, 5239, 5366, 5493, 5620, 5747, 5874, 6001, 6128,
    6255, 6382, 6509, 6636, 6763, 6890, 7017, 7144, 7271, 7398, 7525, 7652, 7779, 7906, 8033, 8160, 8161, 8034, 7907, 7780, 7653, 7526, 7399, 7272, 7145, 7018, 6891, 6764, 6637, 6510, 6383, 6256, 6129, 6002, 5875, 5748, 5621, 5494, 5367, 5240, 5113, 4986, 4859, 4732, 4605, 4478, 4351, 4479, 4606, 4733, 4860, 4987, 5114, 5241, 5368, 5495, 5622, 5749, 5876, 6003, 6130, 6257, 6384, 6511, 6638, 6765, 6892, 7019, 7146, 7273, 7400, 7527, 7654, 7781, 7908, 8035, 8162, 8163, 8036, 7909, 7782, 7655, 7528, 7401, 7274, 7147, 7020, 6893, 6766, 6639, 6512, 6385, 6258, 6131, 6004, 5877, 5750, 5623, 5496, 5369, 5242, 5115, 4988, 4861, 4734, 4607, 4735, 4862, 4989, 5116, 5243, 5370, 5497, 5624, 5751, 5878, 6005, 6132, 6259, 6386, 6513, 6640, 6767, 6894, 7021, 7148, 7275, 7402,
    7529, 7656, 7783, 7910, 8037, 8164, 8165, 8038, 7911, 7784, 7657, 7530, 7403, 7276, 7149, 7022, 6895, 6768, 6641, 6514, 6387, 6260, 6133, 6006, 5879, 5752, 5625, 5498, 5371, 5244, 5117, 4990, 4863, 4991, 5118, 5245, 5372, 5499, 5626, 5753, 5880, 6007, 6134, 6261, 6388, 6515, 6642, 6769, 6896, 7023, 7150, 7277, 7404, 7531, 7658, 7785, 7912, 8039, 8166, 8167, 8040, 7913, 7786, 7659, 7532, 7405, 7278, 7151, 7024, 6897, 6770, 6643, 6516, 6389, 6262, 6135, 6008, 5881, 5754, 5627, 5500, 5373, 5246, 5119, 5247, 5374, 5501, 5628, 5755, 5882, 6009, 6136, 6263, 6390, 6517, 6644, 6771, 6898, 7025, 7152, 7279, 7406, 7533, 7660, 7787, 7914, 8041, 8168, 8169, 8042, 7915, 7788, 7661, 7534, 7407, 7280, 7153, 7026, 6899, 6772, 6645, 6518, 6391, 6264, 6137, 6010, 5883, 5756,
    5629, 5502, 5375, 5503, 5630, 5757, 5884, 6011, 6138, 6265, 6392, 6519, 6646, 6773, 6900, 7027, 7154, 7281, 7408, 7535, 7662, 7789, 7916, 8043, 8170, 8171, 8044, 7917, 7790, 7663, 7536, 7409, 7282, 7155, 7028, 6901, 6774, 6647, 6520, 6393, 6266, 6139, 6012, 5885, 5758, 5631, 5759, 5886, 6013, 6140, 6267, 6394, 6521, 6648, 6775, 6902, 7029, 7156, 7283, 7410, 7537, 7664, 7791, 7918, 8045, 8172, 8173, 8046, 7919, 7792, 7665, 7538, 7411, 7284, 7157, 7030, 6903, 6776, 6649, 6522, 6395, 6268, 6141, 6014, 5887, 6015, 6142, 6269, 6396, 6523, 6650, 6777, 6904, 7031, 7158, 7285, 7412, 7539, 7666, 7793, 7920, 8047, 8174, 8175, 8048, 7921, 7794, 7667, 7540, 7413, 7286, 7159, 7032, 6905, 6778, 6651, 6524, 6397, 6270, 6143, 6271, 6398, 6525, 6652, 6779, 6906, 7033, 7160,
    7287, 7414, 7541, 7668, 7795, 7922, 8049, 8176, 8177, 8050, 7923, 7796, 7669, 7542, 7415, 7288, 7161, 7034, 6907, 6780, 6653, 6526, 6399, 6527, 6654, 6781, 6908, 7035, 7162, 7289, 7416, 7543, 7670, 7797, 7924, 8051, 8178, 8179, 8052, 7925, 7798, 7671, 7544, 7417, 7290, 7163, 7036, 6909, 6782, 6655, 6783, 6910, 7037, 7164, 7291, 7418, 7545, 7672, 7799, 7926, 8053, 8180, 8181, 8054, 7927, 7800, 7673, 7546, 7419, 7292, 7165, 7038, 6911, 7039, 7166, 7293, 7420, 7547, 7674, 7801, 7928, 8055, 8182, 8183, 8056, 7929, 7802, 7675, 7548, 7421, 7294, 7167, 7295, 7422, 7549, 7676, 7803, 7930, 8057, 8184, 8185, 8058, 7931, 7804, 7677, 7550, 7423, 7551, 7678, 7805, 7932, 8059, 8186, 8187, 8060, 7933, 7806, 7679, 7807, 7934, 8061, 8188, 8189, 8062, 7935, 8063, 8190, 8191,

    // 128 x 2:
    0, 1, 2, 4, 3, 5, 6, 8, 7, 9, 10, 12, 11, 13, 14, 16, 15, 17, 18, 20, 19, 21, 22, 24, 23, 25, 26, 28, 27, 29, 30, 32, 31, 33, 34, 36, 35, 37, 38, 40, 39, 41, 42, 44, 43, 45, 46, 48, 47, 49, 50, 52, 51, 53, 54, 56, 55, 57, 58, 60, 59, 61, 62, 64, 63, 65, 66, 68, 67, 69, 70, 72, 71, 73, 74, 76, 75, 77, 78, 80, 79, 81, 82, 84, 83, 85, 86, 88, 87, 89, 90, 92, 91, 93, 94, 96, 95, 97, 98, 100, 99, 101, 102, 104, 103, 105, 106, 108, 107, 109, 110, 112, 111, 113, 114, 116, 115, 117, 118, 120, 119, 121, 122, 124, 123, 125, 126, 128,
    127, 129, 130, 132, 131, 133, 134, 136, 135, 137, 138, 140, 139, 141, 142, 144, 143, 145, 146, 148, 147, 149, 150, 152, 151, 153, 154, 156, 155, 157, 158, 160, 159, 161, 162, 164, 163, 165, 166, 168, 167, 169, 170, 172, 171, 173, 174, 176, 175, 177, 178, 180, 179, 181, 182, 184, 183, 185, 186, 188, 187, 189, 190, 192, 191, 193, 194, 196, 195, 197, 198, 200, 199, 201, 202, 204, 203, 205, 206, 208, 207, 209, 210, 212, 211, 213, 214, 216, 215, 217, 218, 220, 219, 221, 222, 224, 223, 225, 226, 228, 227, 229, 230, 232, 231, 233, 234, 236, 235, 237, 238, 240, 239, 241, 242, 244, 243, 245, 246, 248, 247, 249, 250, 252, 251, 253, 254, 255,

    // 128 x 4:
    0, 1, 4, 8, 5, 2, 3, 6, 9, 12, 16, 13, 10, 7, 11, 14, 17, 20, 24, 21, 18, 15, 19, 22, 25, 28, 32, 29, 26, 23, 27, 30, 33, 36, 40, 37, 34, 31, 35, 38, 41, 44, 48, 45, 42, 39, 43, 46, 49, 52, 56, 53, 50, 47, 51, 54, 57, 60, 64, 61, 58, 55, 59, 62, 65, 68, 72, 69, 66, 63, 67, 70, 73, 76, 80, 77, 74, 71, 75, 78, 81, 84, 88, 85, 82, 79, 83, 86, 89, 92, 96, 93, 90, 87, 91, 94, 97, 100, 104, 101, 98, 95, 99, 102, 105, 108, 112, 109, 106, 103, 107, 110, 113, 116, 120, 117, 114, 111, 115, 118, 121, 124, 128, 125, 122, 119, 123, 126,
    129, 132, 136, 133, 130, 127, 131, 134, 137, 140, 144, 141, 138, 135, 139, 142, 145, 148, 152, 149, 146, 143, 147, 150, 153, 156, 160, 157, 154, 151, 155, 158, 161, 164, 168, 165, 162, 159, 163, 166, 169, 172, 176, 173, 170, 167, 171, 174, 177, 180, 184, 181, 178, 175, 179, 182, 185, 188, 192, 189, 186, 183, 187, 190, 193, 196, 200, 197, 194, 191, 195, 198, 201, 204, 208, 205, 202, 199, 203, 206, 209, 212, 216, 213, 210, 207, 211, 214, 217, 220, 224, 221, 218, 215, 219, 222, 225, 228, 232, 229, 226, 223, 227, 230, 233, 236, 240, 237, 234, 231, 235, 238, 241, 244, 248, 245, 242, 239, 243, 246, 249, 252, 256, 253, 250, 247, 251, 254,
    257, 260, 264, 261, 258, 255, 259, 262, 265, 268, 272, 269, 266, 263, 267, 270, 273, 276, 280, 277, 274, 271, 275, 278, 281, 284, 288, 285, 282, 279, 283, 286, 289, 292, 296, 293, 290, 287, 291, 294, 297, 300, 304, 301, 298, 295, 299, 302, 305, 308, 312, 309, 306, 303, 307, 310, 313, 316, 320, 317, 314, 311, 315, 318, 321, 324, 328, 325, 322, 319, 323, 326, 329, 332, 336, 333, 330, 327, 331, 334, 337, 340, 344, 341, 338, 335, 339, 342, 345, 348, 352, 349, 346, 343, 347, 350, 353, 356, 360, 357, 354, 351, 355, 358, 361, 364, 368, 365, 362, 359, 363, 366, 369, 372, 376, 373, 370, 367, 371, 374, 377, 380, 384, 381, 378, 375, 379, 382,
    385, 388, 392, 389, 386, 383, 387, 390, 393, 396, 400, 397, 394, 391, 395, 398, 401, 404, 408, 405, 402, 399, 403, 406, 409, 412, 416, 413, 410, 407, 411, 414, 417, 420, 424, 421, 418, 415, 419, 422, 425, 428, 432, 429, 426, 423, 427, 430, 433, 436, 440, 437, 434, 431, 435, 438, 441, 444, 448, 445, 442, 439, 443, 446, 449, 452, 456, 453, 450, 447, 451, 454, 457, 460, 464, 461, 458, 455, 459, 462, 465, 468, 472, 469, 466, 463, 467, 470, 473, 476, 480, 477, 474, 471, 475, 478, 481, 484, 488, 485, 482, 479, 483, 486, 489, 492, 496, 493, 490, 487, 491, 494, 497, 500, 504, 501, 498, 495, 499, 502, 505, 508, 509, 506, 503, 507, 510, 511,

    // 128 x 8:
    0, 1, 8, 16, 9, 2, 3, 10, 17, 24, 32, 25, 18, 11, 4, 5, 12, 19, 26, 33, 40, 48, 41, 34, 27, 20, 13, 6, 7, 14, 21, 28, 35, 42, 49, 56, 64, 57, 50, 43, 36, 29, 22, 15, 23, 30, 37, 44, 51, 58, 65, 72, 80, 73, 66, 59, 52, 45, 38, 31, 39, 46, 53, 60, 67, 74, 81, 88, 96, 89, 82, 75, 68, 61, 54, 47, 55, 62, 69, 76, 83, 90, 97, 104, 112, 105, 98, 91, 84, 77, 70, 63, 71, 78, 85, 92, 99, 106, 113, 120, 128, 121, 114, 107, 100, 93, 86, 79, 87, 94, 101, 108, 115, 122, 129, 136, 144, 137, 130, 123, 116, 109, 102, 95, 103, 110, 117, 124,
    131, 138, 145, 152, 160, 153, 146, 139, 132, 125, 118, 111, 119, 126, 133, 140, 147, 154, 161, 168, 176, 169, 162, 155, 148, 141, 134, 127, 135, 142, 149, 156, 163, 170, 177, 184, 192, 185, 178, 171, 164, 157, 150, 143, 151, 158, 165, 172, 179, 186, 193, 200, 208, 201, 194, 187, 180, 173, 166, 159, 167, 174, 181, 188, 195, 202, 209, 216, 224, 217, 210, 203, 196, 189, 182, 175, 183, 190, 197, 204, 211, 218, 225, 232, 240, 233, 226, 219, 212, 205, 198, 191, 199, 206, 213, 220, 227, 234, 241, 248, 256, 249, 242, 235, 228, 221, 214, 207, 215, 222, 229, 236, 243, 250, 257, 264, 272, 265, 258, 251, 244, 237, 230, 223, 231, 238, 245, 252,
    259, 266, 273, 280, 288, 281, 274, 267, 260, 253, 246, 239, 247, 254, 261, 268, 275, 282, 289, 296, 304, 297, 290, 283, 276, 269, 262, 255, 263, 270, 277, 284, 291, 298, 305, 312, 320, 313, 306, 299, 292, 285, 278, 271, 279, 286, 293, 300, 307, 314, 321, 328, 336, 329, 322, 315, 308, 301, 294, 287, 295, 302, 309, 316, 323, 330, 337, 344, 352, 345, 338, 331, 324, 317, 310, 303, 311, 318, 325, 332, 339, 346, 353, 360, 368, 361, 354, 347, 340, 333, 326, 319, 327, 334, 341, 348, 355, 362, 369, 376, 384, 377, 370, 363, 356, 349, 342, 335, 343, 350, 357, 364, 371, 378, 385, 392, 400, 393, 386, 379, 372, 365, 358, 351, 359, 366, 373, 380,
    387, 394, 401, 408, 416, 409, 402, 395, 388, 381, 374, 367, 375, 382, 389, 396, 403, 410, 417, 424, 432, 425, 418, 411, 404, 397, 390, 383, 391, 398, 405, 412, 419, 426, 433, 440, 448, 441, 434, 427, 420, 413, 406, 399, 407, 414, 421, 428, 435, 442, 449, 456, 464, 457, 450, 443, 436, 429, 422, 415, 423, 430, 437, 444, 451, 458, 465, 472, 480, 473, 466, 459, 452, 445, 438, 431, 439, 446, 453, 460, 467, 474, 481, 488, 496, 489, 482, 475, 468, 461, 454, 447, 455, 462, 469, 476, 483, 490, 497, 504, 512, 505, 498, 491, 484, 477, 470, 463, 471, 478, 485, 492, 499, 506, 513, 520, 528, 521, 514, 507, 500, 493, 486, 479, 487, 494, 501, 508,
    515, 522, 529, 536, 544, 537, 530, 523, 516, 509, 502, 495, 503, 510, 517, 524, 531, 538, 545, 552, 560, 553, 546, 539, 532, 525, 518, 511, 519, 526, 533, 540, 547, 554, 561, 568, 576, 569, 562, 555, 548, 541, 534, 527, 535, 542, 549, 556, 563, 570, 577, 584, 592, 585, 578, 571, 564, 557, 550, 543, 551, 558, 565, 572, 579, 586, 593, 600, 608, 601, 594, 587, 580, 573, 566, 559, 567, 574, 581, 588, 595, 602, 609, 616, 624, 617, 610, 603, 596, 589, 582, 575, 583, 590, 597, 604, 611, 618, 625, 632, 640, 633, 626, 619, 612, 605, 598, 591, 599, 606, 613, 620, 627, 634, 641, 648, 656, 649, 642, 635, 628, 621, 614, 607, 615, 622, 629, 636,
    643, 650, 657, 664, 672, 665, 658, 651, 644, 637, 630, 623, 631, 638, 645, 652, 659, 666, 673, 680, 688, 681, 674, 667, 660, 653, 646, 639, 647, 654, 661, 668, 675, 682, 689, 696, 704, 697, 690, 683, 676, 669, 662, 655, 663, 670, 677, 684, 691, 698, 705, 712, 720, 713, 706, 699, 692, 685, 678, 671, 679, 686, 693, 700, 707, 714, 721, 728, 736, 729, 722, 715, 708, 701, 694, 687, 695, 702, 709, 716, 723, 730, 737, 744, 752, 745, 738, 731, 724, 717, 710, 703, 711, 718, 725, 732, 739, 746, 753, 760, 768, 761, 754, 747, 740, 733, 726, 719, 727, 734, 741, 748, 755, 762, 769, 776, 784, 777, 770, 763, 756, 749, 742, 735, 743, 750, 757, 764,
    771, 778, 785, 792, 800, 793, 786, 779, 772, 765, 758, 751, 759, 766, 773, 780, 787, 794, 801, 808, 816, 809, 802, 795, 788, 781, 774, 767, 775, 782, 789, 796, 803, 810, 817, 824, 832, 825, 818, 811, 804, 797, 790, 783, 791, 798, 805, 812, 819, 826, 833, 840, 848, 841, 834, 827, 820, 813, 806, 799, 807, 814, 821, 828, 835, 842, 849, 856, 864, 857, 850, 843, 836, 829, 822, 815, 823, 830, 837, 844, 851, 858, 865, 872, 880, 873, 866, 859, 852, 845, 838, 831, 839, 846, 853, 860, 867, 874, 881, 888, 896, 889, 882, 875, 868, 861, 854, 847, 855, 862, 869, 876, 883, 890, 897, 904, 912, 905, 898, 891, 884, 877, 870, 863, 871, 878, 885, 892,
    899, 906, 913, 920, 928, 921, 914, 907, 900, 893, 886, 879, 887, 894, 901, 908, 915, 922, 929, 936, 944, 937, 930, 923, 916, 909, 902, 895, 903, 910, 917, 924, 931, 938, 945, 952, 960, 953, 946, 939, 932, 925, 918, 911, 919, 926, 933, 940, 947, 954, 961, 968, 976, 969, 962, 955, 948, 941, 934, 927, 935, 942, 949, 956, 963, 970, 977, 984, 992, 985, 978, 971, 964, 957, 950, 943, 951, 958, 965, 972, 979, 986, 993, 1000, 1008, 1001, 994, 987, 980, 973, 966, 959, 967, 974, 981, 988, 995, 1002, 1009, 1016, 1017, 1010, 1003, 996, 989, 982, 975, 983, 990, 997, 1004, 1011, 1018, 1019, 1012, 1005, 998, 991, 999, 1006, 1013, 1020, 1021, 1014, 1007, 1015, 1022, 1023,

    // 128 x 16:
    0, 1, 16, 32, 17, 2, 3, 18, 33, 48, 64, 49, 34, 19, 4, 5, 20, 35, 50, 65, 80, 96, 81, 66, 51, 36, 21, 6, 7, 22, 37, 52, 67, 82, 97, 112, 128, 113, 98, 83, 68, 53, 38, 23, 8, 9, 24, 39, 54, 69, 84, 99, 114, 129, 144, 160, 145, 130, 115, 100, 85, 70, 55, 40, 25, 10, 11, 26, 41, 56, 71, 86, 101, 116, 131, 146, 161, 176, 192, 177, 162, 147, 132, 117, 102, 87, 72, 57, 42, 27, 12, 13, 28, 43, 58, 73, 88, 103, 118, 133, 148, 163, 178, 193, 208, 224, 209, 194, 179, 164, 149, 134, 119, 104, 89, 74, 59, 44, 29, 14, 15, 30, 45, 60, 75, 90, 105, 120,
    135, 150, 165, 180, 195, 210, 225, 240, 256, 241, 226, 211, 196, 181, 166, 151, 136, 121, 106, 91, 76, 61, 46, 31, 47, 62, 77, 92, 107, 122, 137, 152, 167, 182, 197, 212, 227, 242, 257, 272, 288, 273, 258, 243, 228, 213, 198, 183, 168, 153, 138, 123, 108, 93, 78, 63, 79, 94, 109, 124, 139, 154, 169, 184, 199, 214, 229, 244, 259, 274, 289, 304, 320, 305, 290, 275, 260, 245, 230, 215, 200, 185, 170, 155, 140, 125, 110, 95, 111, 126, 141, 156, 171, 186, 201, 216, 231, 246, 261, 276, 291, 306, 321, 336, 352, 337, 322, 307, 292, 277, 262, 247, 232, 217, 202, 187, 172, 157, 142, 127, 143, 158, 173, 188, 203, 218, 233, 248,
    263, 278, 293, 308, 323, 338, 353, 368, 384, 369, 354, 339, 324, 309, 294, 279, 264, 249, 234, 219, 204, 189, 174, 159, 175, 190, 205, 220, 235, 250, 265, 280, 295, 310, 325, 340, 355, 370, 385, 400, 416, 401, 386, 371, 356, 341, 326, 311, 296, 281, 266, 251, 236, 221, 206, 191, 207, 222, 237, 252, 267, 282, 297, 312, 327, 342, 357, 372, 387, 402, 417, 432, 448, 433, 418, 403, 388, 373, 358, 343, 328, 313, 298, 283, 268, 253, 238, 223, 239, 254, 269, 284, 299, 314, 329, 344, 359, 374, 389, 404, 419, 434, 449, 464, 480, 465, 450, 435, 420, 405, 390, 375, 360, 345, 330, 315, 300, 285, 270, 255, 271, 286, 301, 316, 331, 346, 361, 376,
    391, 406, 421, 436, 451, 466, 481, 496, 512, 497, 482, 467, 452, 437, 422, 407, 392, 377, 362, 347, 332, 317, 302, 287, 303, 318, 333, 348, 363, 378, 393, 408, 423, 438, 453, 468, 483, 498, 513, 528, 544, 529, 514, 499, 484, 469, 454, 439, 424, 409, 394, 379, 364, 349, 334, 319, 335, 350, 365, 380, 395, 410, 425, 440, 455, 470, 485, 500, 515, 530, 545, 560, 576, 561, 546, 531, 516, 501, 486, 471, 456, 441, 426, 411, 396, 381, 366, 351, 367, 382, 397, 412, 427, 442, 457, 472, 487, 502, 517, 532, 547, 562, 577, 592, 608, 593, 578, 563, 548, 533, 518, 503, 488, 473, 458, 443, 428, 413, 398, 383, 399, 414, 429, 444, 459, 474, 489, 504,
    519, 534, 549, 564, 579, 594, 609, 624, 640, 625, 610, 595, 580, 565, 550, 535, 520, 505, 490, 475, 460, 445, 430, 415, 431, 446, 461, 476, 491, 506, 521, 536, 551, 566, 581, 596, 611, 626, 641, 656, 672, 657, 642, 627, 612, 597, 582, 567, 552, 537, 522, 507, 492, 477, 462, 447, 463, 478, 493, 508, 523, 538, 553, 568, 583, 598, 613, 628, 643, 658, 673, 688, 704, 689, 674, 659, 644, 629, 614, 599, 584, 569, 554, 539, 524, 509, 494, 479, 495, 510, 525, 540, 555, 570, 585, 600, 615, 630, 645, 660, 675, 690, 705, 720, 736, 721, 706, 691, 676, 661, 646, 631, 616, 601, 586, 571, 556, 541, 526, 511, 527, 542, 557, 572, 587, 602, 617, 632,
    647, 662, 677, 692, 707, 722, 737, 752, 768, 753, 738, 723, 708, 693, 678, 663, 648, 633, 618, 603, 588, 573, 558, 543, 559, 574, 589, 604, 619, 634, 649, 664, 679, 694, 709, 724, 739, 754, 769, 784, 800, 785, 770, 755, 740, 725, 710, 695, 680, 665, 650, 635, 620, 605, 590, 575, 591, 606, 621, 636, 651, 666, 681, 696, 711, 726, 741, 756, 771, 786, 801, 816, 832, 817, 802, 787, 772, 757, 742, 727, 712, 697, 682, 667, 652, 637, 622, 607, 623, 638, 653, 668, 683, 698, 713, 728, 743, 758, 773, 788, 803, 818, 833, 848, 864, 849, 834, 819, 804, 789, 774, 759, 744, 729, 714, 699, 684, 669, 654, 639, 655, 670, 685, 700, 715, 730, 745, 760,
    775, 790, 805, 820, 835, 850, 865, 880, 896, 881, 866, 851, 836, 821, 806, 791, 776, 761, 746, 731, 716, 701, 686, 671, 687, 702, 717, 732, 747, 762, 777, 792, 807, 822, 837, 852, 867, 882, 897, 912, 928, 913, 898, 883, 868, 853, 838, 823, 808, 793, 778, 763, 748, 733, 718, 703, 719, 734, 749, 764, 779, 794, 809, 824, 839, 854, 869, 884, 899, 914, 929, 944, 960, 945, 930, 915, 900, 885, 870, 855, 840, 825, 810, 795, 780, 765, 750, 735, 751, 766, 781, 796, 811, 826, 841, 856, 871, 886, 901, 916, 931, 946, 961, 976, 992, 977, 962, 947, 932, 917, 902, 887, 872, 857, 842, 827, 812, 797, 782, 767, 783, 798, 813, 828, 843, 858, 873, 888,
    903, 918, 933, 948, 963, 978, 993, 1008, 1024, 1009, 994, 979, 964, 949, 934, 919, 904, 889, 874, 859, 844, 829, 814, 799, 815, 830, 845, 860, 875, 890, 905, 920, 935, 950, 965, 980, 995, 1010, 1025, 1040, 1056, 1041, 1026, 1011, 996, 981, 966, 951, 936, 921, 906, 891, 876, 861, 846, 831, 847, 862, 877, 892, 907, 922, 937, 952, 967, 982, 997, 1012, 1027, 1042, 1057, 1072, 1088, 1073, 1058, 1043, 1028, 1013, 998, 983, 968, 953, 938, 923, 908, 893, 878, 863, 879, 894, 909, 924, 939, 954, 969, 984, 999, 1014, 1029, 1044, 1059, 1074, 1089, 1104, 1120, 1105, 1090, 1075, 1060, 1045, 1030, 1015, 1000, 985, 970, 955, 940, 925, 910, 895, 911, 926, 941, 956, 971, 986, 1001, 1016,
    1031, 1046, 1061, 1076, 1091, 1106, 1121, 1136, 1152, 1137, 1122, 1107, 1092, 1077, 1062, 1047, 1032, 1017, 1002, 987, 972, 957, 942, 927, 943, 958, 973, 988, 1003, 1018, 1033, 1048, 1063, 1078, 1093, 1108, 1123, 1138, 1153, 1168, 1184, 1169, 1154, 1139, 1124, 1109, 1094, 1079, 1064, 1049, 1034, 1019, 1004, 989, 974, 959, 975, 990, 1005, 1020, 1035, 1050, 1065, 1080, 1095, 1110, 1125, 1140, 1155, 1170, 1185, 1200, 1216, 1201, 1186, 1171, 1156, 1141, 1126, 1111, 1096, 1081, 1066, 1051, 1036, 1021, 1006, 991, 1007, 1022, 1037, 1052, 1067, 1082, 1097, 1112, 1127, 1142, 1157, 1172, 1187, 1202, 1217, 1232, 1248, 1233, 1218, 1203, 1188, 1173, 1158, 1143, 1128, 1113, 1098, 1083, 1068, 1053, 1038, 1023, 1039, 1054, 1069, 1084, 1099, 1114, 1129, 1144,
    1159, 1174, 1189, 1204, 1219, 1234, 1249, 1264, 1280, 1265, 1250, 1235, 1220, 1205, 1190, 1175, 1160, 1145, 1130, 1115, 1100, 1085, 1070, 1055, 1071, 1086, 1101, 1116, 1131, 1146, 1161, 1176, 1191, 1206, 1221, 1236, 1251, 1266, 1281, 1296, 1312, 1297, 1282, 1267, 1252, 1237, 1222, 1207, 1192, 1177, 1162, 1147, 1132, 1117, 1102, 1087, 1103, 1118, 1133, 1148, 1163, 1178, 1193, 1208, 1223, 1238, 1253, 1268, 1283, 1298, 1313, 1328, 1344, 1329, 1314, 1299, 1284, 1269, 1254, 1239, 1224, 1209, 1194, 1179, 1164, 1149, 1134, 1119, 1135, 1150, 1165, 1180, 1195, 1210, 1225, 1240, 1255, 1270, 1285, 1300, 1315, 1330, 1345, 1360, 1376, 1361, 1346, 1331, 1316, 1301, 1286, 1271, 1256, 1241, 1226, 1211, 1196, 1181, 1166, 1151, 1167, 1182, 1197, 1212, 1227, 1242, 1257, 1272,
    1287, 1302, 1317, 1332, 1347, 1362, 1377, 1392, 1408, 1393, 1378, 1363, 1348, 1333, 1318, 1303, 1288, 1273, 1258, 1243, 1228, 1213, 1198, 1183, 1199, 1214, 1229, 1244, 1259, 1274, 1289, 1304, 1319, 1334, 1349, 1364, 1379, 1394, 1409, 1424, 1440, 1425, 1410, 1395, 1380, 1365, 1350, 1335, 1320, 1305, 1290, 1275, 1260, 1245, 1230, 1215, 1231, 1246, 1261, 1276, 1291, 1306, 1321, 1336, 1351, 1366, 1381, 1396, 1411, 1426, 1441, 1456, 1472, 1457, 1442, 1427, 1412, 1397, 1382, 1367, 1352, 1337, 1322, 1307, 1292, 1277, 1262, 1247, 1263, 1278, 1293, 1308, 1323, 1338, 1353, 1368, 1383, 1398, 1413, 1428, 1443, 1458, 1473, 1488, 1504, 1489, 1474, 1459, 1444, 1429, 1414, 1399, 1384, 1369, 1354, 1339, 1324, 1309, 1294, 1279, 1295, 1310, 1325, 1340, 1355, 1370, 1385, 1400,
    1415, 1430, 1445, 1460, 1475, 1490, 1505, 1520, 1536, 1521, 1506, 1491, 1476, 1461, 1446, 1431, 1416, 1401, 1386, 1371, 1356, 1341, 1326, 1311, 1327, 1342, 1357, 1372, 1387, 1402, 1417, 1432, 1447, 1462, 1477, 1492, 1507, 1522, 1537, 1552, 1568, 1553, 1538, 1523, 1508, 1493, 1478, 1463, 1448, 1433, 1418, 1403, 1388, 1373, 1358, 1343, 1359, 1374, 1389, 1404, 1419, 1434, 1449, 1464, 1479, 1494, 1509, 1524, 1539, 1554, 1569, 1584, 1600, 1585, 1570, 1555, 1540, 1525, 1510, 1495, 1480, 1465, 1450, 1435, 1420, 1405, 1390, 1375, 1391, 1406, 1421, 1436, 1451, 1466, 1481, 1496, 1511, 1526, 1541, 1556, 1571, 1586, 1601, 1616, 1632, 1617, 1602, 1587, 1572, 1557, 1542, 1527, 1512, 1497, 1482, 1467, 1452, 1437, 1422, 1407, 1423, 1438, 1453, 1468, 1483, 1498, 1513, 1528,
    1543, 1558, 1573, 1588, 1603, 1618, 1633, 1648, 1664, 1649, 1634, 1619, 1604, 1589, 1574, 1559, 1544, 1529, 1514, 1499, 1484, 1469, 1454, 1439, 1455, 1470, 1485, 1500, 1515, 1530, 1545, 1560, 1575, 1590, 1605, 1620, 1635, 1650, 1665, 1680, 1696, 1681, 1666, 1651, 1636, 1621, 1606, 1591, 1576, 1561, 1546, 1531, 1516, 1501, 1486, 1471, 1487, 1502, 1517, 1532, 1547, 1562, 1577, 1592, 1607, 1622, 1637, 1652, 1667, 1682, 1697, 1712, 1728, 1713, 1698, 1683, 1668, 1653, 1638, 1623, 1608, 1593, 1578, 1563, 1548, 1533, 1518, 1503, 1519, 1534, 1549, 1564, 1579, 1594, 1609, 1624, 1639, 1654, 1669, 1684, 1699, 1714, 1729, 1744, 1760, 1745, 1730, 1715, 1700, 1685, 1670, 1655, 1640, 1625, 1610, 1595, 1580, 1565, 1550, 1535, 1551, 1566, 1581, 1596, 1611, 1626, 1641, 1656,
    1671, 1686, 1701, 1716, 1731, 1746, 1761, 1776, 1792, 1777, 1762, 1747, 1732, 1717, 1702, 1687, 1672, 1657, 1642, 1627, 1612, 1597, 1582, 1567, 1583, 1598, 1613, 1628, 1643, 1658, 1673, 1688, 1703, 1718, 1733, 1748, 1763, 1778, 1793, 1808, 1824, 1809, 1794, 1779, 1764, 1749, 1734, 1719, 1704, 1689, 1674, 1659, 1644, 1629, 1614, 1599, 1615, 1630, 1645, 1660, 1675, 1690, 1705, 1720, 1735, 1750, 1765, 1780, 1795, 1810, 1825, 1840, 1856, 1841, 1826, 1811, 1796, 1781, 1766, 1751, 1736, 1721, 1706, 1691, 1676, 1661, 1646, 1631, 1647, 1662, 1677, 1692, 1707, 1722, 1737, 1752, 1767, 1782, 1797, 1812, 1827, 1842, 1857, 1872, 1888, 1873, 1858, 1843, 1828, 1813, 1798, 1783, 1768, 1753, 1738, 1723, 1708, 1693, 1678, 1663, 1679, 1694, 1709, 1724, 1739, 1754, 1769, 1784,
    1799, 1814, 1829, 1844, 1859, 1874, 1889, 1904, 1920, 1905, 1890, 1875, 1860, 1845, 1830, 1815, 1800, 1785, 1770, 1755, 1740, 1725, 1710, 1695, 1711, 1726, 1741, 1756, 1771, 1786, 1801, 1816, 1831, 1846, 1861, 1876, 1891, 1906, 1921, 1936, 1952, 1937, 1922, 1907, 1892, 1877, 1862, 1847, 1832, 1817, 1802, 1787, 1772, 1757, 1742, 1727, 1743, 1758, 1773, 1788, 1803, 1818, 1833, 1848, 1863, 1878, 1893, 1908, 1923, 1938, 1953, 1968, 1984, 1969, 1954, 1939, 1924, 1909, 1894, 1879, 1864, 1849, 1834, 1819, 1804, 1789, 1774, 1759, 1775, 1790, 1805, 1820, 1835, 1850, 1865, 1880, 1895, 1910, 1925, 1940, 1955, 1970, 1985, 2000, 2016, 2001, 1986, 1971, 1956, 1941, 1926, 1911, 1896, 1881, 1866, 1851, 1836, 1821, 1806, 1791, 1807, 1822, 1837, 1852, 1867, 1882, 1897, 1912,
    1927, 1942, 1957, 1972, 1987, 2002, 2017, 2032, 2033, 2018, 2003, 1988, 1973, 1958, 1943, 1928, 1913, 1898, 1883, 1868, 1853, 1838, 1823, 1839, 1854, 1869, 1884, 1899, 1914, 1929, 1944, 1959, 1974, 1989, 2004, 2019, 2034, 2035, 2020, 2005, 1990, 1975, 1960, 1945, 1930, 1915, 1900, 1885, 1870, 1855, 1871, 1886, 1901, 1916, 1931, 1946, 1961, 1976, 1991, 2006, 2021, 2036, 2037, 2022, 2007, 1992, 1977, 1962, 1947, 1932, 1917, 1902, 1887, 1903, 1918, 1933, 1948, 1963, 1978, 1993, 2008, 2023, 2038, 2039, 2024, 2009, 1994, 1979, 1964, 1949, 1934, 1919, 1935, 1950, 1965, 1980, 1995, 2010, 2025, 2040, 2041, 2026, 2011, 1996, 1981, 1966, 1951, 1967, 1982, 1997, 2012, 2027, 2042, 2043, 2028, 2013, 1998, 1983, 1999, 2014, 2029, 2044, 2045, 2030, 2015, 2031, 2046, 2047,

    // 128 x 32:
    0, 1, 32, 64, 33, 2, 3, 34, 65, 96, 128, 97, 66, 35, 4, 5, 36, 67, 98, 129, 160, 192, 161, 130, 99, 68, 37, 6, 7, 38, 69, 100, 131, 162, 193, 224, 256, 225, 194, 163, 132, 101, 70, 39, 8, 9, 40, 71, 102, 133, 164, 195, 226, 257, 288, 320, 289, 258, 227, 196, 165, 134, 103, 72, 41, 10, 11, 42, 73, 104, 135, 166, 197, 228, 259, 290, 321, 352, 384, 353, 322, 291, 260, 229, 198, 167, 136, 105, 74, 43, 12, 13, 44, 75, 106, 137, 168, 199, 230, 261, 292, 323, 354, 385, 416, 448, 417, 386, 355, 324, 293, 262, 231, 200, 169, 138, 107, 76, 45, 14, 15, 46, 77, 108, 139, 170, 201, 232,
    263, 294, 325, 356, 387, 418, 449, 480, 512, 481, 450, 419, 388, 357, 326, 295, 264, 233, 202, 171, 140, 109, 78, 47, 16, 17, 48, 79, 110, 141, 172, 203, 234, 265, 296, 327, 358, 389, 420, 451, 482, 513, 544, 576, 545, 514, 483, 452, 421, 390, 359, 328, 297, 266, 235, 204, 173, 142, 111, 80, 49, 18, 19, 50, 81, 112, 143, 174, 205, 236, 267, 298, 329, 360, 391, 422, 453, 484, 515, 546, 577, 608, 640, 609, 578, 547, 516, 485, 454, 423, 392, 361, 330, 299, 268, 237, 206, 175, 144, 113, 82, 51, 20, 21, 52, 83, 114, 145, 176, 207, 238, 269, 300, 331, 362, 393, 424, 455, 486, 517, 548, 579, 610, 641, 672, 704, 673, 642,
    611, 580, 549, 518, 487, 456, 425, 394, 363, 332, 301, 270, 239, 208, 177, 146, 115, 84, 53, 22, 23, 54, 85, 116, 147, 178, 209, 240, 271, 302, 333, 364, 395, 426, 457, 488, 519, 550, 581, 612, 643, 674, 705, 736, 768, 737, 706, 675, 644, 613, 582, 551, 520, 489, 458, 427, 396, 365, 334, 303, 272, 241, 210, 179, 148, 117, 86, 55, 24, 25, 56, 87, 118, 149, 180, 211, 242, 273, 304, 335, 366, 397, 428, 459, 490, 521, 552, 583, 614, 645, 676, 707, 738, 769, 800, 832, 801, 770, 739, 708, 677, 646, 615, 584, 553, 522, 491, 460, 429, 398, 367, 336, 305, 274, 243, 212, 181, 150, 119, 88, 57, 26, 27, 58, 89, 120, 151, 182,
    213, 244, 275, 306, 337, 368, 399, 430, 461, 492, 523, 554, 585, 616, 647, 678, 709, 740, 771, 802, 833, 864, 896, 865, 834, 803, 772, 741, 710, 679, 648, 617, 586, 555, 524, 493, 462, 431, 400, 369, 338, 307, 276, 245, 214, 183, 152, 121, 90, 59, 28, 29, 60, 91, 122, 153, 184, 215, 246, 277, 308, 339, 370, 401, 432, 463, 494, 525, 556, 587, 618, 649, 680, 711, 742, 773, 804, 835, 866, 897, 928, 960, 929, 898, 867, 836, 805, 774, 743, 712, 681, 650, 619, 588, 557, 526, 495, 464, 433, 402, 371, 340, 309, 278, 247, 216, 185, 154, 123, 92, 61, 30, 31, 62, 93, 124, 155, 186, 217, 248, 279, 310, 341, 372, 403, 434, 465, 496,
    527, 558, 589, 620, 651, 682, 713, 744, 775, 806, 837, 868, 899, 930, 961, 992, 1024, 993, 962, 931, 900, 869, 838, 807, 776, 745, 714, 683, 652, 621, 590, 559, 528, 497, 466, 435, 404, 373, 342, 311, 280, 249, 218, 187, 156, 125, 94, 63, 95, 126, 157, 188, 219, 250, 281, 312, 343, 374, 405, 436, 467, 498, 529, 560, 591, 622, 653, 684, 715, 746, 777, 808, 839, 870, 901, 932, 963, 994, 1025, 1056, 1088, 1057, 1026, 995, 964, 933, 902, 871, 840, 809, 778, 747, 716, 685, 654, 623, 592, 561, 530, 499, 468, 437, 406, 375, 344, 313, 282, 251, 220, 189, 158, 127, 159, 190, 221, 252, 283, 314, 345, 376, 407, 438, 469, 500, 531, 562, 593, 624,
    655, 686, 717, 748, 779, 810, 841, 872, 903, 934, 965, 996, 1027, 1058, 1089, 1120, 1152, 1121, 1090, 1059, 1028, 997, 966, 935, 904, 873, 842, 811, 780, 749, 718, 687, 656, 625, 594, 563, 532, 501, 470, 439, 408, 377, 346, 315, 284, 253, 222, 191, 223, 254, 285, 316, 347, 378, 409, 440, 471, 502, 533, 564, 595, 626, 657, 688, 719, 750, 781, 812, 843, 874, 905, 936, 967, 998, 1029, 1060, 1091, 1122, 1153, 1184, 1216, 1185, 1154, 1123, 1092, 1061, 1030, 999, 968, 937, 906, 875, 844, 813, 782, 751, 720, 689, 658, 627, 596, 565, 534, 503, 472, 441, 410, 379, 348, 317, 286, 255, 287, 318, 349, 380, 411, 442, 473, 504, 535, 566, 597, 628, 659, 690, 721, 752,
    783, 814, 845, 876, 907, 938, 969, 1000, 1031, 1062, 1093, 1124, 1155, 1186, 1217, 1248, 1280, 1249, 1218, 1187, 1156, 1125, 1094, 1063, 1032, 1001, 970, 939, 908, 877, 846, 815, 784, 753, 722, 691, 660, 629, 598, 567, 536, 505, 474, 443, 412, 381, 350, 319, 351, 382, 413, 444, 475, 506, 537, 568, 599, 630, 661, 692, 723, 754, 785, 816, 847, 878, 909, 940, 971, 1002, 1033, 1064, 1095, 1126, 1157, 1188, 1219, 1250, 1281, 1312, 1344, 1313, 1282, 1251, 1220, 1189, 1158, 1127, 1096, 1065, 1034, 1003, 972, 941, 910, 879, 848, 817, 786, 755, 724, 693, 662, 631, 600, 569, 538, 507, 476, 445, 414, 383, 415, 446, 477, 508, 539, 570, 601, 632, 663, 694, 725, 756, 787, 818, 849, 880,
    911, 942, 973, 1004, 1035, 1066, 1097, 1128, 1159, 1190, 1221, 1252, 1283, 1314, 1345, 1376, 1408, 1377, 1346, 1315, 1284, 1253, 1222, 1191, 1160, 1129, 1098, 1067, 1036, 1005, 974, 943, 912, 881, 850, 819, 788, 757, 726, 695, 664, 633, 602, 571, 540, 509, 478, 447, 479, 510, 541, 572, 603, 634, 665, 696, 727, 758, 789, 820, 851, 882, 913, 944, 975, 1006, 1037, 1068, 1099, 1130, 1161, 1192, 1223, 1254, 1285, 1316, 1347, 1378, 1409, 1440, 1472, 1441, 1410, 1379, 1348, 1317, 1286, 1255, 1224, 1193, 1162, 1131, 1100, 1069, 1038, 1007, 976, 945, 914, 883, 852, 821, 790, 759, 728, 697, 666, 635, 604, 573, 542, 511, 543, 574, 605, 636, 667, 698, 729, 760, 791, 822, 853, 884, 915, 946, 977, 1008,
    1039, 1070, 1101, 1132, 1163, 1194, 1225, 1256, 1287, 1318, 1349, 1380, 1411, 1442, 1473, 1504, 1536, 1505, 1474, 1443, 1412, 1381, 1350, 1319, 1288, 1257, 1226, 1195, 1164, 1133, 1102, 1071, 1040, 1009, 978, 947, 916, 885, 854, 823, 792, 761, 730, 699, 668, 637, 606, 575, 607, 638, 669, 700, 731, 762, 793, 824, 855, 886, 917, 948, 979, 1010, 1041, 1072, 1103, 1134, 1165, 1196, 1227, 1258, 1289, 1320, 1351, 1382, 1413, 1444, 1475, 1506, 1537, 1568, 1600, 1569, 1538, 1507, 1476, 1445, 1414, 1383, 1352, 1321, 1290, 1259, 1228, 1197, 1166, 1135, 1104, 1073, 1042, 1011, 980, 949, 918, 887, 856, 825, 794, 763, 732, 701, 670, 639, 671, 702, 733, 764, 795, 826, 857, 888, 919, 950, 981, 1012, 1043, 1074, 1105, 1136,
    1167, 1198, 1229, 1260, 1291, 1322, 1353, 1384, 1415, 1446, 1477, 1508, 1539, 1570, 1601, 1632, 1664, 1633, 1602, 1571, 1540, 1509, 1478, 1447, 1416, 1385, 1354, 1323, 1292, 1261, 1230, 1199, 1168, 1137, 1106, 1075, 1044, 1013, 982, 951, 920, 889, 858, 827, 796, 765, 734, 703, 735, 766, 797, 828, 859, 890, 921, 952, 983, 1014, 1045, 1076, 1107, 1138, 1169, 1200, 1231, 1262, 1293, 1324, 1355, 1386, 1417, 1448, 1479, 1510, 1541, 1572, 1603, 1634, 1665, 1696, 1728, 1697, 1666, 1635, 1604, 1573, 1542, 1511, 1480, 1449, 1418, 1387, 1356, 1325, 1294, 1263, 1232, 1201, 1170, 1139, 1108, 1077, 1046, 1015, 984, 953, 922, 891, 860, 829, 798, 767, 799, 830, 861, 892, 923, 954, 985, 1016, 1047, 1078, 1109, 1140, 1171, 1202, 1233, 1264,
    1295, 1326, 1357, 1388, 1419, 1450, 1481, 1512, 1543, 1574, 1605, 1636, 1667, 1698, 1729, 1760, 1792, 1761, 1730, 1699, 1668, 1637, 1606, 1575, 1544, 1513, 1482, 1451, 1420, 1389, 1358, 1327, 1296, 1265, 1234, 1203, 1172, 1141, 1110, 1079, 1048, 1017, 986, 955, 924, 893, 862, 831, 863, 894, 925, 956, 987, 1018, 1049, 1080, 1111, 1142, 1173, 1204, 1235, 1266, 1297, 1328, 1359, 1390, 1421, 1452, 1483, 1514, 1545, 1576, 1607, 1638, 1669, 1700, 1731, 1762, 1793, 1824, 1856, 1825, 1794, 1763, 1732, 1701, 1670, 1639, 1608, 1577, 1546, 1515, 1484, 1453, 1422, 1391, 1360, 1329, 1298, 1267, 1236, 1205, 1174, 1143, 1112, 1081, 1050, 1019, 988, 957, 926, 895, 927, 958, 989, 1020, 1051, 1082, 1113, 1144, 1175, 1206, 1237, 1268, 1299, 1330, 1361, 1392,
    1423, 1454, 1485, 1516, 1547, 1578, 1609, 1640, 1671, 1702, 1733, 1764, 1795, 1826, 1857, 1888, 1920, 1889, 1858, 1827, 1796, 1765, 1734, 1703, 1672, 1641, 1610, 1579, 1548, 1517, 1486, 1455, 1424, 1393, 1362, 1331, 1300, 1269, 1238, 1207, 1176, 1145, 1114, 1083, 1052, 1021, 990, 959, 991, 1022, 1053, 1084, 1115, 1146, 1177, 1208, 1239, 1270, 1301, 1332, 1363, 1394, 1425, 1456, 1487, 1518, 1549, 1580, 1611, 1642, 1673, 1704, 1735, 1766, 1797, 1828, 1859, 1890, 1921, 1952, 1984, 1953, 1922, 1891, 1860, 1829, 1798, 1767, 1736, 1705, 1674, 1643, 1612, 1581, 1550, 1519, 1488, 1457, 1426, 1395, 1364, 1333, 1302, 1271, 1240, 1209, 1178, 1147, 1116, 1085, 1054, 1023, 1055, 1086, 1117, 1148, 1179, 1210, 1241, 1272, 1303, 1334, 1365, 1396, 1427, 1458, 1489, 1520,
    1551, 1582, 1613, 1644, 1675, 1706, 1737, 1768, 1799, 1830, 1861, 1892, 1923, 1954, 1985, 2016, 2048, 2017, 1986, 1955, 1924, 1893, 1862, 1831, 1800, 1769, 1738, 1707, 1676, 1645, 1614, 1583, 1552, 1521, 1490, 1459, 1428, 1397, 1366, 1335, 1304, 1273, 1242, 1211, 1180, 1149, 1118, 1087, 1119, 1150, 1181, 1212, 1243, 1274, 1305, 1336, 1367, 1398, 1429, 1460, 1491, 1522, 1553, 1584, 1615, 1646, 1677, 1708, 1739, 1770, 1801, 1832, 1863, 1894, 1925, 1956, 1987, 2018, 2049, 2080, 2112, 2081, 2050, 2019, 1988, 1957, 1926, 1895, 1864, 1833, 1802, 1771, 1740, 1709, 1678, 1647, 1616, 1585, 1554, 1523, 1492, 1461, 1430, 1399, 1368, 1337, 1306, 1275, 1244, 1213, 1182, 1151, 1183, 1214, 1245, 1276, 1307, 1338, 1369, 1400, 1431, 1462, 1493, 1524, 1555, 1586, 1617, 1648,
    1679, 1710, 1741, 1772, 1803, 1834, 1865, 1896, 1927, 1958, 1989, 2020, 2051, 2082, 2113, 2144, 2176, 2145, 2114, 2083, 2052, 2021, 1990, 1959, 1928, 1897, 1866, 1835, 1804, 1773, 1742, 1711, 1680, 1649, 1618, 1587, 1556, 1525, 1494, 1463, 1432, 1401, 1370, 1339, 1308, 1277, 1246, 1215, 1247, 1278, 1309, 1340, 1371, 1402, 1433, 1464, 1495, 1526, 1557, 1588, 1619, 1650, 1681, 1712, 1743, 1774, 1805, 1836, 1867, 1898, 1929, 1960, 1991, 2022, 2053, 2084, 2115, 2146, 2177, 2208, 2240, 2209, 2178, 2147, 2116, 2085, 2054, 2023, 1992, 1961, 1930, 1899, 1868, 1837, 1806, 1775, 1744, 1713, 1682, 1651, 1620, 1589, 1558, 1527, 1496, 1465, 1434, 1403, 1372, 1341, 1310, 1279, 1311, 1342, 1373, 1404, 1435, 1466, 1497, 1528, 1559, 1590, 1621, 1652, 1683, 1714, 1745, 1776,
    1807, 1838, 1869, 1900, 1931, 1962, 1993, 2024, 2055, 2086, 2117, 2148, 2179, 2210, 2241, 2272, 2304, 2273, 2242, 2211, 2180, 2149, 2118, 2087, 2056, 2025, 1994, 1963, 1932, 1901, 1870, 1839, 1808, 1777, 1746, 1715, 1684, 1653, 1622, 1591, 1560, 1529, 1498, 1467, 1436, 1405, 1374, 1343, 1375, 1406, 1437, 1468, 1499, 1530, 1561, 1592, 1623, 1654, 1685, 1716, 1747, 1778, 1809, 1840, 1871, 1902, 1933, 1964, 1995, 2026, 2057, 2088, 2119, 2150, 2181, 2212, 2243, 2274, 2305, 2336, 2368, 2337, 2306, 2275, 2244, 2213, 2182, 2151, 2120, 2089, 2058, 2027, 1996, 1965, 1934, 1903, 1872, 1841, 1810, 1779, 1748, 1717, 1686, 1655, 1624, 1593, 1562, 1531, 1500, 1469, 1438, 1407, 1439, 1470, 1501, 1532, 1563, 1594, 1625, 1656, 1687, 1718, 1749, 1780, 1811, 1842, 1873, 1904,
    1935, 1966, 1997, 2028, 2059, 2090, 2121, 2152, 2183, 2214, 2245, 2276, 2307, 2338, 2369, 2400, 2432, 2401, 2370, 2339, 2308, 2277, 2246, 2215, 2184, 2153, 2122, 2091, 2060, 2029, 1998, 1967, 1936, 1905, 1874, 1843, 1812, 1781, 1750, 1719, 1688, 1657, 1626, 1595, 1564, 1533, 1502, 1471, 1503, 1534, 1565, 1596, 1627, 1658, 1689, 1720, 1751, 1782, 1813, 1844, 1875, 1906, 1937, 1968, 1999, 2030, 2061, 2092, 2123, 2154, 2185, 2216, 2247, 2278, 2309, 2340, 2371, 2402, 2433, 2464, 2496, 2465, 2434, 2403, 2372, 2341, 2310, 2279, 2248, 2217, 2186, 2155, 2124, 2093, 2062, 2031, 2000, 1969, 1938, 1907, 1876, 1845, 1814, 1783, 1752, 1721, 1690, 1659, 1628, 1597, 1566, 1535, 1567, 1598, 1629, 1660, 1691, 1722, 1753, 1784, 1815, 1846, 1877, 1908, 1939, 1970, 2001, 2032,
    2063, 2094, 2125, 2156, 2187, 2218, 2249, 2280, 2311, 2342, 2373, 2404, 2435, 2466, 2497, 2528, 2560, 2529, 2498, 2467, 2436, 2405, 2374, 2343, 2312, 2281, 2250, 2219, 2188, 2157, 2126, 2095, 2064, 2033, 2002, 1971, 1940, 1909, 1878, 1847, 1816, 1785, 1754, 1723, 1692, 1661, 1630, 1599, 1631, 1662, 1693, 1724, 1755, 1786, 1817, 1848, 1879, 1910, 1941, 1972, 2003, 2034, 2065, 2096, 2127, 2158, 2189, 2220, 2251, 2282, 2313, 2344, 2375, 2406, 2437, 2468, 2499, 2530, 2561, 2592, 2624, 2593, 2562, 2531, 2500, 2469, 2438, 2407, 2376, 2345, 2314, 2283, 2252, 2221, 2190, 2159, 2128, 2097, 2066, 2035, 2004, 1973, 1942, 1911, 1880, 1849, 1818, 1787, 1756, 1725, 1694, 1663, 1695, 1726, 1757, 1788, 1819, 1850, 1881, 1912, 1943, 1974, 2005, 2036, 2067, 2098, 2129, 2160,
    2191, 2222, 2253, 2284, 2315, 2346, 2377, 2408, 2439, 2470, 2501, 2532, 2563, 2594, 2625, 2656, 2688, 2657, 2626, 2595, 2564, 2533, 2502, 2471, 2440, 2409, 2378, 2347, 2316, 2285, 2254, 2223, 2192, 2161, 2130, 2099, 2068, 2037, 2006, 1975, 1944, 1913, 1882, 1851, 1820, 1789, 1758, 1727, 1759, 1790, 1821, 1852, 1883, 1914, 1945, 1976, 2007, 2038, 2069, 2100, 2131, 2162, 2193, 2224, 2255, 2286, 2317, 2348, 2379, 2410, 2441, 2472, 2503, 2534, 2565, 2596, 2627, 2658, 2689, 2720, 2752, 2721, 2690, 2659, 2628, 2597, 2566, 2535, 2504, 2473, 2442, 2411, 2380, 2349, 2318, 2287, 2256, 2225, 2194, 2163, 2132, 2101, 2070, 2039, 2008, 1977, 1946, 1915, 1884, 1853, 1822, 1791, 1823, 1854, 1885, 1916, 1947, 1978, 2009, 2040, 2071, 2102, 2133, 2164, 2195, 2226, 2257, 2288,
    2319, 2350, 2381, 2412, 2443, 2474, 2505, 2536, 2567, 2598, 2629, 2660, 2691, 2722, 2753, 2784, 2816, 2785, 2754, 2723, 2692, 2661, 2630, 2599, 2568, 2537, 2506, 2475, 2444, 2413, 2382, 2351, 2320, 2289, 2258, 2227, 2196, 2165, 2134, 2103, 2072, 2041, 2010, 1979, 1948, 1917, 1886, 1855, 1887, 1918, 1949, 1980, 2011, 2042, 2073, 2104, 2135, 2166, 2197, 2228, 2259, 2290, 2321, 2352, 2383, 2414, 2445, 2476, 2507, 2538, 2569, 2600, 2631, 2662, 2693, 2724, 2755, 2786, 2817, 2848, 2880, 2849, 2818, 2787, 2756, 2725, 2694, 2663, 2632, 2601, 2570, 2539, 2508, 2477, 2446, 2415, 2384, 2353, 2322, 2291, 2260, 2229, 2198, 2167, 2136, 2105, 2074, 2043, 2012, 1981, 1950, 1919, 1951, 1982, 2013, 2044, 2075, 2106, 2137, 2168, 2199, 2230, 2261, 2292, 2323, 2354, 2385, 2416,
    2447, 2478, 2509, 2540, 2571, 2602, 2633, 2664, 2695, 2726, 2757, 2788, 2819, 2850, 2881, 2912, 2944, 2913, 2882, 2851, 2820, 2789, 2758, 2727, 2696, 2665, 2634, 2603, 2572, 2541, 2510, 2479, 2448, 2417, 2386, 2355, 2324, 2293, 2262, 2231, 2200, 2169, 2138, 2107, 2076, 2045, 2014, 1983, 2015, 2046, 2077, 2108, 2139, 2170, 2201, 2232, 2263, 2294, 2325, 2356, 2387, 2418, 2449, 2480, 2511, 2542, 2573, 2604, 2635, 2666, 2697, 2728, 2759, 2790, 2821, 2852, 2883, 2914, 2945, 2976, 3008, 2977, 2946, 2915, 2884, 2853, 2822, 2791, 2760, 2729, 2698, 2667, 2636, 2605, 2574, 2543, 2512, 2481, 2450, 2419, 2388, 2357, 2326, 2295, 2264, 2233, 2202, 2171, 2140, 2109, 2078, 2047, 2079, 2110, 2141, 2172, 2203, 2234, 2265, 2296, 2327, 2358, 2389, 2420, 2451, 2482, 2513, 2544,
    2575, 2606, 2637, 2668, 2699, 2730, 2761, 2792, 2823, 2854, 2885, 2916, 2947, 2978, 3009, 3040, 3072, 3041, 3010, 2979, 2948, 2917, 2886, 2855, 2824, 2793, 2762, 2731, 2700, 2669, 2638, 2607, 2576, 2545, 2514, 2483, 2452, 2421, 2390, 2359, 2328, 2297, 2266, 2235, 2204, 2173, 2142, 2111, 2143, 2174, 2205, 2236, 2267, 2298, 2329, 2360, 2391, 2422, 2453, 2484, 2515, 2546, 2577, 2608, 2639, 2670, 2701, 2732, 2763, 2794, 2825, 2856, 2887, 2918, 2949, 2980, 3011, 3042, 3073, 3104, 3136, 3105, 3074, 3043, 3012, 2981, 2950, 2919, 2888, 2857, 2826, 2795, 2764, 2733, 2702, 2671, 2640, 2609, 2578, 2547, 2516, 2485, 2454, 2423, 2392, 2361, 2330, 2299, 2268, 2237, 2206, 2175, 2207, 2238, 2269, 2300, 2331, 2362, 2393, 2424, 2455, 2486, 2517, 2548, 2579, 2610, 2641, 2672,
    2703, 2734, 2765, 2796, 2827, 2858, 2889, 2920, 2951, 2982, 3013, 3044, 3075, 3106, 3137, 3168, 3200, 3169, 3138, 3107, 3076, 3045, 3014, 2983, 2952, 2921, 2890, 2859, 2828, 2797, 2766, 2735, 2704, 2673, 2642, 2611, 2580, 2549, 2518, 2487, 2456, 2425, 2394, 2363, 2332, 2301, 2270, 2239, 2271, 2302, 2333, 2364, 2395, 2426, 2457, 2488, 2519, 2550, 2581, 2612, 2643, 2674, 2705, 2736, 2767, 2798, 2829, 2860, 2891, 2922, 2953, 2984, 3015, 3046, 3077, 3108, 3139, 3170, 3201, 3232, 3264, 3233, 3202, 3171, 3140, 3109, 3078, 3047, 3016, 2985, 2954, 2923, 2892, 2861, 2830, 2799, 2768, 2737, 2706, 2675, 2644, 2613, 2582, 2551, 2520, 2489, 2458, 2427, 2396, 2365, 2334, 2303, 2335, 2366, 2397, 2428, 2459, 2490, 2521, 2552, 2583, 2614, 2645, 2676, 2707, 2738, 2769, 2800,
    2831, 2862, 2893, 2924, 2955, 2986, 3017, 3048, 3079, 3110, 3141, 3172, 3203, 3234, 3265, 3296, 3328, 3297, 3266, 3235, 3204, 3173, 3142, 3111, 3080, 3049, 3018, 2987, 2956, 2925, 2894, 2863, 2832, 2801, 2770, 2739, 2708, 2677, 2646, 2615, 2584, 2553, 2522, 2491, 2460, 2429, 2398, 2367, 2399, 2430, 2461, 2492, 2523, 2554, 2585, 2616, 2647, 2678, 2709, 2740, 2771, 2802, 2833, 2864, 2895, 2926, 2957, 2988, 3019, 3050, 3081, 3112, 3143, 3174, 3205, 3236, 3267, 3298, 3329, 3360, 3392, 3361, 3330, 3299, 3268, 3237, 3206, 3175, 3144, 3113, 3082, 3051, 3020, 2989, 2958, 2927, 2896, 2865, 2834, 2803, 2772, 2741, 2710, 2679, 2648, 2617, 2586, 2555, 2524, 2493, 2462, 2431, 2463, 2494, 2525, 2556, 2587, 2618, 2649, 2680, 2711, 2742, 2773, 2804, 2835, 2866, 2897, 2928,
    2959, 2990, 3021, 3052, 3083, 3114, 3145, 3176, 3207, 3238, 3269, 3300, 3331, 3362, 3393, 3424, 3456, 3425, 3394, 3363, 3332, 3301, 3270, 3239, 3208, 3177, 3146, 3115, 3084, 3053, 3022, 2991, 2960, 2929, 2898, 2867, 2836, 2805, 2774, 2743, 2712, 2681, 2650, 2619, 2588, 2557, 2526, 2495, 2527, 2558, 2589, 2620, 2651, 2682, 2713, 2744, 2775, 2806, 2837, 2868, 2899, 2930, 2961, 2992, 3023, 3054, 3085, 3116, 3147, 3178, 3209, 3240, 3271, 3302, 3333, 3364, 3395, 3426, 3457, 3488, 3520, 3489, 3458, 3427, 3396, 3365, 3334, 3303, 3272, 3241, 3210, 3179, 3148, 3117, 3086, 3055, 3024, 2993, 2962, 2931, 2900, 2869, 2838, 2807, 2776, 2745, 2714, 2683, 2652, 2621, 2590, 2559, 2591, 2622, 2653, 2684, 2715, 2746, 2777, 2808, 2839, 2870, 2901, 2932, 2963, 2994, 3025, 3056,
    3087, 3118, 3149, 3180, 3211, 3242, 3273, 3304, 3335, 3366, 3397, 3428, 3459, 3490, 3521, 3552, 3584, 3553, 3522, 3491, 3460, 3429, 3398, 3367, 3336, 3305, 3274, 3243, 3212, 3181, 3150, 3119, 3088, 3057, 3026, 2995, 2964, 2933, 2902, 2871, 2840, 2809, 2778, 2747, 2716, 2685, 2654, 2623, 2655, 2686, 2717, 2748, 2779, 2810, 2841, 2872, 2903, 2934, 2965, 2996, 3027, 3058, 3089, 3120, 3151, 3182, 3213, 3244, 3275, 3306, 3337, 3368, 3399, 3430, 3461, 3492, 3523, 3554, 3585, 3616, 3648, 3617, 3586, 3555, 3524, 3493, 3462, 3431, 3400, 3369, 3338, 3307, 3276, 3245, 3214, 3183, 3152, 3121, 3090, 3059, 3028, 2997, 2966, 2935, 2904, 2873, 2842, 2811, 2780, 2749, 2718, 2687, 2719, 2750, 2781, 2812, 2843, 2874, 2905, 2936, 2967, 2998, 3029, 3060, 3091, 3122, 3153, 3184,
    3215, 3246, 3277, 3308, 3339, 3370, 3401, 3432, 3463, 3494, 3525, 3556, 3587, 3618, 3649, 3680, 3712, 3681, 3650, 3619, 3588, 3557, 3526, 3495, 3464, 3433, 3402, 3371, 3340, 3309, 3278, 3247, 3216, 3185, 3154, 3123, 3092, 3061, 3030, 2999, 2968, 2937, 2906, 2875, 2844, 2813, 2782, 2751, 2783, 2814, 2845, 2876, 2907, 2938, 2969, 3000, 3031, 3062, 3093, 3124, 3155, 3186, 3217, 3248, 3279, 3310, 3341, 3372, 3403, 3434, 3465, 3496, 3527, 3558, 3589, 3620, 3651, 3682, 3713, 3744, 3776, 3745, 3714, 3683, 3652, 3621, 3590, 3559, 3528, 3497, 3466, 3435, 3404, 3373, 3342, 3311, 3280, 3249, 3218, 3187, 3156, 3125, 3094, 3063, 3032, 3001, 2970, 2939, 2908, 2877, 2846, 2815, 2847, 2878, 2909, 2940, 2971, 3002, 3033, 3064, 3095, 3126, 3157, 3188, 3219, 3250, 3281, 3312,
    3343, 3374, 3405, 3436, 3467, 3498, 3529, 3560, 3591, 3622, 3653, 3684, 3715, 3746, 3777, 3808, 3840, 3809, 3778, 3747, 3716, 3685, 3654, 3623, 3592, 3561, 3530, 3499, 3468, 3437, 3406, 3375, 3344, 3313, 3282, 3251, 3220, 3189, 3158, 3127, 3096, 3065, 3034, 3003, 2972, 2941, 2910, 2879, 2911, 2942, 2973, 3004, 3035, 3066, 3097, 3128, 3159, 3190, 3221, 3252, 3283, 3314, 3345, 3376, 3407, 3438, 3469, 3500, 3531, 3562, 3593, 3624, 3655, 3686, 3717, 3748, 3779, 3810, 3841, 3872, 3904, 3873, 3842, 3811, 3780, 3749, 3718, 3687, 3656, 3625, 3594, 3563, 3532, 3501, 3470, 3439, 3408, 3377, 3346, 3315, 3284, 3253, 3222, 3191, 3160, 3129, 3098, 3067, 3036, 3005, 2974, 2943, 2975, 3006, 3037, 3068, 3099, 3130, 3161, 3192, 3223, 3254, 3285, 3316, 3347, 3378, 3409, 3440,
    3471, 3502, 3533, 3564, 3595, 3626, 3657, 3688, 3719, 3750, 3781, 3812, 3843, 3874, 3905, 3936, 3968, 3937, 3906, 3875, 3844, 3813, 3782, 3751, 3720, 3689, 3658, 3627, 3596, 3565, 3534, 3503, 3472, 3441, 3410, 3379, 3348, 3317, 3286, 3255, 3224, 3193, 3162, 3131, 3100, 3069, 3038, 3007, 3039, 3070, 3101, 3132, 3163, 3194, 3225, 3256, 3287, 3318, 3349, 3380, 3411, 3442, 3473, 3504, 3535, 3566, 3597, 3628, 3659, 3690, 3721, 3752, 3783, 3814, 3845, 3876, 3907, 3938, 3969, 4000, 4032, 4001, 3970, 3939, 3908, 3877, 3846, 3815, 3784, 3753, 3722, 3691, 3660, 3629, 3598, 3567, 3536, 3505, 3474, 3443, 3412, 3381, 3350, 3319, 3288, 3257, 3226, 3195, 3164, 3133, 3102, 3071, 3103, 3134, 3165, 3196, 3227, 3258, 3289, 3320, 3351, 3382, 3413, 3444, 3475, 3506, 3537, 3568,
    3599, 3630, 3661, 3692, 3723, 3754, 3785, 3816, 3847, 3878, 3909, 3940, 3971, 4002, 4033, 4064, 4065, 4034, 4003, 3972, 3941, 3910, 3879, 3848, 3817, 3786, 3755, 3724, 3693, 3662, 3631, 3600, 3569, 3538, 3507, 3476, 3445, 3414, 3383, 3352, 3321, 3290, 3259, 3228, 3197, 3166, 3135, 3167, 3198, 3229, 3260, 3291, 3322, 3353, 3384, 3415, 3446, 3477, 3508, 3539, 3570, 3601, 3632, 3663, 3694, 3725, 3756, 3787, 3818, 3849, 3880, 3911, 3942, 3973, 4004, 4035, 4066, 4067, 4036, 4005, 3974, 3943, 3912, 3881, 3850, 3819, 3788, 3757, 3726, 3695, 3664, 3633, 3602, 3571, 3540, 3509, 3478, 3447, 3416, 3385, 3354, 3323, 3292, 3261, 3230, 3199, 3231, 3262, 3293, 3324, 3355, 3386, 3417, 3448, 3479, 3510, 3541, 3572, 3603, 3634, 3665, 3696, 3727, 3758, 3789, 3820, 3851, 3882,
    3913, 3944, 3975, 4006, 4037, 4068, 4069, 4038, 4007, 3976, 3945, 3914, 3883, 3852, 3821, 3790, 3759, 3728, 3697, 3666, 3635, 3604, 3573, 3542, 3511, 3480, 3449, 3418, 3387, 3356, 3325, 3294, 3263, 3295, 3326, 3357, 3388, 3419, 3450, 3481, 3512, 3543, 3574, 3605, 3636, 3667, 3698, 3729, 3760, 3791, 3822, 3853, 3884, 3915, 3946, 3977, 4008, 4039, 4070, 4071, 4040, 4009, 3978, 3947, 3916, 3885, 3854, 3823, 3792, 3761, 3730, 3699, 3668, 3637, 3606, 3575, 3544, 3513, 3482, 3451, 3420, 3389, 3358, 3327, 3359, 3390, 3421, 3452, 3483, 3514, 3545, 3576, 3607, 3638, 3669, 3700, 3731, 3762, 3793, 3824, 3855, 3886, 3917, 3948, 3979, 4010, 4041, 4072, 4073, 4042, 4011, 3980, 3949, 3918, 3887, 3856, 3825, 3794, 3763, 3732, 3701, 3670, 3639, 3608, 3577, 3546, 3515, 3484,
    3453, 3422, 3391, 3423, 3454, 3485, 3516, 3547, 3578, 3609, 3640, 3671, 3702, 3733, 3764, 3795, 3826, 3857, 3888, 3919, 3950, 3981, 4012, 4043, 4074, 4075, 4044, 4013, 3982, 3951, 3920, 3889, 3858, 3827, 3796, 3765, 3734, 3703, 3672, 3641, 3610, 3579, 3548, 3517, 3486, 3455, 3487, 3518, 3549, 3580, 3611, 3642, 3673, 3704, 3735, 3766, 3797, 3828, 3859, 3890, 3921, 3952, 3983, 4014, 4045, 4076, 4077, 4046, 4015, 3984, 3953, 3922, 3891, 3860, 3829, 3798, 3767, 3736, 3705, 3674, 3643, 3612, 3581, 3550, 3519, 3551, 3582, 3613, 3644, 3675, 3706, 3737, 3768, 3799, 3830, 3861, 3892, 3923, 3954, 3985, 4016, 4047, 4078, 4079, 4048, 4017, 3986, 3955, 3924, 3893, 3862, 3831, 3800, 3769, 3738, 3707, 3676, 3645, 3614, 3583, 3615, 3646, 3677, 3708, 3739, 3770, 3801, 3832,
    3863, 3894, 3925, 3956, 3987, 4018, 4049, 4080, 4081, 4050, 4019, 3988, 3957, 3926, 3895, 3864, 3833, 3802, 3771, 3740, 3709, 3678, 3647, 3679, 3710, 3741, 3772, 3803, 3834, 3865, 3896, 3927, 3958, 3989, 4020, 4051, 4082, 4083, 4052, 4021, 3990, 3959, 3928, 3897, 3866, 3835, 3804, 3773, 3742, 3711, 3743, 3774, 3805, 3836, 3867, 3898, 3929, 3960, 3991, 4022, 4053, 4084, 4085, 4054, 4023, 3992, 3961, 3930, 3899, 3868, 3837, 3806, 3775, 3807, 3838, 3869, 3900, 3931, 3962, 3993, 4024, 4055, 4086, 4087, 4056, 4025, 3994, 3963, 3932, 3901, 3870, 3839, 3871, 3902, 3933, 3964, 3995, 4026, 4057, 4088, 4089, 4058, 4027, 3996, 3965, 3934, 3903, 3935, 3966, 3997, 4028, 4059, 4090, 4091, 4060, 4029, 3998, 3967, 3999, 4030, 4061, 4092, 4093, 4062, 4031, 4063, 4094, 4095,

    // 128 x 64:
    0, 1, 64, 128, 65, 2, 3, 66, 129, 192, 256, 193, 130, 67, 4, 5, 68, 131, 194, 257, 320, 384, 321, 258, 195, 132, 69, 6, 7, 70, 133, 196, 259, 322, 385, 448, 512, 449, 386, 323, 260, 197, 134, 71, 8, 9, 72, 135, 198, 261, 324, 387, 450, 513, 576, 640, 577, 514, 451, 388, 325, 262, 199, 136, 73, 10, 11, 74, 137, 200, 263, 326, 389, 452, 515, 578, 641, 704, 768, 705, 642, 579, 516, 453, 390, 327, 264, 201, 138, 75, 12, 13, 76, 139, 202, 265, 328, 391, 454, 517, 580, 643, 706, 769, 832, 896, 833, 770, 707, 644, 581, 518, 455, 392, 329, 266, 203, 140, 77, 14, 15, 78, 141, 204, 267, 330, 393, 456,
    519, 582, 645, 708, 771, 834, 897, 960, 1024, 961, 898, 835, 772, 709, 646, 583, 520, 457, 394, 331, 268, 205, 142, 79, 16, 17, 80, 143, 206, 269, 332, 395, 458, 521, 584, 647, 710, 773, 836, 899, 962, 1025, 1088, 1152, 1089, 1026, 963, 900, 837, 774, 711, 648, 585, 522, 459, 396, 333, 270, 207, 144, 81, 18, 19, 82, 145, 208, 271, 334, 397, 460, 523, 586, 649, 712, 775, 838, 901, 964, 1027, 1090, 1153, 1216, 1280, 1217, 1154, 1091, 1028, 965, 902, 839, 776, 713, 650, 587, 524, 461, 398, 335, 272, 209, 146, 83, 20, 21, 84, 147, 210, 273, 336, 399, 462, 525, 588, 651, 714, 777, 840, 903, 966, 1029, 1092, 1155, 1218, 1281, 1344, 1408, 1345, 1282,
    1219, 1156, 1093, 1030, 967, 904, 841, 778, 715, 652, 589, 526, 463, 400, 337, 274, 211, 148, 85, 22, 23, 86, 149, 212, 275, 338, 401, 464, 527, 590, 653, 716, 779, 842, 905, 968, 1031, 1094, 1157, 1220, 1283, 1346, 1409, 1472, 1536, 1473, 1410, 1347, 1284, 1221, 1158, 1095, 1032, 969, 906, 843, 780, 717, 654, 591, 528, 465, 402, 339, 276, 213, 150, 87, 24, 25, 88, 151, 214, 277, 340, 403, 466, 529, 592, 655, 718, 781, 844, 907, 970, 1033, 1096, 1159, 1222, 1285, 1348, 1411, 1474, 1537, 1600, 1664, 1601, 1538, 1475, 1412, 1349, 1286, 1223, 1160, 1097, 1034, 971, 908, 845, 782, 719, 656, 593, 530, 467, 404, 341, 278, 215, 152, 89, 26, 27, 90, 153, 216, 279, 342,
    405, 468, 531, 594, 657, 720, 783, 846, 909, 972, 1035, 1098, 1161, 1224, 1287, 1350, 1413, 1476, 1539, 1602, 1665, 1728, 1792, 1729, 1666, 1603, 1540, 1477, 1414, 1351, 1288, 1225, 1162, 1099, 1036, 973, 910, 847, 784, 721, 658, 595, 532, 469, 406, 343, 280, 217, 154, 91, 28, 29, 92, 155, 218, 281, 344, 407, 470, 533, 596, 659, 722, 785, 848, 911, 974, 1037, 1100, 1163, 1226, 1289, 1352, 1415, 1478, 1541, 1604, 1667, 1730, 1793, 1856, 1920, 1857, 1794, 1731, 1668, 1605, 1542, 1479, 1416, 1353, 1290, 1227, 1164, 1101, 1038, 975, 912, 849, 786, 723, 660, 597, 534, 471, 408, 345, 282, 219, 156, 93, 30, 31, 94, 157, 220, 283, 346, 409, 472, 535, 598, 661, 724, 787, 850, 913, 976,
    1039, 1102, 1165, 1228, 1291, 1354, 1417, 1480, 1543, 1606, 1669, 1732, 1795, 1858, 1921, 1984, 2048, 1985, 1922, 1859, 1796, 1733, 1670, 1607, 1544, 1481, 1418, 1355, 1292, 1229, 1166, 1103, 1040, 977, 914, 851, 788, 725, 662, 599, 536, 473, 410, 347, 284, 221, 158, 95, 32, 33, 96, 159, 222, 285, 348, 411, 474, 537, 600, 663, 726, 789, 852, 915, 978, 1041, 1104, 1167, 1230, 1293, 1356, 1419, 1482, 1545, 1608, 1671, 1734, 1797, 1860, 1923, 1986, 2049, 2112, 2176, 2113, 2050, 1987, 1924, 1861, 1798, 1735, 1672, 1609, 1546, 1483, 1420, 1357, 1294, 1231, 1168, 1105, 1042, 979, 916, 853, 790, 727, 664, 601, 538, 475, 412, 349, 286, 223, 160, 97, 34, 35, 98, 161, 224, 287, 350, 413, 476, 539, 602,
    665, 728, 791, 854, 917, 980, 1043, 1106, 1169, 1232, 1295, 1358, 1421, 1484, 1547, 1610, 1673, 1736, 1799, 1862, 1925, 1988, 2051, 2114, 2177, 2240, 2304, 2241, 2178, 2115, 2052, 1989, 1926, 1863, 1800, 1737, 1674, 1611, 1548, 1485, 1422, 1359, 1296, 1233, 1170, 1107, 1044, 981, 918, 855, 792, 729, 666, 603, 540, 477, 414, 351, 288, 225, 162, 99, 36, 37, 100, 163, 226, 289, 352, 415, 478, 541, 604, 667, 730, 793, 856, 919, 982, 1045, 1108, 1171, 1234, 1297, 1360, 1423, 1486, 1549, 1612, 1675, 1738, 1801, 1864, 1927, 1990, 2053, 2116, 2179, 2242, 2305, 2368, 2432, 2369, 2306, 2243, 2180, 2117, 2054, 1991, 1928, 1865, 1802, 1739, 1676, 1613, 1550, 1487, 1424, 1361, 1298, 1235, 1172, 1109, 1046, 983, 920, 857, 794,
    731, 668, 605, 542, 479, 416, 353, 290, 227, 164, 101, 38, 39, 102, 165, 228, 291, 354, 417, 480, 543, 606, 669, 732, 795, 858, 921, 984, 1047, 1110, 1173, 1236, 1299, 1362, 1425, 1488, 1551, 1614, 1677, 1740, 1803, 1866, 1929, 1992, 2055, 2118, 2181, 2244, 2307, 2370, 2433, 2496, 2560, 2497, 2434, 2371, 2308, 2245, 2182, 2119, 2056, 1993, 1930, 1867, 1804, 1741, 1678, 1615, 1552, 1489, 1426, 1363, 1300, 1237, 1174, 1111, 1048, 985, 922, 859, 796, 733, 670, 607, 544, 481, 418, 355, 292, 229, 166, 103, 40, 41, 104, 167, 230, 293, 356, 419, 482, 545, 608, 671, 734, 797, 860, 923, 986, 1049, 1112, 1175, 1238, 1301, 1364, 1427, 1490, 1553, 1616, 1679, 1742, 1805, 1868, 1931, 1994, 2057, 2120, 2183,
    2246, 2309, 2372, 2435, 2498, 2561, 2624, 2688, 2625, 2562, 2499, 2436, 2373, 2310, 2247, 2184, 2121, 2058, 1995, 1932, 1869, 1806, 1743, 1680, 1617, 1554, 1491, 1428, 1365, 1302, 1239, 1176, 1113, 1050, 987, 924, 861, 798, 735, 672, 609, 546, 483, 420, 357, 294, 231, 168, 105, 42, 43, 106, 169, 232, 295, 358, 421, 484, 547, 610, 673, 736, 799, 862, 925, 988, 1051, 1114, 1177, 1240, 1303, 1366, 1429, 1492, 1555, 1618, 1681, 1744, 1807, 1870, 1933, 1996, 2059, 2122, 2185, 2248, 2311, 2374, 2437, 2500, 2563, 2626, 2689, 2752, 2816, 2753, 2690, 2627, 2564, 2501, 2438, 2375, 2312, 2249, 2186, 2123, 2060, 1997, 1934, 1871, 1808, 1745, 1682, 1619, 1556, 1493, 1430, 1367, 1304, 1241, 1178, 1115, 1052, 989, 926, 863, 800, 737,
    674, 611, 548, 485, 422, 359, 296, 233, 170, 107, 44, 45, 108, 171, 234, 297, 360, 423, 486, 549, 612, 675, 738, 801, 864, 927, 990, 1053, 1116, 1179, 1242, 1305, 1368, 1431, 1494, 1557, 1620, 1683, 1746, 1809, 1872, 1935, 1998, 2061, 2124, 2187, 2250, 2313, 2376, 2439, 2502, 2565, 2628, 2691, 2754, 2817, 2880, 2944, 2881, 2818, 2755, 2692, 2629, 2566, 2503, 2440, 2377, 2314, 2251, 2188, 2125, 2062, 1999, 1936, 1873, 1810, 1747, 1684, 1621, 1558, 1495, 1432, 1369, 1306, 1243, 1180, 1117, 1054, 991, 928, 865, 802, 739, 676, 613, 550, 487, 424, 361, 298, 235, 172, 109, 46, 47, 110, 173, 236, 299, 362, 425, 488, 551, 614, 677, 740, 803, 866, 929, 992, 1055, 1118, 1181, 1244, 1307, 1370, 1433, 1496,
    1559, 1622, 1685, 1748, 1811, 1874, 1937, 2000, 2063, 2126, 2189, 2252, 2315, 2378, 2441, 2504, 2567, 2630, 2693, 2756, 2819, 2882, 2945, 3008, 3072, 3009, 2946, 2883, 2820, 2757, 2694, 2631, 2568, 2505, 2442, 2379, 2316, 2253, 2190, 2127, 2064, 2001, 1938, 1875, 1812, 1749, 1686, 1623, 1560, 1497, 1434, 1371, 1308, 1245, 1182, 1119, 1056, 993, 930, 867, 804, 741, 678, 615, 552, 489, 426, 363, 300, 237, 174, 111, 48, 49, 112, 175, 238, 301, 364, 427, 490, 553, 616, 679, 742, 805, 868, 931, 994, 1057, 1120, 1183, 1246, 1309, 1372, 1435, 1498, 1561, 1624, 1687, 1750, 1813, 1876, 1939, 2002, 2065, 2128, 2191, 2254, 2317, 2380, 2443, 2506, 2569, 2632, 2695, 2758, 2821, 2884, 2947, 3010, 3073, 3136, 3200, 3137, 3074, 3011, 2948,
    2885, 2822, 2759, 2696, 2633, 2570, 2507, 2444, 2381, 2318, 2255, 2192, 2129, 2066, 2003, 1940, 1877, 1814, 1751, 1688, 1625, 1562, 1499, 1436, 1373, 1310, 1247, 1184, 1121, 1058, 995, 932, 869, 806, 743, 680, 617, 554, 491, 428, 365, 302, 239, 176, 113, 50, 51, 114, 177, 240, 303, 366, 429, 492, 555, 618, 681, 744, 807, 870, 933, 996, 1059, 1122, 1185, 1248, 1311, 1374, 1437, 1500, 1563, 1626, 1689, 1752, 1815, 1878, 1941, 2004, 2067, 2130, 2193, 2256, 2319, 2382, 2445, 2508, 2571, 2634, 2697, 2760, 2823, 2886, 2949, 3012, 3075, 3138, 3201, 3264, 3328, 3265, 3202, 3139, 3076, 3013, 2950, 2887, 2824, 2761, 2698, 2635, 2572, 2509, 2446, 2383, 2320, 2257, 2194, 2131, 2068, 2005, 1942, 1879, 1816, 1753, 1690, 1627, 1564, 1501,
    1438, 1375, 1312, 1249, 1186, 1123, 1060, 997, 934, 871, 808, 745, 682, 619, 556, 493, 430, 367, 304, 241, 178, 115, 52, 53, 116, 179, 242, 305, 368, 431, 494, 557, 620, 683, 746, 809, 872, 935, 998, 1061, 1124, 1187, 1250, 1313, 1376, 1439, 1502, 1565, 1628, 1691, 1754, 1817, 1880, 1943, 2006, 2069, 2132, 2195, 2258, 2321, 2384, 2447, 2510, 2573, 2636, 2699, 2762, 2825, 2888, 2951, 3014, 3077, 3140, 3203, 3266, 3329, 3392, 3456, 3393, 3330, 3267, 3204, 3141, 3078, 3015, 2952, 2889, 2826, 2763, 2700, 2637, 2574, 2511, 2448, 2385, 2322, 2259, 2196, 2133, 2070, 2007, 1944, 1881, 1818, 1755, 1692, 1629, 1566, 1503, 1440, 1377, 1314, 1251, 1188, 1125, 1062, 999, 936, 873, 810, 747, 684, 621, 558, 495, 432, 369, 306,
    243, 180, 117, 54, 55, 118, 181, 244, 307, 370, 433, 496, 559, 622, 685, 748, 811, 874, 937, 1000, 1063, 1126, 1189, 1252, 1315, 1378, 1441, 1504, 1567, 1630, 1693, 1756, 1819, 1882, 1945, 2008, 2071, 2134, 2197, 2260, 2323, 2386, 2449, 2512, 2575, 2638, 2701, 2764, 2827, 2890, 2953, 3016, 3079, 3142, 3205, 3268, 3331, 3394, 3457, 3520, 3584, 3521, 3458, 3395, 3332, 3269, 3206, 3143, 3080, 3017, 2954, 2891, 2828, 2765, 2702, 2639, 2576, 2513, 2450, 2387, 2324, 2261, 2198, 2135, 2072, 2009, 1946, 1883, 1820, 1757, 1694, 1631, 1568, 1505, 1442, 1379, 1316, 1253, 1190, 1127, 1064, 1001, 938, 875, 812, 749, 686, 623, 560, 497, 434, 371, 308, 245, 182, 119, 56, 57, 120, 183, 246, 309, 372, 435, 498, 561, 624, 687,
    750, 813, 876, 939, 1002, 1065, 1128, 1191, 1254, 1317, 1380, 1443, 1506, 1569, 1632, 1695, 1758, 1821, 1884, 1947, 2010, 2073, 2136, 2199, 2262, 2325, 2388, 2451, 2514, 2577, 2640, 2703, 2766, 2829, 2892, 2955, 3018, 3081, 3144, 3207, 3270, 3333, 3396, 3459, 3522, 3585, 3648, 3712, 3649, 3586, 3523, 3460, 3397, 3334, 3271, 3208, 3145, 3082, 3019, 2956, 2893, 2830, 2767, 2704, 2641, 2578, 2515, 2452, 2389, 2326, 2263, 2200, 2137, 2074, 2011, 1948, 1885, 1822, 1759, 1696, 1633, 1570, 1507, 1444, 1381, 1318, 1255, 1192, 1129, 1066, 1003, 940, 877, 814, 751, 688, 625, 562, 499, 436, 373, 310, 247, 184, 121, 58, 59, 122, 185, 248, 311, 374, 437, 500, 563, 626, 689, 752, 815, 878, 941, 1004, 1067, 1130, 1193, 1256, 1319, 1382,
    1445, 1508, 1571, 1634, 1697, 1760, 1823, 1886, 1949, 2012, 2075, 2138, 2201, 2264, 2327, 2390, 2453, 2516, 2579, 2642, 2705, 2768, 2831, 2894, 2957, 3020, 3083, 3146, 3209, 3272, 3335, 3398, 3461, 3524, 3587, 3650, 3713, 3776, 3840, 3777, 3714, 3651, 3588, 3525, 3462, 3399, 3336, 3273, 3210, 3147, 3084, 3021, 2958, 2895, 2832, 2769, 2706, 2643, 2580, 2517, 2454, 2391, 2328, 2265, 2202, 2139, 2076, 2013, 1950, 1887, 1824, 1761, 1698, 1635, 1572, 1509, 1446, 1383, 1320, 1257, 1194, 1131, 1068, 1005, 942, 879, 816, 753, 690, 627, 564, 501, 438, 375, 312, 249, 186, 123, 60, 61, 124, 187, 250, 313, 376, 439, 502, 565, 628, 691, 754, 817, 880, 943, 1006, 1069, 1132, 1195, 1258, 1321, 1384, 1447, 1510, 1573, 1636, 1699, 1762, 1825,
    1888, 1951, 2014, 2077, 2140, 2203, 2266, 2329, 2392, 2455, 2518, 2581, 2644, 2707, 2770, 2833, 2896, 2959, 3022, 3085, 3148, 3211, 3274, 3337, 3400, 3463, 3526, 3589, 3652, 3715, 3778, 3841, 3904, 3968, 3905, 3842, 3779, 3716, 3653, 3590, 3527, 3464, 3401, 3338, 3275, 3212, 3149, 3086, 3023, 2960, 2897, 2834, 2771, 2708, 2645, 2582, 2519, 2456, 2393, 2330, 2267, 2204, 2141, 2078, 2015, 1952, 1889, 1826, 1763, 1700, 1637, 1574, 1511, 1448, 1385, 1322, 1259, 1196, 1133, 1070, 1007, 944, 881, 818, 755, 692, 629, 566, 503, 440, 377, 314, 251, 188, 125, 62, 63, 126, 189, 252, 315, 378, 441, 504, 567, 630, 693, 756, 819, 882, 945, 1008, 1071, 1134, 1197, 1260, 1323, 1386, 1449, 1512, 1575, 1638, 1701, 1764, 1827, 1890, 1953, 2016,
    2079, 2142, 2205, 2268, 2331, 2394, 2457, 2520, 2583, 2646, 2709, 2772, 2835, 2898, 2961, 3024, 3087, 3150, 3213, 3276, 3339, 3402, 3465, 3528, 3591, 3654, 3717, 3780, 3843, 3906, 3969, 4032, 4096, 4033, 3970, 3907, 3844, 3781, 3718, 3655, 3592, 3529, 3466, 3403, 3340, 3277, 3214, 3151, 3088, 3025, 2962, 2899, 2836, 2773, 2710, 2647, 2584, 2521, 2458, 2395, 2332, 2269, 2206, 2143, 2080, 2017, 1954, 1891, 1828, 1765, 1702, 1639, 1576, 1513, 1450, 1387, 1324, 1261, 1198, 1135, 1072, 1009, 946, 883, 820, 757, 694, 631, 568, 505, 442, 379, 316, 253, 190, 127, 191, 254, 317, 380, 443, 506, 569, 632, 695, 758, 821, 884, 947, 1010, 1073, 1136, 1199, 1262, 1325, 1388, 1451, 1514, 1577, 1640, 1703, 1766, 1829, 1892, 1955, 2018, 2081, 2144,
    2207, 2270, 2333, 2396, 2459, 2522, 2585, 2648, 2711, 2774, 2837, 2900, 2963, 3026, 3089, 3152, 3215, 3278, 3341, 3404, 3467, 3530, 3593, 3656, 3719, 3782, 3845, 3908, 3971, 4034, 4097, 4160, 4224, 4161, 4098, 4035, 3972, 3909, 3846, 3783, 3720, 3657, 3594, 3531, 3468, 3405, 3342, 3279, 3216, 3153, 3090, 3027, 2964, 2901, 2838, 2775, 2712, 2649, 2586, 2523, 2460, 2397, 2334, 2271, 2208, 2145, 2082, 2019, 1956, 1893, 1830, 1767, 1704, 1641, 1578, 1515, 1452, 1389, 1326, 1263, 1200, 1137, 1074, 1011, 948, 885, 822, 759, 696, 633, 570, 507, 444, 381, 318, 255, 319, 382, 445, 508, 571, 634, 697, 760, 823, 886, 949, 1012, 1075, 1138, 1201, 1264, 1327, 1390, 1453, 1516, 1579, 1642, 1705, 1768, 1831, 1894, 1957, 2020, 2083, 2146, 2209, 2272,
    2335, 2398, 2461, 2524, 2587, 2650, 2713, 2776, 2839, 2902, 2965, 3028, 3091, 3154, 3217, 3280, 3343, 3406, 3469, 3532, 3595, 3658, 3721, 3784, 3847, 3910, 3973, 4036, 4099, 4162, 4225, 4288, 4352, 4289, 4226, 4163, 4100, 4037, 3974, 3911, 3848, 3785, 3722, 3659, 3596, 3533, 3470, 3407, 3344, 3281, 3218, 3155, 3092, 3029, 2966, 2903, 2840, 2777, 2714, 2651, 2588, 2525, 2462, 2399, 2336, 2273, 2210, 2147, 2084, 2021, 1958, 1895, 1832, 1769, 1706, 1643, 1580, 1517, 1454, 1391, 1328, 1265, 1202, 1139, 1076, 1013, 950, 887, 824, 761, 698, 635, 572, 509, 446, 383, 447, 510, 573, 636, 699, 762, 825, 888, 951, 1014, 1077, 1140, 1203, 1266, 1329, 1392, 1455, 1518, 1581, 1644, 1707, 1770, 1833, 1896, 1959, 2022, 2085, 2148, 2211, 2274, 2337, 2400,
    2463, 2526, 2589, 2652, 2715, 2778, 2841, 2904, 2967, 3030, 3093, 3156, 3219, 3282, 3345, 3408, 3471, 3534, 3597, 3660, 3723, 3786, 3849, 3912, 3975, 4038, 4101, 4164, 4227, 4290, 4353, 4416, 4480, 4417, 4354, 4291, 4228, 4165, 4102, 4039, 3976, 3913, 3850, 3787, 3724, 3661, 3598, 3535, 3472, 3409, 3346, 3283, 3220, 3157, 3094, 3031, 2968, 2905, 2842, 2779, 2716, 2653, 2590, 2527, 2464, 2401, 2338, 2275, 2212, 2149, 2086, 2023, 1960, 1897, 1834, 1771, 1708, 1645, 1582, 1519, 1456, 1393, 1330, 1267, 1204, 1141, 1078, 1015, 952, 889, 826, 763, 700, 637, 574, 511, 575, 638, 701, 764, 827, 890, 953, 1016, 1079, 1142, 1205, 1268, 1331, 1394, 1457, 1520, 1583, 1646, 1709, 1772, 1835, 1898, 1961, 2024, 2087, 2150, 2213, 2276, 2339, 2402, 2465, 2528,
    2591, 2654, 2717, 2780, 2843, 2906, 2969, 3032, 3095, 3158, 3221, 3284, 3347, 3410, 3473, 3536, 3599, 3662, 3725, 3788, 3851, 3914, 3977, 4040, 4103, 4166, 4229, 4292, 4355, 4418, 4481, 4544, 4608, 4545, 4482, 4419, 4356, 4293, 4230, 4167, 4104, 4041, 3978, 3915, 3852, 3789, 3726, 3663, 3600, 3537, 3474, 3411, 3348, 3285, 3222, 3159, 3096, 3033, 2970, 2907, 2844, 2781, 2718, 2655, 2592, 2529, 2466, 2403, 2340, 2277, 2214, 2151, 2088, 2025, 1962, 1899, 1836, 1773, 1710, 1647, 1584, 1521, 1458, 1395, 1332, 1269, 1206, 1143, 1080, 1017, 954, 891, 828, 765, 702, 639, 703, 766, 829, 892, 955, 1018, 1081, 1144, 1207, 1270, 1333, 1396, 1459, 1522, 1585, 1648, 1711, 1774, 1837, 1900, 1963, 2026, 2089, 2152, 2215, 2278, 2341, 2404, 2467, 2530, 2593, 2656,
    2719, 2782, 2845, 2908, 2971, 3034, 3097, 3160, 3223, 3286, 3349, 3412, 3475, 3538, 3601, 3664, 3727, 3790, 3853, 3916, 3979, 4042, 4105, 4168, 4231, 4294, 4357, 4420, 4483, 4546, 4609, 4672, 4736, 4673, 4610, 4547, 4484, 4421, 4358, 4295, 4232, 4169, 4106, 4043, 3980, 3917, 3854, 3791, 3728, 3665, 3602, 3539, 3476, 3413, 3350, 3287, 3224, 3161, 3098, 3035, 2972, 2909, 2846, 2783, 2720, 2657, 2594, 2531, 2468, 2405, 2342, 2279, 2216, 2153, 2090, 2027, 1964, 1901, 1838, 1775, 1712, 1649, 1586, 1523, 1460, 1397, 1334, 1271, 1208, 1145, 1082, 1019, 956, 893, 830, 767, 831, 894, 957, 1020, 1083, 1146, 1209, 1272, 1335, 1398, 1461, 1524, 1587, 1650, 1713, 1776, 1839, 1902, 1965, 2028, 2091, 2154, 2217, 2280, 2343, 2406, 2469, 2532, 2595, 2658, 2721, 2784,
    2847, 2910, 2973, 3036, 3099, 3162, 3225, 3288, 3351, 3414, 3477, 3540, 3603, 3666, 3729, 3792, 3855, 3918, 3981, 4044, 4107, 4170, 4233, 4296, 4359, 4422, 4485, 4548, 4611, 4674, 4737, 4800, 4864, 4801, 4738, 4675, 4612, 4549, 4486, 4423, 4360, 4297, 4234, 4171, 4108, 4045, 3982, 3919, 3856, 3793, 3730, 3667, 3604, 3541, 3478, 3415, 3352, 3289, 3226, 3163, 3100, 3037, 2974, 2911, 2848, 2785, 2722, 2659, 2596, 2533, 2470, 2407, 2344, 2281, 2218, 2155, 2092, 2029, 1966, 1903, 1840, 1777, 1714, 1651, 1588, 1525, 1462, 1399, 1336, 1273, 1210, 1147, 1084, 1021, 958, 895, 959, 1022, 1085, 1148, 1211, 1274, 1337, 1400, 1463, 1526, 1589, 1652, 1715, 1778, 1841, 1904, 1967, 2030, 2093, 2156, 2219, 2282, 2345, 2408, 2471, 2534, 2597, 2660, 2723, 2786, 2849, 2912,
    2975, 3038, 3101, 3164, 3227, 3290, 3353, 3416, 3479, 3542, 3605, 3668, 3731, 3794, 3857, 3920, 3983, 4046, 4109, 4172, 4235, 4298, 4361, 4424, 4487, 4550, 4613, 4676, 4739, 4802, 4865, 4928, 4992, 4929, 4866, 4803, 4740, 4677, 4614, 4551, 4488, 4425, 4362, 4299, 4236, 4173, 4110, 4047, 3984, 3921, 3858, 3795, 3732, 3669, 3606, 3543, 3480, 3417, 3354, 3291, 3228, 3165, 3102, 3039, 2976, 2913, 2850, 2787, 2724, 2661, 2598, 2535, 2472, 2409, 2346, 2283, 2220, 2157, 2094, 2031, 1968, 1905, 1842, 1779, 1716, 1653, 1590, 1527, 1464, 1401, 1338, 1275, 1212, 1149, 1086, 1023, 1087, 1150, 1213, 1276, 1339, 1402, 1465, 1528, 1591, 1654, 1717, 1780, 1843, 1906, 1969, 2032, 2095, 2158, 2221, 2284, 2347, 2410, 2473, 2536, 2599, 2662, 2725, 2788, 2851, 2914, 2977, 3040,
    3103, 3166, 3229, 3292, 3355, 3418, 3481, 3544, 3607, 3670, 3733, 3796, 3859, 3922, 3985, 4048, 4111, 4174, 4237, 4300, 4363, 4426, 4489, 4552, 4615, 4678, 4741, 4804, 4867, 4930, 4993, 5056, 5120, 5057, 4994, 4931, 4868, 4805, 4742, 4679, 4616, 4553, 4490, 4427, 4364, 4301, 4238, 4175, 4112, 4049, 3986, 3923, 3860, 3797, 3734, 3671, 3608, 3545, 3482, 3419, 3356, 3293, 3230, 3167, 3104, 3041, 2978, 2915, 2852, 2789, 2726, 2663, 2600, 2537, 2474, 2411, 2348, 2285, 2222, 2159, 2096, 2033, 1970, 1907, 1844, 1781, 1718, 1655, 1592, 1529, 1466, 1403, 1340, 1277, 1214, 1151, 1215, 1278, 1341, 1404, 1467, 1530, 1593, 1656, 1719, 1782, 1845, 1908, 1971, 2034, 2097, 2160, 2223, 2286, 2349, 2412, 2475, 2538, 2601, 2664, 2727, 2790, 2853, 2916, 2979, 3042, 3105, 3168,
    3231, 3294, 3357, 3420, 3483, 3546, 3609, 3672, 3735, 3798, 3861, 3924, 3987, 4050, 4113, 4176, 4239, 4302, 4365, 4428, 4491, 4554, 4617, 4680, 4743, 4806, 4869, 4932, 4995, 5058, 5121, 5184, 5248, 5185, 5122, 5059, 4996, 4933, 4870, 4807, 4744, 4681, 4618, 4555, 4492, 4429, 4366, 4303, 4240, 4177, 4114, 4051, 3988, 3925, 3862, 3799, 3736, 3673, 3610, 3547, 3484, 3421, 3358, 3295, 3232, 3169, 3106, 3043, 2980, 2917, 2854, 2791, 2728, 2665, 2602, 2539, 2476, 2413, 2350, 2287, 2224, 2161, 2098, 2035, 1972, 1909, 1846, 1783, 1720, 1657, 1594, 1531, 1468, 1405, 1342, 1279, 1343, 1406, 1469, 1532, 1595, 1658, 1721, 1784, 1847, 1910, 1973, 2036, 2099, 2162, 2225, 2288, 2351, 2414, 2477, 2540, 2603, 2666, 2729, 2792, 2855, 2918, 2981, 3044, 3107, 3170, 3233, 3296,
    3359, 3422, 3485, 3548, 3611, 3674, 3737, 3800, 3863, 3926, 3989, 4052, 4115, 4178, 4241, 4304, 4367, 4430, 4493, 4556, 4619, 4682, 4745, 4808, 4871, 4934, 4997, 5060, 5123, 5186, 5249, 5312, 5376, 5313, 5250, 5187, 5124, 5061, 4998, 4935, 4872, 4809, 4746, 4683, 4620, 4557, 4494, 4431, 4368, 4305, 4242, 4179, 4116, 4053, 3990, 3927, 3864, 3801, 3738, 3675, 3612, 3549, 3486, 3423, 3360, 3297, 3234, 3171, 3108, 3045, 2982, 2919, 2856, 2793, 2730, 2667, 2604, 2541, 2478, 2415, 2352, 2289, 2226, 2163, 2100, 2037, 1974, 1911, 1848, 1785, 1722, 1659, 1596, 1533, 1470, 1407, 1471, 1534, 1597, 1660, 1723, 1786, 1849, 1912, 1975, 2038, 2101, 2164, 2227, 2290, 2353, 2416, 2479, 2542, 2605, 2668, 2731, 2794, 2857, 2920, 2983, 3046, 3109, 3172, 3235, 3298, 3361, 3424,
    3487, 3550, 3613, 3676, 3739, 3802, 3865, 3928, 3991, 4054, 4117, 4180, 4243, 4306, 4369, 4432, 4495, 4558, 4621, 4684, 4747, 4810, 4873, 4936, 4999, 5062, 5125, 5188, 5251, 5314, 5377, 5440, 5504, 5441, 5378, 5315, 5252, 5189, 5126, 5063, 5000, 4937, 4874, 4811, 4748, 4685, 4622, 4559, 4496, 4433, 4370, 4307, 4244, 4181, 4118, 4055, 3992, 3929, 3866, 3803, 3740, 3677, 3614, 3551, 3488, 3425, 3362, 3299, 3236, 3173, 3110, 3047, 2984, 2921, 2858, 2795, 2732, 2669, 2606, 2543, 2480, 2417, 2354, 2291, 2228, 2165, 2102, 2039, 1976, 1913, 1850, 1787, 1724, 1661, 1598, 1535, 1599, 1662, 1725, 1788, 1851, 1914, 1977, 2040, 2103, 2166, 2229, 2292, 2355, 2418, 2481, 2544, 2607, 2670, 2733, 2796, 2859, 2922, 2985, 3048, 3111, 3174, 3237, 3300, 3363, 3426, 3489, 3552,
    3615, 3678, 3741, 3804, 3867, 3930, 3993, 4056, 4119, 4182, 4245, 4308, 4371, 4434, 4497, 4560, 4623, 4686, 4749, 4812, 4875, 4938, 5001, 5064, 5127, 5190, 5253, 5316, 5379, 5442, 5505, 5568, 5632, 5569, 5506, 5443, 5380, 5317, 5254, 5191, 5128, 5065, 5002, 4939, 4876, 4813, 4750, 4687, 4624, 4561, 4498, 4435, 4372, 4309, 4246, 4183, 4120, 4057, 3994, 3931, 3868, 3805, 3742, 3679, 3616, 3553, 3490, 3427, 3364, 3301, 3238, 3175, 3112, 3049, 2986, 2923, 2860, 2797, 2734, 2671, 2608, 2545, 2482, 2419, 2356, 2293, 2230, 2167, 2104, 2041, 1978, 1915, 1852, 1789, 1726, 1663, 1727, 1790, 1853, 1916, 1979, 2042, 2105, 2168, 2231, 2294, 2357, 2420, 2483, 2546, 2609, 2672, 2735, 2798, 2861, 2924, 2987, 3050, 3113, 3176, 3239, 3302, 3365, 3428, 3491, 3554, 3617, 3680,
    3743, 3806, 3869, 3932, 3995, 4058, 4121, 4184, 4247, 4310, 4373, 4436, 4499, 4562, 4625, 4688, 4751, 4814, 4877, 4940, 5003, 5066, 5129, 5192, 5255, 5318, 5381, 5444, 5507, 5570, 5633, 5696, 5760, 5697, 5634, 5571, 5508, 5445, 5382, 5319, 5256, 5193, 5130, 5067, 5004, 4941, 4878, 4815, 4752, 4689, 4626, 4563, 4500, 4437, 4374, 4311, 4248, 4185, 4122, 4059, 3996, 3933, 3870, 3807, 3744, 3681, 3618, 3555, 3492, 3429, 3366, 3303, 3240, 3177, 3114, 3051, 2988, 2925, 2862, 2799, 2736, 2673, 2610, 2547, 2484, 2421, 2358, 2295, 2232, 2169, 2106, 2043, 1980, 1917, 1854, 1791, 1855, 1918, 1981, 2044, 2107, 2170, 2233, 2296, 2359, 2422, 2485, 2548, 2611, 2674, 2737, 2800, 2863, 2926, 2989, 3052, 3115, 3178, 3241, 3304, 3367, 3430, 3493, 3556, 3619, 3682, 3745, 3808,
    3871, 3934, 3997, 4060, 4123, 4186, 4249, 4312, 4375, 4438, 4501, 4564, 4627, 4690, 4753, 4816, 4879, 4942, 5005, 5068, 5131, 5194, 5257, 5320, 5383, 5446, 5509, 5572, 5635, 5698, 5761, 5824, 5888, 5825, 5762, 5699, 5636, 5573, 5510, 5447, 5384, 5321, 5258, 5195, 5132, 5069, 5006, 4943, 4880, 4817, 4754, 4691, 4628, 4565, 4502, 4439, 4376, 4313, 4250, 4187, 4124, 4061, 3998, 3935, 3872, 3809, 3746, 3683, 3620, 3557, 3494, 3431, 3368, 3305, 3242, 3179, 3116, 3053, 2990, 2927, 2864, 2801, 2738, 2675, 2612, 2549, 2486, 2423, 2360, 2297, 2234, 2171, 2108, 2045, 1982, 1919, 1983, 2046, 2109, 2172, 2235, 2298, 2361, 2424, 2487, 2550, 2613, 2676, 2739, 2802, 2865, 2928, 2991, 3054, 3117, 3180, 3243, 3306, 3369, 3432, 3495, 3558, 3621, 3684, 3747, 3810, 3873, 3936,
    3999, 4062, 4125, 4188, 4251, 4314, 4377, 4440, 4503, 4566, 4629, 4692, 4755, 4818, 4881, 4944, 5007, 5070, 5133, 5196, 5259, 5322, 5385, 5448, 5511, 5574, 5637, 5700, 5763, 5826, 5889, 5952, 6016, 5953, 5890, 5827, 5764, 5701, 5638, 5575, 5512, 5449, 5386, 5323, 5260, 5197, 5134, 5071, 5008, 4945, 4882, 4819, 4756, 4693, 4630, 4567, 4504, 4441, 4378, 4315, 4252, 4189, 4126, 4063, 4000, 3937, 3874, 3811, 3748, 3685, 3622, 3559, 3496, 3433, 3370, 3307, 3244, 3181, 3118, 3055, 2992, 2929, 2866, 2803, 2740, 2677, 2614, 2551, 2488, 2425, 2362, 2299, 2236, 2173, 2110, 2047, 2111, 2174, 2237, 2300, 2363, 2426, 2489, 2552, 2615, 2678, 2741, 2804, 2867, 2930, 2993, 3056, 3119, 3182, 3245, 3308, 3371, 3434, 3497, 3560, 3623, 3686, 3749, 3812, 3875, 3938, 4001, 4064,
    4127, 4190, 4253, 4316, 4379, 4442, 4505, 4568, 4631, 4694, 4757, 4820, 4883, 4946, 5009, 5072, 5135, 5198, 5261, 5324, 5387, 5450, 5513, 5576, 5639, 5702, 5765, 5828, 5891, 5954, 6017, 6080, 6144, 6081, 6018, 5955, 5892, 5829, 5766, 5703, 5640, 5577, 5514, 5451, 5388, 5325, 5262, 5199, 5136, 5073, 5010, 4947, 4884, 4821, 4758, 4695, 4632, 4569, 4506, 4443, 4380, 4317, 4254, 4191, 4128, 4065, 4002, 3939, 3876, 3813, 3750, 3687, 3624, 3561, 3498, 3435, 3372, 3309, 3246, 3183, 3120, 3057, 2994, 2931, 2868, 2805, 2742, 2679, 2616, 2553, 2490, 2427, 2364, 2301, 2238, 2175, 2239, 2302, 2365, 2428, 2491, 2554, 2617, 2680, 2743, 2806, 2869, 2932, 2995, 3058, 3121, 3184, 3247, 3310, 3373, 3436, 3499, 3562, 3625, 3688, 3751, 3814, 3877, 3940, 4003, 4066, 4129, 4192,
    4255, 4318, 4381, 4444, 4507, 4570, 4633, 4696, 4759, 4822, 4885, 4948, 5011, 5074, 5137, 5200, 5263, 5326, 5389, 5452, 5515, 5578, 5641, 5704, 5767, 5830, 5893, 5956, 6019, 6082, 6145, 6208, 6272, 6209, 6146, 6083, 6020, 5957, 5894, 5831, 5768, 5705, 5642, 5579, 5516, 5453, 5390, 5327, 5264, 5201, 5138, 5075, 5012, 4949, 4886, 4823, 4760, 4697, 4634, 4571, 4508, 4445, 4382, 4319, 4256, 4193, 4130, 4067, 4004, 3941, 3878, 3815, 3752, 3689, 3626, 3563, 3500, 3437, 3374, 3311, 3248, 3185, 3122, 3059, 2996, 2933, 2870, 2807, 2744, 2681, 2618, 2555, 2492, 2429, 2366, 2303, 2367, 2430, 2493, 2556, 2619, 2682, 2745, 2808, 2871, 2934, 2997, 3060, 3123, 3186, 3249, 3312, 3375, 3438, 3501, 3564, 3627, 3690, 3753, 3816, 3879, 3942, 4005, 4068, 4131, 4194, 4257, 4320,
    4383, 4446, 4509, 4572, 4635, 4698, 4761, 4824, 4887, 4950, 5013, 5076, 5139, 5202, 5265, 5328, 5391, 5454, 5517, 5580, 5643, 5706, 5769, 5832, 5895, 5958, 6021, 6084, 6147, 6210, 6273, 6336, 6400, 6337, 6274, 6211, 6148, 6085, 6022, 5959, 5896, 5833, 5770, 5707, 5644, 5581, 5518, 5455, 5392, 5329, 5266, 5203, 5140, 5077, 5014, 4951, 4888, 4825, 4762, 4699, 4636, 4573, 4510, 4447, 4384, 4321, 4258, 4195, 4132, 4069, 4006, 3943, 3880, 3817, 3754, 3691, 3628, 3565, 3502, 3439, 3376, 3313, 3250, 3187, 3124, 3061, 2998, 2935, 2872, 2809, 2746, 2683, 2620, 2557, 2494, 2431, 2495, 2558, 2621, 2684, 2747, 2810, 2873, 2936, 2999, 3062, 3125, 3188, 3251, 3314, 3377, 3440, 3503, 3566, 3629, 3692, 3755, 3818, 3881, 3944, 4007, 4070, 4133, 4196, 4259, 4322, 4385, 4448,
    4511, 4574, 4637, 4700, 4763, 4826, 4889, 4952, 5015, 5078, 5141, 5204, 5267, 5330, 5393, 5456, 5519, 5582, 5645, 5708, 5771, 5834, 5897, 5960, 6023, 6086, 6149, 6212, 6275, 6338, 6401, 6464, 6528, 6465, 6402, 6339, 6276, 6213, 6150, 6087, 6024, 5961, 5898, 5835, 5772, 5709, 5646, 5583, 5520, 5457, 5394, 5331, 5268, 5205, 5142, 5079, 5016, 4953, 4890, 4827, 4764, 4701, 4638, 4575, 4512, 4449, 4386, 4323, 4260, 4197, 4134, 4071, 4008, 3945, 3882, 3819, 3756, 3693, 3630, 3567, 3504, 3441, 3378, 3315, 3252, 3189, 3126, 3063, 3000, 2937, 2874, 2811, 2748, 2685, 2622, 2559, 2623, 2686, 2749, 2812, 2875, 2938, 3001, 3064, 3127, 3190, 3253, 3316, 3379, 3442, 3505, 3568, 3631, 3694, 3757, 3820, 3883, 3946, 4009, 4072, 4135, 4198, 4261, 4324, 4387, 4450, 4513, 4576,
    4639, 4702, 4765, 4828, 4891, 4954, 5017, 5080, 5143, 5206, 5269, 5332, 5395, 5458, 5521, 5584, 5647, 5710, 5773, 5836, 5899, 5962, 6025, 6088, 6151, 6214, 6277, 6340, 6403, 6466, 6529, 6592, 6656, 6593, 6530, 6467, 6404, 6341, 6278, 6215, 6152, 6089, 6026, 5963, 5900, 5837, 5774, 5711, 5648, 5585, 5522, 5459, 5396, 5333, 5270, 5207, 5144, 5081, 5018, 4955, 4892, 4829, 4766, 4703, 4640, 4577, 4514, 4451, 4388, 4325, 4262, 4199, 4136, 4073, 4010, 3947, 3884, 3821, 3758, 3695, 3632, 3569, 3506, 3443, 3380, 3317, 3254, 3191, 3128, 3065, 3002, 2939, 2876, 2813, 2750, 2687, 2751, 2814, 2877, 2940, 3003, 3066, 3129, 3192, 3255, 3318, 3381, 3444, 3507, 3570, 3633, 3696, 3759, 3822, 3885, 3948, 4011, 4074, 4137, 4200, 4263, 4326, 4389, 4452, 4515, 4578, 4641, 4704,
    4767, 4830, 4893, 4956, 5019, 5082, 5145, 5208, 5271, 5334, 5397, 5460, 5523, 5586, 5649, 5712, 5775, 5838, 5901, 5964, 6027, 6090, 6153, 6216, 6279, 6342, 6405, 6468, 6531, 6594, 6657, 6720, 6784, 6721, 6658, 6595, 6532, 6469, 6406, 6343, 6280, 6217, 6154, 6091, 6028, 5965, 5902, 5839, 5776, 5713, 5650, 5587, 5524, 5461, 5398, 5335, 5272, 5209, 5146, 5083, 5020, 4957, 4894, 4831, 4768, 4705, 4642, 4579, 4516, 4453, 4390, 4327, 4264, 4201, 4138, 4075, 4012, 3949, 3886, 3823, 3760, 3697, 3634, 3571, 3508, 3445, 3382, 3319, 3256, 3193, 3130, 3067, 3004, 2941, 2878, 2815, 2879, 2942, 3005, 3068, 3131, 3194, 3257, 3320, 3383, 3446, 3509, 3572, 3635, 3698, 3761, 3824, 3887, 3950, 4013, 4076, 4139, 4202, 4265, 4328, 4391, 4454, 4517, 4580, 4643, 4706, 4769, 4832,
    4895, 4958, 5021, 5084, 5147, 5210, 5273, 5336, 5399, 5462, 5525, 5588, 5651, 5714, 5777, 5840, 5903, 5966, 6029, 6092, 6155, 6218, 6281, 6344, 6407, 6470, 6533, 6596, 6659, 6722, 6785, 6848, 6912, 6849, 6786, 6723, 6660, 6597, 6534, 6471, 6408, 6345, 6282, 6219, 6156, 6093, 6030, 5967, 5904, 5841, 5778, 5715, 5652, 5589, 5526, 5463, 5400, 5337, 5274, 5211, 5148, 5085, 5022, 4959, 4896, 4833, 4770, 4707, 4644, 4581, 4518, 4455, 4392, 4329, 4266, 4203, 4140, 4077, 4014, 3951, 3888, 3825, 3762, 3699, 3636, 3573, 3510, 3447, 3384, 3321, 3258, 3195, 3132, 3069, 3006, 2943, 3007, 3070, 3133, 3196, 3259, 3322, 3385, 3448, 3511, 3574, 3637, 3700, 3763, 3826, 3889, 3952, 4015, 4078, 4141, 4204, 4267, 4330, 4393, 4456, 4519, 4582, 4645, 4708, 4771, 4834, 4897, 4960,
    5023, 5086, 5149, 5212, 5275, 5338, 5401, 5464, 5527, 5590, 5653, 5716, 5779, 5842, 5905, 5968, 6031, 6094, 6157, 6220, 6283, 6346, 6409, 6472, 6535, 6598, 6661, 6724, 6787, 6850, 6913, 6976, 7040, 6977, 6914, 6851, 6788, 6725, 6662, 6599, 6536, 6473, 6410, 6347, 6284, 6221, 6158, 6095, 6032, 5969, 5906, 5843, 5780, 5717, 5654, 5591, 5528, 5465, 5402, 5339, 5276, 5213, 5150, 5087, 5024, 4961, 4898, 4835, 4772, 4709, 4646, 4583, 4520, 4457, 4394, 4331, 4268, 4205, 4142, 4079, 4016, 3953, 3890, 3827, 3764, 3701, 3638, 3575, 3512, 3449, 3386, 3323, 3260, 3197, 3134, 3071, 3135, 3198, 3261, 3324, 3387, 3450, 3513, 3576, 3639, 3702, 3765, 3828, 3891, 3954, 4017, 4080, 4143, 4206, 4269, 4332, 4395, 4458, 4521, 4584, 4647, 4710, 4773, 4836, 4899, 4962, 5025, 5088,
    5151, 5214, 5277, 5340, 5403, 5466, 5529, 5592, 5655, 5718, 5781, 5844, 5907, 5970, 6033, 6096, 6159, 6222, 6285, 6348, 6411, 6474, 6537, 6600, 6663, 6726, 6789, 6852, 6915, 6978, 7041, 7104, 7168, 7105, 7042, 6979, 6916, 6853, 6790, 6727, 6664, 6601, 6538, 6475, 6412, 6349, 6286, 6223, 6160, 6097, 6034, 5971, 5908, 5845, 5782, 5719, 5656, 5593, 5530, 5467, 5404, 5341, 5278, 5215, 5152, 5089, 5026, 4963, 4900, 4837, 4774, 4711, 4648, 4585, 4522, 4459, 4396, 4333, 4270, 4207, 4144, 4081, 4018, 3955, 3892, 3829, 3766, 3703, 3640, 3577, 3514, 3451, 3388, 3325, 3262, 3199, 3263, 3326, 3389, 3452, 3515, 3578, 3641, 3704, 3767, 3830, 3893, 3956, 4019, 4082, 4145, 4208, 4271, 4334, 4397, 4460, 4523, 4586, 4649, 4712, 4775, 4838, 4901, 4964, 5027, 5090, 5153, 5216,
    5279, 5342, 5405, 5468, 5531, 5594, 5657, 5720, 5783, 5846, 5909, 5972, 6035, 6098, 6161, 6224, 6287, 6350, 6413, 6476, 6539, 6602, 6665, 6728, 6791, 6854, 6917, 6980, 7043, 7106, 7169, 7232, 7296, 7233, 7170, 7107, 7044, 6981, 6918, 6855, 6792, 6729, 6666, 6603, 6540, 6477, 6414, 6351, 6288, 6225, 6162, 6099, 6036, 5973, 5910, 5847, 5784, 5721, 5658, 5595, 5532, 5469, 5406, 5343, 5280, 5217, 5154, 5091, 5028, 4965, 4902, 4839, 4776, 4713, 4650, 4587, 4524, 4461, 4398, 4335, 4272, 4209, 4146, 4083, 4020, 3957, 3894, 3831, 3768, 3705, 3642, 3579, 3516, 3453, 3390, 3327, 3391, 3454, 3517, 3580, 3643, 3706, 3769, 3832, 3895, 3958, 4021, 4084, 4147, 4210, 4273, 4336, 4399, 4462, 4525, 4588, 4651, 4714, 4777, 4840, 4903, 4966, 5029, 5092, 5155, 5218, 5281, 5344,
    5407, 5470, 5533, 5596, 5659, 5722, 5785, 5848, 5911, 5974, 6037, 6100, 6163, 6226, 6289, 6352, 6415, 6478, 6541, 6604, 6667, 6730, 6793, 6856, 6919, 6982, 7045, 7108, 7171, 7234, 7297, 7360, 7424, 7361, 7298, 7235, 7172, 7109, 7046, 6983, 6920, 6857, 6794, 6731, 6668, 6605, 6542, 6479, 6416, 6353, 6290, 6227, 6164, 6101, 6038, 5975, 5912, 5849, 5786, 5723, 5660, 5597, 5534, 5471, 5408, 5345, 5282, 5219, 5156, 5093, 5030, 4967, 4904, 4841, 4778, 4715, 4652, 4589, 4526, 4463, 4400, 4337, 4274, 4211, 4148, 4085, 4022, 3959, 3896, 3833, 3770, 3707, 3644, 3581, 3518, 3455, 3519, 3582, 3645, 3708, 3771, 3834, 3897, 3960, 4023, 4086, 4149, 4212, 4275, 4338, 4401, 4464, 4527, 4590, 4653, 4716, 4779, 4842, 4905, 4968, 5031, 5094, 5157, 5220, 5283, 5346, 5409, 5472,
    5535, 5598, 5661, 5724, 5787, 5850, 5913, 5976, 6039, 6102, 6165, 6228, 6291, 6354, 6417, 6480, 6543, 6606, 6669, 6732, 6795, 6858, 6921, 6984, 7047, 7110, 7173, 7236, 7299, 7362, 7425, 7488, 7552, 7489, 7426, 7363, 7300, 7237, 7174, 7111, 7048, 6985, 6922, 6859, 6796, 6733, 6670, 6607, 6544, 6481, 6418, 6355, 6292, 6229, 6166, 6103, 6040, 5977, 5914, 5851, 5788, 5725, 5662, 5599, 5536, 5473, 5410, 5347, 5284, 5221, 5158, 5095, 5032, 4969, 4906, 4843, 4780, 4717, 4654, 4591, 4528, 4465, 4402, 4339, 4276, 4213, 4150, 4087, 4024, 3961, 3898, 3835, 3772, 3709, 3646, 3583, 3647, 3710, 3773, 3836, 3899, 3962, 4025, 4088, 4151, 4214, 4277, 4340, 4403, 4466, 4529, 4592, 4655, 4718, 4781, 4844, 4907, 4970, 5033, 5096, 5159, 5222, 5285, 5348, 5411, 5474, 5537, 5600,
    5663, 5726, 5789, 5852, 5915, 5978, 6041, 6104, 6167, 6230, 6293, 6356, 6419, 6482, 6545, 6608, 6671, 6734, 6797, 6860, 6923, 6986, 7049, 7112, 7175, 7238, 7301, 7364, 7427, 7490, 7553, 7616, 7680, 7617, 7554, 7491, 7428, 7365, 7302, 7239, 7176, 7113, 7050, 6987, 6924, 6861, 6798, 6735, 6672, 6609, 6546, 6483, 6420, 6357, 6294, 6231, 6168, 6105, 6042, 5979, 5916, 5853, 5790, 5727, 5664, 5601, 5538, 5475, 5412, 5349, 5286, 5223, 5160, 5097, 5034, 4971, 4908, 4845, 4782, 4719, 4656, 4593, 4530, 4467, 4404, 4341, 4278, 4215, 4152, 4089, 4026, 3963, 3900, 3837, 3774, 3711, 3775, 3838, 3901, 3964, 4027, 4090, 4153, 4216, 4279, 4342, 4405, 4468, 4531, 4594, 4657, 4720, 4783, 4846, 4909, 4972, 5035, 5098, 5161, 5224, 5287, 5350, 5413, 5476, 5539, 5602, 5665, 5728,
    5791, 5854, 5917, 5980, 6043, 6106, 6169, 6232, 6295, 6358, 6421, 6484, 6547, 6610, 6673, 6736, 6799, 6862, 6925, 6988, 7051, 7114, 7177, 7240, 7303, 7366, 7429, 7492, 7555, 7618, 7681, 7744, 7808, 7745, 7682, 7619, 7556, 7493, 7430, 7367, 7304, 7241, 7178, 7115, 7052, 6989, 6926, 6863, 6800, 6737, 6674, 6611, 6548, 6485, 6422, 6359, 6296, 6233, 6170, 6107, 6044, 5981, 5918, 5855, 5792, 5729, 5666, 5603, 5540, 5477, 5414, 5351, 5288, 5225, 5162, 5099, 5036, 4973, 4910, 4847, 4784, 4721, 4658, 4595, 4532, 4469, 4406, 4343, 4280, 4217, 4154, 4091, 4028, 3965, 3902, 3839, 3903, 3966, 4029, 4092, 4155, 4218, 4281, 4344, 4407, 4470, 4533, 4596, 4659, 4722, 4785, 4848, 4911, 4974, 5037, 5100, 5163, 5226, 5289, 5352, 5415, 5478, 5541, 5604, 5667, 5730, 5793, 5856,
    5919, 5982, 6045, 6108, 6171, 6234, 6297, 6360, 6423, 6486, 6549, 6612, 6675, 6738, 6801, 6864, 6927, 6990, 7053, 7116, 7179, 7242, 7305, 7368, 7431, 7494, 7557, 7620, 7683, 7746, 7809, 7872, 7936, 7873, 7810, 7747, 7684, 7621, 7558, 7495, 7432, 7369, 7306, 7243, 7180, 7117, 7054, 6991, 6928, 6865, 6802, 6739, 6676, 6613, 6550, 6487, 6424, 6361, 6298, 6235, 6172, 6109, 6046, 5983, 5920, 5857, 5794, 5731, 5668, 5605, 5542, 5479, 5416, 5353, 5290, 5227, 5164, 5101, 5038, 4975, 4912, 4849, 4786, 4723, 4660, 4597, 4534, 4471, 4408, 4345, 4282, 4219, 4156, 4093, 4030, 3967, 4031, 4094, 4157, 4220, 4283, 4346, 4409, 4472, 4535, 4598, 4661, 4724, 4787, 4850, 4913, 4976, 5039, 5102, 5165, 5228, 5291, 5354, 5417, 5480, 5543, 5606, 5669, 5732, 5795, 5858, 5921, 5984,
    6047, 6110, 6173, 6236, 6299, 6362, 6425, 6488, 6551, 6614, 6677, 6740, 6803, 6866, 6929, 6992, 7055, 7118, 7181, 7244, 7307, 7370, 7433, 7496, 7559, 7622, 7685, 7748, 7811, 7874, 7937, 8000, 8064, 8001, 7938, 7875, 7812, 7749, 7686, 7623, 7560, 7497, 7434, 7371, 7308, 7245, 7182, 7119, 7056, 6993, 6930, 6867, 6804, 6741, 6678, 6615, 6552, 6489, 6426, 6363, 6300, 6237, 6174, 6111, 6048, 5985, 5922, 5859, 5796, 5733, 5670, 5607, 5544, 5481, 5418, 5355, 5292, 5229, 5166, 5103, 5040, 4977, 4914, 4851, 4788, 4725, 4662, 4599, 4536, 4473, 4410, 4347, 4284, 4221, 4158, 4095, 4159, 4222, 4285, 4348, 4411, 4474, 4537, 4600, 4663, 4726, 4789, 4852, 4915, 4978, 5041, 5104, 5167, 5230, 5293, 5356, 5419, 5482, 5545, 5608, 5671, 5734, 5797, 5860, 5923, 5986, 6049, 6112,
    6175, 6238, 6301, 6364, 6427, 6490, 6553, 6616, 6679, 6742, 6805, 6868, 6931, 6994, 7057, 7120, 7183, 7246, 7309, 7372, 7435, 7498, 7561, 7624, 7687, 7750, 7813, 7876, 7939, 8002, 8065, 8128, 8129, 8066, 8003, 7940, 7877, 7814, 7751, 7688, 7625, 7562, 7499, 7436, 7373, 7310, 7247, 7184, 7121, 7058, 6995, 6932, 6869, 6806, 6743, 6680, 6617, 6554, 6491, 6428, 6365, 6302, 6239, 6176, 6113, 6050, 5987, 5924, 5861, 5798, 5735, 5672, 5609, 5546, 5483, 5420, 5357, 5294, 5231, 5168, 5105, 5042, 4979, 4916, 4853, 4790, 4727, 4664, 4601, 4538, 4475, 4412, 4349, 4286, 4223, 4287, 4350, 4413, 4476, 4539, 4602, 4665, 4728, 4791, 4854, 4917, 4980, 5043, 5106, 5169, 5232, 5295, 5358, 5421, 5484, 5547, 5610, 5673, 5736, 5799, 5862, 5925, 5988, 6051, 6114, 6177, 6240, 6303,
    6366, 6429, 6492, 6555, 6618, 6681, 6744, 6807, 6870, 6933, 6996, 7059, 7122, 7185, 7248, 7311, 7374, 7437, 7500, 7563, 7626, 7689, 7752, 7815, 7878, 7941, 8004, 8067, 8130, 8131, 8068, 8005, 7942, 7879, 7816, 7753, 7690, 7627, 7564, 7501, 7438, 7375, 7312, 7249, 7186, 7123, 7060, 6997, 6934, 6871, 6808, 6745, 6682, 6619, 6556, 6493, 6430, 6367, 6304, 6241, 6178, 6115, 6052, 5989, 5926, 5863, 5800, 5737, 5674, 5611, 5548, 5485, 5422, 5359, 5296, 5233, 5170, 5107, 5044, 4981, 4918, 4855, 4792, 4729, 4666, 4603, 4540, 4477, 4414, 4351, 4415, 4478, 4541, 4604, 4667, 4730, 4793, 4856, 4919, 4982, 5045, 5108, 5171, 5234, 5297, 5360, 5423, 5486, 5549, 5612, 5675, 5738, 5801, 5864, 5927, 5990, 6053, 6116, 6179, 6242, 6305, 6368, 6431, 6494, 6557, 6620, 6683, 6746,
    6809, 6872, 6935, 6998, 7061, 7124, 7187, 7250, 7313, 7376, 7439, 7502, 7565, 7628, 7691, 7754, 7817, 7880, 7943, 8006, 8069, 8132, 8133, 8070, 8007, 7944, 7881, 7818, 7755, 7692, 7629, 7566, 7503, 7440, 7377, 7314, 7251, 7188, 7125, 7062, 6999, 6936, 6873, 6810, 6747, 6684, 6621, 6558, 6495, 6432, 6369, 6306, 6243, 6180, 6117, 6054, 5991, 5928, 5865, 5802, 5739, 5676, 5613, 5550, 5487, 5424, 5361, 5298, 5235, 5172, 5109, 5046, 4983, 4920, 4857, 4794, 4731, 4668, 4605, 4542, 4479, 4543, 4606, 4669, 4732, 4795, 4858, 4921, 4984, 5047, 5110, 5173, 5236, 5299, 5362, 5425, 5488, 5551, 5614, 5677, 5740, 5803, 5866, 5929, 5992, 6055, 6118, 6181, 6244, 6307, 6370, 6433, 6496, 6559, 6622, 6685, 6748, 6811, 6874, 6937, 7000, 7063, 7126, 7189, 7252, 7315, 7378, 7441,
    7504, 7567, 7630, 7693, 7756, 7819, 7882, 7945, 8008, 8071, 8134, 8135, 8072, 8009, 7946, 7883, 7820, 7757, 7694, 7631, 7568, 7505, 7442, 7379, 7316, 7253, 7190, 7127, 7064, 7001, 6938, 6875, 6812, 6749, 6686, 6623, 6560, 6497, 6434, 6371, 6308, 6245, 6182, 6119, 6056, 5993, 5930, 5867, 5804, 5741, 5678, 5615, 5552, 5489, 5426, 5363, 5300, 5237, 5174, 5111, 5048, 4985, 4922, 4859, 4796, 4733, 4670, 4607, 4671, 4734, 4797, 4860, 4923, 4986, 5049, 5112, 5175, 5238, 5301, 5364, 5427, 5490, 5553, 5616, 5679, 5742, 5805, 5868, 5931, 5994, 6057, 6120, 6183, 6246, 6309, 6372, 6435, 6498, 6561, 6624, 6687, 6750, 6813, 6876, 6939, 7002, 7065, 7128, 7191, 7254, 7317, 7380, 7443, 7506, 7569, 7632, 7695, 7758, 7821, 7884, 7947, 8010, 8073, 8136, 8137, 8074, 8011, 7948,
    7885, 7822, 7759, 7696, 7633, 7570, 7507, 7444, 7381, 7318, 7255, 7192, 7129, 7066, 7003, 6940, 6877, 6814, 6751, 6688, 6625, 6562, 6499, 6436, 6373, 6310, 6247, 6184, 6121, 6058, 5995, 5932, 5869, 5806, 5743, 5680, 5617, 5554, 5491, 5428, 5365, 5302, 5239, 5176, 5113, 5050, 4987, 4924, 4861, 4798, 4735, 4799, 4862, 4925, 4988, 5051, 5114, 5177, 5240, 5303, 5366, 5429, 5492, 5555, 5618, 5681, 5744, 5807, 5870, 5933, 5996, 6059, 6122, 6185, 6248, 6311, 6374, 6437, 6500, 6563, 6626, 6689, 6752, 6815, 6878, 6941, 7004, 7067, 7130, 7193, 7256, 7319, 7382, 7445, 7508, 7571, 7634, 7697, 7760, 7823, 7886, 7949, 8012, 8075, 8138, 8139, 8076, 8013, 7950, 7887, 7824, 7761, 7698, 7635, 7572, 7509, 7446, 7383, 7320, 7257, 7194, 7131, 7068, 7005, 6942, 6879, 6816, 6753,
    6690, 6627, 6564, 6501, 6438, 6375, 6312, 6249, 6186, 6123, 6060, 5997, 5934, 5871, 5808, 5745, 5682, 5619, 5556, 5493, 5430, 5367, 5304, 5241, 5178, 5115, 5052, 4989, 4926, 4863, 4927, 4990, 5053, 5116, 5179, 5242, 5305, 5368, 5431, 5494, 5557, 5620, 5683, 5746, 5809, 5872, 5935, 5998, 6061, 6124, 6187, 6250, 6313, 6376, 6439, 6502, 6565, 6628, 6691, 6754, 6817, 6880, 6943, 7006, 7069, 7132, 7195, 7258, 7321, 7384, 7447, 7510, 7573, 7636, 7699, 7762, 7825, 7888, 7951, 8014, 8077, 8140, 8141, 8078, 8015, 7952, 7889, 7826, 7763, 7700, 7637, 7574, 7511, 7448, 7385, 7322, 7259, 7196, 7133, 7070, 7007, 6944, 6881, 6818, 6755, 6692, 6629, 6566, 6503, 6440, 6377, 6314, 6251, 6188, 6125, 6062, 5999, 5936, 5873, 5810, 5747, 5684, 5621, 5558, 5495, 5432, 5369, 5306,
    5243, 5180, 5117, 5054, 4991, 5055, 5118, 5181, 5244, 5307, 5370, 5433, 5496, 5559, 5622, 5685, 5748, 5811, 5874, 5937, 6000, 6063, 6126, 6189, 6252, 6315, 6378, 6441, 6504, 6567, 6630, 6693, 6756, 6819, 6882, 6945, 7008, 7071, 7134, 7197, 7260, 7323, 7386, 7449, 7512, 7575, 7638, 7701, 7764, 7827, 7890, 7953, 8016, 8079, 8142, 8143, 8080, 8017, 7954, 7891, 7828, 7765, 7702, 7639, 7576, 7513, 7450, 7387, 7324, 7261, 7198, 7135, 7072, 7009, 6946, 6883, 6820, 6757, 6694, 6631, 6568, 6505, 6442, 6379, 6316, 6253, 6190, 6127, 6064, 6001, 5938, 5875, 5812, 5749, 5686, 5623, 5560, 5497, 5434, 5371, 5308, 5245, 5182, 5119, 5183, 5246, 5309, 5372, 5435, 5498, 5561, 5624, 5687, 5750, 5813, 5876, 5939, 6002, 6065, 6128, 6191, 6254, 6317, 6380, 6443, 6506, 6569, 6632,
    6695, 6758, 6821, 6884, 6947, 7010, 7073, 7136, 7199, 7262, 7325, 7388, 7451, 7514, 7577, 7640, 7703, 7766, 7829, 7892, 7955, 8018, 8081, 8144, 8145, 8082, 8019, 7956, 7893, 7830, 7767, 7704, 7641, 7578, 7515, 7452, 7389, 7326, 7263, 7200, 7137, 7074, 7011, 6948, 6885, 6822, 6759, 6696, 6633, 6570, 6507, 6444, 6381, 6318, 6255, 6192, 6129, 6066, 6003, 5940, 5877, 5814, 5751, 5688, 5625, 5562, 5499, 5436, 5373, 5310, 5247, 5311, 5374, 5437, 5500, 5563, 5626, 5689, 5752, 5815, 5878, 5941, 6004, 6067, 6130, 6193, 6256, 6319, 6382, 6445, 6508, 6571, 6634, 6697, 6760, 6823, 6886, 6949, 7012, 7075, 7138, 7201, 7264, 7327, 7390, 7453, 7516, 7579, 7642, 7705, 7768, 7831, 7894, 7957, 8020, 8083, 8146, 8147, 8084, 8021, 7958, 7895, 7832, 7769, 7706, 7643, 7580, 7517,
    7454, 7391, 7328, 7265, 7202, 7139, 7076, 7013, 6950, 6887, 6824, 6761, 6698, 6635, 6572, 6509, 6446, 6383, 6320, 6257, 6194, 6131, 6068, 6005, 5942, 5879, 5816, 5753, 5690, 5627, 5564, 5501, 5438, 5375, 5439, 5502, 5565, 5628, 5691, 5754, 5817, 5880, 5943, 6006, 6069, 6132, 6195, 6258, 6321, 6384, 6447, 6510, 6573, 6636, 6699, 6762, 6825, 6888, 6951, 7014, 7077, 7140, 7203, 7266, 7329, 7392, 7455, 7518, 7581, 7644, 7707, 7770, 7833, 7896, 7959, 8022, 8085, 8148, 8149, 8086, 8023, 7960, 7897, 7834, 7771, 7708, 7645, 7582, 7519, 7456, 7393, 7330, 7267, 7204, 7141, 7078, 7015, 6952, 6889, 6826, 6763, 6700, 6637, 6574, 6511, 6448, 6385, 6322, 6259, 6196, 6133, 6070, 6007, 5944, 5881, 5818, 5755, 5692, 5629, 5566, 5503, 5567, 5630, 5693, 5756, 5819, 5882, 5945,
    6008, 6071, 6134, 6197, 6260, 6323, 6386, 6449, 6512, 6575, 6638, 6701, 6764, 6827, 6890, 6953, 7016, 7079, 7142, 7205, 7268, 7331, 7394, 7457, 7520, 7583, 7646, 7709, 7772, 7835, 7898, 7961, 8024, 8087, 8150, 8151, 8088, 8025, 7962, 7899, 7836, 7773, 7710, 7647, 7584, 7521, 7458, 7395, 7332, 7269, 7206, 7143, 7080, 7017, 6954, 6891, 6828, 6765, 6702, 6639, 6576, 6513, 6450, 6387, 6324, 6261, 6198, 6135, 6072, 6009, 5946, 5883, 5820, 5757, 5694, 5631, 5695, 5758, 5821, 5884, 5947, 6010, 6073, 6136, 6199, 6262, 6325, 6388, 6451, 6514, 6577, 6640, 6703, 6766, 6829, 6892, 6955, 7018, 7081, 7144, 7207, 7270, 7333, 7396, 7459, 7522, 7585, 7648, 7711, 7774, 7837, 7900, 7963, 8026, 8089, 8152, 8153, 8090, 8027, 7964, 7901, 7838, 7775, 7712, 7649, 7586, 7523, 7460,
    7397, 7334, 7271, 7208, 7145, 7082, 7019, 6956, 6893, 6830, 6767, 6704, 6641, 6578, 6515, 6452, 6389, 6326, 6263, 6200, 6137, 6074, 6011, 5948, 5885, 5822, 5759, 5823, 5886, 5949, 6012, 6075, 6138, 6201, 6264, 6327, 6390, 6453, 6516, 6579, 6642, 6705, 6768, 6831, 6894, 6957, 7020, 7083, 7146, 7209, 7272, 7335, 7398, 7461, 7524, 7587, 7650, 7713, 7776, 7839, 7902, 7965, 8028, 8091, 8154, 8155, 8092, 8029, 7966, 7903, 7840, 7777, 7714, 7651, 7588, 7525, 7462, 7399, 7336, 7273, 7210, 7147, 7084, 7021, 6958, 6895, 6832, 6769, 6706, 6643, 6580, 6517, 6454, 6391, 6328, 6265, 6202, 6139, 6076, 6013, 5950, 5887, 5951, 6014, 6077, 6140, 6203, 6266, 6329, 6392, 6455, 6518, 6581, 6644, 6707, 6770, 6833, 6896, 6959, 7022, 7085, 7148, 7211, 7274, 7337, 7400, 7463, 7526,
    7589, 7652, 7715, 7778, 7841, 7904, 7967, 8030, 8093, 8156, 8157, 8094, 8031, 7968, 7905, 7842, 7779, 7716, 7653, 7590, 7527, 7464, 7401, 7338, 7275, 7212, 7149, 7086, 7023, 6960, 6897, 6834, 6771, 6708, 6645, 6582, 6519, 6456, 6393, 6330, 6267, 6204, 6141, 6078, 6015, 6079, 6142, 6205, 6268, 6331, 6394, 6457, 6520, 6583, 6646, 6709, 6772, 6835, 6898, 6961, 7024, 7087, 7150, 7213, 7276, 7339, 7402, 7465, 7528, 7591, 7654, 7717, 7780, 7843, 7906, 7969, 8032, 8095, 8158, 8159, 8096, 8033, 7970, 7907, 7844, 7781, 7718, 7655, 7592, 7529, 7466, 7403, 7340, 7277, 7214, 7151, 7088, 7025, 6962, 6899, 6836, 6773, 6710, 6647, 6584, 6521, 6458, 6395, 6332, 6269, 6206, 6143, 6207, 6270, 6333, 6396, 6459, 6522, 6585, 6648, 6711, 6774, 6837, 6900, 6963, 7026, 7089, 7152,
    7215, 7278, 7341, 7404, 7467, 7530, 7593, 7656, 7719, 7782, 7845, 7908, 7971, 8034, 8097, 8160, 8161, 8098, 8035, 7972, 7909, 7846, 7783, 7720, 7657, 7594, 7531, 7468, 7405, 7342, 7279, 7216, 7153, 7090, 7027, 6964, 6901, 6838, 6775, 6712, 6649, 6586, 6523, 6460, 6397, 6334, 6271, 6335, 6398, 6461, 6524, 6587, 6650, 6713, 6776, 6839, 6902, 6965, 7028, 7091, 7154, 7217, 7280, 7343, 7406, 7469, 7532, 7595, 7658, 7721, 7784, 7847, 7910, 7973, 8036, 8099, 8162, 8163, 8100, 8037, 7974, 7911, 7848, 7785, 7722, 7659, 7596, 7533, 7470, 7407, 7344, 7281, 7218, 7155, 7092, 7029, 6966, 6903, 6840, 6777, 6714, 6651, 6588, 6525, 6462, 6399, 6463, 6526, 6589, 6652, 6715, 6778, 6841, 6904, 6967, 7030, 7093, 7156, 7219, 7282, 7345, 7408, 7471, 7534, 7597, 7660, 7723, 7786,
    7849, 7912, 7975, 8038, 8101, 8164, 8165, 8102, 8039, 7976, 7913, 7850, 7787, 7724, 7661, 7598, 7535, 7472, 7409, 7346, 7283, 7220, 7157, 7094, 7031, 6968, 6905, 6842, 6779, 6716, 6653, 6590, 6527, 6591, 6654, 6717, 6780, 6843, 6906, 6969, 7032, 7095, 7158, 7221, 7284, 7347, 7410, 7473, 7536, 7599, 7662, 7725, 7788, 7851, 7914, 7977, 8040, 8103, 8166, 8167, 8104, 8041, 7978, 7915, 7852, 7789, 7726, 7663, 7600, 7537, 7474, 7411, 7348, 7285, 7222, 7159, 7096, 7033, 6970, 6907, 6844, 6781, 6718, 6655, 6719, 6782, 6845, 6908, 6971, 7034, 7097, 7160, 7223, 7286, 7349, 7412, 7475, 7538, 7601, 7664, 7727, 7790, 7853, 7916, 7979, 8042, 8105, 8168, 8169, 8106, 8043, 7980, 7917, 7854, 7791, 7728, 7665, 7602, 7539, 7476, 7413, 7350, 7287, 7224, 7161, 7098, 7035, 6972,
    6909, 6846, 6783, 6847, 6910, 6973, 7036, 7099, 7162, 7225, 7288, 7351, 7414, 7477, 7540, 7603, 7666, 7729, 7792, 7855, 7918, 7981, 8044, 8107, 8170, 8171, 8108, 8045, 7982, 7919, 7856, 7793, 7730, 7667, 7604, 7541, 7478, 7415, 7352, 7289, 7226, 7163, 7100, 7037, 6974, 6911, 6975, 7038, 7101, 7164, 7227, 7290, 7353, 7416, 7479, 7542, 7605, 7668, 7731, 7794, 7857, 7920, 7983, 8046, 8109, 8172, 8173, 8110, 8047, 7984, 7921, 7858, 7795, 7732, 7669, 7606, 7543, 7480, 7417, 7354, 7291, 7228, 7165, 7102, 7039, 7103, 7166, 7229, 7292, 7355, 7418, 7481, 7544, 7607, 7670, 7733, 7796, 7859, 7922, 7985, 8048, 8111, 8174, 8175, 8112, 8049, 7986, 7923, 7860, 7797, 7734, 7671, 7608, 7545, 7482, 7419, 7356, 7293, 7230, 7167, 7231, 7294, 7357, 7420, 7483, 7546, 7609, 7672,
    7735, 7798, 7861, 7924, 7987, 8050, 8113, 8176, 8177, 8114, 8051, 7988, 7925, 7862, 7799, 7736, 7673, 7610, 7547, 7484, 7421, 7358, 7295, 7359, 7422, 7485, 7548, 7611, 7674, 7737, 7800, 7863, 7926, 7989, 8052, 8115, 8178, 8179, 8116, 8053, 7990, 7927, 7864, 7801, 7738, 7675, 7612, 7549, 7486, 7423, 7487, 7550, 7613, 7676, 7739, 7802, 7865, 7928, 7991, 8054, 8117, 8180, 8181, 8118, 8055, 7992, 7929, 7866, 7803, 7740, 7677, 7614, 7551, 7615, 7678, 7741, 7804, 7867, 7930, 7993, 8056, 8119, 8182, 8183, 8120, 8057, 7994, 7931, 7868, 7805, 7742, 7679, 7743, 7806, 7869, 7932, 7995, 8058, 8121, 8184, 8185, 8122, 8059, 7996, 7933, 7870, 7807, 7871, 7934, 7997, 8060, 8123, 8186, 8187, 8124, 8061, 7998, 7935, 7999, 8062, 8125, 8188, 8189, 8126, 8063, 8127, 8190, 8191,

    // 128 x 128:
    0, 1, 128, 256, 129, 2, 3, 130, 257, 384, 512, 385, 258, 131, 4, 5, 132, 259, 386, 513, 640, 768, 641, 514, 387, 260, 133, 6, 7, 134, 261, 388, 515, 642, 769, 896, 1024, 897, 770, 643, 516, 389, 262, 135, 8, 9, 136, 263, 390, 517, 644, 771, 898, 1025, 1152, 1280, 1153, 1026, 899, 772, 645, 518, 391, 264, 137, 10, 11, 138, 265, 392, 519, 646, 773, 900, 1027, 1154, 1281, 1408, 1536, 1409, 1282, 1155, 1028, 901, 774, 647, 520, 393, 266, 139, 12, 13, 140, 267, 394, 521, 648, 775, 902, 1029, 1156, 1283, 1410, 1537, 1664, 1792, 1665, 1538, 1411, 1284, 1157, 1030, 903, 776, 649, 522, 395, 268, 141, 14, 15, 142, 269, 396, 523, 650, 777, 904,
    1031, 1158, 1285, 1412, 1539, 1666, 1793, 1920, 2048, 1921, 1794, 1667, 1540, 1413, 1286, 1159, 1032, 905, 778, 651, 524, 397, 270, 143, 16, 17, 144, 271, 398, 525, 652, 779, 906, 1033, 1160, 1287, 1414, 1541, 1668, 1795, 1922, 2049, 2176, 2304, 2177, 2050, 1923, 1796, 1669, 1542, 1415, 1288, 1161, 1034, 907, 780, 653, 526, 399, 272, 145, 18, 19, 146, 273, 400, 527, 654, 781, 908, 1035, 1162, 1289, 1416, 1543, 1670, 1797, 1924, 2051, 2178, 2305, 2432, 2560, 2433, 2306, 2179, 2052, 1925, 1798, 1671, 1544, 1417, 1290, 1163, 1036, 909, 782, 655, 528, 401, 274, 147, 20, 21, 148, 275, 402, 529, 656, 783, 910, 1037, 1164, 1291, 1418, 1545, 1672, 1799, 1926, 2053, 2180, 2307, 2434, 2561, 2688, 2816, 2689, 2562,
    2435, 2308, 2181, 2054, 1927, 1800, 1673, 1546, 1419, 1292, 1165, 1038, 911, 784, 657, 530, 403, 276, 149, 22, 23, 150, 277, 404, 531, 658, 785, 912, 1039, 1166, 1293, 1420, 1547, 1674, 1801, 1928, 2055, 2182, 2309, 2436, 2563, 2690, 2817, 2944, 3072, 2945, 2818, 2691, 2564, 2437, 2310, 2183, 2056, 1929, 1802, 1675, 1548, 1421, 1294, 1167, 1040, 913, 786, 659, 532, 405, 278, 151, 24, 25, 152, 279, 406, 533, 660, 787, 914, 1041, 1168, 1295, 1422, 1549, 1676, 1803, 1930, 2057, 2184, 2311, 2438, 2565, 2692, 2819, 2946, 3073, 3200, 3328, 3201, 3074, 2947, 2820, 2693, 2566, 2439, 2312, 2185, 2058, 1931, 1804, 1677, 1550, 1423, 1296, 1169, 1042, 915, 788, 661, 534, 407, 280, 153, 26, 27, 154, 281, 408, 535, 662,
    789, 916, 1043, 1170, 1297, 1424, 1551, 1678, 1805, 1932, 2059, 2186, 2313, 2440, 2567, 2694, 2821, 2948, 3075, 3202, 3329, 3456, 3584, 3457, 3330, 3203, 3076, 2949, 2822, 2695, 2568, 2441, 2314, 2187, 2060, 1933, 1806, 1679, 1552, 1425, 1298, 1171, 1044, 917, 790, 663, 536, 409, 282, 155, 28, 29, 156, 283, 410, 537, 664, 791, 918, 1045, 1172, 1299, 1426, 1553, 1680, 1807, 1934, 2061, 2188, 2315, 2442, 2569, 2696, 2823, 2950, 3077, 3204, 3331, 3458, 3585, 3712, 3840, 3713, 3586, 3459, 3332, 3205, 3078, 2951, 2824, 2697, 2570, 2443, 2316, 2189, 2062, 1935, 1808, 1681, 1554, 1427, 1300, 1173, 1046, 919, 792, 665, 538, 411, 284, 157, 30, 31, 158, 285, 412, 539, 666, 793, 920, 1047, 1174, 1301, 1428, 1555, 1682, 1809, 1936,
    2063, 2190, 2317, 2444, 2571, 2698, 2825, 2952, 3079, 3206, 3333, 3460, 3587, 3714, 3841, 3968, 4096, 3969, 3842, 3715, 3588, 3461, 3334, 3207, 3080, 2953, 2826, 2699, 2572, 2445, 2318, 2191, 2064, 1937, 1810, 1683, 1556, 1429, 1302, 1175, 1048, 921, 794, 667, 540, 413, 286, 159, 32, 33, 160, 287, 414, 541, 668, 795, 922, 1049, 1176, 1303, 1430, 1557, 1684, 1811, 1938, 2065, 2192, 2319, 2446, 2573, 2700, 2827, 2954, 3081, 3208, 3335, 3462, 3589, 3716, 3843, 3970, 4097, 4224, 4352, 4225, 4098, 3971, 3844, 3717, 3590, 3463, 3336, 3209, 3082, 2955, 2828, 2701, 2574, 2447, 2320, 2193, 2066, 1939, 1812, 1685, 1558, 1431, 1304, 1177, 1050, 923, 796, 669, 542, 415, 288, 161, 34, 35, 162, 289, 416, 543, 670, 797, 924, 1051, 1178,
    1305, 1432, 1559, 1686, 1813, 1940, 2067, 2194, 2321, 2448, 2575, 2702, 2829, 2956, 3083, 3210, 3337, 3464, 3591, 3718, 3845, 3972, 4099, 4226, 4353, 4480, 4608, 4481, 4354, 4227, 4100, 3973, 3846, 3719, 3592, 3465, 3338, 3211, 3084, 2957, 2830, 2703, 2576, 2449, 2322, 2195, 2068, 1941, 1814, 1687, 1560, 1433, 1306, 1179, 1052, 925, 798, 671, 544, 417, 290, 163, 36, 37, 164, 291, 418, 545, 672, 799, 926, 1053, 1180, 1307, 1434, 1561, 1688, 1815, 1942, 2069, 2196, 2323, 2450, 2577, 2704, 2831, 2958, 3085, 3212, 3339, 3466, 3593, 3720, 3847, 3974, 4101, 4228, 4355, 4482, 4609, 4736, 4864, 4737, 4610, 4483, 4356, 4229, 4102, 3975, 3848, 3721, 3594, 3467, 3340, 3213, 3086, 2959, 2832, 2705, 2578, 2451, 2324, 2197, 2070, 1943, 1816, 1689, 1562,
    1435, 1308, 1181, 1054, 927, 800, 673, 546, 419, 292, 165, 38, 39, 166, 293, 420, 547, 674, 801, 928, 1055, 1182, 1309, 1436, 1563, 1690, 1817, 1944, 2071, 2198, 2325, 2452, 2579, 2706, 2833, 2960, 3087, 3214, 3341, 3468, 3595, 3722, 3849, 3976, 4103, 4230, 4357, 4484, 4611, 4738, 4865, 4992, 5120, 4993, 4866, 4739, 4612, 4485, 4358, 4231, 4104, 3977, 3850, 3723, 3596, 3469, 3342, 3215, 3088, 2961, 2834, 2707, 2580, 2453, 2326, 2199, 2072, 1945, 1818, 1691, 1564, 1437, 1310, 1183, 1056, 929, 802, 675, 548, 421, 294, 167, 40, 41, 168, 295, 422, 549, 676, 803, 930, 1057, 1184, 1311, 1438, 1565, 1692, 1819, 1946, 2073, 2200, 2327, 2454, 2581, 2708, 2835, 2962, 3089, 3216, 3343, 3470, 3597, 3724, 3851, 3978, 4105, 4232, 4359,
    4486, 4613, 4740, 4867, 4994, 5121, 5248, 5376, 5249, 5122, 4995, 4868, 4741, 4614, 4487, 4360, 4233, 4106, 3979, 3852, 3725, 3598, 3471, 3344, 3217, 3090, 2963, 2836, 2709, 2582, 2455, 2328, 2201, 2074, 1947, 1820, 1693, 1566, 1439, 1312, 1185, 1058, 931, 804, 677, 550, 423, 296, 169, 42, 43, 170, 297, 424, 551, 678, 805, 932, 1059, 1186, 1313, 1440, 1567, 1694, 1821, 1948, 2075, 2202, 2329, 2456, 2583, 2710, 2837, 2964, 3091, 3218, 3345, 3472, 3599, 3726, 3853, 3980, 4107, 4234, 4361, 4488, 4615, 4742, 4869, 4996, 5123, 5250, 5377, 5504, 5632, 5505, 5378, 5251, 5124, 4997, 4870, 4743, 4616, 4489, 4362, 4235, 4108, 3981, 3854, 3727, 3600, 3473, 3346, 3219, 3092, 2965, 2838, 2711, 2584, 2457, 2330, 2203, 2076, 1949, 1822, 1695, 1568, 1441,
    1314, 1187, 1060, 933, 806, 679, 552, 425, 298, 171, 44, 45, 172, 299, 426, 553, 680, 807, 934, 1061, 1188, 1315, 1442, 1569, 1696, 1823, 1950, 2077, 2204, 2331, 2458, 2585, 2712, 2839, 2966, 3093, 3220, 3347, 3474, 3601, 3728, 3855, 3982, 4109, 4236, 4363, 4490, 4617, 4744, 4871, 4998, 5125, 5252, 5379, 5506, 5633, 5760, 5888, 5761, 5634, 5507, 5380, 5253, 5126, 4999, 4872, 4745, 4618, 4491, 4364, 4237, 4110, 3983, 3856, 3729, 3602, 3475, 3348, 3221, 3094, 2967, 2840, 2713, 2586, 2459, 2332, 2205, 2078, 1951, 1824, 1697, 1570, 1443, 1316, 1189, 1062, 935, 808, 681, 554, 427, 300, 173, 46, 47, 174, 301, 428, 555, 682, 809, 936, 1063, 1190, 1317, 1444, 1571, 1698, 1825, 1952, 2079, 2206, 2333, 2460, 2587, 2714, 2841, 2968,
    3095, 3222, 3349, 3476, 3603, 3730, 3857, 3984, 4111, 4238, 4365, 4492, 4619, 4746, 4873, 5000, 5127, 5254, 5381, 5508, 5635, 5762, 5889, 6016, 6144, 6017, 5890, 5763, 5636, 5509, 5382, 5255, 5128, 5001, 4874, 4747, 4620, 4493, 4366, 4239, 4112, 3985, 3858, 3731, 3604, 3477, 3350, 3223, 3096, 2969, 2842, 2715, 2588, 2461, 2334, 2207, 2080, 1953, 1826, 1699, 1572, 1445, 1318, 1191, 1064, 937, 810, 683, 556, 429, 302, 175, 48, 49, 176, 303, 430, 557, 684, 811, 938, 1065, 1192, 1319, 1446, 1573, 1700, 1827, 1954, 2081, 2208, 2335, 2462, 2589, 2716, 2843, 2970, 3097, 3224, 3351, 3478, 3605, 3732, 3859, 3986, 4113, 4240, 4367, 4494, 4621, 4748, 4875, 5002, 5129, 5256, 5383, 5510, 5637, 5764, 5891, 6018, 6145, 6272, 6400, 6273, 6146, 6019, 5892,
    5765, 5638, 5511, 5384, 5257, 5130, 5003, 4876, 4749, 4622, 4495, 4368, 4241, 4114, 3987, 3860, 3733, 3606, 3479, 3352, 3225, 3098, 2971, 2844, 2717, 2590, 2463, 2336, 2209, 2082, 1955, 1828, 1701, 1574, 1447, 1320, 1193, 1066, 939, 812, 685, 558, 431, 304, 177, 50, 51, 178, 305, 432, 559, 686, 813, 940, 1067, 1194, 1321, 1448, 1575, 1702, 1829, 1956, 2083, 2210, 2337, 2464, 2591, 2718, 2845, 2972, 3099, 3226, 3353, 3480, 3607, 3734, 3861, 3988, 4115, 4242, 4369, 4496, 4623, 4750, 4877, 5004, 5131, 5258, 5385, 5512, 5639, 5766, 5893, 6020, 6147, 6274, 6401, 6528, 6656, 6529, 6402, 6275, 6148, 6021, 5894, 5767, 5640, 5513, 5386, 5259, 5132, 5005, 4878, 4751, 4624, 4497, 4370, 4243, 4116, 3989, 3862, 3735, 3608, 3481, 3354, 3227, 3100, 2973,
    2846, 2719, 2592, 2465, 2338, 2211, 2084, 1957, 1830, 1703, 1576, 1449, 1322, 1195, 1068, 941, 814, 687, 560, 433, 306, 179, 52, 53, 180, 307, 434, 561, 688, 815, 942, 1069, 1196, 1323, 1450, 1577, 1704, 1831, 1958, 2085, 2212, 2339, 2466, 2593, 2720, 2847, 2974, 3101, 3228, 3355, 3482, 3609, 3736, 3863, 3990, 4117, 4244, 4371, 4498, 4625, 4752, 4879, 5006, 5133, 5260, 5387, 5514, 5641, 5768, 5895, 6022, 6149, 6276, 6403, 6530, 6657, 6784, 6912, 6785, 6658, 6531, 6404, 6277, 6150, 6023, 5896, 5769, 5642, 5515, 5388, 5261, 5134, 5007, 4880, 4753, 4626, 4499, 4372, 4245, 4118, 3991, 3864, 3737, 3610, 3483, 3356, 3229, 3102, 2975, 2848, 2721, 2594, 2467, 2340, 2213, 2086, 1959, 1832, 1705, 1578, 1451, 1324, 1197, 1070, 943, 816, 689, 562,
    435, 308, 181, 54, 55, 182, 309, 436, 563, 690, 817, 944, 1071, 1198, 1325, 1452, 1579, 1706, 1833, 1960, 2087, 2214, 2341, 2468, 2595, 2722, 2849, 2976, 3103, 3230, 3357, 3484, 3611, 3738, 3865, 3992, 4119, 4246, 4373, 4500, 4627, 4754, 4881, 5008, 5135, 5262, 5389, 5516, 5643, 5770, 5897, 6024, 6151, 6278, 6405, 6532, 6659, 6786, 6913, 7040, 7168, 7041, 6914, 6787, 6660, 6533, 6406, 6279, 6152, 6025, 5898, 5771, 5644, 5517, 5390, 5263, 5136, 5009, 4882, 4755, 4628, 4501, 4374, 4247, 4120, 3993, 3866, 3739, 3612, 3485, 3358, 3231, 3104, 2977, 2850, 2723, 2596, 2469, 2342, 2215, 2088, 1961, 1834, 1707, 1580, 1453, 1326, 1199, 1072, 945, 818, 691, 564, 437, 310, 183, 56, 57, 184, 311, 438, 565, 692, 819, 946, 1073, 1200, 1327,
    1454, 1581, 1708, 1835, 1962, 2089, 2216, 2343, 2470, 2597, 2724, 2851, 2978, 3105, 3232, 3359, 3486, 3613, 3740, 3867, 3994, 4121, 4248, 4375, 4502, 4629, 4756, 4883, 5010, 5137, 5264, 5391, 5518, 5645, 5772, 5899, 6026, 6153, 6280, 6407, 6534, 6661, 6788, 6915, 7042, 7169, 7296, 7424, 7297, 7170, 7043, 6916, 6789, 6662, 6535, 6408, 6281, 6154, 6027, 5900, 5773, 5646, 5519, 5392, 5265, 5138, 5011, 4884, 4757, 4630, 4503, 4376, 4249, 4122, 3995, 3868, 3741, 3614, 3487, 3360, 3233, 3106, 2979, 2852, 2725, 2598, 2471, 2344, 2217, 2090, 1963, 1836, 1709, 1582, 1455, 1328, 1201, 1074, 947, 820, 693, 566, 439, 312, 185, 58, 59, 186, 313, 440, 567, 694, 821, 948, 1075, 1202, 1329, 1456, 1583, 1710, 1837, 1964, 2091, 2218, 2345, 2472, 2599, 2726,
    2853, 2980, 3107, 3234, 3361, 3488, 3615, 3742, 3869, 3996, 4123, 4250, 4377, 4504, 4631, 4758, 4885, 5012, 5139, 5266, 5393, 5520, 5647, 5774, 5901, 6028, 6155, 6282, 6409, 6536, 6663, 6790, 6917, 7044, 7171, 7298, 7425, 7552, 7680, 7553, 7426, 7299, 7172, 7045, 6918, 6791, 6664, 6537, 6410, 6283, 6156, 6029, 5902, 5775, 5648, 5521, 5394, 5267, 5140, 5013, 4886, 4759, 4632, 4505, 4378, 4251, 4124, 3997, 3870, 3743, 3616, 3489, 3362, 3235, 3108, 2981, 2854, 2727, 2600, 2473, 2346, 2219, 2092, 1965, 1838, 1711, 1584, 1457, 1330, 1203, 1076, 949, 822, 695, 568, 441, 314, 187, 60, 61, 188, 315, 442, 569, 696, 823, 950, 1077, 1204, 1331, 1458, 1585, 1712, 1839, 1966, 2093, 2220, 2347, 2474, 2601, 2728, 2855, 2982, 3109, 3236, 3363, 3490, 3617,
    3744, 3871, 3998, 4125, 4252, 4379, 4506, 4633, 4760, 4887, 5014, 5141, 5268, 5395, 5522, 5649, 5776, 5903, 6030, 6157, 6284, 6411, 6538, 6665, 6792, 6919, 7046, 7173, 7300, 7427, 7554, 7681, 7808, 7936, 7809, 7682, 7555, 7428, 7301, 7174, 7047, 6920, 6793, 6666, 6539, 6412, 6285, 6158, 6031, 5904, 5777, 5650, 5523, 5396, 5269, 5142, 5015, 4888, 4761, 4634, 4507, 4380, 4253, 4126, 3999, 3872, 3745, 3618, 3491, 3364, 3237, 3110, 2983, 2856, 2729, 2602, 2475, 2348, 2221, 2094, 1967, 1840, 1713, 1586, 1459, 1332, 1205, 1078, 951, 824, 697, 570, 443, 316, 189, 62, 63, 190, 317, 444, 571, 698, 825, 952, 1079, 1206, 1333, 1460, 1587, 1714, 1841, 1968, 2095, 2222, 2349, 2476, 2603, 2730, 2857, 2984, 3111, 3238, 3365, 3492, 3619, 3746, 3873, 4000,
    4127, 4254, 4381, 4508, 4635, 4762, 4889, 5016, 5143, 5270, 5397, 5524, 5651, 5778, 5905, 6032, 6159, 6286, 6413, 6540, 6667, 6794, 6921, 7048, 7175, 7302, 7429, 7556, 7683, 7810, 7937, 8064, 8192, 8065, 7938, 7811, 7684, 7557, 7430, 7303, 7176, 7049, 6922, 6795, 6668, 6541, 6414, 6287, 6160, 6033, 5906, 5779, 5652, 5525, 5398, 5271, 5144, 5017, 4890, 4763, 4636, 4509, 4382, 4255, 4128, 4001, 3874, 3747, 3620, 3493, 3366, 3239, 3112, 2985, 2858, 2731, 2604, 2477, 2350, 2223, 2096, 1969, 1842, 1715, 1588, 1461, 1334, 1207, 1080, 953, 826, 699, 572, 445, 318, 191, 64, 65, 192, 319, 446, 573, 700, 827, 954, 1081, 1208, 1335, 1462, 1589, 1716, 1843, 1970, 2097, 2224, 2351, 2478, 2605, 2732, 2859, 2986, 3113, 3240, 3367, 3494, 3621, 3748, 3875,
    4002, 4129, 4256, 4383, 4510, 4637, 4764, 4891, 5018, 5145, 5272, 5399, 5526, 5653, 5780, 5907, 6034, 6161, 6288, 6415, 6542, 6669, 6796, 6923, 7050, 7177, 7304, 7431, 7558, 7685, 7812, 7939, 8066, 8193, 8320, 8448, 8321, 8194, 8067, 7940, 7813, 7686, 7559, 7432, 7305, 7178, 7051, 6924, 6797, 6670, 6543, 6416, 6289, 6162, 6035, 5908, 5781, 5654, 5527, 5400, 5273, 5146, 5019, 4892, 4765, 4638, 4511, 4384, 4257, 4130, 4003, 3876, 3749, 3622, 3495, 3368, 3241, 3114, 2987, 2860, 2733, 2606, 2479, 2352, 2225, 2098, 1971, 1844, 1717, 1590, 1463, 1336, 1209, 1082, 955, 828, 701, 574, 447, 320, 193, 66, 67, 194, 321, 448, 575, 702, 829, 956, 1083, 1210, 1337, 1464, 1591, 1718, 1845, 1972, 2099, 2226, 2353, 2480, 2607, 2734, 2861, 2988, 3115, 3242,
    3369, 3496, 3623, 3750, 3877, 4004, 4131, 4258, 4385, 4512, 4639, 4766, 4893, 5020, 5147, 5274, 5401, 5528, 5655, 5782, 5909, 6036, 6163, 6290, 6417, 6544, 6671, 6798, 6925, 7052, 7179, 7306, 7433, 7560, 7687, 7814, 7941, 8068, 8195, 8322, 8449, 8576, 8704, 8577, 8450, 8323, 8196, 8069, 7942, 7815, 7688, 7561, 7434, 7307, 7180, 7053, 6926, 6799, 6672, 6545, 6418, 6291, 6164, 6037, 5910, 5783, 5656, 5529, 5402, 5275, 5148, 5021, 4894, 4767, 4640, 4513, 4386, 4259, 4132, 4005, 3878, 3751, 3624, 3497, 3370, 3243, 3116, 2989, 2862, 2735, 2608, 2481, 2354, 2227, 2100, 1973, 1846, 1719, 1592, 1465, 1338, 1211, 1084, 957, 830, 703, 576, 449, 322, 195, 68, 69, 196, 323, 450, 577, 704, 831, 958, 1085, 1212, 1339, 1466, 1593, 1720, 1847, 1974, 2101,
    2228, 2355, 2482, 2609, 2736, 2863, 2990, 3117, 3244, 3371, 3498, 3625, 3752, 3879, 4006, 4133, 4260, 4387, 4514, 4641, 4768, 4895, 5022, 5149, 5276, 5403, 5530, 5657, 5784, 5911, 6038, 6165, 6292, 6419, 6546, 6673, 6800, 6927, 7054, 7181, 7308, 7435, 7562, 7689, 7816, 7943, 8070, 8197, 8324, 8451, 8578, 8705, 8832, 8960, 8833, 8706, 8579, 8452, 8325, 8198, 8071, 7944, 7817, 7690, 7563, 7436, 7309, 7182, 7055, 6928, 6801, 6674, 6547, 6420, 6293, 6166, 6039, 5912, 5785, 5658, 5531, 5404, 5277, 5150, 5023, 4896, 4769, 4642, 4515, 4388, 4261, 4134, 4007, 3880, 3753, 3626, 3499, 3372, 3245, 3118, 2991, 2864, 2737, 2610, 2483, 2356, 2229, 2102, 1975, 1848, 1721, 1594, 1467, 1340, 1213, 1086, 959, 832, 705, 578, 451, 324, 197, 70, 71, 198, 325, 452,
    579, 706, 833, 960, 1087, 1214, 1341, 1468, 1595, 1722, 1849, 1976, 2103, 2230, 2357, 2484, 2611, 2738, 2865, 2992, 3119, 3246, 3373, 3500, 3627, 3754, 3881, 4008, 4135, 4262, 4389, 4516, 4643, 4770, 4897, 5024, 5151, 5278, 5405, 5532, 5659, 5786, 5913, 6040, 6167, 6294, 6421, 6548, 6675, 6802, 6929, 7056, 7183, 7310, 7437, 7564, 7691, 7818, 7945, 8072, 8199, 8326, 8453, 8580, 8707, 8834, 8961, 9088, 9216, 9089, 8962, 8835, 8708, 8581, 8454, 8327, 8200, 8073, 7946, 7819, 7692, 7565, 7438, 7311, 7184, 7057, 6930, 6803, 6676, 6549, 6422, 6295, 6168, 6041, 5914, 5787, 5660, 5533, 5406, 5279, 5152, 5025, 4898, 4771, 4644, 4517, 4390, 4263, 4136, 4009, 3882, 3755, 3628, 3501, 3374, 3247, 3120, 2993, 2866, 2739, 2612, 2485, 2358, 2231, 2104, 1977, 1850, 1723,
    1596, 1469, 1342, 1215, 1088, 961, 834, 707, 580, 453, 326, 199, 72, 73, 200, 327, 454, 581, 708, 835, 962, 1089, 1216, 1343, 1470, 1597, 1724, 1851, 1978, 2105, 2232, 2359, 2486, 2613, 2740, 2867, 2994, 3121, 3248, 3375, 3502, 3629, 3756, 3883, 4010, 4137, 4264, 4391, 4518, 4645, 4772, 4899, 5026, 5153, 5280, 5407, 5534, 5661, 5788, 5915, 6042, 6169, 6296, 6423, 6550, 6677, 6804, 6931, 7058, 7185, 7312, 7439, 7566, 7693, 7820, 7947, 8074, 8201, 8328, 8455, 8582, 8709, 8836, 8963, 9090, 9217, 9344, 9472, 9345, 9218, 9091, 8964, 8837, 8710, 8583, 8456, 8329, 8202, 8075, 7948, 7821, 7694, 7567, 7440, 7313, 7186, 7059, 6932, 6805, 6678, 6551, 6424, 6297, 6170, 6043, 5916, 5789, 5662, 5535, 5408, 5281, 5154, 5027, 4900, 4773, 4646, 4519, 4392,
    4265, 4138, 4011, 3884, 3757, 3630, 3503, 3376, 3249, 3122, 2995, 2868, 2741, 2614, 2487, 2360, 2233, 2106, 1979, 1852, 1725, 1598, 1471, 1344, 1217, 1090, 963, 836, 709, 582, 455, 328, 201, 74, 75, 202, 329, 456, 583, 710, 837, 964, 1091, 1218, 1345, 1472, 1599, 1726, 1853, 1980, 2107, 2234, 2361, 2488, 2615, 2742, 2869, 2996, 3123, 3250, 3377, 3504, 3631, 3758, 3885, 4012, 4139, 4266, 4393, 4520, 4647, 4774, 4901, 5028, 5155, 5282, 5409, 5536, 5663, 5790, 5917, 6044, 6171, 6298, 6425, 6552, 6679, 6806, 6933, 7060, 7187, 7314, 7441, 7568, 7695, 7822, 7949, 8076, 8203, 8330, 8457, 8584, 8711, 8838, 8965, 9092, 9219, 9346, 9473, 9600, 9728, 9601, 9474, 9347, 9220, 9093, 8966, 8839, 8712, 8585, 8458, 8331, 8204, 8077, 7950, 7823, 7696, 7569,
    7442, 7315, 7188, 7061, 6934, 6807, 6680, 6553, 6426, 6299, 6172, 6045, 5918, 5791, 5664, 5537, 5410, 5283, 5156, 5029, 4902, 4775, 4648, 4521, 4394, 4267, 4140, 4013, 3886, 3759, 3632, 3505, 3378, 3251, 3124, 2997, 2870, 2743, 2616, 2489, 2362, 2235, 2108, 1981, 1854, 1727, 1600, 1473, 1346, 1219, 1092, 965, 838, 711, 584, 457, 330, 203, 76, 77, 204, 331, 458, 585, 712, 839, 966, 1093, 1220, 1347, 1474, 1601, 1728, 1855, 1982, 2109, 2236, 2363, 2490, 2617, 2744, 2871, 2998, 3125, 3252, 3379, 3506, 3633, 3760, 3887, 4014, 4141, 4268, 4395, 4522, 4649, 4776, 4903, 5030, 5157, 5284, 5411, 5538, 5665, 5792, 5919, 6046, 6173, 6300, 6427, 6554, 6681, 6808, 6935, 7062, 7189, 7316, 7443, 7570, 7697, 7824, 7951, 8078, 8205, 8332, 8459, 8586, 8713,
    8840, 8967, 9094, 9221, 9348, 9475, 9602, 9729, 9856, 9984, 9857, 9730, 9603, 9476, 9349, 9222, 9095, 8968, 8841, 8714, 8587, 8460, 8333, 8206, 8079, 7952, 7825, 7698, 7571, 7444, 7317, 7190, 7063, 6936, 6809, 6682, 6555, 6428, 6301, 6174, 6047, 5920, 5793, 5666, 5539, 5412, 5285, 5158, 5031, 4904, 4777, 4650, 4523, 4396, 4269, 4142, 4015, 3888, 3761, 3634, 3507, 3380, 3253, 3126, 2999, 2872, 2745, 2618, 2491, 2364, 2237, 2110, 1983, 1856, 1729, 1602, 1475, 1348, 1221, 1094, 967, 840, 713, 586, 459, 332, 205, 78, 79, 206, 333, 460, 587, 714, 841, 968, 1095, 1222, 1349, 1476, 1603, 1730, 1857, 1984, 2111, 2238, 2365, 2492, 2619, 2746, 2873, 3000, 3127, 3254, 3381, 3508, 3635, 3762, 3889, 4016, 4143, 4270, 4397, 4524, 4651, 4778, 4905, 5032,
    5159, 5286, 5413, 5540, 5667, 5794, 5921, 6048, 6175, 6302, 6429, 6556, 6683, 6810, 6937, 7064, 7191, 7318, 7445, 7572, 7699, 7826, 7953, 8080, 8207, 8334, 8461, 8588, 8715, 8842, 8969, 9096, 9223, 9350, 9477, 9604, 9731, 9858, 9985, 10112, 10240, 10113, 9986, 9859, 9732, 9605, 9478, 9351, 9224, 9097, 8970, 8843, 8716, 8589, 8462, 8335, 8208, 8081, 7954, 7827, 7700, 7573, 7446, 7319, 7192, 7065, 6938, 6811, 6684, 6557, 6430, 6303, 6176, 6049, 5922, 5795, 5668, 5541, 5414, 5287, 5160, 5033, 4906, 4779, 4652, 4525, 4398, 4271, 4144, 4017, 3890, 3763, 3636, 3509, 3382, 3255, 3128, 3001, 2874, 2747, 2620, 2493, 2366, 2239, 2112, 1985, 1858, 1731, 1604, 1477, 1350, 1223, 1096, 969, 842, 715, 588, 461, 334, 207, 80, 81, 208, 335, 462, 589, 716, 843,
    970, 1097, 1224, 1351, 1478, 1605, 1732, 1859, 1986, 2113, 2240, 2367, 2494, 2621, 2748, 2875, 3002, 3129, 3256, 3383, 3510, 3637, 3764, 3891, 4018, 4145, 4272, 4399, 4526, 4653, 4780, 4907, 5034, 5161, 5288, 5415, 5542, 5669, 5796, 5923, 6050, 6177, 6304, 6431, 6558, 6685, 6812, 6939, 7066, 7193, 7320, 7447, 7574, 7701, 7828, 7955, 8082, 8209, 8336, 8463, 8590, 8717, 8844, 8971, 9098, 9225, 9352, 9479, 9606, 9733, 9860, 9987, 10114, 10241, 10368, 10496, 10369, 10242, 10115, 9988, 9861, 9734, 9607, 9480, 9353, 9226, 9099, 8972, 8845, 8718, 8591, 8464, 8337, 8210, 8083, 7956, 7829, 7702, 7575, 7448, 7321, 7194, 7067, 6940, 6813, 6686, 6559, 6432, 6305, 6178, 6051, 5924, 5797, 5670, 5543, 5416, 5289, 5162, 5035, 4908, 4781, 4654, 4527, 4400, 4273, 4146, 4019, 3892,
    3765, 3638, 3511, 3384, 3257, 3130, 3003, 2876, 2749, 2622, 2495, 2368, 2241, 2114, 1987, 1860, 1733, 1606, 1479, 1352, 1225, 1098, 971, 844, 717, 590, 463, 336, 209, 82, 83, 210, 337, 464, 591, 718, 845, 972, 1099, 1226, 1353, 1480, 1607, 1734, 1861, 1988, 2115, 2242, 2369, 2496, 2623, 2750, 2877, 3004, 3131, 3258, 3385, 3512, 3639, 3766, 3893, 4020, 4147, 4274, 4401, 4528, 4655, 4782, 4909, 5036, 5163, 5290, 5417, 5544, 5671, 5798, 5925, 6052, 6179, 6306, 6433, 6560, 6687, 6814, 6941, 7068, 7195, 7322, 7449, 7576, 7703, 7830, 7957, 8084, 8211, 8338, 8465, 8592, 8719, 8846, 8973, 9100, 9227, 9354, 9481, 9608, 9735, 9862, 9989, 10116, 10243, 10370, 10497, 10624, 10752, 10625, 10498, 10371, 10244, 10117, 9990, 9863, 9736, 9609, 9482, 9355, 9228, 9101,
    8974, 8847, 8720, 8593, 8466, 8339, 8212, 8085, 7958, 7831, 7704, 7577, 7450, 7323, 7196, 7069, 6942, 6815, 6688, 6561, 6434, 6307, 6180, 6053, 5926, 5799, 5672, 5545, 5418, 5291, 5164, 5037, 4910, 4783, 4656, 4529, 4402, 4275, 4148, 4021, 3894, 3767, 3640, 3513, 3386, 3259, 3132, 3005, 2878, 2751, 2624, 2497, 2370, 2243, 2116, 1989, 1862, 1735, 1608, 1481, 1354, 1227, 1100, 973, 846, 719, 592, 465, 338, 211, 84, 85, 212, 339, 466, 593, 720, 847, 974, 1101, 1228, 1355, 1482, 1609, 1736, 1863, 1990, 2117, 2244, 2371, 2498, 2625, 2752, 2879, 3006, 3133, 3260, 3387, 3514, 3641, 3768, 3895, 4022, 4149, 4276, 4403, 4530, 4657, 4784, 4911, 5038, 5165, 5292, 5419, 5546, 5673, 5800, 5927, 6054, 6181, 6308, 6435, 6562, 6689, 6816, 6943, 7070, 7197,
    7324, 7451, 7578, 7705, 7832, 7959, 8086, 8213, 8340, 8467, 8594, 8721, 8848, 8975, 9102, 9229, 9356, 9483, 9610, 9737, 9864, 9991, 10118, 10245, 10372, 10499, 10626, 10753, 10880, 11008, 10881, 10754, 10627, 10500, 10373, 10246, 10119, 9992, 9865, 9738, 9611, 9484, 9357, 9230, 9103, 8976, 8849, 8722, 8595, 8468, 8341, 8214, 8087, 7960, 7833, 7706, 7579, 7452, 7325, 7198, 7071, 6944, 6817, 6690, 6563, 6436, 6309, 6182, 6055, 5928, 5801, 5674, 5547, 5420, 5293, 5166, 5039, 4912, 4785, 4658, 4531, 4404, 4277, 4150, 4023, 3896, 3769, 3642, 3515, 3388, 3261, 3134, 3007, 2880, 2753, 2626, 2499, 2372, 2245, 2118, 1991, 1864, 1737, 1610, 1483, 1356, 1229, 1102, 975, 848, 721, 594, 467, 340, 213, 86, 87, 214, 341, 468, 595, 722, 849, 976, 1103, 1230, 1357, 1484,
    1611, 1738, 1865, 1992, 2119, 2246, 2373, 2500, 2627, 2754, 2881, 3008, 3135, 3262, 3389, 3516, 3643, 3770, 3897, 4024, 4151, 4278, 4405, 4532, 4659, 4786, 4913, 5040, 5167, 5294, 5421, 5548, 5675, 5802, 5929, 6056, 6183, 6310, 6437, 6564, 6691, 6818, 6945, 7072, 7199, 7326, 7453, 7580, 7707, 7834, 7961, 8088, 8215, 8342, 8469, 8596, 8723, 8850, 8977, 9104, 9231, 9358, 9485, 9612, 9739, 9866, 9993, 10120, 10247, 10374, 10501, 10628, 10755, 10882, 11009, 11136, 11264, 11137, 11010, 10883, 10756, 10629, 10502, 10375, 10248, 10121, 9994, 9867, 9740, 9613, 9486, 9359, 9232, 9105, 8978, 8851, 8724, 8597, 8470, 8343, 8216, 8089, 7962, 7835, 7708, 7581, 7454, 7327, 7200, 7073, 6946, 6819, 6692, 6565, 6438, 6311, 6184, 6057, 5930, 5803, 5676, 5549, 5422, 5295, 5168, 5041, 4914, 4787,
    4660, 4533, 4406, 4279, 4152, 4025, 3898, 3771, 3644, 3517, 3390, 3263, 3136, 3009, 2882, 2755, 2628, 2501, 2374, 2247, 2120, 1993, 1866, 1739, 1612, 1485, 1358, 1231, 1104, 977, 850, 723, 596, 469, 342, 215, 88, 89, 216, 343, 470, 597, 724, 851, 978, 1105, 1232, 1359, 1486, 1613, 1740, 1867, 1994, 2121, 2248, 2375, 2502, 2629, 2756, 2883, 3010, 3137, 3264, 3391, 3518, 3645, 3772, 3899, 4026, 4153, 4280, 4407, 4534, 4661, 4788, 4915, 5042, 5169, 5296, 5423, 5550, 5677, 5804, 5931, 6058, 6185, 6312, 6439, 6566, 6693, 6820, 6947, 7074, 7201, 7328, 7455, 7582, 7709, 7836, 7963, 8090, 8217, 8344, 8471, 8598, 8725, 8852, 8979, 9106, 9233, 9360, 9487, 9614, 9741, 9868, 9995, 10122, 10249, 10376, 10503, 10630, 10757, 10884, 11011, 11138, 11265, 11392, 11520,
    11393, 11266, 11139, 11012, 10885, 10758, 10631, 10504, 10377, 10250, 10123, 9996, 9869, 9742, 9615, 9488, 9361, 9234, 9107, 8980, 8853, 8726, 8599, 8472, 8345, 8218, 8091, 7964, 7837, 7710, 7583, 7456, 7329, 7202, 7075, 6948, 6821, 6694, 6567, 6440, 6313, 6186, 6059, 5932, 5805, 5678, 5551, 5424, 5297, 5170, 5043, 4916, 4789, 4662, 4535, 4408, 4281, 4154, 4027, 3900, 3773, 3646, 3519, 3392, 3265, 3138, 3011, 2884, 2757, 2630, 2503, 2376, 2249, 2122, 1995, 1868, 1741, 1614, 1487, 1360, 1233, 1106, 979, 852, 725, 598, 471, 344, 217, 90, 91, 218, 345, 472, 599, 726, 853, 980, 1107, 1234, 1361, 1488, 1615, 1742, 1869, 1996, 2123, 2250, 2377, 2504, 2631, 2758, 2885, 3012, 3139, 3266, 3393, 3520, 3647, 3774, 3901, 4028, 4155, 4282, 4409, 4536, 4663, 4790,
    4917, 5044, 5171, 5298, 5425, 5552, 5679, 5806, 5933, 6060, 6187, 6314, 6441, 6568, 6695, 6822, 6949, 7076, 7203, 7330, 7457, 7584, 7711, 7838, 7965, 8092, 8219, 8346, 8473, 8600, 8727, 8854, 8981, 9108, 9235, 9362, 9489, 9616, 9743, 9870, 9997, 10124, 10251, 10378, 10505, 10632, 10759, 10886, 11013, 11140, 11267, 11394, 11521, 11648, 11776, 11649, 11522, 11395, 11268, 11141, 11014, 10887, 10760, 10633, 10506, 10379, 10252, 10125, 9998, 9871, 9744, 9617, 9490, 9363, 9236, 9109, 8982, 8855, 8728, 8601, 8474, 8347, 8220, 8093, 7966, 7839, 7712, 7585, 7458, 7331, 7204, 7077, 6950, 6823, 6696, 6569, 6442, 6315, 6188, 6061, 5934, 5807, 5680, 5553, 5426, 5299, 5172, 5045, 4918, 4791, 4664, 4537, 4410, 4283, 4156, 4029, 3902, 3775, 3648, 3521, 3394, 3267, 3140, 3013, 2886, 2759, 2632, 2505,
    2378, 2251, 2124, 1997, 1870, 1743, 1616, 1489, 1362, 1235, 1108, 981, 854, 727, 600, 473, 346, 219, 92, 93, 220, 347, 474, 601, 728, 855, 982, 1109, 1236, 1363, 1490, 1617, 1744, 1871, 1998, 2125, 2252, 2379, 2506, 2633, 2760, 2887, 3014, 3141, 3268, 3395, 3522, 3649, 3776, 3903, 4030, 4157, 4284, 4411, 4538, 4665, 4792, 4919, 5046, 5173, 5300, 5427, 5554, 5681, 5808, 5935, 6062, 6189, 6316, 6443, 6570, 6697, 6824, 6951, 7078, 7205, 7332, 7459, 7586, 7713, 7840, 7967, 8094, 8221, 8348, 8475, 8602, 8729, 8856, 8983, 9110, 9237, 9364, 9491, 9618, 9745, 9872, 9999, 10126, 10253, 10380, 10507, 10634, 10761, 10888, 11015, 11142, 11269, 11396, 11523, 11650, 11777, 11904, 12032, 11905, 11778, 11651, 11524, 11397, 11270, 11143, 11016, 10889, 10762, 10635, 10508, 10381, 10254,
    10127, 10000, 9873, 9746, 9619, 9492, 9365, 9238, 9111, 8984, 8857, 8730, 8603, 8476, 8349, 8222, 8095, 7968, 7841, 7714, 7587, 7460, 7333, 7206, 7079, 6952, 6825, 6698, 6571, 6444, 6317, 6190, 6063, 5936, 5809, 5682, 5555, 5428, 5301, 5174, 5047, 4920, 4793, 4666, 4539, 4412, 4285, 4158, 4031, 3904, 3777, 3650, 3523, 3396, 3269, 3142, 3015, 2888, 2761, 2634, 2507, 2380, 2253, 2126, 1999, 1872, 1745, 1618, 1491, 1364, 1237, 1110, 983, 856, 729, 602, 475, 348, 221, 94, 95, 222, 349, 476, 603, 730, 857, 984, 1111, 1238, 1365, 1492, 1619, 1746, 1873, 2000, 2127, 2254, 2381, 2508, 2635, 2762, 2889, 3016, 3143, 3270, 3397, 3524, 3651, 3778, 3905, 4032, 4159, 4286, 4413, 4540, 4667, 4794, 4921, 5048, 5175, 5302, 5429, 5556, 5683, 5810, 5937, 6064,
    6191, 6318, 6445, 6572, 6699, 6826, 6953, 7080, 7207, 7334, 7461, 7588, 7715, 7842, 7969, 8096, 8223, 8350, 8477, 8604, 8731, 8858, 8985, 9112, 9239, 9366, 9493, 9620, 9747, 9874, 10001, 10128, 10255, 10382, 10509, 10636, 10763, 10890, 11017, 11144, 11271, 11398, 11525, 11652, 11779, 11906, 12033, 12160, 12288, 12161, 12034, 11907, 11780, 11653, 11526, 11399, 11272, 11145, 11018, 10891, 10764, 10637, 10510, 10383, 10256, 10129, 10002, 9875, 9748, 9621, 9494, 9367, 9240, 9113, 8986, 8859, 8732, 8605, 8478, 8351, 8224, 8097, 7970, 7843, 7716, 7589, 7462, 7335, 7208, 7081, 6954, 6827, 6700, 6573, 6446, 6319, 6192, 6065, 5938, 5811, 5684, 5557, 5430, 5303, 5176, 5049, 4922, 4795, 4668, 4541, 4414, 4287, 4160, 4033, 3906, 3779, 3652, 3525, 3398, 3271, 3144, 3017, 2890, 2763, 2636, 2509, 2382, 2255,
    2128, 2001, 1874, 1747, 1620, 1493, 1366, 1239, 1112, 985, 858, 731, 604, 477, 350, 223, 96, 97, 224, 351, 478, 605, 732, 859, 986, 1113, 1240, 1367, 1494, 1621, 1748, 1875, 2002, 2129, 2256, 2383, 2510, 2637, 2764, 2891, 3018, 3145, 3272, 3399, 3526, 3653, 3780, 3907, 4034, 4161, 4288, 4415, 4542, 4669, 4796, 4923, 5050, 5177, 5304, 5431, 5558, 5685, 5812, 5939, 6066, 6193, 6320, 6447, 6574, 6701, 6828, 6955, 7082, 7209, 7336, 7463, 7590, 7717, 7844, 7971, 8098, 8225, 8352, 8479, 8606, 8733, 8860, 8987, 9114, 9241, 9368, 9495, 9622, 9749, 9876, 10003, 10130, 10257, 10384, 10511, 10638, 10765, 10892, 11019, 11146, 11273, 11400, 11527, 11654, 11781, 11908, 12035, 12162, 12289, 12416, 12544, 12417, 12290, 12163, 12036, 11909, 11782, 11655, 11528, 11401, 11274, 11147, 11020,
    10893, 10766, 10639, 10512, 10385, 10258, 10131, 10004, 9877, 9750, 9623, 9496, 9369, 9242, 9115, 8988, 8861, 8734, 8607, 8480, 8353, 8226, 8099, 7972, 7845, 7718, 7591, 7464, 7337, 7210, 7083, 6956, 6829, 6702, 6575, 6448, 6321, 6194, 6067, 5940, 5813, 5686, 5559, 5432, 5305, 5178, 5051, 4924, 4797, 4670, 4543, 4416, 4289, 4162, 4035, 3908, 3781, 3654, 3527, 3400, 3273, 3146, 3019, 2892, 2765, 2638, 2511, 2384, 2257, 2130, 2003, 1876, 1749, 1622, 1495, 1368, 1241, 1114, 987, 860, 733, 606, 479, 352, 225, 98, 99, 226, 353, 480, 607, 734, 861, 988, 1115, 1242, 1369, 1496, 1623, 1750, 1877, 2004, 2131, 2258, 2385, 2512, 2639, 2766, 2893, 3020, 3147, 3274, 3401, 3528, 3655, 3782, 3909, 4036, 4163, 4290, 4417, 4544, 4671, 4798, 4925, 5052, 5179, 5306,
    5433, 5560, 5687, 5814, 5941, 6068, 6195, 6322, 6449, 6576, 6703, 6830, 6957, 7084, 7211, 7338, 7465, 7592, 7719, 7846, 7973, 8100, 8227, 8354, 8481, 8608, 8735, 8862, 8989, 9116, 9243, 9370, 9497, 9624, 9751, 9878, 10005, 10132, 10259, 10386, 10513, 10640, 10767, 10894, 11021, 11148, 11275, 11402, 11529, 11656, 11783, 11910, 12037, 12164, 12291, 12418, 12545, 12672, 12800, 12673, 12546, 12419, 12292, 12165, 12038, 11911, 11784, 11657, 11530, 11403, 11276, 11149, 11022, 10895, 10768, 10641, 10514, 10387, 10260, 10133, 10006, 9879, 9752, 9625, 9498, 9371, 9244, 9117, 8990, 8863, 8736, 8609, 8482, 8355, 8228, 8101, 7974, 7847, 7720, 7593, 7466, 7339, 7212, 7085, 6958, 6831, 6704, 6577, 6450, 6323, 6196, 6069, 5942, 5815, 5688, 5561, 5434, 5307, 5180, 5053, 4926, 4799, 4672, 4545, 4418, 4291, 4164, 4037,
    3910, 3783, 3656, 3529, 3402, 3275, 3148, 3021, 2894, 2767, 2640, 2513, 2386, 2259, 2132, 2005, 1878, 1751, 1624, 1497, 1370, 1243, 1116, 989, 862, 735, 608, 481, 354, 227, 100, 101, 228, 355, 482, 609, 736, 863, 990, 1117, 1244, 1371, 1498, 1625, 1752, 1879, 2006, 2133, 2260, 2387, 2514, 2641, 2768, 2895, 3022, 3149, 3276, 3403, 3530, 3657, 3784, 3911, 4038, 4165, 4292, 4419, 4546, 4673, 4800, 4927, 5054, 5181, 5308, 5435, 5562, 5689, 5816, 5943, 6070, 6197, 6324, 6451, 6578, 6705, 6832, 6959, 7086, 7213, 7340, 7467, 7594, 7721, 7848, 7975, 8102, 8229, 8356, 8483, 8610, 8737, 8864, 8991, 9118, 9245, 9372, 9499, 9626, 9753, 9880, 10007, 10134, 10261, 10388, 10515, 10642, 10769, 10896, 11023, 11150, 11277, 11404, 11531, 11658, 11785, 11912, 12039, 12166, 12293,
    12420, 12547, 12674, 12801, 12928, 13056, 12929, 12802, 12675, 12548, 12421, 12294, 12167, 12040, 11913, 11786, 11659, 11532, 11405, 11278, 11151, 11024, 10897, 10770, 10643, 10516, 10389, 10262, 10135, 10008, 9881, 9754, 9627, 9500, 9373, 9246, 9119, 8992, 8865, 8738, 8611, 8484, 8357, 8230, 8103, 7976, 7849, 7722, 7595, 7468, 7341, 7214, 7087, 6960, 6833, 6706, 6579, 6452, 6325, 6198, 6071, 5944, 5817, 5690, 5563, 5436, 5309, 5182, 5055, 4928, 4801, 4674, 4547, 4420, 4293, 4166, 4039, 3912, 3785, 3658, 3531, 3404, 3277, 3150, 3023, 2896, 2769, 2642, 2515, 2388, 2261, 2134, 2007, 1880, 1753, 1626, 1499, 1372, 1245, 1118, 991, 864, 737, 610, 483, 356, 229, 102, 103, 230, 357, 484, 611, 738, 865, 992, 1119, 1246, 1373, 1500, 1627, 1754, 1881, 2008, 2135, 2262, 2389, 2516,
    2643, 2770, 2897, 3024, 3151, 3278, 3405, 3532, 3659, 3786, 3913, 4040, 4167, 4294, 4421, 4548, 4675, 4802, 4929, 5056, 5183, 5310, 5437, 5564, 5691, 5818, 5945, 6072, 6199, 6326, 6453, 6580, 6707, 6834, 6961, 7088, 7215, 7342, 7469, 7596, 7723, 7850, 7977, 8104, 8231, 8358, 8485, 8612, 8739, 8866, 8993, 9120, 9247, 9374, 9501, 9628, 9755, 9882, 10009, 10136, 10263, 10390, 10517, 10644, 10771, 10898, 11025, 11152, 11279, 11406, 11533, 11660, 11787, 11914, 12041, 12168, 12295, 12422, 12549, 12676, 12803, 12930, 13057, 13184, 13312, 13185, 13058, 12931, 12804, 12677, 12550, 12423, 12296, 12169, 12042, 11915, 11788, 11661, 11534, 11407, 11280, 11153, 11026, 10899, 10772, 10645, 10518, 10391, 10264, 10137, 10010, 9883, 9756, 9629, 9502, 9375, 9248, 9121, 8994, 8867, 8740, 8613, 8486, 8359, 8232, 8105, 7978, 7851,
    7724, 7597, 7470, 7343, 7216, 7089, 6962, 6835, 6708, 6581, 6454, 6327, 6200, 6073, 5946, 5819, 5692, 5565, 5438, 5311, 5184, 5057, 4930, 4803, 4676, 4549, 4422, 4295, 4168, 4041, 3914, 3787, 3660, 3533, 3406, 3279, 3152, 3025, 2898, 2771, 2644, 2517, 2390, 2263, 2136, 2009, 1882, 1755, 1628, 1501, 1374, 1247, 1120, 993, 866, 739, 612, 485, 358, 231, 104, 105, 232, 359, 486, 613, 740, 867, 994, 1121, 1248, 1375, 1502, 1629, 1756, 1883, 2010, 2137, 2264, 2391, 2518, 2645, 2772, 2899, 3026, 3153, 3280, 3407, 3534, 3661, 3788, 3915, 4042, 4169, 4296, 4423, 4550, 4677, 4804, 4931, 5058, 5185, 5312, 5439, 5566, 5693, 5820, 5947, 6074, 6201, 6328, 6455, 6582, 6709, 6836, 6963, 7090, 7217, 7344, 7471, 7598, 7725, 7852, 7979, 8106, 8233, 8360, 8487,
    8614, 8741, 8868, 8995, 9122, 9249, 9376, 9503, 9630, 9757, 9884, 10011, 10138, 10265, 10392, 10519, 10646, 10773, 10900, 11027, 11154, 11281, 11408, 11535, 11662, 11789, 11916, 12043, 12170, 12297, 12424, 12551, 12678, 12805, 12932, 13059, 13186, 13313, 13440, 13568, 13441, 13314, 13187, 13060, 12933, 12806, 12679, 12552, 12425, 12298, 12171, 12044, 11917, 11790, 11663, 11536, 11409, 11282, 11155, 11028, 10901, 10774, 10647, 10520, 10393, 10266, 10139, 10012, 9885, 9758, 9631, 9504, 9377, 9250, 9123, 8996, 8869, 8742, 8615, 8488, 8361, 8234, 8107, 7980, 7853, 7726, 7599, 7472, 7345, 7218, 7091, 6964, 6837, 6710, 6583, 6456, 6329, 6202, 6075, 5948, 5821, 5694, 5567, 5440, 5313, 5186, 5059, 4932, 4805, 4678, 4551, 4424, 4297, 4170, 4043, 3916, 3789, 3662, 3535, 3408, 3281, 3154, 3027, 2900, 2773, 2646, 2519, 2392,
    2265, 2138, 2011, 1884, 1757, 1630, 1503, 1376, 1249, 1122, 995, 868, 741, 614, 487, 360, 233, 106, 107, 234, 361, 488, 615, 742, 869, 996, 1123, 1250, 1377, 1504, 1631, 1758, 1885, 2012, 2139, 2266, 2393, 2520, 2647, 2774, 2901, 3028, 3155, 3282, 3409, 3536, 3663, 3790, 3917, 4044, 4171, 4298, 4425, 4552, 4679, 4806, 4933, 5060, 5187, 5314, 5441, 5568, 5695, 5822, 5949, 6076, 6203, 6330, 6457, 6584, 6711, 6838, 6965, 7092, 7219, 7346, 7473, 7600, 7727, 7854, 7981, 8108, 8235, 8362, 8489, 8616, 8743, 8870, 8997, 9124, 9251, 9378, 9505, 9632, 9759, 9886, 10013, 10140, 10267, 10394, 10521, 10648, 10775, 10902, 11029, 11156, 11283, 11410, 11537, 11664, 11791, 11918, 12045, 12172, 12299, 12426, 12553, 12680, 12807, 12934, 13061, 13188, 13315, 13442, 13569, 13696, 13824, 13697,
    13570, 13443, 13316, 13189, 13062, 12935, 12808, 12681, 12554, 12427, 12300, 12173, 12046, 11919, 11792, 11665, 11538, 11411, 11284, 11157, 11030, 10903, 10776, 10649, 10522, 10395, 10268, 10141, 10014, 9887, 9760, 9633, 9506, 9379, 9252, 9125, 8998, 8871, 8744, 8617, 8490, 8363, 8236, 8109, 7982, 7855, 7728, 7601, 7474, 7347, 7220, 7093, 6966, 6839, 6712, 6585, 6458, 6331, 6204, 6077, 5950, 5823, 5696, 5569, 5442, 5315, 5188, 5061, 4934, 4807, 4680, 4553, 4426, 4299, 4172, 4045, 3918, 3791, 3664, 3537, 3410, 3283, 3156, 3029, 2902, 2775, 2648, 2521, 2394, 2267, 2140, 2013, 1886, 1759, 1632, 1505, 1378, 1251, 1124, 997, 870, 743, 616, 489, 362, 235, 108, 109, 236, 363, 490, 617, 744, 871, 998, 1125, 1252, 1379, 1506, 1633, 1760, 1887, 2014, 2141, 2268, 2395, 2522, 2649,
    2776, 2903, 3030, 3157, 3284, 3411, 3538, 3665, 3792, 3919, 4046, 4173, 4300, 4427, 4554, 4681, 4808, 4935, 5062, 5189, 5316, 5443, 5570, 5697, 5824, 5951, 6078, 6205, 6332, 6459, 6586, 6713, 6840, 6967, 7094, 7221, 7348, 7475, 7602, 7729, 7856, 7983, 8110, 8237, 8364, 8491, 8618, 8745, 8872, 8999, 9126, 9253, 9380, 9507, 9634, 9761, 9888, 10015, 10142, 10269, 10396, 10523, 10650, 10777, 10904, 11031, 11158, 11285, 11412, 11539, 11666, 11793, 11920, 12047, 12174, 12301, 12428, 12555, 12682, 12809, 12936, 13063, 13190, 13317, 13444, 13571, 13698, 13825, 13952, 14080, 13953, 13826, 13699, 13572, 13445, 13318, 13191, 13064, 12937, 12810, 12683, 12556, 12429, 12302, 12175, 12048, 11921, 11794, 11667, 11540, 11413, 11286, 11159, 11032, 10905, 10778, 10651, 10524, 10397, 10270, 10143, 10016, 9889, 9762, 9635, 9508, 9381, 9254,
    9127, 9000, 8873, 8746, 8619, 8492, 8365, 8238, 8111, 7984, 7857, 7730, 7603, 7476, 7349, 7222, 7095, 6968, 6841, 6714, 6587, 6460, 6333, 6206, 6079, 5952, 5825, 5698, 5571, 5444, 5317, 5190, 5063, 4936, 4809, 4682, 4555, 4428, 4301, 4174, 4047, 3920, 3793, 3666, 3539, 3412, 3285, 3158, 3031, 2904, 2777, 2650, 2523, 2396, 2269, 2142, 2015, 1888, 1761, 1634, 1507, 1380, 1253, 1126, 999, 872, 745, 618, 491, 364, 237, 110, 111, 238, 365, 492, 619, 746, 873, 1000, 1127, 1254, 1381, 1508, 1635, 1762, 1889, 2016, 2143, 2270, 2397, 2524, 2651, 2778, 2905, 3032, 3159, 3286, 3413, 3540, 3667, 3794, 3921, 4048, 4175, 4302, 4429, 4556, 4683, 4810, 4937, 5064, 5191, 5318, 5445, 5572, 5699, 5826, 5953, 6080, 6207, 6334, 6461, 6588, 6715, 6842, 6969, 7096,
    7223, 7350, 7477, 7604, 7731, 7858, 7985, 8112, 8239, 8366, 8493, 8620, 8747, 8874, 9001, 9128, 9255, 9382, 9509, 9636, 9763, 9890, 10017, 10144, 10271, 10398, 10525, 10652, 10779, 10906, 11033, 11160, 11287, 11414, 11541, 11668, 11795, 11922, 12049, 12176, 12303, 12430, 12557, 12684, 12811, 12938, 13065, 13192, 13319, 13446, 13573, 13700, 13827, 13954, 14081, 14208, 14336, 14209, 14082, 13955, 13828, 13701, 13574, 13447, 13320, 13193, 13066, 12939, 12812, 12685, 12558, 12431, 12304, 12177, 12050, 11923, 11796, 11669, 11542, 11415, 11288, 11161, 11034, 10907, 10780, 10653, 10526, 10399, 10272, 10145, 10018, 9891, 9764, 9637, 9510, 9383, 9256, 9129, 9002, 8875, 8748, 8621, 8494, 8367, 8240, 8113, 7986, 7859, 7732, 7605, 7478, 7351, 7224, 7097, 6970, 6843, 6716, 6589, 6462, 6335, 6208, 6081, 5954, 5827, 5700, 5573, 5446, 5319,
    5192, 5065, 4938, 4811, 4684, 4557, 4430, 4303, 4176, 4049, 3922, 3795, 3668, 3541, 3414, 3287, 3160, 3033, 2906, 2779, 2652, 2525, 2398, 2271, 2144, 2017, 1890, 1763, 1636, 1509, 1382, 1255, 1128, 1001, 874, 747, 620, 493, 366, 239, 112, 113, 240, 367, 494, 621, 748, 875, 1002, 1129, 1256, 1383, 1510, 1637, 1764, 1891, 2018, 2145, 2272, 2399, 2526, 2653, 2780, 2907, 3034, 3161, 3288, 3415, 3542, 3669, 3796, 3923, 4050, 4177, 4304, 4431, 4558, 4685, 4812, 4939, 5066, 5193, 5320, 5447, 5574, 5701, 5828, 5955, 6082, 6209, 6336, 6463, 6590, 6717, 6844, 6971, 7098, 7225, 7352, 7479, 7606, 7733, 7860, 7987, 8114, 8241, 8368, 8495, 8622, 8749, 8876, 9003, 9130, 9257, 9384, 9511, 9638, 9765, 9892, 10019, 10146, 10273, 10400, 10527, 10654, 10781, 10908, 11035,
    11162, 11289, 11416, 11543, 11670, 11797, 11924, 12051, 12178, 12305, 12432, 12559, 12686, 12813, 12940, 13067, 13194, 13321, 13448, 13575, 13702, 13829, 13956, 14083, 14210, 14337, 14464, 14592, 14465, 14338, 14211, 14084, 13957, 13830, 13703, 13576, 13449, 13322, 13195, 13068, 12941, 12814, 12687, 12560, 12433, 12306, 12179, 12052, 11925, 11798, 11671, 11544, 11417, 11290, 11163, 11036, 10909, 10782, 10655, 10528, 10401, 10274, 10147, 10020, 9893, 9766, 9639, 9512, 9385, 9258, 9131, 9004, 8877, 8750, 8623, 8496, 8369, 8242, 8115, 7988, 7861, 7734, 7607, 7480, 7353, 7226, 7099, 6972, 6845, 6718, 6591, 6464, 6337, 6210, 6083, 5956, 5829, 5702, 5575, 5448, 5321, 5194, 5067, 4940, 4813, 4686, 4559, 4432, 4305, 4178, 4051, 3924, 3797, 3670, 3543, 3416, 3289, 3162, 3035, 2908, 2781, 2654, 2527, 2400, 2273, 2146, 2019, 1892,
    1765, 1638, 1511, 1384, 1257, 1130, 1003, 876, 749, 622, 495, 368, 241, 114, 115, 242, 369, 496, 623, 750, 877, 1004, 1131, 1258, 1385, 1512, 1639, 1766, 1893, 2020, 2147, 2274, 2401, 2528, 2655, 2782, 2909, 3036, 3163, 3290, 3417, 3544, 3671, 3798, 3925, 4052, 4179, 4306, 4433, 4560, 4687, 4814, 4941, 5068, 5195, 5322, 5449, 5576, 5703, 5830, 5957, 6084, 6211, 6338, 6465, 6592, 6719, 6846, 6973, 7100, 7227, 7354, 7481, 7608, 7735, 7862, 7989, 8116, 8243, 8370, 8497, 8624, 8751, 8878, 9005, 9132, 9259, 9386, 9513, 9640, 9767, 9894, 10021, 10148, 10275, 10402, 10529, 10656, 10783, 10910, 11037, 11164, 11291, 11418, 11545, 11672, 11799, 11926, 12053, 12180, 12307, 12434, 12561, 12688, 12815, 12942, 13069, 13196, 13323, 13450, 13577, 13704, 13831, 13958, 14085, 14212, 14339, 14466,
    14593, 14720, 14848, 14721, 14594, 14467, 14340, 14213, 14086, 13959, 13832, 13705, 13578, 13451, 13324, 13197, 13070, 12943, 12816, 12689, 12562, 12435, 12308, 12181, 12054, 11927, 11800, 11673, 11546, 11419, 11292, 11165, 11038, 10911, 10784, 10657, 10530, 10403, 10276, 10149, 10022, 9895, 9768, 9641, 9514, 9387, 9260, 9133, 9006, 8879, 8752, 8625, 8498, 8371, 8244, 8117, 7990, 7863, 7736, 7609, 7482, 7355, 7228, 7101, 6974, 6847, 6720, 6593, 6466, 6339, 6212, 6085, 5958, 5831, 5704, 5577, 5450, 5323, 5196, 5069, 4942, 4815, 4688, 4561, 4434, 4307, 4180, 4053, 3926, 3799, 3672, 3545, 3418, 3291, 3164, 3037, 2910, 2783, 2656, 2529, 2402, 2275, 2148, 2021, 1894, 1767, 1640, 1513, 1386, 1259, 1132, 1005, 878, 751, 624, 497, 370, 243, 116, 117, 244, 371, 498, 625, 752, 879, 1006, 1133,
    1260, 1387, 1514, 1641, 1768, 1895, 2022, 2149, 2276, 2403, 2530, 2657, 2784, 2911, 3038, 3165, 3292, 3419, 3546, 3673, 3800, 3927, 4054, 4181, 4308, 4435, 4562, 4689, 4816, 4943, 5070, 5197, 5324, 5451, 5578, 5705, 5832, 5959, 6086, 6213, 6340, 6467, 6594, 6721, 6848, 6975, 7102, 7229, 7356, 7483, 7610, 7737, 7864, 7991, 8118, 8245, 8372, 8499, 8626, 8753, 8880, 9007, 9134, 9261, 9388, 9515, 9642, 9769, 9896, 10023, 10150, 10277, 10404, 10531, 10658, 10785, 10912, 11039, 11166, 11293, 11420, 11547, 11674, 11801, 11928, 12055, 12182, 12309, 12436, 12563, 12690, 12817, 12944, 13071, 13198, 13325, 13452, 13579, 13706, 13833, 13960, 14087, 14214, 14341, 14468, 14595, 14722, 14849, 14976, 15104, 14977, 14850, 14723, 14596, 14469, 14342, 14215, 14088, 13961, 13834, 13707, 13580, 13453, 13326, 13199, 13072, 12945, 12818,
    12691, 12564, 12437, 12310, 12183, 12056, 11929, 11802, 11675, 11548, 11421, 11294, 11167, 11040, 10913, 10786, 10659, 10532, 10405, 10278, 10151, 10024, 9897, 9770, 9643, 9516, 9389, 9262, 9135, 9008, 8881, 8754, 8627, 8500, 8373, 8246, 8119, 7992, 7865, 7738, 7611, 7484, 7357, 7230, 7103, 6976, 6849, 6722, 6595, 6468, 6341, 6214, 6087, 5960, 5833, 5706, 5579, 5452, 5325, 5198, 5071, 4944, 4817, 4690, 4563, 4436, 4309, 4182, 4055, 3928, 3801, 3674, 3547, 3420, 3293, 3166, 3039, 2912, 2785, 2658, 2531, 2404, 2277, 2150, 2023, 1896, 1769, 1642, 1515, 1388, 1261, 1134, 1007, 880, 753, 626, 499, 372, 245, 118, 119, 246, 373, 500, 627, 754, 881, 1008, 1135, 1262, 1389, 1516, 1643, 1770, 1897, 2024, 2151, 2278, 2405, 2532, 2659, 2786, 2913, 3040, 3167, 3294, 3421, 3548,
    3675, 3802, 3929, 4056, 4183, 4310, 4437, 4564, 4691, 4818, 4945, 5072, 5199, 5326, 5453, 5580, 5707, 5834, 5961, 6088, 6215, 6342, 6469, 6596, 6723, 6850, 6977, 7104, 7231, 7358, 7485, 7612, 7739, 7866, 7993, 8120, 8247, 8374, 8501, 8628, 8755, 8882, 9009, 9136, 9263, 9390, 9517, 9644, 9771, 9898, 10025, 10152, 10279, 10406, 10533, 10660, 10787, 10914, 11041, 11168, 11295, 11422, 11549, 11676, 11803, 11930, 12057, 12184, 12311, 12438, 12565, 12692, 12819, 12946, 13073, 13200, 13327, 13454, 13581, 13708, 13835, 13962, 14089, 14216, 14343, 14470, 14597, 14724, 14851, 14978, 15105, 15232, 15360, 15233, 15106, 14979, 14852, 14725, 14598, 14471, 14344, 14217, 14090, 13963, 13836, 13709, 13582, 13455, 13328, 13201, 13074, 12947, 12820, 12693, 12566, 12439, 12312, 12185, 12058, 11931, 11804, 11677, 11550, 11423, 11296, 11169, 11042, 10915,
    10788, 10661, 10534, 10407, 10280, 10153, 10026, 9899, 9772, 9645, 9518, 9391, 9264, 9137, 9010, 8883, 8756, 8629, 8502, 8375, 8248, 8121, 7994, 7867, 7740, 7613, 7486, 7359, 7232, 7105, 6978, 6851, 6724, 6597, 6470, 6343, 6216, 6089, 5962, 5835, 5708, 5581, 5454, 5327, 5200, 5073, 4946, 4819, 4692, 4565, 4438, 4311, 4184, 4057, 3930, 3803, 3676, 3549, 3422, 3295, 3168, 3041, 2914, 2787, 2660, 2533, 2406, 2279, 2152, 2025, 1898, 1771, 1644, 1517, 1390, 1263, 1136, 1009, 882, 755, 628, 501, 374, 247, 120, 121, 248, 375, 502, 629, 756, 883, 1010, 1137, 1264, 1391, 1518, 1645, 1772, 1899, 2026, 2153, 2280, 2407, 2534, 2661, 2788, 2915, 3042, 3169, 3296, 3423, 3550, 3677, 3804, 3931, 4058, 4185, 4312, 4439, 4566, 4693, 4820, 4947, 5074, 5201, 5328, 5455,
    5582, 5709, 5836, 5963, 6090, 6217, 6344, 6471, 6598, 6725, 6852, 6979, 7106, 7233, 7360, 7487, 7614, 7741, 7868, 7995, 8122, 8249, 8376, 8503, 8630, 8757, 8884, 9011, 9138, 9265, 9392, 9519, 9646, 9773, 9900, 10027, 10154, 10281, 10408, 10535, 10662, 10789, 10916, 11043, 11170, 11297, 11424, 11551, 11678, 11805, 11932, 12059, 12186, 12313, 12440, 12567, 12694, 12821, 12948, 13075, 13202, 13329, 13456, 13583, 13710, 13837, 13964, 14091, 14218, 14345, 14472, 14599, 14726, 14853, 14980, 15107, 15234, 15361, 15488, 15616, 15489, 15362, 15235, 15108, 14981, 14854, 14727, 14600, 14473, 14346, 14219, 14092, 13965, 13838, 13711, 13584, 13457, 13330, 13203, 13076, 12949, 12822, 12695, 12568, 12441, 12314, 12187, 12060, 11933, 11806, 11679, 11552, 11425, 11298, 11171, 11044, 10917, 10790, 10663, 10536, 10409, 10282, 10155, 10028, 9901, 9774, 9647, 9520,
    9393, 9266, 9139, 9012, 8885, 8758, 8631, 8504, 8377, 8250, 8123, 7996, 7869, 7742, 7615, 7488, 7361, 7234, 7107, 6980, 6853, 6726, 6599, 6472, 6345, 6218, 6091, 5964, 5837, 5710, 5583, 5456, 5329, 5202, 5075, 4948, 4821, 4694, 4567, 4440, 4313, 4186, 4059, 3932, 3805, 3678, 3551, 3424, 3297, 3170, 3043, 2916, 2789, 2662, 2535, 2408, 2281, 2154, 2027, 1900, 1773, 1646, 1519, 1392, 1265, 1138, 1011, 884, 757, 630, 503, 376, 249, 122, 123, 250, 377, 504, 631, 758, 885, 1012, 1139, 1266, 1393, 1520, 1647, 1774, 1901, 2028, 2155, 2282, 2409, 2536, 2663, 2790, 2917, 3044, 3171, 3298, 3425, 3552, 3679, 3806, 3933, 4060, 4187, 4314, 4441, 4568, 4695, 4822, 4949, 5076, 5203, 5330, 5457, 5584, 5711, 5838, 5965, 6092, 6219, 6346, 6473, 6600, 6727, 6854,
    6981, 7108, 7235, 7362, 7489, 7616, 7743, 7870, 7997, 8124, 8251, 8378, 8505, 8632, 8759, 8886, 9013, 9140, 9267, 9394, 9521, 9648, 9775, 9902, 10029, 10156, 10283, 10410, 10537, 10664, 10791, 10918, 11045, 11172, 11299, 11426, 11553, 11680, 11807, 11934, 12061, 12188, 12315, 12442, 12569, 12696, 12823, 12950, 13077, 13204, 13331, 13458, 13585, 13712, 13839, 13966, 14093, 14220, 14347, 14474, 14601, 14728, 14855, 14982, 15109, 15236, 15363, 15490, 15617, 15744, 15872, 15745, 15618, 15491, 15364, 15237, 15110, 14983, 14856, 14729, 14602, 14475, 14348, 14221, 14094, 13967, 13840, 13713, 13586, 13459, 13332, 13205, 13078, 12951, 12824, 12697, 12570, 12443, 12316, 12189, 12062, 11935, 11808, 11681, 11554, 11427, 11300, 11173, 11046, 10919, 10792, 10665, 10538, 10411, 10284, 10157, 10030, 9903, 9776, 9649, 9522, 9395, 9268, 9141, 9014, 8887, 8760, 8633,
    8506, 8379, 8252, 8125, 7998, 7871, 7744, 7617, 7490, 7363, 7236, 7109, 6982, 6855, 6728, 6601, 6474, 6347, 6220, 6093, 5966, 5839, 5712, 5585, 5458, 5331, 5204, 5077, 4950, 4823, 4696, 4569, 4442, 4315, 4188, 4061, 3934, 3807, 3680, 3553, 3426, 3299, 3172, 3045, 2918, 2791, 2664, 2537, 2410, 2283, 2156, 2029, 1902, 1775, 1648, 1521, 1394, 1267, 1140, 1013, 886, 759, 632, 505, 378, 251, 124, 125, 252, 379, 506, 633, 760, 887, 1014, 1141, 1268, 1395, 1522, 1649, 1776, 1903, 2030, 2157, 2284, 2411, 2538, 2665, 2792, 2919, 3046, 3173, 3300, 3427, 3554, 3681, 3808, 3935, 4062, 4189, 4316, 4443, 4570, 4697, 4824, 4951, 5078, 5205, 5332, 5459, 5586, 5713, 5840, 5967, 6094, 6221, 6348, 6475, 6602, 6729, 6856, 6983, 7110, 7237, 7364, 7491, 7618, 7745,
    7872, 7999, 8126, 8253, 8380, 8507, 8634, 8761, 8888, 9015, 9142, 9269, 9396, 9523, 9650, 9777, 9904, 10031, 10158, 10285, 10412, 10539, 10666, 10793, 10920, 11047, 11174, 11301, 11428, 11555, 11682, 11809, 11936, 12063, 12190, 12317, 12444, 12571, 12698, 12825, 12952, 13079, 13206, 13333, 13460, 13587, 13714, 13841, 13968, 14095, 14222, 14349, 14476, 14603, 14730, 14857, 14984, 15111, 15238, 15365, 15492, 15619, 15746, 15873, 16000, 16128, 16001, 15874, 15747, 15620, 15493, 15366, 15239, 15112, 14985, 14858, 14731, 14604, 14477, 14350, 14223, 14096, 13969, 13842, 13715, 13588, 13461, 13334, 13207, 13080, 12953, 12826, 12699, 12572, 12445, 12318, 12191, 12064, 11937, 11810, 11683, 11556, 11429, 11302, 11175, 11048, 10921, 10794, 10667, 10540, 10413, 10286, 10159, 10032, 9905, 9778, 9651, 9524, 9397, 9270, 9143, 9016, 8889, 8762, 8635, 8508, 8381, 8254,
    8127, 8000, 7873, 7746, 7619, 7492, 7365, 7238, 7111, 6984, 6857, 6730, 6603, 6476, 6349, 6222, 6095, 5968, 5841, 5714, 5587, 5460, 5333, 5206, 5079, 4952, 4825, 4698, 4571, 4444, 4317, 4190, 4063, 3936, 3809, 3682, 3555, 3428, 3301, 3174, 3047, 2920, 2793, 2666, 2539, 2412, 2285, 2158, 2031, 1904, 1777, 1650, 1523, 1396, 1269, 1142, 1015, 888, 761, 634, 507, 380, 253, 126, 127, 254, 381, 508, 635, 762, 889, 1016, 1143, 1270, 1397, 1524, 1651, 1778, 1905, 2032, 2159, 2286, 2413, 2540, 2667, 2794, 2921, 3048, 3175, 3302, 3429, 3556, 3683, 3810, 3937, 4064, 4191, 4318, 4445, 4572, 4699, 4826, 4953, 5080, 5207, 5334, 5461, 5588, 5715, 5842, 5969, 6096, 6223, 6350, 6477, 6604, 6731, 6858, 6985, 7112, 7239, 7366, 7493, 7620, 7747, 7874, 8001, 8128,
    8255, 8382, 8509, 8636, 8763, 8890, 9017, 9144, 9271, 9398, 9525, 9652, 9779, 9906, 10033, 10160, 10287, 10414, 10541, 10668, 10795, 10922, 11049, 11176, 11303, 11430, 11557, 11684, 11811, 11938, 12065, 12192, 12319, 12446, 12573, 12700, 12827, 12954, 13081, 13208, 13335, 13462, 13589, 13716, 13843, 13970, 14097, 14224, 14351, 14478, 14605, 14732, 14859, 14986, 15113, 15240, 15367, 15494, 15621, 15748, 15875, 16002, 16129, 16256, 16257, 16130, 16003, 15876, 15749, 15622, 15495, 15368, 15241, 15114, 14987, 14860, 14733, 14606, 14479, 14352, 14225, 14098, 13971, 13844, 13717, 13590, 13463, 13336, 13209, 13082, 12955, 12828, 12701, 12574, 12447, 12320, 12193, 12066, 11939, 11812, 11685, 11558, 11431, 11304, 11177, 11050, 10923, 10796, 10669, 10542, 10415, 10288, 10161, 10034, 9907, 9780, 9653, 9526, 9399, 9272, 9145, 9018, 8891, 8764, 8637, 8510, 8383, 8256,
    8129, 8002, 7875, 7748, 7621, 7494, 7367, 7240, 7113, 6986, 6859, 6732, 6605, 6478, 6351, 6224, 6097, 5970, 5843, 5716, 5589, 5462, 5335, 5208, 5081, 4954, 4827, 4700, 4573, 4446, 4319, 4192, 4065, 3938, 3811, 3684, 3557, 3430, 3303, 3176, 3049, 2922, 2795, 2668, 2541, 2414, 2287, 2160, 2033, 1906, 1779, 1652, 1525, 1398, 1271, 1144, 1017, 890, 763, 636, 509, 382, 255, 383, 510, 637, 764, 891, 1018, 1145, 1272, 1399, 1526, 1653, 1780, 1907, 2034, 2161, 2288, 2415, 2542, 2669, 2796, 2923, 3050, 3177, 3304, 3431, 3558, 3685, 3812, 3939, 4066, 4193, 4320, 4447, 4574, 4701, 4828, 4955, 5082, 5209, 5336, 5463, 5590, 5717, 5844, 5971, 6098, 6225, 6352, 6479, 6606, 6733, 6860, 6987, 7114, 7241, 7368, 7495, 7622, 7749, 7876, 8003, 8130, 8257, 8384, 8511,
    8638, 8765, 8892, 9019, 9146, 9273, 9400, 9527, 9654, 9781, 9908, 10035, 10162, 10289, 10416, 10543, 10670, 10797, 10924, 11051, 11178, 11305, 11432, 11559, 11686, 11813, 11940, 12067, 12194, 12321, 12448, 12575, 12702, 12829, 12956, 13083, 13210, 13337, 13464, 13591, 13718, 13845, 13972, 14099, 14226, 14353, 14480, 14607, 14734, 14861, 14988, 15115, 15242, 15369, 15496, 15623, 15750, 15877, 16004, 16131, 16258, 16259, 16132, 16005, 15878, 15751, 15624, 15497, 15370, 15243, 15116, 14989, 14862, 14735, 14608, 14481, 14354, 14227, 14100, 13973, 13846, 13719, 13592, 13465, 13338, 13211, 13084, 12957, 12830, 12703, 12576, 12449, 12322, 12195, 12068, 11941, 11814, 11687, 11560, 11433, 11306, 11179, 11052, 10925, 10798, 10671, 10544, 10417, 10290, 10163, 10036, 9909, 9782, 9655, 9528, 9401, 9274, 9147, 9020, 8893, 8766, 8639, 8512, 8385, 8258, 8131, 8004, 7877,
    7750, 7623, 7496, 7369, 7242, 7115, 6988, 6861, 6734, 6607, 6480, 6353, 6226, 6099, 5972, 5845, 5718, 5591, 5464, 5337, 5210, 5083, 4956, 4829, 4702, 4575, 4448, 4321, 4194, 4067, 3940, 3813, 3686, 3559, 3432, 3305, 3178, 3051, 2924, 2797, 2670, 2543, 2416, 2289, 2162, 2035, 1908, 1781, 1654, 1527, 1400, 1273, 1146, 1019, 892, 765, 638, 511, 639, 766, 893, 1020, 1147, 1274, 1401, 1528, 1655, 1782, 1909, 2036, 2163, 2290, 2417, 2544, 2671, 2798, 2925, 3052, 3179, 3306, 3433, 3560, 3687, 3814, 3941, 4068, 4195, 4322, 4449, 4576, 4703, 4830, 4957, 5084, 5211, 5338, 5465, 5592, 5719, 5846, 5973, 6100, 6227, 6354, 6481, 6608, 6735, 6862, 6989, 7116, 7243, 7370, 7497, 7624, 7751, 7878, 8005, 8132, 8259, 8386, 8513, 8640, 8767, 8894, 9021, 9148, 9275, 9402,
    9529, 9656, 9783, 9910, 10037, 10164, 10291, 10418, 10545, 10672, 10799, 10926, 11053, 11180, 11307, 11434, 11561, 11688, 11815, 11942, 12069, 12196, 12323, 12450, 12577, 12704, 12831, 12958, 13085, 13212, 13339, 13466, 13593, 13720, 13847, 13974, 14101, 14228, 14355, 14482, 14609, 14736, 14863, 14990, 15117, 15244, 15371, 15498, 15625, 15752, 15879, 16006, 16133, 16260, 16261, 16134, 16007, 15880, 15753, 15626, 15499, 15372, 15245, 15118, 14991, 14864, 14737, 14610, 14483, 14356, 14229, 14102, 13975, 13848, 13721, 13594, 13467, 13340, 13213, 13086, 12959, 12832, 12705, 12578, 12451, 12324, 12197, 12070, 11943, 11816, 11689, 11562, 11435, 11308, 11181, 11054, 10927, 10800, 10673, 10546, 10419, 10292, 10165, 10038, 9911, 9784, 9657, 9530, 9403, 9276, 9149, 9022, 8895, 8768, 8641, 8514, 8387, 8260, 8133, 8006, 7879, 7752, 7625, 7498, 7371, 7244, 7117, 6990,
    6863, 6736, 6609, 6482, 6355, 6228, 6101, 5974, 5847, 5720, 5593, 5466, 5339, 5212, 5085, 4958, 4831, 4704, 4577, 4450, 4323, 4196, 4069, 3942, 3815, 3688, 3561, 3434, 3307, 3180, 3053, 2926, 2799, 2672, 2545, 2418, 2291, 2164, 2037, 1910, 1783, 1656, 1529, 1402, 1275, 1148, 1021, 894, 767, 895, 1022, 1149, 1276, 1403, 1530, 1657, 1784, 1911, 2038, 2165, 2292, 2419, 2546, 2673, 2800, 2927, 3054, 3181, 3308, 3435, 3562, 3689, 3816, 3943, 4070, 4197, 4324, 4451, 4578, 4705, 4832, 4959, 5086, 5213, 5340, 5467, 5594, 5721, 5848, 5975, 6102, 6229, 6356, 6483, 6610, 6737, 6864, 6991, 7118, 7245, 7372, 7499, 7626, 7753, 7880, 8007, 8134, 8261, 8388, 8515, 8642, 8769, 8896, 9023, 9150, 9277, 9404, 9531, 9658, 9785, 9912, 10039, 10166, 10293, 10420, 10547, 10674, 10801,
    10928, 11055, 11182, 11309, 11436, 11563, 11690, 11817, 11944, 12071, 12198, 12325, 12452, 12579, 12706, 12833, 12960, 13087, 13214, 13341, 13468, 13595, 13722, 13849, 13976, 14103, 14230, 14357, 14484, 14611, 14738, 14865, 14992, 15119, 15246, 15373, 15500, 15627, 15754, 15881, 16008, 16135, 16262, 16263, 16136, 16009, 15882, 15755, 15628, 15501, 15374, 15247, 15120, 14993, 14866, 14739, 14612, 14485, 14358, 14231, 14104, 13977, 13850, 13723, 13596, 13469, 13342, 13215, 13088, 12961, 12834, 12707, 12580, 12453, 12326, 12199, 12072, 11945, 11818, 11691, 11564, 11437, 11310, 11183, 11056, 10929, 10802, 10675, 10548, 10421, 10294, 10167, 10040, 9913, 9786, 9659, 9532, 9405, 9278, 9151, 9024, 8897, 8770, 8643, 8516, 8389, 8262, 8135, 8008, 7881, 7754, 7627, 7500, 7373, 7246, 7119, 6992, 6865, 6738, 6611, 6484, 6357, 6230, 6103, 5976, 5849, 5722, 5595,
    5468, 5341, 5214, 5087, 4960, 4833, 4706, 4579, 4452, 4325, 4198, 4071, 3944, 3817, 3690, 3563, 3436, 3309, 3182, 3055, 2928, 2801, 2674, 2547, 2420, 2293, 2166, 2039, 1912, 1785, 1658, 1531, 1404, 1277, 1150, 1023, 1151, 1278, 1405, 1532, 1659, 1786, 1913, 2040, 2167, 2294, 2421, 2548, 2675, 2802, 2929, 3056, 3183, 3310, 3437, 3564, 3691, 3818, 3945, 4072, 4199, 4326, 4453, 4580, 4707, 4834, 4961, 5088, 5215, 5342, 5469, 5596, 5723, 5850, 5977, 6104, 6231, 6358, 6485, 6612, 6739, 6866, 6993, 7120, 7247, 7374, 7501, 7628, 7755, 7882, 8009, 8136, 8263, 8390, 8517, 8644, 8771, 8898, 9025, 9152, 9279, 9406, 9533, 9660, 9787, 9914, 10041, 10168, 10295, 10422, 10549, 10676, 10803, 10930, 11057, 11184, 11311, 11438, 11565, 11692, 11819, 11946, 12073, 12200, 12327, 12454, 12581, 12708,
    12835, 12962, 13089, 13216, 13343, 13470, 13597, 13724, 13851, 13978, 14105, 14232, 14359, 14486, 14613, 14740, 14867, 14994, 15121, 15248, 15375, 15502, 15629, 15756, 15883, 16010, 16137, 16264, 16265, 16138, 16011, 15884, 15757, 15630, 15503, 15376, 15249, 15122, 14995, 14868, 14741, 14614, 14487, 14360, 14233, 14106, 13979, 13852, 13725, 13598, 13471, 13344, 13217, 13090, 12963, 12836, 12709, 12582, 12455, 12328, 12201, 12074, 11947, 11820, 11693, 11566, 11439, 11312, 11185, 11058, 10931, 10804, 10677, 10550, 10423, 10296, 10169, 10042, 9915, 9788, 9661, 9534, 9407, 9280, 9153, 9026, 8899, 8772, 8645, 8518, 8391, 8264, 8137, 8010, 7883, 7756, 7629, 7502, 7375, 7248, 7121, 6994, 6867, 6740, 6613, 6486, 6359, 6232, 6105, 5978, 5851, 5724, 5597, 5470, 5343, 5216, 5089, 4962, 4835, 4708, 4581, 4454, 4327, 4200, 4073, 3946, 3819, 3692,
    3565, 3438, 3311, 3184, 3057, 2930, 2803, 2676, 2549, 2422, 2295, 2168, 2041, 1914, 1787, 1660, 1533, 1406, 1279, 1407, 1534, 1661, 1788, 1915, 2042, 2169, 2296, 2423, 2550, 2677, 2804, 2931, 3058, 3185, 3312, 3439, 3566, 3693, 3820, 3947, 4074, 4201, 4328, 4455, 4582, 4709, 4836, 4963, 5090, 5217, 5344, 5471, 5598, 5725, 5852, 5979, 6106, 6233, 6360, 6487, 6614, 6741, 6868, 6995, 7122, 7249, 7376, 7503, 7630, 7757, 7884, 8011, 8138, 8265, 8392, 8519, 8646, 8773, 8900, 9027, 9154, 9281, 9408, 9535, 9662, 9789, 9916, 10043, 10170, 10297, 10424, 10551, 10678, 10805, 10932, 11059, 11186, 11313, 11440, 11567, 11694, 11821, 11948, 12075, 12202, 12329, 12456, 12583, 12710, 12837, 12964, 13091, 13218, 13345, 13472, 13599, 13726, 13853, 13980, 14107, 14234, 14361, 14488, 14615, 14742, 14869, 14996, 15123,
    15250, 15377, 15504, 15631, 15758, 15885, 16012, 16139, 16266, 16267, 16140, 16013, 15886, 15759, 15632, 15505, 15378, 15251, 15124, 14997, 14870, 14743, 14616, 14489, 14362, 14235, 14108, 13981, 13854, 13727, 13600, 13473, 13346, 13219, 13092, 12965, 12838, 12711, 12584, 12457, 12330, 12203, 12076, 11949, 11822, 11695, 11568, 11441, 11314, 11187, 11060, 10933, 10806, 10679, 10552, 10425, 10298, 10171, 10044, 9917, 9790, 9663, 9536, 9409, 9282, 9155, 9028, 8901, 8774, 8647, 8520, 8393, 8266, 8139, 8012, 7885, 7758, 7631, 7504, 7377, 7250, 7123, 6996, 6869, 6742, 6615, 6488, 6361, 6234, 6107, 5980, 5853, 5726, 5599, 5472, 5345, 5218, 5091, 4964, 4837, 4710, 4583, 4456, 4329, 4202, 4075, 3948, 3821, 3694, 3567, 3440, 3313, 3186, 3059, 2932, 2805, 2678, 2551, 2424, 2297, 2170, 2043, 1916, 1789, 1662, 1535, 1663, 1790,
    1917, 2044, 2171, 2298, 2425, 2552, 2679, 2806, 2933, 3060, 3187, 3314, 3441, 3568, 3695, 3822, 3949, 4076, 4203, 4330, 4457, 4584, 4711, 4838, 4965, 5092, 5219, 5346, 5473, 5600, 5727, 5854, 5981, 6108, 6235, 6362, 6489, 6616, 6743, 6870, 6997, 7124, 7251, 7378, 7505, 7632, 7759, 7886, 8013, 8140, 8267, 8394, 8521, 8648, 8775, 8902, 9029, 9156, 9283, 9410, 9537, 9664, 9791, 9918, 10045, 10172, 10299, 10426, 10553, 10680, 10807, 10934, 11061, 11188, 11315, 11442, 11569, 11696, 11823, 11950, 12077, 12204, 12331, 12458, 12585, 12712, 12839, 12966, 13093, 13220, 13347, 13474, 13601, 13728, 13855, 13982, 14109, 14236, 14363, 14490, 14617, 14744, 14871, 14998, 15125, 15252, 15379, 15506, 15633, 15760, 15887, 16014, 16141, 16268, 16269, 16142, 16015, 15888, 15761, 15634, 15507, 15380, 15253, 15126, 14999, 14872, 14745, 14618,
    14491, 14364, 14237, 14110, 13983, 13856, 13729, 13602, 13475, 13348, 13221, 13094, 12967, 12840, 12713, 12586, 12459, 12332, 12205, 12078, 11951, 11824, 11697, 11570, 11443, 11316, 11189, 11062, 10935, 10808, 10681, 10554, 10427, 10300, 10173, 10046, 9919, 9792, 9665, 9538, 9411, 9284, 9157, 9030, 8903, 8776, 8649, 8522, 8395, 8268, 8141, 8014, 7887, 7760, 7633, 7506, 7379, 7252, 7125, 6998, 6871, 6744, 6617, 6490, 6363, 6236, 6109, 5982, 5855, 5728, 5601, 5474, 5347, 5220, 5093, 4966, 4839, 4712, 4585, 4458, 4331, 4204, 4077, 3950, 3823, 3696, 3569, 3442, 3315, 3188, 3061, 2934, 2807, 2680, 2553, 2426, 2299, 2172, 2045, 1918, 1791, 1919, 2046, 2173, 2300, 2427, 2554, 2681, 2808, 2935, 3062, 3189, 3316, 3443, 3570, 3697, 3824, 3951, 4078, 4205, 4332, 4459, 4586, 4713, 4840, 4967, 5094, 5221,
    5348, 5475, 5602, 5729, 5856, 5983, 6110, 6237, 6364, 6491, 6618, 6745, 6872, 6999, 7126, 7253, 7380, 7507, 7634, 7761, 7888, 8015, 8142, 8269, 8396, 8523, 8650, 8777, 8904, 9031, 9158, 9285, 9412, 9539, 9666, 9793, 9920, 10047, 10174, 10301, 10428, 10555, 10682, 10809, 10936, 11063, 11190, 11317, 11444, 11571, 11698, 11825, 11952, 12079, 12206, 12333, 12460, 12587, 12714, 12841, 12968, 13095, 13222, 13349, 13476, 13603, 13730, 13857, 13984, 14111, 14238, 14365, 14492, 14619, 14746, 14873, 15000, 15127, 15254, 15381, 15508, 15635, 15762, 15889, 16016, 16143, 16270, 16271, 16144, 16017, 15890, 15763, 15636, 15509, 15382, 15255, 15128, 15001, 14874, 14747, 14620, 14493, 14366, 14239, 14112, 13985, 13858, 13731, 13604, 13477, 13350, 13223, 13096, 12969, 12842, 12715, 12588, 12461, 12334, 12207, 12080, 11953, 11826, 11699, 11572, 11445, 11318, 11191,
    11064, 10937, 10810, 10683, 10556, 10429, 10302, 10175, 10048, 9921, 9794, 9667, 9540, 9413, 9286, 9159, 9032, 8905, 8778, 8651, 8524, 8397, 8270, 8143, 8016, 7889, 7762, 7635, 7508, 7381, 7254, 7127, 7000, 6873, 6746, 6619, 6492, 6365, 6238, 6111, 5984, 5857, 5730, 5603, 5476, 5349, 5222, 5095, 4968, 4841, 4714, 4587, 4460, 4333, 4206, 4079, 3952, 3825, 3698, 3571, 3444, 3317, 3190, 3063, 2936, 2809, 2682, 2555, 2428, 2301, 2174, 2047, 2175, 2302, 2429, 2556, 2683, 2810, 2937, 3064, 3191, 3318, 3445, 3572, 3699, 3826, 3953, 4080, 4207, 4334, 4461, 4588, 4715, 4842, 4969, 5096, 5223, 5350, 5477, 5604, 5731, 5858, 5985, 6112, 6239, 6366, 6493, 6620, 6747, 6874, 7001, 7128, 7255, 7382, 7509, 7636, 7763, 7890, 8017, 8144, 8271, 8398, 8525, 8652, 8779, 8906, 9033, 9160,
    9287, 9414, 9541, 9668, 9795, 9922, 10049, 10176, 10303, 10430, 10557, 10684, 10811, 10938, 11065, 11192, 11319, 11446, 11573, 11700, 11827, 11954, 12081, 12208, 12335, 12462, 12589, 12716, 12843, 12970, 13097, 13224, 13351, 13478, 13605, 13732, 13859, 13986, 14113, 14240, 14367, 14494, 14621, 14748, 14875, 15002, 15129, 15256, 15383, 15510, 15637, 15764, 15891, 16018, 16145, 16272, 16273, 16146, 16019, 15892, 15765, 15638, 15511, 15384, 15257, 15130, 15003, 14876, 14749, 14622, 14495, 14368, 14241, 14114, 13987, 13860, 13733, 13606, 13479, 13352, 13225, 13098, 12971, 12844, 12717, 12590, 12463, 12336, 12209, 12082, 11955, 11828, 11701, 11574, 11447, 11320, 11193, 11066, 10939, 10812, 10685, 10558, 10431, 10304, 10177, 10050, 9923, 9796, 9669, 9542, 9415, 9288, 9161, 9034, 8907, 8780, 8653, 8526, 8399, 8272, 8145, 8018, 7891, 7764, 7637, 7510, 7383, 7256,
    7129, 7002, 6875, 6748, 6621, 6494, 6367, 6240, 6113, 5986, 5859, 5732, 5605, 5478, 5351, 5224, 5097, 4970, 4843, 4716, 4589, 4462, 4335, 4208, 4081, 3954, 3827, 3700, 3573, 3446, 3319, 3192, 3065, 2938, 2811, 2684, 2557, 2430, 2303, 2431, 2558, 2685, 2812, 2939, 3066, 3193, 3320, 3447, 3574, 3701, 3828, 3955, 4082, 4209, 4336, 4463, 4590, 4717, 4844, 4971, 5098, 5225, 5352, 5479, 5606, 5733, 5860, 5987, 6114, 6241, 6368, 6495, 6622, 6749, 6876, 7003, 7130, 7257, 7384, 7511, 7638, 7765, 7892, 8019, 8146, 8273, 8400, 8527, 8654, 8781, 8908, 9035, 9162, 9289, 9416, 9543, 9670, 9797, 9924, 10051, 10178, 10305, 10432, 10559, 10686, 10813, 10940, 11067, 11194, 11321, 11448, 11575, 11702, 11829, 11956, 12083, 12210, 12337, 12464, 12591, 12718, 12845, 12972, 13099, 13226, 13353, 13480, 13607,
    13734, 13861, 13988, 14115, 14242, 14369, 14496, 14623, 14750, 14877, 15004, 15131, 15258, 15385, 15512, 15639, 15766, 15893, 16020, 16147, 16274, 16275, 16148, 16021, 15894, 15767, 15640, 15513, 15386, 15259, 15132, 15005, 14878, 14751, 14624, 14497, 14370, 14243, 14116, 13989, 13862, 13735, 13608, 13481, 13354, 13227, 13100, 12973, 12846, 12719, 12592, 12465, 12338, 12211, 12084, 11957, 11830, 11703, 11576, 11449, 11322, 11195, 11068, 10941, 10814, 10687, 10560, 10433, 10306, 10179, 10052, 9925, 9798, 9671, 9544, 9417, 9290, 9163, 9036, 8909, 8782, 8655, 8528, 8401, 8274, 8147, 8020, 7893, 7766, 7639, 7512, 7385, 7258, 7131, 7004, 6877, 6750, 6623, 6496, 6369, 6242, 6115, 5988, 5861, 5734, 5607, 5480, 5353, 5226, 5099, 4972, 4845, 4718, 4591, 4464, 4337, 4210, 4083, 3956, 3829, 3702, 3575, 3448, 3321, 3194, 3067, 2940, 2813,
    2686, 2559, 2687, 2814, 2941, 3068, 3195, 3322, 3449, 3576, 3703, 3830, 3957, 4084, 4211, 4338, 4465, 4592, 4719, 4846, 4973, 5100, 5227, 5354, 5481, 5608, 5735, 5862, 5989, 6116, 6243, 6370, 6497, 6624, 6751, 6878, 7005, 7132, 7259, 7386, 7513, 7640, 7767, 7894, 8021, 8148, 8275, 8402, 8529, 8656, 8783, 8910, 9037, 9164, 9291, 9418, 9545, 9672, 9799, 9926, 10053, 10180, 10307, 10434, 10561, 10688, 10815, 10942, 11069, 11196, 11323, 11450, 11577, 11704, 11831, 11958, 12085, 12212, 12339, 12466, 12593, 12720, 12847, 12974, 13101, 13228, 13355, 13482, 13609, 13736, 13863, 13990, 14117, 14244, 14371, 14498, 14625, 14752, 14879, 15006, 15133, 15260, 15387, 15514, 15641, 15768, 15895, 16022, 16149, 16276, 16277, 16150, 16023, 15896, 15769, 15642, 15515, 15388, 15261, 15134, 15007, 14880, 14753, 14626, 14499, 14372, 14245, 14118,
    13991, 13864, 13737, 13610, 13483, 13356, 13229, 13102, 12975, 12848, 12721, 12594, 12467, 12340, 12213, 12086, 11959, 11832, 11705, 11578, 11451, 11324, 11197, 11070, 10943, 10816, 10689, 10562, 10435, 10308, 10181, 10054, 9927, 9800, 9673, 9546, 9419, 9292, 9165, 9038, 8911, 8784, 8657, 8530, 8403, 8276, 8149, 8022, 7895, 7768, 7641, 7514, 7387, 7260, 7133, 7006, 6879, 6752, 6625, 6498, 6371, 6244, 6117, 5990, 5863, 5736, 5609, 5482, 5355, 5228, 5101, 4974, 4847, 4720, 4593, 4466, 4339, 4212, 4085, 3958, 3831, 3704, 3577, 3450, 3323, 3196, 3069, 2942, 2815, 2943, 3070, 3197, 3324, 3451, 3578, 3705, 3832, 3959, 4086, 4213, 4340, 4467, 4594, 4721, 4848, 4975, 5102, 5229, 5356, 5483, 5610, 5737, 5864, 5991, 6118, 6245, 6372, 6499, 6626, 6753, 6880, 7007, 7134, 7261, 7388, 7515, 7642, 7769,
    7896, 8023, 8150, 8277, 8404, 8531, 8658, 8785, 8912, 9039, 9166, 9293, 9420, 9547, 9674, 9801, 9928, 10055, 10182, 10309, 10436, 10563, 10690, 10817, 10944, 11071, 11198, 11325, 11452, 11579, 11706, 11833, 11960, 12087, 12214, 12341, 12468, 12595, 12722, 12849, 12976, 13103, 13230, 13357, 13484, 13611, 13738, 13865, 13992, 14119, 14246, 14373, 14500, 14627, 14754, 14881, 15008, 15135, 15262, 15389, 15516, 15643, 15770, 15897, 16024, 16151, 16278, 16279, 16152, 16025, 15898, 15771, 15644, 15517, 15390, 15263, 15136, 15009, 14882, 14755, 14628, 14501, 14374, 14247, 14120, 13993, 13866, 13739, 13612, 13485, 13358, 13231, 13104, 12977, 12850, 12723, 12596, 12469, 12342, 12215, 12088, 11961, 11834, 11707, 11580, 11453, 11326, 11199, 11072, 10945, 10818, 10691, 10564, 10437, 10310, 10183, 10056, 9929, 9802, 9675, 9548, 9421, 9294, 9167, 9040, 8913, 8786, 8659,
    8532, 8405, 8278, 8151, 8024, 7897, 7770, 7643, 7516, 7389, 7262, 7135, 7008, 6881, 6754, 6627, 6500, 6373, 6246, 6119, 5992, 5865, 5738, 5611, 5484, 5357, 5230, 5103, 4976, 4849, 4722, 4595, 4468, 4341, 4214, 4087, 3960, 3833, 3706, 3579, 3452, 3325, 3198, 3071, 3199, 3326, 3453, 3580, 3707, 3834, 3961, 4088, 4215, 4342, 4469, 4596, 4723, 4850, 4977, 5104, 5231, 5358, 5485, 5612, 5739, 5866, 5993, 6120, 6247, 6374, 6501, 6628, 6755, 6882, 7009, 7136, 7263, 7390, 7517, 7644, 7771, 7898, 8025, 8152, 8279, 8406, 8533, 8660, 8787, 8914, 9041, 9168, 9295, 9422, 9549, 9676, 9803, 9930, 10057, 10184, 10311, 10438, 10565, 10692, 10819, 10946, 11073, 11200, 11327, 11454, 11581, 11708, 11835, 11962, 12089, 12216, 12343, 12470, 12597, 12724, 12851, 12978, 13105, 13232, 13359, 13486, 13613, 13740,
    13867, 13994, 14121, 14248, 14375, 14502, 14629, 14756, 14883, 15010, 15137, 15264, 15391, 15518, 15645, 15772, 15899, 16026, 16153, 16280, 16281, 16154, 16027, 15900, 15773, 15646, 15519, 15392, 15265, 15138, 15011, 14884, 14757, 14630, 14503, 14376, 14249, 14122, 13995, 13868, 13741, 13614, 13487, 13360, 13233, 13106, 12979, 12852, 12725, 12598, 12471, 12344, 12217, 12090, 11963, 11836, 11709, 11582, 11455, 11328, 11201, 11074, 10947, 10820, 10693, 10566, 10439, 10312, 10185, 10058, 9931, 9804, 9677, 9550, 9423, 9296, 9169, 9042, 8915, 8788, 8661, 8534, 8407, 8280, 8153, 8026, 7899, 7772, 7645, 7518, 7391, 7264, 7137, 7010, 6883, 6756, 6629, 6502, 6375, 6248, 6121, 5994, 5867, 5740, 5613, 5486, 5359, 5232, 5105, 4978, 4851, 4724, 4597, 4470, 4343, 4216, 4089, 3962, 3835, 3708, 3581, 3454, 3327, 3455, 3582, 3709, 3836, 3963,
    4090, 4217, 4344, 4471, 4598, 4725, 4852, 4979, 5106, 5233, 5360, 5487, 5614, 5741, 5868, 5995, 6122, 6249, 6376, 6503, 6630, 6757, 6884, 7011, 7138, 7265, 7392, 7519, 7646, 7773, 7900, 8027, 8154, 8281, 8408, 8535, 8662, 8789, 8916, 9043, 9170, 9297, 9424, 9551, 9678, 9805, 9932, 10059, 10186, 10313, 10440, 10567, 10694, 10821, 10948, 11075, 11202, 11329, 11456, 11583, 11710, 11837, 11964, 12091, 12218, 12345, 12472, 12599, 12726, 12853, 12980, 13107, 13234, 13361, 13488, 13615, 13742, 13869, 13996, 14123, 14250, 14377, 14504, 14631, 14758, 14885, 15012, 15139, 15266, 15393, 15520, 15647, 15774, 15901, 16028, 16155, 16282, 16283, 16156, 16029, 15902, 15775, 15648, 15521, 15394, 15267, 15140, 15013, 14886, 14759, 14632, 14505, 14378, 14251, 14124, 13997, 13870, 13743, 13616, 13489, 13362, 13235, 13108, 12981, 12854, 12727, 12600, 12473,
    12346, 12219, 12092, 11965, 11838, 11711, 11584, 11457, 11330, 11203, 11076, 10949, 10822, 10695, 10568, 10441, 10314, 10187, 10060, 9933, 9806, 9679, 9552, 9425, 9298, 9171, 9044, 8917, 8790, 8663, 8536, 8409, 8282, 8155, 8028, 7901, 7774, 7647, 7520, 7393, 7266, 7139, 7012, 6885, 6758, 6631, 6504, 6377, 6250, 6123, 5996, 5869, 5742, 5615, 5488, 5361, 5234, 5107, 4980, 4853, 4726, 4599, 4472, 4345, 4218, 4091, 3964, 3837, 3710, 3583, 3711, 3838, 3965, 4092, 4219, 4346, 4473, 4600, 4727, 4854, 4981, 5108, 5235, 5362, 5489, 5616, 5743, 5870, 5997, 6124, 6251, 6378, 6505, 6632, 6759, 6886, 7013, 7140, 7267, 7394, 7521, 7648, 7775, 7902, 8029, 8156, 8283, 8410, 8537, 8664, 8791, 8918, 9045, 9172, 9299, 9426, 9553, 9680, 9807, 9934, 10061, 10188, 10315, 10442, 10569, 10696, 10823, 10950,
    11077, 11204, 11331, 11458, 11585, 11712, 11839, 11966, 12093, 12220, 12347, 12474, 12601, 12728, 12855, 12982, 13109, 13236, 13363, 13490, 13617, 13744, 13871, 13998, 14125, 14252, 14379, 14506, 14633, 14760, 14887, 15014, 15141, 15268, 15395, 15522, 15649, 15776, 15903, 16030, 16157, 16284, 16285, 16158, 16031, 15904, 15777, 15650, 15523, 15396, 15269, 15142, 15015, 14888, 14761, 14634, 14507, 14380, 14253, 14126, 13999, 13872, 13745, 13618, 13491, 13364, 13237, 13110, 12983, 12856, 12729, 12602, 12475, 12348, 12221, 12094, 11967, 11840, 11713, 11586, 11459, 11332, 11205, 11078, 10951, 10824, 10697, 10570, 10443, 10316, 10189, 10062, 9935, 9808, 9681, 9554, 9427, 9300, 9173, 9046, 8919, 8792, 8665, 8538, 8411, 8284, 8157, 8030, 7903, 7776, 7649, 7522, 7395, 7268, 7141, 7014, 6887, 6760, 6633, 6506, 6379, 6252, 6125, 5998, 5871, 5744, 5617, 5490,
    5363, 5236, 5109, 4982, 4855, 4728, 4601, 4474, 4347, 4220, 4093, 3966, 3839, 3967, 4094, 4221, 4348, 4475, 4602, 4729, 4856, 4983, 5110, 5237, 5364, 5491, 5618, 5745, 5872, 5999, 6126, 6253, 6380, 6507, 6634, 6761, 6888, 7015, 7142, 7269, 7396, 7523, 7650, 7777, 7904, 8031, 8158, 8285, 8412, 8539, 8666, 8793, 8920, 9047, 9174, 9301, 9428, 9555, 9682, 9809, 9936, 10063, 10190, 10317, 10444, 10571, 10698, 10825, 10952, 11079, 11206, 11333, 11460, 11587, 11714, 11841, 11968, 12095, 12222, 12349, 12476, 12603, 12730, 12857, 12984, 13111, 13238, 13365, 13492, 13619, 13746, 13873, 14000, 14127, 14254, 14381, 14508, 14635, 14762, 14889, 15016, 15143, 15270, 15397, 15524, 15651, 15778, 15905, 16032, 16159, 16286, 16287, 16160, 16033, 15906, 15779, 15652, 15525, 15398, 15271, 15144, 15017, 14890, 14763, 14636, 14509, 14382, 14255,
    14128, 14001, 13874, 13747, 13620, 13493, 13366, 13239, 13112, 12985, 12858, 12731, 12604, 12477, 12350, 12223, 12096, 11969, 11842, 11715, 11588, 11461, 11334, 11207, 11080, 10953, 10826, 10699, 10572, 10445, 10318, 10191, 10064, 9937, 9810, 9683, 9556, 9429, 9302, 9175, 9048, 8921, 8794, 8667, 8540, 8413, 8286, 8159, 8032, 7905, 7778, 7651, 7524, 7397, 7270, 7143, 7016, 6889, 6762, 6635, 6508, 6381, 6254, 6127, 6000, 5873, 5746, 5619, 5492, 5365, 5238, 5111, 4984, 4857, 4730, 4603, 4476, 4349, 4222, 4095, 4223, 4350, 4477, 4604, 4731, 4858, 4985, 5112, 5239, 5366, 5493, 5620, 5747, 5874, 6001, 6128, 6255, 6382, 6509, 6636, 6763, 6890, 7017, 7144, 7271, 7398, 7525, 7652, 7779, 7906, 8033, 8160, 8287, 8414, 8541, 8668, 8795, 8922, 9049, 9176, 9303, 9430, 9557, 9684, 9811, 9938, 10065, 10192,
    10319, 10446, 10573, 10700, 10827, 10954, 11081, 11208, 11335, 11462, 11589, 11716, 11843, 11970, 12097, 12224, 12351, 12478, 12605, 12732, 12859, 12986, 13113, 13240, 13367, 13494, 13621, 13748, 13875, 14002, 14129, 14256, 14383, 14510, 14637, 14764, 14891, 15018, 15145, 15272, 15399, 15526, 15653, 15780, 15907, 16034, 16161, 16288, 16289, 16162, 16035, 15908, 15781, 15654, 15527, 15400, 15273, 15146, 15019, 14892, 14765, 14638, 14511, 14384, 14257, 14130, 14003, 13876, 13749, 13622, 13495, 13368, 13241, 13114, 12987, 12860, 12733, 12606, 12479, 12352, 12225, 12098, 11971, 11844, 11717, 11590, 11463, 11336, 11209, 11082, 10955, 10828, 10701, 10574, 10447, 10320, 10193, 10066, 9939, 9812, 9685, 9558, 9431, 9304, 9177, 9050, 8923, 8796, 8669, 8542, 8415, 8288, 8161, 8034, 7907, 7780, 7653, 7526, 7399, 7272, 7145, 7018, 6891, 6764, 6637, 6510, 6383, 6256,
    6129, 6002, 5875, 5748, 5621, 5494, 5367, 5240, 5113, 4986, 4859, 4732, 4605, 4478, 4351, 4479, 4606, 4733, 4860, 4987, 5114, 5241, 5368, 5495, 5622, 5749, 5876, 6003, 6130, 6257, 6384, 6511, 6638, 6765, 6892, 7019, 7146, 7273, 7400, 7527, 7654, 7781, 7908, 8035, 8162, 8289, 8416, 8543, 8670, 8797, 8924, 9051, 9178, 9305, 9432, 9559, 9686, 9813, 9940, 10067, 10194, 10321, 10448, 10575, 10702, 10829, 10956, 11083, 11210, 11337, 11464, 11591, 11718, 11845, 11972, 12099, 12226, 12353, 12480, 12607, 12734, 12861, 12988, 13115, 13242, 13369, 13496, 13623, 13750, 13877, 14004, 14131, 14258, 14385, 14512, 14639, 14766, 14893, 15020, 15147, 15274, 15401, 15528, 15655, 15782, 15909, 16036, 16163, 16290, 16291, 16164, 16037, 15910, 15783, 15656, 15529, 15402, 15275, 15148, 15021, 14894, 14767, 14640, 14513, 14386, 14259, 14132, 14005,
    13878, 13751, 13624, 13497, 13370, 13243, 13116, 12989, 12862, 12735, 12608, 12481, 12354, 12227, 12100, 11973, 11846, 11719, 11592, 11465, 11338, 11211, 11084, 10957, 10830, 10703, 10576, 10449, 10322, 10195, 10068, 9941, 9814, 9687, 9560, 9433, 9306, 9179, 9052, 8925, 8798, 8671, 8544, 8417, 8290, 8163, 8036, 7909, 7782, 7655, 7528, 7401, 7274, 7147, 7020, 6893, 6766, 6639, 6512, 6385, 6258, 6131, 6004, 5877, 5750, 5623, 5496, 5369, 5242, 5115, 4988, 4861, 4734, 4607, 4735, 4862, 4989, 5116, 5243, 5370, 5497, 5624, 5751, 5878, 6005, 6132, 6259, 6386, 6513, 6640, 6767, 6894, 7021, 7148, 7275, 7402, 7529, 7656, 7783, 7910, 8037, 8164, 8291, 8418, 8545, 8672, 8799, 8926, 9053, 9180, 9307, 9434, 9561, 9688, 9815, 9942, 10069, 10196, 10323, 10450, 10577, 10704, 10831, 10958, 11085, 11212, 11339, 11466,
    11593, 11720, 11847, 11974, 12101, 12228, 12355, 12482, 12609, 12736, 12863, 12990, 13117, 13244, 13371, 13498, 13625, 13752, 13879, 14006, 14133, 14260, 14387, 14514, 14641, 14768, 14895, 15022, 15149, 15276, 15403, 15530, 15657, 15784, 15911, 16038, 16165, 16292, 16293, 16166, 16039, 15912, 15785, 15658, 15531, 15404, 15277, 15150, 15023, 14896, 14769, 14642, 14515, 14388, 14261, 14134, 14007, 13880, 13753, 13626, 13499, 13372, 13245, 13118, 12991, 12864, 12737, 12610, 12483, 12356, 12229, 12102, 11975, 11848, 11721, 11594, 11467, 11340, 11213, 11086, 10959, 10832, 10705, 10578, 10451, 10324, 10197, 10070, 9943, 9816, 9689, 9562, 9435, 9308, 9181, 9054, 8927, 8800, 8673, 8546, 8419, 8292, 8165, 8038, 7911, 7784, 7657, 7530, 7403, 7276, 7149, 7022, 6895, 6768, 6641, 6514, 6387, 6260, 6133, 6006, 5879, 5752, 5625, 5498, 5371, 5244, 5117, 4990,
    4863, 4991, 5118, 5245, 5372, 5499, 5626, 5753, 5880, 6007, 6134, 6261, 6388, 6515, 6642, 6769, 6896, 7023, 7150, 7277, 7404, 7531, 7658, 7785, 7912, 8039, 8166, 8293, 8420, 8547, 8674, 8801, 8928, 9055, 9182, 9309, 9436, 9563, 9690, 9817, 9944, 10071, 10198, 10325, 10452, 10579, 10706, 10833, 10960, 11087, 11214, 11341, 11468, 11595, 11722, 11849, 11976, 12103, 12230, 12357, 12484, 12611, 12738, 12865, 12992, 13119, 13246, 13373, 13500, 13627, 13754, 13881, 14008, 14135, 14262, 14389, 14516, 14643, 14770, 14897, 15024, 15151, 15278, 15405, 15532, 15659, 15786, 15913, 16040, 16167, 16294, 16295, 16168, 16041, 15914, 15787, 15660, 15533, 15406, 15279, 15152, 15025, 14898, 14771, 14644, 14517, 14390, 14263, 14136, 14009, 13882, 13755, 13628, 13501, 13374, 13247, 13120, 12993, 12866, 12739, 12612, 12485, 12358, 12231, 12104, 11977, 11850, 11723,
    11596, 11469, 11342, 11215, 11088, 10961, 10834, 10707, 10580, 10453, 10326, 10199, 10072, 9945, 9818, 9691, 9564, 9437, 9310, 9183, 9056, 8929, 8802, 8675, 8548, 8421, 8294, 8167, 8040, 7913, 7786, 7659, 7532, 7405, 7278, 7151, 7024, 6897, 6770, 6643, 6516, 6389, 6262, 6135, 6008, 5881, 5754, 5627, 5500, 5373, 5246, 5119, 5247, 5374, 5501, 5628, 5755, 5882, 6009, 6136, 6263, 6390, 6517, 6644, 6771, 6898, 7025, 7152, 7279, 7406, 7533, 7660, 7787, 7914, 8041, 8168, 8295, 8422, 8549, 8676, 8803, 8930, 9057, 9184, 9311, 9438, 9565, 9692, 9819, 9946, 10073, 10200, 10327, 10454, 10581, 10708, 10835, 10962, 11089, 11216, 11343, 11470, 11597, 11724, 11851, 11978, 12105, 12232, 12359, 12486, 12613, 12740, 12867, 12994, 13121, 13248, 13375, 13502, 13629, 13756, 13883, 14010, 14137, 14264, 14391, 14518, 14645, 14772,
    14899, 15026, 15153, 15280, 15407, 15534, 15661, 15788, 15915, 16042, 16169, 16296, 16297, 16170, 16043, 15916, 15789, 15662, 15535, 15408, 15281, 15154, 15027, 14900, 14773, 14646, 14519, 14392, 14265, 14138, 14011, 13884, 13757, 13630, 13503, 13376, 13249, 13122, 12995, 12868, 12741, 12614, 12487, 12360, 12233, 12106, 11979, 11852, 11725, 11598, 11471, 11344, 11217, 11090, 10963, 10836, 10709, 10582, 10455, 10328, 10201, 10074, 9947, 9820, 9693, 9566, 9439, 9312, 9185, 9058, 8931, 8804, 8677, 8550, 8423, 8296, 8169, 8042, 7915, 7788, 7661, 7534, 7407, 7280, 7153, 7026, 6899, 6772, 6645, 6518, 6391, 6264, 6137, 6010, 5883, 5756, 5629, 5502, 5375, 5503, 5630, 5757, 5884, 6011, 6138, 6265, 6392, 6519, 6646, 6773, 6900, 7027, 7154, 7281, 7408, 7535, 7662, 7789, 7916, 8043, 8170, 8297, 8424, 8551, 8678, 8805, 8932, 9059,
    9186, 9313, 9440, 9567, 9694, 9821, 9948, 10075, 10202, 10329, 10456, 10583, 10710, 10837, 10964, 11091, 11218, 11345, 11472, 11599, 11726, 11853, 11980, 12107, 12234, 12361, 12488, 12615, 12742, 12869, 12996, 13123, 13250, 13377, 13504, 13631, 13758, 13885, 14012, 14139, 14266, 14393, 14520, 14647, 14774, 14901, 15028, 15155, 15282, 15409, 15536, 15663, 15790, 15917, 16044, 16171, 16298, 16299, 16172, 16045, 15918, 15791, 15664, 15537, 15410, 15283, 15156, 15029, 14902, 14775, 14648, 14521, 14394, 14267, 14140, 14013, 13886, 13759, 13632, 13505, 13378, 13251, 13124, 12997, 12870, 12743, 12616, 12489, 12362, 12235, 12108, 11981, 11854, 11727, 11600, 11473, 11346, 11219, 11092, 10965, 10838, 10711, 10584, 10457, 10330, 10203, 10076, 9949, 9822, 9695, 9568, 9441, 9314, 9187, 9060, 8933, 8806, 8679, 8552, 8425, 8298, 8171, 8044, 7917, 7790, 7663, 7536, 7409,
    7282, 7155, 7028, 6901, 6774, 6647, 6520, 6393, 6266, 6139, 6012, 5885, 5758, 5631, 5759, 5886, 6013, 6140, 6267, 6394, 6521, 6648, 6775, 6902, 7029, 7156, 7283, 7410, 7537, 7664, 7791, 7918, 8045, 8172, 8299, 8426, 8553, 8680, 8807, 8934, 9061, 9188, 9315, 9442, 9569, 9696, 9823, 9950, 10077, 10204, 10331, 10458, 10585, 10712, 10839, 10966, 11093, 11220, 11347, 11474, 11601, 11728, 11855, 11982, 12109, 12236, 12363, 12490, 12617, 12744, 12871, 12998, 13125, 13252, 13379, 13506, 13633, 13760, 13887, 14014, 14141, 14268, 14395, 14522, 14649, 14776, 14903, 15030, 15157, 15284, 15411, 15538, 15665, 15792, 15919, 16046, 16173, 16300, 16301, 16174, 16047, 15920, 15793, 15666, 15539, 15412, 15285, 15158, 15031, 14904, 14777, 14650, 14523, 14396, 14269, 14142, 14015, 13888, 13761, 13634, 13507, 13380, 13253, 13126, 12999, 12872, 12745, 12618,
    12491, 12364, 12237, 12110, 11983, 11856, 11729, 11602, 11475, 11348, 11221, 11094, 10967, 10840, 10713, 10586, 10459, 10332, 10205, 10078, 9951, 9824, 9697, 9570, 9443, 9316, 9189, 9062, 8935, 8808, 8681, 8554, 8427, 8300, 8173, 8046, 7919, 7792, 7665, 7538, 7411, 7284, 7157, 7030, 6903, 6776, 6649, 6522, 6395, 6268, 6141, 6014, 5887, 6015, 6142, 6269, 6396, 6523, 6650, 6777, 6904, 7031, 7158, 7285, 7412, 7539, 7666, 7793, 7920, 8047, 8174, 8301, 8428, 8555, 8682, 8809, 8936, 9063, 9190, 9317, 9444, 9571, 9698, 9825, 9952, 10079, 10206, 10333, 10460, 10587, 10714, 10841, 10968, 11095, 11222, 11349, 11476, 11603, 11730, 11857, 11984, 12111, 12238, 12365, 12492, 12619, 12746, 12873, 13000, 13127, 13254, 13381, 13508, 13635, 13762, 13889, 14016, 14143, 14270, 14397, 14524, 14651, 14778, 14905, 15032, 15159, 15286, 15413,
    15540, 15667, 15794, 15921, 16048, 16175, 16302, 16303, 16176, 16049, 15922, 15795, 15668, 15541, 15414, 15287, 15160, 15033, 14906, 14779, 14652, 14525, 14398, 14271, 14144, 14017, 13890, 13763, 13636, 13509, 13382, 13255, 13128, 13001, 12874, 12747, 12620, 12493, 12366, 12239, 12112, 11985, 11858, 11731, 11604, 11477, 11350, 11223, 11096, 10969, 10842, 10715, 10588, 10461, 10334, 10207, 10080, 9953, 9826, 9699, 9572, 9445, 9318, 9191, 9064, 8937, 8810, 8683, 8556, 8429, 8302, 8175, 8048, 7921, 7794, 7667, 7540, 7413, 7286, 7159, 7032, 6905, 6778, 6651, 6524, 6397, 6270, 6143, 6271, 6398, 6525, 6652, 6779, 6906, 7033, 7160, 7287, 7414, 7541, 7668, 7795, 7922, 8049, 8176, 8303, 8430, 8557, 8684, 8811, 8938, 9065, 9192, 9319, 9446, 9573, 9700, 9827, 9954, 10081, 10208, 10335, 10462, 10589, 10716, 10843, 10970, 11097, 11224,
    11351, 11478, 11605, 11732, 11859, 11986, 12113, 12240, 12367, 12494, 12621, 12748, 12875, 13002, 13129, 13256, 13383, 13510, 13637, 13764, 13891, 14018, 14145, 14272, 14399, 14526, 14653, 14780, 14907, 15034, 15161, 15288, 15415, 15542, 15669, 15796, 15923, 16050, 16177, 16304, 16305, 16178, 16051, 15924, 15797, 15670, 15543, 15416, 15289, 15162, 15035, 14908, 14781, 14654, 14527, 14400, 14273, 14146, 14019, 13892, 13765, 13638, 13511, 13384, 13257, 13130, 13003, 12876, 12749, 12622, 12495, 12368, 12241, 12114, 11987, 11860, 11733, 11606, 11479, 11352, 11225, 11098, 10971, 10844, 10717, 10590, 10463, 10336, 10209, 10082, 9955, 9828, 9701, 9574, 9447, 9320, 9193, 9066, 8939, 8812, 8685, 8558, 8431, 8304, 8177, 8050, 7923, 7796, 7669, 7542, 7415, 7288, 7161, 7034, 6907, 6780, 6653, 6526, 6399, 6527, 6654, 6781, 6908, 7035, 7162, 7289, 7416, 7543,
    7670, 7797, 7924, 8051, 8178, 8305, 8432, 8559, 8686, 8813, 8940, 9067, 9194, 9321, 9448, 9575, 9702, 9829, 9956, 10083, 10210, 10337, 10464, 10591, 10718, 10845, 10972, 11099, 11226, 11353, 11480, 11607, 11734, 11861, 11988, 12115, 12242, 12369, 12496, 12623, 12750, 12877, 13004, 13131, 13258, 13385, 13512, 13639, 13766, 13893, 14020, 14147, 14274, 14401, 14528, 14655, 14782, 14909, 15036, 15163, 15290, 15417, 15544, 15671, 15798, 15925, 16052, 16179, 16306, 16307, 16180, 16053, 15926, 15799, 15672, 15545, 15418, 15291, 15164, 15037, 14910, 14783, 14656, 14529, 14402, 14275, 14148, 14021, 13894, 13767, 13640, 13513, 13386, 13259, 13132, 13005, 12878, 12751, 12624, 12497, 12370, 12243, 12116, 11989, 11862, 11735, 11608, 11481, 11354, 11227, 11100, 10973, 10846, 10719, 10592, 10465, 10338, 10211, 10084, 9957, 9830, 9703, 9576, 9449, 9322, 9195, 9068, 8941,
    8814, 8687, 8560, 8433, 8306, 8179, 8052, 7925, 7798, 7671, 7544, 7417, 7290, 7163, 7036, 6909, 6782, 6655, 6783, 6910, 7037, 7164, 7291, 7418, 7545, 7672, 7799, 7926, 8053, 8180, 8307, 8434, 8561, 8688, 8815, 8942, 9069, 9196, 9323, 9450, 9577, 9704, 9831, 9958, 10085, 10212, 10339, 10466, 10593, 10720, 10847, 10974, 11101, 11228, 11355, 11482, 11609, 11736, 11863, 11990, 12117, 12244, 12371, 12498, 12625, 12752, 12879, 13006, 13133, 13260, 13387, 13514, 13641, 13768, 13895, 14022, 14149, 14276, 14403, 14530, 14657, 14784, 14911, 15038, 15165, 15292, 15419, 15546, 15673, 15800, 15927, 16054, 16181, 16308, 16309, 16182, 16055, 15928, 15801, 15674, 15547, 15420, 15293, 15166, 15039, 14912, 14785, 14658, 14531, 14404, 14277, 14150, 14023, 13896, 13769, 13642, 13515, 13388, 13261, 13134, 13007, 12880, 12753, 12626, 12499, 12372, 12245, 12118,
    11991, 11864, 11737, 11610, 11483, 11356, 11229, 11102, 10975, 10848, 10721, 10594, 10467, 10340, 10213, 10086, 9959, 9832, 9705, 9578, 9451, 9324, 9197, 9070, 8943, 8816, 8689, 8562, 8435, 8308, 8181, 8054, 7927, 7800, 7673, 7546, 7419, 7292, 7165, 7038, 6911, 7039, 7166, 7293, 7420, 7547, 7674, 7801, 7928, 8055, 8182, 8309, 8436, 8563, 8690, 8817, 8944, 9071, 9198, 9325, 9452, 9579, 9706, 9833, 9960, 10087, 10214, 10341, 10468, 10595, 10722, 10849, 10976, 11103, 11230, 11357, 11484, 11611, 11738, 11865, 11992, 12119, 12246, 12373, 12500, 12627, 12754, 12881, 13008, 13135, 13262, 13389, 13516, 13643, 13770, 13897, 14024, 14151, 14278, 14405, 14532, 14659, 14786, 14913, 15040, 15167, 15294, 15421, 15548, 15675, 15802, 15929, 16056, 16183, 16310, 16311, 16184, 16057, 15930, 15803, 15676, 15549, 15422, 15295, 15168, 15041, 14914, 14787,
    14660, 14533, 14406, 14279, 14152, 14025, 13898, 13771, 13644, 13517, 13390, 13263, 13136, 13009, 12882, 12755, 12628, 12501, 12374, 12247, 12120, 11993, 11866, 11739, 11612, 11485, 11358, 11231, 11104, 10977, 10850, 10723, 10596, 10469, 10342, 10215, 10088, 9961, 9834, 9707, 9580, 9453, 9326, 9199, 9072, 8945, 8818, 8691, 8564, 8437, 8310, 8183, 8056, 7929, 7802, 7675, 7548, 7421, 7294, 7167, 7295, 7422, 7549, 7676, 7803, 7930, 8057, 8184, 8311, 8438, 8565, 8692, 8819, 8946, 9073, 9200, 9327, 9454, 9581, 9708, 9835, 9962, 10089, 10216, 10343, 10470, 10597, 10724, 10851, 10978, 11105, 11232, 11359, 11486, 11613, 11740, 11867, 11994, 12121, 12248, 12375, 12502, 12629, 12756, 12883, 13010, 13137, 13264, 13391, 13518, 13645, 13772, 13899, 14026, 14153, 14280, 14407, 14534, 14661, 14788, 14915, 15042, 15169, 15296, 15423, 15550, 15677, 15804,
    15931, 16058, 16185, 16312, 16313, 16186, 16059, 15932, 15805, 15678, 15551, 15424, 15297, 15170, 15043, 14916, 14789, 14662, 14535, 14408, 14281, 14154, 14027, 13900, 13773, 13646, 13519, 13392, 13265, 13138, 13011, 12884, 12757, 12630, 12503, 12376, 12249, 12122, 11995, 11868, 11741, 11614, 11487, 11360, 11233, 11106, 10979, 10852, 10725, 10598, 10471, 10344, 10217, 10090, 9963, 9836, 9709, 9582, 9455, 9328, 9201, 9074, 8947, 8820, 8693, 8566, 8439, 8312, 8185, 8058, 7931, 7804, 7677, 7550, 7423, 7551, 7678, 7805, 7932, 8059, 8186, 8313, 8440, 8567, 8694, 8821, 8948, 9075, 9202, 9329, 9456, 9583, 9710, 9837, 9964, 10091, 10218, 10345, 10472, 10599, 10726, 10853, 10980, 11107, 11234, 11361, 11488, 11615, 11742, 11869, 11996, 12123, 12250, 12377, 12504, 12631, 12758, 12885, 13012, 13139, 13266, 13393, 13520, 13647, 13774, 13901, 14028, 14155,
    14282, 14409, 14536, 14663, 14790, 14917, 15044, 15171, 15298, 15425, 15552, 15679, 15806, 15933, 16060, 16187, 16314, 16315, 16188, 16061, 15934, 15807, 15680, 15553, 15426, 15299, 15172, 15045, 14918, 14791, 14664, 14537, 14410, 14283, 14156, 14029, 13902, 13775, 13648, 13521, 13394, 13267, 13140, 13013, 12886, 12759, 12632, 12505, 12378, 12251, 12124, 11997, 11870, 11743, 11616, 11489, 11362, 11235, 11108, 10981, 10854, 10727, 10600, 10473, 10346, 10219, 10092, 9965, 9838, 9711, 9584, 9457, 9330, 9203, 9076, 8949, 8822, 8695, 8568, 8441, 8314, 8187, 8060, 7933, 7806, 7679, 7807, 7934, 8061, 8188, 8315, 8442, 8569, 8696, 8823, 8950, 9077, 9204, 9331, 9458, 9585, 9712, 9839, 9966, 10093, 10220, 10347, 10474, 10601, 10728, 10855, 10982, 11109, 11236, 11363, 11490, 11617, 11744, 11871, 11998, 12125, 12252, 12379, 12506, 12633, 12760, 12887, 13014,
    13141, 13268, 13395, 13522, 13649, 13776, 13903, 14030, 14157, 14284, 14411, 14538, 14665, 14792, 14919, 15046, 15173, 15300, 15427, 15554, 15681, 15808, 15935, 16062, 16189, 16316, 16317, 16190, 16063, 15936, 15809, 15682, 15555, 15428, 15301, 15174, 15047, 14920, 14793, 14666, 14539, 14412, 14285, 14158, 14031, 13904, 13777, 13650, 13523, 13396, 13269, 13142, 13015, 12888, 12761, 12634, 12507, 12380, 12253, 12126, 11999, 11872, 11745, 11618, 11491, 11364, 11237, 11110, 10983, 10856, 10729, 10602, 10475, 10348, 10221, 10094, 9967, 9840, 9713, 9586, 9459, 9332, 9205, 9078, 8951, 8824, 8697, 8570, 8443, 8316, 8189, 8062, 7935, 8063, 8190, 8317, 8444, 8571, 8698, 8825, 8952, 9079, 9206, 9333, 9460, 9587, 9714, 9841, 9968, 10095, 10222, 10349, 10476, 10603, 10730, 10857, 10984, 11111, 11238, 11365, 11492, 11619, 11746, 11873, 12000, 12127, 12254, 12381,
    12508, 12635, 12762, 12889, 13016, 13143, 13270, 13397, 13524, 13651, 13778, 13905, 14032, 14159, 14286, 14413, 14540, 14667, 14794, 14921, 15048, 15175, 15302, 15429, 15556, 15683, 15810, 15937, 16064, 16191, 16318, 16319, 16192, 16065, 15938, 15811, 15684, 15557, 15430, 15303, 15176, 15049, 14922, 14795, 14668, 14541, 14414, 14287, 14160, 14033, 13906, 13779, 13652, 13525, 13398, 13271, 13144, 13017, 12890, 12763, 12636, 12509, 12382, 12255, 12128, 12001, 11874, 11747, 11620, 11493, 11366, 11239, 11112, 10985, 10858, 10731, 10604, 10477, 10350, 10223, 10096, 9969, 9842, 9715, 9588, 9461, 9334, 9207, 9080, 8953, 8826, 8699, 8572, 8445, 8318, 8191, 8319, 8446, 8573, 8700, 8827, 8954, 9081, 9208, 9335, 9462, 9589, 9716, 9843, 9970, 10097, 10224, 10351, 10478, 10605, 10732, 10859, 10986, 11113, 11240, 11367, 11494, 11621, 11748, 11875, 12002, 12129, 12256,
    12383, 12510, 12637, 12764, 12891, 13018, 13145, 13272, 13399, 13526, 13653, 13780, 13907, 14034, 14161, 14288, 14415, 14542, 14669, 14796, 14923, 15050, 15177, 15304, 15431, 15558, 15685, 15812, 15939, 16066, 16193, 16320, 16321, 16194, 16067, 15940, 15813, 15686, 15559, 15432, 15305, 15178, 15051, 14924, 14797, 14670, 14543, 14416, 14289, 14162, 14035, 13908, 13781, 13654, 13527, 13400, 13273, 13146, 13019, 12892, 12765, 12638, 12511, 12384, 12257, 12130, 12003, 11876, 11749, 11622, 11495, 11368, 11241, 11114, 10987, 10860, 10733, 10606, 10479, 10352, 10225, 10098, 9971, 9844, 9717, 9590, 9463, 9336, 9209, 9082, 8955, 8828, 8701, 8574, 8447, 8575, 8702, 8829, 8956, 9083, 9210, 9337, 9464, 9591, 9718, 9845, 9972, 10099, 10226, 10353, 10480, 10607, 10734, 10861, 10988, 11115, 11242, 11369, 11496, 11623, 11750, 11877, 12004, 12131, 12258, 12385, 12512, 12639,
    12766, 12893, 13020, 13147, 13274, 13401, 13528, 13655, 13782, 13909, 14036, 14163, 14290, 14417, 14544, 14671, 14798, 14925, 15052, 15179, 15306, 15433, 15560, 15687, 15814, 15941, 16068, 16195, 16322, 16323, 16196, 16069, 15942, 15815, 15688, 15561, 15434, 15307, 15180, 15053, 14926, 14799, 14672, 14545, 14418, 14291, 14164, 14037, 13910, 13783, 13656, 13529, 13402, 13275, 13148, 13021, 12894, 12767, 12640, 12513, 12386, 12259, 12132, 12005, 11878, 11751, 11624, 11497, 11370, 11243, 11116, 10989, 10862, 10735, 10608, 10481, 10354, 10227, 10100, 9973, 9846, 9719, 9592, 9465, 9338, 9211, 9084, 8957, 8830, 8703, 8831, 8958, 9085, 9212, 9339, 9466, 9593, 9720, 9847, 9974, 10101, 10228, 10355, 10482, 10609, 10736, 10863, 10990, 11117, 11244, 11371, 11498, 11625, 11752, 11879, 12006, 12133, 12260, 12387, 12514, 12641, 12768, 12895, 13022, 13149, 13276, 13403, 13530,
    13657, 13784, 13911, 14038, 14165, 14292, 14419, 14546, 14673, 14800, 14927, 15054, 15181, 15308, 15435, 15562, 15689, 15816, 15943, 16070, 16197, 16324, 16325, 16198, 16071, 15944, 15817, 15690, 15563, 15436, 15309, 15182, 15055, 14928, 14801, 14674, 14547, 14420, 14293, 14166, 14039, 13912, 13785, 13658, 13531, 13404, 13277, 13150, 13023, 12896, 12769, 12642, 12515, 12388, 12261, 12134, 12007, 11880, 11753, 11626, 11499, 11372, 11245, 11118, 10991, 10864, 10737, 10610, 10483, 10356, 10229, 10102, 9975, 9848, 9721, 9594, 9467, 9340, 9213, 9086, 8959, 9087, 9214, 9341, 9468, 9595, 9722, 9849, 9976, 10103, 10230, 10357, 10484, 10611, 10738, 10865, 10992, 11119, 11246, 11373, 11500, 11627, 11754, 11881, 12008, 12135, 12262, 12389, 12516, 12643, 12770, 12897, 13024, 13151, 13278, 13405, 13532, 13659, 13786, 13913, 14040, 14167, 14294, 14421, 14548, 14675, 14802, 14929,
    15056, 15183, 15310, 15437, 15564, 15691, 15818, 15945, 16072, 16199, 16326, 16327, 16200, 16073, 15946, 15819, 15692, 15565, 15438, 15311, 15184, 15057, 14930, 14803, 14676, 14549, 14422, 14295, 14168, 14041, 13914, 13787, 13660, 13533, 13406, 13279, 13152, 13025, 12898, 12771, 12644, 12517, 12390, 12263, 12136, 12009, 11882, 11755, 11628, 11501, 11374, 11247, 11120, 10993, 10866, 10739, 10612, 10485, 10358, 10231, 10104, 9977, 9850, 9723, 9596, 9469, 9342, 9215, 9343, 9470, 9597, 9724, 9851, 9978, 10105, 10232, 10359, 10486, 10613, 10740, 10867, 10994, 11121, 11248, 11375, 11502, 11629, 11756, 11883, 12010, 12137, 12264, 12391, 12518, 12645, 12772, 12899, 13026, 13153, 13280, 13407, 13534, 13661, 13788, 13915, 14042, 14169, 14296, 14423, 14550, 14677, 14804, 14931, 15058, 15185, 15312, 15439, 15566, 15693, 15820, 15947, 16074, 16201, 16328, 16329, 16202, 16075, 15948,
    15821, 15694, 15567, 15440, 15313, 15186, 15059, 14932, 14805, 14678, 14551, 14424, 14297, 14170, 14043, 13916, 13789, 13662, 13535, 13408, 13281, 13154, 13027, 12900, 12773, 12646, 12519, 12392, 12265, 12138, 12011, 11884, 11757, 11630, 11503, 11376, 11249, 11122, 10995, 10868, 10741, 10614, 10487, 10360, 10233, 10106, 9979, 9852, 9725, 9598, 9471, 9599, 9726, 9853, 9980, 10107, 10234, 10361, 10488, 10615, 10742, 10869, 10996, 11123, 11250, 11377, 11504, 11631, 11758, 11885, 12012, 12139, 12266, 12393, 12520, 12647, 12774, 12901, 13028, 13155, 13282, 13409, 13536, 13663, 13790, 13917, 14044, 14171, 14298, 14425, 14552, 14679, 14806, 14933, 15060, 15187, 15314, 15441, 15568, 15695, 15822, 15949, 16076, 16203, 16330, 16331, 16204, 16077, 15950, 15823, 15696, 15569, 15442, 15315, 15188, 15061, 14934, 14807, 14680, 14553, 14426, 14299, 14172, 14045, 13918, 13791, 13664, 13537,
    13410, 13283, 13156, 13029, 12902, 12775, 12648, 12521, 12394, 12267, 12140, 12013, 11886, 11759, 11632, 11505, 11378, 11251, 11124, 10997, 10870, 10743, 10616, 10489, 10362, 10235, 10108, 9981, 9854, 9727, 9855, 9982, 10109, 10236, 10363, 10490, 10617, 10744, 10871, 10998, 11125, 11252, 11379, 11506, 11633, 11760, 11887, 12014, 12141, 12268, 12395, 12522, 12649, 12776, 12903, 13030, 13157, 13284, 13411, 13538, 13665, 13792, 13919, 14046, 14173, 14300, 14427, 14554, 14681, 14808, 14935, 15062, 15189, 15316, 15443, 15570, 15697, 15824, 15951, 16078, 16205, 16332, 16333, 16206, 16079, 15952, 15825, 15698, 15571, 15444, 15317, 15190, 15063, 14936, 14809, 14682, 14555, 14428, 14301, 14174, 14047, 13920, 13793, 13666, 13539, 13412, 13285, 13158, 13031, 12904, 12777, 12650, 12523, 12396, 12269, 12142, 12015, 11888, 11761, 11634, 11507, 11380, 11253, 11126, 10999, 10872, 10745, 10618,
    10491, 10364, 10237, 10110, 9983, 10111, 10238, 10365, 10492, 10619, 10746, 10873, 11000, 11127, 11254, 11381, 11508, 11635, 11762, 11889, 12016, 12143, 12270, 12397, 12524, 12651, 12778, 12905, 13032, 13159, 13286, 13413, 13540, 13667, 13794, 13921, 14048, 14175, 14302, 14429, 14556, 14683, 14810, 14937, 15064, 15191, 15318, 15445, 15572, 15699, 15826, 15953, 16080, 16207, 16334, 16335, 16208, 16081, 15954, 15827, 15700, 15573, 15446, 15319, 15192, 15065, 14938, 14811, 14684, 14557, 14430, 14303, 14176, 14049, 13922, 13795, 13668, 13541, 13414, 13287, 13160, 13033, 12906, 12779, 12652, 12525, 12398, 12271, 12144, 12017, 11890, 11763, 11636, 11509, 11382, 11255, 11128, 11001, 10874, 10747, 10620, 10493, 10366, 10239, 10367, 10494, 10621, 10748, 10875, 11002, 11129, 11256, 11383, 11510, 11637, 11764, 11891, 12018, 12145, 12272, 12399, 12526, 12653, 12780, 12907, 13034, 13161, 13288,
    13415, 13542, 13669, 13796, 13923, 14050, 14177, 14304, 14431, 14558, 14685, 14812, 14939, 15066, 15193, 15320, 15447, 15574, 15701, 15828, 15955, 16082, 16209, 16336, 16337, 16210, 16083, 15956, 15829, 15702, 15575, 15448, 15321, 15194, 15067, 14940, 14813, 14686, 14559, 14432, 14305, 14178, 14051, 13924, 13797, 13670, 13543, 13416, 13289, 13162, 13035, 12908, 12781, 12654, 12527, 12400, 12273, 12146, 12019, 11892, 11765, 11638, 11511, 11384, 11257, 11130, 11003, 10876, 10749, 10622, 10495, 10623, 10750, 10877, 11004, 11131, 11258, 11385, 11512, 11639, 11766, 11893, 12020, 12147, 12274, 12401, 12528, 12655, 12782, 12909, 13036, 13163, 13290, 13417, 13544, 13671, 13798, 13925, 14052, 14179, 14306, 14433, 14560, 14687, 14814, 14941, 15068, 15195, 15322, 15449, 15576, 15703, 15830, 15957, 16084, 16211, 16338, 16339, 16212, 16085, 15958, 15831, 15704, 15577, 15450, 15323, 15196, 15069,
    14942, 14815, 14688, 14561, 14434, 14307, 14180, 14053, 13926, 13799, 13672, 13545, 13418, 13291, 13164, 13037, 12910, 12783, 12656, 12529, 12402, 12275, 12148, 12021, 11894, 11767, 11640, 11513, 11386, 11259, 11132, 11005, 10878, 10751, 10879, 11006, 11133, 11260, 11387, 11514, 11641, 11768, 11895, 12022, 12149, 12276, 12403, 12530, 12657, 12784, 12911, 13038, 13165, 13292, 13419, 13546, 13673, 13800, 13927, 14054, 14181, 14308, 14435, 14562, 14689, 14816, 14943, 15070, 15197, 15324, 15451, 15578, 15705, 15832, 15959, 16086, 16213, 16340, 16341, 16214, 16087, 15960, 15833, 15706, 15579, 15452, 15325, 15198, 15071, 14944, 14817, 14690, 14563, 14436, 14309, 14182, 14055, 13928, 13801, 13674, 13547, 13420, 13293, 13166, 13039, 12912, 12785, 12658, 12531, 12404, 12277, 12150, 12023, 11896, 11769, 11642, 11515, 11388, 11261, 11134, 11007, 11135, 11262, 11389, 11516, 11643, 11770, 11897,
    12024, 12151, 12278, 12405, 12532, 12659, 12786, 12913, 13040, 13167, 13294, 13421, 13548, 13675, 13802, 13929, 14056, 14183, 14310, 14437, 14564, 14691, 14818, 14945, 15072, 15199, 15326, 15453, 15580, 15707, 15834, 15961, 16088, 16215, 16342, 16343, 16216, 16089, 15962, 15835, 15708, 15581, 15454, 15327, 15200, 15073, 14946, 14819, 14692, 14565, 14438, 14311, 14184, 14057, 13930, 13803, 13676, 13549, 13422, 13295, 13168, 13041, 12914, 12787, 12660, 12533, 12406, 12279, 12152, 12025, 11898, 11771, 11644, 11517, 11390, 11263, 11391, 11518, 11645, 11772, 11899, 12026, 12153, 12280, 12407, 12534, 12661, 12788, 12915, 13042, 13169, 13296, 13423, 13550, 13677, 13804, 13931, 14058, 14185, 14312, 14439, 14566, 14693, 14820, 14947, 15074, 15201, 15328, 15455, 15582, 15709, 15836, 15963, 16090, 16217, 16344, 16345, 16218, 16091, 15964, 15837, 15710, 15583, 15456, 15329, 15202, 15075, 14948,
    14821, 14694, 14567, 14440, 14313, 14186, 14059, 13932, 13805, 13678, 13551, 13424, 13297, 13170, 13043, 12916, 12789, 12662, 12535, 12408, 12281, 12154, 12027, 11900, 11773, 11646, 11519, 11647, 11774, 11901, 12028, 12155, 12282, 12409, 12536, 12663, 12790, 12917, 13044, 13171, 13298, 13425, 13552, 13679, 13806, 13933, 14060, 14187, 14314, 14441, 14568, 14695, 14822, 14949, 15076, 15203, 15330, 15457, 15584, 15711, 15838, 15965, 16092, 16219, 16346, 16347, 16220, 16093, 15966, 15839, 15712, 15585, 15458, 15331, 15204, 15077, 14950, 14823, 14696, 14569, 14442, 14315, 14188, 14061, 13934, 13807, 13680, 13553, 13426, 13299, 13172, 13045, 12918, 12791, 12664, 12537, 12410, 12283, 12156, 12029, 11902, 11775, 11903, 12030, 12157, 12284, 12411, 12538, 12665, 12792, 12919, 13046, 13173, 13300, 13427, 13554, 13681, 13808, 13935, 14062, 14189, 14316, 14443, 14570, 14697, 14824, 14951, 15078,
    15205, 15332, 15459, 15586, 15713, 15840, 15967, 16094, 16221, 16348, 16349, 16222, 16095, 15968, 15841, 15714, 15587, 15460, 15333, 15206, 15079, 14952, 14825, 14698, 14571, 14444, 14317, 14190, 14063, 13936, 13809, 13682, 13555, 13428, 13301, 13174, 13047, 12920, 12793, 12666, 12539, 12412, 12285, 12158, 12031, 12159, 12286, 12413, 12540, 12667, 12794, 12921, 13048, 13175, 13302, 13429, 13556, 13683, 13810, 13937, 14064, 14191, 14318, 14445, 14572, 14699, 14826, 14953, 15080, 15207, 15334, 15461, 15588, 15715, 15842, 15969, 16096, 16223, 16350, 16351, 16224, 16097, 15970, 15843, 15716, 15589, 15462, 15335, 15208, 15081, 14954, 14827, 14700, 14573, 14446, 14319, 14192, 14065, 13938, 13811, 13684, 13557, 13430, 13303, 13176, 13049, 12922, 12795, 12668, 12541, 12414, 12287, 12415, 12542, 12669, 12796, 12923, 13050, 13177, 13304, 13431, 13558, 13685, 13812, 13939, 14066, 14193, 14320,
    14447, 14574, 14701, 14828, 14955, 15082, 15209, 15336, 15463, 15590, 15717, 15844, 15971, 16098, 16225, 16352, 16353, 16226, 16099, 15972, 15845, 15718, 15591, 15464, 15337, 15210, 15083, 14956, 14829, 14702, 14575, 14448, 14321, 14194, 14067, 13940, 13813, 13686, 13559, 13432, 13305, 13178, 13051, 12924, 12797, 12670, 12543, 12671, 12798, 12925, 13052, 13179, 13306, 13433, 13560, 13687, 13814, 13941, 14068, 14195, 14322, 14449, 14576, 14703, 14830, 14957, 15084, 15211, 15338, 15465, 15592, 15719, 15846, 15973, 16100, 16227, 16354, 16355, 16228, 16101, 15974, 15847, 15720, 15593, 15466, 15339, 15212, 15085, 14958, 14831, 14704, 14577, 14450, 14323, 14196, 14069, 13942, 13815, 13688, 13561, 13434, 13307, 13180, 13053, 12926, 12799, 12927, 13054, 13181, 13308, 13435, 13562, 13689, 13816, 13943, 14070, 14197, 14324, 14451, 14578, 14705, 14832, 14959, 15086, 15213, 15340, 15467, 15594,
    15721, 15848, 15975, 16102, 16229, 16356, 16357, 16230, 16103, 15976, 15849, 15722, 15595, 15468, 15341, 15214, 15087, 14960, 14833, 14706, 14579, 14452, 14325, 14198, 14071, 13944, 13817, 13690, 13563, 13436, 13309, 13182, 13055, 13183, 13310, 13437, 13564, 13691, 13818, 13945, 14072, 14199, 14326, 14453, 14580, 14707, 14834, 14961, 15088, 15215, 15342, 15469, 15596, 15723, 15850, 15977, 16104, 16231, 16358, 16359, 16232, 16105, 15978, 15851, 15724, 15597, 15470, 15343, 15216, 15089, 14962, 14835, 14708, 14581, 14454, 14327, 14200, 14073, 13946, 13819, 13692, 13565, 13438, 13311, 13439, 13566, 13693, 13820, 13947, 14074, 14201, 14328, 14455, 14582, 14709, 14836, 14963, 15090, 15217, 15344, 15471, 15598, 15725, 15852, 15979, 16106, 16233, 16360, 16361, 16234, 16107, 15980, 15853, 15726, 15599, 15472, 15345, 15218, 15091, 14964, 14837, 14710, 14583, 14456, 14329, 14202, 14075, 13948,
    13821, 13694, 13567, 13695, 13822, 13949, 14076, 14203, 14330, 14457, 14584, 14711, 14838, 14965, 15092, 15219, 15346, 15473, 15600, 15727, 15854, 15981, 16108, 16235, 16362, 16363, 16236, 16109, 15982, 15855, 15728, 15601, 15474, 15347, 15220, 15093, 14966, 14839, 14712, 14585, 14458, 14331, 14204, 14077, 13950, 13823, 13951, 14078, 14205, 14332, 14459, 14586, 14713, 14840, 14967, 15094, 15221, 15348, 15475, 15602, 15729, 15856, 15983, 16110, 16237, 16364, 16365, 16238, 16111, 15984, 15857, 15730, 15603, 15476, 15349, 15222, 15095, 14968, 14841, 14714, 14587, 14460, 14333, 14206, 14079, 14207, 14334, 14461, 14588, 14715, 14842, 14969, 15096, 15223, 15350, 15477, 15604, 15731, 15858, 15985, 16112, 16239, 16366, 16367, 16240, 16113, 15986, 15859, 15732, 15605, 15478, 15351, 15224, 15097, 14970, 14843, 14716, 14589, 14462, 14335, 14463, 14590, 14717, 14844, 14971, 15098, 15225, 15352,
    15479, 15606, 15733, 15860, 15987, 16114, 16241, 16368, 16369, 16242, 16115, 15988, 15861, 15734, 15607, 15480, 15353, 15226, 15099, 14972, 14845, 14718, 14591, 14719, 14846, 14973, 15100, 15227, 15354, 15481, 15608, 15735, 15862, 15989, 16116, 16243, 16370, 16371, 16244, 16117, 15990, 15863, 15736, 15609, 15482, 15355, 15228, 15101, 14974, 14847, 14975, 15102, 15229, 15356, 15483, 15610, 15737, 15864, 15991, 16118, 16245, 16372, 16373, 16246, 16119, 15992, 15865, 15738, 15611, 15484, 15357, 15230, 15103, 15231, 15358, 15485, 15612, 15739, 15866, 15993, 16120, 16247, 16374, 16375, 16248, 16121, 15994, 15867, 15740, 15613, 15486, 15359, 15487, 15614, 15741, 15868, 15995, 16122, 16249, 16376, 16377, 16250, 16123, 15996, 15869, 15742, 15615, 15743, 15870, 15997, 16124, 16251, 16378, 16379, 16252, 16125, 15998, 15871, 15999, 16126, 16253, 16380, 16381, 16254, 16127, 16255, 16382, 16383
};