/*
    Copyright (c) 2014, Lukas Holecek <hluk@email.cz>

    This file is part of CopyQ.

    CopyQ is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    CopyQ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CopyQ.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef ITEMEDITOR_H
#define ITEMEDITOR_H

#include <QDateTime>
#include <QFileInfo>
#include <QObject>
#include <QString>

class QProcess;
class QTimer;

class ItemEditor : public QObject
{
    Q_OBJECT

    public:
        ItemEditor(const QByteArray &data, const QString &mime, const QString &editor,
                   QObject *parent = NULL);
        ~ItemEditor();

    public slots:
        /**
         * Execute editor process.
         * @retval true   Editor successfully opened.
         * @retval false  An error occured (failed to create temporary file), editor was not opened.
         */
        bool start();

    signals:
        /**
         * File was modified.
         * @param data  modified data
         * @param mime  MIME type of the data
         */
        void fileModified(const QByteArray &data, const QString &mime);

        /**
         * Editor was closed.
         * @param who  pointer to this object
         */
        void closed(QObject *who);

        /**
         * Failed to run editor command.
         */
        void error(const QString &errorString);

    private slots:
        /**
         * Close editor.
         */
        void close();

        void onError();

        void onTimer();

    private:
        /** Return true only if file was modified and reset this status. */
        bool fileModified();

        void emitError(const QString &errorString);

        QByteArray m_data;
        QString m_mime;
        // hash of original string (saves some memory)
        uint m_hash;

        QString m_editorcmd;
        QProcess *m_editor;
        QTimer *m_timer;

        QFileInfo m_info;
        QDateTime m_lastmodified;
        qint64 m_lastSize;
        bool m_modified;
};

#endif // ITEMEDITOR_H
