

/*
 *  Author: Arvin Schnell
 */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <iostream>

using std::cerr;

#include "pcm-alsa.h"


PCMALSA::PCMALSA (const char* name, pcm_type_t type, pcm_mode_t mode,
		  pcm_format_t format, int channels, int rate, int buffer_size)
    : PCM (name, type, mode, format, channels, rate, buffer_size),
      handle (0)
{
    if (channels != 1 && channels != 2) {
	cerr << "error: Illegal number of channels\n";
	return;
    }

    int ret;

    snd_pcm_stream_t stream = mode == WRITE ? SND_PCM_STREAM_PLAYBACK :
	SND_PCM_STREAM_CAPTURE;
    ret = snd_pcm_open (&handle, name, stream, SND_PCM_NONBLOCK);
    if (ret < 0) {
	cerr << "error: Can't open PCM device " << name << ": "
	     << snd_strerror (ret) << '\n';
	handle = 0;
	return;
    }

    ret = snd_pcm_nonblock (handle, 0);
    if (ret < 0) {
	cerr << "error: Can't set blocking mode: " << snd_strerror (ret) << '\n';
	return;
    }

    snd_pcm_hw_params_t* hwparams;
    snd_pcm_hw_params_alloca (&hwparams);

    ret = snd_pcm_hw_params_any (handle, hwparams);
    if (ret < 0) {
	cerr << "error: Can't configure the PCM device: "
	     << snd_strerror (ret) << '\n';
	return;
    }

    ret = snd_pcm_hw_params_set_access (handle, hwparams,
					SND_PCM_ACCESS_RW_INTERLEAVED);
    if (ret < 0) {
	cerr << "error: Can't set access: " << snd_strerror (ret) << '\n';
	return;
    }

    snd_pcm_format_t format_array[3] = { SND_PCM_FORMAT_S8, SND_PCM_FORMAT_S16,
					 SND_PCM_FORMAT_S32 };
    ret = snd_pcm_hw_params_set_format (handle, hwparams, format_array[format]);
    if (ret < 0) {
	cerr << "error: Can't set format (resolution): "
	     << snd_strerror (ret) << '\n';
	return;
    }

    ret = snd_pcm_hw_params_set_rate_near (handle, hwparams, rate, 0);
    if (ret < 0) {
	cerr << "error: Can't set rate: " << snd_strerror (ret) << '\n';
	return;
    }

    if (abs (rate - ret) > 1) {  // we can tolerate 1 Hz difference
	cerr << "error: Can't set requested rate: " << snd_strerror (ret) << '\n'
	     << "requested rate: " << rate << " Hz, actual rate: "
	     << ret << " Hz\n";
	return;
    }

    ret = snd_pcm_hw_params_set_channels (handle, hwparams, channels);
    if (ret < 0) {
	cerr << "error: Can't set number of channels: " << snd_strerror (ret)
	     << '\n';
	return;
    }

    ret = snd_pcm_hw_params_set_periods (handle, hwparams, 2, 0);
    if (ret < 0) {
	cerr << "error: Can't set number of periods: " << snd_strerror (ret)
	     << '\n';
	return;
    }

    ret = snd_pcm_hw_params_set_buffer_size (handle, hwparams, buffer_size);
    if (ret < 0) {
	cerr << "error: Can't set buffer size: " << snd_strerror (ret) << '\n';
	return;
    }

    ret = snd_pcm_hw_params (handle, hwparams);
    if (ret < 0) {
	cerr << "error: Can't set HW params: " << snd_strerror (ret) << '\n';
	return;
    }

    ok = true;
}


PCMALSA::~PCMALSA ()
{
    if (handle)
	snd_pcm_close (handle);
}


void
PCMALSA::info () const
{
    cerr << "driver: alsa 0.9\n";
    PCM::info ();
}


size_t
PCMALSA::size ()
{
    return (size_t)(-1);
}


off_t
PCMALSA::seek (off_t, int)
{
    return (off_t)(-1);
}


size_t
PCMALSA::read (void* buffer, size_t frames)
{
    int ret = snd_pcm_readi (handle, buffer, frames);

    if (ret < 0) {
	if (ret == -EPIPE) {
	    cerr << "warning: Buffer overrun: "
		 << snd_strerror (ret) << '\n';;
	    snd_pcm_prepare (handle);
	} else {
	    cerr << "error: " << snd_strerror (ret) << '\n';
	    return (size_t)(-1);
	}
    }

    return frames;
}


size_t
PCMALSA::write (void* buffer, size_t frames)
{
    int ret = snd_pcm_writei (handle, buffer, frames);

    if (ret < 0) {
	if (ret == -EPIPE) {
	    cerr << "warning: Buffer underrun: "
		 << snd_strerror (ret) << '\n';
	    snd_pcm_prepare (handle);
	} else {
	    cerr << "error: " << snd_strerror (ret) << '\n';
	    return (size_t)(-1);
	}
    }

    return frames;
}
