

/*
 *  Author: Arvin Schnell
 */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <iostream>

using std::cerr;

#include "pcm-wav.h"


#ifndef __BYTE_ORDER
#  error "undefined endianness"
#elif __BYTE_ORDER != __BIG_ENDIAN && __BYTE_ORDER != __LITTLE_ENDIAN
#  error "unsupported endianness"
#endif


PCMWAV::PCMWAV (const char* name, pcm_type_t type, pcm_mode_t mode,
		pcm_format_t format, int channels, int rate, int buffer_size)
    : PCM (name, type, mode, format, channels, rate, buffer_size),
      fd (-1)
{
    if (channels != 2) {
	cerr << "error: Illegal number of channels\n";
	return;
    }

    if (format != S16) {
	cerr << "error: Illegal format (resolution)\n";
	return;
    }

    if (rate != 44100) {
	cerr << "error: Illegal rate\n";
	return;
    }

    if (buffer_size != 1) {
	cerr << "error: Illegal buffer size\n";
	return;
    }

    switch (mode)
    {
	case READ: {

	    fd = open (name, O_RDONLY);
	    if (fd == -1) {
		cerr << "error: Can't open " << name << ": "
		     << strerror (errno) << '\n';
		return;
	    }

	    struct stat fs;
	    if (fstat (fd, &fs) == -1) {
		cerr << "error: Can't stat " << name << ": "
		     << strerror (errno) << '\n';
		return;
	    }

	    if ((size_t)(fs.st_size) < sizeof (WAVHeader)) {
		cerr << "error: File too small " << name << '\n';
		return;
	    }

	    if ((fs.st_size - sizeof (WAVHeader)) % get_bytes_per_frame () != 0) {
		cerr << "error: Illegal size of " << name << '\n';
		return;
	    }

	    WAVHeader header;
	    if (::read (fd, &header, sizeof (header)) != sizeof (WAVHeader)) {
		cerr << "error: Reading header failed: " << strerror (errno)
		     << '\n';
		return;
	    }

	    if (!check_header (header, fs.st_size)) {
		cerr << "error: Male formed header\n";
		return;
	    }

	} break;

	case WRITE: {

	    fd = open (name, O_WRONLY | O_CREAT | O_TRUNC, 0666);
	    if (fd == -1) {
		cerr << "error: Can't open " << name << ": "
		     << strerror (errno) << '\n';
		return;
	    }

	    WAVHeader header;
	    fill_header (&header, 0);
	    if (::write (fd, &header, sizeof (header)) != sizeof (WAVHeader)) {
		cerr << "error: Writing header failed: " << strerror (errno)
		     << '\n';
		return;
	    }

	} break;
    }

    ok = true;
}


PCMWAV::~PCMWAV ()
{
    if (fd > 0)
    {
	switch (mode)
	{
	    case READ: {
	    } break;

	    case WRITE: {

		struct stat fs;
		fstat (fd, &fs);

		WAVHeader header;
		fill_header (&header, fs.st_size);

		if (::lseek (fd, 0, SEEK_SET) != 0) {
		    cerr << "error: Seeking failed: " << strerror (errno)
			 << '\n';
		}

		if (::write (fd, &header, sizeof (header)) != sizeof (WAVHeader)) {
		    cerr << "error: Writing header failed: " << strerror (errno)
			 << '\n';
		}

	    } break;
	}

	close (fd);
    }
}


void
PCMWAV::info () const
{
    cerr << "driver: wav\n";
    PCM::info ();
}


size_t
PCMWAV::size ()
{
    struct stat fs;
    if (fstat (fd, &fs) == -1)
	return (size_t)(-1);

    return (fs.st_size - sizeof (WAVHeader)) / get_bytes_per_frame ();
}


off_t
PCMWAV::seek (off_t offset, int whence)
{
    off_t i = offset * get_bytes_per_frame () + sizeof (WAVHeader);

    off_t j = ::lseek (fd, i, whence);
    if (j == (off_t)(-1))
	return (off_t)(-1);

    off_t k = (j - sizeof (WAVHeader)) / get_bytes_per_frame ();

    return k;
}


size_t
PCMWAV::read (void* buffer, size_t frames)
{
    size_t i = frames * get_bytes_per_frame ();

    size_t j = ::read (fd, buffer, i);
    if (j == (size_t)(-1))
	return (size_t)(-1);

    size_t k = j / get_bytes_per_frame ();
#if __BYTE_ORDER == __BIG_ENDIAN
    swap_buffer (buffer, k);
#endif
    return k;
}


size_t
PCMWAV::write (void* buffer, size_t frames)
{
#if __BYTE_ORDER == __BIG_ENDIAN
    swap_buffer (buffer, frames);
#endif

    size_t i = frames * get_bytes_per_frame ();

    size_t j = ::write (fd, buffer, i);
    if (j == (size_t)(-1))
	return (size_t)(-1);

    size_t k = j / get_bytes_per_frame ();
    return k;
}


bool
PCMWAV::check_header (const WAVHeader& header, size_t file_size) const
{
    size_t data_size = file_size - sizeof (WAVHeader);

    if (memcmp (&header.chunk_id, "RIFF", 4) != 0)
	return false;

    if (header.chunk_size != uint32_to_le (data_size + sizeof (WAVHeader) - 8))
	return false;

    if (memcmp (&header.format, "WAVE", 4) != 0)
	return false;

    if (memcmp (&header.sub_chunk_id, "fmt ", 4) != 0)
	return false;

    if (header.sub_chunk_size != uint32_to_le (16))
	return false;

    if (header.audio_format != uint16_to_le (1))
	return false;

    if (header.num_channels != uint16_to_le (channels))
	return false;

    if (header.sample_rate != uint32_to_le (rate))
	return false;

    if (header.byte_rate != uint32_to_le (rate * channels * (1 << format)))
	return false;

    if (header.block_align != uint16_to_le (channels * (1 << format)))
	return false;

    if (header.bit_per_sample != uint16_to_le (8 << format))
	return false;

    if (memcmp (&header.data_chunk_id, "data", 4) != 0)
	return false;

    if (header.data_chunk_size != uint32_to_le (data_size))
	return false;

    return true;
}


void
PCMWAV::fill_header (WAVHeader* header, size_t file_size) const
{
    size_t data_size = file_size - sizeof (WAVHeader);

    memcpy (&header->chunk_id, "RIFF", 4);
    header->chunk_size = uint32_to_le (data_size + sizeof (WAVHeader) - 8);
    memcpy (&header->format, "WAVE", 4);
    memcpy (&header->sub_chunk_id, "fmt ", 4);
    header->sub_chunk_size = uint32_to_le (16);
    header->audio_format = uint16_to_le (1);
    header->num_channels = uint16_to_le (channels);
    header->sample_rate = uint32_to_le (rate);
    header->byte_rate = uint32_to_le (rate * channels * (1 << format));
    header->block_align = uint16_to_le (channels * (1 << format));
    header->bit_per_sample = uint16_to_le (8 << format);
    memcpy (&header->data_chunk_id, "data", 4);
    header->data_chunk_size = uint32_to_le (data_size);
}
