/*
 * SYSTEMTIME functions
 *
 * Copyright (C) 2009-2022, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <byte_stream.h>
#include <memory.h>
#include <types.h>

#include "libfdatetime_definitions.h"
#include "libfdatetime_date_time_values.h"
#include "libfdatetime_libcerror.h"
#include "libfdatetime_systemtime.h"
#include "libfdatetime_types.h"

/* Creates a SYSTEMTIME
 * Make sure the value systemtime is referencing, is set to NULL
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_initialize(
     libfdatetime_systemtime_t **systemtime,
     libcerror_error_t **error )
{
	libfdatetime_internal_systemtime_t *internal_systemtime = NULL;
	static char *function                                   = "libfdatetime_systemtime_initialize";

	if( systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	if( *systemtime != NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_ALREADY_SET,
		 "%s: invalid SYSTEMTIME value already set.",
		 function );

		return( -1 );
	}
	internal_systemtime = memory_allocate_structure(
	                       libfdatetime_internal_systemtime_t );

	if( internal_systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_MEMORY,
		 LIBCERROR_MEMORY_ERROR_INSUFFICIENT,
		 "%s: unable to create SYSTEMTIME.",
		 function );

		goto on_error;
	}
	if( memory_set(
	     internal_systemtime,
	     0,
	     sizeof( libfdatetime_internal_systemtime_t ) ) == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_MEMORY,
		 LIBCERROR_MEMORY_ERROR_SET_FAILED,
		 "%s: unable to clear SYSTEMTIME.",
		 function );

		goto on_error;
	}
	*systemtime = (libfdatetime_systemtime_t *) internal_systemtime;

	return( 1 );

on_error:
	if( internal_systemtime != NULL )
	{
		memory_free(
		 internal_systemtime );
	}
	return( -1 );
}

/* Frees a SYSTEMTIME
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_free(
     libfdatetime_systemtime_t **systemtime,
     libcerror_error_t **error )
{
	libfdatetime_internal_systemtime_t *internal_systemtime = NULL;
	static char *function                                   = "libfdatetime_systemtime_free";

	if( systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	if( *systemtime != NULL )
	{
		internal_systemtime = (libfdatetime_internal_systemtime_t *) *systemtime;
		*systemtime         = NULL;

		memory_free(
		 internal_systemtime );
	}
	return( 1 );
}

/* Converts a byte stream into a systemtime
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_copy_from_byte_stream(
     libfdatetime_systemtime_t *systemtime,
     const uint8_t *byte_stream,
     size_t byte_stream_size,
     int byte_order,
     libcerror_error_t **error )
{
	libfdatetime_internal_systemtime_t *internal_systemtime = NULL;
	static char *function                                   = "libfdatetime_systemtime_copy_from_byte_stream";

	if( systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	internal_systemtime = (libfdatetime_internal_systemtime_t *) systemtime;

	if( byte_stream == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream.",
		 function );

		return( -1 );
	}
	if( byte_stream_size < 16 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: byte stream too small.",
		 function );

		return( -1 );
	}
	if( byte_stream_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: byte stream size exceeds maximum.",
		 function );

		return( -1 );
	}
	if( ( byte_order != LIBFDATETIME_ENDIAN_BIG )
	 && ( byte_order != LIBFDATETIME_ENDIAN_LITTLE ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported byte order.",
		 function );

		return( -1 );
	}
	if( byte_order == LIBFDATETIME_ENDIAN_LITTLE )
	{
		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->year );

		byte_stream += 2;

		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->month );

		byte_stream += 2;

		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->day_of_week );

		byte_stream += 2;

		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->day_of_month );

		byte_stream += 2;

		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->hours );

		byte_stream += 2;

		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->minutes );

		byte_stream += 2;

		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->seconds );

		byte_stream += 2;

		byte_stream_copy_to_uint16_little_endian(
		 byte_stream,
		 internal_systemtime->milli_seconds );
	}
	else if( byte_order == LIBFDATETIME_ENDIAN_BIG )
	{
		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->year );

		byte_stream += 2;

		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->month );

		byte_stream += 2;

		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->day_of_week );

		byte_stream += 2;

		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->day_of_month );

		byte_stream += 2;

		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->hours );

		byte_stream += 2;

		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->minutes );

		byte_stream += 2;

		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->seconds );

		byte_stream += 2;

		byte_stream_copy_to_uint16_big_endian(
		 byte_stream,
		 internal_systemtime->milli_seconds );
	}
	return( 1 );
}

/* Converts a SYSTEMTIME into date time values
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_internal_systemtime_copy_to_date_time_values(
     libfdatetime_internal_systemtime_t *internal_systemtime,
     libfdatetime_date_time_values_t *date_time_values,
     libcerror_error_t **error )
{
	static char *function = "libfdatetime_internal_systemtime_copy_to_date_time_values";
	uint8_t days_in_month = 0;

	if( internal_systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	if( date_time_values == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid date time values.",
		 function );

		return( -1 );
	}
	if( internal_systemtime->milli_seconds > 999 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - milli seconds value out of bounds.",
		 function );

		return( -1 );
	}
	date_time_values->nano_seconds  = 0;
	date_time_values->micro_seconds = 0;
	date_time_values->milli_seconds = internal_systemtime->milli_seconds;

	if( internal_systemtime->seconds > 59 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - seconds value out of bounds.",
		 function );

		return( -1 );
	}
	date_time_values->seconds = (uint8_t) internal_systemtime->seconds;

	if( internal_systemtime->minutes > 59 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - minutes value out of bounds.",
		 function );

		return( -1 );
	}
	date_time_values->minutes = (uint8_t) internal_systemtime->minutes;

	if( internal_systemtime->hours > 23 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - hours value out of bounds.",
		 function );

		return( -1 );
	}
	date_time_values->hours = (uint8_t) internal_systemtime->hours;

	if( internal_systemtime->year > 9999 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - year value out of bounds.",
		 function );

		return( -1 );
	}
	date_time_values->year = (uint16_t) internal_systemtime->year;

	if( ( internal_systemtime->month == 0 )
	 || ( internal_systemtime->month > 12 ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - month value out of bounds.",
		 function );

		return( -1 );
	}
	date_time_values->month = (uint8_t) internal_systemtime->month;

	if( internal_systemtime->day_of_week > 6 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - day of week value out of bounds.",
		 function );

		return( -1 );
	}
	/* February (2)
	 */
	if( date_time_values->month == 2 )
	{
		if( ( ( ( date_time_values->year % 4 ) == 0 )
		  &&  ( ( date_time_values->year % 100 ) != 0 ) )
		 || ( ( date_time_values->year % 400 ) == 0 ) )
		{
			days_in_month = 29;
		}
		else
		{
			days_in_month = 28;
		}
	}
	/* April (4), June (6), September (9), November (11)
	 */
	else if( ( date_time_values->month == 4 )
	      || ( date_time_values->month == 6 )
	      || ( date_time_values->month == 9 )
	      || ( date_time_values->month == 11 ) )
	{
		days_in_month = 30;
	}
	/* January (1), March (3), May (5), July (7), August (8), October (10), December (12)
	 */
	else if( ( date_time_values->month == 1 )
	      || ( date_time_values->month == 3 )
	      || ( date_time_values->month == 5 )
	      || ( date_time_values->month == 7 )
	      || ( date_time_values->month == 8 )
	      || ( date_time_values->month == 10 )
	      || ( date_time_values->month == 12 ) )
	{
		days_in_month = 31;
	}
	/* This should never happen, but just in case
	 */
	else
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported month: %d.",
		 function,
		 date_time_values->month );

		return( -1 );
	}
	if( internal_systemtime->day_of_month > (uint16_t) days_in_month )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid SYSTEMTIME - day of month value out of bounds.",
		 function );

		return( -1 );
	}
	date_time_values->day = (uint8_t) internal_systemtime->day_of_month;

	return( 1 );
}

/* Deterimes the size of the string for the SYSTEMTIME
 * The string size includes the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_get_string_size(
     libfdatetime_systemtime_t *systemtime,
     size_t *string_size,
     uint32_t string_format_flags,
     libcerror_error_t **error )
{
	libfdatetime_date_time_values_t date_time_values;

	static char *function = "libfdatetime_systemtime_get_string_size";
	int result            = 0;

	if( systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	if( string_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid string size.",
		 function );

		return( -1 );
	}
	result = libfdatetime_internal_systemtime_copy_to_date_time_values(
	          (libfdatetime_internal_systemtime_t *) systemtime,
	          &date_time_values,
	          error );

	if( result != 1 )
	{
#if defined( HAVE_DEBUG_OUTPUT )
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
		 "%s: unable to set date time values.",
		 function );

/* TODO debug print error */

#endif /* defined( HAVE_DEBUG_OUTPUT ) */

		if( ( error != NULL )
		 && ( *error != NULL ) )
		{
			libcerror_error_free(
			 error );
		}
	}
	else
	{
		result = libfdatetime_date_time_values_get_string_size(
		          &date_time_values,
		          string_size,
		          string_format_flags,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to get string size.",
			 function );

			return( -1 );
		}
	}
	if( result != 1 )
	{
		/* Make sure the string can hold the hexadecimal representation of the SYSTEMTIME
		 */
		*string_size = 58;
	}
	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-8 string in hexadecimal representation
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_internal_systemtime_copy_to_utf8_string_in_hexadecimal(
     libfdatetime_internal_systemtime_t *internal_systemtime,
     uint8_t *utf8_string,
     size_t utf8_string_size,
     size_t *utf8_string_index,
     libcerror_error_t **error )
{
	static char *function = "libfdatetime_internal_systemtime_copy_to_utf8_string_in_hexadecimal";
	size_t string_index   = 0;
	uint8_t byte_value    = 0;
	int8_t byte_shift     = 0;

	if( internal_systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	if( utf8_string == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-8 string.",
		 function );

		return( -1 );
	}
	if( utf8_string_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid UTF-8 string size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( utf8_string_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-8 string index.",
		 function );

		return( -1 );
	}
	if( ( utf8_string_size < 58 )
	 || ( *utf8_string_index > ( utf8_string_size - 58 ) ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: UTF-8 string is too small.",
		 function );

		return( -1 );
	}
	string_index = *utf8_string_index;

	utf8_string[ string_index++ ] = (uint8_t) '(';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->year >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ' ';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->month >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ' ';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->day_of_week >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ' ';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->day_of_month >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ' ';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->hours >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ' ';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->minutes >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ' ';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->seconds >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ' ';
	utf8_string[ string_index++ ] = (uint8_t) '0';
	utf8_string[ string_index++ ] = (uint8_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->milli_seconds >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf8_string[ string_index++ ] = (uint8_t) '0' + byte_value;
		}
		else
		{
			utf8_string[ string_index++ ] = (uint8_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf8_string[ string_index++ ] = (uint8_t) ')';

	utf8_string[ string_index++ ] = 0;

	*utf8_string_index = string_index;

	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-8 string
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_copy_to_utf8_string(
     libfdatetime_systemtime_t *systemtime,
     uint8_t *utf8_string,
     size_t utf8_string_size,
     uint32_t string_format_flags,
     libcerror_error_t **error )
{
	static char *function    = "libfdatetime_systemtime_copy_to_utf8_string";
	size_t utf8_string_index = 0;

	if( libfdatetime_systemtime_copy_to_utf8_string_with_index(
	     systemtime,
	     utf8_string,
	     utf8_string_size,
	     &utf8_string_index,
	     string_format_flags,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_COPY_FAILED,
		 "%s: unable to copy SYSTEMTIME to UTF-8 string.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-8 string
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_copy_to_utf8_string_with_index(
     libfdatetime_systemtime_t *systemtime,
     uint8_t *utf8_string,
     size_t utf8_string_size,
     size_t *utf8_string_index,
     uint32_t string_format_flags,
     libcerror_error_t **error )
{
	libfdatetime_date_time_values_t date_time_values;

	libfdatetime_internal_systemtime_t *internal_systemtime = NULL;
	static char *function                                   = "libfdatetime_systemtime_copy_to_utf8_string_with_index";
	int result                                              = 0;

	if( systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	internal_systemtime = (libfdatetime_internal_systemtime_t *) systemtime;

	result = libfdatetime_internal_systemtime_copy_to_date_time_values(
	          internal_systemtime,
	          &date_time_values,
	          error );

	if( result != 1 )
	{
#if defined( HAVE_DEBUG_OUTPUT )
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
		 "%s: unable to set date time values.",
		 function );

/* TODO debug print error */

#endif /* defined( HAVE_DEBUG_OUTPUT ) */

		if( ( error != NULL )
		 && ( *error != NULL ) )
		{
			libcerror_error_free(
			 error );
		}
	}
	else
	{
		result = libfdatetime_date_time_values_copy_to_utf8_string_with_index(
		          &date_time_values,
		          utf8_string,
		          utf8_string_size,
		          utf8_string_index,
		          string_format_flags,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
			 "%s: unable to copy date time values to UTF-8 string.",
			 function );

			return( -1 );
		}
	}
	if( result != 1 )
	{
		result = libfdatetime_internal_systemtime_copy_to_utf8_string_in_hexadecimal(
		          internal_systemtime,
		          utf8_string,
		          utf8_string_size,
		          utf8_string_index,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
			 "%s: unable to SYSTEMTIME to hexadecimal UTF-8 string.",
			 function );

			return( -1 );
		}
	}
	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-16 string in hexadecimal representation
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_internal_systemtime_copy_to_utf16_string_in_hexadecimal(
     libfdatetime_internal_systemtime_t *internal_systemtime,
     uint16_t *utf16_string,
     size_t utf16_string_size,
     size_t *utf16_string_index,
     libcerror_error_t **error )
{
	static char *function = "libfdatetime_internal_systemtime_copy_to_utf16_string_in_hexadecimal";
	size_t string_index   = 0;
	uint8_t byte_value    = 0;
	int8_t byte_shift     = 0;

	if( internal_systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	if( utf16_string == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-16 string.",
		 function );

		return( -1 );
	}
	if( utf16_string_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid UTF-16 string size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( utf16_string_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-16 string index.",
		 function );

		return( -1 );
	}
	if( ( utf16_string_size < 58 )
	 || ( *utf16_string_index > ( utf16_string_size - 58 ) ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: UTF-16 string is too small.",
		 function );

		return( -1 );
	}
	string_index = *utf16_string_index;

	utf16_string[ string_index++ ] = (uint16_t) '(';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->year >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ' ';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->month >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ' ';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->day_of_week >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ' ';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->day_of_month >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ' ';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->hours >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ' ';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->minutes >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ' ';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->seconds >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ' ';
	utf16_string[ string_index++ ] = (uint16_t) '0';
	utf16_string[ string_index++ ] = (uint16_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->milli_seconds >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf16_string[ string_index++ ] = (uint16_t) '0' + byte_value;
		}
		else
		{
			utf16_string[ string_index++ ] = (uint16_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf16_string[ string_index++ ] = (uint16_t) ')';

	utf16_string[ string_index++ ] = 0;

	*utf16_string_index = string_index;

	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-16 string
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_copy_to_utf16_string(
     libfdatetime_systemtime_t *systemtime,
     uint16_t *utf16_string,
     size_t utf16_string_size,
     uint32_t string_format_flags,
     libcerror_error_t **error )
{
	static char *function     = "libfdatetime_systemtime_copy_to_utf16_string";
	size_t utf16_string_index = 0;

	if( libfdatetime_systemtime_copy_to_utf16_string_with_index(
	     systemtime,
	     utf16_string,
	     utf16_string_size,
	     &utf16_string_index,
	     string_format_flags,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_COPY_FAILED,
		 "%s: unable to copy SYSTEMTIME to UTF-16 string.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-16 string
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_copy_to_utf16_string_with_index(
     libfdatetime_systemtime_t *systemtime,
     uint16_t *utf16_string,
     size_t utf16_string_size,
     size_t *utf16_string_index,
     uint32_t string_format_flags,
     libcerror_error_t **error )
{
	libfdatetime_date_time_values_t date_time_values;

	libfdatetime_internal_systemtime_t *internal_systemtime = NULL;
	static char *function                                   = "libfdatetime_systemtime_copy_to_utf16_string_with_index";
	int result                                              = 0;

	if( systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	internal_systemtime = (libfdatetime_internal_systemtime_t *) systemtime;

	result = libfdatetime_internal_systemtime_copy_to_date_time_values(
	          internal_systemtime,
	          &date_time_values,
	          error );

	if( result != 1 )
	{
#if defined( HAVE_DEBUG_OUTPUT )
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
		 "%s: unable to set date time values.",
		 function );

/* TODO debug print error */

#endif /* defined( HAVE_DEBUG_OUTPUT ) */

		if( ( error != NULL )
		 && ( *error != NULL ) )
		{
			libcerror_error_free(
			 error );
		}
	}
	else
	{
		result = libfdatetime_date_time_values_copy_to_utf16_string_with_index(
		          &date_time_values,
		          utf16_string,
		          utf16_string_size,
		          utf16_string_index,
		          string_format_flags,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
			 "%s: unable to copy date time values to UTF-16 string.",
			 function );

			return( -1 );
		}
	}
	if( result != 1 )
	{
		result = libfdatetime_internal_systemtime_copy_to_utf16_string_in_hexadecimal(
		          internal_systemtime,
		          utf16_string,
		          utf16_string_size,
		          utf16_string_index,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
			 "%s: unable to SYSTEMTIME to hexadecimal UTF-16 string.",
			 function );

			return( -1 );
		}
	}
	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-32 string in hexadecimal representation
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_internal_systemtime_copy_to_utf32_string_in_hexadecimal(
     libfdatetime_internal_systemtime_t *internal_systemtime,
     uint32_t *utf32_string,
     size_t utf32_string_size,
     size_t *utf32_string_index,
     libcerror_error_t **error )
{
	static char *function = "libfdatetime_internal_systemtime_copy_to_utf32_string_in_hexadecimal";
	size_t string_index   = 0;
	uint8_t byte_value    = 0;
	int8_t byte_shift     = 0;

	if( internal_systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	if( utf32_string == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-32 string.",
		 function );

		return( -1 );
	}
	if( utf32_string_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid UTF-32 string size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( utf32_string_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-32 string index.",
		 function );

		return( -1 );
	}
	if( ( utf32_string_size < 58 )
	 || ( *utf32_string_index > ( utf32_string_size - 58 ) ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: UTF-32 string is too small.",
		 function );

		return( -1 );
	}
	string_index = *utf32_string_index;

	utf32_string[ string_index++ ] = (uint32_t) '(';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->year >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ' ';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->month >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ' ';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->day_of_week >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ' ';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->day_of_month >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ' ';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->hours >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ' ';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->minutes >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ' ';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->seconds >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ' ';
	utf32_string[ string_index++ ] = (uint32_t) '0';
	utf32_string[ string_index++ ] = (uint32_t) 'x';

	byte_shift = 14;

	do
	{
		byte_value = ( internal_systemtime->milli_seconds >> byte_shift ) & 0x0f;

		if( byte_value <= 9 )
		{
			utf32_string[ string_index++ ] = (uint32_t) '0' + byte_value;
		}
		else
		{
			utf32_string[ string_index++ ] = (uint32_t) 'a' + byte_value - 10;
		}
		byte_shift -= 4;
	}
	while( byte_shift >= 0 );

	utf32_string[ string_index++ ] = (uint32_t) ')';

	utf32_string[ string_index++ ] = 0;

	*utf32_string_index = string_index;

	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-32 string
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_copy_to_utf32_string(
     libfdatetime_systemtime_t *systemtime,
     uint32_t *utf32_string,
     size_t utf32_string_size,
     uint32_t string_format_flags,
     libcerror_error_t **error )
{
	static char *function     = "libfdatetime_systemtime_copy_to_utf32_string";
	size_t utf32_string_index = 0;

	if( libfdatetime_systemtime_copy_to_utf32_string_with_index(
	     systemtime,
	     utf32_string,
	     utf32_string_size,
	     &utf32_string_index,
	     string_format_flags,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_COPY_FAILED,
		 "%s: unable to copy SYSTEMTIME to UTF-32 string.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Converts the SYSTEMTIME into an UTF-32 string
 * The string size should include the end of string character
 * Returns 1 if successful or -1 on error
 */
int libfdatetime_systemtime_copy_to_utf32_string_with_index(
     libfdatetime_systemtime_t *systemtime,
     uint32_t *utf32_string,
     size_t utf32_string_size,
     size_t *utf32_string_index,
     uint32_t string_format_flags,
     libcerror_error_t **error )
{
	libfdatetime_date_time_values_t date_time_values;

	libfdatetime_internal_systemtime_t *internal_systemtime = NULL;
	static char *function                                   = "libfdatetime_systemtime_copy_to_utf32_string_with_index";
	int result                                              = 0;

	if( systemtime == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid SYSTEMTIME.",
		 function );

		return( -1 );
	}
	internal_systemtime = (libfdatetime_internal_systemtime_t *) systemtime;

	result = libfdatetime_internal_systemtime_copy_to_date_time_values(
	          internal_systemtime,
	          &date_time_values,
	          error );

	if( result != 1 )
	{
#if defined( HAVE_DEBUG_OUTPUT )
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
		 "%s: unable to set date time values.",
		 function );

/* TODO debug print error */

#endif /* defined( HAVE_DEBUG_OUTPUT ) */

		if( ( error != NULL )
		 && ( *error != NULL ) )
		{
			libcerror_error_free(
			 error );
		}
	}
	else
	{
		result = libfdatetime_date_time_values_copy_to_utf32_string_with_index(
		          &date_time_values,
		          utf32_string,
		          utf32_string_size,
		          utf32_string_index,
		          string_format_flags,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
			 "%s: unable to copy date time values to UTF-32 string.",
			 function );

			return( -1 );
		}
	}
	if( result != 1 )
	{
		result = libfdatetime_internal_systemtime_copy_to_utf32_string_in_hexadecimal(
		          internal_systemtime,
		          utf32_string,
		          utf32_string_size,
		          utf32_string_index,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
			 "%s: unable to SYSTEMTIME to hexadecimal UTF-32 string.",
			 function );

			return( -1 );
		}
	}
	return( 1 );
}

